import { LaunchNative } from "capacitor-launch-native";
import { Reader } from "capacitor-reader";
import { use } from "react";

import { DarkContext } from "#/core/GlobalStyles";
import {
  notifyStatusTapThatBrowserWasClosed,
  notifyStatusTapThatBrowserWasOpened,
} from "#/core/listeners/statusTap";
import { isAndroid } from "#/helpers/device";
import store from "#/store";

import { shouldOpenWithInAppBrowser } from "./InAppExternalLink";

export default function useNativeBrowser() {
  const isDark = use(DarkContext);

  return async function openNativeBrowser(href: string) {
    const state = store.getState();

    const { usingSystemDarkMode, pureBlack } = state.settings.appearance.dark;
    const alwaysUseReaderMode =
      state.settings.general.safari.alwaysUseReaderMode;
    const preferNativeApps = state.settings.general.preferNativeApps;

    const toolbarColor = (() => {
      if (usingSystemDarkMode) return undefined;

      if (isAndroid()) {
        if (isDark) {
          if (pureBlack) return "#000000";

          return "#0f1419";
        }

        return "#ffffff";
      } else {
        // iOS clamps so #000 is not true black
        if (isDark) return "#000000";

        return "#ffffff";
      }
    })();

    if (preferNativeApps) {
      const { completed } = await LaunchNative.attempt({ url: href });
      if (completed) return;
    }

    notifyStatusTapThatBrowserWasOpened();

    try {
      await Reader.open({
        url: href,
        toolbarColor,
        entersReaderIfAvailable: alwaysUseReaderMode,
      });
    } catch (e) {
      notifyStatusTapThatBrowserWasClosed();
      throw e;
    }
  };
}

export function useOpenNativeBrowserIfPreferred() {
  const _openNativeBrowser = useNativeBrowser();

  return async function openNativeBrowser(href: string) {
    if (shouldOpenWithInAppBrowser(href)) {
      _openNativeBrowser(href);
      return;
    }

    window.open(href, "_blank");
  };
}
