import * as fs from 'node:fs';
import path from 'node:path';
import slash from 'slash';
import { lookup } from 'mime-types';
import constants from '../config/constants.js';
const getExtension = (file) => path.extname(file).replace('.', '');
const isImageFile = (file) => [
    'apng',
    'bmp',
    'gif',
    'ico',
    'cur',
    'jpg',
    'jpeg',
    'jfif',
    'pjpeg',
    'pjp',
    'png',
    'svg',
    'webp',
].includes(getExtension(file));
const isHtmlFile = (file) => ['html', 'htm'].includes(getExtension(file));
const convertBackslashPathToSlashPath = (backSlashPath) => slash(backSlashPath);
const getAppDir = () => {
    let appPath;
    try {
        appPath = require.resolve('pwa-asset-generator');
    }
    catch (e) {
        if (require.main !== undefined) {
            appPath = require.main.filename;
        }
    }
    return path.join(path.dirname(appPath ?? ''), '..');
};
const getShellHtmlFilePath = () => `${getAppDir()}/static/shell.html`;
const getImageSavePath = (imageName, outputFolder, ext, maskable, isMaskableIcon) => {
    const getMaskablePrefix = () => {
        if (!isMaskableIcon) {
            return '';
        }
        if (maskable) {
            return '.maskable';
        }
        return '';
    };
    return convertBackslashPathToSlashPath(path.join(outputFolder, `${imageName}${getMaskablePrefix()}.${ext}`));
};
const fileUrl = (filePath) => {
    let pathName = filePath;
    pathName = pathName.replace(/\\/g, '/');
    // Windows drive letter must be prefixed with a slash
    if (pathName[0] !== '/') {
        pathName = `/${pathName}`;
    }
    return encodeURI(`file://${pathName}`).replace(/[?#]/g, encodeURIComponent);
};
const getFileUrlOfPath = (source) => fileUrl(path.resolve(source));
const getRelativeFilePath = (referenceFilePath, filePath) => path.relative(path.dirname(path.resolve(referenceFilePath)), path.resolve(filePath));
const getRelativeImagePath = (outputFilePath, imagePath) => {
    if (outputFilePath) {
        return convertBackslashPathToSlashPath(getRelativeFilePath(outputFilePath, imagePath));
    }
    return convertBackslashPathToSlashPath(imagePath);
};
const getImageBase64Url = (imagePath) => `data:${lookup(imagePath)};base64,${fs.readFileSync(imagePath, {
    encoding: 'base64',
})}`;
const getHtmlShell = (imagePath, options, isUrl) => {
    const imageUrl = isUrl ? imagePath : getImageBase64Url(imagePath);
    return `${constants.SHELL_HTML_FOR_LOGO(imageUrl, options.padding, options.background)}`;
};
const isPathAccessible = (filePath, mode) => fs.promises
    .access(filePath, mode)
    .then(() => true)
    .catch(() => false);
const makeDirRecursiveSync = (dirPath) => {
    fs.mkdirSync(dirPath, { recursive: true });
    return dirPath;
};
export default {
    convertBackslashPathToSlashPath,
    getRelativeImagePath,
    getHtmlShell,
    isHtmlFile,
    isImageFile,
    getImageBase64Url,
    getShellHtmlFilePath,
    getImageSavePath,
    getFileUrlOfPath,
    isPathAccessible,
    getRelativeFilePath,
    getAppDir,
    getExtension,
    getFilesInDir: fs.promises.readdir,
    readFile: fs.promises.readFile,
    readFileSync: fs.readFileSync,
    writeFile: fs.promises.writeFile,
    writeFileSync: fs.writeFileSync,
    makeDir: fs.promises.mkdir,
    makeDirSync: fs.mkdirSync,
    copyFileSync: fs.copyFileSync,
    existsSync: fs.existsSync,
    makeDirRecursiveSync,
    normalizePath: path.normalize,
    READ_ACCESS: fs.constants.R_OK,
    WRITE_ACCESS: fs.constants.W_OK,
};
//# sourceMappingURL=file.js.map