/*
 * Copyright 2025 Pachli Association
 *
 * This file is a part of Pachli.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * Pachli is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with Pachli; if not,
 * see <http://www.gnu.org/licenses>.
 */

package app.pachli.core.testing.di

import android.os.Build
import androidx.room.Room
import androidx.sqlite.driver.bundled.BundledSQLiteDriver
import androidx.test.platform.app.InstrumentationRegistry
import app.pachli.core.database.AppDatabase
import app.pachli.core.database.Converters
import app.pachli.core.database.di.DatabaseModule
import app.pachli.core.database.di.InvalidationTracker
import app.pachli.core.database.di.TransactionProvider
import com.squareup.moshi.Moshi
import dagger.Module
import dagger.Provides
import dagger.hilt.components.SingletonComponent
import dagger.hilt.testing.TestInstallIn
import javax.inject.Singleton

@TestInstallIn(
    components = [SingletonComponent::class],
    replaces = [DatabaseModule::class],
)
@Module
object FakeDatabaseModule {
    @Provides
    @Singleton
    fun providesDatabase(moshi: Moshi): AppDatabase {
        val context = InstrumentationRegistry.getInstrumentation().targetContext
        val roomBuilder = Room.inMemoryDatabaseBuilder(context, AppDatabase::class.java)
            .addTypeConverter(Converters(moshi))
            .allowMainThreadQueries()

        // Tests run on the device should use the BundledSQLiteDriver, like the app.
        // Tests run on Robolectic use the default driver provided by Robolectric.
        if (Build.FINGERPRINT != "robolectric") roomBuilder.setDriver(BundledSQLiteDriver())

        return roomBuilder.build()
    }

    @Provides
    @Singleton
    fun provideTransactionProvider(appDatabase: AppDatabase) = TransactionProvider(appDatabase)

    @Provides
    @Singleton
    fun provideInvalidationTracker(appDatabase: AppDatabase) = InvalidationTracker(appDatabase)

    @Provides
    fun provideAccountDao(appDatabase: AppDatabase) = appDatabase.accountDao()

    @Provides
    fun provideInstanceDao(appDatabase: AppDatabase) = appDatabase.instanceDao()

    @Provides
    fun provideConversationsDao(appDatabase: AppDatabase) = appDatabase.conversationDao()

    @Provides
    fun provideTimelineDao(appDatabase: AppDatabase) = appDatabase.timelineDao()

    @Provides
    fun provideDraftDao(appDatabase: AppDatabase) = appDatabase.draftDao()

    @Provides
    fun provideRemoteKeyDao(appDatabase: AppDatabase) = appDatabase.remoteKeyDao()

    @Provides
    fun providesTranslatedStatusDao(appDatabase: AppDatabase) = appDatabase.translatedStatusDao()

    @Provides
    fun providesLogEntryDao(appDatabase: AppDatabase) = appDatabase.logEntryDao()

    @Provides
    fun providesContentFiltersDao(appDatabase: AppDatabase) = appDatabase.contentFiltersDao()

    @Provides
    fun providesListsDao(appDatabase: AppDatabase) = appDatabase.listsDao()

    @Provides
    fun providesAnnouncementsDao(appDatabase: AppDatabase) = appDatabase.announcementsDao()

    @Provides
    fun providesFollowingAccountDao(appDatabase: AppDatabase) = appDatabase.followingAccountDao()

    @Provides
    fun providesNotificationDao(appDatabase: AppDatabase) = appDatabase.notificationDao()

    @Provides
    fun providesStatusDao(appDatabase: AppDatabase) = appDatabase.statusDao()

    @Provides
    fun providesDebugDao(appDatabase: AppDatabase) = appDatabase.debugDao()
}
