/*
 * Copyright 2024 Pachli Association
 *
 * This file is a part of Pachli.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * Pachli is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with Pachli; if not,
 * see <http://www.gnu.org/licenses>.
 */

package app.pachli.mkserverversions

import app.pachli.mkserverversions.fediverseobserver.ServerVersionsQuery
import ch.qos.logback.classic.Level
import ch.qos.logback.classic.Logger
import com.apollographql.apollo3.ApolloClient
import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.core.Context
import com.github.ajalt.clikt.core.UsageError
import com.github.ajalt.clikt.core.main
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.squareup.moshi.Moshi
import com.squareup.moshi.adapter
import io.github.oshai.kotlinlogging.DelegatingKLogger
import io.github.oshai.kotlinlogging.KotlinLogging
import java.nio.file.Path
import java.nio.file.Paths
import java.util.TreeMap
import java.util.TreeSet
import kotlin.io.path.Path
import kotlin.io.path.div
import kotlin.io.path.exists
import kotlinx.coroutines.runBlocking

private val log = KotlinLogging.logger {}

const val DEST_DIR = "core/network/src/test/resources"

class App : CliktCommand() {
    private val verbose by option("-n", "--verbose", help = "show additional information").flag()

    override fun help(context: Context) = "Update server-versions.json5"

    /**
     * Returns the full path to the Pachli `.../core/network/src/test/resources` directory,
     * starting from the given [start] directory, walking up the tree if it can't be found
     * there.
     *
     * @return the path, or null if it's not a subtree of [start] or any of its parents.
     */
    private fun findResourcePath(start: Path): Path? {
        val suffix = Path(DEST_DIR)

        var prefix = start
        var resourcePath: Path
        do {
            resourcePath = prefix / suffix
            if (resourcePath.exists()) return resourcePath
            prefix = prefix.parent
        } while (prefix != prefix.root)

        return null
    }

    @OptIn(ExperimentalStdlibApi::class)
    override fun run() = runBlocking {
        System.setProperty("file.encoding", "UTF8")
        ((log as? DelegatingKLogger<*>)?.underlyingLogger as Logger).level = if (verbose) Level.INFO else Level.WARN

        val cwd = Paths.get("").toAbsolutePath()
        log.info { "working directory: $cwd" }

        val resourcePath = findResourcePath(cwd) ?: throw UsageError("could not find $DEST_DIR in tree")

        val apolloClient = ApolloClient.Builder()
            .serverUrl("https://api.fediverse.observer/")
            .build()

        val response = apolloClient.query(ServerVersionsQuery()).execute()

        if (response.hasErrors()) {
            response.errors?.forEach {
                log.error { it }
            }
            return@runBlocking
        }

        val serverVersionsPath = resourcePath / "server-versions.json5"
        val w = serverVersionsPath.toFile().printWriter()
        w.println("// GENERATED BY \"runtools mkserverversions\", DO NOT HAND EDIT")

        // Use TreeMap and TreeSet for a consistent sort order in the output to
        // minimise git diffs when the output is regenerated
        val m = TreeMap<String, MutableSet<String>>()

        response.data?.nodes.orEmpty().filterNotNull().forEach {
            if (it.fullversion != null && it.softwarename != null) {
                val versions = m.getOrPut(it.softwarename) { TreeSet() }
                versions.add(it.fullversion)
            }
        }

        val moshi = Moshi.Builder().build()
        w.print(moshi.adapter<Map<String, Set<String>>>().indent("  ").toJson(m))
        w.close()
    }
}

fun main(args: Array<String>) = App().main(args)
