package app.pachli.components.account

import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import app.pachli.core.data.repository.AccountManager
import app.pachli.core.eventhub.DomainMuteEvent
import app.pachli.core.eventhub.EventHub
import app.pachli.core.eventhub.ProfileEditedEvent
import app.pachli.core.model.Account
import app.pachli.core.model.Relationship
import app.pachli.core.network.model.asModel
import app.pachli.core.network.retrofit.MastodonApi
import app.pachli.core.ui.getDomain
import app.pachli.usecase.TimelineCases
import app.pachli.util.Error
import app.pachli.util.Loading
import app.pachli.util.Resource
import app.pachli.util.Success
import com.github.michaelbull.result.map
import com.github.michaelbull.result.onFailure
import com.github.michaelbull.result.onSuccess
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import timber.log.Timber

@HiltViewModel(assistedFactory = AccountViewModel.Factory::class)
class AccountViewModel @AssistedInject constructor(
    @Assisted private val pachliAccountId: Long,
    private val mastodonApi: MastodonApi,
    private val eventHub: EventHub,
    private val accountManager: AccountManager,
    private val timelineCases: TimelineCases,
) : ViewModel() {

    val accountData = MutableLiveData<Resource<Account>>()
    val relationshipData = MutableLiveData<Resource<Relationship>>()

    val noteSaved = MutableLiveData<Boolean>()

    val isRefreshing = MutableLiveData<Boolean>().apply { value = false }
    private var isDataLoading = false

    lateinit var accountId: String
    var isSelf = false

    /** The domain of the viewed account **/
    var domain = ""

    /** True if the viewed account has the same domain as the active account */
    var isFromOwnDomain = false

    private var noteUpdateJob: Job? = null

    private val activeAccount = accountManager.activeAccount!!

    init {
        viewModelScope.launch {
            eventHub.events.collect { event ->
                if (event is ProfileEditedEvent && event.newProfileData.id == accountData.value?.data?.id) {
                    accountData.postValue(Success(event.newProfileData))
                }
            }
        }
    }

    private fun obtainAccount(reload: Boolean = false) {
        if (accountData.value == null || reload) {
            isDataLoading = true
            accountData.postValue(Loading())

            viewModelScope.launch {
                mastodonApi.account(accountId)
                    .onSuccess { result ->
                        val account = result.body.asModel()
                        domain = getDomain(account.url)
                        accountData.postValue(Success(account))
                        isDataLoading = false
                        isRefreshing.postValue(false)

                        isFromOwnDomain = domain == activeAccount.domain
                    }
                    .onFailure {
                        Timber.w("failed obtaining account: %s", it)
                        accountData.postValue(Error(cause = it.throwable))
                        isDataLoading = false
                        isRefreshing.postValue(false)
                    }
            }
        }
    }

    private fun obtainRelationship(reload: Boolean = false) {
        if (relationshipData.value == null || reload) {
            relationshipData.postValue(Loading())

            viewModelScope.launch {
                mastodonApi.relationships(listOf(accountId))
                    .onSuccess {
                        val relationships = it.body.asModel()
                        relationshipData.postValue(if (relationships.isNotEmpty()) Success(relationships[0]) else Error())
                    }
                    .onFailure {
                        Timber.w("failed obtaining relationships: %s", it)
                        relationshipData.postValue(Error(cause = it.throwable))
                    }
            }
        }
    }

    fun changeFollowState() {
        val relationship = relationshipData.value?.data
        if (relationship?.following == true || relationship?.requested == true) {
            changeRelationship(RelationShipAction.UNFOLLOW)
        } else {
            changeRelationship(RelationShipAction.FOLLOW)
        }
    }

    fun changeBlockState() {
        if (relationshipData.value?.data?.blocking == true) {
            changeRelationship(RelationShipAction.UNBLOCK)
        } else {
            changeRelationship(RelationShipAction.BLOCK)
        }
    }

    fun muteAccount(notifications: Boolean, duration: Int?) {
        changeRelationship(RelationShipAction.MUTE, notifications, duration)
    }

    fun unmuteAccount() {
        changeRelationship(RelationShipAction.UNMUTE)
    }

    fun changeSubscribingState() {
        val relationship = relationshipData.value?.data
        if (relationship?.notifying == true ||
            // Mastodon 3.3.0rc1
            relationship?.subscribing == true // Pleroma
        ) {
            changeRelationship(RelationShipAction.UNSUBSCRIBE)
        } else {
            changeRelationship(RelationShipAction.SUBSCRIBE)
        }
    }

    fun blockDomain(instance: String) {
        viewModelScope.launch {
            mastodonApi.blockDomain(instance)
                .onSuccess {
                    eventHub.dispatch(DomainMuteEvent(pachliAccountId, instance))
                    val relation = relationshipData.value?.data
                    if (relation != null) {
                        relationshipData.postValue(Success(relation.copy(blockingDomain = true)))
                    }
                }
                .onFailure { e ->
                    Timber.e("Error muting %s: %s", instance, e)
                }
        }
    }

    fun unblockDomain(instance: String) {
        viewModelScope.launch {
            mastodonApi.unblockDomain(instance)
                .onSuccess {
                    val relation = relationshipData.value?.data
                    if (relation != null) {
                        relationshipData.postValue(Success(relation.copy(blockingDomain = false)))
                    }
                }
                .onFailure { e ->
                    Timber.e("Error unmuting %s: %s", instance, e)
                }
        }
    }

    fun changeShowReblogsState() {
        if (relationshipData.value?.data?.showingReblogs == true) {
            changeRelationship(RelationShipAction.HIDE_REBLOGS)
        } else {
            changeRelationship(RelationShipAction.SHOW_REBLOGS)
        }
    }

    /**
     * @param parameter showReblogs if RelationShipAction.FOLLOW, notifications if MUTE
     */
    private fun changeRelationship(
        relationshipAction: RelationShipAction,
        parameter: Boolean? = null,
        duration: Int? = null,
    ) = viewModelScope.launch {
        val relation = relationshipData.value?.data
        val account = accountData.value?.data
        val isMastodon = relationshipData.value?.data?.notifying != null

        if (relation != null && account != null) {
            // optimistically post new state for faster response

            val newRelation = when (relationshipAction) {
                RelationShipAction.FOLLOW -> {
                    if (account.locked) {
                        relation.copy(requested = true)
                    } else {
                        relation.copy(following = true)
                    }
                }
                RelationShipAction.UNFOLLOW -> relation.copy(following = false)
                RelationShipAction.BLOCK -> relation.copy(blocking = true)
                RelationShipAction.UNBLOCK -> relation.copy(blocking = false)
                RelationShipAction.MUTE -> relation.copy(muting = true)
                RelationShipAction.UNMUTE -> relation.copy(muting = false)
                RelationShipAction.SUBSCRIBE -> {
                    if (isMastodon) {
                        relation.copy(notifying = true)
                    } else {
                        relation.copy(subscribing = true)
                    }
                }
                RelationShipAction.UNSUBSCRIBE -> {
                    if (isMastodon) {
                        relation.copy(notifying = false)
                    } else {
                        relation.copy(subscribing = false)
                    }
                }

                RelationShipAction.SHOW_REBLOGS -> relation.copy(showingReblogs = true)
                RelationShipAction.HIDE_REBLOGS -> relation.copy(showingReblogs = false)
            }
            relationshipData.postValue(Loading(newRelation))
        }

        val response = when (relationshipAction) {
            RelationShipAction.FOLLOW -> timelineCases.followAccount(pachliAccountId, accountId, showReblogs = true)
            RelationShipAction.UNFOLLOW -> timelineCases.unfollowAccount(pachliAccountId, accountId)
            RelationShipAction.BLOCK -> timelineCases.blockAccount(pachliAccountId, accountId)
            RelationShipAction.UNBLOCK -> timelineCases.unblockAccount(pachliAccountId, accountId)
            RelationShipAction.MUTE -> timelineCases.muteAccount(
                pachliAccountId,
                accountId,
                notifications = parameter ?: true,
                duration,
            )

            RelationShipAction.UNMUTE -> timelineCases.unmuteAccount(pachliAccountId, accountId)
            RelationShipAction.SUBSCRIBE -> {
                if (isMastodon) {
                    timelineCases.followAccount(pachliAccountId, accountId, notify = true)
                } else {
                    timelineCases.subscribeAccount(pachliAccountId, accountId)
                }
            }
            RelationShipAction.UNSUBSCRIBE -> {
                if (isMastodon) {
                    timelineCases.followAccount(pachliAccountId, accountId, notify = false)
                } else {
                    timelineCases.unsubscribeAccount(pachliAccountId, accountId)
                }
            }

            RelationShipAction.SHOW_REBLOGS -> timelineCases.followAccount(pachliAccountId, accountId, showReblogs = true)
            RelationShipAction.HIDE_REBLOGS -> timelineCases.followAccount(pachliAccountId, accountId, showReblogs = false)
        }

        response
            .map { it.body.asModel() }
            .onSuccess { relationshipData.postValue(Success(it)) }
            .onFailure { e ->
                Timber.w("failed loading relationship: %s", e)
                relationshipData.postValue(Error(relation, cause = e.throwable))
            }
    }

    fun noteChanged(newNote: String) {
        noteSaved.postValue(false)
        noteUpdateJob?.cancel()
        noteUpdateJob = viewModelScope.launch {
            delay(1500)
            mastodonApi.updateAccountNote(accountId, newNote)
                .onSuccess {
                    noteSaved.postValue(true)
                    delay(4000)
                    noteSaved.postValue(false)
                }
                .onFailure { e ->
                    Timber.w("Error updating note: %s", e)
                }
        }
    }

    fun refresh() {
        reload(true)
    }

    private fun reload(isReload: Boolean = false) {
        if (isDataLoading) {
            return
        }
        accountId.let {
            obtainAccount(isReload)
            if (!isSelf) {
                obtainRelationship(isReload)
            }
        }
    }

    fun setAccountInfo(accountId: String) {
        this.accountId = accountId
        this.isSelf = activeAccount.accountId == accountId
        reload(false)
    }

    enum class RelationShipAction {
        FOLLOW,
        UNFOLLOW,
        BLOCK,
        UNBLOCK,
        MUTE,
        UNMUTE,
        SUBSCRIBE,
        UNSUBSCRIBE,
        SHOW_REBLOGS,
        HIDE_REBLOGS,
    }

    @AssistedFactory
    interface Factory {
        /** Creates [AccountViewModel] with [pachliAccountId] as the active account. */
        fun create(pachliAccountId: Long): AccountViewModel
    }
}
