#!/usr/bin/env node

const { execSync } = require("child_process");
const fs = require("fs");
const https = require("https");

const OUTPUT_FILE = "CHANGELOG.md";
const TAGS_TO_INCLUDE = 10; // Number of recent tags to include
const REPO_URL = "https://github.com/CodeWorksCreativeHub/mLauncher";

const HEADER = `# Changelog

All notable changes to this project will be documented in this file. See [conventional commits](https://www.conventionalcommits.org/) for commit guidelines.\n\n`;

const FOOTER = `---
> Generated by CodeWorks Creative Hub`;

// Verbose logging
const VERBOSE = process.argv.includes("--verbose");
function log(...args) {
	if (!VERBOSE) return;
	const cleanArgs = args.map((arg) => (typeof arg === "string" ? arg.replace("### ", "").trim() : arg));
	console.log(...cleanArgs);
}

// Commit parsing rules
const commitParsers = [
	// Skip some "noise" commits
	{ message: /^chore\(release\): prepare for/i, skip: true },
	{ message: /^chore\(deps.*\)/i, skip: true },
	{ message: /^chore\(change.*\)/i, skip: true },
	{ message: /^chore\(pr\)/i, skip: true },
	{ message: /^chore\(pull\)/i, skip: true },
	{ message: /^fixes/i, skip: true },
	{ message: /^build/i, skip: true },

	// Enhancements (new features, improvements, UX, performance)
	{ message: /^feat|^perf|^style|^ui|^ux/i, group: "### :sparkles: Enhancements:" },

	// Bug fixes & hotfixes
	{ message: /^fix|^bug|^hotfix|^emergency/i, group: "### :bug: Bug Fixes:" },

	// Code quality (refactors, cleanup without changing behavior)
	{ message: /^refactor/i, group: "### :wrench: Code Quality:" },

	// Documentation
	{ message: /^doc/i, group: "### :books: Documentation:" },

	// Localization & internationalization
	{ message: /^(lang|i18n)/i, group: "### :globe_with_meridians: Localization:" },

	// Security
	{ message: /^security/i, group: "### :lock: Security:" },

	// Feature removal / drops
	{ message: /^drop|^remove|^deprecated/i, group: "### :x: Feature Removals:" },

	// Reverts
	{ message: /^revert/i, group: "### :rewind: Reverts:" },

	// Build-related
	{ message: /^build/i, group: "### :building_construction: Build:" },

	// Dependencies-related
	{ message: /^dependency|^deps/i, group: "### :package: Dependencies:" },

	// Meta: configuration, CI/CD, versioning, releases
	{ message: /^config|^configuration|^ci|^pipeline|^release|^version|^versioning/i, group: "### :gear: Meta:" },

	// Tests
	{ message: /^test/i, group: "### :test_tube: Tests:" },

	// Infrastructure & Ops
	{ message: /^infra|^infrastructure|^ops/i, group: "### :office: Infrastructure & Ops:" },

	// Chore & cleanup
	{ message: /^chore|^housekeeping|^cleanup|^clean\(up\)/i, group: "### :broom: Maintenance & Cleanup:" },
];

const GROUP_ORDER = commitParsers.filter((p) => !p.skip).map((p) => p.group);

// Helper functions
function run(cmd) {
	log("Running:", cmd);
	return execSync(cmd, { encoding: "utf8" }).trim();
}

function formatDate(dateString) {
	const date = new Date(dateString);
	const day = String(date.getDate()).padStart(2, "0");
	const month = date.toLocaleString("en-US", { month: "long" });
	const year = date.getFullYear();
	return `- (${day}, ${month} ${year})`;
}

function classifyCommit(msg) {
	for (const parser of commitParsers) {
		if (parser.message.test(msg)) {
			if (parser.skip) return null;
			return { group: parser.group, message: msg };
		}
	}
	return null;
}

function cleanMessage(message) {
	return message.replace(/^(feat|fix|fixed|bug|lang|i18n|doc|docs|perf|refactor|style|ui|ux|security|revert|release|dependency|deps|build|ci|pipeline|chore|housekeeping|version|versioning|config|configuration|cleanup|clean\(up\)|drop|remove|deprecated|hotfix|emergency|test|infra|infrastructure|ops|asset|content|exp|experiment|prototype)\s*(\(.+?\))?:\s*/i, "");
}

function linkPR(message) {
	return message.replace(/\(#(\d+)\)/g, (_, num) => `([#${num}](${REPO_URL}/pull/${num}))`);
}

// Get GitHub release title for a tag
function getReleaseTitle(tag) {
	const options = {
		hostname: "api.github.com",
		path: `/repos/CodeWorksCreativeHub/mLauncher/releases/tags/${tag}`,
		method: "GET",
		headers: { "User-Agent": "Node.js" },
	};
	return new Promise((resolve) => {
		const req = https.request(options, (res) => {
			let data = "";
			res.on("data", (chunk) => (data += chunk));
			res.on("end", () => {
				try {
					const json = JSON.parse(data);
					resolve(json.name || tag);
				} catch {
					resolve(tag);
				}
			});
		});
		req.on("error", () => resolve(tag));
		req.end();
	});
}

// Main async function
async function generateChangelog() {
	const allTags = run("git tag --sort=-creatordate").split("\n");
	const tags = allTags.slice(0, TAGS_TO_INCLUDE);
	log("Tags to include:", tags);

	let changelog = HEADER;

	// Coming Soon / Unreleased
	const latestTag = tags[0] || "unreleased";
	const rawUnreleased = run(`git log ${latestTag}..HEAD --pretty=format:"%h|%s"`).split("\n");
	const unreleasedCommits = rawUnreleased
		.map((line) => {
			const [hash, ...msgParts] = line.split("|");
			const message = msgParts.join("|");
			const classified = classifyCommit(message);
			if (!classified) return null;
			return { ...classified, hash };
		})
		.filter(Boolean);

	if (unreleasedCommits.length > 0) {
		log("Unreleased commits found:", unreleasedCommits.length);
		changelog += `## [${latestTag} → Unreleased](https://github.com/CodeWorksCreativeHub/mLauncher/tree/main) - In Development\n\n`;
		const groups = {};
		for (const c of unreleasedCommits) {
			groups[c.group] = groups[c.group] || [];
			groups[c.group].push(`* ${linkPR(cleanMessage(c.message))} ([${c.hash}](${REPO_URL}/commit/${c.hash}))`);
		}
		for (const group of GROUP_ORDER) {
			if (groups[group]) {
				log(`Unreleased group: ${group}, commits: ${groups[group].length}`);
				changelog += `${group}\n\n${groups[group].join("\n")}\n\n`;
			}
		}
	}

	// Generate changelog for each tag
	for (let i = 0; i < tags.length; i++) {
		const currentTag = tags[i];
		const releaseTitle = await getReleaseTitle(currentTag); // GitHub release title
		log(`Processing tag: ${currentTag} (${releaseTitle})`);

		let range;
		if (i === tags.length - 1) {
			const oldestTagIndex = allTags.indexOf(currentTag);
			const parentTag = allTags[oldestTagIndex - 1];
			range = parentTag ? `${parentTag}..${currentTag}` : currentTag;
		} else {
			const previousTagInSlice = tags[i + 1];
			range = `${previousTagInSlice}..${currentTag}`;
		}

		const rawCommits = run(`git log ${range} --pretty=format:"%h|%s"`).split("\n");
		const commits = rawCommits
			.map((line) => {
				const [hash, ...msgParts] = line.split("|");
				const message = msgParts.join("|");
				const classified = classifyCommit(message);
				if (!classified) return null;
				return { ...classified, hash };
			})
			.filter(Boolean);

		if (commits.length === 0) continue;

		const tagDateRaw = run(`git log -1 --format=%ad --date=short ${currentTag}`);
		const tagDateFormatted = formatDate(tagDateRaw);

		changelog += `## [${releaseTitle}](${REPO_URL}/tree/${currentTag}) ${tagDateFormatted}\n\n`;

		const groups = {};
		for (const c of commits) {
			groups[c.group] = groups[c.group] || [];
			groups[c.group].push(`* ${linkPR(cleanMessage(c.message))} ([${c.hash}](${REPO_URL}/commit/${c.hash}))`);
		}

		for (const group of GROUP_ORDER) {
			if (groups[group]) {
				changelog += `${group}\n\n${groups[group].join("\n")}\n\n`;
			}
		}
	}

	changelog += FOOTER;
	fs.writeFileSync(OUTPUT_FILE, changelog, "utf8");
	console.log(`✅ Generated ${OUTPUT_FILE}`);
}

// Run
generateChangelog();
