package app.hypostats.data.local

import android.content.Context
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.longPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import app.hypostats.ui.model.AppTheme
import app.hypostats.ui.model.CarbIcon
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import java.time.Instant

private val Context.dataStore by preferencesDataStore(name = "app")

class PreferencesAppDataStore(
    private val context: Context,
) : AppDataStore {
    private val trackingStartDateKey = longPreferencesKey("tracking_start_date")
    private val appThemeKey = stringPreferencesKey("app_theme")
    private val carbIncrementKey = intPreferencesKey("carb_increment")
    private val carbIconKey = stringPreferencesKey("carb_icon")

    override val trackingStartDate: Flow<Instant?> =
        context.dataStore.data
            .map { preferences ->
                preferences[trackingStartDateKey]?.let { Instant.ofEpochMilli(it) }
            }

    override suspend fun setTrackingStartDate(startDate: Instant) {
        context.dataStore.edit { preferences ->
            preferences[trackingStartDateKey] = startDate.toEpochMilli()
        }
    }

    override val appTheme: Flow<AppTheme?> =
        context.dataStore.data
            .map { preferences ->
                preferences[appThemeKey]?.let { themeName ->
                    AppTheme.valueOf(themeName)
                }
            }

    override suspend fun setAppTheme(theme: AppTheme) {
        context.dataStore.edit { preferences ->
            preferences[appThemeKey] = theme.name
        }
    }

    override val carbIncrement: Flow<Int?> =
        context.dataStore.data
            .map { preferences ->
                preferences[carbIncrementKey]
            }

    override suspend fun setCarbIncrement(increment: Int) {
        context.dataStore.edit { preferences ->
            preferences[carbIncrementKey] = increment
        }
    }

    override val carbIcon: Flow<CarbIcon?> =
        context.dataStore.data
            .map { preferences ->
                preferences[carbIconKey]?.let { iconName ->
                    CarbIcon.valueOf(iconName)
                }
            }

    override suspend fun setCarbIcon(icon: CarbIcon) {
        context.dataStore.edit { preferences ->
            preferences[carbIconKey] = icon.name
        }
    }
}
