"use strict";

import { Animated, Keyboard, KeyboardAvoidingView, Modal, StyleSheet, Text, TouchableWithoutFeedback, View } from 'react-native';
import { Button, IconButton, overlay, useTheme } from 'react-native-paper';
import TimePicker from './TimePicker';
import { clockTypes, getTimeInputTypeIcon, inputTypes, reverseInputTypes } from './timeUtils';
import { memo, useCallback, useEffect, useState } from 'react';
import { sharedStyles } from '../shared/styles';
import { supportedOrientations } from '../shared/utils';
import { jsx as _jsx, jsxs as _jsxs, Fragment as _Fragment } from "react/jsx-runtime";
export function TimePickerModal({
  visible,
  onDismiss,
  onConfirm,
  hours,
  minutes,
  label = 'Select time',
  uppercase: _uppercase,
  cancelLabel = 'Cancel',
  confirmLabel = 'Ok',
  animationType = 'none',
  locale,
  keyboardIcon = 'keyboard-outline',
  clockIcon = 'clock-outline',
  use24HourClock,
  inputFontSize,
  defaultInputType
}) {
  const theme = useTheme();
  const [inputType, setInputType] = useState(defaultInputType || inputTypes.picker);
  const [focused, setFocused] = useState(clockTypes.hours);
  const [localHours, setLocalHours] = useState(getHours(hours));
  const [localMinutes, setLocalMinutes] = useState(getMinutes(minutes));
  useEffect(() => {
    setLocalHours(getHours(hours));
  }, [setLocalHours, hours]);
  useEffect(() => {
    setLocalMinutes(getMinutes(minutes));
  }, [setLocalMinutes, minutes]);
  const onFocusInput = useCallback(type => setFocused(type), []);
  const onChange = useCallback(params => {
    if (params.focused) {
      setFocused(params.focused);
    }
    setLocalHours(params.hours);
    setLocalMinutes(params.minutes);
  }, [setFocused, setLocalHours, setLocalMinutes]);
  const defaultUppercase = !theme.isV3;
  const uppercase = _uppercase ?? defaultUppercase;
  let textFont;
  let labelText = label;
  if (theme.isV3) {
    textFont = theme.fonts.labelMedium;
  } else {
    textFont = theme?.fonts.medium;
  }
  if (inputType === inputTypes.keyboard && !label) {
    labelText = 'Enter time';
  }
  let color;
  if (theme.isV3) {
    color = theme.dark ? theme.colors.elevation.level3 : theme.colors.surface;
  } else {
    color = theme.dark ? overlay(10, theme.colors.surface) : theme.colors.surface;
  }
  return /*#__PURE__*/_jsx(Modal, {
    animationType: animationType,
    transparent: true,
    visible: visible,
    onRequestClose: onDismiss,
    presentationStyle: "overFullScreen",
    supportedOrientations: supportedOrientations,
    statusBarTranslucent: true,
    children: /*#__PURE__*/_jsxs(_Fragment, {
      children: [/*#__PURE__*/_jsx(TouchableWithoutFeedback, {
        onPress: onDismiss,
        children: /*#__PURE__*/_jsx(View, {
          style: [StyleSheet.absoluteFill, sharedStyles.root, {
            backgroundColor: theme.colors?.backdrop
          }]
        })
      }), /*#__PURE__*/_jsx(View, {
        style: [StyleSheet.absoluteFill, styles.center],
        pointerEvents: "box-none",
        children: /*#__PURE__*/_jsx(KeyboardAvoidingView, {
          style: styles.center,
          behavior: "padding",
          children: /*#__PURE__*/_jsxs(Animated.View, {
            style: [styles.modalContent,
            // eslint-disable-next-line react-native/no-inline-styles
            {
              backgroundColor: color,
              borderRadius: theme.isV3 ? 28 : undefined
            }],
            children: [/*#__PURE__*/_jsx(View, {
              style: styles.labelContainer,
              children: /*#__PURE__*/_jsx(Text, {
                maxFontSizeMultiplier: 1.5,
                style: [styles.label, {
                  ...textFont,
                  color: theme?.isV3 ? theme.colors.onSurfaceVariant : theme.colors.text
                }],
                children: uppercase ? labelText.toUpperCase() : labelText
              })
            }), /*#__PURE__*/_jsx(View, {
              style: styles.timePickerContainer,
              children: /*#__PURE__*/_jsx(TimePicker, {
                locale: locale,
                inputType: inputType,
                use24HourClock: use24HourClock,
                inputFontSize: inputFontSize,
                focused: focused,
                hours: localHours,
                minutes: localMinutes,
                onChange: onChange,
                onFocusInput: onFocusInput
              })
            }), /*#__PURE__*/_jsxs(View, {
              style: styles.bottom,
              children: [/*#__PURE__*/_jsx(IconButton, {
                iconColor: theme?.isV3 ? theme.colors.onBackground : theme.colors.text,
                icon: getTimeInputTypeIcon(inputType, {
                  keyboard: keyboardIcon,
                  picker: clockIcon
                }),
                onPress: () => {
                  Keyboard.dismiss();
                  setInputType(reverseInputTypes[inputType]);
                },
                size: 24,
                style: styles.inputTypeToggle,
                accessibilityLabel: "toggle keyboard"
              }), /*#__PURE__*/_jsx(View, {
                style: sharedStyles.root
              }), /*#__PURE__*/_jsx(Button, {
                onPress: onDismiss,
                uppercase: uppercase,
                children: cancelLabel
              }), /*#__PURE__*/_jsx(Button, {
                onPress: () => onConfirm({
                  hours: localHours,
                  minutes: localMinutes
                }),
                uppercase: uppercase,
                children: confirmLabel
              })]
            })]
          })
        })
      })]
    })
  });
}
function getMinutes(minutes) {
  return minutes === undefined || minutes === null ? new Date().getMinutes() : minutes;
}
function getHours(hours) {
  return hours === undefined || hours === null ? new Date().getHours() : hours;
}
const styles = StyleSheet.create({
  bottom: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: 8
  },
  center: {
    justifyContent: 'center',
    alignItems: 'center',
    flex: 1
  },
  inputTypeToggle: {
    margin: 4
  },
  labelContainer: {
    justifyContent: 'flex-end',
    paddingLeft: 24,
    paddingRight: 24,
    paddingTop: 16
  },
  label: {
    letterSpacing: 1,
    fontSize: 13
  },
  modalContent: {
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 5
    },
    shadowOpacity: 0.34,
    shadowRadius: 6.27,
    elevation: 3,
    minWidth: 328,
    paddingVertical: 8
  },
  timePickerContainer: {
    paddingLeft: 24,
    paddingTop: 20,
    paddingBottom: 16,
    paddingRight: 24
  }
});
export default /*#__PURE__*/memo(TimePickerModal);
//# sourceMappingURL=TimePickerModal.js.map