"use strict";

import { StyleSheet, useWindowDimensions, View } from 'react-native';
import { Text, useTheme } from 'react-native-paper';
import { clockTypes, toHourInputFormat, toHourOutputFormat } from './timeUtils';
import TimeInput from './TimeInput';
import AmPmSwitcher from './AmPmSwitcher';
import { useLatest } from '../shared/utils';
import Color from 'color';
import { getTranslation } from '../translations/utils';
import { memo, useCallback, useRef } from 'react';
import { sharedStyles } from '../shared/styles';
import { jsx as _jsx, jsxs as _jsxs, Fragment as _Fragment } from "react/jsx-runtime";
function TimeInputs({
  hours,
  minutes,
  onFocusInput,
  focused,
  inputType,
  onChange,
  is24Hour,
  inputFontSize,
  locale
}) {
  const theme = useTheme();
  const startInput = useRef(null);
  const endInput = useRef(null);
  const dimensions = useWindowDimensions();
  const isLandscape = dimensions.width > dimensions.height;
  const minutesRef = useLatest(minutes);
  const onSubmitStartInput = useCallback(() => {
    if (endInput.current) {
      endInput.current.focus();
    }
  }, [endInput]);
  const onSubmitEndInput = useCallback(() => {
    // TODO: close modal and persist time
  }, []);
  const onChangeHours = useCallback(newHours => {
    onChange({
      hours: newHours,
      minutes: minutesRef.current,
      focused: clockTypes.hours
    });
  }, [onChange, minutesRef]);
  return /*#__PURE__*/_jsxs(View, {
    style: [styles.inputContainer, isLandscape && sharedStyles.root],
    children: [/*#__PURE__*/_jsxs(View, {
      style: styles.column,
      children: [/*#__PURE__*/_jsx(TimeInput, {
        ref: startInput,
        inputFontSize: inputFontSize,
        placeholder: '00',
        value: toHourInputFormat(hours, is24Hour),
        clockType: clockTypes.hours,
        pressed: focused === clockTypes.hours,
        onPress: onFocusInput,
        inputType: inputType,
        maxFontSizeMultiplier: 1.2,
        selectionColor: theme.dark ? Color(theme.colors.primary).darken(0.2).hex() : theme.colors.primary,
        returnKeyType: 'next',
        onSubmitEditing: onSubmitStartInput,
        blurOnSubmit: false,
        onChanged: newHoursFromInput => {
          let newHours = toHourOutputFormat(newHoursFromInput, hours, is24Hour);
          if (newHoursFromInput > 24) {
            newHours = 24;
          }
          onChange({
            hours: newHours,
            minutes
          });
        }
      }), inputType === 'keyboard' ? /*#__PURE__*/_jsx(Text, {
        maxFontSizeMultiplier: 1.5,
        variant: "bodySmall",
        children: getTranslation(locale, 'hour', 'Hour')
      }) : null]
    }), /*#__PURE__*/_jsxs(View, {
      style: [styles.hoursAndMinutesSeparator,
      // eslint-disable-next-line react-native/no-inline-styles
      {
        marginBottom: inputType === 'keyboard' ? 16 : 0
      }],
      children: [/*#__PURE__*/_jsx(View, {
        style: sharedStyles.root
      }), /*#__PURE__*/_jsx(View, {
        style: [styles.dot, {
          backgroundColor: theme?.isV3 ? theme.colors.onSurface : theme.colors.text
        }]
      }), /*#__PURE__*/_jsx(View, {
        style: styles.betweenDot
      }), /*#__PURE__*/_jsx(View, {
        style: [styles.dot, {
          backgroundColor: theme?.isV3 ? theme.colors.onSurface : theme.colors.text
        }]
      }), /*#__PURE__*/_jsx(View, {
        style: sharedStyles.root
      })]
    }), /*#__PURE__*/_jsxs(View, {
      style: styles.column,
      children: [/*#__PURE__*/_jsx(TimeInput, {
        ref: endInput,
        inputFontSize: inputFontSize,
        placeholder: '00',
        value: minutes,
        clockType: clockTypes.minutes,
        pressed: focused === clockTypes.minutes,
        onPress: onFocusInput,
        inputType: inputType,
        maxFontSizeMultiplier: 1.2,
        selectionColor: theme.dark ? Color(theme.colors.primary).darken(0.2).hex() : theme.colors.primary,
        onSubmitEditing: onSubmitEndInput,
        onChanged: newMinutesFromInput => {
          let newMinutes = newMinutesFromInput;
          if (newMinutesFromInput > 59) {
            newMinutes = 59;
          }
          onChange({
            hours,
            minutes: newMinutes
          });
        }
      }), inputType === 'keyboard' ? /*#__PURE__*/_jsx(Text, {
        maxFontSizeMultiplier: 1.5,
        variant: "bodySmall",
        children: getTranslation(locale, 'minute', 'Minute')
      }) : null]
    }), !is24Hour && /*#__PURE__*/_jsxs(_Fragment, {
      children: [/*#__PURE__*/_jsx(View, {
        style: styles.spaceBetweenInputsAndSwitcher
      }), /*#__PURE__*/_jsx(AmPmSwitcher, {
        hours: hours,
        onChange: onChangeHours,
        inputType: inputType
      })]
    })]
  });
}
const styles = StyleSheet.create({
  betweenDot: {
    height: 12
  },
  column: {
    flexDirection: 'column'
  },
  dot: {
    width: 7,
    height: 7,
    borderRadius: 7 / 2
  },
  hoursAndMinutesSeparator: {
    fontSize: 65,
    width: 24,
    alignItems: 'center'
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center'
  },
  spaceBetweenInputsAndSwitcher: {
    width: 12
  }
});
export default /*#__PURE__*/memo(TimeInputs);
//# sourceMappingURL=TimeInputs.js.map