package expo.modules.application

import android.content.Context
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.os.Build
import android.os.RemoteException
import android.provider.Settings
import expo.modules.kotlin.Promise
import expo.modules.kotlin.exception.CodedException
import expo.modules.kotlin.exception.Exceptions
import expo.modules.kotlin.modules.Module
import expo.modules.kotlin.modules.ModuleDefinition

class ApplicationPackageNameNotFoundException(cause: PackageManager.NameNotFoundException) :
  CodedException(message = "Unable to get install time of this application. Could not get package info or package name.", cause = cause)

class ApplicationModule : Module() {
  private val context: Context
    get() = appContext.reactContext ?: throw Exceptions.ReactContextLost()

  override fun definition() = ModuleDefinition {
    Name("ExpoApplication")

    Constants {
      return@Constants mapOf(
        "applicationName" to applicationName,
        "applicationId" to packageName,
        "nativeApplicationVersion" to versionName,
        "nativeBuildVersion" to versionCode.toString()
      )
    }

    Property("androidId") {
      Settings.Secure.getString(context.contentResolver, Settings.Secure.ANDROID_ID)
    }

    AsyncFunction<Double>("getInstallationTimeAsync") {
      val packageManager = context.packageManager
      val packageName = context.packageName
      packageManager
        .getPackageInfoCompat(packageName, 0)
        .firstInstallTime
        .toDouble()
    }

    AsyncFunction<Double>("getLastUpdateTimeAsync") {
      val packageManager = context.packageManager
      val packageName = context.packageName
      packageManager
        .getPackageInfoCompat(packageName, 0)
        .lastUpdateTime
        .toDouble()
    }

    AsyncFunction("getInstallReferrerAsync") { promise: Promise ->
promise.resolve("")
    }
  }

  private val applicationName
    get() = context.applicationInfo.loadLabel(context.packageManager).toString()
  private val packageName
    get() = context.packageName
  private val packageManager
    get() = context.packageManager
  private val versionName
    get() = packageManager.getPackageInfoCompat(packageName, 0).versionName
  private val versionCode
    get() = getLongVersionCode(packageManager.getPackageInfoCompat(packageName, 0)).toInt()
}

private fun PackageManager.getPackageInfoCompat(packageName: String, flags: Int = 0): PackageInfo =
  try {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
      getPackageInfo(packageName, PackageManager.PackageInfoFlags.of(flags.toLong()))
    } else {
      @Suppress("DEPRECATION")
      getPackageInfo(packageName, flags)
    }
  } catch (e: PackageManager.NameNotFoundException) {
    throw ApplicationPackageNameNotFoundException(e)
  }

private fun getLongVersionCode(info: PackageInfo): Long {
  return if (Build.VERSION.SDK_INT >= 28) {
    info.longVersionCode
  } else {
    @Suppress("DEPRECATION")
    info.versionCode.toLong()
  }
}
