export type Executor<T> = ConstructorParameters<typeof Promise<T>>[0];
type Resolver<T> = Parameters<Executor<T>>[0];
type Rejecter = Parameters<Executor<unknown>>[1];
/**
 * Barrier is just a Promise which has resolve and reject exposed as instance
 * methods.
 *
 * It has two generic arguments T and TR which correspond to the argument of
 * the .resolve() method, and to the value resolved by the promise (barrier).
 * For a simple barrier TR equals to T, however for barriers created via .then()
 * chain, T corresponds to the argument of the original barrier, and TR to
 * the value resolved by the latest promise in the chain. Consider this:
 *
 * const b = new Barrier<string>();
 * b.resolve('result');
 * const s = await b; // `s` has `string` type, and equals "result".
 *
 * const b = (new Barrier<string>()).then((s) => s.length);
 * b.resolve('result'); // Chained barrier exposes .resolve() method of
 *                      // the first barrier in the chain, which expects
 *                      // `string` arugment (T), but the chained barrier
 *                      // resolves to `number` (TR).
 * const n = await b; // `n` has `number` type, and equals 6.
 *
 * Docs: https://dr.pogodin.studio/docs/react-utils/docs/api/classes/Barrier
 */
export default class Barrier<T = unknown, TR = T> extends Promise<TR> {
    private pResolve;
    private pReject;
    private pState;
    constructor(executor?: Executor<TR>);
    get resolve(): (arg: Parameters<Resolver<T>>[0]) => this;
    get reject(): (arg: Parameters<Rejecter>[0]) => this;
    get resolved(): boolean;
    get rejected(): boolean;
    get settled(): boolean;
    catch<TR1>(onRejected?: ((reason: unknown) => TR1 | PromiseLike<TR1>) | null): Barrier<T, TR1>;
    finally(onFinally?: (() => void) | null): Barrier<TR>;
    then<TR1, TR2>(onFulfilled?: ((value: TR) => TR1 | PromiseLike<TR1>) | null, onRejected?: ((reason: unknown) => TR2 | PromiseLike<TR2>) | null): Barrier<T, TR1 | TR2>;
}
export {};
