"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Emitter = void 0;
/**
 * Simple listeneable data Emitter.
 */
class Emitter {
  pListeners = [];

  /**
   * Returns "true" if any listener is connected; "false" otherwise.
   * @return {boolean}
   */
  get hasListeners() {
    return !!this.pListeners.length;
  }
  get listeners() {
    return this.pListeners;
  }

  /**
   * Adds `listener` if it is not already connected.
   * @param {function} listener
   * @return {function} Unsubscribe function.
   */
  addListener(listener) {
    if (!this.pListeners.includes(listener)) {
      this.pListeners.push(listener);
    }
    return () => {
      this.removeListener(listener);
    };
  }

  /**
   * Calls every connected listener with the given arguments.
   * @param args
   */
  emit(...args) {
    const listeners = this.pListeners.slice();
    for (const listener of listeners) listener(...args);
  }

  /**
   * Removes all connected listeners.
   */
  removeAllListeners() {
    this.pListeners = [];
  }

  /**
   * Removes specified `listener`, if connected.
   * @param listener
   */
  removeListener(listener) {
    const idx = this.pListeners.indexOf(listener);
    if (idx >= 0) this.pListeners.splice(idx, 1);
  }
}
exports.Emitter = Emitter;
//# sourceMappingURL=Emitter.js.map