"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Cached = void 0;
/** Adds timestamp to the promise. */
function addTimestamp(promise, timestamp) {
  const res = promise;
  res.timestamp = timestamp;
  return res;
}

/** Gets entry timestamp. */
function getTimestamp(e) {
  return Array.isArray(e) ? e[1] : e.timestamp;
}
class Cached {
  pData = {};
  pOldestTimestamp = Number.MAX_SAFE_INTEGER;

  /** For test use only. */
  get data() {
    return this.pData;
  }

  /** For test use only. */
  get oldestTimestamp() {
    return this.pOldestTimestamp;
  }
  constructor(maxage, getter) {
    this.maxage = maxage;
    this.getter = getter;
  }

  /** Removes stale items from the cache, and updates .oldestTimestamp. */
  cleanCache() {
    const deadline = Date.now() - this.maxage;
    this.pOldestTimestamp = Number.MAX_SAFE_INTEGER;
    for (const [key, entry] of Object.entries(this.pData)) {
      const timestamp = getTimestamp(entry);
      if (timestamp < deadline) delete this.pData[key];else if (timestamp < this.pOldestTimestamp) {
        this.pOldestTimestamp = timestamp;
      }
    }
  }

  /**
   * Adds entry to the cache.
   * NOTE: It assumes entry's timestamp is the current moment (for the cache
   * cleaning purposes; if it is not, but it is a past timestamp, nothing bad
   * will happen, just some cleaning operation will be skipped).
   */
  setEntry(id, entry) {
    this.pData[id] = entry;
    const timestamp = getTimestamp(entry);
    if (timestamp < this.pOldestTimestamp) this.pOldestTimestamp = timestamp;else if (this.pOldestTimestamp < timestamp - this.maxage) this.cleanCache();
  }

  /** Adds `datum` to the cache, and removes stale items from the cache. */
  set(id, datum) {
    const res = [datum, Date.now()];
    this.setEntry(id, res);
    return res;
  }

  /**
   * Retrieves envelope of the specified datum, either read from the cache,
   * or retrieved using the getter provided at construction time.
   */
  getEntry(id, forceRefresh) {
    const now = Date.now();
    if (!forceRefresh) {
      const cached = this.pData[id];
      if (cached && getTimestamp(cached) >= now - this.maxage) return cached;
    }
    const itemOrPromise = this.getter(id);
    if (!(itemOrPromise instanceof Promise)) {
      return this.set(id, itemOrPromise);
    }
    const promise = addTimestamp(itemOrPromise.then(item => this.set(id, item)), now);
    this.setEntry(id, promise);
    return promise;
  }

  /** Gets item. */
  get(id, forceRefresh) {
    const entry = this.getEntry(id, forceRefresh);
    return Array.isArray(entry) ? entry[0] : entry.then(e => e[0]);
  }
}
exports.Cached = Cached;
//# sourceMappingURL=Cached.js.map