/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.views.provider;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.INotifyChangedListener;
import org.eclipse.emf.edit.provider.ItemProviderAdapter;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jsr220orm.core.internal.product.OrmProject;
import org.eclipse.jsr220orm.core.nature.OrmNature;
import org.eclipse.jsr220orm.core.nature.OrmNatureEvent;
import org.eclipse.jsr220orm.core.nature.OrmNatureListener;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.provider.MetadataItemProviderAdapterFactory;
import org.eclipse.jsr220orm.metadata.util.MetadataAdapterFactory;
import org.eclipse.swt.widgets.Display;

public class ModelContentProvider implements ITreeContentProvider,
		INotifyChangedListener, OrmNatureListener, IResourceChangeListener {

	private AdapterFactoryContentProvider contentProvider;
    private MetadataAdapterFactory factory = new MetadataItemProviderAdapterFactory();

	private Viewer viewer;

	public ModelContentProvider(ComposedAdapterFactory adapterFactory) {
		contentProvider = new AdapterFactoryContentProvider(adapterFactory);
	}

	public Object[] getElements(Object inputElement) {
		if (inputElement instanceof IWorkspaceRoot) {
			IWorkspaceRoot workspaceRoot = (IWorkspaceRoot) inputElement;
			workspaceRoot.getWorkspace().addResourceChangeListener(this,
					IResourceChangeEvent.POST_CHANGE);
			return (workspaceRoot).getProjects();
		} else if (inputElement instanceof IProject) {
			IProject project = (IProject) inputElement;
			OrmNature nature = null;
			try {
				nature = (OrmNature) project.getNature(OrmNature.ID);
			} catch (CoreException e) {
				return null;
			}
			if (nature == null) {
				return null;
			}
			nature.addOrmNatureListener(this);
            OrmProject ormProject = nature.getActiveOrmProject();
			return contentProvider.getElements(ormProject.getModel());
		} else if (inputElement instanceof OrmProject) {
			OrmProject ormProject = (OrmProject) inputElement;
			return contentProvider.getElements(ormProject.getModel());
		} else if (inputElement instanceof Object[]) {
			return (Object[]) inputElement;
        } else if (inputElement instanceof EStructuralFeatureTreeItem) {
            EStructuralFeatureTreeItem ref = (EStructuralFeatureTreeItem) inputElement;
            Collection children = ref.getChildren();
            if (children != null) {
                return children.toArray();
            }
            return null;
		} else if (inputElement instanceof Notifier) {
            Notifier notifier = (Notifier) inputElement;
            ItemProviderAdapter adapter = (ItemProviderAdapter) factory.createAdapter(notifier);
            Method method;
            try {
                method = adapter.getClass().getMethod("getChildrenFeatures", new Class[]{Object.class});
                Object ret = method.invoke(adapter, new Object[]{notifier});
                Collection children = (Collection)ret;
                ArrayList newList = new ArrayList(children.size());
                for(Iterator it = children.iterator();it.hasNext();){
                    EStructuralFeature ref = (EStructuralFeature) it.next();
                    EStructuralFeatureTreeItem structuralFeatureTreeItem = new EStructuralFeatureTreeItem((EObject) notifier, ref);
//                    if(!structuralFeatureTreeItem.getChildren().isEmpty()){
                        newList.add(structuralFeatureTreeItem);
//                    }
                }
                return newList.toArray();
            } catch (Exception e) {
                return contentProvider.getElements(inputElement);
            }
		} else {
			return contentProvider.getElements(inputElement);
		}
	}

	public void dispose() {
		viewer = null;
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		contentProvider.inputChanged(viewer, oldInput, newInput);
		this.viewer = viewer;
	}

	public Object[] getChildren(Object parentElement) {
		return getElements(parentElement);
	}

	public Object getParent(Object element) {
		if (element instanceof IProject) {
			return null;
		} else if (element instanceof OrmProject) {
			OrmProject ormProject = (OrmProject) element;
			return ormProject.getIProject();
		} else {
			Object parent = contentProvider.getParent(element);
			if (parent instanceof EntityModel) {
				return null;
			}
			return parent;
		}
	}

	public boolean hasChildren(Object element) {
		Object[] children = getChildren(element);
		return children != null && children.length > 0;
	}

	public void notifyChanged(Notification notification) {
		contentProvider.notifyChanged(notification);
	}

    public void projectActivated(OrmNatureEvent e) {
        refreshElement(e.getNature().getProject(), true, true);
        
    }

    public void projectDeactivated(OrmNatureEvent e) {
        refreshElement(e.getNature().getProject(), true, true);
    }

	public void ormNatureRemoved(OrmNatureEvent e) {
		refreshElement(e.getNature().getProject(), false, true);
	}

	public void refreshElement(Object element, boolean updateLabels,
			boolean refreshContent) {
		if (viewer != null && viewer.getControl() != null
				&& !viewer.getControl().isDisposed()) {
			RefreshRunner viewerRefresh = new RefreshRunner(viewer, element,
					updateLabels, refreshContent);
			Display d = viewer.getControl().getDisplay();
			if (d != Display.getCurrent()) {
				d.asyncExec(viewerRefresh);
			} else {
				viewerRefresh.run();
			}
		}
	}

	class RefreshRunner implements Runnable {
		Viewer viewer;

		Object element;

		boolean updateLabels;

		boolean refreshContent;

		public RefreshRunner(Viewer viewer, Object element,
				boolean updateLabels, boolean refreshContent) {
			super();
			this.viewer = viewer;
			this.element = element;
			this.updateLabels = updateLabels;
			this.refreshContent = refreshContent;
		}

		public void run() {
			if (viewer instanceof StructuredViewer) {
				StructuredViewer structuredViewer = (StructuredViewer) viewer;
				if (element != null) {
					if (refreshContent) {
						structuredViewer.refresh(element, updateLabels);
					} else if (updateLabels) {
						structuredViewer.update(element, null);
					}
				} else {
					structuredViewer.refresh(updateLabels);
				}
			} else {
				viewer.refresh();
			}
		}
	}

	public void resourceChanged(IResourceChangeEvent event) {
		System.out.println("OutlineContentProvider.resourceChanged(): "
				+ event);

	}
}
