/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io.ast;

import java.lang.annotation.Annotation;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;

/**
 * Base class for our implementations that have annotations.
 */
public abstract class AstRAnnotatedElement implements RAnnotatedElement {

	protected final AstRClassFactory factory;
	protected final List modifierList;
	protected final ImportListHelper importListHelper;
	protected final int relativeIndexInOwner;
	
	protected Map<Class, Annotation> annotations;	
	protected int modifierBits = -1;
	
	public AstRAnnotatedElement(AstRClassFactory factory, List modifierList,
			ImportListHelper importListHelper, int relativeIndexInOwner) {
		this.factory = factory;
		this.modifierList = modifierList;
		this.importListHelper = importListHelper;
		this.relativeIndexInOwner = relativeIndexInOwner;
	}

	/**
	 * Subclasses must call this at the end of their own constructor to
	 * avoid timing issues with calling {@link #getFullyQualifiedName()}.
	 */
	protected void initAnnotations() {
		annotations = factory.createAnnotations(
				this, modifierList.iterator(), importListHelper);
	}	
	
	public AstRClassFactory getFactory() {
		return factory;
	}

	public ImportListHelper getImportListHelper() {
		return importListHelper;
	}

	public <T extends Annotation> T getAnnotation(Class<T> annotationType) {
		return (T)annotations.get(annotationType);
	}

	public <T extends Annotation> T getAnnotation(Class<T> annotationType, 
			boolean returnDefault) {
		T ans = getAnnotation(annotationType);
		if (ans == null && returnDefault) {
			// create a new empty annotation and put it in the map
			AstAnnotationProxyHandler h = 
				new AstAnnotationProxyHandler(this, 
						annotationType, importListHelper, null, modifierList);
			ans = (T)h.newProxy();
			annotations.put(annotationType, ans);
		}
		return ans;
	}

	public Annotation[] getAnnotations() {
		return getDeclaredAnnotations();
	}

	public Annotation[] getDeclaredAnnotations() {
		Annotation[] all = new Annotation[annotations.size()];
		annotations.values().toArray(all);
		return all;
	}

	public boolean isAnnotationPresent(Class<? extends Annotation> annotationType) {
		return getAnnotation(annotationType) != null;
	}
	
	/**
	 * Get our fully qualified name (e.g. class name with package, field name
	 * with fully qualified name). 
	 */
	public abstract String getFullyQualifiedName();
	
	public int getModifiers() {
		if (modifierBits < 0) {
			modifierBits = 0;
			for (Iterator i = modifierList.iterator(); i.hasNext(); ) {
				Object o = i.next();
				if (o instanceof Modifier) {
					modifierBits |= ((Modifier)o).getKeyword().toFlagValue();
				}
			}
		}
		return modifierBits;
	}

	public int getRelativeIndexInOwner() {
		return relativeIndexInOwner;
	}

	public boolean isTypeEnum() {
		try {
			IType type = factory.getJavaProject().findType(getTypeName());
			return type != null && type.isEnum();
		} catch (JavaModelException e) {
			OrmPlugin.log(e);
			return false;
		}
	}

	public boolean isTypeSerializable() {
		try {
			return isTypeSerializableImp(factory.getJavaProject(), getTypeName());
		} catch (JavaModelException e) {
			OrmPlugin.log(e);
			return false;
		}
	}
	
	protected boolean isTypeSerializableImp(IJavaProject javaProject, String name) 
			throws JavaModelException {
		IType type = javaProject.findType(name);
		if (type == null) {
			return false;
		}
		if ("java.io.Serializable".equals(type.getFullyQualifiedName())) {
			return true;
		}
		for (String n : type.getSuperInterfaceTypeSignatures()) {
			if (isTypeSerializableImp(javaProject, type, n)) {
				return true;
			}
		}
		String n = type.getSuperclassTypeSignature();
		return n != null && isTypeSerializableImp(javaProject, type, n);
	}

	protected boolean isTypeSerializableImp(IJavaProject javaProject, IType type, 
			String n) throws JavaModelException {
		String sq = Signature.getSignatureQualifier(n);
		String sn = Signature.getSignatureSimpleName(n);
		if (sq.length() > 0) {
			n = sq + "." + sn;
		} else {
			n = sn;
		}
		if (type.isBinary()) {
			return isTypeSerializableImp(javaProject, n);
		} else {
			String[][] rt = type.resolveType(n);
			if (rt != null) {
				n = rt[0][0] + "." + rt[0][1];
				if (isTypeSerializableImp(javaProject, n)) {
					return true;
				}
			}
		}
		return false;
	}
	
}
