/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io.ast;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.Annotation;
import org.eclipse.jdt.core.dom.ArrayInitializer;
import org.eclipse.jdt.core.dom.BooleanLiteral;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.MarkerAnnotation;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.Name;
import org.eclipse.jdt.core.dom.NormalAnnotation;
import org.eclipse.jdt.core.dom.NumberLiteral;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.SingleMemberAnnotation;
import org.eclipse.jdt.core.dom.StringLiteral;
import org.eclipse.jdt.core.dom.TypeLiteral;
import org.eclipse.jdt.internal.corext.dom.ASTFlattener;
import org.eclipse.jsr220orm.core.util.JdbcUtils;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.io.AnnotationEx;
import org.eclipse.jsr220orm.generic.io.AnnotationRegistry;

/**
 * This is used to create proxies for java.lang.annotation.Annotation that
 * return data from an Eclipse AST tree (i.e. source code). The proxies
 * also implement {@link AnnotationEx} i.e. they support changing the
 * values of the annotation, extended default values and so on.
 */
public class AstAnnotationProxyHandler implements InvocationHandler {

	protected final AstRAnnotatedElement owner;
	protected final AnnotationRegistry reg;
	protected final Class annotationType;
	protected final ImportListHelper importListHelper;
	protected final List modifierList;
	protected final LifeCycleManager lifeCycleManager;
	protected final AST ast;
	protected Annotation ann;
	protected Map defaultMap;
	protected Map<String, String> errorMap;
	protected boolean marker;
	
	protected static final String[] VALUE = new String[]{"value"};
	protected static final String[] EMPTY = new String[0];
		
	/**
	 * This is used to lazily create the underlying AST elements for an
	 * annotation when needed. It also abstracts the location of the 
	 * annotation (e.g. on a method, nested in another annotation, in an
	 * array initializer and so on).
	 */
	protected interface LifeCycleManager {
		void create(AstAnnotationProxyHandler h);
		void remove(AstAnnotationProxyHandler h);
	}
	
	/**
	 * Create for annotation directly attached to a field, method etc.
	 * TODO refactor out modifierList in favour of lifeCycleManager only
	 */
	public AstAnnotationProxyHandler(AstRAnnotatedElement owner, 
			Class annotationType, ImportListHelper importListHelper,
			Annotation ann, List modList) {
		this.owner = owner;
		this.reg = owner.getFactory().getAnnotationRegistry();
		this.annotationType = annotationType;
		this.importListHelper = importListHelper;
		this.ast = importListHelper.getRoot().getAST();
		this.ann = ann;
		this.lifeCycleManager = null;
		this.errorMap = getAstState().getAnnotationErrorMap(this, false); 
		if (modList == null && ann != null) {
			Object ploc = ann.getParent().getStructuralProperty(
					ann.getLocationInParent());
			if (ploc instanceof List) {
				modifierList = (List)ploc;
			} else {
				modifierList = null;
			}
		} else {
			modifierList = modList;
		}
	}
	
	/**
	 * Create for annotation contained inside another annotation etc.
	 */
	public AstAnnotationProxyHandler(AstRAnnotatedElement owner, 
			Class annotationType, ImportListHelper importListHelper,
			Annotation ann, LifeCycleManager lifeCycleManager) {
		this.owner = owner;
		this.reg = owner.getFactory().getAnnotationRegistry();
		this.annotationType = annotationType;
		this.importListHelper = importListHelper;
		this.ast = importListHelper.getRoot().getAST();
		this.ann = ann;
		this.modifierList = null;
		this.lifeCycleManager = lifeCycleManager;
		this.errorMap = getAstState().getAnnotationErrorMap(this, false); 
		marker = reg.isMarker(annotationType);
	}
	
	public AstRAnnotatedElement getOwner() {
		return owner;
	}

	public Class getAnnotationType() {
		return annotationType;
	}

	/**
	 * Create an implementation of the java.lang.annotation.Annotation 
     * backed by our AST Annotation. This proxy implements annotationCls
     * and AnnotationIO.
	 */
	public java.lang.annotation.Annotation newProxy() {
		return (java.lang.annotation.Annotation)
				Proxy.newProxyInstance(annotationType.getClassLoader(),
						new Class[]{annotationType, AnnotationEx.class}, 
						this);
	}	
	
	protected AstState getAstState() {
		return owner.getFactory().getAstState();
	}
	
	public Object invoke(Object proxy, Method method, Object[] args) 
			throws Throwable {
		String methodName = method.getName();
		if (args != null && args.length == 1 && "get".equals(methodName)) {
			method = annotationType.getDeclaredMethod(
					methodName = (String)args[0], (Class[])null);
			args = null;
		}
		if (args == null) {
			if (ann instanceof NormalAnnotation) {
				MemberValuePair p = findMemberValuePair(methodName);
				if (p != null) {
					return convert(methodName, p.getValue(), 
							method.getReturnType(),	method.getDefaultValue());
				}
			} else if (ann instanceof SingleMemberAnnotation) {
				if ("value".equals(methodName)) {
					SingleMemberAnnotation sa = (SingleMemberAnnotation)ann;
					return convert("value", sa.getValue(), 
							method.getReturnType(),	method.getDefaultValue());
				}
			}
			if ("getErrorMap".equals(methodName)) {
				return errorMap;
			}
			if ("toString".equals(methodName)) {
				return ann == null ? annotationType.getSimpleName() : ann.toString();
			}
			if ("delete".equals(methodName)) {
				if (ann != null) {
					ann.delete();
					ann = null;
				}
				return null;
			}
			if ("getValueCount".equals(methodName)) {
				if (ann instanceof NormalAnnotation) {
					return ((NormalAnnotation)ann).values().size();
				}
				if (ann instanceof SingleMemberAnnotation) {
					return 1;
				}
				return 0;
			}
			if ("getValueNames".equals(methodName)) {
				return getValueNames();
			}
		} else if (args.length == 2) {
			if ("set".equals(methodName)) {
				set((String) args[0], args[1], false);
				return null;
			} else if ("setDefault".equals(methodName)) {
				if (defaultMap == null) {
					defaultMap = new HashMap();
				}
				defaultMap.put(args[0], args[1]);
			} else if ("setArraySize".equals(methodName)) {
				return setArraySize((String)args[0], (Integer)args[1]);
			}
		} else if (args.length == 1) {
			if ("getAsString".equals(methodName)) {
				String s = getAsString((String)args[0]);
				if (s != null) {
					return s;
				}
			} else if ("hasValue".equals(methodName)) {
				if (ann instanceof NormalAnnotation) {
					return findMemberValuePair((String)args[0]) != null;
				} else if (ann instanceof SingleMemberAnnotation) {
					return "value".equals(args[0]);
				} else {
					return false;
				}
			} else if ("getLocation".equals(methodName)) {
				return getLocation((String)args[0]);
			} else if ("setMarker".equals(methodName)) {
				marker = ((Boolean)args[0]).booleanValue();
				if (marker && ann == null) {
					ann = ast.newMarkerAnnotation();
					ann.setTypeName(ast.newSimpleName(
							annotationType.getSimpleName()));
					modifierList.add(chooseModifierListIndex(), ann);
					importListHelper.ensureImport(annotationType.getName());					
				}
			} else if ("getClassValue".equals(methodName)) {
				String s = getAsString((String)args[0]);
				if (s != null) {
					if (s.endsWith(".class")) {
						return s.substring(0, s.length() - 6);
					}
					return s;
				}				
			} else if ("appendArrayElement".equals(methodName)) {
				return appendArrayElement((String)args[0]);
			} else if ("getArraySize".equals(methodName)) {
				return getArraySize((String)args[0]);
			}
		} else if (args.length == 3) {
			if ("set".equals(methodName)) {
				set((String) args[0], args[1], (Boolean)args[2]);
				return null;
			}			
		}
		if (defaultMap != null) {
			Object def = defaultMap.get(methodName);
			if (def != null) {
				return def;
			}
		}
		Class returnType = method.getReturnType();
		if (returnType.isAnnotation()) {
			return convertAnnotation(methodName, null, returnType);
		}
//		if (returnType.isArray()) {
//			Class ct = returnType.getComponentType();
//			if (ct.isAnnotation()) {
//				// make sure the returned Annotation[] can be cast to
//				// AnnotationEx[] by wrapping each element
//				Object[] ans = (Object[])Array.newInstance(annType, 1);
//			}
//		}
		return method.getDefaultValue();
	}

	protected String getAsString(String name) {
		if (ann instanceof NormalAnnotation) {
			MemberValuePair p = findMemberValuePair(name);
			if (p != null) {
				return ASTFlattener.asString(p.getValue());
			}
		} else if (ann instanceof SingleMemberAnnotation) {
			if ("value".equals(name)) {
				SingleMemberAnnotation sa = (SingleMemberAnnotation)ann;
				return ASTFlattener.asString(sa.getValue());
			}
		}		
		return null;
	}

	protected MemberValuePair findMemberValuePair(String name) {
		List values = ((NormalAnnotation)ann).values();
		for (int i = values.size() - 1; i >= 0; i--) {
			MemberValuePair p = (MemberValuePair)values.get(i);
			if (p.getName().getIdentifier().equals(name)) {
				return p;
			}
		}
		return null;
	}

	/**
	 * Convert value into cls. Returns def and registers an error if this is 
	 * not successfull.
	 */
	protected Object convert(String name, Expression value, Class cls, 
			Object def) {
		Object ans = null;
		switch (value.getNodeType()) {
		
		case ASTNode.STRING_LITERAL:
			ans = ((StringLiteral)value).getLiteralValue();
			break;
			
		case ASTNode.QUALIFIED_NAME:
			String qn = ((QualifiedName)value).getFullyQualifiedName();
			ans = reg.getNamedValue(qn);
			if (ans == null) {
				setError(name, "Invalid value: " + qn);
			}
			break;

		case ASTNode.NUMBER_LITERAL:
			try {
				ans = Integer.parseInt(((NumberLiteral) value).getToken());
			} catch (NumberFormatException e) {
				setError(name, e.toString());
			}
			break;

		case ASTNode.BOOLEAN_LITERAL:
			ans = ((BooleanLiteral)value).booleanValue();
			break;
			
		case ASTNode.NORMAL_ANNOTATION:
			ans = convertAnnotation(name, (Annotation)value, cls);
			break;
			
		case ASTNode.ARRAY_INITIALIZER:
			if (cls.isArray()) {
				Class componentType = cls.getComponentType();
				if (componentType.isAnnotation()) {
					ans = convertAnnotationArray(name, (ArrayInitializer)value, 
							cls.getComponentType(), def);
					break;
				} else {
					ArrayInitializer ai = (ArrayInitializer)value;
					int n = ai.expressions().size();
					Object[] a = (Object[])Array.newInstance(componentType, n);
					for (int i = 0; i < n; i++) {
						a[i] = convert(name, 
								(Expression)ai.expressions().get(i), 
								componentType, null);
					}
					if (getError(name) != null) {
						return def;
					}
					return a;
				}
			}
		default:
			setError(name, "Cannot convert to " + cls.getName() + ": " + value);
			return def;
		}		
		if (cls.isPrimitive()) {
			cls = Utils.PRIM_WRAPPER_MAP.get(cls);
		} 
		if (cls.isInstance(ans)) {
			return ans;
		} else if (cls.isArray() && cls.getComponentType().isInstance(ans)) {
			Object[] a = (Object[])Array.newInstance(cls.getComponentType(), 1);
			a[0] = ans;
			return a;
		}
		setError(name, "Expected " + cls.getName() + ": " + ans);			
		return def;
	}

	/**
	 * Set the error associated with name.
	 */
	protected void setError(String name, String msg) {
		if (errorMap == null) {
			errorMap = getAstState().getAnnotationErrorMap(this, true); 
		}
		errorMap.put(name, msg);
	}

	/**
	 * Get the error associated with name or null if none.
	 */
	protected String getError(String name) {
		if (errorMap == null) {
			return null;
		}
		return errorMap.get(name);
	}
	
	/**
	 * Convert an AST value into an appropriate Java object. 
	 */
	protected Object convert(Expression value) {
		switch (value.getNodeType()) {
		
			case ASTNode.STRING_LITERAL:
				return ((StringLiteral)value).getLiteralValue();
				
			case ASTNode.QUALIFIED_NAME:
				QualifiedName qn = (QualifiedName)value;
				return reg.getNamedValue(qn.getFullyQualifiedName());

			case ASTNode.NUMBER_LITERAL:
				try {
					return Integer.parseInt(((NumberLiteral) value).getToken());
				} catch (NumberFormatException e) {
					return 0;
				}

			case ASTNode.BOOLEAN_LITERAL:
				return ((BooleanLiteral)value).booleanValue();
		}		
		return null;
	}	
	
	/**
	 * Convert Java object into appropriate AST Expression. Returns null
	 * if not possible.
	 */
	protected Expression convert(String name, Object value) {
		if (value == null) {
			return null;
		}
		Class cls = value.getClass();
		if (cls.isArray()) {
			ArrayInitializer ai = ast.newArrayInitializer();
			Object[] a = (Object[])value;
			for (int i = 0; i < a.length; i++) {
				Expression e = convert(name, a[i]);
				if (e != null) {
					ai.expressions().add(e);
				}
			}
			return ai;
		}
		if (value instanceof String) {
			if (getValueType(name) == Class.class) {
				Name cname = ast.newName((String)value);
				TypeLiteral t = ast.newTypeLiteral();
				t.setType(ast.newSimpleType(cname));
				return t;				
			}
			StringLiteral v = ast.newStringLiteral();
			v.setLiteralValue((String)value);
			return v;
		} else if (value instanceof Enum) {
			Enum e = (Enum)value;
			QualifiedName qn = ast.newQualifiedName(
					ast.newSimpleName(e.getClass().getSimpleName()),
					ast.newSimpleName(e.name()));
			return qn;
		} else if (value instanceof Integer) {
			return ast.newNumberLiteral(value.toString());
		} else if (value instanceof Boolean) {
			return ast.newBooleanLiteral(((Boolean)value).booleanValue());
		}
		return null;
	}
	
	/**
	 * Get the default value for name from our local map or the registry.
	 */
	protected Object getDefaultValue(String name) {
		if (defaultMap != null && defaultMap.containsKey(name)) {
			return defaultMap.get(name);
		}
		return reg.getDefault(annotationType, name);
	}
	
	/**
	 * Set the value name to value. If treatEmptyAsNull is true and the
	 * value is an empty String then it is considered null. If value is the 
	 * same as the default or null then remove it. Note that if name is in 
	 * the {@link #getErrorMap()} then this is a NOP.
	 * <p>
	 * This converts annotations between types (Normal, Marker, 
	 * SingleMember) to keep the source code as simple as possible.
	 */
	protected void set(String name, Object value, boolean treatEmptyAsNull) {
		if (errorMap != null && errorMap.containsKey(name)) {
			return;
		}
		if (treatEmptyAsNull && value instanceof String 
				&& ((String)value).length() == 0) {
			value = null;
		}
		Expression astValue = null;
		Object def = getDefaultValue(name);
		if (def == value || def != null && def.equals(value)) {
			value = null;
			astValue = null;
		} else {
			if ("jdbcType".equals(name) && value instanceof Integer) {
				String s = JdbcUtils.getQualifiedJdbcTypeName((Integer)value);
				if (s != null) {
					astValue = ast.newName(s);
					importListHelper.ensureImport("java.sql.Types");
				}
			}
			if (astValue == null) {
				astValue = convert(name, value);
			}
		}
		if (astValue != null) {
			if (value instanceof Enum) {
				importListHelper.ensureImport(value.getClass().getName());
			} else {
				Class cls = value.getClass();
				if (cls.isArray() && cls.getComponentType().isEnum()) {
					importListHelper.ensureImport(
							cls.getComponentType().getName());					
				}
			}
		}
		if (ann == null && astValue != null) {
			createOurAstAnnotation();
		}
		if (ann instanceof NormalAnnotation) {
			NormalAnnotation na = (NormalAnnotation)ann;
			List values = na.values();
			for (Iterator i = values.iterator(); i.hasNext(); ) {
				MemberValuePair p = (MemberValuePair)i.next();
				if (p.getName().getFullyQualifiedName().equals(name)) {
					if (astValue == null) {
						values.remove(p);
						if (values.isEmpty()) { 
							if (marker) {
								// convert to marker annotation
								MarkerAnnotation ma = ast.newMarkerAnnotation();
								Name typeName = (Name)ASTNode.copySubtree(ast, 
										na.getTypeName());
								ma.setTypeName(typeName);
								Utils.replace(na, ma);
								ann = ma;
							} else {
								// remove from source
								if (lifeCycleManager != null) {
									lifeCycleManager.remove(this);
								} else {
									ann.delete();
								}
								ann = null;
							}
						}
					} else {
						p.setValue(astValue);
					}
					return;
				}
			}
			if (astValue != null) {
				values.add(chooseIndexForNewPair(name, values),
						createMemberValuePair(name, astValue));
			}
		} else if (ann instanceof MarkerAnnotation) {
			if (astValue != null) {
				convertMarkerToNormal(name, astValue);				
			}
		} else if (ann instanceof SingleMemberAnnotation) {
			SingleMemberAnnotation sa = (SingleMemberAnnotation)ann;
			if (astValue != null) {
				sa.setValue(astValue);
			} else {
				if (reg.isMarker(annotationType)) {
					// convert to marker annotation
					MarkerAnnotation ma = ast.newMarkerAnnotation();
					Name typeName = (Name)ASTNode.copySubtree(ast, 
							sa.getTypeName());
					ma.setTypeName(typeName);
					Utils.replace(sa, ma);
					ann = ma;
				} else {
					// remove from source
					ann.delete();
					ann = null;
				}				
			}
		}
	}
	
	/**
	 * Choose the best index for name in values. 
	 */
	protected int chooseIndexForNewPair(String name, List values) {
		Map<String, Integer> map = reg.getOrderingMap(annotationType);
		Integer pos = map.get(name);
		if (pos == null) {
			return values.size();
		}
		int posi = pos;
		int n = values.size();
		for (int i = 0; i < n; i++) {
			MemberValuePair p = (MemberValuePair)values.get(i);
			String pname = p.getName().getFullyQualifiedName();
			Integer q = map.get(pname);
			if (q == null || posi < q) {
				return i;
			}
		}
		return n;
	}

	/**
	 * Convert our Annotation from a MarkerAnnotation to a NormalAnnotation
	 * with an initial name=astValue pair.
	 */
	protected void convertMarkerToNormal(String name, Expression astValue) {
		NormalAnnotation na = ast.newNormalAnnotation();
		Name typeName = (Name)ASTNode.copySubtree(ast, ann.getTypeName());
		na.setTypeName(typeName);
		na.values().add(createMemberValuePair(name, astValue));
		Utils.replace(ann, na);
		ann = na;
	}

	/**
	 * Create an AST Annotation in the source for us. This will be a
	 * SingleMemberAnnotation or a NormalAnnotation. If we are contained
	 * in an annotation then that might be created as well if it does not
	 * exist.
	 */
	protected void createOurAstAnnotation() {
		if (reg.isSingleValue(annotationType)) {
			ann = ast.newSingleMemberAnnotation();
		} else {
			ann = ast.newNormalAnnotation();
		}
		ann.setTypeName(ast.newSimpleName(annotationType.getSimpleName()));
		if (modifierList == null) {
			lifeCycleManager.create(this);
		} else {
			modifierList.add(chooseModifierListIndex(), ann);
		}
		importListHelper.ensureImport(annotationType.getName());
	}

	protected MemberValuePair createMemberValuePair(String name, 
			Expression value) {
		MemberValuePair p = ast.newMemberValuePair();
		p.setName(ast.newSimpleName(name));
		p.setValue(value);
		return p;
	}	
	
	/**
	 * Select the correct position for us in the modifier list. 
	 */
	protected int chooseModifierListIndex() {
		// sort annotations according to the registry order and place them 
		// before other mods
		int n = modifierList.size();
		int pos = reg.getOrderingIndex(ann.getTypeName().getFullyQualifiedName());
		for (int i = 0; i < n; i++) {
			Object o = modifierList.get(i);
			if (o instanceof Annotation) {
				String s = ((Annotation)o).getTypeName().getFullyQualifiedName();
				if (pos < reg.getOrderingIndex(s)) {
					return i;
				}
			} else {
				return i;
			}
		}		
		return n;
	}	
	
	protected Map getLocation(String name) {
		if (ann == null) {
			return owner.getLocation();
		}
		if (ann instanceof NormalAnnotation) {
			MemberValuePair p = findMemberValuePair(name);
			if (p != null) {
				return Utils.createMarkerLocation(p);
			}
		} 
		return Utils.createMarkerLocation(ann);
	}
	
	protected Object convertAnnotation(final String name, final Annotation e, 
			Class cls) {
		boolean array = cls.isArray();
		Class annType = array ? cls.getComponentType() : cls;
		LifeCycleManager lcm = new LifeCycleManager() {
			
			public void create(AstAnnotationProxyHandler h) {
				if (ann == null) {
					createOurAstAnnotation();
				}
				if (ann instanceof NormalAnnotation) {
					NormalAnnotation na = (NormalAnnotation)ann;
					na.values().add(createMemberValuePair(name, h.ann));
				} else {
					SingleMemberAnnotation sa = (SingleMemberAnnotation)ann;
					sa.setValue(h.ann);
				}
			}

			public void remove(AstAnnotationProxyHandler h) {
				set(name, null, false);
			}
		};
		
		AstAnnotationProxyHandler h = 
			new AstAnnotationProxyHandler(owner, annType, importListHelper, e,
					lcm);
		java.lang.annotation.Annotation p = h.newProxy();
		if (array) {
			Object[] ans = (Object[])Array.newInstance(annType, 1);
			ans[0] = p;
			return ans;
		}
		return p;
	}
	
	protected Object convertAnnotationArray(String name, 
			ArrayInitializer ai, Class annotationType, Object def) {
		int n = ai.expressions().size();
		Object[] ans = (Object[])Array.newInstance(annotationType, n);
		for (int i = 0; i < n; i++) {
			Expression e = (Expression)ai.expressions().get(i);
			if (!(e instanceof Annotation)) {
				setError(name, "Expected " + annotationType.getName() + ": " + e);
				return def;
			}
			AstAnnotationProxyHandler h = 
				new AstAnnotationProxyHandler(owner,
						annotationType, importListHelper, (Annotation)e, (List)null);
			ans[i] = h.newProxy();
		}
		return ans;
	}
	
	/**
	 * Set the size of the array[] valued value name to size. This will
	 * delete excess annotations or create new annotations as required.
	 */
	protected boolean setArraySize(String name, int size) {
		if (errorMap != null && errorMap.containsKey(name)) {
			return false;
		}
		ArrayInitializer ai = getOrCreateArrayInitializer(name);
		List expressions = ai.expressions();
		int n = expressions.size();
		if (n == size) {
			return true;
		}
		if (n > size) {
			for (; n > size; ) {
				((Expression)expressions.get(--n)).delete();
			}
		} else {
			Class componentType = getValueType(name).getComponentType();
			for (int i = n; i < size; i++) {
				Annotation a;
				if (reg.isSingleValue(componentType)) {
					a = ast.newSingleMemberAnnotation();
				} else {
					a = ast.newNormalAnnotation();
				}
				a.setTypeName(ast.newSimpleName(componentType.getSimpleName()));
				expressions.add(a);
				importListHelper.ensureImport(componentType.getName());
			}
		}
		return true;
	}

	/**
	 * Make sure the value name exists and that its value is an 
	 * ArrayInitializer. This will create a new Annotation and 
	 * ArrayInitializer if needed.
	 */
	protected ArrayInitializer getOrCreateArrayInitializer(String name) {
		ArrayInitializer ai;
		if (ann == null) {
			createOurAstAnnotation();
		}
		if (ann instanceof NormalAnnotation) {
			MemberValuePair p = findMemberValuePair(name);
			if (p != null) {
				Expression v = p.getValue();
				if (v instanceof ArrayInitializer) {
					ai = (ArrayInitializer)v;
				} else {
					p.setValue(ai = ast.newArrayInitializer());
				}
			} else {
				NormalAnnotation na = (NormalAnnotation)ann;
				ai = ast.newArrayInitializer();
				na.values().add(createMemberValuePair(name, ai));
			}
		} else if (ann instanceof SingleMemberAnnotation) {
			SingleMemberAnnotation sa = (SingleMemberAnnotation)ann;
			Expression v = sa.getValue();
			if (v instanceof ArrayInitializer) {
				ai = (ArrayInitializer)v;
			} else {
				sa.setValue(ai = ast.newArrayInitializer());
			}
		} else {
			convertMarkerToNormal(name, ai = ast.newArrayInitializer());
		}
		return ai;
	}

	/**
	 * Get the size of the array[] annotation value name. If the value has
	 * not been set then 0 is returned.
	 */
	protected int getArraySize(String name) {
		if (ann instanceof NormalAnnotation) {
			MemberValuePair p = findMemberValuePair(name);
			if (p != null) {
				Expression v = p.getValue();
				if (v instanceof ArrayInitializer) {
					return ((ArrayInitializer)v).expressions().size();
				}
			}
		} else if (ann instanceof SingleMemberAnnotation) {
			SingleMemberAnnotation sa = (SingleMemberAnnotation)ann;
			Expression v = sa.getValue();
			if (v instanceof ArrayInitializer) {
				return ((ArrayInitializer)v).expressions().size();
			}
		}
		return 0;
	}
	
	/**
	 * Create a new proxy that will be appended to the end of the array[] 
	 * value name if any of its values are set to non-defaults.
	 */
	protected Object appendArrayElement(final String name) {
		final Class componentType = getValueType(name).getComponentType();
		LifeCycleManager lcm = new LifeCycleManager() {
			public void create(AstAnnotationProxyHandler h) {
				ArrayInitializer ai = getOrCreateArrayInitializer(name);
				List expressions = ai.expressions();
				expressions.add(h.ann);
				importListHelper.ensureImport(componentType.getName());
			}

			public void remove(AstAnnotationProxyHandler h) {
				if (h.ann != null) {
					h.ann.delete();
					h.ann = null;
				}
			}
		};
		AstAnnotationProxyHandler h = 
			new AstAnnotationProxyHandler(owner,
					componentType, importListHelper, null, lcm);
		return h.newProxy();
	} 
	
	/**
	 * Get the type of the value with name. 
	 */
	protected Class getValueType(String name) {
		try {
			Method m = annotationType.getDeclaredMethod(name, (Class[])null);
			return m.getReturnType();
		} catch (Exception e) {
			// no exceptions should be possible but anyway ...
			throw new IllegalArgumentException(e);
		}
	}
	
	/**
	 * Get the names of the values on this annotatation or an empty array
	 * if none.
	 */
	protected String[] getValueNames() {
		if (ann instanceof NormalAnnotation) {
			List values = ((NormalAnnotation)ann).values();
			int n = values.size();
			String[] a = new String[n];
			for (int i = 0; i < n; i++) {
				MemberValuePair p = (MemberValuePair)values.get(i);
				a[i] = p.getName().getFullyQualifiedName();
			}
			return a;
		}
		if (ann instanceof SingleMemberAnnotation) {
			return VALUE;
		}
		return EMPTY;
	}		
	
}
