/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.messages.Either;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * The document ranges formatting request is sent from the client to the server to format
 * multiple ranges at once in a document.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class DocumentRangesFormattingParams implements WorkDoneProgressParams {
  /**
   * An optional token that a server can use to report work done progress.
   */
  private Either<String, Integer> workDoneToken;

  /**
   * The document to format.
   */
  @NonNull
  private TextDocumentIdentifier textDocument;

  /**
   * The format options.
   */
  @NonNull
  private FormattingOptions options;

  /**
   * The ranges to format.
   */
  @NonNull
  private List<Range> ranges;

  public DocumentRangesFormattingParams() {
    ArrayList<Range> _arrayList = new ArrayList<Range>();
    this.ranges = _arrayList;
  }

  public DocumentRangesFormattingParams(@NonNull final TextDocumentIdentifier textDocument, @NonNull final FormattingOptions options, @NonNull final List<Range> ranges) {
    this.textDocument = Preconditions.<TextDocumentIdentifier>checkNotNull(textDocument, "textDocument");
    this.options = Preconditions.<FormattingOptions>checkNotNull(options, "options");
    this.ranges = Preconditions.<List<Range>>checkNotNull(ranges, "ranges");
  }

  /**
   * An optional token that a server can use to report work done progress.
   */
  @Override
  public Either<String, Integer> getWorkDoneToken() {
    return this.workDoneToken;
  }

  /**
   * An optional token that a server can use to report work done progress.
   */
  public void setWorkDoneToken(final Either<String, Integer> workDoneToken) {
    this.workDoneToken = workDoneToken;
  }

  public void setWorkDoneToken(final String workDoneToken) {
    if (workDoneToken == null) {
      this.workDoneToken = null;
      return;
    }
    this.workDoneToken = Either.forLeft(workDoneToken);
  }

  public void setWorkDoneToken(final Integer workDoneToken) {
    if (workDoneToken == null) {
      this.workDoneToken = null;
      return;
    }
    this.workDoneToken = Either.forRight(workDoneToken);
  }

  /**
   * The document to format.
   */
  @NonNull
  public TextDocumentIdentifier getTextDocument() {
    return this.textDocument;
  }

  /**
   * The document to format.
   */
  public void setTextDocument(@NonNull final TextDocumentIdentifier textDocument) {
    this.textDocument = Preconditions.checkNotNull(textDocument, "textDocument");
  }

  /**
   * The format options.
   */
  @NonNull
  public FormattingOptions getOptions() {
    return this.options;
  }

  /**
   * The format options.
   */
  public void setOptions(@NonNull final FormattingOptions options) {
    this.options = Preconditions.checkNotNull(options, "options");
  }

  /**
   * The ranges to format.
   */
  @NonNull
  public List<Range> getRanges() {
    return this.ranges;
  }

  /**
   * The ranges to format.
   */
  public void setRanges(@NonNull final List<Range> ranges) {
    this.ranges = Preconditions.checkNotNull(ranges, "ranges");
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("workDoneToken", this.workDoneToken);
    b.add("textDocument", this.textDocument);
    b.add("options", this.options);
    b.add("ranges", this.ranges);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    DocumentRangesFormattingParams other = (DocumentRangesFormattingParams) obj;
    if (this.workDoneToken == null) {
      if (other.workDoneToken != null)
        return false;
    } else if (!this.workDoneToken.equals(other.workDoneToken))
      return false;
    if (this.textDocument == null) {
      if (other.textDocument != null)
        return false;
    } else if (!this.textDocument.equals(other.textDocument))
      return false;
    if (this.options == null) {
      if (other.options != null)
        return false;
    } else if (!this.options.equals(other.options))
      return false;
    if (this.ranges == null) {
      if (other.ranges != null)
        return false;
    } else if (!this.ranges.equals(other.ranges))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.workDoneToken== null) ? 0 : this.workDoneToken.hashCode());
    result = prime * result + ((this.textDocument== null) ? 0 : this.textDocument.hashCode());
    result = prime * result + ((this.options== null) ? 0 : this.options.hashCode());
    return prime * result + ((this.ranges== null) ? 0 : this.ranges.hashCode());
  }
}
