/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Represents a collection of completion items to be presented in the editor.
 */
@SuppressWarnings("all")
public class CompletionList {
  /**
   * This list is not complete. Further typing should result in recomputing this list.
   */
  private boolean isIncomplete;

  /**
   * The completion items.
   */
  @NonNull
  private List<CompletionItem> items;

  /**
   * In many cases, the items of an actual completion result share the same
   * value for properties like {@link CompletionItem#commitCharacters} or the range of a text
   * edit. A completion list can therefore define item defaults which will
   * be used if a completion item itself doesn't specify the value.
   * <p>
   * If a completion list specifies a default value and a completion item
   * also specifies a corresponding value, the rules for combining these are
   * defined by {@link CompletionList#applyKind} (if the client supports it),
   * defaulting to {@link ApplyKind#Replace}.
   * <p>
   * Servers are only allowed to return default values if the client
   * signals support for this via the {@link CompletionListCapabilities#itemDefaults}
   * capability.
   */
  @ProtocolSince("3.17.0")
  private CompletionItemDefaults itemDefaults;

  /**
   * Specifies how fields from a completion item should be combined with those
   * from {@link CompletionList#itemDefaults}.
   * <p>
   * If unspecified, all fields will be treated as {@link ApplyKind#Replace}.
   * <p>
   * If a field's value is {@link ApplyKind#Replace}, the value from a completion item
   * (if provided and not {@code null}) will always be used instead of the value
   * from {@link CompletionList#itemDefaults}.
   * <p>
   * If a field's value is {@link ApplyKind#Merge}, the values will be merged using
   * the rules defined against each field in {@link CompletionApplyKind}.
   * <p>
   * Servers are only allowed to return {@link CompletionList#applyKind} if the client
   * signals support for this via the {@link CompletionListCapabilities#applyKindSupport}
   * capability.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  private CompletionApplyKind applyKind;

  public CompletionList() {
    this(new ArrayList<CompletionItem>());
  }

  public CompletionList(@NonNull final List<CompletionItem> items) {
    this.items = Preconditions.<List<CompletionItem>>checkNotNull(items, "items");
  }

  public CompletionList(final boolean isIncomplete, @NonNull final List<CompletionItem> items) {
    this(items);
    this.isIncomplete = isIncomplete;
  }

  public CompletionList(final boolean isIncomplete, @NonNull final List<CompletionItem> items, final CompletionItemDefaults itemDefaults) {
    this(isIncomplete, items);
    this.itemDefaults = itemDefaults;
  }

  /**
   * This list is not complete. Further typing should result in recomputing this list.
   */
  public boolean isIncomplete() {
    return this.isIncomplete;
  }

  /**
   * This list is not complete. Further typing should result in recomputing this list.
   */
  public void setIsIncomplete(final boolean isIncomplete) {
    this.isIncomplete = isIncomplete;
  }

  /**
   * The completion items.
   */
  @NonNull
  public List<CompletionItem> getItems() {
    return this.items;
  }

  /**
   * The completion items.
   */
  public void setItems(@NonNull final List<CompletionItem> items) {
    this.items = Preconditions.checkNotNull(items, "items");
  }

  /**
   * In many cases, the items of an actual completion result share the same
   * value for properties like {@link CompletionItem#commitCharacters} or the range of a text
   * edit. A completion list can therefore define item defaults which will
   * be used if a completion item itself doesn't specify the value.
   * <p>
   * If a completion list specifies a default value and a completion item
   * also specifies a corresponding value, the rules for combining these are
   * defined by {@link CompletionList#applyKind} (if the client supports it),
   * defaulting to {@link ApplyKind#Replace}.
   * <p>
   * Servers are only allowed to return default values if the client
   * signals support for this via the {@link CompletionListCapabilities#itemDefaults}
   * capability.
   */
  @ProtocolSince("3.17.0")
  public CompletionItemDefaults getItemDefaults() {
    return this.itemDefaults;
  }

  /**
   * In many cases, the items of an actual completion result share the same
   * value for properties like {@link CompletionItem#commitCharacters} or the range of a text
   * edit. A completion list can therefore define item defaults which will
   * be used if a completion item itself doesn't specify the value.
   * <p>
   * If a completion list specifies a default value and a completion item
   * also specifies a corresponding value, the rules for combining these are
   * defined by {@link CompletionList#applyKind} (if the client supports it),
   * defaulting to {@link ApplyKind#Replace}.
   * <p>
   * Servers are only allowed to return default values if the client
   * signals support for this via the {@link CompletionListCapabilities#itemDefaults}
   * capability.
   */
  @ProtocolSince("3.17.0")
  public void setItemDefaults(final CompletionItemDefaults itemDefaults) {
    this.itemDefaults = itemDefaults;
  }

  /**
   * Specifies how fields from a completion item should be combined with those
   * from {@link CompletionList#itemDefaults}.
   * <p>
   * If unspecified, all fields will be treated as {@link ApplyKind#Replace}.
   * <p>
   * If a field's value is {@link ApplyKind#Replace}, the value from a completion item
   * (if provided and not {@code null}) will always be used instead of the value
   * from {@link CompletionList#itemDefaults}.
   * <p>
   * If a field's value is {@link ApplyKind#Merge}, the values will be merged using
   * the rules defined against each field in {@link CompletionApplyKind}.
   * <p>
   * Servers are only allowed to return {@link CompletionList#applyKind} if the client
   * signals support for this via the {@link CompletionListCapabilities#applyKindSupport}
   * capability.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public CompletionApplyKind getApplyKind() {
    return this.applyKind;
  }

  /**
   * Specifies how fields from a completion item should be combined with those
   * from {@link CompletionList#itemDefaults}.
   * <p>
   * If unspecified, all fields will be treated as {@link ApplyKind#Replace}.
   * <p>
   * If a field's value is {@link ApplyKind#Replace}, the value from a completion item
   * (if provided and not {@code null}) will always be used instead of the value
   * from {@link CompletionList#itemDefaults}.
   * <p>
   * If a field's value is {@link ApplyKind#Merge}, the values will be merged using
   * the rules defined against each field in {@link CompletionApplyKind}.
   * <p>
   * Servers are only allowed to return {@link CompletionList#applyKind} if the client
   * signals support for this via the {@link CompletionListCapabilities#applyKindSupport}
   * capability.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public void setApplyKind(final CompletionApplyKind applyKind) {
    this.applyKind = applyKind;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("isIncomplete", this.isIncomplete);
    b.add("items", this.items);
    b.add("itemDefaults", this.itemDefaults);
    b.add("applyKind", this.applyKind);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    CompletionList other = (CompletionList) obj;
    if (other.isIncomplete != this.isIncomplete)
      return false;
    if (this.items == null) {
      if (other.items != null)
        return false;
    } else if (!this.items.equals(other.items))
      return false;
    if (this.itemDefaults == null) {
      if (other.itemDefaults != null)
        return false;
    } else if (!this.itemDefaults.equals(other.itemDefaults))
      return false;
    if (this.applyKind == null) {
      if (other.applyKind != null)
        return false;
    } else if (!this.applyKind.equals(other.applyKind))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + (this.isIncomplete ? 1231 : 1237);
    result = prime * result + ((this.items== null) ? 0 : this.items.hashCode());
    result = prime * result + ((this.itemDefaults== null) ? 0 : this.itemDefaults.hashCode());
    return prime * result + ((this.applyKind== null) ? 0 : this.applyKind.hashCode());
  }
}
