/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.btree;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import org.eclipse.rdf4j.common.io.ByteArrayUtil;
import org.eclipse.rdf4j.sail.nativerdf.btree.BTree;
import org.eclipse.rdf4j.sail.nativerdf.btree.NodeListener;

class Node {
    private final int id;
    private final BTree tree;
    private final byte[] data;
    private int valueCount;
    private final AtomicInteger usageCount = new AtomicInteger(0);
    private boolean dataChanged;
    private final ConcurrentLinkedDeque<NodeListener> listeners = new ConcurrentLinkedDeque();

    public Node(int id, BTree tree) {
        if (id <= 0) {
            throw new IllegalArgumentException("id must be larger than 0, is: " + id + " in " + tree.getFile());
        }
        this.id = id;
        this.tree = tree;
        this.valueCount = 0;
        this.data = new byte[tree.nodeSize + tree.slotSize];
    }

    public int getID() {
        return this.id;
    }

    public String toString() {
        return "node " + this.getID();
    }

    public boolean isLeaf() {
        return this.getChildNodeID(0) == 0;
    }

    public int use() {
        return this.usageCount.incrementAndGet();
    }

    public void release() throws IOException {
        int newUsage = this.usageCount.decrementAndGet();
        assert (newUsage >= 0) : "Releasing node while usage count is " + (newUsage + 1);
        if (newUsage == 0) {
            this.tree.releaseNode(this);
        }
    }

    public int getUsageCount() {
        return this.usageCount.get();
    }

    public boolean dataChanged() {
        return this.dataChanged;
    }

    public int getValueCount() {
        return this.valueCount;
    }

    public int getNodeCount() {
        if (this.isLeaf()) {
            return 0;
        }
        return this.valueCount + 1;
    }

    public boolean isEmpty() {
        return this.valueCount == 0;
    }

    public boolean isFull() {
        return this.valueCount == this.tree.branchFactor - 1;
    }

    public byte[] getValue(int valueIdx) {
        assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
        assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
        return ByteArrayUtil.get(this.data, this.valueIdx2offset(valueIdx), this.tree.valueSize);
    }

    public void setValue(int valueIdx, byte[] value) {
        assert (value != null) : "value must not be null";
        assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
        assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
        ByteArrayUtil.put(value, this.data, this.valueIdx2offset(valueIdx));
        this.dataChanged = true;
    }

    public byte[] removeValueRight(int valueIdx) {
        assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
        assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
        byte[] value = this.getValue(valueIdx);
        int endOffset = this.valueIdx2offset(this.valueCount);
        if (valueIdx < this.valueCount - 1) {
            this.shiftData(this.valueIdx2offset(valueIdx + 1), endOffset, -this.tree.slotSize);
        }
        this.clearData(endOffset - this.tree.slotSize, endOffset);
        this.setValueCount(--this.valueCount);
        this.dataChanged = true;
        this.notifyValueRemoved(valueIdx);
        return value;
    }

    public byte[] removeValueLeft(int valueIdx) {
        assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
        assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
        byte[] value = this.getValue(valueIdx);
        int endOffset = this.valueIdx2offset(this.valueCount);
        this.shiftData(this.nodeIdx2offset(valueIdx + 1), endOffset, -this.tree.slotSize);
        this.clearData(endOffset - this.tree.slotSize, endOffset);
        this.setValueCount(--this.valueCount);
        this.dataChanged = true;
        this.notifyValueRemoved(valueIdx);
        return value;
    }

    public int getChildNodeID(int nodeIdx) {
        assert (nodeIdx >= 0) : "nodeIdx must be positive, is: " + nodeIdx;
        assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
        return ByteArrayUtil.getInt(this.data, this.nodeIdx2offset(nodeIdx));
    }

    public void setChildNodeID(int nodeIdx, int nodeID) {
        assert (nodeIdx >= 0) : "nodeIdx must not be negative, is: " + nodeIdx;
        assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
        assert (nodeID >= 0) : "nodeID must not be negative, is: " + nodeID;
        ByteArrayUtil.putInt(nodeID, this.data, this.nodeIdx2offset(nodeIdx));
        this.dataChanged = true;
    }

    public Node getChildNode(int nodeIdx) throws IOException {
        assert (nodeIdx >= 0) : "nodeIdx must be positive, is: " + nodeIdx;
        assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
        int childNodeID = this.getChildNodeID(nodeIdx);
        return this.tree.readNode(childNodeID);
    }

    public int search(byte[] key) {
        int low = 0;
        int high = this.valueCount - 1;
        while (low <= high) {
            int mid = low + high >> 1;
            int diff = this.tree.comparator.compareBTreeValues(key, this.data, this.valueIdx2offset(mid), this.tree.valueSize);
            if (diff < 0) {
                high = mid - 1;
                continue;
            }
            if (diff > 0) {
                low = mid + 1;
                continue;
            }
            return mid;
        }
        return -low - 1;
    }

    public void insertValueNodeIDPair(int valueIdx, byte[] value, int nodeID) {
        assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
        assert (valueIdx <= this.valueCount) : "valueIdx out of range (" + valueIdx + " > " + this.valueCount + ")";
        assert (value != null) : "value must not be null";
        assert (nodeID >= 0) : "nodeID must not be negative, is: " + nodeID;
        int offset = this.valueIdx2offset(valueIdx);
        if (valueIdx < this.valueCount) {
            this.shiftData(offset, this.valueIdx2offset(this.valueCount), this.tree.slotSize);
        }
        ByteArrayUtil.put(value, this.data, offset);
        ByteArrayUtil.putInt(nodeID, this.data, offset + this.tree.valueSize);
        this.setValueCount(++this.valueCount);
        this.notifyValueAdded(valueIdx);
        this.dataChanged = true;
    }

    public void insertNodeIDValuePair(int nodeIdx, int nodeID, byte[] value) {
        assert (nodeIdx >= 0) : "nodeIdx must not be negative, is: " + nodeIdx;
        assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
        assert (nodeID >= 0) : "nodeID must not be negative, is: " + nodeID;
        assert (value != null) : "value must not be null";
        int offset = this.nodeIdx2offset(nodeIdx);
        this.shiftData(offset, this.valueIdx2offset(this.valueCount), this.tree.slotSize);
        ByteArrayUtil.putInt(nodeID, this.data, offset);
        ByteArrayUtil.put(value, this.data, offset + 4);
        this.setValueCount(++this.valueCount);
        this.notifyValueAdded(nodeIdx);
        this.dataChanged = true;
    }

    public byte[] splitAndInsert(byte[] newValue, int newNodeID, int newValueIdx, Node newNode) throws IOException {
        this.insertValueNodeIDPair(newValueIdx, newValue, newNodeID);
        assert (this.valueCount == this.tree.branchFactor) : "Node contains " + this.valueCount + " values, expected " + this.tree.branchFactor;
        int medianIdx = this.tree.branchFactor / 2;
        int medianOffset = this.valueIdx2offset(medianIdx);
        int splitOffset = medianOffset + this.tree.valueSize;
        System.arraycopy(this.data, splitOffset, newNode.data, 4, this.data.length - splitOffset);
        byte[] medianValue = this.getValue(medianIdx);
        this.clearData(medianOffset, this.data.length);
        this.setValueCount(medianIdx);
        newNode.setValueCount(this.tree.branchFactor - medianIdx - 1);
        newNode.dataChanged = true;
        this.notifyNodeSplit(newNode, medianIdx);
        return medianValue;
    }

    public void mergeWithRightSibling(byte[] medianValue, Node rightSibling) throws IOException {
        assert (this.valueCount + rightSibling.getValueCount() + 1 < this.tree.branchFactor) : "Nodes contain too many values to be merged; left: " + this.valueCount + "; right: " + rightSibling.getValueCount();
        this.insertValueNodeIDPair(this.valueCount, medianValue, 0);
        int rightIdx = this.valueCount;
        System.arraycopy(rightSibling.data, 4, this.data, this.nodeIdx2offset(rightIdx), this.valueIdx2offset(rightSibling.valueCount) - 4);
        this.setValueCount(this.valueCount + rightSibling.valueCount);
        rightSibling.clearData(4, this.valueIdx2offset(rightSibling.valueCount));
        rightSibling.setValueCount(0);
        rightSibling.dataChanged = true;
        rightSibling.notifyNodeMerged(this, rightIdx);
    }

    public void rotateLeft(int valueIdx, Node leftChildNode, Node rightChildNode) throws IOException {
        leftChildNode.insertValueNodeIDPair(leftChildNode.getValueCount(), this.getValue(valueIdx), rightChildNode.getChildNodeID(0));
        this.setValue(valueIdx, rightChildNode.removeValueLeft(0));
        this.notifyRotatedLeft(valueIdx, leftChildNode, rightChildNode);
    }

    public void rotateRight(int valueIdx, Node leftChildNode, Node rightChildNode) throws IOException {
        rightChildNode.insertNodeIDValuePair(0, leftChildNode.getChildNodeID(leftChildNode.getValueCount()), this.getValue(valueIdx - 1));
        this.setValue(valueIdx - 1, leftChildNode.removeValueRight(leftChildNode.getValueCount() - 1));
        this.notifyRotatedRight(valueIdx, leftChildNode, rightChildNode);
    }

    public void register(NodeListener listener) {
        this.listeners.add(listener);
    }

    public void deregister(NodeListener listener) {
        this.listeners.removeFirstOccurrence(listener);
    }

    private void notifyValueAdded(int index) {
        this.notifySafeListeners(nl -> nl.valueAdded(this, index));
    }

    private void notifyValueRemoved(int index) {
        this.notifySafeListeners(nl -> nl.valueRemoved(this, index));
    }

    private void notifyRotatedLeft(int index, Node leftChildNode, Node rightChildNode) throws IOException {
        this.notifyListeners(nl -> nl.rotatedLeft(this, index, leftChildNode, rightChildNode));
    }

    private void notifyRotatedRight(int index, Node leftChildNode, Node rightChildNode) throws IOException {
        this.notifyListeners(nl -> nl.rotatedRight(this, index, leftChildNode, rightChildNode));
    }

    private void notifyNodeSplit(Node rightNode, int medianIdx) throws IOException {
        this.notifyListeners(nl -> nl.nodeSplit(this, rightNode, medianIdx));
    }

    private void notifyNodeMerged(Node targetNode, int mergeIdx) throws IOException {
        this.notifyListeners(nl -> nl.nodeMergedWith(this, targetNode, mergeIdx));
    }

    private void notifyListeners(NodeListenerNotifier notifier) throws IOException {
        Iterator<NodeListener> iter = this.listeners.iterator();
        while (iter.hasNext()) {
            boolean deregister = notifier.apply(iter.next());
            if (!deregister) continue;
            iter.remove();
        }
    }

    private void notifySafeListeners(Function<NodeListener, Boolean> notifier) {
        Iterator<NodeListener> iter = this.listeners.iterator();
        while (iter.hasNext()) {
            boolean deregister = notifier.apply(iter.next());
            if (!deregister) continue;
            iter.remove();
        }
    }

    public void read() throws IOException {
        ByteBuffer buf = ByteBuffer.wrap(this.data);
        buf.limit(this.tree.nodeSize);
        int bytesRead = this.tree.nioFile.read(buf, this.tree.nodeID2offset(this.id));
        assert (bytesRead == this.tree.nodeSize) : "Read operation didn't read the entire node (" + bytesRead + " of " + this.tree.nodeSize + " bytes)";
        this.valueCount = ByteArrayUtil.getInt(this.data, 0);
    }

    public void write() throws IOException {
        ByteBuffer buf = ByteBuffer.wrap(this.data);
        buf.limit(this.tree.nodeSize);
        int bytesWritten = this.tree.nioFile.write(buf, this.tree.nodeID2offset(this.id));
        assert (bytesWritten == this.tree.nodeSize) : "Write operation didn't write the entire node (" + bytesWritten + " of " + this.tree.nodeSize + " bytes)";
        this.dataChanged = false;
    }

    private void shiftData(int startOffset, int endOffset, int shift) {
        System.arraycopy(this.data, startOffset, this.data, startOffset + shift, endOffset - startOffset);
    }

    private void clearData(int startOffset, int endOffset) {
        Arrays.fill(this.data, startOffset, endOffset, (byte)0);
    }

    private void setValueCount(int valueCount) {
        this.valueCount = valueCount;
        ByteArrayUtil.putInt(valueCount, this.data, 0);
    }

    private int valueIdx2offset(int id) {
        return 8 + id * this.tree.slotSize;
    }

    private int nodeIdx2offset(int id) {
        return 4 + id * this.tree.slotSize;
    }

    @FunctionalInterface
    private static interface NodeListenerNotifier {
        public boolean apply(NodeListener var1) throws IOException;
    }
}

