package org.eclipse.nebula.widgets.grid;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EventObject;
import java.util.Iterator;

import org.eclipse.nebula.widgets.grid.Grid.GridVisibleRange;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Listener;

/**
 * This support class adds the possibility to get informed when the visual range
 * in the grid is modified. E.g. to implement clever resource management
 * <p>
 * <b>This support is provisional and may change</b>
 * </p>
 */
public class GridVisibleRangeSupport {
	private Collection<VisibleRangeChangedListener> rangeChangeListener;
	private Grid grid;
	private GridVisibleRange oldRange = new GridVisibleRange();

	private Listener paintListener = event -> calculateChange();

	/**
	 * Listener notified when the visible range changes
	 */
	@FunctionalInterface
	public interface VisibleRangeChangedListener {
		/**
		 * Method called when range is changed
		 *
		 * @param event
		 *            the event holding informations about the change
		 */
		public void rangeChanged(RangeChangedEvent event);
	}

	/**
	 * Event informing about the change
	 */
	public static class RangeChangedEvent extends EventObject {
		/**
		 *
		 */
		private static final long serialVersionUID = 1L;

		/**
		 * Rows new in the visible range
		 */
		public GridItem[] addedRows;

		/**
		 * Rows removed from the range
		 */
		public GridItem[] removedRows;

		/**
		 * Columns added to the range
		 */
		public GridColumn[] addedColumns;

		/**
		 * Columns removed from the range
		 */
		public GridColumn[] removedColumns;

		/**
		 * The current visible range
		 */
		public GridVisibleRange visibleRange;

		RangeChangedEvent(Grid grid, GridVisibleRange visibleRange) {
			super(grid);
			this.visibleRange = visibleRange;
		}

	}

	private GridVisibleRangeSupport(Grid grid) {
		this.grid = grid;
		this.grid.setSizeOnEveryItemImageChange(true);
		// FIXME Maybe better to listen to resize, ... ?
		grid.addListener(SWT.Paint, paintListener);
	}

	/**
	 * Add a listener who is informed when the range is changed
	 *
	 * @param listener
	 *            the listener to add
	 */
	public void addRangeChangeListener(VisibleRangeChangedListener listener) {
		if (rangeChangeListener == null) {
			rangeChangeListener = new ArrayList<>();
		}
		rangeChangeListener.add(listener);
	}

	/**
	 * Remove the listener from the ones informed when the range is changed
	 *
	 * @param listener the listener to remove
	 */
	public void removeRangeChangeListener(VisibleRangeChangedListener listener) {
		if (rangeChangeListener != null) {
			rangeChangeListener.remove(listener);
			if (rangeChangeListener.size() == 0) {
				rangeChangeListener = null;
			}
		}
	}

	private void calculateChange() {
		// FIXME Add back
		if (rangeChangeListener == null) {
			return;
		}
		GridVisibleRange range = grid.getVisibleRange();

		ArrayList<GridItem> lOrigItems = new ArrayList<>();
		lOrigItems.addAll(Arrays.asList(oldRange.getItems()));

		ArrayList<GridItem> lNewItems = new ArrayList<>();
		lNewItems.addAll(Arrays.asList(range.getItems()));

		Iterator<GridItem> newItemsIterator = lNewItems.iterator();
		while (newItemsIterator.hasNext()) {
			if (lOrigItems.remove(newItemsIterator.next())) {
				newItemsIterator.remove();
			}
		}

		ArrayList<GridColumn> lOrigColumns = new ArrayList<>();
		lOrigColumns.addAll(Arrays.asList(oldRange.getColumns()));

		ArrayList<GridColumn> lNewColumns = new ArrayList<>();
		lNewColumns.addAll(Arrays.asList(range.getColumns()));

		Iterator<GridColumn> newColumnsIterator = lNewColumns.iterator();
		while (newColumnsIterator.hasNext()) {
			if (lOrigColumns.remove(newColumnsIterator.next())) {
				newColumnsIterator.remove();
			}
		}

		if (lOrigItems.size() != 0 || lNewItems.size() != 0 || lOrigColumns.size() != 0 || lNewColumns.size() != 0) {
			RangeChangedEvent evt = new RangeChangedEvent(grid, range);
			evt.addedRows = new GridItem[lNewItems.size()];
			lNewItems.toArray(evt.addedRows);

			evt.removedRows = new GridItem[lOrigItems.size()];
			lOrigItems.toArray(evt.removedRows);

			evt.addedColumns = new GridColumn[lNewColumns.size()];
			lNewColumns.toArray(evt.addedColumns);

			evt.removedColumns = new GridColumn[lOrigColumns.size()];
			lNewColumns.toArray(evt.removedColumns);
			Iterator<VisibleRangeChangedListener> rangeChangeIterator = rangeChangeListener.iterator();

			while (rangeChangeIterator.hasNext()) {
				rangeChangeIterator.next().rangeChanged(evt);
			}
		}

		oldRange = range;
	}

	/**
	 * Create a range support for the given grid instance
	 *
	 * @param grid
	 *            the grid instance the range support is created for
	 * @return the created range support
	 */
	public static GridVisibleRangeSupport createFor(Grid grid) {
		return new GridVisibleRangeSupport(grid);
	}

}
