/*****************************************************************************
 * Copyright (c) 2014, 2019, 2025 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Benoit Maggi (CEA LIST) benoit.maggi@cea.fr - Initial API and implementation
 *  Ansgar Radermacher (CEA LIST) ansgar.radermacher@cea.fr - bug 541686 (duplicated replationships)
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Issue 15, Issue 18
 *****************************************************************************/
package org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.EcoreUtil.Copier;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmf.runtime.notation.NotationPackage;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.papyrus.infra.core.clipboard.IClipboardAdditionalData;
import org.eclipse.papyrus.infra.core.clipboard.PapyrusClipboard;
import org.eclipse.papyrus.infra.core.resource.ModelSet;
import org.eclipse.papyrus.infra.gmfdiag.common.Activator;
import org.eclipse.papyrus.infra.gmfdiag.common.commands.InsertDiagramCommand;
import org.eclipse.papyrus.infra.gmfdiag.common.model.NotationUtils;
import org.eclipse.papyrus.infra.gmfdiag.common.utils.DiagramUtils;


/**
 * Offer a copy/paste strategy for diagram in model explorer.
 */
public class DiagramPasteStrategy extends AbstractPasteStrategy implements IPasteStrategy {

	/**
	 * key to store diagrams with no owner
	 *
	 * @deprecated, use DIAGRAM_WITH_NO_SEMANTIC_CONTEXT instead
	 */
	@Deprecated
	protected static final String DIAGRAM_WITH_NO_OWNER = "DIAGRAM_WITH_NO_OWNER"; //$NON-NLS-1$

	/** key to store diagrams with no semanticElement */
	protected static final String DIAGRAM_WITH_NO_SEMANTIC_CONTEXT = "DIAGRAM_WITH_NO_SEMANTIC_CONTEXT"; //$NON-NLS-1$

	/** The instance. */
	private static IPasteStrategy instance = new DiagramPasteStrategy();

	/**
	 * Gets the single instance of DiagramPasteStrategy.
	 *
	 * @return single instance of DiagramPasteStrategy
	 */
	public static IPasteStrategy getInstance() {
		return instance;
	}


	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste.IPasteStrategy#getLabel()
	 */
	@Override
	public String getLabel() {
		return "Diagram Strategy"; //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste.IPasteStrategy#getID()
	 */
	@Override
	public String getID() {
		return Activator.ID + ".DiagramStrategy"; //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste.IPasteStrategy#getDescription()
	 */
	@Override
	public String getDescription() {
		return "Copy Diagrams in model explorer"; //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste.IPasteStrategy#getSemanticCommand(org.eclipse.emf.edit.domain.EditingDomain,
	 * org.eclipse.emf.ecore.EObject, org.eclipse.papyrus.infra.core.clipboard.PapyrusClipboard)
	 */
	@Override
	public org.eclipse.emf.common.command.Command getSemanticCommand(final EditingDomain domain, final EObject targetOwner, PapyrusClipboard<Object> papyrusClipboard) {
		CompoundCommand compoundCommand = new CompoundCommand("Copy all Diagrams"); //$NON-NLS-1$

		Map internalClipboardToTargetCopy = papyrusClipboard.getInternalClipboardToTargetCopy();
		Map<Object, ?> additionalDataMap = papyrusClipboard.getAdditionalDataForStrategy(getID());
		if (additionalDataMap != null) {
			Object additionalData = additionalDataMap.get(DIAGRAM_WITH_NO_SEMANTIC_CONTEXT);
			if (additionalData instanceof DiagramClipboardAdditionalData) {
				DiagramClipboardAdditionalData diagramClipboardAdditionnalData = (DiagramClipboardAdditionalData) additionalData;
				Map<Diagram, Diagram> maps = diagramClipboardAdditionnalData.getDuplicatedDiagramsMap(internalClipboardToTargetCopy);
				for (Map.Entry<Diagram, Diagram> currentEntry : maps.entrySet()) {
					final Diagram copiedDiagram = currentEntry.getValue();
					final EObject diagramSemanticContext = currentEntry.getKey().getElement();
					org.eclipse.emf.common.command.Command command = new InsertDiagramCommand((TransactionalEditingDomain) domain, "InsertDiagramCommand", copiedDiagram, diagramSemanticContext, targetOwner); //$NON-NLS-1$
					compoundCommand.append(command);

					// this command allows to update all the views in the duplicated diagram
					org.eclipse.emf.common.command.Command updateCommand = new UpdateDiagramCommand((TransactionalEditingDomain) domain, currentEntry.getKey(), papyrusClipboard);
					compoundCommand.append(updateCommand);
				}
			}

			for (Iterator<Object> iterator = papyrusClipboard.iterator(); iterator.hasNext();) {
				Object object = iterator.next();
				// get target Element
				EObject diagramSemanticContext = papyrusClipboard.getTragetCopyFromInternalClipboardCopy(object);
				if (diagramSemanticContext != null && diagramSemanticContext instanceof EObject) {
					// get affiliate additional data
					additionalData = additionalDataMap.get(object);
					if (additionalData instanceof DiagramClipboardAdditionalData) {
						DiagramClipboardAdditionalData diagramClipboardAdditionnalData = (DiagramClipboardAdditionalData) additionalData;
						Map<Diagram, Diagram> maps = diagramClipboardAdditionnalData.getDuplicatedDiagramsMap(internalClipboardToTargetCopy);
						for (Map.Entry<Diagram, Diagram> currentEntry : maps.entrySet()) {
							final Diagram sourceDiagram = currentEntry.getKey();
							EObject sourceSemanticContext = sourceDiagram.getElement();
							EObject sourceGraphicalContext = DiagramUtils.getOwner(sourceDiagram);

							EObject intermediateSourceSemanticContextCopy = (EObject) papyrusClipboard.getSourceToInternalClipboard().get(sourceSemanticContext);
							EObject intermediateSourceGraphicalContextCopy = (EObject) papyrusClipboard.getSourceToInternalClipboard().get(sourceGraphicalContext);

							EObject targetSemanticContextCopy = (EObject) internalClipboardToTargetCopy.get(intermediateSourceSemanticContextCopy);
							EObject targetGraphicalContextCopy = (EObject) internalClipboardToTargetCopy.get(intermediateSourceGraphicalContextCopy);

							EObject newSemanticContext = targetSemanticContextCopy != null ? targetSemanticContextCopy : sourceSemanticContext;
							EObject newTargetContext = targetGraphicalContextCopy != null ? targetGraphicalContextCopy : sourceGraphicalContext;

							final Diagram copiedDiagram = currentEntry.getValue();
							// the diagram graphicalOwner has already been set by the copy process
							// final EObject diagramGraphicalContext = DiagramUtils.getOwner(copiedDiagram);
							org.eclipse.emf.common.command.Command command = new InsertDiagramCommand((TransactionalEditingDomain) domain, "InsertDiagramCommand", copiedDiagram, newSemanticContext, newTargetContext); //$NON-NLS-1$
							compoundCommand.append(command);

							// this command allows to update all the views in the duplicated diagram
							org.eclipse.emf.common.command.Command updateCommand = new UpdateDiagramCommand((TransactionalEditingDomain) domain, sourceDiagram, papyrusClipboard);
							compoundCommand.append(updateCommand);
						}
					}
				}
			}
		}

		// An empty compound Command can't be executed
		if (compoundCommand.getCommandList().isEmpty()) {
			return null;
		}
		return compoundCommand;
	}

	/**
	 *
	 * This command is used to update all the views in the duplicated diagrams
	 *
	 */
	private class UpdateDiagramCommand extends RecordingCommand {

		/**
		 * the initial diagram
		 */
		private Diagram source;

		private PapyrusClipboard<Object> papyrusClipboard;

		/**
		 *
		 * Constructor.
		 *
		 * @param domain
		 * @param initialDiagram
		 * @param papyrusClipboard
		 */
		public UpdateDiagramCommand(final TransactionalEditingDomain domain, final Diagram initialDiagram, final PapyrusClipboard<Object> papyrusClipboard) {
			super(domain, "UpdateDiagramCommand"); //$NON-NLS-1$
			this.source = initialDiagram;
			this.papyrusClipboard = papyrusClipboard;
		}

		/**
		 * @see org.eclipse.emf.transaction.RecordingCommand#doExecute()
		 *
		 */

		@Override
		protected void doExecute() {
			Map<?, ?> internalClipboardToTargetCopy = papyrusClipboard.getInternalClipboardToTargetCopy();
			Map<?, ?> addition = papyrusClipboard.getAdditionalDataForStrategy(getID());
			DiagramClipboardAdditionalData data = null;
			for (final Object tmp : addition.values()) {
				if (tmp instanceof DiagramClipboardAdditionalData d) {
					data = d;
					break;
				}
			}
			Copier copier = null;
			if (data != null) {
				copier = data.copier;

			}

			final Iterator<EObject> iter = source.eAllContents();
			while (iter.hasNext()) {
				final EObject sourceView = iter.next();
				if (sourceView instanceof View view && !(view instanceof Diagram)) {
					// all the views, excepted diagrams
					final EObject sourceEObject = view.getElement();
					final EObject intermediateSourceCopy = (EObject) papyrusClipboard.getSourceToInternalClipboard().get(sourceEObject);
					final EObject targetSemanticContextCopy = (EObject) internalClipboardToTargetCopy.get(intermediateSourceCopy);
					final EObject finalSemanticElement = targetSemanticContextCopy != null ? targetSemanticContextCopy : sourceEObject;

					final View duplicatedView = (View) copier.get(sourceView);
					if (duplicatedView != null) {
						// set the UML Element
						duplicatedView.setElement(finalSemanticElement);

						// set the other references of the view, assuming all these references are not UML Element, but are Notation elements
						final Iterator<EReference> iterRef = duplicatedView.eClass().getEAllReferences().iterator();
						while (iterRef.hasNext()) {
							final EReference currentRef = iterRef.next();
							if (!currentRef.isContainment() && currentRef.isChangeable() && !currentRef.isDerived() && currentRef != NotationPackage.eINSTANCE.getView_Element()) {
								if (currentRef.isMany()) {
									final Collection<?> oldColl = (Collection<?>) sourceView.eGet(currentRef);
									if (oldColl.size() > 0) {
										final Collection<Object> newColl = new ArrayList<>();
										for (final Object currentObject : oldColl) {
											final EObject duplicatedValue = copier.get(currentObject);
											final Object newValue = duplicatedValue != null ? duplicatedValue : currentObject;
											newColl.add(newValue);
										}
										duplicatedView.eSet(currentRef, newColl);
									}
								} else {
									final EObject sourceValue = (EObject) sourceView.eGet(currentRef);
									final EObject duplicatedRef = copier.get(sourceValue);
									final EObject newRef = duplicatedRef != null ? duplicatedRef : sourceValue;
									duplicatedView.eSet(currentRef, newRef);
								}
							}
						}
					}
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste.IPasteStrategy#dependsOn()
	 */
	@Override
	public IPasteStrategy dependsOn() {
		return DefaultPasteStrategy.getInstance();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.papyrus.infra.gmfdiag.common.strategy.paste.IPasteStrategy#prepare(org.eclipse.papyrus.infra.core.clipboard.PapyrusClipboard)
	 */
	@Override
	public void prepare(PapyrusClipboard<Object> papyrusClipboard, Collection<EObject> selection) {
		Map<Object, IClipboardAdditionalData> mapCopyToClipboardAdditionalData = new HashMap<>();
		Map sourceToInternalClipboard = papyrusClipboard.getSourceToInternalClipboard();
		List<Diagram> extractSelectedWithoutElement = extractDiagramWithoutSemanticContext(selection);
		if (extractSelectedWithoutElement != null && extractSelectedWithoutElement.size() > 0) {
			DiagramClipboardAdditionalData diagramAdditionnalData = new DiagramClipboardAdditionalData(extractSelectedWithoutElement, sourceToInternalClipboard);
			mapCopyToClipboardAdditionalData.put(DIAGRAM_WITH_NO_SEMANTIC_CONTEXT, diagramAdditionnalData);
		}

		// list of already managed diagrams, to avoid to copy diagrams many times, because we can found them by graphical context and by semantic context
		Collection<Diagram> alreadyManagedDiagrams = new ArrayList<>();
		for (Iterator<EObject> iterator = papyrusClipboard.iterateOnSource(); iterator.hasNext();) {
			EObject eObjectSource = iterator.next();
			ResourceSet resourceSet = eObjectSource.eResource().getResourceSet();
			Collection<Diagram> associatedDiagrams = getAssociatedDiagrams(eObjectSource, resourceSet);
			// remove the already managed diagrams
			associatedDiagrams.removeAll(alreadyManagedDiagrams);
			alreadyManagedDiagrams.addAll(associatedDiagrams);

			if (associatedDiagrams != null && associatedDiagrams.size() > 0) {
				DiagramClipboardAdditionalData diagramAdditionnalData = new DiagramClipboardAdditionalData(associatedDiagrams, sourceToInternalClipboard);
				Object copy = papyrusClipboard.getCopyFromSource(eObjectSource);
				mapCopyToClipboardAdditionalData.put(copy, diagramAdditionnalData);
			}

		}
		papyrusClipboard.pushAdditionalData(getID(), mapCopyToClipboardAdditionalData);
	}

	/**
	 * Get the diagrams associated to the elements (as semantic or graphical context)
	 *
	 * @param eobject
	 * @param resourceSet
	 * @return
	 *         the list of diagram referencing the eobject as semantic context (Diagram#element) or as graphical context (Diagram#owner)
	 */
	public static Collection<Diagram> getAssociatedDiagrams(final EObject eobject, final ResourceSet resourceSet) {
		Collection<Diagram> diagrams = new HashSet<>();
		diagrams.addAll(getDiagramsAssociatedBySemanticContext(eobject, resourceSet));
		diagrams.addAll(getDiagramsAssociatedByGraphicalContext(eobject, resourceSet));
		return diagrams;
	}

	/**
	 * Gets the diagrams associated to element by the semantic context
	 *
	 * @param element
	 * @param resourceSet
	 *            can be null, it will then try to retrieve it from the element.
	 * @return the list of diagrams associated with the given element
	 */
	public static List<Diagram> getDiagramsAssociatedBySemanticContext(EObject element, ResourceSet resourceSet) {
		return DiagramUtils.getAssociatedDiagrams(element, resourceSet);
	}

	/**
	 * Gets the diagrams associated to element as graphical context
	 *
	 * @param element
	 * @param resourceSet
	 *            can be null, it will then try to retrieve it from the element.
	 * @return the list of diagrams associated with the given element
	 */
	public static List<Diagram> getDiagramsAssociatedByGraphicalContext(EObject element, ResourceSet resourceSet) {
		if (resourceSet == null) {
			if (element != null && element.eResource() != null) {
				resourceSet = element.eResource().getResourceSet();
			}
		}

		if (resourceSet instanceof ModelSet) {
			Resource notationResource = NotationUtils.getNotationResource((ModelSet) resourceSet);
			return getAssociatedDiagramsByByGraphicalContextFromNotationResource(element, notationResource);
		}

		return DiagramUtils.getAssociatedDiagrams(element, resourceSet);
	}

	/**
	 * Gets the diagrams associated to element as graphical context.
	 *
	 * @param element
	 * @param notationResource
	 *            the notation resource where to look for diagrams
	 * @return the list of diagrams associated with the given element
	 */
	public static List<Diagram> getAssociatedDiagramsByByGraphicalContextFromNotationResource(EObject element, Resource notationResource) {
		if (notationResource != null) {
			LinkedList<Diagram> diagrams = new LinkedList<>();
			for (EObject eObj : notationResource.getContents()) {
				if (eObj instanceof Diagram) {
					Diagram diagram = (Diagram) eObj;
					if (element.equals(org.eclipse.papyrus.infra.gmfdiag.common.utils.DiagramUtils.getOwner(diagram))) {
						diagrams.add(diagram);
					}
				}
			}
			return diagrams;
		}
		return Collections.emptyList();
	}

	/**
	 * Extract Diagram which semanticContext is not in the selection
	 *
	 * @param selection
	 * @return
	 */
	protected List<Diagram> extractDiagramWithoutSemanticContext(Collection<EObject> selection) {
		List<Diagram> diagramWithoutSemanticContenxtOwnerInSelection = new ArrayList<>();
		if (selection != null) {
			for (EObject eObject : selection) {
				if (eObject instanceof Diagram) {
					Diagram diagram = (Diagram) eObject;
					EObject element = diagram.getElement();
					if (!selection.contains(element)) {
						diagramWithoutSemanticContenxtOwnerInSelection.add(diagram);
					}
				}
			}
		}
		return diagramWithoutSemanticContenxtOwnerInSelection;
	}


	protected class DiagramClipboardAdditionalData implements IClipboardAdditionalData {

		/** The diagrams. */
		protected Collection<Diagram> diagrams;

		/**
		 * This map references the source diagram and its copy that we will paste
		 */
		private Map<Diagram, Diagram> sourceVSCopy = new HashMap<>();

		/**
		 * The Copier instance used to Copy the Diagrams
		 */
		private Copier copier;

		/**
		 * @param diagramCopier
		 * @param diagrams
		 */
		public DiagramClipboardAdditionalData(Collection<Diagram> diagrams, Map<? extends EObject, ? extends EObject> alreadyCopied) {
			this.diagrams = duplicateDiagrams(diagrams, alreadyCopied);
			this.sourceVSCopy = createDuplicatedDiagrams(diagrams, alreadyCopied); // we keep a reference to the first duplicated element
		}

		/**
		 * @return
		 * @deprecated use getDuplicatedDiagramsMap instead
		 */
		@Deprecated
		public Collection<Diagram> getDuplicatedDiagrams(Map<? extends EObject, ? extends EObject> alreadyCopied) {
			return duplicateDiagrams(this.diagrams, alreadyCopied);
		}

		/**
		 * @return
		 */
		public Map<Diagram, Diagram> getDuplicatedDiagramsMap(Map<? extends EObject, ? extends EObject> alreadyCopied) {
			return createDuplicatedDiagrams(this.sourceVSCopy.keySet(), alreadyCopied);
		}

		/**
		 * @param diagrams
		 *            to duplicate
		 * @param alreadyCopied
		 * @return duplicated diagrams
		 *
		 * @deprecated //use createDuplicatedDiagrams instead
		 */
		@Deprecated
		protected Collection<Diagram> duplicateDiagrams(Collection<Diagram> diagrams, Map<? extends EObject, ? extends EObject> alreadyCopied) {
			Collection<Diagram> duplicatedDiagrams = new ArrayList<>();
			EcoreUtil.Copier copier = new EcoreUtil.Copier();
			copier.putAll(alreadyCopied);
			for (Diagram diagram : diagrams) {
				copier.copy(diagram);
				// do not copy references (already done in default strategy), avoid duplicates
				EObject copy = copier.get(diagram);
				if (copy instanceof Diagram) {
					duplicatedDiagrams.add((Diagram) copy);
				}
			}
			return duplicatedDiagrams;
		}

		/**
		 * @param diagrams
		 *            to duplicate
		 * @param alreadyCopied
		 * @return duplicated diagrams
		 */
		protected Map<Diagram, Diagram> createDuplicatedDiagrams(Collection<Diagram> diagrams, Map<? extends EObject, ? extends EObject> alreadyCopied) {
			final Map<Diagram, Diagram> sourceVSCopy = new HashMap<>();

			this.copier = new EcoreUtil.Copier();
			copier.putAll(alreadyCopied);
			for (Diagram diagram : diagrams) {
				copier.copy(diagram);
				// do not copy references (already done in default strategy), avoid duplicates
				// probably a bad idea here, because we don't yet do the paste.
				// If we copy the references now, we will attach element to the current models, during the copy, and it is wrong, we probably should do it during the paste itself
				// copier.copyReferences();
				EObject copy = copier.get(diagram);
				if (copy instanceof Diagram tmpDiag) {
					sourceVSCopy.put(diagram, tmpDiag);
				}

			}
			return sourceVSCopy;
		}
	}

}
