/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.messages.Either;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * An inline completion item represents a text snippet that is proposed inline
 * to complete text that is being typed.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class InlineCompletionItem {
  /**
   * The text to replace the range with. Must be set.
   * Is used both for the preview and the accept operation.
   */
  @NonNull
  private Either<String, StringValue> insertText;

  /**
   * A text that is used to decide if this inline completion should be
   * shown. When `falsy`, the {@link InlineCompletionItem#insertText} is
   * used.
   * <p>
   * An inline completion is shown if the text to replace is a prefix of the
   * filter text.
   */
  private String filterText;

  /**
   * The range to replace.
   * Must begin and end on the same line.
   * <p>
   * Prefer replacements over insertions to provide a better experience when
   * the user deletes typed text.
   */
  private Range range;

  /**
   * An optional {@link Command} that is executed *after* inserting this
   * completion.
   */
  private Command command;

  public InlineCompletionItem() {
  }

  public InlineCompletionItem(@NonNull final Either<String, StringValue> insertText) {
    this.insertText = Preconditions.<Either<String, StringValue>>checkNotNull(insertText, "insertText");
  }

  /**
   * The text to replace the range with. Must be set.
   * Is used both for the preview and the accept operation.
   */
  @NonNull
  public Either<String, StringValue> getInsertText() {
    return this.insertText;
  }

  /**
   * The text to replace the range with. Must be set.
   * Is used both for the preview and the accept operation.
   */
  public void setInsertText(@NonNull final Either<String, StringValue> insertText) {
    this.insertText = Preconditions.checkNotNull(insertText, "insertText");
  }

  public void setInsertText(final String insertText) {
    if (insertText == null) {
      Preconditions.checkNotNull(insertText, "insertText");
      this.insertText = null;
      return;
    }
    this.insertText = Either.forLeft(insertText);
  }

  public void setInsertText(final StringValue insertText) {
    if (insertText == null) {
      Preconditions.checkNotNull(insertText, "insertText");
      this.insertText = null;
      return;
    }
    this.insertText = Either.forRight(insertText);
  }

  /**
   * A text that is used to decide if this inline completion should be
   * shown. When `falsy`, the {@link InlineCompletionItem#insertText} is
   * used.
   * <p>
   * An inline completion is shown if the text to replace is a prefix of the
   * filter text.
   */
  public String getFilterText() {
    return this.filterText;
  }

  /**
   * A text that is used to decide if this inline completion should be
   * shown. When `falsy`, the {@link InlineCompletionItem#insertText} is
   * used.
   * <p>
   * An inline completion is shown if the text to replace is a prefix of the
   * filter text.
   */
  public void setFilterText(final String filterText) {
    this.filterText = filterText;
  }

  /**
   * The range to replace.
   * Must begin and end on the same line.
   * <p>
   * Prefer replacements over insertions to provide a better experience when
   * the user deletes typed text.
   */
  public Range getRange() {
    return this.range;
  }

  /**
   * The range to replace.
   * Must begin and end on the same line.
   * <p>
   * Prefer replacements over insertions to provide a better experience when
   * the user deletes typed text.
   */
  public void setRange(final Range range) {
    this.range = range;
  }

  /**
   * An optional {@link Command} that is executed *after* inserting this
   * completion.
   */
  public Command getCommand() {
    return this.command;
  }

  /**
   * An optional {@link Command} that is executed *after* inserting this
   * completion.
   */
  public void setCommand(final Command command) {
    this.command = command;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("insertText", this.insertText);
    b.add("filterText", this.filterText);
    b.add("range", this.range);
    b.add("command", this.command);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    InlineCompletionItem other = (InlineCompletionItem) obj;
    if (this.insertText == null) {
      if (other.insertText != null)
        return false;
    } else if (!this.insertText.equals(other.insertText))
      return false;
    if (this.filterText == null) {
      if (other.filterText != null)
        return false;
    } else if (!this.filterText.equals(other.filterText))
      return false;
    if (this.range == null) {
      if (other.range != null)
        return false;
    } else if (!this.range.equals(other.range))
      return false;
    if (this.command == null) {
      if (other.command != null)
        return false;
    } else if (!this.command.equals(other.command))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.insertText== null) ? 0 : this.insertText.hashCode());
    result = prime * result + ((this.filterText== null) ? 0 : this.filterText.hashCode());
    result = prime * result + ((this.range== null) ? 0 : this.range.hashCode());
    return prime * result + ((this.command== null) ? 0 : this.command.hashCode());
  }
}
