use strict;
use warnings;

package App::Embra::Plugin::GatherDir;
$App::Embra::Plugin::GatherDir::VERSION = '0.001'; # TRIAL
# ABSTRACT: gather all the files in a directory

use Path::Class::Dir;
use List::MoreUtils qw< any >;
use Method::Signatures;

use App::Embra::File;
use Moo;

method mvp_multivalue_args() { qw< exclude_match >; }



has 'from' => (
    is => 'ro',
    required => 1,
    default => sub { '.' },
    coerce => sub { Path::Class::Dir->new( $_[0] )->absolute },
);


has 'include_dotfiles' => (
    is => 'ro',
    default => sub { !1 },
);


has 'exclude_match' => (
    is => 'ro',
    coerce => sub { [ map { qr{$_} } @{ $_[0] } ] },
    default => sub { [] },
);

method gather_files {
    $self->debug( 'looking in '.$self->from );
    $self->from->recurse( callback => func( $file ) {
        return if $file eq $self->from;
        my $skip = $file->basename =~ m/ \A [.] /xms && not $self->include_dotfiles;
        my $exclude = any { $file =~ $_ } @{ $self->exclude_match };
        return $file->PRUNE if $file->is_dir and $skip || $exclude;
        return if $file->is_dir;
        $self->debug( "considering $file" );
        return if $skip or $exclude;
        my $embra_file = App::Embra::File->new( name => $file->stringify );
        $embra_file->name( $file->relative( $self->from )->stringify );
        $self->add_file( $embra_file );
    } );
}

with 'App::Embra::Role::FileGatherer';

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

App::Embra::Plugin::GatherDir - gather all the files in a directory

=head1 VERSION

version 0.001

=head1 DESCRIPTION

This plugin recursively add all files in a directory to the site.

=head1 ATTRIBUTES

=head2 from

The directory to gather files from. Defaults to F<.> (the current directory).

=head2 include_dotfiles

Whether to consider files and directories beginning with C<.> (dot) when gathering files. Defaults to false.

=head2 exclude_match

A regular expression to match files which should not be gathered. May be used multiple times to specify multiple patterns to exclude.

=head1 AUTHOR

Daniel Holz <dgholz@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Daniel Holz.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
