## -*- Mode: CPerl -*-
##
## File: DTA::CAB::Format::Raw::Base.pm
## Author: Bryan Jurish <moocow@cpan.org>
## Description: Datum parser/formatter: raw untokenized text (common)

package DTA::CAB::Format::Raw::Base;
use DTA::CAB::Format;
use DTA::CAB::Datum ':all';
use IO::File;
use Encode qw(encode decode);
use Carp;
use strict;

##==============================================================================
## Globals
##==============================================================================

our @ISA = qw(DTA::CAB::Format);

##==============================================================================
## Constructors etc.
##==============================================================================

## $fmt = CLASS_OR_OBJ->new(%args)
##  + object structure: assumed HASH
##    {
##     ##-- Common
##     #utf8 => $bool,		       ##-- utf8 mode always on
##     #level => $level,                ##-- output level (ignored)
##    }
sub new {
  my $that = shift;
  my $fmt = bless({
		   ##-- common
		   utf8 => 1,
		   #level => 1,

		   ##-- user args
		   @_
		  }, ref($that)||$that);
  return $fmt;
}

##==============================================================================
## Methods: Input
##==============================================================================

# (nothing here)

##==============================================================================
## Methods: Output
##==============================================================================

##--------------------------------------------------------------
## Methods: Output: Generic

## $type = $fmt->mimeType()
##  + default returns text/plain
sub mimeType { return 'text/plain'; }

## $ext = $fmt->defaultExtension()
##  + returns default filename extension for this format
sub defaultExtension { return '.raw'; }

## $str = $fmt->toString(\$str)
## $str = $fmt->toString(\$str,$formatLevel)
##  + select output to byte-string
##  + default implementation wraps $fmt->toFh()

## $fmt_or_undef = $fmt->toFile($filename_or_handle, $formatLevel)
##  + select output to named file $filename.
##  + default implementation wraps $fmt->toFh()

## $fmt_or_undef = $fmt->toFh($fh,$formatLevel)
##  + select output to an open filehandle $fh.
##  + default implementation calls to $fmt->formatString($formatLevel)
sub toFh {
  $_[0]->DTA::CAB::Format::toFh(@_[1..$#_]);
  $_[0]->setLayers();
  return $_[0];
}

##--------------------------------------------------------------
## Methods: Output: API

## $fmt = $fmt->putDocument($doc)
##  + override
sub putDocument {
  my ($fmt,$doc,$bufr) = @_;
  #$bufr = \(my $buf='') if (!defined($bufr));

  my $level = $fmt->{level} // 0;
  my ($s,$w, $txt);
  foreach $s (@{$doc->{body}}) {
    $fmt->{fh}->print(join(' ',
			   map {
			     ($_->{moot} && defined($_->{moot}{word}) ? $_->{moot}{word}
			      : ($_->{dmoot} && defined($_->{dmoot}{tag}) ? $_->{dmoot}{tag}
				 : ($_->{xlit} && defined($_->{xlit}{latin1Text}) ? $_->{xlit}{latin1Text}
				    : $_->{text})));
			   } @{$s->{tokens}}),
		      "\n");
  }
  return $fmt;
}

## $fmt = $fmt->putData($data)
##  + puts raw data (uses forceDocument())
sub putData {
  $_[0]->putDocument($_[0]->forceDocument($_[1]));
}


1; ##-- be happy

__END__

##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl
=pod

##========================================================================
## NAME
=pod

=head1 NAME

DTA::CAB::Format::Raw::Base - Datum parser/formatter: raw untokenized text (common base class)

=cut

##========================================================================
## SYNOPSIS
=pod

=head1 SYNOPSIS

 ##========================================================================
 ## PRELIMINARIES
 
 use DTA::CAB::Format::Raw::Base;
 
 ##========================================================================
 ## Constructors etc.
 
 $fmt = CLASS_OR_OBJ->new(%args);
 
 ##========================================================================
 ## Methods: Output: Generic
 
 $type = $fmt->mimeType();
 $ext = $fmt->defaultExtension();
 
 ##========================================================================
 ## Methods: Output: API
 
 $fmt = $fmt->putDocument($doc);
 $fmt = $fmt->putData($data);
 

=cut

##========================================================================
## DESCRIPTION
=pod

=head1 DESCRIPTION

DTA::CAB::Format::Raw::Base
is an abstract base class for
for untokenized "raw" string I/O.
Input methods are expected to be provided by child subclasses.

This class provides L<DTA::CAB::Format>-compliant output methods
which writes all and only the canonical ("modern", "normalized") surface form
of each token to the output stream.  Individual output tokens are separated by a single space
character (ASCII 0x20, C<" ">) and individual output sentences are separated by a single newline
character (ASCII 0x0A, C<"\n">).


=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::CAB::Format::Raw::Base: Globals
=pod

=head2 Globals

=over 4

=item Variable: @ISA

Inherits from L<DTA::CAB::Format|DTA::CAB::Format>.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::CAB::Format::Raw::Base: Constructors etc.
=pod

=head2 Constructors etc.

=over 4

=item new

 $fmt = CLASS_OR_OBJ->new(%args);

object structure: assumed HASH

    {
     ##-- Common
     #utf8 => $bool,	##-- utf8 mode always on
     #level => $level,  ##-- output level (ignored)
    }

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::CAB::Format::Raw::Base: Methods: Output: Generic
=pod

=head2 Methods: Output: Generic

=over 4

=item mimeType

 $type = $fmt->mimeType();

default returns text/plain

=item defaultExtension

 $ext = $fmt->defaultExtension();

returns default filename extension for this format

=item toFh

thin wrapper for L<DTA::CAB::Format::toFh|DTA::CAB::Format/toFh>-

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::CAB::Format::Raw::Base: Methods: Output: API
=pod

=head2 Methods: Output: API

=over 4

=item putDocument

 $fmt = $fmt->putDocument($doc);

ye olde guttes

=item putData

 $fmt = $fmt->putData($data);

puts raw data (uses forceDocument())

=back

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl


##======================================================================
## Footer
##======================================================================
=pod

=head1 AUTHOR

Bryan Jurish E<lt>jurish@bbaw.deE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2019 by Bryan Jurish

This package is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.20.2 or,
at your option, any later version of Perl 5 you may have available.

=head1 SEE ALSO

L<dta-cab-analyze.perl(1)|dta-cab-analyze.perl>,
L<dta-cab-convert.perl(1)|dta-cab-convert.perl>,
L<dta-cab-http-server.perl(1)|dta-cab-http-server.perl>,
L<dta-cab-http-client.perl(1)|dta-cab-http-client.perl>,
L<dta-cab-xmlrpc-server.perl(1)|dta-cab-xmlrpc-server.perl>,
L<dta-cab-xmlrpc-client.perl(1)|dta-cab-xmlrpc-client.perl>,
L<DTA::CAB::Server(3pm)|DTA::CAB::Server>,
L<DTA::CAB::Client(3pm)|DTA::CAB::Client>,
L<DTA::CAB::Format(3pm)|DTA::CAB::Format>,
L<DTA::CAB(3pm)|DTA::CAB>,
L<perl(1)|perl>,
...



=cut
