#!/usr/bin/perl -w
# SPDX-License-Identifier: CC0-1.0

=head1 NAME

dh_package_notes - Add package metadata to ELF header

=cut

use strict;
use warnings;
use Debian::Debhelper::Dh_Lib
  qw(%dh doit hostarch init isnative sourcepackage);
our $VERSION = '0.6';

=head1 SYNOPSIS

B<dh_package_notes>

=cut

=head1 DESCRIPTION

B<dh_package_notes> is a debhelper program that creates a linker script to
include package metadata in ELF binaries built by packages.

The package metadata specification for ELF binaries can be found at
L<https://systemd.io/COREDUMP_PACKAGE_METADATA/>.

B<dh_package_notes> creates a linker script in a fixed location at
B<debian/.debhelper/notes.ld> with the package type set to B<deb>, the package
name and version set to the source package name and version respectively, and
the B<os> and B<osVersion> fields set to the values of the B<ID> and
B<VERSION_ID> fields found in B</etc/os-release>. Simply add
B<dh-sequence-package-notes> to the Build-Depends or add
B<--with package_notes> to the B<dh $@> call to make this happen.

The package using B<dh_package_notes> also needs to manually set in debian/rules
B<export DEB_LDFLAGS_MAINT_APPEND = -Wl,-dT,debian/.debhelper/notes.ld> as it is
not possible to do so from a debhelper addon.

=cut

init();

isnative( $dh{MAINPACKAGE} );    # Necessary to have $dh{VERSION}
my $source_package = sourcepackage();
my $package_arch   = hostarch();
my %options        = ( 'stdout' => "debian/.debhelper/notes.ld" );
my @cmd            = (
    "/usr/bin/generate-package-notes",
    "--type",    "deb",
    "--os",      "debian",
    "--name",    ${source_package},
    "--architecture", ${package_arch},
    "--version", $dh{VERSION},
    "--debugInfoUrl",  "https://debuginfod.debian.net"
);

if ( not mkdir("debian/.debhelper") ) {
    error("mkdir debian/.debhelper failed: $!");
}

doit( \%options, @cmd );

# FIXME: neither of these work, so appending from each debian/rules file is
# currently the only way to make this work, which is less than ideal
#$ENV{'DEB_LDFLAGS_MAINT_APPEND'} .= ' -Wl,-dT,debian/.debhelper/notes.ld';
#$ENV{'LDFLAGS'}                  .= ' -Wl,-dT,debian/.debhelper/notes.ld';

=head1 SEE ALSO

L<debhelper(7)>

=head1 AUTHOR

Luca Boccassi <bluca@debian.org>

=cut
