/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     dof_admin.c                                                    */
/*                                                                          */
/* description:  implementation of the administration of DOFs               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"

/*--------------------------------------------------------------------------*/
/*      use a bit vector to indicate used/unused dofs			    */
/*      storage needed: one bit per dof                     		    */
/*--------------------------------------------------------------------------*/

#if DOF_FREE_SIZE == 32

DOF_FREE_UNIT dof_free_bit[DOF_FREE_SIZE] =
      { 0x00000001,  0x00000002,  0x00000004,  0x00000008,
	0x00000010,  0x00000020,  0x00000040,  0x00000080, 
	0x00000100,  0x00000200,  0x00000400,  0x00000800, 
	0x00001000,  0x00002000,  0x00004000,  0x00008000, 
	0x00010000,  0x00020000,  0x00040000,  0x00080000, 
	0x00100000,  0x00200000,  0x00400000,  0x00800000, 
	0x01000000,  0x02000000,  0x04000000,  0x08000000, 
	0x10000000,  0x20000000,  0x40000000,  0x80000000 };
DOF_FREE_UNIT dof_free_mod[DOF_FREE_SIZE] =
      { 0xFFFFFFFF,  0xFFFFFFFE,  0xFFFFFFFC,  0xFFFFFFF8,
	0xFFFFFFF0,  0xFFFFFFE0,  0xFFFFFFC0,  0xFFFFFF80, 
	0xFFFFFF00,  0xFFFFFE00,  0xFFFFFC00,  0xFFFFF800, 
	0xFFFFF000,  0xFFFFE000,  0xFFFFC000,  0xFFFF8000, 
	0xFFFF0000,  0xFFFE0000,  0xFFFC0000,  0xFFF80000, 
	0xFFF00000,  0xFFE00000,  0xFFC00000,  0xFF800000, 
	0xFF000000,  0xFE000000,  0xFC000000,  0xF8000000, 
	0xF0000000,  0xE0000000,  0xC0000000,  0x80000000 };

#elif DOF_FREE_SIZE == 64 /* why not? */

const DOF_FREE_UNIT dof_free_bit[DOF_FREE_SIZE] = {
  0x0000000000000001,0x0000000000000002,0x0000000000000004,0x0000000000000008,
  0x0000000000000010,0x0000000000000020,0x0000000000000040,0x0000000000000080, 
  0x0000000000000100,0x0000000000000200,0x0000000000000400,0x0000000000000800, 
  0x0000000000001000,0x0000000000002000,0x0000000000004000,0x0000000000008000, 
  0x0000000000010000,0x0000000000020000,0x0000000000040000,0x0000000000080000, 
  0x0000000000100000,0x0000000000200000,0x0000000000400000,0x0000000000800000, 
  0x0000000001000000,0x0000000002000000,0x0000000004000000,0x0000000008000000, 
  0x0000000010000000,0x0000000020000000,0x0000000040000000,0x0000000080000000,
  0x0000000100000000,0x0000000200000000,0x0000000400000000,0x0000000800000000,
  0x0000001000000000,0x0000002000000000,0x0000004000000000,0x0000008000000000, 
  0x0000010000000000,0x0000020000000000,0x0000040000000000,0x0000080000000000, 
  0x0000100000000000,0x0000200000000000,0x0000400000000000,0x0000800000000000, 
  0x0001000000000000,0x0002000000000000,0x0004000000000000,0x0008000000000000, 
  0x0010000000000000,0x0020000000000000,0x0040000000000000,0x0080000000000000, 
  0x0100000000000000,0x0200000000000000,0x0400000000000000,0x0800000000000000, 
  0x1000000000000000,0x2000000000000000,0x4000000000000000,0x8000000000000000
};
const DOF_FREE_UNIT dof_free_mod[DOF_FREE_SIZE] = {
  0xFFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFFE,0xFFFFFFFFFFFFFFFC,0xFFFFFFFFFFFFFFF8,
  0xFFFFFFFFFFFFFFF0,0xFFFFFFFFFFFFFFE0,0xFFFFFFFFFFFFFFC0,0xFFFFFFFFFFFFFF80, 
  0xFFFFFFFFFFFFFF00,0xFFFFFFFFFFFFFE00,0xFFFFFFFFFFFFFC00,0xFFFFFFFFFFFFF800, 
  0xFFFFFFFFFFFFF000,0xFFFFFFFFFFFFE000,0xFFFFFFFFFFFFC000,0xFFFFFFFFFFFF8000, 
  0xFFFFFFFFFFFF0000,0xFFFFFFFFFFFE0000,0xFFFFFFFFFFFC0000,0xFFFFFFFFFFF80000, 
  0xFFFFFFFFFFF00000,0xFFFFFFFFFFE00000,0xFFFFFFFFFFC00000,0xFFFFFFFFFF800000, 
  0xFFFFFFFFFF000000,0xFFFFFFFFFE000000,0xFFFFFFFFFC000000,0xFFFFFFFFF8000000, 
  0xFFFFFFFFF0000000,0xFFFFFFFFE0000000,0xFFFFFFFFC0000000,0xFFFFFFFF80000000,
  0xFFFFFFFF00000000,0xFFFFFFFE00000000,0xFFFFFFFC00000000,0xFFFFFFF800000000,
  0xFFFFFFF000000000,0xFFFFFFE000000000,0xFFFFFFC000000000,0xFFFFFF8000000000, 
  0xFFFFFF0000000000,0xFFFFFE0000000000,0xFFFFFC0000000000,0xFFFFF80000000000, 
  0xFFFFF00000000000,0xFFFFE00000000000,0xFFFFC00000000000,0xFFFF800000000000, 
  0xFFFF000000000000,0xFFFE000000000000,0xFFFC000000000000,0xFFF8000000000000, 
  0xFFF0000000000000,0xFFE0000000000000,0xFFC0000000000000,0xFF80000000000000, 
  0xFF00000000000000,0xFE00000000000000,0xFC00000000000000,0xF800000000000000, 
  0xF000000000000000,0xE000000000000000,0xC000000000000000,0x8000000000000000
};

#else

DOF_FREE_UNIT dof_free_bit[DOF_FREE_SIZE] =
      { 0x01,  0x02,  0x04,  0x08,
	0x10,  0x20,  0x40,  0x80 };

DOF_FREE_UNIT dof_free_mod[DOF_FREE_SIZE] =
      { 0xFF,  0xFE,  0xFC,  0xF8,
	0xF0,  0xE0,  0xC0,  0x80 };

#endif

/*--------------------------------------------------------------------------*/
/*  SIZE_INCREMENT:                                                         */
/*  default increment for DOF_VECs etc. in enlarge_dof_lists                */
/*--------------------------------------------------------------------------*/

#define SIZE_INCREMENT (DOF_FREE_SIZE * 32)

/*--------------------------------------------------------------------------*/

void free_dof_index(DOF_ADMIN  *admin, int dof)
{
  FUNCNAME("free_dof_index");
#if 0
  int             i, j, col, col2;
#endif
  unsigned int    iunit, ibit;
  DOF_MATRIX      *matrix;
  MATRIX_ROW      *row, *row2;
  DOF_DOWB_MATRIX *dmatrix;
  DOWB_MATRIX_ROW *drow, *drow2;

  DEBUG_TEST_EXIT(admin, "no admin\n");
  DEBUG_TEST_EXIT(admin->used_count > 0, "no dofs in use\n");
  DEBUG_TEST_EXIT((dof >= 0) && (dof < admin->size),
	      "invalid DOF index %d!\n",dof);

  for (matrix = admin->dof_matrix; matrix; matrix = matrix->next)
  {
    if ((row = matrix->matrix_row[dof]))
    {
      do
      {
#if 0
/* This operation is expensive - and leads to wrong results, if fe_space    */
/* and col_fe_space don't match! (DK)                                       */
/* Besides, this sort of cleanup is not done for vectors either.            */

	for (i=0; i<ROW_LENGTH; i++)
	{
	  col = row->col[i];
	  if (ENTRY_USED(col))
	  {
	    if (col != dof)   /* remove symmetric entry if exists */
	    {
	      row2 = matrix->matrix_row[col];
	      while (row2)
	      {
		for (j=0; j<ROW_LENGTH; j++)
		{
		  col2 = row2->col[j];
		  if (col2 == dof) 
		  {
		    row2->col[j] = UNUSED_ENTRY;
		  }
		  else if (col2 == NO_MORE_ENTRIES)
		  {
		    break;
		  }
		}
		row2 = row2->next;
	      }
	    }
	  }
	  else if (col == NO_MORE_ENTRIES)
	  {
	    break;
	  }
	}
#endif
	row2 = row;
	row = row->next;
	free_matrix_row(matrix->row_fe_space, row2);
	
      } while(row);

      matrix->matrix_row[dof] = nil;
    }
  }

  for (dmatrix = admin->dof_dowb_matrix; dmatrix; dmatrix = dmatrix->next)
  {
    if ((drow = dmatrix->matrix_row[dof]))
    {
      do
      {
#if 0
/* This operation is expensive - and leads to wrong results, if row_fe_space*/
/* and col_fe_space don't match! (DK)                                       */
/* Besides, this sort of cleanup is not done for vectors either.            */

	for (i=0; i<ROW_LENGTH; i++)
	{
	  col = drow->col[i];
	  if (ENTRY_USED(col))
	  {
	    if (col != dof)   /* remove symmetric entry if exists */
	    {
	      drow2 = dmatrix->matrix_row[col];
	      while (drow2)
	      {
		for (j=0; j<ROW_LENGTH; j++)
		{
		  col2 = drow2->col[j];
		  if (col2 == dof) 
		  {
		    drow2->col[j] = UNUSED_ENTRY;
		  }
		  else if (col2 == NO_MORE_ENTRIES)
		  {
		    break;
		  }
		}
		drow2 = drow2->next;
	      };
	    }
	  }
	  else if (col == NO_MORE_ENTRIES)
	  {
	    break;
	  }
	}
#endif

	drow2 = drow;
	drow = drow->next;
	free_dowb_matrix_row(dmatrix->row_fe_space, dmatrix->type, drow2);
	
      } while(drow);

      dmatrix->matrix_row[dof] = nil;
    }
  }

  iunit = dof / DOF_FREE_SIZE;
  ibit  = dof % DOF_FREE_SIZE;
  admin->dof_free[iunit] |= dof_free_bit[ibit];
  if (admin->first_hole > iunit) admin->first_hole = iunit;

  admin->used_count--;
  admin->hole_count++;
}

/*--------------------------------------------------------------------------*/

int get_dof_index(DOF_ADMIN *admin)
{
  FUNCNAME("get_dof_index");
  int       dof = 0, i, ibit;

  DEBUG_TEST_EXIT(admin, "no admin\n");

  if (admin->first_hole < admin->dof_free_size) {
    for (ibit = 0; ibit < DOF_FREE_SIZE; ibit++) {
      if (admin->dof_free[admin->first_hole] & dof_free_bit[ibit]) {
	admin->dof_free[admin->first_hole] ^= dof_free_bit[ibit];
	dof = DOF_FREE_SIZE * admin->first_hole + ibit;
	if (admin->dof_free[admin->first_hole] == 0) {
	  for (i = admin->first_hole+1; i < admin->dof_free_size; i++) {
	    if (admin->dof_free[i]) {
	      break;
	    }
	  }
	  admin->first_hole = i;
	}
	break;
      }
    }
    DEBUG_TEST_EXIT(ibit < DOF_FREE_SIZE, "no free bit in first_hole ?\n");
  }
  else {
    enlarge_dof_lists(admin, 0);
    DEBUG_TEST_EXIT(admin->first_hole < admin->dof_free_size,
		"no free entry after enlarge_dof_lists\n");
    DEBUG_TEST_EXIT(admin->dof_free[admin->first_hole] & dof_free_bit[0],
		"no free bit 0\n");
    admin->dof_free[admin->first_hole] ^= dof_free_bit[0];
    dof = DOF_FREE_SIZE * admin->first_hole;
  }

  admin->used_count++;
  if (admin->hole_count > 0) admin->hole_count--;
  admin->size_used = MAX(admin->size_used, dof+1);

  return(dof);
}

/*--------------------------------------------------------------------------*/

void enlarge_dof_lists(DOF_ADMIN *admin, int minsize)
{
  FUNCNAME("enlarge_dof_lists");
  int              old_size, new_size, i, j, new_free_size;
  DOF_INT_VEC      *iv;
  DOF_DOF_VEC      *dv;
  DOF_UCHAR_VEC    *uv;
  DOF_SCHAR_VEC    *sv;
  DOF_REAL_VEC     *rv;
  DOF_REAL_D_VEC   *rdv;
  DOF_PTR_VEC      *pv;
  DOF_MATRIX       *mat;
  DOF_DOWB_MATRIX  *bmat;

  DEBUG_TEST_EXIT(admin, "no admin\n");

  old_size = admin->size;
  if (minsize > 0) {
    if (old_size > minsize) return;
  }

  new_size = MAX(minsize, admin->size + SIZE_INCREMENT);

  new_size += (DOF_FREE_SIZE - (new_size % DOF_FREE_SIZE)) % DOF_FREE_SIZE;
  admin->size = new_size;

  new_free_size = new_size / DOF_FREE_SIZE;
  admin->dof_free = MEM_REALLOC(admin->dof_free, admin->dof_free_size,
				new_free_size, DOF_FREE_UNIT);
  for (i = admin->dof_free_size; i < new_free_size; i++)
    admin->dof_free[i] = DOF_UNIT_ALL_FREE;
  admin->first_hole = admin->dof_free_size;
  admin->dof_free_size = new_free_size;

  /* enlarge all vectors and matrices          */
  /* but int_dof_vecs don't have to be changed */

  for (iv = admin->dof_int_vec; iv; iv = iv->next) {
    if (iv->size < new_size) {
      iv->vec = MEM_REALLOC(iv->vec, iv->size, new_size, int);
      for (i = iv->size; i < new_size; i++) iv->vec[i] = 0;
      iv->size   = new_size;
    }
  }

  for (dv = admin->dof_dof_vec; dv; dv = dv->next) {
    if (dv->size < new_size) {
      dv->vec = MEM_REALLOC(dv->vec, dv->size, new_size, DOF);
      for (i = dv->size; i < new_size; i++) dv->vec[i] = -1;
      dv->size = new_size;
    }
  }

  for (uv = admin->dof_uchar_vec; uv; uv = uv->next) {
    if (uv->size < new_size) {
      uv->vec = MEM_REALLOC(uv->vec, uv->size, new_size, U_CHAR);
      for (i = uv->size; i < new_size; i++) uv->vec[i] = 0;
      uv->size   = new_size;
    }
  }

  for (sv = admin->dof_schar_vec; sv; sv = sv->next) {
    if (sv->size < new_size) {
      sv->vec = MEM_REALLOC(sv->vec, old_size, new_size, S_CHAR);
      for (i = sv->size; i < new_size; i++) sv->vec[i] = 0;
      sv->size   = new_size;
    }
  }

  for (rv = admin->dof_real_vec; rv; rv = rv->next) {
    if (rv->size < new_size) {
      rv->vec = MEM_REALLOC(rv->vec, rv->size, new_size, REAL);
      for (i = rv->size; i < new_size; i++) rv->vec[i] = 0.0;
      rv->size = new_size;
    }
  }

  for (rdv = admin->dof_real_d_vec; rdv; rdv = rdv->next) {
    if (rdv->size < new_size) {
      rdv->vec = MEM_REALLOC(rdv->vec, rdv->size, new_size, REAL_D);
      for (i = rdv->size; i < new_size; i++)
	for (j=0; j<DIM_OF_WORLD; j++)
	  rdv->vec[i][j] = 0.0;
      rdv->size = new_size;
    }
  }

  for (pv = admin->dof_ptr_vec; pv; pv = pv->next) {
    if (pv->size < new_size) {
      pv->vec = MEM_REALLOC(pv->vec, pv->size, new_size, void *);
      for (i = pv->size; i < new_size; i++) pv->vec[i] = nil;
      pv->size = new_size;
    }
  }

  for (mat = admin->dof_matrix; mat; mat = mat->next) {
    if (mat->size < new_size) {
      mat->matrix_row = MEM_REALLOC(mat->matrix_row, mat->size, new_size,
				    MATRIX_ROW *);
      for (i = mat->size; i < new_size; i++)
	mat->matrix_row[i] = nil;
      mat->size = new_size;
    }
  }

  for (bmat = admin->dof_dowb_matrix; bmat; bmat = bmat->next) {
    if (bmat->size < new_size) {
      bmat->matrix_row = MEM_REALLOC(bmat->matrix_row, bmat->size, new_size,
				     DOWB_MATRIX_ROW *);
      for (i = bmat->size; i < new_size; i++)
	bmat->matrix_row[i] = nil;
      bmat->size = new_size;
    }
  }
}


/*--------------------------------------------------------------------------*/
/*  dof_compress: remove holes in dof vectors                               */
/*--------------------------------------------------------------------------*/

typedef struct dof_admin_traverse_data1 {
  int       *new_dof;
  int       *g_n_dof;
  int       *g_n0_dof;
  int       *g_node;
} DOF_ADMIN_TRAVERSE_DATA1;

/*--------------------------------------------------------------------------*/
/*  ATTENTION:                                                              */
/*  new_dof_fct() destroys new_dof !!!!!!!!!!                               */
/*  should be used only at the end of dof_compress()!!!!!                   */
/*--------------------------------------------------------------------------*/

/* CHANGE_DOFS_1 changes old dofs to NEGATIVE new dofs. The index is shifted */
/* by - 2 so that unused element DOFs (-1) are preserved.                    */

#define CHANGE_DOFS_1(el) \
      dof0 = el->dof[n0+i]; \
      dof = dof0 + nd0; \
      if(dof0) \
        for (j = 0; j < nd; j++) \
	  if ((k = dof[j]) >= 0) \
	    /* do it only once! (dofs are visited more than once) */ \
	    dof[j] = - ud->new_dof[k] - 2;

/* CHANGE_DOFS_2 changes NEGATIVE new dofs to POSITIVE. The index shift above*/
/* is undone.                                                                */

#define CHANGE_DOFS_2(el) \
      dof0 = el->dof[n0+i]; \
      dof = dof0 + nd0; \
      if(dof0) \
        for (j = 0; j < nd; j++) \
	  if ((k = dof[j]) < -1) \
  	  /* do it only once! (dofs are visited more than once) */ \
	    dof[j] = - k - 2; 

static void new_dof_fct1(const EL_INFO *el_info, void *data)
{
  DOF_ADMIN_TRAVERSE_DATA1 *ud = (DOF_ADMIN_TRAVERSE_DATA1 *)data;
  EL      *el = el_info->el;
  int     i, j, k, n0, nd, nd0;
  int     dim = el_info->mesh->dim;
  DOF     *dof0, *dof;
  
  if ((nd = ud->g_n_dof[VERTEX]))  {
    nd0 = ud->g_n0_dof[VERTEX];
    n0 = ud->g_node[VERTEX];
    for (i = 0; i < N_VERTICES(dim); i++) {
      CHANGE_DOFS_1(el);
    }
  }

  if(dim > 1 && (nd = ud->g_n_dof[EDGE])) {
    nd0 = ud->g_n0_dof[EDGE];
    n0 = ud->g_node[EDGE];
    for (i = 0; i < N_EDGES(dim); i++) {
      CHANGE_DOFS_1(el);
    }
  }
  
  if(dim == 3 && (nd = ud->g_n_dof[FACE]))  {
    nd0 = ud->g_n0_dof[FACE];
    n0 = ud->g_node[FACE];
    for (i = 0; i < N_FACES_3D; i++) {
      CHANGE_DOFS_1(el);
    }
  }
  
  if ((nd = ud->g_n_dof[CENTER]))  {
    nd0 = ud->g_n0_dof[CENTER];
    n0 = ud->g_node[CENTER];
    i = 0;          /* only one center */
    CHANGE_DOFS_1(el);
  }

  return;
}


static void new_dof_fct2(const EL_INFO *el_info, void *data)
{
  DOF_ADMIN_TRAVERSE_DATA1 *ud = (DOF_ADMIN_TRAVERSE_DATA1 *)data;
  EL      *el = el_info->el;
  int     i, j, k, n0, nd, nd0;
  int     dim = el_info->mesh->dim;
  DOF     *dof0, *dof;
  
  if ((nd = ud->g_n_dof[VERTEX]))  {
    nd0 = ud->g_n0_dof[VERTEX];
    n0 = ud->g_node[VERTEX];
    for (i = 0; i < N_VERTICES(dim); i++) {
      CHANGE_DOFS_2(el);
    }
  }

  if(dim > 1 && (nd = ud->g_n_dof[EDGE])) {
    nd0 = ud->g_n0_dof[EDGE];
    n0 = ud->g_node[EDGE];
    for (i = 0; i < N_EDGES(dim); i++) 
      {
	CHANGE_DOFS_2(el);
      }
  }
  
  if(dim == 3 && (nd = ud->g_n_dof[FACE]))  {
    nd0 = ud->g_n0_dof[FACE];
    n0 = ud->g_node[FACE];
    for (i = 0; i < N_FACES_3D; i++) {
      CHANGE_DOFS_2(el);
    }
  }
  
  if ((nd = ud->g_n_dof[CENTER]))  {
    nd0 = ud->g_n0_dof[CENTER];
    n0 = ud->g_node[CENTER];
    i = 0;          /* only one center */
    CHANGE_DOFS_2(el);
  }

  return;
}

#undef CHANGE_DOFS_1
#undef CHANGE_DOFS_2

/*--------------------------------------------------------------------------*/

extern void dof_compress(MESH *mesh)
{
  FUNCNAME("dof_compress");
  DOF_ADMIN_TRAVERSE_DATA1 td[1] = {{0}};
  DOF_ADMIN       *compress_admin, *search_admin;
  int             i,j,k,n,size,first,last=0,col, iadmin, jadmin;
  FLAGS           fill_flag;
  DOF_INT_VEC     *div;
  DOF_DOF_VEC     *ddv;
  DOF_UCHAR_VEC   *duv;
  DOF_SCHAR_VEC   *dsv;
  DOF_REAL_D_VEC  *drdv;
  DOF_REAL_VEC    *drv;
  DOF_PTR_VEC     *dpv;
  DOF_MATRIX      *dm;
  DOF_DOWB_MATRIX *dbm;
  MATRIX_ROW      *row, *row_next;
  DOWB_MATRIX_ROW *drow, *drow_next;

  TEST_EXIT(mesh, "No mesh given!\n");

  td->g_node = mesh->node;

  for (iadmin = 0; iadmin < mesh->n_dof_admin; iadmin++)
  {
    compress_admin = mesh->dof_admin[iadmin];

    DEBUG_TEST_EXIT(compress_admin,
		"no dof_admin[%d] in mesh\n", iadmin);

    if ((size = compress_admin->size) < 1) continue;
    if (compress_admin->used_count < 1)    continue;
    if (compress_admin->hole_count < 1)    continue;
    
    td->g_n_dof = compress_admin->n_dof;
    td->g_n0_dof = compress_admin->n0_dof;

    td->new_dof = MEM_ALLOC(size, int);
    for (i=0; i<size; i++) {                       /* mark everything unused */
      td->new_dof[i] = -1;
    }

    FOR_ALL_DOFS(compress_admin, td->new_dof[dof] = 1);

    n = 0;
    for (i = 0; i < size; i++) {             /* create a MONOTONE compress */
      if (td->new_dof[i] == 1) {
	td->new_dof[i] = n++;
	last = i;
      }
    }

    DEBUG_TEST_EXIT(n == compress_admin->used_count,
		"count %d != used_count %d\n",
		n, compress_admin->used_count);

    {
      int n1 = n / DOF_FREE_SIZE;

      for (i = 0; i < n1; i++)
	compress_admin->dof_free[i] = 0;
      if (n1 < compress_admin->dof_free_size)
	compress_admin->dof_free[n1] = dof_free_mod[n % DOF_FREE_SIZE];
      for (i = n1+1; i < compress_admin->dof_free_size; i++)
	compress_admin->dof_free[i] = DOF_UNIT_ALL_FREE;
      compress_admin->first_hole = n1;
    }

    compress_admin->hole_count = 0;
    compress_admin->size_used  = n;

    first = 0;
    for (i=0; i<size; i++) {
      if ((td->new_dof[i] < i) && (td->new_dof[i] >= 0)) {
	first = i;
	break;
      }
    }
    if (last >= first) {
      last++;   /* for (i = first; i < last; i++) ... */

      /* compress all vectors associated to admin */

      for (div = compress_admin->dof_int_vec; div; div = div->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    div->vec[k] = div->vec[i];
	}
      }
  
      for (ddv = compress_admin->dof_dof_vec; ddv; ddv = ddv->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    ddv->vec[k] = ddv->vec[i];
	}
	for (i=0; i<n; i++) {
	  if ((k = td->new_dof[ddv->vec[i]]) >= 0)
	    ddv->vec[i] = k;
	}
      }
  
      for (ddv = compress_admin->int_dof_vec; ddv; ddv = ddv->next) {
	for (i=0; i < ddv->size; i++) {
	  if (ddv->vec[i] >= 0) 
	    if ((k = td->new_dof[ddv->vec[i]]) >= 0)
	      ddv->vec[i] = k;
	}
      }
  
      for (duv = compress_admin->dof_uchar_vec; duv; duv = duv->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    duv->vec[k] = duv->vec[i];
	}
      }
  
      for (dsv = compress_admin->dof_schar_vec; dsv; dsv = dsv->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    dsv->vec[k] = dsv->vec[i];
	}
      }
  
      for (drv = compress_admin->dof_real_vec; drv; drv = drv->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    drv->vec[k] = drv->vec[i];
	}
      }
  
      for (drdv = compress_admin->dof_real_d_vec; drdv; drdv = drdv->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    for (j=0; j<DIM_OF_WORLD; j++)
	      drdv->vec[k][j] = drdv->vec[i][j];
	}
      }

      for (dpv = compress_admin->dof_ptr_vec; dpv; dpv = dpv->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0)
	    dpv->vec[k] = dpv->vec[i];
	}
      }

      /* Row corrections */  
      for (dm = compress_admin->dof_matrix; dm; dm = dm->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0) {
	    /* free dm->matrix_row[k]; */
	    for (row = dm->matrix_row[k]; row; row = row_next) {
	      row_next = row->next;
	      free_matrix_row(dm->row_fe_space, row);
	    }
	    dm->matrix_row[k] = dm->matrix_row[i];
	    dm->matrix_row[i] = nil;
	  }
	}
      }

      for (dbm = compress_admin->dof_dowb_matrix; dbm; dbm = dbm->next) {
	for (i=first; i<last; i++) {
	  if ((k = td->new_dof[i]) >= 0) {
	    /* free dm->matrix_row[k]; */
	    for (drow = dbm->matrix_row[k]; drow; drow = drow_next) {
	      drow_next = drow->next;
	      free_dowb_matrix_row(dbm->row_fe_space, dbm->type, drow);
	    }
	    dbm->matrix_row[k] = dbm->matrix_row[i];
	    dbm->matrix_row[i] = nil;
	  }
	}
      }

      /* Column corrections, we search all matrices for ones with */
      /* col_fe_space->admin == compress_admin.                   */

      for (jadmin = 0; jadmin < mesh->n_dof_admin; jadmin++) {
	search_admin = mesh->dof_admin[jadmin];
	
	DEBUG_TEST_EXIT(search_admin,
			"no dof_admin[%d] in mesh\n", jadmin);
	
	for (dm = search_admin->dof_matrix; dm; dm = dm->next)
	  if(dm->col_fe_space && dm->col_fe_space->admin == compress_admin)
	    for (i=0; i<n; i++) { /* change columns */
	      for (row = dm->matrix_row[i]; row; row = row->next) {
		for (j=0; j<ROW_LENGTH; j++) {
		  col = row->col[j];
		  if (ENTRY_USED(col)) row->col[j] = td->new_dof[col];
		}
	      }
	    }
	
	for (dbm = search_admin->dof_dowb_matrix; dbm; dbm = dbm->next)
	  if(dbm->col_fe_space && dbm->col_fe_space->admin == compress_admin)
	    for (i=0; i<n; i++) { /* change columns */
	      for (drow = dbm->matrix_row[i]; drow; drow = drow->next) {
		for (j=0; j<ROW_LENGTH; j++) {
		  col = drow->col[j];
		  if (ENTRY_USED(col)) drow->col[j] = td->new_dof[col];
		}
	      }
	    }
      }
      
      /* now, change dofs in MESH's nodes */
      /* new_dof_fct destroys new_dof[] !!! */

      fill_flag = CALL_EVERY_EL_PREORDER | FILL_NOTHING;

      mesh_traverse(mesh, -1, fill_flag, new_dof_fct1, td);
      mesh_traverse(mesh, -1, fill_flag, new_dof_fct2, td);

    }

    MEM_FREE(td->new_dof, size, int);
    td->new_dof = nil;
  }
  return;
}


/*--------------------------------------------------------------------------*/
/*  matrix administration                            			    */
/*--------------------------------------------------------------------------*/
/*  AI_add_dof_matrix_to_admin():       enter new matrix to DOF_ADMIN list  */
/*                                   (re)alloc matrix_row, el_* if necessary*/
/*  AI_add_dof_dowb_matrix_to_admin():  dito for block matrices             */
/*                                   (re)alloc matrix_row, el_* if necessary*/
/*  AI_add_dof_int_vec_to_admin():      enter new vec to DOF_ADMIN list     */
/*  AI_add_int_dof_vec_to_admin():      enter new vec to DOF_ADMIN list     */
/*  AI_add_dof_dof_vec_to_admin():      enter new vec to DOF_ADMIN list     */
/*  AI_add_dof_uchar_vec_to_admin():    enter new vec to DOF_ADMIN list     */
/*  AI_add_dof_schar_vec_to_admin():    enter new vec to DOF_ADMIN list     */
/*  AI_add_dof_real_vec_to_admin():     enter new vec to DOF_ADMIN list     */
/*  AI_add_dof_real_d_vec_to_admin():   enter new vec to DOF_ADMIN list     */
/*  AI_add_dof_ptr_vec_to_admin():      enter new vec to DOF_ADMIN list     */
/*                                     (re)alloc vector if necessary        */
/*                                                                          */
/*  update_dof_matrix():             add/substract el_mat to/from matrix    */
/*--------------------------------------------------------------------------*/

#define CHECK_IF_PRESENT(admin, TYPE, list, vec) \
  {TYPE *v; for (v=admin->list; v; v = v->next) \
      if (v==vec) {ERROR("dof_vec %s already associated to admin %s\n",\
			 NAME(vec), NAME(admin)); return;} }


#define DEFINE_ADD_DOF_OBJ_TO_ADMIN(TYPE, list, enlarge)	\
void AI_add_##list##_to_admin(TYPE *obj, DOF_ADMIN *admin)	\
{								\
  FUNCNAME("AI_add_"#list"_to_admin");				\
								\
  if (!obj) {							\
    MSG("no obj\n");						\
    return;							\
  }								\
								\
  CHECK_IF_PRESENT(admin, TYPE, list, obj);			\
								\
  if (obj->size < admin->size) {				\
    enlarge;							\
    obj->size = admin->size;					\
  }								\
								\
  obj->next = admin->list;					\
  admin->list = obj;						\
  /* obj->admin = admin; */					\
}

#define DEFINE_ADD_DOF_VEC_TO_ADMIN(TYPE, type, list)			\
 DEFINE_ADD_DOF_OBJ_TO_ADMIN(TYPE, list,				\
    obj->vec = MEM_REALLOC(obj->vec, obj->size, admin->size, type))

#define DEFINE_ADD_DOF_MAT_TO_ADMIN(TYPE, type, list)		\
 DEFINE_ADD_DOF_OBJ_TO_ADMIN(TYPE, list, {			\
   int i;							\
								\
   obj->matrix_row = MEM_REALLOC(obj->matrix_row, obj->size,	\
				    admin->size, type *);	\
   for (i = obj->size; i < admin->size; i++) {			\
     obj->matrix_row[i] = nil;					\
   }								\
 })

#define DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(TYPE, list)			\
void AI_remove_##list##_from_admin(TYPE *obj)				\
{									\
  FUNCNAME("AI_remove_"#list"_from_admin");				\
  DOF_ADMIN    *admin;							\
  TYPE         *obj_last;						\
									\
  if (obj->fe_space && (admin = (DOF_ADMIN *)obj->fe_space->admin))	\
  {									\
    if (admin->list == obj)						\
      admin->list = obj->next;						\
    else								\
    {									\
      obj_last = admin->list;						\
      while (obj_last &&  obj_last->next != obj)			\
	obj_last = obj_last->next;					\
									\
      if (!obj_last)							\
      {									\
	ERROR(""#list" %s not in list of dof admin %s found\n",		\
	      NAME(obj), NAME(admin));					\
      }									\
      else								\
	obj_last->next = obj->next;					\
    }									\
  }									\
  return;								\
}

#define DEFINE_REMOVE_DOF_MAT_FROM_ADMIN(TYPE, list)			\
void AI_remove_##list##_from_admin(TYPE *obj)				\
{									\
  FUNCNAME("AI_remove_"#list"_from_admin");				\
  DOF_ADMIN    *admin;							\
  TYPE         *obj_last;						\
									\
  if (obj->row_fe_space &&                                              \
      (admin = (DOF_ADMIN *)obj->row_fe_space->admin))	                \
  {									\
    if (admin->list == obj)						\
      admin->list = obj->next;						\
    else								\
    {									\
      obj_last = admin->list;						\
      while (obj_last &&  obj_last->next != obj)			\
	obj_last = obj_last->next;					\
									\
      if (!obj_last)							\
      {									\
	ERROR(""#list" %s not in list of dof admin %s found\n",		\
	      NAME(obj), NAME(admin));					\
      }									\
      else								\
	obj_last->next = obj->next;					\
    }									\
  }									\
  return;								\
}

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_INT_VEC, int, dof_int_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_INT_VEC, dof_int_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_DOF_VEC, DOF, dof_dof_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_DOF_VEC, dof_dof_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_DOF_VEC, DOF, int_dof_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_DOF_VEC, int_dof_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_UCHAR_VEC, U_CHAR, dof_uchar_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_UCHAR_VEC, dof_uchar_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_SCHAR_VEC, S_CHAR, dof_schar_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_SCHAR_VEC, dof_schar_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_REAL_VEC, REAL, dof_real_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_REAL_VEC, dof_real_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_REAL_D_VEC, REAL_D, dof_real_d_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_REAL_D_VEC, dof_real_d_vec)

DEFINE_ADD_DOF_VEC_TO_ADMIN(DOF_PTR_VEC, void *, dof_ptr_vec)
DEFINE_REMOVE_DOF_VEC_FROM_ADMIN(DOF_PTR_VEC, dof_ptr_vec)

DEFINE_ADD_DOF_MAT_TO_ADMIN(DOF_MATRIX, MATRIX_ROW, dof_matrix)
DEFINE_REMOVE_DOF_MAT_FROM_ADMIN(DOF_MATRIX, dof_matrix)

DEFINE_ADD_DOF_MAT_TO_ADMIN(DOF_DOWB_MATRIX, DOWB_MATRIX_ROW, dof_dowb_matrix)
DEFINE_REMOVE_DOF_MAT_FROM_ADMIN(DOF_DOWB_MATRIX, dof_dowb_matrix)

#undef CHECK_IF_PRESENT
#undef DEFINE_ADD_DOF_OBJ_TO_ADMIN
#undef DEFINE_ADD_DOF_VEC_TO_ADMIN
#undef DEFINE_ADD_DOF_MAT_TO_ADMIN
#undef DEFINE_REMOVE_DOF_OBJ_FROM_ADMIN

/*--------------------------------------------------------------------------*/

void update_dof_matrix(DOF_MATRIX *matrix, REAL sign, int dim, 
		       const REAL **el_mat, const DOF  *el_dof, 
		       const S_CHAR *el_bound)
{
  FUNCNAME("update_dof_matrix");
  DOF        i, j, k, irow, jcol;
  int        free_col=0;
  MATRIX_ROW *row, *free_row;

  TEST_EXIT(matrix, "no matrix\n");

  for (i = 0; i < dim; i++) 
  {
    irow = el_dof[i];

    if (matrix->matrix_row[irow] == nil)
    {
      row = matrix->matrix_row[irow] = get_matrix_row(matrix->row_fe_space);
      row->col[0] = irow;           /* first entry is diagonal element */

      if (el_bound && el_bound[i] >= DIRICHLET)
      {
	row->entry[0] = 1.0;
	continue;
      }
      else
	row->entry[0] = 0.0;
    }
    else if (el_bound && el_bound[i] >= DIRICHLET)
      continue;

    for (j = 0; j < dim; j++)
    {
      jcol = el_dof[j];
      row = matrix->matrix_row[irow];
      free_row = nil;
      do 
      {
	for (k=0; k<ROW_LENGTH; k++)
	{
	  if (row->col[k] == jcol)
	  {
	    row->entry[k] += sign*el_mat[i][j];
	    break;
	  }
	  if (ENTRY_NOT_USED(row->col[k])) 
	  {
	    free_col = k;
	    free_row = row;
	    if (row->col[k] == NO_MORE_ENTRIES) 
	    {
	      k = ROW_LENGTH;
	      break;
	    }
	  }
	}
	if (k < ROW_LENGTH) break;               /* done? */
	if (row->next || free_row)
	{
	  row = row->next;
	}
	else 
	{
	  free_row = row->next = get_matrix_row(matrix->row_fe_space);
	  free_col = 0;
	  row = nil;
	}
      } while (row);
      
      if (k >= ROW_LENGTH)                       /* not done? */
      {                     
	DEBUG_TEST_EXIT(free_row, "no free_row\n");
	DEBUG_TEST_EXIT(sign > 0.0,
		    "new entry (%d,%d) in matrix %s with sign=%lf\n",
		    irow, jcol, matrix->name, sign);
	free_row->col[free_col]   = jcol;
	free_row->entry[free_col] = sign * el_mat[i][j];
      }
    }
  }
  return;
}

void update_dof_dowb_matrix(DOF_DOWB_MATRIX *matrix, REAL sign, int dim, 
			    const void **el_mat, const DOF  *el_dof, 
			    const S_CHAR *el_bound)
{
  FUNCNAME("update_dof_dobw_matrix");
  DOF             i, j, k, irow, jcol;
  int             free_col=0;
  const REAL_DD   **el_mat_r = (const REAL_DD **)el_mat;
  const REAL_DDS  **el_mat_s = (const REAL_DDS **)el_mat;
  const REAL_D    **el_mat_d = (const REAL_D **)el_mat;
  DOWB_MATRIX_ROW *row, *free_row;

  TEST_EXIT(matrix, "no matrix\n");

  for (i = 0; i < dim; i++) 
  {
    irow = el_dof[i];

    if (matrix->matrix_row[irow] == nil)
    {
      row = matrix->matrix_row[irow] = 
	get_dowb_matrix_row(matrix->row_fe_space,
			    matrix->type);
      row->col[0] = irow;           /* first entry is diagonal element */

      if (el_bound && el_bound[i] >= DIRICHLET)
      {
	switch(matrix->type) {
	case dowbm_full: MSET_DOW(1.0, row->entry.full[0]); break;
	case dowbm_symm: SMSET_DOW(1.0, &row->entry.symm[0]); break;
	case dowbm_diag: DMSET_DOW(1.0, row->entry.diag[0]); break;
	}
	continue;
      }
      else {
	switch(matrix->type) {
	case dowbm_full: MSET_DOW(0.0, row->entry.full[0]); break;
	case dowbm_symm: SMSET_DOW(0.0, &row->entry.symm[0]); break;
	case dowbm_diag: DMSET_DOW(0.0, row->entry.diag[0]); break;
	}
      }
    }
    else if (el_bound && el_bound[i] >= DIRICHLET)
      continue;

    for (j = 0; j < dim; j++)
    {
      jcol = el_dof[j];
      row = matrix->matrix_row[irow];
      free_row = nil;
      do 
      {
	for (k=0; k<ROW_LENGTH; k++)
	{
	  if (row->col[k] == jcol)
	  {
	    switch(matrix->type) {
	    case dowbm_full:
	      MAXPY_DOW(sign, (REAL_D *)el_mat_r[i][j], row->entry.full[k]);
	      break;
	    case dowbm_symm:
	      SMAXPY_DOW(sign, &el_mat_s[i][j], &row->entry.symm[k]);
	      break;
	    case dowbm_diag:
	      DMAXPY_DOW(sign, el_mat_d[i][j], row->entry.diag[k]);
	      break;
	    }
	    break;
	  }
	  if (ENTRY_NOT_USED(row->col[k])) 
	  {
	    free_col = k;
	    free_row = row;
	    if (row->col[k] == NO_MORE_ENTRIES) 
	    {
	      k = ROW_LENGTH;
	      break;
	    }
	  }
	}
	if (k < ROW_LENGTH) break;               /* done? */
	if (row->next || free_row)
	{
	  row = row->next;
	}
	else 
	{
	  free_row = row->next = 
	    get_dowb_matrix_row(matrix->row_fe_space, matrix->type);
	  free_col = 0;
	  row = nil;
	}
      } while (row);
      
      if (k >= ROW_LENGTH)                       /* not done? */
      {                     
	DEBUG_TEST_EXIT(free_row, "no free_row\n");
	DEBUG_TEST_EXIT(sign > 0.0,
		    "new entry (%d,%d) in matrix %s with sign=%lf\n",
		    irow, jcol, matrix->name, sign);
	free_row->col[free_col]   = jcol;
	switch (matrix->type) {
	case dowbm_full:
	  MAXEY_DOW(sign, (REAL_D *)el_mat_r[i][j], free_row->entry.full[free_col]);
	  break;
	case dowbm_symm:
	  SMAXEY_DOW(sign, &el_mat_s[i][j], &free_row->entry.symm[free_col]);
	  break;
	case dowbm_diag:
	  DMAXEY_DOW(sign, el_mat_d[i][j], free_row->entry.diag[free_col]);
	  break;
	}
      }
    }
  }
  return;
}

void update_dof_real_vec(DOF_REAL_VEC *drv, REAL sign, int el_dim, 
			 const REAL *el_vec, const DOF  *el_dof, 
			 const S_CHAR *el_bound)
{
  FUNCNAME("update_dof_real_vec");
  REAL   *vec;
  int    i;

  TEST_EXIT(drv, "no dof_real_vec\n");
  vec = drv->vec;

  if (el_bound)
  {
    for (i = 0; i < el_dim; i++)
      if (el_bound[i] < DIRICHLET)
	vec[el_dof[i]] += sign*el_vec[i];
  }
  else
  {
    for (i = 0; i < el_dim; i++)
      vec[el_dof[i]] += sign*el_vec[i];

  }
  return;
}

void update_dof_real_d_vec(DOF_REAL_D_VEC *drdv, REAL sign, int el_dim, 
			   const REAL_D *el_vec, const DOF  *el_dof, 
			   const S_CHAR *el_bound)
{
  FUNCNAME("update_dof_real_d_vec");
  REAL_D *vec;
  int    i, n;

  TEST_EXIT(drdv, "no dof_real_vec\n");
  vec = drdv->vec;

  if (el_bound)
  {
    for (i = 0; i < el_dim; i++)
      if (el_bound[i] < DIRICHLET)
	for (n = 0; n < DIM_OF_WORLD; n++)
	  vec[el_dof[i]][n] += sign*el_vec[i][n];
  }
  else
  {
    for (i = 0; i < el_dim; i++)
      for (n = 0; n < DIM_OF_WORLD; n++)
	vec[el_dof[i]][n] += sign*el_vec[i][n];
  }
  return;
}



/*--------------------------------------------------------------------------*/
/* some BLAS level 1 routines:                                              */
/* ---------------------------                                              */
/*   dof_asum:  asum = ||X||_l1                                             */
/*   dof_nrm2:  nrm2 = ||X||_l2                                             */
/*   dof_set:   X = alpha                                                   */
/*   dof_scal:  X = alpha * X                                               */
/*   dof_copy:  Y = X                                                       */
/*   dof_dot:   dot = X . Y                                                 */
/*   dof_axpy:  Y = Y + alpha * X                                           */
/*                                                                          */
/* some BLAS level 2 routines:                                              */
/* ---------------------------                                              */
/*   dof_gemv:  y = alpha*A*x + beta*y   or   y = alpha*A'*x + beta*y       */
/*                                                                          */
/* some non-BLAS level 2 routines:                                          */
/* ---------------------------                                              */
/*   dof_mv:    y = A*x   or   y = A'*x                                     */
/*   dof_min:   min of x                                                    */
/*   dof_max:   max of x                                                    */
/*--------------------------------------------------------------------------*/

REAL dof_nrm2(const DOF_REAL_VEC *x)
{
  FUNCNAME("dof_nrm2");
  REAL nrm;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
	    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size,
	    admin->size_used);

  nrm = 0.0;
  FOR_ALL_DOFS(admin, nrm += x->vec[dof] * x->vec[dof] );

  return(sqrt(nrm));
}


REAL dof_asum(const DOF_REAL_VEC *x)
{
  FUNCNAME("dof_asum");
  REAL    nrm;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
	    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size, 
	    admin->size_used);

  nrm = 0.0;
  FOR_ALL_DOFS(admin, nrm += ABS(x->vec[dof]) );

  return(nrm);
}


void dof_set(REAL alpha, DOF_REAL_VEC *x)
{
  FUNCNAME("dof_set");
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
	    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size,
	    admin->size_used);

  FOR_ALL_DOFS(admin, x->vec[dof] = alpha );
}


void dof_scal(REAL alpha, DOF_REAL_VEC *x)
{
  FUNCNAME("dof_scal");
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
	    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size, 
	    admin->size_used);

  FOR_ALL_DOFS(admin, x->vec[dof] *= alpha );
}


REAL dof_dot(const DOF_REAL_VEC *x, const DOF_REAL_VEC *y)
{
  FUNCNAME("dof_dot");
  REAL      dot;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && y, "pointer is nil: %p, %p\n", x,y);
  TEST_EXIT(x->fe_space && y->fe_space,
	    "fe_space is nil: %p, %p\n", x->fe_space,y->fe_space);
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
	    "no admin or different admins: %p, %p\n",
	    x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size, 
	    admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
	    "y->size = %d too small: admin->size_used = %d\n", y->size, 
	    admin->size_used);

  dot = 0.0;
  FOR_ALL_DOFS(admin, dot += x->vec[dof] * y->vec[dof] );

  return(dot);
}


void dof_copy(const DOF_REAL_VEC *x, DOF_REAL_VEC *y)
{
  FUNCNAME("dof_copy");
  REAL      *xvec, *yvec;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && y, "pointer is nil: %p, %p\n", x,y);
  TEST_EXIT(x->fe_space && y->fe_space,
	    "fe_space is nil: %p, %p\n", x->fe_space,y->fe_space);
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
	    "no admin or different admins: %p, %p\n",
	    x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size, 
	    admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
	    "y->size = %d too small: admin->size_used = %d\n", y->size, 
	    admin->size_used);
  xvec = x->vec;
  yvec = y->vec;

  FOR_ALL_DOFS(admin, yvec[dof] = xvec[dof] );
}


void dof_axpy(REAL alpha, const DOF_REAL_VEC *x, DOF_REAL_VEC *y)
{
  FUNCNAME("dof_axpy");
  REAL      *xvec, *yvec;
  const DOF_ADMIN *admin;

  TEST_EXIT(x && y, "pointer is nil: %p, %p\n", x,y);
  TEST_EXIT(x->fe_space && y->fe_space,
	    "fe_space is nil: %p, %p\n", x->fe_space,y->fe_space);
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
	    "no admin or different admins: %p, %p\n",
	    x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size = %d\n",
	    x->size, admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
	    "y->size = %d too small: admin->size = %d\n",
	    y->size, admin->size_used);

  xvec = x->vec;
  yvec = y->vec;

  FOR_ALL_DOFS(admin, yvec[dof] += alpha * xvec[dof] );
}

void dof_xpay(REAL alpha, const DOF_REAL_VEC *x, DOF_REAL_VEC *y)
{
  FUNCNAME("dof_axpy");
  REAL      *xvec, *yvec;
  const DOF_ADMIN *admin;

  TEST_EXIT(x && y, "pointer is nil: %p, %p\n", x,y);
  TEST_EXIT(x->fe_space && y->fe_space,
	    "fe_space is nil: %p, %p\n", x->fe_space,y->fe_space);
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
	    "no admin or different admins: %p, %p\n",
	    x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n",
	    x->size, admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
	    "y->size = %d too small: admin->size_used = %d\n", 
	    y->size, admin->size_used);

  xvec = x->vec;
  yvec = y->vec;

  FOR_ALL_DOFS(admin, yvec[dof] = xvec[dof] + alpha*yvec[dof]);
}


REAL dof_min(const DOF_REAL_VEC *x)
{
  FUNCNAME("dof_min");
  REAL m;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
	    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size, 
	    admin->size_used);

  m = 1.0E30;
  FOR_ALL_DOFS(admin, m = MIN(m, x->vec[dof]) );

  return(m);
}

REAL dof_max(const DOF_REAL_VEC *x)
{
  FUNCNAME("dof_max");
  REAL m;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
	    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", x->size, 
	    admin->size_used);

  m = -1.0E30;
  FOR_ALL_DOFS(admin, m = MAX(m, x->vec[dof]) );

  return(m);
}

/*--------------------------------------------------------------------------*/
/*   now the same routines for REAL_D vectors                               */
/*--------------------------------------------------------------------------*/

void dof_set_d(REAL alpha, DOF_REAL_D_VEC *x)
{
  FUNCNAME("dof_set_d");
  const DOF_ADMIN *admin = nil;
  int             n;

  TEST_EXIT(x && x->fe_space  &&  (admin = x->fe_space->admin),
	    "pointer is nil: x: %p, x->fe_space: %p, x->fe_space->admin :%p\n",
	    x, x->fe_space, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", 
	    x->size, admin->size_used);

  FOR_ALL_DOFS(admin, 
	       for (n = 0; n < DIM_OF_WORLD; n++)
	          x->vec[dof][n] = alpha);
}

void dof_scal_d(REAL alpha, DOF_REAL_D_VEC *x)
{
  FUNCNAME("dof_scal_d");
  const DOF_ADMIN *admin = nil;
  int              n;

  TEST_EXIT(x && x->fe_space  &&  (admin = x->fe_space->admin),
	"pointer is nil: x: %p, x->fe_space: %p, x->fe_space->admin :%p\n",
	    x, admin);
  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n", 
	    x->size, admin->size_used);

  FOR_ALL_DOFS(admin,
	       for (n = 0; n < DIM_OF_WORLD; n++)
	          x->vec[dof][n] *= alpha);
}

REAL dof_dot_d(const DOF_REAL_D_VEC *x, const DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_dot_d");
  REAL            dot;
  const DOF_ADMIN *admin = nil;
  int             n;

  TEST_EXIT(x && y,
	    "pointer to DOF_REAL_D_VEC is nil: x: %p, y: %p\n",
	    x, y);
  TEST_EXIT(x->fe_space && y->fe_space,
	    "pointer to FE_SPACE is nil: x->fe_space: %p, y->fe_space: %p\n",
	    x->fe_space, y->fe_space);
  
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
    "no admin or admins: x->fe_space->admin: %p, y->fe_space->admin: %p\n",
	    x->fe_space->admin, y->fe_space->admin);

  TEST_EXIT(x->size >= admin->size_used,
	    "x->size = %d too small: admin->size_used = %d\n",
	    x->size, admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
    "y->size = %d too small: admin->size_used = %d\n",
     y->size, admin->size_used);

  dot = 0.0;
  FOR_ALL_DOFS(admin, for (n = 0; n < DIM_OF_WORLD; n++)
	                 dot += x->vec[dof][n] * y->vec[dof][n] );

  return(dot);
}

void dof_copy_d(const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_copy_d");
  const DOF_ADMIN *admin = nil;
  int             n;

  TEST_EXIT(x && y,"pointer to DOF_REAL_D_VEC is nil: x: %p, y: %p\n",
		    x, y);
  TEST_EXIT(x->fe_space && y->fe_space,
    "pointer to FE_SPACE is nil: x->fe_space: %p, y->fe_space: %p\n",
     x->fe_space, y->fe_space);

  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
    "no admin or admins: x->fe_space->admin: %p, y->fe_space->admin: %p\n",
     x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
    "x->size = %d too small: admin->size_used = %d\n",
     x->size, admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
    "y->size = %d too small: admin->size_used = %d\n",
     y->size, admin->size_used);

  FOR_ALL_DOFS(admin, for (n = 0; n < DIM_OF_WORLD; n++)
	                y->vec[dof][n] = x->vec[dof][n]);

}

void dof_axpy_d(REAL alpha, const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_axpy_d");
  const DOF_ADMIN *admin;
  int             n;

  TEST_EXIT(x && y,"pointer to DOF_REAL_D_VEC is nil: x: %p, y: %p\n",
		    x, y);
  TEST_EXIT(x->fe_space && y->fe_space,
    "pointer to FE_SPACE is nil: x->fe_space: %p, y->fe_space: %p\n",
     x->fe_space, y->fe_space);
  
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
    "no admin or admins: x->fe_space->admin: %p, y->fe_space->admin: %p\n",
     x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
    "x->size = %d too small: admin->size_used = %d\n",
     x->size, admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
    "y->size = %d too small: admin->size_used = %d\n",
     y->size, admin->size_used);

  FOR_ALL_DOFS(admin, for (n = 0; n < DIM_OF_WORLD; n++)
                         y->vec[dof][n] += alpha * x->vec[dof][n]);
}

REAL dof_nrm2_d(const DOF_REAL_D_VEC *x)
{
  FUNCNAME("dof_nrm2_d");
  REAL            nrm;
  int             n;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
    "x->size = %d too small: admin->size_used = %d\n", x->size, 
     admin->size_used);

  nrm = 0.0;
  FOR_ALL_DOFS(admin, for (n = 0; n < DIM_OF_WORLD; n++)
	                nrm += x->vec[dof][n] * x->vec[dof][n]);

  return(sqrt(nrm));
}

void dof_xpay_d(REAL alpha, const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_xpay_d");
  const DOF_ADMIN *admin;
  int             n;

  TEST_EXIT(x && y,"pointer to DOF_REAL_D_VEC is nil: x: %p, y: %p\n",
		    x, y);
  TEST_EXIT(x->fe_space && y->fe_space,
    "pointer to FE_SPACE is nil: x->fe_space: %p, y->fe_space: %p\n",
     x->fe_space, y->fe_space);
  
  TEST_EXIT((admin = x->fe_space->admin) && (admin == y->fe_space->admin),
    "no admin or admins: x->fe_space->admin: %p, y->fe_space->admin: %p\n",
     x->fe_space->admin, y->fe_space->admin);
  TEST_EXIT(x->size >= admin->size_used,
    "x->size = %d too small: admin->size_used = %d\n",
     x->size, admin->size_used);
  TEST_EXIT(y->size >= admin->size_used,
    "y->size = %d too small: admin->size_used = %d\n",
     y->size, admin->size_used);

  FOR_ALL_DOFS(admin, 
	       for (n = 0; n < DIM_OF_WORLD; n++)
	          y->vec[dof][n] = x->vec[dof][n] + alpha*y->vec[dof][n]);
}

REAL dof_min_d(const DOF_REAL_D_VEC *x)
{
  FUNCNAME("dof_min_d");
  REAL m, norm;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
    "x->size = %d too small: admin->size_used = %d\n", x->size, 
     admin->size_used);

  m = 1.0E30;
  FOR_ALL_DOFS(admin, 
	       norm = NORM_DOW(x->vec[dof]);
	       m = MIN(m, norm) );

  return(m);
}

REAL dof_max_d(const DOF_REAL_D_VEC *x)
{
  FUNCNAME("dof_max_d");
  REAL m, norm;
  const DOF_ADMIN *admin = nil;

  TEST_EXIT(x && x->fe_space && (admin = x->fe_space->admin),
    "pointer is nil: %p, %p\n", x, admin);
  TEST_EXIT(x->size >= admin->size_used,
    "x->size = %d too small: admin->size_used = %d\n", x->size, 
     admin->size_used);

  m = 0.0;
  FOR_ALL_DOFS(admin,  
	       norm = NORM_DOW(x->vec[dof]);
	       m = MAX(m, norm) );

  return(m);
}



void dof_gemv(MatrixTranspose transpose, REAL alpha,
	      const DOF_MATRIX *a, const DOF_REAL_VEC *x,
	      REAL beta, DOF_REAL_VEC *y)
{
  FUNCNAME("dof_gemv");
  int        j, jcol, ysize;
  REAL       sum, ax;
  REAL       *xvec, *yvec;
  MATRIX_ROW *row;
  const DOF_ADMIN *row_admin = nil, *col_admin = nil;

  TEST_EXIT(a && x && y,"pointer is nil: %p, %p, %p\n", a,x,y);
  TEST_EXIT(a->row_fe_space && a->col_fe_space && x->fe_space && y->fe_space,
	    "Missing fe_space!\nMatrix row fe_space: %p\nMatrix column fe_space: %p\nX vector fe_space: %p\nY vector fe_space: %p\n",
	    a->row_fe_space, a->col_fe_space,
	    x->fe_space,y->fe_space);
  TEST_EXIT((row_admin = a->row_fe_space->admin) && 
	    (col_admin = a->col_fe_space->admin) && 
	    (col_admin == x->fe_space->admin) && 
	    (row_admin == y->fe_space->admin),
	    "No admin or non-matching admins!\nMatrix row admin: %p\nMatrix column admin: %p\nX vector admin: %p\nY vector admin: %p\n",
	    a->row_fe_space->admin, a->col_fe_space->admin,
	    x->fe_space->admin, y->fe_space->admin);
  
  TEST_EXIT(x->size >= col_admin->size_used,
    "x->size = %d too small: col_admin->size_used = %d\n",
     x->size, col_admin->size_used);
  TEST_EXIT(y->size >= row_admin->size_used,
    "y->size = %d too small: row_admin->size_used = %d\n",
     y->size, row_admin->size_used);
  TEST_EXIT(a->size >= row_admin->size_used,
    "a->size = %d too small: row_admin->size_used = %d\n",
     a->size, row_admin->size_used);

  xvec = x->vec;
  yvec = y->vec;

  ysize = y->size;

  FOR_ALL_FREE_DOFS(row_admin, if (dof < ysize) yvec[dof] = 0.0);

  if (transpose == NoTranspose) {
    FOR_ALL_DOFS(row_admin, 
	      sum = 0.0;
	      for (row = a->matrix_row[dof]; row; row = row->next) {
		for (j=0; j<ROW_LENGTH; j++) {
		  jcol = row->col[j];
		  if (ENTRY_USED(jcol)) {
		    sum += row->entry[j] * xvec[jcol];
		  }
		  else {
		    if (jcol == NO_MORE_ENTRIES)
		      break;
		  }
		}
	      }
	      yvec[dof] = beta * yvec[dof] + alpha * sum;
      );
  }
  else if (transpose == Transpose) {

    FOR_ALL_DOFS(row_admin, yvec[dof]  *= beta );
    FOR_ALL_DOFS(row_admin, 
		 ax = alpha * xvec[dof];
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol))
		       yvec[jcol] += ax * row->entry[j];
		     else 
		       if (jcol == NO_MORE_ENTRIES) break;
		   }
		 }
      );
  }
  else {
    ERROR_EXIT("transpose=%d\n", transpose);
  }
}


/*--------------------------------------------------------------------------*/

void dof_mv(MatrixTranspose transpose,
	    const DOF_MATRIX *a, const DOF_REAL_VEC *x, DOF_REAL_VEC *y)
{
  FUNCNAME("dof_mv");
  int           j, jcol;
  REAL          sum, ax, *xvec, *yvec;
  MATRIX_ROW    *row;
  const DOF_ADMIN *row_admin = nil, *col_admin = nil;

  TEST_EXIT(a && x && y,"pointer is nil: %p, %p, %p\n", a,x,y);
  TEST_EXIT(a->row_fe_space && a->col_fe_space && x->fe_space && y->fe_space,
	    "Missing fe_space!\nMatrix row fe_space: %p\nMatrix column fe_space: %p\nX vector fe_space: %p\nY vector fe_space: %p\n",
	    a->row_fe_space, a->col_fe_space,
	    x->fe_space,y->fe_space);
  TEST_EXIT((row_admin = a->row_fe_space->admin) && 
	    (col_admin = a->col_fe_space->admin) && 
	    (col_admin == x->fe_space->admin) && 
	    (row_admin == y->fe_space->admin),
	    "No admin or non-matching admins!\nMatrix row admin: %p\nMatrix column admin: %p\nX vector admin: %p\nY vector admin: %p\n",
	    a->row_fe_space->admin, a->col_fe_space->admin,
	    x->fe_space->admin, y->fe_space->admin);
  
  TEST_EXIT(x->size >= col_admin->size_used,
    "x->size = %d too small: col_admin->size_used = %d\n",
     x->size, col_admin->size_used);
  TEST_EXIT(y->size >= row_admin->size_used,
    "y->size = %d too small: row_admin->size_used = %d\n",
     y->size, row_admin->size_used);
  TEST_EXIT(a->size >= row_admin->size_used,
    "a->size = %d too small: row_admin->size_used = %d\n",
     a->size, row_admin->size_used);


  xvec = x->vec;
  yvec = y->vec;

  if (transpose == NoTranspose) {
    int ysize = y->size;
    FOR_ALL_FREE_DOFS(row_admin, if (dof < ysize) yvec[dof] = 0.0);

    FOR_ALL_DOFS(row_admin,
		 sum = 0.0;
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol)) {
		       sum += row->entry[j] * xvec[jcol];
		     }
		     else {
		       if (jcol == NO_MORE_ENTRIES)
			 break;
		     }
		   }
		 }
		 yvec[dof] = sum;
      );
  }
  else if (transpose == Transpose) {
    
    for (j = 0; j < row_admin->size_used; j++)
      yvec[j] = 0.0;

    FOR_ALL_DOFS(row_admin,
		 ax = xvec[dof];
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol)) {
		       yvec[jcol] += ax * row->entry[j];
		     }
		     else {
		       if (jcol == NO_MORE_ENTRIES)
			 break;
		     }
		   }
		 }
      );
  }
  else {
    ERROR_EXIT("transpose=%d\n", transpose);
  }
}


void dof_gemv_d(MatrixTranspose transpose, REAL alpha,
		const DOF_MATRIX *a, const DOF_REAL_D_VEC *x,
		REAL beta, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_gemv_d");
  int               n, j, jcol;
  REAL_D            sum, ax;
  REAL_D            *xvec, *yvec;
  MATRIX_ROW        *row;
  const DOF_ADMIN *row_admin = nil, *col_admin = nil;


  TEST_EXIT(a && x && y,"pointer is nil: %p, %p, %p\n", a,x,y);
  TEST_EXIT(a->row_fe_space && a->col_fe_space && x->fe_space && y->fe_space,
	    "Missing fe_space!\nMatrix row fe_space: %p\nMatrix column fe_space: %p\nX vector fe_space: %p\nY vector fe_space: %p\n",
	    a->row_fe_space, a->col_fe_space,
	    x->fe_space,y->fe_space);
  TEST_EXIT((row_admin = a->row_fe_space->admin) && 
	    (col_admin = a->col_fe_space->admin) && 
	    (col_admin == x->fe_space->admin) && 
	    (row_admin == y->fe_space->admin),
	    "No admin or non-matching admins!\nMatrix row admin: %p\nMatrix column admin: %p\nX vector admin: %p\nY vector admin: %p\n",
	    a->row_fe_space->admin, a->col_fe_space->admin,
	    x->fe_space->admin, y->fe_space->admin);
  
  TEST_EXIT(x->size >= col_admin->size_used,
    "x->size = %d too small: col_admin->size_used = %d\n",
     x->size, col_admin->size_used);
  TEST_EXIT(y->size >= row_admin->size_used,
    "y->size = %d too small: row_admin->size_used = %d\n",
     y->size, row_admin->size_used);
  TEST_EXIT(a->size >= row_admin->size_used,
    "a->size = %d too small: row_admin->size_used = %d\n",
     a->size, row_admin->size_used);



  xvec = x->vec;
  yvec = y->vec;

  if (transpose == NoTranspose) {
    int ysize = y->size;
    FOR_ALL_FREE_DOFS(row_admin, if (dof < ysize) 
		      for (n = 0; n < DIM_OF_WORLD; n++)
		      yvec[dof][n] = 0.0);

    FOR_ALL_DOFS(row_admin,	
		 for (n = 0; n < DIM_OF_WORLD; n++)
		    sum[n] = 0.0;
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol)) 
		     {
		       for (n = 0; n < DIM_OF_WORLD; n++)
			 sum[n] += row->entry[j] * xvec[jcol][n];
		     }
		     else 
		     {
		       if (jcol == NO_MORE_ENTRIES)
			 break;
		     }
		   }
		 }
		 for (n = 0; n < DIM_OF_WORLD; n++)
		    yvec[dof][n] = beta * yvec[dof][n] + alpha * sum[n];
      );
  }
  else if (transpose == Transpose) {
    
    FOR_ALL_DOFS(row_admin,
		 for (n = 0; n < DIM_OF_WORLD; n++) {
		     yvec[dof][n]  *= beta;
		 } );
   FOR_ALL_DOFS(row_admin,
		 for (n = 0; n < DIM_OF_WORLD; n++)
		    ax[n] = alpha * xvec[dof][n];
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol))
		     {
		       for (n = 0; n < DIM_OF_WORLD; n++)
			 yvec[jcol][n] += ax[n] * row->entry[j];
		     }
		     else
		     {
		       if (jcol == NO_MORE_ENTRIES)
			 break;
		     }
		   }
		 }
      );
  }
  else {
    ERROR_EXIT("transpose=%d\n", transpose);
  }
}

void dof_mv_d(MatrixTranspose transpose, const DOF_MATRIX *a, 
	      const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_mv_d");
  int               n, j, jcol;
  REAL_D            sum, ax;
  REAL_D            *xvec, *yvec;
  MATRIX_ROW        *row;
  const DOF_ADMIN *row_admin = nil, *col_admin = nil;


  TEST_EXIT(a && x && y,"pointer is nil: %p, %p, %p\n", a,x,y);
  TEST_EXIT(a->row_fe_space && a->col_fe_space && x->fe_space && y->fe_space,
	    "Missing fe_space!\nMatrix row fe_space: %p\nMatrix column fe_space: %p\nX vector fe_space: %p\nY vector fe_space: %p\n",
	    a->row_fe_space, a->col_fe_space,
	    x->fe_space,y->fe_space);
  TEST_EXIT((row_admin = a->row_fe_space->admin) && 
	    (col_admin = a->col_fe_space->admin) && 
	    (col_admin == x->fe_space->admin) && 
	    (row_admin == y->fe_space->admin),
	    "No admin or non-matching admins!\nMatrix row admin: %p\nMatrix column admin: %p\nX vector admin: %p\nY vector admin: %p\n",
	    a->row_fe_space->admin, a->col_fe_space->admin,
	    x->fe_space->admin, y->fe_space->admin);
  
  TEST_EXIT(x->size >= col_admin->size_used,
    "x->size = %d too small: col_admin->size_used = %d\n",
     x->size, col_admin->size_used);
  TEST_EXIT(y->size >= row_admin->size_used,
    "y->size = %d too small: row_admin->size_used = %d\n",
     y->size, row_admin->size_used);
  TEST_EXIT(a->size >= row_admin->size_used,
    "a->size = %d too small: row_admin->size_used = %d\n",
     a->size, row_admin->size_used);

  xvec = x->vec;
  yvec = y->vec;

  if (transpose == NoTranspose) {
    int ysize = y->size;
    FOR_ALL_FREE_DOFS(row_admin, if (dof < ysize) 
                               for (n = 0; n < DIM_OF_WORLD; n++)
		                  yvec[dof][n] = 0.0);

    FOR_ALL_DOFS(row_admin,	
		 for (n = 0; n < DIM_OF_WORLD; n++)
		    sum[n] = 0.0;
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol)) 
		     {
		       for (n = 0; n < DIM_OF_WORLD; n++)
			 sum[n] += row->entry[j] * xvec[jcol][n];
		     }
		     else 
		     {
		       if (jcol == NO_MORE_ENTRIES)
			 break;
		     }
		   }
		 }
		 for (n = 0; n < DIM_OF_WORLD; n++)
		    yvec[dof][n] = sum[n];
      );
  }
  else if (transpose == Transpose) {
    
    for (j = 0; j < row_admin->size_used; j++)
      for (n = 0; n < DIM_OF_WORLD; n++)
	yvec[j][n] = 0.0;

    FOR_ALL_DOFS(row_admin,
		 for (n = 0; n < DIM_OF_WORLD; n++)
		    ax[n] = xvec[dof][n];
		 for (row = a->matrix_row[dof]; row; row = row->next) {
		   for (j=0; j<ROW_LENGTH; j++) {
		     jcol = row->col[j];
		     if (ENTRY_USED(jcol))
		     {
		       for (n = 0; n < DIM_OF_WORLD; n++)
			 yvec[jcol][n] += ax[n] * row->entry[j];
		     }
		     else
		     {
		       if (jcol == NO_MORE_ENTRIES)
			 break;
		     }
		   }
		 }
      );
  }
  else {
    ERROR_EXIT("transpose=%d\n", transpose);
  }
}

void dof_gemv_dowb(MatrixTranspose transpose, REAL alpha,
		   const DOF_DOWB_MATRIX *a, const DOF_REAL_D_VEC *x,
		   REAL beta, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_gemv_dowb");
  int               j, jcol;
  REAL_D            sum, ax;
  REAL_D            *xvec, *yvec;
  DOWB_MATRIX_ROW        *row;
  const DOF_ADMIN *row_admin = nil, *col_admin = nil;


  TEST_EXIT(a && x && y,"pointer is nil: %p, %p, %p\n", a,x,y);
  TEST_EXIT(a->row_fe_space && a->col_fe_space && x->fe_space && y->fe_space,
	    "Missing fe_space!\nMatrix row fe_space: %p\nMatrix column fe_space: %p\nX vector fe_space: %p\nY vector fe_space: %p\n",
	    a->row_fe_space, a->col_fe_space,
	    x->fe_space,y->fe_space);
  TEST_EXIT((row_admin = a->row_fe_space->admin) && 
	    (col_admin = a->col_fe_space->admin) && 
	    (col_admin == x->fe_space->admin) && 
	    (row_admin == y->fe_space->admin),
	    "No admin or non-matching admins!\nMatrix row admin: %p\nMatrix column admin: %p\nX vector admin: %p\nY vector admin: %p\n",
	    a->row_fe_space->admin, a->col_fe_space->admin,
	    x->fe_space->admin, y->fe_space->admin);
  
  TEST_EXIT(x->size >= col_admin->size_used,
    "x->size = %d too small: col_admin->size_used = %d\n",
     x->size, col_admin->size_used);
  TEST_EXIT(y->size >= row_admin->size_used,
    "y->size = %d too small: row_admin->size_used = %d\n",
     y->size, row_admin->size_used);
  TEST_EXIT(a->size >= row_admin->size_used,
    "a->size = %d too small: row_admin->size_used = %d\n",
     a->size, row_admin->size_used);


  xvec = x->vec;
  yvec = y->vec;

  if (transpose == NoTranspose) {
    int ysize = y->size;
    FOR_ALL_FREE_DOFS(row_admin, if (dof < ysize) SET_DOW(0.0, yvec[dof]));

#undef MAT_BODY
#define MAT_BODY(F, C, P, S)					\
    FOR_ALL_DOFS(row_admin, {					\
      SET_DOW(0.0, sum);					\
								\
      for (row = a->matrix_row[dof]; row; row = row->next) {	\
	for (j=0; j<ROW_LENGTH; j++) {				\
	  jcol = row->col[j];					\
	  if (ENTRY_USED(jcol)) {				\
	    F##V_DOW(C P row->entry.S[j], xvec[jcol], sum);	\
	  }							\
	  else 							\
	  {							\
	    if (jcol == NO_MORE_ENTRIES)			\
	      break;						\
	  }							\
	}							\
      }								\
      AXPBY_DOW(alpha, sum, beta, yvec[dof], yvec[dof]);	\
    });
    MAT_EMIT_BODY_SWITCH(a->type);
    
  } else if (transpose == Transpose) {
    
    FOR_ALL_DOFS(row_admin, AX_DOW(beta, yvec[dof]));

#undef MAT_BODY
#define MAT_BODY(F, C, P, S)					\
    FOR_ALL_DOFS(row_admin, {					\
      COPY_DOW(xvec[dof], ax); AX_DOW(alpha, ax);		\
								\
      for (row = a->matrix_row[dof]; row; row = row->next) {	\
	for (j=0; j<ROW_LENGTH; j++) {				\
	  jcol = row->col[j];					\
	  if (ENTRY_USED(jcol)) {				\
	    F##TV_DOW(C P row->entry.S[j], ax, yvec[jcol]);	\
	  } else {						\
	    if (jcol == NO_MORE_ENTRIES)			\
	      break;						\
	  }							\
	}							\
      }								\
    });
    MAT_EMIT_BODY_SWITCH(a->type);
  }
  else {
    ERROR_EXIT("transpose=%d\n", transpose);
  }
}

/* Multiply a DOF_REAL_D_VEC with a DOF_DOWB_MATRIX
 *
 */
void dof_mv_dowb(MatrixTranspose transpose, const DOF_DOWB_MATRIX *a, 
		 const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y)
{
  FUNCNAME("dof_mv_dowb");
  int               j, jcol;
  REAL_D            sum;
  REAL_D            *xvec, *yvec;
  DOWB_MATRIX_ROW   *row;
  const DOF_ADMIN *row_admin = nil, *col_admin = nil;


  TEST_EXIT(a && x && y,"pointer is nil: %p, %p, %p\n", a,x,y);
  TEST_EXIT(a->row_fe_space && a->col_fe_space && x->fe_space && y->fe_space,
	    "Missing fe_space!\nMatrix row fe_space: %p\nMatrix column fe_space: %p\nX vector fe_space: %p\nY vector fe_space: %p\n",
	    a->row_fe_space, a->col_fe_space,
	    x->fe_space,y->fe_space);
  TEST_EXIT((row_admin = a->row_fe_space->admin) && 
	    (col_admin = a->col_fe_space->admin) && 
	    (col_admin == x->fe_space->admin) && 
	    (row_admin == y->fe_space->admin),
	    "No admin or non-matching admins!\nMatrix row admin: %p\nMatrix column admin: %p\nX vector admin: %p\nY vector admin: %p\n",
	    a->row_fe_space->admin, a->col_fe_space->admin,
	    x->fe_space->admin, y->fe_space->admin);
  
  TEST_EXIT(x->size >= col_admin->size_used,
    "x->size = %d too small: col_admin->size_used = %d\n",
     x->size, col_admin->size_used);
  TEST_EXIT(y->size >= row_admin->size_used,
    "y->size = %d too small: row_admin->size_used = %d\n",
     y->size, row_admin->size_used);
  TEST_EXIT(a->size >= row_admin->size_used,
    "a->size = %d too small: row_admin->size_used = %d\n",
     a->size, row_admin->size_used);

  xvec = x->vec;
  yvec = y->vec;

  if (transpose == NoTranspose) {
    int ysize = y->size;
    FOR_ALL_FREE_DOFS(row_admin, if (dof < ysize) SET_DOW(0.0, yvec[dof]));

#undef MAT_BODY
#define MAT_BODY(F, C, P, S)					\
    FOR_ALL_DOFS(row_admin, {					\
      SET_DOW(0.0, sum);					\
								\
      for (row = a->matrix_row[dof]; row; row = row->next) {	\
	for (j=0; j<ROW_LENGTH; j++) {				\
	  jcol = row->col[j];					\
	  if (ENTRY_USED(jcol)) {				\
	    F##V_DOW(C P row->entry.S[j], xvec[jcol], sum);	\
	  } else {						\
	    if (jcol == NO_MORE_ENTRIES)			\
	      break;						\
	  }							\
	}							\
      }								\
      COPY_DOW(sum, yvec[dof]);					\
    });
    MAT_EMIT_BODY_SWITCH(a->type);
  } else if (transpose == Transpose) {

    TEST_EXIT(x != y,
      "x == y not implemented in transpose mode");
    
    for (j = 0; j < row_admin->size_used; j++) {
      SET_DOW(0.0, yvec[j]);
    }

#undef MAT_BODY	
#define MAT_BODY(F, C, P, S)						\
    FOR_ALL_DOFS(row_admin, {						\
      for (row = a->matrix_row[dof]; row; row = row->next) {		\
	for (j=0; j<ROW_LENGTH; j++) {					\
	  jcol = row->col[j];						\
	  if (ENTRY_USED(jcol)) {					\
	    F##TV_DOW(C P row->entry.S[j], xvec[dof], yvec[jcol]);	\
	  } else {							\
	    if (jcol == NO_MORE_ENTRIES)				\
	      break;							\
	  }								\
	}								\
      }									\
    });
    MAT_EMIT_BODY_SWITCH(a->type);
  } else {
    ERROR_EXIT("transpose=%d\n", transpose);
  }
}

/*--------------------------------------------------------------------------*/
/*  print_dof_matrix: print matrix in compressed format                     */
/*--------------------------------------------------------------------------*/

void print_dof_matrix(const DOF_MATRIX *matrix)
{
  FUNCNAME("print_dof_matrix");
  int  i, j, jcol;
  MATRIX_ROW *row;

  for (i=0; i<matrix->size; i++) { 
    for (row = matrix->matrix_row[i]; row; row = row->next) {
      MSG("row %3d:",i);
      for (j=0; j<ROW_LENGTH; j++) {
	jcol = row->col[j];
	if (ENTRY_USED(jcol)) {
	  print_msg(" (%3d,% .2e)", jcol, row->entry[j]);
	}
      }
      print_msg("\n");
    }
  }
  return;
}

void print_dof_dowb_matrix(const DOF_DOWB_MATRIX *matrix)
{
  FUNCNAME("print_dof_dowb_matrix");
  int  i, j, jcol, n, m;
  DOWB_MATRIX_ROW *row;

  for (i=0; i < matrix->size; i++) { 
    if (matrix->matrix_row[i]) {
      for (n = 0; n < DIM_OF_WORLD; n++) {
	if (n == 0) {
	  MSG("row %3d:",i);
	} else {
	  MSG("        ");
	}
	for (row = matrix->matrix_row[i]; row; row = row->next) {
	  for (j=0; j<ROW_LENGTH; j++) {
	    jcol = row->col[j];
	    if (ENTRY_USED(jcol)) {
	      if (n == 0) {
		print_msg(" |%3d", jcol);
	      } else {
		print_msg(" |   ");
	      }
	      switch (matrix->type) {
	      case dowbm_full:
		for (m = 0; m < DIM_OF_WORLD; m++) {
		  print_msg(" % .2e", row->entry.full[j][n][m]);
		}
		break;
	      case dowbm_symm:
		switch (n) {
		case 0:
		  for (m = 0; m < n; m++) {
		    print_msg("           ");
		  }
		  for (m = 0; m < DIM_OF_WORLD-n; m++) {
		    print_msg(" % .2e", row->entry.symm[j].row0[m]);
		  }
		  break;
		case 1:
		  for (m = 0; m < n; m++) {
		    print_msg("           ");
		  }
		  for (m = 0; m < DIM_OF_WORLD-n; m++) {
		    print_msg(" % .2e", row->entry.symm[j].row1[m]);
		  }
		  break;
		case 2:
		  for (m = 0; m < n; m++) {
		    print_msg("           ");
		  }
		  for (m = 0; m < DIM_OF_WORLD-n; m++) {
		    print_msg(" % .2e", row->entry.symm[j].row2[m]);
		  }
		  break;
		default:
		  ERROR_EXIT("DIM_OF_WORLD == %d unsupported", n + 1);
		}
		break;
	      case dowbm_diag:
		for (m = 0; m < n; m++) {
		  print_msg("           ");
		}
		print_msg(" % .2e", row->entry.diag[j][n]);
		for (; m < DIM_OF_WORLD; m++) {
		  print_msg("           ");
		}
		break;
	      }
	    }
	  }
	}
	print_msg("\n");
      }
    }
  }
}

void print_dof_real_vec(const DOF_REAL_VEC *drv)
{
  FUNCNAME("print_dof_real_vec");
  int  i, j;
  const DOF_ADMIN *admin = nil;
  char      *format;

  if (drv->fe_space) admin = drv->fe_space->admin;

  MSG("Vec `%s':\n", drv->name);
  j = 0;
  if (admin)
  {
    if (admin->size_used > 100)
      format = "%s(%3d,%10.5le)";
    else if (admin->size_used > 10)
      format = "%s(%2d,%10.5le)";
    else
      format = "%s(%1d,%10.5le)";

    FOR_ALL_DOFS(admin,
		 if ((j % 3) == 0) {
                   if (j) print_msg("\n");
		   MSG(format, "", dof, drv->vec[dof]);
                 }
		 else 
		   print_msg(format, " ", dof, drv->vec[dof]);
		 j++;
      );
    print_msg("\n");
  }
  else
  {
    MSG("no DOF_ADMIN, print whole vector.\n");
    
    for (i = 0; i < drv->size; i++) {
      if ((j % 3) == 0)
      {
	if (j) print_msg("\n");
	MSG("(%d,%10.5le)",i,drv->vec[i]);
      }
      else 
	print_msg(" (%d,%10.5le)",i,drv->vec[i]);
      j++;
    }
    print_msg("\n");
  }
  return;
}


void print_dof_real_d_vec(const DOF_REAL_D_VEC *drdv)
{
  FUNCNAME("print_dof_real_d_vec");
  int         i, j, k;
  const DOF_ADMIN   *admin = nil;
#if DIM_OF_WORLD < 2
  static int  per_line = 4;
#else
#if DIM_OF_WORLD < 4
  static int  per_line = 2;
#else
  static int  per_line = 1;
#endif
#endif

  if (drdv->fe_space) admin = drdv->fe_space->admin;

  MSG("Vec `%s':\n", drdv->name);
  j = 0;
  if (admin)
  {
    FOR_ALL_DOFS(admin,
		 if ((j % per_line) == 0) {
                    if (j) print_msg("\n");
		    MSG("(%3d:",dof);
                 }
		 else 
		    print_msg(" (%3d:", dof);
		 for (k=0; k<DIM_OF_WORLD; k++)
		 print_msg("%c%10.5le", (k > 0 ? ',':' '), drdv->vec[dof][k]);
		 print_msg(")");
		 j++;
		 );	 
    print_msg("\n");
  }
  else
  {
    MSG("no DOF_ADMIN, print whole vector.\n");
    
    for (i = 0; i < drdv->size; i++) {
      if ((j % per_line) == 0)
      {
	if (j) print_msg("\n");
	MSG("(%3d:",i);
      }
      else 
	print_msg(" (%3d:", i);
      for (k=0; k<DIM_OF_WORLD; k++)
	print_msg("%c%10.5le", (k > 0 ? ',':' '), drdv->vec[i][k]);
      print_msg(")");
    }
    print_msg("\n");
  }
  return;
}


void print_dof_int_vec(const DOF_INT_VEC *div)
{
  FUNCNAME("print_dof_int_vec");
  int       i, j;
  const DOF_ADMIN *admin = nil;
  char      *format;

  if (div->fe_space) admin = div->fe_space->admin;

  MSG("Vector `%s':\n", div->name);
  j = 0;
  if (admin) {
    if (admin->size_used > 100)
      format = "%s(%3d,%3d)";
    else if (admin->size_used > 10)
      format = "%s(%2d,%3d)";
    else
      format = "%s(%1d,%3d)";

    FOR_ALL_DOFS(admin,
		 if ((j % 5) == 0)
		 {
		   if (j) print_msg("\n");
		   MSG(format, "", dof,div->vec[dof]);
		 }
	         else 
		   print_msg(format, " ", dof, div->vec[dof]);
	         j++;
      );
    print_msg("\n");
  }
  else {
    if (div->size > 100)
      format = "%s(%3d,%3d)";
    else if (div->size > 10)
      format = "%s(%2d,%3d)";
    else
      format = "%s(%1d,%3d)";

    for (i = 0; i < div->size; i++) {
      if ((j % 5) == 0)
	{
	  if (j) print_msg("\n");
	  MSG(format, "", i, div->vec[i]);
	}
      else 
	print_msg(format, " ", i, div->vec[i]);
      j++;
    }
    print_msg("\n");
  }
  return;
}

void print_dof_uchar_vec(const DOF_UCHAR_VEC *duv)
{
  FUNCNAME("print_dof_uchar_vec");
  int  i, j;
  const DOF_ADMIN *admin = nil;
  char      *format;

  if (duv->fe_space) admin = duv->fe_space->admin;

  MSG("Vector `%s':\n", duv->name);
  j = 0;

  if (admin) {
    if (admin->size_used > 100)
      format = "%s(%3d,0x%2X)";
    else if (admin->size_used > 10)
      format = "%s(%2d,0x%2X)";
    else
      format = "%s(%1d,0x%2X)";

    FOR_ALL_DOFS(admin,
	      if ((j % 5) == 0) {
		if (j) print_msg("\n");
		MSG(format, "", dof, duv->vec[dof]);
	      }
	      else 
		print_msg(format, " ", dof, duv->vec[dof]);
	      j++;
      );
    print_msg("\n");
  }
  else {
    if (duv->size > 100)
      format = "%s(%3d,0x%2X)";
    else if (duv->size > 10)
      format = "%s(%2d,0x%2X)";
    else
      format = "%s(%1d,0x%2X)";

    for (i = 0; i < duv->size; i++) {
      if ((j % 5) == 0)
	{
	  if (j) print_msg("\n");
	  MSG(format, "", i, duv->vec[i]);
	}
      else 
	print_msg(format, " ", i, duv->vec[i]);
      j++;
    }
    print_msg("\n");
  }
  return;
}

void print_dof_schar_vec(const DOF_SCHAR_VEC *dsv)
{
  FUNCNAME("print_dof_schar_vec");
  int       i, j;
  const DOF_ADMIN *admin = nil;
  char      *format;

  if (dsv->fe_space) admin = dsv->fe_space->admin;

  MSG("Vector `%s':\n", dsv->name);
  j = 0;


  if (admin) {
    if (admin->size_used > 100)
      format = "%s(%3d,0x%2X)";
    else if (admin->size_used > 10)
      format = "%s(%2d,0x%2X)";
    else
      format = "%s(%1d,0x%2X)";

    FOR_ALL_DOFS(admin,
	      if ((j % 5) == 0)
		{
		  if (j) print_msg("\n");
		  MSG(format, "",dof,dsv->vec[dof]);
		}
	      else 
		print_msg(format, " ",dof,dsv->vec[dof]);
	      j++;
		 );
    print_msg("\n");
  }
  else {
    if (dsv->size > 100)
      format = "%s(%3d,0x%2X)";
    else if (dsv->size > 10)
      format = "%s(%2d,0x%2X)";
    else
      format = "%s(%1d,0x%2X)";

    for (i = 0; i < dsv->size; i++) {
      if ((j % 5) == 0)
	{
	  if (j) print_msg("\n");
	  MSG(format, "", i, dsv->vec[i]);
	}
      else 
	print_msg(format, " ", i, dsv->vec[i]);
      j++;
    }
    print_msg("\n");
  }
  return;
}


void print_dof_ptr_vec(const DOF_PTR_VEC *dpv)
{
  FUNCNAME("print_dof_ptr_vec");
  int       i, j;
  const DOF_ADMIN *admin = nil;
  char      *format;

  if (dpv->fe_space) admin = dpv->fe_space->admin;

  MSG("Vector `%s':\n", dpv->name);
  j = 0;


  if (admin) {
    if (admin->size_used > 100)
      format = "%s(%3d,%p)";
    else if (admin->size_used > 10)
      format = "%s(%2d,%p)";
    else
      format = "%s(%1d,%p)";

    FOR_ALL_DOFS(admin,
	      if ((j % 5) == 0)
		{
		  if (j) print_msg("\n");
		  MSG(format, "", dof,dpv->vec[dof]);
		}
	      else 
		print_msg(format, " ",dof,dpv->vec[dof]);
	      j++;
		 );
    print_msg("\n");
  }
  else {
    if (dpv->size > 100)
      format = "%s(%3d,%p)";
    else if (dpv->size > 10)
      format = "%s(%2d,%p)";
    else
      format = "%s(%1d,%p)";

    for (i = 0; i < dpv->size; i++) {
      if ((j % 5) == 0)
	{
	  if (j) print_msg("\n");
	  MSG(format, "", i, dpv->vec[i]);
	}
      else 
	print_msg(format, " ", i, dpv->vec[i]);
      j++;
    }
    print_msg("\n");
  }
  return;
}

/*--------------------------------------------------------------------------*/
/*  clear_dof_matrix: remove all entries from dof_matrix                    */
/*--------------------------------------------------------------------------*/

void clear_dof_matrix(DOF_MATRIX *matrix)
{
  int        i;
  MATRIX_ROW *row, *next;

  for (i=0; i<matrix->size; i++) {
    for (row = matrix->matrix_row[i]; row; row = next) {
      next = row->next;
      free_matrix_row(matrix->row_fe_space, row);
    }
    matrix->matrix_row[i] = nil;
  }
  return;
}

/*--------------------------------------------------------------------------*/
/*  clear_dof_dobw_matrix: remove all entries from dof_matrix               */
/*--------------------------------------------------------------------------*/

void clear_dof_dowb_matrix(DOF_DOWB_MATRIX *matrix)
{
  int        i;
  DOWB_MATRIX_ROW   *row, *next;

  for (i=0; i<matrix->size; i++) {
    for (row = matrix->matrix_row[i]; row; row = next) {
      next = row->next;
      free_dowb_matrix_row(matrix->row_fe_space, matrix->type, row);
    }
    matrix->matrix_row[i] = nil;
  }
}

/*--------------------------------------------------------------------------*/
/*  test_dof_matrix                                                         */
/*--------------------------------------------------------------------------*/

void test_dof_matrix(DOF_MATRIX *matrix)
{
  FUNCNAME("test_dof_matrix");
  int  i, j, jcol, k,kcol;
  MATRIX_ROW *row, *row2;
  int        non_symmetric, found;
  double     sum;

/* test symmetry */
  non_symmetric = 0;
  for (i=0; i<matrix->size; i++) { 
    sum = 0.0;
    for (row = matrix->matrix_row[i]; row; row = row->next) {
      for (j=0; j<ROW_LENGTH; j++) {
	jcol = row->col[j];
	if (ENTRY_USED(jcol)) {
	  found = 0;
	  if (row->entry[j] != row->entry[j]) {
	    MSG("mat[%d,%d]=%10.5le ???\n", i,jcol,row->entry[j]);
	    WAIT;
	  }
	  for (row2 = matrix->matrix_row[jcol]; row2;
	       row2 = row2 ? row2->next : nil) {
	    for (k=0; k<ROW_LENGTH; k++) {
	      kcol = row2->col[k];
	      if (ENTRY_USED(kcol)) {
		if (kcol==i) {
		  found = 1;
		  if (ABS(row2->entry[k] - row->entry[j]) > 1.E-5) {
		    non_symmetric = 1;
		    MSG("mat[%d,%d]=%10.5le != mat[%d,%d]=%10.5le\n",
			i,jcol,row->entry[j], jcol,i,row2->entry[k]);
		  }
		  row2 = nil;
		  break;
		}
	      }
	    }
	  }
	  if (!found) {
	    non_symmetric = 1;
	    MSG("mat[%d,%d] not found\n",jcol,i);
	  }
	}
      }
    }
    if (ABS(sum) > 1.E-5) {
      MSG("Row sum[%d] = %10.5le\n", i, sum);
    }
  }
  if (non_symmetric) {
    MSG("matrix `%s' not symmetric.\n",matrix->name);
    WAIT;
  }
  else {
    MSG("matrix `%s' is symmetric.\n",matrix->name);
  }
  return;
}

/*--------------------------------------------------------------------------*/
/*  test_dof_dowb_matrix                                                    */
/*--------------------------------------------------------------------------*/

void test_dof_dowb_matrix(DOF_DOWB_MATRIX *matrix)
{
  FUNCNAME("test_dof_matrix");
  int  i, j, jcol, k,kcol;
  DOWB_MATRIX_ROW *row, *row2;
  int        non_symmetric, found;
  double     sum;

/* test symmetry */
  non_symmetric = 0;
  for (i=0; i<matrix->size; i++) { 
    sum = 0.0;
    for (row = matrix->matrix_row[i]; row; row = row->next) {
      for (j=0; j<ROW_LENGTH; j++) {
	jcol = row->col[j];
	if (ENTRY_USED(jcol)) {
	  found = 0;
	  for (row2 = matrix->matrix_row[jcol]; row2;
	       row2 = row2 ? row2->next : nil) {
	    for (k=0; k<ROW_LENGTH; k++) {
	      kcol = row2->col[k];
	      if (ENTRY_USED(kcol)) {
		if (kcol==i) {
		  found = 1;
		  switch (matrix->type) {
		  case dowbm_full:
		    if (MDST2_DOW(row2->entry.full[k],
				  row->entry.full[j]) > 1.E-10) {
		      non_symmetric = 1;
		      MSG("mat[%d,%d]="MFORMAT_DOW \
			  " != mat[%d,%d]="MFORMAT_DOW"\n",
			  i, jcol, MEXPAND_DOW(row->entry.full[j]),
			  jcol, i, MEXPAND_DOW(row2->entry.full[k]));
		    }
		    row2 = nil;
		    break;
		  case dowbm_symm:
		    if (MDST2_DOW(row2->entry.full[k],
				  row->entry.full[j]) > 1.E-10) {
		      non_symmetric = 1;
		      MSG("mat[%d,%d]="SMFORMAT_DOW \
			  " != mat[%d,%d]="SMFORMAT_DOW"\n",
			  i, jcol, SMEXPAND_DOW(&row->entry.symm[j]),
			  jcol, i, SMEXPAND_DOW(&row2->entry.symm[k]));
		    }
		    row2 = nil;
		    break;
		  case dowbm_diag:
		    if (DST2_DOW(row2->entry.diag[k],
				 row->entry.diag[j]) > 1.E-10) {
		      non_symmetric = 1;
		      MSG("mat[%d,%d]="DMFORMAT_DOW \
			  " != mat[%d,%d]="DMFORMAT_DOW"\n",
			  i, jcol, DMEXPAND_DOW(row->entry.diag[j]),
			  jcol, i, DMEXPAND_DOW(row2->entry.diag[k]));
		    }
		    row2 = nil;
		    break;
		  }
		  if (row2 == nil) {
		    break;
		  }
		}
	      }
	    }
	    if (!found) {
	      non_symmetric = 1;
	      MSG("mat[%d,%d] not found\n",jcol,i);
	    }
	  }
	}
      }
      if (ABS(sum) > 1.E-5) {
	MSG("Row sum[%d] = %10.5le\n", i, sum);
      }
    }
  }
  if (non_symmetric) {
    MSG("matrix `%s' not symmetric.\n",matrix->name);
    WAIT;
  }
  else {
    MSG("matrix `%s' is symmetric.\n",matrix->name);
  }
  return;
}


/*--------------------------------------------------------------------------*/
/*   the new update routines:                                               */
/*--------------------------------------------------------------------------*/

void add_element_matrix(DOF_MATRIX *matrix, REAL sign, int n_row, int n_col,
			const DOF *row_dof, const DOF *col_dof, 
			const REAL **el_mat, const S_CHAR *bound)
{
  FUNCNAME("add_element_matrix");
  DOF        i, j, k, irow, jcol;
  int        free_col=0;
  MATRIX_ROW *row, *free_row;

  TEST_EXIT(matrix,"no matrix\n");
  if (!col_dof || n_col <= 0)
  {
    col_dof = row_dof;
    n_col = n_row;
  }
  if (row_dof != col_dof  && bound)
    ERROR_EXIT("no boundary treatment for row_dof != col_dof\n");
    
  for (i = 0; i < n_row; i++) 
  {
    irow = row_dof[i];

    if (matrix->matrix_row[irow] == nil)
    {
      row = matrix->matrix_row[irow] = get_matrix_row(matrix->row_fe_space);
      if (row_dof == col_dof)
      {
	row->col[0] = irow;           /* first entry is diagonal element */
	if (bound && bound[i] >= DIRICHLET)
	{
	  row->entry[0] = 1.0;
	  continue;
	}
	else
	  row->entry[0] = 0.0;
      }
      else
      {
	row->col[0] = UNUSED_ENTRY;
      }
    }
    else if (bound && bound[i] >= DIRICHLET)
      continue;

    for (j = 0; j < n_col; j++)
    {
      jcol = col_dof[j];
      row = matrix->matrix_row[irow];
      free_row = nil;
      do 
      {
	for (k=0; k<ROW_LENGTH; k++)
	{
	  if (row->col[k] == jcol)
	  {
	    row->entry[k] += sign*el_mat[i][j];
	    break;
	  }
	  if (ENTRY_NOT_USED(row->col[k])) 
	  {
	    free_col = k;
	    free_row = row;
	    if (row->col[k] == NO_MORE_ENTRIES) 
	    {
	      k = ROW_LENGTH;
	      break;
	    }
	  }
	}
	if (k < ROW_LENGTH) break;               /* done? */
	if (row->next || free_row)
	{
	  row = row->next;
	}
	else 
	{
	  free_row = row->next = get_matrix_row(matrix->row_fe_space);
	  free_col = 0;
	  row = nil;
	}
      } while (row);
      
      if (k >= ROW_LENGTH)                       /* not done? */
      {                     
	free_row->col[free_col]   = jcol;
	free_row->entry[free_col] = sign * el_mat[i][j];
      }
    }
  }
  return;
}

void add_element_dowb_matrix(DOF_DOWB_MATRIX *matrix,
			     REAL sign, int n_row, int n_col,
			     const DOF *row_dof, const DOF *col_dof, 
			     const void **el_mat, const S_CHAR *bound)
{
  FUNCNAME("add_element_dowb_matrix");
  DOF             i, j, k, irow, jcol;
  int             free_col=0;
  const REAL_DD   **el_mat_r = (const REAL_DD **)el_mat;
  const REAL_DDS  **el_mat_s = (const REAL_DDS **)el_mat;
  const REAL_D    **el_mat_d = (const REAL_D **)el_mat;
  DOWB_MATRIX_ROW *row, *free_row;

  TEST_EXIT(matrix,"no matrix\n");
  if (!col_dof || n_col <= 0)
  {
    col_dof = row_dof;
    n_col = n_row;
  }
  if (row_dof != col_dof  && bound)
    ERROR_EXIT("no boundary treatment for row_dof != col_dof\n");
    
  for (i = 0; i < n_row; i++) 
  {
    irow = row_dof[i];

    if (matrix->matrix_row[irow] == nil)
    {
      row = matrix->matrix_row[irow] =
	get_dowb_matrix_row(matrix->row_fe_space, matrix->type);
      if (row_dof == col_dof)
      {
	row->col[0] = irow;           /* first entry is diagonal element */
	if (bound && bound[i] >= DIRICHLET)
	{
	  switch(matrix->type) {
	  case dowbm_full: MSET_DOW(1.0, row->entry.full[0]); break;
	  case dowbm_symm: SMSET_DOW(1.0, &row->entry.symm[0]); break;
	  case dowbm_diag: DMSET_DOW(1.0, row->entry.diag[0]); break;
	  }
	  continue;
	}
	else {
	  switch(matrix->type) {
	  case dowbm_full: MSET_DOW(0.0, row->entry.full[0]); break;
	  case dowbm_symm: SMSET_DOW(0.0, &row->entry.symm[0]); break;
	  case dowbm_diag: DMSET_DOW(0.0, row->entry.diag[0]); break;
	  }
	}
      }
      else
      {
	row->col[0] = UNUSED_ENTRY;
      }
    }
    else if (bound && bound[i] >= DIRICHLET)
      continue;

    for (j = 0; j < n_col; j++)
    {
      jcol = col_dof[j];
      row = matrix->matrix_row[irow];
      free_row = nil;
      do 
      {
	for (k=0; k<ROW_LENGTH; k++)
	{
	  if (row->col[k] == jcol)
	  {
	    switch(matrix->type) {
	    case dowbm_full:
	      MAXPY_DOW(sign, (REAL_D *)el_mat_r[i][j], row->entry.full[k]);
	      break;
	    case dowbm_symm:
	      SMAXPY_DOW(sign, &el_mat_s[i][j], &row->entry.symm[k]);
	      break;
	    case dowbm_diag:
	      DMAXPY_DOW(sign, el_mat_d[i][j], row->entry.diag[k]);
	      break;
	    }
	    break;
	  }
	  if (ENTRY_NOT_USED(row->col[k])) 
	  {
	    free_col = k;
	    free_row = row;
	    if (row->col[k] == NO_MORE_ENTRIES) 
	    {
	      k = ROW_LENGTH;
	      break;
	    }
	  }
	}
	if (k < ROW_LENGTH) break;               /* done? */
	if (row->next || free_row)
	{
	  row = row->next;
	}
	else 
	{
	  free_row =
	    row->next = get_dowb_matrix_row(matrix->row_fe_space,matrix->type);
	  free_col = 0;
	  row = nil;
	}
      } while (row);
      
      if (k >= ROW_LENGTH)                       /* not done? */
      {                     
	free_row->col[free_col]   = jcol;
	switch (matrix->type) {
	case dowbm_full:
	  MAXEY_DOW(sign, (REAL_D *)el_mat_r[i][j], free_row->entry.full[free_col]);
	  break;
	case dowbm_symm:
	  SMAXEY_DOW(sign, &el_mat_s[i][j], &free_row->entry.symm[free_col]);
	  break;
	case dowbm_diag:
	  DMAXEY_DOW(sign, el_mat_d[i][j], free_row->entry.diag[free_col]);
	  break;
	}
      }
    }
  }
  return;
}

void add_element_vec(DOF_REAL_VEC *drv, REAL sign, int dim, const DOF *dof,
		     const REAL *vec, const S_CHAR *bound)
{
  FUNCNAME("add_element_vec");
  REAL   *drv_vec = nil;
  int    i;

  GET_DOF_VEC(drv_vec,drv);

  if (bound)
  {
    for (i = 0; i < dim; i++)
      if (bound[i] < DIRICHLET)
	drv_vec[dof[i]] += sign*vec[i];
  }
  else
  {
    for (i = 0; i < dim; i++)
      drv_vec[dof[i]] += sign*vec[i];

  }
  return;
}

void add_element_d_vec(DOF_REAL_D_VEC *drdv, REAL sign, int dim, 
		       const DOF  *dof, const REAL_D *vec, const S_CHAR *bound)
{
  FUNCNAME("add_element_d_vec");
  REAL_D *drdv_vec = nil;
  int    i, n;

  GET_DOF_VEC(drdv_vec, drdv);

  if (bound)
  {
    for (i = 0; i < dim; i++)
      if (bound[i] < DIRICHLET)
	for (n = 0; n < DIM_OF_WORLD; n++)
	  drdv_vec[dof[i]][n] += sign*vec[i][n];
  }
  else
  {
    for (i = 0; i < dim; i++)
      for (n = 0; n < DIM_OF_WORLD; n++)
	drdv_vec[dof[i]][n] += sign*vec[i][n];
  }
  return;
}


#define EL_BOUND(fct) !fct ? nil : (*fct)

typedef struct dof_admin_traverse_data2 {
  const EL_MATRIX_INFO      *el_mat_info;
  DOF_MATRIX                *matrix;

  const EL_DOWB_MATRIX_INFO *el_dowb_mat_info;
  DOF_DOWB_MATRIX           *dowb_matrix;

  const EL_VEC_INFO         *el_vec_info;
  DOF_REAL_VEC              *dof_real_vec;

  const EL_VEC_D_INFO       *el_vec_d_info;
  DOF_REAL_D_VEC            *dof_real_d_vec;

} DOF_ADMIN_TRAVERSE_DATA2;


static void update_matrix_fct(const EL_INFO *el_info, void *data)
{
  DOF_ADMIN_TRAVERSE_DATA2 *ud = (DOF_ADMIN_TRAVERSE_DATA2 *)data;
  const REAL               **mat;
  const DOF                *row_dof, *col_dof;
  const S_CHAR             *bound;
  int                      n_row, n_col;
  const EL                 *el = el_info->el;

  mat = (*ud->el_mat_info->el_matrix_fct)(el_info, ud->el_mat_info->fill_info);
  row_dof = ud->el_mat_info->get_row_dof(el, ud->el_mat_info->row_admin, nil);
  n_row = ud->el_mat_info->n_row;
  n_col = ud->el_mat_info->n_col;
  if (n_col > 0 && ud->el_mat_info->get_col_dof && ud->el_mat_info->col_admin)
  {
    col_dof = ud->el_mat_info->get_col_dof(el, ud->el_mat_info->col_admin, nil);
  }
  else
  {
    col_dof = row_dof;
    n_col   = n_row;
  }
  if (col_dof == row_dof)
    bound = EL_BOUND(ud->el_mat_info->get_bound)(el_info, nil);
  else
    bound = nil;

  add_element_matrix(ud->matrix, ud->el_mat_info->factor, n_row, n_col, row_dof,
		     col_dof, mat, bound);
  return;
}

static void update_dowb_matrix_fct(const EL_INFO *el_info, void *data)
{
  DOF_ADMIN_TRAVERSE_DATA2 *ud = (DOF_ADMIN_TRAVERSE_DATA2 *)data;
  const void    **mat;
  const DOF     *row_dof, *col_dof;
  const S_CHAR  *bound;
  int           n_row, n_col;
  const EL      *el = el_info->el;

  mat     = (*ud->el_dowb_mat_info->el_matrix_fct)
    (el_info, ud->el_dowb_mat_info->fill_info);
  row_dof = ud->el_dowb_mat_info->get_row_dof
    (el, ud->el_dowb_mat_info->row_admin, nil);
  n_row   = ud->el_dowb_mat_info->n_row;
  n_col   = ud->el_dowb_mat_info->n_col;
  if (n_col > 0 && ud->el_dowb_mat_info->get_col_dof 
      && ud->el_dowb_mat_info->col_admin)
  {
    col_dof = ud->el_dowb_mat_info->get_col_dof
      (el, ud->el_dowb_mat_info->col_admin, nil);
  }
  else
  {
    col_dof = row_dof;
    n_col   = n_row;
  }
  if (col_dof == row_dof)
    bound = EL_BOUND(ud->el_dowb_mat_info->get_bound)(el_info, nil);
  else
    bound = nil;

  add_element_dowb_matrix(ud->dowb_matrix, ud->el_dowb_mat_info->factor, 
			  n_row, n_col, row_dof, col_dof, mat, bound);
  return;
}

void update_matrix(DOF_MATRIX *dof_matrix, const EL_MATRIX_INFO *minfo)
{
  FUNCNAME("update_matrix");
  DOF_ADMIN_TRAVERSE_DATA2 td[1] = {{0}};
  FLAGS     fill_flag;

  TEST_EXIT(minfo,"no EL_MATRIX_INFO\n");
  TEST_EXIT(dof_matrix,"no DOF_MATRIX\n");
  TEST_EXIT(dof_matrix->row_fe_space,"no row fe_space in DOF_MATRIX\n");
  TEST_EXIT(dof_matrix->col_fe_space,"no column fe_space in DOF_MATRIX\n");

  TEST_EXIT(minfo->row_admin == dof_matrix->row_fe_space->admin,
	    "Row admins of EL_MATRIX_INFO and DOF_MATRIX don't match\n");
  TEST_EXIT(minfo->col_admin == dof_matrix->col_fe_space->admin,
	    "Column admins of EL_MATRIX_INFO and DOF_MATRIX don't match\n");

  TEST_EXIT(minfo->el_matrix_fct,"no el_matrix_fct in EL_MATRIX_INFO\n");

  td->matrix = dof_matrix;
  td->el_mat_info = minfo;
  if (minfo->get_bound)
    fill_flag = minfo->fill_flag|FILL_BOUND;
  else
    fill_flag = minfo->fill_flag;

  mesh_traverse(td->matrix->row_fe_space->mesh, -1, fill_flag, 
		update_matrix_fct, td);
  return;
}

void update_dowb_matrix(DOF_DOWB_MATRIX *dof_matrix,
			const EL_DOWB_MATRIX_INFO *minfo)
{
  FUNCNAME("update_dowb_matrix");
  DOF_ADMIN_TRAVERSE_DATA2 td[1] = {{0}};
  FLAGS     fill_flag;

  TEST_EXIT(minfo,"no EL_DOWB_MATRIX_INFO\n");
  TEST_EXIT(dof_matrix,"no DOF_DOWB_MATRIX\n");
  TEST_EXIT(minfo->row_admin == dof_matrix->row_fe_space->admin,
    "Row admins of EL_DOWB_MATRIX_INFO and DOF_DOWB_MATRIX don't match\n");
  TEST_EXIT(minfo->col_admin == dof_matrix->col_fe_space->admin,
    "Column admins of EL_DOWB_MATRIX_INFO and DOF_DOWB_MATRIX don't match\n");

  TEST_EXIT(minfo->el_matrix_fct,"no el_matrix_fct in EL_DOWB_MATRIX_INFO\n");


  td->dowb_matrix = dof_matrix;
  td->el_dowb_mat_info = minfo;
  if (minfo->get_bound)
    fill_flag = minfo->fill_flag|FILL_BOUND;
  else
    fill_flag = minfo->fill_flag;

  mesh_traverse(dof_matrix->row_fe_space->mesh,
		-1, fill_flag, update_dowb_matrix_fct, td);
  return;
}

static void update_vec_fct(const EL_INFO *el_info, void *data)
{
  DOF_ADMIN_TRAVERSE_DATA2 *ud = (DOF_ADMIN_TRAVERSE_DATA2 *)data;
  const REAL        *vec;
  const DOF         *dof;
  const S_CHAR      *bound;

  vec = (*ud->el_vec_info->el_vec_fct)(el_info, ud->el_vec_info->fill_info);
  dof = (*ud->el_vec_info->get_dof)(el_info->el, ud->el_vec_info->admin, nil);
  bound = EL_BOUND(ud->el_vec_info->get_bound)(el_info, nil);

  add_element_vec(ud->dof_real_vec, ud->el_vec_info->factor, 
		  ud->el_vec_info->n_dof,  dof, vec, bound);
  return;
}

void update_real_vec(DOF_REAL_VEC *drv, const EL_VEC_INFO *vec_info)
{
  FUNCNAME("update_real_vec");
  DOF_ADMIN_TRAVERSE_DATA2 td[1] = {{0}};
  FLAGS fill_flag;

  TEST_EXIT(vec_info,"no EL_VEC_INFO\n");
  TEST_EXIT(vec_info->el_vec_fct,"no el_vec_fct in EL_VEC_INFO\n");
  TEST_EXIT(drv,"no DOF_REAL_VEC\n");

  td->dof_real_vec = drv;
  td->el_vec_info = vec_info;

  if (vec_info->get_bound)
    fill_flag = vec_info->fill_flag|FILL_BOUND;
  else
    fill_flag = vec_info->fill_flag;

  mesh_traverse(drv->fe_space->mesh, -1, fill_flag, update_vec_fct, td);
  return;
}

static void update_vec_d_fct(const EL_INFO *el_info, void *data)
{
  DOF_ADMIN_TRAVERSE_DATA2 *ud = (DOF_ADMIN_TRAVERSE_DATA2 *)data;
  const REAL_D   *vec;
  const DOF      *dof;
  const S_CHAR   *bound;

  vec   = (*ud->el_vec_d_info->el_vec_fct)
    (el_info, ud->el_vec_d_info->fill_info);
  dof   = (*ud->el_vec_d_info->get_dof)
    (el_info->el, ud->el_vec_d_info->admin, nil);
  bound = EL_BOUND(ud->el_vec_d_info->get_bound)(el_info, nil);

  add_element_d_vec(ud->dof_real_d_vec, ud->el_vec_d_info->factor,
		    ud->el_vec_d_info->n_dof, dof, vec, bound);
  return;
}

void update_real_d_vec(DOF_REAL_D_VEC *drdv, const EL_VEC_D_INFO *vecd_info)
{
  FUNCNAME("update_real_d_vec");
  DOF_ADMIN_TRAVERSE_DATA2 td[1] = {{0}};
  FLAGS    fill_flag;

  TEST_EXIT(vecd_info,"no EL_VEC_D_INFO\n");
  TEST_EXIT(vecd_info->el_vec_fct,"no el_vec_fct in EL_VEC_D_INFO\n");
  TEST_EXIT(drdv,"no DOF_REAL_D_VEC\n");

  td->el_vec_d_info = vecd_info;
  td->dof_real_d_vec = drdv;
  if (vecd_info->get_bound)
    fill_flag = vecd_info->fill_flag|FILL_BOUND;
  else
    fill_flag = vecd_info->fill_flag;

  mesh_traverse(drdv->fe_space->mesh, -1, fill_flag, update_vec_d_fct, td);
  return;
}

/*--------------------------------------------------------------------------*/
/* attempt to find a DOF_ADMIN structure which stores vertex DOFs           */
/*--------------------------------------------------------------------------*/

const DOF_ADMIN *get_vertex_admin(MESH *mesh)
{
  int       i, n_admin = mesh->n_dof_admin;
  DOF_ADMIN **admins = mesh->dof_admin;
  const DOF_ADMIN *admin = nil;

  for (i = 0; i < n_admin; i++) {
    if (admins[i]->n_dof[VERTEX]) {
      if (!admin)
	admin = admins[i];
      else if (admins[i]->size < admin->size)
	admin = admins[i];
    }
  }

  if(!admin) {
    const int n_dof[N_NODE_TYPES] = {1,0,0,0};
    const FE_SPACE *fe_space = nil;

    fe_space = get_fe_space(mesh, "Vertex DOF admin", n_dof, nil, 0);
    admin = fe_space->admin;

    MEM_FREE(fe_space, 1, FE_SPACE);
  }

  return(admin);
}


