/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include "gaussian.h"

static double lookup_pixel( double *image, int x, int y, int width,
                            int height, int component )
{
    if( x < 0 ) x = 0;
    if( y < 0 ) y = 0;
    if( x >= width ) x = width - 1;
    if( y >= height ) y = height - 1;
    return image[ (y * width * 4) + (x * 4) + component ];
}

static double lookup_pixel_plane( double *image, int x, int y, int width, int height )
{
    if( x < 0 ) x = 0;
    if( y < 0 ) y = 0;
    if( x >= width ) x = width - 1;
    if( y >= height ) y = height - 1;
    return image[ (y * width) + x ];
}

static int build_gaussian( double *output, double size )
{
    double sigma;
    double sigma2;
    double l;
    int length, n, i;

    sigma = sqrt( -(size * size) / (2.0 * log( 1.0 / 255.0 )) );
    sigma2 = (2.0 * sigma * sigma);
    l = sqrt( -sigma2 * log( 1.0 / 255.0 ) );

    n = ceil( l ) * 2.0;
    if( (n % 2) == 0 ) n++;
    length = n / 2;

    for( i = 0; i < (length + 1); i++ ) {
        double temp = exp( -(i * i) / sigma2 );
        output[ i ] = temp;
    }

    return length + 1;
}

void gaussian_blur( double *dst, double *src, int width, int height,
                    double blur_width )
{
    int gwidth = (int) (blur_width + 1.0);
    double *gaussian = malloc( gwidth * 2 * sizeof( double ) );
    double *dst1 = malloc( width * height * 4 * sizeof( double ) );
    int kwidth;
    double total = 0.0;
    int i, x, y;

    gwidth = build_gaussian( gaussian, blur_width );

    kwidth = (gwidth * 2) - 1;

    for( i = 0; i < kwidth; i++ ) {
        int xpos = i - gwidth + 1;
        if( xpos < 0 ) xpos = -xpos;
        total += gaussian[ xpos ];
    }

    fprintf( stderr, "gaussian: Pass 1 (horizontal).\n" );
    for( y = 0; y < height; y++ ) {
        double *curout = dst1 + (y * width * 4);

        fprintf( stderr, "gaussian: %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r = 0.0;
            double g = 0.0;
            double b = 0.0;
            double a = 0.0;

            for( i = 0; i < kwidth; i++ ) {
                int xpos = i - gwidth + 1;
                double w;
                if( xpos < 0 ) xpos = -xpos;
                w = gaussian[ xpos ];
                r += lookup_pixel( src, x + i - (gwidth - 1), y, width, height, 0 ) * w;
                g += lookup_pixel( src, x + i - (gwidth - 1), y, width, height, 1 ) * w;
                b += lookup_pixel( src, x + i - (gwidth - 1), y, width, height, 2 ) * w;
                a += lookup_pixel( src, x + i - (gwidth - 1), y, width, height, 3 ) * w;
            }
            curout[ (x * 4) + 0 ] = r / total;
            curout[ (x * 4) + 1 ] = g / total;
            curout[ (x * 4) + 2 ] = b / total;
            curout[ (x * 4) + 3 ] = a / total;
        }
    }

    fprintf( stderr, "\ngaussian: Pass 2 (vertical).\n" );
    for( y = 0; y < height; y++ ) {
        double *curout = dst + (y * width * 4);

        fprintf( stderr, "gaussian: %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r = 0.0;
            double g = 0.0;
            double b = 0.0;
            double a = 0.0;

            for( i = 0; i < kwidth; i++ ) {
                int xpos = i - gwidth + 1;
                double w;
                if( xpos < 0 ) xpos = -xpos;
                w = gaussian[ xpos ];
                r += lookup_pixel( dst1, x, y + i - (gwidth - 1), width, height, 0 ) * w;
                g += lookup_pixel( dst1, x, y + i - (gwidth - 1), width, height, 1 ) * w;
                b += lookup_pixel( dst1, x, y + i - (gwidth - 1), width, height, 2 ) * w;
                a += lookup_pixel( dst1, x, y + i - (gwidth - 1), width, height, 3 ) * w;
            }
            curout[ (x * 4) + 0 ] = r / total;
            curout[ (x * 4) + 1 ] = g / total;
            curout[ (x * 4) + 2 ] = b / total;
            curout[ (x * 4) + 3 ] = a / total;
        }
    }
    fprintf( stderr, "\ngaussian: Done.\n" );

    free( gaussian );
    free( dst1 );
}

