﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/eks/EKSRequest.h>
#include <aws/eks/EKS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace EKS {
namespace Model {

/**
 */
class ListUpdatesRequest : public EKSRequest {
 public:
  AWS_EKS_API ListUpdatesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListUpdates"; }

  AWS_EKS_API Aws::String SerializePayload() const override;

  AWS_EKS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the Amazon EKS cluster to list updates for.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ListUpdatesRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the Amazon EKS managed node group to list updates for.</p>
   */
  inline const Aws::String& GetNodegroupName() const { return m_nodegroupName; }
  inline bool NodegroupNameHasBeenSet() const { return m_nodegroupNameHasBeenSet; }
  template <typename NodegroupNameT = Aws::String>
  void SetNodegroupName(NodegroupNameT&& value) {
    m_nodegroupNameHasBeenSet = true;
    m_nodegroupName = std::forward<NodegroupNameT>(value);
  }
  template <typename NodegroupNameT = Aws::String>
  ListUpdatesRequest& WithNodegroupName(NodegroupNameT&& value) {
    SetNodegroupName(std::forward<NodegroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The names of the installed add-ons that have available updates.</p>
   */
  inline const Aws::String& GetAddonName() const { return m_addonName; }
  inline bool AddonNameHasBeenSet() const { return m_addonNameHasBeenSet; }
  template <typename AddonNameT = Aws::String>
  void SetAddonName(AddonNameT&& value) {
    m_addonNameHasBeenSet = true;
    m_addonName = std::forward<AddonNameT>(value);
  }
  template <typename AddonNameT = Aws::String>
  ListUpdatesRequest& WithAddonName(AddonNameT&& value) {
    SetAddonName(std::forward<AddonNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the capability for which you want to list updates.</p>
   */
  inline const Aws::String& GetCapabilityName() const { return m_capabilityName; }
  inline bool CapabilityNameHasBeenSet() const { return m_capabilityNameHasBeenSet; }
  template <typename CapabilityNameT = Aws::String>
  void SetCapabilityName(CapabilityNameT&& value) {
    m_capabilityNameHasBeenSet = true;
    m_capabilityName = std::forward<CapabilityNameT>(value);
  }
  template <typename CapabilityNameT = Aws::String>
  ListUpdatesRequest& WithCapabilityName(CapabilityNameT&& value) {
    SetCapabilityName(std::forward<CapabilityNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>nextToken</code> value returned from a previous paginated request,
   * where <code>maxResults</code> was used and the results exceeded the value of
   * that parameter. Pagination continues from the end of the previous results that
   * returned the <code>nextToken</code> value. This value is null when there are no
   * more results to return.</p>  <p>This token should be treated as an opaque
   * identifier that is used only to retrieve the next items in a list and not for
   * other programmatic purposes.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListUpdatesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results, returned in paginated output. You receive
   * <code>maxResults</code> in a single page, along with a <code>nextToken</code>
   * response element. You can see the remaining results of the initial request by
   * sending another request with the returned <code>nextToken</code> value. This
   * value can be between 1 and 100. If you don't use this parameter, 100 results and
   * a <code>nextToken</code> value, if applicable, are returned.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListUpdatesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_nodegroupName;

  Aws::String m_addonName;

  Aws::String m_capabilityName;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_nameHasBeenSet = false;
  bool m_nodegroupNameHasBeenSet = false;
  bool m_addonNameHasBeenSet = false;
  bool m_capabilityNameHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EKS
}  // namespace Aws
