"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const helper_1 = require("../helper");
const hoverProvider_1 = require("../../src/providers/hoverProvider");
function get_hover_value(hover) {
    if (hover) {
        if (Array.isArray(hover)) {
            return "";
        }
        else {
            if (Object.hasOwn(hover.contents, "value")) {
                return hover.contents["value"];
            }
        }
    }
    return "";
}
function testPlayKeywords(context, textDoc) {
    const tests = [
        {
            word: "name",
            position: { line: 0, character: 4 },
            doc: "Identifier. Can be used for documentation, or in tasks/handlers.",
        },
        {
            word: "host",
            position: { line: 1, character: 4 },
            doc: "A list of groups, hosts or host pattern that translates into a list of hosts that are the play’s target.",
        },
        {
            word: "tasks",
            position: { line: 3, character: 4 },
            doc: "Main list of tasks to execute in the play, they run after roles and before post_tasks.",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            if (actualHover) {
                (0, chai_1.expect)(get_hover_value(actualHover)).includes(doc);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
    });
}
function testTaskKeywords(context, textDoc) {
    const tests = [
        {
            word: "register",
            position: { line: 6, character: 8 },
            doc: "Name of variable that will contain task status and module return data.",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            if (actualHover) {
                (0, chai_1.expect)(get_hover_value(actualHover)).includes(doc);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
    });
}
function testBlockKeywords(context, textDoc) {
    const tests = [
        {
            word: "become",
            position: { line: 11, character: 8 },
            doc: "Boolean that controls if privilege escalation is used or not on Task execution. Implemented by the become plugin.",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            if (actualHover) {
                (0, chai_1.expect)(get_hover_value(actualHover)).includes(doc);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
    });
}
function testRoleKeywords(context, textDoc) {
    const tests = [
        {
            word: "tags",
            position: { line: 6, character: 8 },
            doc: "Tags applied to the task or included tasks, this allows selecting subsets of tasks from the command line.",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            if (actualHover) {
                (0, chai_1.expect)(get_hover_value(actualHover)).includes(doc);
            }
            else {
                (0, chai_1.expect)(false);
            }
        });
    });
}
function testModuleNames(context, textDoc) {
    const tests = [
        {
            word: "ansible.builtin.debug",
            position: { line: 4, character: 8 },
            doc: "Print statements during execution",
        },
        {
            word: "ansible.builtin.debug -> msg",
            position: { line: 5, character: 10 },
            doc: "The customized message that is printed\\. If omitted\\, prints a generic message\\.",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            (0, chai_1.expect)(get_hover_value(actualHover)).includes(doc);
        });
    });
}
function testNoHover(context, textDoc) {
    it("should not provide hovering for values", async function () {
        const actualHover = await (0, hoverProvider_1.doHover)(textDoc, { line: 13, character: 24 }, await context.docsLibrary);
        (0, chai_1.expect)(actualHover).to.be.null;
    });
    it("should not provide hovering for improper module name and options", async function () {
        const actualHover = await (0, hoverProvider_1.doHover)(textDoc, { line: 13, character: 8 }, await context.docsLibrary);
        (0, chai_1.expect)(actualHover).to.be.null;
    });
    it("should not provide hovering for improper module option", async function () {
        const actualHover = await (0, hoverProvider_1.doHover)(textDoc, { line: 14, character: 10 }, await context.docsLibrary);
        (0, chai_1.expect)(actualHover).to.be.null;
    });
}
function testPlaybookAdjacentCollection(context, textDoc) {
    const tests = [
        {
            word: "playbook adjacent module name",
            position: { line: 5, character: 19 },
            doc: "This is a test module for playbook adjacent collection",
        },
        {
            word: "playbook adjacent module option",
            position: { line: 6, character: 11 },
            doc: "Option 1",
        },
        {
            word: "playbook adjacent module sub option",
            position: { line: 7, character: 19 },
            doc: "Sub option 1",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            (0, chai_1.expect)(get_hover_value(actualHover)).includes(doc);
        });
    });
}
function testNonPlaybookAdjacentCollection(context, textDoc) {
    const tests = [
        {
            word: "non playbook adjacent module name",
            position: { line: 5, character: 19 },
            doc: "",
        },
        {
            word: "non playbook adjacent module option",
            position: { line: 6, character: 11 },
            doc: "",
        },
    ];
    tests.forEach(({ word, position, doc }) => {
        it(`should not provide hovering for '${word}'`, async function () {
            const actualHover = await (0, hoverProvider_1.doHover)(textDoc, position, await context.docsLibrary);
            if (!doc) {
                (0, chai_1.expect)(actualHover).to.be.null;
            }
            else {
                (0, chai_1.expect)(get_hover_value(actualHover), `actual hover -> ${actualHover}`).includes(doc);
            }
        });
    });
}
describe("doHover()", () => {
    const workspaceManager = (0, helper_1.createTestWorkspaceManager)();
    let fixtureFilePath = "hover/tasks.yml";
    let fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    let context = workspaceManager.getContext(fixtureFileUri);
    let textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    let docSettings = context?.documentSettings.get(textDoc.uri);
    (0, chai_1.expect)(docSettings !== undefined);
    describe("Play keywords hover", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testPlayKeywords(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testPlayKeywords(context, textDoc);
            }
        });
    });
    describe("Task keywords hover", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testTaskKeywords(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testTaskKeywords(context, textDoc);
            }
        });
    });
    describe("Block keywords hover", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testBlockKeywords(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testBlockKeywords(context, textDoc);
            }
        });
    });
    fixtureFilePath = "hover/roles.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context?.documentSettings.get(textDoc.uri);
    describe("Role keywords hover", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testRoleKeywords(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testRoleKeywords(context, textDoc);
            }
        });
    });
    fixtureFilePath = "hover/tasks.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context?.documentSettings.get(textDoc.uri);
    describe("Module name and options hover", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testModuleNames(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testModuleNames(context, textDoc);
            }
        });
    });
    describe("No hover", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testNoHover(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testNoHover(context, textDoc);
            }
        });
    });
    fixtureFilePath = "playbook_adjacent_collection/playbook.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context?.documentSettings.get(textDoc.uri);
    describe("Hover for playbook adjacent collection", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testPlaybookAdjacentCollection(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testPlaybookAdjacentCollection(context, textDoc);
            }
        });
    });
    fixtureFilePath =
        "playbook_adjacent_collection/non_adjacent_playbooks/playbook2.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context?.documentSettings.get(textDoc.uri);
    describe("Negate hover for non playbook adjacent collection", () => {
        describe("With EE enabled @ee", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testNonPlaybookAdjacentCollection(context, textDoc);
            }
            after(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", () => {
            before(async () => {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testNonPlaybookAdjacentCollection(context, textDoc);
            }
        });
    });
});
