package stellarnet

import (
	"testing"

	"github.com/stellar/go/xdr"
)

var summarytests = []struct {
	b64xdr         string
	summaryPresent string
	summaryPast    string
}{
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAAAAAAA4q3J1m55u5ZBlCFTqY97Sxzg6CFdwaFas/z1jXxaBSQAAAAAC+vCAAAAAAAA",
		"Create account GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT with starting balance of 20.0000000 XLM",
		"Created account GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT with starting balance of 20.0000000 XLM",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAEAAAAA4q3J1m55u5ZBlCFTqY97Sxzg6CFdwaFas/z1jXxaBSQAAAAAAAAAABHhowAAAAAA",
		"Pay 30.0000000 XLM to account GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Paid 30.0000000 XLM to account GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAIAAAABQUJDRAAAAADircnWbnm7lkGUIVOpj3tLHODoIV3BoVqz/PWNfFoFJAAAAAB3NZQAAAAAAOKtydZuebuWQZQhU6mPe0sc4OghXcGhWrP89Y18WgUkAAAAAUVGR0gAAAAA4q3J1m55u5ZBlCFTqY97Sxzg6CFdwaFas/z1jXxaBSQAAAAAB7+kgAAAAAAAAAAA",
		"Pay 13.0000000 EFGH/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT to account GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT using at most 200.0000000 ABCD/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Paid 13.0000000 EFGH/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT to account GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT using at most 200.0000000 ABCD/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAMAAAABQUJDRAAAAADircnWbnm7lkGUIVOpj3tLHODoIV3BoVqz/PWNfFoFJAAAAAFFRkdIAAAAAOKtydZuebuWQZQhU6mPe0sc4OghXcGhWrP89Y18WgUkAAAAADsCM4AAAABDAAAAMgAAAAAAAAAAAAAAAAAA",
		"Create offer selling 99.0000000 ABCD/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT for 1.3400000 to buy EFGH/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Created offer selling 99.0000000 ABCD/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT for 1.3400000 to buy EFGH/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAQAAAAAAAAAAVFXRVIAAAAA4q3J1m55u5ZBlCFTqY97Sxzg6CFdwaFas/z1jXxaBSQAAAAAEUkMgAAAAc0AAAAFAAAAAAAA",
		"Create passive offer selling 29.0000000 XLM for 92.2000000 to buy QWER/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Created passive offer selling 29.0000000 XLM for 92.2000000 to buy QWER/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAUAAAABAAAAAOKtydZuebuWQZQhU6mPe0sc4OghXcGhWrP89Y18WgUkAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
		"Set inflation destination to GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Set inflation destination to GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAYAAAABQkxBSAAAAADircnWbnm7lkGUIVOpj3tLHODoIV3BoVqz/PWNfFoFJH//////////AAAAAAAA",
		"Establish trust line to BLAH/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Established trust line to BLAH/GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAAHN+b9x5HwmNAmIgPPfK5P/YZFHjQkwp3njikB8qNRyXAAAAZAFb5rMAAAAlAAAAAAAAAAAAAAABAAAAAAAAAAcAAAAA4q3J1m55u5ZBlCFTqY97Sxzg6CFdwaFas/z1jXxaBSQAAAABSEFBQQAAAAEAAAAA",
		"Authorize trustline to HAAA for GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
		"Authorized trustline to HAAA for GDRK3SOWNZ43XFSBSQQVHKMPPNFRZYHIEFO4DIK2WP6PLDL4LICSIPAT",
	},
	{
		"AAAAACSsWe+Gqyd5TUPRGt2SkM2V4zD3xDmut3XwM2vSfn3iAAAnEAETAZMAAAF7AAAAAAAAAAEAAAAOZm9yIHRlc3RpbmcgdWkAAAAAAAEAAAABAAAAACSsWe+Gqyd5TUPRGt2SkM2V4zD3xDmut3XwM2vSfn3iAAAADQAAAAAAAAAABfXhAAAAAABzfm/ceR8JjQJiIDz3yuT/2GRR40JMKd544pAfKjUclwAAAAAAAAAABfXhAAAAAAAAAAAA",
		"[Source account GASKYWPPQ2VSO6KNIPIRVXMSSDGZLYZQ67CDTLVXOXYDG26SPZ66EDCQ] Pay at least 10.0000000 XLM to account GBZX4364PEPQTDICMIQDZ56K4T75QZCR4NBEYKO6PDRJAHZKGUOJPCXB using 10.0000000 XLM",
		"[Source account GASKYWPPQ2VSO6KNIPIRVXMSSDGZLYZQ67CDTLVXOXYDG26SPZ66EDCQ] Paid at least 10.0000000 XLM to account GBZX4364PEPQTDICMIQDZ56K4T75QZCR4NBEYKO6PDRJAHZKGUOJPCXB using 10.0000000 XLM",
	},
}

func TestOpSummary(t *testing.T) {
	for i, test := range summarytests {
		var tx xdr.Transaction
		if err := xdr.SafeUnmarshalBase64(test.b64xdr, &tx); err != nil {
			t.Errorf("test %d: unmarshal error %s", i, err)
			continue
		}
		if len(tx.Operations) != 1 {
			t.Errorf("test %d: expected 1 operation, got %d", i, len(tx.Operations))
			continue
		}
		summaryPresent := OpSummary(tx.Operations[0], false)
		if summaryPresent != test.summaryPresent {
			t.Errorf("test %d: summaryPresent: %q, expected %q", i, summaryPresent, test.summaryPresent)
			continue
		}
		summaryPast := OpSummary(tx.Operations[0], true)
		if summaryPast != test.summaryPast {
			t.Errorf("test %d: summaryPast: %q, expected %q", i, summaryPast, test.summaryPast)
			continue
		}
	}
}
