" Vim syntax file
" Language:     Syd v3 profiles
" Author:       Ali Polatel
" Copyright:    Copyright (c) 2024, 2025 Ali Polatel
" Licence:      You may redistribute this under the same terms as Vim itself
"
" Syntax highlighting for Syd version 3 profiles.
"

if &compatible || v:version < 700
    finish
endif

if exists("b:current_syntax")
    finish
endif

syn match Syd3ConfigError /.\+/
hi def link Syd3ConfigError Error

syn keyword Syd3ConfigTodo TODO FIXME XXX contained
hi def link Syd3ConfigTodo Todo

syn keyword Syd3ConfigHeaderKey PROG ARGS DATE
      \ contained nextgroup=Syd3ConfigHeaderSep skipwhite
hi def link Syd3ConfigHeaderKey SpecialComment

syn match Syd3ConfigHeaderSep /:/
      \ contained nextgroup=Syd3ConfigCmd skipwhite
hi def link Syd3ConfigHeaderSep Delimiter

syn match Syd3ConfigCmd /\s*\zs.\+/
      \ contained contains=Syd3DateStamp
hi def link Syd3ConfigCmd PreProc

syn match Syd3DateStamp /\<\d\{8}T\d\{6}Z\>/ contained
hi def link Syd3DateStamp Number

syn match Syd3ConfigHeaderValue /:\s*\zs.\+$/ contained contains=Syd3DateStamp
hi def link Syd3ConfigHeaderValue Constant

syn match Syd3ConfigHeaderTitle /\v\#\s*\zsSyd profile generated by .+$/ contained
hi def link Syd3ConfigHeaderTitle WarningMsg

syn region Syd3Comment start=/^#/ end=/$/
      \ contains=Syd3ConfigTodo,Syd3ConfigHeaderKey,Syd3ConfigHeaderSep,Syd3ConfigHeaderValue,Syd3ConfigHeaderTitle,Syd3DateStamp
hi def link Syd3Comment Comment

syn match Syd3Boolean "\<\(1\|on\|t\|tr\|tru\|true\|✓\|0\|off\|f\|fa\|fal\|fals\|false\|✗\)\>" contained
hi def link Syd3Boolean Boolean

syn keyword Syd3None none off contained
hi def link Syd3None Type

syn match Syd3Size /\v[0-9]+[kKmMgGtTpP]?[bB]?/ contained
hi def link Syd3Size Number

syn match Syd3LockState "\<\(on\|off\|exec\|ipc\|read-only\|readonly\|read\|ro\|1\|0\|x\|r\|i\)\>" contained
hi def link Syd3LockState Boolean

syn keyword Syd3LockType lock contained containedin=Syd3Lock
hi def link Syd3LockType Identifier

syn keyword Syd3IpcType ipc contained containedin=Syd3Ipc
hi def link Syd3IpcType Identifier

syn match Syd3IpcSocket /\v.+$/ contained
hi link Syd3IpcSocket String

syn keyword Syd3DisallowedCommandsType ghost load panic contained
hi def link Syd3DisallowedCommandsType Error

syn match Syd3LinkFam /\v(all|route|usersock|firewall|sock_diag|nflog|xfrm|selinux|iscsi|audit|fib_lookup|connector|netfilter|ip6_fw|dnrtmsg|kobject_uevent|generic|scsitransport|ecryptfs|rdma|crypto|smc)/
      \ contained containedin=Syd3Link
hi link Syd3LinkFam Special

syn match Syd3LinkType /\vallow\/net\/link/
      \ contained containedin=Syd3Link nextgroup=Syd3RuleOps,Syd3LinkFam
hi link Syd3LinkType Identifier

syn match Syd3DefaultAct /\v(warn|filter|deny|panic|stop|abort|kill|exit)$/
      \ contained
hi link Syd3DefaultAct Special

syn match Syd3DefaultType /\vdefault\/(all|rpath|wpath|cpath|dpath|spath|tpath|fown|fattr|inet|crypt|exec|force|lock|mem|pid|proxy|tpe|fs|walk|stat|read|write|exec|ioctl|create|delete|rename|symlink|truncate|chdir|readdir|mkdir|rmdir|chown|chgrp|chmod|chattr|chroot|utime|mkdev|mkfifo|mktemp|net\/bind|net\/connect|net\/sendfd|net)/
      \ contained containedin=Syd3Default
hi link Syd3DefaultType Identifier

syn match Syd3SandboxType /\vsandbox\/(all|rpath|wpath|cpath|dpath|spath|tpath|fown|fattr|inet|crypt|exec|force|lock|mem|pid|proxy|pty|tpe|fs|walk|stat|read|write|exec|ioctl|create|delete|rename|symlink|truncate|chdir|readdir|mkdir|rmdir|chown|chgrp|chmod|chattr|chroot|utime|mkdev|mkfifo|mktemp|net\/bind|net\/connect|net\/sendfd|net)(,(all|rpath|wpath|cpath|dpath|spath|tpath|fown|fattr|inet|crypt|exec|force|lock|mem|pid|proxy|pty|tpe|fs|walk|stat|read|write|exec|ioctl|create|delete|rename|symlink|truncate|chdir|readdir|mkdir|rmdir|chown|chgrp|chmod|chattr|chroot|utime|mkdev|mkfifo|mkdev|net\/bind|net\/connect|net\/sendfd|net))*/
      \ contained containedin=Syd3Sandbox
hi link Syd3SandboxType Identifier

syn match Syd3UnshareType /\vunshare\/(mount|uts|ipc|user|pid|net|cgroup|time)(,(mount|uts|ipc|user|pid|net|cgroup|time))*/
      \ contained containedin=Syd3Unshare
hi link Syd3UnshareType Identifier

syn match Syd3ChrootType /\vroot/
      \ contained containedin=Syd3Chroot nextgroup=Syd3OpSet
hi link Syd3ChrootType Identifier

syn match Syd3RootType /\vroot\/(fake|map)/
      \ contained containedin=Syd3Root
hi link Syd3RootType Identifier

syn match Syd3UtsType /\vuts\/(host|domain)/
      \ contained containedin=Syd3Uts
hi link Syd3UtsType Identifier

syn match Syd3KillType /\v(mem|pid)\/kill/
      \ contained containedin=Syd3Kill
hi link Syd3KillType Identifier

syn match Syd3MemType /\vmem\/(max|vm_max)/
      \ contained containedin=Syd3Mem
hi link Syd3MemType Identifier

syn match Syd3PidType "pid/max"
      \ contained containedin=Syd3Pid
hi link Syd3PidType Identifier

syn match Syd3TpeType /\vtpe\/(negate|root_owned|user_owned|root_mount)/
      \ contained containedin=Syd3Tpe
hi link Syd3TpeType Identifier

syn match Syd3TpeGidType /\vtpe\/gid/
      \ contained containedin=Syd3TpeGid
hi link Syd3TpeGidType Identifier

syn match Syd3PidMax /\v[0-9]+$/
      \ contained containedin=Syd3Pid
hi link Syd3PidMax Number

syn keyword Syd3GidNone none contained containedin=Syd3Gid
hi def link Syd3GidNone Special

syn match Syd3Gid /\v(none|[0-9]+)$/
      \ contained contains=Syd3None containedin=Syd3TpeGid
hi link Syd3Gid Number

syn match Syd3BindType /\vbind[-+^]/
      \ contained contains=SydRuleOps nextgroup=Syd3BindSrc
hi link Syd3BindType Identifier

" Delimiters and Operators
syn match Syd3OpSet /\v:/ contained
hi link Syd3OpSet Delimiter

syn match Syd3OpCmd /\v!/ contained
hi link Syd3OpCmd Delimiter

syn match Syd3OpNext /\v,/ contained
hi link Syd3OpNext Delimiter

syn match Syd3OpEq /\v\=/ contained
hi link Syd3OpEq Operator

syn match Syd3OpAdd /\v\+/ contained
hi link Syd3OpAdd Operator

syn match Syd3OpDec /\v-/ contained
hi link Syd3OpDec Operator

syn match Syd3OpXor /\v\^/ contained
hi link Syd3OpXor Operator

syn match Syd3RuleOps /\v[-+^]/ contained
hi link Syd3RuleOps Operator

syn match Syd3OpVal /\v\=[^0-9,]+/ contained contains=Syd3OpEq
hi link Syd3OpVal String

syn keyword Syd3BindFs contained
      \ ext4 xfs btrfs f2fs zfs
      \ nfs cifs iso9660 squashfs
      \ tmpfs vfat proc sysfs cgroup2
      \ overlay overlayfs
      \ efivarfs configfs
      \ securityfs debugfs
hi link Syd3BindFs Type

syn match Syd3BindSrc /\v[^:]+:/
      \ contained contains=Syd3OpSet,Syd3BindFs nextgroup=Syd3BindDst
hi link Syd3BindSrc String

syn match Syd3BindDst /\v\/[^:]*/
      \ contained nextgroup=Syd3BindOpts
hi link Syd3BindDst String

syn match Syd3BindOpt /\v(ro|nodev|noexec|nosuid|noatime|nodiratime|relatime)/
      \ contained
hi link Syd3BindOpt Special

syn match Syd3BindArg /\v([ug]id|mode|size|lowerdir|upperdir|workdir)\=[^,]+/
      \ contained contains=Syd3OpEq,Syd3OpNext,Syd3Size,Syd3OpVal
hi link Syd3BindArg Special

syn match Syd3BindOpts /\v:([^,]+)(,[^,]+)*$/
      \ contained contains=Syd3OpSet,Syd3BindOpt,Syd3BindArg
hi link Syd3BindOpts None

syn match Syd3CryptType /\vcrypt/ contained
hi link Syd3CryptType Identifier

syn match Syd3CryptTmpType /\vcrypt\/tmp/ contained
hi link Syd3CryptTmpType Identifier

syn match Syd3CryptKeyType /\vcrypt\/key/ contained
hi link Syd3CryptKeyType Identifier

syn match Syd3CryptKeyHex /\v:([0-9a-fA-F]{64})$/
      \ contained contains=Syd3OpSet
hi link Syd3CryptKeyHex Number

syn match Syd3ForceType /\vforce/ contained
hi link Syd3ForceType Identifier

syn match Syd3ForceAddType /\vforce\+/
      \ contained contains=Syd3OpAdd nextgroup=Syd3ForcePath
hi link Syd3ForceAddType Identifier

syn match  Syd3ForcePath /\v\/[^:]+/
      \ contained contains=NONE nextgroup=Syd3OpSet
hi link Syd3ForcePath String

syn match Syd3ForceHash /\v([0-9a-fA-F]{128}|[0-9a-fA-F]{96}|[0-9a-fA-F]{64}|[0-9a-fA-F]{40}|[0-9a-fA-F]{32}|[0-9a-fA-F]{16}|[0-9a-fA-F]{8})/
      \ contained contains=NONE nextgroup=Syd3OpSet,Syd3ForceAct
hi link Syd3ForceHash Number

syn match Syd3ForceAct /\v(warn|filter|deny|panic|stop|abort|kill|exit)$/
      \ contained
hi link Syd3ForceAct Special

syn match Syd3IoctlType /\v(allow|deny)\/ioctl/
      \ contained nextgroup=Syd3IoctlOp
hi link Syd3IoctlType Identifier

syn match Syd3IoctlOp /\v[-+]/ contained
      \ nextgroup=Syd3IoctlNum,Syd3IoctlConst
hi link Syd3IoctlOp Operator

syn match Syd3IoctlNum /\v(0x[0-9A-Fa-f]+|0o[0-7]+|\d+)/
      \ contained
hi def link Syd3IoctlNum Number

" Match ioctl request names (FIOCLEX, FS_IOC_*, TIOC*, etc.)
" with optional exclamation mark suffix for portability.
syn match Syd3IoctlConst /\v<[A-Z][A-Z0-9_]+(!)?/
      \ contained
hi link Syd3IoctlConst Special

syn match Syd3SegvGuardType /\vsegvguard\/(expiry|suspension|max_crashes)/ contained
hi link Syd3SegvGuardType Identifier

syn match Syd3MaskType /\vmask[-+^]/
      \ contained contains=Syd3RuleOps nextgroup=Syd3SetPath
hi link Syd3MaskType Identifier

syn match Syd3SetIdUGid /\v[0-9]+/ contained
hi link Syd3SetIdUGid Number

syn match Syd3SetIdUGidBadDst /\v(0|root)$/ contained
hi link Syd3SetIdUGidBadDst Error

syn match Syd3SetIdAddDelDst /\v[^:]+$/
      \ contained
      \ contains=Syd3SetIdUGidBadDst,Syd3SetIdUGid
hi link Syd3SetIdAddDelDst Type

syn match Syd3SetIdAddDelSrc /\v[^:]+:/
      \ contained
      \ contains=Syd3OpSet,Syd3SetIdUGid
      \ nextgroup=Syd3SetIdAddDelDst
hi link Syd3SetIdAddDelSrc Type

syn match Syd3SetIdRemName /\v[^:]*/ contained contains=Syd3OpSet,Syd3SetIdUGid
hi link Syd3SetIdRemName Type

syn match Syd3SetIdAddDelAct /\v[-+]/ contained nextgroup=Syd3SetIdAddDelSrc
hi link Syd3SetIdAddDelAct Operator

syn match Syd3SetIdRemAct /\v\^/ contained nextgroup=Syd3SetIdRemName
hi link Syd3SetIdRemAct Operator

syn match Syd3SetIdAddDelType /\vset[ug]id/ contained nextgroup=Syd3SetIdAddDelAct
hi link Syd3SetIdAddDelType Identifier

syn match Syd3SetIdRemType /\vset[ug]id/ contained nextgroup=Syd3SetIdRemAct
hi link Syd3SetIdRemType Identifier

syn match Syd3Env /\v.+$/ contained
hi link Syd3Rule String

syn match Syd3SetEnvType /\vsetenv/ contained nextgroup=Syd3OpCmd,Syd3Env
hi link Syd3SetEnvType Identifier

syn match Syd3UnsetEnvType /\vunsetenv/ contained nextgroup=Syd3OpCmd,Syd3Env
hi link Syd3UnsetEnvType Identifier

syn match Syd3ClearEnvType /\vclearenv/ contained nextgroup=Syd3OpCmd
hi link Syd3ClearEnvType Identifier

syn match Syd3RulePath /\v.+$/ contained
hi link Syd3RulePath String

syn match Syd3RuleFsType /\v(allow|warn|filter|deny|panic|stop|abort|kill|exit)\/(all|lpath|rpath|wpath|cpath|dpath|spath|tpath|fown|fattr|fs|walk|stat|read|write|exec|ioctl|create|delete|rename|symlink|truncate|chdir|readdir|mkdir|rmdir|chown|chgrp|chmod|chattr|chroot|utime|mkdev|mkfifo|mktemp|net|inet|bnet|cnet|snet)(,all|,lpath|,rpath|,wpath|,cpath|,dpath|,spath|,tpath|,fown|,fattr|,walk|,stat|,read|,write|,exec|,ioctl|,create|,delete|,rename|,symlink|,truncate|,chdir|,readdir|,mkdir|,rmdir|,chown|,chgrp|,chmod|,chattr|,chroot|,utime|,mkdev|,mkfifo|,mktemp|,net|,inet|,bnet|,cnet|,snet)*[-+^]/
      \ contained contains=Syd3RuleOps nextgroup=Syd3RulePath
syn match Syd3RuleFsType /\vallow\/lock\/(all|rpath|wpath|cpath|dpath|spath|tpath|inet|read|write|exec|ioctl|create|rename|symlink|truncate|readdir|mkdir|rmdir|mkdev|mkfifo|bind|connect|net|inet|bnet|cnet)(,all|,rpath|,wpath|,cpath|,dpath|,spath|,tpath|,inet|,read|,write|,exec|,ioctl|,create|,rename|,symlink|,truncate|,readdir|,mkdir|,rmdir|,mkbdev|,mkcdev|,mkfifo|,bind|,connect|,net|,inet|,bnet|,cnet)*[-+^]/
      \ contained contains=Syd3RuleOps nextgroup=Syd3RulePath
hi link Syd3RuleFsType Identifier

syn match Syd3RulePort /\v[0-9]+(-[0-9]+)?/
      \ contained containedin=Syd3RuleNet
hi link Syd3RulePort Number

syn match Syd3NetOps /\v[@!]/
      \ contained nextgroup=Syd3RulePort
hi link Syd3NetOps Operator

syn match Syd3RuleUnix /\v\/.*$/ contained
hi link Syd3RuleUnix String

syn match Syd3RuleAddr /\v(!unnamed|[A-Fa-f0-9:\.]+\/\d+|[A-Fa-f0-9:\.]+|\/.*$|[aA][nN][yY][46]?|[lL][oO][cC][aA][lL][46]?|[lL][oO][oO][pP][bB][aA][cC][kK][46]?|[lL][iI][nN][kK][lL][oO][cC][aA][lL][46]?)/
      \ contained contains=Syd3RuleAlias,Syd3RuleUnix nextgroup=Syd3NetOps
hi link Syd3RuleAddr Constant

syn match Syd3RuleAlias /\v[aA][nN][yY][46]?/
      \ contained
syn match Syd3RuleAlias /\v[lL][oO][cC][aA][lL][46]?/
      \ contained
syn match Syd3RuleAlias /\v[lL][oO][oO][pP][bB][aA][cC][kK][46]?/
      \ contained
syn match Syd3RuleAlias /\v[lL][iI][nN][kK][lL][oO][cC][aA][lL][46]?/
      \ contained
syn match Syd3RuleAlias /\v!unnamed/
      \ contained
hi link Syd3RuleAlias Type

syn match Syd3RuleNetType /\v(allow|warn|filter|deny|panic|stop|abort|kill|exit)\/net\/(bind|connect|sendfd)[-+^]/
      \ contained contains=Syd3RuleOps nextgroup=Syd3RuleAddr
syn match Syd3RuleNetType /\v(allow|warn|filter|deny|panic|stop|abort|kill|exit)\/(net|inet|bnet|cnet|snet)[-+^]/
hi link Syd3RuleNetType Identifier

syn match Syd3BlockType /\vblock[-+!^]/ contained
hi link Syd3BlockType Identifier

syn match Syd3BlockAddr /\vblock[-+].*$/ contained
      \ contains=Syd3BlockType,Syd3RuleAddr,Syd3EnvVar
hi link Syd3BlockAddr Constant

syn match Syd3AppendType /\vappend[-+^]/ contained
hi link Syd3AppendType Identifier

syn match Syd3SetInt /\v:[0-9]+/ contained contains=Syd3OpSet
hi def link Syd3SetInt Number

syn match Syd3SetName /\v:.+$/ contained contains=Syd3OpSet
hi link Syd3SetName String

syn match Syd3SetPath /\v\/.*$/ contained
hi link Syd3SetPath String

syn keyword Syd3SetTmpfs
      \ tmpfs
      \ tmp
      \ t
      \ contained
hi link Syd3SetTmpfs Type

syn match Syd3StatType /\vstat$/
hi link Syd3StatType Type

let s:trace_options = join([
      \ 'allow_safe_bind',
      \ 'allow_safe_kcapi',
      \ 'allow_safe_setuid',
      \ 'allow_safe_setgid',
      \ 'allow_safe_syslog',
      \ 'deny_dotdot',
      \ 'deny_exec_elf32',
      \ 'deny_exec_elf_dynamic',
      \ 'deny_exec_elf_static',
      \ 'deny_exec_script',
      \ 'deny_tsc',
      \ 'exit_wait_all',
      \ 'force_cloexec',
      \ 'force_local_net',
      \ 'force_no_symlinks',
      \ 'force_rand_fd',
      \ 'force_ro_open',
      \ 'force_no_magiclinks',
      \ 'force_no_xdev',
      \ 'force_umask',
      \ ], '|')
let s:unsafe_trace_options = join([
      \ 'allow_unsafe_any_addr',
      \ 'allow_unsafe_bind',
      \ 'allow_unsafe_cap_fixup',
      \ 'allow_unsafe_caps',
      \ 'allow_unsafe_cbpf',
      \ 'allow_unsafe_chown',
      \ 'allow_unsafe_chroot',
      \ 'allow_unsafe_cpu',
      \ 'allow_unsafe_create',
      \ 'allow_unsafe_deprecated',
      \ 'allow_unsafe_dumpable',
      \ 'allow_unsafe_ebpf',
      \ 'allow_unsafe_env',
      \ 'allow_unsafe_exec_interactive',
      \ 'allow_unsafe_exec_ldso',
      \ 'allow_unsafe_exec_libc',
      \ 'allow_unsafe_exec_nopie',
      \ 'allow_unsafe_exec_null',
      \ 'allow_unsafe_exec_script',
      \ 'allow_unsafe_exec_stack',
      \ 'allow_unsafe_filename',
      \ 'allow_unsafe_hardlinks',
      \ 'allow_unsafe_keyring',
      \ 'allow_unsafe_machine_id',
      \ 'allow_unsafe_madvise',
      \ 'allow_unsafe_magiclinks',
      \ 'allow_unsafe_mbind',
      \ 'allow_unsafe_memfd',
      \ 'allow_unsafe_memory',
      \ 'allow_unsafe_mkbdev',
      \ 'allow_unsafe_mkcdev',
      \ 'allow_unsafe_mqueue',
      \ 'allow_unsafe_msgsnd',
      \ 'allow_unsafe_nice',
      \ 'allow_unsafe_nocookie',
      \ 'allow_unsafe_nomseal',
      \ 'allow_unsafe_notify_bdev',
      \ 'allow_unsafe_notify_cdev',
      \ 'allow_unsafe_oob',
      \ 'allow_unsafe_open_kfd',
      \ 'allow_unsafe_open_path',
      \ 'allow_unsafe_page_cache',
      \ 'allow_unsafe_perf',
      \ 'allow_unsafe_personality',
      \ 'allow_unsafe_pipe',
      \ 'allow_unsafe_pivot_root',
      \ 'allow_unsafe_pkey',
      \ 'allow_unsafe_prctl',
      \ 'allow_unsafe_prlimit',
      \ 'allow_unsafe_proc_pid_status',
      \ 'allow_unsafe_ptrace',
      \ 'allow_unsafe_rseq',
      \ 'allow_unsafe_setsockopt',
      \ 'allow_unsafe_shm',
      \ 'allow_unsafe_sigreturn',
      \ 'allow_unsafe_socket',
      \ 'allow_unsafe_spec_exec',
      \ 'allow_unsafe_stat_bdev',
      \ 'allow_unsafe_stat_cdev',
      \ 'allow_unsafe_symlinks',
      \ 'allow_unsafe_sync',
      \ 'allow_unsafe_sysinfo',
      \ 'allow_unsafe_syslog',
      \ 'allow_unsafe_time',
      \ 'allow_unsafe_uname',
      \ 'allow_unsafe_uring',
      \ 'allow_unsafe_vmsplice',
      \ 'allow_unsafe_xattr',
      \ 'allow_unsupp_socket',
      \ ], '|')
let s:trace_regex = '\vtrace\/(' . s:trace_options . '):'
let s:unsafe_trace_regex = '\vtrace\/(' . s:unsafe_trace_options . '):'
exec 'syn match Syd3TraceType /' . s:trace_regex . '/ contained contains=Syd3OpSet nextgroup=Syd3Boolean'
exec 'syn match Syd3TraceUnsafe /' . s:unsafe_trace_regex . '/ contained contains=Syd3OpSet nextgroup=Syd3Boolean'
hi link Syd3TraceType Identifier
hi link Syd3TraceUnsafe WarningMsg

syn match Syd3Umask /\v(-1|[0-9]+)/ contained
hi def link Syd3Umask Number

syn match Syd3ZeroOrOne /\v[01]/ contained
hi def link Syd3ZeroOrOne Number

syn match Syd3ForceUmaskType /\vtrace\/force_umask:/
      \ contained contains=Syd3OpSet nextgroup=Syd3Umask
hi link Syd3ForceUmaskType Identifier

syn match Syd3MemoryAccessType /\vtrace\/memory_access:/
      \ contained contains=Syd3OpSet nextgroup=Syd3ZeroOrOne
hi link Syd3MemoryAccessType Identifier

syn match Syd3IncludePath /\v.*$/ contained
hi link Syd3IncludePath String

syn match Syd3LogLevel /\vlog\/level:/
      \ contained contains=Syd3OpSet nextgroup=Syd3LogSeverity,Syd3SetInt
hi link Syd3LogLevel Identifier

syn keyword Syd3LogSeverity
      \ emerg
      \ alert
      \ crit
      \ error
      \ warn
      \ notice
      \ info
      \ debug
      \ contained
hi link Syd3LogSeverity Type

syn match Syd3LogType /\vlog\/(verbose|lock\/(same_exec_off|new_exec_on|subdomains_off)):/
      \ contained contains=Syd3OpSet nextgroup=Syd3SetBoolean
hi link Syd3LogType Identifier

syn match Syd3TimeType /\vtime(:|\/(boot|mono):)/
      \ contained contains=Syd3OptSet nextgroup=Syd3None,Syd3SetInt
hi link Syd3TimeType Identifier

syn keyword Syd3IncludeProfilePattern
      \ container
      \ core
      \ debug
      \ enforce
      \ firefox
      \ immutable
      \ landlock
      \ lib
      \ linux
      \ noipv4
      \ noipv6
      \ nomem
      \ nopie
      \ oci
      \ paludis
      \ quiet
      \ rand
      \ readonly
      \ tty
      \ user
      \ contained
hi link Syd3IncludeProfilePattern Type

syn match Syd3IncludeType /\vinclude /
      \ contained nextgroup=Syd3IncludePath
hi link Syd3IncludeType Include

syn match Syd3IncludeProfileType /\vinclude_profile /
      \ contained nextgroup=Syd3IncludeProfilePattern
hi link Syd3IncludeProfileType Include

syn match Syd3EnvEnd /\v\}/
      \ contained containedin=Syd3EnvVar
hi link Syd3EnvEnd Include

syn match Syd3EnvDef /\v(:-[^\}]*)?/
      \ contained containedin=Syd3EnvVar nextgroup=Syd3EnvEnd
hi link Syd3EnvDef String

syn match Syd3EnvVar /\v\$\{[a-zA-Z0-9_-]+/
      \ contained containedin=ALLBUT,Syd3Comment nextgroup=Syd3EnvDef
hi link Syd3EnvVar Include

syn region Syd3Lock start=/\vlock/ end=/$/ oneline keepend
      \ contains=Syd3LockType,Syd3OpSet,Syd3LockState,Syd3EnvVar,Syd3Comment
syn region Syd3Ipc start=/\vipc/ end=/$/ oneline keepend
      \ contains=Syd3IpcType,Syd3OpSet,Syd3IpcSocket,Syd3EnvVar,Syd3Comment
syn region Syd3DisallowedCommands start=/\v(ghost|load|panic)/ end=/$/ oneline keepend
      \ contains=Syd3DisallowedCommandsType,Syd3Comment
syn region Syd3Default start=/\vdefault\// end=/$/ oneline keepend
      \ contains=Syd3DefaultType,Syd3OpSet,Syd3DefaultAct,Syd3EnvVar,Syd3Comment
syn region Syd3Link start=/\vallow\/net\/link/ end=/$/ oneline keepend
      \ contains=Syd3LinkType,Syd3LinkFam,Syd3EnvVar,Syd3Comment
syn region Syd3Sandbox start=/\vsandbox\// end=/$/ oneline keepend
      \ contains=Syd3SandboxType,Syd3OpSet,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3Unshare start=/\vunshare\// end=/$/ oneline keepend
      \ contains=Syd3UnshareType,Syd3OpSet,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3Chroot start=/\vroot:[\/t]/ end=/$/ oneline keepend
      \ contains=Syd3ChrootType,Syd3OpSet,Syd3SetPath,Syd3SetTmpfs,Syd3EnvVar,Syd3Comment
syn region Syd3Root start=/\vroot\// end=/$/ oneline keepend
      \ contains=Syd3RootType,Syd3OpSet,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3Uts start=/\vuts\// end=/$/ oneline keepend
      \ contains=Syd3UtsType,Syd3OpSet,Syd3SetName,Syd3EnvVar,Syd3Comment
syn region Syd3RuleFs start=/\v(allow|warn|filter|deny|panic|stop|abort|kill|exit)\/%(net\/)@!/ end=/$/ oneline keepend
      \ contains=Syd3RuleFsType,Syd3RuleOps,Syd3RulePath,Syd3EnvVar,Syd3Comment
syn region Syd3RuleNet start=/\v(allow|warn|filter|deny|panic|stop|abort|kill|exit)\/net\/%(link)@!/ end=/$/ oneline keepend
      \ contains=Syd3RuleNetType,Syd3RuleOps,Syd3RuleAddr,Syd3RuleUnix,Syd3EnvVar,Syd3Comment
syn region Syd3Kill start=/\v(mem|pid)\/kill:/ end=/$/ oneline keepend
      \ contains=Syd3KillType,Syd3OpSet,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3AppendRem start=/\vappend\^/ end=/$/ oneline keepend
      \ contains=Syd3AppendType,Syd3OpXor,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3AppendDel start=/\vappend-/ end=/$/ oneline keepend
      \ contains=Syd3AppendType,Syd3OpDec,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3AppendAdd start=/\vappend\+/ end=/$/ oneline keepend
      \ contains=Syd3AppendType,Syd3OpAdd,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3BlockAdd start=/\vblock\+/ end=/$/ oneline keepend
      \ contains=Syd3BlockAddr,Syd3EnvVar,Syd3Comment
syn region Syd3BlockDel start=/\vblock-/ end=/$/ oneline keepend
      \ contains=Syd3BlockAddr,Syd3EnvVar,Syd3Comment
syn region Syd3BlockClear start=/\vblock\^/ end=/$/ oneline keepend
      \ contains=Syd3BlockType
syn region Syd3BlockSimplify start=/\vblock!/ end=/$/ oneline keepend
      \ contains=Syd3BlockType
syn region Syd3Mem start=/\vmem\/(max|vm_max):/ end=/$/ oneline keepend
      \ contains=Syd3MemType,Syd3OpSet,Syd3Size,Syd3EnvVar,Syd3Comment
syn region Syd3Pid start=/\vpid\/max:/ end=/$/ oneline keepend
      \ contains=Syd3PidType,Syd3OpSet,Syd3PidMax,Syd3EnvVar,Syd3Comment
syn region Syd3Tpe start=/\vtpe\/(negate|root_owned|user_owned|root_mount):/ end=/$/ oneline keepend
      \ contains=Syd3TpeType,Syd3OpSet,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3TpeGid start=/\vtpe\/gid:/ end=/$/ oneline keepend
      \ contains=Syd3TpeGidType,Syd3OpSet,Syd3Gid,Syd3EnvVar,Syd3Comment
syn region Syd3Bind start=/\vbind[-+^]/ end=/$/ oneline keepend
      \ contains=Syd3BindType,Syd3EnvVar,Syd3Comment
syn region Syd3CryptRem start=/\vcrypt\^/ end=/$/ oneline keepend
      \ contains=Syd3CryptType,Syd3OpXor,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3CryptDel start=/\vcrypt-/ end=/$/ oneline keepend
      \ contains=Syd3CryptType,Syd3OpDec,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3CryptAdd start=/\vcrypt\+/ end=/$/ oneline keepend
      \ contains=Syd3CryptType,Syd3OpAdd,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3CryptKey start=/\vcrypt\/key:/ end=/$/ oneline keepend
      \ contains=Syd3CryptKeyType,Syd3OpSet,Syd3CryptKeyHex,Syd3EnvVar,Syd3Comment
syn region Syd3CryptTmp start=/\vcrypt\/tmp:/ end=/$/ oneline keepend
      \ contains=Syd3CryptTmpType,Syd3OpSet,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3ForceRem start=/\vforce\^/ end=/$/ oneline keepend
      \ contains=Syd3ForceType,Syd3OpXor,Syd3EnvVar,Syd3Comment
syn region Syd3ForceDel start=/\vforce-/ end=/$/ oneline keepend
      \ contains=Syd3ForceType,Syd3OpDec,Syd3SetPath,Syd3EnvVar,Syd3Comment
syn region Syd3ForceAdd start=/\vforce\+/ end=/$/ oneline keepend
      \ contains=Syd3ForceAddType,Syd3ForcePath,Syd3OpSet,Syd3ForceHash,Syd3ForceAct,Syd3EnvVar,Syd3Comment
syn region Syd3Ioctl start=/\v(allow|deny)\/ioctl/ end=/$/ oneline keepend
      \ contains=Syd3IoctlType,Syd3IoctlAct,Syd3IoctlNum,Syd3IoctlConst,Syd3OpNext,Syd3EnvVar,Syd3Comment
syn region Syd3SegvGuard start=/\vsegvguard\// end=/$/ oneline keepend
      \ contains=Syd3SegvGuardType,Syd3SetInt,Syd3EnvVar,Syd3Comment
syn region Syd3Mask start=/\vmask[-+^]/ end=/$/ oneline keepend
      \ contains=Syd3MaskType,Syd3EnvVar,Syd3Comment
syn region Syd3SetIdAddDel start=/\vset[ug]id[-+]/ end=/$/ oneline keepend
      \ contains=Syd3SetIdAddDelType,Syd3Comment
syn region Syd3SetIdRem start=/\vset[ug]id\^/ end=/$/ oneline keepend
      \ contains=Syd3SetIdRemType,Syd3Comment
syn region Syd3SetEnv start=/\vsetenv!/ end=/$/ oneline keepend
      \ contains=Syd3SetEnvType,Syd3EnvVar,Syd3Comment
syn region Syd3UnsetEnv start=/\vunsetenv!/ end=/$/ oneline keepend
      \ contains=Syd3UnsetEnvType,Syd3Comment
syn region Syd3ClearEnv start=/\vclearenv!/ end=/$/ oneline keepend
      \ contains=Syd3ClearEnvType,Syd3Comment
syn region Syd3Stat start=/\vstat/ end=/$/ oneline keepend
      \ contains=Syd3StatType,Syd3EnvVar,Syd3Comment
syn region Syd3Log start=/\vlog\// end=/$/ oneline keepend
      \ contains=Syd3LogLevel,Syd3LogType,Syd3Boolean,Syd3SetInt,Syd3EnvVar,Syd3Comment
syn region Syd3Time start=/\vtime/ end=/$/ oneline keepend
      \ contains=Syd3TimeType,Syd3None,Syd3SetInt,Syd3EnvVar,Syd3Comment
syn region Syd3Trace start=/\vtrace\/([^a]|allow_[^u])/ end=/$/ oneline keepend
      \ contains=Syd3TraceType,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3Unsafe start=/\vtrace\/allow_u/ end=/$/ oneline keepend
      \ contains=Syd3TraceUnsafe,Syd3Boolean,Syd3EnvVar,Syd3Comment
syn region Syd3ForceUmask start=/\vtrace\/force_umask/ end=/$/ oneline keepend
      \ contains=Syd3ForceUmaskType,Syd3EnvVar,Syd3Comment
syn region Syd3MemoryAccess start=/\vtrace\/memory_access/ end=/$/ oneline keepend
      \ contains=Syd3MemoryAccessType,Syd3EnvVar,Syd3Comment
syn region Syd3Include start=/\vinclude / end=/$/ oneline keepend
      \ contains=Syd3IncludeType,Syd3IncludePath,Syd3EnvVar,Syd3Comment
syn region Syd3IncludeProfile start=/\vinclude_profile / end=/$/ oneline keepend
      \ contains=Syd3IncludeProfileType,Syd3IncludeProfilePattern,Syd3EnvVar,Syd3Comment

syn match Syd3TopLevelDirective /^\s*/ nextgroup=Syd3Comment,Syd3Lock,Syd3Ipc,Syd3DisallowedCommands,Syd3Default,Syd3Link,Syd3Sandbox,Syd3Unshare,Syd3Chroot,Syd3Root,Syd3Uts,Syd3RuleFs,Syd3RuleNet,Syd3Kill,Syd3AppendRem,Syd3AppendDel,Syd3AppendAdd,Syd3BlockAdd,Syd3BlockDel,Syd3BlockClear,Syd3BlockSimplify,Syd3Mem,Syd3Pid,Syd3Tpe,Syd3TpeGid,Syd3Bind,Syd3CryptRem,Syd3CryptDel,Syd3CryptAdd,Syd3CryptKey,Syd3CryptTmp,Syd3ForceRem,Syd3ForceDel,Syd3ForceAdd,Syd3Ioctl,Syd3SegvGuard,Syd3Mask,Syd3SetIdAddDel,Syd3SetIdRem,Syd3SetEnv,Syd3UnsetEnv,Syd3ClearEnv,Syd3Stat,Syd3Log,Syd3Time,Syd3Trace,Syd3Unsafe,Syd3ForceUmask,Syd3MemoryAccess,Syd3Include,Syd3IncludeProfile,Syd3ConfigError

let b:current_syntax = "syd-3"
