package jwt_test

import (
	"reflect"
	"strings"
	"testing"

	"github.com/golang-jwt/jwt/v5"
)

var noneTestData = []struct {
	name        string
	tokenString string
	alg         string
	key         interface{}
	claims      map[string]interface{}
	valid       bool
}{
	{
		"Basic",
		"eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJmb28iOiJiYXIifQ.",
		"none",
		jwt.UnsafeAllowNoneSignatureType,
		map[string]interface{}{"foo": "bar"},
		true,
	},
	{
		"Basic - no key",
		"eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJmb28iOiJiYXIifQ.",
		"none",
		nil,
		map[string]interface{}{"foo": "bar"},
		false,
	},
	{
		"Signed",
		"eyJhbGciOiJSUzM4NCIsInR5cCI6IkpXVCJ9.eyJmb28iOiJiYXIifQ.W-jEzRfBigtCWsinvVVuldiuilzVdU5ty0MvpLaSaqK9PlAWWlDQ1VIQ_qSKzwL5IXaZkvZFJXT3yL3n7OUVu7zCNJzdwznbC8Z-b0z2lYvcklJYi2VOFRcGbJtXUqgjk2oGsiqUMUMOLP70TTefkpsgqDxbRh9CDUfpOJgW-dU7cmgaoswe3wjUAUi6B6G2YEaiuXC0XScQYSYVKIzgKXJV8Zw-7AN_DBUI4GkTpsvQ9fVVjZM9csQiEXhYekyrKu1nu_POpQonGd8yqkIyXPECNmmqH5jH4sFiF67XhD7_JpkvLziBpI-uh86evBUadmHhb9Otqw3uV3NTaXLzJw",
		"none",
		jwt.UnsafeAllowNoneSignatureType,
		map[string]interface{}{"foo": "bar"},
		false,
	},
}

func TestNoneVerify(t *testing.T) {
	for _, data := range noneTestData {
		parts := strings.Split(data.tokenString, ".")

		method := jwt.GetSigningMethod(data.alg)
		err := method.Verify(strings.Join(parts[0:2], "."), decodeSegment(t, parts[2]), data.key)
		if data.valid && err != nil {
			t.Errorf("[%v] Error while verifying key: %v", data.name, err)
		}
		if !data.valid && err == nil {
			t.Errorf("[%v] Invalid key passed validation", data.name)
		}
	}
}

func TestNoneSign(t *testing.T) {
	for _, data := range noneTestData {
		if !data.valid {
			continue
		}
		parts := strings.Split(data.tokenString, ".")
		method := jwt.GetSigningMethod(data.alg)
		sig, err := method.Sign(strings.Join(parts[0:2], "."), data.key)
		if err != nil {
			t.Errorf("[%v] Error signing token: %v", data.name, err)
		}
		if !reflect.DeepEqual(sig, decodeSegment(t, parts[2])) {
			t.Errorf("[%v] Incorrect signature.\nwas:\n%v\nexpecting:\n%v", data.name, sig, parts[2])
		}
	}
}
