/*******************************************************************************
* Copyright 2023-2024 Intel Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#include <cstdint>
#include <vector>

namespace dnnl {
namespace impl {
namespace gpu {
namespace intel {
namespace jit {

// clang-format off
const std::vector<uint64_t> &get_conv_lookup_table_data() {
    static std::vector<uint64_t> data = {
        0x0000000000000196, 0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x0018000000010001, 0x6869323363690000, 0x6f3233636f323135, 0x7033686b32313568,
        0x0003000000083168, 0x0000000202020000, 0x0808000000030707, 0x0000000200000003, 0x0d0d000000020a0a, 0x0000000300000004, 0x0a0a000000100202, 0x00200d0d00000010,
        0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000002, 0x3668693032336369, 0x686f303233636f34, 0x31687033686b3436,
        0x0000000300000010, 0x07070000000a0202, 0x0003080800000003, 0x0d0d000000010000, 0x0000000400000008, 0x0909000000200202, 0x00200a0a00000002, 0x000000080d0d0000,
        0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000180000, 0x6f34366869303233, 0x3436686f30323363, 0x001031687033686b,
        0x0202000000030000, 0x000307070000000a, 0x0000000308080000, 0x00020a0a00000002, 0x000000040d0d0000, 0x0020020200000003, 0x000000400a0a0000, 0x0003000000100d0d,
        0x0003000000050000, 0x000a000000030000, 0x000a000000010000, 0x0000002000010000, 0x6936636900000014, 0x646f31636f363264, 0x30647032646b3532, 0x0000000200000010,
        0x0d0d000000020909, 0x0000000000000019, 0x0008020200000004, 0x0000000208080000, 0x0a0a000000080909, 0x0000000000000010, 0x0000000300000005, 0x0000000700000003,
        0x0000000700000001, 0x0016000000100001, 0x6869343663690000, 0x686f3436636f3537, 0x31687033686b3537, 0x0000000200000010, 0x0d0d000000050c0c, 0x0000000100000005,
        0x0003000000040202, 0x0000001002020000, 0x0a0a000000100909, 0x0000000300000040, 0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x0018000000010001,
        0x6930343663690000, 0x303233636f343668, 0x7033686b3436686f, 0x0003000000083168, 0x0000002802020000, 0x0808000000030707, 0x0000000200000003, 0x0d0d000000080a0a,
        0x0000000300000004, 0x0a0a000000100202, 0x00100d0d00000010, 0x0004ffffffff0000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001,
        0x3368693034366369, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000008, 0x0707000000280202, 0x0003080800000003, 0x0a0a000000010000, 0x0000000300000008,
        0x0a0a000000100202, 0x00100d0d00000020, 0x0004ffffffff0000, 0x0003000000030000, 0x0001000000070000, 0x0001000000070000, 0x0000001c00000008, 0x3277693231356369,
        0x6f323135636f3939, 0x777338776b333777, 0x0000000830777034, 0x0008090900000001, 0x0202000000020000, 0x00040a0a00000008, 0x0202000000030000, 0x00400a0a00000010,
        0x000000100d0d0000, 0x0000000400000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x63690000000a0000, 0x3834636f32353131, 0x0000000100000008,
        0x0001000000090202, 0x0000000802020000, 0x0010020200000003, 0x000000080a0a0000, 0x0000000000080d0d, 0x0003000000050000, 0x0007000000030000, 0x0007000000010000,
        0x0000000100010000, 0x693363690000001a, 0x3231636f34323268, 0x34686b3635686f38, 0x0010306870346873, 0x0001000000000000, 0x000000020a0a0000, 0x0008020200000004,
        0x0000000408080000, 0x0d0d000000100a0a, 0x0000000300000010, 0x0000000300000005, 0x0000000a00000003, 0x0000000a00000001, 0x001a000000010001, 0x3268693363690000,
        0x6f383231636f3432, 0x687334686b363568, 0x0000001030687034, 0x00070c0c00000001, 0x0a0a000000010000, 0x0000000400000008, 0x0808000000040202, 0x00100a0a00000004,
        0x000000080d0d0000, 0x0000000400000002, 0x0000000300000003, 0x0000000100000007, 0x0010010100000007, 0x63690000001d0000, 0x3531776938383031, 0x6f323135636f3030,
        0x31776b3030353177, 0x0100000008307770, 0x0001f40d0d000000, 0x0802020000000200, 0x0000080a0a000000, 0x2002020000000300, 0x0000100909000000, 0x03000000200a0a00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000020001000000, 0x3263690000001c00, 0x3635327769303635, 0x776f30383231636f, 0x777031776b363532,
        0x0000010000001030, 0x0200000050020200, 0x0000020a0a000000, 0x03000000080d0d00, 0x0000200202000000, 0x0d000000400a0a00, 0x000003000000100d, 0x0000030000000500,
        0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3635326369000000, 0x636f363532776930, 0x3532776f30383231, 0x1030777031776b36, 0x0200000001000000,
        0x0000020000005002, 0x0d000000020a0a00, 0x000003000000080d, 0x0a00000020020200, 0x100d0d000000200a, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000,
        0x0100000001000000, 0x0000190000000200, 0x6930323931636900, 0x303436636f323368, 0x7033686b3233686f, 0x0003000000103168, 0x0000003c02020000, 0x0808000000030707,
        0x0000000100000003, 0x0004000000020d0d, 0x0000002002020000, 0x0a0a000000020909, 0x00100d0d00000040, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000,
        0x0001000000010000, 0x0000001900000001, 0x6869303239316369, 0x6f303436636f3233, 0x687033686b323368, 0x0000030000001031, 0x070000003c020200, 0x0308080000000307,
        0x0a00000002000000, 0x020d0d000000040a, 0x0200000003000000, 0x200a0a0000002002, 0x0000100d0d000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700,
        0x0100010000000100, 0x6900000018000000, 0x3639303477693463, 0x393034776f34636f, 0x1030777031776b36, 0x0100000000000000, 0x0000020d0d000000, 0x0402020000000400,
        0x0000040808000000, 0x0d000000100a0a00, 0x000003000000100d, 0x0000030000000400, 0x0000040000000100, 0x0000010000000400, 0x1b00000080010100, 0x3635326369000000,
        0x3135636f36356869, 0x31686b3832686f32, 0x0008306870326873, 0x0202000000010000, 0x0000000100000008, 0x0003000000080a0a, 0x0000002002020000, 0x0a0a000000200909,
        0x0000000200000020, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004, 0x0018000000200101, 0x6934383363690000, 0x363532636f363268, 0x7033686b3632686f,
        0x0003000000083168, 0x0000000c02020000, 0x0808000000030707, 0x0000000200000003, 0x0d0d000000040a0a, 0x0000000300000004, 0x0909000000200202, 0x00200a0a00000020,
        0x0005000000030000, 0x0002000000010000, 0x000b000000010000, 0x00010000000b0000, 0x0000001300000800, 0x6f30316869326369, 0x33686b38686f3863, 0x0200000008306870,
        0x0000030707000000, 0x0100000003080800, 0x0000020505000000, 0x0802020000000300, 0x0000100909000000, 0x00000000080a0a00, 0x0100000005000000, 0x0100000002000000,
        0x0800000008000000, 0x0008000001000000, 0x3263690000001300, 0x6f38636f30316869, 0x30687033686b3868, 0x0000000200000010, 0x0808000000030707, 0x0000000000000003,
        0x0010020200000003, 0x0000001009090000, 0x0000000000080a0a, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x3631636900000018,
        0x33636f3830326869, 0x686b383032686f32, 0x0000000831687033, 0x0003070700000002, 0x0000000308080000, 0x00020a0a00000001, 0x0202000000030000, 0x00100a0a00000020,
        0x000000100d0d0000, 0x0000000400000003, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001c0000, 0x3735776930323931, 0x6f30383231636f36,
        0x7031776b36373577, 0x0001000000083077, 0x0000007802020000, 0x00080a0a00000002, 0x000000080d0d0000, 0x0010020200000003, 0x000000200a0a0000, 0x0000000000180d0d,
        0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x393163690000001a, 0x636f363168693032, 0x3631686f30383231, 0x000831687033686b,
        0x0202000000030000, 0x000307070000003c, 0x0000000308080000, 0x0002020200000001, 0x0202000000030000, 0x00200a0a00000010, 0x000000100d0d0000, 0x0000000500000000,
        0x0000000300000003, 0x000000010000000a, 0x004001010000000a, 0x6369000000190000, 0x6f30397769363532, 0x3935776f36353263, 0x103077703233776b, 0x0900000002000000,
        0x3b0d0d0000000209, 0x0200000001000000, 0x0000030000000802, 0x0900000020020200, 0x400a0a0000000809, 0x0400000002000000, 0x0100000003000000, 0x0700000007000000,
        0x0100000001000000, 0x00001a0000000100, 0x6930383231636900, 0x383231636f323368, 0x33686b3233686f30, 0x0300000008316870, 0x0000500202000000, 0x0800000003070700,
        0x0000020000000308, 0x0d000000040a0a00, 0x000003000000020d, 0x0a00000010020200, 0x100d0d000000100a, 0x05ffffffff000000, 0x0300000003000000, 0x0100000007000000,
        0x0100000007000000, 0x0000190000004000, 0x7769363532636900, 0x6f363532636f3039, 0x703233776b393577, 0x0002000000103077, 0x0000004009090000, 0x0002000000040d0d,
        0x0000000802020000, 0x0003000000040a0a, 0x0000001002020000, 0x0d0d000000400a0a, 0xffffffff00000010, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007,
        0x002c000000200101, 0x3868693163690000, 0x3536303031776930, 0x3038686f31636f30, 0x303536303031776f, 0x703332776b31686b, 0x0010313177703068, 0x0808000000010000,
        0x0000000100000003, 0x0004000000080d0d, 0x0000000202020000, 0x0909000000080808, 0x00100a0a00000020, 0x0005000000030000, 0x0001000000030000, 0x0004000000040000,
        0x0101000000010000, 0x0000002c00000020, 0x6930386869316369, 0x6f30353630303177, 0x776f3038686f3163, 0x686b303536303031, 0x3068703332776b31, 0x0000001031317770,
        0x0003080800000001, 0x0d0d000000010000, 0x0000000400000004, 0x0808000000040202, 0x0020090900000008, 0x000000100a0a0000, 0x0000000400000003, 0x0000000100000003,
        0x0000000400000004, 0x0c80010100000001, 0x6369000000190000, 0x6f34367769383434, 0x36776f3038303263, 0x0830777031776b34, 0x0200000001000000, 0x0000020000000e02,
        0x0d000000040a0a00, 0x000003000000080d, 0x0900000020020200, 0x200a0a0000002009, 0x0400000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000,
        0x00001900000c8001, 0x7769383434636900, 0x30383032636f3436, 0x7031776b3436776f, 0x0001000000083077, 0x0000001c02020000, 0x00020a0a00000002, 0x000000080d0d0000,
        0x0010020200000003, 0x0000002009090000, 0x0003000000200a0a, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3231636900000019,
        0x636f323368693038, 0x6b3233686f303436, 0x0000083168703368, 0x5002020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000080a0a000000, 0x03000000020d0d00,
        0x0000100202000000, 0x0d000000200a0a00, 0xffffff000000100d, 0x00000300000004ff, 0x0000070000000100, 0x0000010000000700, 0x1a00000001000100, 0x3832316369000000,
        0x31636f3831686930, 0x6b3631686f303832, 0x0000083068703368, 0x2802020000000300, 0x0000030707000000, 0x0100000003080800, 0x0000020202000000, 0x1002020000000300,
        0x0000200a0a000000, 0x00000000100d0d00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3163690000001800, 0x636f386869303832,
        0x6b38686f30383231, 0x0000083168703368, 0x1402020000000300, 0x0000030707000000, 0x0100000003080800, 0x0000020202000000, 0x2002020000000300, 0x0000100a0a000000,
        0x00000000080d0d00, 0x0300000005000000, 0x0700000003000000, 0x0700000001000000, 0x0000100001000000, 0x3163690000001800, 0x6f38336869343230, 0x6b3833686f363163,
        0x0000103168703368, 0x260d0d0000000100, 0x0300000000000000, 0x0000200202000000, 0x0a00000010090900, 0x000003000000100a, 0x0000010000000500, 0x00000b0000000300,
        0x00000b0000000100, 0x1300000800000100, 0x6869326369000000, 0x38686f38636f3031, 0x000830687033686b, 0x0c0c000000020000, 0x00080d0d00000004, 0x0004000000000000,
        0x0000000202020000, 0x0909000000030808, 0x00080a0a00000008, 0x0005000000000000, 0x0003000000010000, 0x0001000000080000, 0x0001000000080000, 0x0000001300000800,
        0x6f30316869326369, 0x33686b38686f3863, 0x0200000010306870, 0x0000040c0c000000, 0x00000000080d0d00, 0x0200000004000000, 0x0308080000000202, 0x0000100909000000,
        0x00000000100a0a00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3163690000001c00, 0x3635327769303832, 0x776f30383231636f,
        0x777031776b363532, 0x0000010000000830, 0x0200000050020200, 0x0000080a0a000000, 0x03000000040d0d00, 0x0000100202000000, 0x0d000000100a0a00, 0xffffff000000200d,
        0x00000300000004ff, 0x0000070000000300, 0x0000070000000100, 0x2600000003000100, 0x3635326369000000, 0x3631776930316869, 0x31686f363532636f, 0x35686b3631776f30,
        0x777032687035776b, 0x0000010000000832, 0x020000000a0c0c00, 0x0000080202000000, 0x03000000040a0a00, 0x0000100202000000, 0x0d000000400a0a00, 0x000003000000100d,
        0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3239316369000000, 0x636f363532776930, 0x3532776f30383231, 0x0830777031776b36,
        0x0200000001000000, 0x0000020000007802, 0x0d000000080a0a00, 0x000003000000040d, 0x0a00000010020200, 0x200d0d000000100a, 0x05ffffffff000000, 0x0100000003000000,
        0x0700000007000000, 0x0100000001000000, 0x0000170000000100, 0x7769363938636900, 0x776f3233636f3934, 0x30777031776b3934, 0x0000000100000010, 0x0001000000070202,
        0x0000000802020000, 0x0010020200000003, 0x000000100a0a0000, 0x0000000000080d0d, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000,
        0x393163690000001d, 0x3432303177693032, 0x31776f303436636f, 0x777031776b343230, 0x0000010000000830, 0x0200000078020200, 0x0000080a0a000000, 0x03000000040d0d00,
        0x0000100202000000, 0x0d000000200a0a00, 0xffffff000000200d, 0x00000300000005ff, 0x0000010000000200, 0x0000070000000700, 0x3200000010000100, 0x6932336369000000,
        0x6933336869373164, 0x6f3233636f333377, 0x6f3333686f373164, 0x686b33646b333377, 0x7031647033776b33, 0x0000103177703168, 0x0306060000000300, 0x0000030707000000,
        0x0000000003080800, 0x0200000003000000, 0x1009090000002002, 0x0000200a0a000000, 0x0000050000000300, 0x0000020000000300, 0x0000070000000100, 0x4000010000000700,
        0x690000001c000000, 0x3230317769343663, 0x776f383231636f32, 0x3631776b37303031, 0x0200000010307770, 0x0000100808000000, 0x01000000040a0a00, 0x0000080505000000,
        0x4002020000000300, 0x0000200505000000, 0xff000000200a0a00, 0x0300000004ffffff, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3663690000001c00,
        0x3432303177693034, 0x31776f303436636f, 0x777031776b343230, 0x0000010000000830, 0x0200000028020200, 0x0000080a0a000000, 0x03000000080d0d00, 0x0000100202000000,
        0x0d000000100a0a00, 0x000000000000200d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1800000002000100, 0x3032336369000000, 0x3436636f32336869,
        0x33686b3233686f30, 0x0300000010316870, 0x00000a0202000000, 0x0800000003070700, 0x0000010000000308, 0x04000000040d0d00, 0x0000200202000000, 0x0a00000002090900,
        0x080d0d000000200a, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x0000180000000100, 0x6869303233636900, 0x6f303436636f3233,
        0x687033686b323368, 0x0000030000001031, 0x070000000a020200, 0x0308080000000307, 0x0a00000002000000, 0x040d0d000000020a, 0x0200000003000000, 0x200a0a0000002002,
        0x0000080d0d000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000016000000, 0x3636686930323363, 0x6b3436686f34636f,
        0x0000083068703368, 0x0302020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000040202000000, 0x03000000080d0d00, 0x0000200202000000, 0x0d000000080a0a00,
        0xffffff000000080d, 0x00000300000004ff, 0x0000070000000100, 0x0000010000000700, 0x1600000001000100, 0x6869346369000000, 0x6f303233636f3436, 0x687033686b343668,
        0x0000010000000831, 0x0100000003070700, 0x0000020a0a000000, 0x0402020000000400, 0x0000040808000000, 0x0d000000400a0a00, 0x000000000000100d, 0x0000030000000500,
        0x0000040000000100, 0x0000010000000400, 0x3900000020010100, 0x6469336369000000, 0x7769343232686938, 0x6f3631636f343232, 0x6f323131686f3864, 0x6b31646b32313177,
        0x32687333776b3368, 0x6870306470327773, 0x0000001031777031, 0x0003070700000001, 0x0004000000000000, 0x0000000402020000, 0x0909000000080808, 0x00100a0a00000020,
        0x0004000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001b00000001, 0x3368693034366369, 0x686f303436636f32, 0x32687333686b3631,
        0x0300000008316870, 0x00000a0202000000, 0x0800000003070700, 0x0000010000000308, 0x0300000004020200, 0x0000100202000000, 0x0d000000200a0a00, 0x000000000000100d,
        0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1600000002000100, 0x3032336369000000, 0x6f34636f34366869, 0x687033686b343668, 0x0000030000001031,
        0x070000000a020200, 0x0308080000000307, 0x0300000000000000, 0x0000200202000000, 0x0d000000100a0a00, 0x000003000000080d, 0x0000030000000500, 0x0000070000000100,
        0x0000010000000700, 0x1600000001000100, 0x3032336369000000, 0x6f34636f34366869, 0x687033686b343668, 0x0000030000001031, 0x070000000a020200, 0x0308080000000307,
        0x0d00000001000000, 0x000003000000020d, 0x0a00000020020200, 0x080d0d000000100a, 0x0400000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000,
        0x0000180000000100, 0x6869303436636900, 0x6f303436636f3433, 0x687033686b323368, 0x0000030000000830, 0x0700000028020200, 0x0308080000000307, 0x0a00000001000000,
        0x000003000000020a, 0x0a00000010020200, 0x100d0d000000200a, 0x0400000000000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x0000170000000100,
        0x3036686933636900, 0x36686f3233636f38, 0x31687033686b3830, 0x0000000100000008, 0x0001000000030707, 0x000000020d0d0000, 0x0004020200000004, 0x0000000808080000,
        0x0d0d000000100a0a, 0x0000000300000020, 0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x001a000000010001, 0x6936353263690000, 0x3532636f30353168,
        0x686b303531686f36, 0x0000000831687033, 0x0010020200000003, 0x0000000307070000, 0x0001000000030808, 0x000000040a0a0000, 0x0010020200000003, 0x000000400a0a0000,
        0x0003000000100d0d, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x693363690000001a, 0x3436636f30303368, 0x37686b303531686f,
        0x0008336870326873, 0x0707000000020000, 0x0002080800000007, 0x0a0a000000010000, 0x0000000400000002, 0x0808000000040202, 0x00200a0a00000004, 0x0000000f0d0d0000,
        0x0000000500000000, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001a0000, 0x3631686930383231, 0x686f30383231636f, 0x31687033686b3631,
        0x0000000300000010, 0x0707000000050202, 0x0003080800000003, 0x0202000000010000, 0x0000000400000008, 0x0909000000200202, 0x00200a0a00000002, 0x000000100d0d0000,
        0x0000000500000000, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x63690000001a0000, 0x3631686930383231, 0x686f30383231636f, 0x31687033686b3631,
        0x0000000300000010, 0x0707000000050202, 0x0003080800000003, 0x0202000000010000, 0x0000000300000008, 0x0a0a000000200202, 0x00100d0d00000020, 0x0004000000000000,
        0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001c00000001, 0x7769303635326369, 0x383231636f363532, 0x776b363532776f30, 0x0000000830777031,
        0x00a0020200000001, 0x0a0a000000020000, 0x00040d0d00000008, 0x0202000000030000, 0x00100a0a00000010, 0x000000200d0d0000, 0x00000005ffffffff, 0x0000000100000003,
        0x0000000700000007, 0x0002000100000001, 0x63690000001a0000, 0x3436776930363532, 0x776f30383231636f, 0x30777031776b3436, 0x0000000100000010, 0x0002000000500202,
        0x000000020a0a0000, 0x0003000000080d0d, 0x0000002002020000, 0x0d0d000000200a0a, 0x0000000300000008, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007,
        0x001a000000010001, 0x3036353263690000, 0x3231636f34367769, 0x776b3436776f3038, 0x0000001030777031, 0x0014020200000001, 0x0202000000010000, 0x0000000300000004,
        0x0a0a000000200202, 0x00100d0d00000020, 0x0004000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001a00000001, 0x7769303832316369,
        0x30383231636f3436, 0x7031776b3436776f, 0x0001000000083077, 0x0000000a02020000, 0x0008020200000002, 0x000000020d0d0000, 0x0010020200000003, 0x000000100a0a0000,
        0x0000000000200d0d, 0x0001000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x6932636900000014, 0x686f32636f363568, 0x31687033686b3635,
        0x0000000100000008, 0x0000000000030707, 0x0202000000040000, 0x0003080800000002, 0x000000080a0a0000, 0x0000000000070d0d, 0x0003000000040000, 0x0007000000010000,
        0x0001000000070000, 0x0000000100010000, 0x693463690000002f, 0x3177693038303168, 0x6f3633636f303239, 0x3436776f30363368, 0x7333776b33686b30, 0x3068703377733368,
        0x0100000008307770, 0x0000030707000000, 0x020a0a0000000200, 0x0000040d0d000000, 0x0402020000000400, 0x0000040808000000, 0x0d000000200a0a00, 0xffffff000000100d,
        0x00000300000005ff, 0x0000070000000100, 0x0000010000000700, 0x1b00000010000100, 0x3635326369000000, 0x3135636f38336869, 0x33686b3931686f32, 0x0010316870326873,
        0x0202000000030000, 0x0003070700000008, 0x0000000308080000, 0x00040a0a00000002, 0x000000040d0d0000, 0x0020020200000003, 0x0000001009090000, 0x0003000000200a0a,
        0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x323363690000001b, 0x33636f3436686930, 0x686b3233686f3032, 0x0831687032687333,
        0x0200000003000000, 0x0307070000001402, 0x0000030808000000, 0x080a0a0000000200, 0x0000020d0d000000, 0x1002020000000300, 0x0000100a0a000000, 0xff000000100d0d00,
        0x0300000004ffffff, 0x0100000002000000, 0x0700000007000000, 0x0000020001000000, 0x3263690000001a00, 0x6f30303168693635, 0x3031686f36353263, 0x0831687033686b30,
        0x0700000003000000, 0x0308080000000307, 0x0000080a0a000000, 0x0802020000000200, 0x0000040505000000, 0x1002020000000300, 0x0000190505000000, 0x03000000200a0a00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000020001000000, 0x3363690000001c00, 0x3432303177693032, 0x31776f303436636f, 0x777031776b343230,
        0x0000010000001030, 0x020000000a020200, 0x0000020a0a000000, 0x03000000040d0d00, 0x0000200202000000, 0x0d000000200a0a00, 0x000003000000100d, 0x0000030000000500,
        0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3032336369000000, 0x636f343230317769, 0x323031776f303436, 0x1030777031776b34, 0x0200000001000000,
        0x0000020000000a02, 0x0d000000040a0a00, 0x000003000000080d, 0x0a00000020020200, 0x100d0d000000200a, 0x0500000003000000, 0x0100000003000000, 0x0400000004000000,
        0x0100000001000000, 0x00001a0000008000, 0x6869323135636900, 0x6f323135636f3431, 0x32687333686b3768, 0x0300000010316870, 0x0000080202000000, 0x0800000003070700,
        0x0000020000000308, 0x0d000000040a0a00, 0x000003000000080d, 0x0900000040020200, 0x400a0a0000002009, 0x0400000003000000, 0x0100000003000000, 0x0700000007000000,
        0x0100000001000000, 0x0000180000000100, 0x6869303436636900, 0x6f303436636f3436, 0x687033686b343668, 0x0000030000000831, 0x0700000028020200, 0x0308080000000307,
        0x0a00000002000000, 0x020d0d000000080a, 0x0200000003000000, 0x100a0a0000001002, 0x0000200d0d000000, 0x000004ffffffff00, 0x0000010000000100, 0x0000040000000400,
        0x0100010000000100, 0x670000001b000000, 0x3168693263693233, 0x35686f32636f3231, 0x7032687333686b36, 0x0001000000083168, 0x0000000307070000, 0x00020d0d00000001,
        0x0202000000040000, 0x0003080800000002, 0x000000080a0a0000, 0x00000000000e0d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000002000010000,
        0x3233636900000017, 0x33636f3231356869, 0x33686b323135686f, 0x0200000010316870, 0x0000030707000000, 0x0100000003080800, 0x0000080d0d000000, 0x2002020000000300,
        0x0000100909000000, 0x03000000100a0a00, 0x0300000005000000, 0x0700000003000000, 0x0700000001000000, 0x0000040001000000, 0x3863690000002800, 0x3177693034366869,
        0x686f38636f343230, 0x323031776f303436, 0x7033776b33686b34, 0x0000103177703168, 0x200c0c0000000100, 0x0400000000000000, 0x0000080202000000, 0x0a00000003080800,
        0x100d0d000000100a, 0x0400000003000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x0000380000000100, 0x3264693233636900, 0x6934323268693432,
        0x3233636f30363177, 0x32686f343232646f, 0x6b303631776f3432, 0x33776b33686b3364, 0x7770316870316470, 0x0000030000000831, 0x0700000003060600, 0x0308080000000307,
        0x0a00000002000000, 0x020d0d000000020a, 0x0200000003000000, 0x100a0a0000002002, 0x0000140d0d000000, 0x0000040000000300, 0x0000030000000100, 0x0000010000000800,
        0x6400010000000800, 0x690000001e000000, 0x3335393277693163, 0x35776f3031636f32, 0x733031776b353039, 0x0000103077703577, 0x050d0d0000000100, 0x0200000000000000,
        0x0000100909000000, 0x00000000100a0a00, 0x0100000005000000, 0x0800000001000000, 0x0100000008000000, 0x0004000101000000, 0x3233670000001700, 0x6f36356869346369,
        0x686b3635686f3463, 0x0000001031687033, 0x0003070700000001, 0x0d0d000000010000, 0x0000000400000008, 0x0808000000040202, 0x0010090900000003, 0x000000100a0a0000,
        0x0000000500000000, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001c0000, 0x3930347769303436, 0x776f303233636f36, 0x7031776b36393034,
        0x0001000000103077, 0x0000001402020000, 0x00020a0a00000002, 0x000000080d0d0000, 0x0020020200000003, 0x000000400a0a0000, 0x0003000000200d0d, 0x0003000000050000,
        0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x343663690000001c, 0x6f36393034776930, 0x3034776f30323363, 0x30777031776b3639, 0x0000000100000010,
        0x0001000000140202, 0x000000020a0a0000, 0x0020020200000003, 0x000000200a0a0000, 0x0003000000200d0d, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000,
        0x0000008001010000, 0x353263690000001b, 0x636f363931776936, 0x3931776f34323031, 0x0830777031776b36, 0x0200000001000000, 0x0000020000000802, 0x0d000000080a0a00,
        0x000003000000040d, 0x0900000020020200, 0x200a0a0000002009, 0x0400000003000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x00001c0000008001,
        0x6869323135636900, 0x34323031636f3832, 0x7331686b3431686f, 0x0000083068703268, 0x1002020000000100, 0x0900000002000000, 0x080a0a0000000409, 0x0200000003000000,
        0x2009090000002002, 0x0000200a0a000000, 0x0000040000000300, 0x0000030000000300, 0x0000010000000700, 0x0400010000000700, 0x6900000016000000, 0x6f38646936353263,
        0x6b38646f32313563, 0x0000083164703364, 0x080c0c0000000100, 0x0200000002000000, 0x080a0a0000000202, 0x0200000003000000, 0x080a0a0000002002, 0x0000100d0d000000,
        0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x0200010000000100, 0x6900000018000000, 0x3436686930343663, 0x36686f303436636f, 0x1031687033686b34,
        0x0200000003000000, 0x0307070000001402, 0x0000030808000000, 0x080d0d0000000100, 0x0200000004000000, 0x0209090000002002, 0x0000400a0a000000, 0x03000000080d0d00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3663690000001800, 0x636f343668693034, 0x6b3436686f303436, 0x0000103168703368,
        0x1402020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000020a0a000000, 0x03000000040d0d00, 0x0000200202000000, 0x0d000000200a0a00, 0x000003000000100d,
        0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1700000020000100, 0x6932336369000000, 0x6f33636f32333268, 0x7039686b34323268, 0x0002000000103068,
        0x0000000907070000, 0x0001000000090808, 0x000000080d0d0000, 0x0020020200000003, 0x0000002009090000, 0x0003000000100a0a, 0x0003000000050000, 0x0004000000010000,
        0x0001000000040000, 0x0000002000010000, 0x3233636900000017, 0x33636f3233326869, 0x39686b343232686f, 0x0200000010306870, 0x0000090707000000, 0x0100000009080800,
        0x0000080d0d000000, 0x2002020000000300, 0x0000200909000000, 0x03000000100a0a00, 0x0300000005000000, 0x0700000003000000, 0x0700000001000000, 0x0000100001000000,
        0x3363690000001a00, 0x36636f3030336869, 0x686b303531686f34, 0x1033687032687337, 0x0c00000002000000, 0x0a0d0d000000020c, 0x0400000000000000, 0x0000040202000000,
        0x0900000008080800, 0x400a0a0000001009, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001a0000000200, 0x6930323931636900,
        0x383231636f363168, 0x33686b3631686f30, 0x0300000010316870, 0x00003c0202000000, 0x0800000003070700, 0x0000010000000308, 0x04000000020d0d00, 0x0000200202000000,
        0x0a00000002090900, 0x080d0d000000400a, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001a0000000100, 0x6930323931636900,
        0x383231636f363168, 0x33686b3631686f30, 0x0300000010316870, 0x0000080202000000, 0x0800000003070700, 0x0000010000000308, 0x0300000008020200, 0x0000200202000000,
        0x0d000000200a0a00, 0x000000000000100d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1800000010000100, 0x3635326369000000, 0x3532636f38336869,
        0x33686b3833686f36, 0x0300000010316870, 0x0000080202000000, 0x0800000003070700, 0x0000020000000308, 0x0d000000020a0a00, 0x000003000000080d, 0x0900000020020200,
        0x400a0a0000001009, 0x0400000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001c0000000200, 0x6930363532636900, 0x3231636f34343177,
        0x6b343431776f3038, 0x0000083077703177, 0x5002020000000100, 0x0200000001000000, 0x0000040000000202, 0x0900000010020200, 0x100a0a0000000209, 0x0000100d0d000000,
        0x0000050000000000, 0x0000020000000100, 0x00000b0000000100, 0x0000010000000b00, 0x6900000014000008, 0x636f303168693863, 0x33686b38686f3031, 0x0200000008306870,
        0x0000030707000000, 0x0100000003080800, 0x0000020a0a000000, 0x0802020000000300, 0x0000100909000000, 0x00000000080a0a00, 0x0100000005000000, 0x0100000002000000,
        0x0800000008000000, 0x0008000001000000, 0x3863690000001400, 0x3031636f30316869, 0x687033686b38686f, 0x0000020000001030, 0x0800000003070700, 0x0000000000000308,
        0x1002020000000300, 0x0000100909000000, 0x00000000100a0a00, 0x0300000005000000, 0x0a00000003000000, 0x0a00000001000000, 0x0000200101000000, 0x3863690000001400,
        0x6f38636f35326469, 0x647035646b353264, 0x0000020000001032, 0x0d000000190c0c00, 0x000000000000050d, 0x0802020000000300, 0x0000100909000000, 0x02000000100a0a00,
        0x0300000005000000, 0x0400000001000000, 0x0100000004000000, 0x0000010001000000, 0x3163690000001700, 0x636f333168693832, 0x686b3331686f3233, 0x0000001031687033,
        0x0003070700000002, 0x0000000308080000, 0x0004020200000001, 0x0202000000030000, 0x00100a0a00000020, 0x000000080d0d0000, 0x0000000500000000, 0x0000000200000003,
        0x0000000700000001, 0x0010000100000007, 0x63690000001b0000, 0x6f35376869383231, 0x3833686f38323163, 0x687032687333686b, 0x0000030000001031, 0x0800000003070700,
        0x040a0a0000000308, 0x0300000000000000, 0x0000400202000000, 0x0a00000010090900, 0x000003000000200a, 0x0000030000000500, 0x0000070000000300, 0x0000070000000100,
        0x1c00000010000100, 0x3635326369000000, 0x636f353236357769, 0x323635776f383231, 0x1030777031776b35, 0x0d00000001000000, 0x0000020000007d0d, 0x0a00000008020200,
        0x000003000000020a, 0x0900000010020200, 0x400a0a0000001009, 0x0400000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001a0000000100,
        0x3232686933636900, 0x686f383231636f34, 0x34687334686b3635, 0x0100000008306870, 0x0000040707000000, 0x020a0a0000000100, 0x0200000004000000, 0x0408080000000402,
        0x0000200a0a000000, 0x00000000080d0d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000020001000000, 0x3263690000001800, 0x636f386869303635,
        0x6b38686f30383231, 0x0000103168703368, 0x5002020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000020a0a000000, 0x03000000080d0d00, 0x0000200202000000,
        0x0a00000002090900, 0x000003000000400a, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1800000001000100, 0x3635326369000000, 0x3231636f38686930,
        0x33686b38686f3038, 0x0300000010316870, 0x0000500202000000, 0x0800000003070700, 0x0000010000000308, 0x03000000020a0a00, 0x0000200202000000, 0x0d000000100a0a00,
        0x000003000000080d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x2000000001000100, 0x3635326369000000, 0x3434313236327769, 0x32776f383231636f,
        0x31776b3434313236, 0x0100000010307770, 0x0000080202000000, 0x020a0a0000000200, 0x0000020d0d000000, 0x2002020000000300, 0x0000400a0a000000, 0x03000000200d0d00,
        0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000020001000000, 0x3163690000001d00, 0x3230317769303832, 0x776f303436636f34, 0x7031776b34323031,
        0x0001000000103077, 0x0000002802020000, 0x00040a0a00000002, 0x000000080d0d0000, 0x0020020200000003, 0x000000200a0a0000, 0x0003000000200d0d, 0x0003000000050000,
        0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x323163690000001d, 0x3432303177693038, 0x31776f303436636f, 0x777031776b343230, 0x0000010000001030,
        0x0200000028020200, 0x0000040a0a000000, 0x03000000040d0d00, 0x0000200202000000, 0x0d000000200a0a00, 0x000003000000200d, 0x0000030000000400, 0x0000040000000100,
        0x0000010000000400, 0x1900000080010100, 0x3231356369000000, 0x3032636f39347769, 0x776b3934776f3834, 0x0000000830777031, 0x0010020200000001, 0x0909000000020000,
        0x00080a0a00000004, 0x0202000000030000, 0x0020090900000020, 0x000000200a0a0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0020000100000001,
        0x3367000000170000, 0x3635686934636932, 0x6b3635686f34636f, 0x0000083168703368, 0x0307070000000100, 0x0d00000001000000, 0x000004000000080d, 0x0800000008020200,
        0x2009090000000408, 0x0000080a0a000000, 0x0000040000000200, 0x0000010000000300, 0x0000070000000700, 0x2000010000000100, 0x6700000017000000, 0x3568693463693233,
        0x3635686f34636f36, 0x000831687033686b, 0x0707000000010000, 0x0000000100000003, 0x0004000000080d0d, 0x0000000402020000, 0x0909000000040808, 0x00080a0a00000020,
        0x0005000000020000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3668693231356369, 0x686f323135636f34, 0x31687033686b3436,
        0x0000000300000010, 0x0707000000100202, 0x0003080800000003, 0x0a0a000000020000, 0x00040d0d00000004, 0x0202000000030000, 0x00200a0a00000020, 0x000000100d0d0000,
        0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000160000, 0x636f373168693233, 0x686b3731686f3233, 0x0000001031687033,
        0x0002020200000003, 0x0000000307070000, 0x0002000000030808, 0x000000020a0a0000, 0x0003000000040d0d, 0x0000001002020000, 0x0d0d000000100a0a, 0x0000000300000008,
        0x0000000300000004, 0x0000000400000001, 0x0000000100000004, 0x0017000000200001, 0x6869323363690000, 0x686f33636f323135, 0x687033686b323135, 0x0000020000000831,
        0x0800000003070700, 0x0000010000000308, 0x03000000080d0d00, 0x0000200202000000, 0x0a00000010090900, 0x000002000000080a, 0x0000030000000400, 0x0000070000000100,
        0x0000010000000700, 0x1700000020000100, 0x6932336369000000, 0x6f33636f32313568, 0x7033686b32313568, 0x0002000000083168, 0x0000000307070000, 0x0001000000030808,
        0x000000080d0d0000, 0x0020020200000003, 0x0000001009090000, 0x0002000000080a0a, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000,
        0x323163690000001a, 0x636f323168693038, 0x3231686f30383231, 0x000831687033686b, 0x0202000000030000, 0x0003070700000028, 0x0000000308080000, 0x0002020200000002,
        0x000000020d0d0000, 0x0010020200000004, 0x0000000209090000, 0x0d0d000000200a0a, 0xffffffff00000006, 0x0000000100000004, 0x0000000400000001, 0x0000000100000004,
        0x001c000000200101, 0x6369363532670000, 0x6f36393034776931, 0x36393034776f3163, 0x000830777031776b, 0x0000000000000000, 0x0101000000020000, 0x0010090900000020,
        0x0004000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3377693635326369, 0x776f383231636f32, 0x30777031776b3233,
        0x0000000000000008, 0x0008020200000001, 0x0202000000030000, 0x00080a0a00000020, 0x000000080d0d0000, 0x0000000400000000, 0x0000000100000003, 0x0000000400000004,
        0x0001000100000001, 0x6369000000180000, 0x6f32337769363532, 0x3233776f38323163, 0x000830777031776b, 0x0001000000000000, 0x0000000402020000, 0x0040020200000003,
        0x000000080a0a0000, 0x0000000000100d0d, 0x0003000000050000, 0x000a000000030000, 0x000a000000010000, 0x0000004001010000, 0x343663690000001c, 0x636f323230317769,
        0x303031776f383231, 0x3077703631776b37, 0x0000000200000010, 0x0d0d000000020909, 0x0000000200000013, 0x0a0a000000020202, 0x0000000300000002, 0x0909000000200202,
        0x00400a0a00000008, 0x0004000000020000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001c00000001, 0x3177693032336369, 0x303436636f343230,
        0x776b34323031776f, 0x0000000830777031, 0x0014020200000001, 0x0a0a000000020000, 0x00080d0d00000008, 0x0202000000030000, 0x00100a0a00000010, 0x000000200d0d0000,
        0x0000000400000000, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000140000, 0x33636f3968693233, 0x7033686b37686f32, 0x0002000000083068,
        0x0000000307070000, 0x0001000000030808, 0x0000000202020000, 0x0010020200000003, 0x000000100a0a0000, 0x0000000000070d0d, 0x0003000000050000, 0x0004000000010000,
        0x0001000000040000, 0x0000008000010000, 0x3135636900000019, 0x32636f3934776932, 0x6b3934776f383430, 0x0000103077703177, 0x0802020000000100, 0x0900000002000000,
        0x040a0a0000000809, 0x0200000003000000, 0x1009090000004002, 0x0000400a0a000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100,
        0x6900000016000000, 0x636f343668693463, 0x6b3436686f323135, 0x0000103168703368, 0x0307070000000100, 0x0d00000001000000, 0x000004000000040d, 0x0800000004020200,
        0x400a0a0000000408, 0x0000100d0d000000, 0x0000050000000300, 0x0000030000000100, 0x0000010000000b00, 0x0000010000000b00, 0x6900000014000008, 0x636f303168693863,
        0x33686b38686f3031, 0x0200000008306870, 0x0000020c0c000000, 0x00000000080d0d00, 0x0200000003000000, 0x0809090000000802, 0x0000100a0a000000, 0x0000050000000000,
        0x0000030000000100, 0x0000010000000800, 0x0000010000000800, 0x6900000014000008, 0x636f303168693863, 0x33686b38686f3031, 0x0200000010306870, 0x0000040c0c000000,
        0x00000000080d0d00, 0x0200000003000000, 0x1009090000000802, 0x0000100a0a000000, 0x0000040000000000, 0x0000010000000300, 0x0000070000000700, 0x0500010000000100,
        0x6900000018000000, 0x3635686938323163, 0x35686f383231636f, 0x0831687033686b36, 0x0200000003000000, 0x0307070000000202, 0x0000030808000000, 0x0202020000000100,
        0x0200000003000000, 0x200a0a0000002002, 0x00000e0d0d000000, 0x0000040000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001a000000,
        0x3135686938323163, 0x686f383231636f32, 0x687033686b323135, 0x0000030000000831, 0x0700000004020200, 0x0308080000000307, 0x0a00000002000000, 0x080d0d000000040a,
        0x0200000003000000, 0x200a0a0000002002, 0x0000100d0d000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000018000000,
        0x3436686930323363, 0x36686f303233636f, 0x0831687033686b34, 0x0200000003000000, 0x0307070000001402, 0x0000030808000000, 0x080a0a0000000200, 0x0000040d0d000000,
        0x1002020000000300, 0x0000100a0a000000, 0xff000000100d0d00, 0x0300000004ffffff, 0x0400000001000000, 0x0100000004000000, 0x0000010001000000, 0x3363690000001b00,
        0x636f343230316869, 0x6b363532686f3432, 0x3368703468733768, 0x0000000100000008, 0x0002000000070707, 0x000000020a0a0000, 0x0004000000080d0d, 0x0000000402020000,
        0x0a0a000000080808, 0x00200d0d00000010, 0x0005000000010000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001600000002, 0x6f34366869346369,
        0x3436686f30323363, 0x001031687033686b, 0x0707000000010000, 0x0000000100000003, 0x0005000000040d0d, 0x0000000402020000, 0x0909000000040808, 0x00400a0a00000002,
        0x000000080d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x6369000000160000, 0x33636f3436686934, 0x686b3436686f3032,
        0x0000001031687033, 0x0003070700000001, 0x0004000000000000, 0x0000000402020000, 0x0a0a000000040808, 0x00100d0d00000040, 0x0005000000030000, 0x0003000000010000,
        0x0001000000080000, 0x0001000000080000, 0x0000001e00000064, 0x3539327769316369, 0x776f3031636f3233, 0x3031776b35303935, 0x0010307770357773, 0x0d0d000000010000,
        0x0000000000000040, 0x0008080800000003, 0x0000000a09090000, 0x0000000000100a0a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000,
        0x3233636900000038, 0x3268693432326469, 0x6f30363177693432, 0x343232646f323363, 0x31776f343232686f, 0x33686b33646b3036, 0x687031647033776b, 0x0000001031777031,
        0x0003060600000003, 0x0000000307070000, 0x0001000000030808, 0x0000000202020000, 0x0010020200000003, 0x000000200a0a0000, 0x0000000000200d0d, 0x0003000000040000,
        0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x313563690000001a, 0x636f343837776932, 0x343837776f383231, 0x000830777031776b, 0x0202000000010000,
        0x0000000100000010, 0x0003000000080a0a, 0x0000002002020000, 0x0a0a000000200909, 0x0000000200000010, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007,
        0x001c000000020001, 0x3038323163690000, 0x31636f3635327769, 0x363532776f303832, 0x001030777031776b, 0x0202000000010000, 0x0000000200000028, 0x0d0d000000040a0a,
        0x0000000300000004, 0x0a0a000000200202, 0x00200d0d00000020, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001c00000001,
        0x7769303832316369, 0x383231636f363532, 0x776b363532776f30, 0x0000001030777031, 0x0028020200000001, 0x0a0a000000020000, 0x00080d0d00000002, 0x0202000000030000,
        0x00400a0a00000020, 0x000000080d0d0000, 0x0000000500000003, 0x0000000300000003, 0x000000010000000a, 0x002001010000000a, 0x6369000000140000, 0x38636f3532646938,
        0x7035646b3332646f, 0x0001000000103164, 0x000000170c0c0000, 0x0000000300000000, 0x0909000000080202, 0x00100a0a00000010, 0x0004000000020000, 0x0001000000030000,
        0x0004000000040000, 0x0101000000010000, 0x0000001b00000080, 0x3268693635326369, 0x686f363532636f38, 0x32687333686b3431, 0x0300000008316870, 0x0000080202000000,
        0x0800000003070700, 0x0000020000000308, 0x0a00000004090900, 0x000003000000080a, 0x0900000020020200, 0x200a0a0000002009, 0x0400000003000000, 0x0100000003000000,
        0x0400000004000000, 0x0100000001000000, 0x00001b0000008001, 0x6934323031636900, 0x3135636f36393177, 0x776b363931776f32, 0x0000000830777031, 0x0020020200000001,
        0x0a0a000000020000, 0x00040d0d00000008, 0x0202000000030000, 0x0020090900000020, 0x000000200a0a0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004,
        0x0080010100000001, 0x63690000001a0000, 0x3438377769323135, 0x37776f363532636f, 0x30777031776b3438, 0x0000000100000008, 0x0001000000100202, 0x000000080a0a0000,
        0x0020020200000003, 0x0000002009090000, 0x0002000000200a0a, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000001800010000, 0x6932336700000017,
        0x636f363568693863, 0x33686b3635686f38, 0x0100000008316870, 0x0000030707000000, 0x080d0d0000000100, 0x0200000004000000, 0x0408080000000802, 0x00000c0909000000,
        0x02000000080a0a00, 0x0300000005000000, 0x0100000002000000, 0x0700000007000000, 0x0000100001000000, 0x3563690000001d00, 0x3434343177693231, 0x776f34323031636f,
        0x7031776b34343431, 0x0001000000103077, 0x000000200a0a0000, 0x0002020200000002, 0x0000000805050000, 0x0040020200000003, 0x0000001009090000, 0x0003000000200a0a,
        0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x323163690000001a, 0x636f323135686938, 0x323135686f383231, 0x001031687033686b,
        0x0202000000030000, 0x0003070700000004, 0x0000000308080000, 0x00020a0a00000002, 0x000000020d0d0000, 0x0020020200000003, 0x000000400a0a0000, 0x0003000000200d0d,
        0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x3032636900000014, 0x6f3035636f366869, 0x30687033686b3468, 0x0000000200000008,
        0x0808000000030707, 0x0000000000000003, 0x0020020200000003, 0x000000080a0a0000, 0x0000000000040d0d, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000,
        0x0000000200010000, 0x3436636900000018, 0x33636f3639686930, 0x686b3639686f3032, 0x0000000831687033, 0x0028020200000003, 0x0000000307070000, 0x0001000000030808,
        0x000000020d0d0000, 0x0010020200000004, 0x0000000209090000, 0x0d0d000000200a0a, 0x0000000000000010, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004,
        0x002a000000200101, 0x3168693463690000, 0x3239317769303830, 0x3031686f31636f30, 0x30323931776f3038, 0x687033776b33686b, 0x0000000831777031, 0x0003070700000001,
        0x0d0d000000010000, 0x0000000400000002, 0x0808000000040202, 0x0020090900000008, 0x000000080a0a0000, 0x0000000400000002, 0x0000000100000003, 0x0000000700000007,
        0x0001000100000001, 0x6369000000180000, 0x6f34366869303639, 0x3436686f30323363, 0x000831687033686b, 0x0202000000030000, 0x000307070000003c, 0x0000000308080000,
        0x00080a0a00000002, 0x000000040d0d0000, 0x0010020200000003, 0x000000100a0a0000, 0xffff000000100d0d, 0x000300000005ffff, 0x0007000000010000, 0x0001000000070000,
        0x0000000100010000, 0x303263690000000b, 0x10383231636f3834, 0x0200000001000000, 0x0000010000001002, 0x0300000008020200, 0x0000100202000000, 0x0d000000100a0a00,
        0x000000000000080d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1a00000001000100, 0x3231356369000000, 0x35636f3635326869, 0x6b363532686f3231,
        0x0000103168703368, 0x1002020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000040a0a000000, 0x03000000080d0d00, 0x0000200202000000, 0x0d000000200a0a00,
        0x000003000000200d, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3832316369000000, 0x31636f3631686930, 0x686b38686f303832,
        0x0831687032687333, 0x0200000003000000, 0x0307070000001402, 0x0000030808000000, 0x0202020000000100, 0x0200000003000000, 0x100a0a0000002002, 0x0000080d0d000000,
        0x0000050000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001a000000, 0x3135686936353263, 0x686f383231636f32, 0x687033686b323135,
        0x0000030000001031, 0x0700000008020200, 0x0308080000000307, 0x0a00000002000000, 0x040d0d000000040a, 0x0200000003000000, 0x200a0a0000002002, 0x0000200d0d000000,
        0x0000050000000300, 0x0000010000000100, 0x00000b0000000b00, 0x0000010000000100, 0x6900000013000008, 0x636f303168693263, 0x7033686b38686f38, 0x0001000000083068,
        0x0000000307070000, 0x0000000400000000, 0x0808000000020202, 0x00080a0a00000003, 0x000000080d0d0000, 0x0000000500000000, 0x0000000100000001, 0x0000000800000008,
        0x0800000100000001, 0x6369000000130000, 0x38636f3031686932, 0x687033686b38686f, 0x0000010000001030, 0x0000000003070700, 0x0200000004000000, 0x0308080000000202,
        0x0000100a0a000000, 0x00000000080d0d00, 0x0300000005000000, 0x0700000003000000, 0x0700000001000000, 0x0000400001000000, 0x3163690000001700, 0x636f363177693832,
        0x776b3531776f3436, 0x0000001030777032, 0x0000000100000000, 0x0004000000080202, 0x0000000802020000, 0x0a0a000000020808, 0x00100d0d00000010, 0x0004ffffffff0000,
        0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001600000020, 0x3531686930316369, 0x3331686f3032636f, 0x000830687033686b, 0x0707000000020000,
        0x0002080800000003, 0x0a0a000000020000, 0x00040d0d00000004, 0x0202000000040000, 0x0002080800000010, 0x0000002009090000, 0x0003000000080a0a, 0x0001000000050000,
        0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x693163690000002c, 0x3030317769303868, 0x686f31636f303536, 0x36303031776f3038, 0x32776b31686b3035,
        0x3131777030687033, 0x0000000000000010, 0x0000000300000000, 0x0a0a000000170808, 0x00100d0d00000010, 0x0005000000000000, 0x0001000000010000, 0x0004000000040000,
        0x0001000000010000, 0x0000002c00000001, 0x6930386869316369, 0x6f30353630303177, 0x776f3038686f3163, 0x686b303536303031, 0x3068703332776b31, 0x0000001031317770,
        0x0000000100000000, 0x0003000000040d0d, 0x0000001708080000, 0x0d0d000000100a0a, 0x0000000000000010, 0x0000000300000004, 0x0000000700000001, 0x0000000100000007,
        0x0019000000010001, 0x3038323163690000, 0x3231636f30316869, 0x33686b38686f3038, 0x0300000008306870, 0x0000140202000000, 0x0800000003070700, 0x0000010000000308,
        0x0300000002020200, 0x0000200202000000, 0x0d000000100a0a00, 0x000000000000080d, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1900000002000100,
        0x3034366369000000, 0x3231636f34326869, 0x686b3432686f3038, 0x0000000831687033, 0x0028020200000003, 0x0000000307070000, 0x0001000000030808, 0x000000040a0a0000,
        0x0010020200000003, 0x000000200a0a0000, 0xffff000000180d0d, 0x000300000005ffff, 0x0001000000020000, 0x0007000000070000, 0x0000004000010000, 0x3532636900000019,
        0x32636f3039776936, 0x776b3935776f3635, 0x0000103077703233, 0x2008080000000200, 0x0000080a0a000000, 0x0802020000000200, 0x0000040505000000, 0x2002020000000300,
        0x00001e0505000000, 0xff000000200a0a00, 0x0300000005ffffff, 0x0700000001000000, 0x0100000007000000, 0x0000200001000000, 0x3663690000001700, 0x636f363532686934,
        0x686b363532686f33, 0x0000001034687039, 0x0002020200000003, 0x0000000907070000, 0x0001000000090808, 0x000000080d0d0000, 0x0020020200000003, 0x0000002009090000,
        0x0003000000100a0a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x313563690000001a, 0x636f363532686932, 0x363532686f363532,
        0x001031687033686b, 0x0202000000030000, 0x0003070700000010, 0x0000000308080000, 0x00040a0a00000002, 0x000000080d0d0000, 0x0020020200000003, 0x000000200a0a0000,
        0x0003000000200d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x393163690000001d, 0x3432303177693032, 0x31776f303436636f,
        0x777031776b343230, 0x0000010000001030, 0x020000003c020200, 0x0000080a0a000000, 0x03000000040d0d00, 0x0000200202000000, 0x0d000000200a0a00, 0x000003000000200d,
        0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1d00000001000100, 0x3239316369000000, 0x6f34323031776930, 0x3031776f30343663, 0x30777031776b3432,
        0x0000000100000010, 0x00020000003c0202, 0x000000040a0a0000, 0x0003000000040d0d, 0x0000002002020000, 0x0d0d000000200a0a, 0x0000000300000020, 0x0000000300000004,
        0x0000000700000001, 0x0000000100000007, 0x001a000000020001, 0x3036353263690000, 0x3231636f34326869, 0x686b3432686f3038, 0x0000000831687033, 0x00a0020200000003,
        0x0000000307070000, 0x0001000000030808, 0x000000040a0a0000, 0x0010020200000003, 0x000000200a0a0000, 0xffff000000180d0d, 0x000300000005ffff, 0x0007000000010000,
        0x0001000000070000, 0x0000004000010000, 0x323163690000001b, 0x636f323532776938, 0x373332776f363532, 0x103077703631776b, 0x0200000002000000, 0x1008080000000402,
        0x0a00000002000000, 0x080d0d000000040a, 0x0200000003000000, 0x400a0a0000002002, 0x0000200d0d000000, 0x000004ffffffff00, 0x0000010000000300, 0x0000070000000700,
        0x0200010000000100, 0x6900000019000000, 0x3468693038323163, 0x686f303436636f38, 0x31687033686b3834, 0x0000000300000008, 0x0707000000500202, 0x0003080800000003,
        0x0d0d000000010000, 0x0000000400000004, 0x0909000000100202, 0x00200a0a00000002, 0x0000000c0d0d0000, 0x00000005ffffffff, 0x0000000100000001, 0x0000000700000007,
        0x0001000100000001, 0x6369000000140000, 0x32636f3635686932, 0x7033686b3635686f, 0x0001000000103168, 0x0000000307070000, 0x00080d0d00000001, 0x0202000000040000,
        0x0003080800000002, 0x000000100a0a0000, 0x0000000000070d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x3231636900000018,
        0x31636f3868693038, 0x686b38686f303832, 0x0000001031687033, 0x0028020200000003, 0x0000000307070000, 0x0000000000030808, 0x0202000000040000, 0x0002090900000020,
        0x000000200a0a0000, 0x0003000000080d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3231636900000018, 0x31636f3868693038,
        0x686b38686f303832, 0x0000001031687033, 0x0028020200000003, 0x0000000307070000, 0x0001000000030808, 0x000000020a0a0000, 0x0020020200000003, 0x000000100a0a0000,
        0x0003000000080d0d, 0x0003000000050000, 0x0001000000020000, 0x0007000000070000, 0x0000004000010000, 0x323163690000001b, 0x636f323532776938, 0x373332776f363532,
        0x103077703631776b, 0x0800000002000000, 0x080a0a0000001008, 0x0200000002000000, 0x0805050000000402, 0x0200000003000000, 0x2005050000002002, 0x0000200a0a000000,
        0x000004ffffffff00, 0x0000010000000300, 0x0000040000000400, 0x8001010000000100, 0x690000001b000000, 0x3133776936353263, 0x776f3436636f3633, 0x7031776b36333133,
        0x0001000000083077, 0x0000000802020000, 0x00080a0a00000002, 0x000000020d0d0000, 0x0020020200000003, 0x0000001009090000, 0x0003000000080a0a, 0x0003000000040000,
        0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x303163690000001c, 0x636f343168693432, 0x6b37686f38343032, 0x3068703268733168, 0x0000000100000008,
        0x0002000000200202, 0x0000000409090000, 0x0003000000080a0a, 0x0000002002020000, 0x0a0a000000200909, 0x0000000300000020, 0x0000000300000005, 0x0000000700000001,
        0x0000000100000007, 0x0019000000400001, 0x6936353263690000, 0x363532636f303977, 0x3233776b3935776f, 0x0200000010307770, 0x0000080202000000, 0x0100000020080800,
        0x0000040d0d000000, 0x2002020000000300, 0x0000200a0a000000, 0xff000000100d0d00, 0x0300000004ffffff, 0x0700000001000000, 0x0100000007000000, 0x0000020001000000,
        0x3363690000001600, 0x636f363968693032, 0x33686b3639686f34, 0x0300000008316870, 0x00000a0202000000, 0x0800000003070700, 0x0000010000000308, 0x03000000080d0d00,
        0x0000200202000000, 0x0d000000080a0a00, 0x000000000000100d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1a00000020010100, 0x6869336369000000,
        0x6f33636f34323031, 0x7337686b32313568, 0x0000103368703268, 0x0707070000000200, 0x0000020808000000, 0x040d0d0000000100, 0x0200000004000000, 0x0408080000000402,
        0x0000200909000000, 0x03000000100a0a00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3163690000001800, 0x6f32313568693832,
        0x6b323135686f3363, 0x0000103168703368, 0x0402020000000300, 0x0000030707000000, 0x0100000003080800, 0x0000040d0d000000, 0x2002020000000300, 0x0000100a0a000000,
        0x03000000200d0d00, 0x0300000005000000, 0x0100000002000000, 0x0700000007000000, 0x0000400001000000, 0x3163690000001700, 0x636f363177693832, 0x776b3531776f3436,
        0x0000001030777032, 0x0002080800000002, 0x000000020a0a0000, 0x0002020200000001, 0x0202000000030000, 0x0010050500000010, 0x000000200a0a0000, 0x00000005ffffffff,
        0x0000000200000003, 0x0000000700000001, 0x0010000100000007, 0x63690000001b0000, 0x6f35376869363532, 0x3833686f32313563, 0x687032687331686b, 0x0000010000001030,
        0x00000000100a0a00, 0x0200000003000000, 0x1009090000004002, 0x0000200a0a000000, 0x0000050000000300, 0x0000010000000300, 0x0000040000000400, 0x2000010000000100,
        0x6700000019000000, 0x6869363163693233, 0x686f3631636f3431, 0x31687033686b3431, 0x0000000200000010, 0x0808000000030707, 0x0000000100000003, 0x0003000000040d0d,
        0x0000002002020000, 0x0a0a000000200909, 0x0000000300000010, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004, 0x001a000000010001, 0x3268693363690000,
        0x686f3436636f3432, 0x687337686b323131, 0x0000000833687032, 0x0007070700000001, 0x0a0a000000020000, 0x00040d0d00000002, 0x0202000000040000, 0x0008080800000004,
        0x000000100a0a0000, 0x0001000000200d0d, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x693363690000001a, 0x3436636f34323268,
        0x37686b323131686f, 0x0008336870326873, 0x0707000000020000, 0x0002080800000007, 0x0a0a000000020000, 0x00040d0d00000002, 0x0202000000040000, 0x0004080800000004,
        0x000000100a0a0000, 0x0001000000200d0d, 0x0003000000050000, 0x0007000000030000, 0x0007000000010000, 0x0000004000010000, 0x3231636900000018, 0x31636f3831776938,
        0x776b3531776f3832, 0x0000001030777034, 0x0040090900000001, 0x0202000000020000, 0x00020a0a00000002, 0x0202000000030000, 0x00100a0a00000010, 0x000000100d0d0000,
        0x00000004ffffffff, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001a0000, 0x3834686930383231, 0x686f30383231636f, 0x31687033686b3834,
        0x0000000300000008, 0x0707000000500202, 0x0003080800000003, 0x0a0a000000020000, 0x00020d0d00000008, 0x0202000000030000, 0x00200a0a00000010, 0x000000180d0d0000,
        0x00000005ffffffff, 0x0000000200000003, 0x0000000700000001, 0x0040000100000007, 0x6369000000180000, 0x6f38317769383231, 0x3531776f38323163, 0x001030777034776b,
        0x0808000000020000, 0x00040a0a00000004, 0x0505000000010000, 0x0000000300000004, 0x0505000000200202, 0x00200a0a00000008, 0x0005ffffffff0000, 0x0003000000030000,
        0x0001000000070000, 0x0001000000070000, 0x0000001800000040, 0x3177693635326369, 0x776f383231636f38, 0x30777034776b3531, 0x0000000100000010, 0x0002000000080909,
        0x0000000202020000, 0x0003000000080a0a, 0x0000002002020000, 0x0d0d000000100a0a, 0xffffffff00000010, 0x0000000300000005, 0x0000000400000001, 0x0000000100000004,
        0x0015000000010001, 0x6938323163690000, 0x686f3233636f3668, 0x1031687033686b36, 0x0700000002000000, 0x0308080000000307, 0x0200000001000000, 0x0000030000000402,
        0x0a00000020020200, 0x080d0d000000100a, 0x0500000000000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x0000180000000100, 0x6869383231636900,
        0x6f383231636f3835, 0x687033686b363568, 0x0000030000001030, 0x0700000008020200, 0x0308080000000307, 0x0a00000002000000, 0x040d0d000000040a, 0x0200000003000000,
        0x100a0a0000001002, 0x00000e0d0d000000, 0x0000050000000300, 0x0000010000000300, 0x0000040000000400, 0x0100010000000100, 0x6900000018000000, 0x3835686938323163,
        0x35686f383231636f, 0x1030687033686b36, 0x0200000003000000, 0x0307070000000402, 0x0000030808000000, 0x020a0a0000000200, 0x0000080d0d000000, 0x2002020000000300,
        0x0000200a0a000000, 0x03000000070d0d00, 0x0100000005000000, 0x0b00000001000000, 0x010000000b000000, 0x0008000001000000, 0x3863690000001400, 0x3031636f30316869,
        0x687033686b38686f, 0x0000010000001030, 0x0000000003070700, 0x0200000004000000, 0x0308080000000802, 0x0000100a0a000000, 0x00000000080d0d00, 0x0100000005000000,
        0x0800000001000000, 0x0100000008000000, 0x0008000001000000, 0x3863690000001400, 0x3031636f30316869, 0x687033686b38686f, 0x0000010000001030, 0x0000000003070700,
        0x0200000004000000, 0x0308080000000802, 0x0000100a0a000000, 0x00000000080d0d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000400001000000,
        0x3663690000001c00, 0x6f32323031776934, 0x3031776f38323163, 0x77703631776b3730, 0x0000020000001030, 0x0800000002020200, 0x0000020000001008, 0x0d000000020a0a00,
        0x000003000000020d, 0x0a00000020020200, 0x200d0d000000400a, 0x04ffffffff000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x0000180000000100,
        0x7769323135636900, 0x6f383231636f3233, 0x777031776b323377, 0x0000010000000830, 0x0100000002020200, 0x0000080202000000, 0x2002020000000300, 0x0000080a0a000000,
        0x00000000080d0d00, 0x0300000004000000, 0x0400000001000000, 0x0100000004000000, 0x0000010001000000, 0x3563690000001800, 0x636f323377693231, 0x6b3233776f383231,
        0x0000083077703177, 0x0000010000000000, 0x0300000008020200, 0x0000400202000000, 0x0d000000080a0a00, 0x000000000000080d, 0x0000030000000500, 0x0000070000000100,
        0x0000010000000700, 0x1a00000020010100, 0x7769316369000000, 0x3436636f32393138, 0x776b37373138776f, 0x0000103077703631, 0x0208080000000100, 0x0400000000000000,
        0x0000020202000000, 0x0a00000008080800, 0x0d0d0d000000400a, 0x04ffffffff000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000, 0x00001a0000000100,
        0x3033686933636900, 0x31686f3233636f30, 0x32687333686b3035, 0x0100000008316870, 0x0000030707000000, 0x020a0a0000000200, 0x0000020d0d000000, 0x0402020000000400,
        0x0000080808000000, 0x0d000000100a0a00, 0x000003000000190d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x0a00000001000100, 0x3230316369000000,
        0x0000103436636f34, 0x0802020000000100, 0x0200000001000000, 0x0000030000000802, 0x0a00000010020200, 0x080d0d000000100a, 0x0500000000000000, 0x0100000001000000,
        0x0700000007000000, 0x0100000001000000, 0x0000150000000100, 0x6931636934366700, 0x38686f31636f3868, 0x001031687033686b, 0x0707000000010000, 0x0000000000000003,
        0x0010010100000003, 0x0000000308080000, 0x0000000000080d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x323363690000001b,
        0x33636f3436686930, 0x686b3233686f3032, 0x1031687032687333, 0x0200000003000000, 0x0307070000000a02, 0x0000030808000000, 0x040a0a0000000200, 0x0000020d0d000000,
        0x2002020000000300, 0x0000200a0a000000, 0x03000000100d0d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3363690000001b00,
        0x636f343668693032, 0x6b3233686f303233, 0x3168703268733368, 0x0000000300000010, 0x07070000000a0202, 0x0003080800000003, 0x0a0a000000020000, 0x00020d0d00000004,
        0x0202000000030000, 0x00100a0a00000020, 0x000000100d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001c0000,
        0x3930347769303233, 0x776f303233636f36, 0x7031776b36393034, 0x0001000000103077, 0x0000000a02020000, 0x00020a0a00000002, 0x000000080d0d0000, 0x0020020200000003,
        0x000000200a0a0000, 0x0003000000200d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x323363690000001c, 0x6f36393034776930,
        0x3034776f30323363, 0x30777031776b3639, 0x0000000100000010, 0x00020000000a0202, 0x000000020a0a0000, 0x0003000000080d0d, 0x0000002002020000, 0x0d0d000000400a0a,
        0x0000000300000020, 0x0000000300000005, 0x0000000700000003, 0x0000000700000001, 0x001a000000200001, 0x3877693163690000, 0x6f3436636f323931, 0x31776b3737313877,
        0x0000001030777036, 0x0002090900000002, 0x000000110d0d0000, 0x00020a0a00000001, 0x0202000000040000, 0x0010080800000002, 0x0000001009090000, 0xffff000000100a0a,
        0x000300000004ffff, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x6934636900000027, 0x3436776930363368, 0x33686f3633636f30, 0x6b303436776f3036,
        0x31687033776b3368, 0x0100000008317770, 0x0000030707000000, 0x020d0d0000000100, 0x0200000004000000, 0x0808080000000402, 0x0000100a0a000000, 0xff000000200d0d00,
        0x0300000005ffffff, 0x0400000001000000, 0x0100000004000000, 0x0000800001000000, 0x3163690000001a00, 0x6f34383777693832, 0x3837776f32313563, 0x1030777031776b34,
        0x0200000001000000, 0x0000010000000202, 0x03000000040a0a00, 0x0000400202000000, 0x0d000000400a0a00, 0x0000030000001c0d, 0x0000030000000500, 0x0000070000000100,
        0x0000010000000700, 0x1e00000001000100, 0x3231356369000000, 0x6f36333535367769, 0x3536776f36353263, 0x777031776b363335, 0x0000010000001030, 0x0200000010020200,
        0x0000040a0a000000, 0x03000000020d0d00, 0x0000200202000000, 0x0d000000400a0a00, 0x000003000000200d, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700,
        0x1c00000001000100, 0x3230316369000000, 0x636f363931776934, 0x3931776f38343032, 0x1030777031776b36, 0x0200000001000000, 0x0000020000004002, 0x0d000000040a0a00,
        0x000003000000040d, 0x0a00000010020200, 0x070d0d000000200a, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x0000180000004000,
        0x7769363532636900, 0x6f383231636f3831, 0x777034776b353177, 0x0000020000001030, 0x0800000008020200, 0x0000010000000408, 0x03000000080d0d00, 0x0000200202000000,
        0x0a00000008090900, 0xffffff000000200a, 0x00000300000004ff, 0x0000070000000100, 0x0000010000000700, 0x1c00000001000100, 0x3036396369000000, 0x636f343230317769,
        0x323031776f303436, 0x0830777031776b34, 0x0200000001000000, 0x0000020000003c02, 0x0d000000080a0a00, 0x000003000000080d, 0x0a00000010020200, 0x200d0d000000100a,
        0x0400000000000000, 0x0100000001000000, 0x0700000007000000, 0x0100000001000000, 0x00001c0000000100, 0x3163693231356700, 0x6f31636f32337769, 0x776433776b383477,
        0x0000083631777037, 0x0000010000000000, 0x03000000080d0d00, 0x0000080101000000, 0x0d00000003080800, 0x000000000000080d, 0x0000030000000500, 0x0000070000000100,
        0x0000010000000700, 0x1900000001000100, 0x3635326369000000, 0x33636f3938327769, 0x776b393832776f32, 0x0000001030777031, 0x0002020200000001, 0x0202000000010000,
        0x0000000300000008, 0x0a0a000000100202, 0x00080d0d00000010, 0x0004000000000000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000, 0x0000001800000001,
        0x3668693436346369, 0x686f303631636f34, 0x31687033686b3436, 0x0000000300000008, 0x07070000000f0202, 0x0003080800000003, 0x0a0a000000020000, 0x00020d0d00000004,
        0x0202000000030000, 0x00100a0a00000020, 0x000000200d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0020010100000001, 0x6369000000190000,
        0x6f33316869323135, 0x31686f3432303163, 0x1031687033686b33, 0x0200000003000000, 0x0307070000002002, 0x0000030808000000, 0x020a0a0000000200, 0x0000080d0d000000,
        0x1002020000000300, 0x0000200909000000, 0x03000000200a0a00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3263690000001a00,
        0x6f36353268693635, 0x3532686f36353263, 0x1031687033686b36, 0x0200000003000000, 0x0307070000000802, 0x0000030808000000, 0x020a0a0000000200, 0x0000040d0d000000,
        0x2002020000000300, 0x0000400a0a000000, 0x03000000200d0d00, 0x0300000004000000, 0x0400000001000000, 0x0100000004000000, 0x0000800101000000, 0x3163690000001a00,
        0x6f34383777693832, 0x3837776f32313563, 0x0830777031776b34, 0x0200000001000000, 0x0000020000000402, 0x0d000000080a0a00, 0x000003000000040d, 0x0900000020020200,
        0x200a0a0000002009, 0x0400000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001d0000000200, 0x6930383231636900, 0x383231636f343268,
        0x33686b3231686f30, 0x0008316870326873, 0x0202000000030000, 0x0003070700000028, 0x0000000308080000, 0x0002020200000002, 0x000000020d0d0000, 0x0010020200000004,
        0x0000000209090000, 0x0d0d000000200a0a, 0xffffffff00000006, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x0017000000400001, 0x6938323163690000,
        0x6f3436636f363177, 0x777032776b353177, 0x0000020000001030, 0x0800000004020200, 0x0000010000000208, 0x0300000004090900, 0x0000200202000000, 0x0a00000010090900,
        0xffffff000000200a, 0x00000300000005ff, 0x0000010000000200, 0x0000070000000700, 0x3200000010010100, 0x6932336369000000, 0x6933336869373164, 0x6f3233636f333377,
        0x6f3333686f373164, 0x686b33646b333377, 0x7031647033776b33, 0x0000103177703168, 0x0306060000000400, 0x0000030707000000, 0x0a00000003080800, 0x000001000000020a,
        0x0300000002050500, 0x0000200202000000, 0x0a00000010090900, 0x000003000000100a, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1a00000002000100,
        0x3635326369000000, 0x31636f3631686930, 0x6b3631686f303832, 0x0000083168703368, 0xa002020000000300, 0x0000030707000000, 0x0100000003080800, 0x0000040a0a000000,
        0x1002020000000300, 0x0000200a0a000000, 0x02000000100d0d00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3263690000001a00,
        0x6f36316869303635, 0x31686f3038323163, 0x0831687033686b36, 0x0200000003000000, 0x0307070000005002, 0x0000030808000000, 0x0202020000000100, 0x0200000003000000,
        0x200a0a0000001002, 0x0000100d0d000000, 0x0000050000000000, 0x0000020000000300, 0x0000070000000100, 0x0800010000000700, 0x690000001c000000, 0x3932776932313563,
        0x776f323135636f39, 0x34777338776b3337, 0x0200000010307770, 0x0000080808000000, 0x02000000200a0a00, 0x0000080202000000, 0x0300000004050500, 0x0000400202000000,
        0x0a00000010050500, 0x000003000000100a, 0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1c00000002000100, 0x3036396369000000, 0x636f343230317769,
        0x323031776f303436, 0x1030777031776b34, 0x0200000001000000, 0x0000020000001e02, 0x0d000000040a0a00, 0x000003000000080d, 0x0a00000020020200, 0x200d0d000000200a,
        0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001c0000000100, 0x7769303639636900, 0x3436636f34323031, 0x6b34323031776f30,
        0x0000103077703177, 0x1e02020000000100, 0x0a00000002000000, 0x040d0d000000040a, 0x0200000003000000, 0x200a0a0000002002, 0x0000200d0d000000, 0x0000050000000300,
        0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000018000000, 0x3832316469363163, 0x3231646f3631636f, 0x1031647033646b38, 0x0600000003000000,
        0x0307070000000306, 0x0000030808000000, 0x040d0d0000000100, 0x0200000003000000, 0x100a0a0000001002, 0x0000200d0d000000, 0x0000040000000300, 0x0000010000000300,
        0x0000040000000400, 0x0100010000000100, 0x690000002a000000, 0x3038303168693463, 0x636f303239317769, 0x6f30383031686f31, 0x33686b3032393177, 0x777031687033776b,
        0x0000010000000831, 0x0100000003070700, 0x0000040d0d000000, 0x0402020000000400, 0x0000080808000000, 0x0d000000080a0a00, 0xffffff000000200d, 0x00000300000005ff,
        0x0000040000000100, 0x0000010000000400, 0x1800000001000100, 0x3832316369000000, 0x3231636f36316869, 0x33686b3631686f38, 0x0200000010316870, 0x0000030707000000,
        0x0100000003080800, 0x0000040202000000, 0x2002020000000300, 0x0000100a0a000000, 0x00000000080d0d00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000,
        0x0000020001000000, 0x3263690000001c00, 0x3637357769303635, 0x776f30383231636f, 0x777031776b363735, 0x0000010000000830, 0x02000000a0020200, 0x0000080a0a000000,
        0x03000000080d0d00, 0x0000100202000000, 0x0d000000200a0a00, 0x000000000000180d, 0x0000010000000400, 0x0000070000000100, 0x0000010000000700, 0x1d00000001000100,
        0x6932313567000000, 0x636f323377693163, 0x33776b3436776f31, 0x3233777035317764, 0x0000000000000008, 0x00080d0d00000001, 0x0101000000030000, 0x0003080800000008,
        0x000000080d0d0000, 0x0000000500000000, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001c0000, 0x3532776930323931, 0x6f30383231636f36,
        0x7031776b36353277, 0x0001000000103077, 0x0000003c02020000, 0x00040a0a00000002, 0x000000080d0d0000, 0x0020020200000003, 0x000000200a0a0000, 0x0003000000100d0d,
        0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x393163690000001c, 0x6f36353277693032, 0x32776f3038323163, 0x30777031776b3635,
        0x0000000100000010, 0x00020000003c0202, 0x000000040a0a0000, 0x0003000000040d0d, 0x0000002002020000, 0x0d0d000000200a0a, 0x0000000300000010, 0x0000000300000005,
        0x0000000100000002, 0x0000000a0000000a, 0x0015000000200001, 0x3264693163690000, 0x32646f3233636f35, 0x1032647035646b35, 0x0600000004000000, 0x0507070000000506,
        0x0000050808000000, 0x01000000020a0a00, 0x0000080505000000, 0x1002020000000300, 0x0000200909000000, 0x02000000100a0a00, 0x0300000004000000, 0x0400000001000000,
        0x0100000004000000, 0x0000800101000000, 0x3663690000001a00, 0x6f36333133776934, 0x333133776f343663, 0x0830777031776b36, 0x0200000001000000, 0x0000000000000202,
        0x2002020000000300, 0x0000100909000000, 0x00000000400a0a00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000, 0x0000010001000000, 0x3163690000001800,
        0x636f323377693832, 0x6b3233776f323135, 0x0000083077703177, 0x0000010000000000, 0x0300000008020200, 0x0000100202000000, 0x0d000000100a0a00, 0x000000000000100d,
        0x0000030000000400, 0x0000040000000100, 0x0000010000000400, 0x1800000001000100, 0x3832316369000000, 0x3135636f32337769, 0x31776b3233776f32, 0x0000000008307770,
        0x0200000001000000, 0x0000030000000202, 0x0a00000040020200, 0x080d0d000000080a, 0x0400000000000000, 0x0100000003000000, 0x0400000004000000, 0x0100000001000000,
        0x0000260000000100, 0x3633686931636900, 0x636f303436776930, 0x776f303633686f34, 0x776b33686b303436, 0x0831777031687033, 0x0700000001000000, 0x0000010000000307,
        0x04000000040d0d00, 0x0000040202000000, 0x0a00000008080800, 0x200d0d000000080a, 0x04ffffffff000000, 0x0100000001000000, 0x0700000007000000, 0x0100000001000000,
        0x00001b0000000100, 0x3163693231356700, 0x6f31636f32337769, 0x776433776b323377, 0x0000000834777033, 0x0000000100000000, 0x0003000000040d0d, 0x0000000801010000,
        0x0d0d000000030808, 0xffffffff00000008, 0x0000000300000005, 0x0000000100000002, 0x0000000700000007, 0x001d000000100001, 0x6936353263690000, 0x31636f3434343177,
        0x343431776f343230, 0x1030777031776b34, 0x0a00000001000000, 0x000002000000200a, 0x0500000002020200, 0x0000030000000405, 0x0900000040020200, 0x200a0a0000001009,
        0x0500000003000000, 0x0300000003000000, 0x0100000007000000, 0x0100000007000000, 0x00001b0000004000, 0x7769383231636900, 0x363532636f323532, 0x31776b373332776f,
        0x0000001030777036, 0x0004090900000002, 0x0000000f0d0d0000, 0x0004020200000002, 0x000000040a0a0000, 0x0020020200000003, 0x000000400a0a0000, 0xffff000000100d0d,
        0x000300000004ffff, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x353263690000001a, 0x636f323168693036, 0x3231686f30383231, 0x000831687033686b,
        0x0202000000030000, 0x00030707000000a0, 0x0000000308080000, 0x00020d0d00000001, 0x0202000000040000, 0x0002090900000010, 0x000000100a0a0000, 0xffff000000060d0d,
        0x000300000004ffff, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x3231636900000017, 0x36636f3233686938, 0x33686b3233686f34, 0x0200000008316870,
        0x0000030707000000, 0x0100000003080800, 0x0000040202000000, 0x2002020000000300, 0x0000100a0a000000, 0x00000000080d0d00, 0x0100000005000000, 0x0700000001000000,
        0x0100000007000000, 0x0000010001000000, 0x3436670000001700, 0x6f36316869316369, 0x686b3631686f3163, 0x0000001031687033, 0x0003070700000001, 0x0d0d000000010000,
        0x0000000300000002, 0x0808000000100101, 0x00080d0d00000003, 0x0004000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001c00000001,
        0x3477693034366369, 0x303233636f363930, 0x776b36393034776f, 0x0000000830777031, 0x0028020200000001, 0x0a0a000000020000, 0x00040d0d00000004, 0x0202000000030000,
        0x00200a0a00000010, 0x000000200d0d0000, 0x00000004ffffffff, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x6369000000180000, 0x6f36396869303436,
        0x3639686f30343663, 0x000831687033686b, 0x0202000000030000, 0x0003070700000028, 0x0000000308080000, 0x00040a0a00000002, 0x000000040d0d0000, 0x0010020200000003,
        0x000000200a0a0000, 0xffff000000180d0d, 0x000300000005ffff, 0x0001000000020000, 0x0007000000070000, 0x0000004000010000, 0x3532636900000018, 0x31636f3831776936,
        0x776b3531776f3832, 0x0000001030777034, 0x0004080800000002, 0x000000040a0a0000, 0x0002020200000002, 0x0000000805050000, 0x0020020200000003, 0x0000001009090000,
        0xffff000000200a0a, 0x000300000004ffff, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x3031636900000016, 0x3032636f35316869, 0x7033686b3331686f,
        0x0002000000083068, 0x0000000307070000, 0x0001000000020808, 0x000000020d0d0000, 0x0010020200000004, 0x0000000208080000, 0x0d0d000000080a0a, 0x0000000300000008,
        0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x0018000000010001, 0x3036353263690000, 0x383231636f386869, 0x7033686b38686f30, 0x0003000000083168,
        0x0000002802020000, 0x0808000000030707, 0x0000000100000003, 0x0003000000020202, 0x0000002002020000, 0x0d0d000000100a0a, 0x0000000000000008, 0x0000000300000005,
        0x0000000700000001, 0x0000000100000007, 0x001c000000020001, 0x3038323163690000, 0x3231636f36316869, 0x33686b38686f3038, 0x0010316870326873, 0x0202000000030000,
        0x0003070700000028, 0x0000000308080000, 0x00020a0a00000002, 0x000000080d0d0000, 0x0020020200000003, 0x0000000209090000, 0x0003000000400a0a, 0x0003000000050000,
        0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x323163690000001c, 0x636f363168693038, 0x6b38686f30383231, 0x3168703268733368, 0x0000000300000010,
        0x0707000000280202, 0x0003080800000003, 0x0a0a000000010000, 0x0000000300000008, 0x0a0a000000200202, 0x00080d0d00000010, 0x0004000000030000, 0x0001000000030000,
        0x0007000000070000, 0x0001000000010000, 0x0000001b00000002, 0x3468693034366369, 0x686f303436636f38, 0x32687333686b3432, 0x0300000008316870, 0x0000280202000000,
        0x0800000003070700, 0x0000010000000308, 0x03000000080a0a00, 0x0000100202000000, 0x0d000000100a0a00, 0xffffff000000180d, 0x00000300000004ff, 0x0000040000000100,
        0x0000010000000400, 0x1600000080010100, 0x3231356369000000, 0x323135636f376869, 0x687033686b37686f, 0x0000030000000831, 0x0700000010020200, 0x0308080000000307,
        0x0900000002000000, 0x040a0a0000000409, 0x0200000003000000, 0x2009090000002002, 0x0000200a0a000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700,
        0x0100010000000100, 0x690000001d000000, 0x6f34323268693363, 0x31686f3432303163, 0x3168733431686b36, 0x0000001030687034, 0x000e070700000002, 0x0000000208080000,
        0x00080a0a00000002, 0x000000020d0d0000, 0x0004020200000004, 0x0000000808080000, 0x0d0d000000100a0a, 0x0000000000000008, 0x0000000300000004, 0x0000000700000001,
        0x0000000100000007, 0x001c000000080001, 0x3268693363690000, 0x6f383637636f3432, 0x733631686b343168, 0x0008306870363168, 0x0707000000020000, 0x0004080800000010,
        0x0a0a000000010000, 0x0000000400000002, 0x0808000000040202, 0x00400a0a00000004, 0x0000000e0d0d0000, 0x0000000500000000, 0x0000000100000003, 0x0000000700000007,
        0x0001000100000001, 0x63690000001a0000, 0x3832316869323135, 0x31686f323135636f, 0x31687033686b3832, 0x0000000300000010, 0x0707000000100202, 0x0003080800000003,
        0x0a0a000000020000, 0x00080d0d00000004, 0x0202000000030000, 0x00400a0a00000020, 0x000000100d0d0000, 0x0000000400000003, 0x0000000100000003, 0x0000000700000007,
        0x0002000100000001, 0x63690000001d0000, 0x3332776930383231, 0x6f303436636f3430, 0x31776b3430333277, 0x0100000008307770, 0x0000500202000000, 0x040a0a0000000200,
        0x0000080d0d000000, 0x1002020000000300, 0x0000200a0a000000, 0xff000000180d0d00, 0x0100000005ffffff, 0x0b00000003000000, 0x0b00000001000000, 0x0008000001000000,
        0x3863690000001300, 0x6f38636f30316869, 0x30687033686b3868, 0x0000000200000008, 0x0d0d000000080c0c, 0x0000000000000002, 0x0008020200000003, 0x0000000809090000,
        0x0000000000080a0a, 0x0001000000050000, 0x0008000000030000, 0x0008000000010000, 0x0000080000010000, 0x6938636900000013, 0x686f38636f303168, 0x1030687033686b38,
        0x0c00000002000000, 0x040d0d000000080c, 0x0300000000000000, 0x0000080202000000, 0x0a00000010090900, 0x000000000000100a, 0x0000030000000500, 0x0000070000000100,
        0x0000010000000700, 0x1800000002000100, 0x3036396369000000, 0x3233636f34366869, 0x33686b3436686f30, 0x0300000010316870, 0x00001e0202000000, 0x0800000003070700,
        0x0000010000000308, 0x04000000080d0d00, 0x0000200202000000, 0x0a00000002090900, 0x080d0d000000200a, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000,
        0x0100000001000000, 0x0000180000000100, 0x6869303639636900, 0x6f303233636f3436, 0x687033686b343668, 0x0000030000001031, 0x070000001e020200, 0x0308080000000307,
        0x0a00000002000000, 0x040d0d000000020a, 0x0200000003000000, 0x400a0a0000002002, 0x0000100d0d000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700,
        0x0200010000000100, 0x6900000019000000, 0x3368693038323163, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000010, 0x0707000000280202, 0x0003080800000003,
        0x0d0d000000010000, 0x0000000400000002, 0x0909000000200202, 0x00400a0a00000002, 0x000000100d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007,
        0x0001000100000001, 0x6369000000190000, 0x3233686930383231, 0x33686f303436636f, 0x1031687033686b32, 0x0200000003000000, 0x0307070000002802, 0x0000030808000000,
        0x020a0a0000000200, 0x0000040d0d000000, 0x2002020000000300, 0x0000400a0a000000, 0x03000000080d0d00, 0x0300000005000000, 0x0700000001000000, 0x0100000007000000,
        0x0000400001000000, 0x3163690000001800, 0x636f383177693832, 0x6b3531776f383231, 0x0000103077703477, 0x0402020000000200, 0x0000040808000000, 0x080d0d0000000100,
        0x0200000003000000, 0x0809090000002002, 0x0000200a0a000000, 0x000004ffffffff00, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001d000000,
        0x3177693038323163, 0x303436636f343230, 0x776b34323031776f, 0x0000000830777031, 0x0050020200000001, 0x0a0a000000020000, 0x00080d0d00000004, 0x0202000000030000,
        0x00200a0a00000010, 0x000000200d0d0000, 0x00000004ffffffff, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x6369000000170000, 0x636f323332686933,
        0x6b343232686f3233, 0x0000083068703968, 0x0907070000000200, 0x0000020808000000, 0x020a0a0000000200, 0x0000040d0d000000, 0x0402020000000400, 0x0000080808000000,
        0x0d000000100a0a00, 0x0000010000001c0d, 0x0000030000000500, 0x0000010000000200, 0x0000070000000700, 0x1900000010000100, 0x3230316369000000, 0x33636f3833686934,
        0x686b3833686f3432, 0x0000001031687033, 0x0003070700000003, 0x0000000308080000, 0x00020000000b0a0a, 0x0000000402020000, 0x0003000000080505, 0x0000004002020000,
        0x0a0a000000100909, 0x0000000300000020, 0x0000000100000004, 0x0000000700000001, 0x0000000100000007, 0x0019000000800001, 0x3163693034670000, 0x31636f3035316869,
        0x33686b303531686f, 0x0100000008316870, 0x0000030707000000, 0x080d0d0000000100, 0x0100000003000000, 0x0308080000001001, 0x0000080909000000, 0x0000040000000100,
        0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000019000000, 0x3368693032393163, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000008,
        0x0707000000780202, 0x0003080800000003, 0x0a0a000000020000, 0x00020d0d00000008, 0x0202000000030000, 0x00200a0a00000010, 0x000000100d0d0000, 0x00000004ffffffff,
        0x0000000100000003, 0x0000000700000007, 0x0006000100000001, 0x6369000000260000, 0x6938326869323135, 0x323135636f313277, 0x3132776f3832686f, 0x687033776b33686b,
        0x0000000831777031, 0x0008020200000003, 0x0000000307070000, 0x0001000000030808, 0x0000000202020000, 0x0020020200000003, 0x000000100a0a0000, 0x0000000000150d0d,
        0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x3436636900000018, 0x33636f3436686930, 0x686b3436686f3032, 0x0000001031687033,
        0x0014020200000003, 0x0000000307070000, 0x0001000000030808, 0x000000080d0d0000, 0x0020020200000004, 0x0000000209090000, 0x0d0d000000200a0a, 0x0000000300000008,
        0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x0018000000010001, 0x6930343663690000, 0x303233636f343668, 0x7033686b3436686f, 0x0003000000103168,
        0x0000001402020000, 0x0808000000030707, 0x0000000200000003, 0x0d0d000000020a0a, 0x0000000300000004, 0x0a0a000000200202, 0x00100d0d00000040, 0x0005000000030000,
        0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001a00000002, 0x6869303635326369, 0x30383231636f3631, 0x7033686b3631686f, 0x0003000000103168,
        0x0000000a02020000, 0x0808000000030707, 0x0000000100000003, 0x0004000000080202, 0x0000002002020000, 0x0a0a000000020909, 0x00100d0d00000020, 0x0005000000000000,
        0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001a00000001, 0x6869303635326369, 0x30383231636f3631, 0x7033686b3631686f, 0x0003000000103168,
        0x0000000a02020000, 0x0808000000030707, 0x0000000100000003, 0x0003000000080202, 0x0000002002020000, 0x0d0d000000200a0a, 0x0000000000000010, 0x0000000300000005,
        0x0000000700000003, 0x0000000700000001, 0x001b000000100001, 0x6938323163690000, 0x383231636f353768, 0x7333686b3833686f, 0x0000103168703268, 0x130c0c0000000100,
        0x0200000002000000, 0x020a0a0000000402, 0x0200000003000000, 0x1009090000002002, 0x0000400a0a000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700,
        0x0100010000000100, 0x690000001c000000, 0x3034776930323363, 0x6f303233636f3639, 0x31776b3639303477, 0x0100000008307770, 0x0000140202000000, 0x040a0a0000000200,
        0x0000040d0d000000, 0x1002020000000300, 0x0000200a0a000000, 0xff000000200d0d00, 0x0300000005ffffff, 0x0700000003000000, 0x0700000001000000, 0x0000180001000000,
        0x3233670000001700, 0x6f36356869386369, 0x686b3635686f3863, 0x0000001031687033, 0x00020c0c00000002, 0x0000000e0d0d0000, 0x0000000300000000, 0x0909000000080202,
        0x00100a0a00000020, 0x0004000000000000, 0x0001000000010000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x6931636932313567, 0x776f31636f323377,
        0x31777033776b3233, 0x0000000000000008, 0x0000000300000000, 0x0808000000080101, 0x00080d0d00000003, 0x0005000000000000, 0x0001000000030000, 0x0007000000070000,
        0x0001000000010000, 0x0000000a00000001, 0x636f323531316369, 0x0001000000103834, 0x0000000902020000, 0x0008020200000001, 0x0202000000030000, 0x00100a0a00000010,
        0x000000080d0d0000, 0x0000000400000000, 0x0000000300000003, 0x0000000100000007, 0x0008000100000007, 0x63690000001b0000, 0x6f33377769323135, 0x3533776f32313563,
        0x777032777334776b, 0x0000010000000830, 0x0200000008090900, 0x0000080202000000, 0x03000000080a0a00, 0x0000200202000000, 0x0d000000200a0a00, 0x000003000000100d,
        0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1800000001000100, 0x3832316369000000, 0x3231636f32336869, 0x33686b3233686f38, 0x0200000008316870,
        0x0000030707000000, 0x0100000003080800, 0x0000080202000000, 0x1002020000000300, 0x0000200a0a000000, 0x00000000100d0d00, 0x0300000004000000, 0x0700000001000000,
        0x0100000007000000, 0x0000020001000000, 0x3663690000001b00, 0x6f36373577693034, 0x35776f3038323163, 0x30777031776b3637, 0x0000000100000008, 0x0002000000280202,
        0x000000080a0a0000, 0x0003000000080d0d, 0x0000001002020000, 0x0d0d000000200a0a, 0x0000000000000018, 0x0000000300000005, 0x0000000700000001, 0x0000000100000007,
        0x0018000000020001, 0x6930363963690000, 0x303436636f323368, 0x7033686b3233686f, 0x0003000000103168, 0x0000001e02020000, 0x0808000000030707, 0x0000000100000003,
        0x0004000000020d0d, 0x0000002002020000, 0x0a0a000000020909, 0x00100d0d00000040, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000,
        0x0000001800000001, 0x3368693036396369, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000010, 0x07070000001e0202, 0x0003080800000003, 0x0a0a000000020000,
        0x00040d0d00000002, 0x0202000000030000, 0x00200a0a00000020, 0x000000080d0d0000, 0x0000000400000003, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001,
        0x63690000001a0000, 0x3631686930383231, 0x686f30383231636f, 0x31687033686b3631, 0x0000000300000008, 0x0707000000500202, 0x0003080800000003, 0x0a0a000000010000,
        0x0000000300000002, 0x0a0a000000100202, 0x00100d0d00000020, 0x0004000000020000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001a00000001,
        0x6869303832316369, 0x30383231636f3631, 0x7033686b3631686f, 0x0003000000083168, 0x0000002802020000, 0x0808000000030707, 0x0000000100000003, 0x0003000000020202,
        0x0000001002020000, 0x0d0d000000200a0a, 0x0000000000000010, 0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x0015000000010001, 0x6932313563690000,
        0x686f3432636f3568, 0x0831687033686b35, 0x0200000003000000, 0x0307070000000402, 0x0000030808000000, 0x0802020000000100, 0x0200000003000000, 0x080a0a0000001002,
        0x0000080d0d000000, 0x0000050000000000, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x6900000015000000, 0x6f32336869343663, 0x686b3233686f3163,
        0x0000001031687033, 0x0002020200000003, 0x0000000307070000, 0x0001000000030808, 0x000000040d0d0000, 0x0020020200000003, 0x000000100a0a0000, 0x0003000000080d0d,
        0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3532636900000014, 0x6f36636f35686936, 0x31687033686b3568, 0x0000000300000008,
        0x0707000000020202, 0x0003080800000003, 0x0202000000010000, 0x0000000300000008, 0x0a0a000000100202, 0x00080d0d00000008, 0x0004000000000000, 0x0001000000030000,
        0x0007000000070000, 0x0001000000010000, 0x0000001c00000002, 0x3277693036396369, 0x303436636f343033, 0x776b34303332776f, 0x0000000830777031, 0x003c020200000001,
        0x0a0a000000020000, 0x00080d0d00000004, 0x0202000000030000, 0x00200a0a00000010, 0x000000180d0d0000, 0x00000005ffffffff, 0x0000000300000003, 0x000000010000000a,
        0x002001010000000a, 0x6369000000300000, 0x3268693632646936, 0x36636f3732776938, 0x3632686f3632646f, 0x6b33646b3632776f, 0x31647038776b3368, 0x0010337770306870,
        0x0c0c000000020000, 0x001a0d0d0000000d, 0x0003000000000000, 0x0000000808080000, 0x0a0a000000100909, 0x0000000200000010, 0x0000000300000004, 0x0000000400000001,
        0x0000000100000004, 0x0018000000800101, 0x6936353263690000, 0x363532636f343168, 0x7033686b3431686f, 0x0003000000083168, 0x0000000802020000, 0x0808000000030707,
        0x0000000200000003, 0x0a0a000000040909, 0x0000000300000004, 0x0909000000200202, 0x00200a0a00000020, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000,
        0x0001000000010000, 0x0000001700000020, 0x6869346369323367, 0x35686f34636f3635, 0x1031687033686b36, 0x0700000002000000, 0x0308080000000307, 0x0d00000001000000,
        0x000003000000080d, 0x0900000010020200, 0x100a0a0000002009, 0x0500000000000000, 0x0300000003000000, 0x0100000007000000, 0x0100000007000000, 0x0000180000001000,
        0x6869363532636900, 0x6f363532636f3833, 0x687033686b383368, 0x0000020000001031, 0x0d000000260c0c00, 0x000002000000020d, 0x0a00000008020200, 0x000003000000020a,
        0x0900000020020200, 0x400a0a0000001009, 0x0500000003000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001b0000001000, 0x6869363532636900,
        0x6f323135636f3537, 0x687331686b383368, 0x0000001030687032, 0x0008020200000001, 0x0a0a000000020000, 0x00020d0d00000004, 0x0202000000030000, 0x0010090900000020,
        0x000000400a0a0000, 0x0000000500000003, 0x0000000200000003, 0x0000000700000001, 0x0010000100000007, 0x6369000000180000, 0x6f38336869363532, 0x3833686f36353263,
        0x001031687033686b, 0x0707000000030000, 0x0003080800000003, 0x000000080a0a0000, 0x0002020200000002, 0x0000000805050000, 0x0040020200000003, 0x0000001009090000,
        0x0003000000200a0a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x343663690000001b, 0x636f363532776930, 0x3532776f30383231,
        0x1030777031776b36, 0x0200000001000000, 0x0000020000001402, 0x0d000000080a0a00, 0x000003000000040d, 0x0a00000020020200, 0x100d0d000000200a, 0x0500000003000000,
        0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001b0000000100, 0x7769303436636900, 0x383231636f363532, 0x776b363532776f30, 0x0000001030777031,
        0x0014020200000001, 0x0a0a000000020000, 0x00080d0d00000002, 0x0202000000030000, 0x00200a0a00000020, 0x000000100d0d0000, 0x0000000400000003, 0x0000000100000003,
        0x0000000700000007, 0x0001000100000001, 0x6369000000160000, 0x6f30316869323135, 0x686b3031686f3663, 0x0000000831687033, 0x0004020200000003, 0x0000000307070000,
        0x0001000000030808, 0x0000000802020000, 0x0010020200000003, 0x000000080a0a0000, 0x0000000000080d0d, 0x0003000000050000, 0x0004000000010000, 0x0001000000040000,
        0x0000002001010000, 0x3431636900000016, 0x3431636f32686934, 0x7033686b32686f34, 0x0003000000103168, 0x0000000502020000, 0x0808000000030707, 0x0000000200000003,
        0x0a0a000000020909, 0x0000000300000002, 0x0909000000200202, 0x00100a0a00000010, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000,
        0x0000001900000010, 0x6869343230316369, 0x6f343233636f3833, 0x687033686b383368, 0x0000030000001031, 0x0700000020020200, 0x0308080000000307, 0x0a00000002000000,
        0x080d0d000000020a, 0x0200000003000000, 0x1009090000002002, 0x0000400a0a000000, 0x0000050000000300, 0x0000010000000300, 0x0000070000000700, 0x1000010000000100,
        0x690000001b000000, 0x3537686938323163, 0x33686f383231636f, 0x7032687333686b38, 0x0003000000103168, 0x0000000402020000, 0x0808000000030707, 0x0000000200000003,
        0x0d0d000000020a0a, 0x0000000300000004, 0x0909000000200202, 0x00400a0a00000010, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000,
        0x0000001c00000002, 0x3177693034366369, 0x303436636f343230, 0x776b34323031776f, 0x0000001030777031, 0x0014020200000001, 0x0a0a000000020000, 0x00080d0d00000004,
        0x0202000000030000, 0x00400a0a00000020, 0x000000100d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x63690000001c0000,
        0x3230317769303436, 0x776f303436636f34, 0x7031776b34323031, 0x0001000000103077, 0x0000001402020000, 0x00040a0a00000002, 0x000000080d0d0000, 0x0020020200000003,
        0x000000400a0a0000, 0x0003000000080d0d, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000002001010000, 0x3231636900000019, 0x636f303168693038,
        0x6b3031686f363435, 0x0000083168703368, 0x5002020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000080a0a000000, 0x03000000020d0d00, 0x0000100202000000,
        0x0a00000020090900, 0x000003000000100a, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1a00000001000100, 0x3635326369000000, 0x31636f3436776930,
        0x6b3436776f303832, 0x0000083077703177, 0x1402020000000100, 0x0200000002000000, 0x020d0d0000000802, 0x0200000003000000, 0x100a0a0000001002, 0x0000200d0d000000,
        0x0000050000000000, 0x0000030000000300, 0x0000010000000700, 0x1000010000000700, 0x6900000016000000, 0x6f35686936353263, 0x6b35686f36383463, 0x0000103168703368,
        0x050c0c0000000200, 0x0000050d0d000000, 0x040a0a0000000100, 0x0200000003000000, 0x1009090000002002, 0x0000200a0a000000, 0x0000040000000300, 0x0000010000000300,
        0x0000070000000700, 0x0100010000000100, 0x6900000018000000, 0x3233686930363963, 0x33686f303436636f, 0x0831687033686b32, 0x0200000003000000, 0x0307070000003c02,
        0x0000030808000000, 0x080a0a0000000100, 0x0200000003000000, 0x200a0a0000001002, 0x0000100d0d000000, 0x000004ffffffff00, 0x0000010000000300, 0x0000040000000400,
        0x8001010000000100, 0x6900000019000000, 0x3477693834303263, 0x776f323135636f39, 0x30777031776b3934, 0x0000000100000008, 0x0002000000400202, 0x000000040a0a0000,
        0x0003000000040d0d, 0x0000002002020000, 0x0a0a000000200909, 0x0000000300000020, 0x0000000100000005, 0x0000000800000003, 0x0000000800000001, 0x0014000000200001,
        0x3264693863690000, 0x3532646f38636f35, 0x001032647035646b, 0x0d0d000000010000, 0x0000000000000019, 0x0008020200000003, 0x0000001009090000, 0x0000000000100a0a,
        0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x353263690000001a, 0x636f323135686936, 0x323135686f363532, 0x001031687033686b,
        0x0202000000030000, 0x0003070700000008, 0x0000000308080000, 0x00040a0a00000002, 0x000000040d0d0000, 0x0020020200000003, 0x000000400a0a0000, 0x0003000000200d0d,
        0x0003000000050000, 0x0004000000010000, 0x0001000000040000, 0x0000000100010000, 0x3436636900000038, 0x3268693432326469, 0x6f30363177693432, 0x343232646f323363,
        0x31776f343232686f, 0x33686b33646b3036, 0x687031647033776b, 0x0000001031777031, 0x0003060600000003, 0x0000000307070000, 0x0001000000030808, 0x0000000202020000,
        0x0020020200000003, 0x000000200a0a0000, 0x0000000000200d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3436636900000038,
        0x3268693432326469, 0x6f30363177693432, 0x343232646f323363, 0x31776f343232686f, 0x33686b33646b3036, 0x687031647033776b, 0x0000001031777031, 0x0004020200000004,
        0x0000000306060000, 0x0808000000030707, 0x0000000000000003, 0x0010020200000003, 0x000000200a0a0000, 0x0003000000200d0d, 0x0003000000050000, 0x0001000000020000,
        0x0007000000070000, 0x0000001000010000, 0x3135636900000018, 0x34636f3931686932, 0x686b3931686f3638, 0x0000001031687033, 0x0003070700000003, 0x0000000308080000,
        0x0002000000100a0a, 0x0000000202020000, 0x0003000000080505, 0x0000004002020000, 0x0a0a000000100909, 0x0000000300000020, 0x0000000300000004, 0x0000000400000001,
        0x0000000100000004, 0x0031000000010001, 0x3432303163690000, 0x3177693030316869, 0x6f323135636f3137, 0x3731776f30303168, 0x6433776b33686b31, 0x3268703177643168,
        0x0300000008327770, 0x0000200202000000, 0x0800000003070700, 0x0000010000000308, 0x03000000080a0a00, 0x0000200202000000, 0x0d000000400a0a00, 0x000003000000100d,
        0x0000010000000500, 0x0000010000000200, 0x00000b0000000b00, 0x1300000800000100, 0x6869386369000000, 0x38686f38636f3031, 0x000830687033686b, 0x0707000000020000,
        0x0003080800000003, 0x0505000000010000, 0x0000000300000002, 0x0909000000080202, 0x00080a0a00000010, 0x0005000000000000, 0x0002000000010000, 0x0008000000010000,
        0x0001000000080000, 0x0000001300000800, 0x6f30316869386369, 0x33686b38686f3863, 0x0200000010306870, 0x0000030707000000, 0x0100000003080800, 0x0000020505000000,
        0x1002020000000300, 0x0000100909000000, 0x00000000080a0a00, 0x0300000005000000, 0x0400000001000000, 0x0100000004000000, 0x0000200001000000, 0x3233670000001700,
        0x6f38326869386369, 0x686b3832686f3863, 0x0000001031687033, 0x0003070700000002, 0x0000000308080000, 0x00040d0d00000001, 0x0202000000030000, 0x0020090900000020,
        0x000000100a0a0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x6369000000180000, 0x6f34316869363532, 0x3431686f36353263,
        0x000831687033686b, 0x0707000000020000, 0x0003080800000003, 0x0202000000010000, 0x0000000300000008, 0x0a0a000000200202, 0x00080d0d00000020, 0x0005000000000000,
        0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000002, 0x3368693034366369, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000010,
        0x0707000000140202, 0x0003080800000003, 0x0a0a000000010000, 0x0000000300000008, 0x0a0a000000200202, 0x00200d0d00000020, 0x0005000000030000, 0x0001000000030000,
        0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3368693034366369, 0x686f303436636f32, 0x31687033686b3233, 0x0000000300000010, 0x0707000000140202,
        0x0003080800000003, 0x0a0a000000010000, 0x0000000300000008, 0x0a0a000000200202, 0x00200d0d00000010, 0x0005000000030000, 0x0001000000030000, 0x0007000000070000,
        0x0101000000010000, 0x0000003200000010, 0x3731646932336369, 0x3333776933336869, 0x3731646f3233636f, 0x3333776f3333686f, 0x776b33686b33646b, 0x7031687031647033,
        0x0004000000103177, 0x0000000202020000, 0x0707000000030606, 0x0003080800000003, 0x0d0d000000010000, 0x0000000300000002, 0x0909000000100202, 0x00200a0a00000010,
        0x0004000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3377693832316369, 0x776f363532636f32, 0x30777031776b3233,
        0x0000000100000008, 0x0002000000040202, 0x000000040a0a0000, 0x0003000000040d0d, 0x0000002002020000, 0x0d0d000000080a0a, 0xffffffff00000008, 0x0000000300000004,
        0x0000000400000001, 0x0000000100000004, 0x0018000000010001, 0x6938323163690000, 0x363532636f323377, 0x7031776b3233776f, 0x0001000000083077, 0x0000000202020000,
        0x00080a0a00000002, 0x000000020d0d0000, 0x0040020200000003, 0x000000080a0a0000, 0xffff000000080d0d, 0x000100000004ffff, 0x0007000000010000, 0x0001000000070000,
        0x0000000100010000, 0x323135670000001b, 0x6f32337769316369, 0x776b3233776f3163, 0x0832777031776433, 0x0100000000000000, 0x0000040d0d000000, 0x0801010000000300,
        0x0000030808000000, 0xff000000080d0d00, 0x0300000004ffffff, 0x0100000002000000, 0x0700000007000000, 0x0000100001000000, 0x3163690000001f00, 0x3035317769383830,
        0x6f323135636f3030, 0x776b303030353177, 0x0000000830777031, 0x00100a0a00000001, 0x0202000000020000, 0x0008050500000004, 0x0202000000030000, 0x0010090900000020,
        0x000000200a0a0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x63690000001a0000, 0x3233686930383231, 0x686f30383231636f,
        0x31687033686b3233, 0x0000000300000010, 0x0707000000280202, 0x0003080800000003, 0x0d0d000000010000, 0x0000000400000004, 0x0909000000200202, 0x00200a0a00000002,
        0x000000080d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007, 0x0001000100000001, 0x63690000001a0000, 0x3233686930383231, 0x686f30383231636f,
        0x31687033686b3233, 0x0000000300000010, 0x0707000000050202, 0x0003080800000003, 0x0202000000010000, 0x0000000300000008, 0x0a0a000000200202, 0x00200d0d00000020,
        0x0004000000000000, 0x0001000000030000, 0x0004000000040000, 0x0101000000010000, 0x0000001900000020, 0x3168693231356369, 0x6f34323031636f33, 0x687033686b333168,
        0x0000030000000831, 0x0700000010020200, 0x0308080000000307, 0x0a00000002000000, 0x020d0d000000080a, 0x0200000003000000, 0x2009090000002002, 0x0000100a0a000000,
        0x0000040000000300, 0x0000010000000300, 0x0000070000000700, 0x2001010000000100, 0x6900000019000000, 0x3331686932313563, 0x686f34323031636f, 0x31687033686b3331,
        0x0000000300000008, 0x0707000000200202, 0x0003080800000003, 0x0a0a000000010000, 0x0000000300000008, 0x0909000000100202, 0x00200a0a00000020, 0x0005000000020000,
        0x0001000000030000, 0x0007000000070000, 0x0101000000010000, 0x0000002a00000020, 0x3830316869346369, 0x6f30323931776930, 0x30383031686f3163, 0x686b30323931776f,
        0x7031687033776b33, 0x0001000000103177, 0x0000000307070000, 0x00080d0d00000001, 0x0202000000040000, 0x0004080800000004, 0x0000002009090000, 0x0003000000100a0a,
        0x0003000000050000, 0x0004000000010000, 0x0001000000040000, 0x0000002001010000, 0x693463690000002a, 0x3177693038303168, 0x686f31636f303239, 0x3931776f30383031,
        0x33776b33686b3032, 0x0010317770316870, 0x0707000000010000, 0x0000000000000003, 0x0004020200000004, 0x0000000808080000, 0x0a0a000000200909, 0x0000000300000010,
        0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x0019000000010001, 0x6930343663690000, 0x383231636f363168, 0x33686b3631686f30, 0x0300000008316870,
        0x0000140202000000, 0x0800000003070700, 0x0000010000000308, 0x0300000002020200, 0x0000100202000000, 0x0d000000200a0a00, 0x000000000000100d, 0x0000030000000400,
        0x0000040000000100, 0x0000010000000400, 0x1700000020000100, 0x6369323367000000, 0x38636f3832686938, 0x7033686b3832686f, 0x0001000000083168, 0x0000000307070000,
        0x00080d0d00000001, 0x0202000000040000, 0x0004080800000008, 0x0000002009090000, 0x0002000000080a0a, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000,
        0x0000000200010000, 0x393163690000001a, 0x636f343268693032, 0x3432686f30383231, 0x000831687033686b, 0x0202000000030000, 0x0003070700000078, 0x0000000308080000,
        0x00040a0a00000001, 0x0202000000030000, 0x00200a0a00000010, 0x000000180d0d0000, 0x00000005ffffffff, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001,
        0x6369000000170000, 0x6f32336869383231, 0x6b3233686f343663, 0x0000103168703368, 0x0307070000000200, 0x0000030808000000, 0x0402020000000100, 0x0200000003000000,
        0x100a0a0000002002, 0x0000080d0d000000, 0x0000050000000000, 0x0000010000000300, 0x0000040000000400, 0x0100010000000100, 0x6900000017000000, 0x3431686938323163,
        0x3431686f3233636f, 0x001031687033686b, 0x0707000000020000, 0x0003080800000003, 0x0202000000010000, 0x0000000300000004, 0x0a0a000000200202, 0x00080d0d00000010,
        0x0004000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000005, 0x3168693231356369, 0x686f323135636f34, 0x31687033686b3431,
        0x0000000300000008, 0x0707000000040202, 0x0003080800000003, 0x0202000000010000, 0x0000000300000004, 0x0a0a000000200202, 0x000e0d0d00000010, 0x0005000000000000,
        0x0003000000030000, 0x0001000000070000, 0x0001000000070000, 0x0000001800000010, 0x3168693231356369, 0x686f363834636f30, 0x31687033686b3031, 0x0000000200000010,
        0x0d0d0000000a0c0c, 0x000000000000000a, 0x0020020200000003, 0x0000001009090000, 0x0003000000400a0a, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000,
        0x0000008001010000, 0x353263690000001c, 0x6f36333133776936, 0x3133776f38323163, 0x30777031776b3633, 0x0000000100000008, 0x0002000000080202, 0x000000080a0a0000,
        0x0003000000040d0d, 0x0000002002020000, 0x0a0a000000100909, 0x0000000300000010, 0x0000000300000004, 0x0000000700000001, 0x0000000100000007, 0x001c000000020001,
        0x6930343663690000, 0x33636f3631323977, 0x36313239776f3032, 0x000830777031776b, 0x0202000000010000, 0x0000000200000028, 0x0d0d000000020a0a, 0x0000000300000008,
        0x0a0a000000100202, 0x00200d0d00000020, 0x0004ffffffff0000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000001, 0x3368693032336369,
        0x686f303436636f32, 0x31687033686b3233, 0x0000000300000008, 0x0707000000140202, 0x0003080800000003, 0x0a0a000000010000, 0x0000000300000002, 0x0a0a000000100202,
        0x00100d0d00000020, 0x0005ffffffff0000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001c00000008, 0x3432326869336369, 0x31686f383637636f,
        0x3168733631686b34, 0x0000001030687036, 0x0010070700000002, 0x0000000408080000, 0x00080a0a00000001, 0x0202000000040000, 0x0004080800000004, 0x000000400a0a0000,
        0x00030000000e0d0d, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000, 0x0000008001010000, 0x3231636900000018, 0x31636f3832686938, 0x686b3832686f3832,
        0x0000000831687033, 0x0004020200000003, 0x0000000307070000, 0x0002000000030808, 0x000000040a0a0000, 0x0003000000040d0d, 0x0000002002020000, 0x0a0a000000200909,
        0x0000000300000020, 0x0000000300000005, 0x0000000400000001, 0x0000000100000004, 0x0018000000800001, 0x6938323163690000, 0x383231636f383268, 0x7033686b3832686f,
        0x0003000000103168, 0x0000000202020000, 0x0808000000030707, 0x0000000200000003, 0x0d0d000000020a0a, 0x0000000300000004, 0x0909000000400202, 0x00400a0a00000020,
        0x0005000000030000, 0x0001000000010000, 0x0004000000040000, 0x0001000000010000, 0x0000003000000001, 0x6931636938383267, 0x3532776938323168, 0x3231686f31636f36,
        0x686b363532776f38, 0x6433686435776b35, 0x3877703868703377, 0x0000000100000010, 0x0000000000050707, 0x0101000000030000, 0x0005080800000010, 0x000000100d0d0000,
        0x0000000400000000, 0x0000000100000001, 0x0000000400000004, 0x0020010100000001, 0x32670000001c0000, 0x3377693163693432, 0x776f31636f363331, 0x7031776b36333133,
        0x0000000000083077, 0x0002000000000000, 0x0000002001010000, 0x0000000000100909, 0x0003000000040000, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000,
        0x3931636900000019, 0x636f383468693032, 0x6b3834686f303436, 0x0000083168703368, 0x7802020000000300, 0x0000030707000000, 0x0100000003080800, 0x0000040d0d000000,
        0x1002020000000400, 0x0000020909000000, 0x0d000000200a0a00, 0xffffff0000000c0d, 0x00000300000005ff, 0x0000070000000100, 0x0000010000000700, 0x1700000020010100,
        0x6869336369000000, 0x6f3631636f363134, 0x7033686b36313468, 0x0001000000103168, 0x0000000307070000, 0x00080d0d00000001, 0x0202000000040000, 0x0004080800000004,
        0x0000002009090000, 0x0003000000100a0a, 0x0003000000050000, 0x0004000000010000, 0x0001000000040000, 0x0000002001010000, 0x6933636900000017, 0x3631636f36313468,
        0x33686b363134686f, 0x0100000010316870, 0x0000030707000000, 0x0000040000000000, 0x0800000004020200, 0x2009090000000808, 0x0000100a0a000000, 0x0000040000000300,
        0x0000010000000300, 0x0000040000000400, 0x8001010000000100, 0x690000001a000000, 0x3431686932313563, 0x37686f323135636f, 0x687032687333686b, 0x0000030000000831,
        0x0700000010020200, 0x0308080000000307, 0x0900000002000000, 0x080a0a0000000409, 0x0200000003000000, 0x2009090000002002, 0x0000200a0a000000, 0x0000050000000300,
        0x0000010000000300, 0x0000070000000700, 0x0200010000000100, 0x690000001a000000, 0x3677693038323163, 0x6f30383231636f34, 0x777031776b343677, 0x0000010000001030,
        0x0200000028020200, 0x0000020a0a000000, 0x03000000080d0d00, 0x0000200202000000, 0x0d000000200a0a00, 0x000003000000080d, 0x0000030000000500, 0x0000070000000100,
        0x0000010000000700, 0x1a00000001000100, 0x3832316369000000, 0x31636f3436776930, 0x6b3436776f303832, 0x0000103077703177, 0x2802020000000100, 0x0a00000002000000,
        0x020d0d000000020a, 0x0200000003000000, 0x100a0a0000002002, 0x0000100d0d000000, 0x0000050000000300, 0x0000010000000300, 0x0000040000000400, 0x8000010000000100,
        0x690000001b000000, 0x3331337769343663, 0x776f363532636f36, 0x7031776b36333133, 0x0000000000103077, 0x0a0a000000010000, 0x0000000300000004, 0x0a0a000000400202,
        0x00200d0d00000040, 0x0005000000030000, 0x0002000000030000, 0x0007000000010000, 0x0001000000070000, 0x0000001c00000010, 0x3577693635326369, 0x383231636f353236,
        0x776b35323635776f, 0x0000001030777031, 0x00040a0a00000001, 0x0202000000010000, 0x0000000300000004, 0x0909000000400202, 0x00200a0a00000010, 0x0005000000030000,
        0x0003000000030000, 0x0001000000070000, 0x0001000000070000, 0x0000001800000020, 0x3168693231356369, 0x686f363834636f39, 0x31687033686b3931, 0x0000000100000010,
        0x0000000000130d0d, 0x0202000000030000, 0x0010090900000020, 0x000000400a0a0000, 0x0000000400000003, 0x0000000100000001, 0x0000000700000007, 0x0080000100000001,
        0x31670000001c0000, 0x3168693163693434, 0x39686f31636f3139, 0x7032687333686b35, 0x0001000000083068, 0x0000000307070000, 0x0000000300000000, 0x0808000000400101,
        0x0008090900000003, 0x0004000000000000, 0x0001000000010000, 0x0004000000040000, 0x0001000000010000, 0x0000001700000001, 0x6869326369323367, 0x32686f32636f3832,
        0x0831687033686b38, 0x0700000001000000, 0x0000010000000307, 0x04000000020d0d00, 0x0000020202000000, 0x0a00000003080800, 0x0e0d0d000000080a, 0x0400000000000000,
        0x0300000001000000, 0x0100000008000000, 0x0100000008000000, 0x00001a0000000100, 0x3232686933636900, 0x686f383231636f34, 0x34687334686b3635, 0x0100000010306870,
        0x00001c0c0c000000, 0x0000030000000000, 0x0a00000003020200, 0x080d0d000000200a, 0x0500000000000000, 0x0100000001000000, 0x0800000008000000, 0x0100000001000000,
        0x00001e0000006400, 0x3932776931636900, 0x6f3031636f323335, 0x31776b3530393577, 0x1030777035777330, 0x0000000000000000, 0x0800000003000000, 0x0a09090000000a08,
        0x0000100a0a000000, 0x0000040000000000, 0x0000020000000300, 0x0000070000000100, 0x0100010000000700, 0x690000002f000000, 0x3038303168693463, 0x636f303239317769,
        0x6f303633686f3633, 0x6b33686b30343677, 0x3377733368733377, 0x0008307770306870, 0x0707000000030000, 0x0003080800000003, 0x000000030a0a0000, 0x0002050500000001,
        0x0202000000030000, 0x001e050500000008, 0x000000100a0a0000, 0x00000005ffffffff, 0x0000000300000003, 0x0000000100000007, 0x0010000100000007, 0x63690000001f0000,
        0x3035317769383231, 0x34323031636f3030, 0x6b3030303531776f, 0x0000103077703177, 0xfa0d0d0000000100, 0x0200000002000000, 0x080a0a0000000402, 0x0200000003000000,
        0x1009090000002002, 0x0000200a0a000000, 0x0000040000000300, 0x0000010000000300, 0x0000070000000700, 0x0100010000000100, 0x690000001b000000, 0x3532776930343663,
        0x6f30383231636f36, 0x7031776b36353277, 0x0001000000083077, 0x0000002802020000, 0x00080a0a00000002, 0x000000040d0d0000, 0x0010020200000003, 0x000000100a0a0000,
        0xffff000000200d0d, 0x000100000005ffff, 0x0008000000030000, 0x0008000000010000, 0x0000002000010000, 0x6936636900000030, 0x6938326869363264, 0x646f36636f373277,
        0x776f3632686f3632, 0x33686b33646b3632, 0x687031647038776b, 0x0000001033777030, 0x00020c0c00000002, 0x0000001a0d0d0000, 0x0002020200000001, 0x0808000000030000,
        0x0010090900000008, 0x000000100a0a0000, 0x0000000500000000, 0x0000000200000003, 0x0000000700000001, 0x0010000100000007, 0x6369000000180000, 0x3833686934323031,
        0x3833686f3631636f, 0x001031687033686b, 0x0707000000020000, 0x0003080800000003, 0x0202000000020000, 0x0002050500000002, 0x0202000000030000, 0x0010090900000040,
        0x000000100a0a0000, 0x0000000400000003, 0x0000000100000003, 0x0000000400000004, 0x0001000100000001, 0x6369000000150000, 0x31636f3233686933, 0x33686b3033686f30,
        0x0200000008306870, 0x0000030707000000, 0x0100000003080800, 0x0000040d0d000000, 0x2002020000000300, 0x0000080a0a000000, 0x03000000080d0d00, 0x0300000004000000,
        0x0400000001000000, 0x0100000004000000, 0x0000800101000000, 0x3363690000001a00, 0x36636f3432326869, 0x686b323131686f34, 0x0833687032687337, 0x0700000001000000,
        0x0000020000000707, 0x0d000000020a0a00, 0x000004000000080d, 0x0800000004020200, 0x2009090000000808, 0x0000200a0a000000, 0x0000040000000300, 0x0000010000000300,
        0x0000070000000700, 0x0100010000000100, 0x690000001c000000, 0x3034776930363963, 0x6f303233636f3639, 0x31776b3639303477, 0x0100000008307770, 0x00003c0202000000,
        0x040a0a0000000200, 0x0000040d0d000000, 0x1002020000000300, 0x0000200a0a000000, 0xff000000200d0d00, 0x0300000005ffffff, 0x0700000001000000, 0x0100000007000000,
        0x0000020001000000, 0x3663690000001b00, 0x636f323368693034, 0x6b3631686f303436, 0x3168703268733368, 0x0000000300000010, 0x0707000000140202, 0x0003080800000003,
        0x0d0d000000010000, 0x0000000400000002, 0x0909000000200202, 0x00400a0a00000002, 0x000000080d0d0000, 0x0000000500000003, 0x0000000100000003, 0x0000000700000007,
        0x0001000100000001, 0x63690000001b0000, 0x6f32336869303436, 0x3631686f30343663, 0x687032687333686b, 0x0000030000001031, 0x0700000014020200, 0x0308080000000307,
        0x0a00000002000000, 0x020d0d000000080a, 0x0200000003000000, 0x100a0a0000002002, 0x0000080d0d000000, 0x0000040000000300, 0x0000010000000300, 0x0000040000000400,
        0x0100010000000100, 0x6900000018000000, 0x6f38323168693363, 0x686b3436686f3863, 0x0830687032687333, 0x0700000001000000, 0x0000010000000307, 0x04000000020d0d00,
        0x0000040202000000, 0x0a00000008080800, 0x080d0d000000080a, 0x0500000003000000, 0x0300000003000000, 0x0100000007000000, 0x0100000007000000, 0x00001c0000004000,
        0x3177693436636900, 0x383231636f323230, 0x776b37303031776f, 0x0000103077703631, 0x0409090000000200, 0x0000070d0d000000, 0x0202020000000200, 0x0000020a0a000000,
        0x2002020000000300, 0x0000400a0a000000, 0xff000000100d0d00, 0x0300000005ffffff, 0x0700000003000000, 0x0700000001000000, 0x0000100001000000, 0x3163690000001900,
        0x6f38336869343230, 0x3833686f34323363, 0x001031687033686b, 0x0c0c000000020000, 0x00260d0d00000002, 0x0a0a000000010000, 0x0000000300000004, 0x0909000000200202,
        0x00200a0a00000010, 0x0004000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000002, 0x3968693036396369, 0x686f303233636f36,
        0x31687033686b3639, 0x0000000300000008, 0x07070000003c0202, 0x0003080800000003, 0x0a0a000000020000, 0x00040d0d00000004, 0x0202000000030000, 0x00200a0a00000010,
        0x000000180d0d0000, 0x00000004ffffffff, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x6369000000160000, 0x33636f3639686934, 0x686b3639686f3032,
        0x0000000831687033, 0x0003070700000001, 0x0d0d000000010000, 0x0000000500000002, 0x0808000000040202, 0x0002090900000004, 0x000000200a0a0000, 0xffff000000100d0d,
        0x000300000005ffff, 0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x3436636900000019, 0x31636f3631686930, 0x6b3631686f303832, 0x0000103168703368,
        0x1402020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000020a0a000000, 0x03000000020d0d00, 0x0000200202000000, 0x0d000000100a0a00, 0x000003000000080d,
        0x0000030000000500, 0x0000070000000100, 0x0000010000000700, 0x1900000001000100, 0x3034366369000000, 0x3231636f36316869, 0x686b3631686f3038, 0x0000001031687033,
        0x0005020200000003, 0x0000000307070000, 0x0001000000030808, 0x0000000402020000, 0x0020020200000003, 0x000000200a0a0000, 0x0000000000100d0d, 0x0003000000040000,
        0x0007000000010000, 0x0001000000070000, 0x0000000200010000, 0x323363690000001b, 0x33636f3639686930, 0x686b3834686f3032, 0x0831687032687333, 0x0200000003000000,
        0x0307070000001402, 0x0000030808000000, 0x040d0d0000000100, 0x0200000004000000, 0x0209090000001002, 0x0000200a0a000000, 0x000000000c0d0d00, 0x0100000004000000,
        0x0700000001000000, 0x0100000007000000, 0x0000800001000000, 0x3431670000001c00, 0x3531686931636934, 0x3537686f31636f31, 0x687032687333686b, 0x0000010000000830,
        0x0000000003070700, 0x0100000003000000, 0x0308080000004001, 0x0000080909000000, 0x0000040000000000, 0x0000030000000100, 0x0000010000000800, 0x0000010000000800,
        0x6900000013000008, 0x636f303168693263, 0x7033686b38686f38, 0x0001000000083068, 0x000000080c0c0000, 0x0000000300000000, 0x0909000000080202, 0x00080a0a00000010,
        0x0004000000000000, 0x0001000000030000, 0x0004000000040000, 0x0001000000010000, 0x0000001700000020, 0x3332686932336369, 0x3232686f33636f32, 0x0830687039686b34,
        0x0700000002000000, 0x0908080000000907, 0x0d00000001000000, 0x000003000000040d, 0x0900000020020200, 0x080a0a0000002009, 0x0400000001000000, 0x0100000003000000,
        0x0400000004000000, 0x0100000001000000, 0x0000170000000100, 0x3268693233636900, 0x32686f33636f3233, 0x30687039686b3432, 0x0000000200000008, 0x0808000000090707,
        0x0000000100000009, 0x0003000000020d0d, 0x0000002002020000, 0x0d0d000000080a0a, 0x000000010000001c, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004,
        0x0016000000010001, 0x6869363163690000, 0x686f3631636f3631, 0x30687033686b3431, 0x0000000200000008, 0x0808000000030707, 0x0000000100000002, 0x0004000000020d0d,
        0x0000001002020000, 0x0a0a000000020808, 0x00080d0d00000008, 0x0004000000030000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000002,
        0x3468693036396369, 0x686f303436636f38, 0x31687033686b3834, 0x0000000300000008, 0x07070000003c0202, 0x0003080800000003, 0x0d0d000000010000, 0x0000000400000004,
        0x0909000000100202, 0x00200a0a00000002, 0x0000000c0d0d0000, 0x00000004ffffffff, 0x0000000100000003, 0x0000000700000007, 0x0002000100000001, 0x6369000000180000,
        0x6f36396869303233, 0x3639686f30323363, 0x000831687033686b, 0x0202000000030000, 0x0003070700000014, 0x0000000308080000, 0x00020d0d00000001, 0x0202000000040000,
        0x0002090900000010, 0x000000200a0a0000, 0x0000000000100d0d, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000, 0x0000000100010000, 0x3231636900000015,
        0x3233636f37686938, 0x687033686b37686f, 0x0000020000001031, 0x0800000003070700, 0x0000010000000308, 0x0300000008020200, 0x0000100202000000, 0x0d000000100a0a00,
        0x000000000000080d, 0x0000030000000500, 0x0000040000000100, 0x0000010000000400, 0x1500000001000100, 0x3832316369000000, 0x6f3233636f376869, 0x31687033686b3768,
        0x0000000200000010, 0x0808000000030707, 0x0000000100000003, 0x0003000000040202, 0x0000002002020000, 0x0d0d000000100a0a, 0x0000000000000008, 0x0000000100000005,
        0x0000000b00000001, 0x000000010000000b, 0x0013000008000001, 0x3168693863690000, 0x6b38686f38636f30, 0x0000083068703368, 0x0307070000000100, 0x0400000000000000,
        0x0000080202000000, 0x0a00000003080800, 0x080d0d000000080a, 0x0500000000000000, 0x0100000001000000, 0x0800000008000000, 0x0100000001000000, 0x0000130000080000,
        0x3031686938636900, 0x686b38686f38636f, 0x0000001030687033, 0x0003070700000001, 0x0004000000000000, 0x0000000802020000, 0x0a0a000000030808, 0x00080d0d00000010,
        0x0004000000000000, 0x0001000000030000, 0x0007000000070000, 0x0001000000010000, 0x0000001800000002, 0x3468693034366369, 0x686f303436636f38, 0x31687033686b3834,
        0x0000000300000008, 0x0707000000280202, 0x0003080800000003, 0x0d0d000000010000, 0x0000000400000004, 0x0909000000100202, 0x00200a0a00000002, 0x0000000c0d0d0000,
        0x0000000500000000, 0x0000000300000003, 0x000000010000000a, 0x004001010000000a, 0x63690000001b0000, 0x3235327769383231, 0x32776f363532636f, 0x77703631776b3733,
        0x0000020000001030, 0x0d00000002090900, 0x0000020000004f0d, 0x0a00000004020200, 0x000003000000020a, 0x0900000020020200, 0x400a0a0000000809, 0x0500000002000000,
        0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001c0000000200, 0x7769303639636900, 0x3233636f36393034, 0x6b36393034776f30, 0x0000103077703177,
        0x1e02020000000100, 0x0a00000002000000, 0x080d0d000000020a, 0x0200000003000000, 0x400a0a0000002002, 0x0000200d0d000000, 0x0000050000000300, 0x0000010000000300,
        0x0000070000000700, 0x0100010000000100, 0x690000001c000000, 0x3034776930363963, 0x6f303233636f3639, 0x31776b3639303477, 0x0100000010307770, 0x00001e0202000000,
        0x040a0a0000000200, 0x0000040d0d000000, 0x2002020000000300, 0x0000200a0a000000, 0x03000000200d0d00, 0x0300000004000000, 0x0700000001000000, 0x0100000007000000,
        0x0000020001000000, 0x3363690000001800, 0x636f383468693032, 0x6b3834686f303436, 0x0000083168703368, 0x1402020000000300, 0x0000030707000000, 0x0100000003080800,
        0x0000040d0d000000, 0x1002020000000400, 0x0000020909000000, 0x0d000000200a0a00, 0x0000000000000c0d, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700,
        0x1800000001000100, 0x3032336369000000, 0x3233636f36366869, 0x33686b3436686f30, 0x0300000008306870, 0x0000140202000000, 0x0800000003070700, 0x0000020000000308,
        0x0d000000080a0a00, 0x000003000000020d, 0x0a00000010020200, 0x200d0d000000100a, 0x0400000000000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000,
        0x00001d0000000200, 0x6930323931636900, 0x36636f3430333277, 0x34303332776f3034, 0x000830777031776b, 0x0202000000010000, 0x0000000200000078, 0x0d0d000000040a0a,
        0x0000000300000008, 0x0a0a000000100202, 0x00180d0d00000020, 0x0005ffffffff0000, 0x0003000000010000, 0x0001000000080000, 0x0001000000080000, 0x0000001400000020,
        0x6f35326469386369, 0x646b3332646f3863, 0x0000001031647035, 0x00170d0d00000001, 0x0003000000000000, 0x0000000802020000, 0x0a0a000000100909, 0x0000000000000010,
        0x0000000300000005, 0x0000000700000001, 0x0000000100000007, 0x0015000000010001, 0x6936353263690000, 0x686f3231636f3568, 0x1031687033686b35, 0x0200000003000000,
        0x0307070000000202, 0x0000030808000000, 0x0802020000000100, 0x0200000003000000, 0x100a0a0000001002, 0x0000080d0d000000, 0x0000040000000000, 0x0000010000000300,
        0x0000040000000400, 0x8001010000000100, 0x690000001b000000, 0x3177693432303163, 0x6f363532636f3639, 0x7031776b36393177, 0x0001000000083077, 0x0000002002020000,
        0x00080a0a00000002, 0x000000040d0d0000, 0x0020020200000003, 0x0000002009090000, 0x0003000000200a0a, 0x0003000000040000, 0x0004000000010000, 0x0001000000040000,
        0x0000008001010000, 0x3436636900000016, 0x3436636f36356869, 0x7033686b3635686f, 0x0003000000083168, 0x0000000202020000, 0x0808000000030707, 0x0000000200000003,
        0x0d0d000000020a0a, 0x0000000300000008, 0x0909000000200202, 0x00200a0a00000020, 0x0004000000030000, 0x0001000000010000, 0x0004000000040000, 0x0001000000010000,
        0x0000002c00000001, 0x6930386869316369, 0x6f30353630303177, 0x776f3038686f3163, 0x686b303536303031, 0x3068703332776b31, 0x0000000831317770, 0x0000000100000000,
        0x0003000000020d0d, 0x0000001708080000, 0x0d0d000000080a0a, 0x000000000000000f, 0x0000000100000004, 0x0000000700000001, 0x0000000100000007, 0x002c000000010001,
        0x3868693163690000, 0x3536303031776930, 0x3038686f31636f30, 0x303536303031776f, 0x703332776b31686b, 0x0008313177703068, 0x0808000000010000, 0x0000000100000003,
        0x0003000000080d0d, 0x0000000808080000, 0x0d0d000000080a0a, 0x0000000100000010, 0x0000000300000004, 0x0000000400000001, 0x0000000100000004, 0x001b000000800101,
        0x6938323163690000, 0x383231636f363568, 0x7333686b3832686f, 0x0000083168703268, 0x0402020000000300, 0x0000030707000000, 0x0200000003080800, 0x0000040a0a000000,
        0x03000000080d0d00, 0x0000200202000000, 0x0a00000020090900, 0x000003000000200a, 0x0000030000000400, 0x0000070000000100, 0x0000010000000700, 0x1c00000002000100,
        0x3036396369000000, 0x636f363132397769, 0x313239776f303233, 0x0830777031776b36, 0x0200000001000000, 0x0000020000003c02, 0x0d000000040a0a00, 0x000003000000080d,
        0x0a00000010020200, 0x200d0d000000200a, 0x04ffffffff000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00001a0000000200, 0x6930383231636900,
        0x383231636f343268, 0x33686b3432686f30, 0x0300000008316870, 0x0000500202000000, 0x0800000003070700, 0x0000010000000308, 0x03000000040a0a00, 0x0000100202000000,
        0x0d000000200a0a00, 0xffffff000000180d, 0x00000300000004ff, 0x0000040000000100, 0x0000010000000400, 0x2c00000020010100, 0x6869316369000000, 0x3630303177693038,
        0x38686f31636f3035, 0x3536303031776f30, 0x3332776b31686b30, 0x0831317770306870, 0x0800000001000000, 0x0000010000000308, 0x04000000080d0d00, 0x0000040202000000,
        0x0900000008080800, 0x080a0a0000002009, 0x0400000001000000, 0x0100000003000000, 0x0700000007000000, 0x0100000001000000, 0x00002c0000002001, 0x3038686931636900,
        0x3035363030317769, 0x6f3038686f31636f, 0x6b30353630303177, 0x68703332776b3168, 0x0000083131777030, 0x0308080000000100, 0x0d00000001000000, 0x000004000000080d,
        0x0800000002020200, 0x2009090000000808, 0x0000080a0a000000, 0x0000040000000100, 0x0000010000000300, 0x0000070000000700, 0x0200010000000100, 0x690000001c000000,
        0x3332776930323363, 0x6f303436636f3430, 0x31776b3430333277, 0x0100000008307770, 0x0000140202000000, 0x040a0a0000000200, 0x0000080d0d000000, 0x1002020000000300,
        0x0000200a0a000000, 0xff000000180d0d00, 0x0300000004ffffff, 0x0400000001000000, 0x0100000004000000, 0x0000800101000000, 0x3663690000001b00, 0x6f36333133776934,
        0x3133776f36353263, 0x30777031776b3633, 0x0000000100000008, 0x0001000000020202, 0x000000080d0d0000, 0x0020020200000003, 0x0000002009090000, 0x0002000000200a0a,
        0x0003000000040000, 0x0007000000030000, 0x0007000000010000, 0x0000000100010000, 0x693363690000001a, 0x3231636f34323268, 0x34686b3635686f38, 0x0008306870346873,
        0x0c0c000000010000, 0x0000000100000008, 0x0004000000020a0a, 0x0000000402020000, 0x0a0a000000040808, 0x00100d0d00000008, 0x0005000000000000, 0x0003000000030000,
        0x0001000000070000, 0x0001000000070000, 0x0000001a00000018, 0x7769383430326369, 0x38343032636f3934, 0x7031776b3934776f, 0x0001000000103077, 0x000000310d0d0000,
        0x0004020200000002, 0x000000040a0a0000, 0x0020020200000003, 0x0000001009090000, 0x0003000000400a0a, 0x0003000000050000, 0x0007000000010000, 0x0001000000070000,
        0x0000000100010000, 0x3938636900000018, 0x31636f3436776936, 0x776b3436776f3832, 0x0000001030777031, 0x0007020200000001, 0x0202000000010000, 0x0000000300000008,
        0x0a0a000000100202, 0x00080d0d00000010, 0x0000000000000000
    };
    return data;
};
// clang-format on

} // namespace jit
} // namespace intel
} // namespace gpu
} // namespace impl
} // namespace dnnl
