/*
*  qm_browser.cpp
*  QUIMUP library & playlist window
*  © 2008-2024 Johan Spee
*  SPDX-License-Identifier: GPL-3.0-or-later
*/

#include <QTimer>
#include "qm_browser.h"
#include "qm_modes_ids.h"

qm_Browser::qm_Browser(qm_Config *cfg)
{
    if (objectName().isEmpty())
        setObjectName("quimup browser");

    config = cfg;

    set_themed_icons(config->set_dark_theme);

    // init_vars
    mpd_cmd = nullptr;
    stream_fetcher = nullptr;
    playlist_view = nullptr;
    library_view = nullptr;
    b_mpd_connected = false;
    b_startup = true;
    b_wayland = config->this_is_wayland;
    resize(config->browser_W, config->browser_H );

    setup_widgets();
    retranslate();
}


void qm_Browser::setup_widgets()
{
    if (objectName().isEmpty())
        setObjectName("browser");

    setWindowIcon(QPixmap(":/qm_main_icon"));

    QSizePolicy sizePolicy(static_cast<QSizePolicy::Policy>(5), static_cast<QSizePolicy::Policy>(5));
    sizePolicy.setHorizontalStretch(0);
    sizePolicy.setVerticalStretch(0);
    sizePolicy.setHeightForWidth(this->sizePolicy().hasHeightForWidth());
    setSizePolicy(sizePolicy);
    setMinimumSize(QSize(620, 400));
    setMaximumSize(QSize(32767, 32767));

    main_widget = new QWidget(this);
    vbox_all = new QVBoxLayout(main_widget);
    setCentralWidget(main_widget);
    vbox_all->setContentsMargins(4,4,4,4);
    vbox_all->setSpacing(0);

    sp_splitter = new QSplitter(this);
    sp_splitter->setOrientation(Qt::Horizontal);
    sp_splitter->setHandleWidth(8);
    sp_splitter->setChildrenCollapsible(false);

    lb_left_group = new QLabel(sp_splitter);
    lb_left_group->setMinimumSize(QSize(300, 0));
    lb_left_group->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    vbox_lefthalf = new QVBoxLayout(lb_left_group);
    vbox_lefthalf->setSpacing(4);
    vbox_lefthalf->setContentsMargins(2,2,2,2);
    lb_left_title = new QLabel();
    lb_left_title->setTextFormat(Qt::RichText);
    lb_left_title->setAlignment(Qt::AlignCenter);
    lb_left_title->setAutoFillBackground(true);
    lb_left_title->setContentsMargins(1,1,1,1);
    vbox_lefthalf->addWidget(lb_left_title);
    hbox_leftop = new QHBoxLayout();
    hbox_leftop->setSpacing(4);
    hbox_leftop->setContentsMargins(0,0,0,0);
    cb_dbasemode = new QComboBox();
    cb_dbasemode->setMinimumSize(QSize(100, 26));
    cb_dbasemode->setMaximumHeight(26);
    hbox_leftop->addWidget(cb_dbasemode);
    spacer_left = new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum );
    hbox_leftop->addItem(spacer_left);
    bt_update = new QPushButton();
    bt_update->setMinimumSize(QSize(100, 26));
    bt_update->setMaximumHeight(26);
    bt_update->setIcon(ic_update);
    hbox_leftop->addWidget(bt_update);
    vbox_lefthalf->addLayout(hbox_leftop);
    library_view = new qm_libraryView(lb_left_group, config);
    library_view->setHorizontalScrollBarPolicy (Qt::ScrollBarAlwaysOn);
    library_view->setVerticalScrollBarPolicy (Qt::ScrollBarAlwaysOn); // avoid problems with viewport()->width();
    library_view->setSizePolicy (QSizePolicy::Expanding, QSizePolicy::Expanding);
    library_view->setUniformRowHeights(true);
    library_view->setAlternatingRowColors(true);
    vbox_lefthalf->addWidget(library_view);
    lb_dbasestats = new QLabel();
    lb_dbasestats->setAlignment(Qt::AlignHCenter);
    lb_dbasestats->setAutoFillBackground(true);
    vbox_lefthalf->addWidget(lb_dbasestats);
    hbox_leftbottom = new QHBoxLayout();
    hbox_leftbottom->setSpacing(4);
    hbox_leftbottom->setContentsMargins(0,0,0,0);
    cb_searchmode = new QComboBox();
    cb_searchmode->setMinimumSize(QSize(100, 26));
    cb_searchmode->setMaximumHeight(26);
    hbox_leftbottom->addWidget(cb_searchmode);
    le_search = new QLineEdit();
    le_search->setMinimumSize(QSize(0, 24));
    le_search->setMaximumHeight(24);
    hbox_leftbottom->addWidget(le_search);
    bt_search = new QPushButton();
    bt_search->setMinimumSize(QSize(100, 26));
    bt_search->setMaximumHeight(26);
    bt_search->setIcon(ic_bt_search);
    hbox_leftbottom->addWidget(bt_search);
    vbox_lefthalf->addLayout(hbox_leftbottom);
    sp_splitter->addWidget(lb_left_group);

    lb_right_group = new QLabel(sp_splitter);
    lb_right_group->setMinimumSize(QSize(300, 0));
    lb_right_group->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    vbox_righthalf = new QVBoxLayout(lb_right_group);
    vbox_righthalf->setSpacing(4);
    vbox_righthalf->setContentsMargins(2,2,2,2);
    lb_right_title = new QLabel();
    lb_right_title->setTextFormat(Qt::RichText);
    lb_right_title->setAlignment(Qt::AlignCenter);
    lb_right_title->setAutoFillBackground(true);
    lb_right_title->setContentsMargins(1,1,1,1);
    vbox_righthalf->addWidget(lb_right_title);
    hbox_rightop = new QHBoxLayout();
    hbox_rightop->setSpacing(4);
    hbox_rightop->setContentsMargins(0,0,0,0);
    bt_stream = new QPushButton();
    bt_stream->setMinimumSize(QSize(100, 26));
    bt_stream->setMaximumHeight(26);
    bt_stream->setIcon(ic_bt_stream);
    hbox_rightop->addWidget(bt_stream);
    spacer_right = new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum );
    hbox_rightop->addItem(spacer_right);
    vbox_righthalf->addLayout(hbox_rightop);
    playlist_view = new qm_playlistView(lb_right_group, config);
    playlist_view->setHorizontalScrollBarPolicy (Qt::ScrollBarAlwaysOn);
    playlist_view->setVerticalScrollBarPolicy (Qt::ScrollBarAlwaysOn); // avoid problems with viewport()->width();
    playlist_view->setSizePolicy (QSizePolicy::Expanding, QSizePolicy::Expanding);
    playlist_view->setUniformRowHeights(true);
    playlist_view->setAlternatingRowColors(true);
    vbox_righthalf->addWidget(playlist_view);
    lb_pliststats = new QLabel();
    lb_pliststats->setAlignment(Qt::AlignHCenter);
    lb_pliststats->setAutoFillBackground(true);
    vbox_righthalf->addWidget(lb_pliststats);
    hbox_rightbottom = new QHBoxLayout();
    hbox_rightbottom->setSpacing(4);
    hbox_rightbottom->setContentsMargins(0,0,0,0);
    cb_selectmode = new QComboBox();
    cb_selectmode->setMinimumSize(QSize(100, 26));
    cb_selectmode->setMaximumHeight(26);
    hbox_rightbottom->addWidget(cb_selectmode);
    le_select = new QLineEdit();
    le_select->setMinimumSize(QSize(0, 24));
    le_select->setMaximumHeight(24);
    hbox_rightbottom->addWidget(le_select);
    bt_select = new QPushButton();
    bt_select->setMinimumSize(QSize(100, 26));
    bt_select->setMaximumHeight(26);
    bt_select->setIcon(ic_bt_select);
    hbox_rightbottom->addWidget(bt_select);
    vbox_righthalf->addLayout(hbox_rightbottom);
    sp_splitter->addWidget(lb_right_group);
    vbox_all->addWidget(sp_splitter);

    // set the splitter pos
    QList<int> List;
    List.append(config->browser_L_splitter);
    List.append(config->browser_R_splitter);
    sp_splitter->setSizes(List);
    // icons set in set_themed_icons
    cb_dbasemode->addItem (tr("Artists"));
    cb_dbasemode->addItem (tr("Albums"));
    cb_dbasemode->addItem (tr("Timestamp"));
    cb_dbasemode->addItem (tr("Year"));
    cb_dbasemode->addItem (tr("Titles"));
    cb_dbasemode->addItem (tr("Genres"));
    cb_dbasemode->addItem (tr("Playlists"));
    cb_dbasemode->addItem (tr("Folders"));
    cb_dbasemode->addItem (tr("Searched"));
    cb_dbasemode->setIconSize(QSize(18, 18));
    cb_dbasemode->blockSignals(true);
    cb_dbasemode->setCurrentIndex(config->browser_libmode);
    cb_dbasemode->blockSignals(false);

    cb_dbasemode->setItemIcon (0, ic_artist);
    cb_dbasemode->setItemIcon (1, ic_album);
    cb_dbasemode->setItemIcon (2, ic_album);
    cb_dbasemode->setItemIcon (3, ic_album);
    cb_dbasemode->setItemIcon (4, ic_titles);
    cb_dbasemode->setItemIcon (5, ic_genre);
    cb_dbasemode->setItemIcon (6, ic_playlist);
    cb_dbasemode->setItemIcon (7, ic_dir);
    cb_dbasemode->setItemIcon (8, ic_search);

    cb_searchmode->addItem (tr("Artists"));
    cb_searchmode->addItem (tr("Albums"));
    cb_searchmode->addItem (tr("Titles"));
    cb_searchmode->addItem (tr("Genres"));
    cb_searchmode->addItem (tr("Folders"));
    cb_searchmode->setIconSize(QSize(18, 18));
    cb_searchmode->setCurrentIndex(config->browser_searchmode);
    if (config->browser_libmode != LB_ID_SEARCH)
        config->search_string = "";

    cb_searchmode->setItemIcon ( 0, ic_dot);
    cb_searchmode->setItemIcon ( 1, ic_dot);
    cb_searchmode->setItemIcon ( 2, ic_dot);
    cb_searchmode->setItemIcon ( 3, ic_dot);
    cb_searchmode->setItemIcon ( 4, ic_dot);

    cb_selectmode->addItem (tr("Artists"));
    cb_selectmode->addItem (tr("Albums"));
    cb_selectmode->addItem (tr("Titles"));
    cb_selectmode->addItem (tr("All"));
    cb_selectmode->setIconSize(QSize(18, 18));

    cb_selectmode->setItemIcon ( 0, ic_dot);
    cb_selectmode->setItemIcon ( 1, ic_dot);
    cb_selectmode->setItemIcon ( 2, ic_dot);
    cb_selectmode->setItemIcon ( 3, ic_dot);

    stream_fetcher = new qm_streamFetcher();

    tb_options = new QToolBar();
    tb_options->setIconSize(QSize(18,18));
    tb_options->setMinimumSize(QSize(130, 26));
    tb_options->setMaximumHeight(26);
    hbox_rightop->addWidget(tb_options);

    a_random = new QAction(tb_options);
    a_random->setText(tr("Random playback"));
    a_random->setCheckable(true);
    a_random->setIcon(ic_a_random);
    connect( a_random, SIGNAL(triggered()), this, SLOT(set_random()) );
    tb_options->addAction(a_random);

    a_repeat = new QAction(tb_options);
    a_repeat->setText(tr("Repeat list/single"));
    a_repeat->setCheckable(true);
    a_repeat->setIcon(ic_a_repeat);
    connect( a_repeat, SIGNAL(triggered()), this, SLOT(set_repeat()) );
    tb_options->addAction(a_repeat);

    a_single = new QAction(tb_options);
    a_single->setText(tr("Single mode"));
    a_single->setCheckable(true);
    a_single->setIcon(ic_a_single);

    connect( a_single, SIGNAL(triggered()), this, SLOT(set_single()) );
    tb_options->addAction(a_single);

    a_consume = new QAction(tb_options);
    a_consume->setText(tr("Consume mode"));
    a_consume->setCheckable(true);
    a_consume->setIcon(ic_a_consume);
    connect( a_consume, SIGNAL(triggered()), this, SLOT(set_consume()) );
    tb_options->addAction(a_consume);

    a_xfade = new QAction(tb_options);
    a_xfade->setText(tr("Cross fade"));
    a_xfade->setCheckable(true);
    a_xfade->setIcon(ic_a_xfade);
    connect( a_xfade, SIGNAL(triggered()), this, SLOT(set_xfade()) );
    tb_options->addAction(a_xfade);

    cb_selectmode->setCurrentIndex(config->browser_selectmode);
    config->select_string = "";
    le_select->setText(config->select_string);
    le_search->setText(config->search_string);

    connect( stream_fetcher, SIGNAL(streamdownload_done(QStringList)), this, SLOT(on_streamdownload_done(QStringList)) );
    connect( cb_dbasemode, SIGNAL(activated(int)), this, SLOT(on_dbase_newmode(int)) );
    connect( cb_searchmode, SIGNAL(activated(int)), this, SLOT(set_searchmode(int)) );
    connect( cb_selectmode, SIGNAL(activated(int)), this, SLOT(set_selectmode(int)) );
    connect( bt_search, SIGNAL(clicked()), this, SLOT(search_in_library()) );
    connect( le_search, SIGNAL(returnPressed()), this, SLOT(search_in_library()) );
    connect( bt_select, SIGNAL(clicked()), this, SLOT(select_in_plist()));
    connect( le_select, SIGNAL(returnPressed()), this, SLOT(select_in_plist()) );
    connect( bt_update, SIGNAL(clicked()), this, SLOT(on_update_dbase()) );
    connect( bt_stream, SIGNAL(clicked()), this, SLOT(get_the_stream()) );
    connect( playlist_view, SIGNAL(signal_playlist_statistics(QString)), this, SLOT(set_playlist_stats(QString)) );
    connect( playlist_view, SIGNAL(signal_playlist_dropreceived(int)), library_view, SLOT(on_playlist_dropreceived(int)) );
    connect( sp_splitter, SIGNAL(splitterMoved( int, int )), this, SLOT(splittermoved()) );
} // << setup_widgets



void qm_Browser::get_the_stream()
{
    QString suggested_name = "http://";
    bool ok;
    QString input = QInputDialog::getText(this, tr("Enter URL"),
                                          tr("Location of the stream's playlist or audio file:"), QLineEdit::Normal,
                                          suggested_name, &ok);

    if ( !ok || input.isEmpty() || input == "http:// or https://")
        return;

    if (!input.startsWith("http:") && !input.startsWith("https:"))
    {
        QMessageBox::warning( this, tr("Error"), "URL must begin with \"http://\" or \"https://\"");
        return;
    }

    stream_fetcher->download_this(input);
    // and wait for signal_on_streamdownload_done
}


void qm_Browser::on_streamdownload_done(QStringList result)
{
    qm_commandList cmdList;
    if (result.value(0) == "error")
    {
        QMessageBox::warning( this, tr("Error"), result.value(1));
        return;
    }

    if (result.value(0) == "nothing")
    {
        QString message = tr("Expected file was not found.\nTry a Winamp compatible URL that\npoints to a playlist or audio file.");
        QMessageBox::warning( this, tr("Error"), message);
        return;
    }

    if (result.value(0) == "timeout")
    {
        QString message = tr("Download takes too long:") + " " + result.value(1) + " " + tr("% in 10 sec.\nDownload aborted.");
        QMessageBox::warning( this, tr("Error"), message);
        return;
    }

    for (int i = 2; i < result.size(); ++i)
    {
        qm_mpd_command newCommand;
        newCommand.cmd = CMD_ADD;
        newCommand.uri = result.value(i);
        newCommand.moveto = 0;
        newCommand.artist = "Stream";
        newCommand.title = result.value(i);
        newCommand.time = 0;
        newCommand.album = tr("Info in player");
        newCommand.track = "";
        newCommand.type = TP_STREAM;
        cmdList.push_back(newCommand);
    }

    int count = mpd_cmd->execute_commands(cmdList, true );
    QString message =  tr("Downloaded file:") + " " + result.value(1) +  "\n"  + QString::number(count) +  " " + tr("items added to the playlist.");
    QMessageBox::information( this, tr("Confirmation"), message);
}


void qm_Browser::on_dbase_newmode(int i)
{

    config->browser_libmode = i;
    library_view->reload_root();
}


void qm_Browser::set_searchmode(int i)
{
    config->browser_searchmode = i;
}


void qm_Browser::set_selectmode(int i)
{
    config->browser_selectmode = i;
}


void qm_Browser::search_in_library()
{
    config->search_string = le_search->text();
    cb_dbasemode->blockSignals(true);
    cb_dbasemode->setCurrentIndex(LB_ID_SEARCH);
    cb_dbasemode->blockSignals(false);
    on_dbase_newmode(LB_ID_SEARCH);
    library_view->setFocus();
}


void qm_Browser::select_in_plist()
{
    config->select_string = le_select->text();
    playlist_view->select_it(cb_selectmode->currentIndex() );
    playlist_view->setFocus();
}


void qm_Browser::retranslate()
{
    setWindowTitle(tr("Quimup Media Browser"));

    if (config->show_tooltips)
    {
        bt_stream->setToolTip(tr("Enter a URL to an internet audio stream"));
        bt_update->setToolTip(tr("Scan the entire music collection for changes"));
        cb_searchmode->setToolTip(tr("Search mode"));
        cb_selectmode->setToolTip(tr("Select mode"));
        cb_dbasemode->setToolTip(tr("Library mode"));
    }

    bt_select->setText(tr("Select"));
    bt_search->setText(tr("Search"));
    bt_update->setText(tr("Update"));
    bt_stream->setText(tr("Stream"));
    lb_left_title->setText( "<b>" + tr("Library") +  "</b>");
    lb_right_title->setText("<b>" + tr("Playlist") + "</b>");
}


void qm_Browser::splittermoved()
{
    if (sp_splitter->sizes().value(0) != 0 && sp_splitter->sizes().value(1) != 0)
    {
        config->browser_L_splitter = sp_splitter->sizes().value(0);
        config->browser_R_splitter = sp_splitter->sizes().value(1);
    }
}


void qm_Browser::hideEvent(QHideEvent *event)
{
    if (!b_wayland)
    {
        config->browser_X = x();
        config->browser_Y = y();
    }
    config->browser_W = width();
    config->browser_H = height();
    event->accept();
}


void qm_Browser::set_playlist_stats(QString stats)
{
    lb_pliststats->setText(stats);
}


void qm_Browser::set_dbase_stats(QString stats)
{
    lb_dbasestats->setText(stats);
}


void qm_Browser::showEvent(QShowEvent *event)
{
    if (!b_wayland)
        move(config->browser_X, config->browser_Y);
    event->accept();
}


void qm_Browser::closeEvent( QCloseEvent *event )
{
    this->hide();
    event->ignore(); // only hide, don't close
}


void qm_Browser::set_connected(qm_mpdCommand *cmd, bool isconnected)
{
    mpd_cmd = cmd;
    if (mpd_cmd == nullptr)
        b_mpd_connected = false;
    else
        b_mpd_connected = isconnected;

    if (b_mpd_connected)
    {
        connect( mpd_cmd, SIGNAL(signal_statistics(QString)), this, SLOT(set_dbase_stats(QString)) );
        connect( mpd_cmd, SIGNAL(signal_update_ready()), this, SLOT(on_update_ready()) );

        bt_stream->setEnabled(true);
        bt_select->setEnabled(true);
        bt_search->setEnabled(true);

        if (config->mpd_stats_allowed)
            mpd_cmd->get_statistics();
        else
            set_dbase_stats("not connected");

        if (config->mpd_update_allowed)
            bt_update->setEnabled(true);
        else
            bt_update->setEnabled(false);

        if (config->mpd_single_allowed)
            a_single->setEnabled(true);
        else
            a_single->setEnabled(false);

        if (config->mpd_consume_allowed)
            a_consume->setEnabled(true);
        else
            a_consume->setEnabled(false);

        if (config->mpd_xfade_allowed)
            a_xfade->setEnabled(true);
        else
            a_xfade->setEnabled(false);

        if (config->mpd_repeat_allowed)
            a_repeat->setEnabled(true);
        else
            a_repeat->setEnabled(false);

        if (config->mpd_random_allowed)
            a_random->setEnabled(true);
        else
            a_random->setEnabled(false);
    }
    else
    {
        set_dbase_stats("...");
        set_playlist_stats("...");
        bt_update->setEnabled(false);
        bt_stream->setEnabled(false);
        bt_select->setEnabled(false);
        bt_search->setEnabled(false);
        a_single->setEnabled(false);
        a_consume->setEnabled(false);
        a_xfade->setEnabled(false);
        a_repeat->setEnabled(false);
        a_random->setEnabled(false);
    }
}


void qm_Browser::on_update_dbase()
{
    if (!config->mpd_update_allowed)
        return;

    bt_update->setEnabled(false);
    bt_update->setIcon(ic_busy);
    mpd_cmd->update_all();
}


void qm_Browser::on_update_ready()
{
    bt_update->setDisabled(false);
    bt_update->setIcon(ic_update);
    library_view->reload_root();
}


void qm_Browser::set_menu_random(bool isitso)
{
    a_random->blockSignals(true);
    a_random->setChecked(isitso);
    a_random->blockSignals(false);
}


void qm_Browser::set_menu_repeat(bool isitso)
{
    a_repeat->blockSignals(true);
    a_repeat->setChecked(isitso);
    a_repeat->blockSignals(false);
}


void qm_Browser::set_menu_single(bool isitso)
{
    a_single->blockSignals(true);
    a_single->setChecked(isitso);
    a_single->blockSignals(false);
}


void qm_Browser::set_menu_consume(bool isitso)
{
    a_consume->blockSignals(true);
    a_consume->setChecked(isitso);
    a_consume->blockSignals(false);
}

// called by player
void qm_Browser::set_menu_xfade(bool isitso)
{
    a_xfade->blockSignals(true);
    a_xfade->setChecked(isitso);
    config->xfade_is_on = isitso;
    a_xfade->blockSignals(false);
}


void qm_Browser::set_random()
{
    if (b_mpd_connected)
        mpd_cmd->set_random(a_random->isChecked());
    else
        set_menu_random(!a_random->isChecked());
}


void qm_Browser::set_repeat()
{
    if (b_mpd_connected)
        mpd_cmd->set_repeat(a_repeat->isChecked());
    else
        set_menu_repeat(!a_repeat->isChecked());
}


void qm_Browser::set_single()
{
    if (b_mpd_connected)
        mpd_cmd->set_single(a_single->isChecked());
    else
        set_menu_single(!a_single->isChecked());
}


void qm_Browser::set_consume()
{
    if (b_mpd_connected)
        mpd_cmd->set_consume(a_consume->isChecked());
    else
        set_menu_consume(!a_consume->isChecked());
}


void qm_Browser::set_xfade()
{
    if (b_mpd_connected) //else a_xfade is disabled
    {
        if (a_xfade->isChecked())
        {
            mpd_cmd->set_xfade(config->xfade_time);
            config->xfade_is_on = true; // settings window needs this
        }
        else
        {
            mpd_cmd->set_xfade(0);
            config->xfade_is_on = false;
        }
    }
}


void qm_Browser::set_colors(QPalette pal)
{
    lb_left_title->setPalette(pal);
    lb_right_title->setPalette(pal);
    lb_dbasestats->setPalette(pal);
    lb_pliststats->setPalette(pal);
}

void qm_Browser::keyPressEvent(QKeyEvent *event)
{
    switch (event->key())
    {
        case Qt::Key_MediaPause:
        {
            emit signal_keypressed(event);
            break;

        }
        case Qt::Key_MediaPlay:
        {
            emit signal_keypressed(event);
            break;

        }
        case Qt::Key_MediaNext:
        {
            emit signal_keypressed(event);
            break;

        }
        case Qt::Key_MediaPrevious:
        {
            emit signal_keypressed(event);
            break;

        }
        case Qt::Key_MediaStop:
        {
            emit signal_keypressed(event);;
            break;

        }
        default:
            QMainWindow::keyPressEvent(event);
    }
}


void qm_Browser::set_themed_icons(bool dark_theme)
{
    if (!dark_theme)
    {
        ic_expand_out = QIcon(":/br_expand_out");
        ic_expand_in = QIcon(":/br_expand_in");
        ic_update = QIcon(":/pl_br_update");
        ic_busy = QIcon(":/br_busy");
        ic_dot = QIcon(":/br_dot");
        ic_bt_search = QIcon(":/br_pointleft");
        ic_bt_stream = QIcon(":/br_stream");
        ic_bt_select = QIcon(":/br_pointleft");
        ic_a_random = QIcon(":/br_random");
        ic_a_repeat = QIcon(":/br_repeat");
        ic_a_single = QIcon(":/br_single");
        ic_a_consume = QIcon(":/br_consume");
        ic_a_xfade = QIcon(":/br_xfade");
        ic_artist = QIcon(":/br_artist");
        ic_album = QIcon(":/br_album");
        ic_timestamp = QIcon(":/br_timestamp");
        ic_albumyear = QIcon(":/br_albumyear");
        ic_titles = QIcon(":/br_track");
        ic_genre = QIcon(":/br_genre");
        ic_playlist = QIcon(":/br_playlist");
        ic_search = QIcon(":/br_search");
        ic_dir = QIcon(":/pl_br_dir");
    }
    else
    {
        ic_expand_out = QIcon(":/br_expand_out.alt");
        ic_expand_in = QIcon(":/br_expand_in.alt");
        ic_update = QIcon(":/pl_br_update.alt");
        ic_busy = QIcon(":/br_busy.alt");
        ic_dot = QIcon(":/br_dot.alt");
        ic_bt_search = QIcon(":/br_pointleft.alt");
        ic_bt_stream = QIcon(":/br_stream.alt");
        ic_bt_select = QIcon(":/br_pointleft.alt");
        ic_a_random = QIcon(":/br_random.alt");
        ic_a_repeat = QIcon(":/br_repeat.alt");
        ic_a_single = QIcon(":/br_single.alt");
        ic_a_consume = QIcon(":/br_consume.alt");
        ic_a_xfade = QIcon(":/br_xfade.alt");
        ic_artist = QIcon(":/br_artist.alt");
        ic_album = QIcon(":/br_album.alt");
        ic_timestamp = QIcon(":/br_timestamp.alt");
        ic_albumyear = QIcon(":/br_albumyear.alt");
        ic_titles = QIcon(":/br_track.alt");
        ic_genre = QIcon(":/br_genre.alt");
        ic_playlist = QIcon(":/br_playlist.alt");
        ic_search = QIcon(":/br_search.alt");
        ic_dir = QIcon(":/pl_br_dir.alt");
    }
}

// called from changeEvent()
void qm_Browser::resize_columns()
{
    playlist_view->resize_columns();
    library_view->resize_columns();
}

// TODO : fixed in later QT version?
// when maximized or restored playlist & library are not resized
void qm_Browser::changeEvent(QEvent *event)
{
    bool make_changes = false;

    // if maximized or restored
    if (event->type() == QEvent::WindowStateChange)
        make_changes = true;

    // allow the changeEvent (resets event)
    event->accept();

    if(make_changes)
    {
        // still have to wait a few ms for the changes
        QTimer::singleShot(20, this, SLOT(resize_columns()));
    }
}


qm_Browser::~qm_Browser()
{}
