{-# LANGUAGE DeriveDataTypeable #-}
{-# LANGUAGE OverloadedStrings #-}

-- | An OptionalConfiguration is just like a 'Configuration', except
--   all of its fields are optional. The user can set options in two
--   places: the command-line, and a configuration file. Obviously if
--   a parameter is set in one place, it doesn't need to be set in the
--   other. Thus, the latter needs to be optional.
--

module OptionalConfiguration (
  OptionalConfiguration(..),
  from_rc,
  merge_maybes )
where

import qualified Data.Configurator as DC (
  Worth(Optional),
  load,
  lookup )
import Data.Data ( Data )
import Paths_mailbox_count ( getSysconfDir )
import System.Directory ( getHomeDirectory )
import System.FilePath ( (</>) )
import System.IO.Error ( catchIOError )


-- | The same as Configuration, except everything is optional. It's easy to
--   merge two of these by simply dropping the Nothings in favor of
--   the Justs. The 'feed_hosts' are left un-maybed so that cmdargs
--   can parse more than one of them.
--
data OptionalConfiguration =
  OptionalConfiguration {
    database :: Maybe String,
    detail   :: Maybe Bool,
    detail_query :: Maybe String,
    host     :: Maybe String,
    password :: Maybe String,
    port     :: Maybe Int,
    summary_query :: Maybe String,
    username :: Maybe String }
  deriving (Show, Data)


-- | Combine two Maybes into one, essentially mashing them
--   together. We give precedence to the second argument when both are
--   Justs.
merge_maybes :: (Maybe a) -> (Maybe a) -> (Maybe a)
merge_maybes Nothing Nothing   = Nothing
merge_maybes (Just x) Nothing  = Just x
merge_maybes Nothing (Just x)  = Just x
merge_maybes (Just _) (Just y) = Just y


-- | The Semigroup instance for these lets us "combine" two
--   configurations. The "combine" operation that we'd like to perform
--   is, essentially, to mash them together. So if we have two
--   OptionalConfigurations, each half full, we could combine them
--   into one big one.
--
--   This is used to merge command-line and config-file settings.
--
instance Semigroup OptionalConfiguration where
  -- | Combine @cfg1@ and @cfg2@, giving precedence to @cfg2@.
  cfg1 <> cfg2 =
    OptionalConfiguration
      (merge_maybes (database cfg1) (database cfg2))
      (merge_maybes (detail cfg1) (detail cfg2))
      (merge_maybes (detail_query cfg1) (detail_query cfg2))
      (merge_maybes (host cfg1) (host cfg2))
      (merge_maybes (password cfg1) (password cfg2))
      (merge_maybes (port cfg1) (port cfg2))
      (merge_maybes (summary_query cfg1) (summary_query cfg2))
      (merge_maybes (username cfg1) (username cfg2))


-- | The Monoid instance essentially only provides the "empty
--   configuration."
instance Monoid OptionalConfiguration where
  -- | An empty OptionalConfiguration.
  mempty = OptionalConfiguration
             Nothing
             Nothing
             Nothing
             Nothing
             Nothing
             Nothing
             Nothing
             Nothing
  mappend = (<>)


-- | Obtain an OptionalConfiguration from mailbox-countrc in either
--   the global configuration directory or the user's home
--   directory. The one in $HOME is prefixed by a dot so that it is
--   hidden.
--
--   We make an attempt at cross-platform compatibility; we will try
--   to find the correct directory even on Windows. But if the calls
--   to getHomeDirectory/getSysconfDir fail for whatever reason, we
--   fall back to using the Unix-specific /etc and $HOME.
--
from_rc :: IO OptionalConfiguration
from_rc = do
  etc  <- catchIOError getSysconfDir (\e -> do
                                        putStrLn $ "ERROR: " ++ (show e)
                                        return "/etc")
  home <- catchIOError getHomeDirectory (\e -> do
                                           putStrLn $ "ERROR: " ++ (show e)
                                           return "$(HOME)")
  let global_config_path = etc </> "mailbox-countrc"
  let user_config_path = home </> ".mailbox-countrc"
  cfg <- DC.load [ DC.Optional global_config_path,
                   DC.Optional user_config_path ]
  cfg_database <- DC.lookup cfg "database"
  cfg_detail <- DC.lookup cfg "detail"
  cfg_detail_query <- DC.lookup cfg "detail_query"
  cfg_host <- DC.lookup cfg "host"
  cfg_password <- DC.lookup cfg "password"
  cfg_port <- DC.lookup cfg "port"
  cfg_summary_query <- DC.lookup cfg "summary_query"
  cfg_username <- DC.lookup cfg "username"

  return $ OptionalConfiguration
             cfg_database
             cfg_detail
             cfg_detail_query
             cfg_host
             cfg_password
             cfg_port
             cfg_summary_query
             cfg_username
