#include <stdlib.h>
#include <string.h>

#include "tr_cvar.h"
#include "vkimpl.h"
#include "tr_globals.h"
#include "vk_depth_attachment.h"
#include "vk_frame.h"
#include "vk_image.h"
#include "vk_instance.h"
#include "vk_pipelines.h"
#include "vk_shade_geometry.h"
#include "vk_shaders.h"

struct Vk_Instance vk;

//
// Vulkan API functions used by the renderer.
//
PFN_vkGetInstanceProcAddr qvkGetInstanceProcAddr;

PFN_vkCreateInstance qvkCreateInstance;
PFN_vkEnumerateInstanceExtensionProperties qvkEnumerateInstanceExtensionProperties;

PFN_vkCreateDevice qvkCreateDevice;
PFN_vkDestroyInstance qvkDestroyInstance;
PFN_vkEnumerateDeviceExtensionProperties qvkEnumerateDeviceExtensionProperties;
PFN_vkEnumeratePhysicalDevices qvkEnumeratePhysicalDevices;
PFN_vkGetDeviceProcAddr qvkGetDeviceProcAddr;
PFN_vkGetPhysicalDeviceFeatures qvkGetPhysicalDeviceFeatures;
PFN_vkGetPhysicalDeviceFormatProperties qvkGetPhysicalDeviceFormatProperties;
PFN_vkGetPhysicalDeviceMemoryProperties qvkGetPhysicalDeviceMemoryProperties;
PFN_vkGetPhysicalDeviceProperties qvkGetPhysicalDeviceProperties;
PFN_vkGetPhysicalDeviceQueueFamilyProperties qvkGetPhysicalDeviceQueueFamilyProperties;

PFN_vkDestroySurfaceKHR qvkDestroySurfaceKHR;
PFN_vkGetPhysicalDeviceSurfaceCapabilitiesKHR qvkGetPhysicalDeviceSurfaceCapabilitiesKHR;
PFN_vkGetPhysicalDeviceSurfaceFormatsKHR qvkGetPhysicalDeviceSurfaceFormatsKHR;
PFN_vkGetPhysicalDeviceSurfacePresentModesKHR qvkGetPhysicalDeviceSurfacePresentModesKHR;
PFN_vkGetPhysicalDeviceSurfaceSupportKHR qvkGetPhysicalDeviceSurfaceSupportKHR;

#ifndef NDEBUG
PFN_vkCreateDebugReportCallbackEXT qvkCreateDebugReportCallbackEXT;
PFN_vkDestroyDebugReportCallbackEXT qvkDestroyDebugReportCallbackEXT;
#endif

PFN_vkAllocateCommandBuffers qvkAllocateCommandBuffers;
PFN_vkAllocateDescriptorSets qvkAllocateDescriptorSets;
PFN_vkAllocateMemory qvkAllocateMemory;
PFN_vkBeginCommandBuffer qvkBeginCommandBuffer;
PFN_vkBindBufferMemory qvkBindBufferMemory;
PFN_vkBindImageMemory qvkBindImageMemory;
PFN_vkCmdBeginRenderPass qvkCmdBeginRenderPass;
PFN_vkCmdBindDescriptorSets qvkCmdBindDescriptorSets;
PFN_vkCmdBindIndexBuffer qvkCmdBindIndexBuffer;
PFN_vkCmdBindPipeline qvkCmdBindPipeline;
PFN_vkCmdBindVertexBuffers qvkCmdBindVertexBuffers;
PFN_vkCmdBlitImage qvkCmdBlitImage;
PFN_vkCmdClearAttachments qvkCmdClearAttachments;
PFN_vkCmdCopyBufferToImage qvkCmdCopyBufferToImage;
PFN_vkCmdCopyImage qvkCmdCopyImage;
PFN_vkCmdCopyImageToBuffer qvkCmdCopyImageToBuffer;
PFN_vkCmdDraw qvkCmdDraw;
PFN_vkCmdDrawIndexed qvkCmdDrawIndexed;
PFN_vkCmdEndRenderPass qvkCmdEndRenderPass;
PFN_vkCmdPipelineBarrier qvkCmdPipelineBarrier;
PFN_vkCmdPushConstants qvkCmdPushConstants;
PFN_vkCmdSetDepthBias qvkCmdSetDepthBias;
PFN_vkCmdSetScissor qvkCmdSetScissor;
PFN_vkCmdSetViewport qvkCmdSetViewport;
PFN_vkCreateBuffer qvkCreateBuffer;
PFN_vkCreateCommandPool qvkCreateCommandPool;
PFN_vkCreateDescriptorPool qvkCreateDescriptorPool;
PFN_vkCreateDescriptorSetLayout qvkCreateDescriptorSetLayout;
PFN_vkCreateFence qvkCreateFence;
PFN_vkCreateFramebuffer qvkCreateFramebuffer;
PFN_vkCreateGraphicsPipelines qvkCreateGraphicsPipelines;
PFN_vkCreateImage qvkCreateImage;
PFN_vkCreateImageView qvkCreateImageView;
PFN_vkCreatePipelineLayout qvkCreatePipelineLayout;
PFN_vkCreateRenderPass qvkCreateRenderPass;
PFN_vkCreateSampler qvkCreateSampler;
PFN_vkCreateSemaphore qvkCreateSemaphore;
PFN_vkCreateShaderModule qvkCreateShaderModule;
PFN_vkDestroyBuffer qvkDestroyBuffer;
PFN_vkDestroyCommandPool qvkDestroyCommandPool;
PFN_vkDestroyDescriptorPool qvkDestroyDescriptorPool;
PFN_vkDestroyDescriptorSetLayout qvkDestroyDescriptorSetLayout;
PFN_vkDestroyDevice qvkDestroyDevice;
PFN_vkDestroyFence qvkDestroyFence;
PFN_vkDestroyFramebuffer qvkDestroyFramebuffer;
PFN_vkDestroyImage qvkDestroyImage;
PFN_vkDestroyImageView qvkDestroyImageView;
PFN_vkDestroyPipeline qvkDestroyPipeline;
PFN_vkDestroyPipelineLayout qvkDestroyPipelineLayout;
PFN_vkDestroyRenderPass qvkDestroyRenderPass;
PFN_vkDestroySampler qvkDestroySampler;
PFN_vkDestroySemaphore qvkDestroySemaphore;
PFN_vkDestroyShaderModule qvkDestroyShaderModule;
PFN_vkDeviceWaitIdle qvkDeviceWaitIdle;
PFN_vkEndCommandBuffer qvkEndCommandBuffer;
PFN_vkFreeCommandBuffers qvkFreeCommandBuffers;
PFN_vkFreeDescriptorSets qvkFreeDescriptorSets;
PFN_vkFreeMemory qvkFreeMemory;
PFN_vkGetBufferMemoryRequirements qvkGetBufferMemoryRequirements;
PFN_vkGetDeviceQueue qvkGetDeviceQueue;
PFN_vkGetImageMemoryRequirements qvkGetImageMemoryRequirements;
PFN_vkGetImageSubresourceLayout qvkGetImageSubresourceLayout;
PFN_vkMapMemory qvkMapMemory;
PFN_vkUnmapMemory qvkUnmapMemory;
PFN_vkQueueSubmit qvkQueueSubmit;
PFN_vkQueueWaitIdle qvkQueueWaitIdle;
PFN_vkResetDescriptorPool qvkResetDescriptorPool;
PFN_vkResetFences qvkResetFences;
PFN_vkUpdateDescriptorSets qvkUpdateDescriptorSets;
PFN_vkWaitForFences qvkWaitForFences;
PFN_vkAcquireNextImageKHR qvkAcquireNextImageKHR;
PFN_vkCreateSwapchainKHR qvkCreateSwapchainKHR;
PFN_vkDestroySwapchainKHR qvkDestroySwapchainKHR;
PFN_vkGetSwapchainImagesKHR qvkGetSwapchainImagesKHR;
PFN_vkQueuePresentKHR qvkQueuePresentKHR;

#ifndef NDEBUG

static VKAPI_ATTR VkBool32 VKAPI_CALL vk_DebugCallback(VkDebugReportFlagsEXT flags, VkDebugReportObjectTypeEXT object_type,
												uint64_t object, size_t location, int32_t message_code,
												const char *layer_prefix, const char *message, void *user_data) {
	ri.Printf(PRINT_WARNING, "%s\n", message);
	return VK_FALSE;
}

static void vk_createDebugCallback(PFN_vkDebugReportCallbackEXT qvkDebugCB) {
	VkDebugReportCallbackCreateInfoEXT desc;
	ri.Printf(PRINT_DEVELOPER, " vk_createDebugCallback()\n");
	desc.sType = VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT;
	desc.pNext = NULL;
	desc.flags =
		VK_DEBUG_REPORT_WARNING_BIT_EXT | VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT | VK_DEBUG_REPORT_ERROR_BIT_EXT;
	desc.pfnCallback = qvkDebugCB;
	desc.pUserData = NULL;

	VK_CHECK(qvkCreateDebugReportCallbackEXT(vk.instance, &desc, NULL, &vk.h_debugCB));
}

#endif

static void vk_createInstance(void) {
	VkApplicationInfo appInfo;
	VkInstanceCreateInfo instanceCreateInfo;
	uint32_t nInsExt = 0;
	uint32_t i = 0;
	VkExtensionProperties *pInsExt;
	VkResult e;
	const char **ppInstanceExt;
	qboolean enableValidationLayers = r_enablevalidationlayers->integer;

	// There is no global state in Vulkan and all per-application state
	// is stored in a VkInstance object. Creating a VkInstance object
	// initializes the Vulkan library and allows the application to pass
	// information about itself to the implementation.
	ri.Printf(PRINT_DEVELOPER, " Creating instance: vk.instance\n");

	// The version of Vulkan that is supported by an instance may be
	// different than the version of Vulkan supported by a device or
	// physical device. Because Vulkan 1.0 implementations may fail
	// with VK_ERROR_INCOMPATIBLE_DRIVER, applications should determine
	// the version of Vulkan available before calling vkCreateInstance.
	// If the vkGetInstanceProcAddr returns NULL for vkEnumerateInstanceVersion,
	// it is a Vulkan 1.0 implementation. Otherwise, the application can
	// call vkEnumerateInstanceVersion to determine the version of Vulkan.
	appInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
	appInfo.pNext = NULL;
	appInfo.pApplicationName = CLIENT_WINDOW_TITLE;
	appInfo.applicationVersion = VK_MAKE_VERSION(1, 0, 0);
	appInfo.pEngineName = CLIENT_WINDOW_TITLE;
	appInfo.engineVersion = VK_MAKE_VERSION(1, 0, 0);
	// apiVersion must be the highest version of Vulkan that the
	// application is designed to use, encoded as described in the
	// API Version Numbers and Semantics section. The patch version
	// number specified in apiVersion is ignored when creating an
	// instance object. Only the major and minor versions of the
	// instance must match those requested in apiVersion.
	appInfo.apiVersion = VK_MAKE_VERSION(1, 0, 0);

	instanceCreateInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
	// pNext is NULL or a pointer to an extension-specific structure.
	instanceCreateInfo.pNext = NULL;
	// flags is reserved for future use.
	instanceCreateInfo.flags = 0;
	// pApplicationInfo is NULL or a pointer to an instance of
	// VkApplicationInfo. If not NULL, this information helps
	// implementations recognize behavior inherent to classes
	// of applications.
	instanceCreateInfo.pApplicationInfo = &appInfo;

	// check extensions availability
	//
	// Extensions may define new Vulkan commands, structures, and enumerants.
	// For compilation purposes, the interfaces defined by registered extensions,
	// including new structures and enumerants as well as function pointer types
	// for new commands, are defined in the Khronos-supplied vulkan_core.h
	// together with the core API. However, commands defined by extensions may
	// not be available for static linking - in which case function pointers to
	// these commands should be queried at runtime as described in Command Function
	// Pointers.
	// Extensions may be provided by layers as well as by a Vulkan implementation.
	//
	// check extensions availability
	// To retrieve a list of supported extensions before creating an instance
	VK_CHECK(qvkEnumerateInstanceExtensionProperties(NULL, &nInsExt, NULL));

	assert(nInsExt > 0);

	ri.Printf(PRINT_ALL, "--- Total %d instance extensions. ---\n", nInsExt);

	pInsExt = (VkExtensionProperties *)ri.Hunk_AllocateTempMemory(sizeof(VkExtensionProperties) * nInsExt);

	ppInstanceExt = (const char **)ri.Hunk_AllocateTempMemory(sizeof(char *) * (nInsExt));

	VK_CHECK(qvkEnumerateInstanceExtensionProperties(NULL, &nInsExt, pInsExt));

	// Each platform-specific extension is an instance extension.
	// The application must enable instance extensions with vkCreateInstance
	// before using them.

	// TODO: CHECK OUT
	// All of the instance wxtention enabled, Does this reasonable ?

	for (i = 0; i < nInsExt; i++) {
		ppInstanceExt[i] = pInsExt[i].extensionName;
	}

	instanceCreateInfo.enabledExtensionCount = nInsExt;
	instanceCreateInfo.ppEnabledExtensionNames = ppInstanceExt;

	instanceCreateInfo.enabledLayerCount = 0;
	instanceCreateInfo.ppEnabledLayerNames = NULL;

	if (enableValidationLayers) {
		static const char *validation_layer_name[] = {"VK_LAYER_KHRONOS_validation"};
		instanceCreateInfo.ppEnabledLayerNames = validation_layer_name;
		instanceCreateInfo.enabledLayerCount = 1;
	}

	e = qvkCreateInstance(&instanceCreateInfo, NULL, &vk.instance);
	if (e == VK_SUCCESS) {
		ri.Printf(PRINT_ALL, "--- Vulkan create instance success! ---\n\n");
	} else if (e == VK_ERROR_INCOMPATIBLE_DRIVER) {
		// The requested version of Vulkan is not supported by the driver
		// or is otherwise incompatible for implementation-specific reasons.
		ri.Error(ERR_FATAL, "The requested version of Vulkan is not supported by the driver.");
	} else if (e == VK_ERROR_EXTENSION_NOT_PRESENT) {
		ri.Error(ERR_FATAL, "Cannot find a specified extension library.");
	} else {
		ri.Error(ERR_FATAL, "%d, returned by qvkCreateInstance.", e);
	}

	ri.Hunk_FreeTempMemory((void*)ppInstanceExt);
	ri.Hunk_FreeTempMemory(pInsExt);
}

static void vk_loadGlobalFunctions(void) {
	ri.Printf(PRINT_ALL, " Loading vulkan instance functions\n");

	vk_getInstanceProcAddrImpl();

#define INIT_INSTANCE_FUNCTION(func)                                                                                   \
	q##func = (PFN_##func)qvkGetInstanceProcAddr(vk.instance, #func);                                                  \
	if (q##func == NULL) {                                                                                             \
		ri.Error(ERR_FATAL, "Failed to find entrypoint %s", #func);                                                    \
	}

	INIT_INSTANCE_FUNCTION(vkCreateInstance)
	INIT_INSTANCE_FUNCTION(vkEnumerateInstanceExtensionProperties)

	//
	// Get instance level functions.
	//
	vk_createInstance();

	INIT_INSTANCE_FUNCTION(vkCreateDevice)
	INIT_INSTANCE_FUNCTION(vkDestroyInstance)
	INIT_INSTANCE_FUNCTION(vkDestroySurfaceKHR)
	INIT_INSTANCE_FUNCTION(vkEnumerateDeviceExtensionProperties)
	INIT_INSTANCE_FUNCTION(vkEnumeratePhysicalDevices)
	INIT_INSTANCE_FUNCTION(vkGetDeviceProcAddr)

	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceFeatures)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceProperties)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceFormatProperties)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceMemoryProperties)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceQueueFamilyProperties)

	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceSurfaceCapabilitiesKHR)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceSurfaceFormatsKHR)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceSurfacePresentModesKHR)
	INIT_INSTANCE_FUNCTION(vkGetPhysicalDeviceSurfaceSupportKHR)

#ifndef NDEBUG
	INIT_INSTANCE_FUNCTION(vkCreateDebugReportCallbackEXT)
	INIT_INSTANCE_FUNCTION(vkDestroyDebugReportCallbackEXT) //
#endif

#undef INIT_INSTANCE_FUNCTION

	ri.Printf(PRINT_ALL, " Init global functions done.\n");
}

////////////////////////////////

static const char *renderer_name(const VkPhysicalDeviceProperties *props) {
	static char buf[sizeof(props->deviceName) + 64];
	const char *device_type;

	switch (props->deviceType) {
	case VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU:
		device_type = "Integrated";
		break;
	case VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU:
		device_type = "Discrete";
		break;
	case VK_PHYSICAL_DEVICE_TYPE_VIRTUAL_GPU:
		device_type = "Virtual";
		break;
	case VK_PHYSICAL_DEVICE_TYPE_CPU:
		device_type = "CPU";
		break;
	default:
		device_type = "OTHER";
		break;
	}

	Com_sprintf(buf, sizeof(buf), "%s %s, 0x%04x", device_type, props->deviceName, props->deviceID);

	return buf;
}

static void vk_selectPhysicalDevice(void) {
	VkPhysicalDevice *pPhyDev;
	int i, device_index = -1;

	// After initializing the Vulkan library through a VkInstance
	// we need to look for and select a graphics card in the system
	// that supports the features we need. In fact we can select any
	// number of graphics cards and use them simultaneously.
	uint32_t gpu_count = 0;

	// Initial call to query gpu_count, then second call for gpu info.
	qvkEnumeratePhysicalDevices(vk.instance, &gpu_count, NULL);

	if (gpu_count <= 0)
		ri.Error(ERR_FATAL, "Vulkan: no physical device found");

	pPhyDev = (VkPhysicalDevice *)ri.Hunk_AllocateTempMemory(sizeof(VkPhysicalDevice) * gpu_count);

	// TODO: multi graphic cards selection support
	VK_CHECK(qvkEnumeratePhysicalDevices(vk.instance, &gpu_count, pPhyDev));
	for (i = 0; i < gpu_count; i++) {
		VkPhysicalDeviceProperties props;
		qvkGetPhysicalDeviceProperties(pPhyDev[i], &props);
		ri.Printf(PRINT_ALL, " %i: %s\n", i, renderer_name(&props));
		if (device_index == -1 && props.deviceType == VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU) {
			device_index = i;
		} else if (device_index == -2 && props.deviceType == VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU) {
			device_index = i;
		}
	}

	vk.physical_device = pPhyDev[device_index];

	ri.Hunk_FreeTempMemory(pPhyDev);

	ri.Printf(PRINT_ALL, " Total %d graphics card, %i is choosed.\n", gpu_count, device_index);

	ri.Printf(PRINT_ALL, " Get physical device memory properties: vk.devMemProperties\n");
	qvkGetPhysicalDeviceMemoryProperties(vk.physical_device, &vk.devMemProperties);
}

static void vk_selectSurfaceFormat(void) {
	uint32_t nSurfmt;
	VkSurfaceFormatKHR *pSurfFmts;
	VkFormatProperties props;

	ri.Printf(PRINT_ALL, "\n -------- vk_selectSurfaceFormat() --------\n");

	// Get the numbers of VkFormat's that are supported
	// "vk.surface" is the surface that will be associated with the swapchain.
	// "vk.surface" must be a valid VkSurfaceKHR handle
	VK_CHECK(qvkGetPhysicalDeviceSurfaceFormatsKHR(vk.physical_device, vk.surface, &nSurfmt, NULL));
	if (nSurfmt <= 0) {
		ri.Error(ERR_FATAL, "Vulkan: no surface formats found");
	}

	pSurfFmts = (VkSurfaceFormatKHR *)ri.Hunk_AllocateTempMemory(nSurfmt * sizeof(VkSurfaceFormatKHR));

	// To query the supported swapchain format-color space pairs for a surface
	VK_CHECK(qvkGetPhysicalDeviceSurfaceFormatsKHR(vk.physical_device, vk.surface, &nSurfmt, pSurfFmts));

	// If the format list includes just one entry of VK_FORMAT_UNDEFINED, the surface
	// has no preferred format. Otherwise, at least one supported format will be returned.
	if ((nSurfmt == 1) && (pSurfFmts[0].format == VK_FORMAT_UNDEFINED)) {
		// special case that means we can choose any format
		vk.surface_format.format = VK_FORMAT_B8G8R8A8_UNORM;
		vk.surface_format.colorSpace = VK_COLORSPACE_SRGB_NONLINEAR_KHR;
		ri.Printf(PRINT_ALL, "VK_FORMAT_R8G8B8A8_UNORM\n");
		ri.Printf(PRINT_ALL, "VK_COLORSPACE_SRGB_NONLINEAR_KHR\n");
	} else {
		uint32_t i;
		ri.Printf(PRINT_ALL, " Total %d surface formats supported, we choose:\n", nSurfmt);

		for (i = 0; i < nSurfmt; i++) {
			if ((pSurfFmts[i].format == VK_FORMAT_B8G8R8A8_UNORM) &&
				(pSurfFmts[i].colorSpace == VK_COLORSPACE_SRGB_NONLINEAR_KHR)) {

				ri.Printf(PRINT_ALL, " format = VK_FORMAT_B8G8R8A8_UNORM\n");
				ri.Printf(PRINT_ALL, " colorSpace = VK_COLORSPACE_SRGB_NONLINEAR_KHR\n");

				vk.surface_format = pSurfFmts[i];
				break;
			}
		}

		if (i == nSurfmt)
			vk.surface_format = pSurfFmts[0];
	}

	ri.Hunk_FreeTempMemory(pSurfFmts);

	// To query the basic capabilities of a surface, needed in order to create a swapchain
	VK_CHECK(qvkGetPhysicalDeviceSurfaceCapabilitiesKHR(vk.physical_device, vk.surface, &vk.surface_caps));

	// VK_IMAGE_USAGE_TRANSFER_DST_BIT is required by image clear operations.
	if ((vk.surface_caps.supportedUsageFlags & VK_IMAGE_USAGE_TRANSFER_DST_BIT) == 0)
		ri.Error(ERR_FATAL, "VK_IMAGE_USAGE_TRANSFER_DST_BIT is not supported by you GPU.");

	// VK_IMAGE_USAGE_TRANSFER_SRC_BIT is required in order to take screenshots.
	if ((vk.surface_caps.supportedUsageFlags & VK_IMAGE_USAGE_TRANSFER_SRC_BIT) == 0)
		ri.Error(ERR_FATAL, "VK_IMAGE_USAGE_TRANSFER_SRC_BIT is not supported by you GPU.");

	// To query supported format features which are properties of the physical device

	// To determine the set of valid usage bits for a given format,
	// call vkGetPhysicalDeviceFormatProperties.

	// ========================= color ================
	qvkGetPhysicalDeviceFormatProperties(vk.physical_device, vk.surface_format.format, &props);

	// Check if the device supports blitting to linear images
	if (props.linearTilingFeatures & VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT)
		ri.Printf(PRINT_ALL, "--- Linear TilingFeatures supported. ---\n");

	if (props.linearTilingFeatures & VK_FORMAT_FEATURE_BLIT_DST_BIT) {
		ri.Printf(PRINT_ALL, "--- Blitting from linear tiled images supported. ---\n");
	}

	if (props.optimalTilingFeatures & VK_FORMAT_FEATURE_BLIT_SRC_BIT) {
		ri.Printf(PRINT_ALL, "--- Blitting from optimal tiled images supported. ---\n");
		vk.isBlitSupported = VK_TRUE;
	}

	//=========================== depth =====================================
	qvkGetPhysicalDeviceFormatProperties(vk.physical_device, VK_FORMAT_D24_UNORM_S8_UINT, &props);
	if (props.optimalTilingFeatures & VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT) {
		ri.Printf(PRINT_ALL, " VK_FORMAT_D24_UNORM_S8_UINT optimal Tiling feature supported.\n");
		vk.fmt_DepthStencil = VK_FORMAT_D24_UNORM_S8_UINT;
	} else {
		qvkGetPhysicalDeviceFormatProperties(vk.physical_device, VK_FORMAT_D32_SFLOAT_S8_UINT, &props);

		if (props.optimalTilingFeatures & VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT) {
			ri.Printf(PRINT_ALL, " VK_FORMAT_D32_SFLOAT_S8_UINT optimal Tiling feature supported.\n");
			vk.fmt_DepthStencil = VK_FORMAT_D32_SFLOAT_S8_UINT;
		} else {
			// formats[0] = VK_FORMAT_X8_D24_UNORM_PACK32;
			// formats[1] = VK_FORMAT_D32_SFLOAT;
			// never get here.
			ri.Error(ERR_FATAL, " Failed to find depth attachment format.");
		}
	}

	ri.Printf(PRINT_ALL, " -------- --------------------------- --------\n");
}

static void vk_selectQueueFamilyForPresentation(void) {
	// Almosty every operation in Vulkan, anything from drawing textures,
	// requires commands to be submitted to a queue. There are different
	// types of queues that originate from differnet queue families and
	// each family of queues allows only a subset of commands.
	// For example, there could be a queue family allows processing of
	// compute commands or one that only allows memory transfer related
	// commands. We need to check which queue families are supported by
	// the device and which one of these supports the commands that we use.
	uint32_t i;
	uint32_t nSurfmt;
	VkQueueFamilyProperties *pQueueFamilies;

	qvkGetPhysicalDeviceQueueFamilyProperties(vk.physical_device, &nSurfmt, NULL);

	assert(nSurfmt > 0);

	pQueueFamilies = (VkQueueFamilyProperties *)ri.Hunk_AllocateTempMemory(nSurfmt * sizeof(VkQueueFamilyProperties));

	// To query properties of queues available on a physical device
	qvkGetPhysicalDeviceQueueFamilyProperties(vk.physical_device, &nSurfmt, pQueueFamilies);

	// Select queue family with presentation and graphics support
	// Iterate over each queue to learn whether it supports presenting:
	vk.queue_family_index = -1;

	for (i = 0; i < nSurfmt; ++i) {
		// To look for a queue family that has the capability of presenting
		// to our window surface

		VkBool32 presentation_supported = VK_FALSE;
		VK_CHECK(qvkGetPhysicalDeviceSurfaceSupportKHR(vk.physical_device, i, vk.surface, &presentation_supported));

		if (presentation_supported && (pQueueFamilies[i].queueFlags & VK_QUEUE_GRAPHICS_BIT) != 0) {
			vk.queue_family_index = i;

			ri.Printf(PRINT_ALL, " Queue family for presentation selected: %d\n", vk.queue_family_index);

			break;
		}
	}

	ri.Hunk_FreeTempMemory(pQueueFamilies);

	if (vk.queue_family_index == -1)
		ri.Error(ERR_FATAL, "Vulkan: failed to find queue family");
}

static void vk_createLogicalDevice(void) {
	static const char *device_extensions[1] = {VK_KHR_SWAPCHAIN_EXTENSION_NAME};

	//  Not all graphics cards are capble of presenting images directly
	//  to a screen for various reasons, for example because they are
	//  designed for servers and don't have any display outputs.
	//  Secondly, since image presentation is heavily tied into the
	//  window system and the surfaces associated with windows, it is
	//  not actually part of the vulkan core. You have to enable the
	//  VK_KHR_swapchain device extension after querying for its support.
	uint32_t nDevExts = 0;
	VkBool32 swapchainExtFound = 0;
	VkExtensionProperties *pDeviceExt;
	const float priority = 1.0f;
	VkDeviceQueueCreateInfo queue_desc;
	VkPhysicalDeviceFeatures features;
	VkDeviceCreateInfo device_desc;
	uint32_t j;

	// To query the numbers of extensions available to a given physical device
	ri.Printf(PRINT_ALL, " Check for VK_KHR_swapchain extension.\n");

	qvkEnumerateDeviceExtensionProperties(vk.physical_device, NULL, &nDevExts, NULL);

	pDeviceExt = (VkExtensionProperties *)ri.Hunk_AllocateTempMemory(sizeof(VkExtensionProperties) * nDevExts);

	qvkEnumerateDeviceExtensionProperties(vk.physical_device, NULL, &nDevExts, pDeviceExt);

	for (j = 0; j < nDevExts; j++) {
		if (!strcmp(device_extensions[0], pDeviceExt[j].extensionName)) {
			swapchainExtFound = VK_TRUE;
			break;
		}
	}
	if (VK_FALSE == swapchainExtFound)
		ri.Error(ERR_FATAL, "VK_KHR_SWAPCHAIN_EXTENSION_NAME is not available");

	ri.Hunk_FreeTempMemory(pDeviceExt);

	queue_desc.sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
	queue_desc.pNext = NULL;
	queue_desc.flags = 0;
	queue_desc.queueFamilyIndex = vk.queue_family_index;
	queue_desc.queueCount = 1;
	queue_desc.pQueuePriorities = &priority;

	// Query fine-grained feature support for this device. If APP
	// has specific feature requirements it should check supported
	// features based on this query.

	qvkGetPhysicalDeviceFeatures(vk.physical_device, &features);
	if (features.shaderClipDistance == VK_FALSE)
		ri.Error(ERR_FATAL, "vk_create_device: shaderClipDistance feature is not supported");
	if (features.fillModeNonSolid == VK_FALSE)
		ri.Error(ERR_FATAL, "vk_create_device: fillModeNonSolid feature is not supported");

	device_desc.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;
	device_desc.pNext = NULL;
	device_desc.flags = 0;
	device_desc.queueCreateInfoCount = 1;
	device_desc.pQueueCreateInfos = &queue_desc;
	device_desc.enabledLayerCount = 0;
	device_desc.ppEnabledLayerNames = NULL;
	device_desc.enabledExtensionCount = 1;
	device_desc.ppEnabledExtensionNames = device_extensions;
	device_desc.pEnabledFeatures = &features;

	// After selecting a physical device to use we need to set up a
	// logical device to interface with it. The logical device
	// creation process id similar to the instance creation process
	// and describes the features we want to use. we also need to
	// specify which queues to create now that we've queried which
	// queue families are available. You can create multiple logical
	// devices from the same physical device if you have varying requirements.
	ri.Printf(PRINT_ALL, " Create logical device: vk.device\n");
	VK_CHECK(qvkCreateDevice(vk.physical_device, &device_desc, NULL, &vk.device));
}

static void vk_loadDeviceFunctions(void) {
	ri.Printf(PRINT_ALL, " Loading device level function.\n");

#define INIT_DEVICE_FUNCTION(func)                                                                                     \
	q##func = (PFN_##func)qvkGetDeviceProcAddr(vk.device, #func);                                                      \
	if (q##func == NULL) {                                                                                             \
		ri.Error(ERR_FATAL, "Failed to find entrypoint %s", #func);                                                    \
	}

	INIT_DEVICE_FUNCTION(vkAllocateCommandBuffers)
	INIT_DEVICE_FUNCTION(vkAllocateDescriptorSets)
	INIT_DEVICE_FUNCTION(vkAllocateMemory)
	INIT_DEVICE_FUNCTION(vkBeginCommandBuffer)
	INIT_DEVICE_FUNCTION(vkBindBufferMemory)
	INIT_DEVICE_FUNCTION(vkBindImageMemory)
	INIT_DEVICE_FUNCTION(vkCmdBeginRenderPass)
	INIT_DEVICE_FUNCTION(vkCmdBindDescriptorSets)
	INIT_DEVICE_FUNCTION(vkCmdBindIndexBuffer)
	INIT_DEVICE_FUNCTION(vkCmdBindPipeline)
	INIT_DEVICE_FUNCTION(vkCmdBindVertexBuffers)
	INIT_DEVICE_FUNCTION(vkCmdBlitImage)
	INIT_DEVICE_FUNCTION(vkCmdClearAttachments)
	INIT_DEVICE_FUNCTION(vkCmdCopyBufferToImage)
	INIT_DEVICE_FUNCTION(vkCmdCopyImage)
	INIT_DEVICE_FUNCTION(vkCmdCopyImageToBuffer)
	INIT_DEVICE_FUNCTION(vkCmdDraw)
	INIT_DEVICE_FUNCTION(vkCmdDrawIndexed)
	INIT_DEVICE_FUNCTION(vkCmdEndRenderPass)
	INIT_DEVICE_FUNCTION(vkCmdPipelineBarrier)
	INIT_DEVICE_FUNCTION(vkCmdPushConstants)
	INIT_DEVICE_FUNCTION(vkCmdSetDepthBias)
	INIT_DEVICE_FUNCTION(vkCmdSetScissor)
	INIT_DEVICE_FUNCTION(vkCmdSetViewport)
	INIT_DEVICE_FUNCTION(vkCreateBuffer)
	INIT_DEVICE_FUNCTION(vkCreateCommandPool)
	INIT_DEVICE_FUNCTION(vkCreateDescriptorPool)
	INIT_DEVICE_FUNCTION(vkCreateDescriptorSetLayout)
	INIT_DEVICE_FUNCTION(vkCreateFence)
	INIT_DEVICE_FUNCTION(vkCreateFramebuffer)
	INIT_DEVICE_FUNCTION(vkCreateGraphicsPipelines)
	INIT_DEVICE_FUNCTION(vkCreateImage)
	INIT_DEVICE_FUNCTION(vkCreateImageView)
	INIT_DEVICE_FUNCTION(vkCreatePipelineLayout)
	INIT_DEVICE_FUNCTION(vkCreateRenderPass)
	INIT_DEVICE_FUNCTION(vkCreateSampler)
	INIT_DEVICE_FUNCTION(vkCreateSemaphore)
	INIT_DEVICE_FUNCTION(vkCreateShaderModule)
	INIT_DEVICE_FUNCTION(vkDestroyBuffer)
	INIT_DEVICE_FUNCTION(vkDestroyCommandPool)
	INIT_DEVICE_FUNCTION(vkDestroyDescriptorPool)
	INIT_DEVICE_FUNCTION(vkDestroyDescriptorSetLayout)
	INIT_DEVICE_FUNCTION(vkDestroyDevice)
	INIT_DEVICE_FUNCTION(vkDestroyFence)
	INIT_DEVICE_FUNCTION(vkDestroyFramebuffer)
	INIT_DEVICE_FUNCTION(vkDestroyImage)
	INIT_DEVICE_FUNCTION(vkDestroyImageView)
	INIT_DEVICE_FUNCTION(vkDestroyPipeline)
	INIT_DEVICE_FUNCTION(vkDestroyPipelineLayout)
	INIT_DEVICE_FUNCTION(vkDestroyRenderPass)
	INIT_DEVICE_FUNCTION(vkDestroySampler)
	INIT_DEVICE_FUNCTION(vkDestroySemaphore)
	INIT_DEVICE_FUNCTION(vkDestroyShaderModule)
	INIT_DEVICE_FUNCTION(vkDeviceWaitIdle)
	INIT_DEVICE_FUNCTION(vkEndCommandBuffer)
	INIT_DEVICE_FUNCTION(vkFreeCommandBuffers)
	INIT_DEVICE_FUNCTION(vkFreeDescriptorSets)
	INIT_DEVICE_FUNCTION(vkFreeMemory)
	INIT_DEVICE_FUNCTION(vkGetBufferMemoryRequirements)
	INIT_DEVICE_FUNCTION(vkGetDeviceQueue)
	INIT_DEVICE_FUNCTION(vkGetImageMemoryRequirements)
	INIT_DEVICE_FUNCTION(vkGetImageSubresourceLayout)
	INIT_DEVICE_FUNCTION(vkMapMemory)
	INIT_DEVICE_FUNCTION(vkUnmapMemory)
	INIT_DEVICE_FUNCTION(vkQueueSubmit)
	INIT_DEVICE_FUNCTION(vkQueueWaitIdle)
	INIT_DEVICE_FUNCTION(vkResetDescriptorPool)
	INIT_DEVICE_FUNCTION(vkResetFences)
	INIT_DEVICE_FUNCTION(vkUpdateDescriptorSets)
	INIT_DEVICE_FUNCTION(vkWaitForFences)

	INIT_DEVICE_FUNCTION(vkCreateSwapchainKHR)
	INIT_DEVICE_FUNCTION(vkDestroySwapchainKHR)
	INIT_DEVICE_FUNCTION(vkGetSwapchainImagesKHR)
	INIT_DEVICE_FUNCTION(vkAcquireNextImageKHR)
	INIT_DEVICE_FUNCTION(vkQueuePresentKHR)

#undef INIT_DEVICE_FUNCTION
}

void vk_getProcAddress(void) {
	vk_loadGlobalFunctions();

#ifndef NDEBUG
	// Create debug callback.
	vk_createDebugCallback(vk_DebugCallback);
#endif

	// The window surface needs to be created right after the instance creation,
	// because it can actually influence the presentation mode selection.
	vk_createSurfaceImpl();

	// select physical device
	vk_selectPhysicalDevice();

	vk_selectSurfaceFormat();

	vk_selectQueueFamilyForPresentation();

	vk_createLogicalDevice();

	// Get device level functions.
	vk_loadDeviceFunctions();

	// a call to retrieve queue handle
	qvkGetDeviceQueue(vk.device, vk.queue_family_index, 0, &vk.queue);
}

void vk_clearProcAddress(void) {
	ri.Printf(PRINT_ALL, " Destroy logical device: vk.device.\n");
	// Device queues are implicitly cleaned up when the device is destroyed
	// so we don't need to do anything in clean up
	qvkDestroyDevice(vk.device, NULL);

	ri.Printf(PRINT_ALL, " Destroy surface: vk.surface.\n");
	// make sure that the surface is destroyed before the instance
	qvkDestroySurfaceKHR(vk.instance, vk.surface, NULL);

#ifndef NDEBUG
	ri.Printf(PRINT_ALL, " Destroy callback function: vk.h_debugCB.\n");

	qvkDestroyDebugReportCallbackEXT(vk.instance, vk.h_debugCB, NULL);
#endif

	ri.Printf(PRINT_ALL, " Destroy instance: vk.instance.\n");
	qvkDestroyInstance(vk.instance, NULL);

	// ===========================================================
	ri.Printf(PRINT_ALL, " clear all proc address\n");

	qvkCreateInstance = NULL;
	qvkEnumerateInstanceExtensionProperties = NULL;

	qvkCreateDevice = NULL;
	qvkDestroyInstance = NULL;
	qvkEnumerateDeviceExtensionProperties = NULL;
	qvkEnumeratePhysicalDevices = NULL;
	qvkGetDeviceProcAddr = NULL;
	qvkGetPhysicalDeviceFeatures = NULL;
	qvkGetPhysicalDeviceFormatProperties = NULL;
	qvkGetPhysicalDeviceMemoryProperties = NULL;
	qvkGetPhysicalDeviceProperties = NULL;
	qvkGetPhysicalDeviceQueueFamilyProperties = NULL;

	qvkDestroySurfaceKHR = NULL;
	qvkGetPhysicalDeviceSurfaceCapabilitiesKHR = NULL;
	qvkGetPhysicalDeviceSurfaceFormatsKHR = NULL;
	qvkGetPhysicalDeviceSurfacePresentModesKHR = NULL;
	qvkGetPhysicalDeviceSurfaceSupportKHR = NULL;
#ifndef NDEBUG
	qvkCreateDebugReportCallbackEXT = NULL;
	qvkDestroyDebugReportCallbackEXT = NULL;
#endif

	qvkAllocateCommandBuffers = NULL;
	qvkAllocateDescriptorSets = NULL;
	qvkAllocateMemory = NULL;
	qvkBeginCommandBuffer = NULL;
	qvkBindBufferMemory = NULL;
	qvkBindImageMemory = NULL;
	qvkCmdBeginRenderPass = NULL;
	qvkCmdBindDescriptorSets = NULL;
	qvkCmdBindIndexBuffer = NULL;
	qvkCmdBindPipeline = NULL;
	qvkCmdBindVertexBuffers = NULL;
	qvkCmdBlitImage = NULL;
	qvkCmdClearAttachments = NULL;
	qvkCmdCopyBufferToImage = NULL;
	qvkCmdCopyImage = NULL;
	qvkCmdCopyImageToBuffer = NULL;
	qvkCmdDraw = NULL;
	qvkCmdDrawIndexed = NULL;
	qvkCmdEndRenderPass = NULL;
	qvkCmdPipelineBarrier = NULL;
	qvkCmdPushConstants = NULL;
	qvkCmdSetDepthBias = NULL;
	qvkCmdSetScissor = NULL;
	qvkCmdSetViewport = NULL;
	qvkCreateBuffer = NULL;
	qvkCreateCommandPool = NULL;
	qvkCreateDescriptorPool = NULL;
	qvkCreateDescriptorSetLayout = NULL;
	qvkCreateFence = NULL;
	qvkCreateFramebuffer = NULL;
	qvkCreateGraphicsPipelines = NULL;
	qvkCreateImage = NULL;
	qvkCreateImageView = NULL;
	qvkCreatePipelineLayout = NULL;
	qvkCreateRenderPass = NULL;
	qvkCreateSampler = NULL;
	qvkCreateSemaphore = NULL;
	qvkCreateShaderModule = NULL;
	qvkDestroyBuffer = NULL;
	qvkDestroyCommandPool = NULL;
	qvkDestroyDescriptorPool = NULL;
	qvkDestroyDescriptorSetLayout = NULL;
	qvkDestroyDevice = NULL;
	qvkDestroyFence = NULL;
	qvkDestroyFramebuffer = NULL;
	qvkDestroyImage = NULL;
	qvkDestroyImageView = NULL;
	qvkDestroyPipeline = NULL;
	qvkDestroyPipelineLayout = NULL;
	qvkDestroyRenderPass = NULL;
	qvkDestroySampler = NULL;
	qvkDestroySemaphore = NULL;
	qvkDestroyShaderModule = NULL;
	qvkDeviceWaitIdle = NULL;
	qvkEndCommandBuffer = NULL;
	qvkFreeCommandBuffers = NULL;
	qvkFreeDescriptorSets = NULL;
	qvkFreeMemory = NULL;
	qvkGetBufferMemoryRequirements = NULL;
	qvkGetDeviceQueue = NULL;
	qvkGetImageMemoryRequirements = NULL;
	qvkGetImageSubresourceLayout = NULL;
	qvkMapMemory = NULL;
	qvkUnmapMemory = NULL;
	qvkQueueSubmit = NULL;
	qvkQueueWaitIdle = NULL;
	qvkResetDescriptorPool = NULL;
	qvkResetFences = NULL;
	qvkUpdateDescriptorSets = NULL;
	qvkWaitForFences = NULL;
	qvkAcquireNextImageKHR = NULL;
	qvkCreateSwapchainKHR = NULL;
	qvkDestroySwapchainKHR = NULL;
	qvkGetSwapchainImagesKHR = NULL;
	qvkQueuePresentKHR = NULL;
}

const char *cvtResToStr(VkResult result) {
	switch (result) {
	case VK_SUCCESS:
		return "VK_SUCCESS";
	case VK_NOT_READY:
		return "VK_NOT_READY";
	case VK_TIMEOUT:
		return "VK_TIMEOUT";
	case VK_EVENT_SET:
		return "VK_EVENT_SET";
	case VK_EVENT_RESET:
		return "VK_EVENT_RESET";
	case VK_INCOMPLETE:
		return "VK_INCOMPLETE";
	case VK_ERROR_OUT_OF_HOST_MEMORY:
		return "VK_ERROR_OUT_OF_HOST_MEMORY";
	case VK_ERROR_OUT_OF_DEVICE_MEMORY:
		return "VK_ERROR_OUT_OF_DEVICE_MEMORY";
	case VK_ERROR_INITIALIZATION_FAILED:
		return "VK_ERROR_INITIALIZATION_FAILED";
	case VK_ERROR_DEVICE_LOST:
		return "VK_ERROR_DEVICE_LOST";
	case VK_ERROR_MEMORY_MAP_FAILED:
		return "VK_ERROR_MEMORY_MAP_FAILED";
	case VK_ERROR_LAYER_NOT_PRESENT:
		return "VK_ERROR_LAYER_NOT_PRESENT";
	case VK_ERROR_EXTENSION_NOT_PRESENT:
		return "VK_ERROR_EXTENSION_NOT_PRESENT";
	case VK_ERROR_FEATURE_NOT_PRESENT:
		return "VK_ERROR_FEATURE_NOT_PRESENT";
	case VK_ERROR_INCOMPATIBLE_DRIVER:
		return "VK_ERROR_INCOMPATIBLE_DRIVER";
	case VK_ERROR_TOO_MANY_OBJECTS:
		return "VK_ERROR_TOO_MANY_OBJECTS";
	case VK_ERROR_FORMAT_NOT_SUPPORTED:
		return "VK_ERROR_FORMAT_NOT_SUPPORTED";
	case VK_ERROR_FRAGMENTED_POOL:
		return "VK_ERROR_FRAGMENTED_POOL";
	case VK_ERROR_SURFACE_LOST_KHR:
		return "VK_ERROR_SURFACE_LOST_KHR";
	case VK_ERROR_NATIVE_WINDOW_IN_USE_KHR:
		return "VK_ERROR_NATIVE_WINDOW_IN_USE_KHR";
	case VK_SUBOPTIMAL_KHR:
		return "VK_SUBOPTIMAL_KHR";
	case VK_ERROR_OUT_OF_DATE_KHR:
		return "VK_ERROR_OUT_OF_DATE_KHR";
	case VK_ERROR_INCOMPATIBLE_DISPLAY_KHR:
		return "VK_ERROR_INCOMPATIBLE_DISPLAY_KHR";
	case VK_ERROR_VALIDATION_FAILED_EXT:
		return "VK_ERROR_VALIDATION_FAILED_EXT";
	case VK_ERROR_OUT_OF_POOL_MEMORY_KHR:
		return "VK_ERROR_OUT_OF_POOL_MEMORY_KHR";
	case VK_ERROR_INVALID_SHADER_NV:
		return "VK_ERROR_INVALID_SHADER_NV";
		//
	case VK_ERROR_INVALID_EXTERNAL_HANDLE:
		return "VK_ERROR_INVALID_EXTERNAL_HANDLE";
	case VK_ERROR_NOT_PERMITTED_EXT:
		return "VK_ERROR_NOT_PERMITTED_EXT";
		//
	case VK_RESULT_MAX_ENUM:
		return "VK_RESULT_MAX_ENUM";
	case VK_RESULT_RANGE_SIZE:
		return "VK_RESULT_RANGE_SIZE";
	case VK_ERROR_FRAGMENTATION_EXT:
		return "VK_ERROR_FRAGMENTATION_EXT";
	}

	return "UNKNOWN_ERROR";
}
