/** @file
 * IPRT - Symbol Mangling.
 *
 * This header is used to mangle public IPRT symbol to make it possible to have
 * several IPRT version loaded into one symbol space at the same time.  To
 * enable symbol mangling you create a header which the compiler includes for
 * every compilation unit (check out the -include option of gcc).  Your header
 * will define RT_MANGLER(name) and then include this header to set up the
 * actual mappings.
 */

/*
 * Copyright (C) 2011-2024 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * The contents of this file may alternatively be used under the terms
 * of the Common Development and Distribution License Version 1.0
 * (CDDL), a copy of it is provided in the "COPYING.CDDL" file included
 * in the VirtualBox distribution, in which case the provisions of the
 * CDDL are applicable instead of those of the GPL.
 *
 * You may elect to license modified versions of this file under the
 * terms and conditions of either the GPL or the CDDL or both.
 *
 * SPDX-License-Identifier: GPL-3.0-only OR CDDL-1.0
 */

#ifndef IPRT_INCLUDED_mangling_h
#define IPRT_INCLUDED_mangling_h
#ifndef RT_WITHOUT_PRAGMA_ONCE
# pragma once
#endif

#ifndef RT_MANGLER
# error "RT_MANGLER is not defined."
#endif

#ifndef DOXYGEN_RUNNING

/** @def RT_WITH_MANGLING
 * Indicates that we're mangling symbols.  */
# define RT_WITH_MANGLING


/*
 * Stable functions (alphabetical order):
 */
/* ASM*:
 grep -h DECLASM include/iprt/asm.h include/iprt/asm-amd64-x86.h \
  | kmk_sed -e 's/^DECLASM.[^)]*. *\(ASM[^(]*\)[(].*$/# define \1 :RT_MANGLER(\1)\n# define \1_EndProc :RT_MANGLER(\1_EndProc)/' \
  | sort \
  | awk -F: '{ printf("%-55s %s\n", $1, $2);' */
# define ASMAddFlags                                    RT_MANGLER(ASMAddFlags)
# define ASMAddFlags_EndProc                            RT_MANGLER(ASMAddFlags_EndProc)
# define ASMAtomicAddU16                                RT_MANGLER(ASMAtomicAddU16)
# define ASMAtomicAddU16_EndProc                        RT_MANGLER(ASMAtomicAddU16_EndProc)
# define ASMAtomicAddU32                                RT_MANGLER(ASMAtomicAddU32)
# define ASMAtomicAddU32_EndProc                        RT_MANGLER(ASMAtomicAddU32_EndProc)
# define ASMAtomicAddU64                                RT_MANGLER(ASMAtomicAddU64)
# define ASMAtomicAddU64_EndProc                        RT_MANGLER(ASMAtomicAddU64_EndProc)
# define ASMAtomicAndU32                                RT_MANGLER(ASMAtomicAndU32)
# define ASMAtomicAndU32_EndProc                        RT_MANGLER(ASMAtomicAndU32_EndProc)
# define ASMAtomicAndU64                                RT_MANGLER(ASMAtomicAndU64)
# define ASMAtomicAndU64_EndProc                        RT_MANGLER(ASMAtomicAndU64_EndProc)
# define ASMAtomicBitClear                              RT_MANGLER(ASMAtomicBitClear)
# define ASMAtomicBitClear_EndProc                      RT_MANGLER(ASMAtomicBitClear_EndProc)
# define ASMAtomicBitSet                                RT_MANGLER(ASMAtomicBitSet)
# define ASMAtomicBitSet_EndProc                        RT_MANGLER(ASMAtomicBitSet_EndProc)
# define ASMAtomicBitTestAndClear                       RT_MANGLER(ASMAtomicBitTestAndClear)
# define ASMAtomicBitTestAndClear_EndProc               RT_MANGLER(ASMAtomicBitTestAndClear_EndProc)
# define ASMAtomicBitTestAndSet                         RT_MANGLER(ASMAtomicBitTestAndSet)
# define ASMAtomicBitTestAndSet_EndProc                 RT_MANGLER(ASMAtomicBitTestAndSet_EndProc)
# define ASMAtomicBitTestAndToggle                      RT_MANGLER(ASMAtomicBitTestAndToggle)
# define ASMAtomicBitTestAndToggle_EndProc              RT_MANGLER(ASMAtomicBitTestAndToggle_EndProc)
# define ASMAtomicBitToggle                             RT_MANGLER(ASMAtomicBitToggle)
# define ASMAtomicBitToggle_EndProc                     RT_MANGLER(ASMAtomicBitToggle_EndProc)
# define ASMAtomicCmpXchgExU32                          RT_MANGLER(ASMAtomicCmpXchgExU32)
# define ASMAtomicCmpXchgExU32_EndProc                  RT_MANGLER(ASMAtomicCmpXchgExU32_EndProc)
# define ASMAtomicCmpXchgExU64                          RT_MANGLER(ASMAtomicCmpXchgExU64)
# define ASMAtomicCmpXchgExU64_EndProc                  RT_MANGLER(ASMAtomicCmpXchgExU64_EndProc)
# define ASMAtomicCmpXchgU32                            RT_MANGLER(ASMAtomicCmpXchgU32)
# define ASMAtomicCmpXchgU32_EndProc                    RT_MANGLER(ASMAtomicCmpXchgU32_EndProc)
# define ASMAtomicCmpXchgU64                            RT_MANGLER(ASMAtomicCmpXchgU64)
# define ASMAtomicCmpXchgU64_EndProc                    RT_MANGLER(ASMAtomicCmpXchgU64_EndProc)
# define ASMAtomicCmpXchgU8                             RT_MANGLER(ASMAtomicCmpXchgU8)
# define ASMAtomicCmpXchgU8_EndProc                     RT_MANGLER(ASMAtomicCmpXchgU8_EndProc)
# define ASMAtomicDecU16                                RT_MANGLER(ASMAtomicDecU16)
# define ASMAtomicDecU16_EndProc                        RT_MANGLER(ASMAtomicDecU16_EndProc)
# define ASMAtomicDecU32                                RT_MANGLER(ASMAtomicDecU32)
# define ASMAtomicDecU32_EndProc                        RT_MANGLER(ASMAtomicDecU32_EndProc)
# define ASMAtomicDecU64                                RT_MANGLER(ASMAtomicDecU64)
# define ASMAtomicDecU64_EndProc                        RT_MANGLER(ASMAtomicDecU64_EndProc)
# define ASMAtomicIncU16                                RT_MANGLER(ASMAtomicIncU16)
# define ASMAtomicIncU16_EndProc                        RT_MANGLER(ASMAtomicIncU16_EndProc)
# define ASMAtomicIncU32                                RT_MANGLER(ASMAtomicIncU32)
# define ASMAtomicIncU32_EndProc                        RT_MANGLER(ASMAtomicIncU32_EndProc)
# define ASMAtomicIncU64                                RT_MANGLER(ASMAtomicIncU64)
# define ASMAtomicIncU64_EndProc                        RT_MANGLER(ASMAtomicIncU64_EndProc)
# define ASMAtomicOrU32                                 RT_MANGLER(ASMAtomicOrU32)
# define ASMAtomicOrU32_EndProc                         RT_MANGLER(ASMAtomicOrU32_EndProc)
# define ASMAtomicOrU64                                 RT_MANGLER(ASMAtomicOrU64)
# define ASMAtomicOrU64_EndProc                         RT_MANGLER(ASMAtomicOrU64_EndProc)
# define ASMAtomicReadU64                               RT_MANGLER(ASMAtomicReadU64)
# define ASMAtomicReadU64_EndProc                       RT_MANGLER(ASMAtomicReadU64_EndProc)
# define ASMAtomicUoAndU32                              RT_MANGLER(ASMAtomicUoAndU32)
# define ASMAtomicUoAndU32_EndProc                      RT_MANGLER(ASMAtomicUoAndU32_EndProc)
# define ASMAtomicUoAndU64                              RT_MANGLER(ASMAtomicUoAndU64)
# define ASMAtomicUoAndU64_EndProc                      RT_MANGLER(ASMAtomicUoAndU64_EndProc)
# define ASMAtomicUoDecU32                              RT_MANGLER(ASMAtomicUoDecU32)
# define ASMAtomicUoDecU32_EndProc                      RT_MANGLER(ASMAtomicUoDecU32_EndProc)
# define ASMAtomicUoIncU32                              RT_MANGLER(ASMAtomicUoIncU32)
# define ASMAtomicUoIncU32_EndProc                      RT_MANGLER(ASMAtomicUoIncU32_EndProc)
# define ASMAtomicUoOrU32                               RT_MANGLER(ASMAtomicUoOrU32)
# define ASMAtomicUoOrU32_EndProc                       RT_MANGLER(ASMAtomicUoOrU32_EndProc)
# define ASMAtomicUoOrU64                               RT_MANGLER(ASMAtomicUoOrU64)
# define ASMAtomicUoOrU64_EndProc                       RT_MANGLER(ASMAtomicUoOrU64_EndProc)
# define ASMAtomicUoReadU64                             RT_MANGLER(ASMAtomicUoReadU64)
# define ASMAtomicUoReadU64_EndProc                     RT_MANGLER(ASMAtomicUoReadU64_EndProc)
# define ASMAtomicUoXorU32                              RT_MANGLER(ASMAtomicUoXorU32)
# define ASMAtomicXchgU16                               RT_MANGLER(ASMAtomicXchgU16)
# define ASMAtomicXchgU16_EndProc                       RT_MANGLER(ASMAtomicXchgU16_EndProc)
# define ASMAtomicXchgU32                               RT_MANGLER(ASMAtomicXchgU32)
# define ASMAtomicXchgU32_EndProc                       RT_MANGLER(ASMAtomicXchgU32_EndProc)
# define ASMAtomicXchgU64                               RT_MANGLER(ASMAtomicXchgU64)
# define ASMAtomicXchgU64_EndProc                       RT_MANGLER(ASMAtomicXchgU64_EndProc)
# define ASMAtomicXchgU8                                RT_MANGLER(ASMAtomicXchgU8)
# define ASMAtomicXchgU8_EndProc                        RT_MANGLER(ASMAtomicXchgU8_EndProc)
# define ASMBitClear                                    RT_MANGLER(ASMBitClear)
# define ASMBitClear_EndProc                            RT_MANGLER(ASMBitClear_EndProc)
# define ASMBitFirstClear                               RT_MANGLER(ASMBitFirstClear)
# define ASMBitFirstClear_EndProc                       RT_MANGLER(ASMBitFirstClear_EndProc)
# define ASMBitFirstSet                                 RT_MANGLER(ASMBitFirstSet)
# define ASMBitFirstSet_EndProc                         RT_MANGLER(ASMBitFirstSet_EndProc)
# define ASMBitFirstSetU16                              RT_MANGLER(ASMBitFirstSetU16)
# define ASMBitFirstSetU16_EndProc                      RT_MANGLER(ASMBitFirstSetU16_EndProc)
# define ASMBitFirstSetU32                              RT_MANGLER(ASMBitFirstSetU32)
# define ASMBitFirstSetU32_EndProc                      RT_MANGLER(ASMBitFirstSetU32_EndProc)
# define ASMBitFirstSetU64                              RT_MANGLER(ASMBitFirstSetU64)
# define ASMBitFirstSetU64_EndProc                      RT_MANGLER(ASMBitFirstSetU64_EndProc)
# define ASMBitLastSetU16                               RT_MANGLER(ASMBitLastSetU16)
# define ASMBitLastSetU16_EndProc                       RT_MANGLER(ASMBitLastSetU16_EndProc)
# define ASMBitLastSetU32                               RT_MANGLER(ASMBitLastSetU32)
# define ASMBitLastSetU32_EndProc                       RT_MANGLER(ASMBitLastSetU32_EndProc)
# define ASMBitLastSetU64                               RT_MANGLER(ASMBitLastSetU64)
# define ASMBitLastSetU64_EndProc                       RT_MANGLER(ASMBitLastSetU64_EndProc)
# define ASMBitNextClear                                RT_MANGLER(ASMBitNextClear)
# define ASMBitNextClear_EndProc                        RT_MANGLER(ASMBitNextClear_EndProc)
# define ASMBitNextSet                                  RT_MANGLER(ASMBitNextSet)
# define ASMBitNextSet_EndProc                          RT_MANGLER(ASMBitNextSet_EndProc)
# define ASMBitSet                                      RT_MANGLER(ASMBitSet)
# define ASMBitSet_EndProc                              RT_MANGLER(ASMBitSet_EndProc)
# define ASMBitTest                                     RT_MANGLER(ASMBitTest)
# define ASMBitTest_EndProc                             RT_MANGLER(ASMBitTest_EndProc)
# define ASMBitTestAndClear                             RT_MANGLER(ASMBitTestAndClear)
# define ASMBitTestAndClear_EndProc                     RT_MANGLER(ASMBitTestAndClear_EndProc)
# define ASMBitTestAndSet                               RT_MANGLER(ASMBitTestAndSet)
# define ASMBitTestAndSet_EndProc                       RT_MANGLER(ASMBitTestAndSet_EndProc)
# define ASMBitTestAndToggle                            RT_MANGLER(ASMBitTestAndToggle)
# define ASMBitTestAndToggle_EndProc                    RT_MANGLER(ASMBitTestAndToggle_EndProc)
# define ASMBitToggle                                   RT_MANGLER(ASMBitToggle)
# define ASMBitToggle_EndProc                           RT_MANGLER(ASMBitToggle_EndProc)
# define ASMByteSwapU16                                 RT_MANGLER(ASMByteSwapU16)
# define ASMByteSwapU16_EndProc                         RT_MANGLER(ASMByteSwapU16_EndProc)
# define ASMByteSwapU32                                 RT_MANGLER(ASMByteSwapU32)
# define ASMByteSwapU32_EndProc                         RT_MANGLER(ASMByteSwapU32_EndProc)
# define ASMChangeFlags                                 RT_MANGLER(ASMChangeFlags)
# define ASMChangeFlags_EndProc                         RT_MANGLER(ASMChangeFlags_EndProc)
# define ASMClearFlags                                  RT_MANGLER(ASMClearFlags)
# define ASMClearFlags_EndProc                          RT_MANGLER(ASMClearFlags_EndProc)
# define ASMCpuId                                       RT_MANGLER(ASMCpuId)
# define ASMCpuId_EAX                                   RT_MANGLER(ASMCpuId_EAX)
# define ASMCpuId_EAX_EndProc                           RT_MANGLER(ASMCpuId_EAX_EndProc)
# define ASMCpuId_EBX                                   RT_MANGLER(ASMCpuId_EBX)
# define ASMCpuId_EBX_EndProc                           RT_MANGLER(ASMCpuId_EBX_EndProc)
# define ASMCpuId_ECX                                   RT_MANGLER(ASMCpuId_ECX)
# define ASMCpuId_ECX_EDX                               RT_MANGLER(ASMCpuId_ECX_EDX)
# define ASMCpuId_ECX_EDX_EndProc                       RT_MANGLER(ASMCpuId_ECX_EDX_EndProc)
# define ASMCpuId_ECX_EndProc                           RT_MANGLER(ASMCpuId_ECX_EndProc)
# define ASMCpuId_EDX                                   RT_MANGLER(ASMCpuId_EDX)
# define ASMCpuId_EDX_EndProc                           RT_MANGLER(ASMCpuId_EDX_EndProc)
# define ASMCpuId_EndProc                               RT_MANGLER(ASMCpuId_EndProc)
# define ASMCpuId_Idx_ECX                               RT_MANGLER(ASMCpuId_Idx_ECX)
# define ASMCpuId_Idx_ECX_EndProc                       RT_MANGLER(ASMCpuId_Idx_ECX_EndProc)
# define ASMCpuIdExSlow                                 RT_MANGLER(ASMCpuIdExSlow)
# define ASMCpuIdExSlow_EndProc                         RT_MANGLER(ASMCpuIdExSlow_EndProc)
# define ASMGetAndClearDR6                              RT_MANGLER(ASMGetAndClearDR6)
# define ASMGetAndClearDR6_EndProc                      RT_MANGLER(ASMGetAndClearDR6_EndProc)
# define ASMGetApicId                                   RT_MANGLER(ASMGetApicId)
# define ASMGetApicId_EndProc                           RT_MANGLER(ASMGetApicId_EndProc)
# define ASMGetCR0                                      RT_MANGLER(ASMGetCR0)
# define ASMGetCR0_EndProc                              RT_MANGLER(ASMGetCR0_EndProc)
# define ASMGetCR2                                      RT_MANGLER(ASMGetCR2)
# define ASMGetCR2_EndProc                              RT_MANGLER(ASMGetCR2_EndProc)
# define ASMGetCR3                                      RT_MANGLER(ASMGetCR3)
# define ASMGetCR3_EndProc                              RT_MANGLER(ASMGetCR3_EndProc)
# define ASMGetCR4                                      RT_MANGLER(ASMGetCR4)
# define ASMGetCR4_EndProc                              RT_MANGLER(ASMGetCR4_EndProc)
# define ASMGetCR8                                      RT_MANGLER(ASMGetCR8)
# define ASMGetCR8_EndProc                              RT_MANGLER(ASMGetCR8_EndProc)
# define ASMGetCS                                       RT_MANGLER(ASMGetCS)
# define ASMGetCS_EndProc                               RT_MANGLER(ASMGetCS_EndProc)
# define ASMGetDR0                                      RT_MANGLER(ASMGetDR0)
# define ASMGetDR0_EndProc                              RT_MANGLER(ASMGetDR0_EndProc)
# define ASMGetDR1                                      RT_MANGLER(ASMGetDR1)
# define ASMGetDR1_EndProc                              RT_MANGLER(ASMGetDR1_EndProc)
# define ASMGetDR2                                      RT_MANGLER(ASMGetDR2)
# define ASMGetDR2_EndProc                              RT_MANGLER(ASMGetDR2_EndProc)
# define ASMGetDR3                                      RT_MANGLER(ASMGetDR3)
# define ASMGetDR3_EndProc                              RT_MANGLER(ASMGetDR3_EndProc)
# define ASMGetDR6                                      RT_MANGLER(ASMGetDR6)
# define ASMGetDR6_EndProc                              RT_MANGLER(ASMGetDR6_EndProc)
# define ASMGetDR7                                      RT_MANGLER(ASMGetDR7)
# define ASMGetDR7_EndProc                              RT_MANGLER(ASMGetDR7_EndProc)
# define ASMGetDS                                       RT_MANGLER(ASMGetDS)
# define ASMGetDS_EndProc                               RT_MANGLER(ASMGetDS_EndProc)
# define ASMGetES                                       RT_MANGLER(ASMGetES)
# define ASMGetES_EndProc                               RT_MANGLER(ASMGetES_EndProc)
# define ASMGetFlags                                    RT_MANGLER(ASMGetFlags)
# define ASMGetFlags_EndProc                            RT_MANGLER(ASMGetFlags_EndProc)
# define ASMGetFS                                       RT_MANGLER(ASMGetFS)
# define ASMGetFS_EndProc                               RT_MANGLER(ASMGetFS_EndProc)
# define ASMGetGDTR                                     RT_MANGLER(ASMGetGDTR)
# define ASMGetGDTR_EndProc                             RT_MANGLER(ASMGetGDTR_EndProc)
# define ASMGetGS                                       RT_MANGLER(ASMGetGS)
# define ASMGetGS_EndProc                               RT_MANGLER(ASMGetGS_EndProc)
# define ASMGetIDTR                                     RT_MANGLER(ASMGetIDTR)
# define ASMGetIDTR_EndProc                             RT_MANGLER(ASMGetIDTR_EndProc)
# define ASMGetIdtrLimit                                RT_MANGLER(ASMGetIdtrLimit)
# define ASMGetIdtrLimit_EndProc                        RT_MANGLER(ASMGetIdtrLimit_EndProc)
# define ASMGetLDTR                                     RT_MANGLER(ASMGetLDTR)
# define ASMGetLDTR_EndProc                             RT_MANGLER(ASMGetLDTR_EndProc)
# define ASMGetSegAttr                                  RT_MANGLER(ASMGetSegAttr)
# define ASMGetSegAttr_EndProc                          RT_MANGLER(ASMGetSegAttr_EndProc)
# define ASMGetSS                                       RT_MANGLER(ASMGetSS)
# define ASMGetSS_EndProc                               RT_MANGLER(ASMGetSS_EndProc)
# define ASMGetTR                                       RT_MANGLER(ASMGetTR)
# define ASMGetTR_EndProc                               RT_MANGLER(ASMGetTR_EndProc)
# define ASMGetXcr0                                     RT_MANGLER(ASMGetXcr0)
# define ASMGetXcr0_EndProc                             RT_MANGLER(ASMGetXcr0_EndProc)
# define ASMHalt                                        RT_MANGLER(ASMHalt)
# define ASMHalt_EndProc                                RT_MANGLER(ASMHalt_EndProc)
# define ASMInStrU16                                    RT_MANGLER(ASMInStrU16)
# define ASMInStrU16_EndProc                            RT_MANGLER(ASMInStrU16_EndProc)
# define ASMInStrU32                                    RT_MANGLER(ASMInStrU32)
# define ASMInStrU32_EndProc                            RT_MANGLER(ASMInStrU32_EndProc)
# define ASMInStrU8                                     RT_MANGLER(ASMInStrU8)
# define ASMInStrU8_EndProc                             RT_MANGLER(ASMInStrU8_EndProc)
# define ASMIntDisable                                  RT_MANGLER(ASMIntDisable)
# define ASMIntDisable_EndProc                          RT_MANGLER(ASMIntDisable_EndProc)
# define ASMIntDisableFlags                             RT_MANGLER(ASMIntDisableFlags)
# define ASMIntDisableFlags_EndProc                     RT_MANGLER(ASMIntDisableFlags_EndProc)
# define ASMIntEnable                                   RT_MANGLER(ASMIntEnable)
# define ASMIntEnable_EndProc                           RT_MANGLER(ASMIntEnable_EndProc)
# define ASMInU16                                       RT_MANGLER(ASMInU16)
# define ASMInU16_EndProc                               RT_MANGLER(ASMInU16_EndProc)
# define ASMInU32                                       RT_MANGLER(ASMInU32)
# define ASMInU32_EndProc                               RT_MANGLER(ASMInU32_EndProc)
# define ASMInU8                                        RT_MANGLER(ASMInU8)
# define ASMInU8_EndProc                                RT_MANGLER(ASMInU8_EndProc)
# define ASMInvalidateInternalCaches                    RT_MANGLER(ASMInvalidateInternalCaches)
# define ASMInvalidateInternalCaches_EndProc            RT_MANGLER(ASMInvalidateInternalCaches_EndProc)
# define ASMInvalidatePage                              RT_MANGLER(ASMInvalidatePage)
# define ASMInvalidatePage_EndProc                      RT_MANGLER(ASMInvalidatePage_EndProc)
# define ASMMemFill32                                   RT_MANGLER(ASMMemFill32)
# define ASMMemFill32_EndProc                           RT_MANGLER(ASMMemFill32_EndProc)
# define ASMMemFirstNonZero                             RT_MANGLER(ASMMemFirstNonZero)
# define ASMMemFirstNonZero_EndProc                     RT_MANGLER(ASMMemFirstNonZero_EndProc)
# define ASMMemFirstMismatchingU8                       RT_MANGLER(ASMMemFirstMismatchingU8)
# define ASMMemFirstMismatchingU8_EndProc               RT_MANGLER(ASMMemFirstMismatchingU8_EndProc)
# define ASMMemFirstMismatchingU32                      RT_MANGLER(ASMMemFirstMismatchingU32)
# define ASMMemFirstMismatchingU32_EndProc              RT_MANGLER(ASMMemFirstMismatchingU32_EndProc)
# define ASMMemIsZero                                   RT_MANGLER(ASMMemIsZero)
# define ASMMemIsZero_EndProc                           RT_MANGLER(ASMMemIsZero_EndProc)
# define ASMMemIsAllU8                                  RT_MANGLER(ASMMemIsAllU8)
# define ASMMemIsAllU8_EndProc                          RT_MANGLER(ASMMemIsAllU8_EndProc)
# define ASMMemZero32                                   RT_MANGLER(ASMMemZero32)
# define ASMMemZero32_EndProc                           RT_MANGLER(ASMMemZero32_EndProc)
# define ASMMultU64ByU32DivByU32                        RT_MANGLER(ASMMultU64ByU32DivByU32)
# define ASMMultU64ByU32DivByU32_EndProc                RT_MANGLER(ASMMultU64ByU32DivByU32_EndProc)
# define ASMNopPause                                    RT_MANGLER(ASMNopPause)
# define ASMNopPause_EndProc                            RT_MANGLER(ASMNopPause_EndProc)
# define ASMOutStrU16                                   RT_MANGLER(ASMOutStrU16)
# define ASMOutStrU16_EndProc                           RT_MANGLER(ASMOutStrU16_EndProc)
# define ASMOutStrU32                                   RT_MANGLER(ASMOutStrU32)
# define ASMOutStrU32_EndProc                           RT_MANGLER(ASMOutStrU32_EndProc)
# define ASMOutStrU8                                    RT_MANGLER(ASMOutStrU8)
# define ASMOutStrU8_EndProc                            RT_MANGLER(ASMOutStrU8_EndProc)
# define ASMOutU16                                      RT_MANGLER(ASMOutU16)
# define ASMOutU16_EndProc                              RT_MANGLER(ASMOutU16_EndProc)
# define ASMOutU32                                      RT_MANGLER(ASMOutU32)
# define ASMOutU32_EndProc                              RT_MANGLER(ASMOutU32_EndProc)
# define ASMOutU8                                       RT_MANGLER(ASMOutU8)
# define ASMOutU8_EndProc                               RT_MANGLER(ASMOutU8_EndProc)
# define ASMProbeReadByte                               RT_MANGLER(ASMProbeReadByte)
# define ASMProbeReadByte_EndProc                       RT_MANGLER(ASMProbeReadByte_EndProc)
# define ASMRdMsr                                       RT_MANGLER(ASMRdMsr)
# define ASMRdMsr_EndProc                               RT_MANGLER(ASMRdMsr_EndProc)
# define ASMRdMsr_High                                  RT_MANGLER(ASMRdMsr_High)
# define ASMRdMsr_High_EndProc                          RT_MANGLER(ASMRdMsr_High_EndProc)
# define ASMRdMsr_Low                                   RT_MANGLER(ASMRdMsr_Low)
# define ASMRdMsr_Low_EndProc                           RT_MANGLER(ASMRdMsr_Low_EndProc)
# define ASMRdMsrEx                                     RT_MANGLER(ASMRdMsrEx)
# define ASMRdMsrEx_EndProc                             RT_MANGLER(ASMRdMsrEx_EndProc)
# define ASMReadTSC                                     RT_MANGLER(ASMReadTSC)
# define ASMReadTSC_EndProc                             RT_MANGLER(ASMReadTSC_EndProc)
# define ASMReadTscWithAux                              RT_MANGLER(ASMReadTscWithAux)
# define ASMReadTscWithAux_EndProc                      RT_MANGLER(ASMReadTscWithAux_EndProc)
# define ASMReloadCR3                                   RT_MANGLER(ASMReloadCR3)
# define ASMReloadCR3_EndProc                           RT_MANGLER(ASMReloadCR3_EndProc)
# define ASMRotateLeftU32                               RT_MANGLER(ASMRotateLeftU32)
# define ASMRotateLeftU32_EndProc                       RT_MANGLER(ASMRotateLeftU32_EndProc)
# define ASMRotateRightU32                              RT_MANGLER(ASMRotateRightU32)
# define ASMRotateRightU32_EndProc                      RT_MANGLER(ASMRotateRightU32_EndProc)
# define ASMSerializeInstructionCpuId                   RT_MANGLER(ASMSerializeInstructionCpuId)
# define ASMSerializeInstructionCpuId_EndProc           RT_MANGLER(ASMSerializeInstructionCpuId_EndProc)
# define ASMSerializeInstructionIRet                    RT_MANGLER(ASMSerializeInstructionIRet)
# define ASMSerializeInstructionIRet_EndProc            RT_MANGLER(ASMSerializeInstructionIRet_EndProc)
# define ASMSerializeInstructionRdTscp                  RT_MANGLER(ASMSerializeInstructionRdTscp)
# define ASMSerializeInstructionRdTscp_EndProc          RT_MANGLER(ASMSerializeInstructionRdTscp_EndProc)
# define ASMSetCR0                                      RT_MANGLER(ASMSetCR0)
# define ASMSetCR0_EndProc                              RT_MANGLER(ASMSetCR0_EndProc)
# define ASMSetCR2                                      RT_MANGLER(ASMSetCR2)
# define ASMSetCR2_EndProc                              RT_MANGLER(ASMSetCR2_EndProc)
# define ASMSetCR3                                      RT_MANGLER(ASMSetCR3)
# define ASMSetCR3_EndProc                              RT_MANGLER(ASMSetCR3_EndProc)
# define ASMSetCR4                                      RT_MANGLER(ASMSetCR4)
# define ASMSetCR4_EndProc                              RT_MANGLER(ASMSetCR4_EndProc)
# define ASMSetDR0                                      RT_MANGLER(ASMSetDR0)
# define ASMSetDR0_EndProc                              RT_MANGLER(ASMSetDR0_EndProc)
# define ASMSetDR1                                      RT_MANGLER(ASMSetDR1)
# define ASMSetDR1_EndProc                              RT_MANGLER(ASMSetDR1_EndProc)
# define ASMSetDR2                                      RT_MANGLER(ASMSetDR2)
# define ASMSetDR2_EndProc                              RT_MANGLER(ASMSetDR2_EndProc)
# define ASMSetDR3                                      RT_MANGLER(ASMSetDR3)
# define ASMSetDR3_EndProc                              RT_MANGLER(ASMSetDR3_EndProc)
# define ASMSetDR6                                      RT_MANGLER(ASMSetDR6)
# define ASMSetDR6_EndProc                              RT_MANGLER(ASMSetDR6_EndProc)
# define ASMSetDR7                                      RT_MANGLER(ASMSetDR7)
# define ASMSetDR7_EndProc                              RT_MANGLER(ASMSetDR7_EndProc)
# define ASMSetFlags                                    RT_MANGLER(ASMSetFlags)
# define ASMSetFlags_EndProc                            RT_MANGLER(ASMSetFlags_EndProc)
# define ASMSetGDTR                                     RT_MANGLER(ASMSetGDTR)
# define ASMSetGDTR_EndProc                             RT_MANGLER(ASMSetGDTR_EndProc)
# define ASMSetIDTR                                     RT_MANGLER(ASMSetIDTR)
# define ASMSetIDTR_EndProc                             RT_MANGLER(ASMSetIDTR_EndProc)
# define ASMSetXcr0                                     RT_MANGLER(ASMSetXcr0)
# define ASMSetXcr0_EndProc                             RT_MANGLER(ASMSetXcr0_EndProc)
# define ASMWriteBackAndInvalidateCaches                RT_MANGLER(ASMWriteBackAndInvalidateCaches)
# define ASMWriteBackAndInvalidateCaches_EndProc        RT_MANGLER(ASMWriteBackAndInvalidateCaches_EndProc)
# define ASMWrMsr                                       RT_MANGLER(ASMWrMsr)
# define ASMWrMsr_EndProc                               RT_MANGLER(ASMWrMsr_EndProc)
# define ASMWrMsrEx                                     RT_MANGLER(ASMWrMsrEx)
# define ASMWrMsrEx_EndProc                             RT_MANGLER(ASMWrMsrEx_EndProc)
# define ASMXRstor                                      RT_MANGLER(ASMXRstor)
# define ASMXRstor_EndProc                              RT_MANGLER(ASMXRstor_EndProc)
# define ASMXSave                                       RT_MANGLER(ASMXSave)
# define ASMXSave_EndProc                               RT_MANGLER(ASMXSave_EndProc)
# define ASMFxRstor                                     RT_MANGLER(ASMFxRstor)
# define ASMFxRstor_EndProc                             RT_MANGLER(ASMFxRstor_EndProc)
# define ASMFxSave                                      RT_MANGLER(ASMFxSave)
# define ASMFxSave_EndProc                              RT_MANGLER(ASMFxSave_EndProc)

# define RTAssertAreQuiet                               RT_MANGLER(RTAssertAreQuiet)
# define RTAssertMayPanic                               RT_MANGLER(RTAssertMayPanic)
# define RTAssertMsg1                                   RT_MANGLER(RTAssertMsg1)
# define RTAssertMsg1Weak                               RT_MANGLER(RTAssertMsg1Weak)
# define RTAssertMsg2                                   RT_MANGLER(RTAssertMsg2)
# define RTAssertMsg2Add                                RT_MANGLER(RTAssertMsg2Add)
# define RTAssertMsg2AddV                               RT_MANGLER(RTAssertMsg2AddV)
# define RTAssertMsg2AddWeak                            RT_MANGLER(RTAssertMsg2AddWeak)
# define RTAssertMsg2AddWeakV                           RT_MANGLER(RTAssertMsg2AddWeakV)
# define RTAssertMsg2V                                  RT_MANGLER(RTAssertMsg2V)
# define RTAssertMsg2Weak                               RT_MANGLER(RTAssertMsg2Weak)
# define RTAssertMsg2WeakV                              RT_MANGLER(RTAssertMsg2WeakV)
# define RTAssertSetMayPanic                            RT_MANGLER(RTAssertSetMayPanic)
# define RTAssertSetQuiet                               RT_MANGLER(RTAssertSetQuiet)
# define RTAssertShouldPanic                            RT_MANGLER(RTAssertShouldPanic)
# define RTAvlGCPhysDestroy                             RT_MANGLER(RTAvlGCPhysDestroy)
# define RTAvlGCPhysDoWithAll                           RT_MANGLER(RTAvlGCPhysDoWithAll)
# define RTAvlGCPhysGet                                 RT_MANGLER(RTAvlGCPhysGet)
# define RTAvlGCPhysGetBestFit                          RT_MANGLER(RTAvlGCPhysGetBestFit)
# define RTAvlGCPhysInsert                              RT_MANGLER(RTAvlGCPhysInsert)
# define RTAvlGCPhysRemove                              RT_MANGLER(RTAvlGCPhysRemove)
# define RTAvlGCPhysRemoveBestFit                       RT_MANGLER(RTAvlGCPhysRemoveBestFit)
# define RTAvlGCPtrDestroy                              RT_MANGLER(RTAvlGCPtrDestroy)
# define RTAvlGCPtrDoWithAll                            RT_MANGLER(RTAvlGCPtrDoWithAll)
# define RTAvlGCPtrGet                                  RT_MANGLER(RTAvlGCPtrGet)
# define RTAvlGCPtrGetBestFit                           RT_MANGLER(RTAvlGCPtrGetBestFit)
# define RTAvlGCPtrInsert                               RT_MANGLER(RTAvlGCPtrInsert)
# define RTAvlGCPtrRemove                               RT_MANGLER(RTAvlGCPtrRemove)
# define RTAvlGCPtrRemoveBestFit                        RT_MANGLER(RTAvlGCPtrRemoveBestFit)
# define RTAvlHCPhysDestroy                             RT_MANGLER(RTAvlHCPhysDestroy)
# define RTAvlHCPhysDoWithAll                           RT_MANGLER(RTAvlHCPhysDoWithAll)
# define RTAvlHCPhysGet                                 RT_MANGLER(RTAvlHCPhysGet)
# define RTAvlHCPhysGetBestFit                          RT_MANGLER(RTAvlHCPhysGetBestFit)
# define RTAvlHCPhysInsert                              RT_MANGLER(RTAvlHCPhysInsert)
# define RTAvlHCPhysRemove                              RT_MANGLER(RTAvlHCPhysRemove)
# define RTAvlHCPhysRemoveBestFit                       RT_MANGLER(RTAvlHCPhysRemoveBestFit)
# define RTAvllU32Destroy                               RT_MANGLER(RTAvllU32Destroy)
# define RTAvllU32DoWithAll                             RT_MANGLER(RTAvllU32DoWithAll)
# define RTAvllU32Get                                   RT_MANGLER(RTAvllU32Get)
# define RTAvllU32GetBestFit                            RT_MANGLER(RTAvllU32GetBestFit)
# define RTAvllU32Insert                                RT_MANGLER(RTAvllU32Insert)
# define RTAvllU32Remove                                RT_MANGLER(RTAvllU32Remove)
# define RTAvllU32RemoveBestFit                         RT_MANGLER(RTAvllU32RemoveBestFit)
# define RTAvllU32RemoveNode                            RT_MANGLER(RTAvllU32RemoveNode)
# define RTAvloGCPhysDestroy                            RT_MANGLER(RTAvloGCPhysDestroy)
# define RTAvloGCPhysDoWithAll                          RT_MANGLER(RTAvloGCPhysDoWithAll)
# define RTAvloGCPhysGet                                RT_MANGLER(RTAvloGCPhysGet)
# define RTAvloGCPhysGetBestFit                         RT_MANGLER(RTAvloGCPhysGetBestFit)
# define RTAvloGCPhysInsert                             RT_MANGLER(RTAvloGCPhysInsert)
# define RTAvloGCPhysRemove                             RT_MANGLER(RTAvloGCPhysRemove)
# define RTAvloGCPhysRemoveBestFit                      RT_MANGLER(RTAvloGCPhysRemoveBestFit)
# define RTAvloGCPtrDestroy                             RT_MANGLER(RTAvloGCPtrDestroy)
# define RTAvloGCPtrDoWithAll                           RT_MANGLER(RTAvloGCPtrDoWithAll)
# define RTAvloGCPtrGet                                 RT_MANGLER(RTAvloGCPtrGet)
# define RTAvloGCPtrGetBestFit                          RT_MANGLER(RTAvloGCPtrGetBestFit)
# define RTAvloGCPtrInsert                              RT_MANGLER(RTAvloGCPtrInsert)
# define RTAvloGCPtrRemove                              RT_MANGLER(RTAvloGCPtrRemove)
# define RTAvloGCPtrRemoveBestFit                       RT_MANGLER(RTAvloGCPtrRemoveBestFit)
# define RTAvloHCPhysDestroy                            RT_MANGLER(RTAvloHCPhysDestroy)
# define RTAvloHCPhysDoWithAll                          RT_MANGLER(RTAvloHCPhysDoWithAll)
# define RTAvloHCPhysGet                                RT_MANGLER(RTAvloHCPhysGet)
# define RTAvloHCPhysGetBestFit                         RT_MANGLER(RTAvloHCPhysGetBestFit)
# define RTAvloHCPhysInsert                             RT_MANGLER(RTAvloHCPhysInsert)
# define RTAvloHCPhysRemove                             RT_MANGLER(RTAvloHCPhysRemove)
# define RTAvloHCPhysRemoveBestFit                      RT_MANGLER(RTAvloHCPhysRemoveBestFit)
# define RTAvloIOPortDestroy                            RT_MANGLER(RTAvloIOPortDestroy)
# define RTAvloIOPortDoWithAll                          RT_MANGLER(RTAvloIOPortDoWithAll)
# define RTAvloIOPortGet                                RT_MANGLER(RTAvloIOPortGet)
# define RTAvloIOPortGetBestFit                         RT_MANGLER(RTAvloIOPortGetBestFit)
# define RTAvloIOPortInsert                             RT_MANGLER(RTAvloIOPortInsert)
# define RTAvloIOPortRemove                             RT_MANGLER(RTAvloIOPortRemove)
# define RTAvloIOPortRemoveBestFit                      RT_MANGLER(RTAvloIOPortRemoveBestFit)
# define RTAvloU32Destroy                               RT_MANGLER(RTAvloU32Destroy)
# define RTAvloU32DoWithAll                             RT_MANGLER(RTAvloU32DoWithAll)
# define RTAvloU32Get                                   RT_MANGLER(RTAvloU32Get)
# define RTAvloU32GetBestFit                            RT_MANGLER(RTAvloU32GetBestFit)
# define RTAvloU32Insert                                RT_MANGLER(RTAvloU32Insert)
# define RTAvloU32Remove                                RT_MANGLER(RTAvloU32Remove)
# define RTAvloU32RemoveBestFit                         RT_MANGLER(RTAvloU32RemoveBestFit)
# define RTAvlPVDestroy                                 RT_MANGLER(RTAvlPVDestroy)
# define RTAvlPVDoWithAll                               RT_MANGLER(RTAvlPVDoWithAll)
# define RTAvlPVGet                                     RT_MANGLER(RTAvlPVGet)
# define RTAvlPVGetBestFit                              RT_MANGLER(RTAvlPVGetBestFit)
# define RTAvlPVInsert                                  RT_MANGLER(RTAvlPVInsert)
# define RTAvlPVRemove                                  RT_MANGLER(RTAvlPVRemove)
# define RTAvlPVRemoveBestFit                           RT_MANGLER(RTAvlPVRemoveBestFit)
# define RTAvlrFileOffsetDestroy                        RT_MANGLER(RTAvlrFileOffsetDestroy)
# define RTAvlrFileOffsetDoWithAll                      RT_MANGLER(RTAvlrFileOffsetDoWithAll)
# define RTAvlrFileOffsetGet                            RT_MANGLER(RTAvlrFileOffsetGet)
# define RTAvlrFileOffsetGetBestFit                     RT_MANGLER(RTAvlrFileOffsetGetBestFit)
# define RTAvlrFileOffsetGetLeft                        RT_MANGLER(RTAvlrFileOffsetGetLeft)
# define RTAvlrFileOffsetGetRight                       RT_MANGLER(RTAvlrFileOffsetGetRight)
# define RTAvlrFileOffsetGetRoot                        RT_MANGLER(RTAvlrFileOffsetGetRoot)
# define RTAvlrFileOffsetInsert                         RT_MANGLER(RTAvlrFileOffsetInsert)
# define RTAvlrFileOffsetRangeGet                       RT_MANGLER(RTAvlrFileOffsetRangeGet)
# define RTAvlrFileOffsetRangeRemove                    RT_MANGLER(RTAvlrFileOffsetRangeRemove)
# define RTAvlrFileOffsetRemove                         RT_MANGLER(RTAvlrFileOffsetRemove)
# define RTAvlrGCPtrDestroy                             RT_MANGLER(RTAvlrGCPtrDestroy)
# define RTAvlrGCPtrDoWithAll                           RT_MANGLER(RTAvlrGCPtrDoWithAll)
# define RTAvlrGCPtrGet                                 RT_MANGLER(RTAvlrGCPtrGet)
# define RTAvlrGCPtrGetBestFit                          RT_MANGLER(RTAvlrGCPtrGetBestFit)
# define RTAvlrGCPtrGetLeft                             RT_MANGLER(RTAvlrGCPtrGetLeft)
# define RTAvlrGCPtrGetRight                            RT_MANGLER(RTAvlrGCPtrGetRight)
# define RTAvlrGCPtrGetRoot                             RT_MANGLER(RTAvlrGCPtrGetRoot)
# define RTAvlrGCPtrInsert                              RT_MANGLER(RTAvlrGCPtrInsert)
# define RTAvlrGCPtrRangeGet                            RT_MANGLER(RTAvlrGCPtrRangeGet)
# define RTAvlrGCPtrRangeRemove                         RT_MANGLER(RTAvlrGCPtrRangeRemove)
# define RTAvlrGCPtrRemove                              RT_MANGLER(RTAvlrGCPtrRemove)
# define RTAvlroGCPhysDestroy                           RT_MANGLER(RTAvlroGCPhysDestroy)
# define RTAvlroGCPhysDoWithAll                         RT_MANGLER(RTAvlroGCPhysDoWithAll)
# define RTAvlroGCPhysGet                               RT_MANGLER(RTAvlroGCPhysGet)
# define RTAvlroGCPhysGetBestFit                        RT_MANGLER(RTAvlroGCPhysGetBestFit)
# define RTAvlroGCPhysGetLeft                           RT_MANGLER(RTAvlroGCPhysGetLeft)
# define RTAvlroGCPhysGetRight                          RT_MANGLER(RTAvlroGCPhysGetRight)
# define RTAvlroGCPhysGetRoot                           RT_MANGLER(RTAvlroGCPhysGetRoot)
# define RTAvlroGCPhysInsert                            RT_MANGLER(RTAvlroGCPhysInsert)
# define RTAvlroGCPhysRangeGet                          RT_MANGLER(RTAvlroGCPhysRangeGet)
# define RTAvlroGCPhysRangeRemove                       RT_MANGLER(RTAvlroGCPhysRangeRemove)
# define RTAvlroGCPhysRemove                            RT_MANGLER(RTAvlroGCPhysRemove)
# define RTAvlroGCPtrDestroy                            RT_MANGLER(RTAvlroGCPtrDestroy)
# define RTAvlroGCPtrDoWithAll                          RT_MANGLER(RTAvlroGCPtrDoWithAll)
# define RTAvlroGCPtrGet                                RT_MANGLER(RTAvlroGCPtrGet)
# define RTAvlroGCPtrGetBestFit                         RT_MANGLER(RTAvlroGCPtrGetBestFit)
# define RTAvlroGCPtrGetLeft                            RT_MANGLER(RTAvlroGCPtrGetLeft)
# define RTAvlroGCPtrGetRight                           RT_MANGLER(RTAvlroGCPtrGetRight)
# define RTAvlroGCPtrGetRoot                            RT_MANGLER(RTAvlroGCPtrGetRoot)
# define RTAvlroGCPtrInsert                             RT_MANGLER(RTAvlroGCPtrInsert)
# define RTAvlroGCPtrRangeGet                           RT_MANGLER(RTAvlroGCPtrRangeGet)
# define RTAvlroGCPtrRangeRemove                        RT_MANGLER(RTAvlroGCPtrRangeRemove)
# define RTAvlroGCPtrRemove                             RT_MANGLER(RTAvlroGCPtrRemove)
# define RTAvlroIOPortDestroy                           RT_MANGLER(RTAvlroIOPortDestroy)
# define RTAvlroIOPortDoWithAll                         RT_MANGLER(RTAvlroIOPortDoWithAll)
# define RTAvlroIOPortGet                               RT_MANGLER(RTAvlroIOPortGet)
# define RTAvlroIOPortInsert                            RT_MANGLER(RTAvlroIOPortInsert)
# define RTAvlroIOPortRangeGet                          RT_MANGLER(RTAvlroIOPortRangeGet)
# define RTAvlroIOPortRangeRemove                       RT_MANGLER(RTAvlroIOPortRangeRemove)
# define RTAvlroIOPortRemove                            RT_MANGLER(RTAvlroIOPortRemove)
# define RTAvlrooGCPtrDestroy                           RT_MANGLER(RTAvlrooGCPtrDestroy)
# define RTAvlrooGCPtrDoWithAll                         RT_MANGLER(RTAvlrooGCPtrDoWithAll)
# define RTAvlrooGCPtrGet                               RT_MANGLER(RTAvlrooGCPtrGet)
# define RTAvlrooGCPtrGetBestFit                        RT_MANGLER(RTAvlrooGCPtrGetBestFit)
# define RTAvlrooGCPtrGetLeft                           RT_MANGLER(RTAvlrooGCPtrGetLeft)
# define RTAvlrooGCPtrGetNextEqual                      RT_MANGLER(RTAvlrooGCPtrGetNextEqual)
# define RTAvlrooGCPtrGetRight                          RT_MANGLER(RTAvlrooGCPtrGetRight)
# define RTAvlrooGCPtrGetRoot                           RT_MANGLER(RTAvlrooGCPtrGetRoot)
# define RTAvlrooGCPtrInsert                            RT_MANGLER(RTAvlrooGCPtrInsert)
# define RTAvlrooGCPtrRangeGet                          RT_MANGLER(RTAvlrooGCPtrRangeGet)
# define RTAvlrooGCPtrRangeRemove                       RT_MANGLER(RTAvlrooGCPtrRangeRemove)
# define RTAvlrooGCPtrRemove                            RT_MANGLER(RTAvlrooGCPtrRemove)
# define RTAvlrPVDestroy                                RT_MANGLER(RTAvlrPVDestroy)
# define RTAvlrPVDoWithAll                              RT_MANGLER(RTAvlrPVDoWithAll)
# define RTAvlrPVGet                                    RT_MANGLER(RTAvlrPVGet)
# define RTAvlrPVGetBestFit                             RT_MANGLER(RTAvlrPVGetBestFit)
# define RTAvlrPVInsert                                 RT_MANGLER(RTAvlrPVInsert)
# define RTAvlrPVRangeGet                               RT_MANGLER(RTAvlrPVRangeGet)
# define RTAvlrPVRangeRemove                            RT_MANGLER(RTAvlrPVRangeRemove)
# define RTAvlrPVRemove                                 RT_MANGLER(RTAvlrPVRemove)
# define RTAvlrPVRemoveBestFit                          RT_MANGLER(RTAvlrPVRemoveBestFit)
# define RTAvlrU64Destroy                               RT_MANGLER(RTAvlrU64Destroy)
# define RTAvlrU64DoWithAll                             RT_MANGLER(RTAvlrU64DoWithAll)
# define RTAvlrU64Get                                   RT_MANGLER(RTAvlrU64Get)
# define RTAvlrU64GetBestFit                            RT_MANGLER(RTAvlrU64GetBestFit)
# define RTAvlrU64Insert                                RT_MANGLER(RTAvlrU64Insert)
# define RTAvlrU64RangeGet                              RT_MANGLER(RTAvlrU64RangeGet)
# define RTAvlrU64RangeRemove                           RT_MANGLER(RTAvlrU64RangeRemove)
# define RTAvlrU64Remove                                RT_MANGLER(RTAvlrU64Remove)
# define RTAvlrU64RemoveBestFit                         RT_MANGLER(RTAvlrU64RemoveBestFit)
# define RTAvlrUIntPtrDestroy                           RT_MANGLER(RTAvlrUIntPtrDestroy)
# define RTAvlrUIntPtrDoWithAll                         RT_MANGLER(RTAvlrUIntPtrDoWithAll)
# define RTAvlrUIntPtrGet                               RT_MANGLER(RTAvlrUIntPtrGet)
# define RTAvlrUIntPtrGetBestFit                        RT_MANGLER(RTAvlrUIntPtrGetBestFit)
# define RTAvlrUIntPtrGetLeft                           RT_MANGLER(RTAvlrUIntPtrGetLeft)
# define RTAvlrUIntPtrGetRight                          RT_MANGLER(RTAvlrUIntPtrGetRight)
# define RTAvlrUIntPtrGetRoot                           RT_MANGLER(RTAvlrUIntPtrGetRoot)
# define RTAvlrUIntPtrInsert                            RT_MANGLER(RTAvlrUIntPtrInsert)
# define RTAvlrUIntPtrRangeGet                          RT_MANGLER(RTAvlrUIntPtrRangeGet)
# define RTAvlrUIntPtrRangeRemove                       RT_MANGLER(RTAvlrUIntPtrRangeRemove)
# define RTAvlrUIntPtrRemove                            RT_MANGLER(RTAvlrUIntPtrRemove)
# define RTAvlU32Destroy                                RT_MANGLER(RTAvlU32Destroy)
# define RTAvlU32DoWithAll                              RT_MANGLER(RTAvlU32DoWithAll)
# define RTAvlU32Get                                    RT_MANGLER(RTAvlU32Get)
# define RTAvlU32GetBestFit                             RT_MANGLER(RTAvlU32GetBestFit)
# define RTAvlU32Insert                                 RT_MANGLER(RTAvlU32Insert)
# define RTAvlU32Remove                                 RT_MANGLER(RTAvlU32Remove)
# define RTAvlU32RemoveBestFit                          RT_MANGLER(RTAvlU32RemoveBestFit)
# define RTAvlU64Destroy                                RT_MANGLER(RTAvlU64Destroy)
# define RTAvlU64DoWithAll                              RT_MANGLER(RTAvlU64DoWithAll)
# define RTAvlU64Get                                    RT_MANGLER(RTAvlU64Get)
# define RTAvlU64GetBestFit                             RT_MANGLER(RTAvlU64GetBestFit)
# define RTAvlU64Insert                                 RT_MANGLER(RTAvlU64Insert)
# define RTAvlU64Remove                                 RT_MANGLER(RTAvlU64Remove)
# define RTAvlU64RemoveBestFit                          RT_MANGLER(RTAvlU64RemoveBestFit)
# define RTAvlUIntPtrDestroy                            RT_MANGLER(RTAvlUIntPtrDestroy)
# define RTAvlUIntPtrDoWithAll                          RT_MANGLER(RTAvlUIntPtrDoWithAll)
# define RTAvlUIntPtrGet                                RT_MANGLER(RTAvlUIntPtrGet)
# define RTAvlUIntPtrGetBestFit                         RT_MANGLER(RTAvlUIntPtrGetBestFit)
# define RTAvlUIntPtrGetLeft                            RT_MANGLER(RTAvlUIntPtrGetLeft)
# define RTAvlUIntPtrGetRight                           RT_MANGLER(RTAvlUIntPtrGetRight)
# define RTAvlUIntPtrGetRoot                            RT_MANGLER(RTAvlUIntPtrGetRoot)
# define RTAvlUIntPtrInsert                             RT_MANGLER(RTAvlUIntPtrInsert)
# define RTAvlUIntPtrRemove                             RT_MANGLER(RTAvlUIntPtrRemove)
# define RTAvlULDestroy                                 RT_MANGLER(RTAvlULDestroy)
# define RTAvlULDoWithAll                               RT_MANGLER(RTAvlULDoWithAll)
# define RTAvlULGet                                     RT_MANGLER(RTAvlULGet)
# define RTAvlULGetBestFit                              RT_MANGLER(RTAvlULGetBestFit)
# define RTAvlULInsert                                  RT_MANGLER(RTAvlULInsert)
# define RTAvlULRemove                                  RT_MANGLER(RTAvlULRemove)
# define RTAvlULRemoveBestFit                           RT_MANGLER(RTAvlULRemoveBestFit)
# define RTBase64Decode                                 RT_MANGLER(RTBase64Decode)
# define RTBase64DecodeEx                               RT_MANGLER(RTBase64DecodeEx)
# define RTBase64DecodedSize                            RT_MANGLER(RTBase64DecodedSize)
# define RTBase64DecodedSizeEx                          RT_MANGLER(RTBase64DecodedSizeEx)
# define RTBase64DecodeUtf16                            RT_MANGLER(RTBase64DecodeUtf16)
# define RTBase64DecodeUtf16Ex                          RT_MANGLER(RTBase64DecodeUtf16Ex)
# define RTBase64DecodedUtf16Size                       RT_MANGLER(RTBase64DecodedUtf16Size)
# define RTBase64DecodedUtf16SizeEx                     RT_MANGLER(RTBase64DecodedUtf16SizeEx)
# define RTBase64Encode                                 RT_MANGLER(RTBase64Encode)
# define RTBase64EncodeEx                               RT_MANGLER(RTBase64EncodeEx)
# define RTBase64EncodedLength                          RT_MANGLER(RTBase64EncodedLength)
# define RTBase64EncodedLengthEx                        RT_MANGLER(RTBase64EncodedLengthEx)
# define RTBase64EncodeUtf16                            RT_MANGLER(RTBase64EncodeUtf16)
# define RTBase64EncodeUtf16Ex                          RT_MANGLER(RTBase64EncodeUtf16Ex)
# define RTBase64EncodedUtf16Length                     RT_MANGLER(RTBase64EncodedUtf16Length)
# define RTBase64EncodedUtf16LengthEx                   RT_MANGLER(RTBase64EncodedUtf16LengthEx)
# define RTBldCfgCompiler                               RT_MANGLER(RTBldCfgCompiler)
# define RTBldCfgRevision                               RT_MANGLER(RTBldCfgRevision)
# define RTBldCfgRevisionStr                            RT_MANGLER(RTBldCfgRevisionStr)
# define RTBldCfgTarget                                 RT_MANGLER(RTBldCfgTarget)
# define RTBldCfgTargetArch                             RT_MANGLER(RTBldCfgTargetArch)
# define RTBldCfgTargetDotArch                          RT_MANGLER(RTBldCfgTargetDotArch)
# define RTBldCfgType                                   RT_MANGLER(RTBldCfgType)
# define RTBldCfgVersion                                RT_MANGLER(RTBldCfgVersion)
# define RTBldCfgVersionBuild                           RT_MANGLER(RTBldCfgVersionBuild)
# define RTBldCfgVersionMajor                           RT_MANGLER(RTBldCfgVersionMajor)
# define RTBldCfgVersionMinor                           RT_MANGLER(RTBldCfgVersionMinor)
# define RTCdromOpen                                    RT_MANGLER(RTCdromOpen)
# define RTCdromRetain                                  RT_MANGLER(RTCdromRetain)
# define RTCdromRelease                                 RT_MANGLER(RTCdromRelease)
# define RTCdromQueryMountPoint                         RT_MANGLER(RTCdromQueryMountPoint)
# define RTCdromUnmount                                 RT_MANGLER(RTCdromUnmount)
# define RTCdromEject                                   RT_MANGLER(RTCdromEject)
# define RTCdromLock                                    RT_MANGLER(RTCdromLock)
# define RTCdromUnlock                                  RT_MANGLER(RTCdromUnlock)
# define RTCdromCount                                   RT_MANGLER(RTCdromCount)
# define RTCdromOrdinalToName                           RT_MANGLER(RTCdromOrdinalToName)
# define RTCdromOpenByOrdinal                           RT_MANGLER(RTCdromOpenByOrdinal)
# define RTCidrStrToIPv4                                RT_MANGLER(RTCidrStrToIPv4)
# define RTCircBufAcquireReadBlock                      RT_MANGLER(RTCircBufAcquireReadBlock)
# define RTCircBufAcquireWriteBlock                     RT_MANGLER(RTCircBufAcquireWriteBlock)
# define RTCircBufCreate                                RT_MANGLER(RTCircBufCreate)
# define RTCircBufDestroy                               RT_MANGLER(RTCircBufDestroy)
# define RTCircBufFree                                  RT_MANGLER(RTCircBufFree)
# define RTCircBufIsReading                             RT_MANGLER(RTCircBufIsReading)
# define RTCircBufIsWriting                             RT_MANGLER(RTCircBufIsWriting)
# define RTCircBufOffsetRead                            RT_MANGLER(RTCircBufOffsetRead)
# define RTCircBufOffsetWrite                           RT_MANGLER(RTCircBufOffsetWrite)
# define RTCircBufReleaseReadBlock                      RT_MANGLER(RTCircBufReleaseReadBlock)
# define RTCircBufReleaseWriteBlock                     RT_MANGLER(RTCircBufReleaseWriteBlock)
# define RTCircBufReset                                 RT_MANGLER(RTCircBufReset)
# define RTCircBufSize                                  RT_MANGLER(RTCircBufSize)
# define RTCircBufUsed                                  RT_MANGLER(RTCircBufUsed)
# define RTCoreDumperDisable                            RT_MANGLER(RTCoreDumperDisable)  /* solaris */
# define RTCoreDumperSetup                              RT_MANGLER(RTCoreDumperSetup)    /* solaris */
# define RTCoreDumperTakeDump                           RT_MANGLER(RTCoreDumperTakeDump) /* solaris */
# define RTCrc16Ccitt                                   RT_MANGLER(RTCrc16Ccitt)
# define RTCrc16CcittProcess                            RT_MANGLER(RTCrc16CcittProcess)
# define RTCrc16CcittFinish                             RT_MANGLER(RTCrc16CcittFinish)
# define RTCrc16CcittStart                              RT_MANGLER(RTCrc16CcittStart)
# define RTCrc32                                        RT_MANGLER(RTCrc32)
# define RTCrc32Finish                                  RT_MANGLER(RTCrc32Finish)
# define RTCrc32Process                                 RT_MANGLER(RTCrc32Process)
# define RTCrc32Start                                   RT_MANGLER(RTCrc32Start)
# define RTCrc32C                                       RT_MANGLER(RTCrc32C)
# define RTCrc32CFinish                                 RT_MANGLER(RTCrc32CFinish)
# define RTCrc32CProcess                                RT_MANGLER(RTCrc32CProcess)
# define RTCrc32CStart                                  RT_MANGLER(RTCrc32CStart)
# define RTCrc64                                        RT_MANGLER(RTCrc64)
# define RTCrc64Finish                                  RT_MANGLER(RTCrc64Finish)
# define RTCrc64Process                                 RT_MANGLER(RTCrc64Process)
# define RTCrc64Start                                   RT_MANGLER(RTCrc64Start)
# define RTCrcAdler32                                   RT_MANGLER(RTCrcAdler32)
# define RTCrcAdler32Finish                             RT_MANGLER(RTCrcAdler32Finish)
# define RTCrcAdler32Process                            RT_MANGLER(RTCrcAdler32Process)
# define RTCrcAdler32Start                              RT_MANGLER(RTCrcAdler32Start)
# define RTCritSectDelete                               RT_MANGLER(RTCritSectDelete)
# define RTCritSectEnter                                RT_MANGLER(RTCritSectEnter)
# define RTCritSectEnterDebug                           RT_MANGLER(RTCritSectEnterDebug)
# define RTCritSectEnterMultiple                        RT_MANGLER(RTCritSectEnterMultiple)
# define RTCritSectEnterMultipleDebug                   RT_MANGLER(RTCritSectEnterMultipleDebug)
# define RTCritSectInit                                 RT_MANGLER(RTCritSectInit)
# define RTCritSectInitEx                               RT_MANGLER(RTCritSectInitEx)
# define RTCritSectInitNamed                            RT_MANGLER(RTCritSectInitNamed)
# define RTCritSectLeave                                RT_MANGLER(RTCritSectLeave)
# define RTCritSectLeaveMultiple                        RT_MANGLER(RTCritSectLeaveMultiple)
# define RTCritSectSetSubClass                          RT_MANGLER(RTCritSectSetSubClass)
# define RTCritSectTryEnter                             RT_MANGLER(RTCritSectTryEnter)
# define RTCritSectTryEnterDebug                        RT_MANGLER(RTCritSectTryEnterDebug)
# define RTCritSectRwDelete                             RT_MANGLER(RTCritSectRwDelete)
# define RTCritSectRwEnterExcl                          RT_MANGLER(RTCritSectRwEnterExcl)
# define RTCritSectRwEnterExclDebug                     RT_MANGLER(RTCritSectRwEnterExclDebug)
# define RTCritSectRwEnterShared                        RT_MANGLER(RTCritSectRwEnterShared)
# define RTCritSectRwEnterSharedDebug                   RT_MANGLER(RTCritSectRwEnterSharedDebug)
# define RTCritSectRwGetReadCount                       RT_MANGLER(RTCritSectRwGetReadCount)
# define RTCritSectRwGetWriteRecursion                  RT_MANGLER(RTCritSectRwGetWriteRecursion)
# define RTCritSectRwGetWriterReadRecursion             RT_MANGLER(RTCritSectRwGetWriterReadRecursion)
# define RTCritSectRwInit                               RT_MANGLER(RTCritSectRwInit)
# define RTCritSectRwInitEx                             RT_MANGLER(RTCritSectRwInitEx)
# define RTCritSectRwInitNamed                          RT_MANGLER(RTCritSectRwInitNamed)
# define RTCritSectRwIsReadOwner                        RT_MANGLER(RTCritSectRwIsReadOwner)
# define RTCritSectRwIsWriteOwner                       RT_MANGLER(RTCritSectRwIsWriteOwner)
# define RTCritSectRwLeaveExcl                          RT_MANGLER(RTCritSectRwLeaveExcl)
# define RTCritSectRwLeaveShared                        RT_MANGLER(RTCritSectRwLeaveShared)
# define RTCritSectRwSetSubClass                        RT_MANGLER(RTCritSectRwSetSubClass)
# define RTCritSectRwTryEnterExcl                       RT_MANGLER(RTCritSectRwTryEnterExcl)
# define RTCritSectRwTryEnterExclDebug                  RT_MANGLER(RTCritSectRwTryEnterExclDebug)
# define RTCritSectRwTryEnterShared                     RT_MANGLER(RTCritSectRwTryEnterShared)
# define RTCritSectRwTryEnterSharedDebug                RT_MANGLER(RTCritSectRwTryEnterSharedDebug)
# define RTDbgAsCreate                                  RT_MANGLER(RTDbgAsCreate)
# define RTDbgAsCreateF                                 RT_MANGLER(RTDbgAsCreateF)
# define RTDbgAsCreateV                                 RT_MANGLER(RTDbgAsCreateV)
# define RTDbgAsFirstAddr                               RT_MANGLER(RTDbgAsFirstAddr)
# define RTDbgAsLastAddr                                RT_MANGLER(RTDbgAsLastAddr)
# define RTDbgAsLineAdd                                 RT_MANGLER(RTDbgAsLineAdd)
# define RTDbgAsLineByAddr                              RT_MANGLER(RTDbgAsLineByAddr)
# define RTDbgAsLineByAddrA                             RT_MANGLER(RTDbgAsLineByAddrA)
# define RTDbgAsLockExcl                                RT_MANGLER(RTDbgAsLockExcl)
# define RTDbgAsModuleByAddr                            RT_MANGLER(RTDbgAsModuleByAddr)
# define RTDbgAsModuleByIndex                           RT_MANGLER(RTDbgAsModuleByIndex)
# define RTDbgAsModuleByName                            RT_MANGLER(RTDbgAsModuleByName)
# define RTDbgAsModuleCount                             RT_MANGLER(RTDbgAsModuleCount)
# define RTDbgAsModuleLink                              RT_MANGLER(RTDbgAsModuleLink)
# define RTDbgAsModuleLinkSeg                           RT_MANGLER(RTDbgAsModuleLinkSeg)
# define RTDbgAsModuleQueryMapByIndex                   RT_MANGLER(RTDbgAsModuleQueryMapByIndex)
# define RTDbgAsModuleUnlink                            RT_MANGLER(RTDbgAsModuleUnlink)
# define RTDbgAsModuleUnlinkByAddr                      RT_MANGLER(RTDbgAsModuleUnlinkByAddr)
# define RTDbgAsName                                    RT_MANGLER(RTDbgAsName)
# define RTDbgAsRelease                                 RT_MANGLER(RTDbgAsRelease)
# define RTDbgAsRetain                                  RT_MANGLER(RTDbgAsRetain)
# define RTDbgAsSymbolAdd                               RT_MANGLER(RTDbgAsSymbolAdd)
# define RTDbgAsSymbolByAddr                            RT_MANGLER(RTDbgAsSymbolByAddr)
# define RTDbgAsSymbolByAddrA                           RT_MANGLER(RTDbgAsSymbolByAddrA)
# define RTDbgAsSymbolByName                            RT_MANGLER(RTDbgAsSymbolByName)
# define RTDbgAsSymbolByNameA                           RT_MANGLER(RTDbgAsSymbolByNameA)
# define RTDbgAsUnlockExcl                              RT_MANGLER(RTDbgAsUnlockExcl)
# define RTDbgCfgCreate                                 RT_MANGLER(RTDbgCfgCreate)
# define RTDbgCfgRetain                                 RT_MANGLER(RTDbgCfgRetain)
# define RTDbgCfgRelease                                RT_MANGLER(RTDbgCfgRelease)
# define RTDbgCfgChangeString                           RT_MANGLER(RTDbgCfgChangeString)
# define RTDbgCfgChangeUInt                             RT_MANGLER(RTDbgCfgChangeUInt)
# define RTDbgCfgQueryString                            RT_MANGLER(RTDbgCfgQueryString)
# define RTDbgCfgQueryUInt                              RT_MANGLER(RTDbgCfgQueryUInt)
# define RTDbgCfgOpenEx                                 RT_MANGLER(RTDbgCfgOpenEx)
# define RTDbgCfgOpenDbg                                RT_MANGLER(RTDbgCfgOpenDbg)
# define RTDbgCfgOpenDsymBundle                         RT_MANGLER(RTDbgCfgOpenDsymBundle)
# define RTDbgCfgOpenMachOImage                         RT_MANGLER(RTDbgCfgOpenMachOImage)
# define RTDbgCfgOpenDwo                                RT_MANGLER(RTDbgCfgOpenDwo)
# define RTDbgCfgOpenDwoBuildId                         RT_MANGLER(RTDbgCfgOpenDwoBuildId)
# define RTDbgCfgOpenPdb70                              RT_MANGLER(RTDbgCfgOpenPdb70)
# define RTDbgCfgOpenPdb20                              RT_MANGLER(RTDbgCfgOpenPdb20)
# define RTDbgCfgOpenPeImage                            RT_MANGLER(RTDbgCfgOpenPeImage)
# define RTDbgCfgSetLogCallback                         RT_MANGLER(RTDbgCfgSetLogCallback)
# define RTDbgLineAlloc                                 RT_MANGLER(RTDbgLineAlloc)
# define RTDbgLineDup                                   RT_MANGLER(RTDbgLineDup)
# define RTDbgLineFree                                  RT_MANGLER(RTDbgLineFree)
# define RTDbgModCreate                                 RT_MANGLER(RTDbgModCreate)
# define RTDbgModCreateFromDbg                          RT_MANGLER(RTDbgModCreateFromDbg)
# define RTDbgModCreateFromDwo                          RT_MANGLER(RTDbgModCreateFromDwo)
# define RTDbgModCreateFromImage                        RT_MANGLER(RTDbgModCreateFromImage)
# define RTDbgModCreateFromMap                          RT_MANGLER(RTDbgModCreateFromMap)
# define RTDbgModCreateFromPdb                          RT_MANGLER(RTDbgModCreateFromPdb)
# define RTDbgModCreateFromPeImage                      RT_MANGLER(RTDbgModCreateFromPeImage)
# define RTDbgModCreateFromMachOImage                   RT_MANGLER(RTDbgModCreateFromMachOImage)
# define RTDbgModGetTag                                 RT_MANGLER(RTDbgModGetTag)
# define RTDbgModImageGetArch                           RT_MANGLER(RTDbgModImageGetArch)
# define RTDbgModImageGetFormat                         RT_MANGLER(RTDbgModImageGetFormat)
# define RTDbgModImageSize                              RT_MANGLER(RTDbgModImageSize)
# define RTDbgModImageQueryProp                         RT_MANGLER(RTDbgModImageQueryProp)
# define RTDbgModIsDeferred                             RT_MANGLER(RTDbgModIsDeferred)
# define RTDbgModIsExports                              RT_MANGLER(RTDbgModIsExports)
# define RTDbgModLineAdd                                RT_MANGLER(RTDbgModLineAdd)
# define RTDbgModLineByAddr                             RT_MANGLER(RTDbgModLineByAddr)
# define RTDbgModLineByAddrA                            RT_MANGLER(RTDbgModLineByAddrA)
# define RTDbgModLineByOrdinal                          RT_MANGLER(RTDbgModLineByOrdinal)
# define RTDbgModLineByOrdinalA                         RT_MANGLER(RTDbgModLineByOrdinalA)
# define RTDbgModLineCount                              RT_MANGLER(RTDbgModLineCount)
# define RTDbgModName                                   RT_MANGLER(RTDbgModName)
# define RTDbgModDebugFile                              RT_MANGLER(RTDbgModDebugFile)
# define RTDbgModImageFile                              RT_MANGLER(RTDbgModImageFile)
# define RTDbgModImageFileUsed                          RT_MANGLER(RTDbgModImageFileUsed)
# define RTDbgModRelease                                RT_MANGLER(RTDbgModRelease)
# define RTDbgModRemoveAll                              RT_MANGLER(RTDbgModRemoveAll)
# define RTDbgModRetain                                 RT_MANGLER(RTDbgModRetain)
# define RTDbgModRvaToSegOff                            RT_MANGLER(RTDbgModRvaToSegOff)
# define RTDbgModSegmentAdd                             RT_MANGLER(RTDbgModSegmentAdd)
# define RTDbgModSegmentByIndex                         RT_MANGLER(RTDbgModSegmentByIndex)
# define RTDbgModSegmentCount                           RT_MANGLER(RTDbgModSegmentCount)
# define RTDbgModSegmentRva                             RT_MANGLER(RTDbgModSegmentRva)
# define RTDbgModSegmentSize                            RT_MANGLER(RTDbgModSegmentSize)
# define RTDbgModSetTag                                 RT_MANGLER(RTDbgModSetTag)
# define RTDbgModSymbolAdd                              RT_MANGLER(RTDbgModSymbolAdd)
# define RTDbgModSymbolByAddr                           RT_MANGLER(RTDbgModSymbolByAddr)
# define RTDbgModSymbolByAddrA                          RT_MANGLER(RTDbgModSymbolByAddrA)
# define RTDbgModSymbolByName                           RT_MANGLER(RTDbgModSymbolByName)
# define RTDbgModSymbolByNameA                          RT_MANGLER(RTDbgModSymbolByNameA)
# define RTDbgModSymbolByOrdinal                        RT_MANGLER(RTDbgModSymbolByOrdinal)
# define RTDbgModSymbolByOrdinalA                       RT_MANGLER(RTDbgModSymbolByOrdinalA)
# define RTDbgModSymbolCount                            RT_MANGLER(RTDbgModSymbolCount)
# define RTDbgModUnwindFrame                            RT_MANGLER(RTDbgModUnwindFrame)
# define RTDbgStackDumpSelf                             RT_MANGLER(RTDbgStackDumpSelf)
# define RTDbgStackDumpSelf_EndProc                     RT_MANGLER(RTDbgStackDumpSelf_EndProc)
# define RTDbgSymbolAlloc                               RT_MANGLER(RTDbgSymbolAlloc)
# define RTDbgSymbolDup                                 RT_MANGLER(RTDbgSymbolDup)
# define RTDbgSymbolFree                                RT_MANGLER(RTDbgSymbolFree)
# define RTDirClose                                     RT_MANGLER(RTDirClose)
# define RTDirCreate                                    RT_MANGLER(RTDirCreate)
# define RTDirCreateFullPath                            RT_MANGLER(RTDirCreateFullPath)
# define RTDirCreateFullPathEx                          RT_MANGLER(RTDirCreateFullPathEx)
# define RTDirCreateTemp                                RT_MANGLER(RTDirCreateTemp)
# define RTDirCreateTempSecure                          RT_MANGLER(RTDirCreateTempSecure)
# define RTDirCreateUniqueNumbered                      RT_MANGLER(RTDirCreateUniqueNumbered)
# define RTDirEntryIsStdDotLink                         RT_MANGLER(RTDirEntryIsStdDotLink)
# define RTDirEntryExIsStdDotLink                       RT_MANGLER(RTDirEntryExIsStdDotLink)
# define RTDirExists                                    RT_MANGLER(RTDirExists)
# define RTDirFlush                                     RT_MANGLER(RTDirFlush)
# define RTDirFlushParent                               RT_MANGLER(RTDirFlushParent)
# define RTDirIsValid                                   RT_MANGLER(RTDirIsValid)
# define RTDirOpen                                      RT_MANGLER(RTDirOpen)
# define RTDirOpenFiltered                              RT_MANGLER(RTDirOpenFiltered)
# define RTDirQueryInfo                                 RT_MANGLER(RTDirQueryInfo)
# define RTDirQueryUnknownType                          RT_MANGLER(RTDirQueryUnknownType)
# define RTDirQueryUnknownTypeEx                        RT_MANGLER(RTDirQueryUnknownTypeEx)
# define RTDirRead                                      RT_MANGLER(RTDirRead)
# define RTDirReadEx                                    RT_MANGLER(RTDirReadEx)
# define RTDirReadExA                                   RT_MANGLER(RTDirReadExA)
# define RTDirReadExAFree                               RT_MANGLER(RTDirReadExAFree)
# define RTDirRemove                                    RT_MANGLER(RTDirRemove)
# define RTDirRemoveRecursive                           RT_MANGLER(RTDirRemoveRecursive)
# define RTDirRename                                    RT_MANGLER(RTDirRename)
# define RTDirRewind                                    RT_MANGLER(RTDirRewind)
# define RTDirSetMode                                   RT_MANGLER(RTDirSetMode)
# define RTDirSetTimes                                  RT_MANGLER(RTDirSetTimes)
# define RTDirRelFileOpen                               RT_MANGLER(RTDirRelFileOpen)
# define RTDirRelDirOpen                                RT_MANGLER(RTDirRelDirOpen)
# define RTDirRelDirOpenFiltered                        RT_MANGLER(RTDirRelDirOpenFiltered)
# define RTDirRelDirCreate                              RT_MANGLER(RTDirRelDirCreate)
# define RTDirRelDirRemove                              RT_MANGLER(RTDirRelDirRemove)
# define RTDirRelPathQueryInfo                          RT_MANGLER(RTDirRelPathQueryInfo)
# define RTDirRelPathSetMode                            RT_MANGLER(RTDirRelPathSetMode)
# define RTDirRelPathSetTimes                           RT_MANGLER(RTDirRelPathSetTimes)
# define RTDirRelPathSetOwner                           RT_MANGLER(RTDirRelPathSetOwner)
# define RTDirRelPathRename                             RT_MANGLER(RTDirRelPathRename)
# define RTDirRelPathUnlink                             RT_MANGLER(RTDirRelPathUnlink)
# define RTDirRelSymlinkCreate                          RT_MANGLER(RTDirRelSymlinkCreate)
# define RTDirRelSymlinkRead                            RT_MANGLER(RTDirRelSymlinkRead)
# define RTVfsDirOpenDir                                RT_MANGLER(RTVfsDirOpenDir)
# define RTVfsDirFromRTDir                              RT_MANGLER(RTVfsDirFromRTDir)
# define RTVfsDirOpenNormal                             RT_MANGLER(RTVfsDirOpenNormal)
# define RTVfsDirIsStdDir                               RT_MANGLER(RTVfsDirIsStdDir)
# define RTDvmCreate                                    RT_MANGLER(RTDvmCreate)
# define RTDvmCreateFromVfsFile                         RT_MANGLER(RTDvmCreateFromVfsFile)
# define RTDvmRetain                                    RT_MANGLER(RTDvmRetain)
# define RTDvmRelease                                   RT_MANGLER(RTDvmRelease)
# define RTDvmMapOpen                                   RT_MANGLER(RTDvmMapOpen)
# define RTDvmMapInitialize                             RT_MANGLER(RTDvmMapInitialize)
# define RTDvmMapGetFormatName                          RT_MANGLER(RTDvmMapGetFormatName)
# define RTDvmMapGetFormatType                          RT_MANGLER(RTDvmMapGetFormatType)
# define RTDvmMapGetValidVolumes                        RT_MANGLER(RTDvmMapGetValidVolumes)
# define RTDvmMapGetMaxVolumes                          RT_MANGLER(RTDvmMapGetMaxVolumes)
# define RTDvmMapQueryBlockStatus                       RT_MANGLER(RTDvmMapQueryBlockStatus)
# define RTDvmMapQueryFirstVolume                       RT_MANGLER(RTDvmMapQueryFirstVolume)
# define RTDvmMapQueryNextVolume                        RT_MANGLER(RTDvmMapQueryNextVolume)
# define RTDvmMapQueryDiskUuid                          RT_MANGLER(RTDvmMapQueryDiskUuid)
# define RTDvmMapQueryTableLocations                    RT_MANGLER(RTDvmMapQueryTableLocations)
# define RTDvmVolumeRetain                              RT_MANGLER(RTDvmVolumeRetain)
# define RTDvmVolumeRelease                             RT_MANGLER(RTDvmVolumeRelease)
# define RTDvmVolumeGetIndex                            RT_MANGLER(RTDvmVolumeGetIndex)
# define RTDvmVolumeGetPropU64                          RT_MANGLER(RTDvmVolumeGetPropU64)
# define RTDvmVolumeGetSize                             RT_MANGLER(RTDvmVolumeGetSize)
# define RTDvmVolumeQueryName                           RT_MANGLER(RTDvmVolumeQueryName)
# define RTDvmVolumeQueryProp                           RT_MANGLER(RTDvmVolumeQueryProp)
# define RTDvmVolumeQueryTableLocation                  RT_MANGLER(RTDvmVolumeQueryTableLocation)
# define RTDvmVolumeGetType                             RT_MANGLER(RTDvmVolumeGetType)
# define RTDvmVolumeGetFlags                            RT_MANGLER(RTDvmVolumeGetFlags)
# define RTDvmVolumeQueryRange                          RT_MANGLER(RTDvmVolumeQueryRange)
# define RTDvmVolumeRead                                RT_MANGLER(RTDvmVolumeRead)
# define RTDvmVolumeWrite                               RT_MANGLER(RTDvmVolumeWrite)
# define RTDvmVolumeSetQueryBlockStatusCallback         RT_MANGLER(RTDvmVolumeSetQueryBlockStatusCallback)
# define RTDvmVolumeTypeGetDescr                        RT_MANGLER(RTDvmVolumeTypeGetDescr)
# define RTDvmVolumeCreateVfsFile                       RT_MANGLER(RTDvmVolumeCreateVfsFile)
# define RTEfiGuidCompare                               RT_MANGLER(RTEfiGuidCompare)
# define RTEfiGuidFromUuid                              RT_MANGLER(RTEfiGuidFromUuid)
# define RTEfiGuidToUuid                                RT_MANGLER(RTEfiGuidToUuid)
# define RTEfiSigDbAddFromExistingDb                    RT_MANGLER(RTEfiSigDbAddFromExistingDb)
# define RTEfiSigDbAddSignatureFromFile                 RT_MANGLER(RTEfiSigDbAddSignatureFromFile)
# define RTEfiSigDbAddSignatureFromBuf                  RT_MANGLER(RTEfiSigDbAddSignatureFromBuf)
# define RTEfiSigDbCreate                               RT_MANGLER(RTEfiSigDbCreate)
# define RTEfiSigDbDestroy                              RT_MANGLER(RTEfiSigDbDestroy)
# define RTEfiSigDbEnum                                 RT_MANGLER(RTEfiSigDbEnum)
# define RTEfiSigDbTypeGetGuid                          RT_MANGLER(RTEfiSigDbTypeGetGuid)
# define RTEfiSigDbTypeStringify                        RT_MANGLER(RTEfiSigDbTypeStringify)
# define RTEfiSigDbWriteToFile                          RT_MANGLER(RTEfiSigDbWriteToFile)
# define RTEfiTimeFromTimeSpec                          RT_MANGLER(RTEfiTimeFromTimeSpec)
# define RTEfiTimeToTimeSpec                            RT_MANGLER(RTEfiTimeToTimeSpec)
# define RTEfiVarStoreCreate                            RT_MANGLER(RTEfiVarStoreCreate)
# define RTEfiVarStoreOpenAsVfs                         RT_MANGLER(RTEfiVarStoreOpenAsVfs)
# define RTEnvApplyChanges                              RT_MANGLER(RTEnvApplyChanges)
# define RTEnvClone                                     RT_MANGLER(RTEnvClone)
# define RTEnvCloneUtf16Block                           RT_MANGLER(RTEnvCloneUtf16Block)
# define RTEnvCountEx                                   RT_MANGLER(RTEnvCountEx)
# define RTEnvCreate                                    RT_MANGLER(RTEnvCreate)
# define RTEnvCreateEx                                  RT_MANGLER(RTEnvCreateEx)
# define RTEnvCreateChangeRecord                        RT_MANGLER(RTEnvCreateChangeRecord)
# define RTEnvCreateChangeRecordEx                      RT_MANGLER(RTEnvCreateChangeRecordEx)
# define RTEnvDestroy                                   RT_MANGLER(RTEnvDestroy)
# define RTEnvDup                                       RT_MANGLER(RTEnvDup)
# define RTEnvDupEx                                     RT_MANGLER(RTEnvDupEx)
# define RTEnvExist                                     RT_MANGLER(RTEnvExist)
# define RTEnvExistsBad                                 RT_MANGLER(RTEnvExistsBad)
# define RTEnvExistsUtf8                                RT_MANGLER(RTEnvExistsUtf8)
# define RTEnvExistEx                                   RT_MANGLER(RTEnvExistEx)
# define RTEnvFreeUtf8Block                             RT_MANGLER(RTEnvFreeUtf8Block)
# define RTEnvFreeUtf16Block                            RT_MANGLER(RTEnvFreeUtf16Block)
# define RTEnvGet                                       RT_MANGLER(RTEnvGet)
# define RTEnvGetBad                                    RT_MANGLER(RTEnvGetBad)
# define RTEnvGetByIndexEx                              RT_MANGLER(RTEnvGetByIndexEx)
# define RTEnvGetByIndexRawEx                           RT_MANGLER(RTEnvGetByIndexRawEx)
# define RTEnvGetUtf8                                   RT_MANGLER(RTEnvGetUtf8)
# define RTEnvGetEx                                     RT_MANGLER(RTEnvGetEx)
# define RTEnvGetExecEnvP                               RT_MANGLER(RTEnvGetExecEnvP)
# define RTEnvIsChangeRecord                            RT_MANGLER(RTEnvIsChangeRecord)
# define RTEnvPut                                       RT_MANGLER(RTEnvPut)
# define RTEnvPutBad                                    RT_MANGLER(RTEnvPutBad)
# define RTEnvPutUtf8                                   RT_MANGLER(RTEnvPutUtf8)
# define RTEnvPutEx                                     RT_MANGLER(RTEnvPutEx)
# define RTEnvQueryUtf16Block                           RT_MANGLER(RTEnvQueryUtf16Block)
# define RTEnvQueryUtf8Block                            RT_MANGLER(RTEnvQueryUtf8Block)
# define RTEnvReset                                     RT_MANGLER(RTEnvReset)
# define RTEnvSet                                       RT_MANGLER(RTEnvSet)
# define RTEnvSetBad                                    RT_MANGLER(RTEnvSetBad)
# define RTEnvSetUtf8                                   RT_MANGLER(RTEnvSetUtf8)
# define RTEnvSetEx                                     RT_MANGLER(RTEnvSetEx)
# define RTEnvUnset                                     RT_MANGLER(RTEnvUnset)
# define RTEnvUnsetBad                                  RT_MANGLER(RTEnvUnsetBad)
# define RTEnvUnsetUtf8                                 RT_MANGLER(RTEnvUnsetUtf8)
# define RTEnvUnsetEx                                   RT_MANGLER(RTEnvUnsetEx)
# define RTErrCOMGet                                    RT_MANGLER(RTErrCOMGet)
# define RTErrConvertFromErrno                          RT_MANGLER(RTErrConvertFromErrno)
# define RTErrConvertToErrno                            RT_MANGLER(RTErrConvertToErrno)
# define RTErrIsKnown                                   RT_MANGLER(RTErrIsKnown)
# define RTErrQueryDefine                               RT_MANGLER(RTErrQueryDefine)
# define RTErrQueryMsgShort                             RT_MANGLER(RTErrQueryMsgShort)
# define RTErrQueryMsgFull                              RT_MANGLER(RTErrQueryMsgFull)
# define RTErrFormatDefine                              RT_MANGLER(RTErrFormatDefine)
# define RTErrFormatMsgShort                            RT_MANGLER(RTErrFormatMsgShort)
# define RTErrFormatMsgFull                             RT_MANGLER(RTErrFormatMsgFull)
# define RTErrFormatMsgAll                              RT_MANGLER(RTErrFormatMsgAll)
# define RTErrInfoAlloc                                 RT_MANGLER(RTErrInfoAlloc)
# define RTErrInfoAllocEx                               RT_MANGLER(RTErrInfoAllocEx)
# define RTErrInfoFree                                  RT_MANGLER(RTErrInfoFree)
# define RTErrInfoSet                                   RT_MANGLER(RTErrInfoSet)
# define RTErrInfoSetF                                  RT_MANGLER(RTErrInfoSetF)
# define RTErrInfoSetV                                  RT_MANGLER(RTErrInfoSetV)
# define RTErrInfoLogAndSet                             RT_MANGLER(RTErrInfoLogAndSet)
# define RTErrInfoLogAndSetF                            RT_MANGLER(RTErrInfoLogAndSetF)
# define RTErrInfoLogAndSetV                            RT_MANGLER(RTErrInfoLogAndSetV)
# define RTErrInfoLogAndAdd                             RT_MANGLER(RTErrInfoLogAndAdd)
# define RTErrInfoLogAndAddF                            RT_MANGLER(RTErrInfoLogAndAddF)
# define RTErrInfoLogAndAddV                            RT_MANGLER(RTErrInfoLogAndAddV)
# define RTErrVarsAreEqual                              RT_MANGLER(RTErrVarsAreEqual)
# define RTErrVarsHaveChanged                           RT_MANGLER(RTErrVarsHaveChanged)
# define RTErrVarsRestore                               RT_MANGLER(RTErrVarsRestore)
# define RTErrVarsSave                                  RT_MANGLER(RTErrVarsSave)
# define RTFdtAddMemoryReservation                      RT_MANGLER(RTFdtAddMemoryReservation)
# define RTFdtCreateEmpty                               RT_MANGLER(RTFdtCreateEmpty)
# define RTFdtCreateFromFile                            RT_MANGLER(RTFdtCreateFromFile)
# define RTFdtCreateFromVfsIoStrm                       RT_MANGLER(RTFdtCreateFromVfsIoStrm)
# define RTFdtDestroy                                   RT_MANGLER(RTFdtDestroy)
# define RTFdtDumpToFile                                RT_MANGLER(RTFdtDumpToFile)
# define RTFdtDumpToVfsIoStrm                           RT_MANGLER(RTFdtDumpToVfsIoStrm)
# define RTFdtFinalize                                  RT_MANGLER(RTFdtFinalize)
# define RTFdtNodeAdd                                   RT_MANGLER(RTFdtNodeAdd)
# define RTFdtNodeAddF                                  RT_MANGLER(RTFdtNodeAddF)
# define RTFdtNodeAddV                                  RT_MANGLER(RTFdtNodeAddV)
# define RTFdtNodeFinalize                              RT_MANGLER(RTFdtNodeFinalize)
# define RTFdtNodePropertyAddCellsU32                   RT_MANGLER(RTFdtNodePropertyAddCellsU32)
# define RTFdtNodePropertyAddCellsU32AsArray            RT_MANGLER(RTFdtNodePropertyAddCellsU32AsArray)
# define RTFdtNodePropertyAddCellsU32V                  RT_MANGLER(RTFdtNodePropertyAddCellsU32V)
# define RTFdtNodePropertyAddCellsU64                   RT_MANGLER(RTFdtNodePropertyAddCellsU64)
# define RTFdtNodePropertyAddCellsU64V                  RT_MANGLER(RTFdtNodePropertyAddCellsU64V)
# define RTFdtNodePropertyAddEmpty                      RT_MANGLER(RTFdtNodePropertyAddEmpty)
# define RTFdtNodePropertyAddString                     RT_MANGLER(RTFdtNodePropertyAddString)
# define RTFdtNodePropertyAddStringList                 RT_MANGLER(RTFdtNodePropertyAddStringList)
# define RTFdtNodePropertyAddStringListV                RT_MANGLER(RTFdtNodePropertyAddStringListV)
# define RTFdtNodePropertyAddU32                        RT_MANGLER(RTFdtNodePropertyAddU32)
# define RTFdtNodePropertyAddU64                        RT_MANGLER(RTFdtNodePropertyAddU64)
# define RTFdtPHandleAllocate                           RT_MANGLER(RTFdtPHandleAllocate)
# define RTFdtSetPhysBootCpuId                          RT_MANGLER(RTFdtSetPhysBootCpuId)
# define RTFileAioCtxAssociateWithFile                  RT_MANGLER(RTFileAioCtxAssociateWithFile)
# define RTFileAioCtxCreate                             RT_MANGLER(RTFileAioCtxCreate)
# define RTFileAioCtxDestroy                            RT_MANGLER(RTFileAioCtxDestroy)
# define RTFileAioCtxGetMaxReqCount                     RT_MANGLER(RTFileAioCtxGetMaxReqCount)
# define RTFileAioCtxSubmit                             RT_MANGLER(RTFileAioCtxSubmit)
# define RTFileAioCtxWait                               RT_MANGLER(RTFileAioCtxWait)
# define RTFileAioCtxWakeup                             RT_MANGLER(RTFileAioCtxWakeup)
# define RTFileAioGetLimits                             RT_MANGLER(RTFileAioGetLimits)
# define RTFileAioReqCancel                             RT_MANGLER(RTFileAioReqCancel)
# define RTFileAioReqCreate                             RT_MANGLER(RTFileAioReqCreate)
# define RTFileAioReqDestroy                            RT_MANGLER(RTFileAioReqDestroy)
# define RTFileAioReqGetRC                              RT_MANGLER(RTFileAioReqGetRC)
# define RTFileAioReqGetUser                            RT_MANGLER(RTFileAioReqGetUser)
# define RTFileAioReqPrepareFlush                       RT_MANGLER(RTFileAioReqPrepareFlush)
# define RTFileAioReqPrepareRead                        RT_MANGLER(RTFileAioReqPrepareRead)
# define RTFileAioReqPrepareWrite                       RT_MANGLER(RTFileAioReqPrepareWrite)
# define RTFileChangeLock                               RT_MANGLER(RTFileChangeLock)
# define RTFileClose                                    RT_MANGLER(RTFileClose)
# define RTFileCompare                                  RT_MANGLER(RTFileCompare)
# define RTFileCompareByHandles                         RT_MANGLER(RTFileCompareByHandles)
# define RTFileCompareByHandlesEx                       RT_MANGLER(RTFileCompareByHandlesEx)
# define RTFileCompareEx                                RT_MANGLER(RTFileCompareEx)
# define RTFileCopy                                     RT_MANGLER(RTFileCopy)
# define RTFileCopyAttributes                           RT_MANGLER(RTFileCopyAttributes)
# define RTFileCopyByHandles                            RT_MANGLER(RTFileCopyByHandles)
# define RTFileCopyByHandlesEx                          RT_MANGLER(RTFileCopyByHandlesEx)
# define RTFileCopyEx                                   RT_MANGLER(RTFileCopyEx)
# define RTFileCopyPart                                 RT_MANGLER(RTFileCopyPart)
# define RTFileCopyPartCleanup                          RT_MANGLER(RTFileCopyPartCleanup)
# define RTFileCopyPartEx                               RT_MANGLER(RTFileCopyPartEx)
# define RTFileCopyPartPrep                             RT_MANGLER(RTFileCopyPartPrep)
# define RTFileCreateUnique                             RT_MANGLER(RTFileCreateUnique)
# define RTFileCreateTemp                               RT_MANGLER(RTFileCreateTemp)
# define RTFileCreateTempSecure                         RT_MANGLER(RTFileCreateTempSecure)
# define RTFileDelete                                   RT_MANGLER(RTFileDelete)
# define RTFileDup                                      RT_MANGLER(RTFileDup)
# define RTFileExists                                   RT_MANGLER(RTFileExists)
# define RTFileFlush                                    RT_MANGLER(RTFileFlush)
# define RTFileFromNative                               RT_MANGLER(RTFileFromNative)
# define RTFileGetMaxSize                               RT_MANGLER(RTFileGetMaxSize)
# define RTFileQueryMaxSizeEx                           RT_MANGLER(RTFileQueryMaxSizeEx)
# define RTFileQuerySizeByPath                          RT_MANGLER(RTFileQuerySizeByPath)
# define RTFileIoCtl                                    RT_MANGLER(RTFileIoCtl)
# define RTFileIsValid                                  RT_MANGLER(RTFileIsValid)
# define RTFileLock                                     RT_MANGLER(RTFileLock)
# define RTFileModeToFlags                              RT_MANGLER(RTFileModeToFlags)
# define RTFileModeToFlagsEx                            RT_MANGLER(RTFileModeToFlagsEx)
# define RTFileMove                                     RT_MANGLER(RTFileMove)
# define RTFileOpen                                     RT_MANGLER(RTFileOpen)
# define RTFileOpenBitBucket                            RT_MANGLER(RTFileOpenBitBucket)
# define RTFileOpenEx                                   RT_MANGLER(RTFileOpenEx)
# define RTFileOpenF                                    RT_MANGLER(RTFileOpenF)
# define RTFileOpenV                                    RT_MANGLER(RTFileOpenV)
# define RTFileOpenTemp                                 RT_MANGLER(RTFileOpenTemp)
# define RTFileQueryFsSizes                             RT_MANGLER(RTFileQueryFsSizes)
# define RTFileQueryInfo                                RT_MANGLER(RTFileQueryInfo)
# define RTFileQuerySectorSize                          RT_MANGLER(RTFileQuerySectorSize)
# define RTFileQuerySize                                RT_MANGLER(RTFileQuerySize)
# define RTFileRead                                     RT_MANGLER(RTFileRead)
# define RTFileReadAll                                  RT_MANGLER(RTFileReadAll)
# define RTFileReadAllByHandle                          RT_MANGLER(RTFileReadAllByHandle)
# define RTFileReadAllByHandleEx                        RT_MANGLER(RTFileReadAllByHandleEx)
# define RTFileReadAllEx                                RT_MANGLER(RTFileReadAllEx)
# define RTFileReadAllFree                              RT_MANGLER(RTFileReadAllFree)
# define RTFileReadAt                                   RT_MANGLER(RTFileReadAt)
# define RTFileRename                                   RT_MANGLER(RTFileRename)
# define RTFileSeek                                     RT_MANGLER(RTFileSeek)
# define RTFileSetAllocationSize                        RT_MANGLER(RTFileSetAllocationSize)
# define RTFileSetForceFlags                            RT_MANGLER(RTFileSetForceFlags)
# define RTFileSetMode                                  RT_MANGLER(RTFileSetMode)
# define RTFileSetOwner                                 RT_MANGLER(RTFileSetOwner)
# define RTFileSetSize                                  RT_MANGLER(RTFileSetSize)
# define RTFileSetTimes                                 RT_MANGLER(RTFileSetTimes)
# define RTFileSgRead                                   RT_MANGLER(RTFileSgRead)
# define RTFileSgReadAt                                 RT_MANGLER(RTFileSgReadAt)
# define RTFileSgWrite                                  RT_MANGLER(RTFileSgWrite)
# define RTFileSgWriteAt                                RT_MANGLER(RTFileSgWriteAt)
# define RTFileTell                                     RT_MANGLER(RTFileTell)
# define RTFileToNative                                 RT_MANGLER(RTFileToNative)
# define RTFileUnlock                                   RT_MANGLER(RTFileUnlock)
# define RTFileWrite                                    RT_MANGLER(RTFileWrite)
# define RTFileWriteAt                                  RT_MANGLER(RTFileWriteAt)
# define RTFilesystemVfsFromFile                        RT_MANGLER(RTFilesystemVfsFromFile)
# define RTFsIsCaseSensitive                            RT_MANGLER(RTFsIsCaseSensitive)
# define RTFsMountpointsEnum                            RT_MANGLER(RTFsMountpointsEnum)
# define RTFsQueryProperties                            RT_MANGLER(RTFsQueryProperties)
# define RTFsQuerySerial                                RT_MANGLER(RTFsQuerySerial)
# define RTFsQuerySizes                                 RT_MANGLER(RTFsQuerySizes)
# define RTFsQueryType                                  RT_MANGLER(RTFsQueryType)
# define RTFsTypeName                                   RT_MANGLER(RTFsTypeName)
# define RTFsExtVolOpen                                 RT_MANGLER(RTFsExtVolOpen)
# define RTFsFatVolOpen                                 RT_MANGLER(RTFsFatVolOpen)
# define RTFsFatVolFormat                               RT_MANGLER(RTFsFatVolFormat)
# define RTFsFatVolFormat144                            RT_MANGLER(RTFsFatVolFormat144)
# define RTFsFatVolFormat288                            RT_MANGLER(RTFsFatVolFormat288)
# define RTFsCmdLs                                      RT_MANGLER(RTFsCmdLs)
# define RTFsIso9660VolOpen                             RT_MANGLER(RTFsIso9660VolOpen)
# define RTFsIsoMakerCreate                             RT_MANGLER(RTFsIsoMakerCreate)
# define RTFsIsoMakerRetain                             RT_MANGLER(RTFsIsoMakerRetain)
# define RTFsIsoMakerRelease                            RT_MANGLER(RTFsIsoMakerRelease)
# define RTFsIsoMakerBootCatSetFile                     RT_MANGLER(RTFsIsoMakerBootCatSetFile)
# define RTFsIsoMakerBootCatSetValidationEntry          RT_MANGLER(RTFsIsoMakerBootCatSetValidationEntry)
# define RTFsIsoMakerBootCatSetSectionEntry             RT_MANGLER(RTFsIsoMakerBootCatSetSectionEntry)
# define RTFsIsoMakerBootCatSetSectionHeaderEntry       RT_MANGLER(RTFsIsoMakerBootCatSetSectionHeaderEntry)
# define RTFsIsoMakerQueryObjIdxForBootCatalog          RT_MANGLER(RTFsIsoMakerQueryObjIdxForBootCatalog)
# define RTFsIsoMakerGetPopulatedNamespaces             RT_MANGLER(RTFsIsoMakerGetPopulatedNamespaces)
# define RTFsIsoMakerGetIso9660Level                    RT_MANGLER(RTFsIsoMakerGetIso9660Level)
# define RTFsIsoMakerGetRockRidgeLevel                  RT_MANGLER(RTFsIsoMakerGetRockRidgeLevel)
# define RTFsIsoMakerGetJolietRockRidgeLevel            RT_MANGLER(RTFsIsoMakerGetJolietRockRidgeLevel)
# define RTFsIsoMakerSetImagePadding                    RT_MANGLER(RTFsIsoMakerSetImagePadding)
# define RTFsIsoMakerSetIso9660Level                    RT_MANGLER(RTFsIsoMakerSetIso9660Level)
# define RTFsIsoMakerSetJolietUcs2Level                 RT_MANGLER(RTFsIsoMakerSetJolietUcs2Level)
# define RTFsIsoMakerSetRockRidgeLevel                  RT_MANGLER(RTFsIsoMakerSetRockRidgeLevel)
# define RTFsIsoMakerSetJolietRockRidgeLevel            RT_MANGLER(RTFsIsoMakerSetJolietRockRidgeLevel)
# define RTFsIsoMakerSetAttribInheritStyle              RT_MANGLER(RTFsIsoMakerSetAttribInheritStyle)
# define RTFsIsoMakerSetDefaultDirMode                  RT_MANGLER(RTFsIsoMakerSetDefaultDirMode)
# define RTFsIsoMakerSetDefaultFileMode                 RT_MANGLER(RTFsIsoMakerSetDefaultFileMode)
# define RTFsIsoMakerSetForcedDirMode                   RT_MANGLER(RTFsIsoMakerSetForcedDirMode)
# define RTFsIsoMakerSetForcedFileMode                  RT_MANGLER(RTFsIsoMakerSetForcedFileMode)
# define RTFsIsoMakerSetPathGroupId                     RT_MANGLER(RTFsIsoMakerSetPathGroupId)
# define RTFsIsoMakerSetPathMode                        RT_MANGLER(RTFsIsoMakerSetPathMode)
# define RTFsIsoMakerSetPathOwnerId                     RT_MANGLER(RTFsIsoMakerSetPathOwnerId)
# define RTFsIsoMakerSetSysAreaContent                  RT_MANGLER(RTFsIsoMakerSetSysAreaContent)
# define RTFsIsoMakerSetStringProp                      RT_MANGLER(RTFsIsoMakerSetStringProp)
# define RTFsIsoMakerGetObjIdxForPath                   RT_MANGLER(RTFsIsoMakerGetObjIdxForPath)
# define RTFsIsoMakerObjEnableBootInfoTablePatching     RT_MANGLER(RTFsIsoMakerObjEnableBootInfoTablePatching)
# define RTFsIsoMakerObjQueryDataSize                   RT_MANGLER(RTFsIsoMakerObjQueryDataSize)
# define RTFsIsoMakerObjRemove                          RT_MANGLER(RTFsIsoMakerObjRemove)
# define RTFsIsoMakerObjSetPath                         RT_MANGLER(RTFsIsoMakerObjSetPath)
# define RTFsIsoMakerObjSetNameAndParent                RT_MANGLER(RTFsIsoMakerObjSetNameAndParent)
# define RTFsIsoMakerObjSetRockName                     RT_MANGLER(RTFsIsoMakerObjSetRockName)
# define RTFsIsoMakerAddUnnamedDir                      RT_MANGLER(RTFsIsoMakerAddUnnamedDir)
# define RTFsIsoMakerAddDir                             RT_MANGLER(RTFsIsoMakerAddDir)
# define RTFsIsoMakerAddFileWithSrcPath                 RT_MANGLER(RTFsIsoMakerAddFileWithSrcPath)
# define RTFsIsoMakerAddFileWithVfsFile                 RT_MANGLER(RTFsIsoMakerAddFileWithVfsFile)
# define RTFsIsoMakerAddUnnamedFileWithSrcPath          RT_MANGLER(RTFsIsoMakerAddUnnamedFileWithSrcPath)
# define RTFsIsoMakerAddUnnamedFileWithVfsFile          RT_MANGLER(RTFsIsoMakerAddUnnamedFileWithVfsFile)
# define RTFsIsoMakerAddUnnamedFileWithCommonSrc        RT_MANGLER(RTFsIsoMakerAddUnnamedFileWithCommonSrc)
# define RTFsIsoMakerAddSymlink                         RT_MANGLER(RTFsIsoMakerAddSymlink)
# define RTFsIsoMakerAddUnnamedSymlink                  RT_MANGLER(RTFsIsoMakerAddUnnamedSymlink)
# define RTFsIsoMakerAddCommonSourceFile                RT_MANGLER(RTFsIsoMakerAddCommonSourceFile)
# define RTFsIsoMakerImport                             RT_MANGLER(RTFsIsoMakerImport)
# define RTFsIsoMakerFinalize                           RT_MANGLER(RTFsIsoMakerFinalize)
# define RTFsIsoMakerCreateVfsOutputFile                RT_MANGLER(RTFsIsoMakerCreateVfsOutputFile)
# define RTFsIsoMakerCmd                                RT_MANGLER(RTFsIsoMakerCmd)
# define RTFsIsoMakerCmdEx                              RT_MANGLER(RTFsIsoMakerCmdEx)
# define RTFsNtfsVolOpen                                RT_MANGLER(RTFsNtfsVolOpen)
# define RTFsPdbVolOpen                                 RT_MANGLER(RTFsPdbVolOpen)
# define RTFtpServerCreate                              RT_MANGLER(RTFtpServerCreate)
# define RTFtpServerDestroy                             RT_MANGLER(RTFtpServerDestroy)
# define RTFuzzCmdMaster                                RT_MANGLER(RTFuzzCmdMaster)
# define RTFuzzCfgCreateFromFile                        RT_MANGLER(RTFuzzCfgCreateFromFile)
# define RTFuzzCfgCreateFromVfsFile                     RT_MANGLER(RTFuzzCfgCreateFromVfsFile)
# define RTFuzzCfgRetain                                RT_MANGLER(RTFuzzCfgRetain)
# define RTFuzzCfgRelease                               RT_MANGLER(RTFuzzCfgRelease)
# define RTFuzzCfgImport                                RT_MANGLER(RTFuzzCfgImport)
# define RTFuzzCfgQueryCustomCfg                        RT_MANGLER(RTFuzzCfgQueryCustomCfg)
# define RTFuzzCtxCfgGetBehavioralFlags                 RT_MANGLER(RTFuzzCtxCfgGetBehavioralFlags)
# define RTFuzzCtxCfgGetInputSeedMaximum                RT_MANGLER(RTFuzzCtxCfgGetInputSeedMaximum)
# define RTFuzzCtxCfgGetTmpDirectory                    RT_MANGLER(RTFuzzCtxCfgGetTmpDirectory)
# define RTFuzzCtxCfgSetBehavioralFlags                 RT_MANGLER(RTFuzzCtxCfgSetBehavioralFlags)
# define RTFuzzCtxCfgSetInputSeedMaximum                RT_MANGLER(RTFuzzCtxCfgSetInputSeedMaximum)
# define RTFuzzCtxCfgSetMutationRange                   RT_MANGLER(RTFuzzCtxCfgSetMutationRange)
# define RTFuzzCtxCfgSetTmpDirectory                    RT_MANGLER(RTFuzzCtxCfgSetTmpDirectory)
# define RTFuzzCtxCorpusInputAdd                        RT_MANGLER(RTFuzzCtxCorpusInputAdd)
# define RTFuzzCtxCorpusInputAddEx                        RT_MANGLER(RTFuzzCtxCorpusInputAddEx)
# define RTFuzzCtxCorpusInputAddFromDirPath             RT_MANGLER(RTFuzzCtxCorpusInputAddFromDirPath)
# define RTFuzzCtxCorpusInputAddFromFile                RT_MANGLER(RTFuzzCtxCorpusInputAddFromFile)
# define RTFuzzCtxCorpusInputAddFromFileEx              RT_MANGLER(RTFuzzCtxCorpusInputAddFromFileEx)
# define RTFuzzCtxCorpusInputAddFromVfsFile             RT_MANGLER(RTFuzzCtxCorpusInputAddFromVfsFile)
# define RTFuzzCtxCorpusInputAddFromVfsFileEx           RT_MANGLER(RTFuzzCtxCorpusInputAddFromVfsFileEx)
# define RTFuzzCtxCorpusInputAddFromVfsIoStrm           RT_MANGLER(RTFuzzCtxCorpusInputAddFromVfsIoStrm)
# define RTFuzzCtxCorpusInputAddFromVfsIoStrmEx         RT_MANGLER(RTFuzzCtxCorpusInputAddFromVfsIoStrmEx)
# define RTFuzzCtxCreate                                RT_MANGLER(RTFuzzCtxCreate)
# define RTFuzzCtxCreateFromState                       RT_MANGLER(RTFuzzCtxCreateFromState)
# define RTFuzzCtxCreateFromStateFile                   RT_MANGLER(RTFuzzCtxCreateFromStateFile)
# define RTFuzzCtxCreateFromStateMem                    RT_MANGLER(RTFuzzCtxCreateFromStateMem)
# define RTFuzzCtxInputGenerate                         RT_MANGLER(RTFuzzCtxInputGenerate)
# define RTFuzzCtxQueryStats                            RT_MANGLER(RTFuzzCtxQueryStats)
# define RTFuzzCtxRelease                               RT_MANGLER(RTFuzzCtxRelease)
# define RTFuzzCtxReseed                                RT_MANGLER(RTFuzzCtxReseed)
# define RTFuzzCtxRetain                                RT_MANGLER(RTFuzzCtxRetain)
# define RTFuzzCtxStateExport                           RT_MANGLER(RTFuzzCtxStateExport)
# define RTFuzzCtxStateExportToFile                     RT_MANGLER(RTFuzzCtxStateExportToFile)
# define RTFuzzCtxStateExportToMem                      RT_MANGLER(RTFuzzCtxStateExportToMem)
# define RTFuzzInputAddToCtxCorpus                      RT_MANGLER(RTFuzzInputAddToCtxCorpus)
# define RTFuzzInputMutateStreamData                    RT_MANGLER(RTFuzzInputMutateStreamData)
# define RTFuzzInputQueryBlobData                       RT_MANGLER(RTFuzzInputQueryBlobData)
# define RTFuzzInputQueryDigestString                   RT_MANGLER(RTFuzzInputQueryDigestString)
# define RTFuzzInputRelease                             RT_MANGLER(RTFuzzInputRelease)
# define RTFuzzInputRemoveFromCtxCorpus                 RT_MANGLER(RTFuzzInputRemoveFromCtxCorpus)
# define RTFuzzInputRetain                              RT_MANGLER(RTFuzzInputRetain)
# define RTFuzzInputWriteToFile                         RT_MANGLER(RTFuzzInputWriteToFile)
# define RTFuzzObsCreate                                RT_MANGLER(RTFuzzObsCreate)
# define RTFuzzObsDestroy                               RT_MANGLER(RTFuzzObsDestroy)
# define RTFuzzObsExecStart                             RT_MANGLER(RTFuzzObsExecStart)
# define RTFuzzObsExecStop                              RT_MANGLER(RTFuzzObsExecStop)
# define RTFuzzObsQueryCtx                              RT_MANGLER(RTFuzzObsQueryCtx)
# define RTFuzzObsQueryStats                            RT_MANGLER(RTFuzzObsQueryStats)
# define RTFuzzObsSetResultDirectory                    RT_MANGLER(RTFuzzObsSetResultDirectory)
# define RTFuzzObsSetTestBinary                         RT_MANGLER(RTFuzzObsSetTestBinary)
# define RTFuzzObsSetTestBinaryArgs                     RT_MANGLER(RTFuzzObsSetTestBinaryArgs)
# define RTFuzzObsSetTestBinaryEnv                      RT_MANGLER(RTFuzzObsSetTestBinaryEnv)
# define RTFuzzObsSetTestBinarySanitizers               RT_MANGLER(RTFuzzObsSetTestBinarySanitizers)
# define RTFuzzObsSetTestBinaryTimeout                  RT_MANGLER(RTFuzzObsSetTestBinaryTimeout)
# define RTFuzzObsSetTmpDirectory                       RT_MANGLER(RTFuzzObsSetTmpDirectory)
# define RTFuzzTgtRecorderCreate                        RT_MANGLER(RTFuzzTgtRecorderCreate)
# define RTFuzzTgtRecorderCreateNewState                RT_MANGLER(RTFuzzTgtRecorderCreateNewState)
# define RTFuzzTgtRecorderRelease                       RT_MANGLER(RTFuzzTgtRecorderRelease)
# define RTFuzzTgtRecorderRetain                        RT_MANGLER(RTFuzzTgtRecorderRetain)
# define RTFuzzTgtStateAddProcSts                       RT_MANGLER(RTFuzzTgtStateAddProcSts)
# define RTFuzzTgtStateAddSanCovReportFromFile          RT_MANGLER(RTFuzzTgtStateAddSanCovReportFromFile)
# define RTFuzzTgtStateAddToRecorder                    RT_MANGLER(RTFuzzTgtStateAddToRecorder)
# define RTFuzzTgtStateAppendStderrFromBuf              RT_MANGLER(RTFuzzTgtStateAppendStderrFromBuf)
# define RTFuzzTgtStateAppendStderrFromPipe             RT_MANGLER(RTFuzzTgtStateAppendStderrFromPipe)
# define RTFuzzTgtStateAppendStdoutFromBuf              RT_MANGLER(RTFuzzTgtStateAppendStdoutFromBuf)
# define RTFuzzTgtStateAppendStdoutFromPipe             RT_MANGLER(RTFuzzTgtStateAppendStdoutFromPipe)
# define RTFuzzTgtStateDumpToDir                        RT_MANGLER(RTFuzzTgtStateDumpToDir)
# define RTFuzzTgtStateFinalize                         RT_MANGLER(RTFuzzTgtStateFinalize)
# define RTFuzzTgtStateRelease                          RT_MANGLER(RTFuzzTgtStateRelease)
# define RTFuzzTgtStateReset                            RT_MANGLER(RTFuzzTgtStateReset)
# define RTFuzzTgtStateRetain                           RT_MANGLER(RTFuzzTgtStateRetain)
# define RTGetOpt                                       RT_MANGLER(RTGetOpt)
# define RTGetOptArgvFree                               RT_MANGLER(RTGetOptArgvFree)
# define RTGetOptArgvFreeEx                             RT_MANGLER(RTGetOptArgvFreeEx)
# define RTGetOptArgvFromString                         RT_MANGLER(RTGetOptArgvFromString)
# define RTGetOptArgvToString                           RT_MANGLER(RTGetOptArgvToString)
# define RTGetOptArgvToUtf16String                      RT_MANGLER(RTGetOptArgvToUtf16String)
# define RTGetOptFetchValue                             RT_MANGLER(RTGetOptFetchValue)
# define RTGetOptInit                                   RT_MANGLER(RTGetOptInit)
# define RTGetOptNonOptionArrayPtr                      RT_MANGLER(RTGetOptNonOptionArrayPtr)
# define RTGetOptFormatError                            RT_MANGLER(RTGetOptFormatError)
# define RTGetOptPrintError                             RT_MANGLER(RTGetOptPrintError)
# define RTHandleClose                                  RT_MANGLER(RTHandleClose)
# define RTHandleGetStandard                            RT_MANGLER(RTHandleGetStandard)
# define RTHandleTableAlloc                             RT_MANGLER(RTHandleTableAlloc)
# define RTHandleTableAllocWithCtx                      RT_MANGLER(RTHandleTableAllocWithCtx)
# define RTHandleTableCreate                            RT_MANGLER(RTHandleTableCreate)
# define RTHandleTableCreateEx                          RT_MANGLER(RTHandleTableCreateEx)
# define RTHandleTableDestroy                           RT_MANGLER(RTHandleTableDestroy)
# define RTHandleTableFree                              RT_MANGLER(RTHandleTableFree)
# define RTHandleTableFreeWithCtx                       RT_MANGLER(RTHandleTableFreeWithCtx)
# define RTHandleTableLookup                            RT_MANGLER(RTHandleTableLookup)
# define RTHandleTableLookupWithCtx                     RT_MANGLER(RTHandleTableLookupWithCtx)
# define RTHeapOffsetAlloc                              RT_MANGLER(RTHeapOffsetAlloc)
# define RTHeapOffsetAllocZ                             RT_MANGLER(RTHeapOffsetAllocZ)
# define RTHeapOffsetDump                               RT_MANGLER(RTHeapOffsetDump)
# define RTHeapOffsetFree                               RT_MANGLER(RTHeapOffsetFree)
# define RTHeapOffsetGetFreeSize                        RT_MANGLER(RTHeapOffsetGetFreeSize)
# define RTHeapOffsetGetHeapSize                        RT_MANGLER(RTHeapOffsetGetHeapSize)
# define RTHeapOffsetInit                               RT_MANGLER(RTHeapOffsetInit)
# define RTHeapOffsetSize                               RT_MANGLER(RTHeapOffsetSize)
# define RTHeapSimpleAlloc                              RT_MANGLER(RTHeapSimpleAlloc)
# define RTHeapSimpleAllocZ                             RT_MANGLER(RTHeapSimpleAllocZ)
# define RTHeapSimpleDump                               RT_MANGLER(RTHeapSimpleDump)
# define RTHeapSimpleFree                               RT_MANGLER(RTHeapSimpleFree)
# define RTHeapSimpleGetFreeSize                        RT_MANGLER(RTHeapSimpleGetFreeSize)
# define RTHeapSimpleGetHeapSize                        RT_MANGLER(RTHeapSimpleGetHeapSize)
# define RTHeapSimpleInit                               RT_MANGLER(RTHeapSimpleInit)
# define RTHeapSimpleRelocate                           RT_MANGLER(RTHeapSimpleRelocate)
# define RTHeapSimpleSize                               RT_MANGLER(RTHeapSimpleSize)
# define RTHttpGetFile                                  RT_MANGLER(RTHttpGetFile)
# define RTHttpGetFollowRedirects                       RT_MANGLER(RTHttpGetFollowRedirects)
# define RTHttpSetFollowRedirects                       RT_MANGLER(RTHttpSetFollowRedirects)
# define RTHttpGetVerifyPeer                            RT_MANGLER(RTHttpGetVerifyPeer)
# define RTHttpHeaderListInit                           RT_MANGLER(RTHttpHeaderListInit)
# define RTHttpHeaderListDestroy                        RT_MANGLER(RTHttpHeaderListDestroy)
# define RTHttpHeaderListSet                            RT_MANGLER(RTHttpHeaderListSet)
# define RTHttpHeaderListAddRaw                         RT_MANGLER(RTHttpHeaderListAddRaw)
# define RTHttpHeaderListAdd                            RT_MANGLER(RTHttpHeaderListAdd)
# define RTHttpHeaderListGet                            RT_MANGLER(RTHttpHeaderListGet)
# define RTHttpHeaderListGetCount                       RT_MANGLER(RTHttpHeaderListGetCount)
# define RTHttpHeaderListGetByOrdinal                   RT_MANGLER(RTHttpHeaderListGetByOrdinal)
# define RTHttpMethodToStr                              RT_MANGLER(RTHttpMethodToStr)
# define RTHttpSetVerifyPeer                            RT_MANGLER(RTHttpSetVerifyPeer)
# define RTHttpUseSystemProxySettings                   RT_MANGLER(RTHttpUseSystemProxySettings)
# define RTHttpServerCreate                             RT_MANGLER(RTHttpServerCreate)
# define RTHttpServerDestroy                            RT_MANGLER(RTHttpServerDestroy)
# define RTHttpServerResponseInitEx                     RT_MANGLER(RTHttpServerResponseInitEx)
# define RTHttpServerResponseInit                       RT_MANGLER(RTHttpServerResponseInit)
# define RTHttpServerResponseDestroy                    RT_MANGLER(RTHttpServerResponseDestroy)
# define RTHttpStatusToStr                              RT_MANGLER(RTHttpStatusToStr)
# define RTIniFileCreateFromVfsFile                     RT_MANGLER(RTIniFileCreateFromVfsFile)
# define RTIniFileRetain                                RT_MANGLER(RTIniFileRetain)
# define RTIniFileRelease                               RT_MANGLER(RTIniFileRelease)
# define RTIniFileQueryPair                             RT_MANGLER(RTIniFileQueryPair)
# define RTIniFileQueryValue                            RT_MANGLER(RTIniFileQueryValue)
# define RTIoQueueCommit                                RT_MANGLER(RTIoQueueCommit)
# define RTIoQueueCreate                                RT_MANGLER(RTIoQueueCreate)
# define RTIoQueueDestroy                               RT_MANGLER(RTIoQueueDestroy)
# define RTIoQueueEvtWait                               RT_MANGLER(RTIoQueueEvtWait)
# define RTIoQueueEvtWaitWakeup                         RT_MANGLER(RTIoQueueEvtWaitWakeup)
# define RTIoQueueHandleDeregister                      RT_MANGLER(RTIoQueueHandleDeregister)
# define RTIoQueueHandleRegister                        RT_MANGLER(RTIoQueueHandleRegister)
# define RTIoQueueProviderGetBestForHndType             RT_MANGLER(RTIoQueueProviderGetBestForHndType)
# define RTIoQueueProviderGetById                       RT_MANGLER(RTIoQueueProviderGetById)
# define RTIoQueueRequestPrepare                        RT_MANGLER(RTIoQueueRequestPrepare)
# define RTIoQueueRequestPrepareSg                      RT_MANGLER(RTIoQueueRequestPrepareSg)
# define RTJsonIteratorBegin                            RT_MANGLER(RTJsonIteratorBegin)
# define RTJsonIteratorBeginArray                       RT_MANGLER(RTJsonIteratorBeginArray)
# define RTJsonIteratorBeginObject                      RT_MANGLER(RTJsonIteratorBeginObject)
# define RTJsonIteratorFree                             RT_MANGLER(RTJsonIteratorFree)
# define RTJsonIteratorNext                             RT_MANGLER(RTJsonIteratorNext)
# define RTJsonIteratorQueryValue                       RT_MANGLER(RTJsonIteratorQueryValue)
# define RTJsonParseFromBuf                             RT_MANGLER(RTJsonParseFromBuf)
# define RTJsonParseFromFile                            RT_MANGLER(RTJsonParseFromFile)
# define RTJsonParseFromString                          RT_MANGLER(RTJsonParseFromString)
# define RTJsonParseFromVfsFile                         RT_MANGLER(RTJsonParseFromVfsFile)
# define RTJsonValueGetArraySize                        RT_MANGLER(RTJsonValueGetArraySize)
# define RTJsonValueGetString                           RT_MANGLER(RTJsonValueGetString)
# define RTJsonValueGetType                             RT_MANGLER(RTJsonValueGetType)
# define RTJsonValueQueryArraySizeEx                    RT_MANGLER(RTJsonValueQueryArraySize)
# define RTJsonValueQueryBooleanByName                  RT_MANGLER(RTJsonValueQueryBooleanByName)
# define RTJsonValueQueryByIndex                        RT_MANGLER(RTJsonValueQueryByIndex)
# define RTJsonValueQueryByName                         RT_MANGLER(RTJsonValueQueryByName)
# define RTJsonValueQueryInteger                        RT_MANGLER(RTJsonValueQueryInteger)
# define RTJsonValueQueryIntegerByName                  RT_MANGLER(RTJsonValueQueryIntegerByName)
# define RTJsonValueQueryNumber                         RT_MANGLER(RTJsonValueQueryNumber)
# define RTJsonValueQueryNumberByName                   RT_MANGLER(RTJsonValueQueryNumberByName)
# define RTJsonValueQueryString                         RT_MANGLER(RTJsonValueQueryString)
# define RTJsonValueQueryStringByName                   RT_MANGLER(RTJsonValueQueryStringByName)
# define RTJsonValueRelease                             RT_MANGLER(RTJsonValueRelease)
# define RTJsonValueRetain                              RT_MANGLER(RTJsonValueRetain)
# define RTJsonValueTypeName                            RT_MANGLER(RTJsonValueTypeName)
# define RTKrnlModInfoGetFilePath                       RT_MANGLER(RTKrnlModInfoGetFilePath)
# define RTKrnlModInfoGetLoadAddr                       RT_MANGLER(RTKrnlModInfoGetLoadAddr)
# define RTKrnlModInfoGetName                           RT_MANGLER(RTKrnlModInfoGetName)
# define RTKrnlModInfoGetRefCnt                         RT_MANGLER(RTKrnlModInfoGetRefCnt)
# define RTKrnlModInfoGetSize                           RT_MANGLER(RTKrnlModInfoGetSize)
# define RTKrnlModInfoQueryRefModInfo                   RT_MANGLER(RTKrnlModInfoQueryRefModInfo)
# define RTKrnlModInfoRetain                            RT_MANGLER(RTKrnlModInfoRetain)
# define RTKrnlModInfoRelease                           RT_MANGLER(RTKrnlModInfoRelease)
# define RTKrnlModLoadByName                            RT_MANGLER(RTKrnlModLoadByName)
# define RTKrnlModLoadByPath                            RT_MANGLER(RTKrnlModLoadByPath)
# define RTKrnlModLoadedGetCount                        RT_MANGLER(RTKrnlModLoadedGetCount)
# define RTKrnlModLoadedQueryInfo                       RT_MANGLER(RTKrnlModLoadedQueryInfo)
# define RTKrnlModLoadedQueryInfoAll                    RT_MANGLER(RTKrnlModLoadedQueryInfoAll)
# define RTKrnlModQueryLoaded                           RT_MANGLER(RTKrnlModQueryLoaded)
# define RTKrnlModUnloadByName                          RT_MANGLER(RTKrnlModUnloadByName)
# define RTLatin1CalcUtf16Len                           RT_MANGLER(RTLatin1CalcUtf16Len)
# define RTLatin1CalcUtf16LenEx                         RT_MANGLER(RTLatin1CalcUtf16LenEx)
# define RTLatin1CalcUtf8Len                            RT_MANGLER(RTLatin1CalcUtf8Len)
# define RTLatin1CalcUtf8LenEx                          RT_MANGLER(RTLatin1CalcUtf8LenEx)
# define RTLatin1ToUtf16ExTag                           RT_MANGLER(RTLatin1ToUtf16ExTag)
# define RTLatin1ToUtf16Tag                             RT_MANGLER(RTLatin1ToUtf16Tag)
# define RTLatin1ToUtf8ExTag                            RT_MANGLER(RTLatin1ToUtf8ExTag)
# define RTLatin1ToUtf8Tag                              RT_MANGLER(RTLatin1ToUtf8Tag)
# define RTLdrArchName                                  RT_MANGLER(RTLdrArchName)
# define RTLdrClose                                     RT_MANGLER(RTLdrClose)
# define RTLdrEnumDbgInfo                               RT_MANGLER(RTLdrEnumDbgInfo)
# define RTLdrEnumSegments                              RT_MANGLER(RTLdrEnumSegments)
# define RTLdrEnumSymbols                               RT_MANGLER(RTLdrEnumSymbols)
# define RTLdrGetArch                                   RT_MANGLER(RTLdrGetArch)
# define RTLdrGetBits                                   RT_MANGLER(RTLdrGetBits)
# define RTLdrGetEndian                                 RT_MANGLER(RTLdrGetEndian)
# define RTLdrGetFormat                                 RT_MANGLER(RTLdrGetFormat)
# define RTLdrGetFunction                               RT_MANGLER(RTLdrGetFunction)
# define RTLdrGetHostArch                               RT_MANGLER(RTLdrGetHostArch)
# define RTLdrGetNativeHandle                           RT_MANGLER(RTLdrGetNativeHandle)
# define RTLdrGetSuff                                   RT_MANGLER(RTLdrGetSuff)
# define RTLdrGetSymbol                                 RT_MANGLER(RTLdrGetSymbol)
# define RTLdrGetSymbolEx                               RT_MANGLER(RTLdrGetSymbolEx)
# define RTLdrGetSystemSymbol                           RT_MANGLER(RTLdrGetSystemSymbol)
# define RTLdrGetSystemSymbolEx                         RT_MANGLER(RTLdrGetSystemSymbolEx)
# define RTLdrGetType                                   RT_MANGLER(RTLdrGetType)
# define RTLdrIsLoadable                                RT_MANGLER(RTLdrIsLoadable)
# define RTLdrLinkAddressToRva                          RT_MANGLER(RTLdrLinkAddressToRva)
# define RTLdrLinkAddressToSegOffset                    RT_MANGLER(RTLdrLinkAddressToSegOffset)
# define RTLdrLoad                                      RT_MANGLER(RTLdrLoad)
# define RTLdrLoadAppPriv                               RT_MANGLER(RTLdrLoadAppPriv)
# define RTLdrLoadEx                                    RT_MANGLER(RTLdrLoadEx)
# define RTLdrLoadSystem                                RT_MANGLER(RTLdrLoadSystem)
# define RTLdrLoadSystemEx                              RT_MANGLER(RTLdrLoadSystemEx)
# define RTLdrLxSetSegmentSelectors                     RT_MANGLER(RTLdrLxSetSegmentSelectors)
# define RTLdrOpen                                      RT_MANGLER(RTLdrOpen)
# define RTLdrOpenEx                                    RT_MANGLER(RTLdrOpenEx)
# define RTLdrOpenInMemory                              RT_MANGLER(RTLdrOpenInMemory)
# define RTLdrOpenVfsChain                              RT_MANGLER(RTLdrOpenVfsChain)
# define RTLdrRelocate                                  RT_MANGLER(RTLdrRelocate)
# define RTLdrRvaToSegOffset                            RT_MANGLER(RTLdrRvaToSegOffset)
# define RTLdrQueryForwarderInfo                        RT_MANGLER(RTLdrQueryForwarderInfo)
# define RTLdrQueryProp                                 RT_MANGLER(RTLdrQueryProp)
# define RTLdrSegOffsetToRva                            RT_MANGLER(RTLdrSegOffsetToRva)
# define RTLdrSize                                      RT_MANGLER(RTLdrSize)
# define RTLdrUnwindFrame                               RT_MANGLER(RTLdrUnwindFrame)
# define RTLinuxCheckDevicePath                         RT_MANGLER(RTLinuxCheckDevicePath)
# define RTLinuxCheckDevicePathV                        RT_MANGLER(RTLinuxCheckDevicePathV)
# define RTLinuxConstructPath                           RT_MANGLER(RTLinuxConstructPath)
# define RTLinuxConstructPathV                          RT_MANGLER(RTLinuxConstructPathV)
# define RTLinuxSysFsClose                              RT_MANGLER(RTLinuxSysFsClose)
# define RTLinuxSysFsExists                             RT_MANGLER(RTLinuxSysFsExists)
# define RTLinuxSysFsExistsEx                           RT_MANGLER(RTLinuxSysFsExistsEx)
# define RTLinuxSysFsExistsExV                          RT_MANGLER(RTLinuxSysFsExistsExV)
# define RTLinuxSysFsExistsV                            RT_MANGLER(RTLinuxSysFsExistsV)
# define RTLinuxSysFsGetLinkDest                        RT_MANGLER(RTLinuxSysFsGetLinkDest)
# define RTLinuxSysFsGetLinkDestV                       RT_MANGLER(RTLinuxSysFsGetLinkDestV)
# define RTLinuxSysFsOpen                               RT_MANGLER(RTLinuxSysFsOpen)
# define RTLinuxSysFsOpenEx                             RT_MANGLER(RTLinuxSysFsOpenEx)
# define RTLinuxSysFsOpenExV                            RT_MANGLER(RTLinuxSysFsOpenExV)
# define RTLinuxSysFsOpenV                              RT_MANGLER(RTLinuxSysFsOpenV)
# define RTLinuxSysFsReadDevNumFile                     RT_MANGLER(RTLinuxSysFsReadDevNumFile)
# define RTLinuxSysFsReadDevNumFileV                    RT_MANGLER(RTLinuxSysFsReadDevNumFileV)
# define RTLinuxSysFsReadFile                           RT_MANGLER(RTLinuxSysFsReadFile)
# define RTLinuxSysFsReadIntFile                        RT_MANGLER(RTLinuxSysFsReadIntFile)
# define RTLinuxSysFsReadIntFileV                       RT_MANGLER(RTLinuxSysFsReadIntFileV)
# define RTLinuxSysFsReadStr                            RT_MANGLER(RTLinuxSysFsReadStr)
# define RTLinuxSysFsReadStrFile                        RT_MANGLER(RTLinuxSysFsReadStrFile)
# define RTLinuxSysFsReadStrFileV                       RT_MANGLER(RTLinuxSysFsReadStrFileV)
# define RTLinuxSysFsWriteFile                          RT_MANGLER(RTLinuxSysFsWriteFile)
# define RTLinuxSysFsWriteStr                           RT_MANGLER(RTLinuxSysFsWriteStr)
# define RTLinuxSysFsWriteStrFile                       RT_MANGLER(RTLinuxSysFsWriteStrFile)
# define RTLinuxSysFsWriteStrFileV                      RT_MANGLER(RTLinuxSysFsWriteStrFileV)
# define RTLinuxSysFsWriteU8File                        RT_MANGLER(RTLinuxSysFsWriteU8File)
# define RTLinuxSysFsWriteU8FileV                       RT_MANGLER(RTLinuxSysFsWriteU8FileV)
# define RTLinuxSysFsWriteU16File                       RT_MANGLER(RTLinuxSysFsWriteU16File)
# define RTLinuxSysFsWriteU16FileV                      RT_MANGLER(RTLinuxSysFsWriteU16FileV)
# define RTLinuxSysFsWriteU32File                       RT_MANGLER(RTLinuxSysFsWriteU32File)
# define RTLinuxSysFsWriteU32FileV                      RT_MANGLER(RTLinuxSysFsWriteU32FileV)
# define RTLinuxSysFsWriteU64File                       RT_MANGLER(RTLinuxSysFsWriteU64File)
# define RTLinuxSysFsWriteU64FileV                      RT_MANGLER(RTLinuxSysFsWriteU64FileV)
# define RTLocalIpcServerCreate                         RT_MANGLER(RTLocalIpcServerCreate)
# define RTLocalIpcServerDestroy                        RT_MANGLER(RTLocalIpcServerDestroy)
# define RTLocalIpcServerGrantGroupAccess               RT_MANGLER(RTLocalIpcServerGrantGroupAccess)
# define RTLocalIpcServerSetAccessMode                  RT_MANGLER(RTLocalIpcServerSetAccessMode);
# define RTLocalIpcServerCancel                         RT_MANGLER(RTLocalIpcServerCancel)
# define RTLocalIpcServerListen                         RT_MANGLER(RTLocalIpcServerListen)
# define RTLocalIpcSessionConnect                       RT_MANGLER(RTLocalIpcSessionConnect)
# define RTLocalIpcSessionClose                         RT_MANGLER(RTLocalIpcSessionClose)
# define RTLocalIpcSessionCancel                        RT_MANGLER(RTLocalIpcSessionCancel)
# define RTLocalIpcSessionRead                          RT_MANGLER(RTLocalIpcSessionRead)
# define RTLocalIpcSessionReadNB                        RT_MANGLER(RTLocalIpcSessionReadNB)
# define RTLocalIpcSessionRetain                        RT_MANGLER(RTLocalIpcSessionRetain)
# define RTLocalIpcSessionRelease                       RT_MANGLER(RTLocalIpcSessionRelease)
# define RTLocalIpcSessionWrite                         RT_MANGLER(RTLocalIpcSessionWrite)
# define RTLocalIpcSessionFlush                         RT_MANGLER(RTLocalIpcSessionFlush)
# define RTLocalIpcSessionWaitForData                   RT_MANGLER(RTLocalIpcSessionWaitForData)
# define RTLocalIpcSessionQueryProcess                  RT_MANGLER(RTLocalIpcSessionQueryProcess)
# define RTLocalIpcSessionQueryUserId                   RT_MANGLER(RTLocalIpcSessionQueryUserId)
# define RTLocalIpcSessionQueryGroupId                  RT_MANGLER(RTLocalIpcSessionQueryGroupId)
# define RTLocaleQueryLocaleName                        RT_MANGLER(RTLocaleQueryLocaleName)
# define RTLocaleQueryNormalizedBaseLocaleName          RT_MANGLER(RTLocaleQueryNormalizedBaseLocaleName)
# define RTLocaleQueryUserCountryCode                   RT_MANGLER(RTLocaleQueryUserCountryCode)
# define RTLockValidatorClassAddPriorClass              RT_MANGLER(RTLockValidatorClassAddPriorClass)
# define RTLockValidatorClassCreate                     RT_MANGLER(RTLockValidatorClassCreate)
# define RTLockValidatorClassCreateEx                   RT_MANGLER(RTLockValidatorClassCreateEx)
# define RTLockValidatorClassCreateExV                  RT_MANGLER(RTLockValidatorClassCreateExV)
# define RTLockValidatorClassCreateUnique               RT_MANGLER(RTLockValidatorClassCreateUnique)
# define RTLockValidatorClassEnforceStrictReleaseOrder  RT_MANGLER(RTLockValidatorClassEnforceStrictReleaseOrder)
# define RTLockValidatorClassFindForSrcPos              RT_MANGLER(RTLockValidatorClassFindForSrcPos)
# define RTLockValidatorClassForSrcPos                  RT_MANGLER(RTLockValidatorClassForSrcPos)
# define RTLockValidatorClassRelease                    RT_MANGLER(RTLockValidatorClassRelease)
# define RTLockValidatorClassRetain                     RT_MANGLER(RTLockValidatorClassRetain)
# define RTLockValidatorHoldsLocksInClass               RT_MANGLER(RTLockValidatorHoldsLocksInClass)
# define RTLockValidatorHoldsLocksInSubClass            RT_MANGLER(RTLockValidatorHoldsLocksInSubClass)
# define RTLockValidatorIsBlockedThreadInValidator      RT_MANGLER(RTLockValidatorIsBlockedThreadInValidator)
# define RTLockValidatorIsEnabled                       RT_MANGLER(RTLockValidatorIsEnabled)
# define RTLockValidatorIsQuiet                         RT_MANGLER(RTLockValidatorIsQuiet)
# define RTLockValidatorMayPanic                        RT_MANGLER(RTLockValidatorMayPanic)
# define RTLockValidatorQueryBlocking                   RT_MANGLER(RTLockValidatorQueryBlocking)
# define RTLockValidatorReadLockDec                     RT_MANGLER(RTLockValidatorReadLockDec)
# define RTLockValidatorReadLockGetCount                RT_MANGLER(RTLockValidatorReadLockGetCount)
# define RTLockValidatorReadLockInc                     RT_MANGLER(RTLockValidatorReadLockInc)
# define RTLockValidatorRecExclCheckBlocking            RT_MANGLER(RTLockValidatorRecExclCheckBlocking)
# define RTLockValidatorRecExclCheckOrder               RT_MANGLER(RTLockValidatorRecExclCheckOrder)
# define RTLockValidatorRecExclCheckOrderAndBlocking    RT_MANGLER(RTLockValidatorRecExclCheckOrderAndBlocking)
# define RTLockValidatorRecExclCreate                   RT_MANGLER(RTLockValidatorRecExclCreate)
# define RTLockValidatorRecExclCreateV                  RT_MANGLER(RTLockValidatorRecExclCreateV)
# define RTLockValidatorRecExclDelete                   RT_MANGLER(RTLockValidatorRecExclDelete)
# define RTLockValidatorRecExclDestroy                  RT_MANGLER(RTLockValidatorRecExclDestroy)
# define RTLockValidatorRecExclInit                     RT_MANGLER(RTLockValidatorRecExclInit)
# define RTLockValidatorRecExclInitV                    RT_MANGLER(RTLockValidatorRecExclInitV)
# define RTLockValidatorRecExclRecursion                RT_MANGLER(RTLockValidatorRecExclRecursion)
# define RTLockValidatorRecExclRecursionMixed           RT_MANGLER(RTLockValidatorRecExclRecursionMixed)
# define RTLockValidatorRecExclReleaseOwner             RT_MANGLER(RTLockValidatorRecExclReleaseOwner)
# define RTLockValidatorRecExclReleaseOwnerUnchecked    RT_MANGLER(RTLockValidatorRecExclReleaseOwnerUnchecked)
# define RTLockValidatorRecExclSetOwner                 RT_MANGLER(RTLockValidatorRecExclSetOwner)
# define RTLockValidatorRecExclSetSubClass              RT_MANGLER(RTLockValidatorRecExclSetSubClass)
# define RTLockValidatorRecExclUnwind                   RT_MANGLER(RTLockValidatorRecExclUnwind)
# define RTLockValidatorRecExclUnwindMixed              RT_MANGLER(RTLockValidatorRecExclUnwindMixed)
# define RTLockValidatorRecMakeSiblings                 RT_MANGLER(RTLockValidatorRecMakeSiblings)
# define RTLockValidatorRecSharedAddOwner               RT_MANGLER(RTLockValidatorRecSharedAddOwner)
# define RTLockValidatorRecSharedCheckAndRelease        RT_MANGLER(RTLockValidatorRecSharedCheckAndRelease)
# define RTLockValidatorRecSharedCheckBlocking          RT_MANGLER(RTLockValidatorRecSharedCheckBlocking)
# define RTLockValidatorRecSharedCheckOrder             RT_MANGLER(RTLockValidatorRecSharedCheckOrder)
# define RTLockValidatorRecSharedCheckOrderAndBlocking  RT_MANGLER(RTLockValidatorRecSharedCheckOrderAndBlocking)
# define RTLockValidatorRecSharedCheckSignaller         RT_MANGLER(RTLockValidatorRecSharedCheckSignaller)
# define RTLockValidatorRecSharedCreate                 RT_MANGLER(RTLockValidatorRecSharedCreate)
# define RTLockValidatorRecSharedCreateV                RT_MANGLER(RTLockValidatorRecSharedCreateV)
# define RTLockValidatorRecSharedDelete                 RT_MANGLER(RTLockValidatorRecSharedDelete)
# define RTLockValidatorRecSharedDestroy                RT_MANGLER(RTLockValidatorRecSharedDestroy)
# define RTLockValidatorRecSharedInit                   RT_MANGLER(RTLockValidatorRecSharedInit)
# define RTLockValidatorRecSharedInitV                  RT_MANGLER(RTLockValidatorRecSharedInitV)
# define RTLockValidatorRecSharedIsOwner                RT_MANGLER(RTLockValidatorRecSharedIsOwner)
# define RTLockValidatorRecSharedRemoveOwner            RT_MANGLER(RTLockValidatorRecSharedRemoveOwner)
# define RTLockValidatorRecSharedResetOwner             RT_MANGLER(RTLockValidatorRecSharedResetOwner)
# define RTLockValidatorRecSharedSetSubClass            RT_MANGLER(RTLockValidatorRecSharedSetSubClass)
# define RTLockValidatorSetEnabled                      RT_MANGLER(RTLockValidatorSetEnabled)
# define RTLockValidatorSetMayPanic                     RT_MANGLER(RTLockValidatorSetMayPanic)
# define RTLockValidatorSetQuiet                        RT_MANGLER(RTLockValidatorSetQuiet)
# define RTLockValidatorWriteLockDec                    RT_MANGLER(RTLockValidatorWriteLockDec)
# define RTLockValidatorWriteLockGetCount               RT_MANGLER(RTLockValidatorWriteLockGetCount)
# define RTLockValidatorWriteLockInc                    RT_MANGLER(RTLockValidatorWriteLockInc)
# define RTLogAssert                                    RT_MANGLER(RTLogAssert)
# define RTLogAssertV                                   RT_MANGLER(RTLogAssertV)
# define RTLogBackdoorPrintf                            RT_MANGLER(RTLogBackdoorPrintf) /* r0drv-guest */
# define RTLogBackdoorPrintfV                           RT_MANGLER(RTLogBackdoorPrintfV) /* r0drv-guest */
# define RTLogBulkUpdate                                RT_MANGLER(RTLogBulkUpdate)
# define RTLogBulkWrite                                 RT_MANGLER(RTLogBulkWrite)
# define RTLogBulkNestedWrite                           RT_MANGLER(RTLogBulkNestedWrite)
# define RTLogChangeDestinations                        RT_MANGLER(RTLogChangeDestinations)
# define RTLogChangeFlags                               RT_MANGLER(RTLogChangeFlags)
# define RTLogCheckGroupFlags                           RT_MANGLER(RTLogCheckGroupFlags)
# define RTLogClearFileDelayFlag                        RT_MANGLER(RTLogClearFileDelayFlag)
# define RTLogCloneRC                                   RT_MANGLER(RTLogCloneRC)
# define RTLogComPrintf                                 RT_MANGLER(RTLogComPrintf)
# define RTLogComPrintfV                                RT_MANGLER(RTLogComPrintfV)
# define RTLogCreate                                    RT_MANGLER(RTLogCreate)
# define RTLogCreateEx                                  RT_MANGLER(RTLogCreateEx)
# define RTLogCreateExV                                 RT_MANGLER(RTLogCreateExV)
# define RTLogDefaultInit                               RT_MANGLER(RTLogDefaultInit)
# define RTLogDefaultInstance                           RT_MANGLER(RTLogDefaultInstance)
# define RTLogDefaultInstanceEx                         RT_MANGLER(RTLogDefaultInstanceEx)
# define RTLogDestinations                              RT_MANGLER(RTLogDestinations)
# define RTLogDestroy                                   RT_MANGLER(RTLogDestroy)
# define RTLogFlags                                     RT_MANGLER(RTLogFlags)
# define RTLogFlush                                     RT_MANGLER(RTLogFlush)
# define RTLogFormatV                                   RT_MANGLER(RTLogFormatV)
# define RTLogGetDefaultInstance                        RT_MANGLER(RTLogGetDefaultInstance)
# define RTLogGetDefaultInstanceEx                      RT_MANGLER(RTLogGetDefaultInstanceEx)
# define RTLogGetDestinations                           RT_MANGLER(RTLogGetDestinations)
# define RTLogGetFlags                                  RT_MANGLER(RTLogGetFlags)
# define RTLogGroupSettings                             RT_MANGLER(RTLogGroupSettings)
# define RTLogLogger                                    RT_MANGLER(RTLogLogger)
# define RTLogLoggerWeak                                RT_MANGLER(RTLogLoggerWeak)
# define RTLogLoggerEx                                  RT_MANGLER(RTLogLoggerEx)
# define RTLogLoggerExWeak                              RT_MANGLER(RTLogLoggerExWeak)
# define RTLogLoggerExV                                 RT_MANGLER(RTLogLoggerExV)
# define RTLogLoggerV                                   RT_MANGLER(RTLogLoggerV)
# define RTLogPrintf                                    RT_MANGLER(RTLogPrintf)
# define RTLogPrintfV                                   RT_MANGLER(RTLogPrintfV)
# define RTLogDumpPrintfV                               RT_MANGLER(RTLogDumpPrintfV)
# define RTLogQueryBulk                                 RT_MANGLER(RTLogQueryBulk)
# define RTLogQueryDestinations                         RT_MANGLER(RTLogQueryDestinations)
# define RTLogQueryFlags                                RT_MANGLER(RTLogQueryFlags)
# define RTLogQueryGroupSettings                        RT_MANGLER(RTLogQueryGroupSettings)
# define RTLogRelGetDefaultInstance                     RT_MANGLER(RTLogRelGetDefaultInstance)
# define RTLogRelGetDefaultInstanceEx                   RT_MANGLER(RTLogRelGetDefaultInstanceEx)
# define RTLogRelLogger                                 RT_MANGLER(RTLogRelLogger)
# define RTLogRelLoggerV                                RT_MANGLER(RTLogRelLoggerV)
# define RTLogRelPrintf                                 RT_MANGLER(RTLogRelPrintf)
# define RTLogRelPrintfV                                RT_MANGLER(RTLogRelPrintfV)
# define RTLogRelSetBuffering                           RT_MANGLER(RTLogRelSetBuffering)
# define RTLogRelSetDefaultInstance                     RT_MANGLER(RTLogRelSetDefaultInstance)
# define RTLogSetBuffering                              RT_MANGLER(RTLogSetBuffering)
# define RTLogSetCustomPrefixCallback                   RT_MANGLER(RTLogSetCustomPrefixCallback)
# define RTLogSetFlushCallback                          RT_MANGLER(RTLogSetFlushCallback)
# define RTLogSetDefaultInstance                        RT_MANGLER(RTLogSetDefaultInstance)
# define RTLogSetDefaultInstanceThread                  RT_MANGLER(RTLogSetDefaultInstanceThread) /* r0drv */
# define RTLogSetGroupLimit                             RT_MANGLER(RTLogSetGroupLimit)
# define RTLogSetR0ProgramStart                         RT_MANGLER(RTLogSetR0ProgramStart) /* r0drv */
# define RTLogSetR0ThreadNameF                          RT_MANGLER(RTLogSetR0ThreadNameF) /* r0drv */
# define RTLogSetR0ThreadNameV                          RT_MANGLER(RTLogSetR0ThreadNameV) /* r0drv */
# define RTLogWriteCom                                  RT_MANGLER(RTLogWriteCom)
# define RTLogWriteDebugger                             RT_MANGLER(RTLogWriteDebugger)
# define RTLogWriteStdErr                               RT_MANGLER(RTLogWriteStdErr)
# define RTLogWriteStdOut                               RT_MANGLER(RTLogWriteStdOut)
# define RTLogWriteUser                                 RT_MANGLER(RTLogWriteUser)
# define RTLogWriteVmm                                  RT_MANGLER(RTLogWriteVmm)
# define RTLogWriteVmm_EndProc                          RT_MANGLER(RTLogWriteVmm_EndProc)
# define RTManifestCreate                               RT_MANGLER(RTManifestCreate)
# define RTManifestDup                                  RT_MANGLER(RTManifestDup)
# define RTManifestEntryAdd                             RT_MANGLER(RTManifestEntryAdd)
# define RTManifestEntryAddIoStream                     RT_MANGLER(RTManifestEntryAddIoStream)
# define RTManifestEntryAddPassthruIoStream             RT_MANGLER(RTManifestEntryAddPassthruIoStream)
# define RTManifestEntryExists                          RT_MANGLER(RTManifestEntryExists)
# define RTManifestEntryRemove                          RT_MANGLER(RTManifestEntryRemove)
# define RTManifestEntryQueryAttr                       RT_MANGLER(RTManifestEntryQueryAttr)
# define RTManifestEntrySetAttr                         RT_MANGLER(RTManifestEntrySetAttr)
# define RTManifestEntryUnsetAttr                       RT_MANGLER(RTManifestEntryUnsetAttr)
# define RTManifestEquals                               RT_MANGLER(RTManifestEquals)
# define RTManifestEqualsEx                             RT_MANGLER(RTManifestEqualsEx)
# define RTManifestPtIosAddEntryNow                     RT_MANGLER(RTManifestPtIosAddEntryNow)
# define RTManifestPtIosIsInstanceOf                    RT_MANGLER(RTManifestPtIosIsInstanceOf)
# define RTManifestQueryAllAttrTypes                    RT_MANGLER(RTManifestQueryAllAttrTypes)
# define RTManifestQueryAttr                            RT_MANGLER(RTManifestQueryAttr)
# define RTManifestReadStandard                         RT_MANGLER(RTManifestReadStandard)
# define RTManifestReadStandardEx                       RT_MANGLER(RTManifestReadStandardEx)
# define RTManifestReadStandardFromFile                 RT_MANGLER(RTManifestReadStandardFromFile)
# define RTManifestRelease                              RT_MANGLER(RTManifestRelease)
# define RTManifestRetain                               RT_MANGLER(RTManifestRetain)
# define RTManifestSetAttr                              RT_MANGLER(RTManifestSetAttr)
# define RTManifestUnsetAttr                            RT_MANGLER(RTManifestUnsetAttr)
# define RTManifestVerify                               RT_MANGLER(RTManifestVerify)
# define RTManifestVerifyDigestType                     RT_MANGLER(RTManifestVerifyDigestType)
# define RTManifestVerifyFiles                          RT_MANGLER(RTManifestVerifyFiles)
# define RTManifestVerifyFilesBuf                       RT_MANGLER(RTManifestVerifyFilesBuf)
# define RTManifestWriteFiles                           RT_MANGLER(RTManifestWriteFiles)
# define RTManifestWriteFilesBuf                        RT_MANGLER(RTManifestWriteFilesBuf)
# define RTManifestWriteStandard                        RT_MANGLER(RTManifestWriteStandard)
# define RTManifestWriteStandardToFile                  RT_MANGLER(RTManifestWriteStandardToFile)
# define RTMd4                                          RT_MANGLER(RTMd4)
# define RTMd4Final                                     RT_MANGLER(RTMd4Final)
# define RTMd4FromString                                RT_MANGLER(RTMd4FromString)
# define RTMd4Init                                      RT_MANGLER(RTMd4Init)
# define RTMd4ToString                                  RT_MANGLER(RTMd4ToString)
# define RTMd4Update                                    RT_MANGLER(RTMd4Update)
# define RTMd5                                          RT_MANGLER(RTMd5)
# define RTMd5Final                                     RT_MANGLER(RTMd5Final)
# define RTMd5FromString                                RT_MANGLER(RTMd5FromString)
# define RTMd5Init                                      RT_MANGLER(RTMd5Init)
# define RTMd5ToString                                  RT_MANGLER(RTMd5ToString)
# define RTMd5Update                                    RT_MANGLER(RTMd5Update)
# define RTMemAllocExTag                                RT_MANGLER(RTMemAllocExTag)
# define RTMemAllocTag                                  RT_MANGLER(RTMemAllocTag)
# define RTMemAllocVarTag                               RT_MANGLER(RTMemAllocVarTag)
# define RTMemAllocZTag                                 RT_MANGLER(RTMemAllocZTag)
# define RTMemAllocZVarTag                              RT_MANGLER(RTMemAllocZVarTag)
# define RTMemCacheAlloc                                RT_MANGLER(RTMemCacheAlloc)
# define RTMemCacheAllocEx                              RT_MANGLER(RTMemCacheAllocEx)
# define RTMemCacheCreate                               RT_MANGLER(RTMemCacheCreate)
# define RTMemCacheDestroy                              RT_MANGLER(RTMemCacheDestroy)
# define RTMemCacheFree                                 RT_MANGLER(RTMemCacheFree)
# define RTMemContAlloc                                 RT_MANGLER(RTMemContAlloc) /* r0drv */
# define RTMemContFree                                  RT_MANGLER(RTMemContFree) /* r0drv */
# define RTMemDump                                      RT_MANGLER(RTMemDump)
# define RTMemDumpFreed                                 RT_MANGLER(RTMemDumpFreed)
# define RTMemDupExTag                                  RT_MANGLER(RTMemDupExTag)
# define RTMemDupTag                                    RT_MANGLER(RTMemDupTag)
# define RTMemEfAlloc                                   RT_MANGLER(RTMemEfAlloc)
# define RTMemEfAllocNP                                 RT_MANGLER(RTMemEfAllocNP)
# define RTMemEfAllocVar                                RT_MANGLER(RTMemEfAllocVar)
# define RTMemEfAllocVarNP                              RT_MANGLER(RTMemEfAllocVarNP)
# define RTMemEfAllocZ                                  RT_MANGLER(RTMemEfAllocZ)
# define RTMemEfAllocZNP                                RT_MANGLER(RTMemEfAllocZNP)
# define RTMemEfAllocZVar                               RT_MANGLER(RTMemEfAllocZVar)
# define RTMemEfAllocZVarNP                             RT_MANGLER(RTMemEfAllocZVarNP)
# define RTMemEfDup                                     RT_MANGLER(RTMemEfDup)
# define RTMemEfDupEx                                   RT_MANGLER(RTMemEfDupEx)
# define RTMemEfDupExNP                                 RT_MANGLER(RTMemEfDupExNP)
# define RTMemEfDupNP                                   RT_MANGLER(RTMemEfDupNP)
# define RTMemEfFree                                    RT_MANGLER(RTMemEfFree)
# define RTMemEfFreeNP                                  RT_MANGLER(RTMemEfFreeNP)
# define RTMemEfFreeZ                                   RT_MANGLER(RTMemEfFreeZ)
# define RTMemEfFreeZNP                                 RT_MANGLER(RTMemEfFreeZNP)
# define RTMemEfRealloc                                 RT_MANGLER(RTMemEfRealloc)
# define RTMemEfReallocNP                               RT_MANGLER(RTMemEfReallocNP)
# define RTMemEfReallocZ                                RT_MANGLER(RTMemEfReallocZ)
# define RTMemEfReallocZNP                              RT_MANGLER(RTMemEfReallocZNP)
# define RTMemEfTmpAlloc                                RT_MANGLER(RTMemEfTmpAlloc)
# define RTMemEfTmpAllocNP                              RT_MANGLER(RTMemEfTmpAllocNP)
# define RTMemEfTmpAllocZ                               RT_MANGLER(RTMemEfTmpAllocZ)
# define RTMemEfTmpAllocZNP                             RT_MANGLER(RTMemEfTmpAllocZNP)
# define RTMemEfTmpFree                                 RT_MANGLER(RTMemEfTmpFree)
# define RTMemEfTmpFreeNP                               RT_MANGLER(RTMemEfTmpFreeNP)
# define RTMemEfTmpFreeZ                                RT_MANGLER(RTMemEfTmpFreeZ)
# define RTMemEfTmpFreeZNP                              RT_MANGLER(RTMemEfTmpFreeZNP)
# define RTMemFree                                      RT_MANGLER(RTMemFree)
# define RTMemFreeZ                                     RT_MANGLER(RTMemFreeZ)
# define RTMemFreeEx                                    RT_MANGLER(RTMemFreeEx)
# define RTMemPageAllocTag                              RT_MANGLER(RTMemPageAllocTag)
# define RTMemPageAllocExTag                            RT_MANGLER(RTMemPageAllocExTag)
# define RTMemPageAllocZTag                             RT_MANGLER(RTMemPageAllocZTag)
# define RTMemPageFree                                  RT_MANGLER(RTMemPageFree)
# define RTMemPoolAlloc                                 RT_MANGLER(RTMemPoolAlloc)
# define RTMemPoolAllocZ                                RT_MANGLER(RTMemPoolAllocZ)
# define RTMemPoolCreate                                RT_MANGLER(RTMemPoolCreate)
# define RTMemPoolDestroy                               RT_MANGLER(RTMemPoolDestroy)
# define RTMemPoolDup                                   RT_MANGLER(RTMemPoolDup)
# define RTMemPoolDupEx                                 RT_MANGLER(RTMemPoolDupEx)
# define RTMemPoolFree                                  RT_MANGLER(RTMemPoolFree)
# define RTMemPoolRealloc                               RT_MANGLER(RTMemPoolRealloc)
# define RTMemPoolRefCount                              RT_MANGLER(RTMemPoolRefCount)
# define RTMemPoolRelease                               RT_MANGLER(RTMemPoolRelease)
# define RTMemPoolRetain                                RT_MANGLER(RTMemPoolRetain)
# define RTMemProtect                                   RT_MANGLER(RTMemProtect)
# define RTMemReallocTag                                RT_MANGLER(RTMemReallocTag)
# define RTMemReallocZTag                               RT_MANGLER(RTMemReallocZTag)
# define RTMemTmpAllocTag                               RT_MANGLER(RTMemTmpAllocTag)
# define RTMemTmpAllocZTag                              RT_MANGLER(RTMemTmpAllocZTag)
# define RTMemTmpFree                                   RT_MANGLER(RTMemTmpFree)
# define RTMemTmpFreeZ                                  RT_MANGLER(RTMemTmpFreeZ)
# define RTMemTrackerDumpAllToFile                      RT_MANGLER(RTMemTrackerDumpAllToFile)
# define RTMemTrackerDumpAllToLog                       RT_MANGLER(RTMemTrackerDumpAllToLog)
# define RTMemTrackerDumpAllToLogRel                    RT_MANGLER(RTMemTrackerDumpAllToLogRel)
# define RTMemTrackerDumpAllToStdErr                    RT_MANGLER(RTMemTrackerDumpAllToStdErr)
# define RTMemTrackerDumpAllToStdOut                    RT_MANGLER(RTMemTrackerDumpAllToStdOut)
# define RTMemTrackerDumpStatsToFile                    RT_MANGLER(RTMemTrackerDumpStatsToFile)
# define RTMemTrackerDumpStatsToLog                     RT_MANGLER(RTMemTrackerDumpStatsToLog)
# define RTMemTrackerDumpStatsToLogRel                  RT_MANGLER(RTMemTrackerDumpStatsToLogRel)
# define RTMemTrackerDumpStatsToStdErr                  RT_MANGLER(RTMemTrackerDumpStatsToStdErr)
# define RTMemTrackerDumpStatsToStdOut                  RT_MANGLER(RTMemTrackerDumpStatsToStdOut)
# define RTMemTrackerHdrAlloc                           RT_MANGLER(RTMemTrackerHdrAlloc)
# define RTMemTrackerHdrFree                            RT_MANGLER(RTMemTrackerHdrFree)
# define RTMemTrackerHdrReallocDone                     RT_MANGLER(RTMemTrackerHdrReallocDone)
# define RTMemTrackerHdrReallocPrep                     RT_MANGLER(RTMemTrackerHdrReallocPrep)
# define RTMemWipeThoroughly                            RT_MANGLER(RTMemWipeThoroughly)
# define RTMpCpuId                                      RT_MANGLER(RTMpCpuId)
# define RTMpCpuIdFromSetIndex                          RT_MANGLER(RTMpCpuIdFromSetIndex)
# define RTMpCpuIdToSetIndex                            RT_MANGLER(RTMpCpuIdToSetIndex)
# define RTMpCurSetIndex                                RT_MANGLER(RTMpCurSetIndex)
# define RTMpCurSetIndexAndId                           RT_MANGLER(RTMpCurSetIndexAndId)
# define RTMpGetArraySize                               RT_MANGLER(RTMpGetArraySize)
# define RTMpGetCount                                   RT_MANGLER(RTMpGetCount)
# define RTMpGetCurFrequency                            RT_MANGLER(RTMpGetCurFrequency)
# define RTMpGetDescription                             RT_MANGLER(RTMpGetDescription)
# define RTMpGetCpuGroupCounts                          RT_MANGLER(RTMpGetCpuGroupCounts)
# define RTMpGetMaxCpuGroupCount                        RT_MANGLER(RTMpGetMaxCpuGroupCount)
# define RTMpGetMaxCpuId                                RT_MANGLER(RTMpGetMaxCpuId)
# define RTMpGetMaxFrequency                            RT_MANGLER(RTMpGetMaxFrequency)
# define RTMpGetOnlineCount                             RT_MANGLER(RTMpGetOnlineCount)
# define RTMpGetOnlineCoreCount                         RT_MANGLER(RTMpGetOnlineCoreCount)
# define RTMpGetOnlineSet                               RT_MANGLER(RTMpGetOnlineSet)
# define RTMpGetPresentCount                            RT_MANGLER(RTMpGetPresentCount)
# define RTMpGetPresentCoreCount                        RT_MANGLER(RTMpGetPresentCoreCount)
# define RTMpGetPresentSet                              RT_MANGLER(RTMpGetPresentSet)
# define RTMpGetSet                                     RT_MANGLER(RTMpGetSet)
# define RTMpGetCoreCount                               RT_MANGLER(RTMpGetCoreCount)
# define RTMpIsCpuOnline                                RT_MANGLER(RTMpIsCpuOnline)
# define RTMpIsCpuPossible                              RT_MANGLER(RTMpIsCpuPossible) /* r0drv */
# define RTMpIsCpuPresent                               RT_MANGLER(RTMpIsCpuPresent)
# define RTMpIsCpuWorkPending                           RT_MANGLER(RTMpIsCpuWorkPending)
# define RTMpNotificationDeregister                     RT_MANGLER(RTMpNotificationDeregister) /* r0drv */
# define RTMpNotificationRegister                       RT_MANGLER(RTMpNotificationRegister)   /* r0drv */
# define RTMpOnAll                                      RT_MANGLER(RTMpOnAll)                  /* r0drv */
# define RTMpOnAllIsConcurrentSafe                      RT_MANGLER(RTMpOnAllIsConcurrentSafe)  /* r0drv */
# define RTMpOnOthers                                   RT_MANGLER(RTMpOnOthers)               /* r0drv */
# define RTMpOnPair                                     RT_MANGLER(RTMpOnPair)                 /* r0drv */
# define RTMpOnPairIsConcurrentExecSupported            RT_MANGLER(RTMpOnPairIsConcurrentExecSupported) /* r0drv */
# define RTMpOnSpecific                                 RT_MANGLER(RTMpOnSpecific)             /* r0drv */
# define RTMpPokeCpu                                    RT_MANGLER(RTMpPokeCpu)                /* r0drv */
# define RTMpSetIndexFromCpuGroupMember                 RT_MANGLER(RTMpSetIndexFromCpuGroupMember)
# define RTMsgError                                     RT_MANGLER(RTMsgError)
# define RTMsgErrorExit                                 RT_MANGLER(RTMsgErrorExit)
# define RTMsgErrorExitV                                RT_MANGLER(RTMsgErrorExitV)
# define RTMsgErrorExitFailure                          RT_MANGLER(RTMsgErrorExitFailure)
# define RTMsgErrorExitFailureV                         RT_MANGLER(RTMsgErrorExitFailureV)
# define RTMsgErrorRc                                   RT_MANGLER(RTMsgErrorRc)
# define RTMsgErrorRcV                                  RT_MANGLER(RTMsgErrorRcV)
# define RTMsgErrorV                                    RT_MANGLER(RTMsgErrorV)
# define RTMsgInfo                                      RT_MANGLER(RTMsgInfo)
# define RTMsgInfoV                                     RT_MANGLER(RTMsgInfoV)
# define RTMsgInitFailure                               RT_MANGLER(RTMsgInitFailure)
# define RTMsgSetProgName                               RT_MANGLER(RTMsgSetProgName)
# define RTMsgSyntax                                    RT_MANGLER(RTMsgSyntax)
# define RTMsgSyntaxV                                   RT_MANGLER(RTMsgSyntaxV)
# define RTMsgWarning                                   RT_MANGLER(RTMsgWarning)
# define RTMsgWarningV                                  RT_MANGLER(RTMsgWarningV)
# define RTMsgRefEntryPrintStringTable                  RT_MANGLER(RTMsgRefEntryPrintStringTable)
# define RTMsgRefEntrySynopsisEx                        RT_MANGLER(RTMsgRefEntrySynopsisEx)
# define RTMsgRefEntrySynopsis                          RT_MANGLER(RTMsgRefEntrySynopsis)
# define RTMsgRefEntryHelpEx                            RT_MANGLER(RTMsgRefEntryHelpEx)
# define RTMsgRefEntryHelp                              RT_MANGLER(RTMsgRefEntryHelp)
# define RTNetIPv4AddDataChecksum                       RT_MANGLER(RTNetIPv4AddDataChecksum)
# define RTNetIPv4AddTCPChecksum                        RT_MANGLER(RTNetIPv4AddTCPChecksum)
# define RTNetIPv4AddUDPChecksum                        RT_MANGLER(RTNetIPv4AddUDPChecksum)
# define RTNetIPv4FinalizeChecksum                      RT_MANGLER(RTNetIPv4FinalizeChecksum)
# define RTNetIPv4HdrChecksum                           RT_MANGLER(RTNetIPv4HdrChecksum)
# define RTNetIPv4IsDHCPValid                           RT_MANGLER(RTNetIPv4IsDHCPValid)
# define RTNetIPv4IsHdrValid                            RT_MANGLER(RTNetIPv4IsHdrValid)
# define RTNetIPv4IsTCPSizeValid                        RT_MANGLER(RTNetIPv4IsTCPSizeValid)
# define RTNetIPv4IsTCPValid                            RT_MANGLER(RTNetIPv4IsTCPValid)
# define RTNetIPv4IsUDPSizeValid                        RT_MANGLER(RTNetIPv4IsUDPSizeValid)
# define RTNetIPv4IsUDPValid                            RT_MANGLER(RTNetIPv4IsUDPValid)
# define RTNetIPv4PseudoChecksum                        RT_MANGLER(RTNetIPv4PseudoChecksum)
# define RTNetIPv4PseudoChecksumBits                    RT_MANGLER(RTNetIPv4PseudoChecksumBits)
# define RTNetIPv4TCPChecksum                           RT_MANGLER(RTNetIPv4TCPChecksum)
# define RTNetIPv4UDPChecksum                           RT_MANGLER(RTNetIPv4UDPChecksum)
# define RTNetIPv6PseudoChecksum                        RT_MANGLER(RTNetIPv6PseudoChecksum)
# define RTNetIPv6PseudoChecksumBits                    RT_MANGLER(RTNetIPv6PseudoChecksumBits)
# define RTNetIPv6PseudoChecksumEx                      RT_MANGLER(RTNetIPv6PseudoChecksumEx)
# define RTNetIsIPv4AddrStr                             RT_MANGLER(RTNetIsIPv4AddrStr)
# define RTNetIsIPv6AddrStr                             RT_MANGLER(RTNetIsIPv6AddrStr)
# define RTNetMaskToPrefixIPv4                          RT_MANGLER(RTNetMaskToPrefixIPv4)
# define RTNetMaskToPrefixIPv6                          RT_MANGLER(RTNetMaskToPrefixIPv6)
# define RTNetPrefixToMaskIPv4                          RT_MANGLER(RTNetPrefixToMaskIPv4)
# define RTNetPrefixToMaskIPv6                          RT_MANGLER(RTNetPrefixToMaskIPv6)
# define RTNetStrIsIPv4AddrAny                          RT_MANGLER(RTNetStrIsIPv4AddrAny)
# define RTNetStrIsIPv6AddrAny                          RT_MANGLER(RTNetStrIsIPv6AddrAny)
# define RTNetStrToIPv4Addr                             RT_MANGLER(RTNetStrToIPv4Addr)
# define RTNetStrToIPv4AddrEx                           RT_MANGLER(RTNetStrToIPv4AddrEx)
# define RTNetStrToIPv4Cidr                             RT_MANGLER(RTNetStrToIPv4Cidr)
# define RTNetStrToIPv6Addr                             RT_MANGLER(RTNetStrToIPv6Addr)
# define RTNetStrToIPv6AddrEx                           RT_MANGLER(RTNetStrToIPv6AddrEx)
# define RTNetStrToIPv6Cidr                             RT_MANGLER(RTNetStrToIPv6Cidr)
# define RTNetStrToMacAddr                              RT_MANGLER(RTNetStrToMacAddr)
# define RTNetTCPChecksum                               RT_MANGLER(RTNetTCPChecksum)
# define RTNetUDPChecksum                               RT_MANGLER(RTNetUDPChecksum)
# define RTOnceSlow                                     RT_MANGLER(RTOnceSlow)
# define RTOnceReset                                    RT_MANGLER(RTOnceReset)
# define RTPathAbs                                      RT_MANGLER(RTPathAbs)
# define RTPathAbsDup                                   RT_MANGLER(RTPathAbsDup)
# define RTPathAbsEx                                    RT_MANGLER(RTPathAbsEx)
# define RTPathAbsExDup                                 RT_MANGLER(RTPathAbsExDup)
# define RTPathAppDocs                                  RT_MANGLER(RTPathAppDocs)
# define RTPathAppend                                   RT_MANGLER(RTPathAppend)
# define RTPathAppendEx                                 RT_MANGLER(RTPathAppendEx)
# define RTPathAppPrivateArch                           RT_MANGLER(RTPathAppPrivateArch)
# define RTPathAppPrivateArchTop                        RT_MANGLER(RTPathAppPrivateArchTop)
# define RTPathAppPrivateNoArch                         RT_MANGLER(RTPathAppPrivateNoArch)
# define RTPathCalcRelative                             RT_MANGLER(RTPathCalcRelative)
# define RTPathChangeToDosSlashes                       RT_MANGLER(RTPathChangeToDosSlashes)
# define RTPathChangeToUnixSlashes                      RT_MANGLER(RTPathChangeToUnixSlashes)
# define RTPathCompare                                  RT_MANGLER(RTPathCompare)
# define RTPathCopyComponents                           RT_MANGLER(RTPathCopyComponents)
# define RTPathCountComponents                          RT_MANGLER(RTPathCountComponents)
# define RTPathEnsureTrailingSeparator                  RT_MANGLER(RTPathEnsureTrailingSeparator)
# define RTPathEnsureTrailingSeparatorEx                RT_MANGLER(RTPathEnsureTrailingSeparatorEx)
# define RTPathExecDir                                  RT_MANGLER(RTPathExecDir)
# define RTPathExists                                   RT_MANGLER(RTPathExists)
# define RTPathExistsEx                                 RT_MANGLER(RTPathExistsEx)
# define RTPathSuffix                                   RT_MANGLER(RTPathSuffix)
# define RTPathFilename                                 RT_MANGLER(RTPathFilename)
# define RTPathFilenameUtf16                            RT_MANGLER(RTPathFilenameUtf16)
# define RTPathFilenameEx                               RT_MANGLER(RTPathFilenameEx)
# define RTPathFilenameExUtf16                          RT_MANGLER(RTPathFilenameExUtf16)
# define RTPathFindCommon                               RT_MANGLER(RTPathFindCommon)
# define RTPathFindCommonEx                             RT_MANGLER(RTPathFindCommonEx)
# define RTPathGetCurrent                               RT_MANGLER(RTPathGetCurrent)
# define RTPathGetCurrentDrive                          RT_MANGLER(RTPathGetCurrentDrive)
# define RTPathGetCurrentOnDrive                        RT_MANGLER(RTPathGetCurrentOnDrive)
# define RTPathGetMode                                  RT_MANGLER(RTPathGetMode)
# define RTPathGlob                                     RT_MANGLER(RTPathGlob)
# define RTPathGlobFree                                 RT_MANGLER(RTPathGlobFree)
# define RTPathHasSuffix                                RT_MANGLER(RTPathHasSuffix)
# define RTPathHasPath                                  RT_MANGLER(RTPathHasPath)
# define RTPathIsSame                                   RT_MANGLER(RTPathIsSame)
# define RTPathJoin                                     RT_MANGLER(RTPathJoin)
# define RTPathJoinA                                    RT_MANGLER(RTPathJoinA)
# define RTPathJoinEx                                   RT_MANGLER(RTPathJoinEx)
# define RTPathParentLength                             RT_MANGLER(RTPathParentLength)
# define RTPathParentLengthEx                           RT_MANGLER(RTPathParentLengthEx)
# define RTPathParse                                    RT_MANGLER(RTPathParse)
# define RTPathParsedReassemble                         RT_MANGLER(RTPathParsedReassemble)
# define RTPathParseSimple                              RT_MANGLER(RTPathParseSimple)
# define RTPathPurgeFilename                            RT_MANGLER(RTPathPurgeFilename)
# define RTPathQueryInfo                                RT_MANGLER(RTPathQueryInfo)
# define RTPathQueryInfoEx                              RT_MANGLER(RTPathQueryInfoEx)
# define RTPathReal                                     RT_MANGLER(RTPathReal)
# define RTPathRealDup                                  RT_MANGLER(RTPathRealDup)
# define RTPathRename                                   RT_MANGLER(RTPathRename)
# define RTPathRmCmd                                    RT_MANGLER(RTPathRmCmd)
# define RTPathSetCurrent                               RT_MANGLER(RTPathSetCurrent)
# define RTPathSetMode                                  RT_MANGLER(RTPathSetMode)                   /* not-win */
# define RTPathSetOwner                                 RT_MANGLER(RTPathSetOwner)                  /* not-win */
# define RTPathSetOwnerEx                               RT_MANGLER(RTPathSetOwnerEx)                /* not-win */
# define RTPathSetTimes                                 RT_MANGLER(RTPathSetTimes)
# define RTPathSetTimesEx                               RT_MANGLER(RTPathSetTimesEx)
# define RTPathSharedLibs                               RT_MANGLER(RTPathSharedLibs)
# define RTPathSkipRootSpec                             RT_MANGLER(RTPathSkipRootSpec)
# define RTPathSplit                                    RT_MANGLER(RTPathSplit)
# define RTPathSplitATag                                RT_MANGLER(RTPathSplitATag)
# define RTPathSplitFree                                RT_MANGLER(RTPathSplitFree)
# define RTPathSplitReassemble                          RT_MANGLER(RTPathSplitReassemble)
# define RTPathStartsWith                               RT_MANGLER(RTPathStartsWith)
# define RTPathStartsWithRoot                           RT_MANGLER(RTPathStartsWithRoot)
# define RTPathStripSuffix                              RT_MANGLER(RTPathStripSuffix)
# define RTPathStripFilename                            RT_MANGLER(RTPathStripFilename)
# define RTPathStripTrailingSlash                       RT_MANGLER(RTPathStripTrailingSlash)
# define RTPathTemp                                     RT_MANGLER(RTPathTemp)
# define RTPathTraverseList                             RT_MANGLER(RTPathTraverseList)
# define RTPathUnlink                                   RT_MANGLER(RTPathUnlink)
# define RTPathUserDocuments                            RT_MANGLER(RTPathUserDocuments)
# define RTPathUserHome                                 RT_MANGLER(RTPathUserHome)
# define RTPipeClose                                    RT_MANGLER(RTPipeClose)
# define RTPipeCloseEx                                  RT_MANGLER(RTPipeCloseEx)
# define RTPipeCreate                                   RT_MANGLER(RTPipeCreate)
# define RTPipeFlush                                    RT_MANGLER(RTPipeFlush)
# define RTPipeFromNative                               RT_MANGLER(RTPipeFromNative)
# define RTPipeQueryInfo                                RT_MANGLER(RTPipeQueryInfo)
# define RTPipeQueryReadable                            RT_MANGLER(RTPipeQueryReadable)
# define RTPipeRead                                     RT_MANGLER(RTPipeRead)
# define RTPipeReadBlocking                             RT_MANGLER(RTPipeReadBlocking)
# define RTPipeSelectOne                                RT_MANGLER(RTPipeSelectOne)
# define RTPipeToNative                                 RT_MANGLER(RTPipeToNative)
# define RTPipeWrite                                    RT_MANGLER(RTPipeWrite)
# define RTPipeWriteBlocking                            RT_MANGLER(RTPipeWriteBlocking)
# define RTPoll                                         RT_MANGLER(RTPoll)
# define RTPollNoResume                                 RT_MANGLER(RTPollNoResume)
# define RTPollSetAdd                                   RT_MANGLER(RTPollSetAdd)
# define RTPollSetCreate                                RT_MANGLER(RTPollSetCreate)
# define RTPollSetDestroy                               RT_MANGLER(RTPollSetDestroy)
# define RTPollSetEventsChange                          RT_MANGLER(RTPollSetEventsChange)
# define RTPollSetGetCount                              RT_MANGLER(RTPollSetGetCount)
# define RTPollSetQueryHandle                           RT_MANGLER(RTPollSetQueryHandle)
# define RTPollSetRemove                                RT_MANGLER(RTPollSetRemove)
# define RTPowerNotificationDeregister                  RT_MANGLER(RTPowerNotificationDeregister) /* r0drv */
# define RTPowerNotificationRegister                    RT_MANGLER(RTPowerNotificationRegister) /* r0drv */
# define RTPowerSignalEvent                             RT_MANGLER(RTPowerSignalEvent)         /* r0drv */
# define RTPrintf                                       RT_MANGLER(RTPrintf)
# define RTPrintfV                                      RT_MANGLER(RTPrintfV)
# define RTProcCreate                                   RT_MANGLER(RTProcCreate)
# define RTProcCreateEx                                 RT_MANGLER(RTProcCreateEx)
# define RTProcDaemonize                                RT_MANGLER(RTProcDaemonize)
# define RTProcDaemonizeUsingFork                       RT_MANGLER(RTProcDaemonizeUsingFork)
# define RTProcExecutablePath                           RT_MANGLER(RTProcExecutablePath)
# define RTProcGetAffinityMask                          RT_MANGLER(RTProcGetAffinityMask)
# define RTProcGetExecutablePath                        RT_MANGLER(RTProcGetExecutablePath)
# define RTProcGetPriority                              RT_MANGLER(RTProcGetPriority)
# define RTProcIsRunningByName                          RT_MANGLER(RTProcIsRunningByName)
# define RTProcQueryParent                              RT_MANGLER(RTProcQueryParent)
# define RTProcQueryUsername                            RT_MANGLER(RTProcQueryUsername)
# define RTProcQueryUsernameA                           RT_MANGLER(RTProcQueryUsernameA)
# define RTProcSelf                                     RT_MANGLER(RTProcSelf)
# define RTProcSetPriority                              RT_MANGLER(RTProcSetPriority)
# define RTProcShortName                                RT_MANGLER(RTProcShortName)
# define RTProcSignalName                               RT_MANGLER(RTProcSignalName)
# define RTProcTerminate                                RT_MANGLER(RTProcTerminate)
# define RTProcWait                                     RT_MANGLER(RTProcWait)
# define RTProcWaitNoResume                             RT_MANGLER(RTProcWaitNoResume)
# define RTR0AssertPanicSystem                          RT_MANGLER(RTR0AssertPanicSystem)      /* r0drv */
# define RTR0DbgKrnlInfoOpen                            RT_MANGLER(RTR0DbgKrnlInfoOpen)        /* r0drv */
# define RTR0DbgKrnlInfoQueryMember                     RT_MANGLER(RTR0DbgKrnlInfoQueryMember) /* r0drv */
# define RTR0DbgKrnlInfoQuerySize                       RT_MANGLER(RTR0DbgKrnlInfoQuerySize)   /* r0drv */
# define RTR0DbgKrnlInfoQuerySymbol                     RT_MANGLER(RTR0DbgKrnlInfoQuerySymbol) /* r0drv */
# define RTR0DbgKrnlInfoGetSymbol                       RT_MANGLER(RTR0DbgKrnlInfoGetSymbol)   /* r0drv */
# define RTR0DbgKrnlInfoRelease                         RT_MANGLER(RTR0DbgKrnlInfoRelease)     /* r0drv */
# define RTR0DbgKrnlInfoRetain                          RT_MANGLER(RTR0DbgKrnlInfoRetain)      /* r0drv */
# define RTR0Init                                       RT_MANGLER(RTR0Init)                   /* r0drv */
# define RTR0MemAreKrnlAndUsrDifferent                  RT_MANGLER(RTR0MemAreKrnlAndUsrDifferent) /* r0drv */
# define RTR0MemKernelIsValidAddr                       RT_MANGLER(RTR0MemKernelIsValidAddr)   /* r0drv */
# define RTR0MemObjAddress                              RT_MANGLER(RTR0MemObjAddress)          /* r0drv */
# define RTR0MemObjAddressR3                            RT_MANGLER(RTR0MemObjAddressR3)        /* r0drv */
# define RTR0MemKernelCopyFrom                          RT_MANGLER(RTR0MemKernelCopyFrom)      /* r0drv */
# define RTR0MemKernelCopyTo                            RT_MANGLER(RTR0MemKernelCopyTo)        /* r0drv */
# define RTR0MemObjAllocContTag                         RT_MANGLER(RTR0MemObjAllocContTag)     /* r0drv */
# define RTR0MemObjAllocLargeTag                        RT_MANGLER(RTR0MemObjAllocLargeTag)    /* r0drv */
# define RTR0MemObjAllocLowTag                          RT_MANGLER(RTR0MemObjAllocLowTag)      /* r0drv */
# define RTR0MemObjAllocPageTag                         RT_MANGLER(RTR0MemObjAllocPageTag)     /* r0drv */
# define RTR0MemObjAllocPhysExTag                       RT_MANGLER(RTR0MemObjAllocPhysExTag)   /* r0drv */
# define RTR0MemObjAllocPhysNCTag                       RT_MANGLER(RTR0MemObjAllocPhysNCTag)   /* r0drv */
# define RTR0MemObjAllocPhysTag                         RT_MANGLER(RTR0MemObjAllocPhysTag)     /* r0drv */
# define RTR0MemObjEnterPhysTag                         RT_MANGLER(RTR0MemObjEnterPhysTag)     /* r0drv */
# define RTR0MemObjFree                                 RT_MANGLER(RTR0MemObjFree)             /* r0drv */
# define RTR0MemObjGetPagePhysAddr                      RT_MANGLER(RTR0MemObjGetPagePhysAddr)  /* r0drv */
# define RTR0MemObjIsMapping                            RT_MANGLER(RTR0MemObjIsMapping)        /* r0drv */
# define RTR0MemObjLockKernelTag                        RT_MANGLER(RTR0MemObjLockKernelTag)    /* r0drv */
# define RTR0MemObjLockUserTag                          RT_MANGLER(RTR0MemObjLockUserTag)      /* r0drv */
# define RTR0MemObjMapKernelExTag                       RT_MANGLER(RTR0MemObjMapKernelExTag)   /* r0drv */
# define RTR0MemObjMapKernelTag                         RT_MANGLER(RTR0MemObjMapKernelTag)     /* r0drv */
# define RTR0MemObjMapUserTag                           RT_MANGLER(RTR0MemObjMapUserTag)       /* r0drv */
# define RTR0MemObjMapUserExTag                         RT_MANGLER(RTR0MemObjMapUserExTag)     /* r0drv */
# define RTR0MemObjProtect                              RT_MANGLER(RTR0MemObjProtect)          /* r0drv */
# define RTR0MemObjReserveKernelTag                     RT_MANGLER(RTR0MemObjReserveKernelTag) /* r0drv */
# define RTR0MemObjReserveUserTag                       RT_MANGLER(RTR0MemObjReserveUserTag)   /* r0drv */
# define RTR0MemObjSize                                 RT_MANGLER(RTR0MemObjSize)             /* r0drv */
# define RTR0MemObjWasZeroInitialized                   RT_MANGLER(RTR0MemObjWasZeroInitialized)/* r0drv */
# define RTR0MemObjZeroInitialize                       RT_MANGLER(RTR0MemObjZeroInitialize)   /* r0drv */
# define RTR0MemUserCopyFrom                            RT_MANGLER(RTR0MemUserCopyFrom)        /* r0drv */
# define RTR0MemUserCopyTo                              RT_MANGLER(RTR0MemUserCopyTo)          /* r0drv */
# define RTR0MemUserIsValidAddr                         RT_MANGLER(RTR0MemUserIsValidAddr)     /* r0drv */
# define rtR0MemObjLinuxVirtToPage                      RT_MANGLER(rtR0MemObjLinuxVirtToPage)  /* r0drv linux-only */
# define RTR0ProcHandleSelf                             RT_MANGLER(RTR0ProcHandleSelf)         /* r0drv */
# define RTR0Term                                       RT_MANGLER(RTR0Term)                   /* r0drv */
# define RTR0TermForced                                 RT_MANGLER(RTR0TermForced)             /* r0drv */
# define RTR3InitDll                                    RT_MANGLER(RTR3InitDll)
# define RTR3InitExe                                    RT_MANGLER(RTR3InitExe)
# define RTR3InitExeNoArguments                         RT_MANGLER(RTR3InitExeNoArguments)
# define RTR3InitEx                                     RT_MANGLER(RTR3InitEx)
# define RTR3InitIsInitialized                          RT_MANGLER(RTR3InitIsInitialized)
# define RTR3InitIsUnobtrusive                          RT_MANGLER(RTR3InitIsUnobtrusive)
# define rtR3MemAlloc                                   RT_MANGLER(rtR3MemAlloc)
# define rtR3MemFree                                    RT_MANGLER(rtR3MemFree)
# define rtR3MemRealloc                                 RT_MANGLER(rtR3MemRealloc)
# define RTRCInit                                       RT_MANGLER(RTRCInit)
# define RTRCTerm                                       RT_MANGLER(RTRCTerm)
# define RTRandAdvBytes                                 RT_MANGLER(RTRandAdvBytes)
# define RTRandAdvCreateParkMiller                      RT_MANGLER(RTRandAdvCreateParkMiller)
# define RTRandAdvCreateSystemFaster                    RT_MANGLER(RTRandAdvCreateSystemFaster)
# define RTRandAdvCreateSystemTruer                     RT_MANGLER(RTRandAdvCreateSystemTruer)
# define RTRandAdvDestroy                               RT_MANGLER(RTRandAdvDestroy)
# define RTRandAdvRestoreState                          RT_MANGLER(RTRandAdvRestoreState)
# define RTRandAdvS32                                   RT_MANGLER(RTRandAdvS32)
# define RTRandAdvS32Ex                                 RT_MANGLER(RTRandAdvS32Ex)
# define RTRandAdvS64                                   RT_MANGLER(RTRandAdvS64)
# define RTRandAdvS64Ex                                 RT_MANGLER(RTRandAdvS64Ex)
# define RTRandAdvSaveState                             RT_MANGLER(RTRandAdvSaveState)
# define RTRandAdvSeed                                  RT_MANGLER(RTRandAdvSeed)
# define RTRandAdvU32                                   RT_MANGLER(RTRandAdvU32)
# define RTRandAdvU32Ex                                 RT_MANGLER(RTRandAdvU32Ex)
# define RTRandAdvU64                                   RT_MANGLER(RTRandAdvU64)
# define RTRandAdvU64Ex                                 RT_MANGLER(RTRandAdvU64Ex)
# define RTRandBytes                                    RT_MANGLER(RTRandBytes)
# define RTRandS32                                      RT_MANGLER(RTRandS32)
# define RTRandS32Ex                                    RT_MANGLER(RTRandS32Ex)
# define RTRandS64                                      RT_MANGLER(RTRandS64)
# define RTRandS64Ex                                    RT_MANGLER(RTRandS64Ex)
# define RTRandU32                                      RT_MANGLER(RTRandU32)
# define RTRandU32Ex                                    RT_MANGLER(RTRandU32Ex)
# define RTRandU64                                      RT_MANGLER(RTRandU64)
# define RTRandU64Ex                                    RT_MANGLER(RTRandU64Ex)
# define RTReqPoolAlloc                                 RT_MANGLER(RTReqPoolAlloc)
# define RTReqPoolCallEx                                RT_MANGLER(RTReqPoolCallEx)
# define RTReqPoolCallExV                               RT_MANGLER(RTReqPoolCallExV)
# define RTReqPoolCallWait                              RT_MANGLER(RTReqPoolCallWait)
# define RTReqPoolCallNoWait                            RT_MANGLER(RTReqPoolCallNoWait)
# define RTReqPoolCallVoidWait                          RT_MANGLER(RTReqPoolCallVoidWait)
# define RTReqPoolCallVoidNoWait                        RT_MANGLER(RTReqPoolCallVoidNoWait)
# define RTReqPoolCreate                                RT_MANGLER(RTReqPoolCreate)
# define RTReqPoolGetCfgVar                             RT_MANGLER(RTReqPoolGetCfgVar)
# define RTReqPoolGetStat                               RT_MANGLER(RTReqPoolGetStat)
# define RTReqPoolRetain                                RT_MANGLER(RTReqPoolRetain)
# define RTReqPoolRelease                               RT_MANGLER(RTReqPoolRelease)
# define RTReqPoolSetCfgVar                             RT_MANGLER(RTReqPoolSetCfgVar)
# define RTReqQueueAlloc                                RT_MANGLER(RTReqQueueAlloc)
# define RTReqQueueCall                                 RT_MANGLER(RTReqQueueCall)
# define RTReqQueueCallEx                               RT_MANGLER(RTReqQueueCallEx)
# define RTReqQueueCallV                                RT_MANGLER(RTReqQueueCallV)
# define RTReqQueueCallVoid                             RT_MANGLER(RTReqQueueCallVoid)
# define RTReqQueueCreate                               RT_MANGLER(RTReqQueueCreate)
# define RTReqQueueDestroy                              RT_MANGLER(RTReqQueueDestroy)
# define RTReqQueueIsBusy                               RT_MANGLER(RTReqQueueIsBusy)
# define RTReqQueueProcess                              RT_MANGLER(RTReqQueueProcess)
# define RTReqCancel                                    RT_MANGLER(RTReqCancel)
# define RTReqRelease                                   RT_MANGLER(RTReqRelease)
# define RTReqRetain                                    RT_MANGLER(RTReqRetain)
# define RTReqSubmit                                    RT_MANGLER(RTReqSubmit)
# define RTReqWait                                      RT_MANGLER(RTReqWait)
# define RTReqGetStatus                                 RT_MANGLER(RTReqGetStatus)
# define RTS3BucketsDestroy                             RT_MANGLER(RTS3BucketsDestroy)
# define RTS3Create                                     RT_MANGLER(RTS3Create)
# define RTS3CreateBucket                               RT_MANGLER(RTS3CreateBucket)
# define RTS3DeleteBucket                               RT_MANGLER(RTS3DeleteBucket)
# define RTS3DeleteKey                                  RT_MANGLER(RTS3DeleteKey)
# define RTS3Destroy                                    RT_MANGLER(RTS3Destroy)
# define RTS3GetBucketKeys                              RT_MANGLER(RTS3GetBucketKeys)
# define RTS3GetBuckets                                 RT_MANGLER(RTS3GetBuckets)
# define RTS3GetKey                                     RT_MANGLER(RTS3GetKey)
# define RTS3KeysDestroy                                RT_MANGLER(RTS3KeysDestroy)
# define RTS3PutKey                                     RT_MANGLER(RTS3PutKey)
# define RTS3SetProgressCallback                        RT_MANGLER(RTS3SetProgressCallback)
# define RTScriptLexConsumeToken                        RT_MANGLER(RTScriptLexConsumeToken)
# define RTScriptLexConsumeCh                           RT_MANGLER(RTScriptLexConsumeCh)
# define RTScriptLexConsumeChEx                         RT_MANGLER(RTScriptLexConsumeChEx)
# define RTScriptLexCreateFromReader                    RT_MANGLER(RTScriptLexCreateFromReader)
# define RTScriptLexCreateFromString                    RT_MANGLER(RTScriptLexCreateFromString)
# define RTScriptLexCreateFromFile                      RT_MANGLER(RTScriptLexCreateFromFile)
# define RTScriptLexDestroy                             RT_MANGLER(RTScriptLexDestroy)
# define RTScriptLexGetCh                               RT_MANGLER(RTScriptLexGetCh)
# define RTScriptLexGetChEx                             RT_MANGLER(RTScriptLexGetChEx)
# define RTScriptLexGetTokenType                        RT_MANGLER(RTScriptLexGetTokenType)
# define RTScriptLexPeekCh                              RT_MANGLER(RTScriptLexPeekCh)
# define RTScriptLexPeekChEx                            RT_MANGLER(RTScriptLexPeekChEx)
# define RTScriptLexPeekNextTokenType                   RT_MANGLER(RTScriptLexPeekNextTokenType)
# define RTScriptLexProduceTokError                     RT_MANGLER(RTScriptLexProduceTokError)
# define RTScriptLexProduceTokIde                       RT_MANGLER(RTScriptLexProduceTokIde)
# define RTScriptLexQueryToken                          RT_MANGLER(RTScriptLexQueryToken)
# define RTScriptLexScanIdentifier                      RT_MANGLER(RTScriptLexScanIdentifier)
# define RTScriptLexScanNumber                          RT_MANGLER(RTScriptLexScanNumber)
# define RTScriptLexScanStringLiteralC                  RT_MANGLER(RTScriptLexScanStringLiteralC)
# define RTScriptLexScanStringLiteralPascal             RT_MANGLER(RTScriptLexScanStringLiteralPascal)
# define RTScriptLexSkipWhitespace                      RT_MANGLER(RTScriptLexSkipWhitespace)
# define RTSemEventAddSignaller                         RT_MANGLER(RTSemEventAddSignaller)
# define RTSemEventCreate                               RT_MANGLER(RTSemEventCreate)
# define RTSemEventCreateEx                             RT_MANGLER(RTSemEventCreateEx)
# define RTSemEventDestroy                              RT_MANGLER(RTSemEventDestroy)
# define RTSemEventGetResolution                        RT_MANGLER(RTSemEventGetResolution)    /* r0drv */
# define RTSemEventIsSignalSafe                         RT_MANGLER(RTSemEventIsSignalSafe)     /* r0drv */
# define RTSemEventMultiAddSignaller                    RT_MANGLER(RTSemEventMultiAddSignaller)
# define RTSemEventMultiCreate                          RT_MANGLER(RTSemEventMultiCreate)
# define RTSemEventMultiCreateEx                        RT_MANGLER(RTSemEventMultiCreateEx)
# define RTSemEventMultiDestroy                         RT_MANGLER(RTSemEventMultiDestroy)
# define RTSemEventMultiGetResolution                   RT_MANGLER(RTSemEventMultiGetResolution) /* r0drv */
# define RTSemEventMultiIsSignalSafe                    RT_MANGLER(RTSemEventMultiIsSignalSafe) /* r0drv */
# define RTSemEventMultiRemoveSignaller                 RT_MANGLER(RTSemEventMultiRemoveSignaller)
# define RTSemEventMultiReset                           RT_MANGLER(RTSemEventMultiReset)
# define RTSemEventMultiSetSignaller                    RT_MANGLER(RTSemEventMultiSetSignaller)
# define RTSemEventMultiSignal                          RT_MANGLER(RTSemEventMultiSignal)
# define RTSemEventMultiWait                            RT_MANGLER(RTSemEventMultiWait)
# define RTSemEventMultiWaitEx                          RT_MANGLER(RTSemEventMultiWaitEx)
# define RTSemEventMultiWaitEx                          RT_MANGLER(RTSemEventMultiWaitEx)      /* r0drv */
# define RTSemEventMultiWaitExDebug                     RT_MANGLER(RTSemEventMultiWaitExDebug)
# define RTSemEventMultiWaitExDebug                     RT_MANGLER(RTSemEventMultiWaitExDebug) /* r0drv */
# define RTSemEventMultiWaitNoResume                    RT_MANGLER(RTSemEventMultiWaitNoResume)
# define RTSemEventRemoveSignaller                      RT_MANGLER(RTSemEventRemoveSignaller)
# define RTSemEventSetSignaller                         RT_MANGLER(RTSemEventSetSignaller)
# define RTSemEventSignal                               RT_MANGLER(RTSemEventSignal)
# define RTSemEventWait                                 RT_MANGLER(RTSemEventWait)
# define RTSemEventWaitEx                               RT_MANGLER(RTSemEventWaitEx)           /* r0drv */
# define RTSemEventWaitExDebug                          RT_MANGLER(RTSemEventWaitExDebug)      /* r0drv */
# define RTSemEventWaitNoResume                         RT_MANGLER(RTSemEventWaitNoResume)
# define RTSemFastMutexCreate                           RT_MANGLER(RTSemFastMutexCreate)
# define RTSemFastMutexDestroy                          RT_MANGLER(RTSemFastMutexDestroy)
# define RTSemFastMutexRelease                          RT_MANGLER(RTSemFastMutexRelease)
# define RTSemFastMutexRequest                          RT_MANGLER(RTSemFastMutexRequest)
# define RTSemMutexCreate                               RT_MANGLER(RTSemMutexCreate)
# define RTSemMutexCreateEx                             RT_MANGLER(RTSemMutexCreateEx)
# define RTSemMutexDestroy                              RT_MANGLER(RTSemMutexDestroy)
# define RTSemMutexIsOwned                              RT_MANGLER(RTSemMutexIsOwned)
# define RTSemMutexRelease                              RT_MANGLER(RTSemMutexRelease)
# define RTSemMutexRequest                              RT_MANGLER(RTSemMutexRequest)
# define RTSemMutexRequestDebug                         RT_MANGLER(RTSemMutexRequestDebug)
# define RTSemMutexRequestNoResume                      RT_MANGLER(RTSemMutexRequestNoResume)
# define RTSemMutexRequestNoResumeDebug                 RT_MANGLER(RTSemMutexRequestNoResumeDebug)
# define RTSemMutexSetSubClass                          RT_MANGLER(RTSemMutexSetSubClass)
# define RTSemPing                                      RT_MANGLER(RTSemPing)
# define RTSemPingPongDelete                            RT_MANGLER(RTSemPingPongDelete)
# define RTSemPingPongInit                              RT_MANGLER(RTSemPingPongInit)
# define RTSemPingWait                                  RT_MANGLER(RTSemPingWait)
# define RTSemPong                                      RT_MANGLER(RTSemPong)
# define RTSemPongWait                                  RT_MANGLER(RTSemPongWait)
# define RTSemRWCreate                                  RT_MANGLER(RTSemRWCreate)
# define RTSemRWCreateEx                                RT_MANGLER(RTSemRWCreateEx)
# define RTSemRWDestroy                                 RT_MANGLER(RTSemRWDestroy)
# define RTSemRWGetReadCount                            RT_MANGLER(RTSemRWGetReadCount)
# define RTSemRWGetWriteRecursion                       RT_MANGLER(RTSemRWGetWriteRecursion)
# define RTSemRWGetWriterReadRecursion                  RT_MANGLER(RTSemRWGetWriterReadRecursion)
# define RTSemRWIsReadOwner                             RT_MANGLER(RTSemRWIsReadOwner)
# define RTSemRWIsWriteOwner                            RT_MANGLER(RTSemRWIsWriteOwner)
# define RTSemRWReleaseRead                             RT_MANGLER(RTSemRWReleaseRead)
# define RTSemRWReleaseWrite                            RT_MANGLER(RTSemRWReleaseWrite)
# define RTSemRWRequestRead                             RT_MANGLER(RTSemRWRequestRead)
# define RTSemRWRequestReadDebug                        RT_MANGLER(RTSemRWRequestReadDebug)
# define RTSemRWRequestReadNoResume                     RT_MANGLER(RTSemRWRequestReadNoResume)
# define RTSemRWRequestReadNoResumeDebug                RT_MANGLER(RTSemRWRequestReadNoResumeDebug)
# define RTSemRWRequestWrite                            RT_MANGLER(RTSemRWRequestWrite)
# define RTSemRWRequestWriteDebug                       RT_MANGLER(RTSemRWRequestWriteDebug)
# define RTSemRWRequestWriteNoResume                    RT_MANGLER(RTSemRWRequestWriteNoResume)
# define RTSemRWRequestWriteNoResumeDebug               RT_MANGLER(RTSemRWRequestWriteNoResumeDebug)
# define RTSemRWSetSubClass                             RT_MANGLER(RTSemRWSetSubClass)
# define RTSemSpinMutexCreate                           RT_MANGLER(RTSemSpinMutexCreate)
# define RTSemSpinMutexDestroy                          RT_MANGLER(RTSemSpinMutexDestroy)
# define RTSemSpinMutexRelease                          RT_MANGLER(RTSemSpinMutexRelease)
# define RTSemSpinMutexRequest                          RT_MANGLER(RTSemSpinMutexRequest)
# define RTSemSpinMutexTryRequest                       RT_MANGLER(RTSemSpinMutexTryRequest)
# define RTSemXRoadsCreate                              RT_MANGLER(RTSemXRoadsCreate)
# define RTSemXRoadsDestroy                             RT_MANGLER(RTSemXRoadsDestroy)
# define RTSemXRoadsEWEnter                             RT_MANGLER(RTSemXRoadsEWEnter)
# define RTSemXRoadsEWLeave                             RT_MANGLER(RTSemXRoadsEWLeave)
# define RTSemXRoadsNSEnter                             RT_MANGLER(RTSemXRoadsNSEnter)
# define RTSemXRoadsNSLeave                             RT_MANGLER(RTSemXRoadsNSLeave)
# define RTSerialPortOpen                               RT_MANGLER(RTSerialPortOpen)
# define RTSerialPortClose                              RT_MANGLER(RTSerialPortClose)
# define RTSerialPortToNative                           RT_MANGLER(RTSerialPortToNative)
# define RTSerialPortRead                               RT_MANGLER(RTSerialPortRead)
# define RTSerialPortReadNB                             RT_MANGLER(RTSerialPortReadNB)
# define RTSerialPortWrite                              RT_MANGLER(RTSerialPortWrite)
# define RTSerialPortWriteNB                            RT_MANGLER(RTSerialPortWriteNB)
# define RTSerialPortCfgQueryCurrent                    RT_MANGLER(RTSerialPortCfgQueryCurrent)
# define RTSerialPortCfgSet                             RT_MANGLER(RTSerialPortCfgSet)
# define RTSerialPortEvtPoll                            RT_MANGLER(RTSerialPortEvtPoll)
# define RTSerialPortEvtPollInterrupt                   RT_MANGLER(RTSerialPortEvtPollInterrupt)
# define RTSerialPortChgBreakCondition                  RT_MANGLER(RTSerialPortChgBreakCondition)
# define RTSerialPortChgStatusLines                     RT_MANGLER(RTSerialPortChgStatusLines)
# define RTSerialPortQueryStatusLines                   RT_MANGLER(RTSerialPortQueryStatusLines)
# define RTSgBufAdvance                                 RT_MANGLER(RTSgBufAdvance)
# define RTSgBufClone                                   RT_MANGLER(RTSgBufClone)
# define RTSgBufCmp                                     RT_MANGLER(RTSgBufCmp)
# define RTSgBufCmpEx                                   RT_MANGLER(RTSgBufCmpEx)
# define RTSgBufCopy                                    RT_MANGLER(RTSgBufCopy)
# define RTSgBufCopyFromBuf                             RT_MANGLER(RTSgBufCopyFromBuf)
# define RTSgBufCopyFromFn                              RT_MANGLER(RTSgBufCopyFromFn)
# define RTSgBufCopyToBuf                               RT_MANGLER(RTSgBufCopyToBuf)
# define RTSgBufCopyToFn                                RT_MANGLER(RTSgBufCopyToFn)
# define RTSgBufInit                                    RT_MANGLER(RTSgBufInit)
# define RTSgBufIsZero                                  RT_MANGLER(RTSgBufIsZero)
# define RTSgBufReset                                   RT_MANGLER(RTSgBufReset)
# define RTSgBufSegArrayCreate                          RT_MANGLER(RTSgBufSegArrayCreate)
# define RTSgBufSet                                     RT_MANGLER(RTSgBufSet)
# define RTSgBufGetNextSegment                          RT_MANGLER(RTSgBufGetNextSegment)
# define RTSha1                                         RT_MANGLER(RTSha1)
# define RTSha1Check                                    RT_MANGLER(RTSha1Check)
# define RTSha1Digest                                   RT_MANGLER(RTSha1Digest)
# define RTSha1DigestFromFile                           RT_MANGLER(RTSha1DigestFromFile)
# define RTSha1Final                                    RT_MANGLER(RTSha1Final)
# define RTSha1FromString                               RT_MANGLER(RTSha1FromString)
# define RTSha1Init                                     RT_MANGLER(RTSha1Init)
# define RTSha1ToString                                 RT_MANGLER(RTSha1ToString)
# define RTSha1Update                                   RT_MANGLER(RTSha1Update)
# define RTSha224                                       RT_MANGLER(RTSha224)
# define RTSha224Check                                  RT_MANGLER(RTSha224Check)
# define RTSha224Final                                  RT_MANGLER(RTSha224Final)
# define RTSha224FromString                             RT_MANGLER(RTSha224FromString)
# define RTSha224Init                                   RT_MANGLER(RTSha224Init)
# define RTSha224ToString                               RT_MANGLER(RTSha224ToString)
# define RTSha224Update                                 RT_MANGLER(RTSha224Update)
# define RTSha224Digest                                 RT_MANGLER(RTSha224Digest)
# define RTSha224DigestFromFile                         RT_MANGLER(RTSha224DigestFromFile)
# define RTSha256                                       RT_MANGLER(RTSha256)
# define RTSha256Check                                  RT_MANGLER(RTSha256Check)
# define RTSha256Final                                  RT_MANGLER(RTSha256Final)
# define RTSha256FromString                             RT_MANGLER(RTSha256FromString)
# define RTSha256Init                                   RT_MANGLER(RTSha256Init)
# define RTSha256ToString                               RT_MANGLER(RTSha256ToString)
# define RTSha256Update                                 RT_MANGLER(RTSha256Update)
# define RTSha256Digest                                 RT_MANGLER(RTSha256Digest)
# define RTSha256DigestFromFile                         RT_MANGLER(RTSha256DigestFromFile)
# define RTSha384                                       RT_MANGLER(RTSha384)
# define RTSha384Check                                  RT_MANGLER(RTSha384Check)
# define RTSha384Final                                  RT_MANGLER(RTSha384Final)
# define RTSha384FromString                             RT_MANGLER(RTSha384FromString)
# define RTSha384Init                                   RT_MANGLER(RTSha384Init)
# define RTSha384ToString                               RT_MANGLER(RTSha384ToString)
# define RTSha384Update                                 RT_MANGLER(RTSha384Update)
# define RTSha512                                       RT_MANGLER(RTSha512)
# define RTSha512Check                                  RT_MANGLER(RTSha512Check)
# define RTSha512Final                                  RT_MANGLER(RTSha512Final)
# define RTSha512FromString                             RT_MANGLER(RTSha512FromString)
# define RTSha512Init                                   RT_MANGLER(RTSha512Init)
# define RTSha512ToString                               RT_MANGLER(RTSha512ToString)
# define RTSha512Update                                 RT_MANGLER(RTSha512Update)
# define RTSha512t224                                   RT_MANGLER(RTSha512t224)
# define RTSha512t224Check                              RT_MANGLER(RTSha512t224Check)
# define RTSha512t224Final                              RT_MANGLER(RTSha512t224Final)
# define RTSha512t224FromString                         RT_MANGLER(RTSha512t224FromString)
# define RTSha512t224Init                               RT_MANGLER(RTSha512t224Init)
# define RTSha512t224ToString                           RT_MANGLER(RTSha512t224ToString)
# define RTSha512t224Update                             RT_MANGLER(RTSha512t224Update)
# define RTSha512t256                                   RT_MANGLER(RTSha512t256)
# define RTSha512t256Check                              RT_MANGLER(RTSha512t256Check)
# define RTSha512t256Final                              RT_MANGLER(RTSha512t256Final)
# define RTSha512t256FromString                         RT_MANGLER(RTSha512t256FromString)
# define RTSha512t256Init                               RT_MANGLER(RTSha512t256Init)
# define RTSha512t256ToString                           RT_MANGLER(RTSha512t256ToString)
# define RTSha512t256Update                             RT_MANGLER(RTSha512t256Update)
# define RTSha3t224                                     RT_MANGLER(RTSha3t224)
# define RTSha3t224Check                                RT_MANGLER(RTSha3t224Check)
# define RTSha3t224Cleanup                              RT_MANGLER(RTSha3t224Cleanup)
# define RTSha3t224Clone                                RT_MANGLER(RTSha3t224Clone)
# define RTSha3t224Init                                 RT_MANGLER(RTSha3t224Init)
# define RTSha3t224Final                                RT_MANGLER(RTSha3t224Final)
# define RTSha3t224FromString                           RT_MANGLER(RTSha3t224FromString)
# define RTSha3t224ToString                             RT_MANGLER(RTSha3t224ToString)
# define RTSha3t224Update                               RT_MANGLER(RTSha3t224Update)
# define RTSha3t256                                     RT_MANGLER(RTSha3t256)
# define RTSha3t256Check                                RT_MANGLER(RTSha3t256Check)
# define RTSha3t256Cleanup                              RT_MANGLER(RTSha3t256Cleanup)
# define RTSha3t256Clone                                RT_MANGLER(RTSha3t256Clone)
# define RTSha3t256Init                                 RT_MANGLER(RTSha3t256Init)
# define RTSha3t256Final                                RT_MANGLER(RTSha3t256Final)
# define RTSha3t256FromString                           RT_MANGLER(RTSha3t256FromString)
# define RTSha3t256ToString                             RT_MANGLER(RTSha3t256ToString)
# define RTSha3t256Update                               RT_MANGLER(RTSha3t256Update)
# define RTSha3t384                                     RT_MANGLER(RTSha3t384)
# define RTSha3t384Check                                RT_MANGLER(RTSha3t384Check)
# define RTSha3t384Cleanup                              RT_MANGLER(RTSha3t384Cleanup)
# define RTSha3t384Clone                                RT_MANGLER(RTSha3t384Clone)
# define RTSha3t384Init                                 RT_MANGLER(RTSha3t384Init)
# define RTSha3t384Final                                RT_MANGLER(RTSha3t384Final)
# define RTSha3t384FromString                           RT_MANGLER(RTSha3t384FromString)
# define RTSha3t384ToString                             RT_MANGLER(RTSha3t384ToString)
# define RTSha3t384Update                               RT_MANGLER(RTSha3t384Update)
# define RTSha3t512                                     RT_MANGLER(RTSha3t512)
# define RTSha3t512Check                                RT_MANGLER(RTSha3t512Check)
# define RTSha3t512Cleanup                              RT_MANGLER(RTSha3t512Cleanup)
# define RTSha3t512Clone                                RT_MANGLER(RTSha3t512Clone)
# define RTSha3t512Init                                 RT_MANGLER(RTSha3t512Init)
# define RTSha3t512Final                                RT_MANGLER(RTSha3t512Final)
# define RTSha3t512FromString                           RT_MANGLER(RTSha3t512FromString)
# define RTSha3t512ToString                             RT_MANGLER(RTSha3t512ToString)
# define RTSha3t512Update                               RT_MANGLER(RTSha3t512Update)
# define RTShMemClose                                   RT_MANGLER(RTShMemClose)
# define RTShMemDelete                                  RT_MANGLER(RTShMemDelete)
# define RTShMemMapRegion                               RT_MANGLER(RTShMemMapRegion)
# define RTShMemOpen                                    RT_MANGLER(RTShMemOpen)
# define RTShMemQuerySize                               RT_MANGLER(RTShMemQuerySize)
# define RTShMemRefCount                                RT_MANGLER(RTShMemRefCount)
# define RTShMemSetSize                                 RT_MANGLER(RTShMemSetSize)
# define RTShMemUnmapRegion                             RT_MANGLER(RTShMemUnmapRegion)
# define RTSocketClose                                  RT_MANGLER(RTSocketClose)
# define RTSocketFromNative                             RT_MANGLER(RTSocketFromNative)
# define RTSocketQueryAddressStr                        RT_MANGLER(RTSocketQueryAddressStr)
# define RTSocketGetLocalAddress                        RT_MANGLER(RTSocketGetLocalAddress)
# define RTSocketGetPeerAddress                         RT_MANGLER(RTSocketGetPeerAddress)
# define RTSocketParseInetAddress                       RT_MANGLER(RTSocketParseInetAddress)
# define RTSocketRead                                   RT_MANGLER(RTSocketRead)
# define RTSocketReadFrom                               RT_MANGLER(RTSocketReadFrom)
# define RTSocketReadNB                                 RT_MANGLER(RTSocketReadNB)
# define RTSocketRelease                                RT_MANGLER(RTSocketRelease)
# define RTSocketRetain                                 RT_MANGLER(RTSocketRetain)
# define RTSocketSelectOne                              RT_MANGLER(RTSocketSelectOne)
# define RTSocketSelectOneEx                            RT_MANGLER(RTSocketSelectOneEx)
# define RTSocketSetInheritance                         RT_MANGLER(RTSocketSetInheritance)
# define RTSocketSgWrite                                RT_MANGLER(RTSocketSgWrite)
# define RTSocketSgWriteL                               RT_MANGLER(RTSocketSgWriteL)
# define RTSocketSgWriteLNB                             RT_MANGLER(RTSocketSgWriteLNB)
# define RTSocketSgWriteLV                              RT_MANGLER(RTSocketSgWriteLV)
# define RTSocketSgWriteLVNB                            RT_MANGLER(RTSocketSgWriteLVNB)
# define RTSocketSgWriteNB                              RT_MANGLER(RTSocketSgWriteNB)
# define RTSocketShutdown                               RT_MANGLER(RTSocketShutdown)
# define RTSocketToNative                               RT_MANGLER(RTSocketToNative)
# define RTSocketWrite                                  RT_MANGLER(RTSocketWrite)
# define RTSocketWriteNB                                RT_MANGLER(RTSocketWriteNB)
# define RTSocketWriteTo                                RT_MANGLER(RTSocketWriteTo)
# define RTSocketWriteToNB                              RT_MANGLER(RTSocketWriteToNB)
# define RTSortApvIsSorted                              RT_MANGLER(RTSortApvIsSorted)
# define RTSortApvShell                                 RT_MANGLER(RTSortApvShell)
# define RTSortIsSorted                                 RT_MANGLER(RTSortIsSorted)
# define RTSortShell                                    RT_MANGLER(RTSortShell)
# define RTSpinlockAcquire                              RT_MANGLER(RTSpinlockAcquire)
# define RTSpinlockAcquireNoInts                        RT_MANGLER(RTSpinlockAcquireNoInts)
# define RTSpinlockCreate                               RT_MANGLER(RTSpinlockCreate)
# define RTSpinlockDestroy                              RT_MANGLER(RTSpinlockDestroy)
# define RTSpinlockRelease                              RT_MANGLER(RTSpinlockRelease)
# define RTStrAAppendExNVTag                            RT_MANGLER(RTStrAAppendExNVTag)
# define RTStrAAppendNTag                               RT_MANGLER(RTStrAAppendNTag)
# define RTStrAAppendTag                                RT_MANGLER(RTStrAAppendTag)
# define RTStrAllocExTag                                RT_MANGLER(RTStrAllocExTag)
# define RTStrAllocTag                                  RT_MANGLER(RTStrAllocTag)
# define RTStrAPrintf2VTag                              RT_MANGLER(RTStrAPrintf2VTag)
# define RTStrAPrintfVTag                               RT_MANGLER(RTStrAPrintfVTag)
# define RTStrATruncateTag                              RT_MANGLER(RTStrATruncateTag)
# define RTStrCacheCreate                               RT_MANGLER(RTStrCacheCreate)
# define RTStrCacheDestroy                              RT_MANGLER(RTStrCacheDestroy)
# define RTStrCacheEnter                                RT_MANGLER(RTStrCacheEnter)
# define RTStrCacheEnterLower                           RT_MANGLER(RTStrCacheEnterLower)
# define RTStrCacheEnterLowerN                          RT_MANGLER(RTStrCacheEnterLowerN)
# define RTStrCacheEnterN                               RT_MANGLER(RTStrCacheEnterN)
# define RTStrCacheGetStats                             RT_MANGLER(RTStrCacheGetStats)
# define RTStrCacheIsRealImpl                           RT_MANGLER(RTStrCacheIsRealImpl)
# define RTStrCacheLength                               RT_MANGLER(RTStrCacheLength)
# define RTStrCacheRelease                              RT_MANGLER(RTStrCacheRelease)
# define RTStrCacheRetain                               RT_MANGLER(RTStrCacheRetain)
# define RTStrCalcLatin1Len                             RT_MANGLER(RTStrCalcLatin1Len)
# define RTStrCalcLatin1LenEx                           RT_MANGLER(RTStrCalcLatin1LenEx)
# define RTStrCalcUtf16Len                              RT_MANGLER(RTStrCalcUtf16Len)
# define RTStrCalcUtf16LenEx                            RT_MANGLER(RTStrCalcUtf16LenEx)
# define RTStrCat                                       RT_MANGLER(RTStrCat)
# define RTStrCatEx                                     RT_MANGLER(RTStrCatEx)
# define RTStrCatP                                      RT_MANGLER(RTStrCatP)
# define RTStrCatPEx                                    RT_MANGLER(RTStrCatPEx)
# define RTStrCmp                                       RT_MANGLER(RTStrCmp)
# define RTStrConvertHexBytes                           RT_MANGLER(RTStrConvertHexBytes)
# define RTStrConvertHexBytesEx                         RT_MANGLER(RTStrConvertHexBytesEx)
# define RTStrCopy                                      RT_MANGLER(RTStrCopy)
# define RTStrCopyEx                                    RT_MANGLER(RTStrCopyEx)
# define RTStrCopyP                                     RT_MANGLER(RTStrCopyP)
# define RTStrCopyPEx                                   RT_MANGLER(RTStrCopyPEx)
# define RTStrCurrentCPToUtf8Tag                        RT_MANGLER(RTStrCurrentCPToUtf8Tag)
# define RTStrConsoleCPToUtf8Tag                        RT_MANGLER(RTStrConsoleCPToUtf8Tag)
# define RTStrDupExTag                                  RT_MANGLER(RTStrDupExTag)
# define RTStrDupNTag                                   RT_MANGLER(RTStrDupNTag)
# define RTStrDupNExTag                                 RT_MANGLER(RTStrDupNExTag)
# define RTStrDupTag                                    RT_MANGLER(RTStrDupTag)
# define RTStrEnd                                       RT_MANGLER(RTStrEnd)
# define RTStrEnd_EndProc                               RT_MANGLER(RTStrEnd_EndProc)
# define RTStrFormat                                    RT_MANGLER(RTStrFormat)
# define RTStrFormatNumber                              RT_MANGLER(RTStrFormatNumber)
# define RTStrFormatR32                                 RT_MANGLER(RTStrFormatR32)
# define RTStrFormatR64                                 RT_MANGLER(RTStrFormatR64)
# define RTStrFormatR80                                 RT_MANGLER(RTStrFormatR80)
# define RTStrFormatR80u2                               RT_MANGLER(RTStrFormatR80u2)
# define RTStrFormatTypeDeregister                      RT_MANGLER(RTStrFormatTypeDeregister)
# define RTStrFormatTypeRegister                        RT_MANGLER(RTStrFormatTypeRegister)
# define RTStrFormatTypeSetUser                         RT_MANGLER(RTStrFormatTypeSetUser)
# define RTStrFormatU128                                RT_MANGLER(RTStrFormatU128)
# define RTStrFormatU256                                RT_MANGLER(RTStrFormatU256)
# define RTStrFormatU512                                RT_MANGLER(RTStrFormatU512)
# define RTStrFormatU16                                 RT_MANGLER(RTStrFormatU16)
# define RTStrFormatU32                                 RT_MANGLER(RTStrFormatU32)
# define RTStrFormatU64                                 RT_MANGLER(RTStrFormatU64)
# define RTStrFormatU8                                  RT_MANGLER(RTStrFormatU8)
# define RTStrFormatV                                   RT_MANGLER(RTStrFormatV)
# define RTStrFree                                      RT_MANGLER(RTStrFree)
# define RTStrGetCpExInternal                           RT_MANGLER(RTStrGetCpExInternal)
# define RTStrGetCpInternal                             RT_MANGLER(RTStrGetCpInternal)
# define RTStrGetCpNExInternal                          RT_MANGLER(RTStrGetCpNExInternal)
# define RTStrHash1                                     RT_MANGLER(RTStrHash1)
# define RTStrHash1ExN                                  RT_MANGLER(RTStrHash1ExN)
# define RTStrHash1ExNV                                 RT_MANGLER(RTStrHash1ExNV)
# define RTStrHash1N                                    RT_MANGLER(RTStrHash1N)
# define RTStrICmp                                      RT_MANGLER(RTStrICmp)
# define RTStrICmpAscii                                 RT_MANGLER(RTStrICmpAscii)
# define RTStrIStartsWith                               RT_MANGLER(RTStrIStartsWith)
# define RTStrIStr                                      RT_MANGLER(RTStrIStr)
# define RTStrIsCaseFoldable                            RT_MANGLER(RTStrIsCaseFoldable)
# define RTStrIsLowerCased                              RT_MANGLER(RTStrIsLowerCased)
# define RTStrIsUpperCased                              RT_MANGLER(RTStrIsUpperCased)
# define RTStrIsValidEncoding                           RT_MANGLER(RTStrIsValidEncoding)
# define RTStrMemFind16                                 RT_MANGLER(RTStrMemFind16)
# define RTStrMemFind16_EndProc                         RT_MANGLER(RTStrMemFind16_EndProc)
# define RTStrMemFind32                                 RT_MANGLER(RTStrMemFind32)
# define RTStrMemFind32_EndProc                         RT_MANGLER(RTStrMemFind32_EndProc)
# define RTStrMemFind64                                 RT_MANGLER(RTStrMemFind64)
# define RTStrMemFind64_EndProc                         RT_MANGLER(RTStrMemFind64_EndProc)
# define RTStrSplit                                     RT_MANGLER(RTStrSplit)
# define RTStrmClearError                               RT_MANGLER(RTStrmClearError)
# define RTStrmClose                                    RT_MANGLER(RTStrmClose)
# define RTStrmError                                    RT_MANGLER(RTStrmError)
# define RTStrmFlush                                    RT_MANGLER(RTStrmFlush)
# define RTStrmGetCh                                    RT_MANGLER(RTStrmGetCh)
# define RTStrmGetLine                                  RT_MANGLER(RTStrmGetLine)
# define RTStrmOpen                                     RT_MANGLER(RTStrmOpen)
# define RTStrmOpenF                                    RT_MANGLER(RTStrmOpenF)
# define RTStrmOpenFV                                   RT_MANGLER(RTStrmOpenFV)
# define RTStrmOpenFileHandle                           RT_MANGLER(RTStrmOpenFileHandle)
# define RTStrmQueryFileHandle                          RT_MANGLER(RTStrmQueryFileHandle)
# define RTStrmPrintf                                   RT_MANGLER(RTStrmPrintf)
# define RTStrmPrintfV                                  RT_MANGLER(RTStrmPrintfV)
# define RTStrmWrappedPrintf                            RT_MANGLER(RTStrmWrappedPrintf)
# define RTStrmWrappedPrintfV                           RT_MANGLER(RTStrmWrappedPrintfV)
# define RTStrmDumpPrintfV                              RT_MANGLER(RTStrmDumpPrintfV)
# define RTStrmPutCh                                    RT_MANGLER(RTStrmPutCh)
# define RTStrmPutStr                                   RT_MANGLER(RTStrmPutStr)
# define RTStrmReadEx                                   RT_MANGLER(RTStrmReadEx)
# define RTStrmRewind                                   RT_MANGLER(RTStrmRewind)
# define RTStrmSetBufferingMode                         RT_MANGLER(RTStrmSetBufferingMode)
# define RTStrmSetMode                                  RT_MANGLER(RTStrmSetMode)
# define RTStrmSeek                                     RT_MANGLER(RTStrmSeek)
# define RTStrmTell                                     RT_MANGLER(RTStrmTell)
# define RTStrmWriteEx                                  RT_MANGLER(RTStrmWriteEx)
# define RTStrmIsTerminal                               RT_MANGLER(RTStrmIsTerminal)
# define RTStrmInputGetEchoChars                        RT_MANGLER(RTStrmInputGetEchoChars)
# define RTStrmInputSetEchoChars                        RT_MANGLER(RTStrmInputSetEchoChars)
# define RTStrmQueryTerminalWidth                       RT_MANGLER(RTStrmQueryTerminalWidth)
# define RTStrNanLongDouble                             RT_MANGLER(RTStrNanLongDouble)
# define RTStrNanDouble                                 RT_MANGLER(RTStrNanDouble)
# define RTStrNanFloat                                  RT_MANGLER(RTStrNanFloat)
# define RTStrNCmp                                      RT_MANGLER(RTStrNCmp)
# define RTStrNICmp                                     RT_MANGLER(RTStrNICmp)
# define RTStrNICmpAscii                                RT_MANGLER(RTStrNICmpAscii)
# define RTStrNLen                                      RT_MANGLER(RTStrNLen)
# define RTStrNLenEx                                    RT_MANGLER(RTStrNLenEx)
# define RTStrPrevCp                                    RT_MANGLER(RTStrPrevCp)
# define RTStrPrintf                                    RT_MANGLER(RTStrPrintf)
# define RTStrPrintfEx                                  RT_MANGLER(RTStrPrintfEx)
# define RTStrPrintfExV                                 RT_MANGLER(RTStrPrintfExV)
# define RTStrPrintfV                                   RT_MANGLER(RTStrPrintfV)
# define RTStrPrintf2                                   RT_MANGLER(RTStrPrintf2)
# define RTStrPrintf2Ex                                 RT_MANGLER(RTStrPrintf2Ex)
# define RTStrPrintf2ExV                                RT_MANGLER(RTStrPrintf2ExV)
# define RTStrPrintf2V                                  RT_MANGLER(RTStrPrintf2V)
# define RTStrPrintHexBytes                             RT_MANGLER(RTStrPrintHexBytes)
# define RTStrPurgeEncoding                             RT_MANGLER(RTStrPurgeEncoding)
# define RTStrPurgeComplementSet                        RT_MANGLER(RTStrPurgeComplementSet)
# define RTStrPutCpInternal                             RT_MANGLER(RTStrPutCpInternal)
# define RTStrReallocTag                                RT_MANGLER(RTStrReallocTag)
# define RTStrSimplePatternMatch                        RT_MANGLER(RTStrSimplePatternMatch)
# define RTStrSimplePatternMultiMatch                   RT_MANGLER(RTStrSimplePatternMultiMatch)
# define RTStrSimplePatternNMatch                       RT_MANGLER(RTStrSimplePatternNMatch)
# define RTStrSpaceDestroy                              RT_MANGLER(RTStrSpaceDestroy)
# define RTStrSpaceEnumerate                            RT_MANGLER(RTStrSpaceEnumerate)
# define RTStrSpaceGet                                  RT_MANGLER(RTStrSpaceGet)
# define RTStrSpaceGetN                                 RT_MANGLER(RTStrSpaceGetN)
# define RTStrSpaceInsert                               RT_MANGLER(RTStrSpaceInsert)
# define RTStrSpaceRemove                               RT_MANGLER(RTStrSpaceRemove)
# define RTStrStartsWith                                RT_MANGLER(RTStrStartsWith)
# define RTStrStr                                       RT_MANGLER(RTStrStr)
# define RTStrStrip                                     RT_MANGLER(RTStrStrip)
# define RTStrStripL                                    RT_MANGLER(RTStrStripL)
# define RTStrStripR                                    RT_MANGLER(RTStrStripR)
# define RTStrToInt16                                   RT_MANGLER(RTStrToInt16)
# define RTStrToInt16Ex                                 RT_MANGLER(RTStrToInt16Ex)
# define RTStrToInt16Full                               RT_MANGLER(RTStrToInt16Full)
# define RTStrToInt32                                   RT_MANGLER(RTStrToInt32)
# define RTStrToInt32Ex                                 RT_MANGLER(RTStrToInt32Ex)
# define RTStrToInt32Full                               RT_MANGLER(RTStrToInt32Full)
# define RTStrToInt64                                   RT_MANGLER(RTStrToInt64)
# define RTStrToInt64Ex                                 RT_MANGLER(RTStrToInt64Ex)
# define RTStrToInt64Full                               RT_MANGLER(RTStrToInt64Full)
# define RTStrToInt8                                    RT_MANGLER(RTStrToInt8)
# define RTStrToInt8Ex                                  RT_MANGLER(RTStrToInt8Ex)
# define RTStrToInt8Full                                RT_MANGLER(RTStrToInt8Full)
# define RTStrToLatin1ExTag                             RT_MANGLER(RTStrToLatin1ExTag)
# define RTStrToLatin1Tag                               RT_MANGLER(RTStrToLatin1Tag)
# define RTStrToLower                                   RT_MANGLER(RTStrToLower)
# define RTStrToUInt16                                  RT_MANGLER(RTStrToUInt16)
# define RTStrToUInt16Ex                                RT_MANGLER(RTStrToUInt16Ex)
# define RTStrToUInt16Full                              RT_MANGLER(RTStrToUInt16Full)
# define RTStrToUInt32                                  RT_MANGLER(RTStrToUInt32)
# define RTStrToUInt32Ex                                RT_MANGLER(RTStrToUInt32Ex)
# define RTStrToUInt32Full                              RT_MANGLER(RTStrToUInt32Full)
# define RTStrToUInt64                                  RT_MANGLER(RTStrToUInt64)
# define RTStrToUInt64Ex                                RT_MANGLER(RTStrToUInt64Ex)
# define RTStrToUInt64Full                              RT_MANGLER(RTStrToUInt64Full)
# define RTStrToUInt8                                   RT_MANGLER(RTStrToUInt8)
# define RTStrToUInt8Ex                                 RT_MANGLER(RTStrToUInt8Ex)
# define RTStrToUInt8Full                               RT_MANGLER(RTStrToUInt8Full)
# define RTStrToFloatEx                                 RT_MANGLER(RTStrToFloatEx)
# define RTStrToDoubleEx                                RT_MANGLER(RTStrToDoubleEx)
# define RTStrToLongDoubleEx                            RT_MANGLER(RTStrToLongDoubleEx)
# define RTStrToUni                                     RT_MANGLER(RTStrToUni)
# define RTStrToUniEx                                   RT_MANGLER(RTStrToUniEx)
# define RTStrToUpper                                   RT_MANGLER(RTStrToUpper)
# define RTStrToUtf16BigExTag                           RT_MANGLER(RTStrToUtf16BigExTag)
# define RTStrToUtf16BigTag                             RT_MANGLER(RTStrToUtf16BigTag)
# define RTStrToUtf16ExTag                              RT_MANGLER(RTStrToUtf16ExTag)
# define RTStrToUtf16Tag                                RT_MANGLER(RTStrToUtf16Tag)
# define RTStrUniLen                                    RT_MANGLER(RTStrUniLen)
# define RTStrUniLenEx                                  RT_MANGLER(RTStrUniLenEx)
# define RTStrUtf8ToCurrentCPTag                        RT_MANGLER(RTStrUtf8ToCurrentCPTag)
# define RTStrUtf8ToCurrentCPExTag                      RT_MANGLER(RTStrUtf8ToCurrentCPExTag)
# define RTStrValidateEncoding                          RT_MANGLER(RTStrValidateEncoding)
# define RTStrValidateEncodingEx                        RT_MANGLER(RTStrValidateEncodingEx)
# define RTStrVersionCompare                            RT_MANGLER(RTStrVersionCompare)
# define RTSymlinkCreate                                RT_MANGLER(RTSymlinkCreate)
# define RTSymlinkDelete                                RT_MANGLER(RTSymlinkDelete)
# define RTSymlinkExists                                RT_MANGLER(RTSymlinkExists)
# define RTSymlinkIsDangling                            RT_MANGLER(RTSymlinkIsDangling)
# define RTSymlinkRead                                  RT_MANGLER(RTSymlinkRead)
# define RTSymlinkReadA                                 RT_MANGLER(RTSymlinkReadA)
# define RTSystemGetPageSize                            RT_MANGLER(RTSystemGetPageSize)
# define RTSystemGetPageOffsetMask                      RT_MANGLER(RTSystemGetPageOffsetMask)
# define RTSystemGetPageShift                           RT_MANGLER(RTSystemGetPageShift)
# define RTSystemPageAlignSize                          RT_MANGLER(RTSystemPageAlignSize)
# define RTSystemQueryFirmwareType                      RT_MANGLER(RTSystemQueryFirmwareType)
# define RTSystemQueryFirmwareBoolean                   RT_MANGLER(RTSystemQueryFirmwareBoolean)
# define RTSystemFirmwareTypeName                       RT_MANGLER(RTSystemFirmwareTypeName)
# define RTSystemIsInsideVM                             RT_MANGLER(RTSystemIsInsideVM)
# define RTSystemQueryAvailableRam                      RT_MANGLER(RTSystemQueryAvailableRam)
# define RTSystemQueryDmiString                         RT_MANGLER(RTSystemQueryDmiString)
# define RTSystemQueryOSInfo                            RT_MANGLER(RTSystemQueryOSInfo)
# define RTSystemQueryTotalRam                          RT_MANGLER(RTSystemQueryTotalRam)
# define RTSystemShutdown                               RT_MANGLER(RTSystemShutdown)
# define RTTarClose                                     RT_MANGLER(RTTarClose)
# define RTTarFileClose                                 RT_MANGLER(RTTarFileClose)
# define RTTarFileGetSize                               RT_MANGLER(RTTarFileGetSize)
# define RTTarFileOpen                                  RT_MANGLER(RTTarFileOpen)
# define RTTarFileReadAt                                RT_MANGLER(RTTarFileReadAt)
# define RTTarFileSetSize                               RT_MANGLER(RTTarFileSetSize)
# define RTTarFileWriteAt                               RT_MANGLER(RTTarFileWriteAt)
# define RTTarOpen                                      RT_MANGLER(RTTarOpen)
# define RTTcpClientCancelConnect                       RT_MANGLER(RTTcpClientCancelConnect)
# define RTTcpClientClose                               RT_MANGLER(RTTcpClientClose)
# define RTTcpClientCloseEx                             RT_MANGLER(RTTcpClientCloseEx)
# define RTTcpClientConnect                             RT_MANGLER(RTTcpClientConnect)
# define RTTcpClientConnectEx                           RT_MANGLER(RTTcpClientConnectEx)
# define RTTcpCreatePair                                RT_MANGLER(RTTcpCreatePair)
# define RTTcpFlush                                     RT_MANGLER(RTTcpFlush)
# define RTTcpGetLocalAddress                           RT_MANGLER(RTTcpGetLocalAddress)
# define RTTcpGetPeerAddress                            RT_MANGLER(RTTcpGetPeerAddress)
# define RTTcpRead                                      RT_MANGLER(RTTcpRead)
# define RTTcpReadNB                                    RT_MANGLER(RTTcpReadNB)
# define RTTcpSelectOne                                 RT_MANGLER(RTTcpSelectOne)
# define RTTcpSelectOneEx                               RT_MANGLER(RTTcpSelectOneEx)
# define RTTcpServerCreate                              RT_MANGLER(RTTcpServerCreate)
# define RTTcpServerCreateEx                            RT_MANGLER(RTTcpServerCreateEx)
# define RTTcpServerDestroy                             RT_MANGLER(RTTcpServerDestroy)
# define RTTcpServerDisconnectClient                    RT_MANGLER(RTTcpServerDisconnectClient)
# define RTTcpServerDisconnectClient2                   RT_MANGLER(RTTcpServerDisconnectClient2)
# define RTTcpServerListen                              RT_MANGLER(RTTcpServerListen)
# define RTTcpServerListen2                             RT_MANGLER(RTTcpServerListen2)
# define RTTcpServerShutdown                            RT_MANGLER(RTTcpServerShutdown)
# define RTTcpSetBufferSize                             RT_MANGLER(RTTcpSetBufferSize)
# define RTTcpSetKeepAlive                              RT_MANGLER(RTTcpSetKeepAlive)
# define RTTcpSetSendCoalescing                         RT_MANGLER(RTTcpSetSendCoalescing)
# define RTTcpSgWrite                                   RT_MANGLER(RTTcpSgWrite)
# define RTTcpSgWriteL                                  RT_MANGLER(RTTcpSgWriteL)
# define RTTcpSgWriteLNB                                RT_MANGLER(RTTcpSgWriteLNB)
# define RTTcpSgWriteLV                                 RT_MANGLER(RTTcpSgWriteLV)
# define RTTcpSgWriteLVNB                               RT_MANGLER(RTTcpSgWriteLVNB)
# define RTTcpSgWriteNB                                 RT_MANGLER(RTTcpSgWriteNB)
# define RTTcpWrite                                     RT_MANGLER(RTTcpWrite)
# define RTTcpWriteNB                                   RT_MANGLER(RTTcpWriteNB)
# define RTTermDeregisterCallback                       RT_MANGLER(RTTermDeregisterCallback)
# define RTTermRegisterCallback                         RT_MANGLER(RTTermRegisterCallback)
# define RTTermRunCallbacks                             RT_MANGLER(RTTermRunCallbacks)
# define RTTestBanner                                   RT_MANGLER(RTTestBanner)
# define RTTestChangeName                               RT_MANGLER(RTTestChangeName)
# define RTTestCreate                                   RT_MANGLER(RTTestCreate)
# define RTTestCreateChild                              RT_MANGLER(RTTestCreateChild)
# define RTTestCreateEx                                 RT_MANGLER(RTTestCreateEx)
# define RTTestDestroy                                  RT_MANGLER(RTTestDestroy)
# define RTTestDisableAssertions                        RT_MANGLER(RTTestDisableAssertions)
# define RTTestErrContext                               RT_MANGLER(RTTestErrContext)
# define RTTestErrContextV                              RT_MANGLER(RTTestErrContextV)
# define RTTestErrorCount                               RT_MANGLER(RTTestErrorCount)
# define RTTestErrorInc                                 RT_MANGLER(RTTestErrorInc)
# define RTTestFailed                                   RT_MANGLER(RTTestFailed)
# define RTTestFailedV                                  RT_MANGLER(RTTestFailedV)
# define RTTestFailureDetails                           RT_MANGLER(RTTestFailureDetails)
# define RTTestFailureDetailsV                          RT_MANGLER(RTTestFailureDetailsV)
# define RTTestGuardedAlloc                             RT_MANGLER(RTTestGuardedAlloc)
# define RTTestGuardedAllocHead                         RT_MANGLER(RTTestGuardedAllocHead)
# define RTTestGuardedAllocTail                         RT_MANGLER(RTTestGuardedAllocTail)
# define RTTestGuardedFree                              RT_MANGLER(RTTestGuardedFree)
# define RTTestIDisableAssertions                       RT_MANGLER(RTTestIDisableAssertions)
# define RTTestIErrContext                              RT_MANGLER(RTTestIErrContext)
# define RTTestIErrContextV                             RT_MANGLER(RTTestIErrContextV)
# define RTTestIErrorCount                              RT_MANGLER(RTTestIErrorCount)
# define RTTestIErrorInc                                RT_MANGLER(RTTestIErrorInc)
# define RTTestIFailed                                  RT_MANGLER(RTTestIFailed)
# define RTTestIFailedRc                                RT_MANGLER(RTTestIFailedRc)
# define RTTestIFailedRcV                               RT_MANGLER(RTTestIFailedRcV)
# define RTTestIFailedV                                 RT_MANGLER(RTTestIFailedV)
# define RTTestIFailureDetails                          RT_MANGLER(RTTestIFailureDetails)
# define RTTestIFailureDetailsV                         RT_MANGLER(RTTestIFailureDetailsV)
# define RTTestInitAndCreate                            RT_MANGLER(RTTestInitAndCreate)
# define RTTestInitExAndCreate                          RT_MANGLER(RTTestInitExAndCreate)
# define RTTestIPassed                                  RT_MANGLER(RTTestIPassed)
# define RTTestIPassedV                                 RT_MANGLER(RTTestIPassedV)
# define RTTestIPrintf                                  RT_MANGLER(RTTestIPrintf)
# define RTTestIPrintfV                                 RT_MANGLER(RTTestIPrintfV)
# define RTTestIRestoreAssertions                       RT_MANGLER(RTTestIRestoreAssertions)
# define RTTestISub                                     RT_MANGLER(RTTestISub)
# define RTTestISubDone                                 RT_MANGLER(RTTestISubDone)
# define RTTestISubF                                    RT_MANGLER(RTTestISubF)
# define RTTestISubV                                    RT_MANGLER(RTTestISubV)
# define RTTestISubSub                                  RT_MANGLER(RTTestISubSub)
# define RTTestISubSubDone                              RT_MANGLER(RTTestISubSubDone)
# define RTTestISubSubF                                 RT_MANGLER(RTTestISubSubF)
# define RTTestISubSubV                                 RT_MANGLER(RTTestISubSubV)
# define RTTestIValue                                   RT_MANGLER(RTTestIValue)
# define RTTestIValueF                                  RT_MANGLER(RTTestIValueF)
# define RTTestIValueV                                  RT_MANGLER(RTTestIValueV)
# define RTTestPassed                                   RT_MANGLER(RTTestPassed)
# define RTTestPassedV                                  RT_MANGLER(RTTestPassedV)
# define RTTestPrintf                                   RT_MANGLER(RTTestPrintf)
# define RTTestPrintfNl                                 RT_MANGLER(RTTestPrintfNl)
# define RTTestPrintfNlV                                RT_MANGLER(RTTestPrintfNlV)
# define RTTestPrintfV                                  RT_MANGLER(RTTestPrintfV)
# define RTTestRestoreAssertions                        RT_MANGLER(RTTestRestoreAssertions)
# define RTTestSetDefault                               RT_MANGLER(RTTestSetDefault)
# define RTTestSkipAndDestroy                           RT_MANGLER(RTTestSkipAndDestroy)
# define RTTestSkipAndDestroyV                          RT_MANGLER(RTTestSkipAndDestroyV)
# define RTTestSkipped                                  RT_MANGLER(RTTestSkipped)
# define RTTestSkippedV                                 RT_MANGLER(RTTestSkippedV)
# define RTTestSub                                      RT_MANGLER(RTTestSub)
# define RTTestSubDone                                  RT_MANGLER(RTTestSubDone)
# define RTTestSubErrorCount                            RT_MANGLER(RTTestSubErrorCount)
# define RTTestSubF                                     RT_MANGLER(RTTestSubF)
# define RTTestSubV                                     RT_MANGLER(RTTestSubV)
# define RTTestSubSub                                   RT_MANGLER(RTTestSubSub)
# define RTTestSubSubDone                               RT_MANGLER(RTTestSubSubDone)
# define RTTestSubSubErrorCount                         RT_MANGLER(RTTestSubSubErrorCount)
# define RTTestSubSubF                                  RT_MANGLER(RTTestSubSubF)
# define RTTestSubSubV                                  RT_MANGLER(RTTestSubSubV)
# define RTTestSummaryAndDestroy                        RT_MANGLER(RTTestSummaryAndDestroy)
# define RTTestValue                                    RT_MANGLER(RTTestValue)
# define RTTestValueF                                   RT_MANGLER(RTTestValueF)
# define RTTestValueV                                   RT_MANGLER(RTTestValueV)
# define RTThreadAdopt                                  RT_MANGLER(RTThreadAdopt)
# define RTThreadBlocking                               RT_MANGLER(RTThreadBlocking)
# define RTThreadCreate                                 RT_MANGLER(RTThreadCreate)
# define RTThreadCreateF                                RT_MANGLER(RTThreadCreateF)
# define RTThreadCreateV                                RT_MANGLER(RTThreadCreateV)
# define RTThreadCtxHookIsEnabled                       RT_MANGLER(RTThreadCtxHookIsEnabled)    /* r0drv */
# define RTThreadCtxHookCreate                          RT_MANGLER(RTThreadCtxHookCreate)       /* r0drv */
# define RTThreadCtxHookDestroy                         RT_MANGLER(RTThreadCtxHookDestroy)      /* r0drv */
# define RTThreadCtxHookDisable                         RT_MANGLER(RTThreadCtxHookDisable)      /* r0drv */
# define RTThreadCtxHookEnable                          RT_MANGLER(RTThreadCtxHookEnable)       /* r0drv */
# define RTThreadFromNative                             RT_MANGLER(RTThreadFromNative)
# define RTThreadGetAffinity                            RT_MANGLER(RTThreadGetAffinity)
# define RTThreadGetExecutionTimeMilli                  RT_MANGLER(RTThreadGetExecutionTimeMilli)
# define RTThreadGetName                                RT_MANGLER(RTThreadGetName)
# define RTThreadGetNative                              RT_MANGLER(RTThreadGetNative)
# define RTThreadGetNativeHandle                        RT_MANGLER(RTThreadGetNativeHandle)
# define RTThreadGetNativeState                         RT_MANGLER(RTThreadGetNativeState)
# define RTThreadGetReallySleeping                      RT_MANGLER(RTThreadGetReallySleeping)
# define RTThreadGetState                               RT_MANGLER(RTThreadGetState)
# define RTThreadGetType                                RT_MANGLER(RTThreadGetType)
# define RTThreadIsInInterrupt                          RT_MANGLER(RTThreadIsInInterrupt)      /* r0drv */
# define RTThreadIsInitialized                          RT_MANGLER(RTThreadIsInitialized)
# define RTThreadIsMain                                 RT_MANGLER(RTThreadIsMain)
# define RTThreadIsSelfAlive                            RT_MANGLER(RTThreadIsSelfAlive)
# define RTThreadIsSelfKnown                            RT_MANGLER(RTThreadIsSelfKnown)
# define RTThreadNativeSelf                             RT_MANGLER(RTThreadNativeSelf)
# define RTThreadControlPokeSignal                      RT_MANGLER(RTThreadControlPokeSignal) /* not-win not-os2 */
# define RTThreadPoke                                   RT_MANGLER(RTThreadPoke) /* not-win not-os2 */
# define RTThreadPreemptDisable                         RT_MANGLER(RTThreadPreemptDisable)     /* r0drv */
# define RTThreadPreemptIsEnabled                       RT_MANGLER(RTThreadPreemptIsEnabled)   /* r0drv */
# define RTThreadPreemptIsPending                       RT_MANGLER(RTThreadPreemptIsPending)   /* r0drv */
# define RTThreadPreemptIsPendingTrusty                 RT_MANGLER(RTThreadPreemptIsPendingTrusty) /* r0drv */
# define RTThreadPreemptIsPossible                      RT_MANGLER(RTThreadPreemptIsPossible)  /* r0drv */
# define RTThreadPreemptRestore                         RT_MANGLER(RTThreadPreemptRestore)     /* r0drv */
# define RTThreadQueryTerminationStatus                 RT_MANGLER(RTThreadQueryTerminationStatus) /* r0drv */
# define RTThreadSelf                                   RT_MANGLER(RTThreadSelf)
# define RTThreadSelfAutoAdopt                          RT_MANGLER(RTThreadSelfAutoAdopt)
# define RTThreadSelfName                               RT_MANGLER(RTThreadSelfName)
# define RTThreadSetAffinity                            RT_MANGLER(RTThreadSetAffinity)
# define RTThreadSetAffinityToCpu                       RT_MANGLER(RTThreadSetAffinityToCpu)
# define RTThreadSetName                                RT_MANGLER(RTThreadSetName)
# define RTThreadSetType                                RT_MANGLER(RTThreadSetType)
# define RTThreadSleep                                  RT_MANGLER(RTThreadSleep)
# define RTThreadSleepNoLog                             RT_MANGLER(RTThreadSleepNoLog)
# define RTThreadStateName                              RT_MANGLER(RTThreadStateName)
# define RTThreadUnblocked                              RT_MANGLER(RTThreadUnblocked)
# define RTThreadUserReset                              RT_MANGLER(RTThreadUserReset)
# define RTThreadUserSignal                             RT_MANGLER(RTThreadUserSignal)
# define RTThreadUserWait                               RT_MANGLER(RTThreadUserWait)
# define RTThreadUserWaitNoResume                       RT_MANGLER(RTThreadUserWaitNoResume)
# define RTThreadWait                                   RT_MANGLER(RTThreadWait)
# define RTThreadWaitNoResume                           RT_MANGLER(RTThreadWaitNoResume)
# define RTThreadYield                                  RT_MANGLER(RTThreadYield)
# define RTTimeCompare                                  RT_MANGLER(RTTimeCompare)
# define RTTimeConvertToZulu                            RT_MANGLER(RTTimeConvertToZulu)
# define RTTimeDbgBad                                   RT_MANGLER(RTTimeDbgBad)
# define RTTimeDbgExpired                               RT_MANGLER(RTTimeDbgExpired)
# define RTTimeDbgRaces                                 RT_MANGLER(RTTimeDbgRaces)
# define RTTimeDbgSteps                                 RT_MANGLER(RTTimeDbgSteps)
# define RTTimeFormatDuration                           RT_MANGLER(RTTimeFormatDuration)
# define RTTimeFormatDurationEx                         RT_MANGLER(RTTimeFormatDurationEx)
# define RTTimeExplode                                  RT_MANGLER(RTTimeExplode)
# define RTTimeImplode                                  RT_MANGLER(RTTimeImplode)
# define RTTimeIsLeapYear                               RT_MANGLER(RTTimeIsLeapYear)
# define RTTimeLocalDeltaNano                           RT_MANGLER(RTTimeLocalDeltaNano)
# define RTTimeLocalDeltaNanoFor                        RT_MANGLER(RTTimeLocalDeltaNanoFor)
# define RTTimeLocalExplode                             RT_MANGLER(RTTimeLocalExplode)
# define RTTimeLocalNormalize                           RT_MANGLER(RTTimeLocalNormalize)
# define RTTimeLocalNow                                 RT_MANGLER(RTTimeLocalNow)
# define RTTimeMilliTS                                  RT_MANGLER(RTTimeMilliTS)
# define RTTimeNanoTS                                   RT_MANGLER(RTTimeNanoTS)
# define RTTimeNanoTSLegacyAsync                                RT_MANGLER(RTTimeNanoTSLegacyAsync)
# define RTTimeNanoTSLegacyAsync_EndProc                        RT_MANGLER(RTTimeNanoTSLegacyAsync_EndProc)
# define RTTimeNanoTSLegacyAsyncUseApicId                       RT_MANGLER(RTTimeNanoTSLegacyAsyncUseApicId)
# define RTTimeNanoTSLegacyAsyncUseApicId_EndProc               RT_MANGLER(RTTimeNanoTSLegacyAsyncUseApicId_EndProc)
# define RTTimeNanoTSLegacyAsyncUseApicIdExt0B                  RT_MANGLER(RTTimeNanoTSLegacyAsyncUseApicIdExt0B)
# define RTTimeNanoTSLegacyAsyncUseApicIdExt0B_EndProc          RT_MANGLER(RTTimeNanoTSLegacyAsyncUseApicIdExt0B_EndProc)
# define RTTimeNanoTSLegacyAsyncUseApicIdExt8000001E            RT_MANGLER(RTTimeNanoTSLegacyAsyncUseApicIdExt8000001E)
# define RTTimeNanoTSLegacyAsyncUseApicIdExt8000001E_EndProc    RT_MANGLER(RTTimeNanoTSLegacyAsyncUseApicIdExt8000001E_EndProc)
# define RTTimeNanoTSLegacyAsyncUseRdtscp                       RT_MANGLER(RTTimeNanoTSLegacyAsyncUseRdtscp)
# define RTTimeNanoTSLegacyAsyncUseRdtscp_EndProc               RT_MANGLER(RTTimeNanoTSLegacyAsyncUseRdtscp_EndProc)
# define RTTimeNanoTSLegacyAsyncUseRdtscpGroupChNumCl           RT_MANGLER(RTTimeNanoTSLegacyAsyncUseRdtscpGroupChNumCl)
# define RTTimeNanoTSLegacyAsyncUseRdtscpGroupChNumCl_EndProc   RT_MANGLER(RTTimeNanoTSLegacyAsyncUseRdtscpGroupChNumCl_EndProc)
# define RTTimeNanoTSLegacyAsyncUseIdtrLim                      RT_MANGLER(RTTimeNanoTSLegacyAsyncUseIdtrLim)
# define RTTimeNanoTSLegacyAsyncUseIdtrLim_EndProc              RT_MANGLER(RTTimeNanoTSLegacyAsyncUseIdtrLim_EndProc)
# define RTTimeNanoTSLegacySyncInvarNoDelta                     RT_MANGLER(RTTimeNanoTSLegacySyncInvarNoDelta)
# define RTTimeNanoTSLegacySyncInvarNoDelta_EndProc             RT_MANGLER(RTTimeNanoTSLegacySyncInvarNoDelta_EndProc)
# define RTTimeNanoTSLegacySyncInvarWithDelta                   RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDelta)
# define RTTimeNanoTSLegacySyncInvarWithDelta_EndProc           RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDelta_EndProc)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseApicId                    RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseApicId)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseApicId_EndProc            RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseApicId_EndProc)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt0B               RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt0B)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt0B_EndProc       RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt0B_EndProc)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt8000001E         RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt8000001E)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt8000001E_EndProc RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseApicIdExt8000001E_EndProc)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseRdtscp          RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseRdtscp)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseRdtscp_EndProc  RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseRdtscp_EndProc)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseIdtrLim         RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseIdtrLim)
# define RTTimeNanoTSLegacySyncInvarWithDeltaUseIdtrLim_EndProc RT_MANGLER(RTTimeNanoTSLegacySyncInvarWithDeltaUseIdtrLim_EndProc)
# define RTTimeNanoTSLFenceAsync                                RT_MANGLER(RTTimeNanoTSLFenceAsync)
# define RTTimeNanoTSLFenceAsync_EndProc                        RT_MANGLER(RTTimeNanoTSLFenceAsync_EndProc)
# define RTTimeNanoTSLFenceAsyncUseApicId                       RT_MANGLER(RTTimeNanoTSLFenceAsyncUseApicId)
# define RTTimeNanoTSLFenceAsyncUseApicId_EndProc               RT_MANGLER(RTTimeNanoTSLFenceAsyncUseApicId_EndProc)
# define RTTimeNanoTSLFenceAsyncUseApicIdExt0B                  RT_MANGLER(RTTimeNanoTSLFenceAsyncUseApicIdExt0B)
# define RTTimeNanoTSLFenceAsyncUseApicIdExt0B_EndProc          RT_MANGLER(RTTimeNanoTSLFenceAsyncUseApicIdExt0B_EndProc)
# define RTTimeNanoTSLFenceAsyncUseApicIdExt8000001E            RT_MANGLER(RTTimeNanoTSLFenceAsyncUseApicIdExt8000001E)
# define RTTimeNanoTSLFenceAsyncUseApicIdExt8000001E_EndProc    RT_MANGLER(RTTimeNanoTSLFenceAsyncUseApicIdExt8000001E_EndProc)
# define RTTimeNanoTSLFenceAsyncUseRdtscp                       RT_MANGLER(RTTimeNanoTSLFenceAsyncUseRdtscp)
# define RTTimeNanoTSLFenceAsyncUseRdtscp_EndProc               RT_MANGLER(RTTimeNanoTSLFenceAsyncUseRdtscp_EndProc)
# define RTTimeNanoTSLFenceAsyncUseRdtscpGroupChNumCl           RT_MANGLER(RTTimeNanoTSLFenceAsyncUseRdtscpGroupChNumCl)
# define RTTimeNanoTSLFenceAsyncUseRdtscpGroupChNumCl_EndProc   RT_MANGLER(RTTimeNanoTSLFenceAsyncUseRdtscpGroupChNumCl_EndProc)
# define RTTimeNanoTSLFenceAsyncUseIdtrLim                      RT_MANGLER(RTTimeNanoTSLFenceAsyncUseIdtrLim)
# define RTTimeNanoTSLFenceAsyncUseIdtrLim_EndProc              RT_MANGLER(RTTimeNanoTSLFenceAsyncUseIdtrLim_EndProc)
# define RTTimeNanoTSLFenceSyncInvarNoDelta                     RT_MANGLER(RTTimeNanoTSLFenceSyncInvarNoDelta)
# define RTTimeNanoTSLFenceSyncInvarNoDelta_EndProc             RT_MANGLER(RTTimeNanoTSLFenceSyncInvarNoDelta_EndProc)
# define RTTimeNanoTSLFenceSyncInvarWithDelta                   RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDelta)
# define RTTimeNanoTSLFenceSyncInvarWithDelta_EndProc           RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDelta_EndProc)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicId                     RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicId)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicId_EndProc             RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicId_EndProc)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt0B                RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt0B)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt0B_EndProc        RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt0B_EndProc)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt8000001E          RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt8000001E)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt8000001E_EndProc  RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseApicIdExt8000001E_EndProc)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseRdtscp          RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseRdtscp)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseRdtscp_EndProc  RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseRdtscp_EndProc)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseIdtrLim         RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseIdtrLim)
# define RTTimeNanoTSLFenceSyncInvarWithDeltaUseIdtrLim_EndProc RT_MANGLER(RTTimeNanoTSLFenceSyncInvarWithDeltaUseIdtrLim_EndProc)
# define RTTimeNanoTSWorkerName                         RT_MANGLER(RTTimeNanoTSWorkerName)
# define RTTimeNormalize                                RT_MANGLER(RTTimeNormalize)
# define RTTimeNow                                      RT_MANGLER(RTTimeNow)
# define RTTimeProgramMicroTS                           RT_MANGLER(RTTimeProgramMicroTS)
# define RTTimeProgramMilliTS                           RT_MANGLER(RTTimeProgramMilliTS)
# define RTTimeProgramNanoTS                            RT_MANGLER(RTTimeProgramNanoTS)
# define RTTimeProgramSecTS                             RT_MANGLER(RTTimeProgramSecTS)
# define RTTimeProgramStartNanoTS                       RT_MANGLER(RTTimeProgramStartNanoTS)
# define RTTimerCanDoHighResolution                     RT_MANGLER(RTTimerCanDoHighResolution)
# define RTTimerChangeInterval                          RT_MANGLER(RTTimerChangeInterval)
# define RTTimerCreate                                  RT_MANGLER(RTTimerCreate)
# define RTTimerCreateEx                                RT_MANGLER(RTTimerCreateEx)
# define RTTimerDestroy                                 RT_MANGLER(RTTimerDestroy)
# define RTTimerGetSystemGranularity                    RT_MANGLER(RTTimerGetSystemGranularity) /* r0drv */
# define RTTimerLRCreate                                RT_MANGLER(RTTimerLRCreate)
# define RTTimerLRCreateEx                              RT_MANGLER(RTTimerLRCreateEx)
# define RTTimerLRDestroy                               RT_MANGLER(RTTimerLRDestroy)
# define RTTimerLRStart                                 RT_MANGLER(RTTimerLRStart)
# define RTTimerLRStop                                  RT_MANGLER(RTTimerLRStop)
# define RTTimerLRChangeInterval                        RT_MANGLER(RTTimerLRChangeInterval)
# define RTTimerReleaseSystemGranularity                RT_MANGLER(RTTimerReleaseSystemGranularity) /* r0drv */
# define RTTimerRequestSystemGranularity                RT_MANGLER(RTTimerRequestSystemGranularity) /* r0drv */
# define RTTimerStart                                   RT_MANGLER(RTTimerStart)
# define RTTimerStop                                    RT_MANGLER(RTTimerStop)
# define RTTimeSet                                      RT_MANGLER(RTTimeSet)
# define RTTimeSpecFromString                           RT_MANGLER(RTTimeSpecFromString)
# define RTTimeSpecToString                             RT_MANGLER(RTTimeSpecToString)
# define RTTimeSystemMilliTS                            RT_MANGLER(RTTimeSystemMilliTS)
# define RTTimeSystemNanoTS                             RT_MANGLER(RTTimeSystemNanoTS)
# define RTTimeFromString                               RT_MANGLER(RTTimeFromString)
# define RTTimeFromRfc2822                              RT_MANGLER(RTTimeFromRfc2822)
# define RTTimeToString                                 RT_MANGLER(RTTimeToString)
# define RTTimeToStringEx                               RT_MANGLER(RTTimeToStringEx)
# define RTTimeToRfc2822                                RT_MANGLER(RTTimeToRfc2822)
# define RTTimeZoneGetInfoByUnixName                    RT_MANGLER(RTTimeZoneGetInfoByUnixName)
# define RTTimeZoneGetInfoByWindowsName                 RT_MANGLER(RTTimeZoneGetInfoByWindowsName)
# define RTTimeZoneGetInfoByWindowsIndex                RT_MANGLER(RTTimeZoneGetInfoByWindowsIndex)
# define RTTimeZoneGetCurrent                           RT_MANGLER(RTTimeZoneGetCurrent)
# define RTTlsAlloc                                     RT_MANGLER(RTTlsAlloc)
# define RTTlsAllocEx                                   RT_MANGLER(RTTlsAllocEx)
# define RTTlsFree                                      RT_MANGLER(RTTlsFree)
# define RTTlsGet                                       RT_MANGLER(RTTlsGet)
# define RTTlsGetEx                                     RT_MANGLER(RTTlsGetEx)
# define RTTlsSet                                       RT_MANGLER(RTTlsSet)
# define RTTpmOpen                                      RT_MANGLER(RTTpmOpen)
# define RTTpmClose                                     RT_MANGLER(RTTpmClose)
# define RTTpmGetLocalityMax                            RT_MANGLER(RTTpmGetLocalityMax)
# define RTTpmGetVersion                                RT_MANGLER(RTTpmGetVersion)
# define RTTpmReqCancel                                 RT_MANGLER(RTTpmReqCancel)
# define RTTpmReqExec                                   RT_MANGLER(RTTpmReqExec)
# define RTTraceBufAddMsg                               RT_MANGLER(RTTraceBufAddMsg)
# define RTTraceBufAddMsgEx                             RT_MANGLER(RTTraceBufAddMsgEx)
# define RTTraceBufAddMsgF                              RT_MANGLER(RTTraceBufAddMsgF)
# define RTTraceBufAddMsgV                              RT_MANGLER(RTTraceBufAddMsgV)
# define RTTraceBufAddPos                               RT_MANGLER(RTTraceBufAddPos)
# define RTTraceBufAddPosMsg                            RT_MANGLER(RTTraceBufAddPosMsg)
# define RTTraceBufAddPosMsgEx                          RT_MANGLER(RTTraceBufAddPosMsgEx)
# define RTTraceBufAddPosMsgF                           RT_MANGLER(RTTraceBufAddPosMsgF)
# define RTTraceBufAddPosMsgV                           RT_MANGLER(RTTraceBufAddPosMsgV)
# define RTTraceBufCarve                                RT_MANGLER(RTTraceBufCarve)
# define RTTraceBufCreate                               RT_MANGLER(RTTraceBufCreate)
# define RTTraceBufDisable                              RT_MANGLER(RTTraceBufDisable)
# define RTTraceBufDumpToAssert                         RT_MANGLER(RTTraceBufDumpToAssert)
# define RTTraceBufDumpToLog                            RT_MANGLER(RTTraceBufDumpToLog)
# define RTTraceBufEnable                               RT_MANGLER(RTTraceBufEnable)
# define RTTraceBufEnumEntries                          RT_MANGLER(RTTraceBufEnumEntries)
# define RTTraceBufGetEntryCount                        RT_MANGLER(RTTraceBufGetEntryCount)
# define RTTraceBufGetEntrySize                         RT_MANGLER(RTTraceBufGetEntrySize)
# define RTTraceBufRelease                              RT_MANGLER(RTTraceBufRelease)
# define RTTraceBufRetain                               RT_MANGLER(RTTraceBufRetain)
# define RTTraceGetDefaultBuf                           RT_MANGLER(RTTraceGetDefaultBuf)
# define RTTraceLogRdrCreate                            RT_MANGLER(RTTraceLogRdrCreate)
# define RTTraceLogRdrCreateFromFile                    RT_MANGLER(RTTraceLogRdrCreateFromFile)
# define RTTraceLogRdrDestroy                           RT_MANGLER(RTTraceLogRdrDestroy)
# define RTTraceLogRdrEvtFillVals                       RT_MANGLER(RTTraceLogRdrEvtFillVals)
# define RTTraceLogRdrEvtGetDesc                        RT_MANGLER(RTTraceLogRdrEvtGetDesc)
# define RTTraceLogRdrEvtGetSeqNo                       RT_MANGLER(RTTraceLogRdrEvtGetSeqNo)
# define RTTraceLogRdrEvtGetTs                          RT_MANGLER(RTTraceLogRdrEvtGetTs)
# define RTTraceLogRdrEvtIsGrouped                      RT_MANGLER(RTTraceLogRdrEvtIsGrouped)
# define RTTraceLogRdrEvtMapToStruct                    RT_MANGLER(RTTraceLogRdrEvtMapToStruct)
# define RTTraceLogRdrEvtMapFree                        RT_MANGLER(RTTraceLogRdrEvtMapFree)
# define RTTraceLogRdrEvtPoll                           RT_MANGLER(RTTraceLogRdrEvtPoll)
# define RTTraceLogRdrEvtQueryVal                       RT_MANGLER(RTTraceLogRdrEvtQueryVal)
# define RTTraceLogRdrIteratorFree                      RT_MANGLER(RTTraceLogRdrIteratorFree)
# define RTTraceLogRdrIteratorNext                      RT_MANGLER(RTTraceLogRdrIteratorNext)
# define RTTraceLogRdrIteratorQueryEvent                RT_MANGLER(RTTraceLogRdrIteratorQueryEvent)
# define RTTraceLogRdrQueryIterator                     RT_MANGLER(RTTraceLogRdrQueryIterator)
# define RTTraceLogRdrQueryLastEvt                      RT_MANGLER(RTTraceLogRdrQueryLastEvt)
# define RTTraceLogWrAddEvtDesc                         RT_MANGLER(RTTraceLogWrAddEvtDesc)
# define RTTraceLogWrCreate                             RT_MANGLER(RTTraceLogWrCreate)
# define RTTraceLogWrCreateFile                         RT_MANGLER(RTTraceLogWrCreateFile)
# define RTTraceLogWrCreateTcpClient                    RT_MANGLER(RTTraceLogWrCreateTcpClient)
# define RTTraceLogWrCreateTcpServer                    RT_MANGLER(RTTraceLogWrCreateTcpServer)
# define RTTraceLogWrDestroy                            RT_MANGLER(RTTraceLogWrDestroy)
# define RTTraceLogWrEvtAdd                             RT_MANGLER(RTTraceLogWrEvtAdd)
# define RTTraceLogWrEvtAddL                            RT_MANGLER(RTTraceLogWrEvtAddL)
# define RTTraceLogWrEvtAddLV                           RT_MANGLER(RTTraceLogWrEvtAddLV)
# define RTTraceLogWrEvtAddSg                           RT_MANGLER(RTTraceLogWrEvtAddSg)
# define RTTraceSetDefaultBuf                           RT_MANGLER(RTTraceSetDefaultBuf)
# define RTUdpCreateClientSocket                        RT_MANGLER(RTUdpCreateClientSocket)
# define RTUdpCreateServerSocket                        RT_MANGLER(RTUdpCreateServerSocket)
# define RTUdpRead                                      RT_MANGLER(RTUdpRead)
# define RTUdpServerCreate                              RT_MANGLER(RTUdpServerCreate)
# define RTUdpServerCreateEx                            RT_MANGLER(RTUdpServerCreateEx)
# define RTUdpServerDestroy                             RT_MANGLER(RTUdpServerDestroy)
# define RTUdpServerListen                              RT_MANGLER(RTUdpServerListen)
# define RTUdpServerShutdown                            RT_MANGLER(RTUdpServerShutdown)
# define RTUdpWrite                                     RT_MANGLER(RTUdpWrite)
# define RTUniFree                                      RT_MANGLER(RTUniFree)
# define RTUriCreate                                    RT_MANGLER(RTUriCreate)
# define RTUriFileCreate                                RT_MANGLER(RTUriFileCreate)
# define RTUriFileCreateEx                              RT_MANGLER(RTUriFileCreateEx)
# define RTUriFilePath                                  RT_MANGLER(RTUriFilePath)
# define RTUriFilePathEx                                RT_MANGLER(RTUriFilePathEx)
# define RTUriParse                                     RT_MANGLER(RTUriParse)
# define RTUriParsedAuthority                           RT_MANGLER(RTUriParsedAuthority)
# define RTUriParsedAuthorityHost                       RT_MANGLER(RTUriParsedAuthorityHost)
# define RTUriParsedAuthorityPassword                   RT_MANGLER(RTUriParsedAuthorityPassword)
# define RTUriParsedAuthorityPort                       RT_MANGLER(RTUriParsedAuthorityPort)
# define RTUriParsedAuthorityUsername                   RT_MANGLER(RTUriParsedAuthorityUsername)
# define RTUriParsedFragment                            RT_MANGLER(RTUriParsedFragment)
# define RTUriParsedPath                                RT_MANGLER(RTUriParsedPath)
# define RTUriParsedScheme                              RT_MANGLER(RTUriParsedScheme)
# define RTUriParsedQuery                               RT_MANGLER(RTUriParsedQuery)
# define RTUriIsSchemeMatch                             RT_MANGLER(RTUriIsSchemeMatch)
# define RTUtf16AllocTag                                RT_MANGLER(RTUtf16AllocTag)
# define RTUtf16ReallocTag                              RT_MANGLER(RTUtf16ReallocTag)
# define RTUtf16CalcLatin1Len                           RT_MANGLER(RTUtf16CalcLatin1Len)
# define RTUtf16CalcLatin1LenEx                         RT_MANGLER(RTUtf16CalcLatin1LenEx)
# define RTUtf16CalcUtf8Len                             RT_MANGLER(RTUtf16CalcUtf8Len)
# define RTUtf16CalcUtf8LenEx                           RT_MANGLER(RTUtf16CalcUtf8LenEx)
# define RTUtf16BigCalcUtf8Len                          RT_MANGLER(RTUtf16BigCalcUtf8Len)
# define RTUtf16BigCalcUtf8LenEx                        RT_MANGLER(RTUtf16BigCalcUtf8LenEx)
# define RTUtf16LittleCalcUtf8Len                       RT_MANGLER(RTUtf16LittleCalcUtf8Len)
# define RTUtf16LittleCalcUtf8LenEx                     RT_MANGLER(RTUtf16LittleCalcUtf8LenEx)
# define RTUtf16Cmp                                     RT_MANGLER(RTUtf16Cmp)
# define RTUtf16CmpAscii                                RT_MANGLER(RTUtf16CmpAscii)
# define RTUtf16CmpUtf8                                 RT_MANGLER(RTUtf16CmpUtf8)
# define RTUtf16DupExTag                                RT_MANGLER(RTUtf16DupExTag)
# define RTUtf16DupTag                                  RT_MANGLER(RTUtf16DupTag)
# define RTUtf16Free                                    RT_MANGLER(RTUtf16Free)
# define RTUtf16GetCpExInternal                         RT_MANGLER(RTUtf16GetCpExInternal)
# define RTUtf16GetCpNExInternal                        RT_MANGLER(RTUtf16GetCpNExInternal)
# define RTUtf16BigGetCpExInternal                      RT_MANGLER(RTUtf16BigGetCpExInternal)
# define RTUtf16GetCpInternal                           RT_MANGLER(RTUtf16GetCpInternal)
# define RTUtf16BigGetCpInternal                        RT_MANGLER(RTUtf16BigGetCpInternal)
# define RTUtf16NCmp                                    RT_MANGLER(RTUtf16NCmp)
# define RTUtf16NCmpAscii                               RT_MANGLER(RTUtf16NCmpAscii)
# define RTUtf16NCmpUtf8                                RT_MANGLER(RTUtf16NCmpUtf8)
# define RTUtf16ICmp                                    RT_MANGLER(RTUtf16ICmp)
# define RTUtf16BigICmp                                 RT_MANGLER(RTUtf16BigICmp)
# define RTUtf16ICmpUtf8                                RT_MANGLER(RTUtf16ICmpUtf8)
# define RTUtf16NICmp                                   RT_MANGLER(RTUtf16NICmp)
# define RTUtf16BigNICmp                                RT_MANGLER(RTUtf16BigNICmp)
# define RTUtf16FindAscii                               RT_MANGLER(RTUtf16FindAscii)
# define RTUtf16IsValidEncoding                         RT_MANGLER(RTUtf16IsValidEncoding)
# define RTUtf16Len                                     RT_MANGLER(RTUtf16Len)
# define RTUtf16LocaleICmp                              RT_MANGLER(RTUtf16LocaleICmp)
# define RTUtf16PutCpInternal                           RT_MANGLER(RTUtf16PutCpInternal)
# define RTUtf16BigPutCpInternal                        RT_MANGLER(RTUtf16BigPutCpInternal)
# define RTUtf16ToLatin1ExTag                           RT_MANGLER(RTUtf16ToLatin1ExTag)
# define RTUtf16ToLatin1Tag                             RT_MANGLER(RTUtf16ToLatin1Tag)
# define RTUtf16ToLower                                 RT_MANGLER(RTUtf16ToLower)
# define RTUtf16ToUpper                                 RT_MANGLER(RTUtf16ToUpper)
# define RTUtf16PurgeComplementSet                      RT_MANGLER(RTUtf16PurgeComplementSet)
# define RTUtf16ToUtf8ExTag                             RT_MANGLER(RTUtf16ToUtf8ExTag)
# define RTUtf16BigToUtf8ExTag                          RT_MANGLER(RTUtf16BigToUtf8ExTag)
# define RTUtf16LittleToUtf8ExTag                       RT_MANGLER(RTUtf16LittleToUtf8ExTag)
# define RTUtf16ToUtf8Tag                               RT_MANGLER(RTUtf16ToUtf8Tag)
# define RTUtf16BigToUtf8Tag                            RT_MANGLER(RTUtf16BigToUtf8Tag)
# define RTUtf16LittleToUtf8Tag                         RT_MANGLER(RTUtf16LittleToUtf8Tag)
# define RTUtf16ValidateEncoding                        RT_MANGLER(RTUtf16ValidateEncoding)
# define RTUtf16ValidateEncodingEx                      RT_MANGLER(RTUtf16ValidateEncodingEx)
# define RTUtf16Printf                                  RT_MANGLER(RTUtf16Printf)
# define RTUtf16PrintfV                                 RT_MANGLER(RTUtf16PrintfV)
# define RTUtf16PrintfEx                                RT_MANGLER(RTUtf16PrintfEx)
# define RTUtf16PrintfExV                               RT_MANGLER(RTUtf16PrintfExV)
# define RTUuidClear                                    RT_MANGLER(RTUuidClear)
# define RTUuidCompare                                  RT_MANGLER(RTUuidCompare)
# define RTUuidCompare2Strs                             RT_MANGLER(RTUuidCompare2Strs)
# define RTUuidCompareStr                               RT_MANGLER(RTUuidCompareStr)
# define RTUuidCreate                                   RT_MANGLER(RTUuidCreate)
# define RTUuidFromStr                                  RT_MANGLER(RTUuidFromStr)
# define RTUuidFromUtf16                                RT_MANGLER(RTUuidFromUtf16)
# define RTUuidIsNull                                   RT_MANGLER(RTUuidIsNull)
# define RTUuidToStr                                    RT_MANGLER(RTUuidToStr)
# define RTUuidToUtf16                                  RT_MANGLER(RTUuidToUtf16)
# define RTVfsChainElementDeregisterProvider            RT_MANGLER(RTVfsChainElementDeregisterProvider)
# define RTVfsChainElementRegisterProvider              RT_MANGLER(RTVfsChainElementRegisterProvider)
# define RTVfsChainIsSpec                               RT_MANGLER(RTVfsChainIsSpec)
# define RTVfsChainMsgError                             RT_MANGLER(RTVfsChainMsgError)
# define RTVfsChainMsgErrorExitFailure                  RT_MANGLER(RTVfsChainMsgErrorExitFailure)
# define RTVfsChainOpenObj                              RT_MANGLER(RTVfsChainOpenObj)
# define RTVfsChainOpenDir                              RT_MANGLER(RTVfsChainOpenDir)
# define RTVfsChainOpenParentDir                        RT_MANGLER(RTVfsChainOpenParentDir)
# define RTVfsChainOpenFile                             RT_MANGLER(RTVfsChainOpenFile)
# define RTVfsChainOpenIoStream                         RT_MANGLER(RTVfsChainOpenIoStream)
# define RTVfsChainQueryFinalPath                       RT_MANGLER(RTVfsChainQueryFinalPath)
# define RTVfsChainQueryInfo                            RT_MANGLER(RTVfsChainQueryInfo)
# define RTVfsChainSpecCheckAndSetup                    RT_MANGLER(RTVfsChainSpecCheckAndSetup)
# define RTVfsChainSpecFree                             RT_MANGLER(RTVfsChainSpecFree)
# define RTVfsChainSpecParse                            RT_MANGLER(RTVfsChainSpecParse)
# define RTVfsChainSplitOffFinalPath                    RT_MANGLER(RTVfsChainSplitOffFinalPath)
# define RTVfsChainValidateOpenFileOrIoStream           RT_MANGLER(RTVfsChainValidateOpenFileOrIoStream)
# define RTVfsDirRelease                                RT_MANGLER(RTVfsDirRelease)
# define RTVfsDirRetain                                 RT_MANGLER(RTVfsDirRetain)
# define RTVfsDirRetainDebug                            RT_MANGLER(RTVfsDirRetainDebug)
# define RTVfsDirOpen                                   RT_MANGLER(RTVfsDirOpen)
# define RTVfsDirOpenDir                                RT_MANGLER(RTVfsDirOpenDir)
# define RTVfsDirCreateDir                              RT_MANGLER(RTVfsDirCreateDir)
# define RTVfsDirOpenFile                               RT_MANGLER(RTVfsDirOpenFile)
# define RTVfsDirOpenFileAsIoStream                     RT_MANGLER(RTVfsDirOpenFileAsIoStream)
# define RTVfsDirOpenObj                                RT_MANGLER(RTVfsDirOpenObj)
# define RTVfsDirQueryPathInfo                          RT_MANGLER(RTVfsDirQueryPathInfo)
# define RTVfsDirReadEx                                 RT_MANGLER(RTVfsDirReadEx)
# define RTVfsDirRemoveDir                              RT_MANGLER(RTVfsDirRemoveDir)
# define RTVfsDirRewind                                 RT_MANGLER(RTVfsDirRewind)
# define RTVfsDirSetPathMode                            RT_MANGLER(RTVfsDirSetPathMode)
# define RTVfsDirToPrivate                              RT_MANGLER(RTVfsDirToPrivate)
# define RTVfsFileFlush                                 RT_MANGLER(RTVfsFileFlush)
# define RTVfsFileFromBuffer                            RT_MANGLER(RTVfsFileFromBuffer)
# define RTVfsFileFromRTFile                            RT_MANGLER(RTVfsFileFromRTFile)
# define RTVfsFileGetOpenFlags                          RT_MANGLER(RTVfsFileGetOpenFlags)
# define RTVfsFileQuerySize                             RT_MANGLER(RTVfsFileQuerySize)
# define RTVfsFileGetMaxSize                            RT_MANGLER(RTVfsFileGetMaxSize)
# define RTVfsFileOpen                                  RT_MANGLER(RTVfsFileOpen)
# define RTVfsFileOpenNormal                            RT_MANGLER(RTVfsFileOpenNormal)
# define RTVfsFilePoll                                  RT_MANGLER(RTVfsFilePoll)
# define RTVfsFilePrintf                                RT_MANGLER(RTVfsFilePrintf)
# define RTVfsFilePrintfV                               RT_MANGLER(RTVfsFilePrintfV)
# define RTVfsFileQueryInfo                             RT_MANGLER(RTVfsFileQueryInfo)
# define RTVfsFileQueryMaxSize                          RT_MANGLER(RTVfsFileQueryMaxSize)
# define RTVfsFileRead                                  RT_MANGLER(RTVfsFileRead)
# define RTVfsFileReadAll                               RT_MANGLER(RTVfsFileReadAll)
# define RTVfsFileReadAllFree                           RT_MANGLER(RTVfsFileReadAllFree)
# define RTVfsFileReadAt                                RT_MANGLER(RTVfsFileReadAt)
# define RTVfsFileRelease                               RT_MANGLER(RTVfsFileRelease)
# define RTVfsFileRetain                                RT_MANGLER(RTVfsFileRetain)
# define RTVfsFileRetainDebug                           RT_MANGLER(RTVfsFileRetainDebug)
# define RTVfsFileSeek                                  RT_MANGLER(RTVfsFileSeek)
# define RTVfsFileSetSize                               RT_MANGLER(RTVfsFileSetSize)
# define RTVfsFileSgRead                                RT_MANGLER(RTVfsFileSgRead)
# define RTVfsFileSgWrite                               RT_MANGLER(RTVfsFileSgWrite)
# define RTVfsFileTell                                  RT_MANGLER(RTVfsFileTell)
# define RTVfsFileToIoStream                            RT_MANGLER(RTVfsFileToIoStream)
# define RTVfsFileWrite                                 RT_MANGLER(RTVfsFileWrite)
# define RTVfsFileWriteAt                               RT_MANGLER(RTVfsFileWriteAt)
# define RTVfsFsStreamToPrivate                         RT_MANGLER(RTVfsFsStreamToPrivate)
# define RTVfsFsStrmAdd                                 RT_MANGLER(RTVfsFsStrmAdd)
# define RTVfsFsStrmEnd                                 RT_MANGLER(RTVfsFsStrmEnd)
# define RTVfsFsStrmNext                                RT_MANGLER(RTVfsFsStrmNext)
# define RTVfsFsStrmPushFile                            RT_MANGLER(RTVfsFsStrmPushFile)
# define RTVfsFsStrmQueryInfo                           RT_MANGLER(RTVfsFsStrmQueryInfo)
# define RTVfsFsStrmRelease                             RT_MANGLER(RTVfsFsStrmRelease)
# define RTVfsFsStrmRetain                              RT_MANGLER(RTVfsFsStrmRetain)
# define RTVfsFsStrmRetainDebug                         RT_MANGLER(RTVfsFsStrmRetainDebug)
# define RTVfsFsStrmToDir                               RT_MANGLER(RTVfsFsStrmToDir)
# define RTVfsFsStrmToNormalDir                         RT_MANGLER(RTVfsFsStrmToNormalDir)
# define RTVfsFsStrmToDirUndo                           RT_MANGLER(RTVfsFsStrmToDirUndo)
# define RTVfsIoStreamToPrivate                         RT_MANGLER(RTVfsIoStreamToPrivate)
# define RTVfsIoStrmFlush                               RT_MANGLER(RTVfsIoStrmFlush)
# define RTVfsIoStrmFromBuffer                          RT_MANGLER(RTVfsIoStrmFromBuffer)
# define RTVfsIoStrmFromRTFile                          RT_MANGLER(RTVfsIoStrmFromRTFile)
# define RTVfsIoStrmFromRTPipe                          RT_MANGLER(RTVfsIoStrmFromRTPipe)
# define RTVfsIoStrmFromStdHandle                       RT_MANGLER(RTVfsIoStrmFromStdHandle)
# define RTVfsIoStrmGetOpenFlags                        RT_MANGLER(RTVfsIoStrmGetOpenFlags)
# define RTVfsIoStrmIsAtEnd                             RT_MANGLER(RTVfsIoStrmIsAtEnd)
# define RTVfsIoStrmOpenNormal                          RT_MANGLER(RTVfsIoStrmOpenNormal)
# define RTVfsIoStrmPoll                                RT_MANGLER(RTVfsIoStrmPoll)
# define RTVfsIoStrmPrintf                              RT_MANGLER(RTVfsIoStrmPrintf)
# define RTVfsIoStrmPrintfV                             RT_MANGLER(RTVfsIoStrmPrintfV)
# define RTVfsIoStrmQueryInfo                           RT_MANGLER(RTVfsIoStrmQueryInfo)
# define RTVfsIoStrmRead                                RT_MANGLER(RTVfsIoStrmRead)
# define RTVfsIoStrmReadAt                              RT_MANGLER(RTVfsIoStrmReadAt)
# define RTVfsIoStrmReadAll                             RT_MANGLER(RTVfsIoStrmReadAll)
# define RTVfsIoStrmReadAllFree                         RT_MANGLER(RTVfsIoStrmReadAllFree)
# define RTVfsIoStrmRelease                             RT_MANGLER(RTVfsIoStrmRelease)
# define RTVfsIoStrmRetain                              RT_MANGLER(RTVfsIoStrmRetain)
# define RTVfsIoStrmRetainDebug                         RT_MANGLER(RTVfsIoStrmRetainDebug)
# define RTVfsIoStrmSgRead                              RT_MANGLER(RTVfsIoStrmSgRead)
# define RTVfsIoStrmSgWrite                             RT_MANGLER(RTVfsIoStrmSgWrite)
# define RTVfsIoStrmSkip                                RT_MANGLER(RTVfsIoStrmSkip)
# define RTVfsIoStrmStrOutputCallback                   RT_MANGLER(RTVfsIoStrmStrOutputCallback)
# define RTVfsIoStrmTell                                RT_MANGLER(RTVfsIoStrmTell)
# define RTVfsIoStrmToFile                              RT_MANGLER(RTVfsIoStrmToFile)
# define RTVfsIoStrmValidateUtf8Encoding                RT_MANGLER(RTVfsIoStrmValidateUtf8Encoding)
# define RTVfsIoStrmWrite                               RT_MANGLER(RTVfsIoStrmWrite)
# define RTVfsIoStrmWriteAt                             RT_MANGLER(RTVfsIoStrmWriteAt)
# define RTVfsIoStrmZeroFill                            RT_MANGLER(RTVfsIoStrmZeroFill)
# define RTVfsQueryLabel                                RT_MANGLER(RTVfsQueryLabel)
# define RTVfsQueryRangeState                           RT_MANGLER(RTVfsQueryRangeState)
# define RTVfsLockAcquireReadSlow                       RT_MANGLER(RTVfsLockAcquireReadSlow)
# define RTVfsLockAcquireWriteSlow                      RT_MANGLER(RTVfsLockAcquireWriteSlow)
# define RTVfsLockRelease                               RT_MANGLER(RTVfsLockRelease)
# define RTVfsLockReleaseReadSlow                       RT_MANGLER(RTVfsLockReleaseReadSlow)
# define RTVfsLockReleaseWriteSlow                      RT_MANGLER(RTVfsLockReleaseWriteSlow)
# define RTVfsLockRetain                                RT_MANGLER(RTVfsLockRetain)
# define RTVfsLockRetainDebug                           RT_MANGLER(RTVfsLockRetainDebug)
# define RTVfsMemFileCreate                             RT_MANGLER(RTVfsMemFileCreate)
# define RTVfsMemIoStrmCreate                           RT_MANGLER(RTVfsMemIoStrmCreate)
# define RTVfsMemorizeIoStreamAsFile                    RT_MANGLER(RTVfsMemorizeIoStreamAsFile)
# define RTVfsNew                                       RT_MANGLER(RTVfsNew)
# define RTVfsNewBaseObj                                RT_MANGLER(RTVfsNewBaseObj)
# define RTVfsNewDir                                    RT_MANGLER(RTVfsNewDir)
# define RTVfsNewFile                                   RT_MANGLER(RTVfsNewFile)
# define RTVfsNewFsStream                               RT_MANGLER(RTVfsNewFsStream)
# define RTVfsNewIoStream                               RT_MANGLER(RTVfsNewIoStream)
# define RTVfsNewSymlink                                RT_MANGLER(RTVfsNewSymlink)
# define RTVfsObjFromDir                                RT_MANGLER(RTVfsObjFromDir)
# define RTVfsObjFromFile                               RT_MANGLER(RTVfsObjFromFile)
# define RTVfsObjFromFsStream                           RT_MANGLER(RTVfsObjFromFsStream)
# define RTVfsObjFromIoStream                           RT_MANGLER(RTVfsObjFromIoStream)
# define RTVfsObjFromSymlink                            RT_MANGLER(RTVfsObjFromSymlink)
# define RTVfsObjFromVfs                                RT_MANGLER(RTVfsObjFromVfs)
# define RTVfsObjGetType                                RT_MANGLER(RTVfsObjGetType)
# define RTVfsObjOpen                                   RT_MANGLER(RTVfsObjOpen)
# define RTVfsObjQueryInfo                              RT_MANGLER(RTVfsObjQueryInfo)
# define RTVfsObjRelease                                RT_MANGLER(RTVfsObjRelease)
# define RTVfsObjRetain                                 RT_MANGLER(RTVfsObjRetain)
# define RTVfsObjRetainDebug                            RT_MANGLER(RTVfsObjRetainDebug)
# define RTVfsObjSetMode                                RT_MANGLER(RTVfsObjSetMode)
# define RTVfsObjSetOwner                               RT_MANGLER(RTVfsObjSetOwner)
# define RTVfsObjSetTimes                               RT_MANGLER(RTVfsObjSetTimes)
# define RTVfsObjToDir                                  RT_MANGLER(RTVfsObjToDir)
# define RTVfsObjToFile                                 RT_MANGLER(RTVfsObjToFile)
# define RTVfsObjToFsStream                             RT_MANGLER(RTVfsObjToFsStream)
# define RTVfsObjToIoStream                             RT_MANGLER(RTVfsObjToIoStream)
# define RTVfsObjToPrivate                              RT_MANGLER(RTVfsObjToPrivate)
# define RTVfsObjToSymlink                              RT_MANGLER(RTVfsObjToSymlink)
# define RTVfsObjToVfs                                  RT_MANGLER(RTVfsObjToVfs)
# define RTVfsParsePath                                 RT_MANGLER(RTVfsParsePath)
# define RTVfsParsePathA                                RT_MANGLER(RTVfsParsePathA)
# define RTVfsParsePathAppend                           RT_MANGLER(RTVfsParsePathAppend)
# define RTVfsParsePathFree                             RT_MANGLER(RTVfsParsePathFree)
# define RTVfsRelease                                   RT_MANGLER(RTVfsRelease)
# define RTVfsOpenRoot                                  RT_MANGLER(RTVfsOpenRoot)
# define RTVfsQuerPathInfo                              RT_MANGLER(RTVfsQueryPathInfo)
# define RTVfsMountVol                                  RT_MANGLER(RTVfsMountVol)
# define RTVfsRetain                                    RT_MANGLER(RTVfsRetain)
# define RTVfsRetainDebug                               RT_MANGLER(RTVfsRetainDebug)
# define RTVfsSymlinkQueryInfo                          RT_MANGLER(RTVfsSymlinkQueryInfo)
# define RTVfsSymlinkRead                               RT_MANGLER(RTVfsSymlinkRead)
# define RTVfsSymlinkRelease                            RT_MANGLER(RTVfsSymlinkRelease)
# define RTVfsSymlinkRetain                             RT_MANGLER(RTVfsSymlinkRetain)
# define RTVfsSymlinkRetainDebug                        RT_MANGLER(RTVfsSymlinkRetainDebug)
# define RTVfsSymlinkSetMode                            RT_MANGLER(RTVfsSymlinkSetMode)
# define RTVfsSymlinkSetOwner                           RT_MANGLER(RTVfsSymlinkSetOwner)
# define RTVfsSymlinkSetTimes                           RT_MANGLER(RTVfsSymlinkSetTimes)
# define RTVfsSymlinkToPrivate                          RT_MANGLER(RTVfsSymlinkToPrivate)
# define RTVfsTypeName                                  RT_MANGLER(RTVfsTypeName)
# define RTVfsUtilDummyPollOne                          RT_MANGLER(RTVfsUtilDummyPollOne)
# define RTVfsUtilPumpIoStreams                         RT_MANGLER(RTVfsUtilPumpIoStreams)
# define RTVfsCreateProgressForFile                     RT_MANGLER(RTVfsCreateProgressForFile)
# define RTVfsCreateProgressForIoStream                 RT_MANGLER(RTVfsCreateProgressForIoStream)
# define RTVfsCreateReadAheadForFile                    RT_MANGLER(RTVfsCreateReadAheadForFile)
# define RTVfsCreateReadAheadForIoStream                RT_MANGLER(RTVfsCreateReadAheadForIoStream)
# define RTZipBlockCompress                             RT_MANGLER(RTZipBlockCompress)
# define RTZipBlockDecompress                           RT_MANGLER(RTZipBlockDecompress)
# define RTZipCompCreate                                RT_MANGLER(RTZipCompCreate)
# define RTZipCompDestroy                               RT_MANGLER(RTZipCompDestroy)
# define RTZipCompFinish                                RT_MANGLER(RTZipCompFinish)
# define RTZipCompress                                  RT_MANGLER(RTZipCompress)
# define RTZipDecompCreate                              RT_MANGLER(RTZipDecompCreate)
# define RTZipDecompDestroy                             RT_MANGLER(RTZipDecompDestroy)
# define RTZipDecompress                                RT_MANGLER(RTZipDecompress)
# define RTZipGzipCompressIoStream                      RT_MANGLER(RTZipGzipCompressIoStream)
# define RTZipGzipDecompressIoStream                    RT_MANGLER(RTZipGzipDecompressIoStream)
# define RTZipGzipCmd                                   RT_MANGLER(RTZipGzipCmd)
# define RTZipPkzipFsStreamFromIoStream                 RT_MANGLER(RTZipPkzipFsStreamFromIoStream)
# define RTZipPkzipMemDecompress                        RT_MANGLER(RTZipPkzipMemDecompress)
# define RTZipTarCmd                                    RT_MANGLER(RTZipTarCmd)
# define RTZipUnzipCmd                                  RT_MANGLER(RTZipUnzipCmd)
# define RTZipTarFsStreamFromIoStream                   RT_MANGLER(RTZipTarFsStreamFromIoStream)
# define RTZipTarFsStreamToIoStream                     RT_MANGLER(RTZipTarFsStreamToIoStream)
# define RTZipTarFsStreamSetOwner                       RT_MANGLER(RTZipTarFsStreamSetOwner)
# define RTZipTarFsStreamSetGroup                       RT_MANGLER(RTZipTarFsStreamSetGroup)
# define RTZipTarFsStreamSetPrefix                      RT_MANGLER(RTZipTarFsStreamSetPrefix)
# define RTZipTarFsStreamSetFileMode                    RT_MANGLER(RTZipTarFsStreamSetFileMode)
# define RTZipTarFsStreamSetDirMode                     RT_MANGLER(RTZipTarFsStreamSetDirMode)
# define RTZipTarFsStreamSetModTime                     RT_MANGLER(RTZipTarFsStreamSetModTime)
# define RTZipTarFsStreamTruncate                       RT_MANGLER(RTZipTarFsStreamTruncate)
# define RTZipXarFsStreamFromIoStream                   RT_MANGLER(RTZipXarFsStreamFromIoStream)
# define RTZipTarFsStreamForFile                        RT_MANGLER(RTZipTarFsStreamForFile)
# define RTZipCpioFsStreamFromIoStream                  RT_MANGLER(RTZipCpioFsStreamFromIoStream)
# define RTZipXzCompressIoStream                        RT_MANGLER(RTZipXzCompressIoStream)
# define RTZipXzDecompressIoStream                      RT_MANGLER(RTZipXzDecompressIoStream)

/* sort/merge into the above later: */
# define RTAsn1ContentAllocZ                            RT_MANGLER(RTAsn1ContentAllocZ)
# define RTAsn1ContentDup                               RT_MANGLER(RTAsn1ContentDup)
# define RTAsn1ContentFree                              RT_MANGLER(RTAsn1ContentFree)
# define RTAsn1ContentReallocZ                          RT_MANGLER(RTAsn1ContentReallocZ)
# define RTAsn1ContextTagN_Clone                        RT_MANGLER(RTAsn1ContextTagN_Clone)
# define RTAsn1ContextTagN_Init                         RT_MANGLER(RTAsn1ContextTagN_Init)
# define RTAsn1Dummy_InitEx                             RT_MANGLER(RTAsn1Dummy_InitEx)
# define RTAsn1MemAllocZ                                RT_MANGLER(RTAsn1MemAllocZ)
# define RTAsn1MemDup                                   RT_MANGLER(RTAsn1MemDup)
# define RTAsn1MemFree                                  RT_MANGLER(RTAsn1MemFree)
# define RTAsn1MemFreeArray                             RT_MANGLER(RTAsn1MemFreeArray)
# define RTAsn1MemResizeArray                           RT_MANGLER(RTAsn1MemResizeArray)
# define RTAsn1MemInitAllocation                        RT_MANGLER(RTAsn1MemInitAllocation)
# define RTAsn1MemInitArrayAllocation                   RT_MANGLER(RTAsn1MemInitArrayAllocation)
# define RTAsn1SeqOfCore_Clone                          RT_MANGLER(RTAsn1SeqOfCore_Clone)
# define RTAsn1SeqOfCore_Init                           RT_MANGLER(RTAsn1SeqOfCore_Init)
# define RTAsn1SequenceCore_Clone                       RT_MANGLER(RTAsn1SequenceCore_Clone)
# define RTAsn1SequenceCore_Init                        RT_MANGLER(RTAsn1SequenceCore_Init)
# define RTAsn1SetCore_Clone                            RT_MANGLER(RTAsn1SetCore_Clone)
# define RTAsn1SetCore_Init                             RT_MANGLER(RTAsn1SetCore_Init)
# define RTAsn1SetOfCore_Clone                          RT_MANGLER(RTAsn1SetOfCore_Clone)
# define RTAsn1SetOfCore_Init                           RT_MANGLER(RTAsn1SetOfCore_Init)
# define RTAsn1VtCheckSanity                            RT_MANGLER(RTAsn1VtCheckSanity)
# define RTAsn1VtClone                                  RT_MANGLER(RTAsn1VtClone)
# define RTAsn1VtCompare                                RT_MANGLER(RTAsn1VtCompare)
# define RTAsn1VtDeepEnum                               RT_MANGLER(RTAsn1VtDeepEnum)
# define RTAsn1VtDelete                                 RT_MANGLER(RTAsn1VtDelete)
# define RTAsn1CursorCheckEnd                           RT_MANGLER(RTAsn1CursorCheckEnd)
# define RTAsn1CursorCheckOctStrEnd                     RT_MANGLER(RTAsn1CursorCheckOctStrEnd)
# define RTAsn1CursorCheckSeqEnd                        RT_MANGLER(RTAsn1CursorCheckSeqEnd)
# define RTAsn1CursorCheckSetEnd                        RT_MANGLER(RTAsn1CursorCheckSetEnd)
# define RTAsn1CursorGetBitString                       RT_MANGLER(RTAsn1CursorGetBitString)
# define RTAsn1CursorGetBitStringEx                     RT_MANGLER(RTAsn1CursorGetBitStringEx)
# define RTAsn1CursorGetBmpString                       RT_MANGLER(RTAsn1CursorGetBmpString)
# define RTAsn1CursorGetBoolean                         RT_MANGLER(RTAsn1CursorGetBoolean)
# define RTAsn1CursorGetContextTagNCursor               RT_MANGLER(RTAsn1CursorGetContextTagNCursor)
# define RTAsn1CursorGetCore                            RT_MANGLER(RTAsn1CursorGetCore)
# define RTAsn1CursorGetDynType                         RT_MANGLER(RTAsn1CursorGetDynType)
# define RTAsn1CursorGetIa5String                       RT_MANGLER(RTAsn1CursorGetIa5String)
# define RTAsn1CursorGetInteger                         RT_MANGLER(RTAsn1CursorGetInteger)
# define RTAsn1CursorGetNull                            RT_MANGLER(RTAsn1CursorGetNull)
# define RTAsn1CursorGetObjId                           RT_MANGLER(RTAsn1CursorGetObjId)
# define RTAsn1CursorGetOctetString                     RT_MANGLER(RTAsn1CursorGetOctetString)
# define RTAsn1CursorGetSequenceCursor                  RT_MANGLER(RTAsn1CursorGetSequenceCursor)
# define RTAsn1CursorGetSetCursor                       RT_MANGLER(RTAsn1CursorGetSetCursor)
# define RTAsn1CursorGetString                          RT_MANGLER(RTAsn1CursorGetString)
# define RTAsn1CursorGetTime                            RT_MANGLER(RTAsn1CursorGetTime)
# define RTAsn1CursorGetUtf8String                      RT_MANGLER(RTAsn1CursorGetUtf8String)
# define RTAsn1CursorInitAllocation                     RT_MANGLER(RTAsn1CursorInitAllocation)
# define RTAsn1CursorInitArrayAllocation                RT_MANGLER(RTAsn1CursorInitArrayAllocation)
# define RTAsn1CursorInitPrimary                        RT_MANGLER(RTAsn1CursorInitPrimary)
# define RTAsn1CursorInitSub                            RT_MANGLER(RTAsn1CursorInitSub)
# define RTAsn1CursorInitSubFromCore                    RT_MANGLER(RTAsn1CursorInitSubFromCore)
# define RTAsn1CursorIsNextEx                           RT_MANGLER(RTAsn1CursorIsNextEx)
# define RTAsn1CursorIsEnd                              RT_MANGLER(RTAsn1CursorIsEnd)
# define RTAsn1CursorMatchTagClassFlagsEx               RT_MANGLER(RTAsn1CursorMatchTagClassFlagsEx)
# define RTAsn1CursorPeek                               RT_MANGLER(RTAsn1CursorPeek)
# define RTAsn1CursorReadHdr                            RT_MANGLER(RTAsn1CursorReadHdr)
# define RTAsn1CursorSetInfo                            RT_MANGLER(RTAsn1CursorSetInfo)
# define RTAsn1CursorSetInfoV                           RT_MANGLER(RTAsn1CursorSetInfoV)
# define RTAsn1Dump                                     RT_MANGLER(RTAsn1Dump)
# define RTAsn1QueryObjIdName                           RT_MANGLER(RTAsn1QueryObjIdName)
# define RTAsn1EncodePrepare                            RT_MANGLER(RTAsn1EncodePrepare)
# define RTAsn1EncodeRecalcHdrSize                      RT_MANGLER(RTAsn1EncodeRecalcHdrSize)
# define RTAsn1EncodeToBuffer                           RT_MANGLER(RTAsn1EncodeToBuffer)
# define RTAsn1EncodeQueryRawBits                       RT_MANGLER(RTAsn1EncodeQueryRawBits)
# define RTAsn1EncodeWrite                              RT_MANGLER(RTAsn1EncodeWrite)
# define RTAsn1EncodeWriteHeader                        RT_MANGLER(RTAsn1EncodeWriteHeader)
# define RTAsn1BitString_CheckSanity                    RT_MANGLER(RTAsn1BitString_CheckSanity)
# define RTAsn1BitString_Clone                          RT_MANGLER(RTAsn1BitString_Clone)
# define RTAsn1BitString_Compare                        RT_MANGLER(RTAsn1BitString_Compare)
# define RTAsn1BitString_Delete                         RT_MANGLER(RTAsn1BitString_Delete)
# define RTAsn1BitString_Enum                           RT_MANGLER(RTAsn1BitString_Enum)
# define RTAsn1BitString_GetAsUInt64                    RT_MANGLER(RTAsn1BitString_GetAsUInt64)
# define RTAsn1BitString_Init                           RT_MANGLER(RTAsn1BitString_Init)
# define RTAsn1BitString_InitWithData                   RT_MANGLER(RTAsn1BitString_InitWithData)
# define RTAsn1BitString_AreContentBitsValid            RT_MANGLER(RTAsn1BitString_AreContentBitsValid)
# define RTAsn1BitString_RefreshContent                 RT_MANGLER(RTAsn1BitString_RefreshContent)
# define RTAsn1SeqOfBitStrings_CheckSanity              RT_MANGLER(RTAsn1SeqOfBitStrings_CheckSanity)
# define RTAsn1SeqOfBitStrings_Clone                    RT_MANGLER(RTAsn1SeqOfBitStrings_Clone)
# define RTAsn1SeqOfBitStrings_Compare                  RT_MANGLER(RTAsn1SeqOfBitStrings_Compare)
# define RTAsn1SeqOfBitStrings_Delete                   RT_MANGLER(RTAsn1SeqOfBitStrings_Delete)
# define RTAsn1SeqOfBitStrings_Enum                     RT_MANGLER(RTAsn1SeqOfBitStrings_Enum)
# define RTAsn1SeqOfBitStrings_Init                     RT_MANGLER(RTAsn1SeqOfBitStrings_Init)
# define RTAsn1SetOfBitStrings_CheckSanity              RT_MANGLER(RTAsn1SetOfBitStrings_CheckSanity)
# define RTAsn1SetOfBitStrings_Clone                    RT_MANGLER(RTAsn1SetOfBitStrings_Clone)
# define RTAsn1SetOfBitStrings_Compare                  RT_MANGLER(RTAsn1SetOfBitStrings_Compare)
# define RTAsn1SetOfBitStrings_Delete                   RT_MANGLER(RTAsn1SetOfBitStrings_Delete)
# define RTAsn1SetOfBitStrings_Enum                     RT_MANGLER(RTAsn1SetOfBitStrings_Enum)
# define RTAsn1SetOfBitStrings_Init                     RT_MANGLER(RTAsn1SetOfBitStrings_Init)
# define RTAsn1BitString_DecodeAsn1                     RT_MANGLER(RTAsn1BitString_DecodeAsn1)
# define RTAsn1BitString_DecodeAsn1Ex                   RT_MANGLER(RTAsn1BitString_DecodeAsn1Ex)
# define RTAsn1SeqOfBitStrings_DecodeAsn1               RT_MANGLER(RTAsn1SeqOfBitStrings_DecodeAsn1)
# define RTAsn1SetOfBitStrings_DecodeAsn1               RT_MANGLER(RTAsn1SetOfBitStrings_DecodeAsn1)
# define RTAsn1Boolean_CheckSanity                      RT_MANGLER(RTAsn1Boolean_CheckSanity)
# define RTAsn1Boolean_Clone                            RT_MANGLER(RTAsn1Boolean_Clone)
# define RTAsn1Boolean_Compare                          RT_MANGLER(RTAsn1Boolean_Compare)
# define RTAsn1Boolean_Delete                           RT_MANGLER(RTAsn1Boolean_Delete)
# define RTAsn1Boolean_Enum                             RT_MANGLER(RTAsn1Boolean_Enum)
# define RTAsn1Boolean_Init                             RT_MANGLER(RTAsn1Boolean_Init)
# define RTAsn1Boolean_InitDefault                      RT_MANGLER(RTAsn1Boolean_InitDefault)
# define RTAsn1Boolean_Set                              RT_MANGLER(RTAsn1Boolean_Set)
# define RTAsn1SeqOfBooleans_CheckSanity                RT_MANGLER(RTAsn1SeqOfBooleans_CheckSanity)
# define RTAsn1SeqOfBooleans_Clone                      RT_MANGLER(RTAsn1SeqOfBooleans_Clone)
# define RTAsn1SeqOfBooleans_Compare                    RT_MANGLER(RTAsn1SeqOfBooleans_Compare)
# define RTAsn1SeqOfBooleans_Delete                     RT_MANGLER(RTAsn1SeqOfBooleans_Delete)
# define RTAsn1SeqOfBooleans_Enum                       RT_MANGLER(RTAsn1SeqOfBooleans_Enum)
# define RTAsn1SeqOfBooleans_Init                       RT_MANGLER(RTAsn1SeqOfBooleans_Init)
# define RTAsn1SetOfBooleans_CheckSanity                RT_MANGLER(RTAsn1SetOfBooleans_CheckSanity)
# define RTAsn1SetOfBooleans_Clone                      RT_MANGLER(RTAsn1SetOfBooleans_Clone)
# define RTAsn1SetOfBooleans_Compare                    RT_MANGLER(RTAsn1SetOfBooleans_Compare)
# define RTAsn1SetOfBooleans_Delete                     RT_MANGLER(RTAsn1SetOfBooleans_Delete)
# define RTAsn1SetOfBooleans_Enum                       RT_MANGLER(RTAsn1SetOfBooleans_Enum)
# define RTAsn1SetOfBooleans_Init                       RT_MANGLER(RTAsn1SetOfBooleans_Init)
# define RTAsn1Boolean_DecodeAsn1                       RT_MANGLER(RTAsn1Boolean_DecodeAsn1)
# define RTAsn1SeqOfBooleans_DecodeAsn1                 RT_MANGLER(RTAsn1SeqOfBooleans_DecodeAsn1)
# define RTAsn1SetOfBooleans_DecodeAsn1                 RT_MANGLER(RTAsn1SetOfBooleans_DecodeAsn1)
# define RTAsn1Core_ChangeTag                           RT_MANGLER(RTAsn1Core_ChangeTag)
# define RTAsn1Core_CheckSanity                         RT_MANGLER(RTAsn1Core_CheckSanity)
# define RTAsn1Core_Clone                               RT_MANGLER(RTAsn1Core_Clone)
# define RTAsn1Core_CloneContent                        RT_MANGLER(RTAsn1Core_CloneContent)
# define RTAsn1Core_CloneNoContent                      RT_MANGLER(RTAsn1Core_CloneNoContent)
# define RTAsn1Core_Compare                             RT_MANGLER(RTAsn1Core_Compare)
# define RTAsn1Core_CompareEx                           RT_MANGLER(RTAsn1Core_CompareEx)
# define RTAsn1Core_Delete                              RT_MANGLER(RTAsn1Core_Delete)
# define RTAsn1Core_Enum                                RT_MANGLER(RTAsn1Core_Enum)
# define RTAsn1Core_Init                                RT_MANGLER(RTAsn1Core_Init)
# define RTAsn1Core_InitDefault                         RT_MANGLER(RTAsn1Core_InitDefault)
# define RTAsn1Core_InitEx                              RT_MANGLER(RTAsn1Core_InitEx)
# define RTAsn1Core_ResetImplict                        RT_MANGLER(RTAsn1Core_ResetImplict)
# define RTAsn1Core_SetTagAndFlags                      RT_MANGLER(RTAsn1Core_SetTagAndFlags)
# define RTAsn1SeqOfCores_CheckSanity                   RT_MANGLER(RTAsn1SeqOfCores_CheckSanity)
# define RTAsn1SeqOfCores_Clone                         RT_MANGLER(RTAsn1SeqOfCores_Clone)
# define RTAsn1SeqOfCores_Compare                       RT_MANGLER(RTAsn1SeqOfCores_Compare)
# define RTAsn1SeqOfCores_Delete                        RT_MANGLER(RTAsn1SeqOfCores_Delete)
# define RTAsn1SeqOfCores_Enum                          RT_MANGLER(RTAsn1SeqOfCores_Enum)
# define RTAsn1SeqOfCores_Init                          RT_MANGLER(RTAsn1SeqOfCores_Init)
# define RTAsn1SetOfCores_CheckSanity                   RT_MANGLER(RTAsn1SetOfCores_CheckSanity)
# define RTAsn1SetOfCores_Clone                         RT_MANGLER(RTAsn1SetOfCores_Clone)
# define RTAsn1SetOfCores_Compare                       RT_MANGLER(RTAsn1SetOfCores_Compare)
# define RTAsn1SetOfCores_Delete                        RT_MANGLER(RTAsn1SetOfCores_Delete)
# define RTAsn1SetOfCores_Enum                          RT_MANGLER(RTAsn1SetOfCores_Enum)
# define RTAsn1SetOfCores_Init                          RT_MANGLER(RTAsn1SetOfCores_Init)
# define RTAsn1Core_DecodeAsn1                          RT_MANGLER(RTAsn1Core_DecodeAsn1)
# define RTAsn1SeqOfCores_DecodeAsn1                    RT_MANGLER(RTAsn1SeqOfCores_DecodeAsn1)
# define RTAsn1SetOfCores_DecodeAsn1                    RT_MANGLER(RTAsn1SetOfCores_DecodeAsn1)
# define RTAsn1DynType_SetToNull                        RT_MANGLER(RTAsn1DynType_SetToNull)
# define RTAsn1DynType_SetToObjId                       RT_MANGLER(RTAsn1DynType_SetToObjId)
# define RTAsn1DynType_CheckSanity                      RT_MANGLER(RTAsn1DynType_CheckSanity)
# define RTAsn1DynType_Clone                            RT_MANGLER(RTAsn1DynType_Clone)
# define RTAsn1DynType_Compare                          RT_MANGLER(RTAsn1DynType_Compare)
# define RTAsn1DynType_Delete                           RT_MANGLER(RTAsn1DynType_Delete)
# define RTAsn1DynType_Enum                             RT_MANGLER(RTAsn1DynType_Enum)
# define RTAsn1DynType_Init                             RT_MANGLER(RTAsn1DynType_Init)
# define RTAsn1DynType_DecodeAsn1                       RT_MANGLER(RTAsn1DynType_DecodeAsn1)
# define RTAsn1Integer_CheckSanity                      RT_MANGLER(RTAsn1Integer_CheckSanity)
# define RTAsn1Integer_Clone                            RT_MANGLER(RTAsn1Integer_Clone)
# define RTAsn1Integer_Compare                          RT_MANGLER(RTAsn1Integer_Compare)
# define RTAsn1Integer_Delete                           RT_MANGLER(RTAsn1Integer_Delete)
# define RTAsn1Integer_Enum                             RT_MANGLER(RTAsn1Integer_Enum)
# define RTAsn1Integer_FromBigNum                       RT_MANGLER(RTAsn1Integer_FromBigNum)
# define RTAsn1Integer_Init                             RT_MANGLER(RTAsn1Integer_Init)
# define RTAsn1Integer_InitDefault                      RT_MANGLER(RTAsn1Integer_InitDefault)
# define RTAsn1Integer_InitU64                          RT_MANGLER(RTAsn1Integer_InitU64)
# define RTAsn1Integer_ToBigNum                         RT_MANGLER(RTAsn1Integer_ToBigNum)
# define RTAsn1Integer_ToString                         RT_MANGLER(RTAsn1Integer_ToString)
# define RTAsn1Integer_UnsignedCompare                  RT_MANGLER(RTAsn1Integer_UnsignedCompare)
# define RTAsn1Integer_UnsignedCompareWithU32           RT_MANGLER(RTAsn1Integer_UnsignedCompareWithU32)
# define RTAsn1Integer_UnsignedCompareWithU64           RT_MANGLER(RTAsn1Integer_UnsignedCompareWithU64)
# define RTAsn1Integer_UnsignedLastBit                  RT_MANGLER(RTAsn1Integer_UnsignedLastBit)
# define RTAsn1SeqOfIntegers_CheckSanity                RT_MANGLER(RTAsn1SeqOfIntegers_CheckSanity)
# define RTAsn1SeqOfIntegers_Clone                      RT_MANGLER(RTAsn1SeqOfIntegers_Clone)
# define RTAsn1SeqOfIntegers_Compare                    RT_MANGLER(RTAsn1SeqOfIntegers_Compare)
# define RTAsn1SeqOfIntegers_Delete                     RT_MANGLER(RTAsn1SeqOfIntegers_Delete)
# define RTAsn1SeqOfIntegers_Enum                       RT_MANGLER(RTAsn1SeqOfIntegers_Enum)
# define RTAsn1SeqOfIntegers_Init                       RT_MANGLER(RTAsn1SeqOfIntegers_Init)
# define RTAsn1SetOfIntegers_CheckSanity                RT_MANGLER(RTAsn1SetOfIntegers_CheckSanity)
# define RTAsn1SetOfIntegers_Clone                      RT_MANGLER(RTAsn1SetOfIntegers_Clone)
# define RTAsn1SetOfIntegers_Compare                    RT_MANGLER(RTAsn1SetOfIntegers_Compare)
# define RTAsn1SetOfIntegers_Delete                     RT_MANGLER(RTAsn1SetOfIntegers_Delete)
# define RTAsn1SetOfIntegers_Enum                       RT_MANGLER(RTAsn1SetOfIntegers_Enum)
# define RTAsn1SetOfIntegers_Init                       RT_MANGLER(RTAsn1SetOfIntegers_Init)
# define RTAsn1Integer_DecodeAsn1                       RT_MANGLER(RTAsn1Integer_DecodeAsn1)
# define RTAsn1SeqOfIntegers_DecodeAsn1                 RT_MANGLER(RTAsn1SeqOfIntegers_DecodeAsn1)
# define RTAsn1SetOfIntegers_DecodeAsn1                 RT_MANGLER(RTAsn1SetOfIntegers_DecodeAsn1)
# define RTAsn1Null_CheckSanity                         RT_MANGLER(RTAsn1Null_CheckSanity)
# define RTAsn1Null_Clone                               RT_MANGLER(RTAsn1Null_Clone)
# define RTAsn1Null_Compare                             RT_MANGLER(RTAsn1Null_Compare)
# define RTAsn1Null_Delete                              RT_MANGLER(RTAsn1Null_Delete)
# define RTAsn1Null_Enum                                RT_MANGLER(RTAsn1Null_Enum)
# define RTAsn1Null_Init                                RT_MANGLER(RTAsn1Null_Init)
# define RTAsn1Null_DecodeAsn1                          RT_MANGLER(RTAsn1Null_DecodeAsn1)
# define RTAsn1ObjIdCountComponents                     RT_MANGLER(RTAsn1ObjIdCountComponents)
# define RTAsn1ObjIdGetComponentsAsUInt32               RT_MANGLER(RTAsn1ObjIdGetComponentsAsUInt32)
# define RTAsn1ObjIdGetLastComponentsAsUInt32           RT_MANGLER(RTAsn1ObjIdGetLastComponentsAsUInt32)
# define RTAsn1ObjId_CheckSanity                        RT_MANGLER(RTAsn1ObjId_CheckSanity)
# define RTAsn1ObjId_Clone                              RT_MANGLER(RTAsn1ObjId_Clone)
# define RTAsn1ObjId_Compare                            RT_MANGLER(RTAsn1ObjId_Compare)
# define RTAsn1ObjId_CompareWithString                  RT_MANGLER(RTAsn1ObjId_CompareWithString)
# define RTAsn1ObjId_Delete                             RT_MANGLER(RTAsn1ObjId_Delete)
# define RTAsn1ObjId_Enum                               RT_MANGLER(RTAsn1ObjId_Enum)
# define RTAsn1ObjId_Init                               RT_MANGLER(RTAsn1ObjId_Init)
# define RTAsn1ObjId_InitFromString                     RT_MANGLER(RTAsn1ObjId_InitFromString)
# define RTAsn1ObjId_SetFromString                      RT_MANGLER(RTAsn1ObjId_SetFromString)
# define RTAsn1ObjId_StartsWith                         RT_MANGLER(RTAsn1ObjId_StartsWith)
# define RTAsn1SeqOfObjIds_CheckSanity                  RT_MANGLER(RTAsn1SeqOfObjIds_CheckSanity)
# define RTAsn1SeqOfObjIds_Clone                        RT_MANGLER(RTAsn1SeqOfObjIds_Clone)
# define RTAsn1SeqOfObjIds_Compare                      RT_MANGLER(RTAsn1SeqOfObjIds_Compare)
# define RTAsn1SeqOfObjIds_Delete                       RT_MANGLER(RTAsn1SeqOfObjIds_Delete)
# define RTAsn1SeqOfObjIds_Enum                         RT_MANGLER(RTAsn1SeqOfObjIds_Enum)
# define RTAsn1SeqOfObjIds_Init                         RT_MANGLER(RTAsn1SeqOfObjIds_Init)
# define RTAsn1SetOfObjIds_CheckSanity                  RT_MANGLER(RTAsn1SetOfObjIds_CheckSanity)
# define RTAsn1SetOfObjIds_Clone                        RT_MANGLER(RTAsn1SetOfObjIds_Clone)
# define RTAsn1SetOfObjIds_Compare                      RT_MANGLER(RTAsn1SetOfObjIds_Compare)
# define RTAsn1SetOfObjIds_Delete                       RT_MANGLER(RTAsn1SetOfObjIds_Delete)
# define RTAsn1SetOfObjIds_Enum                         RT_MANGLER(RTAsn1SetOfObjIds_Enum)
# define RTAsn1SetOfObjIds_Init                         RT_MANGLER(RTAsn1SetOfObjIds_Init)
# define RTAsn1SeqOfObjIdSeqs_CheckSanity               RT_MANGLER(RTAsn1SeqOfObjIdSeqs_CheckSanity)
# define RTAsn1SeqOfObjIdSeqs_Clone                     RT_MANGLER(RTAsn1SeqOfObjIdSeqs_Clone)
# define RTAsn1SeqOfObjIdSeqs_Compare                   RT_MANGLER(RTAsn1SeqOfObjIdSeqs_Compare)
# define RTAsn1SetOfObjIdSeqs_DecodeAsn1                RT_MANGLER(RTAsn1SetOfObjIdSeqs_DecodeAsn1)
# define RTAsn1SeqOfObjIdSeqs_Delete                    RT_MANGLER(RTAsn1SeqOfObjIdSeqs_Delete)
# define RTAsn1SeqOfObjIdSeqs_Enum                      RT_MANGLER(RTAsn1SeqOfObjIdSeqs_Enum)
# define RTAsn1SeqOfObjIdSeqs_Init                      RT_MANGLER(RTAsn1SeqOfObjIdSeqs_Init)
# define RTAsn1SetOfObjIdSeqs_CheckSanity               RT_MANGLER(RTAsn1SetOfObjIdSeqs_CheckSanity)
# define RTAsn1SetOfObjIdSeqs_Clone                     RT_MANGLER(RTAsn1SetOfObjIdSeqs_Clone)
# define RTAsn1SetOfObjIdSeqs_Compare                   RT_MANGLER(RTAsn1SetOfObjIdSeqs_Compare)
# define RTAsn1SetOfObjIdSeqs_Delete                    RT_MANGLER(RTAsn1SetOfObjIdSeqs_Delete)
# define RTAsn1SetOfObjIdSeqs_Enum                      RT_MANGLER(RTAsn1SetOfObjIdSeqs_Enum)
# define RTAsn1SetOfObjIdSeqs_Init                      RT_MANGLER(RTAsn1SetOfObjIdSeqs_Init)
# define RTAsn1ObjId_DecodeAsn1                         RT_MANGLER(RTAsn1ObjId_DecodeAsn1)
# define RTAsn1SeqOfObjIds_DecodeAsn1                   RT_MANGLER(RTAsn1SeqOfObjIds_DecodeAsn1)
# define RTAsn1SetOfObjIds_DecodeAsn1                   RT_MANGLER(RTAsn1SetOfObjIds_DecodeAsn1)
# define RTAsn1OctetString_AllocContent                 RT_MANGLER(RTAsn1OctetString_AllocContent)
# define RTAsn1OctetString_SetContent                   RT_MANGLER(RTAsn1OctetString_SetContent)
# define RTAsn1OctetString_CheckSanity                  RT_MANGLER(RTAsn1OctetString_CheckSanity)
# define RTAsn1OctetString_Clone                        RT_MANGLER(RTAsn1OctetString_Clone)
# define RTAsn1OctetString_Compare                      RT_MANGLER(RTAsn1OctetString_Compare)
# define RTAsn1OctetString_Delete                       RT_MANGLER(RTAsn1OctetString_Delete)
# define RTAsn1OctetString_Enum                         RT_MANGLER(RTAsn1OctetString_Enum)
# define RTAsn1OctetString_Init                         RT_MANGLER(RTAsn1OctetString_Init)
# define RTAsn1OctetString_AreContentBytesValid         RT_MANGLER(RTAsn1OctetString_AreContentBytesValid)
# define RTAsn1OctetString_RefreshContent               RT_MANGLER(RTAsn1OctetString_RefreshContent)
# define RTAsn1SeqOfOctetStrings_CheckSanity            RT_MANGLER(RTAsn1SeqOfOctetStrings_CheckSanity)
# define RTAsn1SeqOfOctetStrings_Clone                  RT_MANGLER(RTAsn1SeqOfOctetStrings_Clone)
# define RTAsn1SeqOfOctetStrings_Compare                RT_MANGLER(RTAsn1SeqOfOctetStrings_Compare)
# define RTAsn1SeqOfOctetStrings_Delete                 RT_MANGLER(RTAsn1SeqOfOctetStrings_Delete)
# define RTAsn1SeqOfOctetStrings_Enum                   RT_MANGLER(RTAsn1SeqOfOctetStrings_Enum)
# define RTAsn1SeqOfOctetStrings_Init                   RT_MANGLER(RTAsn1SeqOfOctetStrings_Init)
# define RTAsn1SetOfOctetStrings_CheckSanity            RT_MANGLER(RTAsn1SetOfOctetStrings_CheckSanity)
# define RTAsn1SetOfOctetStrings_Clone                  RT_MANGLER(RTAsn1SetOfOctetStrings_Clone)
# define RTAsn1SetOfOctetStrings_Compare                RT_MANGLER(RTAsn1SetOfOctetStrings_Compare)
# define RTAsn1SetOfOctetStrings_Delete                 RT_MANGLER(RTAsn1SetOfOctetStrings_Delete)
# define RTAsn1SetOfOctetStrings_Enum                   RT_MANGLER(RTAsn1SetOfOctetStrings_Enum)
# define RTAsn1SetOfOctetStrings_Init                   RT_MANGLER(RTAsn1SetOfOctetStrings_Init)
# define RTAsn1OctetString_DecodeAsn1                   RT_MANGLER(RTAsn1OctetString_DecodeAsn1)
# define RTAsn1SeqOfOctetStrings_DecodeAsn1             RT_MANGLER(RTAsn1SeqOfOctetStrings_DecodeAsn1)
# define RTAsn1SetOfOctetStrings_DecodeAsn1             RT_MANGLER(RTAsn1SetOfOctetStrings_DecodeAsn1)
# define RTAsn1BmpString_CheckSanity                    RT_MANGLER(RTAsn1BmpString_CheckSanity)
# define RTAsn1BmpString_Clone                          RT_MANGLER(RTAsn1BmpString_Clone)
# define RTAsn1BmpString_Compare                        RT_MANGLER(RTAsn1BmpString_Compare)
# define RTAsn1BmpString_Delete                         RT_MANGLER(RTAsn1BmpString_Delete)
# define RTAsn1BmpString_Enum                           RT_MANGLER(RTAsn1BmpString_Enum)
# define RTAsn1BmpString_Init                           RT_MANGLER(RTAsn1BmpString_Init)
# define RTAsn1GeneralString_CheckSanity                RT_MANGLER(RTAsn1GeneralString_CheckSanity)
# define RTAsn1GeneralString_Clone                      RT_MANGLER(RTAsn1GeneralString_Clone)
# define RTAsn1GeneralString_Compare                    RT_MANGLER(RTAsn1GeneralString_Compare)
# define RTAsn1GeneralString_Delete                     RT_MANGLER(RTAsn1GeneralString_Delete)
# define RTAsn1GeneralString_Enum                       RT_MANGLER(RTAsn1GeneralString_Enum)
# define RTAsn1GeneralString_Init                       RT_MANGLER(RTAsn1GeneralString_Init)
# define RTAsn1GraphicString_CheckSanity                RT_MANGLER(RTAsn1GraphicString_CheckSanity)
# define RTAsn1GraphicString_Clone                      RT_MANGLER(RTAsn1GraphicString_Clone)
# define RTAsn1GraphicString_Compare                    RT_MANGLER(RTAsn1GraphicString_Compare)
# define RTAsn1GraphicString_Delete                     RT_MANGLER(RTAsn1GraphicString_Delete)
# define RTAsn1GraphicString_Enum                       RT_MANGLER(RTAsn1GraphicString_Enum)
# define RTAsn1GraphicString_Init                       RT_MANGLER(RTAsn1GraphicString_Init)
# define RTAsn1Ia5String_CheckSanity                    RT_MANGLER(RTAsn1Ia5String_CheckSanity)
# define RTAsn1Ia5String_Clone                          RT_MANGLER(RTAsn1Ia5String_Clone)
# define RTAsn1Ia5String_Compare                        RT_MANGLER(RTAsn1Ia5String_Compare)
# define RTAsn1Ia5String_Delete                         RT_MANGLER(RTAsn1Ia5String_Delete)
# define RTAsn1Ia5String_Enum                           RT_MANGLER(RTAsn1Ia5String_Enum)
# define RTAsn1Ia5String_Init                           RT_MANGLER(RTAsn1Ia5String_Init)
# define RTAsn1NumericString_CheckSanity                RT_MANGLER(RTAsn1NumericString_CheckSanity)
# define RTAsn1NumericString_Clone                      RT_MANGLER(RTAsn1NumericString_Clone)
# define RTAsn1NumericString_Compare                    RT_MANGLER(RTAsn1NumericString_Compare)
# define RTAsn1NumericString_Delete                     RT_MANGLER(RTAsn1NumericString_Delete)
# define RTAsn1NumericString_Enum                       RT_MANGLER(RTAsn1NumericString_Enum)
# define RTAsn1NumericString_Init                       RT_MANGLER(RTAsn1NumericString_Init)
# define RTAsn1PrintableString_CheckSanity              RT_MANGLER(RTAsn1PrintableString_CheckSanity)
# define RTAsn1PrintableString_Clone                    RT_MANGLER(RTAsn1PrintableString_Clone)
# define RTAsn1PrintableString_Compare                  RT_MANGLER(RTAsn1PrintableString_Compare)
# define RTAsn1PrintableString_Delete                   RT_MANGLER(RTAsn1PrintableString_Delete)
# define RTAsn1PrintableString_Enum                     RT_MANGLER(RTAsn1PrintableString_Enum)
# define RTAsn1PrintableString_Init                     RT_MANGLER(RTAsn1PrintableString_Init)
# define RTAsn1SeqOfStrings_CheckSanity                 RT_MANGLER(RTAsn1SeqOfStrings_CheckSanity)
# define RTAsn1SeqOfStrings_Clone                       RT_MANGLER(RTAsn1SeqOfStrings_Clone)
# define RTAsn1SeqOfStrings_Compare                     RT_MANGLER(RTAsn1SeqOfStrings_Compare)
# define RTAsn1SeqOfStrings_Delete                      RT_MANGLER(RTAsn1SeqOfStrings_Delete)
# define RTAsn1SeqOfStrings_Enum                        RT_MANGLER(RTAsn1SeqOfStrings_Enum)
# define RTAsn1SeqOfStrings_Init                        RT_MANGLER(RTAsn1SeqOfStrings_Init)
# define RTAsn1SetOfStrings_CheckSanity                 RT_MANGLER(RTAsn1SetOfStrings_CheckSanity)
# define RTAsn1SetOfStrings_Clone                       RT_MANGLER(RTAsn1SetOfStrings_Clone)
# define RTAsn1SetOfStrings_Compare                     RT_MANGLER(RTAsn1SetOfStrings_Compare)
# define RTAsn1SetOfStrings_Delete                      RT_MANGLER(RTAsn1SetOfStrings_Delete)
# define RTAsn1SetOfStrings_Enum                        RT_MANGLER(RTAsn1SetOfStrings_Enum)
# define RTAsn1SetOfStrings_Init                        RT_MANGLER(RTAsn1SetOfStrings_Init)
# define RTAsn1String_CheckSanity                       RT_MANGLER(RTAsn1String_CheckSanity)
# define RTAsn1String_Clone                             RT_MANGLER(RTAsn1String_Clone)
# define RTAsn1String_Compare                           RT_MANGLER(RTAsn1String_Compare)
# define RTAsn1String_CompareEx                         RT_MANGLER(RTAsn1String_CompareEx)
# define RTAsn1String_CompareValues                     RT_MANGLER(RTAsn1String_CompareValues)
# define RTAsn1String_CompareWithString                 RT_MANGLER(RTAsn1String_CompareWithString)
# define RTAsn1String_Delete                            RT_MANGLER(RTAsn1String_Delete)
# define RTAsn1String_Enum                              RT_MANGLER(RTAsn1String_Enum)
# define RTAsn1String_Init                              RT_MANGLER(RTAsn1String_Init)
# define RTAsn1String_InitEx                            RT_MANGLER(RTAsn1String_InitEx)
# define RTAsn1String_InitWithValue                     RT_MANGLER(RTAsn1String_InitWithValue)
# define RTAsn1String_QueryUtf8                         RT_MANGLER(RTAsn1String_QueryUtf8)
# define RTAsn1String_QueryUtf8Len                      RT_MANGLER(RTAsn1String_QueryUtf8Len)
# define RTAsn1String_RecodeAsUtf8                      RT_MANGLER(RTAsn1String_RecodeAsUtf8)
# define RTAsn1T61String_CheckSanity                    RT_MANGLER(RTAsn1T61String_CheckSanity)
# define RTAsn1T61String_Clone                          RT_MANGLER(RTAsn1T61String_Clone)
# define RTAsn1T61String_Compare                        RT_MANGLER(RTAsn1T61String_Compare)
# define RTAsn1T61String_Delete                         RT_MANGLER(RTAsn1T61String_Delete)
# define RTAsn1T61String_Enum                           RT_MANGLER(RTAsn1T61String_Enum)
# define RTAsn1T61String_Init                           RT_MANGLER(RTAsn1T61String_Init)
# define RTAsn1UniversalString_CheckSanity              RT_MANGLER(RTAsn1UniversalString_CheckSanity)
# define RTAsn1UniversalString_Clone                    RT_MANGLER(RTAsn1UniversalString_Clone)
# define RTAsn1UniversalString_Compare                  RT_MANGLER(RTAsn1UniversalString_Compare)
# define RTAsn1UniversalString_Delete                   RT_MANGLER(RTAsn1UniversalString_Delete)
# define RTAsn1UniversalString_Enum                     RT_MANGLER(RTAsn1UniversalString_Enum)
# define RTAsn1UniversalString_Init                     RT_MANGLER(RTAsn1UniversalString_Init)
# define RTAsn1Utf8String_CheckSanity                   RT_MANGLER(RTAsn1Utf8String_CheckSanity)
# define RTAsn1Utf8String_Clone                         RT_MANGLER(RTAsn1Utf8String_Clone)
# define RTAsn1Utf8String_Compare                       RT_MANGLER(RTAsn1Utf8String_Compare)
# define RTAsn1Utf8String_Delete                        RT_MANGLER(RTAsn1Utf8String_Delete)
# define RTAsn1Utf8String_Enum                          RT_MANGLER(RTAsn1Utf8String_Enum)
# define RTAsn1Utf8String_Init                          RT_MANGLER(RTAsn1Utf8String_Init)
# define RTAsn1VisibleString_CheckSanity                RT_MANGLER(RTAsn1VisibleString_CheckSanity)
# define RTAsn1VisibleString_Clone                      RT_MANGLER(RTAsn1VisibleString_Clone)
# define RTAsn1VisibleString_Compare                    RT_MANGLER(RTAsn1VisibleString_Compare)
# define RTAsn1VisibleString_Delete                     RT_MANGLER(RTAsn1VisibleString_Delete)
# define RTAsn1VisibleString_Enum                       RT_MANGLER(RTAsn1VisibleString_Enum)
# define RTAsn1VisibleString_Init                       RT_MANGLER(RTAsn1VisibleString_Init)
# define RTAsn1BmpString_DecodeAsn1                     RT_MANGLER(RTAsn1BmpString_DecodeAsn1)
# define RTAsn1GeneralString_DecodeAsn1                 RT_MANGLER(RTAsn1GeneralString_DecodeAsn1)
# define RTAsn1GraphicString_DecodeAsn1                 RT_MANGLER(RTAsn1GraphicString_DecodeAsn1)
# define RTAsn1Ia5String_DecodeAsn1                     RT_MANGLER(RTAsn1Ia5String_DecodeAsn1)
# define RTAsn1NumericString_DecodeAsn1                 RT_MANGLER(RTAsn1NumericString_DecodeAsn1)
# define RTAsn1PrintableString_DecodeAsn1               RT_MANGLER(RTAsn1PrintableString_DecodeAsn1)
# define RTAsn1SeqOfStrings_DecodeAsn1                  RT_MANGLER(RTAsn1SeqOfStrings_DecodeAsn1)
# define RTAsn1SetOfStrings_DecodeAsn1                  RT_MANGLER(RTAsn1SetOfStrings_DecodeAsn1)
# define RTAsn1String_DecodeAsn1                        RT_MANGLER(RTAsn1String_DecodeAsn1)
# define RTAsn1T61String_DecodeAsn1                     RT_MANGLER(RTAsn1T61String_DecodeAsn1)
# define RTAsn1UniversalString_DecodeAsn1               RT_MANGLER(RTAsn1UniversalString_DecodeAsn1)
# define RTAsn1Utf8String_DecodeAsn1                    RT_MANGLER(RTAsn1Utf8String_DecodeAsn1)
# define RTAsn1VisibleString_DecodeAsn1                 RT_MANGLER(RTAsn1VisibleString_DecodeAsn1)
# define RTAsn1GeneralizedTime_CheckSanity              RT_MANGLER(RTAsn1GeneralizedTime_CheckSanity)
# define RTAsn1GeneralizedTime_Clone                    RT_MANGLER(RTAsn1GeneralizedTime_Clone)
# define RTAsn1GeneralizedTime_Compare                  RT_MANGLER(RTAsn1GeneralizedTime_Compare)
# define RTAsn1GeneralizedTime_Delete                   RT_MANGLER(RTAsn1GeneralizedTime_Delete)
# define RTAsn1GeneralizedTime_Enum                     RT_MANGLER(RTAsn1GeneralizedTime_Enum)
# define RTAsn1GeneralizedTime_Init                     RT_MANGLER(RTAsn1GeneralizedTime_Init)
# define RTAsn1SeqOfTimes_CheckSanity                   RT_MANGLER(RTAsn1SeqOfTimes_CheckSanity)
# define RTAsn1SeqOfTimes_Clone                         RT_MANGLER(RTAsn1SeqOfTimes_Clone)
# define RTAsn1SeqOfTimes_Compare                       RT_MANGLER(RTAsn1SeqOfTimes_Compare)
# define RTAsn1SeqOfTimes_Delete                        RT_MANGLER(RTAsn1SeqOfTimes_Delete)
# define RTAsn1SeqOfTimes_Enum                          RT_MANGLER(RTAsn1SeqOfTimes_Enum)
# define RTAsn1SeqOfTimes_Init                          RT_MANGLER(RTAsn1SeqOfTimes_Init)
# define RTAsn1SetOfTimes_CheckSanity                   RT_MANGLER(RTAsn1SetOfTimes_CheckSanity)
# define RTAsn1SetOfTimes_Clone                         RT_MANGLER(RTAsn1SetOfTimes_Clone)
# define RTAsn1SetOfTimes_Compare                       RT_MANGLER(RTAsn1SetOfTimes_Compare)
# define RTAsn1SetOfTimes_Delete                        RT_MANGLER(RTAsn1SetOfTimes_Delete)
# define RTAsn1SetOfTimes_Enum                          RT_MANGLER(RTAsn1SetOfTimes_Enum)
# define RTAsn1SetOfTimes_Init                          RT_MANGLER(RTAsn1SetOfTimes_Init)
# define RTAsn1Time_CheckSanity                         RT_MANGLER(RTAsn1Time_CheckSanity)
# define RTAsn1Time_Clone                               RT_MANGLER(RTAsn1Time_Clone)
# define RTAsn1Time_Compare                             RT_MANGLER(RTAsn1Time_Compare)
# define RTAsn1Time_CompareWithTimeSpec                 RT_MANGLER(RTAsn1Time_CompareWithTimeSpec)
# define RTAsn1Time_Delete                              RT_MANGLER(RTAsn1Time_Delete)
# define RTAsn1Time_Enum                                RT_MANGLER(RTAsn1Time_Enum)
# define RTAsn1Time_Init                                RT_MANGLER(RTAsn1Time_Init)
# define RTAsn1Time_InitEx                              RT_MANGLER(RTAsn1Time_InitEx)
# define RTAsn1Time_InitWithTime                        RT_MANGLER(RTAsn1Time_InitWithTime)
# define RTAsn1Time_SetTime                             RT_MANGLER(RTAsn1Time_SetTime)
# define RTAsn1Time_SetTimeSpec                         RT_MANGLER(RTAsn1Time_SetTimeSpec)
# define RTAsn1UtcTime_CheckSanity                      RT_MANGLER(RTAsn1UtcTime_CheckSanity)
# define RTAsn1UtcTime_Clone                            RT_MANGLER(RTAsn1UtcTime_Clone)
# define RTAsn1UtcTime_Compare                          RT_MANGLER(RTAsn1UtcTime_Compare)
# define RTAsn1UtcTime_Delete                           RT_MANGLER(RTAsn1UtcTime_Delete)
# define RTAsn1UtcTime_Enum                             RT_MANGLER(RTAsn1UtcTime_Enum)
# define RTAsn1UtcTime_Init                             RT_MANGLER(RTAsn1UtcTime_Init)
# define RTAsn1GeneralizedTime_DecodeAsn1               RT_MANGLER(RTAsn1GeneralizedTime_DecodeAsn1)
# define RTAsn1SeqOfTimes_DecodeAsn1                    RT_MANGLER(RTAsn1SeqOfTimes_DecodeAsn1)
# define RTAsn1SetOfTimes_DecodeAsn1                    RT_MANGLER(RTAsn1SetOfTimes_DecodeAsn1)
# define RTAsn1Time_DecodeAsn1                          RT_MANGLER(RTAsn1Time_DecodeAsn1)
# define RTAsn1UtcTime_DecodeAsn1                       RT_MANGLER(RTAsn1UtcTime_DecodeAsn1)
# define RTMd2                                          RT_MANGLER(RTMd2)
# define RTMd2Final                                     RT_MANGLER(RTMd2Final)
# define RTMd2Init                                      RT_MANGLER(RTMd2Init)
# define RTMd2Update                                    RT_MANGLER(RTMd2Update)
# define RTMd2FromString                                RT_MANGLER(RTMd2FromString)
# define RTMd2ToString                                  RT_MANGLER(RTMd2ToString)
# define RTCrCipherDecrypt                              RT_MANGLER(RTCrCipherDecrypt)
# define RTCrCipherEncrypt                              RT_MANGLER(RTCrCipherEncrypt)
# define RTCrCipherGetBlockSize                         RT_MANGLER(RTCrCipherGetBlockSize)
# define RTCrCipherGetInitializationVectorLength        RT_MANGLER(RTCrCipherGetInitializationVectorLength)
# define RTCrCipherGetKeyLength                         RT_MANGLER(RTCrCipherGetKeyLength)
# define RTCrCipherOpenByType                           RT_MANGLER(RTCrCipherOpenByType)
# define RTCrCipherRetain                               RT_MANGLER(RTCrCipherRetain)
# define RTCrCipherRelease                              RT_MANGLER(RTCrCipherRelease)
# define RTCrCipherCtxFree                              RT_MANGLER(RTCrCipherCtxFree)
# define RTCrCipherCtxDecryptInit                       RT_MANGLER(RTCrCipherCtxDecryptInit)
# define RTCrCipherCtxDecryptFinish                     RT_MANGLER(RTCrCipherCtxDecryptFinish)
# define RTCrCipherCtxDecryptProcess                    RT_MANGLER(RTCrCipherCtxDecryptProcess)
# define RTCrCipherCtxEncryptInit                       RT_MANLGER(RTCrCipherCtxEncryptInit)
# define RTCrCipherCtxEncryptFinish                     RT_MANGLER(RTCrCipherCtxEncryptFinish)
# define RTCrCipherCtxEncryptProcess                    RT_MANGLER(RTCrCipherCtxEncryptProcess)
# define RTCrCipherDecrypt                              RT_MANGLER(RTCrCipherDecrypt)
# define RTCrCipherDecryptEx                            RT_MANGLER(RTCrCipherDecryptEx)
# define RTCrCipherEncrypt                              RT_MANGLER(RTCrCipherEncrypt)
# define RTCrCipherEncryptEx                            RT_MANGLER(RTCrCipherEncryptEx)
# define RTCrDigestClone                                RT_MANGLER(RTCrDigestClone)
# define RTCrDigestCreate                               RT_MANGLER(RTCrDigestCreate)
# define RTCrDigestFinal                                RT_MANGLER(RTCrDigestFinal)
# define RTCrDigestGetConsumedSize                      RT_MANGLER(RTCrDigestGetConsumedSize)
# define RTCrDigestGetFlags                             RT_MANGLER(RTCrDigestGetFlags)
# define RTCrDigestGetHash                              RT_MANGLER(RTCrDigestGetHash)
# define RTCrDigestGetHashSize                          RT_MANGLER(RTCrDigestGetHashSize)
# define RTCrDigestGetType                              RT_MANGLER(RTCrDigestGetType)
# define RTCrDigestGetAlgorithmOid                      RT_MANGLER(RTCrDigestGetAlgorithmOid)
# define RTCrDigestIsFinalized                          RT_MANGLER(RTCrDigestIsFinalized)
# define RTCrDigestMatch                                RT_MANGLER(RTCrDigestMatch)
# define RTCrDigestRelease                              RT_MANGLER(RTCrDigestRelease)
# define RTCrDigestReset                                RT_MANGLER(RTCrDigestReset)
# define RTCrDigestRetain                               RT_MANGLER(RTCrDigestRetain)
# define RTCrDigestUpdate                               RT_MANGLER(RTCrDigestUpdate)
# define RTCrDigestUpdateFromVfsFile                    RT_MANGLER(RTCrDigestUpdateFromVfsFile)
# define RTCrDigestCreateByObjId                        RT_MANGLER(RTCrDigestCreateByObjId)
# define RTCrDigestCreateByObjIdString                  RT_MANGLER(RTCrDigestCreateByObjIdString)
# define RTCrDigestCreateByType                         RT_MANGLER(RTCrDigestCreateByType)
# define RTCrDigestFindByObjId                          RT_MANGLER(RTCrDigestFindByObjId)
# define RTCrDigestFindByObjIdString                    RT_MANGLER(RTCrDigestFindByObjIdString)
# define RTCrDigestFindByType                           RT_MANGLER(RTCrDigestFindByType)
# define RTCrDigestTypeToAlgorithmOid                   RT_MANGLER(RTCrDigestTypeToAlgorithmOid)
# define RTCrDigestTypeToName                           RT_MANGLER(RTCrDigestTypeToName)
# define RTCrDigestTypeToHashSize                       RT_MANGLER(RTCrDigestTypeToHashSize)
# define RTCrKeyCreateFromBuffer                        RT_MANGLER(RTCrKeyCreateFromBuffer)
# define RTCrKeyCreateFromFile                          RT_MANGLER(RTCrKeyCreateFromFile)
# define RTCrKeyCreateFromPemSection                    RT_MANGLER(RTCrKeyCreateFromPemSection)
# define RTCrKeyCreateFromPublicAlgorithmAndBits        RT_MANGLER(RTCrKeyCreateFromPublicAlgorithmAndBits)
# define RTCrKeyCreateFromSubjectPublicKeyInfo          RT_MANGLER(RTCrKeyCreateFromSubjectPublicKeyInfo)
# define RTCrKeyCreateNewRsa                            RT_MANGLER(RTCrKeyCreateNewRsa)
# define RTCrKeyGetBitCount                             RT_MANGLER(RTCrKeyGetBitCount)
# define RTCrKeyGetType                                 RT_MANGLER(RTCrKeyGetType)
# define RTCrKeyHasPrivatePart                          RT_MANGLER(RTCrKeyHasPrivatePart)
# define RTCrKeyHasPublicPart                           RT_MANGLER(RTCrKeyHasPublicPart)
# define RTCrKeyRelease                                 RT_MANGLER(RTCrKeyRelease)
# define RTCrKeyRetain                                  RT_MANGLER(RTCrKeyRetain)
# define RTCrKeyQueryRsaModulus                         RT_MANGLER(RTCrKeyQueryRsaModulus)
# define RTCrKeyQueryRsaPrivateExponent                 RT_MANGLER(RTCrKeyQueryRsaPrivateExponent)
# define RTCrKeyVerifyParameterCompatibility            RT_MANGLER(RTCrKeyVerifyParameterCompatibility)
# define RTCrRc4                                        RT_MANGLER(RTCrRc4)
# define RTCrRc4SetKey                                  RT_MANGLER(RTCrRc4SetKey)
# define RTCrRsaDigestInfo_DecodeAsn1                   RT_MANGLER(RTCrRsaDigestInfo_DecodeAsn1)
# define RTCrRsaOtherPrimeInfo_DecodeAsn1               RT_MANGLER(RTCrRsaOtherPrimeInfo_DecodeAsn1)
# define RTCrRsaOtherPrimeInfos_DecodeAsn1              RT_MANGLER(RTCrRsaOtherPrimeInfos_DecodeAsn1)
# define RTCrRsaPrivateKey_DecodeAsn1                   RT_MANGLER(RTCrRsaPrivateKey_DecodeAsn1)
# define RTCrRsaPublicKey_DecodeAsn1                    RT_MANGLER(RTCrRsaPublicKey_DecodeAsn1)
# define RTCrRsaDigestInfo_Compare                      RT_MANGLER(RTCrRsaDigestInfo_Compare)
# define RTCrRsaDigestInfo_Delete                       RT_MANGLER(RTCrRsaDigestInfo_Delete)
# define RTCrRsaDigestInfo_Enum                         RT_MANGLER(RTCrRsaDigestInfo_Enum)
# define RTCrRsaOtherPrimeInfo_Compare                  RT_MANGLER(RTCrRsaOtherPrimeInfo_Compare)
# define RTCrRsaOtherPrimeInfo_Delete                   RT_MANGLER(RTCrRsaOtherPrimeInfo_Delete)
# define RTCrRsaOtherPrimeInfo_Enum                     RT_MANGLER(RTCrRsaOtherPrimeInfo_Enum)
# define RTCrRsaOtherPrimeInfos_Compare                 RT_MANGLER(RTCrRsaOtherPrimeInfos_Compare)
# define RTCrRsaOtherPrimeInfos_Delete                  RT_MANGLER(RTCrRsaOtherPrimeInfos_Delete)
# define RTCrRsaOtherPrimeInfos_Enum                    RT_MANGLER(RTCrRsaOtherPrimeInfos_Enum)
# define RTCrRsaPrivateKey_Compare                      RT_MANGLER(RTCrRsaPrivateKey_Compare)
# define RTCrRsaPrivateKey_Delete                       RT_MANGLER(RTCrRsaPrivateKey_Delete)
# define RTCrRsaPrivateKey_Enum                         RT_MANGLER(RTCrRsaPrivateKey_Enum)
# define RTCrRsaPublicKey_Compare                       RT_MANGLER(RTCrRsaPublicKey_Compare)
# define RTCrRsaPublicKey_Delete                        RT_MANGLER(RTCrRsaPublicKey_Delete)
# define RTCrRsaPublicKey_Enum                          RT_MANGLER(RTCrRsaPublicKey_Enum)
# define RTCrRsaDigestInfo_Clone                        RT_MANGLER(RTCrRsaDigestInfo_Clone)
# define RTCrRsaDigestInfo_Init                         RT_MANGLER(RTCrRsaDigestInfo_Init)
# define RTCrRsaOtherPrimeInfo_Clone                    RT_MANGLER(RTCrRsaOtherPrimeInfo_Clone)
# define RTCrRsaOtherPrimeInfo_Init                     RT_MANGLER(RTCrRsaOtherPrimeInfo_Init)
# define RTCrRsaOtherPrimeInfos_Clone                   RT_MANGLER(RTCrRsaOtherPrimeInfos_Clone)
# define RTCrRsaOtherPrimeInfos_Init                    RT_MANGLER(RTCrRsaOtherPrimeInfos_Init)
# define RTCrRsaPrivateKey_Clone                        RT_MANGLER(RTCrRsaPrivateKey_Clone)
# define RTCrRsaPrivateKey_Init                         RT_MANGLER(RTCrRsaPrivateKey_Init)
# define RTCrRsaPublicKey_Clone                         RT_MANGLER(RTCrRsaPublicKey_Clone)
# define RTCrRsaPublicKey_Init                          RT_MANGLER(RTCrRsaPublicKey_Init)
# define RTCrRsaDigestInfo_CheckSanity                  RT_MANGLER(RTCrRsaDigestInfo_CheckSanity)
# define RTCrRsaOtherPrimeInfo_CheckSanity              RT_MANGLER(RTCrRsaOtherPrimeInfo_CheckSanity)
# define RTCrRsaOtherPrimeInfos_CheckSanity             RT_MANGLER(RTCrRsaOtherPrimeInfos_CheckSanity)
# define RTCrRsaPrivateKey_CheckSanity                  RT_MANGLER(RTCrRsaPrivateKey_CheckSanity)
# define RTCrRsaPrivateKey_CanHandleDigestType          RT_MANGLER(RTCrRsaPrivateKey_CanHandleDigestType)
# define RTCrRsaPublicKey_CheckSanity                   RT_MANGLER(RTCrRsaPublicKey_CheckSanity)
# define RTCrRsaPublicKey_CanHandleDigestType           RT_MANGLER(RTCrRsaPublicKey_CanHandleDigestType)
# define RTCrPemFindFirstSectionInContent               RT_MANGLER(RTCrPemFindFirstSectionInContent)
# define RTCrPemFreeSections                            RT_MANGLER(RTCrPemFreeSections)
# define RTCrPemParseContent                            RT_MANGLER(RTCrPemParseContent)
# define RTCrPemReadFile                                RT_MANGLER(RTCrPemReadFile)
# define RTCrPemWriteBlob                               RT_MANGLER(RTCrPemWriteBlob)
# define RTCrPemWriteBlobToVfsIoStrm                    RT_MANGLER(RTCrPemWriteBlobToVfsIoStrm)
# define RTCrPemWriteBlobToVfsFile                      RT_MANGLER(RTCrPemWriteBlobToVfsFile)
# define RTCrPemWriteAsn1                               RT_MANGLER(RTCrPemWriteAsn1)
# define RTCrPemWriteAsn1ToVfsIoStrm                    RT_MANGLER(RTCrPemWriteAsn1ToVfsIoStrm)
# define RTCrPemWriteAsn1ToVfsFile                      RT_MANGLER(RTCrPemWriteAsn1ToVfsFile)
# define RTCrPkcs5Pbkdf2Hmac                            RT_MANGLER(RTCrPkcs5Pbkdf2Hmac)
# define RTCrPkcs7_ReadFromBuffer                       RT_MANGLER(RTCrPkcs7_ReadFromBuffer)
# define RTCrPkcs7Attribute_SetAppleMultiCdPlist        RT_MANGLER(RTCrPkcs7Attribute_SetAppleMultiCdPlist)
# define RTCrPkcs7Attribute_SetContentType              RT_MANGLER(RTCrPkcs7Attribute_SetContentType)
# define RTCrPkcs7Attribute_SetCounterSignatures        RT_MANGLER(RTCrPkcs7Attribute_SetCounterSignatures)
# define RTCrPkcs7Attribute_SetMessageDigest            RT_MANGLER(RTCrPkcs7Attribute_SetMessageDigest)
# define RTCrPkcs7Attribute_SetMsStatementType          RT_MANGLER(RTCrPkcs7Attribute_SetMsStatementType)
# define RTCrPkcs7Attribute_SetMsNestedSignature        RT_MANGLER(RTCrPkcs7Attribute_SetMsNestedSignature)
# define RTCrPkcs7Attribute_SetMsTimestamp              RT_MANGLER(RTCrPkcs7Attribute_SetMsTimestamp)
# define RTCrPkcs7Attribute_SetSigningTime              RT_MANGLER(RTCrPkcs7Attribute_SetSigningTime)
# define RTCrPkcs7Attributes_HashAttributes             RT_MANGLER(RTCrPkcs7Attributes_HashAttributes)
# define RTCrPkcs7Attribute_DecodeAsn1                  RT_MANGLER(RTCrPkcs7Attribute_DecodeAsn1)
# define RTCrPkcs7Attributes_DecodeAsn1                 RT_MANGLER(RTCrPkcs7Attributes_DecodeAsn1)
# define RTCrPkcs7ContentInfo_DecodeAsn1                RT_MANGLER(RTCrPkcs7ContentInfo_DecodeAsn1)
# define RTCrPkcs7DigestInfo_DecodeAsn1                 RT_MANGLER(RTCrPkcs7DigestInfo_DecodeAsn1)
# define RTCrPkcs7IssuerAndSerialNumber_DecodeAsn1      RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_DecodeAsn1)
# define RTCrPkcs7SignedData_SetCertificates            RT_MANGLER(RTCrPkcs7SignedData_SetCertificates)
# define RTCrPkcs7SignedData_SetCrls                    RT_MANGLER(RTCrPkcs7SignedData_SetCrls)
# define RTCrPkcs7SignedData_DecodeAsn1                 RT_MANGLER(RTCrPkcs7SignedData_DecodeAsn1)
# define RTCrPkcs7SignerInfo_SetAuthenticatedAttributes RT_MANGLER(RTCrPkcs7SignerInfo_SetAuthenticatedAttributes)
# define RTCrPkcs7SignerInfo_SetUnauthenticatedAttributes RT_MANGLER(RTCrPkcs7SignerInfo_SetUnauthenticatedAttributes)
# define RTCrPkcs7SignerInfo_DecodeAsn1                 RT_MANGLER(RTCrPkcs7SignerInfo_DecodeAsn1)
# define RTCrPkcs7SignerInfos_DecodeAsn1                RT_MANGLER(RTCrPkcs7SignerInfos_DecodeAsn1)
# define RTCrPkcs7Attribute_Compare                     RT_MANGLER(RTCrPkcs7Attribute_Compare)
# define RTCrPkcs7Attribute_Delete                      RT_MANGLER(RTCrPkcs7Attribute_Delete)
# define RTCrPkcs7Attribute_Enum                        RT_MANGLER(RTCrPkcs7Attribute_Enum)
# define RTCrPkcs7Attributes_Compare                    RT_MANGLER(RTCrPkcs7Attributes_Compare)
# define RTCrPkcs7Attributes_Delete                     RT_MANGLER(RTCrPkcs7Attributes_Delete)
# define RTCrPkcs7Attributes_Enum                       RT_MANGLER(RTCrPkcs7Attributes_Enum)
# define RTCrPkcs7ContentInfo_Compare                   RT_MANGLER(RTCrPkcs7ContentInfo_Compare)
# define RTCrPkcs7ContentInfo_Delete                    RT_MANGLER(RTCrPkcs7ContentInfo_Delete)
# define RTCrPkcs7ContentInfo_Enum                      RT_MANGLER(RTCrPkcs7ContentInfo_Enum)
# define RTCrPkcs7ContentInfo_IsSignedData              RT_MANGLER(RTCrPkcs7ContentInfo_IsSignedData)
# define RTCrPkcs7DigestInfo_Compare                    RT_MANGLER(RTCrPkcs7DigestInfo_Compare)
# define RTCrPkcs7DigestInfo_Delete                     RT_MANGLER(RTCrPkcs7DigestInfo_Delete)
# define RTCrPkcs7DigestInfo_Enum                       RT_MANGLER(RTCrPkcs7DigestInfo_Enum)
# define RTCrPkcs7IssuerAndSerialNumber_Compare         RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_Compare)
# define RTCrPkcs7IssuerAndSerialNumber_Delete          RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_Delete)
# define RTCrPkcs7IssuerAndSerialNumber_Enum            RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_Enum)
# define RTCrPkcs7SignedData_Compare                    RT_MANGLER(RTCrPkcs7SignedData_Compare)
# define RTCrPkcs7SignedData_Delete                     RT_MANGLER(RTCrPkcs7SignedData_Delete)
# define RTCrPkcs7SignedData_Enum                       RT_MANGLER(RTCrPkcs7SignedData_Enum)
# define RTCrPkcs7SignerInfo_Compare                    RT_MANGLER(RTCrPkcs7SignerInfo_Compare)
# define RTCrPkcs7SignerInfo_Delete                     RT_MANGLER(RTCrPkcs7SignerInfo_Delete)
# define RTCrPkcs7SignerInfo_Enum                       RT_MANGLER(RTCrPkcs7SignerInfo_Enum)
# define RTCrPkcs7SignerInfo_GetSigningTime             RT_MANGLER(RTCrPkcs7SignerInfo_GetSigningTime)
# define RTCrPkcs7SignerInfo_GetMsTimestamp             RT_MANGLER(RTCrPkcs7SignerInfo_GetMsTimestamp)
# define RTCrPkcs7SignerInfos_Compare                   RT_MANGLER(RTCrPkcs7SignerInfos_Compare)
# define RTCrPkcs7SignerInfos_Delete                    RT_MANGLER(RTCrPkcs7SignerInfos_Delete)
# define RTCrPkcs7SignerInfos_Enum                      RT_MANGLER(RTCrPkcs7SignerInfos_Enum)
# define RTCrPkcs7Attribute_Clone                       RT_MANGLER(RTCrPkcs7Attribute_Clone)
# define RTCrPkcs7Attribute_Init                        RT_MANGLER(RTCrPkcs7Attribute_Init)
# define RTCrPkcs7Attributes_Clone                      RT_MANGLER(RTCrPkcs7Attributes_Clone)
# define RTCrPkcs7Attributes_Init                       RT_MANGLER(RTCrPkcs7Attributes_Init)
# define RTCrPkcs7ContentInfo_Clone                     RT_MANGLER(RTCrPkcs7ContentInfo_Clone)
# define RTCrPkcs7ContentInfo_Init                      RT_MANGLER(RTCrPkcs7ContentInfo_Init)
# define RTCrPkcs7DigestInfo_Clone                      RT_MANGLER(RTCrPkcs7DigestInfo_Clone)
# define RTCrPkcs7DigestInfo_Init                       RT_MANGLER(RTCrPkcs7DigestInfo_Init)
# define RTCrPkcs7IssuerAndSerialNumber_Clone           RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_Clone)
# define RTCrPkcs7IssuerAndSerialNumber_Init            RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_Init)
# define RTCrPkcs7SignedData_Clone                      RT_MANGLER(RTCrPkcs7SignedData_Clone)
# define RTCrPkcs7SignedData_Init                       RT_MANGLER(RTCrPkcs7SignedData_Init)
# define RTCrPkcs7SignerInfo_Clone                      RT_MANGLER(RTCrPkcs7SignerInfo_Clone)
# define RTCrPkcs7SignerInfo_Init                       RT_MANGLER(RTCrPkcs7SignerInfo_Init)
# define RTCrPkcs7SignerInfos_Clone                     RT_MANGLER(RTCrPkcs7SignerInfos_Clone)
# define RTCrPkcs7SignerInfos_Init                      RT_MANGLER(RTCrPkcs7SignerInfos_Init)
# define RTCrPkcs7Attribute_CheckSanity                 RT_MANGLER(RTCrPkcs7Attribute_CheckSanity)
# define RTCrPkcs7Attributes_CheckSanity                RT_MANGLER(RTCrPkcs7Attributes_CheckSanity)
# define RTCrPkcs7ContentInfo_CheckSanity               RT_MANGLER(RTCrPkcs7ContentInfo_CheckSanity)
# define RTCrPkcs7DigestInfo_CheckSanity                RT_MANGLER(RTCrPkcs7DigestInfo_CheckSanity)
# define RTCrPkcs7IssuerAndSerialNumber_CheckSanity     RT_MANGLER(RTCrPkcs7IssuerAndSerialNumber_CheckSanity)
# define RTCrPkcs7SignedData_CheckSanity                RT_MANGLER(RTCrPkcs7SignedData_CheckSanity)
# define RTCrPkcs7SignerInfo_CheckSanity                RT_MANGLER(RTCrPkcs7SignerInfo_CheckSanity)
# define RTCrPkcs7SignerInfos_CheckSanity               RT_MANGLER(RTCrPkcs7SignerInfos_CheckSanity)
# define RTCrPkcs7SimpleSignSignedData                  RT_MANGLER(RTCrPkcs7SimpleSignSignedData)
# define RTCrPkcs7VerifyCertCallbackCodeSigning         RT_MANGLER(RTCrPkcs7VerifyCertCallbackCodeSigning)
# define RTCrPkcs7VerifyCertCallbackDefault             RT_MANGLER(RTCrPkcs7VerifyCertCallbackDefault)
# define RTCrPkcs7VerifySignedData                      RT_MANGLER(RTCrPkcs7VerifySignedData)
# define RTCrPkcs7VerifySignedDataWithExternalData      RT_MANGLER(RTCrPkcs7VerifySignedDataWithExternalData)
# define RTCrPkcs7Cert_SetX509Cert                      RT_MANGLER(RTCrPkcs7Cert_SetX509Cert)
# define RTCrPkcs7Cert_SetExtendedCert                  RT_MANGLER(RTCrPkcs7Cert_SetExtendedCert)
# define RTCrPkcs7Cert_SetAcV1                          RT_MANGLER(RTCrPkcs7Cert_SetAcV1)
# define RTCrPkcs7Cert_SetAcV2                          RT_MANGLER(RTCrPkcs7Cert_SetAcV2)
# define RTCrPkcs7Cert_SetOtherCert                     RT_MANGLER(RTCrPkcs7Cert_SetOtherCert)
# define RTCrPkcs7Cert_CheckSanity                      RT_MANGLER(RTCrPkcs7Cert_CheckSanity)
# define RTCrPkcs7Cert_Clone                            RT_MANGLER(RTCrPkcs7Cert_Clone)
# define RTCrPkcs7Cert_Compare                          RT_MANGLER(RTCrPkcs7Cert_Compare)
# define RTCrPkcs7Cert_DecodeAsn1                       RT_MANGLER(RTCrPkcs7Cert_DecodeAsn1)
# define RTCrPkcs7Cert_Delete                           RT_MANGLER(RTCrPkcs7Cert_Delete)
# define RTCrPkcs7Cert_Enum                             RT_MANGLER(RTCrPkcs7Cert_Enum)
# define RTCrPkcs7Cert_Init                             RT_MANGLER(RTCrPkcs7Cert_Init)
# define RTCrPkcs7SetOfCerts_CheckSanity                RT_MANGLER(RTCrPkcs7SetOfCerts_CheckSanity)
# define RTCrPkcs7SetOfCerts_Clone                      RT_MANGLER(RTCrPkcs7SetOfCerts_Clone)
# define RTCrPkcs7SetOfCerts_Compare                    RT_MANGLER(RTCrPkcs7SetOfCerts_Compare)
# define RTCrPkcs7SetOfCerts_DecodeAsn1                 RT_MANGLER(RTCrPkcs7SetOfCerts_DecodeAsn1)
# define RTCrPkcs7SetOfCerts_Delete                     RT_MANGLER(RTCrPkcs7SetOfCerts_Delete)
# define RTCrPkcs7SetOfCerts_Enum                       RT_MANGLER(RTCrPkcs7SetOfCerts_Enum)
# define RTCrPkcs7SetOfCerts_Init                       RT_MANGLER(RTCrPkcs7SetOfCerts_Init)
# define RTCrPkcs7SetOfCerts_FindX509ByIssuerAndSerialNumber RT_MANGLER(RTCrPkcs7SetOfCerts_FindX509ByIssuerAndSerialNumber)
# define RTCrPkcs7SetOfContentInfos_CheckSanity         RT_MANGLER(RTCrPkcs7SetOfContentInfos_CheckSanity)
# define RTCrPkcs7SetOfContentInfos_Clone               RT_MANGLER(RTCrPkcs7SetOfContentInfos_Clone)
# define RTCrPkcs7SetOfContentInfos_Compare             RT_MANGLER(RTCrPkcs7SetOfContentInfos_Compare)
# define RTCrPkcs7SetOfContentInfos_DecodeAsn1          RT_MANGLER(RTCrPkcs7SetOfContentInfos_DecodeAsn1)
# define RTCrPkcs7SetOfContentInfos_Delete              RT_MANGLER(RTCrPkcs7SetOfContentInfos_Delete)
# define RTCrPkcs7SetOfContentInfos_Enum                RT_MANGLER(RTCrPkcs7SetOfContentInfos_Enum)
# define RTCrPkcs7SetOfContentInfos_Init                RT_MANGLER(RTCrPkcs7SetOfContentInfos_Init)
# define RTCrPkcs7SetOfSignedData_CheckSanity           RT_MANGLER(RTCrPkcs7SetOfSignedData_CheckSanity)
# define RTCrPkcs7SetOfSignedData_Clone                 RT_MANGLER(RTCrPkcs7SetOfSignedData_Clone)
# define RTCrPkcs7SetOfSignedData_Compare               RT_MANGLER(RTCrPkcs7SetOfSignedData_Compare)
# define RTCrPkcs7SetOfSignedData_DecodeAsn1            RT_MANGLER(RTCrPkcs7SetOfSignedData_DecodeAsn1)
# define RTCrPkcs7SetOfSignedData_Delete                RT_MANGLER(RTCrPkcs7SetOfSignedData_Delete)
# define RTCrPkcs7SetOfSignedData_Enum                  RT_MANGLER(RTCrPkcs7SetOfSignedData_Enum)
# define RTCrPkcs7SetOfSignedData_Init                  RT_MANGLER(RTCrPkcs7SetOfSignedData_Init)
# define RTCrPkcs8PrivateKeyInfo_CheckSanity            RT_MANGLER(RTCrPkcs8PrivateKeyInfo_CheckSanity)
# define RTCrPkcs8PrivateKeyInfo_Clone                  RT_MANGLER(RTCrPkcs8PrivateKeyInfo_Clone)
# define RTCrPkcs8PrivateKeyInfo_Compare                RT_MANGLER(RTCrPkcs8PrivateKeyInfo_Compare)
# define RTCrPkcs8PrivateKeyInfo_DecodeAsn1             RT_MANGLER(RTCrPkcs8PrivateKeyInfo_DecodeAsn1)
# define RTCrPkcs8PrivateKeyInfo_Delete                 RT_MANGLER(RTCrPkcs8PrivateKeyInfo_Delete)
# define RTCrPkcs8PrivateKeyInfo_Enum                   RT_MANGLER(RTCrPkcs8PrivateKeyInfo_Enum)
# define RTCrPkcs8PrivateKeyInfo_Init                   RT_MANGLER(RTCrPkcs8PrivateKeyInfo_Init)
# define RTCrPkcs8EncryptedPrivateKeyInfo_CheckSanity   RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_CheckSanity)
# define RTCrPkcs8EncryptedPrivateKeyInfo_Clone         RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_Clone)
# define RTCrPkcs8EncryptedPrivateKeyInfo_Compare       RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_Compare)
# define RTCrPkcs8EncryptedPrivateKeyInfo_DecodeAsn1    RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_DecodeAsn1)
# define RTCrPkcs8EncryptedPrivateKeyInfo_Delete        RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_Delete)
# define RTCrPkcs8EncryptedPrivateKeyInfo_Enum          RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_Enum)
# define RTCrPkcs8EncryptedPrivateKeyInfo_Init          RT_MANGLER(RTCrPkcs8EncryptedPrivateKeyInfo_Init)
# define RTCrPkixSignatureCreateByObjId                 RT_MANGLER(RTCrPkixSignatureCreateByObjId)
# define RTCrPkixSignatureCreateByObjIdString           RT_MANGLER(RTCrPkixSignatureCreateByObjIdString)
# define RTCrPkixSignatureCreate                        RT_MANGLER(RTCrPkixSignatureCreate)
# define RTCrPkixSignatureFindByObjId                   RT_MANGLER(RTCrPkixSignatureFindByObjId)
# define RTCrPkixSignatureFindByObjIdString             RT_MANGLER(RTCrPkixSignatureFindByObjIdString)
# define RTCrPkixSignatureRelease                       RT_MANGLER(RTCrPkixSignatureRelease)
# define RTCrPkixSignatureRetain                        RT_MANGLER(RTCrPkixSignatureRetain)
# define RTCrPkixSignatureSign                          RT_MANGLER(RTCrPkixSignatureSign)
# define RTCrPkixSignatureVerify                        RT_MANGLER(RTCrPkixSignatureVerify)
# define RTCrPkixSignatureVerifyBitString               RT_MANGLER(RTCrPkixSignatureVerifyBitString)
# define RTCrPkixSignatureVerifyOctetString             RT_MANGLER(RTCrPkixSignatureVerifyOctetString)
# define RTCrPkixGetCiperOidFromSignatureAlgorithm      RT_MANGLER(RTCrPkixGetCiperOidFromSignatureAlgorithm)
# define RTCrPkixGetCiperOidFromSignatureAlgorithmOid   RT_MANGLER(RTCrPkixGetCiperOidFromSignatureAlgorithmOid)
# define RTCrPkixPubKeySignDigest                       RT_MANGLER(RTCrPkixPubKeySignDigest)
# define RTCrPkixPubKeyVerifySignature                  RT_MANGLER(RTCrPkixPubKeyVerifySignature)
# define RTCrPkixPubKeyVerifySignedDigest               RT_MANGLER(RTCrPkixPubKeyVerifySignedDigest)
# define RTCrPkixPubKeyVerifySignedDigestByCertPubKeyInfo RT_MANGLER(RTCrPkixPubKeyVerifySignedDigestByCertPubKeyInfo)
# define RTCrPkixPubKeyCanHandleDigestType              RT_MANGLER(RTCrPkixPubKeyCanHandleDigestType)
# define RTCrPkixCanCertHandleDigestType                RT_MANGLER(RTCrPkixCanCertHandleDigestType)
# define RTCrRandBytes                                  RT_MANGLER(RTCrRandBytes)
# define RTCrShaCryptGenerateSalt                       RT_MANGLER(RTCrShaCryptGenerateSalt)
# define RTCrShaCrypt256                                RT_MANGLER(RTCrShaCrypt256)
# define RTCrShaCrypt256Ex                              RT_MANGLER(RTCrShaCrypt256Ex)
# define RTCrShaCrypt256ToString                        RT_MANGLER(RTCrShaCrypt256ToString)
# define RTCrShaCrypt512                                RT_MANGLER(RTCrShaCrypt512)
# define RTCrShaCrypt512Ex                              RT_MANGLER(RTCrShaCrypt512Ex)
# define RTCrShaCrypt512ToString                        RT_MANGLER(RTCrShaCrypt512ToString)
# define RTCrSpcAttributeTypeAndOptionalValue_SetPeImage RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_SetPeImage)
# define RTCrSpcAttributeTypeAndOptionalValue_DecodeAsn1 RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_DecodeAsn1)
# define RTCrSpcIndirectDataContent_DecodeAsn1          RT_MANGLER(RTCrSpcIndirectDataContent_DecodeAsn1)
# define RTCrSpcLink_DecodeAsn1                         RT_MANGLER(RTCrSpcLink_DecodeAsn1)
# define RTCrSpcPeImageData_SetFile                     RT_MANGLER(RTCrSpcPeImageData_SetFile)
# define RTCrSpcPeImageData_SetFlags                    RT_MANGLER(RTCrSpcPeImageData_SetFlags)
# define RTCrSpcPeImageData_DecodeAsn1                  RT_MANGLER(RTCrSpcPeImageData_DecodeAsn1)
# define RTCrSpcSerializedObjectAttribute_SetV1Hashes   RT_MANGLER(RTCrSpcSerializedObjectAttribute_SetV1Hashes)
# define RTCrSpcSerializedObjectAttribute_SetV2Hashes   RT_MANGLER(RTCrSpcSerializedObjectAttribute_SetV2Hashes)
# define RTCrSpcSerializedObjectAttribute_DecodeAsn1    RT_MANGLER(RTCrSpcSerializedObjectAttribute_DecodeAsn1)
# define RTCrSpcSerializedObjectAttributes_DecodeAsn1   RT_MANGLER(RTCrSpcSerializedObjectAttributes_DecodeAsn1)
# define RTCrSpcSerializedObject_DecodeAsn1             RT_MANGLER(RTCrSpcSerializedObject_DecodeAsn1)
# define RTCrSpcSerializedPageHashes_DecodeAsn1         RT_MANGLER(RTCrSpcSerializedPageHashes_DecodeAsn1)
# define RTCrSpcString_DecodeAsn1                       RT_MANGLER(RTCrSpcString_DecodeAsn1)
# define RTCrSpcAttributeTypeAndOptionalValue_Compare   RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_Compare)
# define RTCrSpcAttributeTypeAndOptionalValue_Delete    RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_Delete)
# define RTCrSpcAttributeTypeAndOptionalValue_Enum      RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_Enum)
# define RTCrSpcIndirectDataContent_Compare             RT_MANGLER(RTCrSpcIndirectDataContent_Compare)
# define RTCrSpcIndirectDataContent_Delete              RT_MANGLER(RTCrSpcIndirectDataContent_Delete)
# define RTCrSpcIndirectDataContent_Enum                RT_MANGLER(RTCrSpcIndirectDataContent_Enum)
# define RTCrSpcIndirectDataContent_GetPeImageObjAttrib RT_MANGLER(RTCrSpcIndirectDataContent_GetPeImageObjAttrib)
# define RTCrSpcLink_SetFile                            RT_MANGLER(RTCrSpcLink_SetFile)
# define RTCrSpcLink_SetMoniker                         RT_MANGLER(RTCrSpcLink_SetMoniker)
# define RTCrSpcLink_SetUrl                             RT_MANGLER(RTCrSpcLink_SetUrl)
# define RTCrSpcLink_Compare                            RT_MANGLER(RTCrSpcLink_Compare)
# define RTCrSpcLink_Delete                             RT_MANGLER(RTCrSpcLink_Delete)
# define RTCrSpcLink_Enum                               RT_MANGLER(RTCrSpcLink_Enum)
# define RTCrSpcPeImageData_Compare                     RT_MANGLER(RTCrSpcPeImageData_Compare)
# define RTCrSpcPeImageData_Delete                      RT_MANGLER(RTCrSpcPeImageData_Delete)
# define RTCrSpcPeImageData_Enum                        RT_MANGLER(RTCrSpcPeImageData_Enum)
# define RTCrSpcSerializedObjectAttribute_Compare       RT_MANGLER(RTCrSpcSerializedObjectAttribute_Compare)
# define RTCrSpcSerializedObjectAttribute_Delete        RT_MANGLER(RTCrSpcSerializedObjectAttribute_Delete)
# define RTCrSpcSerializedObjectAttribute_Enum          RT_MANGLER(RTCrSpcSerializedObjectAttribute_Enum)
# define RTCrSpcSerializedObjectAttributes_Compare      RT_MANGLER(RTCrSpcSerializedObjectAttributes_Compare)
# define RTCrSpcSerializedObjectAttributes_Delete       RT_MANGLER(RTCrSpcSerializedObjectAttributes_Delete)
# define RTCrSpcSerializedObjectAttributes_Enum         RT_MANGLER(RTCrSpcSerializedObjectAttributes_Enum)
# define RTCrSpcSerializedObject_Compare                RT_MANGLER(RTCrSpcSerializedObject_Compare)
# define RTCrSpcSerializedObject_Delete                 RT_MANGLER(RTCrSpcSerializedObject_Delete)
# define RTCrSpcSerializedObject_Enum                   RT_MANGLER(RTCrSpcSerializedObject_Enum)
# define RTCrSpcSerializedPageHashes_Compare            RT_MANGLER(RTCrSpcSerializedPageHashes_Compare)
# define RTCrSpcSerializedPageHashes_Delete             RT_MANGLER(RTCrSpcSerializedPageHashes_Delete)
# define RTCrSpcSerializedPageHashes_Enum               RT_MANGLER(RTCrSpcSerializedPageHashes_Enum)
# define RTCrSpcSerializedPageHashes_UpdateDerivedData  RT_MANGLER(RTCrSpcSerializedPageHashes_UpdateDerivedData)
# define RTCrSpcString_Compare                          RT_MANGLER(RTCrSpcString_Compare)
# define RTCrSpcString_Delete                           RT_MANGLER(RTCrSpcString_Delete)
# define RTCrSpcString_Enum                             RT_MANGLER(RTCrSpcString_Enum)
# define RTCrSpcAttributeTypeAndOptionalValue_Clone     RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_Clone)
# define RTCrSpcAttributeTypeAndOptionalValue_Init      RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_Init)
# define RTCrSpcIndirectDataContent_Clone               RT_MANGLER(RTCrSpcIndirectDataContent_Clone)
# define RTCrSpcIndirectDataContent_Init                RT_MANGLER(RTCrSpcIndirectDataContent_Init)
# define RTCrSpcString_SetAscii                         RT_MANGLER(RTCrSpcString_SetAscii)
# define RTCrSpcString_SetUcs2                          RT_MANGLER(RTCrSpcString_SetUcs2)
# define RTCrSpcLink_Clone                              RT_MANGLER(RTCrSpcLink_Clone)
# define RTCrSpcLink_Init                               RT_MANGLER(RTCrSpcLink_Init)
# define RTCrSpcPeImageData_Clone                       RT_MANGLER(RTCrSpcPeImageData_Clone)
# define RTCrSpcPeImageData_Init                        RT_MANGLER(RTCrSpcPeImageData_Init)
# define RTCrSpcSerializedObjectAttribute_Clone         RT_MANGLER(RTCrSpcSerializedObjectAttribute_Clone)
# define RTCrSpcSerializedObjectAttribute_Init          RT_MANGLER(RTCrSpcSerializedObjectAttribute_Init)
# define RTCrSpcSerializedObjectAttributes_Clone        RT_MANGLER(RTCrSpcSerializedObjectAttributes_Clone)
# define RTCrSpcSerializedObjectAttributes_Init         RT_MANGLER(RTCrSpcSerializedObjectAttributes_Init)
# define RTCrSpcSerializedObject_Clone                  RT_MANGLER(RTCrSpcSerializedObject_Clone)
# define RTCrSpcSerializedObject_Init                   RT_MANGLER(RTCrSpcSerializedObject_Init)
# define RTCrSpcSerializedPageHashes_Clone              RT_MANGLER(RTCrSpcSerializedPageHashes_Clone)
# define RTCrSpcSerializedPageHashes_Init               RT_MANGLER(RTCrSpcSerializedPageHashes_Init)
# define RTCrSpcString_Clone                            RT_MANGLER(RTCrSpcString_Clone)
# define RTCrSpcString_Init                             RT_MANGLER(RTCrSpcString_Init)
# define RTCrSpcAttributeTypeAndOptionalValue_CheckSanity RT_MANGLER(RTCrSpcAttributeTypeAndOptionalValue_CheckSanity)
# define RTCrSpcIndirectDataContent_CheckSanity         RT_MANGLER(RTCrSpcIndirectDataContent_CheckSanity)
# define RTCrSpcIndirectDataContent_CheckSanityEx       RT_MANGLER(RTCrSpcIndirectDataContent_CheckSanityEx)
# define RTCrSpcLink_CheckSanity                        RT_MANGLER(RTCrSpcLink_CheckSanity)
# define RTCrSpcPeImageData_CheckSanity                 RT_MANGLER(RTCrSpcPeImageData_CheckSanity)
# define RTCrSpcSerializedObjectAttribute_CheckSanity   RT_MANGLER(RTCrSpcSerializedObjectAttribute_CheckSanity)
# define RTCrSpcSerializedObjectAttributes_CheckSanity  RT_MANGLER(RTCrSpcSerializedObjectAttributes_CheckSanity)
# define RTCrSpcSerializedObject_CheckSanity            RT_MANGLER(RTCrSpcSerializedObject_CheckSanity)
# define RTCrSpcSerializedPageHashes_CheckSanity        RT_MANGLER(RTCrSpcSerializedPageHashes_CheckSanity)
# define RTCrSpcString_CheckSanity                      RT_MANGLER(RTCrSpcString_CheckSanity)
# define RTCrSslCreate                                  RT_MANGLER(RTCrSslCreate)
# define RTCrSslCreateSessionForNativeSocket            RT_MANGLER(RTCrSslCreateSessionForNativeSocket)
# define RTCrSslLoadTrustedRootCerts                    RT_MANGLER(RTCrSslLoadTrustedRootCerts)
# define RTCrSslRelease                                 RT_MANGLER(RTCrSslRelease)
# define RTCrSslRetain                                  RT_MANGLER(RTCrSslRetain)
# define RTCrSslSessionAccept                           RT_MANGLER(RTCrSslSessionAccept)
# define RTCrSslSessionConnect                          RT_MANGLER(RTCrSslSessionConnect)
# define RTCrSslSessionGetCertIssuerNameAsString        RT_MANGLER(RTCrSslSessionGetCertIssuerNameAsString)
# define RTCrSslSessionGetVersion                       RT_MANGLER(RTCrSslSessionGetVersion)
# define RTCrSslSessionPending                          RT_MANGLER(RTCrSslSessionPending)
# define RTCrSslSessionRead                             RT_MANGLER(RTCrSslSessionRead)
# define RTCrSslSessionRelease                          RT_MANGLER(RTCrSslSessionRelease)
# define RTCrSslSessionRetain                           RT_MANGLER(RTCrSslSessionRetain)
# define RTCrSslSessionWrite                            RT_MANGLER(RTCrSslSessionWrite)
# define RTCrSslSetCertificateFile                      RT_MANGLER(RTCrSslSetCertificateFile)
# define RTCrSslSetNoPeerVerify                         RT_MANGLER(RTCrSslSetNoPeerVerify)
# define RTCrSslSetPrivateKeyFile                       RT_MANGLER(RTCrSslSetPrivateKeyFile)
# define RTCrX509AlgorithmIdentifier_DecodeAsn1         RT_MANGLER(RTCrX509AlgorithmIdentifier_DecodeAsn1)
# define RTCrX509AlgorithmIdentifiers_DecodeAsn1        RT_MANGLER(RTCrX509AlgorithmIdentifiers_DecodeAsn1)
# define RTCrX509AttributeTypeAndValue_DecodeAsn1       RT_MANGLER(RTCrX509AttributeTypeAndValue_DecodeAsn1)
# define RTCrX509AttributeTypeAndValues_DecodeAsn1      RT_MANGLER(RTCrX509AttributeTypeAndValues_DecodeAsn1)
# define RTCrX509AuthorityKeyIdentifier_DecodeAsn1      RT_MANGLER(RTCrX509AuthorityKeyIdentifier_DecodeAsn1)
# define RTCrX509BasicConstraints_DecodeAsn1            RT_MANGLER(RTCrX509BasicConstraints_DecodeAsn1)
# define RTCrX509CertificatePolicies_DecodeAsn1         RT_MANGLER(RTCrX509CertificatePolicies_DecodeAsn1)
# define RTCrX509Certificate_DecodeAsn1                 RT_MANGLER(RTCrX509Certificate_DecodeAsn1)
# define RTCrX509Certificates_DecodeAsn1                RT_MANGLER(RTCrX509Certificates_DecodeAsn1)
# define RTCrX509Extension_DecodeAsn1                   RT_MANGLER(RTCrX509Extension_DecodeAsn1)
# define RTCrX509Extension_ExtnValue_DecodeAsn1         RT_MANGLER(RTCrX509Extension_ExtnValue_DecodeAsn1)
# define RTCrX509Extensions_DecodeAsn1                  RT_MANGLER(RTCrX509Extensions_DecodeAsn1)
# define RTCrX509GeneralName_DecodeAsn1                 RT_MANGLER(RTCrX509GeneralName_DecodeAsn1)
# define RTCrX509GeneralNames_DecodeAsn1                RT_MANGLER(RTCrX509GeneralNames_DecodeAsn1)
# define RTCrX509GeneralSubtree_DecodeAsn1              RT_MANGLER(RTCrX509GeneralSubtree_DecodeAsn1)
# define RTCrX509GeneralSubtrees_DecodeAsn1             RT_MANGLER(RTCrX509GeneralSubtrees_DecodeAsn1)
# define RTCrX509NameConstraints_DecodeAsn1             RT_MANGLER(RTCrX509NameConstraints_DecodeAsn1)
# define RTCrX509Name_DecodeAsn1                        RT_MANGLER(RTCrX509Name_DecodeAsn1)
# define RTCrX509OldAuthorityKeyIdentifier_DecodeAsn1   RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_DecodeAsn1)
# define RTCrX509OtherName_DecodeAsn1                   RT_MANGLER(RTCrX509OtherName_DecodeAsn1)
# define RTCrX509PolicyConstraints_DecodeAsn1           RT_MANGLER(RTCrX509PolicyConstraints_DecodeAsn1)
# define RTCrX509PolicyInformation_DecodeAsn1           RT_MANGLER(RTCrX509PolicyInformation_DecodeAsn1)
# define RTCrX509PolicyMapping_DecodeAsn1               RT_MANGLER(RTCrX509PolicyMapping_DecodeAsn1)
# define RTCrX509PolicyMappings_DecodeAsn1              RT_MANGLER(RTCrX509PolicyMappings_DecodeAsn1)
# define RTCrX509PolicyQualifierInfo_DecodeAsn1         RT_MANGLER(RTCrX509PolicyQualifierInfo_DecodeAsn1)
# define RTCrX509PolicyQualifierInfos_DecodeAsn1        RT_MANGLER(RTCrX509PolicyQualifierInfos_DecodeAsn1)
# define RTCrX509SubjectPublicKeyInfo_DecodeAsn1        RT_MANGLER(RTCrX509SubjectPublicKeyInfo_DecodeAsn1)
# define RTCrX509TbsCertificate_DecodeAsn1              RT_MANGLER(RTCrX509TbsCertificate_DecodeAsn1)
# define RTCrX509Validity_DecodeAsn1                    RT_MANGLER(RTCrX509Validity_DecodeAsn1)
# define RTCrX509CertPathsBuild                         RT_MANGLER(RTCrX509CertPathsBuild)
# define RTCrX509CertPathsCreate                        RT_MANGLER(RTCrX509CertPathsCreate)
# define RTCrX509CertPathsCreateEx                      RT_MANGLER(RTCrX509CertPathsCreateEx)
# define RTCrX509CertPathsDumpAll                       RT_MANGLER(RTCrX509CertPathsDumpAll)
# define RTCrX509CertPathsDumpOne                       RT_MANGLER(RTCrX509CertPathsDumpOne)
# define RTCrX509CertPathsGetPathCount                  RT_MANGLER(RTCrX509CertPathsGetPathCount)
# define RTCrX509CertPathsGetPathLength                 RT_MANGLER(RTCrX509CertPathsGetPathLength)
# define RTCrX509CertPathsGetPathNodeCert               RT_MANGLER(RTCrX509CertPathsGetPathNodeCert)
# define RTCrX509CertPathsGetPathVerifyResult           RT_MANGLER(RTCrX509CertPathsGetPathVerifyResult)
# define RTCrX509CertPathsQueryPathInfo                 RT_MANGLER(RTCrX509CertPathsQueryPathInfo)
# define RTCrX509CertPathsRelease                       RT_MANGLER(RTCrX509CertPathsRelease)
# define RTCrX509CertPathsRetain                        RT_MANGLER(RTCrX509CertPathsRetain)
# define RTCrX509CertPathsSetTrustedStore               RT_MANGLER(RTCrX509CertPathsSetTrustedStore)
# define RTCrX509CertPathsSetTrustAnchorChecks          RT_MANGLER(RTCrX509CertPathsSetTrustAnchorChecks)
# define RTCrX509CertPathsSetUntrustedArray             RT_MANGLER(RTCrX509CertPathsSetUntrustedArray)
# define RTCrX509CertPathsSetUntrustedSet               RT_MANGLER(RTCrX509CertPathsSetUntrustedSet)
# define RTCrX509CertPathsSetUntrustedStore             RT_MANGLER(RTCrX509CertPathsSetUntrustedStore)
# define RTCrX509CertPathsSetValidTime                  RT_MANGLER(RTCrX509CertPathsSetValidTime)
# define RTCrX509CertPathsSetValidTimeSpec              RT_MANGLER(RTCrX509CertPathsSetValidTimeSpec)
# define RTCrX509CertPathsValidateAll                   RT_MANGLER(RTCrX509CertPathsValidateAll)
# define RTCrX509CertPathsValidateOne                   RT_MANGLER(RTCrX509CertPathsValidateOne)
# define RTCrX509AlgorithmIdentifier_CombineEncryptionAndDigest             RT_MANGLER(RTCrX509AlgorithmIdentifier_CombineEncryptionAndDigest)
# define RTCrX509AlgorithmIdentifier_CombineEncryptionOidAndDigestOid       RT_MANGLER(RTCrX509AlgorithmIdentifier_CombineEncryptionOidAndDigestOid)
# define RTCrX509AlgorithmIdentifier_Compare                                RT_MANGLER(RTCrX509AlgorithmIdentifier_Compare)
# define RTCrX509AlgorithmIdentifier_CompareDigestAndEncryptedDigest        RT_MANGLER(RTCrX509AlgorithmIdentifier_CompareDigestAndEncryptedDigest)
# define RTCrX509AlgorithmIdentifier_CompareDigestOidAndEncryptedDigestOid  RT_MANGLER(RTCrX509AlgorithmIdentifier_CompareDigestOidAndEncryptedDigestOid)
# define RTCrX509AlgorithmIdentifier_CompareWithString  RT_MANGLER(RTCrX509AlgorithmIdentifier_CompareWithString)
# define RTCrX509AlgorithmIdentifier_Delete             RT_MANGLER(RTCrX509AlgorithmIdentifier_Delete)
# define RTCrX509AlgorithmIdentifier_Enum               RT_MANGLER(RTCrX509AlgorithmIdentifier_Enum)
# define RTCrX509AlgorithmIdentifier_GetDigestSize      RT_MANGLER(RTCrX509AlgorithmIdentifier_GetDigestSize)
# define RTCrX509AlgorithmIdentifier_GetDigestType      RT_MANGLER(RTCrX509AlgorithmIdentifier_GetDigestType)
# define RTCrX509AlgorithmIdentifier_GetEncryptionOid   RT_MANGLER(RTCrX509AlgorithmIdentifier_GetEncryptionOid)
# define RTCrX509AlgorithmIdentifier_GetEncryptionOidFromOid RT_MANGLER(RTCrX509AlgorithmIdentifier_GetEncryptionOidFromOid)
# define RTCrX509AlgorithmIdentifiers_Compare           RT_MANGLER(RTCrX509AlgorithmIdentifiers_Compare)
# define RTCrX509AlgorithmIdentifiers_Delete            RT_MANGLER(RTCrX509AlgorithmIdentifiers_Delete)
# define RTCrX509AlgorithmIdentifiers_Enum              RT_MANGLER(RTCrX509AlgorithmIdentifiers_Enum)
# define RTCrX509AttributeTypeAndValue_Compare          RT_MANGLER(RTCrX509AttributeTypeAndValue_Compare)
# define RTCrX509AttributeTypeAndValue_Delete           RT_MANGLER(RTCrX509AttributeTypeAndValue_Delete)
# define RTCrX509AttributeTypeAndValue_Enum             RT_MANGLER(RTCrX509AttributeTypeAndValue_Enum)
# define RTCrX509AttributeTypeAndValues_Compare         RT_MANGLER(RTCrX509AttributeTypeAndValues_Compare)
# define RTCrX509AttributeTypeAndValues_Delete          RT_MANGLER(RTCrX509AttributeTypeAndValues_Delete)
# define RTCrX509AttributeTypeAndValues_Enum            RT_MANGLER(RTCrX509AttributeTypeAndValues_Enum)
# define RTCrX509AuthorityKeyIdentifier_Compare         RT_MANGLER(RTCrX509AuthorityKeyIdentifier_Compare)
# define RTCrX509AuthorityKeyIdentifier_Delete          RT_MANGLER(RTCrX509AuthorityKeyIdentifier_Delete)
# define RTCrX509AuthorityKeyIdentifier_Enum            RT_MANGLER(RTCrX509AuthorityKeyIdentifier_Enum)
# define RTCrX509BasicConstraints_Compare               RT_MANGLER(RTCrX509BasicConstraints_Compare)
# define RTCrX509BasicConstraints_Delete                RT_MANGLER(RTCrX509BasicConstraints_Delete)
# define RTCrX509BasicConstraints_Enum                  RT_MANGLER(RTCrX509BasicConstraints_Enum)
# define RTCrX509CertificatePolicies_Compare            RT_MANGLER(RTCrX509CertificatePolicies_Compare)
# define RTCrX509CertificatePolicies_Delete             RT_MANGLER(RTCrX509CertificatePolicies_Delete)
# define RTCrX509CertificatePolicies_Enum               RT_MANGLER(RTCrX509CertificatePolicies_Enum)
# define RTCrX509Certificate_Compare                    RT_MANGLER(RTCrX509Certificate_Compare)
# define RTCrX509Certificate_Delete                     RT_MANGLER(RTCrX509Certificate_Delete)
# define RTCrX509Certificate_Enum                       RT_MANGLER(RTCrX509Certificate_Enum)
# define RTCrX509Certificate_IsSelfSigned               RT_MANGLER(RTCrX509Certificate_IsSelfSigned)
# define RTCrX509Certificate_MatchIssuerAndSerialNumber RT_MANGLER(RTCrX509Certificate_MatchIssuerAndSerialNumber)
# define RTCrX509Certificate_MatchSubjectOrAltSubjectByRfc5280 RT_MANGLER(RTCrX509Certificate_MatchSubjectOrAltSubjectByRfc5280)
# define RTCrX509Certificates_Compare                   RT_MANGLER(RTCrX509Certificates_Compare)
# define RTCrX509Certificates_Delete                    RT_MANGLER(RTCrX509Certificates_Delete)
# define RTCrX509Certificates_Enum                      RT_MANGLER(RTCrX509Certificates_Enum)
# define RTCrX509Certificates_FindByIssuerAndSerialNumber RT_MANGLER(RTCrX509Certificates_FindByIssuerAndSerialNumber)
# define RTCrX509Extension_Compare                      RT_MANGLER(RTCrX509Extension_Compare)
# define RTCrX509Extension_Delete                       RT_MANGLER(RTCrX509Extension_Delete)
# define RTCrX509Extension_Enum                         RT_MANGLER(RTCrX509Extension_Enum)
# define RTCrX509Extensions_Compare                     RT_MANGLER(RTCrX509Extensions_Compare)
# define RTCrX509Extensions_Delete                      RT_MANGLER(RTCrX509Extensions_Delete)
# define RTCrX509Extensions_Enum                        RT_MANGLER(RTCrX509Extensions_Enum)
# define RTCrX509GeneralName_Compare                    RT_MANGLER(RTCrX509GeneralName_Compare)
# define RTCrX509GeneralName_ConstraintMatch            RT_MANGLER(RTCrX509GeneralName_ConstraintMatch)
# define RTCrX509GeneralName_Delete                     RT_MANGLER(RTCrX509GeneralName_Delete)
# define RTCrX509GeneralName_Enum                       RT_MANGLER(RTCrX509GeneralName_Enum)
# define RTCrX509GeneralNames_Compare                   RT_MANGLER(RTCrX509GeneralNames_Compare)
# define RTCrX509GeneralNames_Delete                    RT_MANGLER(RTCrX509GeneralNames_Delete)
# define RTCrX509GeneralNames_Enum                      RT_MANGLER(RTCrX509GeneralNames_Enum)
# define RTCrX509GeneralSubtree_Compare                 RT_MANGLER(RTCrX509GeneralSubtree_Compare)
# define RTCrX509GeneralSubtree_ConstraintMatch         RT_MANGLER(RTCrX509GeneralSubtree_ConstraintMatch)
# define RTCrX509GeneralSubtree_Delete                  RT_MANGLER(RTCrX509GeneralSubtree_Delete)
# define RTCrX509GeneralSubtree_Enum                    RT_MANGLER(RTCrX509GeneralSubtree_Enum)
# define RTCrX509GeneralSubtrees_Compare                RT_MANGLER(RTCrX509GeneralSubtrees_Compare)
# define RTCrX509GeneralSubtrees_Delete                 RT_MANGLER(RTCrX509GeneralSubtrees_Delete)
# define RTCrX509GeneralSubtrees_Enum                   RT_MANGLER(RTCrX509GeneralSubtrees_Enum)
# define RTCrX509NameConstraints_Compare                RT_MANGLER(RTCrX509NameConstraints_Compare)
# define RTCrX509NameConstraints_Delete                 RT_MANGLER(RTCrX509NameConstraints_Delete)
# define RTCrX509NameConstraints_Enum                   RT_MANGLER(RTCrX509NameConstraints_Enum)
# define RTCrX509Name_Compare                           RT_MANGLER(RTCrX509Name_Compare)
# define RTCrX509Name_ConstraintMatch                   RT_MANGLER(RTCrX509Name_ConstraintMatch)
# define RTCrX509Name_Delete                            RT_MANGLER(RTCrX509Name_Delete)
# define RTCrX509Name_Enum                              RT_MANGLER(RTCrX509Name_Enum)
# define RTCrX509Name_FormatAsString                    RT_MANGLER(RTCrX509Name_FormatAsString)
# define RTCrX509Name_MatchByRfc5280                    RT_MANGLER(RTCrX509Name_MatchByRfc5280)
# define RTCrX509Name_MatchWithString                   RT_MANGLER(RTCrX509Name_MatchWithString)
# define RTCrX509Name_GetShortRdn                       RT_MANGLER(RTCrX509Name_GetShortRdn)
# define RTCrX509OldAuthorityKeyIdentifier_Compare      RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_Compare)
# define RTCrX509OldAuthorityKeyIdentifier_Delete       RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_Delete)
# define RTCrX509OldAuthorityKeyIdentifier_Enum         RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_Enum)
# define RTCrX509OtherName_Compare                      RT_MANGLER(RTCrX509OtherName_Compare)
# define RTCrX509OtherName_Delete                       RT_MANGLER(RTCrX509OtherName_Delete)
# define RTCrX509OtherName_Enum                         RT_MANGLER(RTCrX509OtherName_Enum)
# define RTCrX509PolicyConstraints_Compare              RT_MANGLER(RTCrX509PolicyConstraints_Compare)
# define RTCrX509PolicyConstraints_Delete               RT_MANGLER(RTCrX509PolicyConstraints_Delete)
# define RTCrX509PolicyConstraints_Enum                 RT_MANGLER(RTCrX509PolicyConstraints_Enum)
# define RTCrX509PolicyInformation_Compare              RT_MANGLER(RTCrX509PolicyInformation_Compare)
# define RTCrX509PolicyInformation_Delete               RT_MANGLER(RTCrX509PolicyInformation_Delete)
# define RTCrX509PolicyInformation_Enum                 RT_MANGLER(RTCrX509PolicyInformation_Enum)
# define RTCrX509PolicyMapping_Compare                  RT_MANGLER(RTCrX509PolicyMapping_Compare)
# define RTCrX509PolicyMapping_Delete                   RT_MANGLER(RTCrX509PolicyMapping_Delete)
# define RTCrX509PolicyMapping_Enum                     RT_MANGLER(RTCrX509PolicyMapping_Enum)
# define RTCrX509PolicyMappings_Compare                 RT_MANGLER(RTCrX509PolicyMappings_Compare)
# define RTCrX509PolicyMappings_Delete                  RT_MANGLER(RTCrX509PolicyMappings_Delete)
# define RTCrX509PolicyMappings_Enum                    RT_MANGLER(RTCrX509PolicyMappings_Enum)
# define RTCrX509PolicyQualifierInfo_Compare            RT_MANGLER(RTCrX509PolicyQualifierInfo_Compare)
# define RTCrX509PolicyQualifierInfo_Delete             RT_MANGLER(RTCrX509PolicyQualifierInfo_Delete)
# define RTCrX509PolicyQualifierInfo_Enum               RT_MANGLER(RTCrX509PolicyQualifierInfo_Enum)
# define RTCrX509PolicyQualifierInfos_Compare           RT_MANGLER(RTCrX509PolicyQualifierInfos_Compare)
# define RTCrX509PolicyQualifierInfos_Delete            RT_MANGLER(RTCrX509PolicyQualifierInfos_Delete)
# define RTCrX509PolicyQualifierInfos_Enum              RT_MANGLER(RTCrX509PolicyQualifierInfos_Enum)
# define RTCrX509SubjectPublicKeyInfo_Compare           RT_MANGLER(RTCrX509SubjectPublicKeyInfo_Compare)
# define RTCrX509SubjectPublicKeyInfo_Delete            RT_MANGLER(RTCrX509SubjectPublicKeyInfo_Delete)
# define RTCrX509SubjectPublicKeyInfo_Enum              RT_MANGLER(RTCrX509SubjectPublicKeyInfo_Enum)
# define RTCrX509TbsCertificate_Compare                 RT_MANGLER(RTCrX509TbsCertificate_Compare)
# define RTCrX509TbsCertificate_Delete                  RT_MANGLER(RTCrX509TbsCertificate_Delete)
# define RTCrX509TbsCertificate_Enum                    RT_MANGLER(RTCrX509TbsCertificate_Enum)
# define RTCrX509TbsCertificate_ReprocessExtensions     RT_MANGLER(RTCrX509TbsCertificate_ReprocessExtensions)
# define RTCrX509Validity_Compare                       RT_MANGLER(RTCrX509Validity_Compare)
# define RTCrX509Validity_Delete                        RT_MANGLER(RTCrX509Validity_Delete)
# define RTCrX509Validity_Enum                          RT_MANGLER(RTCrX509Validity_Enum)
# define RTCrX509Validity_IsValidAtTimeSpec             RT_MANGLER(RTCrX509Validity_IsValidAtTimeSpec)
# define RTCrX509Certificate_ReadFromFile               RT_MANGLER(RTCrX509Certificate_ReadFromFile)
# define RTCrX509Certificate_ReadFromBuffer             RT_MANGLER(RTCrX509Certificate_ReadFromBuffer)
# define RTCrX509AlgorithmIdentifier_Clone              RT_MANGLER(RTCrX509AlgorithmIdentifier_Clone)
# define RTCrX509AlgorithmIdentifier_Init               RT_MANGLER(RTCrX509AlgorithmIdentifier_Init)
# define RTCrX509AlgorithmIdentifiers_Clone             RT_MANGLER(RTCrX509AlgorithmIdentifiers_Clone)
# define RTCrX509AlgorithmIdentifiers_Init              RT_MANGLER(RTCrX509AlgorithmIdentifiers_Init)
# define RTCrX509AttributeTypeAndValue_Clone            RT_MANGLER(RTCrX509AttributeTypeAndValue_Clone)
# define RTCrX509AttributeTypeAndValue_Init             RT_MANGLER(RTCrX509AttributeTypeAndValue_Init)
# define RTCrX509AttributeTypeAndValues_Clone           RT_MANGLER(RTCrX509AttributeTypeAndValues_Clone)
# define RTCrX509AttributeTypeAndValues_Init            RT_MANGLER(RTCrX509AttributeTypeAndValues_Init)
# define RTCrX509AuthorityKeyIdentifier_Clone           RT_MANGLER(RTCrX509AuthorityKeyIdentifier_Clone)
# define RTCrX509AuthorityKeyIdentifier_Init            RT_MANGLER(RTCrX509AuthorityKeyIdentifier_Init)
# define RTCrX509BasicConstraints_Clone                 RT_MANGLER(RTCrX509BasicConstraints_Clone)
# define RTCrX509BasicConstraints_Init                  RT_MANGLER(RTCrX509BasicConstraints_Init)
# define RTCrX509CertificatePolicies_Clone              RT_MANGLER(RTCrX509CertificatePolicies_Clone)
# define RTCrX509CertificatePolicies_Init               RT_MANGLER(RTCrX509CertificatePolicies_Init)
# define RTCrX509Certificate_Clone                      RT_MANGLER(RTCrX509Certificate_Clone)
# define RTCrX509Certificate_Init                       RT_MANGLER(RTCrX509Certificate_Init)
# define RTCrX509Certificates_Clone                     RT_MANGLER(RTCrX509Certificates_Clone)
# define RTCrX509Certificates_Init                      RT_MANGLER(RTCrX509Certificates_Init)
# define RTCrX509Extension_Clone                        RT_MANGLER(RTCrX509Extension_Clone)
# define RTCrX509Extension_Init                         RT_MANGLER(RTCrX509Extension_Init)
# define RTCrX509Extensions_Clone                       RT_MANGLER(RTCrX509Extensions_Clone)
# define RTCrX509Extensions_Init                        RT_MANGLER(RTCrX509Extensions_Init)
# define RTCrX509GeneralName_Clone                      RT_MANGLER(RTCrX509GeneralName_Clone)
# define RTCrX509GeneralName_Init                       RT_MANGLER(RTCrX509GeneralName_Init)
# define RTCrX509GeneralNames_Clone                     RT_MANGLER(RTCrX509GeneralNames_Clone)
# define RTCrX509GeneralNames_Init                      RT_MANGLER(RTCrX509GeneralNames_Init)
# define RTCrX509GeneralSubtree_Clone                   RT_MANGLER(RTCrX509GeneralSubtree_Clone)
# define RTCrX509GeneralSubtree_Init                    RT_MANGLER(RTCrX509GeneralSubtree_Init)
# define RTCrX509GeneralSubtrees_Clone                  RT_MANGLER(RTCrX509GeneralSubtrees_Clone)
# define RTCrX509GeneralSubtrees_Init                   RT_MANGLER(RTCrX509GeneralSubtrees_Init)
# define RTCrX509NameConstraints_Clone                  RT_MANGLER(RTCrX509NameConstraints_Clone)
# define RTCrX509NameConstraints_Init                   RT_MANGLER(RTCrX509NameConstraints_Init)
# define RTCrX509Name_Clone                             RT_MANGLER(RTCrX509Name_Clone)
# define RTCrX509Name_Init                              RT_MANGLER(RTCrX509Name_Init)
# define RTCrX509Name_RecodeAsUtf8                      RT_MANGLER(RTCrX509Name_RecodeAsUtf8)
# define RTCrX509OldAuthorityKeyIdentifier_Clone        RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_Clone)
# define RTCrX509OldAuthorityKeyIdentifier_Init         RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_Init)
# define RTCrX509OtherName_Clone                        RT_MANGLER(RTCrX509OtherName_Clone)
# define RTCrX509OtherName_Init                         RT_MANGLER(RTCrX509OtherName_Init)
# define RTCrX509PolicyConstraints_Clone                RT_MANGLER(RTCrX509PolicyConstraints_Clone)
# define RTCrX509PolicyConstraints_Init                 RT_MANGLER(RTCrX509PolicyConstraints_Init)
# define RTCrX509PolicyInformation_Clone                RT_MANGLER(RTCrX509PolicyInformation_Clone)
# define RTCrX509PolicyInformation_Init                 RT_MANGLER(RTCrX509PolicyInformation_Init)
# define RTCrX509PolicyMapping_Clone                    RT_MANGLER(RTCrX509PolicyMapping_Clone)
# define RTCrX509PolicyMapping_Init                     RT_MANGLER(RTCrX509PolicyMapping_Init)
# define RTCrX509PolicyMappings_Clone                   RT_MANGLER(RTCrX509PolicyMappings_Clone)
# define RTCrX509PolicyMappings_Init                    RT_MANGLER(RTCrX509PolicyMappings_Init)
# define RTCrX509PolicyQualifierInfo_Clone              RT_MANGLER(RTCrX509PolicyQualifierInfo_Clone)
# define RTCrX509PolicyQualifierInfo_Init               RT_MANGLER(RTCrX509PolicyQualifierInfo_Init)
# define RTCrX509PolicyQualifierInfos_Clone             RT_MANGLER(RTCrX509PolicyQualifierInfos_Clone)
# define RTCrX509PolicyQualifierInfos_Init              RT_MANGLER(RTCrX509PolicyQualifierInfos_Init)
# define RTCrRsaPrivateKey_ReadFromFile                 RT_MANGLER(RTCrRsaPrivateKey_ReadFromFile)
# define RTCrRsaPrivateKey_ReadFromBuffer               RT_MANGLER(RTCrRsaPrivateKey_ReadFromBuffer)
# define RTCrRsaPublicKey_ReadFromFile                  RT_MANGLER(RTCrRsaPublicKey_ReadFromFile)
# define RTCrRsaPublicKey_ReadFromBuffer                RT_MANGLER(RTCrRsaPublicKey_ReadFromBuffer)
# define RTCrX509SubjectPublicKeyInfo_Clone             RT_MANGLER(RTCrX509SubjectPublicKeyInfo_Clone)
# define RTCrX509SubjectPublicKeyInfo_Init              RT_MANGLER(RTCrX509SubjectPublicKeyInfo_Init)
# define RTCrX509TbsCertificate_Clone                   RT_MANGLER(RTCrX509TbsCertificate_Clone)
# define RTCrX509TbsCertificate_Init                    RT_MANGLER(RTCrX509TbsCertificate_Init)
# define RTCrX509Validity_Clone                         RT_MANGLER(RTCrX509Validity_Clone)
# define RTCrX509Validity_Init                          RT_MANGLER(RTCrX509Validity_Init)
# define RTCrX509AlgorithmIdentifier_CheckSanity        RT_MANGLER(RTCrX509AlgorithmIdentifier_CheckSanity)
# define RTCrX509AlgorithmIdentifiers_CheckSanity       RT_MANGLER(RTCrX509AlgorithmIdentifiers_CheckSanity)
# define RTCrX509AttributeTypeAndValue_CheckSanity      RT_MANGLER(RTCrX509AttributeTypeAndValue_CheckSanity)
# define RTCrX509AttributeTypeAndValues_CheckSanity     RT_MANGLER(RTCrX509AttributeTypeAndValues_CheckSanity)
# define RTCrX509AuthorityKeyIdentifier_CheckSanity     RT_MANGLER(RTCrX509AuthorityKeyIdentifier_CheckSanity)
# define RTCrX509BasicConstraints_CheckSanity           RT_MANGLER(RTCrX509BasicConstraints_CheckSanity)
# define RTCrX509CertificatePolicies_CheckSanity        RT_MANGLER(RTCrX509CertificatePolicies_CheckSanity)
# define RTCrX509Certificate_CheckSanity                RT_MANGLER(RTCrX509Certificate_CheckSanity)
# define RTCrX509Certificates_CheckSanity               RT_MANGLER(RTCrX509Certificates_CheckSanity)
# define RTCrX509Extension_CheckSanity                  RT_MANGLER(RTCrX509Extension_CheckSanity)
# define RTCrX509Extensions_CheckSanity                 RT_MANGLER(RTCrX509Extensions_CheckSanity)
# define RTCrX509GeneralName_CheckSanity                RT_MANGLER(RTCrX509GeneralName_CheckSanity)
# define RTCrX509GeneralNames_CheckSanity               RT_MANGLER(RTCrX509GeneralNames_CheckSanity)
# define RTCrX509GeneralSubtree_CheckSanity             RT_MANGLER(RTCrX509GeneralSubtree_CheckSanity)
# define RTCrX509GeneralSubtrees_CheckSanity            RT_MANGLER(RTCrX509GeneralSubtrees_CheckSanity)
# define RTCrX509NameConstraints_CheckSanity            RT_MANGLER(RTCrX509NameConstraints_CheckSanity)
# define RTCrX509Name_CheckSanity                       RT_MANGLER(RTCrX509Name_CheckSanity)
# define RTCrX509OldAuthorityKeyIdentifier_CheckSanity  RT_MANGLER(RTCrX509OldAuthorityKeyIdentifier_CheckSanity)
# define RTCrX509OtherName_CheckSanity                  RT_MANGLER(RTCrX509OtherName_CheckSanity)
# define RTCrX509PolicyConstraints_CheckSanity          RT_MANGLER(RTCrX509PolicyConstraints_CheckSanity)
# define RTCrX509PolicyInformation_CheckSanity          RT_MANGLER(RTCrX509PolicyInformation_CheckSanity)
# define RTCrX509PolicyMapping_CheckSanity              RT_MANGLER(RTCrX509PolicyMapping_CheckSanity)
# define RTCrX509PolicyMappings_CheckSanity             RT_MANGLER(RTCrX509PolicyMappings_CheckSanity)
# define RTCrX509PolicyQualifierInfo_CheckSanity        RT_MANGLER(RTCrX509PolicyQualifierInfo_CheckSanity)
# define RTCrX509PolicyQualifierInfos_CheckSanity       RT_MANGLER(RTCrX509PolicyQualifierInfos_CheckSanity)
# define RTCrX509SubjectPublicKeyInfo_CheckSanity       RT_MANGLER(RTCrX509SubjectPublicKeyInfo_CheckSanity)
# define RTCrX509TbsCertificate_CheckSanity             RT_MANGLER(RTCrX509TbsCertificate_CheckSanity)
# define RTCrX509Validity_CheckSanity                   RT_MANGLER(RTCrX509Validity_CheckSanity)
# define RTCrX509Certificate_VerifySignature            RT_MANGLER(RTCrX509Certificate_VerifySignature)
# define RTCrX509Certificate_VerifySignatureSelfSigned  RT_MANGLER(RTCrX509Certificate_VerifySignatureSelfSigned)
# define RTCrX509Certificate_GenerateSelfSignedRsa      RT_MANGLER(RTCrX509Certificate_GenerateSelfSignedRsa)
# define RTCrTafCertPathControls_DecodeAsn1             RT_MANGLER(RTCrTafCertPathControls_DecodeAsn1)
# define RTCrTafTrustAnchorChoice_DecodeAsn1            RT_MANGLER(RTCrTafTrustAnchorChoice_DecodeAsn1)
# define RTCrTafTrustAnchorInfo_DecodeAsn1              RT_MANGLER(RTCrTafTrustAnchorInfo_DecodeAsn1)
# define RTCrTafTrustAnchorList_DecodeAsn1              RT_MANGLER(RTCrTafTrustAnchorList_DecodeAsn1)
# define RTCrTafCertPathControls_Compare                RT_MANGLER(RTCrTafCertPathControls_Compare)
# define RTCrTafCertPathControls_Delete                 RT_MANGLER(RTCrTafCertPathControls_Delete)
# define RTCrTafCertPathControls_Enum                   RT_MANGLER(RTCrTafCertPathControls_Enum)
# define RTCrTafTrustAnchorChoice_Compare               RT_MANGLER(RTCrTafTrustAnchorChoice_Compare)
# define RTCrTafTrustAnchorChoice_Delete                RT_MANGLER(RTCrTafTrustAnchorChoice_Delete)
# define RTCrTafTrustAnchorChoice_Enum                  RT_MANGLER(RTCrTafTrustAnchorChoice_Enum)
# define RTCrTafTrustAnchorInfo_Compare                 RT_MANGLER(RTCrTafTrustAnchorInfo_Compare)
# define RTCrTafTrustAnchorInfo_Delete                  RT_MANGLER(RTCrTafTrustAnchorInfo_Delete)
# define RTCrTafTrustAnchorInfo_Enum                    RT_MANGLER(RTCrTafTrustAnchorInfo_Enum)
# define RTCrTafTrustAnchorList_Compare                 RT_MANGLER(RTCrTafTrustAnchorList_Compare)
# define RTCrTafTrustAnchorList_Delete                  RT_MANGLER(RTCrTafTrustAnchorList_Delete)
# define RTCrTafTrustAnchorList_Enum                    RT_MANGLER(RTCrTafTrustAnchorList_Enum)
# define RTCrTafCertPathControls_Clone                  RT_MANGLER(RTCrTafCertPathControls_Clone)
# define RTCrTafCertPathControls_Init                   RT_MANGLER(RTCrTafCertPathControls_Init)
# define RTCrTafTrustAnchorChoice_Clone                 RT_MANGLER(RTCrTafTrustAnchorChoice_Clone)
# define RTCrTafTrustAnchorChoice_Init                  RT_MANGLER(RTCrTafTrustAnchorChoice_Init)
# define RTCrTafTrustAnchorInfo_Clone                   RT_MANGLER(RTCrTafTrustAnchorInfo_Clone)
# define RTCrTafTrustAnchorInfo_Init                    RT_MANGLER(RTCrTafTrustAnchorInfo_Init)
# define RTCrTafTrustAnchorList_Clone                   RT_MANGLER(RTCrTafTrustAnchorList_Clone)
# define RTCrTafTrustAnchorList_Init                    RT_MANGLER(RTCrTafTrustAnchorList_Init)
# define RTCrTafCertPathControls_CheckSanity            RT_MANGLER(RTCrTafCertPathControls_CheckSanity)
# define RTCrTafTrustAnchorChoice_CheckSanity           RT_MANGLER(RTCrTafTrustAnchorChoice_CheckSanity)
# define RTCrTafTrustAnchorInfo_CheckSanity             RT_MANGLER(RTCrTafTrustAnchorInfo_CheckSanity)
# define RTCrTafTrustAnchorList_CheckSanity             RT_MANGLER(RTCrTafTrustAnchorList_CheckSanity)
# define RTCrTspAccuracy_CheckSanity                    RT_MANGLER(RTCrTspAccuracy_CheckSanity)
# define RTCrTspAccuracy_Clone                          RT_MANGLER(RTCrTspAccuracy_Clone)
# define RTCrTspAccuracy_Compare                        RT_MANGLER(RTCrTspAccuracy_Compare)
# define RTCrTspAccuracy_DecodeAsn1                     RT_MANGLER(RTCrTspAccuracy_DecodeAsn1)
# define RTCrTspAccuracy_Delete                         RT_MANGLER(RTCrTspAccuracy_Delete)
# define RTCrTspAccuracy_Enum                           RT_MANGLER(RTCrTspAccuracy_Enum)
# define RTCrTspAccuracy_Init                           RT_MANGLER(RTCrTspAccuracy_Init)
# define RTCrTspMessageImprint_CheckSanity              RT_MANGLER(RTCrTspMessageImprint_CheckSanity)
# define RTCrTspMessageImprint_Clone                    RT_MANGLER(RTCrTspMessageImprint_Clone)
# define RTCrTspMessageImprint_Compare                  RT_MANGLER(RTCrTspMessageImprint_Compare)
# define RTCrTspMessageImprint_DecodeAsn1               RT_MANGLER(RTCrTspMessageImprint_DecodeAsn1)
# define RTCrTspMessageImprint_Delete                   RT_MANGLER(RTCrTspMessageImprint_Delete)
# define RTCrTspMessageImprint_Enum                     RT_MANGLER(RTCrTspMessageImprint_Enum)
# define RTCrTspMessageImprint_Init                     RT_MANGLER(RTCrTspMessageImprint_Init)
# define RTCrTspTstInfo_CheckSanity                     RT_MANGLER(RTCrTspTstInfo_CheckSanity)
# define RTCrTspTstInfo_Clone                           RT_MANGLER(RTCrTspTstInfo_Clone)
# define RTCrTspTstInfo_Compare                         RT_MANGLER(RTCrTspTstInfo_Compare)
# define RTCrTspTstInfo_DecodeAsn1                      RT_MANGLER(RTCrTspTstInfo_DecodeAsn1)
# define RTCrTspTstInfo_Delete                          RT_MANGLER(RTCrTspTstInfo_Delete)
# define RTCrTspTstInfo_Enum                            RT_MANGLER(RTCrTspTstInfo_Enum)
# define RTCrTspTstInfo_Init                            RT_MANGLER(RTCrTspTstInfo_Init)
# define RTCrCertCtxRelease                             RT_MANGLER(RTCrCertCtxRelease)
# define RTCrCertCtxRetain                              RT_MANGLER(RTCrCertCtxRetain)
# define RTCrStoreCertAddEncoded                        RT_MANGLER(RTCrStoreCertAddEncoded)
# define RTCrStoreCertAddX509                           RT_MANGLER(RTCrStoreCertAddX509)
# define RTCrStoreCertByIssuerAndSerialNo               RT_MANGLER(RTCrStoreCertByIssuerAndSerialNo)
# define RTCrStoreCertCount                             RT_MANGLER(RTCrStoreCertCount)
# define RTCrStoreCertFindAll                           RT_MANGLER(RTCrStoreCertFindAll)
# define RTCrStoreCertFindBySubjectOrAltSubjectByRfc5280 RT_MANGLER(RTCrStoreCertFindBySubjectOrAltSubjectByRfc5280)
# define RTCrStoreCertSearchDestroy                     RT_MANGLER(RTCrStoreCertSearchDestroy)
# define RTCrStoreCertSearchNext                        RT_MANGLER(RTCrStoreCertSearchNext)
# define RTCrStoreConvertToOpenSslCertStack             RT_MANGLER(RTCrStoreConvertToOpenSslCertStack)
# define RTCrStoreConvertToOpenSslCertStore             RT_MANGLER(RTCrStoreConvertToOpenSslCertStore)
# define RTCrStoreRelease                               RT_MANGLER(RTCrStoreRelease)
# define RTCrStoreRetain                                RT_MANGLER(RTCrStoreRetain)
# define RTCrStoreCreateInMem                           RT_MANGLER(RTCrStoreCreateInMem)
# define RTCrStoreCreateInMemEx                         RT_MANGLER(RTCrStoreCreateInMemEx)
# define RTCrStoreCreateSnapshotById                    RT_MANGLER(RTCrStoreCreateSnapshotById)
# define RTCrStoreCreateSnapshotOfUserAndSystemTrustedCAsAndCerts RT_MANGLER(RTCrStoreCreateSnapshotOfUserAndSystemTrustedCAsAndCerts)
# define RTCrStoreCertAddFromDir                        RT_MANGLER(RTCrStoreCertAddFromDir)
# define RTCrStoreCertAddFromFile                       RT_MANGLER(RTCrStoreCertAddFromFile)
# define RTCrStoreCertAddFromJavaKeyStore               RT_MANGLER(RTCrStoreCertAddFromJavaKeyStore)
# define RTCrStoreCertAddFromJavaKeyStoreInMem          RT_MANGLER(RTCrStoreCertAddFromJavaKeyStoreInMem)
# define RTCrStoreCertAddFromStore                      RT_MANGLER(RTCrStoreCertAddFromStore)
# define RTCrStoreCertAddWantedFromDir                  RT_MANGLER(RTCrStoreCertAddWantedFromDir)
# define RTCrStoreCertAddWantedFromFile                 RT_MANGLER(RTCrStoreCertAddWantedFromFile)
# define RTCrStoreCertAddWantedFromStore                RT_MANGLER(RTCrStoreCertAddWantedFromStore)
# define RTCrStoreCertAddWantedFromFishingExpedition    RT_MANGLER(RTCrStoreCertAddWantedFromFishingExpedition)
# define RTCrStoreCertCheckWanted                       RT_MANGLER(RTCrStoreCertCheckWanted)
# define RTCrStoreCertExportAsPem                       RT_MANGLER(RTCrStoreCertExportAsPem)
# define RTErrInfoAdd                                   RT_MANGLER(RTErrInfoAdd)
# define RTErrInfoAddF                                  RT_MANGLER(RTErrInfoAddF)
# define RTErrInfoAddV                                  RT_MANGLER(RTErrInfoAddV)
# define RTExprEvalCreate                               RT_MANGLER(RTExprEvalCreate)
# define RTExprEvalRelease                              RT_MANGLER(RTExprEvalRelease)
# define RTExprEvalRetain                               RT_MANGLER(RTExprEvalRetain)
# define RTExprEvalToBool                               RT_MANGLER(RTExprEvalToBool)
# define RTExprEvalToInteger                            RT_MANGLER(RTExprEvalToInteger)
# define RTExprEvalToString                             RT_MANGLER(RTExprEvalToString)
# define RTLdrHashImage                                 RT_MANGLER(RTLdrHashImage)
# define RTLdrOpenWithReader                            RT_MANGLER(RTLdrOpenWithReader)
# define RTLdrQueryPropEx                               RT_MANGLER(RTLdrQueryPropEx)
# define RTLdrVerifySignature                           RT_MANGLER(RTLdrVerifySignature)
# define RTBigNumAdd                                    RT_MANGLER(RTBigNumAdd)
# define RTBigNumAssign                                 RT_MANGLER(RTBigNumAssign)
# define RTBigNumBitWidth                               RT_MANGLER(RTBigNumBitWidth)
# define RTBigNumByteWidth                              RT_MANGLER(RTBigNumByteWidth)
# define RTBigNumClone                                  RT_MANGLER(RTBigNumClone)
# define RTBigNumCompare                                RT_MANGLER(RTBigNumCompare)
# define RTBigNumCompareWithS64                         RT_MANGLER(RTBigNumCompareWithS64)
# define RTBigNumCompareWithU64                         RT_MANGLER(RTBigNumCompareWithU64)
# define RTBigNumDestroy                                RT_MANGLER(RTBigNumDestroy)
# define RTBigNumDivide                                 RT_MANGLER(RTBigNumDivide)
# define RTBigNumDivideKnuth                            RT_MANGLER(RTBigNumDivideKnuth)
# define RTBigNumDivideLong                             RT_MANGLER(RTBigNumDivideLong)
# define RTBigNumExponentiate                           RT_MANGLER(RTBigNumExponentiate)
# define RTBigNumInit                                   RT_MANGLER(RTBigNumInit)
# define RTBigNumInitZero                               RT_MANGLER(RTBigNumInitZero)
# define RTBigNumModExp                                 RT_MANGLER(RTBigNumModExp)
# define RTBigNumModulo                                 RT_MANGLER(RTBigNumModulo)
# define RTBigNumMultiply                               RT_MANGLER(RTBigNumMultiply)
# define RTBigNumNegate                                 RT_MANGLER(RTBigNumNegate)
# define RTBigNumNegateThis                             RT_MANGLER(RTBigNumNegateThis)
# define RTBigNumShiftLeft                              RT_MANGLER(RTBigNumShiftLeft)
# define RTBigNumShiftRight                             RT_MANGLER(RTBigNumShiftRight)
# define RTBigNumSubtract                               RT_MANGLER(RTBigNumSubtract)
# define RTBigNumToBytesBigEndian                       RT_MANGLER(RTBigNumToBytesBigEndian)
# define RTUInt128MulByU64                              RT_MANGLER(RTUInt128MulByU64)
# define RTUInt128MulByU64_EndProc                      RT_MANGLER(RTUInt128MulByU64_EndProc)
# define RTUInt128MulByU64Ex                            RT_MANGLER(RTUInt128MulByU64Ex)
# define RTUInt128MulByU64Ex_EndProc                    RT_MANGLER(RTUInt128MulByU64Ex_EndProc)
# define RTUtf16Copy                                    RT_MANGLER(RTUtf16Copy)
# define RTUtf16CopyAscii                               RT_MANGLER(RTUtf16CopyAscii)
# define RTUtf16CopyEx                                  RT_MANGLER(RTUtf16CopyEx)
# define RTUtf16Cat                                     RT_MANGLER(RTUtf16Cat)
# define RTUtf16CatAscii                                RT_MANGLER(RTUtf16CatAscii)
# define RTUtf16Chr                                     RT_MANGLER(RTUtf16Chr)
# define RTUtf16End                                     RT_MANGLER(RTUtf16End)
# define RTUtf16ICmpAscii                               RT_MANGLER(RTUtf16ICmpAscii)
# define RTUtf16NICmpAscii                              RT_MANGLER(RTUtf16NICmpAscii)
# define RTUtf16NLen                                    RT_MANGLER(RTUtf16NLen)
# define RTUtf16NLenEx                                  RT_MANGLER(RTUtf16NLenEx)
# define RTUtf16PrintHexBytes                           RT_MANGLER(RTUtf16PrintHexBytes)
# define RTMemSaferAllocZExTag                          RT_MANGLER(RTMemSaferAllocZExTag)
# define RTMemSaferAllocZTag                            RT_MANGLER(RTMemSaferAllocZTag)
# define RTMemSaferFree                                 RT_MANGLER(RTMemSaferFree)
# define RTMemSaferGetSize                              RT_MANGLER(RTMemSaferGetSize)
# define RTMemSaferReallocZExTag                        RT_MANGLER(RTMemSaferReallocZExTag)
# define RTMemSaferReallocZTag                          RT_MANGLER(RTMemSaferReallocZTag)
# define RTMemSaferScramble                             RT_MANGLER(RTMemSaferScramble)
# define RTMemSaferUnscramble                           RT_MANGLER(RTMemSaferUnscramble)
# define RTErrConvertFromDarwin                         RT_MANGLER(RTErrConvertFromDarwin)
# define RTErrConvertFromDarwinCOM                      RT_MANGLER(RTErrConvertFromDarwinCOM)
# define RTErrConvertFromDarwinIO                       RT_MANGLER(RTErrConvertFromDarwinIO)
# define RTErrConvertFromDarwinKern                     RT_MANGLER(RTErrConvertFromDarwinKern)
# define RTErrConvertFromDarwin                         RT_MANGLER(RTErrConvertFromDarwin)
# define RTErrConvertFromDarwinIO                       RT_MANGLER(RTErrConvertFromDarwinIO)
# define RTErrConvertFromDarwinKern                     RT_MANGLER(RTErrConvertFromDarwinKern)

# define RTAsn1SeqOfBitStrings_Erase                    RT_MANGLER(RTAsn1SeqOfBitStrings_Erase)
# define RTAsn1SeqOfBitStrings_InsertEx                 RT_MANGLER(RTAsn1SeqOfBitStrings_InsertEx)
# define RTAsn1SeqOfBooleans_Erase                      RT_MANGLER(RTAsn1SeqOfBooleans_Erase)
# define RTAsn1SeqOfBooleans_InsertEx                   RT_MANGLER(RTAsn1SeqOfBooleans_InsertEx)
# define RTAsn1SeqOfCores_Erase                         RT_MANGLER(RTAsn1SeqOfCores_Erase)
# define RTAsn1SeqOfCores_InsertEx                      RT_MANGLER(RTAsn1SeqOfCores_InsertEx)
# define RTAsn1SeqOfIntegers_Erase                      RT_MANGLER(RTAsn1SeqOfIntegers_Erase)
# define RTAsn1SeqOfIntegers_InsertEx                   RT_MANGLER(RTAsn1SeqOfIntegers_InsertEx)
# define RTAsn1SeqOfObjIds_Erase                        RT_MANGLER(RTAsn1SeqOfObjIds_Erase)
# define RTAsn1SeqOfObjIds_InsertEx                     RT_MANGLER(RTAsn1SeqOfObjIds_InsertEx)
# define RTAsn1SeqOfOctetStrings_Erase                  RT_MANGLER(RTAsn1SeqOfOctetStrings_Erase)
# define RTAsn1SeqOfOctetStrings_InsertEx               RT_MANGLER(RTAsn1SeqOfOctetStrings_InsertEx)
# define RTAsn1SeqOfStrings_Erase                       RT_MANGLER(RTAsn1SeqOfStrings_Erase)
# define RTAsn1SeqOfStrings_InsertEx                    RT_MANGLER(RTAsn1SeqOfStrings_InsertEx)
# define RTAsn1SeqOfTimes_Erase                         RT_MANGLER(RTAsn1SeqOfTimes_Erase)
# define RTAsn1SeqOfTimes_InsertEx                      RT_MANGLER(RTAsn1SeqOfTimes_InsertEx)
# define RTAsn1SetOfBitStrings_Erase                    RT_MANGLER(RTAsn1SetOfBitStrings_Erase)
# define RTAsn1SetOfBitStrings_InsertEx                 RT_MANGLER(RTAsn1SetOfBitStrings_InsertEx)
# define RTAsn1SetOfBooleans_Erase                      RT_MANGLER(RTAsn1SetOfBooleans_Erase)
# define RTAsn1SetOfBooleans_InsertEx                   RT_MANGLER(RTAsn1SetOfBooleans_InsertEx)
# define RTAsn1SetOfCores_Erase                         RT_MANGLER(RTAsn1SetOfCores_Erase)
# define RTAsn1SetOfCores_InsertEx                      RT_MANGLER(RTAsn1SetOfCores_InsertEx)
# define RTAsn1SetOfIntegers_Erase                      RT_MANGLER(RTAsn1SetOfIntegers_Erase)
# define RTAsn1SetOfIntegers_InsertEx                   RT_MANGLER(RTAsn1SetOfIntegers_InsertEx)
# define RTAsn1SetOfObjIds_Erase                        RT_MANGLER(RTAsn1SetOfObjIds_Erase)
# define RTAsn1SetOfObjIds_InsertEx                     RT_MANGLER(RTAsn1SetOfObjIds_InsertEx)
# define RTAsn1SetOfObjIdSeqs_Erase                     RT_MANGLER(RTAsn1SetOfObjIdSeqs_Erase)
# define RTAsn1SetOfObjIdSeqs_InsertEx                  RT_MANGLER(RTAsn1SetOfObjIdSeqs_InsertEx)
# define RTAsn1SetOfOctetStrings_Erase                  RT_MANGLER(RTAsn1SetOfOctetStrings_Erase)
# define RTAsn1SetOfOctetStrings_InsertEx               RT_MANGLER(RTAsn1SetOfOctetStrings_InsertEx)
# define RTAsn1SetOfStrings_Erase                       RT_MANGLER(RTAsn1SetOfStrings_Erase)
# define RTAsn1SetOfStrings_InsertEx                    RT_MANGLER(RTAsn1SetOfStrings_InsertEx)
# define RTAsn1SetOfTimes_Erase                         RT_MANGLER(RTAsn1SetOfTimes_Erase)
# define RTAsn1SetOfTimes_InsertEx                      RT_MANGLER(RTAsn1SetOfTimes_InsertEx)
# define RTCrPkcs7Attributes_Erase                      RT_MANGLER(RTCrPkcs7Attributes_Erase)
# define RTCrPkcs7Attributes_InsertEx                   RT_MANGLER(RTCrPkcs7Attributes_InsertEx)
# define RTCrPkcs7SetOfCerts_Erase                      RT_MANGLER(RTCrPkcs7SetOfCerts_Erase)
# define RTCrPkcs7SetOfCerts_InsertEx                   RT_MANGLER(RTCrPkcs7SetOfCerts_InsertEx)
# define RTCrPkcs7SetOfContentInfos_Erase               RT_MANGLER(RTCrPkcs7SetOfContentInfos_Erase)
# define RTCrPkcs7SetOfContentInfos_InsertEx            RT_MANGLER(RTCrPkcs7SetOfContentInfos_InsertEx)
# define RTCrPkcs7SetOfSignedData_Erase                 RT_MANGLER(RTCrPkcs7SetOfSignedData_Erase)
# define RTCrPkcs7SetOfSignedData_InsertEx              RT_MANGLER(RTCrPkcs7SetOfSignedData_InsertEx)
# define RTCrPkcs7SignerInfos_Erase                     RT_MANGLER(RTCrPkcs7SignerInfos_Erase)
# define RTCrPkcs7SignerInfos_InsertEx                  RT_MANGLER(RTCrPkcs7SignerInfos_InsertEx)
# define RTCrRsaOtherPrimeInfos_Erase                   RT_MANGLER(RTCrRsaOtherPrimeInfos_Erase)
# define RTCrRsaOtherPrimeInfos_InsertEx                RT_MANGLER(RTCrRsaOtherPrimeInfos_InsertEx)
# define RTCrSpcSerializedObjectAttributes_Erase        RT_MANGLER(RTCrSpcSerializedObjectAttributes_Erase)
# define RTCrSpcSerializedObjectAttributes_InsertEx     RT_MANGLER(RTCrSpcSerializedObjectAttributes_InsertEx)
# define RTCrTafTrustAnchorList_Erase                   RT_MANGLER(RTCrTafTrustAnchorList_Erase)
# define RTCrTafTrustAnchorList_InsertEx                RT_MANGLER(RTCrTafTrustAnchorList_InsertEx)
# define RTCrX509AlgorithmIdentifiers_Erase             RT_MANGLER(RTCrX509AlgorithmIdentifiers_Erase)
# define RTCrX509AlgorithmIdentifiers_InsertEx          RT_MANGLER(RTCrX509AlgorithmIdentifiers_InsertEx)
# define RTCrX509AttributeTypeAndValues_Erase           RT_MANGLER(RTCrX509AttributeTypeAndValues_Erase)
# define RTCrX509AttributeTypeAndValues_InsertEx        RT_MANGLER(RTCrX509AttributeTypeAndValues_InsertEx)
# define RTCrX509CertificatePolicies_Erase              RT_MANGLER(RTCrX509CertificatePolicies_Erase)
# define RTCrX509CertificatePolicies_InsertEx           RT_MANGLER(RTCrX509CertificatePolicies_InsertEx)
# define RTCrX509Certificates_Erase                     RT_MANGLER(RTCrX509Certificates_Erase)
# define RTCrX509Certificates_InsertEx                  RT_MANGLER(RTCrX509Certificates_InsertEx)
# define RTCrX509Extensions_Erase                       RT_MANGLER(RTCrX509Extensions_Erase)
# define RTCrX509Extensions_InsertEx                    RT_MANGLER(RTCrX509Extensions_InsertEx)
# define RTCrX509GeneralNames_Erase                     RT_MANGLER(RTCrX509GeneralNames_Erase)
# define RTCrX509GeneralNames_InsertEx                  RT_MANGLER(RTCrX509GeneralNames_InsertEx)
# define RTCrX509GeneralSubtrees_Erase                  RT_MANGLER(RTCrX509GeneralSubtrees_Erase)
# define RTCrX509GeneralSubtrees_InsertEx               RT_MANGLER(RTCrX509GeneralSubtrees_InsertEx)
# define RTCrX509Name_Erase                             RT_MANGLER(RTCrX509Name_Erase)
# define RTCrX509Name_InsertEx                          RT_MANGLER(RTCrX509Name_InsertEx)
# define RTCrX509PolicyMappings_Erase                   RT_MANGLER(RTCrX509PolicyMappings_Erase)
# define RTCrX509PolicyMappings_InsertEx                RT_MANGLER(RTCrX509PolicyMappings_InsertEx)
# define RTCrX509PolicyQualifierInfos_Erase             RT_MANGLER(RTCrX509PolicyQualifierInfos_Erase)
# define RTCrX509PolicyQualifierInfos_InsertEx          RT_MANGLER(RTCrX509PolicyQualifierInfos_InsertEx)


/*
 * Stable variables (alphabetical order):
 */
# define g_apfnRTZlibDeps                               RT_MANGLER(g_apfnRTZlibDeps) /* os2 win solaris */
# define g_aRTUniFlagsRanges                            RT_MANGLER(g_aRTUniFlagsRanges)
# define g_aRTUniLowerRanges                            RT_MANGLER(g_aRTUniLowerRanges)
# define g_aRTUniUpperRanges                            RT_MANGLER(g_aRTUniUpperRanges)
# define g_fRTAlignmentChecks                           RT_MANGLER(g_fRTAlignmentChecks)
# define g_hKrnlDbgInfo                                 RT_MANGLER(g_hKrnlDbgInfo) /* solaris */
# define g_pStdErr                                      RT_MANGLER(g_pStdErr)
# define g_pStdIn                                       RT_MANGLER(g_pStdIn)
# define g_pStdOut                                      RT_MANGLER(g_pStdOut)
# define g_pfnRTLogAssert                               RT_MANGLER(g_pfnRTLogAssert)
# define g_pfnRTLogAssertV                              RT_MANGLER(g_pfnRTLogAssertV)
# define g_pfnRTLogGetDefaultInstance                   RT_MANGLER(g_pfnRTLogGetDefaultInstance)
# define g_pfnRTLogGetDefaultInstanceEx                 RT_MANGLER(g_pfnRTLogGetDefaultInstanceEx)
# define g_pfnRTLogLoggerExV                            RT_MANGLER(g_pfnRTLogLoggerExV)
# define g_pfnRTLogRelGetDefaultInstance                RT_MANGLER(g_pfnRTLogRelGetDefaultInstance)
# define g_pfnRTLogRelGetDefaultInstanceEx              RT_MANGLER(g_pfnRTLogRelGetDefaultInstanceEx)
# define g_pszRTAssertExpr                              RT_MANGLER(g_pszRTAssertExpr)
# define g_pszRTAssertFile                              RT_MANGLER(g_pszRTAssertFile)
# define g_pszRTAssertFunction                          RT_MANGLER(g_pszRTAssertFunction)
# define g_szRTAssertMsg1                               RT_MANGLER(g_szRTAssertMsg1)
# define g_szRTAssertMsg2                               RT_MANGLER(g_szRTAssertMsg2)
# define g_u32RTAssertLine                              RT_MANGLER(g_u32RTAssertLine)

/* sort/merge into the above later: */
# define g_RTAsn1Time_Vtable                            RT_MANGLER(g_RTAsn1Time_Vtable)
# define g_RTAsn1String_Vtable                          RT_MANGLER(g_RTAsn1String_Vtable)
# define g_RTAsn1OctetString_Vtable                     RT_MANGLER(g_RTAsn1OctetString_Vtable)
# define g_RTAsn1ObjId_Vtable                           RT_MANGLER(g_RTAsn1ObjId_Vtable)
# define g_RTAsn1Null_Vtable                            RT_MANGLER(g_RTAsn1Null_Vtable)
# define g_RTAsn1Integer_Vtable                         RT_MANGLER(g_RTAsn1Integer_Vtable)
# define g_RTAsn1Core_Vtable                            RT_MANGLER(g_RTAsn1Core_Vtable)
# define g_RTAsn1Boolean_Vtable                         RT_MANGLER(g_RTAsn1Boolean_Vtable)
# define g_RTAsn1BitString_Vtable                       RT_MANGLER(g_RTAsn1BitString_Vtable)
# define g_RTAsn1DefaultAllocator                       RT_MANGLER(g_RTAsn1DefaultAllocator)
# define g_RTAsn1EFenceAllocator                        RT_MANGLER(g_RTAsn1EFenceAllocator)
# define g_RTAsn1SaferAllocator                         RT_MANGLER(g_RTAsn1SaferAllocator)
# define g_aRTCrPkcs7Markers                            RT_MANGLER(g_aRTCrPkcs7Markers)
# define g_cRTCrPkcs7Markers                            RT_MANGLER(g_cRTCrPkcs7Markers)
# define g_RTCrPkcs8PrivateKeyInfo_Vtable               RT_MANGLER(g_RTCrPkcs8PrivateKeyInfo_Vtable)
# define g_RTCrPkcs8EncryptedPrivateKeyInfo_Vtable      RT_MANGLER(g_RTCrPkcs8EncryptedPrivateKeyInfo_Vtable)
# define g_aRTCrX509CertificateMarkers                  RT_MANGLER(g_aRTCrX509CertificateMarkers)
# define g_cRTCrX509CertificateMarkers                  RT_MANGLER(g_cRTCrX509CertificateMarkers)
# define g_aRTCrKeyPublicMarkers                        RT_MANGLER(g_aRTCrKeyPublicMarkers)
# define g_cRTCrKeyPublicMarkers                        RT_MANGLER(g_cRTCrKeyPublicMarkers)
# define g_aRTCrKeyPrivateMarkers                       RT_MANGLER(g_aRTCrKeyPrivateMarkers)
# define g_cRTCrKeyPrivateMarkers                       RT_MANGLER(g_cRTCrKeyPrivateMarkers)
# define g_aRTCrKeyAllMarkers                           RT_MANGLER(g_aRTCrKeyAllMarkers)
# define g_cRTCrKeyAllMarkers                           RT_MANGLER(g_cRTCrKeyAllMarkers)
# define g_acRTThreadTypeStats                          RT_MANGLER(g_acRTThreadTypeStats) /* internal */
# define g_RTIoQueueStdFileProv                         RT_MANGLER(g_RTIoQueueStdFileProv) /* internal */
# define g_RTIoQueueAioFileProv                         RT_MANGLER(g_RTIoQueueAioFileProv) /* internal */
# define g_RTIoQueueLnxIoURingProv                      RT_MANGLER(g_RTIoQueueLnxIoURingProv) /* internal */

#if 0 /* Disabled for now as I'm not sure the assmbler supports mangling yet. */
# define g_abRTZeroPage                                 RT_MANGLER(g_abRTZeroPage)
# define g_abRTZero4K                                   RT_MANGLER(g_abRTZero4K)
# define g_abRTZero8K                                   RT_MANGLER(g_abRTZero8K)
# define g_abRTZero16K                                  RT_MANGLER(g_abRTZero16K)
# define g_abRTZero32K                                  RT_MANGLER(g_abRTZero32K)
# define g_abRTZero64K                                  RT_MANGLER(g_abRTZero64K)
#endif


/*
 * Unstable functions (alphabetical order):
 */
/** @todo the list is incomplete! See the .def files + libraries. */


/*
 * Unstable variables (alphabetical order):
 */
/* none */

#endif /* !DOXYGEN_RUNNING */

#endif /* !IPRT_INCLUDED_mangling_h */
