package aQute.maven.bnd;

import static java.nio.file.Files.createDirectories;
import static java.nio.file.Files.isDirectory;
import static java.nio.file.Files.newDirectoryStream;

import java.io.Closeable;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;

import aQute.maven.repo.api.Program;
import aQute.maven.repo.api.Revision;
import aQute.service.reporter.Reporter;

/**
 * Implements a maven revisions repository on the file system. Initially planned
 * to use the Java 7 directory watching but that was a no started due to lack of
 * implementations and missing events. So this one is now polled. The refresh
 * method will do a scan of the directories (a pom file is seen as a hit of a
 * project directory). It will do a callback with the added/deleted entries from
 * a previous scan.
 */
public class LocalRepoWatcher implements Closeable {

	public interface ProgramsChanged {
		public boolean changed(Set<Program> added, Set<Program> removed) throws Exception;
	}

	private final Path							root;
	private final AtomicReference<Set<Program>>	older			= new AtomicReference<Set<Program>>(
			Collections.EMPTY_SET);
	private ProgramsChanged						callback;
	private final ReentrantLock					callbackLock	= new ReentrantLock();
	private final AtomicBoolean					busy			= new AtomicBoolean();
	private ScheduledFuture< ? >				schedule;

	public LocalRepoWatcher(Path root, ProgramsChanged callback, Reporter reporter, ScheduledExecutorService executor,
			long timeoutinMs)
			throws Exception {
		createDirectories(root);
		this.root = root;
		this.callback = callback;
		if (executor != null) {
			setupScheduler(executor, timeoutinMs);
		}
	}

	void setupScheduler(ScheduledExecutorService executor, long timeoutinMs) {
		schedule = executor.scheduleAtFixedRate(new Runnable() {

			@Override
			public void run() {
				int priority = Thread.currentThread().getPriority();
				try {
					Thread.currentThread().setPriority(Thread.MIN_PRIORITY);

					if (busy.getAndSet(true) == false) {
						try {
							refresh();
						} catch (IOException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						} finally {
							busy.set(false);
						}
					}
				} finally {
					Thread.currentThread().setPriority(priority);
				}
			}

		}, timeoutinMs, timeoutinMs, TimeUnit.MILLISECONDS);
	}

	public void open() throws IOException {
		refresh();
	}

	public void refresh() throws IOException {
		Set<Program> added;
		Set<Program> removed;
		Set<Program> newer = new HashSet<>();

		traverse(root, newer, 0);

		if (this.older != null && this.older.equals(newer))
			return;

		added = new HashSet<>(newer);
		added.removeAll(older.get());

		removed = new HashSet<>(older.get());
		removed.removeAll(newer);

		older.set(newer);

		if (callback == null)
			return;

		callbackLock.lock();

		try {
			callback.changed(added, removed);
		} catch (Exception e) {
			// TODO
			e.printStackTrace();
		} finally {
			callbackLock.unlock();
		}
	}

	void traverse(Path dir, Set<Program> programs, int depth) throws IOException {
		try (DirectoryStream<Path> stream = newDirectoryStream(dir);) {
			for (Path path : stream) {
				if (isDirectory(path)) {
					if (depth >= 2) {
						Path artifact = path.getName(path.getNameCount() - 2);
						Path version = path.getName(path.getNameCount() - 1);

						Path p = path.resolve(artifact + "-" + version + ".pom");
						if (Files.isRegularFile(p)) {
							Path revisionDir = root.relativize(path);
							Revision r = Revision.fromProjectPath(revisionDir);
							programs.add(r.program);
							return; // TODO we don't look at siblings. Is this
									// ok?
						}
					}
					traverse(path, programs, depth + 1);
				}

				if (Thread.currentThread().isInterrupted())
					return;

				Thread.yield();
			}
		}
	}

	@Override
	public void close() throws IOException {
		if (schedule != null)
			schedule.cancel(true);
	}

	public Set<Program> getLocalPrograms() {
		return older.get();
	}

}
