/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * (c) ZeroTier, Inc.
 * https://www.zerotier.com/
 */

#include "node/Buffer.hpp"
#include "node/CertificateOfMembership.hpp"
#include "node/Constants.hpp"
#include "node/Dictionary.hpp"
#include "node/ECC.hpp"
#include "node/Hashtable.hpp"
#include "node/Identity.hpp"
#include "node/IncomingPacket.hpp"
#include "node/InetAddress.hpp"
#include "node/MAC.hpp"
#include "node/NetworkConfig.hpp"
#include "node/Node.hpp"
#include "node/Packet.hpp"
#include "node/Peer.hpp"
#include "node/Poly1305.hpp"
#include "node/RuntimeEnvironment.hpp"
#include "node/SHA512.hpp"
#include "node/Salsa20.hpp"
#include "node/Utils.hpp"
#include "osdep/OSUtils.hpp"
#include "osdep/Phy.hpp"
#include "osdep/PortMapper.hpp"
#include "osdep/Thread.hpp"

#include <iostream>
#include <stdexcept>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <string>
#include <thread>
#include <time.h>
#include <vector>

#if defined(ZT_USE_X64_ASM_SALSA2012) && defined(ZT_ARCH_X64)
#include "ext/x64-salsa2012-asm/salsa2012.h"
#endif
#ifdef ZT_USE_ARM32_NEON_ASM_SALSA2012
#include "ext/arm32-neon-salsa2012-asm/salsa2012.h"
#endif

#ifdef __WINDOWS__
#include <tchar.h>
#endif

using namespace ZeroTier;

//////////////////////////////////////////////////////////////////////////////

#define KNOWN_GOOD_IDENTITY                                                                                                                                                                                                                    \
	"8e4df28b72:0:ac3d46abe0c21f3cfe7a6c8d6a85cfcffcb82fbd55af6a4d6350657c68200843fa2e16f9418bbd9702cae365f2af5fb4c420908b803a681d4daef6114d78a2d7:"                                                                                           \
	"bd8dd6e4ce7022d2f812797a80c6ee8ad180dc4ebf301dec8b06d1be08832bddd63a2f1cfa7b2c504474c75bdc8898ba476ef92e8e2d0509f8441985171ff16e"
#define KNOWN_BAD_IDENTITY                                                                                                                                                                                                                     \
	"9e4df28b72:0:ac3d46abe0c21f3cfe7a6c8d6a85cfcffcb82fbd55af6a4d6350657c68200843fa2e16f9418bbd9702cae365f2af5fb4c420908b803a681d4daef6114d78a2d7:"                                                                                           \
	"bd8dd6e4ce7022d2f812797a80c6ee8ad180dc4ebf301dec8b06d1be08832bddd63a2f1cfa7b2c504474c75bdc8898ba476ef92e8e2d0509f8441985171ff16e"

static const unsigned char s20TV0Key[32] = { 0x0f, 0x62, 0xb5, 0x08, 0x5b, 0xae, 0x01, 0x54, 0xa7, 0xfa, 0x4d, 0xa0, 0xf3, 0x46, 0x99, 0xec, 0x3f, 0x92, 0xe5, 0x38, 0x8b, 0xde, 0x31, 0x84, 0xd7, 0x2a, 0x7d, 0xd0, 0x23, 0x76, 0xc9, 0x1c };
static const unsigned char s20TV0Iv[8] = { 0x28, 0x8f, 0xf6, 0x5d, 0xc4, 0x2b, 0x92, 0xf9 };
static const unsigned char s20TV0Ks[64] = { 0x5e, 0x5e, 0x71, 0xf9, 0x01, 0x99, 0x34, 0x03, 0x04, 0xab, 0xb2, 0x2a, 0x37, 0xb6, 0x62, 0x5b, 0xf8, 0x83, 0xfb, 0x89, 0xce, 0x3b, 0x21, 0xf5, 0x4a, 0x10, 0xb8, 0x10, 0x66, 0xef, 0x87, 0xda,
											0x30, 0xb7, 0x76, 0x99, 0xaa, 0x73, 0x79, 0xda, 0x59, 0x5c, 0x77, 0xdd, 0x59, 0x54, 0x2d, 0xa2, 0x08, 0xe5, 0x95, 0x4f, 0x89, 0xe4, 0x0e, 0xb7, 0xaa, 0x80, 0xa8, 0x4a, 0x61, 0x76, 0x66, 0x3f };

static const unsigned char s2012TV0Key[32] = { 0x0f, 0x62, 0xb5, 0x08, 0x5b, 0xae, 0x01, 0x54, 0xa7, 0xfa, 0x4d, 0xa0, 0xf3, 0x46, 0x99, 0xec, 0x3f, 0x92, 0xe5, 0x38, 0x8b, 0xde, 0x31, 0x84, 0xd7, 0x2a, 0x7d, 0xd0, 0x23, 0x76, 0xc9, 0x1c };
static const unsigned char s2012TV0Iv[8] = { 0x28, 0x8f, 0xf6, 0x5d, 0xc4, 0x2b, 0x92, 0xf9 };
static const unsigned char s2012TV0Ks[64] = { 0x99, 0xDB, 0x33, 0xAD, 0x11, 0xCE, 0x0C, 0xCB, 0x3B, 0xFD, 0xBF, 0x8D, 0x0C, 0x18, 0x16, 0x04, 0x52, 0xD0, 0x14, 0xCD, 0xE9, 0x89, 0xB4, 0xC4, 0x11, 0xA5, 0x59, 0xFF, 0x7C, 0x20, 0xA1, 0x69,
											  0xE6, 0xDC, 0x99, 0x09, 0xD8, 0x16, 0xBE, 0xCE, 0xDC, 0x40, 0x63, 0xCE, 0x07, 0xCE, 0xA8, 0x28, 0xF4, 0x4B, 0xF9, 0xB6, 0xC9, 0xA0, 0xA0, 0xB2, 0x00, 0xE1, 0xB5, 0x2A, 0xF4, 0x18, 0x59, 0xC5 };

static const unsigned char poly1305TV0Input[32] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
													0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const unsigned char poly1305TV0Key[32] = {
	0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x33, 0x32, 0x2d, 0x62, 0x79, 0x74, 0x65, 0x20, 0x6b, 0x65, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x50, 0x6f, 0x6c, 0x79, 0x31, 0x33, 0x30, 0x35
};
static const unsigned char poly1305TV0Tag[16] = { 0x49, 0xec, 0x78, 0x09, 0x0e, 0x48, 0x1e, 0xc6, 0xc2, 0x6b, 0x33, 0xb9, 0x1c, 0xcc, 0x03, 0x07 };

static const unsigned char poly1305TV1Input[12] = { 0x48, 0x65, 0x6c, 0x6c, 0x6f, 0x20, 0x77, 0x6f, 0x72, 0x6c, 0x64, 0x21 };
static const unsigned char poly1305TV1Key[32] = {
	0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x33, 0x32, 0x2d, 0x62, 0x79, 0x74, 0x65, 0x20, 0x6b, 0x65, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x50, 0x6f, 0x6c, 0x79, 0x31, 0x33, 0x30, 0x35
};
static const unsigned char poly1305TV1Tag[16] = { 0xa6, 0xf7, 0x45, 0x00, 0x8f, 0x81, 0xc9, 0x16, 0xa2, 0x0d, 0xcc, 0x74, 0xee, 0xf2, 0xb2, 0xf0 };

static const char* sha512TV0Input = "supercalifragilisticexpealidocious";
static const unsigned char sha512TV0Digest[64] = { 0x18, 0x2a, 0x85, 0x59, 0x69, 0xe5, 0xd3, 0xe6, 0xcb, 0xf6, 0x05, 0x24, 0xad, 0xf2, 0x88, 0xd1, 0xbb, 0xf2, 0x52, 0x92, 0x81, 0x24,
												   0x31, 0xf6, 0xd2, 0x52, 0xf1, 0xdb, 0xc1, 0xcb, 0x44, 0xdf, 0x21, 0x57, 0x3d, 0xe1, 0xb0, 0x6b, 0x68, 0x75, 0x95, 0x9f, 0x3b, 0x6f,
												   0x87, 0xb1, 0x13, 0x81, 0xd0, 0xbc, 0x79, 0x2c, 0x43, 0x3a, 0x13, 0x55, 0x3c, 0xe0, 0x84, 0xc2, 0x92, 0x55, 0x31, 0x1c };

struct C25519TestVector {
	unsigned char pub1[64];
	unsigned char priv1[64];
	unsigned char pub2[64];
	unsigned char priv2[64];
	unsigned char agreement[64];
	unsigned char agreementSignedBy1[96];
	unsigned char agreementSignedBy2[96];
};

#define ZT_NUM_C25519_TEST_VECTORS 32

static const C25519TestVector C25519_TEST_VECTORS[ZT_NUM_C25519_TEST_VECTORS] = {
	{ { 0xa1, 0xfc, 0x7a, 0xb4, 0x6d, 0xdf, 0x7d, 0xcf, 0xe7, 0xec, 0x75, 0xe5, 0xfa, 0xdd, 0x11, 0xcb, 0xcc, 0x37, 0xf8, 0x84, 0x5d, 0x1c, 0x92, 0x4e, 0x09, 0x89, 0x65, 0xfc, 0xd8, 0xe9, 0x5a, 0x30,
		0xda, 0xe4, 0x86, 0xa3, 0x35, 0xb4, 0x19, 0x0c, 0xbc, 0x7b, 0xcb, 0x3e, 0xb9, 0x4c, 0xbd, 0x16, 0xe8, 0x3d, 0x13, 0x2b, 0xc9, 0xc3, 0x39, 0xea, 0xf1, 0x42, 0xe7, 0x6f, 0x69, 0x78, 0x9a, 0xb7 },
	  { 0xe5, 0xf3, 0x7b, 0xd4, 0x0e, 0xc9, 0xdc, 0x77, 0x50, 0x86, 0xdc, 0xf4, 0x2e, 0xbc, 0xdb, 0x27, 0xf0, 0x73, 0xd4, 0x58, 0x73, 0xc4, 0x4b, 0x71, 0x8b, 0x3c, 0xc5, 0x4f, 0xa8, 0x7c, 0xa4, 0x84,
		0xd9, 0x96, 0x23, 0x73, 0xb4, 0x03, 0x16, 0xbf, 0x1e, 0xa1, 0x2d, 0xd8, 0xc4, 0x8a, 0xe7, 0x82, 0x10, 0xda, 0xc9, 0xe5, 0x45, 0x9b, 0x01, 0xdc, 0x73, 0xa6, 0xc9, 0x17, 0xa8, 0x15, 0x31, 0x6d },
	  { 0x3e, 0x49, 0xa4, 0x0e, 0x3a, 0xaf, 0xa3, 0x07, 0x3d, 0xf7, 0x2a, 0xec, 0x43, 0xb1, 0xd4, 0x09, 0x1a, 0xcb, 0x8e, 0x92, 0xf9, 0x65, 0x95, 0x04, 0x6d, 0x2d, 0x9b, 0x34, 0xa3, 0xbf, 0x51, 0x00,
		0xe2, 0xee, 0x23, 0xf5, 0x28, 0x0a, 0xa9, 0xb1, 0x57, 0x0b, 0x96, 0x56, 0x62, 0xba, 0x12, 0x94, 0xaf, 0xc6, 0x5f, 0xb5, 0x61, 0x43, 0x0f, 0xde, 0x0b, 0xab, 0xfa, 0x4f, 0xfe, 0xc5, 0xe7, 0x18 },
	  { 0x00, 0x4d, 0x41, 0x8d, 0xe4, 0x69, 0x23, 0xae, 0x98, 0xc4, 0x3e, 0x77, 0x0f, 0x1d, 0x94, 0x5d, 0x29, 0x3e, 0x94, 0x5a, 0x38, 0x39, 0x20, 0x0f, 0xd3, 0x6f, 0x76, 0xa2, 0x29, 0x02, 0x03, 0xcb,
		0x0b, 0x7f, 0x4f, 0x1a, 0x29, 0x51, 0x13, 0x33, 0x7c, 0x99, 0xb3, 0x81, 0x82, 0x39, 0x44, 0x05, 0x97, 0xfb, 0x0d, 0xf2, 0x93, 0xa2, 0x40, 0x94, 0xf4, 0xff, 0x5d, 0x09, 0x61, 0xe4, 0x5f, 0x76 },
	  { 0xab, 0xce, 0xd2, 0x24, 0xe8, 0x93, 0xb0, 0xe7, 0x72, 0x14, 0xdc, 0xbb, 0x7d, 0x0f, 0xd8, 0x94, 0x16, 0x9e, 0xb5, 0x7f, 0xd7, 0x19, 0x5f, 0x3e, 0x2d, 0x45, 0xd5, 0xf7, 0x90, 0x0b, 0x3e, 0x05,
		0x18, 0x2e, 0x2b, 0xf4, 0xfa, 0xd4, 0xec, 0x62, 0x4a, 0x4f, 0x48, 0x50, 0xaf, 0x1c, 0xe8, 0x9f, 0x1a, 0xe1, 0x3d, 0x70, 0x49, 0x00, 0xa7, 0xe3, 0x5b, 0x1e, 0xa1, 0x9b, 0x68, 0x1e, 0xa1, 0x73 },
	  { 0xed, 0xb6, 0xd0, 0xf0, 0x06, 0x6e, 0x33, 0x9c, 0x86, 0xfb, 0xe8, 0xc3, 0x6c, 0x8d, 0xde, 0xdd, 0xa6, 0xa0, 0x2d, 0xb9, 0x07, 0x29, 0xa3, 0x13, 0xbb, 0xa4, 0xba, 0xec, 0x48, 0xc8, 0xf4, 0x56,
		0x82, 0x79, 0xe2, 0xb1, 0xd3, 0x3d, 0x83, 0x9f, 0x10, 0xe8, 0x52, 0xe6, 0x8b, 0x1c, 0x33, 0x9e, 0x2b, 0xd2, 0xdb, 0x62, 0x1c, 0x56, 0xfd, 0x50, 0x40, 0x77, 0x81, 0xab, 0x21, 0x67, 0x3e, 0x09,
		0x4f, 0xf2, 0x51, 0xac, 0x7d, 0xe7, 0xd1, 0x5d, 0x4b, 0xe2, 0x08, 0xc6, 0x3f, 0x6a, 0x4d, 0xc8, 0x5d, 0x74, 0xf6, 0x3b, 0xec, 0x8e, 0xc6, 0x0c, 0x32, 0x27, 0x2f, 0x9c, 0x09, 0x48, 0x59, 0x10 },
	  { 0x23, 0x0f, 0xa3, 0xe2, 0x69, 0xce, 0xb9, 0xb9, 0xd1, 0x1c, 0x4e, 0xab, 0x63, 0xc9, 0x2e, 0x1e, 0x7e, 0xa2, 0xa2, 0xa0, 0x49, 0x2e, 0x78, 0xe4, 0x8a, 0x02, 0x3b, 0xa7, 0xab, 0x1f, 0xd4, 0xce,
		0x05, 0xe2, 0x80, 0x09, 0x09, 0x3c, 0x61, 0xc7, 0x10, 0x3a, 0x9c, 0xf4, 0x95, 0xac, 0x89, 0x6f, 0x23, 0xb3, 0x09, 0xe2, 0x24, 0x3f, 0xf6, 0x96, 0x02, 0x36, 0x41, 0x16, 0x32, 0xe1, 0x66, 0x05,
		0x4f, 0xf2, 0x51, 0xac, 0x7d, 0xe7, 0xd1, 0x5d, 0x4b, 0xe2, 0x08, 0xc6, 0x3f, 0x6a, 0x4d, 0xc8, 0x5d, 0x74, 0xf6, 0x3b, 0xec, 0x8e, 0xc6, 0x0c, 0x32, 0x27, 0x2f, 0x9c, 0x09, 0x48, 0x59, 0x10 } },
	{ { 0xfd, 0x81, 0x14, 0xf1, 0x67, 0x07, 0x44, 0xbb, 0x93, 0x84, 0xa2, 0xdc, 0x36, 0xdc, 0xcc, 0xb3, 0x9e, 0x82, 0xd4, 0x8b, 0x42, 0x56, 0xfb, 0xf2, 0x6e, 0x83, 0x3b, 0x16, 0x2c, 0x29, 0xfb, 0x39,
		0x29, 0x48, 0x85, 0xe3, 0xe3, 0xf7, 0xe7, 0x80, 0x49, 0xd3, 0x01, 0x30, 0x5a, 0x2c, 0x3f, 0x4c, 0xea, 0x13, 0xeb, 0xda, 0xf4, 0x56, 0x75, 0x8d, 0x50, 0x1e, 0x19, 0x2d, 0x29, 0x2b, 0xfb, 0xdb },
	  { 0x85, 0x34, 0x4d, 0xf7, 0x39, 0xbf, 0x98, 0x79, 0x8c, 0x98, 0xeb, 0x8d, 0x61, 0x27, 0xec, 0x87, 0x56, 0xcd, 0xd0, 0xa6, 0x55, 0x77, 0xee, 0xf0, 0x20, 0xd0, 0x59, 0x39, 0x95, 0xab, 0x29, 0x82,
		0x8e, 0x61, 0xf8, 0xad, 0xed, 0xb6, 0x27, 0xc3, 0xd8, 0x16, 0xce, 0x67, 0x78, 0xe2, 0x04, 0x4b, 0x0c, 0x2d, 0x2f, 0xc3, 0x24, 0x72, 0xbc, 0x53, 0xbd, 0xfe, 0x39, 0x23, 0xd4, 0xaf, 0x27, 0x84 },
	  { 0x11, 0xbe, 0x5f, 0x5a, 0x73, 0xe7, 0x42, 0xef, 0xff, 0x3c, 0x47, 0x6a, 0x0e, 0x6b, 0x9e, 0x96, 0x21, 0xa3, 0xdf, 0x49, 0xe9, 0x3f, 0x40, 0xfc, 0xab, 0xb3, 0x66, 0xd3, 0x3d, 0xfa, 0x02, 0x29,
		0xf3, 0x43, 0x45, 0x3c, 0x70, 0xa3, 0x5d, 0x39, 0xf7, 0xc0, 0x6a, 0xcd, 0xfa, 0x1d, 0xbe, 0x3b, 0x91, 0x41, 0xe4, 0xb0, 0x60, 0xc0, 0x22, 0xf7, 0x2c, 0x11, 0x2b, 0x1c, 0x5f, 0x24, 0xef, 0x53 },
	  { 0xfd, 0x3f, 0x09, 0x06, 0xc9, 0x39, 0x8d, 0x48, 0xfa, 0x6b, 0xc9, 0x80, 0xbf, 0xf6, 0xd6, 0x76, 0xb3, 0x62, 0x70, 0x88, 0x4f, 0xde, 0xde, 0xb9, 0xb4, 0xf0, 0xce, 0xf3, 0x74, 0x0d, 0xea, 0x00,
		0x9e, 0x9c, 0x29, 0xe1, 0xa2, 0x1b, 0xbd, 0xb5, 0x83, 0xcc, 0x12, 0xd8, 0x48, 0x08, 0x5b, 0xe5, 0xd6, 0xf9, 0x11, 0x5c, 0xe0, 0xd9, 0xc3, 0x3c, 0x26, 0xbd, 0x69, 0x9f, 0x5c, 0x6f, 0x0c, 0x6f },
	  { 0xca, 0xd4, 0x76, 0x32, 0x8b, 0xbe, 0x0c, 0x65, 0x75, 0x43, 0x73, 0xc2, 0xf2, 0xfd, 0x7f, 0xeb, 0xe4, 0x62, 0xc5, 0x0d, 0x0f, 0xf9, 0x01, 0xc8, 0xb9, 0xfa, 0xca, 0xb4, 0x12, 0x1c, 0xb4, 0xac,
		0x0e, 0x5f, 0x18, 0xfc, 0x0c, 0x7f, 0x2a, 0x55, 0xc5, 0xfd, 0x4d, 0x83, 0xb2, 0x02, 0x31, 0x6a, 0x3f, 0x14, 0xee, 0x9d, 0x11, 0xa8, 0x06, 0xad, 0xeb, 0x93, 0x19, 0x79, 0xb1, 0xf2, 0x78, 0x05 },
	  { 0x85, 0xe6, 0xe2, 0xf2, 0x96, 0xe7, 0xa2, 0x8b, 0x7e, 0x36, 0xbd, 0x7b, 0xf4, 0x28, 0x6a, 0xd7, 0xbc, 0x2a, 0x6a, 0x59, 0xfd, 0xc0, 0xc8, 0x3d, 0x50, 0x0f, 0x0c, 0x2b, 0x12, 0x3a, 0x75, 0xc7,
		0x56, 0xbb, 0x7f, 0x7d, 0x4e, 0xd4, 0x03, 0xb8, 0x7b, 0xde, 0xde, 0x99, 0x65, 0x9e, 0xc4, 0xa6, 0x6e, 0xfe, 0x00, 0x88, 0xeb, 0x9d, 0xa4, 0xa9, 0x9d, 0x37, 0xc9, 0x4a, 0xcf, 0x69, 0xc4, 0x01,
		0xba, 0xa8, 0xce, 0xeb, 0x72, 0xcb, 0x64, 0x8b, 0x9f, 0xc1, 0x1f, 0x9a, 0x9e, 0x99, 0xcc, 0x39, 0xec, 0xd9, 0xbb, 0xd9, 0xce, 0xc2, 0x74, 0x6f, 0xd0, 0x2a, 0xb9, 0xc6, 0xe3, 0xf5, 0xe7, 0xf4 },
	  { 0xb1, 0x39, 0x50, 0xb1, 0x1a, 0x08, 0x42, 0x2b, 0xdd, 0x6d, 0x20, 0x9f, 0x0f, 0x37, 0xba, 0x69, 0x97, 0x21, 0x30, 0x7a, 0x71, 0x2f, 0xce, 0x98, 0x09, 0x04, 0xa2, 0x98, 0x6a, 0xed, 0x02, 0x1d,
		0x5d, 0x30, 0x8f, 0x03, 0x47, 0x6b, 0x89, 0xfd, 0xf7, 0x1a, 0xca, 0x46, 0x6f, 0x51, 0x69, 0x9a, 0x2b, 0x18, 0x77, 0xe4, 0xad, 0x0d, 0x7a, 0x66, 0xd2, 0x2c, 0x28, 0xa0, 0xd3, 0x0a, 0x99, 0x0d,
		0xba, 0xa8, 0xce, 0xeb, 0x72, 0xcb, 0x64, 0x8b, 0x9f, 0xc1, 0x1f, 0x9a, 0x9e, 0x99, 0xcc, 0x39, 0xec, 0xd9, 0xbb, 0xd9, 0xce, 0xc2, 0x74, 0x6f, 0xd0, 0x2a, 0xb9, 0xc6, 0xe3, 0xf5, 0xe7, 0xf4 } },
	{ { 0x02, 0x3a, 0x7e, 0x0c, 0x6d, 0x96, 0x3c, 0x5d, 0x44, 0x56, 0x5d, 0xc1, 0x49, 0x94, 0x35, 0x12, 0x9d, 0xff, 0x8a, 0x5d, 0x91, 0x74, 0xa8, 0x15, 0xee, 0x5d, 0x1e, 0x72, 0xbe, 0x86, 0x15, 0x68,
		0xe7, 0x36, 0xa2, 0x4a, 0xb8, 0xa2, 0xa4, 0x4c, 0xd8, 0x95, 0xe3, 0xc7, 0xbb, 0x32, 0x21, 0x90, 0x64, 0x52, 0x32, 0xeb, 0x26, 0xd3, 0x4f, 0xf0, 0x8e, 0x27, 0x40, 0xea, 0xed, 0xdb, 0xf5, 0xc4 },
	  { 0x76, 0x99, 0x64, 0x70, 0xf4, 0x50, 0xc8, 0xcc, 0x4a, 0x5a, 0xa5, 0x0f, 0xeb, 0x2d, 0xc7, 0x0e, 0x73, 0xd0, 0x65, 0x7d, 0xc3, 0xce, 0x73, 0x03, 0x20, 0x2f, 0xad, 0x65, 0xfd, 0x12, 0xe4, 0x7f,
		0xfd, 0x45, 0x3a, 0x6e, 0xc5, 0x9a, 0x06, 0x67, 0x0e, 0xa6, 0x7b, 0x21, 0x49, 0x2d, 0x01, 0x1b, 0x8e, 0x03, 0x6e, 0x10, 0x08, 0x0c, 0x68, 0xd9, 0x60, 0x47, 0xa4, 0xe2, 0x52, 0xfd, 0x3c, 0xf4 },
	  { 0xa3, 0xe2, 0x5f, 0x16, 0x39, 0x78, 0x96, 0xf7, 0x47, 0x6f, 0x93, 0x5d, 0x27, 0x7b, 0x58, 0xe0, 0xc5, 0xdb, 0x71, 0x7d, 0xa9, 0x6f, 0xf8, 0x8b, 0x69, 0xdd, 0x50, 0xea, 0x91, 0x0d, 0x66, 0x77,
		0xaf, 0x8f, 0xd5, 0x9f, 0x8a, 0x26, 0x69, 0x4c, 0x64, 0x37, 0x62, 0x81, 0x6f, 0x05, 0x9a, 0x08, 0x0d, 0xe1, 0x69, 0x24, 0x77, 0x3f, 0x50, 0xb2, 0x49, 0x4d, 0x93, 0xef, 0x2e, 0x87, 0xff, 0xde },
	  { 0xb3, 0x32, 0xe2, 0x67, 0x79, 0x32, 0x5f, 0x64, 0x47, 0x49, 0x1c, 0xd3, 0x8f, 0x95, 0x44, 0xfd, 0x4c, 0x7e, 0xbf, 0x6b, 0xb7, 0xaf, 0x2c, 0xdd, 0x8f, 0xa5, 0xd8, 0x2f, 0xbf, 0xa0, 0x8a, 0x6b,
		0x58, 0x25, 0xc9, 0x12, 0x23, 0x6f, 0xe6, 0x05, 0xa8, 0xd0, 0x68, 0x6e, 0x0c, 0xee, 0x70, 0xe4, 0xa3, 0x86, 0x51, 0x04, 0x6d, 0xca, 0xd5, 0xed, 0xcf, 0x74, 0x1d, 0x60, 0x9e, 0x86, 0x2d, 0x05 },
	  { 0x91, 0xf4, 0x5f, 0x4a, 0xcb, 0xd8, 0xfd, 0x5f, 0xb9, 0x3d, 0x04, 0xb8, 0xec, 0x35, 0x85, 0x4f, 0x58, 0x20, 0xd1, 0x1f, 0x47, 0xc4, 0xf4, 0xcb, 0x21, 0x4e, 0x9a, 0xf1, 0x6e, 0xbf, 0xe3, 0xd3,
		0x62, 0xe3, 0x82, 0xf6, 0xba, 0xa8, 0xdf, 0x92, 0xe2, 0x3c, 0xe5, 0xf0, 0x16, 0x8a, 0xeb, 0xa4, 0xbb, 0xc7, 0x81, 0xaf, 0x15, 0x19, 0x87, 0x5f, 0xb7, 0xe0, 0x4c, 0x12, 0xff, 0x2c, 0xa9, 0xc8 },
	  { 0xaf, 0x85, 0xe0, 0x36, 0x43, 0xdf, 0x41, 0x17, 0xda, 0xde, 0x5e, 0xb6, 0x33, 0xd0, 0xce, 0x62, 0x70, 0x5f, 0x85, 0x24, 0x6c, 0x3e, 0x1b, 0xe1, 0x52, 0xc1, 0x9b, 0x1c, 0xcd, 0x61, 0x80, 0x9c,
		0xa0, 0xe8, 0x18, 0xee, 0x40, 0x91, 0x93, 0x82, 0xdb, 0x33, 0x44, 0xff, 0xd4, 0xf6, 0x6f, 0x5d, 0xf0, 0x0e, 0x92, 0x92, 0x81, 0x55, 0x46, 0x06, 0xac, 0x58, 0x81, 0x3b, 0x04, 0xc7, 0xf7, 0x0d,
		0xd2, 0x0c, 0x08, 0x6d, 0x46, 0xdb, 0x43, 0x28, 0x31, 0xd8, 0xcd, 0x87, 0x50, 0xbb, 0xd3, 0x07, 0xf5, 0x72, 0x0b, 0x15, 0x7c, 0x16, 0xab, 0x03, 0xd9, 0x4b, 0x07, 0x38, 0x97, 0xe8, 0xd6, 0xb5 },
	  { 0x93, 0xff, 0x6d, 0xc3, 0x62, 0xf7, 0xcc, 0x20, 0x95, 0xc2, 0x2f, 0x7d, 0x1d, 0x9b, 0xd1, 0x63, 0xfc, 0x61, 0x47, 0xb3, 0x22, 0x0f, 0xca, 0xb0, 0x16, 0xcf, 0x29, 0x53, 0x46, 0x97, 0xb1, 0x36,
		0x46, 0xac, 0x48, 0x13, 0x92, 0xe4, 0x46, 0x68, 0xcf, 0x09, 0x4e, 0xfa, 0x59, 0x45, 0x24, 0x08, 0xdb, 0xb4, 0x6f, 0x20, 0x55, 0x12, 0xd9, 0x75, 0x9d, 0x8e, 0x0b, 0xf8, 0x63, 0xe0, 0xf9, 0x01,
		0xd2, 0x0c, 0x08, 0x6d, 0x46, 0xdb, 0x43, 0x28, 0x31, 0xd8, 0xcd, 0x87, 0x50, 0xbb, 0xd3, 0x07, 0xf5, 0x72, 0x0b, 0x15, 0x7c, 0x16, 0xab, 0x03, 0xd9, 0x4b, 0x07, 0x38, 0x97, 0xe8, 0xd6, 0xb5 } },
	{ { 0x14, 0x35, 0xa6, 0x7d, 0xc1, 0xb5, 0x71, 0xca, 0x42, 0x50, 0x90, 0xa7, 0x72, 0x85, 0xbe, 0x78, 0x7a, 0x5f, 0x83, 0x1e, 0xbe, 0xef, 0x6a, 0xbe, 0x48, 0xc5, 0x68, 0x14, 0x0c, 0xf7, 0x44, 0x5c,
		0x2e, 0xfd, 0x1b, 0xcc, 0xee, 0x09, 0x23, 0x82, 0x31, 0xad, 0xaf, 0x4b, 0x73, 0x9c, 0xf2, 0x88, 0x3c, 0xf3, 0xb5, 0x43, 0x8b, 0x53, 0xf9, 0xac, 0x17, 0x86, 0x1c, 0xc2, 0x53, 0x43, 0xec, 0x03 },
	  { 0x7b, 0x36, 0x6c, 0xcc, 0xb5, 0xb2, 0x23, 0x3d, 0x7c, 0xe5, 0xe7, 0xcf, 0x06, 0xe2, 0x32, 0x0b, 0xc5, 0x3b, 0x7f, 0x86, 0x40, 0xfc, 0xaf, 0xba, 0x94, 0xe0, 0x88, 0x58, 0x5b, 0xac, 0xe8, 0xc3,
		0xe8, 0xc3, 0xdf, 0xc4, 0x45, 0x29, 0xe8, 0xf0, 0x1c, 0x10, 0x0d, 0x50, 0x81, 0x29, 0x30, 0xa8, 0x27, 0xb5, 0x3e, 0xb8, 0x25, 0xf1, 0x17, 0x30, 0xc6, 0x05, 0xe3, 0x3e, 0x45, 0x38, 0xa8, 0x3c },
	  { 0xce, 0xd9, 0x45, 0x28, 0xb0, 0xce, 0xa5, 0x47, 0xa8, 0x29, 0x32, 0x76, 0x99, 0x73, 0x8d, 0x74, 0xf9, 0xed, 0x0a, 0xd0, 0xf1, 0xd8, 0x7e, 0x44, 0x63, 0x9e, 0x9a, 0xcf, 0x7c, 0x35, 0x8a, 0x29,
		0xbb, 0x71, 0x66, 0x8d, 0xa7, 0xfc, 0x05, 0x3d, 0xd4, 0x4b, 0x65, 0x20, 0xf5, 0xa4, 0x64, 0xd8, 0x9d, 0x16, 0x80, 0x9c, 0xb2, 0x3c, 0x3e, 0xd4, 0x9d, 0x09, 0x88, 0x8e, 0xbb, 0x58, 0xf8, 0x77 },
	  { 0xe1, 0x29, 0xb3, 0x16, 0xe6, 0xa0, 0xdb, 0x64, 0x08, 0x36, 0xdc, 0x33, 0xad, 0x8b, 0x30, 0x26, 0x17, 0x56, 0xd7, 0x34, 0x17, 0xd1, 0xdd, 0x23, 0x38, 0x58, 0x25, 0x01, 0x42, 0x5a, 0x9d, 0x18,
		0x3e, 0xac, 0x31, 0xfa, 0x43, 0x28, 0xc4, 0x65, 0xfb, 0x30, 0x2f, 0x8c, 0x16, 0x52, 0x32, 0x1b, 0x19, 0xb7, 0x31, 0xf6, 0x67, 0xa7, 0xd8, 0xed, 0x9a, 0xa3, 0x95, 0x01, 0xd7, 0xb9, 0xe7, 0xcc },
	  { 0x81, 0x2d, 0x11, 0xa9, 0x11, 0xf1, 0x22, 0xe2, 0x67, 0x70, 0xc4, 0xba, 0x34, 0xa1, 0x75, 0x8c, 0xf6, 0x0c, 0x63, 0xe7, 0x01, 0x3c, 0x64, 0x6c, 0xe8, 0xd0, 0xf8, 0x8e, 0x88, 0xdf, 0x5c, 0x61,
		0x68, 0x5d, 0x1f, 0xeb, 0x83, 0x1f, 0x40, 0xb8, 0xa8, 0x56, 0x57, 0x26, 0x81, 0x2c, 0xa3, 0x0e, 0x48, 0x4c, 0x45, 0x4d, 0x0d, 0x3d, 0x6e, 0x99, 0x52, 0xbd, 0x0b, 0xd8, 0x05, 0xc5, 0xf9, 0x61 },
	  { 0x92, 0x45, 0xbe, 0xe6, 0xb4, 0x7a, 0xfa, 0x28, 0xd4, 0x5b, 0x6b, 0x17, 0xc6, 0x13, 0x61, 0x5d, 0x5f, 0xd7, 0x90, 0xbb, 0x89, 0x35, 0x7a, 0x02, 0x50, 0x57, 0x56, 0x5f, 0x19, 0xb5, 0xb6, 0xc5,
		0x77, 0x1e, 0x1b, 0xc0, 0xd7, 0x7a, 0x29, 0xbd, 0xe7, 0x24, 0x01, 0x2d, 0x37, 0xc0, 0x38, 0x6f, 0xc8, 0x35, 0xa1, 0x1b, 0xe0, 0xea, 0x16, 0xad, 0xbc, 0xdc, 0xd4, 0x8d, 0x4e, 0x71, 0xdb, 0x05,
		0x9e, 0xb5, 0x53, 0x6b, 0x5c, 0xf1, 0x7d, 0x15, 0x8b, 0xd7, 0xc7, 0x8b, 0x89, 0x9d, 0xfd, 0x28, 0x7c, 0xa1, 0x31, 0xe2, 0xf0, 0x2c, 0x3a, 0x8d, 0x0e, 0x23, 0x85, 0x4e, 0xf0, 0xd1, 0xc0, 0x83 },
	  { 0x7b, 0x88, 0xeb, 0x45, 0x1c, 0x7f, 0xfd, 0xbe, 0xba, 0xac, 0x53, 0x28, 0x59, 0xe8, 0xad, 0x28, 0xf1, 0x97, 0x2d, 0x6c, 0x31, 0xa6, 0xae, 0x47, 0x10, 0x69, 0x68, 0x55, 0xa6, 0x9c, 0x03, 0x62,
		0xb7, 0x2f, 0x31, 0x46, 0x2a, 0x2b, 0x98, 0xdd, 0xe9, 0xf9, 0xfe, 0x77, 0x71, 0x41, 0x54, 0xf8, 0x59, 0x02, 0x7a, 0xe3, 0x45, 0x67, 0xb6, 0xf7, 0x94, 0x31, 0x3e, 0x62, 0x62, 0x2a, 0xf9, 0x0a,
		0x9e, 0xb5, 0x53, 0x6b, 0x5c, 0xf1, 0x7d, 0x15, 0x8b, 0xd7, 0xc7, 0x8b, 0x89, 0x9d, 0xfd, 0x28, 0x7c, 0xa1, 0x31, 0xe2, 0xf0, 0x2c, 0x3a, 0x8d, 0x0e, 0x23, 0x85, 0x4e, 0xf0, 0xd1, 0xc0, 0x83 } },
	{ { 0x27, 0x4d, 0x84, 0x08, 0x95, 0x84, 0xc8, 0xeb, 0x1c, 0x9a, 0x0f, 0xca, 0x09, 0x6f, 0x48, 0x8b, 0x2b, 0x06, 0xa0, 0xae, 0xf2, 0xe3, 0x8a, 0xfe, 0xd7, 0x52, 0x4b, 0xf2, 0xc6, 0x7c, 0xc1, 0x55,
		0x87, 0x2e, 0x5a, 0xb4, 0xc2, 0x43, 0x0a, 0x0d, 0xd0, 0x00, 0xa8, 0xe1, 0x46, 0x68, 0x79, 0xd8, 0x8c, 0x01, 0x36, 0xb7, 0x5a, 0x61, 0x04, 0xe9, 0x7e, 0xbb, 0xc9, 0xee, 0xaa, 0x12, 0x13, 0xda },
	  { 0x78, 0x66, 0xd0, 0xa2, 0x50, 0x82, 0x8d, 0xb0, 0xa0, 0x20, 0xac, 0xa4, 0xb6, 0xa0, 0x31, 0xf7, 0x7d, 0x93, 0x37, 0x67, 0xbb, 0x60, 0xa2, 0x1e, 0x36, 0xce, 0x3d, 0x48, 0x1d, 0x79, 0x99, 0xa5,
		0x19, 0xd8, 0x89, 0x1b, 0xcb, 0x14, 0x87, 0xb7, 0x62, 0xfd, 0xd2, 0xef, 0xbb, 0x13, 0x41, 0x4d, 0xf1, 0x77, 0x5c, 0x7f, 0x6c, 0x3b, 0x94, 0x7d, 0xb4, 0xba, 0x87, 0x3e, 0xc8, 0xe1, 0x3c, 0x0a },
	  { 0xd9, 0x9e, 0x14, 0x89, 0xd6, 0xf8, 0x49, 0xa2, 0xe2, 0x19, 0xfe, 0x94, 0xaa, 0xf7, 0x35, 0xf9, 0x4a, 0xf8, 0xf3, 0x18, 0x68, 0x96, 0x47, 0xc6, 0x23, 0x7c, 0xb0, 0x53, 0xcb, 0xd8, 0x90, 0x31,
		0xb7, 0x50, 0x0e, 0x06, 0xc3, 0x84, 0x75, 0xf1, 0xac, 0x16, 0x4d, 0xc1, 0xbe, 0xf1, 0x80, 0x33, 0x47, 0x56, 0x6f, 0x33, 0x94, 0x5c, 0x81, 0x03, 0x4c, 0x2f, 0x6d, 0xac, 0x73, 0xba, 0x91, 0x3c },
	  { 0x2f, 0xa9, 0xb6, 0xe8, 0x73, 0xe2, 0xef, 0x6d, 0x6d, 0xd7, 0x2e, 0xa0, 0x51, 0x61, 0x24, 0x81, 0x8c, 0xa8, 0x47, 0x40, 0xe1, 0xc7, 0x75, 0x79, 0xc8, 0xec, 0xb2, 0x23, 0x41, 0xad, 0x61, 0x3b,
		0xea, 0x8a, 0xdf, 0x63, 0xed, 0xe1, 0x8e, 0x50, 0x70, 0x6e, 0x86, 0xed, 0xb0, 0xba, 0x27, 0x48, 0x8e, 0xb9, 0x63, 0x39, 0x78, 0x58, 0x4f, 0x1e, 0xbc, 0x45, 0xf3, 0xf2, 0x3a, 0x73, 0x9b, 0x8c },
	  { 0xad, 0x42, 0xc5, 0x84, 0xca, 0xe1, 0xe1, 0x23, 0x2a, 0x73, 0x15, 0x3c, 0x9a, 0xfe, 0x85, 0x8d, 0xa3, 0x2c, 0xcf, 0x46, 0x8d, 0x7f, 0x1c, 0x61, 0xd7, 0x0e, 0xb1, 0xa6, 0xb4, 0xae, 0xab, 0x63,
		0xc4, 0x0e, 0xf2, 0xa0, 0x5d, 0xa6, 0xf3, 0x5d, 0x35, 0x41, 0xea, 0x03, 0x91, 0xb1, 0x3a, 0x07, 0xe6, 0xed, 0x6c, 0x8c, 0xcb, 0x75, 0x27, 0xf1, 0x26, 0x58, 0xf0, 0x62, 0x57, 0xe4, 0x33, 0x00 },
	  { 0x1f, 0xed, 0x53, 0xc6, 0xef, 0x38, 0x26, 0xa4, 0x18, 0x88, 0x8f, 0x5c, 0x49, 0x1c, 0x15, 0x7d, 0x77, 0x90, 0x06, 0x39, 0xe0, 0x7c, 0x25, 0xed, 0x79, 0x05, 0x66, 0xe0, 0x5e, 0x94, 0xe3, 0x46,
		0x6f, 0x96, 0xd8, 0xc1, 0x11, 0xa4, 0x11, 0x6f, 0x78, 0x42, 0x8e, 0x89, 0xc7, 0xc3, 0xed, 0xd2, 0x9e, 0x68, 0x47, 0x79, 0x89, 0x23, 0x70, 0x14, 0x21, 0x60, 0x2d, 0xfe, 0x37, 0x4b, 0xc8, 0x0a,
		0x16, 0x73, 0x7c, 0xc4, 0x55, 0x3f, 0x25, 0x04, 0x08, 0x75, 0x74, 0x68, 0xbc, 0xe4, 0x3a, 0xae, 0x4c, 0x0e, 0xd2, 0x85, 0xa1, 0xbc, 0x81, 0xc0, 0xc9, 0xfe, 0x9a, 0x44, 0x7b, 0x83, 0xdf, 0xc7 },
	  { 0x27, 0x77, 0x97, 0x84, 0x0f, 0x2d, 0x8d, 0x33, 0xb8, 0x4e, 0xdb, 0x8b, 0xea, 0x58, 0x52, 0x88, 0x95, 0x88, 0x55, 0x5f, 0xb8, 0xc4, 0xc9, 0xd6, 0x1f, 0x1e, 0xee, 0x60, 0xb5, 0xeb, 0x78, 0x72,
		0xb5, 0xe5, 0x22, 0x2b, 0x7f, 0x5e, 0xc7, 0x9b, 0x29, 0x55, 0x8e, 0x2a, 0xfc, 0x65, 0x55, 0x4a, 0x02, 0xad, 0x64, 0x06, 0xd4, 0x25, 0xe1, 0x96, 0x6f, 0xee, 0x96, 0xcd, 0x29, 0xc6, 0x64, 0x00,
		0x16, 0x73, 0x7c, 0xc4, 0x55, 0x3f, 0x25, 0x04, 0x08, 0x75, 0x74, 0x68, 0xbc, 0xe4, 0x3a, 0xae, 0x4c, 0x0e, 0xd2, 0x85, 0xa1, 0xbc, 0x81, 0xc0, 0xc9, 0xfe, 0x9a, 0x44, 0x7b, 0x83, 0xdf, 0xc7 } },
	{ { 0x5e, 0xc5, 0x5b, 0x9c, 0xdb, 0x14, 0x05, 0x18, 0x6b, 0xe2, 0x1d, 0x16, 0x77, 0x22, 0x0e, 0xd2, 0xe4, 0x57, 0x82, 0x6e, 0x5b, 0xc5, 0x6a, 0xb9, 0x34, 0x20, 0xdb, 0x72, 0xe2, 0xe1, 0xeb, 0x1b,
		0x34, 0x00, 0x04, 0xbf, 0x83, 0xf6, 0x4f, 0x12, 0x45, 0x08, 0xf0, 0x95, 0x2a, 0xdc, 0x3a, 0x14, 0xb3, 0x29, 0x0b, 0x99, 0xcd, 0x73, 0x31, 0xbd, 0x04, 0xbb, 0x49, 0x1c, 0xde, 0xcf, 0x09, 0x9e },
	  { 0x15, 0x80, 0x3e, 0x2a, 0xfb, 0xc0, 0x8d, 0x62, 0x19, 0x27, 0x83, 0x04, 0xcc, 0xf5, 0xd1, 0xbb, 0x40, 0x41, 0xbe, 0x93, 0x59, 0x6e, 0x27, 0x6d, 0x95, 0x24, 0x0a, 0x07, 0x27, 0x86, 0x10, 0x75,
		0xf7, 0x0a, 0x11, 0xfc, 0x53, 0xd0, 0x4c, 0x15, 0xf8, 0x6e, 0x22, 0x3f, 0xeb, 0x12, 0x97, 0x8a, 0x3d, 0x69, 0xd8, 0x96, 0xc9, 0x53, 0x10, 0x9c, 0x02, 0x95, 0xe4, 0xd3, 0x1a, 0xd5, 0x43, 0x82 },
	  { 0x40, 0x09, 0x2c, 0x17, 0x7e, 0xba, 0xce, 0x1f, 0xfc, 0xc1, 0x8e, 0xc3, 0x1c, 0xa2, 0x34, 0x52, 0x78, 0x16, 0x23, 0x71, 0x82, 0x40, 0xf8, 0x6d, 0x67, 0x65, 0x67, 0x50, 0x53, 0xd9, 0xc8, 0x5e,
		0x7e, 0x8a, 0x98, 0xa3, 0xc6, 0x2a, 0x4d, 0x27, 0xf3, 0xb9, 0xbb, 0xae, 0x43, 0x29, 0x6e, 0x02, 0x1c, 0xe9, 0x01, 0xd6, 0xcd, 0xd8, 0x91, 0x44, 0x95, 0x2b, 0x9e, 0xa5, 0x4f, 0xd0, 0x00, 0xb9 },
	  { 0x3a, 0xe8, 0x3d, 0xb3, 0x32, 0xdc, 0xc2, 0xc8, 0xe3, 0x36, 0x2f, 0xc9, 0x30, 0x3a, 0xc0, 0x76, 0x56, 0xd3, 0x0b, 0x06, 0xbe, 0x8f, 0xe7, 0xf1, 0x66, 0x61, 0x25, 0x42, 0x28, 0xdc, 0x08, 0x81,
		0x84, 0x3a, 0x57, 0x96, 0x27, 0xa6, 0xcf, 0xd6, 0x8f, 0x35, 0xa2, 0xc3, 0x76, 0x86, 0x4f, 0xcf, 0x5f, 0xa1, 0x85, 0x28, 0x4f, 0x4a, 0x3a, 0xbb, 0x5c, 0x25, 0x4b, 0xcc, 0x46, 0xfe, 0xf2, 0x04 },
	  { 0x62, 0xc8, 0xa2, 0x0a, 0x59, 0xb8, 0x97, 0xd2, 0x68, 0x94, 0x00, 0x3b, 0x01, 0xac, 0x91, 0x6e, 0x97, 0x8e, 0x08, 0xe3, 0xfe, 0x9f, 0x9e, 0x9f, 0x4b, 0xcc, 0x5d, 0x1d, 0xb9, 0xbf, 0x07, 0x83,
		0xfe, 0x51, 0x2a, 0xdf, 0x79, 0x2e, 0x07, 0xc9, 0x98, 0x9b, 0xbe, 0xb6, 0xe4, 0x0a, 0x20, 0x44, 0x86, 0xea, 0xb1, 0x61, 0x58, 0x11, 0x32, 0x8e, 0x7b, 0xb9, 0x67, 0x2d, 0xf0, 0x78, 0xb2, 0x93 },
	  { 0x1a, 0x65, 0xb3, 0x6f, 0xa2, 0x45, 0x29, 0x53, 0xd7, 0x23, 0x4d, 0xff, 0x8e, 0xe9, 0xb9, 0xef, 0x16, 0xa0, 0xdd, 0x48, 0xdf, 0x70, 0xd2, 0xe1, 0x56, 0xca, 0xd1, 0xd0, 0x4a, 0x9d, 0x63, 0x92,
		0x2b, 0xfd, 0x7b, 0x87, 0x39, 0x3c, 0x12, 0xc7, 0xe5, 0x91, 0x31, 0x95, 0x78, 0xc4, 0x58, 0x95, 0x89, 0x6e, 0x2c, 0x90, 0xb4, 0x0b, 0xb2, 0xfe, 0x52, 0xc0, 0x86, 0xc4, 0x2e, 0x56, 0x97, 0x0c,
		0x20, 0xf2, 0xbc, 0x6a, 0x9b, 0x89, 0xfb, 0xe9, 0x85, 0x95, 0xd6, 0x22, 0x5e, 0x4d, 0x6d, 0x83, 0x9d, 0xf4, 0xbe, 0x66, 0x05, 0x32, 0xb6, 0xe2, 0xf1, 0x96, 0x42, 0xa4, 0xc8, 0x8c, 0x1b, 0xec },
	  { 0x43, 0x85, 0xff, 0xb9, 0xcf, 0x04, 0x83, 0x40, 0x70, 0x3a, 0x9c, 0x48, 0xb4, 0xc2, 0x99, 0x3b, 0xa0, 0x39, 0xf1, 0x39, 0x58, 0x7f, 0xd2, 0x49, 0x94, 0x3c, 0xc3, 0xe1, 0xb6, 0x56, 0x38, 0x55,
		0x6f, 0xb5, 0x1a, 0x90, 0xa2, 0x04, 0x2f, 0x19, 0xf8, 0xb1, 0x65, 0x5a, 0xad, 0xcd, 0x1c, 0x56, 0x42, 0x38, 0xc2, 0x52, 0x09, 0xd6, 0x41, 0x98, 0x5d, 0x5f, 0xa5, 0xe7, 0xc2, 0x55, 0xa1, 0x09,
		0x20, 0xf2, 0xbc, 0x6a, 0x9b, 0x89, 0xfb, 0xe9, 0x85, 0x95, 0xd6, 0x22, 0x5e, 0x4d, 0x6d, 0x83, 0x9d, 0xf4, 0xbe, 0x66, 0x05, 0x32, 0xb6, 0xe2, 0xf1, 0x96, 0x42, 0xa4, 0xc8, 0x8c, 0x1b, 0xec } },
	{ { 0xf2, 0x4a, 0x96, 0x57, 0xc3, 0x2f, 0xe6, 0x9f, 0xed, 0x7f, 0xcc, 0xe9, 0xea, 0xbe, 0xd2, 0x23, 0x4e, 0x47, 0x13, 0xd9, 0x53, 0x19, 0x31, 0x14, 0x0a, 0xd3, 0x9b, 0x95, 0xa7, 0x9c, 0x88, 0x5e,
		0x08, 0xb2, 0x16, 0xda, 0x45, 0x61, 0x1d, 0x6b, 0xdf, 0xb1, 0x14, 0x0c, 0x66, 0xfd, 0x3a, 0xbe, 0x25, 0xdc, 0xfd, 0xcd, 0xcc, 0x5e, 0x28, 0x77, 0x5a, 0xa9, 0x8b, 0x84, 0x77, 0x26, 0x9d, 0xa6 },
	  { 0xea, 0xde, 0x4d, 0xab, 0x09, 0x02, 0xbf, 0x90, 0xf8, 0xae, 0x8b, 0x50, 0x01, 0xb2, 0x9d, 0x7c, 0x0a, 0x3b, 0x60, 0xda, 0x34, 0xa9, 0xbb, 0x4d, 0xa5, 0x53, 0x18, 0x65, 0xec, 0xaa, 0xc9, 0x29,
		0xb2, 0xf7, 0x74, 0x14, 0x63, 0x5f, 0x88, 0xcf, 0x4e, 0x70, 0x1b, 0x11, 0x64, 0x73, 0x15, 0x6b, 0x5a, 0x8c, 0xb8, 0x4e, 0x0f, 0x83, 0xae, 0x4b, 0x5c, 0x52, 0x1c, 0x6a, 0x0f, 0x54, 0x77, 0xc8 },
	  { 0xae, 0xff, 0x55, 0xbf, 0x78, 0xb5, 0xde, 0x33, 0xeb, 0x87, 0xea, 0x13, 0x7d, 0x36, 0x22, 0x06, 0x32, 0xc4, 0x7e, 0xca, 0x65, 0x37, 0xcc, 0x83, 0x0e, 0xda, 0x54, 0xb3, 0xd2, 0xe6, 0xe7, 0x7f,
		0xe1, 0x90, 0x11, 0x25, 0x16, 0x83, 0x25, 0x43, 0xb4, 0x38, 0x06, 0xbb, 0x6c, 0x62, 0x7d, 0x84, 0x1f, 0xf3, 0x7b, 0xeb, 0xae, 0x50, 0xd8, 0xfb, 0xb9, 0xf2, 0xf9, 0xc3, 0x6f, 0x59, 0xb7, 0xb0 },
	  { 0x95, 0x15, 0x83, 0x19, 0x56, 0x9c, 0x11, 0xd8, 0x31, 0x87, 0x1d, 0xe3, 0x3f, 0x07, 0x89, 0xb2, 0xcb, 0x81, 0xf0, 0xeb, 0x0b, 0x1e, 0x74, 0x08, 0xa2, 0x4a, 0x0e, 0x82, 0xc6, 0x45, 0x8c, 0x32,
		0xb4, 0x8f, 0xfd, 0x76, 0xeb, 0x5e, 0xc7, 0x62, 0xdc, 0xcb, 0xee, 0xad, 0xcf, 0xcf, 0xea, 0x33, 0x9d, 0xb0, 0x02, 0x64, 0x66, 0x77, 0x14, 0x97, 0x0c, 0x6e, 0x79, 0xe8, 0x58, 0x32, 0x0f, 0xe6 },
	  { 0xcb, 0x2f, 0xaf, 0x53, 0xd8, 0x41, 0x48, 0x41, 0x6f, 0x36, 0x78, 0x80, 0x83, 0x5c, 0x0d, 0x4c, 0x1b, 0xf4, 0x39, 0xe0, 0x34, 0x4f, 0xc2, 0xb2, 0x4e, 0xf0, 0xac, 0xc2, 0xf8, 0x15, 0x7a, 0x81,
		0x9f, 0x46, 0x2b, 0xe3, 0xb9, 0x39, 0x05, 0x89, 0xa2, 0xda, 0x1a, 0x63, 0x51, 0xb4, 0x78, 0x0f, 0xfe, 0x2f, 0x9d, 0xce, 0x99, 0x38, 0xa9, 0x7e, 0xcb, 0x80, 0x57, 0x9f, 0xa2, 0x28, 0x0f, 0x6a },
	  { 0x1b, 0xec, 0x67, 0x50, 0xd1, 0x28, 0x65, 0x55, 0xb8, 0xde, 0x3b, 0x2e, 0x1e, 0x33, 0xd8, 0x1b, 0xba, 0x2e, 0x78, 0x6a, 0xb8, 0x0b, 0x8c, 0xa0, 0x55, 0x34, 0x25, 0x90, 0x9a, 0xe2, 0xf5, 0xaa,
		0x95, 0x0c, 0x6f, 0x2a, 0xb0, 0x92, 0x1d, 0x48, 0x5b, 0x56, 0x8c, 0x82, 0x8f, 0xa7, 0x15, 0x75, 0x26, 0x61, 0x85, 0xc8, 0x7d, 0xda, 0xf5, 0x2a, 0xf3, 0x3c, 0x34, 0xc1, 0x20, 0x67, 0xbb, 0x04,
		0xec, 0x7c, 0xe2, 0xcb, 0x31, 0xcf, 0x23, 0xda, 0x5d, 0x8a, 0x05, 0x00, 0x9b, 0x23, 0x34, 0xd0, 0xed, 0x56, 0x10, 0x0a, 0x90, 0x6b, 0x73, 0x26, 0x6b, 0xf0, 0xd7, 0xbc, 0xd8, 0xc7, 0x89, 0xc8 },
	  { 0x90, 0x43, 0x54, 0x87, 0x44, 0x00, 0x07, 0xca, 0xa8, 0x2b, 0xec, 0x55, 0xa0, 0xd2, 0x8c, 0x07, 0x03, 0xaa, 0x61, 0x1a, 0x7d, 0x0f, 0x90, 0x13, 0x67, 0x99, 0x46, 0x20, 0xcd, 0x70, 0xcb, 0xa7,
		0x96, 0xdf, 0x0c, 0x13, 0xc4, 0x41, 0x11, 0xd6, 0xc3, 0x33, 0x02, 0x96, 0x4f, 0x1d, 0xbd, 0x06, 0xa9, 0xa1, 0x31, 0x0a, 0xc3, 0xdf, 0x6d, 0x52, 0x6c, 0xc6, 0xbe, 0xc5, 0xb6, 0x2a, 0xb1, 0x0f,
		0xec, 0x7c, 0xe2, 0xcb, 0x31, 0xcf, 0x23, 0xda, 0x5d, 0x8a, 0x05, 0x00, 0x9b, 0x23, 0x34, 0xd0, 0xed, 0x56, 0x10, 0x0a, 0x90, 0x6b, 0x73, 0x26, 0x6b, 0xf0, 0xd7, 0xbc, 0xd8, 0xc7, 0x89, 0xc8 } },
	{ { 0x4f, 0x3a, 0xdd, 0x0f, 0xcf, 0x7f, 0x27, 0xda, 0x27, 0xc4, 0xa6, 0x2b, 0x6b, 0xd1, 0x9f, 0x59, 0x73, 0x5f, 0xd4, 0xb7, 0xf0, 0x86, 0x16, 0xc9, 0xdd, 0xa6, 0xf9, 0x9b, 0x17, 0xb2, 0xb9, 0x71,
		0xe7, 0x4c, 0xa1, 0x17, 0x79, 0xe0, 0xcc, 0xae, 0x10, 0xec, 0x28, 0x3a, 0x09, 0xf2, 0x8b, 0x34, 0x9c, 0xac, 0x16, 0x2a, 0xa9, 0x21, 0xe8, 0xa7, 0x18, 0xc0, 0xc4, 0x9f, 0x30, 0xa0, 0x25, 0x62 },
	  { 0x23, 0x4c, 0xd4, 0xae, 0x52, 0x30, 0xf6, 0x64, 0xb9, 0xe1, 0x47, 0xca, 0xf8, 0xf3, 0x3a, 0x6b, 0x8b, 0xf3, 0x29, 0xe2, 0x9b, 0x5d, 0xbb, 0x0a, 0x60, 0x52, 0x03, 0x40, 0x53, 0x5c, 0x9e, 0x35,
		0x03, 0xd4, 0xec, 0xd7, 0x67, 0xf4, 0x92, 0xd2, 0x98, 0x96, 0xf2, 0xa7, 0xf4, 0x25, 0x6a, 0x80, 0x9c, 0x75, 0xc6, 0xf2, 0x1f, 0x67, 0x11, 0x00, 0x0d, 0xda, 0x1e, 0xb2, 0x58, 0xa7, 0x8c, 0x39 },
	  { 0x55, 0x1b, 0x80, 0xbb, 0xf3, 0xc5, 0x1a, 0x84, 0x34, 0xf5, 0x0a, 0x8a, 0x8a, 0xe1, 0x8c, 0xea, 0xa6, 0xfb, 0xd0, 0x26, 0xc9, 0xa2, 0x30, 0x37, 0x3e, 0xba, 0x98, 0xfe, 0x81, 0x8a, 0x52, 0x37,
		0x0b, 0x74, 0x4e, 0x3d, 0x26, 0x8f, 0x82, 0x4b, 0xc0, 0x6a, 0x01, 0x10, 0x91, 0x8f, 0x89, 0xb5, 0x62, 0x3f, 0x1e, 0x70, 0xcc, 0x25, 0x77, 0x39, 0x74, 0x88, 0xdd, 0xbc, 0xbe, 0x72, 0x08, 0x63 },
	  { 0xe2, 0x9a, 0x46, 0xd2, 0x74, 0xdc, 0x0f, 0x8a, 0xa3, 0xbd, 0x20, 0xb7, 0xc7, 0xd9, 0x83, 0x4b, 0x58, 0xa6, 0xe3, 0xbd, 0xc5, 0x00, 0xb6, 0x18, 0x04, 0x25, 0x81, 0xbd, 0x99, 0xb3, 0xb1, 0x2a,
		0x7a, 0x68, 0x6d, 0xe1, 0x3e, 0x23, 0x8d, 0x29, 0x9e, 0x7a, 0x30, 0x56, 0x4c, 0x22, 0xb6, 0xf4, 0x7d, 0x7d, 0x4f, 0xfd, 0x76, 0xa5, 0x9d, 0x05, 0x41, 0x7c, 0x7a, 0x2d, 0x7b, 0xbe, 0xcf, 0x73 },
	  { 0x7b, 0xae, 0x11, 0x86, 0x8a, 0x38, 0xbd, 0x56, 0x3c, 0xf3, 0x3c, 0x9c, 0x49, 0xa4, 0x68, 0x0f, 0x2b, 0xdf, 0xf2, 0xa1, 0xbc, 0xc2, 0xed, 0x08, 0x09, 0x96, 0xd0, 0x7e, 0x9b, 0xe3, 0x0a, 0x72,
		0x13, 0x03, 0xd4, 0x35, 0x0a, 0x94, 0x60, 0x09, 0x4a, 0xaa, 0xca, 0x35, 0x8e, 0xed, 0x12, 0xdd, 0x26, 0x8f, 0xf8, 0xa9, 0xa2, 0x8a, 0x7f, 0xac, 0xf3, 0x09, 0xc7, 0x22, 0xc5, 0x73, 0xec, 0xa0 },
	  { 0xe9, 0xc5, 0x57, 0x0d, 0x85, 0xbf, 0x10, 0xe2, 0xd1, 0xf5, 0xd7, 0x22, 0xe9, 0x6a, 0x67, 0x8d, 0xd3, 0x9f, 0x1a, 0xef, 0x7f, 0xc0, 0x2b, 0xe1, 0xfd, 0x2c, 0xc2, 0x5f, 0x39, 0xf9, 0x34, 0xd0,
		0x87, 0x94, 0x41, 0x8a, 0x65, 0xa5, 0x20, 0x48, 0xa4, 0x20, 0x5f, 0x7a, 0xc7, 0x37, 0x00, 0x60, 0x59, 0x84, 0x2a, 0x1d, 0xff, 0x02, 0xc3, 0xe8, 0x20, 0xaa, 0x39, 0x13, 0xac, 0xf3, 0xd7, 0x05,
		0xbd, 0xef, 0x11, 0x66, 0x71, 0xb8, 0x9f, 0x1e, 0xe5, 0xee, 0x2e, 0x37, 0xfb, 0x34, 0xed, 0xc5, 0xa4, 0x40, 0x6e, 0x38, 0x31, 0x0a, 0x1c, 0xaf, 0x0d, 0xd3, 0x98, 0xac, 0x12, 0x40, 0xea, 0x9c },
	  { 0xc6, 0xcd, 0x7a, 0xbd, 0x14, 0xdb, 0xe4, 0xed, 0xbf, 0x46, 0x70, 0x23, 0xbd, 0xdb, 0xc3, 0xce, 0x60, 0xd5, 0x6b, 0x17, 0x4c, 0x23, 0xfa, 0x78, 0x05, 0xcc, 0x18, 0xed, 0x42, 0x03, 0xa5, 0xb7,
		0xdf, 0x28, 0x0e, 0xd4, 0x5d, 0x31, 0xd8, 0xb9, 0xdc, 0xe9, 0xf6, 0x26, 0xc5, 0xe1, 0xb3, 0x80, 0x0d, 0x62, 0xaf, 0x2d, 0xbd, 0xd6, 0xe4, 0xbb, 0x16, 0x82, 0xc8, 0x13, 0x2a, 0x6f, 0xb9, 0x06,
		0xbd, 0xef, 0x11, 0x66, 0x71, 0xb8, 0x9f, 0x1e, 0xe5, 0xee, 0x2e, 0x37, 0xfb, 0x34, 0xed, 0xc5, 0xa4, 0x40, 0x6e, 0x38, 0x31, 0x0a, 0x1c, 0xaf, 0x0d, 0xd3, 0x98, 0xac, 0x12, 0x40, 0xea, 0x9c } },
	{ { 0x6f, 0x46, 0xcd, 0x96, 0xc4, 0x13, 0xf4, 0x11, 0x62, 0x49, 0x8c, 0x5c, 0x78, 0x27, 0xef, 0xc8, 0xb9, 0xe2, 0x7d, 0xf1, 0x0d, 0x37, 0xf2, 0xfe, 0x85, 0x35, 0x82, 0x60, 0x23, 0xb6, 0x7b, 0x17,
		0xd2, 0x91, 0xef, 0x01, 0x9e, 0x99, 0x35, 0xab, 0xc7, 0xfb, 0xa1, 0xa3, 0x13, 0x44, 0x3f, 0x3c, 0x16, 0xcb, 0xd8, 0xf0, 0xbf, 0x9e, 0x65, 0x4d, 0x07, 0xe0, 0xfd, 0x8e, 0x32, 0x61, 0x95, 0xd5 },
	  { 0xb7, 0x81, 0x16, 0x2f, 0xcb, 0xa4, 0x30, 0x4e, 0x6d, 0xf5, 0xf0, 0x3f, 0xfe, 0xd9, 0x81, 0x20, 0xa6, 0x0e, 0x2b, 0xa8, 0xc5, 0xed, 0x0d, 0x9a, 0x28, 0x9c, 0xe3, 0xa9, 0xb7, 0xbf, 0x87, 0x0f,
		0xa5, 0xf9, 0x33, 0xe7, 0xa6, 0x7f, 0x9b, 0xac, 0xb6, 0xcc, 0xaf, 0xfc, 0xa7, 0x4a, 0x4d, 0x36, 0x39, 0xa9, 0xb6, 0xf5, 0x09, 0xde, 0x8d, 0x37, 0x11, 0x07, 0xd1, 0x8a, 0xf5, 0x7b, 0x66, 0xe1 },
	  { 0xcc, 0xe0, 0x07, 0x62, 0xbe, 0x10, 0x8c, 0x3a, 0xa2, 0x96, 0x5d, 0x11, 0xc7, 0xd5, 0x50, 0xc3, 0xbb, 0x55, 0x21, 0xc5, 0x40, 0x27, 0x7d, 0xdb, 0xad, 0xd2, 0x61, 0x2a, 0x42, 0x5f, 0x94, 0x23,
		0x77, 0x83, 0x3a, 0x99, 0xe8, 0xda, 0x79, 0x8c, 0x1e, 0xa8, 0x44, 0x04, 0xec, 0xf5, 0xd1, 0x55, 0x1e, 0x58, 0xf1, 0x6e, 0x4d, 0x27, 0xa4, 0x91, 0xec, 0x59, 0xc8, 0x17, 0x36, 0x58, 0x2a, 0x1f },
	  { 0x6d, 0xf8, 0x73, 0xa3, 0x38, 0x61, 0x1d, 0x95, 0x09, 0xde, 0xe5, 0x26, 0x1b, 0x15, 0x16, 0xfb, 0xf5, 0x16, 0xa8, 0xf3, 0x9e, 0x3a, 0x6b, 0xb5, 0x8c, 0xee, 0xa8, 0x66, 0x79, 0xc3, 0x9e, 0xb4,
		0xe1, 0xc2, 0x85, 0x0e, 0x86, 0x10, 0x5a, 0x4e, 0x8b, 0x4c, 0x0a, 0x7a, 0xd8, 0x8a, 0x48, 0xf4, 0xa0, 0x79, 0x37, 0xe3, 0xa5, 0x90, 0x05, 0x5e, 0xbd, 0xa1, 0xf6, 0x09, 0x58, 0x9c, 0x6f, 0x09 },
	  { 0x66, 0x47, 0x6d, 0x60, 0x06, 0x2d, 0x90, 0x8f, 0xae, 0x6c, 0x01, 0xe9, 0xb0, 0xf9, 0x6b, 0xa5, 0x4a, 0xe1, 0xdb, 0xd3, 0x64, 0x42, 0x37, 0x5c, 0x11, 0x40, 0x7a, 0xce, 0x4e, 0x83, 0xc3, 0x2c,
		0x2e, 0xd2, 0x67, 0x76, 0xfb, 0x8c, 0x5d, 0xab, 0xe8, 0xb8, 0xd6, 0x2b, 0xf8, 0x86, 0xff, 0x96, 0xf3, 0xa8, 0x0e, 0x2b, 0x1a, 0x68, 0xf5, 0xe4, 0xee, 0x49, 0xa6, 0x8c, 0x41, 0x1f, 0x97, 0xbf },
	  { 0x81, 0x92, 0x4e, 0xc6, 0xab, 0x00, 0xdd, 0xf9, 0xf9, 0xb7, 0xe0, 0x0a, 0xa9, 0x3f, 0x0a, 0xf9, 0x32, 0x73, 0xf6, 0x22, 0xec, 0x95, 0xd9, 0x20, 0x8a, 0x3f, 0xeb, 0x0d, 0xc7, 0x79, 0x6f, 0xb3,
		0x85, 0xf4, 0xe1, 0x11, 0xe1, 0xcc, 0xaa, 0x1b, 0xfd, 0xf3, 0x43, 0xff, 0x66, 0x73, 0x0f, 0x09, 0xcc, 0xa4, 0x6c, 0xb8, 0x2a, 0x0f, 0x53, 0x58, 0x63, 0x32, 0x06, 0xd9, 0x6b, 0x1a, 0x14, 0x04,
		0x85, 0x3f, 0x2f, 0x2b, 0x05, 0xfb, 0xed, 0xe9, 0x08, 0x0d, 0x21, 0x49, 0xc9, 0x79, 0xdf, 0x6f, 0x77, 0x89, 0xd7, 0x74, 0x09, 0x57, 0x1a, 0xd2, 0xa7, 0x43, 0xbf, 0x08, 0x8e, 0x98, 0xbc, 0x2f },
	  { 0xe3, 0xb1, 0xc4, 0x81, 0xe6, 0xec, 0x07, 0x58, 0xa4, 0xcb, 0x7e, 0xd5, 0xae, 0x9d, 0x43, 0xf1, 0xb7, 0xe2, 0x0a, 0x1f, 0xd5, 0xe8, 0x14, 0xba, 0x22, 0xff, 0xb7, 0x20, 0x76, 0x08, 0xdc, 0x9a,
		0x44, 0x4c, 0x1c, 0xcd, 0x38, 0x4d, 0xb5, 0xd8, 0xa9, 0x1b, 0x9d, 0xbb, 0x13, 0x5a, 0x6c, 0xe9, 0x5d, 0xa4, 0x42, 0x0e, 0xde, 0x9a, 0x47, 0x8a, 0x2a, 0x97, 0x42, 0x86, 0x87, 0x98, 0x3f, 0x04,
		0x85, 0x3f, 0x2f, 0x2b, 0x05, 0xfb, 0xed, 0xe9, 0x08, 0x0d, 0x21, 0x49, 0xc9, 0x79, 0xdf, 0x6f, 0x77, 0x89, 0xd7, 0x74, 0x09, 0x57, 0x1a, 0xd2, 0xa7, 0x43, 0xbf, 0x08, 0x8e, 0x98, 0xbc, 0x2f } },
	{ { 0xff, 0xe3, 0x69, 0x7b, 0x62, 0x45, 0x40, 0x5f, 0x1c, 0x49, 0x65, 0xd6, 0xae, 0x24, 0x16, 0x84, 0xfa, 0x69, 0x6c, 0x1f, 0x6c, 0x65, 0xee, 0x52, 0xe9, 0x6c, 0x54, 0xc7, 0x31, 0x9b, 0xc2, 0x74,
		0x4f, 0xc0, 0x16, 0xb8, 0xf8, 0x75, 0x5f, 0x45, 0xb5, 0xf3, 0xa0, 0xd9, 0xbe, 0x25, 0x82, 0xbd, 0x3c, 0x03, 0xe0, 0x14, 0x15, 0x6a, 0xd5, 0x64, 0x08, 0x65, 0x13, 0x33, 0xc2, 0xab, 0xe0, 0x45 },
	  { 0x6f, 0x5a, 0x90, 0x80, 0x25, 0x13, 0xc2, 0xa7, 0xfe, 0x1c, 0xa1, 0x07, 0x81, 0x4b, 0x09, 0xd3, 0xbd, 0xda, 0x55, 0xa8, 0xaa, 0x62, 0x19, 0x03, 0xe9, 0x9f, 0x77, 0xef, 0xff, 0xd4, 0x5e, 0x53,
		0xbc, 0x9d, 0x71, 0xb8, 0xc4, 0xc2, 0x85, 0xb9, 0xb4, 0x3d, 0x95, 0xb8, 0xfd, 0x44, 0xb7, 0xc8, 0x6f, 0x93, 0x15, 0x04, 0x16, 0x7e, 0x01, 0xf2, 0x09, 0x23, 0x96, 0x69, 0xe5, 0x65, 0x52, 0x34 },
	  { 0xaf, 0xfe, 0x4f, 0x34, 0x4e, 0xfe, 0x51, 0xa5, 0xb2, 0xd8, 0x31, 0x74, 0x7b, 0xae, 0xfb, 0xb9, 0x33, 0xc1, 0xdc, 0x66, 0xe6, 0x95, 0x9e, 0xce, 0x77, 0x7d, 0x55, 0x3c, 0xa6, 0x6c, 0x09, 0x23,
		0x5a, 0x1a, 0x5e, 0x1a, 0x41, 0xd3, 0xad, 0x5f, 0x86, 0xd0, 0x14, 0xf5, 0xe0, 0xda, 0xf1, 0xce, 0x19, 0x90, 0x45, 0x0c, 0x4c, 0xb1, 0xd3, 0xc8, 0x4c, 0xdb, 0x7e, 0x49, 0xf5, 0xac, 0xde, 0xff },
	  { 0x1b, 0x9b, 0x6b, 0x30, 0xd3, 0x19, 0x37, 0x83, 0xad, 0x05, 0xca, 0xba, 0x22, 0x85, 0x33, 0x7f, 0x55, 0x60, 0xe3, 0x14, 0x8c, 0x39, 0x87, 0xd1, 0x4c, 0x21, 0x27, 0xa0, 0xae, 0x4a, 0x56, 0x15,
		0x50, 0x6c, 0x99, 0xca, 0xff, 0xde, 0x10, 0xc6, 0x9f, 0x6c, 0x70, 0xd1, 0x66, 0xb4, 0x87, 0xd8, 0xfc, 0x46, 0xf2, 0xcf, 0x0c, 0xd8, 0xc3, 0x14, 0x5d, 0x27, 0xbd, 0xed, 0x32, 0x36, 0x7c, 0xed },
	  { 0x64, 0x6b, 0x74, 0xc7, 0x60, 0x36, 0xc5, 0xe4, 0xb6, 0xde, 0x02, 0x1a, 0x09, 0xaf, 0x65, 0xb1, 0x94, 0xa3, 0xf4, 0x95, 0xf5, 0xb0, 0xef, 0x86, 0xb5, 0x13, 0x26, 0x0b, 0xe8, 0xc5, 0x5c, 0x77,
		0xf5, 0xe6, 0xb6, 0x10, 0x36, 0x87, 0xa3, 0xd2, 0x7c, 0x17, 0x2c, 0xb9, 0xb0, 0x90, 0x9e, 0x8c, 0x0a, 0x7d, 0x73, 0xb2, 0x29, 0xeb, 0xa7, 0x85, 0xd7, 0x04, 0x14, 0xf9, 0x77, 0xb7, 0xf4, 0x89 },
	  { 0x7f, 0x1c, 0x5a, 0x57, 0x14, 0xf6, 0x30, 0x07, 0xf9, 0xfe, 0x42, 0x98, 0xcb, 0x3d, 0xac, 0x04, 0x30, 0x0d, 0xc6, 0xd0, 0x4f, 0x8a, 0xbc, 0xdd, 0x3e, 0xc3, 0xb7, 0x74, 0xc8, 0x3b, 0x1a, 0xcc,
		0x6a, 0x54, 0x9e, 0xb9, 0xbe, 0xf0, 0x7c, 0x35, 0x35, 0x1a, 0x50, 0x4c, 0xc2, 0x38, 0x41, 0x46, 0xc8, 0xc4, 0x81, 0x2b, 0x26, 0x56, 0x6f, 0x8a, 0x9f, 0x74, 0x87, 0xe0, 0x01, 0x82, 0xe2, 0x09,
		0xf3, 0x9a, 0xc5, 0x33, 0x5a, 0x7d, 0xb6, 0xbb, 0xff, 0x20, 0x4d, 0xc1, 0x99, 0x3d, 0xcc, 0x5a, 0xc7, 0xd1, 0xbe, 0x4c, 0xcf, 0xc8, 0x09, 0x79, 0x15, 0x5e, 0x0c, 0xc6, 0x26, 0x36, 0xe6, 0xd9 },
	  { 0x4d, 0x2f, 0x08, 0x84, 0x32, 0xcf, 0xe0, 0x3b, 0xa8, 0x3e, 0xa5, 0xf8, 0x3a, 0xe8, 0xa9, 0x04, 0x5a, 0x74, 0x67, 0xcb, 0x41, 0x22, 0xc5, 0xc4, 0x9a, 0xa5, 0xc1, 0xa7, 0x94, 0x8b, 0xa5, 0x35,
		0x00, 0x00, 0x1a, 0xaf, 0xfb, 0xed, 0x40, 0xb8, 0x2b, 0x28, 0xf1, 0xb1, 0x02, 0xd3, 0x8b, 0xc0, 0x32, 0x4a, 0xa5, 0x0a, 0xa4, 0xc3, 0xbf, 0xb3, 0xf5, 0xb7, 0x65, 0x8e, 0x88, 0xdf, 0xd0, 0x0e,
		0xf3, 0x9a, 0xc5, 0x33, 0x5a, 0x7d, 0xb6, 0xbb, 0xff, 0x20, 0x4d, 0xc1, 0x99, 0x3d, 0xcc, 0x5a, 0xc7, 0xd1, 0xbe, 0x4c, 0xcf, 0xc8, 0x09, 0x79, 0x15, 0x5e, 0x0c, 0xc6, 0x26, 0x36, 0xe6, 0xd9 } },
	{ { 0xc8, 0x8e, 0x1c, 0xea, 0x02, 0x6a, 0xfd, 0x88, 0x8b, 0xa9, 0x9d, 0xdd, 0xba, 0xea, 0x77, 0x30, 0x88, 0x1a, 0x93, 0x49, 0xda, 0x05, 0x18, 0xbb, 0x4a, 0x6a, 0x11, 0xc4, 0x48, 0x72, 0x77, 0x1f,
		0x6e, 0x2b, 0x9a, 0xe3, 0x27, 0xbe, 0xe1, 0x75, 0x32, 0x30, 0xa6, 0x12, 0x26, 0x44, 0xbf, 0xb2, 0xa5, 0x51, 0x0b, 0x48, 0x3a, 0xea, 0xc5, 0xd4, 0x24, 0x3f, 0x4e, 0xe8, 0xe5, 0xc3, 0xfb, 0xc2 },
	  { 0xcb, 0x56, 0x3c, 0x00, 0x28, 0x15, 0x72, 0x16, 0x23, 0x4e, 0x2e, 0x2c, 0x8c, 0xe8, 0x7c, 0x44, 0x82, 0x2a, 0xe0, 0x57, 0xa3, 0x0a, 0xc4, 0x42, 0xb5, 0x07, 0xe1, 0x1b, 0x78, 0x8b, 0x3d, 0x4d,
		0xcb, 0xe4, 0x56, 0x72, 0x0b, 0x85, 0x52, 0xd8, 0x55, 0xe2, 0xcd, 0x38, 0xd2, 0x83, 0xb6, 0x05, 0xd2, 0x9f, 0x63, 0x9e, 0x7f, 0xca, 0xe5, 0x95, 0x36, 0x61, 0x9b, 0xca, 0x09, 0x27, 0x53, 0x82 },
	  { 0x24, 0x67, 0x10, 0xd6, 0x8a, 0x1a, 0x8e, 0xb8, 0x53, 0xef, 0xb7, 0x67, 0x2a, 0xfd, 0xb8, 0xd6, 0xe3, 0xf7, 0x41, 0x95, 0x8c, 0x50, 0xca, 0x1d, 0x21, 0x21, 0x41, 0xd1, 0xef, 0x2d, 0x9b, 0x53,
		0xa9, 0x42, 0xcd, 0xda, 0x6d, 0x12, 0x1b, 0xbd, 0x0a, 0xe1, 0x4d, 0x95, 0xc6, 0xaa, 0x40, 0xfd, 0x98, 0xfb, 0x26, 0x21, 0x5e, 0xaf, 0x8e, 0x6b, 0xc9, 0x36, 0x2c, 0x66, 0x31, 0x24, 0x45, 0x87 },
	  { 0x5e, 0xf9, 0x1d, 0x10, 0xb5, 0x79, 0x1f, 0x80, 0x85, 0x90, 0xc3, 0x7f, 0x2b, 0x73, 0xbf, 0x83, 0x0b, 0x5d, 0x46, 0xae, 0x79, 0xef, 0x09, 0x71, 0x29, 0xfb, 0x83, 0xde, 0x1f, 0xe2, 0xdb, 0x1b,
		0xa2, 0x22, 0xee, 0x50, 0x21, 0x9d, 0x9c, 0x35, 0x14, 0x48, 0x13, 0xa5, 0xd1, 0x68, 0xf4, 0x61, 0x1f, 0xd7, 0xe2, 0xd6, 0x42, 0x1c, 0xdc, 0x58, 0xec, 0x8b, 0x03, 0x6b, 0xdf, 0x64, 0x06, 0x30 },
	  { 0xf9, 0xa6, 0x88, 0x74, 0x07, 0x19, 0x15, 0x38, 0xaf, 0xac, 0x07, 0x10, 0xe0, 0xd9, 0x22, 0xf3, 0x78, 0xb0, 0xbf, 0x60, 0xa3, 0x0f, 0xea, 0x0f, 0xa8, 0x64, 0xa9, 0xa3, 0x82, 0xe1, 0x4c, 0x29,
		0x36, 0x22, 0x6d, 0x43, 0x9c, 0xde, 0x22, 0xbf, 0xc6, 0x85, 0xf7, 0xe9, 0xe0, 0x79, 0x80, 0xfe, 0x9d, 0xd6, 0x24, 0xbd, 0x29, 0xa4, 0x8c, 0x35, 0x21, 0x87, 0x45, 0x7f, 0x88, 0xd9, 0x9a, 0x9d },
	  { 0x49, 0x43, 0x19, 0x14, 0xcc, 0x4a, 0x11, 0x01, 0x05, 0xd1, 0x4e, 0x39, 0x6d, 0xb0, 0x22, 0x65, 0x32, 0x6e, 0x67, 0x04, 0x50, 0x85, 0x53, 0x42, 0x90, 0x2c, 0xc0, 0x63, 0x2f, 0xbd, 0x15, 0x90,
		0x1b, 0x3f, 0x03, 0x90, 0x16, 0x7f, 0x7b, 0x49, 0x74, 0xd0, 0x3d, 0x81, 0x80, 0x1e, 0x9e, 0x2e, 0xa9, 0x13, 0x6a, 0x10, 0x14, 0xc1, 0xfd, 0xf9, 0x25, 0x3a, 0x1d, 0x52, 0x93, 0x0a, 0x77, 0x03,
		0xa2, 0xdd, 0xce, 0x9f, 0x2a, 0x35, 0xc9, 0x93, 0x7c, 0xa2, 0x2c, 0xf6, 0x38, 0x73, 0xb3, 0xab, 0x7f, 0x55, 0xb6, 0x62, 0xa2, 0x8d, 0x6a, 0x3e, 0x88, 0x04, 0x9b, 0xa2, 0x19, 0x64, 0x55, 0x01 },
	  { 0x22, 0x03, 0x49, 0x58, 0x76, 0x3c, 0x85, 0x45, 0x5e, 0x73, 0x78, 0x8f, 0x65, 0xc9, 0x50, 0xf8, 0xd7, 0x16, 0x92, 0xa4, 0xd1, 0x79, 0xce, 0xf3, 0x00, 0x34, 0x38, 0xb8, 0xcc, 0x96, 0x9f, 0xa6,
		0x87, 0x28, 0xcb, 0x19, 0x28, 0xad, 0x83, 0xb5, 0x09, 0x96, 0x54, 0xe8, 0x2a, 0xb9, 0x9b, 0xff, 0x60, 0x85, 0x31, 0x28, 0x62, 0x36, 0xd2, 0x0e, 0xad, 0x2a, 0xe1, 0x84, 0x80, 0xeb, 0x6f, 0x00,
		0xa2, 0xdd, 0xce, 0x9f, 0x2a, 0x35, 0xc9, 0x93, 0x7c, 0xa2, 0x2c, 0xf6, 0x38, 0x73, 0xb3, 0xab, 0x7f, 0x55, 0xb6, 0x62, 0xa2, 0x8d, 0x6a, 0x3e, 0x88, 0x04, 0x9b, 0xa2, 0x19, 0x64, 0x55, 0x01 } },
	{ { 0xeb, 0x18, 0x95, 0x94, 0x5f, 0x15, 0x8c, 0xb8, 0x4d, 0x6e, 0x7d, 0xc0, 0x96, 0x6c, 0x52, 0xa2, 0x5f, 0x43, 0x67, 0xc2, 0x3a, 0x10, 0x5b, 0xf1, 0x8f, 0x21, 0x89, 0x06, 0x77, 0xe9, 0xab, 0x2e,
		0xcd, 0x17, 0x9c, 0x9a, 0xd7, 0x89, 0x7e, 0x53, 0x58, 0x60, 0x9b, 0xce, 0x90, 0xd9, 0x13, 0x2d, 0x78, 0xc4, 0x2c, 0x1c, 0x4c, 0xe8, 0x23, 0x70, 0xff, 0xa0, 0x42, 0x98, 0x25, 0x40, 0xd6, 0xd8 },
	  { 0xb6, 0xfb, 0xdd, 0x5d, 0x35, 0xf2, 0x2b, 0x89, 0xda, 0x8e, 0x90, 0xee, 0x03, 0x4e, 0x75, 0xdb, 0x4c, 0x45, 0xc8, 0x00, 0xde, 0x06, 0x27, 0xde, 0x44, 0xb5, 0x5b, 0xc7, 0x56, 0xc3, 0xf5, 0xbb,
		0xee, 0xa6, 0x21, 0xd4, 0xd9, 0xb9, 0x24, 0x9c, 0x4c, 0xbc, 0x23, 0xe5, 0xeb, 0x05, 0xb6, 0xd0, 0xd0, 0xbf, 0x49, 0x95, 0x01, 0xb4, 0x97, 0xad, 0xb5, 0x71, 0x8d, 0x4b, 0x32, 0xd0, 0xdd, 0x1a },
	  { 0xfd, 0x11, 0xd7, 0xe4, 0x46, 0xcd, 0xd8, 0x44, 0x89, 0x0a, 0xe7, 0x44, 0x59, 0xe9, 0xcf, 0x9f, 0xd6, 0xf1, 0x74, 0x56, 0x04, 0x78, 0xfa, 0x29, 0x46, 0x8a, 0x8d, 0x1b, 0xbe, 0x41, 0x92, 0x1c,
		0x8d, 0x74, 0x01, 0x1b, 0xc1, 0xf8, 0x26, 0xf4, 0xc2, 0x68, 0xc3, 0x23, 0x8c, 0x68, 0x7c, 0x0a, 0xad, 0xdd, 0x50, 0x10, 0xcf, 0xdb, 0x78, 0xc5, 0x79, 0x28, 0x37, 0x63, 0x92, 0x1a, 0x1d, 0xea },
	  { 0xd2, 0x2a, 0xf0, 0x66, 0x15, 0x8b, 0xcb, 0x83, 0xcf, 0x34, 0xa1, 0x33, 0x6b, 0xd5, 0xa8, 0x98, 0x3b, 0xd7, 0x09, 0x0d, 0x70, 0xa5, 0x8a, 0xc0, 0x73, 0xcf, 0xde, 0x59, 0xd5, 0x13, 0x41, 0xd2,
		0x43, 0x8b, 0xb4, 0xc3, 0x5b, 0x6f, 0xf1, 0xed, 0x47, 0x76, 0xe6, 0x5e, 0xb8, 0x2a, 0x7e, 0x20, 0x91, 0xa0, 0x9d, 0xc1, 0xa2, 0x0a, 0x6d, 0x97, 0x7d, 0xeb, 0xe3, 0x64, 0x5f, 0x86, 0xff, 0x3e },
	  { 0x45, 0xd8, 0xdc, 0xe4, 0x3a, 0x3a, 0x44, 0xdc, 0x7f, 0xa8, 0x92, 0x11, 0x1b, 0x4f, 0xfa, 0xcf, 0x21, 0xff, 0xfb, 0x20, 0xb0, 0x02, 0x6d, 0x0e, 0x1c, 0xde, 0xe8, 0x51, 0xd8, 0x2c, 0x72, 0x0e,
		0xbf, 0xf6, 0x9a, 0xd3, 0xd3, 0xfe, 0xfa, 0x98, 0x4e, 0xc2, 0xf0, 0x16, 0xda, 0x39, 0x93, 0xc4, 0xe0, 0x33, 0x9a, 0x43, 0xe8, 0x7a, 0xc5, 0x0f, 0x0b, 0xa4, 0x45, 0xf0, 0x5e, 0x7a, 0xa9, 0x42 },
	  { 0xdb, 0x4e, 0x17, 0x76, 0x8b, 0x3c, 0x98, 0x7f, 0x58, 0x76, 0x97, 0xc9, 0x3f, 0x99, 0x01, 0x05, 0x42, 0x7e, 0xfd, 0x83, 0x99, 0xaa, 0x19, 0xb5, 0x72, 0x4c, 0x69, 0xed, 0x6e, 0x21, 0x79, 0x6e,
		0x3b, 0x71, 0xe5, 0xab, 0x23, 0x84, 0xe7, 0xfe, 0x58, 0x2b, 0x0d, 0x1e, 0x75, 0x7c, 0x29, 0xb3, 0x2d, 0x66, 0xc2, 0x45, 0x88, 0xac, 0x86, 0x29, 0xe4, 0xaa, 0x9e, 0x71, 0xa1, 0x88, 0xf9, 0x06,
		0xda, 0xa3, 0xdd, 0x7b, 0x6c, 0xd9, 0xc9, 0x73, 0xe9, 0x56, 0xd1, 0xee, 0x5b, 0xf9, 0xae, 0xc0, 0x29, 0xbe, 0x20, 0x6c, 0xc7, 0xf9, 0xc5, 0x2d, 0x6d, 0xad, 0x8f, 0x49, 0xf8, 0x17, 0xdb, 0x7a },
	  { 0xb8, 0xb7, 0xec, 0xeb, 0x3e, 0x40, 0x77, 0x6c, 0xab, 0x10, 0xfe, 0x9f, 0xd1, 0x40, 0xfe, 0xd2, 0x88, 0x8e, 0xb0, 0x55, 0xae, 0x75, 0xb1, 0xcc, 0x9d, 0x6c, 0x11, 0x28, 0x95, 0x38, 0x9f, 0xb9,
		0x59, 0xe2, 0x29, 0xc3, 0xbc, 0x09, 0x16, 0x1f, 0x17, 0x9e, 0x15, 0x78, 0x09, 0x61, 0x07, 0x9e, 0xad, 0x67, 0x98, 0xa9, 0x24, 0xff, 0xf9, 0x4b, 0xa2, 0x76, 0x09, 0xa0, 0xd7, 0x1b, 0xed, 0x05,
		0xda, 0xa3, 0xdd, 0x7b, 0x6c, 0xd9, 0xc9, 0x73, 0xe9, 0x56, 0xd1, 0xee, 0x5b, 0xf9, 0xae, 0xc0, 0x29, 0xbe, 0x20, 0x6c, 0xc7, 0xf9, 0xc5, 0x2d, 0x6d, 0xad, 0x8f, 0x49, 0xf8, 0x17, 0xdb, 0x7a } },
	{ { 0xc3, 0x92, 0x4d, 0x01, 0x9c, 0xea, 0x5a, 0x8d, 0xbd, 0x5c, 0x12, 0x58, 0x6d, 0x03, 0x26, 0xbf, 0xa4, 0xdd, 0xf7, 0x26, 0xa4, 0x0d, 0x22, 0xe0, 0xbd, 0xcc, 0x6f, 0x30, 0x9e, 0xf9, 0x4c, 0x1f,
		0x03, 0x52, 0xab, 0x38, 0xe9, 0x9c, 0x08, 0x9c, 0x09, 0xe5, 0x87, 0x5c, 0x24, 0x1a, 0xe2, 0x75, 0xcb, 0x18, 0x8a, 0x63, 0x50, 0xd1, 0x23, 0x45, 0x49, 0x93, 0x40, 0x2c, 0x09, 0xd4, 0xac, 0x39 },
	  { 0xd4, 0xe7, 0xb7, 0x05, 0xfd, 0xd6, 0xf3, 0x57, 0xfb, 0xc2, 0x2f, 0x2c, 0x71, 0x80, 0xf5, 0xc3, 0xa6, 0x0a, 0x23, 0x9d, 0x1d, 0xa8, 0x68, 0x10, 0x8a, 0xfa, 0x68, 0x9d, 0x2b, 0xcf, 0x96, 0xa9,
		0xe6, 0x0e, 0x07, 0x32, 0x23, 0x09, 0x87, 0x16, 0xc5, 0xbb, 0x76, 0x22, 0xfc, 0xb4, 0x59, 0x6d, 0x67, 0xfd, 0x29, 0x51, 0x95, 0x4c, 0xe2, 0x8c, 0x18, 0xab, 0xda, 0x84, 0xc3, 0x62, 0x80, 0x14 },
	  { 0xc9, 0xa1, 0xfe, 0xc3, 0x48, 0x0d, 0xee, 0x54, 0x44, 0xff, 0x9c, 0x46, 0x04, 0x0e, 0x74, 0xda, 0xa4, 0x6a, 0x56, 0x02, 0x5f, 0x76, 0x0e, 0xb5, 0xc1, 0xc9, 0xe9, 0xb2, 0x6e, 0x07, 0x49, 0x0c,
		0xf7, 0x4b, 0xee, 0xd6, 0x0a, 0xad, 0x94, 0x03, 0x58, 0x2d, 0x60, 0x95, 0xf8, 0x16, 0x7b, 0x49, 0x0b, 0x01, 0x66, 0x3e, 0x17, 0x01, 0xe5, 0x54, 0x7d, 0xd7, 0xbb, 0x10, 0xd1, 0xad, 0xad, 0x79 },
	  { 0xb2, 0xd8, 0x10, 0x29, 0xeb, 0xb8, 0x4e, 0x2b, 0x39, 0x85, 0x5c, 0xb3, 0xdc, 0xf5, 0x87, 0xca, 0xca, 0x9c, 0x7a, 0x8c, 0x2b, 0x08, 0xe8, 0x25, 0xe2, 0xcf, 0x70, 0xe2, 0xe6, 0xfb, 0xdb, 0x0c,
		0xc3, 0x0d, 0x71, 0x11, 0x83, 0x65, 0xf2, 0x71, 0x08, 0x1b, 0x32, 0x6e, 0x6c, 0x51, 0x50, 0xf1, 0xf6, 0x4b, 0x54, 0x63, 0x16, 0x7f, 0xfd, 0x80, 0x05, 0x61, 0x63, 0xf1, 0x80, 0x6a, 0x0b, 0xfd },
	  { 0xa7, 0x4b, 0x75, 0x38, 0x90, 0x64, 0x96, 0x7b, 0xda, 0x5e, 0x08, 0x9b, 0x80, 0xc4, 0x72, 0x3f, 0x73, 0xb2, 0xdb, 0xd3, 0x4a, 0xed, 0xa4, 0xdc, 0x5c, 0x79, 0xe5, 0x0f, 0x7a, 0xd3, 0x0c, 0xac,
		0xf9, 0x99, 0x5c, 0x1a, 0x0f, 0xb3, 0x1a, 0x0f, 0x5c, 0xc3, 0x9e, 0x1a, 0x2b, 0xfa, 0xc3, 0xf0, 0x40, 0xe5, 0x5f, 0x36, 0xd2, 0x98, 0x31, 0xa1, 0xaf, 0x18, 0x5f, 0xae, 0x92, 0xf3, 0x9e, 0xc0 },
	  { 0xf9, 0xbf, 0x52, 0xe6, 0xd3, 0xe1, 0x5d, 0xd3, 0x30, 0xf3, 0xa1, 0x0c, 0xc8, 0x5a, 0x97, 0x55, 0xab, 0x67, 0x67, 0xd0, 0x00, 0x62, 0x7b, 0x80, 0x70, 0xbf, 0x24, 0xd0, 0x09, 0x8b, 0x07, 0x77,
		0xeb, 0x3e, 0xf0, 0x5d, 0xdf, 0x7b, 0xa9, 0x7d, 0xa4, 0x6a, 0x0d, 0xf1, 0xac, 0x83, 0x7d, 0x64, 0xb5, 0xf4, 0xc6, 0xc4, 0x12, 0x0c, 0x55, 0x9f, 0x67, 0xbb, 0xd5, 0xe3, 0xd3, 0xdb, 0x17, 0x0f,
		0x90, 0x2f, 0x8f, 0xc9, 0xfd, 0x4e, 0x6c, 0x8b, 0xe6, 0x99, 0xfa, 0xda, 0x8f, 0x1f, 0xe6, 0xc3, 0xeb, 0xd8, 0x14, 0x20, 0xcc, 0x3c, 0x1c, 0x23, 0x77, 0x28, 0x9b, 0x22, 0x9a, 0x5a, 0x0c, 0x43 },
	  { 0xa2, 0x78, 0x37, 0xc9, 0x63, 0xe1, 0x31, 0x36, 0xc2, 0x58, 0xac, 0xca, 0xbb, 0xa2, 0x84, 0xaa, 0xb3, 0x82, 0xe2, 0x19, 0xb7, 0x14, 0x96, 0x27, 0x77, 0xfa, 0xa1, 0x02, 0xaa, 0xff, 0x55, 0x82,
		0xba, 0xc0, 0x38, 0x1a, 0x69, 0x35, 0x48, 0x87, 0xc2, 0xeb, 0x48, 0x08, 0xea, 0xc5, 0x6b, 0xfc, 0x84, 0x60, 0x4e, 0xce, 0xd7, 0xd2, 0x86, 0x8b, 0x76, 0xf3, 0x46, 0xe1, 0x87, 0x1f, 0xff, 0x09,
		0x90, 0x2f, 0x8f, 0xc9, 0xfd, 0x4e, 0x6c, 0x8b, 0xe6, 0x99, 0xfa, 0xda, 0x8f, 0x1f, 0xe6, 0xc3, 0xeb, 0xd8, 0x14, 0x20, 0xcc, 0x3c, 0x1c, 0x23, 0x77, 0x28, 0x9b, 0x22, 0x9a, 0x5a, 0x0c, 0x43 } },
	{ { 0x0e, 0xa6, 0x0c, 0xef, 0x12, 0xd6, 0x7d, 0x71, 0xd4, 0x88, 0x73, 0x86, 0x9a, 0x88, 0x8f, 0x5b, 0xd1, 0xb6, 0x12, 0xc4, 0x93, 0x8b, 0x5f, 0xee, 0xdd, 0x9c, 0x2a, 0x7f, 0x4d, 0xfd, 0xba, 0x00,
		0x09, 0x45, 0x77, 0xd2, 0xcf, 0xcd, 0x3a, 0x6f, 0x27, 0x44, 0xe2, 0x55, 0x3e, 0x79, 0x88, 0x4d, 0x5f, 0x38, 0x34, 0xe8, 0xe7, 0xc6, 0x3a, 0xde, 0xef, 0x99, 0x15, 0xea, 0x88, 0x79, 0xd7, 0xca },
	  { 0xa0, 0x9a, 0x0a, 0x3a, 0x42, 0x35, 0x54, 0x78, 0xb9, 0x82, 0x52, 0xb4, 0xc8, 0x5c, 0x4a, 0x03, 0xa1, 0xb9, 0x27, 0xcc, 0x99, 0xec, 0x03, 0xdf, 0xdd, 0x6e, 0xde, 0xef, 0x8f, 0x7f, 0xdc, 0x5a,
		0xc3, 0xcb, 0x0e, 0xa2, 0x7e, 0x93, 0xe6, 0xdd, 0xbd, 0xf1, 0x1b, 0x03, 0x29, 0x63, 0x72, 0x11, 0x72, 0x3d, 0x24, 0x6f, 0xdf, 0x8e, 0xed, 0xa4, 0xe2, 0x2a, 0x4c, 0x00, 0xe2, 0xc4, 0x55, 0x1b },
	  { 0xb2, 0xf1, 0xff, 0xf6, 0x3a, 0x26, 0xe1, 0x74, 0x52, 0xba, 0xee, 0x28, 0xb6, 0x56, 0x90, 0x59, 0xde, 0x92, 0x5f, 0x84, 0xd1, 0x87, 0xe2, 0x64, 0xce, 0xdc, 0x94, 0x3c, 0xb4, 0xf8, 0x01, 0x0a,
		0x86, 0x2f, 0xfe, 0x79, 0x03, 0x72, 0xfc, 0x26, 0x21, 0xc3, 0x1e, 0xec, 0x63, 0x29, 0x64, 0xcb, 0x5f, 0xcc, 0xb6, 0x78, 0xf7, 0xc8, 0xd1, 0xf8, 0x5c, 0xc4, 0x4b, 0xc0, 0xc3, 0x75, 0x3e, 0x46 },
	  { 0x03, 0x4b, 0xb9, 0xd1, 0x50, 0xa3, 0x79, 0xbe, 0x74, 0xa3, 0xb5, 0xd8, 0x28, 0x1b, 0x6d, 0x72, 0x68, 0x0a, 0x9b, 0x19, 0xc9, 0x13, 0xc4, 0x04, 0x94, 0x0a, 0xcb, 0x72, 0xff, 0x7d, 0xb6, 0x9a,
		0x1c, 0xfd, 0xe4, 0xa3, 0x75, 0x13, 0x57, 0x36, 0xfe, 0x4a, 0xf6, 0xbc, 0xca, 0xd9, 0x34, 0x9b, 0xef, 0x90, 0x02, 0xd9, 0xbd, 0xdd, 0x6f, 0x22, 0x54, 0x36, 0xb2, 0x3f, 0x22, 0x65, 0xef, 0xe7 },
	  { 0x04, 0xd4, 0x43, 0xe8, 0x8c, 0xc4, 0xfb, 0xe5, 0x55, 0xd0, 0xa4, 0xea, 0x20, 0xf8, 0xe1, 0x8f, 0xc2, 0xbc, 0x1f, 0x55, 0xf1, 0x8d, 0xda, 0xc0, 0x85, 0xa4, 0xef, 0x36, 0x97, 0x22, 0x8b, 0x8e,
		0x77, 0x4c, 0x1a, 0xa4, 0xa0, 0x6f, 0xe1, 0xdc, 0x32, 0x47, 0xc4, 0x3a, 0xd8, 0x8a, 0xbd, 0x19, 0x30, 0x1c, 0x96, 0x7a, 0xb2, 0x23, 0x7c, 0x16, 0x03, 0xa7, 0x4f, 0xfd, 0xa6, 0x50, 0xd9, 0xf7 },
	  { 0xdf, 0xc2, 0x59, 0xd2, 0xa9, 0x9b, 0x1e, 0xca, 0xf0, 0x39, 0x2f, 0xf8, 0xc2, 0xf3, 0x91, 0x55, 0x1b, 0xba, 0x81, 0x3a, 0x67, 0x1a, 0xd4, 0xf4, 0xb0, 0x9f, 0xb6, 0x18, 0x38, 0x65, 0x3e, 0x67,
		0xa0, 0x37, 0xc2, 0x9a, 0xc7, 0xee, 0x72, 0x8e, 0x13, 0x64, 0xd1, 0x0a, 0xda, 0xbd, 0x8d, 0xa4, 0x28, 0x55, 0x3a, 0x2c, 0x78, 0x41, 0xc6, 0xfc, 0x1c, 0x0f, 0xf8, 0xd7, 0x5f, 0xe6, 0xde, 0x0b,
		0xd5, 0xc0, 0xaa, 0x2c, 0x5c, 0xac, 0x46, 0xeb, 0xa4, 0x35, 0x2a, 0xab, 0x00, 0x2e, 0xc0, 0x8b, 0x42, 0x65, 0x2f, 0x2f, 0x13, 0x84, 0x60, 0x15, 0xa3, 0x69, 0xee, 0xab, 0x0e, 0x50, 0xbf, 0x5f },
	  { 0xc1, 0xb0, 0xac, 0x4c, 0xfa, 0x62, 0x52, 0x22, 0xae, 0x8c, 0x94, 0x38, 0xd9, 0x6e, 0x10, 0x94, 0xe7, 0xaa, 0xc0, 0x92, 0x93, 0x06, 0x55, 0xf9, 0x2e, 0xd9, 0x10, 0x4d, 0xcb, 0x82, 0x19, 0x1f,
		0x27, 0x16, 0x81, 0xdd, 0xea, 0x7a, 0xa8, 0xce, 0x5a, 0xdd, 0x37, 0x77, 0x24, 0x57, 0xfb, 0x40, 0x3d, 0x1b, 0x48, 0x88, 0xda, 0xce, 0xe8, 0xd2, 0xed, 0xe0, 0x6e, 0x29, 0xeb, 0xdb, 0x95, 0x09,
		0xd5, 0xc0, 0xaa, 0x2c, 0x5c, 0xac, 0x46, 0xeb, 0xa4, 0x35, 0x2a, 0xab, 0x00, 0x2e, 0xc0, 0x8b, 0x42, 0x65, 0x2f, 0x2f, 0x13, 0x84, 0x60, 0x15, 0xa3, 0x69, 0xee, 0xab, 0x0e, 0x50, 0xbf, 0x5f } },
	{ { 0x3a, 0x79, 0x39, 0x60, 0xe9, 0x93, 0xad, 0x78, 0xf9, 0x0b, 0x99, 0x64, 0x71, 0x76, 0xad, 0xdc, 0x63, 0xa3, 0x38, 0xbf, 0x0a, 0x36, 0x22, 0xcf, 0x4f, 0x84, 0x3e, 0x34, 0xaf, 0x0b, 0xd4, 0x5c,
		0xc0, 0xa4, 0x01, 0x7c, 0x07, 0xc3, 0xb4, 0xcb, 0xdb, 0x39, 0xdd, 0x39, 0xc7, 0x5c, 0xbd, 0xcf, 0x61, 0x8b, 0x72, 0x74, 0xd6, 0x85, 0xdc, 0x5c, 0x08, 0x93, 0x6d, 0xe6, 0xf1, 0xeb, 0xb9, 0x7c },
	  { 0x71, 0x12, 0x20, 0xbb, 0x37, 0xa6, 0xd8, 0x71, 0xf7, 0x58, 0xaa, 0xbd, 0x30, 0xfb, 0xac, 0x94, 0x62, 0x45, 0xf0, 0x1a, 0xc3, 0x4a, 0x07, 0x78, 0x6d, 0x17, 0xf5, 0x8d, 0x69, 0x3d, 0x2e, 0x15,
		0x96, 0x48, 0x1a, 0xb0, 0x7e, 0xdd, 0xf5, 0x2d, 0xe1, 0x56, 0xfc, 0xe9, 0x26, 0x91, 0x51, 0xfe, 0x5e, 0x2a, 0xdc, 0x23, 0x89, 0x09, 0x14, 0xe6, 0x17, 0xa9, 0x14, 0x8c, 0x8c, 0xe8, 0xe3, 0x71 },
	  { 0xe4, 0xd0, 0xa7, 0x5a, 0xce, 0x93, 0x1d, 0x55, 0xa2, 0x3d, 0xdd, 0x7e, 0x10, 0x66, 0x6d, 0xc6, 0x5c, 0x87, 0x9f, 0x7a, 0x52, 0x5e, 0x76, 0x3f, 0x09, 0x9e, 0xe5, 0x8e, 0x60, 0x39, 0x5e, 0x3c,
		0x28, 0x31, 0xa4, 0x12, 0x39, 0xfd, 0xba, 0xda, 0xc8, 0x59, 0xdd, 0x5b, 0x26, 0x78, 0x8f, 0x33, 0xd2, 0xc8, 0x22, 0x77, 0x49, 0xcf, 0x34, 0x61, 0xbe, 0x7a, 0xa6, 0x31, 0xbe, 0xe5, 0xab, 0xc2 },
	  { 0x60, 0xf5, 0x52, 0xbd, 0xb1, 0x9e, 0x06, 0xa3, 0x94, 0xad, 0xe0, 0x82, 0x33, 0x7c, 0x41, 0x17, 0x5b, 0x8a, 0xbc, 0x7c, 0xce, 0xd1, 0x7e, 0xfd, 0x39, 0x17, 0xfd, 0x90, 0x5a, 0x53, 0x89, 0x27,
		0x9f, 0x27, 0x7a, 0x08, 0xb2, 0x66, 0xda, 0xb5, 0xbf, 0x3b, 0x80, 0xe2, 0x1a, 0x30, 0x80, 0x45, 0x13, 0xf3, 0x4b, 0x0c, 0x4a, 0xe9, 0x0a, 0x6e, 0xf2, 0x3e, 0xa3, 0x70, 0x3d, 0x89, 0xd3, 0xb2 },
	  { 0x23, 0x41, 0x08, 0x8d, 0xa8, 0x0b, 0x6a, 0xe0, 0x65, 0xb1, 0x42, 0x50, 0x49, 0xdd, 0xd3, 0xe8, 0x89, 0x13, 0x7a, 0x04, 0xf0, 0xd6, 0x2f, 0x6e, 0x73, 0xcd, 0xdc, 0x10, 0xbb, 0x02, 0x6b, 0xa2,
		0x25, 0x58, 0xa3, 0x08, 0x37, 0x7c, 0x8b, 0x1f, 0x4a, 0x81, 0x38, 0x88, 0xbd, 0xf4, 0x4f, 0x24, 0xe8, 0xd6, 0x9f, 0x2f, 0x13, 0xeb, 0x79, 0x60, 0x80, 0x90, 0x52, 0x6b, 0x8e, 0xed, 0xcb, 0x77 },
	  { 0x5b, 0x88, 0x63, 0xaf, 0xf9, 0xe2, 0x44, 0x23, 0xc8, 0x02, 0xe0, 0x22, 0x15, 0x3d, 0x2a, 0xb7, 0x40, 0x76, 0xe8, 0x95, 0xfd, 0xa9, 0xe3, 0x85, 0x94, 0xa3, 0xbb, 0xce, 0x61, 0x19, 0x0d, 0xe2,
		0x95, 0xdf, 0x81, 0x11, 0x53, 0x77, 0xcd, 0xf2, 0xd8, 0x4f, 0xbf, 0x19, 0x6a, 0x3d, 0x4b, 0xda, 0xa4, 0x56, 0xa4, 0xcd, 0x9d, 0x4f, 0x52, 0x53, 0x7d, 0xd8, 0xac, 0xe0, 0xfb, 0x9a, 0x71, 0x0c,
		0x59, 0xf9, 0x0b, 0x03, 0xf1, 0x7b, 0xaf, 0x33, 0xc3, 0xe5, 0x1e, 0x8d, 0x4f, 0xbe, 0x21, 0xed, 0x6b, 0x15, 0xdd, 0xd2, 0xeb, 0x7c, 0xe4, 0x59, 0x6c, 0xf9, 0x91, 0xc1, 0x3a, 0x3a, 0xb6, 0x2b },
	  { 0x5e, 0x54, 0xe5, 0x1b, 0x3d, 0x2c, 0x00, 0x80, 0xdd, 0xe4, 0x10, 0x50, 0x98, 0xb6, 0x0e, 0x3a, 0xf7, 0xde, 0x67, 0x2c, 0x8e, 0x7b, 0xb4, 0x73, 0x0b, 0xc7, 0x12, 0xb0, 0x66, 0x6b, 0x3b, 0x99,
		0xd9, 0x33, 0x78, 0x5f, 0x45, 0xe5, 0xec, 0x15, 0x02, 0xfa, 0x8b, 0x86, 0xfd, 0xe0, 0xb7, 0x84, 0x72, 0xf2, 0x68, 0x5c, 0xd6, 0x2e, 0x37, 0xe9, 0x49, 0x32, 0x2f, 0xcd, 0xcd, 0x1e, 0x99, 0x0f,
		0x59, 0xf9, 0x0b, 0x03, 0xf1, 0x7b, 0xaf, 0x33, 0xc3, 0xe5, 0x1e, 0x8d, 0x4f, 0xbe, 0x21, 0xed, 0x6b, 0x15, 0xdd, 0xd2, 0xeb, 0x7c, 0xe4, 0x59, 0x6c, 0xf9, 0x91, 0xc1, 0x3a, 0x3a, 0xb6, 0x2b } },
	{ { 0xfc, 0xb9, 0x4e, 0x4e, 0x11, 0xfe, 0xe1, 0xc5, 0xc7, 0x49, 0x54, 0xd2, 0x2f, 0x13, 0x34, 0x7c, 0x91, 0x7d, 0x98, 0x43, 0xe4, 0xb7, 0x48, 0xea, 0xe8, 0x26, 0xcb, 0x26, 0x1f, 0xe4, 0x99, 0x10,
		0xb9, 0x34, 0xc2, 0xac, 0xa3, 0x2c, 0xbd, 0x9e, 0x80, 0xd4, 0x12, 0x3b, 0xb3, 0xf0, 0x01, 0xae, 0x91, 0x9f, 0xba, 0x77, 0x32, 0x4d, 0x9d, 0xac, 0x1f, 0x8d, 0xad, 0xa7, 0x46, 0x44, 0x85, 0xfb },
	  { 0x65, 0x05, 0x0b, 0xd2, 0x41, 0xd3, 0x58, 0x2a, 0x14, 0xbc, 0x7b, 0x15, 0x4a, 0x6a, 0x6a, 0x18, 0x71, 0x09, 0x25, 0x33, 0xac, 0x73, 0x53, 0xab, 0xd9, 0x0d, 0x8d, 0xdf, 0x95, 0x59, 0x7e, 0x02,
		0x4c, 0x03, 0x11, 0x5c, 0xdc, 0x80, 0x19, 0xd5, 0x13, 0x66, 0x7f, 0xf7, 0xd7, 0x23, 0x18, 0x40, 0x84, 0x16, 0x6b, 0x52, 0x82, 0x96, 0x05, 0x1b, 0xfa, 0xcb, 0x4b, 0x77, 0x00, 0x12, 0xa0, 0x28 },
	  { 0x13, 0xe0, 0x16, 0x1e, 0x24, 0x24, 0xe9, 0xde, 0x9c, 0x86, 0xa9, 0xcf, 0x02, 0x96, 0xdf, 0x8c, 0x64, 0xcb, 0x3d, 0x7d, 0x8a, 0x2a, 0x73, 0x18, 0x20, 0xc8, 0xb0, 0xac, 0x10, 0xa0, 0x52, 0x0c,
		0x6c, 0x17, 0xd9, 0xbd, 0x3c, 0x3e, 0xe5, 0x0c, 0x4a, 0xdb, 0x59, 0xcc, 0x59, 0x15, 0x08, 0x1e, 0xfe, 0xaa, 0xe3, 0xd6, 0xa1, 0x37, 0xd6, 0xd5, 0x6d, 0x8e, 0xcd, 0x57, 0xa9, 0x81, 0xb3, 0x43 },
	  { 0x46, 0x28, 0x2b, 0xa0, 0xe5, 0xe3, 0xf0, 0x72, 0xa7, 0xbc, 0x8d, 0xec, 0x45, 0x31, 0x6e, 0xdb, 0xb2, 0x4b, 0x20, 0xbf, 0x64, 0x74, 0x26, 0x70, 0x9b, 0xd6, 0xd3, 0x7f, 0x9f, 0xc1, 0x59, 0x03,
		0x2d, 0xda, 0x6f, 0xaa, 0x7c, 0x92, 0xc6, 0xe0, 0xe8, 0xaa, 0x1e, 0x26, 0xf0, 0x1e, 0xcc, 0xef, 0x6d, 0x87, 0x04, 0x3c, 0xed, 0x52, 0x15, 0xb3, 0x9f, 0x01, 0x4e, 0xe3, 0x3c, 0xb6, 0xbb, 0xac },
	  { 0x86, 0x1a, 0x25, 0x8e, 0x41, 0x85, 0xf9, 0xba, 0x98, 0x15, 0xb1, 0xec, 0x50, 0xb4, 0xd0, 0xab, 0x55, 0x54, 0xbb, 0x3b, 0x61, 0xfc, 0x54, 0xf3, 0x09, 0xea, 0xaa, 0x6e, 0xbf, 0x03, 0xc3, 0x58,
		0x1d, 0x24, 0xb5, 0xd5, 0x45, 0x5a, 0x7a, 0x14, 0xc3, 0x6a, 0xa9, 0xd8, 0x6f, 0x41, 0xc3, 0xb4, 0x9a, 0x05, 0x71, 0xbc, 0x23, 0x67, 0xc2, 0xa8, 0xf5, 0x7b, 0x69, 0xa5, 0xe1, 0x7a, 0x35, 0x1d },
	  { 0x3b, 0xf5, 0xa8, 0xc0, 0x2a, 0x7d, 0x85, 0x88, 0xd4, 0xf4, 0x26, 0xd3, 0xf4, 0xe3, 0x52, 0x35, 0x37, 0x06, 0x1e, 0x71, 0xc2, 0x3b, 0x7b, 0xeb, 0xf0, 0x07, 0x30, 0x6b, 0x37, 0x31, 0xb9, 0x27,
		0xd8, 0x0b, 0x17, 0xae, 0xff, 0xd4, 0x7c, 0x59, 0xd7, 0x2d, 0xea, 0xcb, 0x92, 0x2f, 0x93, 0xc7, 0xd7, 0xc3, 0xaf, 0x75, 0x73, 0x6a, 0x3f, 0x89, 0xe5, 0x13, 0x0c, 0x28, 0x47, 0xf4, 0xa4, 0x07,
		0xfb, 0xd9, 0x77, 0xb4, 0x1e, 0xb2, 0x70, 0xca, 0x85, 0x22, 0x58, 0xc6, 0x0b, 0x19, 0xc2, 0xa5, 0xba, 0xc3, 0xc9, 0xb6, 0x4a, 0xdb, 0x7d, 0x4d, 0x66, 0xde, 0xeb, 0x8c, 0x1a, 0x23, 0xb8, 0x4c },
	  { 0x8c, 0x57, 0x0e, 0x9f, 0x0a, 0xb2, 0xf4, 0x07, 0xdd, 0x7b, 0x46, 0xf8, 0xa0, 0xb1, 0x33, 0x4c, 0x2b, 0x1e, 0x1a, 0xe0, 0x28, 0x17, 0x14, 0xba, 0x14, 0x06, 0x40, 0x1f, 0x30, 0x0a, 0x19, 0xcd,
		0xe7, 0xca, 0xfb, 0xdb, 0xb9, 0x76, 0xf8, 0x8a, 0x81, 0x3d, 0x03, 0x86, 0x7e, 0x66, 0x75, 0x1d, 0xec, 0xff, 0x6b, 0xa7, 0xea, 0x4c, 0x8c, 0x60, 0xd2, 0x1f, 0x72, 0x11, 0x4c, 0x5d, 0xeb, 0x01,
		0xfb, 0xd9, 0x77, 0xb4, 0x1e, 0xb2, 0x70, 0xca, 0x85, 0x22, 0x58, 0xc6, 0x0b, 0x19, 0xc2, 0xa5, 0xba, 0xc3, 0xc9, 0xb6, 0x4a, 0xdb, 0x7d, 0x4d, 0x66, 0xde, 0xeb, 0x8c, 0x1a, 0x23, 0xb8, 0x4c } },
	{ { 0x05, 0x64, 0x16, 0x53, 0xbb, 0xb2, 0x6e, 0x81, 0xfc, 0xe6, 0xec, 0xc8, 0x0c, 0xc1, 0x75, 0x59, 0x23, 0xe2, 0x4b, 0xd8, 0x6a, 0x70, 0x34, 0x50, 0x37, 0xc6, 0xc2, 0xbd, 0x27, 0xfd, 0xad, 0x4c,
		0xee, 0xe4, 0xf7, 0xfc, 0x91, 0x05, 0x48, 0x3c, 0xd4, 0x09, 0x78, 0x00, 0xce, 0x15, 0x37, 0xdc, 0xe7, 0xce, 0x48, 0x09, 0x3e, 0x7f, 0x01, 0x9b, 0x03, 0xc8, 0x2f, 0x9b, 0xe6, 0x42, 0xe1, 0x71 },
	  { 0x64, 0xbf, 0x63, 0x91, 0xe5, 0x3e, 0x90, 0x89, 0x96, 0xea, 0x59, 0x51, 0x60, 0x7b, 0x5f, 0xfe, 0x0f, 0x76, 0x86, 0x19, 0x45, 0x82, 0xd9, 0x5e, 0x1a, 0xd1, 0xf6, 0x04, 0xc6, 0xaa, 0x71, 0xda,
		0x80, 0xed, 0x75, 0x51, 0xc8, 0x9a, 0x27, 0x09, 0xc3, 0x50, 0xe4, 0x14, 0xa1, 0xc3, 0xf8, 0x3a, 0x6c, 0x84, 0xff, 0x87, 0xd5, 0xf0, 0xb0, 0x3c, 0x5a, 0x57, 0x14, 0x90, 0xc7, 0x31, 0xf8, 0x47 },
	  { 0x88, 0x7d, 0xcc, 0x81, 0x2b, 0xbb, 0x7e, 0x96, 0xbe, 0x78, 0xe1, 0xb1, 0xf2, 0xed, 0x6f, 0xd8, 0xff, 0xbd, 0x7f, 0x8e, 0xe5, 0xeb, 0x7f, 0x7b, 0xca, 0xaf, 0x9b, 0x08, 0x1a, 0x77, 0x69, 0x1d,
		0xc2, 0xa4, 0x7c, 0x4d, 0xa6, 0x74, 0x8e, 0x33, 0x24, 0xff, 0x43, 0xe1, 0x8c, 0x59, 0xae, 0x5f, 0x95, 0xa4, 0x35, 0x9e, 0x61, 0xb8, 0xcc, 0x4c, 0x87, 0xb9, 0x76, 0x53, 0x20, 0xa3, 0xf3, 0xf5 },
	  { 0x13, 0x2a, 0xcc, 0x07, 0xb1, 0x5f, 0xc7, 0xf1, 0x08, 0x0e, 0x7d, 0x7e, 0x26, 0x56, 0xd8, 0x16, 0x9c, 0xae, 0xac, 0xc4, 0xf5, 0x9c, 0x15, 0x67, 0xae, 0xc4, 0xcc, 0x3f, 0xc0, 0xaf, 0x53, 0x28,
		0x1f, 0x65, 0x14, 0xe5, 0x7f, 0x0c, 0xf5, 0x7a, 0xe3, 0x93, 0xc1, 0xa3, 0xd1, 0x4a, 0x09, 0x7d, 0x24, 0xab, 0x22, 0xc4, 0xc4, 0xce, 0x85, 0x37, 0x86, 0xa8, 0x9c, 0x39, 0x33, 0xba, 0x1b, 0x83 },
	  { 0x6d, 0x3e, 0x92, 0x5a, 0xa8, 0xfa, 0xe6, 0x71, 0x98, 0xa8, 0x82, 0x38, 0xcc, 0xed, 0xd6, 0x92, 0x7e, 0x3e, 0xcb, 0xb2, 0x82, 0x92, 0x7a, 0x56, 0x9e, 0xd6, 0x29, 0x45, 0x42, 0x04, 0x76, 0x82,
		0xa5, 0xfc, 0xd9, 0x0c, 0x12, 0x4c, 0x98, 0x04, 0x2a, 0x3a, 0x98, 0x01, 0xb8, 0x62, 0xe8, 0xe6, 0x7c, 0x51, 0xe3, 0x7d, 0x97, 0xf5, 0x45, 0xb4, 0x13, 0xdf, 0x15, 0x68, 0xc3, 0x00, 0x75, 0x40 },
	  { 0x7e, 0x89, 0x3d, 0x7c, 0x78, 0x36, 0x3c, 0x85, 0xda, 0xb6, 0x9b, 0x6d, 0xbc, 0x52, 0x7d, 0xc6, 0xaa, 0xfd, 0x90, 0x62, 0xe4, 0xc4, 0x1a, 0x5a, 0x2e, 0xa1, 0x57, 0xd7, 0xda, 0x57, 0xf4, 0x58,
		0xc5, 0x23, 0x61, 0x21, 0xe1, 0x93, 0xfa, 0x06, 0x22, 0xed, 0x41, 0x66, 0x24, 0x47, 0xb9, 0xed, 0xc8, 0x84, 0x25, 0x28, 0x39, 0xec, 0xfb, 0x29, 0xa1, 0xcd, 0xe1, 0x9d, 0x02, 0x48, 0x6f, 0x0a,
		0xe2, 0x9f, 0x98, 0xfd, 0x3d, 0x18, 0xa1, 0x24, 0x9c, 0xc6, 0x75, 0xb8, 0x99, 0x76, 0x2a, 0xa4, 0x9e, 0xb1, 0x97, 0x2d, 0x1c, 0x99, 0x65, 0x5f, 0x1f, 0xda, 0x14, 0x4f, 0x10, 0x49, 0xf1, 0x7a },
	  { 0x2c, 0xec, 0x27, 0x63, 0xd2, 0x77, 0x14, 0x2d, 0x01, 0x18, 0x10, 0xe0, 0x23, 0x1b, 0xa2, 0x25, 0x61, 0xd4, 0x52, 0xd9, 0x90, 0xde, 0x97, 0x7e, 0xb8, 0xfa, 0x38, 0x25, 0xf2, 0x91, 0x07, 0x3e,
		0xc4, 0xa9, 0x3e, 0xb5, 0x67, 0x02, 0x28, 0x94, 0x5c, 0x34, 0xa1, 0x0a, 0x5c, 0x54, 0x53, 0xd9, 0xb4, 0xc4, 0x5a, 0x8e, 0x57, 0x18, 0xc3, 0x35, 0xea, 0x47, 0x75, 0xe0, 0x44, 0x01, 0x71, 0x09,
		0xe2, 0x9f, 0x98, 0xfd, 0x3d, 0x18, 0xa1, 0x24, 0x9c, 0xc6, 0x75, 0xb8, 0x99, 0x76, 0x2a, 0xa4, 0x9e, 0xb1, 0x97, 0x2d, 0x1c, 0x99, 0x65, 0x5f, 0x1f, 0xda, 0x14, 0x4f, 0x10, 0x49, 0xf1, 0x7a } },
	{ { 0x41, 0x10, 0xd9, 0x7f, 0xb8, 0x83, 0x9e, 0x42, 0x43, 0x7a, 0xb0, 0x6d, 0xa6, 0xcf, 0xa5, 0x7a, 0x50, 0x93, 0x2d, 0x13, 0x94, 0x37, 0xa8, 0x92, 0x26, 0x1f, 0xad, 0xe0, 0x25, 0x19, 0x91, 0x62,
		0x28, 0xfb, 0x18, 0xbf, 0x89, 0xb0, 0x42, 0x80, 0x14, 0xcd, 0xd2, 0x72, 0x84, 0x1c, 0xfd, 0xe5, 0xc3, 0x71, 0x3c, 0x3f, 0x12, 0x5e, 0xdd, 0x53, 0x39, 0xf6, 0x4b, 0x9f, 0xb3, 0x5c, 0xe3, 0x15 },
	  { 0xd0, 0xc7, 0x18, 0x4d, 0x68, 0x9f, 0xdd, 0xec, 0x81, 0xf8, 0xc6, 0x0e, 0x83, 0x43, 0x23, 0x3d, 0xfc, 0xf3, 0x66, 0x55, 0xa8, 0x65, 0x8b, 0xd7, 0x9b, 0x3c, 0x74, 0x23, 0xcd, 0xae, 0x60, 0xe7,
		0x61, 0xed, 0x2c, 0x7e, 0xe7, 0xa7, 0x63, 0x7d, 0x72, 0x47, 0x6a, 0x33, 0x1c, 0xaa, 0x81, 0xba, 0x6f, 0xd4, 0x00, 0xe7, 0xa9, 0x58, 0xb2, 0xad, 0xee, 0x3f, 0x9c, 0x70, 0xff, 0x2f, 0x13, 0x6f },
	  { 0x56, 0x7b, 0x19, 0x66, 0x42, 0x9a, 0x99, 0x51, 0x23, 0x4f, 0xb6, 0xe7, 0xcf, 0x98, 0xff, 0x20, 0x5a, 0xc3, 0x0e, 0x36, 0xc9, 0xc6, 0x20, 0x25, 0x0c, 0x56, 0x98, 0xfb, 0xbd, 0xd6, 0x66, 0x4f,
		0x6f, 0x94, 0x85, 0x8a, 0x35, 0xf3, 0x50, 0xad, 0x87, 0xde, 0x95, 0x9e, 0xae, 0x2a, 0xd8, 0xdd, 0x78, 0x87, 0x96, 0x2b, 0xe0, 0x12, 0x95, 0xd9, 0x3b, 0xb2, 0x2a, 0x06, 0xe2, 0xf0, 0x06, 0xd4 },
	  { 0x42, 0x24, 0xdd, 0x0a, 0xd1, 0x11, 0x31, 0x7e, 0x56, 0x45, 0xb0, 0x0e, 0x86, 0xc1, 0x5d, 0x8c, 0x03, 0x01, 0xb8, 0x33, 0x20, 0xbd, 0x08, 0x10, 0xe5, 0x70, 0x92, 0x2b, 0x5b, 0x86, 0xd3, 0x50,
		0x4c, 0x1e, 0xe3, 0xd1, 0x2a, 0x4e, 0x40, 0x02, 0x19, 0x0b, 0xf6, 0x91, 0xd9, 0x9e, 0xaa, 0x54, 0x7c, 0x3d, 0xba, 0xc5, 0x5a, 0x9e, 0xb2, 0xbb, 0x4e, 0x0d, 0x5b, 0xdd, 0x90, 0xc9, 0x7b, 0xc2 },
	  { 0x54, 0x95, 0xd5, 0xdc, 0x7e, 0x7e, 0xec, 0xd4, 0x67, 0x08, 0xdc, 0x58, 0xa9, 0x80, 0x8a, 0x03, 0x6a, 0xf8, 0x40, 0xca, 0x0d, 0x5b, 0x6c, 0xe4, 0xc9, 0x71, 0xa5, 0xaf, 0x2a, 0xaa, 0xe8, 0x95,
		0x45, 0xe7, 0xe2, 0xc3, 0x47, 0x84, 0xc6, 0xbe, 0xe5, 0x65, 0xaf, 0xcd, 0x7c, 0x20, 0x5f, 0x8b, 0x19, 0x61, 0xe4, 0xc9, 0xc1, 0x86, 0xa5, 0x6f, 0x96, 0xf3, 0x9c, 0x13, 0x28, 0x1b, 0xcf, 0x07 },
	  { 0xc4, 0x7f, 0xf2, 0x6f, 0xcc, 0x4a, 0xf8, 0xa4, 0x1f, 0x1d, 0x6e, 0x5e, 0x30, 0xb2, 0x99, 0x8f, 0x5d, 0x7c, 0x26, 0x1c, 0x52, 0x6f, 0xd0, 0x33, 0xa7, 0xf8, 0xca, 0x2a, 0xc3, 0x8c, 0xa8, 0xd1,
		0x50, 0x4f, 0xa7, 0xe8, 0xf2, 0x10, 0x4c, 0xcd, 0x8a, 0x31, 0x03, 0xc8, 0x93, 0x2c, 0xd7, 0xe4, 0x21, 0xdb, 0xa2, 0x62, 0x7b, 0x1f, 0x28, 0x14, 0x69, 0x7e, 0x87, 0xac, 0xf9, 0xb4, 0x97, 0x00,
		0x62, 0x86, 0x14, 0xd7, 0xe4, 0x65, 0xdd, 0x9e, 0x1c, 0x64, 0x5f, 0x3e, 0xef, 0xfe, 0xa6, 0x60, 0x68, 0x91, 0x94, 0x8a, 0x1c, 0x89, 0xae, 0xe4, 0xcf, 0x3a, 0xdd, 0xc0, 0xb4, 0x47, 0xe8, 0x8f },
	  { 0x12, 0x80, 0x00, 0xda, 0xce, 0xc4, 0x80, 0x8f, 0xa9, 0xa1, 0x5d, 0x98, 0x7d, 0x2c, 0xb2, 0x9c, 0x71, 0xde, 0x62, 0x89, 0x6a, 0xe1, 0x92, 0xd7, 0x96, 0xdc, 0xcd, 0xc8, 0x08, 0x0e, 0x48, 0xbf,
		0x2a, 0x53, 0x72, 0x90, 0x31, 0x71, 0x49, 0x02, 0xda, 0x4e, 0x19, 0x05, 0x10, 0xcb, 0x41, 0x97, 0x44, 0xdc, 0x2d, 0x1e, 0x48, 0xe5, 0x0e, 0x41, 0x9d, 0x7d, 0x03, 0xa3, 0xe2, 0x65, 0xd4, 0x01,
		0x62, 0x86, 0x14, 0xd7, 0xe4, 0x65, 0xdd, 0x9e, 0x1c, 0x64, 0x5f, 0x3e, 0xef, 0xfe, 0xa6, 0x60, 0x68, 0x91, 0x94, 0x8a, 0x1c, 0x89, 0xae, 0xe4, 0xcf, 0x3a, 0xdd, 0xc0, 0xb4, 0x47, 0xe8, 0x8f } },
	{ { 0x00, 0x4b, 0x0b, 0xf5, 0x1f, 0x07, 0x1e, 0x23, 0xe3, 0x93, 0x7b, 0x31, 0x41, 0x2a, 0x0a, 0x50, 0x35, 0xe2, 0xbb, 0xfe, 0x51, 0x77, 0x6c, 0xc9, 0xc5, 0x13, 0xb9, 0x87, 0x79, 0x65, 0x68, 0x20,
		0xcc, 0x09, 0x90, 0xa9, 0xe4, 0xef, 0x9f, 0x1a, 0xe1, 0x69, 0x76, 0x14, 0x82, 0x42, 0x88, 0x4b, 0xdc, 0xe0, 0x10, 0x22, 0xe2, 0xd6, 0x36, 0x7c, 0x0b, 0xd9, 0x08, 0xea, 0xfa, 0xe4, 0xfd, 0x45 },
	  { 0x57, 0x5c, 0x1e, 0x20, 0xb4, 0xae, 0x9e, 0x9d, 0x04, 0xfb, 0x1a, 0xd7, 0x23, 0xd8, 0x8a, 0x6b, 0x1b, 0xb2, 0xef, 0xa9, 0x06, 0x38, 0xbb, 0x9b, 0x43, 0x2e, 0xf1, 0x81, 0x0b, 0x76, 0xec, 0x20,
		0x46, 0x1b, 0xc4, 0x71, 0x19, 0x3e, 0x79, 0xe8, 0xcf, 0xea, 0xdc, 0x4b, 0x3f, 0x0b, 0xeb, 0x05, 0x13, 0x1a, 0x2c, 0xfe, 0x16, 0xe9, 0xf0, 0xc4, 0x9c, 0x41, 0xab, 0x45, 0x1b, 0xba, 0x05, 0xec },
	  { 0x06, 0x0b, 0x73, 0xec, 0x30, 0x74, 0x0d, 0x8d, 0x13, 0x4b, 0xef, 0xac, 0x3b, 0x05, 0xb6, 0xed, 0x2b, 0x05, 0xd1, 0xa7, 0x65, 0xb0, 0xcb, 0x69, 0x00, 0xeb, 0x47, 0xe3, 0x1c, 0x07, 0x8b, 0x15,
		0xbf, 0x69, 0xff, 0x27, 0xb4, 0xdb, 0x77, 0xaf, 0xe9, 0x9a, 0xfb, 0xb2, 0x28, 0xa4, 0xf9, 0x05, 0xe4, 0x3c, 0x66, 0x56, 0x00, 0x1a, 0x2c, 0x41, 0xf2, 0xe1, 0x11, 0x09, 0xfa, 0xe1, 0x50, 0x49 },
	  { 0xbc, 0x4d, 0x6f, 0x75, 0x79, 0x77, 0x64, 0x6b, 0xec, 0xac, 0x1a, 0x26, 0x73, 0x9c, 0xf3, 0xf1, 0x4d, 0x79, 0xbe, 0x6f, 0x0c, 0x07, 0x22, 0xd1, 0xa1, 0x31, 0x75, 0xa8, 0x9c, 0xb6, 0x00, 0x63,
		0x0d, 0x40, 0x17, 0xec, 0x83, 0xda, 0x82, 0x2c, 0x3b, 0xfd, 0x90, 0xe3, 0xbc, 0xc2, 0x2c, 0xf5, 0x3e, 0x41, 0xe9, 0x98, 0x57, 0xa2, 0xb7, 0xce, 0x5f, 0x31, 0xbb, 0x0b, 0x05, 0x61, 0x0f, 0x55 },
	  { 0xb7, 0xab, 0xb2, 0x84, 0xf1, 0x67, 0x24, 0x16, 0x61, 0xe9, 0x20, 0x33, 0x0b, 0xff, 0x22, 0x61, 0x70, 0xa0, 0x5d, 0xf6, 0xa8, 0x33, 0xc9, 0x30, 0x73, 0xe5, 0x89, 0x36, 0x59, 0xea, 0xa8, 0xe7,
		0x03, 0xf6, 0x14, 0xc1, 0x79, 0xb6, 0x42, 0xa5, 0xc8, 0x6c, 0xb8, 0x94, 0x29, 0x24, 0x00, 0x09, 0xb5, 0x54, 0x3f, 0xe1, 0x6b, 0xfb, 0x4d, 0x2d, 0xa9, 0x9a, 0x02, 0xa1, 0xa5, 0x09, 0xf4, 0xcb },
	  { 0x92, 0xfa, 0x18, 0x84, 0x3e, 0xdb, 0xdf, 0x7d, 0x87, 0xd6, 0x2d, 0x07, 0x05, 0x2c, 0xba, 0xe4, 0x30, 0x76, 0xa2, 0xe8, 0x71, 0x3b, 0x1b, 0x93, 0x5b, 0xce, 0x2e, 0xec, 0x50, 0x6e, 0x4a, 0x0b,
		0x2d, 0xbe, 0xa3, 0x76, 0x92, 0xf8, 0xc8, 0x4a, 0x71, 0x66, 0xec, 0xfa, 0x36, 0xc5, 0xdb, 0xab, 0x99, 0x9c, 0xbf, 0x99, 0x07, 0xe8, 0xfe, 0xf4, 0x2f, 0x90, 0x16, 0x5d, 0xdc, 0xbe, 0xfa, 0x08,
		0x93, 0xde, 0x13, 0xf5, 0x32, 0x45, 0x9a, 0xde, 0xa2, 0x5d, 0xb9, 0xe0, 0x38, 0x4c, 0x6a, 0xcc, 0x13, 0x46, 0x27, 0x28, 0xbf, 0xf8, 0x7a, 0x9c, 0x2e, 0xde, 0x6f, 0xfe, 0xe1, 0x86, 0x41, 0x79 },
	  { 0xa7, 0x32, 0x52, 0x76, 0x4f, 0x3e, 0x1b, 0xab, 0x82, 0x18, 0x14, 0xe7, 0x42, 0x32, 0xb8, 0xa4, 0x98, 0xde, 0xa4, 0xd7, 0xae, 0x42, 0x84, 0xda, 0x71, 0xf7, 0x78, 0x40, 0x56, 0x94, 0x64, 0x49,
		0x34, 0x37, 0xeb, 0xe3, 0x05, 0x4c, 0xb9, 0xbb, 0xce, 0xb2, 0x72, 0xc0, 0x75, 0x1c, 0xc4, 0xd5, 0x1e, 0x3a, 0xc1, 0x43, 0xda, 0xd1, 0x81, 0x82, 0xa9, 0xd5, 0x0e, 0x0a, 0x5e, 0xc2, 0xd7, 0x04,
		0x93, 0xde, 0x13, 0xf5, 0x32, 0x45, 0x9a, 0xde, 0xa2, 0x5d, 0xb9, 0xe0, 0x38, 0x4c, 0x6a, 0xcc, 0x13, 0x46, 0x27, 0x28, 0xbf, 0xf8, 0x7a, 0x9c, 0x2e, 0xde, 0x6f, 0xfe, 0xe1, 0x86, 0x41, 0x79 } },
	{ { 0xa3, 0xdf, 0x4a, 0xfd, 0xe6, 0x74, 0xb8, 0xeb, 0xed, 0xe7, 0x7e, 0xd2, 0xae, 0xf8, 0x40, 0x80, 0x3a, 0x55, 0x58, 0x1d, 0x6b, 0xa4, 0x32, 0x6c, 0x15, 0xbb, 0x67, 0xdf, 0x9e, 0xb5, 0x70, 0x4b,
		0x7f, 0x4d, 0xfe, 0x34, 0x42, 0x0c, 0x4d, 0xe3, 0x97, 0x87, 0x6d, 0x08, 0xe8, 0x4d, 0x8a, 0xa9, 0xbc, 0xbf, 0x1b, 0xb7, 0x66, 0x32, 0xf4, 0x7f, 0x93, 0xca, 0xa4, 0xd2, 0x8f, 0x02, 0x7b, 0xfa },
	  { 0xea, 0xac, 0xdf, 0x25, 0x39, 0xf3, 0x28, 0xb6, 0xbe, 0xa8, 0x4a, 0x32, 0x59, 0x4b, 0x4f, 0xb5, 0xd2, 0xf7, 0xf5, 0x75, 0x43, 0x8b, 0xb3, 0x6a, 0x98, 0x8c, 0x14, 0xc9, 0x3f, 0x7e, 0x5c, 0x05,
		0xf0, 0xeb, 0x1d, 0xc5, 0xe6, 0x1b, 0x5d, 0x7f, 0x38, 0x5d, 0x9a, 0xbe, 0xc8, 0x97, 0x09, 0x65, 0x62, 0x88, 0x99, 0xda, 0x95, 0x13, 0x93, 0xd9, 0xa3, 0x19, 0x0a, 0xa7, 0x4a, 0xb2, 0x81, 0xa4 },
	  { 0x6e, 0x70, 0x65, 0xaa, 0x1b, 0x16, 0xcb, 0xc1, 0x59, 0x6b, 0xc9, 0x4d, 0xd1, 0x0a, 0x9d, 0x8c, 0x76, 0x70, 0x3c, 0xc1, 0xc1, 0x66, 0xa6, 0x9f, 0xfc, 0xca, 0xb0, 0x3f, 0x0e, 0xe9, 0xa9, 0x36,
		0x09, 0x4f, 0x94, 0xf3, 0x32, 0x25, 0x34, 0xf6, 0xe4, 0xf9, 0x0b, 0x0c, 0xe6, 0xe0, 0x6d, 0x9e, 0xa5, 0x52, 0x82, 0x9c, 0xd4, 0x43, 0xa4, 0xd1, 0xd1, 0x63, 0x20, 0xce, 0xbc, 0x4f, 0x43, 0xdc },
	  { 0x35, 0xd6, 0xc1, 0x68, 0xa6, 0xd7, 0xd3, 0x36, 0x82, 0x2a, 0x0f, 0x29, 0x3e, 0xd6, 0x15, 0x29, 0x19, 0x73, 0x14, 0x78, 0x87, 0x86, 0xca, 0x9f, 0x6e, 0x17, 0xea, 0xaf, 0x24, 0x37, 0xd6, 0xb4,
		0xb0, 0xee, 0x84, 0x90, 0x2d, 0x18, 0xbd, 0x26, 0xc3, 0xd4, 0x39, 0x4f, 0x45, 0xfa, 0x2f, 0x70, 0xf2, 0xe2, 0x2a, 0x2a, 0x5c, 0x65, 0x15, 0xcb, 0xaf, 0x92, 0x9a, 0xfc, 0x06, 0xe0, 0x8a, 0x1b },
	  { 0x5d, 0xfa, 0xc0, 0x2b, 0xc3, 0x94, 0x19, 0xb4, 0xd6, 0x13, 0xe3, 0xcf, 0x91, 0xad, 0x8c, 0xe1, 0x97, 0x46, 0xfe, 0xea, 0x74, 0xe0, 0x0c, 0x03, 0xf7, 0x2e, 0x51, 0xa7, 0xf2, 0xbc, 0xce, 0xe8,
		0x6b, 0xfd, 0x2f, 0x54, 0x52, 0x12, 0x00, 0x8d, 0x95, 0x91, 0xc3, 0xf6, 0x25, 0xf8, 0x65, 0x6a, 0x9c, 0x79, 0x6b, 0x71, 0xc0, 0x0c, 0x29, 0xfb, 0xe7, 0x14, 0x9f, 0x2f, 0x1a, 0x07, 0x53, 0x50 },
	  { 0xe9, 0xd4, 0x46, 0x0b, 0x51, 0x3f, 0xf1, 0xbe, 0x0a, 0x23, 0xa5, 0x38, 0xa0, 0xe3, 0x70, 0x14, 0x63, 0xf0, 0x94, 0xbb, 0x1c, 0x4f, 0x23, 0x05, 0x1b, 0x62, 0x40, 0x9b, 0xf9, 0x52, 0x1b, 0x41,
		0x51, 0x57, 0x2a, 0x99, 0x73, 0xda, 0xe1, 0xcf, 0xc5, 0x4c, 0x65, 0x3a, 0xc2, 0x9d, 0x73, 0xda, 0xc9, 0x59, 0xf1, 0xdf, 0xab, 0x2b, 0x27, 0xe1, 0x59, 0x8b, 0xa7, 0x48, 0xf9, 0x36, 0xcb, 0x08,
		0xe3, 0x5e, 0x1d, 0xdd, 0xf9, 0x20, 0x4f, 0x64, 0xa9, 0x26, 0x74, 0x97, 0xf2, 0x2d, 0x31, 0xac, 0x8c, 0x20, 0x77, 0x09, 0xa9, 0x8f, 0xed, 0x23, 0x77, 0x7e, 0xd7, 0x34, 0x93, 0x84, 0xe7, 0xaa },
	  { 0xaa, 0xf7, 0x64, 0xdf, 0x34, 0x59, 0x1c, 0x2c, 0xbc, 0x47, 0x08, 0x6a, 0x25, 0xbf, 0x9d, 0x48, 0x54, 0xcf, 0xa0, 0x6c, 0xfc, 0xd4, 0x10, 0x39, 0x9f, 0x64, 0x46, 0xce, 0xd9, 0x95, 0x28, 0x89,
		0xdf, 0x94, 0x5e, 0x74, 0x0b, 0x55, 0x46, 0x82, 0xd9, 0x3d, 0x82, 0x97, 0x7d, 0xd0, 0x3e, 0xd7, 0xf6, 0x6f, 0xaa, 0x97, 0x3e, 0xdf, 0xa7, 0xde, 0xe3, 0xc5, 0xaf, 0xd3, 0xa0, 0x5a, 0x30, 0x0d,
		0xe3, 0x5e, 0x1d, 0xdd, 0xf9, 0x20, 0x4f, 0x64, 0xa9, 0x26, 0x74, 0x97, 0xf2, 0x2d, 0x31, 0xac, 0x8c, 0x20, 0x77, 0x09, 0xa9, 0x8f, 0xed, 0x23, 0x77, 0x7e, 0xd7, 0x34, 0x93, 0x84, 0xe7, 0xaa } },
	{ { 0x96, 0x4e, 0xf2, 0x1e, 0x3a, 0xe5, 0x77, 0xbf, 0xa7, 0x1c, 0x3d, 0x66, 0x08, 0x06, 0xca, 0x55, 0x43, 0x7a, 0x08, 0xf8, 0xff, 0x55, 0xb3, 0xbc, 0x9a, 0x83, 0x9a, 0x2e, 0xe6, 0x97, 0x14, 0x32,
		0x36, 0x57, 0x5c, 0xa4, 0x04, 0x78, 0xb1, 0x92, 0xf4, 0x23, 0x94, 0xe6, 0x2a, 0xef, 0xd4, 0xe7, 0xc4, 0x02, 0x9f, 0xa9, 0x79, 0x77, 0x61, 0x90, 0xd6, 0xdb, 0x6e, 0x28, 0x7e, 0xc0, 0x1d, 0x70 },
	  { 0xc5, 0xd1, 0x5c, 0x34, 0x15, 0xa9, 0x1e, 0x42, 0x2a, 0x1b, 0x0d, 0xf0, 0x56, 0x83, 0x10, 0xc3, 0xc9, 0x21, 0xfd, 0x05, 0xfa, 0x51, 0x0e, 0x11, 0x28, 0xcc, 0x84, 0xac, 0x35, 0xb5, 0xd8, 0xc8,
		0x5c, 0x80, 0x11, 0x1f, 0x60, 0x1c, 0x72, 0x25, 0x82, 0x45, 0xb5, 0x4f, 0x66, 0x6b, 0x52, 0xb1, 0xf7, 0x28, 0x0f, 0x80, 0x76, 0x44, 0xdc, 0x15, 0x70, 0x39, 0xe9, 0xaf, 0xc7, 0x0a, 0xa0, 0x43 },
	  { 0xff, 0x20, 0x5e, 0x3b, 0x75, 0xe9, 0x38, 0x7c, 0xa3, 0x5c, 0x8b, 0x1a, 0xec, 0x17, 0x8d, 0xf0, 0xef, 0xb3, 0x53, 0x9b, 0x16, 0xa9, 0x44, 0xf9, 0x34, 0x45, 0x13, 0x66, 0x80, 0x24, 0xdc, 0x22,
		0x0e, 0x51, 0x94, 0xed, 0xe6, 0x83, 0x36, 0x32, 0x63, 0x23, 0x1b, 0xf8, 0x78, 0xb4, 0x04, 0x7f, 0x5a, 0x50, 0x54, 0x12, 0x19, 0x04, 0x61, 0xdd, 0x25, 0xf0, 0x48, 0x29, 0x04, 0xc1, 0x44, 0xe2 },
	  { 0x46, 0x32, 0x2d, 0xc7, 0xbc, 0x05, 0x2a, 0xd3, 0xb5, 0xce, 0x7d, 0x47, 0x5e, 0xfc, 0x90, 0x38, 0xef, 0xfa, 0x6f, 0x42, 0xf0, 0x66, 0x05, 0x89, 0x7c, 0x9a, 0xc1, 0xfd, 0xa2, 0xe8, 0xa7, 0x38,
		0x18, 0x6d, 0x7f, 0x9e, 0xfb, 0xbd, 0x06, 0x0c, 0x70, 0xd7, 0x29, 0x10, 0x88, 0x04, 0x9f, 0x24, 0x28, 0x9d, 0xc7, 0x84, 0xdf, 0xb6, 0xec, 0xb2, 0xc7, 0x1b, 0xd1, 0xc1, 0x9d, 0x56, 0xb0, 0x83 },
	  { 0xda, 0xd7, 0x34, 0xee, 0x62, 0x13, 0x8f, 0x47, 0xad, 0xb4, 0x9c, 0x98, 0xe4, 0xc5, 0xb3, 0x29, 0x31, 0x11, 0x64, 0xad, 0xf5, 0x0b, 0x60, 0xe1, 0x0e, 0x18, 0x28, 0x30, 0x3c, 0xa2, 0xe3, 0x29,
		0x89, 0x0a, 0x7e, 0x18, 0xba, 0x30, 0x9e, 0x7d, 0x53, 0xf1, 0x82, 0xd5, 0x27, 0xe5, 0xf3, 0xab, 0x15, 0xcd, 0x62, 0x7e, 0xdf, 0xf0, 0x0e, 0x42, 0xfa, 0x6b, 0x7b, 0x54, 0xd2, 0x74, 0x19, 0x8f },
	  { 0x29, 0x4d, 0x28, 0x80, 0x62, 0xb5, 0x77, 0xbb, 0x69, 0x70, 0xb0, 0xb7, 0x10, 0x2e, 0xed, 0xfc, 0x13, 0x34, 0x93, 0x7f, 0xd8, 0xfc, 0xb5, 0x7b, 0xfe, 0x34, 0x0a, 0xa3, 0x95, 0x5b, 0xb1, 0xa7,
		0xc6, 0xab, 0x82, 0x79, 0x25, 0x23, 0x94, 0x12, 0xa4, 0x34, 0xec, 0x23, 0xca, 0xcb, 0xd0, 0xa3, 0xf9, 0x31, 0x32, 0xce, 0x50, 0x31, 0x73, 0x23, 0x98, 0x94, 0xe3, 0x08, 0xd9, 0x1e, 0xc3, 0x0b,
		0x39, 0xe3, 0x3b, 0xf2, 0xe8, 0xb7, 0x26, 0x28, 0x9d, 0xb3, 0x12, 0x8d, 0x16, 0xca, 0x89, 0x26, 0xa9, 0x1c, 0xa3, 0x1f, 0x36, 0x10, 0x60, 0x6a, 0x29, 0x85, 0xe7, 0x2c, 0xee, 0xc1, 0xb6, 0xae },
	  { 0x68, 0xed, 0x3c, 0x64, 0xe6, 0x87, 0xf0, 0x14, 0x64, 0xfc, 0x38, 0x3a, 0x0f, 0xd9, 0x7a, 0x5b, 0x52, 0x32, 0x10, 0xca, 0xc6, 0x83, 0x0b, 0xae, 0x17, 0x0e, 0xfe, 0x77, 0xe0, 0xe7, 0x83, 0xa1,
		0x2c, 0x78, 0x62, 0x9c, 0x79, 0x08, 0x2b, 0xd4, 0x85, 0x72, 0x27, 0x8d, 0x97, 0x78, 0x62, 0x33, 0x34, 0xeb, 0x5c, 0xde, 0x5d, 0xaa, 0x4d, 0xfa, 0xd1, 0x67, 0xa4, 0xea, 0x45, 0xad, 0xf9, 0x06,
		0x39, 0xe3, 0x3b, 0xf2, 0xe8, 0xb7, 0x26, 0x28, 0x9d, 0xb3, 0x12, 0x8d, 0x16, 0xca, 0x89, 0x26, 0xa9, 0x1c, 0xa3, 0x1f, 0x36, 0x10, 0x60, 0x6a, 0x29, 0x85, 0xe7, 0x2c, 0xee, 0xc1, 0xb6, 0xae } },
	{ { 0xd9, 0x64, 0xb2, 0xe1, 0x9f, 0x0a, 0x35, 0xfc, 0x9f, 0xc3, 0xa5, 0x2a, 0xa3, 0x84, 0xb4, 0xf3, 0x23, 0xc4, 0xf3, 0x5a, 0x9d, 0xf8, 0x7f, 0x35, 0xa9, 0xf5, 0x5b, 0x68, 0xfc, 0x19, 0x69, 0x63,
		0x6a, 0x13, 0x19, 0x32, 0xcc, 0x9d, 0x0c, 0x3c, 0x7d, 0xdd, 0x85, 0x16, 0xa8, 0xd9, 0x2b, 0x75, 0x08, 0x4b, 0x9a, 0xa5, 0x6e, 0xf3, 0xe9, 0xeb, 0xed, 0x5d, 0x2e, 0xfd, 0x2e, 0x0c, 0x60, 0xa2 },
	  { 0x0f, 0xf6, 0x8c, 0x3f, 0x6e, 0xee, 0x56, 0x4f, 0x43, 0x6f, 0x54, 0xbd, 0x7a, 0xe4, 0xbe, 0xa8, 0x77, 0x05, 0x99, 0xe7, 0x9e, 0x59, 0x22, 0x85, 0x9b, 0xc6, 0xe4, 0x2a, 0x61, 0x9c, 0x19, 0xb1,
		0x5a, 0xeb, 0x7a, 0xf8, 0x41, 0x4e, 0xe5, 0x2a, 0xd0, 0xf7, 0x44, 0xf0, 0x16, 0xea, 0x0c, 0x04, 0x19, 0x6c, 0xb6, 0x30, 0x3c, 0x6e, 0x2d, 0x79, 0x9a, 0x8f, 0x08, 0x90, 0x11, 0xf1, 0xc0, 0x4d },
	  { 0x68, 0xe7, 0x1d, 0x40, 0xf1, 0x07, 0xc0, 0xc6, 0xb2, 0x87, 0x9c, 0xa2, 0x19, 0x43, 0x7a, 0xdf, 0x8a, 0x5a, 0x0f, 0xe2, 0x24, 0x97, 0xa0, 0x38, 0x79, 0x20, 0x38, 0xa9, 0x9c, 0x77, 0xc4, 0x37,
		0xa6, 0x02, 0xe0, 0x93, 0x47, 0xa4, 0x55, 0x21, 0xc2, 0x69, 0xbe, 0x09, 0x05, 0xaa, 0x87, 0x28, 0xf1, 0x95, 0x2f, 0xdb, 0xf0, 0xbf, 0xd2, 0x9e, 0x5e, 0x3a, 0xfa, 0xc6, 0x2f, 0x13, 0x09, 0xaf },
	  { 0xe1, 0x9e, 0xc8, 0x4f, 0xc9, 0xdd, 0x61, 0x60, 0x94, 0xbc, 0xd3, 0xd6, 0xde, 0x11, 0x6e, 0xec, 0x84, 0xc4, 0xdd, 0xbe, 0x20, 0x46, 0x6c, 0xef, 0xf6, 0x9d, 0x37, 0x07, 0x53, 0x72, 0x57, 0xf9,
		0x02, 0xb5, 0x64, 0x1f, 0xe2, 0x56, 0xa4, 0x38, 0x6d, 0xa4, 0xed, 0x23, 0x9e, 0xa3, 0xf4, 0x4d, 0x77, 0x52, 0xdc, 0x8c, 0x51, 0xfc, 0x88, 0x18, 0xbc, 0x83, 0x2a, 0xac, 0xc1, 0x1d, 0x3d, 0x59 },
	  { 0x08, 0x4f, 0x78, 0x21, 0xfd, 0x4b, 0x85, 0x86, 0x4e, 0x25, 0xdd, 0x47, 0x60, 0x7f, 0x7e, 0xc6, 0xd3, 0xa1, 0xab, 0x91, 0x3f, 0xeb, 0xf6, 0x40, 0x7e, 0x1b, 0xbd, 0x99, 0x9c, 0x7c, 0x2f, 0x4f,
		0xca, 0x68, 0xa5, 0xf6, 0x8c, 0x1e, 0xcb, 0xb8, 0x76, 0xe2, 0x87, 0x5b, 0x49, 0x68, 0x97, 0x2c, 0x21, 0x5c, 0x7c, 0x93, 0x79, 0x9a, 0x95, 0xa1, 0x3a, 0x49, 0xc9, 0x6d, 0x34, 0x6b, 0xa1, 0x98 },
	  { 0xb9, 0x88, 0x25, 0x9a, 0x3b, 0x53, 0x56, 0xa1, 0x48, 0x0f, 0xf0, 0x92, 0xde, 0x4e, 0x3e, 0x3a, 0xcf, 0x02, 0xdc, 0x5c, 0xc2, 0xc3, 0x78, 0xad, 0x8a, 0x0c, 0x3c, 0xc7, 0xdd, 0xdd, 0x71, 0x6e,
		0x3f, 0xd9, 0x3a, 0x57, 0x2a, 0x19, 0xa5, 0x3b, 0x5c, 0x46, 0x7b, 0xc9, 0x0f, 0x16, 0xb3, 0x58, 0xa6, 0x85, 0xfa, 0x91, 0x2c, 0x9a, 0x9c, 0x12, 0xb6, 0xd6, 0x7d, 0x9a, 0xf0, 0x9d, 0xe9, 0x02,
		0xad, 0x12, 0x87, 0xda, 0x85, 0x58, 0x6b, 0xff, 0x68, 0x96, 0x05, 0x33, 0xba, 0x7f, 0x08, 0xf9, 0xa9, 0xa2, 0xa9, 0x46, 0x43, 0xe5, 0x03, 0x12, 0xe4, 0xbe, 0x74, 0xaa, 0x46, 0x4e, 0x51, 0xb3 },
	  { 0x61, 0x70, 0x17, 0x50, 0x26, 0xfa, 0x51, 0x83, 0xe0, 0xca, 0xa9, 0xb1, 0xc3, 0xc4, 0x83, 0xa9, 0xb6, 0x43, 0x6b, 0x7a, 0x5b, 0xe4, 0x21, 0x5a, 0x6b, 0xd4, 0x34, 0xf8, 0xee, 0x95, 0x86, 0x2d,
		0x03, 0xbf, 0xca, 0xd0, 0xfa, 0x68, 0x53, 0xb2, 0x97, 0x50, 0xad, 0x89, 0x2f, 0x99, 0x63, 0x67, 0x18, 0x57, 0x1f, 0x57, 0x41, 0xbc, 0xb7, 0xc0, 0x18, 0xe7, 0xb6, 0xf3, 0x0f, 0xc4, 0x49, 0x0d,
		0xad, 0x12, 0x87, 0xda, 0x85, 0x58, 0x6b, 0xff, 0x68, 0x96, 0x05, 0x33, 0xba, 0x7f, 0x08, 0xf9, 0xa9, 0xa2, 0xa9, 0x46, 0x43, 0xe5, 0x03, 0x12, 0xe4, 0xbe, 0x74, 0xaa, 0x46, 0x4e, 0x51, 0xb3 } },
	{ { 0xc5, 0xdf, 0x86, 0x8f, 0xf1, 0xa7, 0xad, 0x57, 0xfd, 0xb4, 0x53, 0xc3, 0x92, 0x1b, 0x9e, 0x2e, 0xdd, 0xc5, 0xa4, 0x3b, 0x72, 0xa6, 0x9b, 0x4a, 0x15, 0xca, 0x35, 0xed, 0x3c, 0x1a, 0x3b, 0x38,
		0x36, 0xd6, 0xf2, 0x03, 0xb6, 0x97, 0x1f, 0xcb, 0x40, 0x5d, 0x3c, 0x25, 0xfc, 0xe7, 0xff, 0xc6, 0xbe, 0x61, 0xe1, 0x98, 0x31, 0x13, 0xa9, 0xbe, 0x05, 0x86, 0xfe, 0x5c, 0xf6, 0xcc, 0xaa, 0xf5 },
	  { 0xd2, 0x57, 0x19, 0x98, 0xf8, 0x74, 0x90, 0xb7, 0x69, 0x6e, 0xdd, 0x44, 0xf1, 0x8b, 0xb1, 0x9c, 0xfd, 0x5b, 0x6b, 0xc0, 0x45, 0xf2, 0x49, 0xa5, 0x4b, 0xff, 0x8b, 0x7f, 0x87, 0xe3, 0xf9, 0x71,
		0xab, 0xfa, 0xc8, 0x17, 0xed, 0xeb, 0x19, 0xc6, 0x3c, 0xee, 0x78, 0xba, 0x89, 0x97, 0x49, 0x85, 0x39, 0x68, 0x29, 0x88, 0x0b, 0x1c, 0xd1, 0x42, 0x8b, 0xe8, 0x1a, 0x3b, 0xeb, 0x4d, 0xef, 0x3b },
	  { 0xea, 0xfb, 0xec, 0x27, 0xc3, 0x92, 0xc3, 0x68, 0x0d, 0x3c, 0x5b, 0x20, 0x20, 0x9c, 0x96, 0xa7, 0x39, 0xfa, 0x80, 0x91, 0xef, 0x86, 0x7d, 0xa8, 0x87, 0xf6, 0xef, 0x14, 0x01, 0x46, 0xf0, 0x68,
		0x0a, 0x8b, 0xae, 0x83, 0x91, 0x7e, 0xa0, 0x14, 0x14, 0xde, 0xf9, 0xa8, 0xfd, 0x67, 0x57, 0x17, 0x20, 0x46, 0x43, 0x49, 0x07, 0xf0, 0x3e, 0xc8, 0xbe, 0x66, 0xaf, 0x58, 0x3a, 0xbd, 0xd8, 0x00 },
	  { 0x35, 0xf5, 0xc8, 0x2c, 0x0e, 0x4b, 0x56, 0xe0, 0xef, 0x08, 0x34, 0x38, 0x57, 0xe9, 0xde, 0xdb, 0x1d, 0xe1, 0x28, 0x05, 0x01, 0xed, 0x62, 0x3d, 0xa9, 0x6e, 0xea, 0x5b, 0x95, 0x09, 0xe0, 0x04,
		0x46, 0xff, 0xdc, 0x34, 0xf6, 0xf7, 0x63, 0xb1, 0x76, 0xb8, 0x3c, 0x03, 0xef, 0x36, 0x0f, 0x82, 0x1b, 0x5b, 0x6f, 0xe2, 0x86, 0xd9, 0x10, 0x01, 0xe6, 0x73, 0x75, 0x0d, 0x50, 0x30, 0x11, 0x68 },
	  { 0x27, 0xb6, 0x3b, 0x78, 0x79, 0xf3, 0x22, 0x78, 0x8f, 0x0c, 0x14, 0x8b, 0x3f, 0x68, 0xc2, 0xab, 0x9f, 0x9f, 0x05, 0x70, 0x7e, 0xee, 0x4b, 0x1b, 0x6b, 0xfc, 0x04, 0x72, 0xca, 0xf1, 0x9a, 0xba,
		0xe3, 0x65, 0x9d, 0xdb, 0x01, 0x33, 0xc5, 0xdb, 0xf6, 0x87, 0xe4, 0x73, 0x5a, 0x0f, 0x94, 0xa9, 0x2e, 0xfe, 0x8f, 0x3e, 0xd1, 0x0a, 0x6d, 0xa1, 0x21, 0x2a, 0x92, 0x8c, 0x4b, 0x43, 0x13, 0x2f },
	  { 0xa3, 0xa8, 0x3b, 0xb4, 0x4f, 0x8a, 0xac, 0xab, 0x8a, 0x4c, 0x39, 0x7e, 0xb8, 0x2f, 0xb1, 0x01, 0x2e, 0xbe, 0x0e, 0x7d, 0x28, 0x8a, 0x18, 0x4a, 0xda, 0x58, 0x1a, 0xfb, 0x95, 0x97, 0xf3, 0x63,
		0x58, 0xbe, 0x8c, 0x30, 0x13, 0x9b, 0xba, 0x9f, 0x4e, 0xac, 0x8d, 0x95, 0xf2, 0x07, 0xbb, 0x85, 0xa1, 0x41, 0x4c, 0x33, 0xe3, 0x58, 0x8e, 0x5c, 0xa1, 0x05, 0x45, 0xab, 0x5c, 0x0c, 0xe4, 0x02,
		0xc3, 0xa0, 0xa0, 0x72, 0xdb, 0x9a, 0x9d, 0xbf, 0x13, 0x29, 0x94, 0x70, 0x8b, 0xe4, 0xe8, 0xdb, 0x0e, 0x0b, 0xd0, 0xa0, 0x25, 0xad, 0x71, 0xa0, 0x27, 0x9c, 0x1d, 0x77, 0xb0, 0x98, 0xa8, 0x03 },
	  { 0xe1, 0x84, 0xa5, 0xea, 0xa5, 0xd8, 0x1b, 0x29, 0xce, 0xd7, 0xa3, 0x72, 0xa7, 0xc9, 0xa5, 0xea, 0xf1, 0x02, 0xf3, 0x0c, 0xb0, 0x65, 0x12, 0xbc, 0xa4, 0xf2, 0x5d, 0x69, 0x00, 0xa4, 0x7f, 0x5a,
		0x52, 0x09, 0xb6, 0x7b, 0x30, 0xf2, 0x99, 0x03, 0x39, 0x9d, 0xee, 0x6f, 0xb5, 0xf7, 0x9e, 0x7a, 0x97, 0x8b, 0x81, 0x03, 0x8c, 0xdd, 0x35, 0xfc, 0x1f, 0x0a, 0xc6, 0xa4, 0x60, 0x7b, 0xc8, 0x0a,
		0xc3, 0xa0, 0xa0, 0x72, 0xdb, 0x9a, 0x9d, 0xbf, 0x13, 0x29, 0x94, 0x70, 0x8b, 0xe4, 0xe8, 0xdb, 0x0e, 0x0b, 0xd0, 0xa0, 0x25, 0xad, 0x71, 0xa0, 0x27, 0x9c, 0x1d, 0x77, 0xb0, 0x98, 0xa8, 0x03 } },
	{ { 0x67, 0xe9, 0x62, 0x76, 0x3a, 0x90, 0x9b, 0x6b, 0x19, 0x1d, 0x65, 0xb2, 0x2a, 0x2f, 0xf7, 0x50, 0xaa, 0x54, 0xa5, 0xbb, 0x53, 0xb5, 0xf9, 0xee, 0x0c, 0x04, 0x3a, 0x3c, 0x29, 0x4b, 0x66, 0x3e,
		0x7b, 0xb6, 0xaa, 0xd2, 0x10, 0x89, 0xcc, 0x89, 0x2c, 0x47, 0xbe, 0x23, 0xd6, 0x52, 0x81, 0x5d, 0xc8, 0xbc, 0x49, 0xd6, 0x6a, 0xcd, 0x62, 0x99, 0x30, 0xff, 0x16, 0xa5, 0x50, 0x44, 0xd8, 0x7a },
	  { 0xd6, 0xcd, 0xfe, 0xd4, 0x44, 0x4a, 0x9e, 0x90, 0x44, 0x73, 0x8a, 0xff, 0xbb, 0x82, 0x08, 0xb6, 0x7f, 0xf2, 0x87, 0xcb, 0xa5, 0x0b, 0x56, 0xd3, 0x9e, 0x91, 0xb8, 0x52, 0x6b, 0x25, 0xa6, 0x5d,
		0x50, 0xaf, 0x9b, 0xd5, 0xfb, 0x9f, 0x7e, 0x2d, 0x57, 0xdf, 0x30, 0x78, 0x8d, 0x1a, 0xc3, 0xac, 0x9c, 0x5a, 0xbf, 0xab, 0x5a, 0x0d, 0xc9, 0xb6, 0x4b, 0x18, 0xd4, 0xe7, 0x55, 0x40, 0xde, 0x7e },
	  { 0xc2, 0xa9, 0x7e, 0x5c, 0x26, 0xf4, 0x7d, 0xce, 0x9e, 0x73, 0xae, 0x50, 0xde, 0xe7, 0xa6, 0xf9, 0x8b, 0x57, 0xf9, 0x7a, 0x4c, 0x38, 0x82, 0xf6, 0x30, 0x80, 0x12, 0xf7, 0xf6, 0x66, 0x80, 0x46,
		0x4d, 0x41, 0x53, 0x63, 0xd9, 0x65, 0x90, 0xe7, 0xee, 0x24, 0x07, 0xb0, 0x4f, 0xeb, 0x3e, 0x8e, 0x83, 0x21, 0xa3, 0x40, 0x03, 0xc0, 0x64, 0x52, 0xc6, 0xb2, 0x12, 0x9d, 0x8d, 0x86, 0xdd, 0x19 },
	  { 0xe2, 0xd5, 0x49, 0x5e, 0x2a, 0x6e, 0x4e, 0xd9, 0x31, 0x26, 0x53, 0x13, 0x98, 0x5e, 0x2f, 0x23, 0xea, 0xa0, 0x30, 0xee, 0xef, 0x62, 0x2b, 0xdc, 0x93, 0x65, 0x90, 0xad, 0x9a, 0xf1, 0x74, 0x12,
		0xf5, 0x24, 0x33, 0xcc, 0xc3, 0xda, 0x42, 0x54, 0xa6, 0x6c, 0x86, 0x99, 0xb9, 0xb5, 0xf7, 0x07, 0x90, 0xd8, 0x85, 0x7f, 0x69, 0xfb, 0x19, 0x2a, 0x2c, 0xc0, 0x11, 0x81, 0x64, 0x37, 0x38, 0x07 },
	  { 0xc7, 0xb3, 0xf5, 0xe4, 0x4b, 0x55, 0xcf, 0xd8, 0x2b, 0x72, 0xde, 0x62, 0xfc, 0x66, 0xea, 0x82, 0xee, 0x2e, 0xe5, 0x4f, 0x66, 0xba, 0x19, 0x63, 0x01, 0x0b, 0x2d, 0x89, 0xb4, 0xaa, 0x76, 0xb3,
		0x7e, 0xc5, 0xbe, 0xdd, 0x57, 0x90, 0x5e, 0xff, 0x5b, 0x9a, 0x71, 0xe1, 0x47, 0xf9, 0xec, 0xe5, 0xf0, 0x19, 0x89, 0x17, 0x65, 0x3e, 0x56, 0x4a, 0x98, 0xb2, 0x3c, 0x3b, 0xf0, 0x14, 0x13, 0x1b },
	  { 0xc0, 0x72, 0x26, 0x96, 0x6b, 0xf5, 0x50, 0xa1, 0x65, 0xcd, 0xfe, 0x92, 0xa5, 0x5a, 0xb3, 0x56, 0x27, 0x5b, 0x2f, 0x4a, 0x8f, 0x67, 0xaa, 0xf4, 0xa1, 0x6e, 0x3c, 0x66, 0xcc, 0xb7, 0x71, 0x70,
		0xff, 0x70, 0x1f, 0x9e, 0x09, 0xae, 0x31, 0xcb, 0x2a, 0xd5, 0x8a, 0x38, 0xa9, 0xaf, 0xbc, 0x94, 0xa2, 0xa8, 0xe9, 0x77, 0x1c, 0xc3, 0xfa, 0xd1, 0x45, 0xd2, 0xe2, 0xff, 0x7d, 0xf2, 0x44, 0x00,
		0xa0, 0xc3, 0xc1, 0xdd, 0xa0, 0x4c, 0xfb, 0xed, 0x1a, 0xbd, 0x0c, 0x05, 0x3b, 0xa9, 0xc8, 0x98, 0xb0, 0x7d, 0x6a, 0x77, 0xcb, 0x08, 0x70, 0x64, 0x31, 0x9d, 0x9c, 0x7b, 0x40, 0x9e, 0xbb, 0xf4 },
	  { 0xbc, 0x88, 0x9d, 0x36, 0xae, 0xbc, 0x92, 0x47, 0x63, 0x85, 0x41, 0xe3, 0x1e, 0x1c, 0x39, 0xf5, 0xd3, 0xc2, 0x0a, 0x7d, 0x18, 0x7a, 0x8f, 0xd3, 0x0c, 0x37, 0x50, 0x28, 0x35, 0x93, 0x77, 0x4b,
		0xcb, 0xba, 0x35, 0x4e, 0x94, 0x48, 0xe4, 0x0c, 0xa7, 0x36, 0x4f, 0x74, 0x2b, 0xf9, 0xb5, 0xb5, 0xeb, 0x91, 0x50, 0x3c, 0x67, 0x9b, 0x4d, 0x25, 0xd4, 0x0e, 0x0d, 0xb9, 0x5b, 0x77, 0xf3, 0x0e,
		0xa0, 0xc3, 0xc1, 0xdd, 0xa0, 0x4c, 0xfb, 0xed, 0x1a, 0xbd, 0x0c, 0x05, 0x3b, 0xa9, 0xc8, 0x98, 0xb0, 0x7d, 0x6a, 0x77, 0xcb, 0x08, 0x70, 0x64, 0x31, 0x9d, 0x9c, 0x7b, 0x40, 0x9e, 0xbb, 0xf4 } },
	{ { 0x44, 0xdd, 0x62, 0x9e, 0x0f, 0xee, 0x20, 0x11, 0x37, 0xfc, 0xd0, 0x5c, 0xe4, 0xe1, 0x0a, 0xb8, 0xc2, 0xe0, 0x9c, 0x2c, 0x3e, 0x1b, 0x31, 0x1c, 0xdb, 0xa3, 0x84, 0x9a, 0xb7, 0x4e, 0x40, 0x74,
		0x21, 0xfd, 0xfc, 0x65, 0xbd, 0x38, 0x8a, 0x55, 0x6f, 0x1e, 0xc3, 0x14, 0xfc, 0x66, 0x04, 0x7b, 0xc4, 0x61, 0xb0, 0xcb, 0xfa, 0xdd, 0x50, 0x45, 0x4b, 0x2e, 0xf0, 0x6d, 0x0f, 0x26, 0x6d, 0xbf },
	  { 0xe6, 0xbc, 0x35, 0x73, 0xb3, 0x11, 0x38, 0xc6, 0x31, 0x82, 0x96, 0x80, 0x1d, 0xa9, 0xd9, 0x17, 0x85, 0x4e, 0xad, 0x0f, 0x5c, 0xb7, 0xe8, 0x78, 0x62, 0x2f, 0x3c, 0x10, 0x0e, 0xdc, 0xf2, 0x7e,
		0xf5, 0x02, 0x6d, 0x1a, 0x50, 0xc2, 0x50, 0x7d, 0x0d, 0x14, 0x77, 0x77, 0xfc, 0xbe, 0x23, 0x02, 0x81, 0x0a, 0xdc, 0xa3, 0x16, 0xfd, 0xab, 0xb9, 0x7c, 0xb6, 0x7e, 0x8a, 0xde, 0x1f, 0x22, 0xeb },
	  { 0xab, 0xf3, 0xea, 0x63, 0xc0, 0x25, 0xa2, 0xc7, 0x6a, 0xfe, 0x91, 0x4a, 0x0a, 0x91, 0xdd, 0x6d, 0x6f, 0x8c, 0xf9, 0xa8, 0x1c, 0x9f, 0xb5, 0xe5, 0xd2, 0xac, 0xe6, 0x51, 0x9a, 0xd3, 0x87, 0x17,
		0x82, 0x12, 0x0a, 0x58, 0x99, 0x7f, 0x81, 0x2d, 0x8d, 0x27, 0x2d, 0x1b, 0xb0, 0x02, 0x7e, 0x0d, 0xd6, 0x18, 0x89, 0x5e, 0x0c, 0x2b, 0x57, 0xa6, 0x56, 0x35, 0xff, 0x71, 0x4e, 0xb0, 0x49, 0x38 },
	  { 0x36, 0xdf, 0x1d, 0x1c, 0xf6, 0xa7, 0x4d, 0x87, 0x7e, 0x2c, 0x3f, 0xb4, 0xda, 0xd7, 0x80, 0x71, 0x0b, 0xf3, 0x2a, 0x47, 0x20, 0xe6, 0x9a, 0x3d, 0x17, 0x9a, 0x97, 0xc9, 0x4e, 0x53, 0xa6, 0xe2,
		0x23, 0xea, 0x94, 0x4d, 0xf9, 0xeb, 0x2c, 0x03, 0x2c, 0x88, 0xa2, 0xe6, 0xc5, 0x94, 0xa5, 0x6f, 0xc3, 0x98, 0xa9, 0x8b, 0xa7, 0x41, 0x7d, 0xd3, 0x82, 0x01, 0x13, 0xb6, 0x0f, 0x39, 0x1e, 0xd2 },
	  { 0x08, 0x28, 0xc3, 0x1c, 0xec, 0x21, 0x3a, 0xb4, 0x4c, 0xb1, 0xfa, 0xb9, 0x0c, 0xfe, 0xc2, 0x50, 0xc5, 0x99, 0x62, 0xa0, 0x11, 0x74, 0xcf, 0x05, 0x1e, 0x2b, 0xdf, 0x6d, 0x22, 0x8e, 0x6e, 0x55,
		0x19, 0x21, 0x9c, 0xa1, 0x98, 0x56, 0x45, 0x90, 0x40, 0x3a, 0x8e, 0xad, 0x76, 0x4d, 0xd3, 0x95, 0x27, 0x67, 0x4e, 0x02, 0x16, 0xc3, 0xfe, 0x5a, 0x79, 0x4e, 0x2d, 0x6f, 0xd0, 0xe4, 0x4f, 0x62 },
	  { 0x40, 0x14, 0xe1, 0x88, 0x3d, 0xcc, 0x51, 0xcb, 0x98, 0x86, 0x06, 0x4d, 0xe4, 0x52, 0x71, 0xe2, 0x2e, 0x2b, 0x80, 0xfd, 0x81, 0x65, 0xaf, 0x93, 0x31, 0x87, 0xe0, 0xff, 0x31, 0xab, 0xff, 0x53,
		0x0e, 0x2d, 0xb1, 0x47, 0xe6, 0x44, 0xb7, 0x29, 0xab, 0x0f, 0x51, 0x3a, 0x53, 0x84, 0x36, 0x58, 0x8c, 0x5f, 0x7b, 0x65, 0x6a, 0xb7, 0x6f, 0xdc, 0xad, 0xc1, 0xa3, 0xe4, 0x21, 0xfc, 0x22, 0x0e,
		0xc1, 0x10, 0xd1, 0x7d, 0x9f, 0xd3, 0x1e, 0x33, 0xb4, 0xca, 0xb9, 0xff, 0xd8, 0x27, 0xb8, 0xca, 0xde, 0x49, 0x6f, 0xdc, 0xf0, 0xe8, 0x70, 0x36, 0xdb, 0x90, 0x00, 0x07, 0x9e, 0x77, 0x39, 0xfe },
	  { 0xc9, 0x93, 0x4b, 0xe6, 0x47, 0x7e, 0x1d, 0x86, 0x15, 0x46, 0xe8, 0x27, 0xf5, 0x84, 0x67, 0x4e, 0x42, 0xe3, 0x2b, 0x8a, 0x4e, 0x90, 0x7b, 0x87, 0xcc, 0xdf, 0xaa, 0x04, 0x06, 0x05, 0xe6, 0x72,
		0xff, 0x6f, 0x44, 0x1b, 0x08, 0xad, 0x79, 0x3e, 0xb7, 0xdd, 0xd7, 0x2c, 0x73, 0xf0, 0xf0, 0xc4, 0x6e, 0xb7, 0x37, 0xe1, 0x02, 0xf5, 0x42, 0xe7, 0xef, 0xa1, 0xdd, 0x50, 0x9a, 0xc5, 0x8d, 0x00,
		0xc1, 0x10, 0xd1, 0x7d, 0x9f, 0xd3, 0x1e, 0x33, 0xb4, 0xca, 0xb9, 0xff, 0xd8, 0x27, 0xb8, 0xca, 0xde, 0x49, 0x6f, 0xdc, 0xf0, 0xe8, 0x70, 0x36, 0xdb, 0x90, 0x00, 0x07, 0x9e, 0x77, 0x39, 0xfe } },
	{ { 0x3e, 0x0c, 0x21, 0xc4, 0x3d, 0x64, 0x61, 0xc1, 0x9d, 0xa1, 0x83, 0x10, 0x74, 0x1d, 0x56, 0x12, 0xaf, 0x29, 0x5c, 0x6c, 0x12, 0x48, 0x0a, 0xc7, 0xe5, 0x12, 0xb6, 0x42, 0x6b, 0x54, 0xf4, 0x42,
		0x0c, 0x43, 0x42, 0x2e, 0x78, 0xc2, 0xe7, 0x26, 0x09, 0x41, 0x4a, 0x2f, 0xa1, 0xb0, 0x1f, 0xcd, 0x63, 0x76, 0x1e, 0xa1, 0x6f, 0xf6, 0xe2, 0xc2, 0x08, 0x89, 0x0d, 0x28, 0xbf, 0x1b, 0x56, 0x5b },
	  { 0x3e, 0x2e, 0xf2, 0xcc, 0x81, 0xca, 0xa7, 0x5d, 0x01, 0xd2, 0x82, 0xfd, 0x45, 0xee, 0xc0, 0xf5, 0x49, 0x3b, 0xe2, 0xa4, 0x2a, 0x4d, 0x5f, 0x40, 0x0d, 0xbc, 0xb9, 0x3d, 0x6e, 0xda, 0xe2, 0x86,
		0xe1, 0x23, 0x8b, 0x5f, 0x0d, 0xa2, 0x35, 0x15, 0x1d, 0x22, 0x23, 0xa5, 0x69, 0x56, 0x34, 0x78, 0xb3, 0xb3, 0x55, 0xef, 0x63, 0x8a, 0x17, 0x63, 0xda, 0xf0, 0x64, 0x99, 0x8a, 0x8a, 0xba, 0xd6 },
	  { 0x68, 0x79, 0x36, 0xa7, 0x6b, 0xe3, 0x76, 0x1c, 0xe3, 0x38, 0x0b, 0xa3, 0x91, 0xb6, 0xb0, 0x82, 0x37, 0xfa, 0x52, 0x74, 0xf1, 0xb5, 0xd5, 0xd9, 0x07, 0x06, 0x9e, 0xda, 0x87, 0x6b, 0x0f, 0x24,
		0x4f, 0xbe, 0xc9, 0xff, 0x03, 0x41, 0xaf, 0x77, 0x68, 0xed, 0xe7, 0x71, 0xba, 0x2d, 0xde, 0x27, 0xa1, 0xbf, 0xa8, 0xa7, 0x30, 0x7c, 0xcb, 0x79, 0x72, 0x89, 0x1a, 0xdc, 0xc1, 0xe4, 0xb2, 0x9d },
	  { 0x94, 0xa3, 0x11, 0xf4, 0x44, 0x80, 0xd0, 0xa3, 0x47, 0x93, 0x36, 0xe2, 0xbd, 0x04, 0xe4, 0x74, 0x3d, 0x00, 0x60, 0xad, 0xd0, 0x2d, 0x86, 0x66, 0xa1, 0x72, 0x1a, 0xb9, 0x1c, 0x14, 0xa2, 0x9b,
		0x4b, 0x04, 0x7d, 0x5b, 0xcd, 0xf8, 0x01, 0x33, 0xde, 0x34, 0x10, 0x29, 0xc4, 0x72, 0x56, 0xff, 0x11, 0xcd, 0xd8, 0x61, 0x2c, 0xb6, 0xb7, 0xf4, 0x24, 0x8b, 0x44, 0xb4, 0xe7, 0x34, 0x50, 0xb8 },
	  { 0x72, 0xf6, 0xd4, 0xa3, 0x24, 0xf9, 0xef, 0xf4, 0x55, 0x8d, 0x3c, 0x07, 0xca, 0x10, 0xdd, 0x54, 0x87, 0x13, 0x32, 0x78, 0x5c, 0x64, 0x10, 0x08, 0x62, 0x7e, 0xf4, 0x34, 0x0f, 0x1c, 0xcd, 0xcc,
		0x3b, 0x42, 0xfe, 0x60, 0x41, 0x70, 0x2c, 0x6b, 0xd4, 0x6c, 0xf7, 0xb8, 0x24, 0xf6, 0xd7, 0x07, 0xb3, 0x46, 0xb0, 0x7d, 0x14, 0x24, 0x9b, 0x72, 0x79, 0xf4, 0x23, 0x2a, 0xec, 0x02, 0xe7, 0x69 },
	  { 0xe5, 0xbe, 0x84, 0xc3, 0x92, 0x47, 0x15, 0xd3, 0xac, 0x06, 0x44, 0x72, 0x41, 0xeb, 0xb6, 0x5a, 0x17, 0x06, 0x90, 0xd9, 0x55, 0x3d, 0xe4, 0x87, 0x7d, 0x5a, 0x11, 0x9f, 0x02, 0x6d, 0xd3, 0x4e,
		0x71, 0xd1, 0x5e, 0x16, 0x9f, 0xb2, 0xc0, 0x7f, 0xcb, 0x78, 0x8b, 0x89, 0x11, 0xae, 0x43, 0xe8, 0x85, 0xb7, 0xf9, 0xc8, 0x48, 0x5a, 0xb2, 0x96, 0xaf, 0x8f, 0xab, 0x71, 0x84, 0x9d, 0x40, 0x09,
		0x30, 0xd4, 0x32, 0x6e, 0xa2, 0x77, 0x97, 0x71, 0x37, 0xce, 0x22, 0x6b, 0xca, 0xc9, 0x79, 0xef, 0xc0, 0xb2, 0xb4, 0x3d, 0x30, 0xbf, 0x77, 0xe9, 0xc3, 0x8d, 0xec, 0x15, 0x04, 0x08, 0xfa, 0x15 },
	  { 0x4b, 0xf3, 0x7f, 0xb2, 0x78, 0x75, 0x45, 0xd4, 0xce, 0x5e, 0x3d, 0xaf, 0x92, 0x63, 0x3d, 0x90, 0xc0, 0xa7, 0x23, 0x62, 0x7f, 0x37, 0x58, 0x8d, 0x12, 0xe0, 0xb8, 0x6c, 0x46, 0x38, 0xaa, 0xf7,
		0xe1, 0x03, 0x9e, 0x1f, 0x31, 0xf9, 0x5a, 0xa4, 0x59, 0x0d, 0xec, 0xc5, 0x1f, 0x17, 0x88, 0x25, 0xcc, 0xed, 0x69, 0x2b, 0x91, 0x73, 0x6a, 0x3f, 0xcb, 0xe5, 0x9c, 0x1e, 0x26, 0x3e, 0xec, 0x0b,
		0x30, 0xd4, 0x32, 0x6e, 0xa2, 0x77, 0x97, 0x71, 0x37, 0xce, 0x22, 0x6b, 0xca, 0xc9, 0x79, 0xef, 0xc0, 0xb2, 0xb4, 0x3d, 0x30, 0xbf, 0x77, 0xe9, 0xc3, 0x8d, 0xec, 0x15, 0x04, 0x08, 0xfa, 0x15 } },
	{ { 0xc5, 0x1d, 0xcd, 0x70, 0xb2, 0x9e, 0x53, 0x29, 0x05, 0x78, 0x83, 0x5d, 0x56, 0x30, 0x89, 0xee, 0x02, 0xd7, 0xac, 0x57, 0x0a, 0xd2, 0xa0, 0x9c, 0x96, 0x0c, 0xbf, 0xf2, 0x30, 0xbf, 0x1a, 0x2b,
		0xee, 0x0e, 0x9f, 0x1e, 0x1c, 0x65, 0x7d, 0xb5, 0x48, 0xad, 0x6f, 0x51, 0xa0, 0x91, 0x61, 0xe4, 0xe6, 0x83, 0x9f, 0x58, 0x7c, 0x76, 0x2b, 0x52, 0x94, 0x87, 0x3c, 0x8d, 0x36, 0x4c, 0x37, 0x3c },
	  { 0x59, 0x3b, 0x0d, 0x38, 0xab, 0x93, 0xca, 0xfb, 0x67, 0x44, 0x30, 0x96, 0xec, 0xbd, 0x00, 0x1d, 0x93, 0xd0, 0xb3, 0x3d, 0x3c, 0xd4, 0x4e, 0x3d, 0xd8, 0x29, 0x93, 0xb2, 0xb3, 0x77, 0xfc, 0x57,
		0x31, 0x20, 0xe3, 0x90, 0x0d, 0xf4, 0x91, 0x2f, 0x8b, 0x43, 0xce, 0xfe, 0x99, 0x03, 0x03, 0xa2, 0x90, 0x8d, 0xcf, 0xa8, 0xc0, 0x21, 0x00, 0xca, 0xcc, 0xcb, 0x4b, 0x2f, 0xa5, 0x39, 0xa8, 0x0b },
	  { 0xca, 0xf6, 0xf9, 0xbb, 0x53, 0xcb, 0x97, 0x76, 0xb6, 0x9c, 0x2c, 0x18, 0x21, 0x43, 0x13, 0x48, 0x13, 0xc9, 0x0e, 0xeb, 0x40, 0xea, 0xce, 0x1f, 0x3a, 0xe9, 0xd2, 0x9e, 0x29, 0xdb, 0xe2, 0x79,
		0xe2, 0x1a, 0x9f, 0x84, 0x9d, 0xe4, 0x55, 0x82, 0x17, 0xeb, 0x87, 0xf6, 0xc3, 0xef, 0xcd, 0x54, 0x14, 0xee, 0xc8, 0x5b, 0xd7, 0x67, 0x05, 0xe2, 0x34, 0xa2, 0x7e, 0x81, 0x83, 0x21, 0x7a, 0x02 },
	  { 0xc5, 0x03, 0xd9, 0x75, 0xdf, 0x17, 0x15, 0xe3, 0x5b, 0x7b, 0x4f, 0x66, 0x9c, 0x15, 0x4e, 0x01, 0xdf, 0x3d, 0x16, 0xb6, 0x52, 0xcc, 0xcf, 0x28, 0x40, 0xdb, 0x20, 0xee, 0x8b, 0x69, 0xb1, 0x2b,
		0xc0, 0x6e, 0xe4, 0xd2, 0xf5, 0xd1, 0x49, 0x3f, 0xf3, 0x0a, 0x12, 0xcd, 0x13, 0xbd, 0x9d, 0x3d, 0x5b, 0x28, 0x5c, 0xb0, 0x0d, 0x0e, 0xb6, 0xed, 0xec, 0x65, 0xeb, 0x25, 0x28, 0x2e, 0x65, 0x2f },
	  { 0xed, 0xa7, 0x05, 0xc1, 0xa6, 0x81, 0xf2, 0x7a, 0x69, 0x68, 0x17, 0x8e, 0xf7, 0xc9, 0x14, 0x80, 0x9f, 0x81, 0xfe, 0x16, 0xfd, 0x81, 0x93, 0xb4, 0x0b, 0x05, 0x5b, 0x4e, 0xef, 0x6e, 0x7a, 0x67,
		0x9d, 0x99, 0x4c, 0x17, 0xcd, 0x1c, 0x16, 0xfd, 0x31, 0x35, 0xd5, 0x3e, 0xa3, 0x00, 0xbf, 0xbe, 0xda, 0xd6, 0xe2, 0x37, 0x9b, 0x13, 0x1b, 0xca, 0x29, 0x90, 0x4b, 0xf2, 0x09, 0x57, 0x2f, 0xe9 },
	  { 0xd7, 0xba, 0x23, 0xd3, 0xa0, 0x6e, 0x14, 0x6a, 0xf0, 0x77, 0xb7, 0xe6, 0xe3, 0xc9, 0x3b, 0x38, 0xbb, 0xe7, 0xbe, 0x54, 0x75, 0xf8, 0xb7, 0x42, 0x29, 0xe2, 0x83, 0xde, 0x20, 0x22, 0x41, 0xcf,
		0x5f, 0x6f, 0x80, 0x60, 0xf3, 0x44, 0x04, 0x21, 0xd5, 0x03, 0x68, 0x42, 0xde, 0x81, 0xea, 0xe8, 0x7e, 0x5b, 0x80, 0x0f, 0x1b, 0x2d, 0x06, 0xc7, 0xce, 0xe9, 0x46, 0xc7, 0xf7, 0xb3, 0xa2, 0x02,
		0x21, 0xb5, 0x4d, 0xc2, 0x36, 0xea, 0xe6, 0x7b, 0xb3, 0x61, 0xe6, 0x18, 0x40, 0x5b, 0xce, 0x5b, 0xc2, 0xee, 0xa5, 0xde, 0xe9, 0xe6, 0xe0, 0xa8, 0x58, 0x58, 0x03, 0x34, 0x26, 0x27, 0x65, 0x2a },
	  { 0xfa, 0x43, 0xa6, 0xc4, 0x32, 0xa1, 0x2f, 0xb6, 0x37, 0x05, 0xf4, 0xa4, 0xa7, 0x36, 0xdd, 0x1c, 0x45, 0x10, 0x95, 0x83, 0x67, 0x89, 0x79, 0x18, 0x34, 0xad, 0xe7, 0x57, 0x7f, 0x0d, 0x48, 0x9b,
		0x14, 0xdf, 0x5f, 0xc8, 0xd7, 0x0f, 0x78, 0x47, 0x88, 0x20, 0xff, 0x7f, 0xb1, 0x21, 0x27, 0x14, 0x58, 0x32, 0x12, 0xfb, 0x97, 0xe0, 0x81, 0x0e, 0x92, 0xf4, 0x5c, 0x0e, 0x44, 0x48, 0x4e, 0x01,
		0x21, 0xb5, 0x4d, 0xc2, 0x36, 0xea, 0xe6, 0x7b, 0xb3, 0x61, 0xe6, 0x18, 0x40, 0x5b, 0xce, 0x5b, 0xc2, 0xee, 0xa5, 0xde, 0xe9, 0xe6, 0xe0, 0xa8, 0x58, 0x58, 0x03, 0x34, 0x26, 0x27, 0x65, 0x2a } },
	{ { 0x1e, 0x89, 0x12, 0xe8, 0xab, 0xca, 0xeb, 0x96, 0x78, 0x43, 0x89, 0x79, 0x26, 0x61, 0x86, 0x2e, 0x37, 0xd7, 0x94, 0xb5, 0xb9, 0xf7, 0xc9, 0xe7, 0x04, 0x6c, 0x96, 0x1c, 0x54, 0x0d, 0xb0, 0x6c,
		0xd3, 0x68, 0x9b, 0x53, 0xa7, 0x56, 0x34, 0x1b, 0x65, 0xff, 0xf9, 0xee, 0xf1, 0xc6, 0xfd, 0x7e, 0xa8, 0x42, 0x59, 0x60, 0x06, 0x5f, 0xc2, 0x89, 0x8b, 0xfc, 0xf8, 0x6c, 0x9a, 0x0d, 0xb1, 0x36 },
	  { 0x52, 0x3d, 0x83, 0x25, 0x0f, 0x57, 0x81, 0x76, 0x7b, 0x21, 0xf7, 0x96, 0xd6, 0x1f, 0xfe, 0xd7, 0x7c, 0xc1, 0x32, 0xb5, 0xbc, 0x05, 0x46, 0xdb, 0x6f, 0x25, 0xd8, 0x7a, 0x68, 0xe2, 0x01, 0x81,
		0xf8, 0x9a, 0xc5, 0x29, 0x78, 0x1c, 0x01, 0xc5, 0x4d, 0x61, 0x4e, 0x75, 0xdf, 0x9f, 0xc3, 0x22, 0x96, 0x7c, 0xf9, 0xa7, 0xed, 0x41, 0x6f, 0x64, 0xfd, 0xd4, 0x61, 0x58, 0x0d, 0x49, 0xc9, 0xa4 },
	  { 0x4a, 0xf7, 0xda, 0xef, 0xe0, 0x3b, 0x33, 0x19, 0x79, 0x02, 0x7a, 0xbb, 0xd3, 0x53, 0xf4, 0x8c, 0x8a, 0x16, 0xfb, 0xbd, 0x35, 0xd9, 0x70, 0xb2, 0x0a, 0x06, 0x05, 0x14, 0xd0, 0x9e, 0xf6, 0x13,
		0x44, 0xbb, 0xb7, 0x93, 0x86, 0x1b, 0x3c, 0xb0, 0x54, 0xa7, 0x48, 0xc2, 0xa7, 0x10, 0xda, 0x65, 0xb2, 0xdb, 0x0f, 0x85, 0x23, 0x57, 0x77, 0x44, 0x23, 0x20, 0x6d, 0x2e, 0xde, 0x20, 0x01, 0xed },
	  { 0x9c, 0xb8, 0x68, 0xeb, 0xbb, 0x8b, 0xaf, 0x81, 0x9c, 0x2f, 0x90, 0x4c, 0xc2, 0x62, 0x17, 0xfc, 0xf2, 0xa5, 0xab, 0x4c, 0x2e, 0x69, 0xcb, 0x82, 0x5f, 0x4c, 0x3c, 0x82, 0xcd, 0x6a, 0xcb, 0x15,
		0xa2, 0xfc, 0x50, 0x54, 0x5e, 0x2e, 0x83, 0x52, 0x48, 0x29, 0x51, 0xcc, 0x50, 0xaa, 0x27, 0xa3, 0xf3, 0x71, 0xdb, 0x2c, 0x1c, 0xa9, 0x8a, 0xa5, 0x95, 0xab, 0x3e, 0x6f, 0xcd, 0xba, 0x22, 0x7c },
	  { 0xf7, 0x5d, 0xb5, 0x20, 0x65, 0xfe, 0xa9, 0xe7, 0x1f, 0x8e, 0xd6, 0xc0, 0xf2, 0x3f, 0x1b, 0x8c, 0x7a, 0x02, 0x54, 0xd8, 0xa7, 0x0e, 0x6f, 0x68, 0x94, 0x81, 0xff, 0x30, 0x0e, 0x6d, 0x1a, 0x96,
		0x1b, 0x86, 0x07, 0xaa, 0xbf, 0x37, 0xc5, 0x5e, 0x26, 0xa2, 0xdf, 0x0b, 0xd0, 0x7f, 0x94, 0x35, 0x30, 0xa4, 0x9e, 0x47, 0xaf, 0xad, 0x9c, 0xc9, 0x02, 0x21, 0x55, 0x94, 0x04, 0x13, 0xff, 0x64 },
	  { 0x9c, 0x8d, 0x18, 0x63, 0x83, 0xad, 0x01, 0xcc, 0xbb, 0xe6, 0x00, 0xda, 0x15, 0xce, 0xc6, 0x6e, 0x7a, 0x37, 0x6a, 0x81, 0x44, 0xb3, 0xfc, 0xb7, 0xcd, 0x05, 0xee, 0x4a, 0x6f, 0x29, 0xe4, 0x79,
		0x63, 0x52, 0x7e, 0x14, 0xc9, 0x14, 0x77, 0xa8, 0x19, 0x94, 0x03, 0xc6, 0x51, 0x57, 0xf1, 0xcc, 0x11, 0x29, 0xde, 0x86, 0x08, 0xfe, 0x41, 0x02, 0x71, 0xb7, 0xbf, 0xd7, 0xe7, 0x83, 0x3e, 0x0c,
		0x9a, 0x59, 0x7e, 0xe8, 0x61, 0x36, 0x56, 0x9a, 0xbf, 0x64, 0xfd, 0xf3, 0xb7, 0xb9, 0x2f, 0x9e, 0x56, 0x1f, 0x57, 0x45, 0x2e, 0x19, 0x0f, 0x6f, 0x70, 0x01, 0xc2, 0x48, 0x05, 0x23, 0x9b, 0x2f },
	  { 0xb5, 0x4e, 0xe7, 0xcc, 0x7b, 0x66, 0x7a, 0xf8, 0xec, 0xcd, 0x1b, 0x0c, 0x0f, 0xec, 0x04, 0x27, 0xa0, 0x61, 0xfd, 0x12, 0x2d, 0xab, 0xc9, 0xc5, 0x8e, 0xee, 0x36, 0xc2, 0xef, 0x67, 0xd5, 0x87,
		0x95, 0x6c, 0x12, 0xb7, 0x12, 0x81, 0x55, 0xe0, 0x7b, 0xdb, 0x8f, 0x67, 0xea, 0x04, 0x55, 0x91, 0x9b, 0x50, 0x65, 0x05, 0xc1, 0xf1, 0x0b, 0x04, 0x91, 0x66, 0x3c, 0x32, 0x53, 0x72, 0x01, 0x04,
		0x9a, 0x59, 0x7e, 0xe8, 0x61, 0x36, 0x56, 0x9a, 0xbf, 0x64, 0xfd, 0xf3, 0xb7, 0xb9, 0x2f, 0x9e, 0x56, 0x1f, 0x57, 0x45, 0x2e, 0x19, 0x0f, 0x6f, 0x70, 0x01, 0xc2, 0x48, 0x05, 0x23, 0x9b, 0x2f } },
	{ { 0xc8, 0x37, 0x10, 0xdc, 0xdb, 0xfc, 0x51, 0x91, 0xae, 0x37, 0xa4, 0xe0, 0xcf, 0xbb, 0xdd, 0x92, 0x93, 0x5f, 0x6b, 0xd6, 0x81, 0xbf, 0x9b, 0x24, 0x5e, 0x0d, 0xf1, 0xe4, 0x04, 0x89, 0xd1, 0x1b,
		0xb2, 0x68, 0x56, 0x3a, 0xdc, 0x59, 0xd0, 0x8a, 0x93, 0x37, 0x5d, 0xa5, 0x40, 0x5e, 0xfe, 0xc9, 0x41, 0x0b, 0x8a, 0x50, 0xd2, 0xa0, 0x94, 0x86, 0xf7, 0x46, 0x3b, 0x7e, 0x1d, 0xea, 0x2b, 0xa8 },
	  { 0x1b, 0xe2, 0xe6, 0x48, 0x86, 0xa8, 0x65, 0xfd, 0x2b, 0xae, 0xc7, 0x7d, 0x41, 0xee, 0xb2, 0x80, 0x33, 0x1c, 0x0a, 0xdc, 0x42, 0xea, 0x99, 0xd0, 0x1f, 0x6d, 0xc8, 0x80, 0x51, 0x70, 0xd4, 0x19,
		0xae, 0xfc, 0x66, 0x16, 0xa2, 0x53, 0x27, 0x19, 0x7a, 0xf2, 0x9a, 0x25, 0x0c, 0x39, 0x8c, 0xbf, 0xe7, 0xa3, 0x7a, 0xd6, 0xa3, 0x43, 0x62, 0xd2, 0x4a, 0xc2, 0xf1, 0x96, 0x7e, 0xe3, 0x83, 0x13 },
	  { 0xf5, 0xb1, 0x2a, 0xc5, 0x4d, 0xcc, 0xdf, 0x56, 0xde, 0x92, 0x96, 0x46, 0x03, 0x11, 0xfc, 0xa0, 0xbc, 0xa2, 0x22, 0xf7, 0x25, 0x74, 0x2a, 0x1f, 0x27, 0x34, 0x18, 0xe8, 0x06, 0xa4, 0x77, 0x26,
		0x1a, 0x51, 0x5e, 0xfb, 0x77, 0xbc, 0x55, 0xb1, 0xf8, 0xa5, 0x19, 0x23, 0x00, 0x97, 0xf7, 0xbb, 0xe4, 0xcd, 0x41, 0x9e, 0xd9, 0x5e, 0x0c, 0x6b, 0x1b, 0x8a, 0xba, 0x52, 0x93, 0xbe, 0x2c, 0xf3 },
	  { 0xb3, 0x02, 0xeb, 0x44, 0x3c, 0x05, 0xae, 0x9c, 0x94, 0xa9, 0x1f, 0x72, 0x41, 0xbc, 0x81, 0x66, 0x5f, 0x50, 0xc0, 0x57, 0xb4, 0x44, 0xf0, 0xe1, 0x2a, 0xa9, 0x88, 0x69, 0xa6, 0x1c, 0x05, 0x85,
		0xda, 0xc7, 0xb2, 0xe1, 0x8c, 0x2f, 0x7c, 0x49, 0x37, 0xa2, 0xf2, 0x56, 0xab, 0x12, 0x9f, 0x12, 0x4b, 0x1b, 0x73, 0x75, 0x3f, 0x30, 0x0f, 0x40, 0xf1, 0xf9, 0x1d, 0xa7, 0x2c, 0x98, 0x8c, 0x91 },
	  { 0xcb, 0xd3, 0x39, 0x60, 0x56, 0xe3, 0xbd, 0x65, 0x86, 0x1a, 0x58, 0x40, 0xc0, 0xa4, 0xc4, 0x8b, 0xe5, 0xf7, 0x49, 0x0a, 0xf2, 0x09, 0x51, 0x32, 0x6e, 0x06, 0x5a, 0x27, 0x19, 0x78, 0x2e, 0x3a,
		0x04, 0xf9, 0x34, 0x80, 0x49, 0x39, 0x93, 0xcd, 0x89, 0x67, 0x7b, 0xc0, 0x8d, 0x9d, 0x8d, 0x4c, 0x83, 0x20, 0x80, 0xfc, 0x00, 0xf2, 0x8a, 0x8f, 0xa4, 0x4d, 0x8e, 0x8f, 0x58, 0x51, 0x5b, 0x71 },
	  { 0x71, 0x3f, 0x90, 0x41, 0xb8, 0x74, 0xbc, 0x7a, 0x85, 0xf5, 0xab, 0xca, 0x7e, 0xf2, 0x70, 0x41, 0xbc, 0x36, 0xb5, 0xc3, 0x4e, 0xf1, 0x2b, 0x17, 0x35, 0x40, 0xdb, 0x3c, 0xdb, 0xd2, 0xec, 0x0b,
		0x99, 0xc1, 0x43, 0x17, 0xad, 0x38, 0x45, 0x2d, 0x07, 0x31, 0xd7, 0xb6, 0x95, 0x1c, 0x89, 0x25, 0xe4, 0x89, 0x97, 0xd3, 0xcf, 0x11, 0x2f, 0x63, 0x31, 0x51, 0xa2, 0x18, 0xfc, 0x12, 0x04, 0x0a,
		0xb0, 0x33, 0xce, 0x0b, 0x57, 0xc0, 0x8c, 0x58, 0x25, 0xf8, 0x9b, 0x50, 0x22, 0x1c, 0x5c, 0x7b, 0x02, 0xc7, 0xed, 0xfc, 0x98, 0x8b, 0xbd, 0xd2, 0x4e, 0xfc, 0x78, 0x91, 0x7f, 0x4c, 0x99, 0x24 },
	  { 0xfc, 0x46, 0xe4, 0x85, 0x0c, 0x52, 0x14, 0xf8, 0x8a, 0xa4, 0x97, 0x17, 0x10, 0xb2, 0x93, 0xef, 0xa0, 0x66, 0x3c, 0xfd, 0x61, 0x42, 0x24, 0x30, 0x70, 0x4b, 0xfd, 0x0b, 0x86, 0xc8, 0x97, 0xd7,
		0x04, 0xc2, 0xa6, 0x61, 0x41, 0xaf, 0xcc, 0x1d, 0x52, 0xc9, 0xf3, 0xca, 0xe1, 0x90, 0x7c, 0xbd, 0xce, 0xaf, 0x30, 0xc4, 0xb4, 0x7d, 0x81, 0x7e, 0xbd, 0xe2, 0x09, 0x70, 0x1e, 0x6b, 0xb9, 0x03,
		0xb0, 0x33, 0xce, 0x0b, 0x57, 0xc0, 0x8c, 0x58, 0x25, 0xf8, 0x9b, 0x50, 0x22, 0x1c, 0x5c, 0x7b, 0x02, 0xc7, 0xed, 0xfc, 0x98, 0x8b, 0xbd, 0xd2, 0x4e, 0xfc, 0x78, 0x91, 0x7f, 0x4c, 0x99, 0x24 } },
	{ { 0x5f, 0x01, 0x6d, 0xec, 0x82, 0x02, 0x96, 0x47, 0x74, 0xd9, 0x73, 0x2e, 0x2e, 0x17, 0x00, 0xb6, 0xe0, 0xa4, 0x13, 0x17, 0xae, 0x7f, 0x85, 0xcb, 0xff, 0xe7, 0x96, 0x99, 0xdb, 0x9f, 0xad, 0x21,
		0x60, 0xd9, 0x12, 0xdc, 0x41, 0x01, 0x33, 0x66, 0x4c, 0x24, 0x8b, 0x25, 0x17, 0xd7, 0x22, 0x14, 0x12, 0x4d, 0xad, 0x82, 0x9a, 0x85, 0x69, 0x5e, 0x35, 0x10, 0xe0, 0xd7, 0x1a, 0x82, 0x88, 0x14 },
	  { 0xab, 0x5f, 0x2c, 0x7d, 0xa2, 0xe5, 0x67, 0x5f, 0xe4, 0x92, 0x03, 0x93, 0xd7, 0x13, 0xa1, 0xfa, 0x4a, 0xb7, 0x18, 0x4a, 0x8e, 0x8c, 0x78, 0x9a, 0x0c, 0x60, 0x02, 0xe8, 0x2d, 0x50, 0x05, 0x0f,
		0x92, 0xee, 0x9f, 0x81, 0xde, 0x6b, 0x20, 0xe4, 0x9b, 0x17, 0x2e, 0x99, 0x0f, 0x01, 0x31, 0xa7, 0xc5, 0xc4, 0x53, 0x70, 0xda, 0x03, 0xc6, 0xf7, 0x22, 0x87, 0x98, 0x87, 0x19, 0x36, 0xa6, 0x49 },
	  { 0x93, 0xab, 0x22, 0xc4, 0x39, 0x6c, 0x97, 0x80, 0xd2, 0xe2, 0x36, 0xfa, 0x31, 0x74, 0x67, 0xcc, 0x50, 0x1b, 0x95, 0xbe, 0x77, 0xe0, 0xd1, 0x00, 0x74, 0x04, 0xe1, 0x4d, 0xca, 0x44, 0x35, 0x72,
		0x74, 0x69, 0x82, 0x23, 0x56, 0x9b, 0xcc, 0x34, 0x5a, 0xcb, 0xa2, 0xa3, 0x31, 0x12, 0x4a, 0x84, 0x4c, 0xe9, 0x37, 0x3a, 0x58, 0xf8, 0x79, 0x65, 0x4a, 0x66, 0x79, 0x82, 0xf4, 0x5d, 0x75, 0xc3 },
	  { 0x2d, 0x5d, 0xac, 0x4f, 0xb5, 0x00, 0x68, 0x3b, 0x5f, 0x2e, 0xdd, 0xcb, 0x14, 0x4a, 0x7f, 0xad, 0x12, 0x45, 0x91, 0xd1, 0x84, 0xd8, 0x14, 0xff, 0xcb, 0x64, 0x43, 0x6d, 0x65, 0xe7, 0x19, 0x68,
		0x2b, 0x5e, 0x53, 0x05, 0x74, 0x66, 0xed, 0xac, 0x2f, 0x5a, 0x8f, 0x70, 0x96, 0xab, 0x29, 0xf3, 0x9a, 0x59, 0xa2, 0xe2, 0xef, 0xd3, 0xc9, 0xd7, 0x53, 0xf8, 0xf5, 0xa3, 0xd6, 0xf4, 0x34, 0xf8 },
	  { 0x1d, 0x14, 0xf3, 0xfd, 0xb0, 0x66, 0x20, 0xff, 0xfc, 0x79, 0x47, 0xc7, 0x4c, 0xe9, 0x45, 0x67, 0xf5, 0x97, 0x14, 0xea, 0x7c, 0x63, 0xc5, 0x3f, 0x0b, 0x46, 0xe0, 0x88, 0xd6, 0x9b, 0x67, 0x71,
		0xba, 0xa6, 0x15, 0x28, 0x94, 0x54, 0x83, 0x68, 0x00, 0x3a, 0x33, 0xa6, 0x1a, 0x05, 0x6a, 0x68, 0x72, 0x98, 0x48, 0x71, 0xea, 0x5b, 0x47, 0xf5, 0x80, 0x46, 0xa9, 0x57, 0x84, 0xec, 0xad, 0xfc },
	  { 0xa3, 0x1d, 0x87, 0xd3, 0x28, 0x62, 0xc6, 0xf7, 0xdb, 0xfb, 0xfa, 0xfc, 0xf3, 0x27, 0x5c, 0x31, 0xd3, 0x32, 0x26, 0x0e, 0x0f, 0x41, 0x49, 0xec, 0x05, 0x16, 0xf7, 0xa5, 0x63, 0xb3, 0xbc, 0xe5,
		0x0d, 0x1e, 0x6f, 0x97, 0x4f, 0x68, 0x40, 0xc0, 0xd4, 0x6c, 0x4f, 0x9e, 0x25, 0xd0, 0xab, 0x8d, 0x2a, 0xb9, 0x3e, 0x06, 0x4d, 0x9d, 0x3d, 0x2d, 0x79, 0x8d, 0x93, 0xdc, 0xfc, 0x6f, 0x0b, 0x04,
		0x48, 0x7c, 0x19, 0x5c, 0xa9, 0xc8, 0x44, 0xe5, 0xf6, 0x4f, 0x51, 0xd8, 0x72, 0x63, 0x41, 0xda, 0x62, 0xac, 0x78, 0x73, 0xb3, 0x3e, 0xc8, 0xb2, 0xf1, 0x3f, 0x89, 0xf2, 0x0e, 0x95, 0xdf, 0xed },
	  { 0xfd, 0x69, 0xb1, 0x9a, 0xdb, 0xae, 0x95, 0x87, 0xe2, 0xc6, 0x8a, 0x97, 0x0c, 0xee, 0xc4, 0x22, 0x60, 0x4e, 0x96, 0xa9, 0x72, 0xb9, 0x6f, 0x86, 0x97, 0xa8, 0xdf, 0x83, 0xc5, 0x18, 0x18, 0x6e,
		0xc9, 0x43, 0x30, 0x7e, 0x5b, 0xcf, 0x37, 0x0f, 0xc1, 0xd7, 0xe5, 0xab, 0xb1, 0x31, 0xe0, 0x97, 0xc7, 0x53, 0xb7, 0xfd, 0xd7, 0xdf, 0x00, 0x43, 0x0e, 0x41, 0x62, 0x80, 0x0b, 0xe3, 0xe0, 0x06,
		0x48, 0x7c, 0x19, 0x5c, 0xa9, 0xc8, 0x44, 0xe5, 0xf6, 0x4f, 0x51, 0xd8, 0x72, 0x63, 0x41, 0xda, 0x62, 0xac, 0x78, 0x73, 0xb3, 0x3e, 0xc8, 0xb2, 0xf1, 0x3f, 0x89, 0xf2, 0x0e, 0x95, 0xdf, 0xed } },
	{ { 0x98, 0x29, 0xf7, 0x57, 0xfd, 0xbd, 0x44, 0x3f, 0xd9, 0x90, 0x98, 0x19, 0x97, 0xf2, 0x60, 0x27, 0xfd, 0x08, 0xfc, 0x8a, 0xc6, 0xaf, 0x87, 0x22, 0x7f, 0x74, 0x4a, 0x80, 0xaf, 0x72, 0x00, 0x01,
		0x70, 0x9b, 0x47, 0x2a, 0xd2, 0x8e, 0x41, 0x0a, 0xea, 0x6a, 0xdf, 0xb7, 0x61, 0x54, 0x89, 0x5e, 0x01, 0x9f, 0x76, 0x64, 0x29, 0xee, 0x8d, 0x85, 0x20, 0xff, 0x30, 0x58, 0xc2, 0xa3, 0x2a, 0x56 },
	  { 0xea, 0x69, 0x8e, 0x6b, 0x8e, 0xdd, 0x55, 0x22, 0x45, 0x61, 0xd4, 0x92, 0x66, 0x8e, 0x96, 0xaf, 0x7e, 0x40, 0x28, 0x72, 0xc4, 0x46, 0xe7, 0x88, 0xd4, 0x6c, 0x74, 0xb7, 0x48, 0x7f, 0xe8, 0xe1,
		0x5e, 0xa5, 0x85, 0x62, 0x8f, 0xd6, 0xfc, 0x27, 0x0a, 0xb2, 0x4b, 0x38, 0x94, 0x59, 0x52, 0x0d, 0x6a, 0x4d, 0xe5, 0x61, 0xce, 0x0d, 0x44, 0x03, 0xa6, 0x2a, 0xc2, 0xd4, 0xd4, 0xe2, 0x71, 0xe3 },
	  { 0x40, 0xf0, 0x82, 0xf0, 0x8d, 0xaa, 0xad, 0xa9, 0x9f, 0x9b, 0x85, 0x02, 0xcf, 0x57, 0x15, 0x41, 0x13, 0x59, 0xf2, 0xba, 0xdd, 0xbf, 0x93, 0xe5, 0x40, 0x2e, 0xaf, 0xdd, 0x43, 0x52, 0xc8, 0x7f,
		0x40, 0xad, 0x91, 0x5b, 0x58, 0xd1, 0xa1, 0xe8, 0x6f, 0x77, 0xc3, 0x41, 0x35, 0x5e, 0xf7, 0x03, 0xba, 0xe4, 0xed, 0x2c, 0x28, 0x59, 0xd6, 0x48, 0xfe, 0x50, 0xcc, 0xf9, 0x80, 0xd1, 0x49, 0xd1 },
	  { 0xd7, 0xa5, 0xd9, 0x13, 0xdf, 0x7d, 0xf6, 0xc6, 0x25, 0x0f, 0x52, 0xc2, 0x57, 0x61, 0x20, 0xf2, 0xf0, 0xdb, 0x47, 0x49, 0x56, 0xaf, 0x89, 0x11, 0xa7, 0x8d, 0x09, 0x3a, 0xfe, 0x45, 0x43, 0xef,
		0x9f, 0x0c, 0x42, 0xaf, 0xa8, 0xcc, 0x60, 0x48, 0xc0, 0x1c, 0x7c, 0xbe, 0x01, 0xe2, 0x88, 0xcc, 0x6c, 0x3e, 0x97, 0x91, 0xf3, 0xd9, 0xb2, 0xb2, 0x09, 0x7e, 0x35, 0xb1, 0x78, 0xb4, 0x03, 0xf6 },
	  { 0x08, 0xc4, 0x1a, 0x3a, 0xc3, 0xe3, 0x26, 0xbd, 0x8d, 0xee, 0x5d, 0xf0, 0xba, 0xb6, 0x65, 0xff, 0x77, 0xc0, 0x99, 0xd1, 0xca, 0xdc, 0xf5, 0x4b, 0x50, 0x50, 0x0a, 0x9e, 0x13, 0x33, 0x76, 0x86,
		0x9b, 0x39, 0x79, 0x78, 0x73, 0x5c, 0x2f, 0x69, 0xa9, 0x9e, 0x0b, 0xeb, 0x11, 0x1e, 0x12, 0xaa, 0xc1, 0x09, 0x83, 0x0f, 0xca, 0xcb, 0x95, 0x10, 0xde, 0x85, 0xe3, 0x75, 0x62, 0x4a, 0xc2, 0x4c },
	  { 0x68, 0x78, 0x6c, 0xce, 0x2f, 0x72, 0x80, 0xfe, 0x83, 0x88, 0x63, 0x37, 0xa7, 0xa1, 0x5a, 0x0b, 0x84, 0x8a, 0xda, 0x28, 0x84, 0xf1, 0x6a, 0x63, 0x24, 0x1c, 0x72, 0xda, 0x84, 0xee, 0x1d, 0xe0,
		0x77, 0xf0, 0xf6, 0xce, 0x7e, 0x79, 0x0a, 0x55, 0x03, 0x01, 0x13, 0x0f, 0xf7, 0x6b, 0x45, 0xe7, 0xcb, 0xfd, 0xb0, 0x37, 0x93, 0x4b, 0x40, 0x69, 0xe0, 0x77, 0x67, 0x72, 0x65, 0xee, 0x35, 0x08,
		0x00, 0xc0, 0x07, 0x10, 0xd8, 0x6e, 0x55, 0x83, 0x5a, 0xbc, 0xfa, 0x67, 0x80, 0x8f, 0xfa, 0x21, 0x3e, 0x56, 0x53, 0x5b, 0xbc, 0x9d, 0xff, 0x16, 0xd9, 0x57, 0xcf, 0x2b, 0x78, 0x06, 0x5a, 0x89 },
	  { 0xdf, 0x32, 0x1a, 0x01, 0x84, 0xe5, 0xb8, 0x2c, 0x70, 0x6c, 0xeb, 0xd1, 0xf0, 0xb4, 0x9b, 0x32, 0xc8, 0xd0, 0x81, 0xc4, 0xea, 0xb2, 0x7c, 0x32, 0x1a, 0x02, 0x61, 0xf2, 0xd9, 0x4d, 0xe5, 0x85,
		0xad, 0xfc, 0xc6, 0x70, 0xee, 0x85, 0x77, 0x07, 0x9b, 0x5d, 0x5f, 0x88, 0xef, 0xb6, 0xd8, 0xdf, 0x2b, 0xa2, 0x4d, 0x90, 0x11, 0x2d, 0x38, 0x3f, 0xa8, 0x84, 0xf0, 0x76, 0xdd, 0x31, 0xd0, 0x09,
		0x00, 0xc0, 0x07, 0x10, 0xd8, 0x6e, 0x55, 0x83, 0x5a, 0xbc, 0xfa, 0x67, 0x80, 0x8f, 0xfa, 0x21, 0x3e, 0x56, 0x53, 0x5b, 0xbc, 0x9d, 0xff, 0x16, 0xd9, 0x57, 0xcf, 0x2b, 0x78, 0x06, 0x5a, 0x89 } },
	{ { 0x25, 0x87, 0x1e, 0x6f, 0xe8, 0xd0, 0xde, 0x1d, 0xd5, 0xf2, 0xd3, 0x5b, 0xff, 0x9e, 0x67, 0x99, 0x60, 0xb4, 0x0e, 0xb7, 0x98, 0x1b, 0x2a, 0x3a, 0x9c, 0xec, 0xc1, 0xe1, 0x2e, 0x2b, 0xc0, 0x3e,
		0x3c, 0xfb, 0x64, 0x91, 0x72, 0xc6, 0x7e, 0x57, 0x47, 0x00, 0x97, 0xbf, 0x8e, 0x0e, 0xbf, 0xad, 0xd9, 0x28, 0x86, 0x7c, 0xfd, 0x41, 0x91, 0xae, 0x2d, 0xee, 0xc0, 0xb2, 0x32, 0x7d, 0x99, 0x7d },
	  { 0x63, 0xc1, 0xf9, 0x61, 0x9c, 0x9e, 0x1a, 0xd7, 0xca, 0xa3, 0x71, 0xd6, 0x34, 0x3d, 0xa7, 0x08, 0x36, 0x0c, 0xec, 0x37, 0x35, 0x94, 0x1a, 0x45, 0xa9, 0xfa, 0xf2, 0xb5, 0x25, 0x92, 0xbf, 0xd1,
		0x1e, 0xca, 0xdd, 0x5a, 0x23, 0xad, 0x9e, 0x45, 0xc3, 0x66, 0xcb, 0x8f, 0xda, 0xa3, 0xd1, 0xe6, 0x27, 0x38, 0x11, 0x54, 0x67, 0x31, 0x03, 0x64, 0x35, 0xe0, 0x68, 0x0b, 0x93, 0xee, 0x81, 0x17 },
	  { 0x8b, 0x01, 0xe9, 0x99, 0x54, 0x54, 0x73, 0x15, 0x0b, 0xac, 0x38, 0x7b, 0xe9, 0xe3, 0x17, 0x4f, 0x02, 0x3e, 0xe3, 0x8e, 0xda, 0x41, 0xa0, 0x9d, 0x10, 0xe0, 0xda, 0x11, 0xfe, 0xec, 0x2f, 0x42,
		0xe7, 0xc8, 0xb3, 0xde, 0x2f, 0x7b, 0xfd, 0xdf, 0x7c, 0x34, 0x3b, 0x5e, 0xac, 0x22, 0x8c, 0x99, 0x3d, 0xa1, 0xa9, 0xd9, 0x81, 0xb6, 0x51, 0xc8, 0xaf, 0x3e, 0x75, 0xed, 0x45, 0xcf, 0xf7, 0xb9 },
	  { 0xaf, 0xe9, 0x9c, 0x16, 0x4a, 0x8f, 0x3b, 0x0f, 0xef, 0x71, 0x2f, 0xaa, 0x8d, 0x7d, 0xce, 0xed, 0xea, 0x31, 0x93, 0xaf, 0x2c, 0x75, 0xc6, 0xfa, 0xda, 0x3e, 0xa6, 0xea, 0x2a, 0x3e, 0x7b, 0x72,
		0xb6, 0xf8, 0xd7, 0x9a, 0x88, 0xcb, 0x0b, 0x81, 0x97, 0x24, 0x29, 0x3b, 0x11, 0x23, 0x69, 0xc2, 0xff, 0x98, 0x39, 0x25, 0x99, 0xae, 0xe1, 0x07, 0x3e, 0x97, 0xde, 0x10, 0x21, 0x23, 0x7a, 0x2d },
	  { 0xbe, 0x2f, 0xb9, 0x4c, 0x41, 0x5a, 0x9a, 0xf6, 0xfb, 0xf8, 0x26, 0x9d, 0x81, 0x7f, 0x39, 0x91, 0xaf, 0x5b, 0xf1, 0xd7, 0x93, 0x0a, 0xdf, 0x18, 0x19, 0x4a, 0x80, 0x74, 0x14, 0x98, 0x2b, 0xf2,
		0x3b, 0x25, 0xc5, 0xe8, 0xfc, 0x07, 0x3f, 0x5d, 0xa1, 0x39, 0x27, 0x4e, 0x1c, 0xd2, 0x7a, 0xfe, 0x3e, 0x7b, 0x03, 0x35, 0x15, 0x9e, 0x35, 0x2b, 0xd0, 0xbe, 0x67, 0x48, 0x42, 0xdd, 0xa4, 0xdd },
	  { 0xbd, 0xcd, 0xd7, 0xbf, 0xb1, 0x0a, 0xdb, 0x9f, 0x85, 0x42, 0xba, 0xf4, 0xc8, 0xff, 0xb0, 0xe1, 0x9a, 0x18, 0x6d, 0x1a, 0xe0, 0x37, 0xc1, 0xa2, 0xe1, 0x1c, 0x38, 0x55, 0x14, 0xbf, 0x64, 0x67,
		0x84, 0x47, 0xb6, 0x0a, 0xf6, 0x93, 0xf1, 0x10, 0xab, 0x09, 0xf0, 0x60, 0x84, 0xe2, 0x4e, 0x4b, 0x5e, 0xa2, 0xd2, 0xd1, 0x19, 0x22, 0xd7, 0xc4, 0x85, 0x13, 0x23, 0xa3, 0x6a, 0xb6, 0x75, 0x0f,
		0x43, 0xe6, 0xde, 0x7b, 0x67, 0x2a, 0x73, 0x77, 0x9e, 0xb4, 0x94, 0x6c, 0xc3, 0x9a, 0x67, 0x51, 0xcf, 0xe9, 0x47, 0x46, 0x0e, 0x3a, 0x12, 0x7d, 0x7c, 0x66, 0x73, 0x6c, 0xd5, 0x4a, 0x21, 0x4d },
	  { 0x89, 0x7e, 0xd0, 0xbf, 0x2e, 0x9f, 0x0c, 0xff, 0x6e, 0x56, 0x25, 0x9b, 0x79, 0x99, 0x52, 0x27, 0xc2, 0x3a, 0xaa, 0xf0, 0x47, 0x6d, 0xed, 0x05, 0xa1, 0xeb, 0x9c, 0x92, 0x28, 0x7f, 0x1b, 0xc8,
		0x1c, 0x57, 0x76, 0xab, 0x05, 0xe3, 0xd3, 0xb7, 0xa3, 0xf5, 0xac, 0xa8, 0x21, 0x33, 0x7c, 0xb7, 0xe7, 0xc2, 0xd0, 0x25, 0x6f, 0xdf, 0x34, 0xd1, 0xb0, 0x34, 0x41, 0x46, 0x30, 0x9c, 0x76, 0x07,
		0x43, 0xe6, 0xde, 0x7b, 0x67, 0x2a, 0x73, 0x77, 0x9e, 0xb4, 0x94, 0x6c, 0xc3, 0x9a, 0x67, 0x51, 0xcf, 0xe9, 0x47, 0x46, 0x0e, 0x3a, 0x12, 0x7d, 0x7c, 0x66, 0x73, 0x6c, 0xd5, 0x4a, 0x21, 0x4d } }
};

//////////////////////////////////////////////////////////////////////////////

static int testCrypto()
{
	static unsigned char buf1[16384];
	static unsigned char buf2[sizeof(buf1)], buf3[sizeof(buf1)];
	static char hexbuf[1024];

	for (int i = 0; i < 3; ++i) {
		Utils::getSecureRandom(buf1, 64);
		std::cout << "[crypto] getSecureRandom: " << Utils::hex(buf1, 64, hexbuf) << std::endl;
	}

	std::cout << "[crypto] Testing Salsa20... ";
	std::cout.flush();
	for (unsigned int i = 0; i < 4; ++i) {
		for (unsigned int k = 0; k < sizeof(buf1); ++k)
			buf1[k] = (unsigned char)rand();
		memset(buf2, 0, sizeof(buf2));
		memset(buf3, 0, sizeof(buf3));
		Salsa20 s20;
		s20.init("12345678123456781234567812345678", "12345678");
		s20.crypt20(buf1, buf2, sizeof(buf1));
		s20.init("12345678123456781234567812345678", "12345678");
		s20.crypt20(buf2, buf3, sizeof(buf2));
		if (memcmp(buf1, buf3, sizeof(buf1))) {
			std::cout << "FAIL (encrypt/decrypt test)" << std::endl;
			return -1;
		}
	}
	Salsa20 s20(s20TV0Key, s20TV0Iv);
	memset(buf1, 0, sizeof(buf1));
	memset(buf2, 0, sizeof(buf2));
	s20.crypt20(buf1, buf2, 64);
	if (memcmp(buf2, s20TV0Ks, 64)) {
		std::cout << "FAIL (test vector 0)" << std::endl;
		return -1;
	}
	s20.init(s2012TV0Key, s2012TV0Iv);
	memset(buf1, 0, sizeof(buf1));
	memset(buf2, 0, sizeof(buf2));
	s20.crypt12(buf1, buf2, 64);
	if (memcmp(buf2, s2012TV0Ks, 64)) {
		std::cout << "FAIL (test vector 1)" << std::endl;
		return -1;
	}
	std::cout << "PASS" << std::endl;

#ifdef ZT_SALSA20_SSE
	std::cout << "[crypto] Salsa20 SSE: ENABLED" << std::endl;
#else
	std::cout << "[crypto] Salsa20 SSE: DISABLED" << std::endl;
#endif

	std::cout << "[crypto] Benchmarking Salsa20/12... ";
	std::cout.flush();
	{
		unsigned char* bb = (unsigned char*)::malloc(1234567);
		for (unsigned int i = 0; i < 1234567; ++i)
			bb[i] = (unsigned char)i;
		Salsa20 s20(s20TV0Key, s20TV0Iv);
		long double bytes = 0.0;
		uint64_t start = OSUtils::now();
		for (unsigned int i = 0; i < 200; ++i) {
			s20.crypt12(bb, bb, 1234567);
			bytes += 1234567.0;
		}
		uint64_t end = OSUtils::now();
		SHA512(buf1, bb, 1234567);
		std::cout << ((bytes / 1048576.0) / ((long double)(end - start) / 1024.0)) << " MiB/second (" << Utils::hex(buf1, 16, hexbuf) << ')' << std::endl;
		::free((void*)bb);
	}

#if defined(ZT_USE_X64_ASM_SALSA2012) && defined(ZT_ARCH_X64)
	std::cout << "[crypto] Benchmarking Salsa20/12 fast x64 ASM... ";
	std::cout.flush();
	{
		unsigned char* bb = (unsigned char*)::malloc(1234567);
		double bytes = 0.0;
		uint64_t start = OSUtils::now();
		for (unsigned int i = 0; i < 200; ++i) {
			zt_salsa2012_amd64_xmm6(bb, 1234567, s20TV0Iv, s20TV0Key);
			bytes += 1234567.0;
		}
		uint64_t end = OSUtils::now();
		std::cout << ((bytes / 1048576.0) / ((double)(end - start) / 1024.0)) << " MiB/second" << std::endl;
		::free((void*)bb);
	}
#endif

#ifdef ZT_USE_ARM32_NEON_ASM_SALSA2012
	if (zt_arm_has_neon()) {
		std::cout << "[crypto] Benchmarking Salsa20/12 fast arm32/neon ASM... ";
		std::cout.flush();
		{
			unsigned char* bb = (unsigned char*)::malloc(1234567);
			double bytes = 0.0;
			uint64_t start = OSUtils::now();
			for (unsigned int i = 0; i < 200; ++i) {
				zt_salsa2012_armneon3_xor(bb, (const unsigned char*)0, 1234567, s20TV0Iv, s20TV0Key);
				bytes += 1234567.0;
			}
			uint64_t end = OSUtils::now();
			std::cout << ((bytes / 1048576.0) / ((double)(end - start) / 1024.0)) << " MiB/second" << std::endl;
			::free((void*)bb);
		}
	}
#endif

	std::cout << "[crypto] Benchmarking Salsa20/20... ";
	std::cout.flush();
	{
		unsigned char* bb = (unsigned char*)::malloc(1234567);
		for (unsigned int i = 0; i < 1234567; ++i)
			bb[i] = (unsigned char)i;
		Salsa20 s20(s20TV0Key, s20TV0Iv);
		long double bytes = 0.0;
		uint64_t start = OSUtils::now();
		for (unsigned int i = 0; i < 200; ++i) {
			s20.crypt20(bb, bb, 1234567);
			bytes += 1234567.0;
		}
		uint64_t end = OSUtils::now();
		SHA512(buf1, bb, 1234567);
		std::cout << ((bytes / 1048576.0) / ((long double)(end - start) / 1024.0)) << " MiB/second (" << Utils::hex(buf1, 16, hexbuf) << ')' << std::endl;
		::free((void*)bb);
	}

	/*
	{
		AES k0,k1;
		k0.init("00000000000000000000000000000000");
		k1.init("11111111111111111111111111111111");
		uint8_t test_pt[65536];
		uint8_t test_ct[65536];
		uint8_t test_aad[65536];
		uint8_t ct_hash[48];
		char hex_tmp[128];
		for(unsigned int i=0;i<65536;++i) {
			test_pt[i] = (uint8_t)i;
			test_aad[i] = (uint8_t)i;
		}
		AES::GMACSIVEncryptor enc(k0,k1);
		for(unsigned int test_length=0;test_length<65536;test_length+=777) {
			memset(test_ct, 0, test_length);
			enc.init((uint64_t)test_length, test_ct);
			enc.aad(test_aad, test_length);
			enc.update1(test_pt, test_length);
			enc.finish1();
			enc.update2(test_pt, test_length);
			const void *tag = enc.finish2();
			SHA384(ct_hash, test_ct, test_length);
			std::cout << "(" << test_length << ", \"" << Utils::hex(ct_hash, 48, hex_tmp) << "\", \"" << Utils::hex(tag, 16, hex_tmp) << "\")," <<std::endl;
		}
	}
	*/

	std::cout << "[crypto] Benchmarking AES-GMAC-SIV... ";
	std::cout.flush();
	{
		uint64_t end, start = OSUtils::now();
		uint64_t bytes = 0;
		AES k0, k1;
		k0.init(buf1);
		k1.init(buf2);
		AES::GMACSIVEncryptor enc(k0, k1);
		for (;;) {
			for (unsigned int i = 0; i < 10000; ++i) {
				enc.init(i, buf2);
				enc.update1(buf1, sizeof(buf1));
				enc.finish1();
				enc.update2(buf1, sizeof(buf1));
				enc.finish2();
				buf1[0] = buf2[0];
				bytes += sizeof(buf1);
			}
			end = OSUtils::now();
			if ((end - start) >= 5000)
				break;
		}
		std::cout << (((double)bytes / 1048576.0) / ((double)(end - start) / 1024.0)) << " MiB/second" << std::endl;
	}

	std::cout << "[crypto] Testing SHA-512... ";
	std::cout.flush();
	SHA512(buf1, sha512TV0Input, (unsigned int)strlen(sha512TV0Input));
	if (memcmp(buf1, sha512TV0Digest, 64)) {
		std::cout << "FAIL" << std::endl;
		return -1;
	}
	std::cout << "PASS" << std::endl;

	std::cout << "[crypto] Testing Poly1305... ";
	std::cout.flush();
	Poly1305::compute(buf1, poly1305TV0Input, sizeof(poly1305TV0Input), poly1305TV0Key);
	if (memcmp(buf1, poly1305TV0Tag, 16)) {
		std::cout << "FAIL (1)" << std::endl;
		return -1;
	}
	Poly1305::compute(buf1, poly1305TV1Input, sizeof(poly1305TV1Input), poly1305TV1Key);
	if (memcmp(buf1, poly1305TV1Tag, 16)) {
		std::cout << "FAIL (2)" << std::endl;
		return -1;
	}
	std::cout << "PASS" << std::endl;

	std::cout << "[crypto] Benchmarking Poly1305... ";
	std::cout.flush();
	{
		unsigned char* bb = (unsigned char*)::malloc(1234567);
		for (unsigned int i = 0; i < 1234567; ++i)
			bb[i] = (unsigned char)i;
		long double bytes = 0.0;
		uint64_t start = OSUtils::now();
		for (unsigned int i = 0; i < 200; ++i) {
			Poly1305::compute(buf1, bb, 1234567, poly1305TV0Key);
			bytes += 1234567.0;
		}
		uint64_t end = OSUtils::now();
		std::cout << ((bytes / 1048576.0) / ((long double)(end - start) / 1000.0)) << " MiB/second" << std::endl;
		::free((void*)bb);
	}

	/*
	for(unsigned int d=8;d<=10;++d) {
		for(int k=0;k<8;++k) {
			std::cout << "[crypto] computeSalsa2012Sha512ProofOfWork(" << d << ",\"foobarbaz\",9) == "; std::cout.flush();
			unsigned char result[16];
			uint64_t start = OSUtils::now();
			IncomingPacket::computeSalsa2012Sha512ProofOfWork(d,"foobarbaz",9,result);
			uint64_t end = OSUtils::now();
			std::cout << Utils::hex(result,16) << " -- valid: " << IncomingPacket::testSalsa2012Sha512ProofOfWorkResult(d,"foobarbaz",9,result) << ", " << (end - start) << "ms" << std::endl;
		}
	}
	*/

	std::cout << "[crypto] Testing C25519 and Ed25519 against test vectors... ";
	std::cout.flush();
	for (int k = 0; k < ZT_NUM_C25519_TEST_VECTORS; ++k) {
		ECC::Pair p1, p2;
		memcpy(p1.pub.data, C25519_TEST_VECTORS[k].pub1, ZT_ECC_PUBLIC_KEY_SET_LEN);
		memcpy(p1.priv.data, C25519_TEST_VECTORS[k].priv1, ZT_ECC_PRIVATE_KEY_SET_LEN);
		memcpy(p2.pub.data, C25519_TEST_VECTORS[k].pub2, ZT_ECC_PUBLIC_KEY_SET_LEN);
		memcpy(p2.priv.data, C25519_TEST_VECTORS[k].priv2, ZT_ECC_PRIVATE_KEY_SET_LEN);
		ECC::agree(p1, p2.pub, buf1, 64);
		ECC::agree(p2, p1.pub, buf2, 64);
		if (memcmp(buf1, buf2, 64)) {
			std::cout << "FAIL (1)" << std::endl;
			return -1;
		}
		if (memcmp(buf1, C25519_TEST_VECTORS[k].agreement, 64)) {
			std::cout << "FAIL (2)" << std::endl;
			return -1;
		}
		ECC::Signature sig1 = ECC::sign(p1, buf1, 64);
		if (memcmp(sig1.data, C25519_TEST_VECTORS[k].agreementSignedBy1, 64)) {
			std::cout << "FAIL (3)" << std::endl;
			return -1;
		}
		ECC::Signature sig2 = ECC::sign(p2, buf1, 64);
		if (memcmp(sig2.data, C25519_TEST_VECTORS[k].agreementSignedBy2, 64)) {
			std::cout << "FAIL (4)" << std::endl;
			return -1;
		}
	}
	std::cout << "PASS" << std::endl;

	std::cout << "[crypto] Testing C25519 ECC key agreement... ";
	std::cout.flush();
	for (unsigned int i = 0; i < 100; ++i) {
		memset(buf1, 64, sizeof(buf1));
		memset(buf2, 64, sizeof(buf2));
		memset(buf3, 64, sizeof(buf3));
		ECC::Pair p1 = ECC::generate();
		ECC::Pair p2 = ECC::generate();
		ECC::Pair p3 = ECC::generate();
		ECC::agree(p1, p2.pub, buf1, 64);
		ECC::agree(p2, p1.pub, buf2, 64);
		ECC::agree(p3, p1.pub, buf3, 64);
		// p1<>p2 should equal p2<>p1
		if (memcmp(buf1, buf2, 64)) {
			std::cout << "FAIL (1)" << std::endl;
			return -1;
		}
		// p2<>p1 should not equal p3<>p1
		if (! memcmp(buf2, buf3, 64)) {
			std::cout << "FAIL (2)" << std::endl;
			return -1;
		}
	}
	std::cout << "PASS" << std::endl;

	std::cout << "[crypto] Benchmarking C25519 ECC key agreement... ";
	std::cout.flush();
	ECC::Pair bp[8];
	for (int k = 0; k < 8; ++k)
		bp[k] = ECC::generate();
	uint64_t st = OSUtils::now();
	for (unsigned int k = 0; k < 50; ++k) {
		ECC::agree(bp[~k & 7], bp[k & 7].pub, buf1, 64);
	}
	uint64_t et = OSUtils::now();
	std::cout << ((double)(et - st) / 50.0) << "ms per agreement." << std::endl;

	std::cout << "[crypto] Testing Ed25519 ECC signatures... ";
	std::cout.flush();
	ECC::Pair didntSign = ECC::generate();
	for (unsigned int i = 0; i < 10; ++i) {
		ECC::Pair p1 = ECC::generate();
		for (unsigned int k = 0; k < sizeof(buf1); ++k)
			buf1[k] = (unsigned char)rand();
		ECC::Signature sig = ECC::sign(p1, buf1, sizeof(buf1));
		if (! ECC::verify(p1.pub, buf1, sizeof(buf1), sig)) {
			std::cout << "FAIL (1)" << std::endl;
			return -1;
		}
		++buf1[17];
		if (ECC::verify(p1.pub, buf1, sizeof(buf1), sig)) {
			std::cout << "FAIL (2)" << std::endl;
			return -1;
		}
		--buf1[17];
		if (! ECC::verify(p1.pub, buf1, sizeof(buf1), sig)) {
			std::cout << "FAIL (3)" << std::endl;
			return -1;
		}
		if (ECC::verify(didntSign.pub, buf1, sizeof(buf1), sig)) {
			std::cout << "FAIL (2)" << std::endl;
			return -1;
		}
		for (unsigned int k = 0; k < 64; ++k) {
			ECC::Signature sig2(sig);
			sig2.data[rand() % ZT_ECC_SIGNATURE_LEN] ^= (unsigned char)(1 << (rand() & 7));
			if (ECC::verify(p1.pub, buf1, sizeof(buf1), sig2)) {
				std::cout << "FAIL (5)" << std::endl;
				return -1;
			}
		}
	}
	std::cout << "PASS" << std::endl;

	std::cout << "[crypto] Benchmarking Ed25519 ECC signatures... ";
	std::cout.flush();
	st = OSUtils::now();
	for (int k = 0; k < 1000; ++k) {
		ECC::Signature sig;
		ECC::sign(didntSign.priv, didntSign.pub, buf1, sizeof(buf1), sig.data);
	}
	et = OSUtils::now();
	std::cout << ((double)(et - st) / 50.0) << "ms per signature." << std::endl;

	return 0;
}

static int testIdentity()
{
	Identity id;
	Buffer<512> buf;
	char buf2[1024];

	std::cout << "[identity] Validate known-good identity... ";
	std::cout.flush();
	if (! id.fromString(KNOWN_GOOD_IDENTITY)) {
		std::cout << "FAIL (1)" << std::endl;
		return -1;
	}
	const uint64_t vst = OSUtils::now();
	for (int k = 0; k < 10; ++k) {
		if (! id.locallyValidate()) {
			std::cout << "FAIL (2)" << std::endl;
			return -1;
		}
	}
	const uint64_t vet = OSUtils::now();
	std::cout << "PASS (" << ((double)(vet - vst) / 10.0) << "ms per validation)" << std::endl;

	std::cout << "[identity] Validate known-bad identity... ";
	std::cout.flush();
	if (! id.fromString(KNOWN_BAD_IDENTITY)) {
		std::cout << "FAIL (1)" << std::endl;
		return -1;
	}
	if (id.locallyValidate()) {
		std::cout << "FAIL (2)" << std::endl;
		return -1;
	}
	std::cout << "PASS (i.e. it failed)" << std::endl;

	for (unsigned int k = 0; k < 4; ++k) {
		std::cout << "[identity] Generate identity... ";
		std::cout.flush();
		uint64_t genstart = OSUtils::now();
		id.generate();
		uint64_t genend = OSUtils::now();
		std::cout << "(took " << (genend - genstart) << "ms): " << id.toString(true, buf2) << std::endl;
		std::cout << "[identity] Locally validate identity: ";
		if (id.locallyValidate()) {
			std::cout << "PASS" << std::endl;
		}
		else {
			std::cout << "FAIL" << std::endl;
			return -1;
		}
	}

	{
		Identity id2;
		buf.clear();
		id.serialize(buf, true);
		id2.deserialize(buf);
		std::cout << "[identity] Serialize and deserialize (w/private): ";
		if ((id == id2) && (id2.locallyValidate())) {
			std::cout << "PASS" << std::endl;
		}
		else {
			std::cout << "FAIL" << std::endl;
			return -1;
		}
	}

	{
		Identity id2;
		buf.clear();
		id.serialize(buf, false);
		id2.deserialize(buf);
		std::cout << "[identity] Serialize and deserialize (no private): ";
		if ((id == id2) && (id2.locallyValidate())) {
			std::cout << "PASS" << std::endl;
		}
		else {
			std::cout << "FAIL" << std::endl;
			return -1;
		}
	}

	{
		Identity id2;
		id2.fromString(id.toString(true, buf2));
		std::cout << "[identity] Serialize and deserialize (ASCII w/private): ";
		if ((id == id2) && (id2.locallyValidate())) {
			std::cout << "PASS" << std::endl;
		}
		else {
			std::cout << "FAIL" << std::endl;
			return -1;
		}
	}

	{
		Identity id2;
		id2.fromString(id.toString(false, buf2));
		std::cout << "[identity] Serialize and deserialize (ASCII no private): ";
		if ((id == id2) && (id2.locallyValidate())) {
			std::cout << "PASS" << std::endl;
		}
		else {
			std::cout << "FAIL" << std::endl;
			return -1;
		}
	}

	return 0;
}

static int testCertificate()
{
	char buf[4096];

	Identity authority;
	std::cout << "[certificate] Generating identity to act as authority... ";
	std::cout.flush();
	authority.generate();
	std::cout << authority.address().toString(buf) << std::endl;

	Identity idA, idB;
	std::cout << "[certificate] Generating identities A and B... ";
	std::cout.flush();
	idA.generate();
	idB.generate();
	std::cout << idA.address().toString(buf) << ", " << idB.address().toString(buf) << std::endl;

	std::cout << "[certificate] Generating certificates A and B...";
	CertificateOfMembership cA(10000, 100, 1, idA);
	CertificateOfMembership cB(10099, 100, 1, idB);
	std::cout << std::endl;

	std::cout << "[certificate] Signing certificates A and B with authority...";
	cA.sign(authority);
	cB.sign(authority);
	std::cout << std::endl;

	// std::cout << "[certificate] A: " << cA.toString() << std::endl;
	// std::cout << "[certificate] B: " << cB.toString() << std::endl;

	std::cout << "[certificate] A agrees with B and B with A... ";
	if (cA.agreesWith(cB, idB))
		std::cout << "yes, ";
	else {
		std::cout << "FAIL" << std::endl;
		return -1;
	}
	if (cB.agreesWith(cA, idA))
		std::cout << "yes." << std::endl;
	else {
		std::cout << "FAIL" << std::endl;
		return -1;
	}

	std::cout << "[certificate] Generating two certificates that should not agree...";
	cA = CertificateOfMembership(10000, 100, 1, idA);
	cB = CertificateOfMembership(10101, 100, 1, idB);
	std::cout << std::endl;

	std::cout << "[certificate] A agrees with B and B with A... ";
	if (! cA.agreesWith(cB, idB))
		std::cout << "no, ";
	else {
		std::cout << "FAIL" << std::endl;
		return -1;
	}
	if (! cB.agreesWith(cA, idA))
		std::cout << "no." << std::endl;
	else {
		std::cout << "FAIL" << std::endl;
		return -1;
	}

	return 0;
}

static int testPacket()
{
	unsigned char salsaKey[32];
	Packet a, b;

	a.burn();
	b.burn();

	for (unsigned int i = 0; i < 32; ++i)
		salsaKey[i] = (unsigned char)rand();

	std::cout << "[packet] Testing Packet encoder/decoder... ";

	a.reset(Address(), Address(), Packet::VERB_HELLO);
	for (int i = 0; i < 32; ++i)
		a.append("supercalifragilisticexpealidocious", (unsigned int)strlen("supercalifragilisticexpealidocious"));

	b = a;
	if (a != b) {
		std::cout << "FAIL (assign)" << std::endl;
		return -1;
	}

	a.compress();
	unsigned int complen = a.size();
	a.uncompress();

	std::cout << "(compressed: " << complen << ", decompressed: " << a.size() << ") ";
	if (a != b) {
		std::cout << "FAIL (compression)" << std::endl;
		return -1;
	}

	/*
	a.armor(salsaKey, true, false, nullptr);
	if (! a.dearmor(salsaKey, nullptr)) {
		std::cout << "FAIL (encrypt-decrypt/verify)" << std::endl;
		return -1;
	}
	*/

	std::cout << "PASS" << std::endl;
	return 0;
}

static int testOther()
{
	char buf[1024];
	char buf2[4096];
	char buf3[1024];

	std::cout << "[other] Testing hex/unhex... ";
	std::cout.flush();
	Utils::getSecureRandom(buf, (unsigned int)sizeof(buf));
	Utils::hex(buf, (unsigned int)sizeof(buf), buf2);
	Utils::unhex(buf2, buf3, (unsigned int)sizeof(buf3));
	if (memcmp(buf, buf3, sizeof(buf)) == 0) {
		std::cout << "PASS" << std::endl;
	}
	else {
		std::cout << "FAIL!" << std::endl;
		buf2[78] = 0;
		std::cout << buf2 << std::endl;
		Utils::hex(buf3, (unsigned int)sizeof(buf3), buf2);
		buf2[78] = 0;
		std::cout << buf2 << std::endl;
		return -1;
	}

	std::cout << "[other] Testing InetAddress encode/decode...";
	std::cout.flush();
	std::cout << " " << InetAddress("127.0.0.1/9993").toString(buf);
	std::cout << " " << InetAddress("feed:dead:babe:dead:beef:f00d:1234:5678/12345").toString(buf);
	std::cout << " " << InetAddress("0/9993").toString(buf);
	std::cout << " " << InetAddress("").toString(buf);
	std::cout << std::endl;

#if 0
	std::cout << "[other] Testing Hashtable... "; std::cout.flush();
	{
		Hashtable<uint64_t,std::string> ht;
		std::map<uint64_t,std::string> ref; // assume std::map works correctly :)
		for(int x=0;x<2;++x) {
			for(int i=0;i<77777;++i) {
				uint64_t k = rand();
				while ((k == 0)||(ref.count(k) > 0))
					++k;
				std::string v("!");
				for(int j=0;j<(int)(k % 64);++j)
					v.push_back("0123456789"[rand() % 10]);
				ref[k] = v;
				ht.set(0xffffffffffffffffULL,v);
				std::string &vref = ht[k];
				vref = v;
				ht.erase(0xffffffffffffffffULL);
			}
			if (ht.size() != ref.size()) {
				std::cout << "FAILED! (size mismatch, original)" << std::endl;
				return -1;
			}
			{
				Hashtable<uint64_t,std::string>::Iterator i(ht);
				uint64_t *k = (uint64_t *)0;
				std::string *v = (std::string *)0;
				while(i.next(k,v)) {
					if (ref.find(*k)->second != *v) {
						std::cout << "FAILED! (data mismatch!)" << std::endl;
						return -1;
					}
				}
			}
			for(std::map<uint64_t,std::string>::const_iterator i(ref.begin());i!=ref.end();++i) {
				if (ht[i->first] != i->second) {
					std::cout << "FAILED! (data mismatch!)" << std::endl;
					return -1;
				}
			}

			Hashtable<uint64_t,std::string> ht2;
			ht2 = ht;
			Hashtable<uint64_t,std::string> ht3(ht2);
			if (ht2.size() != ref.size()) {
				std::cout << "FAILED! (size mismatch, assigned)" << std::endl;
				return -1;
			}
			if (ht3.size() != ref.size()) {
				std::cout << "FAILED! (size mismatch, copied)" << std::endl;
				return -1;
			}

			for(std::map<uint64_t,std::string>::iterator i(ref.begin());i!=ref.end();++i) {
				std::string *v = ht.get(i->first);
				if (!v) {
					std::cout << "FAILED! (key " << i->first << " not found, original)" << std::endl;
					return -1;
				}
				if (*v != i->second) {
					std::cout << "FAILED! (key " << i->first << "  not equal, original)" << std::endl;
					return -1;
				}
				v = ht2.get(i->first);
				if (!v) {
					std::cout << "FAILED! (key " << i->first << "  not found, assigned)" << std::endl;
					return -1;
				}
				if (*v != i->second) {
					std::cout << "FAILED! (key " << i->first << "  not equal, assigned)" << std::endl;
					return -1;
				}
				v = ht3.get(i->first);
				if (!v) {
					std::cout << "FAILED! (key " << i->first << "  not found, copied)" << std::endl;
					return -1;
				}
				if (*v != i->second) {
					std::cout << "FAILED! (key " << i->first << "  not equal, copied)" << std::endl;
					return -1;
				}
			}
			{
				uint64_t *k;
				std::string *v;
				Hashtable<uint64_t,std::string>::Iterator i(ht);
				unsigned long ic = 0;
				while (i.next(k,v)) {
					if (ref[*k] != *v) {
						std::cout << "FAILED! (iterate)" << std::endl;
						return -1;
					}
					++ic;
				}
				if (ic != ht.size()) {
					std::cout << "FAILED! (iterate coverage)" << std::endl;
					return -1;
				}
			}
			for(std::map<uint64_t,std::string>::iterator i(ref.begin());i!=ref.end();) {
				if (!ht.get(i->first)) {
					std::cout << "FAILED! (erase, check if exists)" << std::endl;
					return -1;
				}
				ht.erase(i->first);
				if (ht.get(i->first)) {
					std::cout << "FAILED! (erase, check if erased)" << std::endl;
					return -1;
				}
				ref.erase(i++);
				if (ht.size() != ref.size()) {
					std::cout << "FAILED! (erase, size)" << std::endl;
					return -1;
				}
			}
			if (!ht.empty()) {
				std::cout << "FAILED! (erase, empty)" << std::endl;
				return -1;
			}
			for(int i=0;i<10000;++i) {
				uint64_t k = rand();
				while ((k == 0)||(ref.count(k) > 0))
					++k;
				std::string v;
				for(int j=0;j<(int)(k % 64);++j)
					v.push_back("0123456789"[rand() % 10]);
				ht.set(k,v);
				ref[k] = v;
			}
			if (ht.size() != ref.size()) {
				std::cout << "FAILED! (second populate)" << std::endl;
				return -1;
			}
			ht.clear();
			ref.clear();
			if (ht.size() != ref.size()) {
				std::cout << "FAILED! (clear)" << std::endl;
				return -1;
			}
			for(int i=0;i<10000;++i) {
				uint64_t k = rand();
				while ((k == 0)||(ref.count(k) > 0))
					++k;
				std::string v;
				for(int j=0;j<(int)(k % 64);++j)
					v.push_back("0123456789"[rand() % 10]);
				ht.set(k,v);
				ref[k] = v;
			}
			{
				Hashtable<uint64_t,std::string>::Iterator i(ht);
				uint64_t *k;
				std::string *v;
				while (i.next(k,v))
					ht.erase(*k);
			}
			ref.clear();
			if (ht.size() != ref.size()) {
				std::cout << "FAILED! (clear by iterate, " << ht.size() << ")" << std::endl;
				return -1;
			}
		}
	}
	std::cout << "PASS" << std::endl;
#endif

	std::cout << "[other] Testing/fuzzing Dictionary... ";
	std::cout.flush();
	for (int k = 0; k < 1000; ++k) {
		Dictionary<8194>* test = new Dictionary<8194>();
		char key[32][16];
		char value[32][128];
		memset(key, 0, sizeof(key));
		memset(value, 0, sizeof(value));
		for (unsigned int q = 0; q < 32; ++q) {
			Utils::hex((uint32_t)((rand() % 1000) + (q * 1000)), key[q]);
			int r = rand() % 128;
			for (int x = 0; x < r; ++x)
				value[q][x] = ("0123456789\0\t\r\n= ")[rand() % 16];
			value[q][r] = (char)0;
			test->add(key[q], value[q], r);
		}
		for (unsigned int q = 0; q < 1024; ++q) {
			int r = rand() % 32;
			char tmp[128];
			if (test->get(key[r], tmp, sizeof(tmp)) >= 0) {
				if (strcmp(value[r], tmp)) {
					std::cout << "FAILED (invalid value '" << value[r] << "' != '" << tmp << "')!" << std::endl;
					return -1;
				}
			}
			else {
				std::cout << "FAILED (can't find key '" << key[r] << "')!" << std::endl;
				return -1;
			}
		}
		delete test;
	}
	int foo = 0;
	volatile int* volatile bar = &foo;	 // force compiler not to optimize out test.get() below
	for (int k = 0; k < 200; ++k) {
		int r = rand() % 8194;
		unsigned char* tmp = new unsigned char[8194];
		for (int q = 0; q < r; ++q)
			tmp[q] = (unsigned char)((rand() % 254) + 1);	   // don't put nulls since those will always just terminate scan
		tmp[r] = (r % 32) ? (char)(rand() & 0xff) : (char)0;   // every 32nd iteration don't terminate the string maybe...
		Dictionary<8194>* test = new Dictionary<8194>((const char*)tmp);
		for (unsigned int q = 0; q < 100; ++q) {
			char tmp[128];
			for (unsigned int x = 0; x < 128; ++x)
				tmp[x] = (char)(rand() & 0xff);
			tmp[127] = (char)0;
			char value[8194];
			*bar += test->get(tmp, value, sizeof(value));
		}
		delete test;
		delete[] tmp;
	}
	std::cout << "PASS (junk value to prevent optimization-out of test: " << foo << ")" << std::endl;

	return 0;
}

#define ZT_TEST_PHY_NUM_UDP_PACKETS			 10000
#define ZT_TEST_PHY_UDP_PACKET_SIZE			 1000
#define ZT_TEST_PHY_NUM_VALID_TCP_CONNECTS	 10
#define ZT_TEST_PHY_NUM_INVALID_TCP_CONNECTS 2
#define ZT_TEST_PHY_TCP_MESSAGE_SIZE		 1000000
#define ZT_TEST_PHY_TIMEOUT_MS				 20000
static unsigned long phyTestUdpPacketCount = 0;
static unsigned long phyTestTcpByteCount = 0;
static unsigned long phyTestTcpConnectSuccessCount = 0;
static unsigned long phyTestTcpConnectFailCount = 0;
static unsigned long phyTestTcpAcceptCount = 0;
struct TestPhyHandlers;
static Phy<TestPhyHandlers*>* testPhyInstance = (Phy<TestPhyHandlers*>*)0;
struct TestPhyHandlers {
	inline void phyOnDatagram(PhySocket* sock, void** uptr, const struct sockaddr* localAddr, const struct sockaddr* from, void* data, unsigned long len)
	{
		++phyTestUdpPacketCount;
	}

	inline void phyOnTcpConnect(PhySocket* sock, void** uptr, bool success)
	{
		if (success) {
			++phyTestTcpConnectSuccessCount;
		}
		else {
			++phyTestTcpConnectFailCount;
		}
	}

	inline void phyOnTcpAccept(PhySocket* sockL, PhySocket* sockN, void** uptrL, void** uptrN, const struct sockaddr* from)
	{
		++phyTestTcpAcceptCount;
		*uptrN = new std::string(ZT_TEST_PHY_TCP_MESSAGE_SIZE, (char)0xff);
		testPhyInstance->setNotifyWritable(sockN, true);
	}

	inline void phyOnTcpClose(PhySocket* sock, void** uptr)
	{
		delete (std::string*)*uptr;	  // delete testMessage if any
	}

	inline void phyOnTcpData(PhySocket* sock, void** uptr, void* data, unsigned long len)
	{
		phyTestTcpByteCount += len;
	}

	inline void phyOnTcpWritable(PhySocket* sock, void** uptr)
	{
		std::string* testMessage = (std::string*)*uptr;
		if ((testMessage) && (testMessage->length() > 0)) {
			long sent = testPhyInstance->streamSend(sock, (const void*)testMessage->data(), (unsigned long)testMessage->length(), true);
			if (sent > 0)
				testMessage->erase(0, sent);
		}
		if ((! testMessage) || (! testMessage->length())) {
			testPhyInstance->close(sock, true);
		}
	}

#ifdef __UNIX_LIKE__
	inline void phyOnUnixAccept(PhySocket* sockL, PhySocket* sockN, void** uptrL, void** uptrN)
	{
	}
	inline void phyOnUnixClose(PhySocket* sock, void** uptr)
	{
	}
	inline void phyOnUnixData(PhySocket* sock, void** uptr, void* data, unsigned long len)
	{
	}
	inline void phyOnUnixWritable(PhySocket* sock, void** uptr)
	{
	}
#endif	 // __UNIX_LIKE__

	inline void phyOnFileDescriptorActivity(PhySocket* sock, void** uptr, bool readable, bool writable)
	{
	}
};
static int testPhy()
{
	char udpTestPayload[ZT_TEST_PHY_UDP_PACKET_SIZE];
	memset(udpTestPayload, 0xff, sizeof(udpTestPayload));

	struct sockaddr_in bindaddr;
	memset(&bindaddr, 0, sizeof(bindaddr));
	bindaddr.sin_family = AF_INET;
	bindaddr.sin_port = Utils::hton((uint16_t)60002);
	bindaddr.sin_addr.s_addr = Utils::hton((uint32_t)0x7f000001);
	struct sockaddr_in invalidAddr;
	memset(&bindaddr, 0, sizeof(bindaddr));
	bindaddr.sin_family = AF_INET;
	bindaddr.sin_port = Utils::hton((uint16_t)60004);
	bindaddr.sin_addr.s_addr = Utils::hton((uint32_t)0x7f000001);

	std::cout << "[phy] Creating phy endpoint..." << std::endl;
	TestPhyHandlers testPhyHandlers;
	testPhyInstance = new Phy<TestPhyHandlers*>(&testPhyHandlers, false, true);

	std::cout << "[phy] Binding UDP listen socket to 127.0.0.1/60002... ";
	PhySocket* udpListenSock = testPhyInstance->udpBind((const struct sockaddr*)&bindaddr);
	if (! udpListenSock) {
		std::cout << "FAILED." << std::endl;
		return -1;
	}
	std::cout << "OK" << std::endl;

	std::cout << "[phy] Binding TCP listen socket to 127.0.0.1/60002... ";
	PhySocket* tcpListenSock = testPhyInstance->tcpListen((const struct sockaddr*)&bindaddr);
	if (! tcpListenSock) {
		std::cout << "FAILED." << std::endl;
		return -1;
	}
	std::cout << "OK" << std::endl;

	unsigned long phyTestUdpPacketsSent = 0;
	unsigned long phyTestTcpValidConnectionsAttempted = 0;
	unsigned long phyTestTcpInvalidConnectionsAttempted = 0;

	std::cout << "[phy] Testing UDP send/receive... ";
	std::cout.flush();
	uint64_t timeoutAt = OSUtils::now() + ZT_TEST_PHY_TIMEOUT_MS;
	while ((OSUtils::now() < timeoutAt) && (phyTestUdpPacketCount < ZT_TEST_PHY_NUM_UDP_PACKETS)) {
		if (phyTestUdpPacketsSent < ZT_TEST_PHY_NUM_UDP_PACKETS) {
			if (! testPhyInstance->udpSend(udpListenSock, (const struct sockaddr*)&bindaddr, udpTestPayload, sizeof(udpTestPayload))) {
				std::cout << "FAILED." << std::endl;
				return -1;
			}
			else
				++phyTestUdpPacketsSent;
		}
		testPhyInstance->poll(100);
	}
	std::cout << "got " << phyTestUdpPacketCount << " packets, OK" << std::endl;

	std::cout << "[phy] Testing TCP... ";
	std::cout.flush();
	timeoutAt = OSUtils::now() + ZT_TEST_PHY_TIMEOUT_MS;
	while ((OSUtils::now() < timeoutAt) && (phyTestTcpByteCount < (ZT_TEST_PHY_NUM_VALID_TCP_CONNECTS * ZT_TEST_PHY_TCP_MESSAGE_SIZE))) {
		if (phyTestTcpValidConnectionsAttempted < ZT_TEST_PHY_NUM_VALID_TCP_CONNECTS) {
			++phyTestTcpValidConnectionsAttempted;
			bool connected = false;
			if (! testPhyInstance->tcpConnect((const struct sockaddr*)&bindaddr, connected, (void*)0, true))
				++phyTestTcpConnectFailCount;
		}
		if (phyTestTcpInvalidConnectionsAttempted < ZT_TEST_PHY_NUM_INVALID_TCP_CONNECTS) {
			++phyTestTcpInvalidConnectionsAttempted;
			bool connected = false;
			if (! testPhyInstance->tcpConnect((const struct sockaddr*)&invalidAddr, connected, (void*)0, true))
				++phyTestTcpConnectFailCount;
		}
		testPhyInstance->poll(100);
	}
	if (phyTestTcpByteCount < (ZT_TEST_PHY_NUM_VALID_TCP_CONNECTS * ZT_TEST_PHY_TCP_MESSAGE_SIZE)) {
		std::cout << "got " << phyTestTcpConnectSuccessCount << " connect successes, " << phyTestTcpConnectFailCount << " failures, and " << phyTestTcpByteCount << " bytes, FAILED." << std::endl;
		return -1;
	}
	else {
		std::cout << "got " << phyTestTcpConnectSuccessCount << " connect successes, " << phyTestTcpConnectFailCount << " failures, and " << phyTestTcpByteCount << " bytes, OK" << std::endl;
	}

	return 0;
}

#ifdef __WINDOWS__
int __cdecl _tmain(int argc, _TCHAR* argv[])
#else
int main(int argc, char** argv)
#endif
{
	int r = 0;

#ifdef __WINDOWS__
	WSADATA wsaData;
	WSAStartup(MAKEWORD(2, 2), &wsaData);
#endif

	// Code to generate the C25519 test vectors -- did this once and then
	// put these up top so that we can ensure that every platform produces
	// the same result.
	/*
	for(int k=0;k<32;++k) {
		ECC::Pair p1 = ECC::generate();
		ECC::Pair p2 = ECC::generate();
		unsigned char agg[64];
		ECC::agree(p1,p2.pub,agg,64);
		ECC::Signature sig1 = ECC::sign(p1,agg,64);
		ECC::Signature sig2 = ECC::sign(p2,agg,64);
		printf("{{");
		for(int i=0;i<64;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)p1.pub.data[i]);
		printf("},{");
		for(int i=0;i<64;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)p1.priv.data[i]);
		printf("},{");
		for(int i=0;i<64;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)p2.pub.data[i]);
		printf("},{");
		for(int i=0;i<64;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)p2.priv.data[i]);
		printf("},{");
		for(int i=0;i<64;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)agg[i]);
		printf("},{");
		for(int i=0;i<96;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)sig1.data[i]);
		printf("},{");
		for(int i=0;i<96;++i)
			printf("%s0x%.2x",((i > 0) ? "," : ""),(unsigned int)sig2.data[i]);
		printf("}}\n");
	}
	exit(0);
	*/

	std::cout << "[info] sizeof(void *) == " << sizeof(void*) << std::endl;
	std::cout << "[info] OSUtils::now() == " << OSUtils::now() << std::endl;
	std::cout << "[info] hardware concurrency == " << std::thread::hardware_concurrency() << std::endl;
	std::cout << "[info] sizeof(NetworkConfig) == " << sizeof(ZeroTier::NetworkConfig) << std::endl;

	srand((unsigned int)time(0));

	///*
	r |= testOther();
	r |= testCrypto();
	r |= testPacket();
	r |= testIdentity();
	r |= testCertificate();
	r |= testPhy();
	//*/

	if (r)
		std::cout << std::endl << "SOMETHING FAILED!" << std::endl;

	/*
#ifdef ZT_USE_MINIUPNPC
	std::cout << std::endl;
	std::cout << "[portmapper] Starting port mapper and waiting forever... use CTRL+C to exit. (enable ZT_PORTMAPPER_TRACE in PortMapper.cpp for output)" << std::endl;
	PortMapper mapper(12345,"ZeroTier/__selftest");
	Thread::sleep(0xffffffff);
#endif
	*/

	return r;
}
