/*
 * Copyright (C) 2004-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_gen_bus.h"

#ifdef NEED_SMM
#define INT_STATE_C	unsigned int state,
#define INT_C		unsigned int,
#define STATE_C		state,
#else
#define INT_STATE_C
#define INT_C
#define STATE_C
#endif

#ifdef NEED_ADDR_TYPE
int
bus_(addr_type)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int type
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t, unsigned int);
		void *func_s;

		if (nr == b->member_count) {
			/* Not found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		func = b->member[nr].f->addr_type;
		func_s = b->member[nr].s;
		if (func
		 && func(func_s, STATE_C addr, type) == 0) {
			return 0;
		}
	}
}

int
bus_(read_data)(
	struct bus *b,
	void *s,
	unsigned int bs,
	bus_data_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, bus_data_t *);
		void *func_s;

		if (nr == b->member_count) {
			/* Not found. */
			*valp = (bus_data_t) -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		func = b->member[nr].f->read_data;
		func_s = b->member[nr].s;
		if (func
		 && func(func_s, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(write_data)(
	struct bus *b,
	void *s,
	unsigned int bs,
	bus_data_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, bus_data_t);
		void *func_s;

		if (nr == b->member_count) {
			/* Not found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		func = b->member[nr].f->write_data;
		func_s = b->member[nr].s;
		if (func
		 && func(func_s, bs, val) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_ADDR_TYPE */

#ifdef NEED_C0
int
bus_(c0r)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, unsigned int, bus_data_t *);

		if (nr == b->member_count) {
			*valp = (bus_data_t) -1;
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->c0r;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(c0w)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, unsigned int, bus_data_t);

		if (nr == b->member_count) {
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->c0w;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_C0 */

#ifdef NEED_C1
int
bus_(c1r)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, unsigned int, bus_data_t *);

		if (nr == b->member_count) {
			*valp = (bus_data_t) -1;
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->c1r;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(c1w)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	unsigned int nr;

	assert((addr & (sizeof(bus_data_t) - 1)) == 0);

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, unsigned int, bus_data_t);

		if (nr == b->member_count) {
			return 1;
		}
		if (! b->member[nr].f
		 || b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->c1w;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_C1 */

#ifdef NEED_IO_LEN
int
bus_(inb)(
	struct bus *b,
	void *s,
	unsigned char *valp,
	bus_addr_t port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xff;
			return -1;
		}
		if (! b->member[nr].f->inb
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->inb(b->member[nr].s, valp, port) == 0) {
			return 0;
		}
	}
}

int
bus_(inw)(
	struct bus *b,
	void *s,
	unsigned short *valp,
	bus_addr_t port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			*valp = 0xffff;
			return -1;
		}
		if (! b->member[nr].f->inw
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->inw(b->member[nr].s, valp, port) == 0) {
			return 0;
		}
	}
}

int
bus_(outb)(
	struct bus *b,
	void *s,
	unsigned char val,
	bus_addr_t port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->outb
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->outb(b->member[nr].s, val, port) == 0) {
			return 0;
		}
	}
}

int
bus_(outw)(
	struct bus *b,
	void *s,
	unsigned short val,
	bus_addr_t port
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return -1;
		}
		if (! b->member[nr].f->outw
		 || b->member[nr].s == s) {
			continue;
		}
		if (b->member[nr].f->outw(b->member[nr].s, val, port) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_IO_LEN */

#ifdef NEED_IO_BS
static inline __attribute__((always_inline)) unsigned int
bus_(_io_hash)(bus_addr_t port, unsigned int bs)
{
	unsigned int val;

	val = port / sizeof(bus_addr_t);
	val ^= bs;
	val %= BUS_HASH_SIZE;

	return val;
}

static inline __attribute__((always_inline)) int
bus_(_ior_lookup)(
	struct bus *b,
	bus_addr_t port,
	unsigned int bs,
	int (**f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t *valp),
	void **s
)
{
	unsigned int hash;
	struct bus_(io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (unlikely(! m)) {
			/* Not found. */
			return 0;
		}
		if (likely(m->type == BUS_(TYPE_IOR)
		 && m->port == port
		 && m->bs == bs)) {
			/* Found. */
			*f = m->ior;
			*s = m->s;
			return 1;
		}
	}
}

static void
bus_(_ior_add)(
	struct bus *b,
	bus_addr_t port,
	unsigned int bs,
	int (*f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t *valp),
	void *s
)
{
	unsigned int hash;
	struct bus_(io) *m;

#ifdef NEED_LOCK
	lock_spinlock(&b->cache_lock);
#endif

	m = b->io_lru_last;

	/* Remove from LRU list. */
	m->lru_prev->lru_next = 0;
	b->io_lru_last = m->lru_prev;

	if (m->port != -1) {
		/* Remove from HASH list. */
		hash = bus_(_io_hash)(m->port, m->bs);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->io_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->io_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_IOR);
	m->port = port;
	m->bs = bs;
	m->ior = f;
	m->s = s;

	/* Add to HASH list. */
	hash = bus_(_io_hash)(port, bs);
	m->hash_prev = 0;
	m->hash_next = b->io_hash_first[hash];
	b->io_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->io_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->io_lru_first;
	b->io_lru_first = m;
	m->lru_next->lru_prev = m;

#ifdef NEED_LOCK
	unlock_spinlock(&b->cache_lock);
#endif
}

int
bus_(ior_info)(
	struct bus *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*info_f)(void *, bus_addr_t, unsigned int,
			int (**)(void *, bus_addr_t, unsigned int, bus_data_t *),
			void **);
		void *info_s;

		if (nr == b->member_count) {
			/* No info found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		info_f = b->member[nr].f->ior_info;
		info_s = b->member[nr].s;
		if (info_f
		 && info_f(info_s, port, bs, cfp, csp) == 0) {
			return 0;
		}
	}
}

void
bus_(ior_info_flush)(
	struct bus *b,
	void *s,
	bus_addr_t port,
	unsigned int bs
)
{
	unsigned int hash;
	struct bus_(io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/* Not found. */
			return;
		}
		if (m->type == BUS_(TYPE_IOR)
		 && port == m->port
		 && (bs == m->bs
		  || bs == 0)) {
			/* Found. */
			/* Remove from LRU list. */
#ifdef NEED_LOCK
			lock_spinlock(&b->cache_lock);
#endif

			if (m->lru_prev) {
				m->lru_prev->lru_next = m->lru_next;
			} else {
				b->io_lru_first = m->lru_next;
			}
			if (m->lru_next) {
				m->lru_next->lru_prev = m->lru_prev;
			} else {
				b->io_lru_last = m->lru_prev;
			}

			/* Remove from HASH list. */
			hash = bus_(_io_hash)(port, bs);
			if (m->hash_prev) {
				m->hash_prev->hash_next = m->hash_next;
			} else {
				b->io_hash_first[hash] = m->hash_next;
			}
			if (m->hash_next) {
				m->hash_next->hash_prev = m->hash_prev;
			} else {
				b->io_hash_last[hash] = m->hash_prev;
			}

			/* Remove info. */
			m->port = -1;
			m->bs = -1;

			/* Don't add empty entry to HASH list. */

			/* Add to LRU list. */
			m->lru_prev = b->io_lru_last;
			m->lru_next = 0;
			m->lru_prev->lru_next = m;
			b->io_lru_last = m;

#ifdef NEED_LOCK
			unlock_spinlock(&b->cache_lock);
#endif
			return;
		}
	}
}

static int
bus_(_ior_dummy)(
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	*valp = (bus_data_t) -1;
	return 1;
}

int
bus_(ior)(
	struct bus *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t *);
	void *cs;
	unsigned int nr;

	assert(! (port & (sizeof(bus_data_t) - 1)));

	if (bus_(_ior_lookup)(b, port, bs, &cf, &cs)) {
		return (*cf)(cs, port, bs, valp);
	}

	if (bus_(ior_info)(b, s, port, bs, &cf, &cs) == 0) {
		bus_(_ior_add)(b, port, bs, cf, cs);
		return cf(cs, port, bs, valp);
	}

#if 0
	fprintf(stderr, "INFO: No IOR info for 0x%04x 0x%x\n", port, bs);
#endif

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			break;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		cf = b->member[nr].f->ior;
		cs = b->member[nr].s;
		if (cf
		 && cf(cs, port, bs, valp) == 0) {
			bus_(_ior_add)(b, port, bs, cf, cs);
			return 0;
		}
	}

	cf = bus_(_ior_dummy);
	cs = 0;
	bus_(_ior_add)(b, port, bs, cf, cs);
	return 1;
}

static inline __attribute__((always_inline)) int
bus_(_iow_lookup)(
	struct bus *b,
	bus_addr_t port,
	unsigned int bs,
	int (**f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t val),
	void **s
)
{
	unsigned int hash;
	struct bus_(io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (unlikely(! m)) {
			/* Not found. */
			return 0;
		}
		if (likely(m->type == BUS_(TYPE_IOW)
		 && m->port == port
		 && m->bs == bs)) {
			/* Found. */
			*f = m->iow;
			*s = m->s;
			return 1;
		}
	}
}

static void
bus_(_iow_add)(
	struct bus *b,
	bus_addr_t port,
	unsigned int bs,
	int (*f)(void *s, bus_addr_t port, unsigned int bs, bus_data_t val),
	void *s
)
{
	unsigned int hash;
	struct bus_(io) *m;

#ifdef NEED_LOCK
	lock_spinlock(&b->cache_lock);
#endif

	m = b->io_lru_last;

	/* Remove from LRU list. */
	m->lru_prev->lru_next = 0;
	b->io_lru_last = m->lru_prev;

	if (m->port != -1) {
		/* Remove from HASH list. */
		hash = bus_(_io_hash)(m->port, m->bs);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->io_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->io_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_IOW);
	m->port = port;
	m->bs = bs;
	m->iow = f;
	m->s = s;

	/* Add to HASH list. */
	hash = bus_(_io_hash)(port, bs);
	m->hash_prev = 0;
	m->hash_next = b->io_hash_first[hash];
	b->io_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->io_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->io_lru_first;
	b->io_lru_first = m;
	m->lru_next->lru_prev = m;

#ifdef NEED_LOCK
	unlock_spinlock(&b->cache_lock);
#endif
}

int
bus_(iow_info)(
	struct bus *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*info_f)(void *, bus_addr_t, unsigned int,
			int (**)(void *, bus_addr_t, unsigned int, bus_data_t),
			void **);
		void *info_s;

		if (nr == b->member_count) {
			/* No info found. */
			return -1;
		}
		if (b->member[nr].s == s) {
			/* Don't ask myself. */
			continue;
		}
		info_f = b->member[nr].f->iow_info;
		info_s = b->member[nr].s;
		if (info_f
		 && info_f(info_s, port, bs, cfp, csp) == 0) {
			return 0;
		}
	}
}

void
bus_(iow_info_flush)(
	struct bus *b,
	void *s,
	bus_addr_t port,
	unsigned int bs
)
{
	unsigned int hash;
	struct bus_(io) *m;

	hash = bus_(_io_hash)(port, bs);

	for (m = b->io_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/* Not found. */
			return;
		}
		if (m->type == BUS_(TYPE_IOW)
		 && port == m->port
		 && (bs == m->bs
		  || bs == 0)) {
			/* Found. */
			/* Remove from LRU list. */
#ifdef NEED_LOCK
			lock_spinlock(&b->cache_lock);
#endif

			if (m->lru_prev) {
				m->lru_prev->lru_next = m->lru_next;
			} else {
				b->io_lru_first = m->lru_next;
			}
			if (m->lru_next) {
				m->lru_next->lru_prev = m->lru_prev;
			} else {
				b->io_lru_last = m->lru_prev;
			}

			/* Remove from HASH list. */
			hash = bus_(_io_hash)(port, bs);
			if (m->hash_prev) {
				m->hash_prev->hash_next = m->hash_next;
			} else {
				b->io_hash_first[hash] = m->hash_next;
			}
			if (m->hash_next) {
				m->hash_next->hash_prev = m->hash_prev;
			} else {
				b->io_hash_last[hash] = m->hash_prev;
			}

			/* Remove info. */
			m->port = -1;
			m->bs = -1;

			/* Don't add empty entry to HASH list. */

			/* Add to LRU list. */
			m->lru_prev = b->io_lru_last;
			m->lru_next = 0;
			m->lru_prev->lru_next = m;
			b->io_lru_last = m;

#ifdef NEED_LCOK
			unlock_spinlock(&b->cache_lock);
#endif
			return;
		}
	}
}

static int
bus_(_iow_dummy)(
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	return 1;
}

int
bus_(iow)(
	struct bus *b,
	void *s,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t);
	void *cs;
	unsigned int nr;

	assert((port & (sizeof(bus_data_t) - 1)) == 0);

	if (bus_(_iow_lookup)(b, port, bs, &cf, &cs)) {
		return cf(cs, port, bs, val);
	}

	if (bus_(iow_info)(b, s, port, bs, &cf, &cs) == 0) {
		bus_(_iow_add)(b, port, bs, cf, cs);
		return cf(cs, port, bs, val);
	}

#if 0
	fprintf(stderr, "INFO: No IOW info for 0x%04x 0x%x\n", port, bs);
#endif

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			break;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		cf = b->member[nr].f->iow;
		cs = b->member[nr].s;
		if (cf
		 && cf(cs, port, bs, val) == 0) {
			bus_(_iow_add)(b, port, bs, cf, cs);
			return 0;
		}
	}

	cf = bus_(_iow_dummy);
	cs = 0;
	bus_(_iow_add)(b, port, bs, cf, cs);
	return 1;
}
#endif /* NEED_IO_BS */

#ifdef NEED_MEM_LEN
int
bus_(readb)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	uint8_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, uint8_t *);

		if (nr == b->member_count) {
			*valp = 0xff;
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->readb;
		if (func
		 && func(b->member[nr].s, addr, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(readw)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	uint16_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, uint16_t *);

		if (nr == b->member_count) {
			*valp = 0xffff;
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->readw;
		if (func
		 && func(b->member[nr].s, addr, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(readl)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	uint32_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, uint32_t *);

		if (nr == b->member_count) {
			*valp = 0xffffffff;
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->readl;
		if (func
		 && func(b->member[nr].s, addr, valp) == 0) {
			return 0;
		}
	}
}

int
bus_(writeb)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	uint8_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, uint8_t);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->writeb;
		if (func
		 && func(b->member[nr].s, addr, val) == 0) {
			return 0;
		}
	}
}

int
bus_(writew)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	uint16_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, uint16_t);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->writew;
		if (func
		 && func(b->member[nr].s, addr, val) == 0) {
			return 0;
		}
	}
}

int
bus_(writel)(
	struct bus *b,
	void *s,
	bus_addr_t addr,
	uint32_t val
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, bus_addr_t, uint32_t);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->writel;
		if (func
		 && func(b->member[nr].s, addr, val) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_MEM_LEN */

#ifdef NEED_MEM_BS
static inline unsigned int
bus_(_map_hash)(bus_addr_t addr)
{
	return (addr >> 12) % BUS_HASH_SIZE;
}

static void
bus_(_map_flush)(
	struct bus *b,
	struct bus_(map) *m
)
{
	unsigned int hash;

	/* Remove from LRU list. */
#ifdef NEED_LOCK
	lock_spinlock(&b->cache_lock);
#endif

	if (m->lru_prev) {
		m->lru_prev->lru_next = m->lru_next;
	} else {
		b->map_lru_first = m->lru_next;
	}
	if (m->lru_next) {
		m->lru_next->lru_prev = m->lru_prev;
	} else {
		b->map_lru_last = m->lru_prev;
	}

	/* Remove from HASH list. */
	hash = bus_(_map_hash)(m->addr);
	if (m->hash_prev) {
		m->hash_prev->hash_next = m->hash_next;
	} else {
		b->map_hash_first[hash] = m->hash_next;
	}
	if (m->hash_next) {
		m->hash_next->hash_prev = m->hash_prev;
	} else {
		b->map_hash_last[hash] = m->hash_prev;
	}

	/* Unmap page. */
#ifdef NEED_SMM
	m->state = -1;
#endif
	m->addr = -1;

	/* Don't add empty entry to HASH list. */

	/* Add to LRU list. */
	m->lru_prev = b->map_lru_last;
	m->lru_next = 0;
	m->lru_prev->lru_next = m;
	b->map_lru_last = m;

#ifdef NEED_LOCK
	unlock_spinlock(&b->cache_lock);
#endif
}
#endif /* NEED_MEM_BS */

#ifdef NEED_MEM_BS
static inline __attribute__((always_inline)) struct bus_(map) *
bus_(_map_lookup)(
	struct bus *b,
	unsigned int type,
	INT_STATE_C
	bus_addr_t addr
)
{
	unsigned int hash;
	struct bus_(map) *m;

	addr &= ~0xfff;

	hash = bus_(_map_hash)(addr);

	for (m = b->map_hash_first[hash]; ; m = m->hash_next) {
		if (! m) {
			/*
			 * Not Found
			 */
			return NULL;
		}
		if (m->type == type
#ifdef NEED_SMM
		 && m->state == state
#endif
		 && m->addr == addr) {
			/*
			 * Found
			 */
			/* Don't update LRU list. */
			/* Costs too much... */
			return m;
		}
	}
}

static inline __attribute__((always_inline)) struct bus_(map) *
bus_(_map_r_add)(
	struct bus *b,
	INT_STATE_C
	bus_addr_t addr
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t *);
	void *cs;
	char *haddr;
	unsigned int hash;
	struct bus_(map) *m;

	addr &= ~0xfff;

	if (bus_(map_r_check)(b, b, STATE_C addr)
	 || bus_(map_r)(b, b, STATE_C addr, &cf, &cs, &haddr)) {
		cf = NULL;
		cs = NULL;
		haddr = NULL;
	}

	/* Get entry from LRU list. */
#ifdef NEED_LOCK
	lock_spinlock(&b->cache_lock);
#endif

	m = b->map_lru_last;

	/* Remove from LRU list. */
	assert(m->lru_prev);
	m->lru_prev->lru_next = 0;
	b->map_lru_last = m->lru_prev;

	if (m->addr != -1) {
		/* Remove from HASH list. */
		hash = bus_(_map_hash)(m->addr);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->map_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->map_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_MR);
#ifdef NEED_SMM
	m->state = state;
#endif
	m->addr = addr;
	m->mr_map = (bus_data_t *) haddr;
	m->mr_func = cf;
	m->mr_cpssp = cs;

	/* Add to HASH list. */
	hash = bus_(_map_hash)(addr);
	m->hash_prev = 0;
	m->hash_next = b->map_hash_first[hash];
	b->map_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->map_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->map_lru_first;
	b->map_lru_first = m;
	m->lru_next->lru_prev = m;

#ifdef NEED_LOCK
	unlock_spinlock(&b->cache_lock);
#endif

	return m;
}

int
bus_(mr)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(map) *m;
	const bus_data_t *from;
	unsigned int nr;

	m = bus_(_map_lookup)(b, BUS_(TYPE_MR), STATE_C addr);
	if (! m) {
		m = bus_(_map_r_add)(b, STATE_C addr);
	}
	if (m->mr_map) {
		from = (const bus_data_t *) m->mr_map
				+ (addr & 0xfff) / sizeof(bus_data_t);

		/* No need to obey `bs'. */
		*valp = *from;

		return 0;

	} else if (m->mr_func) {
		return m->mr_func(m->mr_cpssp, addr, bs, valp);
	}
// fprintf(stderr, "%s %08x\n", __FUNCTION__, (uint32_t) addr);

	/*
	 * Not mappable. Call function...
	 * Should go away. FIXME
	 */
	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t, unsigned int, bus_data_t *);

		if (nr == b->member_count) {
			*valp = (bus_data_t) -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->mr;
		if (func
		 && func(b->member[nr].s, STATE_C addr, bs, valp) == 0) {
			return 0;
		}
	}
}

static inline __attribute__((always_inline)) struct bus_(map) *
bus_(_map_w_add)(
	struct bus *b,
	INT_STATE_C
	bus_addr_t addr
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t);
	void *cs;
	char *haddr;
	unsigned int hash;
	struct bus_(map) *m;

	addr &= ~0xfff;

	if (bus_(map_w_check)(b, b, STATE_C addr)
	 || bus_(map_w)(b, b, STATE_C addr, &cf, &cs, &haddr)) {
		cf = NULL;
		cs = NULL;
		haddr = NULL;
	}

#ifdef NEED_LOCK
	lock_spinlock(&b->cache_lock);
#endif

	/* Get entry from LRU list. */
	m = b->map_lru_last;

	/* Remove from LRU list. */
	assert(m->lru_prev);
	m->lru_prev->lru_next = 0;
	b->map_lru_last = m->lru_prev;

	if (m->addr != -1) {
		/* Remove from HASH list. */
		hash = bus_(_map_hash)(m->addr);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->map_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->map_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_MW);
#ifdef NEED_SMM
	m->state = state;
#endif
	m->addr = addr;
	m->mw_map = (bus_data_t *) haddr;
	m->mw_func = cf;
	m->mw_cpssp = cs;

	/* Add to HASH list. */
	hash = bus_(_map_hash)(addr);
	m->hash_prev = 0;
	m->hash_next = b->map_hash_first[hash];
	b->map_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->map_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->map_lru_first;
	b->map_lru_first = m;
	m->lru_next->lru_prev = m;

#ifdef NEED_LOCK
	unlock_spinlock(&b->cache_lock);
#endif

	return m;
}

int
bus_(mw)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(map) *m;
	bus_data_t *to;
	unsigned int nr;

	m = bus_(_map_lookup)(b, BUS_(TYPE_MW), STATE_C addr);
	if (! m) {
		m = bus_(_map_w_add)(b, STATE_C addr);
	}
	if (m->mw_map) {
		to = (bus_data_t *) m->mw_map + (addr & 0xfff) / sizeof(bus_data_t);

		if ((sizeof(bus_data_t) == 1 && bs == 0x01)
		 || (sizeof(bus_data_t) == 2 && bs == 0x03)
		 || (sizeof(bus_data_t) == 4 && bs == 0x0f)
		 || (sizeof(bus_data_t) == 8 && bs == 0xff)) {
			/* Shortcut... */
			*to = val;

		} else {
			/*
			 * Instead of using uint8_t, uint16_t, ... we
			 * would like to always use bus_data_t. Unfortunately
			 * GCC produces warnings.
			 */
			if ((bs >> 0) & 1) {
				*to &= ~((uint64_t) 0xff << 0);
				*to |= val & ((uint64_t) 0xff << 0);
			}
			if (2 <= sizeof(bus_data_t)
			 && (bs >> 1) & 1) {
				*to &= ~((uint64_t) 0xff << 8);
				*to |= val & ((uint64_t) 0xff << 8);
			}
			if (3 <= sizeof(bus_data_t)
			 && (bs >> 2) & 1) {
				*to &= ~((uint64_t) 0xff << 16);
				*to |= val & ((uint64_t) 0xff << 16);
			}
			if (4 <= sizeof(bus_data_t)
			 && (bs >> 3) & 1) {
				*to &= ~((uint64_t) 0xff << 24);
				*to |= val & ((uint64_t) 0xff << 24);
			}
			if (5 <= sizeof(bus_data_t)
			 && (bs >> 4) & 1) {
				*to &= ~((uint64_t) 0xff << 32);
				*to |= val & ((uint64_t) 0xff << 32);
			}
			if (6 <= sizeof(bus_data_t)
			 && (bs >> 5) & 1) {
				*to &= ~((uint64_t) 0xff << 40);
				*to |= val & ((uint64_t) 0xff << 40);
			}
			if (7 <= sizeof(bus_data_t)
			 && (bs >> 6) & 1) {
				*to &= ~((uint64_t) 0xff << 48);
				*to |= val & ((uint64_t) 0xff << 48);
			}
			if (8 <= sizeof(bus_data_t)
			 && (bs >> 7) & 1) {
				*to &= ~((uint64_t) 0xff << 56);
				*to |= val & ((uint64_t) 0xff << 56);
			}
		}
		return 0;

	} else if (m->mw_func) {
		return m->mw_func(m->mw_cpssp, addr, bs, val);
	}
// fprintf(stderr, "%s %08x\n", __FUNCTION__, (uint32_t) addr);

	/* Should go away. FIXME */
	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t, unsigned int, bus_data_t);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->mw;
		if (func
		 && func(b->member[nr].s, STATE_C addr, bs, val) == 0) {
			return 0;
		}
	}
}

#ifdef NEED_X
static inline __attribute__((always_inline)) struct bus_(map) *
bus_(_map_x_add)(
	struct bus *b,
	INT_STATE_C
	bus_addr_t addr
)
{
	int (*cf)(void *, bus_addr_t, unsigned int, bus_data_t *);
	void *cs;
	char *haddr;
	unsigned int hash;
	struct bus_(map) *m;

	addr &= ~0xfff;

	if (bus_(map_x)(b, b, STATE_C addr, &cf, &cs, &haddr)) {
		cf = NULL;
		cs = NULL;
		haddr = NULL;
	}

	/* Get entry from LRU list. */
#ifdef NEED_LOCK
	lock_spinlock(&b->cache_lock);
#endif

	m = b->map_lru_last;

	/* Remove from LRU list. */
	assert(m->lru_prev);
	m->lru_prev->lru_next = 0;
	b->map_lru_last = m->lru_prev;

	if (m->addr != -1) {
		/* Remove from HASH list. */
		hash = bus_(_map_hash)(m->addr);
		if (m->hash_prev) {
			m->hash_prev->hash_next = m->hash_next;
		} else {
			b->map_hash_first[hash] = m->hash_next;
		}
		if (m->hash_next) {
			m->hash_next->hash_prev = m->hash_prev;
		} else {
			b->map_hash_last[hash] = m->hash_prev;
		}
	}

	/* Add new info. */
	m->type = BUS_(TYPE_MX);
#ifdef NEED_SMM
	m->state = state;
#endif
	m->addr = addr;
	m->mx_map = (bus_data_t *) haddr;
	m->mx_func = cf;
	m->mx_cpssp = cs;

	/* Add to HASH list. */
	hash = bus_(_map_hash)(addr);
	m->hash_prev = 0;
	m->hash_next = b->map_hash_first[hash];
	b->map_hash_first[hash] = m;
	if (m->hash_next) {
		m->hash_next->hash_prev = m;
	} else {
		b->map_hash_last[hash] = m;
	}

	/* Add to LRU list. */
	m->lru_prev = 0;
	m->lru_next = b->map_lru_first;
	b->map_lru_first = m;
	m->lru_next->lru_prev = m;

#ifdef NEED_LOCK
	unlock_spinlock(&b->cache_lock);
#endif

	return m;
}

int
bus_(mx)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(map) *m;
	const bus_data_t *from;
	unsigned int nr;

	m = bus_(_map_lookup)(b, BUS_(TYPE_MX), STATE_C addr);
	if (! m) {
		m = bus_(_map_x_add)(b, STATE_C addr);
	}
	if (m->mx_map) {
		from = (const bus_data_t *) m->mx_map
				+ (addr & 0xfff) / sizeof(bus_data_t);

		/* No need to obey `bs'. */
		*valp = *from;

		return 0;

	} else if (m->mx_func) {
		return m->mx_func(m->mx_cpssp, addr, bs, valp);
	}

	/*
	 * Not mappable. Call function...
	 * Should go away. FIXME
	 */
	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int, bus_addr_t, unsigned int, bus_data_t *);

		if (nr == b->member_count) {
			*valp = (bus_data_t) -1;
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->mx;
		if (func
		 && func(b->member[nr].s, STATE_C addr, bs, valp) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_X */
#endif /* NEED_MEM_BS */

#ifdef NEED_MEM_MAP
int
bus_(map_r_check)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr
)
{
#ifdef NEED_MEM_BS
	unsigned int hash;
	struct bus_(map) *m;
#endif
	unsigned int nr;

#ifdef NEED_MEM_BS
	hash = bus_(_map_hash)(addr);

	for (m = b->map_hash_first[hash]; m; m = m->hash_next) {
		if (m->type == BUS_(TYPE_MW)
#ifdef NEED_SMM
		 && m->state == state
#endif
		 && m->addr == addr) {
			bus_(_map_flush)(b, m);
		}
	}
#endif /* NEED_MEM_BS */

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t);

		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_r_check;
		if (func
		 && func(b->member[nr].s, STATE_C addr)) {
			return 1;
		}
	}
}

int
bus_(map_r)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddr_p
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t,
				int (**)(void *, bus_addr_t, unsigned int, bus_data_t *),
				void **,
				char **);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_r;
		if (func
		 && func(b->member[nr].s, STATE_C addr, cfp, csp, haddr_p) == 0) {
			return 0;
		}
	}
}

int
bus_(map_w_check)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr
)
{
#ifdef NEED_MEM_BS
	unsigned int hash;
	struct bus_(map) *m;
#endif
	unsigned int nr;

#ifdef NEED_MEM_BS
	hash = bus_(_map_hash)(addr);

	for (m = b->map_hash_first[hash]; m; m = m->hash_next) {
		if (m->type == BUS_(TYPE_MR)
#ifdef NEED_SMM
		 && m->state == state
#endif
		 && m->addr == addr) {
			bus_(_map_flush)(b, m);
		}
	}
#endif /* NEED_MEM_BS */

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t);

		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_w_check;
		if (func
		 && func(b->member[nr].s,
		 		STATE_C
				addr)) {
			return 1;
		}
	}
}

int
bus_(map_w)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp,
	char **haddr_p
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t,
				int (**)(void *, bus_addr_t, unsigned int, bus_data_t),
				void **,
				char **);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_w;
		if (func
		 && func(b->member[nr].s, STATE_C addr, cfp, csp, haddr_p) == 0) {
			return 0;
		}
	}
}

#ifdef NEED_X
int
bus_(map_x_check)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr
)
{
	unsigned int hash;
	struct bus_(map) *m;
	unsigned int nr;

	hash = bus_(_map_hash)(addr);

	for (m = b->map_hash_first[hash]; m; m = m->hash_next) {
		if (m->type == BUS_(TYPE_MW)
		 && m->state == state
		 && m->addr == addr) {
			bus_(_map_flush)(b, m);
		}
	}

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t);

		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_x_check;
		if (func
		 && func(b->member[nr].s, STATE_C addr) == 0) {
			return 1;
		}
	}
}

int
bus_(map_x)(
	struct bus *b,
	void *s,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddr_p
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, INT_C bus_addr_t,
				int (**)(void *, bus_addr_t, unsigned int, bus_data_t *),
				void **,
				char **);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map_x;
		if (func
		 && func(b->member[nr].s, STATE_C addr, cfp, csp, haddr_p) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_X */

void
bus_(unmap)(struct bus *b, void *s, bus_addr_t addr, bus_addr_t len)
{
#ifdef NEED_MEM_BS
	unsigned int i;
	struct bus_(map) *m;
#endif
	unsigned int nr;

	addr &= ~0xfff;
	len = (len + 0xfff) & ~(bus_addr_t) 0xfff;

#ifdef NEED_MEM_BS
	for (i = 0; i < sizeof(b->map) / sizeof(b->map[0]); i++) {
		m = &b->map[i];

		if (m->addr == -1
		 || m->addr < addr
		 || addr + len - 1 < m->addr) {
			continue;
		}

		bus_(_map_flush)(b, m);
	}
#endif /* NEED_MEM_BS */

	for (nr = 0; ; nr++) {
		void (*func)(void *, bus_addr_t, bus_addr_t);

		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->unmap;
		if (func) {
			func(b->member[nr].s, addr, len);
		}
	}
}
#endif /* NEED_MEM_MAP */

#ifdef NEED_INTA
int
bus_(inta_addr)(
	struct bus *b,
	void *s
)
{
	unsigned int nr;
	int (*func)(void *);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->inta_addr;
		if (func
		 && func(b->member[nr].s) == 0) {
			return 0;
		}
	}
}

int
bus_(inta_data)(
	struct bus *b,
	void *s,
	uint8_t *valp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, uint8_t *);

		if (nr == b->member_count) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->inta_data;
		if (func
		 && func(b->member[nr].s, valp) == 0) {
			return 0;
		}
	}
}
#endif /* NEED_INTA */

void
bus_(connect)(
	struct bus *b,
	void *s,
	const struct bus_(funcs) *f
)
{
	assert(b);
	assert(b->type == SIG_ID);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

#ifdef NEED_ADDR_TYPE
static int
bus_(s0_addr_type)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int type
)
{
	struct bus_(merge) *f = _f;

	return bus_(addr_type)(f->s1, f, STATE_C addr, type);
}

static int
bus_(s0_read_data)(
	void *_f,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(read_data)(f->s1, f, bs, valp);
}

static int
bus_(s0_write_data)(
	void *_f,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(write_data)(f->s1, f, bs, val);
}
#endif /* NEED_ADDR_TYPE */

#ifdef NEED_C0
static int
bus_(s0_c0r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(c0r)(f->s1, f, addr, bs, valp);
}

static int
bus_(s0_c0w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(c0w)(f->s1, f, addr, bs, val);
}
#endif /* NEED_C0 */

#ifdef NEED_C1
static int
bus_(s0_c1r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(c1r)(f->s1, f, addr, bs, valp);
}

static int
bus_(s0_c1w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(c1w)(f->s1, f, addr, bs, val);
}
#endif /* NEED_C1 */

#ifdef NEED_IO_LEN
static int
bus_(s0_inb)(void *_f, unsigned char *valp, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(inb)(f->s1, f, valp, port);
}

static int
bus_(s0_inw)(void *_f, unsigned short *valp, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(inw)(f->s1, f, valp, port);
}

static int
bus_(s0_outb)(void *_f, unsigned char val, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(outb)(f->s1, f, val, port);
}

static int
bus_(s0_outw)(void *_f, unsigned short val, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(outw)(f->s1, f, val, port);
}
#endif /* NEED_IO_LEN */

#ifdef NEED_IO_BS
static int
bus_(s0_ior)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(ior)(f->s1, f, port, bs, valp);
}

static int
bus_(s0_iow)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(iow)(f->s1, f, port, bs, val);
}

static int
bus_(s0_ior_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp
)
{
	struct bus_(merge) *f = _f;

	return bus_(ior_info)(f->s1, f, port, bs, cfp, csp);
}

static int
bus_(s0_iow_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp
)
{
	struct bus_(merge) *f = _f;

	return bus_(iow_info)(f->s1, f, port, bs, cfp, csp);
}

static void
bus_(s0_ior_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(merge) *f = _f;

	bus_(ior_info_flush)(f->s1, f, port, bs);
}

static void
bus_(s0_iow_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(merge) *f = _f;

	bus_(iow_info_flush)(f->s1, f, port, bs);
}
#endif /* NEED_IO_BS */

#ifdef NEED_MEM_LEN
static int
bus_(s0_readb)(void *_f, bus_addr_t addr, uint8_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(readb)(f->s1, f, addr, valp);
}

static int
bus_(s0_readw)(void *_f, bus_addr_t addr, uint16_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(readw)(f->s1, f, addr, valp);
}

static int
bus_(s0_readl)(void *_f, bus_addr_t addr, uint32_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(readl)(f->s1, f, addr, valp);
}

static int
bus_(s0_writeb)(void *_f, bus_addr_t addr, uint8_t val)
{
	struct bus_(merge) *f = _f;

	return bus_(writeb)(f->s1, f, addr, val);
}

static int
bus_(s0_writew)(void *_f, bus_addr_t addr, uint16_t val)
{
	struct bus_(merge) *f = _f;

	return bus_(writew)(f->s1, f, addr, val);
}

static int
bus_(s0_writel)(void *_f, bus_addr_t addr, uint32_t val)
{
	struct bus_(merge) *f = _f;

	return bus_(writel)(f->s1, f, addr, val);
}
#endif /* NEED_MEM_LEN */

#ifdef NEED_MEM_BS
static int
bus_(s0_mr)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(mr)(f->s1, f, STATE_C addr, bs, valp);
}

static int
bus_(s0_mw)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(mw)(f->s1, f, STATE_C addr, bs, val);
}

#ifdef NEED_X
static int
bus_(s0_mx)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(mx)(f->s1, f, STATE_C addr, bs, valp);
}
#endif /* NEED_X */
#endif /* NEED_MEM_BS */

#ifdef NEED_MEM_MAP
static int
bus_(s0_map_r_check)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_r_check)(f->s1, f, STATE_C addr);
}

static int
bus_(s0_map_r)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddr_p
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_r)(f->s1, f, STATE_C addr, cfp, csp, haddr_p);
}

static int
bus_(s0_map_w_check)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_w_check)(f->s1, f, STATE_C addr);
}

static int
bus_(s0_map_w)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp,
	char **haddr_p
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_w)(f->s1, f, STATE_C addr, cfp, csp, haddr_p);
}

#ifdef NEED_X
static int
bus_(s0_map_x_check)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_x_check)(f->s1, f, STATE_C addr);
}

static int
bus_(s0_map_x)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddr_p
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_x)(f->s1, f, STATE_C addr, cfp, csp, haddr_p);
}
#endif /* NEED_X */

static void
bus_(s0_unmap)(
	void *_f,
	bus_addr_t addr,
	bus_addr_t len
)
{
	struct bus_(merge) *f = _f;

	bus_(unmap)(f->s1, f, addr, len);
}
#endif /* NEED_MEM_MAP */

#ifdef NEED_INTA
static int
bus_(s0_inta_addr)(void *_f)
{
	struct bus_(merge) *f = _f;

	return bus_(inta_addr)(f->s1, f);
}

static int
bus_(s0_inta_data)(void *_f, uint8_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(inta_data)(f->s1, f, valp);
}
#endif /* NEED_INTA */

#ifdef NEED_ADDR_TYPE
static int
bus_(s1_addr_type)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int type
)
{
	struct bus_(merge) *f = _f;

	return bus_(addr_type)(f->s0, f, STATE_C addr, type);
}

static int
bus_(s1_read_data)(
	void *_f,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(read_data)(f->s0, f, bs, valp);
}

static int
bus_(s1_write_data)(
	void *_f,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(write_data)(f->s0, f, bs, val);
}
#endif /* NEED_ADDR_TYPE */

#ifdef NEED_C0
static int
bus_(s1_c0r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(c0r)(f->s0, f, addr, bs, valp);
}

static int
bus_(s1_c0w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(c0w)(f->s0, f, addr, bs, val);
}
#endif /* NEED_C0 */

#ifdef NEED_C1
static int
bus_(s1_c1r)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(c1r)(f->s0, f, addr, bs, valp);
}

static int
bus_(s1_c1w)(
	void *_f,
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(c1w)(f->s0, f, addr, bs, val);
}
#endif /* NEED_C1 */

#ifdef NEED_IO_LEN
static int
bus_(s1_inb)(void *_f, unsigned char *valp, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(inb)(f->s0, f, valp, port);
}

static int
bus_(s1_inw)(void *_f, unsigned short *valp, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(inw)(f->s0, f, valp, port);
}

static int
bus_(s1_outb)(void *_f, unsigned char val, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(outb)(f->s0, f, val, port);
}

static int
bus_(s1_outw)(void *_f, unsigned short val, bus_addr_t port)
{
	struct bus_(merge) *f = _f;

	return bus_(outw)(f->s0, f, val, port);
}
#endif /* NEED_IO_LEN */

#ifdef NEED_IO_BS
static int
bus_(s1_ior)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(ior)(f->s0, f, port, bs, valp);
}

static int
bus_(s1_iow)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(iow)(f->s0, f, port, bs, val);
}

static int
bus_(s1_ior_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp
)
{
	struct bus_(merge) *f = _f;

	return bus_(ior_info)(f->s0, f, port, bs, cfp, csp);
}

static int
bus_(s1_iow_info)(
	void *_f,
	bus_addr_t port,
	unsigned int bs,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp
)
{
	struct bus_(merge) *f = _f;

	return bus_(iow_info)(f->s0, f, port, bs, cfp, csp);
}

static void
bus_(s1_ior_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(merge) *f = _f;

	bus_(ior_info_flush)(f->s0, f, port, bs);
}

static void
bus_(s1_iow_info_flush)(
	void *_f,
	bus_addr_t port,
	unsigned int bs
)
{
	struct bus_(merge) *f = _f;

	bus_(iow_info_flush)(f->s0, f, port, bs);
}
#endif /* NEED_IO_BS */

#ifdef NEED_MEM_LEN
static int
bus_(s1_readb)(void *_f, bus_addr_t addr, uint8_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(readb)(f->s0, f, addr, valp);
}

static int
bus_(s1_readw)(void *_f, bus_addr_t addr, uint16_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(readw)(f->s0, f, addr, valp);
}

static int
bus_(s1_readl)(void *_f, bus_addr_t addr, uint32_t *valp)
{
	struct bus_(merge) *f = _f;

	return bus_(readl)(f->s0, f, addr, valp);
}

static int
bus_(s1_writeb)(void *_f, bus_addr_t addr, uint8_t val)
{
	struct bus_(merge) *f = _f;

	return bus_(writeb)(f->s0, f, addr, val);
}

static int
bus_(s1_writew)(void *_f, bus_addr_t addr, uint16_t val)
{
	struct bus_(merge) *f = _f;

	return bus_(writew)(f->s0, f, addr, val);
}

static int
bus_(s1_writel)(void *_f, bus_addr_t addr, uint32_t val)
{
	struct bus_(merge) *f = _f;

	return bus_(writel)(f->s0, f, addr, val);
}
#endif /* NEED_MEM_LEN */

#ifdef NEED_MEM_BS
static int
bus_(s1_mr)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(mr)(f->s0, f, STATE_C addr, bs, valp);
}

static int
bus_(s1_mw)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t val
)
{
	struct bus_(merge) *f = _f;

	return bus_(mw)(f->s0, f, STATE_C addr, bs, val);
}

#ifdef NEED_X
static int
bus_(s1_mx)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	unsigned int bs,
	bus_data_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(mx)(f->s0, f, STATE_C addr, bs, valp);
}
#endif /* NEED_X */
#endif /* NEED_MEM_BS */

#ifdef NEED_MEM_MAP
static int
bus_(s1_map_r_check)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_r_check)(f->s0, f, STATE_C addr);
}

static int
bus_(s1_map_r)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddr_p
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_r)(f->s0, f, STATE_C addr, cfp, csp, haddr_p);
}

static int
bus_(s1_map_w_check)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_w_check)(f->s0, f, STATE_C addr);
}

static int
bus_(s1_map_w)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp,
	char **haddr_p
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_w)(f->s0, f, STATE_C addr, cfp, csp, haddr_p);
}

#ifdef NEED_X
static int
bus_(s1_map_x_check)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_x_check)(f->s0, f, STATE_C addr);
}

static int
bus_(s1_map_x)(
	void *_f,
	INT_STATE_C
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddr_p
)
{
	struct bus_(merge) *f = _f;

	return bus_(map_x)(f->s0, f, STATE_C addr, cfp, csp, haddr_p);
}
#endif /* NEED_X */

static void
bus_(s1_unmap)(
	void *_f,
	bus_addr_t addr,
	bus_addr_t len
)
{
	struct bus_(merge) *f = _f;

	bus_(unmap)(f->s0, f, addr, len);
}
#endif /* NEED_MEM_MAP */

#ifdef NEED_INTA
static int
bus_(s1_inta_addr)(
	void *_f
)
{
	struct bus_(merge) *f = _f;

	return bus_(inta_addr)(f->s0, f);
}

static int
bus_(s1_inta_data)(
	void *_f,
	uint8_t *valp
)
{
	struct bus_(merge) *f = _f;

	return bus_(inta_data)(f->s0, f, valp);
}
#endif /* NEED_INTA */

struct bus_(merge) *
bus_(merge)(struct bus *s0, struct bus *s1)
{
	static const struct bus_(funcs) s0_funcs = {
#ifdef NEED_ADDR_TYPE
		.addr_type = bus_(s0_addr_type),
		.read_data = bus_(s0_read_data),
		.write_data = bus_(s0_write_data),
#endif /* NEED_ADDR_TYPE */
#ifdef NEED_C0
		.c0r = bus_(s0_c0r),
		.c0w = bus_(s0_c0w),
#endif /* NEED_C0 */
#ifdef NEED_C1
		.c1r = bus_(s0_c1r),
		.c1w = bus_(s0_c1w),
#endif /* NEED_C1 */
#ifdef NEED_IO_LEN
		.inb = bus_(s0_inb),
		.inw = bus_(s0_inw),
		.outb = bus_(s0_outb),
		.outw = bus_(s0_outw),
#endif /* NEED_IO_LEN */
#ifdef NEED_IO_BS
		.ior = bus_(s0_ior),
		.iow = bus_(s0_iow),
		.ior_info = bus_(s0_ior_info),
		.iow_info = bus_(s0_iow_info),
		.ior_info_flush = bus_(s0_ior_info_flush),
		.iow_info_flush = bus_(s0_iow_info_flush),
#endif /* NEED_IO_BS */
#ifdef NEED_MEM_LEN
		.readb = bus_(s0_readb),
		.readw = bus_(s0_readw),
		.readl = bus_(s0_readl),
		.writeb = bus_(s0_writeb),
		.writew = bus_(s0_writew),
		.writel = bus_(s0_writel),
#endif /* NEED_MEM_LEN */
#ifdef NEED_MEM_BS
		.mr = bus_(s0_mr),
		.mw = bus_(s0_mw),
#ifdef NEED_X
		.mx = bus_(s0_mx),
#endif /* NEED_X */
#endif /* NEED_MEM_BS */
#ifdef NEED_MEM_MAP
		.map_r_check = bus_(s0_map_r_check),
		.map_r = bus_(s0_map_r),
		.map_w_check = bus_(s0_map_w_check),
		.map_w = bus_(s0_map_w),
#ifdef NEED_X
		.map_x_check = bus_(s0_map_x_check),
		.map_x = bus_(s0_map_x),
#endif /* NEED_X */
		.unmap = bus_(s0_unmap),
#endif /* NEED_MEM_MAP */
#ifdef NEED_INTA
		.inta_addr = bus_(s0_inta_addr),
		.inta_data = bus_(s0_inta_data),
#endif /* NEED_INTA */
	};
	static const struct bus_(funcs) s1_funcs = {
#ifdef NEED_ADDR_TYPE
		.addr_type = bus_(s1_addr_type),
		.read_data = bus_(s1_read_data),
		.write_data = bus_(s1_write_data),
#endif /* NEED_ADDR_TYPE */
#ifdef NEED_C0
		.c0r = bus_(s1_c0r),
		.c0w = bus_(s1_c0w),
#endif /* NEED_C0 */
#ifdef NEED_C1
		.c1r = bus_(s1_c1r),
		.c1w = bus_(s1_c1w),
#endif /* NEED_C1 */
#ifdef NEED_IO_LEN
		.inb = bus_(s1_inb),
		.inw = bus_(s1_inw),
		.outb = bus_(s1_outb),
		.outw = bus_(s1_outw),
#endif /* NEED_IO_LEN */
#ifdef NEED_IO_BS
		.ior = bus_(s1_ior),
		.iow = bus_(s1_iow),
		.ior_info = bus_(s1_ior_info),
		.iow_info = bus_(s1_iow_info),
		.ior_info_flush = bus_(s1_ior_info_flush),
		.iow_info_flush = bus_(s1_iow_info_flush),
#endif /* NEED_IO_BS */
#ifdef NEED_MEM_LEN
		.readb = bus_(s1_readb),
		.readw = bus_(s1_readw),
		.readl = bus_(s1_readl),
		.writeb = bus_(s1_writeb),
		.writew = bus_(s1_writew),
		.writel = bus_(s1_writel),
#endif /* NEED_MEM_LEN */
#ifdef NEED_MEM_BS
		.mr = bus_(s1_mr),
		.mw = bus_(s1_mw),
#ifdef NEED_X
		.mx = bus_(s1_mx),
#endif /* NEED_X */
#endif /* NEED_MEM_BS */
#ifdef NEED_MEM_MAP
		.map_r_check = bus_(s1_map_r_check),
		.map_r = bus_(s1_map_r),
		.map_w_check = bus_(s1_map_w_check),
		.map_w = bus_(s1_map_w),
#ifdef NEED_X
		.map_x_check = bus_(s1_map_x_check),
		.map_x = bus_(s1_map_x),
#endif /* NEED_X */
		.unmap = bus_(s1_unmap),
#endif /* NEED_MEM_MAP */
#ifdef NEED_INTA
		.inta_addr = bus_(s1_inta_addr),
		.inta_data = bus_(s1_inta_data),
#endif /* NEED_INTA */
	};
	struct bus_(merge) *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	bus_(connect)(s0, m, &s0_funcs);
	m->s1 = s1;
	bus_(connect)(s1, m, &s1_funcs);

	return m;
}

void
bus_(split)(struct bus_(merge) *m)
{
	fixme();
}

struct bus *
bus_(create)(const char *name)
{
	struct bus *b;
#ifdef NEED_IO_BS
	struct bus_(io) *io;
	unsigned int i;
#endif
#ifdef NEED_MEM_BS
	struct bus_(map) *m;
	unsigned int j;
#endif

	b = shm_alloc(sizeof(*b));
	assert(b);

	b->type = SIG_ID;
	b->member_count = 0;

#ifdef NEED_IO_BS
	b->io_lru_first = 0;
	b->io_lru_last = 0;
	for (i = 0; i < sizeof(b->io_hash_first) / sizeof(b->io_hash_first[0]); i++) {
		b->io_hash_first[i] = NULL;
		b->io_hash_last[i] = NULL;
	}
	for (i = 0; i < sizeof(b->io) / sizeof(b->io[0]); i++) {
		io = &b->io[i];
		io->port = -1;
		io->bs = -1;

		/* Don't add empty entry to HASH list. */

		/* Add to LRU list. */
		io->lru_prev = b->io_lru_last;
		io->lru_next = 0;
		if (io->lru_prev) {
			io->lru_prev->lru_next = io;
		} else {
			b->io_lru_first = io;
		}
		b->io_lru_last = io;
	}
#endif /* NEED_IO_BS */

#ifdef NEED_MEM_BS
	b->map_lru_first = 0;
	b->map_lru_last = 0;
	for (j = 0; j < sizeof(b->map_hash_first) / sizeof(b->map_hash_first[0]); j++) {
		b->map_hash_first[j] = 0;
		b->map_hash_last[j] = 0;
	}
	for (j = 0; j < sizeof(b->map) / sizeof(b->map[0]); j++) {
		m = &b->map[j];
#ifdef NEED_SMM
		m->state = -1;
#endif
		m->addr = -1;
		m->mr_map = NULL;
		m->mr_func = NULL;
		m->mr_cpssp = NULL;
		m->mw_map = NULL;
		m->mw_func = NULL;
		m->mw_cpssp = NULL;
#ifdef NEED_X
		m->mx_map = NULL;
		m->mx_func = NULL;
		m->mx_cpssp = NULL;
#endif

		/* Don't add empty entry to HASH list. */

		/* Add to LRU list. */
		m->lru_prev = b->map_lru_last;
		m->lru_next = 0;
		if (m->lru_prev) {
			m->lru_prev->lru_next = m;
		} else {
			b->map_lru_first = m;
		}
		b->map_lru_last = m;
	}
#endif /* NEED_MEM_BS */

#ifdef NEED_LOCK
	init_spinlock(&b->cache_lock);
#endif

	return b;
}

void
bus_(destroy)(struct bus *b)
{
	assert(b);
	assert(b->type == SIG_ID);

	shm_free(b);
}

void
bus_(suspend)(struct bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
bus_(resume)(struct bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}
