/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "wattmeter.h"

#define COMP_(x)	wattmeter_ ## x

#define FREQ	1

struct cpssp {
	struct sig_std_logic *port_output;
	unsigned int state_voltage;
	unsigned int state_current;
	struct sig_integer *port_watt;

	double energy;

	unsigned long long time;
};

static void
COMP_(input_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	sig_std_logic_set(cpssp->port_output, cpssp, val);
}

static double
COMP_(input_supply)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	double e;

	e = sig_std_logic_supply(cpssp->port_output, cpssp);

	cpssp->energy += e;

	return e;
}

static void
COMP_(event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	sig_integer_set(cpssp->port_watt, cpssp,
			(int) (cpssp->energy * 1000.0 * FREQ));

	cpssp->energy = 0.0;

	cpssp->time += TIME_HZ / FREQ;
	time_call_at(cpssp->time, COMP_(event), cpssp);
}

void *
COMP_(create)(
	const char *name,
        struct sig_manage *port_manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_input,
	struct sig_std_logic *port_output,
	struct sig_integer *port_watt
)
{
	static const struct sig_std_logic_funcs input_funcs = {
		.set_ext = COMP_(input_set),
		.supply_ext = COMP_(input_supply),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->energy = 0.0;

	/* Out */
	cpssp->port_output = port_output;
	sig_std_logic_connect_out(port_output, cpssp, SIG_STD_LOGIC_0);

	cpssp->port_watt = port_watt;
	sig_integer_connect_out(port_watt, cpssp, 0);

	/* In */
	sig_std_logic_connect_in(port_input, cpssp, &input_funcs);

	/* Time */
	cpssp->time = 0;
	time_call_at(cpssp->time, COMP_(event), cpssp);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
