// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/data_decoder/web_bundle_builder.h"

#include "base/big_endian.h"
#include "base/files/file_path.h"
#include "base/files/file_util.h"
#include "base/path_service.h"
#include "base/test/task_environment.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace data_decoder {

namespace {

std::string kFallbackUrl = "https://test.example.org/";

std::string GetTestFileContents(const base::FilePath& path) {
  base::FilePath test_data_dir;
  base::PathService::Get(base::DIR_SOURCE_ROOT, &test_data_dir);
  test_data_dir = test_data_dir.Append(
      base::FilePath(FILE_PATH_LITERAL("components/test/data/web_package")));

  std::string contents;
  EXPECT_TRUE(base::ReadFileToString(test_data_dir.Append(path), &contents));
  return contents;
}

std::vector<uint8_t> GetStringAsBytes(base::StringPiece contents) {
  auto bytes = base::as_bytes(base::make_span(contents));
  return std::vector<uint8_t>(bytes.begin(), bytes.end());
}

}  // namespace

class WebBundleBuilderTest : public testing::Test {
 private:
  base::test::TaskEnvironment task_environment_;
};

TEST_F(WebBundleBuilderTest, CorrectWebBundleSizeIsWritten) {
  WebBundleBuilder builder(kFallbackUrl);
  std::vector<mojom::SerializedResourceInfoPtr> exchanges;
  mojom::SerializedResourceInfoPtr ptr = mojom::SerializedResourceInfo::New(
      GURL("https://test.example.org/index.html"), "text/html", 0);
  exchanges.emplace_back(std::move(ptr));
  std::vector<base::Optional<mojo_base::BigBuffer>> bodies;
  bodies.emplace_back();
  std::vector<uint8_t> bundle =
      builder.CreateBundle(std::move(exchanges), std::move(bodies));
  char written_size[8];
  memcpy(written_size, bundle.data() + bundle.size() - 8, 8);
  uint64_t written_size_int;
  base::ReadBigEndian(written_size, &written_size_int);
  EXPECT_EQ(bundle.size(), written_size_int);
}

TEST_F(WebBundleBuilderTest, ByteByByteComparison) {
  WebBundleBuilder builder(kFallbackUrl);
  std::vector<mojom::SerializedResourceInfoPtr> exchanges;
  std::vector<base::Optional<mojo_base::BigBuffer>> bodies;
  exchanges.emplace_back(mojom::SerializedResourceInfo::New(
      GURL("https://test.example.org/"), "text/html; charset=UTF-8", 46));
  bodies.emplace_back(base::Optional<mojo_base::BigBuffer>(
      GetStringAsBytes("<a href='index.html'>click for web bundles</a>")));
  exchanges.emplace_back(mojom::SerializedResourceInfo::New(
      GURL("https://test.example.org/index.html"), "text/html; charset=UTF-8",
      25));
  bodies.emplace_back(base::Optional<mojo_base::BigBuffer>(
      GetStringAsBytes("<p>Hello Web Bundles!</p>")));
  std::vector<uint8_t> bundle =
      builder.CreateBundle(std::move(exchanges), std::move(bodies));
  std::vector<uint8_t> expected_bundle = GetStringAsBytes(
      GetTestFileContents(base::FilePath(FILE_PATH_LITERAL("simple.wbn"))));
  EXPECT_EQ(bundle, expected_bundle);
}

}  // namespace data_decoder
