# -*- coding: utf-8 -*-
# vim:fenc=utf-8

# Copyright (C) 2012-2020 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# X2Go Session Broker is free software; you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# X2Go Session Broker is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

# Python X2GoBroker modules
from . import base_nameservice as base

_users = [ 'maja', 'willi', 'flip', 'kassandra', 'thekla' ]
_groups = {
    'male': ['willi', 'flip'],
    'female': ['maja', 'kassandra', 'thekla'],
    'bees': ['maja', 'willi', 'kassandra'],
    'grasshoppers': ['flip'],
    'spiders': ['thekla'],
}


class X2GoBrokerNameService(base.X2GoBrokerNameService):

    def get_users(self):
        """\
        Retrieve hard-coded list of users that we can use for
        unit testing.

        :returns: list of known user names
        :rtype: ``list``

        """
        return _users

    def get_primary_group(self, username):
        """\
        In POSIX, the primary group name is equal to the user name.
        As this is the only straw we can grab during unit tests, we
        return the username here.

        :param username: name of the user to get the primary group for
        :type username: ``str``

        :returns: name of the primary group of the given user
        :rtype: ``str``

        """
        return username

    def get_groups(self):
        """\
        Retrieve hard-coded list of groups that we can use for unit
        testing.

        :returns: list of known group names
        :rtype: ``list``

        """
        return list(_groups.keys()) + _users

    def get_group_members(self, group, primary_groups=False):
        """\
        Retrieve a list of users being members of a given group. For unit
        testing, the group membership relations have been hard-coded.

        Optionally, primary group memberships can be considered (or not).

        :param group: name of the group to retrieve members of
        :type group: ``str``
        :param primary_groups: take primary group membership into consideration
            or not
        :type primary_groups: ``bool``

        :returns: list of users that are members of the given group
        :rtype: ``list``

        """
        _members = []
        if group in list(_groups.keys()):
            _members.extend(_groups[group])
        if primary_groups:
            for username in self.get_users():
                if group == self.get_primary_group(username):
                    _members.append(username)
        return _members
