!***********************************************************************
! This file is part of OpenMolcas.                                     *
!                                                                      *
! OpenMolcas is free software; you can redistribute it and/or modify   *
! it under the terms of the GNU Lesser General Public License, v. 2.1. *
! OpenMolcas is distributed in the hope that it will be useful, but it *
! is provided "as is" and without any express or implied warranties.   *
! For more details see the full text of the license in the file        *
! LICENSE or in <http://www.gnu.org/licenses/>.                        *
!***********************************************************************

subroutine GenVoronoi(nR_Eff,Alpha,rm,iNQ)
!***********************************************************************
!                                                                      *
!     This version of GenVoronoi computes the radial quadrature points *
!     and computes data useful for the angular quadrature.             *
!     The angular part is generated by Subblock.                       *
!                                                                      *
!***********************************************************************

use NQ_Structure, only: lMax_NQ, NQ_Data
use nq_Info, only: L_Quad, nR, Quadrature
use stdalloc, only: mma_allocate
use Constants, only: Zero, One, Three, Five, Seven, Half
use Definitions, only: wp, iwp, u6

implicit none
integer(kind=iwp), intent(out) :: nR_Eff
real(kind=wp), intent(inout) :: Alpha(2), rm(2)
integer(kind=iwp), intent(in) :: iNQ
integer(kind=iwp) :: iANr, l_Max, mR
#ifdef _DEBUGPRINT_
integer(kind=iwp) :: iR
#endif
real(kind=wp) :: Dum(2,1), Radius_Max, RBS
logical(kind=iwp) :: Process
real(kind=wp), parameter :: TA_R(36) = [0.8_wp,0.9_wp,1.8_wp,1.4_wp,1.3_wp,1.1_wp,0.9_wp,0.9_wp,0.9_wp,0.9_wp,1.4_wp,1.3_wp, &
                                        1.3_wp,1.2_wp,1.1_wp,1.0_wp,1.0_wp,1.0_wp,1.5_wp,1.4_wp,1.3_wp,1.2_wp,1.2_wp,1.2_wp, &
                                        1.2_wp,1.2_wp,1.2_wp,1.1_wp,1.1_wp,1.1_wp,1.1_wp,1.0_wp,0.9_wp,0.9_wp,0.9_wp,0.9_wp]
real(kind=wp), external :: Bragg_Slater, Eval_RMax

!                                                                      *
!***********************************************************************
!                                                                      *
!#define _DEBUGPRINT_
#ifdef _DEBUGPRINT_
write(u6,*) 'nR,L_Quad=',nR,L_Quad
#endif
if (L_Quad > lMax_NQ) then
  call WarningMessage(2,'GenVoronoi: L_Quad > lMax_NQ')
  write(u6,*) 'Redimension lMax_NQ in nq_structure'
  write(u6,*) 'lMax_NQ=',lMax_NQ
  write(u6,*) 'L_Quad=',L_Quad
  call Abend()
end if
l_Max = int(rm(1))
Radius_Max = Eval_RMax(Alpha(1),l_Max,rm(2))
#ifdef _DEBUGPRINT_
write(u6,*) 'Alpha(1)=',Alpha(1)
write(u6,*) 'l_max=',l_max
write(u6,*) 'rm(2)=',rm(2)
write(u6,*) 'Radius_Max=',Radius_Max
write(u6,*)
#endif
!                                                                      *
!***********************************************************************
!                                                                      *
! Generate radial quadrature points. Observe that the integrand
! vanishes at (r=0.0).

mR = nR
select case (Quadrature)

  case ('MHL')

    iANr = NQ_Data(iNQ)%Atom_Nr
    RBS = Bragg_Slater(iANr)
    Alpha(1) = RBS
    mR = nR-1
    call mma_allocate(NQ_Data(iNQ)%R_Quad,2,mR,Label='R_Quad')
    NQ_Data(iNQ)%R_Quad(:,:) = Zero
    call GenRadQuad_MHL(NQ_Data(iNQ)%R_Quad,nR,nR_Eff,Alpha(1))
    call Truncate_Grid(NQ_Data(iNQ)%R_Quad,mR,nR_Eff,Radius_Max)
    mR = nR_Eff

  case ('LOG3')

    rm(1) = Three
    ! alpha=5 (alpha=7 for alkali and rare earth metals)
    iANr = NQ_Data(iNQ)%Atom_Nr
    select case (iANr)
      case (3,4,11,12,19,20,37,38,55,56,87,88)
        Alpha(1) = Seven
      case default
        Alpha(1) = Five
    end select
    mR = nR-1
    call mma_allocate(NQ_Data(iNQ)%R_Quad,2,mR,Label='R_Quad')
    NQ_Data(iNQ)%R_Quad(:,:) = Zero
    call GenRadQuad_MK(NQ_Data(iNQ)%R_Quad,nR,nR_Eff,rm(1),Alpha(1))
    call Truncate_Grid(NQ_Data(iNQ)%r_Quad,mR,nR_Eff,Radius_Max)
    mR = nR_Eff

  case ('BECKE')

    iANr = NQ_Data(iNQ)%Atom_Nr
    RBS = Bragg_Slater(iANr)
    if (iANr == 1) then
      Alpha(1) = RBS
    else
      Alpha(1) = Half*RBS
    end if
    mR = nR-1
    call mma_allocate(NQ_Data(iNQ)%R_Quad,2,mR,Label='R_Quad')
    NQ_Data(iNQ)%R_Quad(:,:) = Zero
    call GenRadQuad_B(NQ_Data(iNQ)%R_Quad,nR,nR_Eff,Alpha(1))
    call Truncate_Grid(NQ_Data(iNQ)%R_Quad,mR,nR_Eff,Radius_Max)
    mR = nR_Eff

  case ('TA')

    Alpha(1) = -One
    iANr = NQ_Data(iNQ)%Atom_Nr
    if ((iANr < lbound(TA_R,1)) .or. (iANr > ubound(TA_R,1))) then
      call WarningMessage(2,'TA grid not defined')
      write(u6,*) ' TA grid not defined for atom number:',iANR
      call Abend()
    else
      Alpha(1) = TA_R(iANr)
    end if
    mR = nR-1
    call mma_allocate(NQ_Data(iNQ)%R_Quad,2,mR,Label='R_Quad')
    NQ_Data(iNQ)%R_Quad(:,:) = Zero
    call GenRadQuad_TA(NQ_Data(iNQ)%R_Quad,nR,nR_Eff,Alpha(1))
    call Truncate_Grid(NQ_Data(iNQ)%R_Quad,mR,nR_Eff,Radius_Max)
    mR = nR_Eff

  case ('LMG')

    !                                                                  *
    !*******************************************************************
    !                                                                  *
    ! Generate radial quadrature. The first call will generate
    ! the size of the grid.

    nR = 1 ! Dummy size on the first call.
    Process = .false.
    call GenRadQuad_PAM(nR_Eff,rm,Alpha(1),Process,Dum,nR)

    nR = nR_Eff
    call mma_allocate(NQ_Data(iNQ)%R_Quad,2,nR,Label='R_Quad')
    NQ_Data(iNQ)%R_Quad(:,:) = Zero
    Process = .true.
    call GenRadQuad_PAM(nR_Eff,rm,Alpha(1),Process,NQ_Data(iNQ)%R_Quad,nR)
    mR = nR
    !                                                                  *
    !*******************************************************************
    !                                                                  *
#   ifdef _DEBUGPRINT_
    write(u6,*) 'GenRadQuad_PAM ----> GenVoronoi'
    write(u6,*) 'nR_Eff=',nR_Eff
    write(u6,*) 'rm : ',rm(1),rm(2)
    write(u6,*) 'Alpha : ',Alpha(1),Alpha(2)
#   endif

  case default

    call WarningMessage(2,'Invalid quadrature scheme:'//Quadrature)
    call Quit_OnUserError()

end select

NQ_Data(iNQ)%R_max = NQ_Data(iNQ)%R_Quad(1,mR)

#ifdef _DEBUGPRINT_
write(u6,*)
write(u6,*) ' ******** The radial grid ********'
write(u6,*)
write(u6,*) 'Initial number of radial grid points=',nR
write(u6,*) 'iNQ=',iNQ
write(u6,*) 'Effective number of radial grid points=',nR_Eff
do iR=1,nR_Eff
  write(u6,*) NQ_Data(iNQ)%R_Quad(1,iR),NQ_Data(iNQ)%R_Quad(2,iR)
end do
write(u6,*)
write(u6,*) ' *********************************'
write(u6,*)
#endif
!                                                                      *
!***********************************************************************
!                                                                      *

return

end subroutine GenVoronoi
