! This program takes a list of xy data files and interpolates the data from each onto an x grid which is the 
! union of all of the x grids, then adds all of y data.

! Written by J. J. Kas - 1/25/2010
! modified by Bruce Ravel 6 October 2015 for inclusion in feff8l
!   BR stripped out the file-base IO, added thiseps input array and
!   the NdataTot, energy, and loss output parameters
!   This subroutine now takes an array of epsilon data for the unique
!   potentials and returns the loss data.  No file IO.

subroutine AddEps(NFiles,Weights,thiseps, NDataTot, energy, loss)
  !!! prior interface
  ! Files:      character array of file names as written by write_eps (original) (deprecated)
  ! NFiles:     number of ipots, i.e. nph+1
  ! Weights:    array of number densities (original)
  ! print_eps:  boolean signaling printing of epsilon.dat diagnostic file (original) (deprecated)
  !!! feff8l interface
  ! thiseps:    (epsmax,3,nphx) array of epsilon data for each unique ipot
  ! energy:     return energy array
  ! loss:       return loss array
  !use IOMod
  !use IOFiles
  use dimsmod
  implicit none

  !character*(*) Files(NFiles)
  !logical print_eps
  integer NFiles, iFile, iData, NDataTot, iTot, iSort, iNew
  real(8), allocatable :: E(:,:), Eps1(:,:), Eps2(:,:), ETot(:), Eps1Tot(:),Eps2Tot(:)
  real(8) xTmp, yTmp, Weights(NFiles)
  integer,allocatable :: NData(:) !, iUnit(:)
  ! Get the filenames.
  !PRINT*, "# Enter the number of files."
  !READ*, NFiles

  integer, parameter :: epsmax = 700
  integer i1, iph !, iUnit
  real(8) thiseps(epsmax,3,0:nphx)
  real(8) energy(epsmax), loss(epsmax)

  !allocate(iUnit(NFiles))
  
  !DO iFile = 1, NFiles
  !   PRINT*, '# Enter file #', iFile 
  !   READ '(a)', Files(iFile)
  !END DO

  ! Get the number of data points for each file.
  allocate(NData(NFiles))
  NDataTot = 0
!   do iFile = 1, NFiles
!      NData(iFile) = 0
!      call OpenFl(Files(iFile))
!      call GetIOFileInfo(Files(iFile), UnitNumber = iUnit(iFile))
!      ! Read comments from header. Assumes that comments are ONLY in header and starts with #.
!      call RdCmt(iUnit(iFile),'#cC!')
!      do
!         read(iUnit(iFile),*,end = 5)
!         NData(iFile) = NData(iFile) + 1
!      end do
! 5    continue
!      NDataTot = NDataTot + NData(iFile)
!      call CloseFl(Files(iFile))
!   end do

  ! Get the number of data points for each unique potential
  do iph = 0,NFiles-1
     do i1 = 1,epsmax
        if (abs(thiseps(i1,1,iph)) .lt. 10d-20) then
           NDataTot = NDataTot + i1 - 1
           NData(iph+1) = i1 - 1
           exit
        end if
     end do
  end do
  
  ! Make space for x data.
  allocate(E(maxval(NData),NFiles),Eps1(maxval(NData),NFiles),Eps2(maxval(NData),NFiles))
  allocate(ETot(NDataTot),Eps1Tot(NDataTot),Eps2Tot(NDataTot))

  ! Read in x and y data.
  iTot = 0
  do iFile = 1, NFiles
     ! call OpenFl(Files(iFile))
     ! call GetIOFileInfo(Files(iFile), UnitNumber = iUnit(iFile))
     
     ! call RdCmt(iUnit(iFile),'#   ')
     do iData = 1, NData(iFile)
        ! read(iUnit(iFile),*) E(iData,iFile), Eps1(iData,iFile), Eps2(iData,iFile)
        E(iData,iFile) = thiseps(iData,1,iFile-1)
        Eps1(iData,iFile) = thiseps(iData,2,iFile-1)
        Eps2(iData,iFile) = thiseps(iData,3,iFile-1)
        
        iTot = iTot + 1
        ! Keep xTot sorted and unique.
        xTmp = E(iData,iFile)
        iNew = iTot
        if(iTot.eq.1) ETot(1) = xTmp
        do iSort = iTot - 1, 1, -1
           if( xTmp.lt.ETot(iSort) ) then
              ! Change iNew.
              iNew = iSort
           elseif (abs(xTmp - ETot(iSort)) .lt. 1d-20) THEN
           !elseif (xTmp .eq. ETot(iSort)) then
              ! Don't keep this point.
              iTot = iTot -1
           else
              ! xTot is sorted. Fill it and exit loop.
              if(iNew.ne.iTot) then
                 ! Shift xTot
                 ETot(iNew + 1:iTot) = ETot(iNew:iTot - 1)
                 ETot(iNew) = xTmp
              else
                 ETot(iNew) = xTmp
              end if
              exit
           end if
        end do
     end do
     ! close(iUnit(iFile))
  end do
  NDataTot = iTot
  Eps1Tot(:) = 0.d0
  Eps2Tot(:) = 0.d0
  do iTot = 1, NDataTot
     do iFile = 1, NFiles
        ! Interpolate each data set onto xTot, and add to yTot
        call terp(E(:,iFile),Eps1(:,iFile),NData(iFile),1,ETot(iTot),yTmp)
        Eps1Tot(iTot) = Eps1Tot(iTot) + yTmp*Weights(iFile)
        call terp(E(:,iFile),Eps2(:,iFile),NData(iFile),1,ETot(iTot),yTmp)
        Eps2Tot(iTot) = Eps2Tot(iTot) + yTmp*Weights(iFile)
     end do
  end do

  do iTot = 1, NDataTot
     energy(iTot) = ETot(iTot)
     loss(iTot) = Eps2Tot(iTot)/((Eps1Tot(iTot)+1)**2 + Eps2Tot(iTot)**2)
  end do
  ! if (print_eps) then
  !    call OpenFl('epsilon.dat')
  !    call GetIOFileInfo('epsilon.dat', UnitNumber = iUnit)
  !    write(iUnit,'(A)') '# E(eV)    eps1    eps2'
  !    do iTot = 1, NDataTot
  !       write(iUnit,*), energy(iTot), Eps1Tot(iTot)+1.d0, Eps2Tot(iTot)
  !    end do
  !    call CloseFl('epsilon.dat')
  ! end if

  deallocate(NData,E,Eps1,Eps2,Eps1Tot,Eps2Tot)
  return
end subroutine AddEps

!subroutine epsdb(iz,n)
subroutine epsdb(nz, thiseps)
  !  integer,intent(IN) :: n, iz(n)
  integer, parameter :: epsmax = 700
  integer,intent(IN) :: nz
  real(8) epsData(epsmax,3,100), thiseps(epsmax,3)
  character(12) files(100)

  files(:) = 'NULL'
  
  data (epsData(1,i,1), i=1,3) / 0.2506580000E-02, -0.9726400000E+02, 0.1960790000E+03 /
  data (epsData(2,i,1), i=1,3) / 0.4693440000E-02, -0.7332500000E+02, 0.6094470000E+02 /
  data (epsData(3,i,1), i=1,3) / 0.7560590000E-02, -0.3322180000E+02, 0.2602180000E+02 /
  data (epsData(4,i,1), i=1,3) / 0.1110800000E-01, -0.1598020000E+02, 0.1350880000E+02 /
  data (epsData(5,i,1), i=1,3) / 0.1533570000E-01, -0.7919730000E+01, 0.7996130000E+01 /
  data (epsData(6,i,1), i=1,3) / 0.2024370000E-01, -0.3763810000E+01, 0.5194390000E+01 /
  data (epsData(7,i,1), i=1,3) / 0.2583200000E-01, -0.1442820000E+01, 0.3613460000E+01 /
  data (epsData(8,i,1), i=1,3) / 0.3210060000E-01, -0.6129380000E-01, 0.2647550000E+01 /
  data (epsData(9,i,1), i=1,3) / 0.3904940000E-01, 0.8044290000E+00, 0.2019520000E+01 /
  data (epsData(10,i,1), i=1,3) / 0.4667860000E-01, 0.1370330000E+01, 0.1590350000E+01 /
  data (epsData(11,i,1), i=1,3) / 0.5498800000E-01, 0.1753500000E+01, 0.1284960000E+01 /
  data (epsData(12,i,1), i=1,3) / 0.6397770000E-01, 0.2020800000E+01, 0.1060260000E+01 /
  data (epsData(13,i,1), i=1,3) / 0.7364770000E-01, 0.2212110000E+01, 0.8902510000E+00 /
  data (epsData(14,i,1), i=1,3) / 0.8399800000E-01, 0.2352090000E+01, 0.7585400000E+00 /
  data (epsData(15,i,1), i=1,3) / 0.1067390000E+00, 0.2535790000E+01, 0.5706650000E+00 /
  data (epsData(16,i,1), i=1,3) / 0.1459540000E+00, 0.2682340000E+01, 0.3982800000E+00 /
  data (epsData(17,i,1), i=1,3) / 0.2077630000E+00, 0.2773230000E+01, 0.2689490000E+00 /
  data (epsData(18,i,1), i=1,3) / 0.3421210000E+00, 0.2833970000E+01, 0.1572210000E+00 /
  data (epsData(19,i,1), i=1,3) / 0.7743970000E+00, 0.2870920000E+01, 0.7021350000E-01 /
  data (epsData(20,i,1), i=1,3) / 0.3247730000E+01, 0.2962910000E+01, 0.2163230000E-01 /
  data (epsData(21,i,1), i=1,3) / 0.6081600000E+01, 0.3234980000E+01, 0.1691100000E-01 /
  data (epsData(22,i,1), i=1,3) / 0.8247440000E+01, 0.3645450000E+01, 0.1866280000E-01 /
  data (epsData(23,i,1), i=1,3) / 0.9912920000E+01, 0.4197950000E+01, 0.2366810000E-01 /
  data (epsData(24,i,1), i=1,3) / 0.1110830000E+02, 0.4865900000E+01, 0.3232920000E-01 /
  data (epsData(25,i,1), i=1,3) / 0.1185810000E+02, 0.5539960000E+01, 0.4472890000E-01 /
  data (epsData(26,i,1), i=1,3) / 0.1237170000E+02, 0.6254550000E+01, 0.6302500000E-01 /
  data (epsData(27,i,1), i=1,3) / 0.1276390000E+02, 0.7119380000E+01, 0.9743720000E-01 /
  data (epsData(28,i,1), i=1,3) / 0.1289610000E+02, 0.7545070000E+01, 0.1134610000E+00 /
  data (epsData(29,i,1), i=1,3) / 0.1302890000E+02, 0.8097930000E+01, 0.1669510000E+00 /
  data (epsData(30,i,1), i=1,3) / 0.1316240000E+02, 0.8899010000E+01, 0.2196710000E+00 /
  data (epsData(31,i,1), i=1,3) / 0.1329650000E+02, 0.1001670000E+02, 0.6294520000E+00 /
  data (epsData(32,i,1), i=1,3) / 0.1343140000E+02, 0.1185440000E+02, 0.1162780000E+01 /
  data (epsData(33,i,1), i=1,3) / 0.1356690000E+02, 0.1375500000E+02, 0.4836270000E+01 /
  data (epsData(34,i,1), i=1,3) / 0.1360850000E+02, 0.1370240000E+02, 0.6842480000E+01 /
  data (epsData(35,i,1), i=1,3) / 0.1368710000E+02, 0.1224940000E+02, 0.9962460000E+01 /
  data (epsData(36,i,1), i=1,3) / 0.1375000000E+02, 0.9921220000E+01, 0.1240950000E+02 /
  data (epsData(37,i,1), i=1,3) / 0.1377990000E+02, 0.8097920000E+01, 0.1355760000E+02 /
  data (epsData(38,i,1), i=1,3) / 0.1379590000E+02, 0.6597650000E+01, 0.1416670000E+02 /
  data (epsData(39,i,1), i=1,3) / 0.1381260000E+02, 0.4925890000E+01, 0.1375070000E+02 /
  data (epsData(40,i,1), i=1,3) / 0.1382990000E+02, 0.3619940000E+01, 0.1296750000E+02 /
  data (epsData(41,i,1), i=1,3) / 0.1386670000E+02, 0.1813990000E+01, 0.1132170000E+02 /
  data (epsData(42,i,1), i=1,3) / 0.1390610000E+02, 0.5781720000E+00, 0.9573150000E+01 /
  data (epsData(43,i,1), i=1,3) / 0.1394830000E+02, -0.2277390000E+00, 0.7725620000E+01 /
  data (epsData(44,i,1), i=1,3) / 0.1399320000E+02, -0.6194780000E+00, 0.5783190000E+01 /
  data (epsData(45,i,1), i=1,3) / 0.1401670000E+02, -0.6457850000E+00, 0.4777610000E+01 /
  data (epsData(46,i,1), i=1,3) / 0.1404080000E+02, -0.5390220000E+00, 0.3749950000E+01 /
  data (epsData(47,i,1), i=1,3) / 0.1406570000E+02, -0.2636490000E+00, 0.2700630000E+01 /
  data (epsData(48,i,1), i=1,3) / 0.1409120000E+02, 0.2693450000E+00, 0.1630200000E+01 /
  data (epsData(49,i,1), i=1,3) / 0.1411740000E+02, 0.1014870000E+01, 0.8508700000E+00 /
  data (epsData(50,i,1), i=1,3) / 0.1414420000E+02, 0.1892310000E+01, 0.2193710000E+00 /
  data (epsData(51,i,1), i=1,3) / 0.1417180000E+02, 0.2788050000E+01, 0.0000000000E+00 /
  data (epsData(52,i,1), i=1,3) / 0.1420000000E+02, 0.3508630000E+01, 0.0000000000E+00 /
  data (epsData(53,i,1), i=1,3) / 0.1422890000E+02, 0.4068310000E+01, 0.0000000000E+00 /
  data (epsData(54,i,1), i=1,3) / 0.1428880000E+02, 0.4979580000E+01, 0.0000000000E+00 /
  data (epsData(55,i,1), i=1,3) / 0.1451970000E+02, 0.7738900000E+01, 0.0000000000E+00 /
  data (epsData(56,i,1), i=1,3) / 0.1459190000E+02, 0.8894610000E+01, 0.0000000000E+00 /
  data (epsData(57,i,1), i=1,3) / 0.1462890000E+02, 0.9734990000E+01, 0.2137060000E+00 /
  data (epsData(58,i,1), i=1,3) / 0.1466670000E+02, 0.1037380000E+02, 0.1131260000E+01 /
  data (epsData(59,i,1), i=1,3) / 0.1470510000E+02, 0.1051430000E+02, 0.1980650000E+01 /
  data (epsData(60,i,1), i=1,3) / 0.1503710000E+02, 0.1139340000E+02, 0.5061120000E+01 /
  data (epsData(61,i,1), i=1,3) / 0.1521940000E+02, 0.1127140000E+02, 0.7197200000E+01 /
  data (epsData(62,i,1), i=1,3) / 0.1541260000E+02, 0.1030740000E+02, 0.9025530000E+01 /
  data (epsData(63,i,1), i=1,3) / 0.1561670000E+02, 0.8916990000E+01, 0.1055690000E+02 /
  data (epsData(64,i,1), i=1,3) / 0.1594320000E+02, 0.6423210000E+01, 0.1166430000E+02 /
  data (epsData(65,i,1), i=1,3) / 0.1629430000E+02, 0.4051480000E+01, 0.1178210000E+02 /
  data (epsData(66,i,1), i=1,3) / 0.1666980000E+02, 0.2095600000E+01, 0.1128580000E+02 /
  data (epsData(67,i,1), i=1,3) / 0.1706980000E+02, 0.5828010000E+00, 0.1046750000E+02 /
  data (epsData(68,i,1), i=1,3) / 0.1764120000E+02, -0.8084750000E+00, 0.9175030000E+01 /
  data (epsData(69,i,1), i=1,3) / 0.1825620000E+02, -0.1644550000E+01, 0.7907440000E+01 /
  data (epsData(70,i,1), i=1,3) / 0.1900010000E+02, -0.2151470000E+01, 0.6652450000E+01 /
  data (epsData(71,i,1), i=1,3) / 0.1989130000E+02, -0.2396100000E+01, 0.5499610000E+01 /
  data (epsData(72,i,1), i=1,3) / 0.2105050000E+02, -0.2459110000E+01, 0.4413320000E+01 /
  data (epsData(73,i,1), i=1,3) / 0.2263720000E+02, -0.2382020000E+01, 0.3391560000E+01 /
  data (epsData(74,i,1), i=1,3) / 0.2462060000E+02, -0.2207970000E+01, 0.2526770000E+01 /
  data (epsData(75,i,1), i=1,3) / 0.2707000000E+02, -0.1959100000E+01, 0.1811460000E+01 /
  data (epsData(76,i,1), i=1,3) / 0.2984550000E+02, -0.1680380000E+01, 0.1287010000E+01 /
  data (epsData(77,i,1), i=1,3) / 0.3269750000E+02, -0.1432690000E+01, 0.9445500000E+00 /
  data (epsData(78,i,1), i=1,3) / 0.3591900000E+02, -0.1214680000E+01, 0.6948600000E+00 /
  data (epsData(79,i,1), i=1,3) / 0.3988130000E+02, -0.1013880000E+01, 0.4868300000E+00 /
  data (epsData(80,i,1), i=1,3) / 0.4427840000E+02, -0.8375990000E+00, 0.3287370000E+00 /
  data (epsData(81,i,1), i=1,3) / 0.4881180000E+02, -0.6904250000E+00, 0.2226440000E+00 /
  data (epsData(82,i,1), i=1,3) / 0.5349890000E+02, -0.5701800000E+00, 0.1536940000E+00 /
  data (epsData(83,i,1), i=1,3) / 0.5864590000E+02, -0.4688920000E+00, 0.1064140000E+00 /
  data (epsData(84,i,1), i=1,3) / 0.6447770000E+02, -0.3829750000E+00, 0.7311790000E-01 /
  data (epsData(85,i,1), i=1,3) / 0.7080530000E+02, -0.3137650000E+00, 0.5029590000E-01 /
  data (epsData(86,i,1), i=1,3) / 0.7762890000E+02, -0.2578920000E+00, 0.3447520000E-01 /
  data (epsData(87,i,1), i=1,3) / 0.8494850000E+02, -0.2127220000E+00, 0.2366260000E-01 /
  data (epsData(88,i,1), i=1,3) / 0.9336280000E+02, -0.1738420000E+00, 0.1583680000E-01 /
  data (epsData(89,i,1), i=1,3) / 0.1033120000E+03, -0.1400010000E+00, 0.1035130000E-01 /
  data (epsData(90,i,1), i=1,3) / 0.1139570000E+03, -0.1137240000E+00, 0.6859130000E-02 /
  data (epsData(91,i,1), i=1,3) / 0.1260310000E+03, -0.9193780000E-01, 0.4472000000E-02 /
  data (epsData(92,i,1), i=1,3) / 0.1396720000E+03, -0.7411470000E-01, 0.2883680000E-02 /
  data (epsData(93,i,1), i=1,3) / 0.1550280000E+03, -0.5957370000E-01, 0.1868180000E-02 /
  data (epsData(94,i,1), i=1,3) / 0.1731430000E+03, -0.4740740000E-01, 0.1154780000E-02 /
  data (epsData(95,i,1), i=1,3) / 0.1924590000E+03, -0.3810370000E-01, 0.7436990000E-03 /
  data (epsData(96,i,1), i=1,3) / 0.2139810000E+03, -0.3063340000E-01, 0.4700670000E-03 /
  data (epsData(97,i,1), i=1,3) / 0.2384250000E+03, -0.2454390000E-01, 0.2929230000E-03 /
  data (epsData(98,i,1), i=1,3) / 0.2660800000E+03, -0.1961880000E-01, 0.1809310000E-03 /
  data (epsData(99,i,1), i=1,3) / 0.2972510000E+03, -0.1565970000E-01, 0.1113270000E-03 /
  data (epsData(100,i,1), i=1,3) / 0.3322620000E+03, -0.1249370000E-01, 0.6871990000E-04 /
  data (epsData(101,i,1), i=1,3) / 0.3714570000E+03, -0.9968990000E-02, 0.4202970000E-04 /
  data (epsData(102,i,1), i=1,3) / 0.4151970000E+03, -0.7961460000E-02, 0.2667890000E-04 /
  data (epsData(103,i,1), i=1,3) / 0.4654040000E+03, -0.6325640000E-02, 0.1581900000E-04 /
  data (epsData(104,i,1), i=1,3) / 0.5211220000E+03, -0.5037290000E-02, 0.9791050000E-05 /
  data (epsData(105,i,1), i=1,3) / 0.5836480000E+03, -0.4010680000E-02, 0.5798750000E-05 /
  data (epsData(106,i,1), i=1,3) / 0.6545100000E+03, -0.3186010000E-02, 0.3410300000E-05 /
  data (epsData(107,i,1), i=1,3) / 0.7335340000E+03, -0.2534380000E-02, 0.2133630000E-05 /
  data (epsData(108,i,1), i=1,3) / 0.8224370000E+03, -0.2014700000E-02, 0.1297180000E-05 /
  data (epsData(109,i,1), i=1,3) / 0.9221170000E+03, -0.1601780000E-02, 0.7675840000E-06 /
  data (epsData(110,i,1), i=1,3) / 0.1033520000E+04, -0.1274480000E-02, 0.4515950000E-06 /
  data (epsData(111,i,1), i=1,3) / 0.1158900000E+04, -0.1013270000E-02, 0.2781460000E-06 /
  data (epsData(112,i,1), i=1,3) / 0.1299510000E+04, -0.8056170000E-03, 0.1706840000E-06 /
  data (epsData(113,i,1), i=1,3) / 0.1456710000E+04, -0.6409750000E-03, 0.1020790000E-06 /
  data (epsData(114,i,1), i=1,3) / 0.1633400000E+04, -0.5097030000E-03, 0.5501030000E-07 /
  data (epsData(115,i,1), i=1,3) / 0.1831330000E+04, -0.4054150000E-03, 0.3320610000E-07 /
  data (epsData(116,i,1), i=1,3) / 0.2054020000E+04, -0.3222320000E-03, 0.1997600000E-07 /
  data (epsData(117,i,1), i=1,3) / 0.2303730000E+04, -0.2561370000E-03, 0.1203160000E-07 /
  data (epsData(118,i,1), i=1,3) / 0.2584690000E+04, -0.2034610000E-03, 0.7258890000E-08 /
  data (epsData(119,i,1), i=1,3) / 0.2899770000E+04, -0.1616380000E-03, 0.4360760000E-08 /
  data (epsData(120,i,1), i=1,3) / 0.3251970000E+04, -0.1285160000E-03, 0.2630430000E-08 /
  data (epsData(121,i,1), i=1,3) / 0.3648880000E+04, -0.1020730000E-03, 0.1572960000E-08 /
  data (epsData(122,i,1), i=1,3) / 0.4092250000E+04, -0.8115070000E-04, 0.9481310000E-09 /
  data (epsData(123,i,1), i=1,3) / 0.4591060000E+04, -0.6447320000E-04, 0.5700030000E-09 /
  data (epsData(124,i,1), i=1,3) / 0.5150360000E+04, -0.5122960000E-04, 0.3416050000E-09 /
  data (epsData(125,i,1), i=1,3) / 0.5778270000E+04, -0.4069980000E-04, 0.2050620000E-09 /
  data (epsData(126,i,1), i=1,3) / 0.6483900000E+04, -0.3232290000E-04, 0.1236730000E-09 /
  data (epsData(127,i,1), i=1,3) / 0.7274230000E+04, -0.2568050000E-04, 0.7397210000E-10 /
  data (epsData(128,i,1), i=1,3) / 0.8160000000E+04, -0.2040770000E-04, 0.4430780000E-10 /
  data (epsData(129,i,1), i=1,3) / 0.9156630000E+04, -0.1620690000E-04, 0.2636490000E-10 /
  data (epsData(130,i,1), i=1,3) / 0.1027420000E+05, -0.1287280000E-04, 0.1579740000E-10 /
  data (epsData(131,i,1), i=1,3) / 0.1152740000E+05, -0.1022600000E-04, 0.9440060000E-11 /
  data (epsData(132,i,1), i=1,3) / 0.1293230000E+05, -0.8124820000E-05, 0.5573950000E-11 /
  data (epsData(133,i,1), i=1,3) / 0.1451140000E+05, -0.6452720000E-05, 0.3687010000E-11 /
  data (epsData(134,i,1), i=1,3) / 0.1628080000E+05, -0.5126420000E-05, 0.1968090000E-11 /
  data (epsData(135,i,1), i=1,3) / 0.1826640000E+05, -0.4072440000E-05, 0.1228630000E-11 /
  data (epsData(136,i,1), i=1,3) / 0.2049300000E+05, -0.3235580000E-05, 0.1891860000E-11 /
  data (epsData(137,i,1), i=1,3) / 0.2299280000E+05, -0.2570250000E-05, 0.7231780000E-12 /
  data (epsData(138,i,1), i=1,3) / 0.2579640000E+05, -0.2041940000E-05, 0.1672520000E-11 /
  data (epsData(139,i,1), i=1,3) / 0.2894300000E+05, -0.1622090000E-05, 0.6881620000E-12 /
  data (epsData(140,i,1), i=1,3) / 0.3247650000E+05, -0.1288320000E-05, 0.5860190000E-12 /
  data (epsData(141,i,1), i=1,3) / 0.3643860000E+05, -0.1023380000E-05, 0.2366280000E-11 /
  data (epsData(142,i,1), i=1,3) / 0.4088940000E+05, -0.8127130000E-06, 0.2593070000E-12 /
  data (epsData(143,i,1), i=1,3) / 0.4588080000E+05, -0.6455000000E-06, 0.1550210000E-11 /
  data (epsData(144,i,1), i=1,3) / 0.5147720000E+05, -0.5127760000E-06, 0.2887150000E-11 /
  data (epsData(145,i,1), i=1,3) / 0.5776600000E+05, -0.4072070000E-06, 0.2731860000E-11 /
  data (epsData(146,i,1), i=1,3) / 0.6481650000E+05, -0.3234360000E-06, 0.7799650000E-13 /
  data (epsData(147,i,1), i=1,3) / 0.7272970000E+05, -0.2568850000E-06, 0.2099830000E-11 /
  data (epsData(148,i,1), i=1,3) / 0.8160330000E+05, -0.2040540000E-06, 0.2578190000E-12 /
  data (epsData(149,i,1), i=1,3) / 0.9156670000E+05, -0.1620640000E-06, 0.1970900000E-11 /
  data (epsData(150,i,1), i=1,3) / 0.1000000000E+06, -0.1359020000E-06, 0.3395380000E-11 /
  data (epsData(1,i,2), i=1,3) / 0.5635290000E+01, 0.1857450000E+01, 0.5676010000E-02 /
  data (epsData(2,i,2), i=1,3) / 0.1062200000E+02, 0.2030620000E+01, 0.4410340000E-02 /
  data (epsData(3,i,2), i=1,3) / 0.1453460000E+02, 0.2305930000E+01, 0.4868790000E-02 /
  data (epsData(4,i,2), i=1,3) / 0.1748310000E+02, 0.2671910000E+01, 0.6194250000E-02 /
  data (epsData(5,i,2), i=1,3) / 0.1954560000E+02, 0.3095670000E+01, 0.8329910000E-02 /
  data (epsData(6,i,2), i=1,3) / 0.2104080000E+02, 0.3583320000E+01, 0.1157410000E-01 /
  data (epsData(7,i,2), i=1,3) / 0.2206830000E+02, 0.4098880000E+01, 0.1604540000E-01 /
  data (epsData(8,i,2), i=1,3) / 0.2294310000E+02, 0.4785570000E+01, 0.2427470000E-01 /
  data (epsData(9,i,2), i=1,3) / 0.2347620000E+02, 0.5448290000E+01, 0.3575960000E-01 /
  data (epsData(10,i,2), i=1,3) / 0.2383500000E+02, 0.6132350000E+01, 0.5201830000E-01 /
  data (epsData(11,i,2), i=1,3) / 0.2401550000E+02, 0.6628510000E+01, 0.7087180000E-01 /
  data (epsData(12,i,2), i=1,3) / 0.2419660000E+02, 0.7362010000E+01, 0.9448650000E-01 /
  data (epsData(13,i,2), i=1,3) / 0.2437840000E+02, 0.8592980000E+01, 0.2540720000E+00 /
  data (epsData(14,i,2), i=1,3) / 0.2456080000E+02, 0.1015340000E+02, 0.1843920000E+01 /
  data (epsData(15,i,2), i=1,3) / 0.2460540000E+02, 0.1023890000E+02, 0.2866030000E+01 /
  data (epsData(16,i,2), i=1,3) / 0.2476460000E+02, 0.9464350000E+01, 0.4989800000E+01 /
  data (epsData(17,i,2), i=1,3) / 0.2481260000E+02, 0.8781660000E+01, 0.5463600000E+01 /
  data (epsData(18,i,2), i=1,3) / 0.2494830000E+02, 0.7682880000E+01, 0.5563030000E+01 /
  data (epsData(19,i,2), i=1,3) / 0.2525850000E+02, 0.6470460000E+01, 0.5743790000E+01 /
  data (epsData(20,i,2), i=1,3) / 0.2595000000E+02, 0.5035450000E+01, 0.6011540000E+01 /
  data (epsData(21,i,2), i=1,3) / 0.2735520000E+02, 0.3205640000E+01, 0.6385340000E+01 /
  data (epsData(22,i,2), i=1,3) / 0.2902050000E+02, 0.1398570000E+01, 0.6461710000E+01 /
  data (epsData(23,i,2), i=1,3) / 0.3034840000E+02, 0.1079760000E+00, 0.6138710000E+01 /
  data (epsData(24,i,2), i=1,3) / 0.3155560000E+02, -0.8456310000E+00, 0.5568760000E+01 /
  data (epsData(25,i,2), i=1,3) / 0.3287770000E+02, -0.1557200000E+01, 0.4776820000E+01 /
  data (epsData(26,i,2), i=1,3) / 0.3443010000E+02, -0.1976780000E+01, 0.3845310000E+01 /
  data (epsData(27,i,2), i=1,3) / 0.3599100000E+02, -0.2093130000E+01, 0.3055600000E+01 /
  data (epsData(28,i,2), i=1,3) / 0.3780060000E+02, -0.2032420000E+01, 0.2358470000E+01 /
  data (epsData(29,i,2), i=1,3) / 0.4003190000E+02, -0.1850010000E+01, 0.1759340000E+01 /
  data (epsData(30,i,2), i=1,3) / 0.4275030000E+02, -0.1603820000E+01, 0.1289160000E+01 /
  data (epsData(31,i,2), i=1,3) / 0.4568910000E+02, -0.1370720000E+01, 0.9699260000E+00 /
  data (epsData(32,i,2), i=1,3) / 0.4903040000E+02, -0.1160240000E+01, 0.7378980000E+00 /
  data (epsData(33,i,2), i=1,3) / 0.5300900000E+02, -0.9692240000E+00, 0.5591910000E+00 /
  data (epsData(34,i,2), i=1,3) / 0.5771080000E+02, -0.7994250000E+00, 0.4208850000E+00 /
  data (epsData(35,i,2), i=1,3) / 0.6209180000E+02, -0.6732640000E+00, 0.3400710000E+00 /
  data (epsData(36,i,2), i=1,3) / 0.6722520000E+02, -0.5700590000E+00, 0.2834680000E+00 /
  data (epsData(37,i,2), i=1,3) / 0.7449510000E+02, -0.4743290000E+00, 0.2250410000E+00 /
  data (epsData(38,i,2), i=1,3) / 0.8720030000E+02, -0.3745470000E+00, 0.1427210000E+00 /
  data (epsData(39,i,2), i=1,3) / 0.9751630000E+02, -0.3072140000E+00, 0.9129840000E-01 /
  data (epsData(40,i,2), i=1,3) / 0.1079440000E+03, -0.2501910000E+00, 0.5927550000E-01 /
  data (epsData(41,i,2), i=1,3) / 0.1176400000E+03, -0.2084720000E+00, 0.4127900000E-01 /
  data (epsData(42,i,2), i=1,3) / 0.1289960000E+03, -0.1708140000E+00, 0.2869960000E-01 /
  data (epsData(43,i,2), i=1,3) / 0.1418040000E+03, -0.1397800000E+00, 0.2028030000E-01 /
  data (epsData(44,i,2), i=1,3) / 0.1566210000E+03, -0.1135560000E+00, 0.1402760000E-01 /
  data (epsData(45,i,2), i=1,3) / 0.1727690000E+03, -0.9261850000E-01, 0.9675100000E-02 /
  data (epsData(46,i,2), i=1,3) / 0.1912730000E+03, -0.7480330000E-01, 0.6323090000E-02 /
  data (epsData(47,i,2), i=1,3) / 0.2133910000E+03, -0.5946400000E-01, 0.4019640000E-02 /
  data (epsData(48,i,2), i=1,3) / 0.2374250000E+03, -0.4761230000E-01, 0.2658820000E-02 /
  data (epsData(49,i,2), i=1,3) / 0.2640390000E+03, -0.3819310000E-01, 0.1757660000E-02 /
  data (epsData(50,i,2), i=1,3) / 0.2934280000E+03, -0.3070230000E-01, 0.1147930000E-02 /
  data (epsData(51,i,2), i=1,3) / 0.3257970000E+03, -0.2473620000E-01, 0.7529750000E-03 /
  data (epsData(52,i,2), i=1,3) / 0.3627140000E+03, -0.1983770000E-01, 0.4894270000E-03 /
  data (epsData(53,i,2), i=1,3) / 0.4046340000E+03, -0.1585390000E-01, 0.3130290000E-03 /
  data (epsData(54,i,2), i=1,3) / 0.4520490000E+03, -0.1264880000E-01, 0.2078460000E-03 /
  data (epsData(55,i,2), i=1,3) / 0.5046720000E+03, -0.1010710000E-01, 0.1300330000E-03 /
  data (epsData(56,i,2), i=1,3) / 0.5646270000E+03, -0.8049920000E-02, 0.8123480000E-04 /
  data (epsData(57,i,2), i=1,3) / 0.6308290000E+03, -0.6429520000E-02, 0.5234590000E-04 /
  data (epsData(58,i,2), i=1,3) / 0.7047010000E+03, -0.5139010000E-02, 0.3299500000E-04 /
  data (epsData(59,i,2), i=1,3) / 0.7889260000E+03, -0.4091810000E-02, 0.2049770000E-04 /
  data (epsData(60,i,2), i=1,3) / 0.8845430000E+03, -0.3249110000E-02, 0.1236820000E-04 /
  data (epsData(61,i,2), i=1,3) / 0.9903740000E+03, -0.2588000000E-02, 0.8035690000E-05 /
  data (epsData(62,i,2), i=1,3) / 0.1109620000E+04, -0.2059170000E-02, 0.4841800000E-05 /
  data (epsData(63,i,2), i=1,3) / 0.1243530000E+04, -0.1637780000E-02, 0.3134890000E-05 /
  data (epsData(64,i,2), i=1,3) / 0.1394760000E+04, -0.1300830000E-02, 0.1828430000E-05 /
  data (epsData(65,i,2), i=1,3) / 0.1563510000E+04, -0.1034390000E-02, 0.1112880000E-05 /
  data (epsData(66,i,2), i=1,3) / 0.1752760000E+04, -0.8225710000E-03, 0.6839890000E-06 /
  data (epsData(67,i,2), i=1,3) / 0.1965980000E+04, -0.6535050000E-03, 0.4197040000E-06 /
  data (epsData(68,i,2), i=1,3) / 0.2205340000E+04, -0.5191300000E-03, 0.2568170000E-06 /
  data (epsData(69,i,2), i=1,3) / 0.2473210000E+04, -0.4126350000E-03, 0.1567220000E-06 /
  data (epsData(70,i,2), i=1,3) / 0.2773960000E+04, -0.3279220000E-03, 0.9567620000E-07 /
  data (epsData(71,i,2), i=1,3) / 0.3110540000E+04, -0.2607380000E-03, 0.5842230000E-07 /
  data (epsData(72,i,2), i=1,3) / 0.3488230000E+04, -0.2072950000E-03, 0.3560710000E-07 /
  data (epsData(73,i,2), i=1,3) / 0.3912900000E+04, -0.1647170000E-03, 0.2164460000E-07 /
  data (epsData(74,i,2), i=1,3) / 0.4388870000E+04, -0.1309120000E-03, 0.1314250000E-07 /
  data (epsData(75,i,2), i=1,3) / 0.4923250000E+04, -0.1040260000E-03, 0.7970640000E-08 /
  data (epsData(76,i,2), i=1,3) / 0.5523970000E+04, -0.8262450000E-04, 0.4823980000E-08 /
  data (epsData(77,i,2), i=1,3) / 0.6197040000E+04, -0.6564710000E-04, 0.2978490000E-08 /
  data (epsData(78,i,2), i=1,3) / 0.6951860000E+04, -0.5216280000E-04, 0.1806120000E-08 /
  data (epsData(79,i,2), i=1,3) / 0.7798910000E+04, -0.4144560000E-04, 0.1089770000E-08 /
  data (epsData(80,i,2), i=1,3) / 0.8749830000E+04, -0.3292550000E-04, 0.6537430000E-09 /
  data (epsData(81,i,2), i=1,3) / 0.9817550000E+04, -0.2615250000E-04, 0.3931510000E-09 /
  data (epsData(82,i,2), i=1,3) / 0.1101250000E+05, -0.2078450000E-04, 0.2366390000E-09 /
  data (epsData(83,i,2), i=1,3) / 0.1235390000E+05, -0.1651560000E-04, 0.1420300000E-09 /
  data (epsData(84,i,2), i=1,3) / 0.1385920000E+05, -0.1312270000E-04, 0.8544000000E-10 /
  data (epsData(85,i,2), i=1,3) / 0.1554810000E+05, -0.1042650000E-04, 0.5121740000E-10 /
  data (epsData(86,i,2), i=1,3) / 0.1744610000E+05, -0.8281200000E-05, 0.3064380000E-10 /
  data (epsData(87,i,2), i=1,3) / 0.1957200000E+05, -0.6579860000E-05, 0.1840710000E-10 /
  data (epsData(88,i,2), i=1,3) / 0.2196210000E+05, -0.5225610000E-05, 0.1095950000E-10 /
  data (epsData(89,i,2), i=1,3) / 0.2464020000E+05, -0.4151370000E-05, 0.6526820000E-11 /
  data (epsData(90,i,2), i=1,3) / 0.2764400000E+05, -0.3298200000E-05, 0.4002740000E-11 /
  data (epsData(91,i,2), i=1,3) / 0.3101530000E+05, -0.2620160000E-05, 0.2464000000E-11 /
  data (epsData(92,i,2), i=1,3) / 0.3480030000E+05, -0.2081190000E-05, 0.1404580000E-11 /
  data (epsData(93,i,2), i=1,3) / 0.3904360000E+05, -0.1653400000E-05, 0.9191080000E-12 /
  data (epsData(94,i,2), i=1,3) / 0.4380850000E+05, -0.1313290000E-05, 0.5795550000E-12 /
  data (epsData(95,i,2), i=1,3) / 0.4915810000E+05, -0.1043000000E-05, 0.4715960000E-12 /
  data (epsData(96,i,2), i=1,3) / 0.5516190000E+05, -0.8283170000E-06, 0.1832530000E-12 /
  data (epsData(97,i,2), i=1,3) / 0.6189180000E+05, -0.6579750000E-06, 0.1022750000E-12 /
  data (epsData(98,i,2), i=1,3) / 0.6944490000E+05, -0.5226290000E-06, 0.4066290000E-12 /
  data (epsData(99,i,2), i=1,3) / 0.7792610000E+05, -0.4150580000E-06, 0.4600230000E-13 /
  data (epsData(100,i,2), i=1,3) / 0.8744010000E+05, -0.3296500000E-06, 0.2395200000E-12 /
  data (epsData(101,i,2), i=1,3) / 0.9811300000E+05, -0.2618310000E-06, 0.2846000000E-12 /
  data (epsData(102,i,2), i=1,3) / 0.1000000000E+06, -0.2520820000E-06, 0.4644810000E-12 /
  data (epsData(1,i,3), i=1,3) / 0.2506580000E-02, -0.2744470000E+03, 0.9288570000E+03 /
  data (epsData(2,i,3), i=1,3) / 0.4693440000E-02, -0.2509210000E+03, 0.3334770000E+03 /
  data (epsData(3,i,3), i=1,3) / 0.7560590000E-02, -0.1113860000E+03, 0.1631440000E+03 /
  data (epsData(4,i,3), i=1,3) / 0.1110800000E-01, -0.4970810000E+02, 0.9543220000E+02 /
  data (epsData(5,i,3), i=1,3) / 0.1533570000E-01, -0.2054600000E+02, 0.6248030000E+02 /
  data (epsData(6,i,3), i=1,3) / 0.2024370000E-01, -0.5411510000E+01, 0.4412630000E+02 /
  data (epsData(7,i,3), i=1,3) / 0.2583200000E-01, 0.3078910000E+01, 0.3288210000E+02 /
  data (epsData(8,i,3), i=1,3) / 0.3210060000E-01, 0.8150450000E+01, 0.2549420000E+02 /
  data (epsData(9,i,3), i=1,3) / 0.3904940000E-01, 0.1133860000E+02, 0.2037470000E+02 /
  data (epsData(10,i,3), i=1,3) / 0.4667860000E-01, 0.1342990000E+02, 0.1667660000E+02 /
  data (epsData(11,i,3), i=1,3) / 0.5498800000E-01, 0.1485250000E+02, 0.1391490000E+02 /
  data (epsData(12,i,3), i=1,3) / 0.6397770000E-01, 0.1585230000E+02, 0.1179570000E+02 /
  data (epsData(13,i,3), i=1,3) / 0.7364770000E-01, 0.1657740000E+02, 0.1013250000E+02 /
  data (epsData(14,i,3), i=1,3) / 0.9502860000E-01, 0.1756450000E+02, 0.7720770000E+01 /
  data (epsData(15,i,3), i=1,3) / 0.1191310000E+00, 0.1813620000E+02, 0.6268570000E+01 /
  data (epsData(16,i,3), i=1,3) / 0.1754980000E+00, 0.1847660000E+02, 0.4400910000E+01 /
  data (epsData(17,i,3), i=1,3) / 0.2804580000E+00, 0.1860120000E+02, 0.2826020000E+01 /
  data (epsData(18,i,3), i=1,3) / 0.4838160000E+00, 0.1873440000E+02, 0.1728300000E+01 /
  data (epsData(19,i,3), i=1,3) / 0.1018280000E+01, 0.1904340000E+02, 0.9741000000E+00 /
  data (epsData(20,i,3), i=1,3) / 0.1949930000E+01, 0.2035580000E+02, 0.7414130000E+00 /
  data (epsData(21,i,3), i=1,3) / 0.2799080000E+01, 0.2278970000E+02, 0.7921470000E+00 /
  data (epsData(22,i,3), i=1,3) / 0.3450210000E+01, 0.2609760000E+02, 0.9977130000E+00 /
  data (epsData(23,i,3), i=1,3) / 0.3946480000E+01, 0.3025670000E+02, 0.1341220000E+01 /
  data (epsData(24,i,3), i=1,3) / 0.4321360000E+01, 0.3535330000E+02, 0.1879690000E+01 /
  data (epsData(25,i,3), i=1,3) / 0.4554460000E+01, 0.4016190000E+02, 0.2545100000E+01 /
  data (epsData(26,i,3), i=1,3) / 0.4713260000E+01, 0.4482630000E+02, 0.3609840000E+01 /
  data (epsData(27,i,3), i=1,3) / 0.4793680000E+01, 0.4776530000E+02, 0.4229550000E+01 /
  data (epsData(28,i,3), i=1,3) / 0.4874780000E+01, 0.5189510000E+02, 0.4811430000E+01 /
  data (epsData(29,i,3), i=1,3) / 0.4956560000E+01, 0.5658540000E+02, 0.7943400000E+01 /
  data (epsData(30,i,3), i=1,3) / 0.5122160000E+01, 0.6540450000E+02, 0.1586200000E+02 /
  data (epsData(31,i,3), i=1,3) / 0.5205980000E+01, 0.7279130000E+02, 0.2049530000E+02 /
  data (epsData(32,i,3), i=1,3) / 0.5303060000E+01, 0.7747890000E+02, 0.4011400000E+02 /
  data (epsData(33,i,3), i=1,3) / 0.5422790000E+01, 0.7046760000E+02, 0.5915930000E+02 /
  data (epsData(34,i,3), i=1,3) / 0.5495920000E+01, 0.6077090000E+02, 0.6996230000E+02 /
  data (epsData(35,i,3), i=1,3) / 0.5566670000E+01, 0.5008950000E+02, 0.7165440000E+02 /
  data (epsData(36,i,3), i=1,3) / 0.5817360000E+01, 0.2707100000E+02, 0.7468620000E+02 /
  data (epsData(37,i,3), i=1,3) / 0.6051370000E+01, 0.1174530000E+02, 0.7080650000E+02 /
  data (epsData(38,i,3), i=1,3) / 0.6255460000E+01, 0.1668280000E+01, 0.6629720000E+02 /
  data (epsData(39,i,3), i=1,3) / 0.6524510000E+01, -0.8127340000E+01, 0.5976230000E+02 /
  data (epsData(40,i,3), i=1,3) / 0.6872820000E+01, -0.1683690000E+02, 0.5088210000E+02 /
  data (epsData(41,i,3), i=1,3) / 0.7316700000E+01, -0.2352680000E+02, 0.3977760000E+02 /
  data (epsData(42,i,3), i=1,3) / 0.7757530000E+01, -0.2632060000E+02, 0.2934880000E+02 /
  data (epsData(43,i,3), i=1,3) / 0.8178960000E+01, -0.2623430000E+02, 0.2079380000E+02 /
  data (epsData(44,i,3), i=1,3) / 0.8566720000E+01, -0.2456700000E+02, 0.1450380000E+02 /
  data (epsData(45,i,3), i=1,3) / 0.8978980000E+01, -0.2198470000E+02, 0.9482670000E+01 /
  data (epsData(46,i,3), i=1,3) / 0.9490890000E+01, -0.1839150000E+02, 0.5237760000E+01 /
  data (epsData(47,i,3), i=1,3) / 0.9956200000E+01, -0.1530610000E+02, 0.2819420000E+01 /
  data (epsData(48,i,3), i=1,3) / 0.1036270000E+02, -0.1286590000E+02, 0.1520640000E+01 /
  data (epsData(49,i,3), i=1,3) / 0.1078610000E+02, -0.1066700000E+02, 0.7320940000E+00 /
  data (epsData(50,i,3), i=1,3) / 0.1122660000E+02, -0.8723390000E+01, 0.3250520000E+00 /
  data (epsData(51,i,3), i=1,3) / 0.1159130000E+02, -0.7373310000E+01, 0.2369660000E+00 /
  data (epsData(52,i,3), i=1,3) / 0.1206240000E+02, -0.5953910000E+01, 0.3180750000E+00 /
  data (epsData(53,i,3), i=1,3) / 0.1255050000E+02, -0.4813790000E+01, 0.5643420000E+00 /
  data (epsData(54,i,3), i=1,3) / 0.1305560000E+02, -0.3946330000E+01, 0.8975560000E+00 /
  data (epsData(55,i,3), i=1,3) / 0.1357770000E+02, -0.3336360000E+01, 0.1266760000E+01 /
  data (epsData(56,i,3), i=1,3) / 0.1411680000E+02, -0.2958030000E+01, 0.1605980000E+01 /
  data (epsData(57,i,3), i=1,3) / 0.1467300000E+02, -0.2763070000E+01, 0.1853820000E+01 /
  data (epsData(58,i,3), i=1,3) / 0.1536280000E+02, -0.2670300000E+01, 0.1994820000E+01 /
  data (epsData(59,i,3), i=1,3) / 0.1681580000E+02, -0.2535990000E+01, 0.1877470000E+01 /
  data (epsData(60,i,3), i=1,3) / 0.2102310000E+02, -0.1911680000E+01, 0.1166720000E+01 /
  data (epsData(61,i,3), i=1,3) / 0.2424450000E+02, -0.1497640000E+01, 0.7872750000E+00 /
  data (epsData(62,i,3), i=1,3) / 0.2741770000E+02, -0.1151590000E+01, 0.5201640000E+00 /
  data (epsData(63,i,3), i=1,3) / 0.3028040000E+02, -0.8802450000E+00, 0.3570530000E+00 /
  data (epsData(64,i,3), i=1,3) / 0.3312220000E+02, -0.6479060000E+00, 0.2517640000E+00 /
  data (epsData(65,i,3), i=1,3) / 0.3550600000E+02, -0.4795100000E+00, 0.1962190000E+00 /
  data (epsData(66,i,3), i=1,3) / 0.3819890000E+02, -0.3148530000E+00, 0.1564800000E+00 /
  data (epsData(67,i,3), i=1,3) / 0.4078670000E+02, -0.1755220000E+00, 0.1339890000E+00 /
  data (epsData(68,i,3), i=1,3) / 0.4485200000E+02, 0.2555210000E-01, 0.1129980000E+00 /
  data (epsData(69,i,3), i=1,3) / 0.4696740000E+02, 0.1361470000E+00, 0.1047350000E+00 /
  data (epsData(70,i,3), i=1,3) / 0.4865070000E+02, 0.2379920000E+00, 0.9878190000E-01 /
  data (epsData(71,i,3), i=1,3) / 0.5012020000E+02, 0.3475180000E+00, 0.9380080000E-01 /
  data (epsData(72,i,3), i=1,3) / 0.5136340000E+02, 0.4707310000E+00, 0.8999630000E-01 /
  data (epsData(73,i,3), i=1,3) / 0.5211750000E+02, 0.5710700000E+00, 0.8848710000E-01 /
  data (epsData(74,i,3), i=1,3) / 0.5287770000E+02, 0.7102560000E+00, 0.8837760000E-01 /
  data (epsData(75,i,3), i=1,3) / 0.5338790000E+02, 0.8454050000E+00, 0.9010820000E-01 /
  data (epsData(76,i,3), i=1,3) / 0.5364400000E+02, 0.9377040000E+00, 0.9248890000E-01 /
  data (epsData(77,i,3), i=1,3) / 0.5390080000E+02, 0.1060890000E+01, 0.9725840000E-01 /
  data (epsData(78,i,3), i=1,3) / 0.5415830000E+02, 0.1245280000E+01, 0.1091990000E+00 /
  data (epsData(79,i,3), i=1,3) / 0.5441650000E+02, 0.1543260000E+01, 0.1749490000E+00 /
  data (epsData(80,i,3), i=1,3) / 0.5467530000E+02, 0.1908310000E+01, 0.5926550000E+00 /
  data (epsData(81,i,3), i=1,3) / 0.5472760000E+02, 0.1937070000E+01, 0.7895270000E+00 /
  data (epsData(82,i,3), i=1,3) / 0.5485000000E+02, 0.1851600000E+01, 0.1185630000E+01 /
  data (epsData(83,i,3), i=1,3) / 0.5489590000E+02, 0.1735410000E+01, 0.1333270000E+01 /
  data (epsData(84,i,3), i=1,3) / 0.5496670000E+02, 0.1549850000E+01, 0.1374540000E+01 /
  data (epsData(85,i,3), i=1,3) / 0.5524420000E+02, 0.1188840000E+01, 0.1477660000E+01 /
  data (epsData(86,i,3), i=1,3) / 0.5558470000E+02, 0.8801960000E+00, 0.1498670000E+01 /
  data (epsData(87,i,3), i=1,3) / 0.5592450000E+02, 0.6720160000E+00, 0.1443140000E+01 /
  data (epsData(88,i,3), i=1,3) / 0.5631940000E+02, 0.5522370000E+00, 0.1370740000E+01 /
  data (epsData(89,i,3), i=1,3) / 0.5704320000E+02, 0.4349140000E+00, 0.1366830000E+01 /
  data (epsData(90,i,3), i=1,3) / 0.5733410000E+02, 0.3452150000E+00, 0.1416030000E+01 /
  data (epsData(91,i,3), i=1,3) / 0.5764190000E+02, 0.2118260000E+00, 0.1410520000E+01 /
  data (epsData(92,i,3), i=1,3) / 0.5810140000E+02, 0.7610510000E-01, 0.1308310000E+01 /
  data (epsData(93,i,3), i=1,3) / 0.5874120000E+02, -0.8897240000E-02, 0.1177830000E+01 /
  data (epsData(94,i,3), i=1,3) / 0.5943610000E+02, -0.3882930000E-01, 0.1070400000E+01 /
  data (epsData(95,i,3), i=1,3) / 0.6044840000E+02, -0.4436170000E-01, 0.9767040000E+00 /
  data (epsData(96,i,3), i=1,3) / 0.6175320000E+02, -0.3389930000E-01, 0.9196630000E+00 /
  data (epsData(97,i,3), i=1,3) / 0.6362670000E+02, -0.5277350000E-01, 0.9073300000E+00 /
  data (epsData(98,i,3), i=1,3) / 0.6723900000E+02, -0.1573620000E+00, 0.8816650000E+00 /
  data (epsData(99,i,3), i=1,3) / 0.7086490000E+02, -0.2664450000E+00, 0.8061180000E+00 /
  data (epsData(100,i,3), i=1,3) / 0.7545170000E+02, -0.3380820000E+00, 0.6812600000E+00 /
  data (epsData(101,i,3), i=1,3) / 0.8136710000E+02, -0.3669270000E+00, 0.5520000000E+00 /
  data (epsData(102,i,3), i=1,3) / 0.8909650000E+02, -0.3694640000E+00, 0.4195020000E+00 /
  data (epsData(103,i,3), i=1,3) / 0.9780820000E+02, -0.3429520000E+00, 0.3144050000E+00 /
  data (epsData(104,i,3), i=1,3) / 0.1099420000E+03, -0.3014720000E+00, 0.2144020000E+00 /
  data (epsData(105,i,3), i=1,3) / 0.1220570000E+03, -0.2569070000E+00, 0.1500770000E+00 /
  data (epsData(106,i,3), i=1,3) / 0.1347110000E+03, -0.2177730000E+00, 0.1080120000E+00 /
  data (epsData(107,i,3), i=1,3) / 0.1491680000E+03, -0.1819760000E+00, 0.7602780000E-01 /
  data (epsData(108,i,3), i=1,3) / 0.1656020000E+03, -0.1496210000E+00, 0.5212040000E-01 /
  data (epsData(109,i,3), i=1,3) / 0.1821000000E+03, -0.1241090000E+00, 0.3726540000E-01 /
  data (epsData(110,i,3), i=1,3) / 0.1997420000E+03, -0.1033680000E+00, 0.2710660000E-01 /
  data (epsData(111,i,3), i=1,3) / 0.2223270000E+03, -0.8340090000E-01, 0.1822850000E-01 /
  data (epsData(112,i,3), i=1,3) / 0.2475680000E+03, -0.6698990000E-01, 0.1227780000E-01 /
  data (epsData(113,i,3), i=1,3) / 0.2756740000E+03, -0.5375600000E-01, 0.8227210000E-02 /
  data (epsData(114,i,3), i=1,3) / 0.3068610000E+03, -0.4310260000E-01, 0.5486000000E-02 /
  data (epsData(115,i,3), i=1,3) / 0.3401060000E+03, -0.3482530000E-01, 0.3733240000E-02 /
  data (epsData(116,i,3), i=1,3) / 0.3780560000E+03, -0.2800390000E-01, 0.2486160000E-02 /
  data (epsData(117,i,3), i=1,3) / 0.4176720000E+03, -0.2277300000E-01, 0.1690990000E-02 /
  data (epsData(118,i,3), i=1,3) / 0.4603190000E+03, -0.1861700000E-01, 0.1182440000E-02 /
  data (epsData(119,i,3), i=1,3) / 0.5155710000E+03, -0.1475370000E-01, 0.7439990000E-03 /
  data (epsData(120,i,3), i=1,3) / 0.5743500000E+03, -0.1182040000E-01, 0.5019750000E-03 /
  data (epsData(121,i,3), i=1,3) / 0.6366560000E+03, -0.9564350000E-02, 0.3329900000E-03 /
  data (epsData(122,i,3), i=1,3) / 0.7119100000E+03, -0.7609190000E-02, 0.2108230000E-03 /
  data (epsData(123,i,3), i=1,3) / 0.7967550000E+03, -0.6049060000E-02, 0.1330440000E-03 /
  data (epsData(124,i,3), i=1,3) / 0.8888780000E+03, -0.4839430000E-02, 0.8565140000E-04 /
  data (epsData(125,i,3), i=1,3) / 0.9920290000E+03, -0.3870660000E-02, 0.5505010000E-04 /
  data (epsData(126,i,3), i=1,3) / 0.1107170000E+04, -0.3097280000E-02, 0.3504090000E-04 /
  data (epsData(127,i,3), i=1,3) / 0.1236570000E+04, -0.2476050000E-02, 0.2249690000E-04 /
  data (epsData(128,i,3), i=1,3) / 0.1382860000E+04, -0.1975280000E-02, 0.1409560000E-04 /
  data (epsData(129,i,3), i=1,3) / 0.1549070000E+04, -0.1570920000E-02, 0.8453300000E-05 /
  data (epsData(130,i,3), i=1,3) / 0.1734140000E+04, -0.1251060000E-02, 0.5238210000E-05 /
  data (epsData(131,i,3), i=1,3) / 0.1942810000E+04, -0.9952070000E-03, 0.3252300000E-05 /
  data (epsData(132,i,3), i=1,3) / 0.2177270000E+04, -0.7913880000E-03, 0.2010370000E-05 /
  data (epsData(133,i,3), i=1,3) / 0.2439840000E+04, -0.6295490000E-03, 0.1246050000E-05 /
  data (epsData(134,i,3), i=1,3) / 0.2734870000E+04, -0.5006040000E-03, 0.7684910000E-06 /
  data (epsData(135,i,3), i=1,3) / 0.3065290000E+04, -0.3982040000E-03, 0.4728020000E-06 /
  data (epsData(136,i,3), i=1,3) / 0.3436330000E+04, -0.3166640000E-03, 0.2903420000E-06 /
  data (epsData(137,i,3), i=1,3) / 0.3853830000E+04, -0.2516430000E-03, 0.1776090000E-06 /
  data (epsData(138,i,3), i=1,3) / 0.4322060000E+04, -0.1999910000E-03, 0.1086240000E-06 /
  data (epsData(139,i,3), i=1,3) / 0.4848090000E+04, -0.1588930000E-03, 0.6615420000E-07 /
  data (epsData(140,i,3), i=1,3) / 0.5437100000E+04, -0.1262970000E-03, 0.4038270000E-07 /
  data (epsData(141,i,3), i=1,3) / 0.6097440000E+04, -0.1004000000E-03, 0.2458500000E-07 /
  data (epsData(142,i,3), i=1,3) / 0.6838440000E+04, -0.7980590000E-04, 0.1500510000E-07 /
  data (epsData(143,i,3), i=1,3) / 0.7670490000E+04, -0.6342180000E-04, 0.9133600000E-08 /
  data (epsData(144,i,3), i=1,3) / 0.8605130000E+04, -0.5038690000E-04, 0.5516770000E-08 /
  data (epsData(145,i,3), i=1,3) / 0.9655180000E+04, -0.4001920000E-04, 0.3366570000E-08 /
  data (epsData(146,i,3), i=1,3) / 0.1083100000E+05, -0.3179910000E-04, 0.2071990000E-08 /
  data (epsData(147,i,3), i=1,3) / 0.1215170000E+05, -0.2526100000E-04, 0.1363480000E-08 /
  data (epsData(148,i,3), i=1,3) / 0.1363450000E+05, -0.2006420000E-04, 0.9119710000E-09 /
  data (epsData(149,i,3), i=1,3) / 0.1529900000E+05, -0.1593520000E-04, 0.5211690000E-09 /
  data (epsData(150,i,3), i=1,3) / 0.1716590000E+05, -0.1265710000E-04, 0.4178930000E-09 /
  data (epsData(151,i,3), i=1,3) / 0.1926310000E+05, -0.1005080000E-04, 0.1892460000E-09 /
  data (epsData(152,i,3), i=1,3) / 0.2161130000E+05, -0.7985190000E-05, 0.1218830000E-09 /
  data (epsData(153,i,3), i=1,3) / 0.2424960000E+05, -0.6342050000E-05, 0.1826190000E-09 /
  data (epsData(154,i,3), i=1,3) / 0.2720400000E+05, -0.5039280000E-05, 0.9084530000E-10 /
  data (epsData(155,i,3), i=1,3) / 0.3052790000E+05, -0.4001650000E-05, 0.1725220000E-09 /
  data (epsData(156,i,3), i=1,3) / 0.3425490000E+05, -0.3178160000E-05, 0.7382140000E-10 /
  data (epsData(157,i,3), i=1,3) / 0.3843500000E+05, -0.2524520000E-05, 0.8702300000E-10 /
  data (epsData(158,i,3), i=1,3) / 0.4312360000E+05, -0.2005390000E-05, 0.6550370000E-10 /
  data (epsData(159,i,3), i=1,3) / 0.4837380000E+05, -0.1593720000E-05, 0.1242640000E-10 /
  data (epsData(160,i,3), i=1,3) / 0.5426040000E+05, -0.1266710000E-05, 0.2836110000E-10 /
  data (epsData(161,i,3), i=1,3) / 0.6087190000E+05, -0.1006440000E-05, 0.3482770000E-10 /
  data (epsData(162,i,3), i=1,3) / 0.6830470000E+05, -0.7993320000E-06, 0.1533630000E-10 /
  data (epsData(163,i,3), i=1,3) / 0.7663450000E+05, -0.6350080000E-06, 0.1689130000E-10 /
  data (epsData(164,i,3), i=1,3) / 0.8598330000E+05, -0.5044340000E-06, 0.1890200000E-10 /
  data (epsData(165,i,3), i=1,3) / 0.9646470000E+05, -0.4007630000E-06, 0.1465260000E-10 /
  data (epsData(166,i,3), i=1,3) / 0.1000000000E+06, -0.3730220000E-06, 0.2988220000E-10 /
  data (epsData(1,i,4), i=1,3) / 0.2506580000E-02, -0.7337840000E+03, 0.2878000000E+04 /
  data (epsData(2,i,4), i=1,3) / 0.4693440000E-02, -0.7133740000E+03, 0.1059420000E+04 /
  data (epsData(3,i,4), i=1,3) / 0.7560590000E-02, -0.3104390000E+03, 0.5287890000E+03 /
  data (epsData(4,i,4), i=1,3) / 0.1110800000E-01, -0.1308810000E+03, 0.3140600000E+03 /
  data (epsData(5,i,4), i=1,3) / 0.1533570000E-01, -0.4554250000E+02, 0.2079650000E+03 /
  data (epsData(6,i,4), i=1,3) / 0.2024370000E-01, -0.8252860000E+00, 0.1481280000E+03 /
  data (epsData(7,i,4), i=1,3) / 0.2583200000E-01, 0.2506660000E+02, 0.1110940000E+03 /
  data (epsData(8,i,4), i=1,3) / 0.3210060000E-01, 0.4156380000E+02, 0.8765660000E+02 /
  data (epsData(9,i,4), i=1,3) / 0.3904940000E-01, 0.5101490000E+02, 0.7486420000E+02 /
  data (epsData(10,i,4), i=1,3) / 0.4667860000E-01, 0.5514540000E+02, 0.6440080000E+02 /
  data (epsData(11,i,4), i=1,3) / 0.5498800000E-01, 0.5718100000E+02, 0.5583270000E+02 /
  data (epsData(12,i,4), i=1,3) / 0.6397770000E-01, 0.5826250000E+02, 0.4877700000E+02 /
  data (epsData(13,i,4), i=1,3) / 0.8399800000E-01, 0.5919340000E+02, 0.3802800000E+02 /
  data (epsData(14,i,4), i=1,3) / 0.1191310000E+00, 0.5955850000E+02, 0.2739580000E+02 /
  data (epsData(15,i,4), i=1,3) / 0.1754980000E+00, 0.5968740000E+02, 0.1890030000E+02 /
  data (epsData(16,i,4), i=1,3) / 0.2612640000E+00, 0.5998060000E+02, 0.1283740000E+02 /
  data (epsData(17,i,4), i=1,3) / 0.3866320000E+00, 0.6091920000E+02, 0.9338710000E+01 /
  data (epsData(18,i,4), i=1,3) / 0.6500010000E+00, 0.6219240000E+02, 0.6567610000E+01 /
  data (epsData(19,i,4), i=1,3) / 0.1018280000E+01, 0.6593480000E+02, 0.5808780000E+01 /
  data (epsData(20,i,4), i=1,3) / 0.1337820000E+01, 0.7157330000E+02, 0.6118380000E+01 /
  data (epsData(21,i,4), i=1,3) / 0.1606050000E+01, 0.7903070000E+02, 0.7048880000E+01 /
  data (epsData(22,i,4), i=1,3) / 0.1848280000E+01, 0.8916960000E+02, 0.8512030000E+01 /
  data (epsData(23,i,4), i=1,3) / 0.2054310000E+01, 0.1017390000E+03, 0.1149760000E+02 /
  data (epsData(24,i,4), i=1,3) / 0.2161410000E+01, 0.1120970000E+03, 0.1397180000E+02 /
  data (epsData(25,i,4), i=1,3) / 0.2327150000E+01, 0.1310120000E+03, 0.2162660000E+02 /
  data (epsData(26,i,4), i=1,3) / 0.2383760000E+01, 0.1414990000E+03, 0.2372820000E+02 /
  data (epsData(27,i,4), i=1,3) / 0.2441050000E+01, 0.1568220000E+03, 0.2893810000E+02 /
  data (epsData(28,i,4), i=1,3) / 0.2499020000E+01, 0.1700410000E+03, 0.4868820000E+02 /
  data (epsData(29,i,4), i=1,3) / 0.2617000000E+01, 0.1819890000E+03, 0.8351280000E+02 /
  data (epsData(30,i,4), i=1,3) / 0.2677010000E+01, 0.1903350000E+03, 0.9882470000E+02 /
  data (epsData(31,i,4), i=1,3) / 0.2737710000E+01, 0.2072090000E+03, 0.1187650000E+03 /
  data (epsData(32,i,4), i=1,3) / 0.2799080000E+01, 0.2088560000E+03, 0.1829360000E+03 /
  data (epsData(33,i,4), i=1,3) / 0.2861130000E+01, 0.1814760000E+03, 0.2295540000E+03 /
  data (epsData(34,i,4), i=1,3) / 0.2987270000E+01, 0.1022790000E+03, 0.2872230000E+03 /
  data (epsData(35,i,4), i=1,3) / 0.3027550000E+01, 0.6302620000E+02, 0.3034570000E+03 /
  data (epsData(36,i,4), i=1,3) / 0.3048980000E+01, 0.3638910000E+02, 0.2976280000E+03 /
  data (epsData(37,i,4), i=1,3) / 0.3098300000E+01, 0.7167600000E-01, 0.2786930000E+03 /
  data (epsData(38,i,4), i=1,3) / 0.3179940000E+01, -0.3834740000E+02, 0.2495850000E+03 /
  data (epsData(39,i,4), i=1,3) / 0.3306130000E+01, -0.7616430000E+02, 0.2094700000E+03 /
  data (epsData(40,i,4), i=1,3) / 0.3416670000E+01, -0.9819470000E+02, 0.1728610000E+03 /
  data (epsData(41,i,4), i=1,3) / 0.3544230000E+01, -0.1082430000E+03, 0.1339130000E+03 /
  data (epsData(42,i,4), i=1,3) / 0.3628920000E+01, -0.1081760000E+03, 0.1108050000E+03 /
  data (epsData(43,i,4), i=1,3) / 0.3850360000E+01, -0.1007400000E+03, 0.7013450000E+02 /
  data (epsData(44,i,4), i=1,3) / 0.3955460000E+01, -0.9368670000E+02, 0.5481870000E+02 /
  data (epsData(45,i,4), i=1,3) / 0.4224510000E+01, -0.7713700000E+02, 0.2963570000E+02 /
  data (epsData(46,i,4), i=1,3) / 0.4437100000E+01, -0.6473960000E+02, 0.1847160000E+02 /
  data (epsData(47,i,4), i=1,3) / 0.4572820000E+01, -0.5732930000E+02, 0.1304420000E+02 /
  data (epsData(48,i,4), i=1,3) / 0.4862620000E+01, -0.4460880000E+02, 0.6172370000E+01 /
  data (epsData(49,i,4), i=1,3) / 0.5122830000E+01, -0.3563130000E+02, 0.3038540000E+01 /
  data (epsData(50,i,4), i=1,3) / 0.5343240000E+01, -0.2950310000E+02, 0.1901260000E+01 /
  data (epsData(51,i,4), i=1,3) / 0.5515690000E+01, -0.2546100000E+02, 0.1439200000E+01 /
  data (epsData(52,i,4), i=1,3) / 0.5755150000E+01, -0.2098750000E+02, 0.1298570000E+01 /
  data (epsData(53,i,4), i=1,3) / 0.6069780000E+01, -0.1653350000E+02, 0.1677510000E+01 /
  data (epsData(54,i,4), i=1,3) / 0.6401420000E+01, -0.1318010000E+02, 0.2282470000E+01 /
  data (epsData(55,i,4), i=1,3) / 0.6750070000E+01, -0.1068790000E+02, 0.2901050000E+01 /
  data (epsData(56,i,4), i=1,3) / 0.7115720000E+01, -0.8821940000E+01, 0.3432100000E+01 /
  data (epsData(57,i,4), i=1,3) / 0.7576950000E+01, -0.7154320000E+01, 0.3915760000E+01 /
  data (epsData(58,i,4), i=1,3) / 0.8212590000E+01, -0.5573000000E+01, 0.4332950000E+01 /
  data (epsData(59,i,4), i=1,3) / 0.9144240000E+01, -0.4043300000E+01, 0.4765600000E+01 /
  data (epsData(60,i,4), i=1,3) / 0.9862620000E+01, -0.3336590000E+01, 0.5191490000E+01 /
  data (epsData(61,i,4), i=1,3) / 0.1051570000E+02, -0.3171540000E+01, 0.5640780000E+01 /
  data (epsData(62,i,4), i=1,3) / 0.1106980000E+02, -0.3413710000E+01, 0.5830680000E+01 /
  data (epsData(63,i,4), i=1,3) / 0.1175010000E+02, -0.3872010000E+01, 0.5596470000E+01 /
  data (epsData(64,i,4), i=1,3) / 0.1265620000E+02, -0.4130130000E+01, 0.4811370000E+01 /
  data (epsData(65,i,4), i=1,3) / 0.1383720000E+02, -0.3944480000E+01, 0.3863360000E+01 /
  data (epsData(66,i,4), i=1,3) / 0.1575560000E+02, -0.3391790000E+01, 0.2866950000E+01 /
  data (epsData(67,i,4), i=1,3) / 0.1794610000E+02, -0.2852950000E+01, 0.2163190000E+01 /
  data (epsData(68,i,4), i=1,3) / 0.2053900000E+02, -0.2363550000E+01, 0.1604030000E+01 /
  data (epsData(69,i,4), i=1,3) / 0.2343190000E+02, -0.1937900000E+01, 0.1178150000E+01 /
  data (epsData(70,i,4), i=1,3) / 0.2646600000E+02, -0.1586360000E+01, 0.8788610000E+00 /
  data (epsData(71,i,4), i=1,3) / 0.2977210000E+02, -0.1291670000E+01, 0.6653190000E+00 /
  data (epsData(72,i,4), i=1,3) / 0.3335040000E+02, -0.1054550000E+01, 0.5128970000E+00 /
  data (epsData(73,i,4), i=1,3) / 0.3740050000E+02, -0.8590670000E+00, 0.3947280000E+00 /
  data (epsData(74,i,4), i=1,3) / 0.4239650000E+02, -0.6861640000E+00, 0.2898600000E+00 /
  data (epsData(75,i,4), i=1,3) / 0.4825100000E+02, -0.5354430000E+00, 0.1990300000E+00 /
  data (epsData(76,i,4), i=1,3) / 0.5381340000E+02, -0.4210990000E+00, 0.1389390000E+00 /
  data (epsData(77,i,4), i=1,3) / 0.5947060000E+02, -0.3273990000E+00, 0.9896450000E-01 /
  data (epsData(78,i,4), i=1,3) / 0.6545720000E+02, -0.2485950000E+00, 0.7162590000E-01 /
  data (epsData(79,i,4), i=1,3) / 0.7177310000E+02, -0.1819600000E+00, 0.5290950000E-01 /
  data (epsData(80,i,4), i=1,3) / 0.7872810000E+02, -0.1216110000E+00, 0.3909000000E-01 /
  data (epsData(81,i,4), i=1,3) / 0.8669620000E+02, -0.6077920000E-01, 0.2812980000E-01 /
  data (epsData(82,i,4), i=1,3) / 0.9134400000E+02, -0.2533020000E-01, 0.2329120000E-01 /
  data (epsData(83,i,4), i=1,3) / 0.9440230000E+02, -0.1075180000E-02, 0.2114360000E-01 /
  data (epsData(84,i,4), i=1,3) / 0.9681910000E+02, 0.1990200000E-01, 0.1936410000E-01 /
  data (epsData(85,i,4), i=1,3) / 0.9891720000E+02, 0.4031840000E-01, 0.1804620000E-01 /
  data (epsData(86,i,4), i=1,3) / 0.1013960000E+03, 0.6823540000E-01, 0.1682650000E-01 /
  data (epsData(87,i,4), i=1,3) / 0.1035470000E+03, 0.9827600000E-01, 0.1589410000E-01 /
  data (epsData(88,i,4), i=1,3) / 0.1053580000E+03, 0.1311280000E+00, 0.1525210000E-01 /
  data (epsData(89,i,4), i=1,3) / 0.1068200000E+03, 0.1668710000E+00, 0.1501570000E-01 /
  data (epsData(90,i,4), i=1,3) / 0.1079230000E+03, 0.2039130000E+00, 0.1511370000E-01 /
  data (epsData(91,i,4), i=1,3) / 0.1086610000E+03, 0.2375900000E+00, 0.1542710000E-01 /
  data (epsData(92,i,4), i=1,3) / 0.1094030000E+03, 0.2847820000E+00, 0.1624490000E-01 /
  data (epsData(93,i,4), i=1,3) / 0.1097750000E+03, 0.3174030000E+00, 0.1706310000E-01 /
  data (epsData(94,i,4), i=1,3) / 0.1101470000E+03, 0.3609890000E+00, 0.1845580000E-01 /
  data (epsData(95,i,4), i=1,3) / 0.1105200000E+03, 0.4252420000E+00, 0.2122630000E-01 /
  data (epsData(96,i,4), i=1,3) / 0.1108940000E+03, 0.5370500000E+00, 0.2959320000E-01 /
  data (epsData(97,i,4), i=1,3) / 0.1112690000E+03, 0.7590740000E+00, 0.9411130000E-01 /
  data (epsData(98,i,4), i=1,3) / 0.1115050000E+03, 0.9198810000E+00, 0.3992280000E+00 /
  data (epsData(99,i,4), i=1,3) / 0.1115870000E+03, 0.9175520000E+00, 0.5912650000E+00 /
  data (epsData(100,i,4), i=1,3) / 0.1116650000E+03, 0.8297620000E+00, 0.7731680000E+00 /
  data (epsData(101,i,4), i=1,3) / 0.1116960000E+03, 0.7524140000E+00, 0.8464000000E+00 /
  data (epsData(102,i,4), i=1,3) / 0.1117480000E+03, 0.6216990000E+00, 0.8575110000E+00 /
  data (epsData(103,i,4), i=1,3) / 0.1119910000E+03, 0.2834200000E+00, 0.8653190000E+00 /
  data (epsData(104,i,4), i=1,3) / 0.1120440000E+03, 0.2042620000E+00, 0.8274000000E+00 /
  data (epsData(105,i,4), i=1,3) / 0.1121590000E+03, 0.1070230000E+00, 0.7289470000E+00 /
  data (epsData(106,i,4), i=1,3) / 0.1122510000E+03, 0.6744330000E-01, 0.6492410000E+00 /
  data (epsData(107,i,4), i=1,3) / 0.1123170000E+03, 0.6348540000E-01, 0.6003490000E+00 /
  data (epsData(108,i,4), i=1,3) / 0.1125670000E+03, 0.5198440000E-01, 0.4919710000E+00 /
  data (epsData(109,i,4), i=1,3) / 0.1126440000E+03, 0.6213250000E-01, 0.4692320000E+00 /
  data (epsData(110,i,4), i=1,3) / 0.1129370000E+03, 0.7717490000E-01, 0.4177990000E+00 /
  data (epsData(111,i,4), i=1,3) / 0.1133130000E+03, 0.9038260000E-01, 0.3897930000E+00 /
  data (epsData(112,i,4), i=1,3) / 0.1136770000E+03, 0.9444180000E-01, 0.3811750000E+00 /
  data (epsData(113,i,4), i=1,3) / 0.1140750000E+03, 0.8100500000E-01, 0.3802400000E+00 /
  data (epsData(114,i,4), i=1,3) / 0.1143790000E+03, 0.6857030000E-01, 0.3615800000E+00 /
  data (epsData(115,i,4), i=1,3) / 0.1153220000E+03, 0.6318860000E-01, 0.3375180000E+00 /
  data (epsData(116,i,4), i=1,3) / 0.1165630000E+03, 0.6056860000E-01, 0.3248610000E+00 /
  data (epsData(117,i,4), i=1,3) / 0.1179780000E+03, 0.5016200000E-01, 0.3313340000E+00 /
  data (epsData(118,i,4), i=1,3) / 0.1196720000E+03, 0.1862900000E-01, 0.3363940000E+00 /
  data (epsData(119,i,4), i=1,3) / 0.1216800000E+03, -0.1675810000E-01, 0.3240840000E+00 /
  data (epsData(120,i,4), i=1,3) / 0.1252420000E+03, -0.5363590000E-01, 0.2923680000E+00 /
  data (epsData(121,i,4), i=1,3) / 0.1294940000E+03, -0.7413920000E-01, 0.2607230000E+00 /
  data (epsData(122,i,4), i=1,3) / 0.1407010000E+03, -0.1021980000E+00, 0.2083720000E+00 /
  data (epsData(123,i,4), i=1,3) / 0.1519850000E+03, -0.1114500000E+00, 0.1703380000E+00 /
  data (epsData(124,i,4), i=1,3) / 0.1651090000E+03, -0.1157310000E+00, 0.1323880000E+00 /
  data (epsData(125,i,4), i=1,3) / 0.1806860000E+03, -0.1091930000E+00, 0.9906810000E-01 /
  data (epsData(126,i,4), i=1,3) / 0.2024170000E+03, -0.9633690000E-01, 0.6820500000E-01 /
  data (epsData(127,i,4), i=1,3) / 0.2263200000E+03, -0.8172440000E-01, 0.4629570000E-01 /
  data (epsData(128,i,4), i=1,3) / 0.2503900000E+03, -0.6879510000E-01, 0.3259870000E-01 /
  data (epsData(129,i,4), i=1,3) / 0.2800830000E+03, -0.5628550000E-01, 0.2196050000E-01 /
  data (epsData(130,i,4), i=1,3) / 0.3084880000E+03, -0.4685160000E-01, 0.1550670000E-01 /
  data (epsData(131,i,4), i=1,3) / 0.3391030000E+03, -0.3895230000E-01, 0.1101340000E-01 /
  data (epsData(132,i,4), i=1,3) / 0.3719280000E+03, -0.3240610000E-01, 0.7836100000E-02 /
  data (epsData(133,i,4), i=1,3) / 0.4075980000E+03, -0.2691110000E-01, 0.5581560000E-02 /
  data (epsData(134,i,4), i=1,3) / 0.4489360000E+03, -0.2209240000E-01, 0.3884130000E-02 /
  data (epsData(135,i,4), i=1,3) / 0.4965850000E+03, -0.1791050000E-01, 0.2659040000E-02 /
  data (epsData(136,i,4), i=1,3) / 0.5527870000E+03, -0.1439710000E-01, 0.1796140000E-02 /
  data (epsData(137,i,4), i=1,3) / 0.6045910000E+03, -0.1194430000E-01, 0.1279400000E-02 /
  data (epsData(138,i,4), i=1,3) / 0.6749190000E+03, -0.9511110000E-02, 0.8300020000E-03 /
  data (epsData(139,i,4), i=1,3) / 0.7546020000E+03, -0.7556950000E-02, 0.5328660000E-03 /
  data (epsData(140,i,4), i=1,3) / 0.8395540000E+03, -0.6063270000E-02, 0.3551280000E-03 /
  data (epsData(141,i,4), i=1,3) / 0.9297740000E+03, -0.4910550000E-02, 0.2387220000E-03 /
  data (epsData(142,i,4), i=1,3) / 0.1029730000E+04, -0.3978830000E-02, 0.1593000000E-03 /
  data (epsData(143,i,4), i=1,3) / 0.1143720000E+04, -0.3206970000E-02, 0.1047460000E-03 /
  data (epsData(144,i,4), i=1,3) / 0.1273160000E+04, -0.2574770000E-02, 0.6814760000E-04 /
  data (epsData(145,i,4), i=1,3) / 0.1418240000E+04, -0.2065470000E-02, 0.4405760000E-04 /
  data (epsData(146,i,4), i=1,3) / 0.1580320000E+04, -0.1657170000E-02, 0.2835200000E-04 /
  data (epsData(147,i,4), i=1,3) / 0.1762340000E+04, -0.1327230000E-02, 0.1778650000E-04 /
  data (epsData(148,i,4), i=1,3) / 0.1970870000E+04, -0.1057700000E-02, 0.1123710000E-04 /
  data (epsData(149,i,4), i=1,3) / 0.2205280000E+04, -0.8423960000E-03, 0.7073470000E-05 /
  data (epsData(150,i,4), i=1,3) / 0.2467900000E+04, -0.6710220000E-03, 0.4438090000E-05 /
  data (epsData(151,i,4), i=1,3) / 0.2763100000E+04, -0.5341990000E-03, 0.2773730000E-05 /
  data (epsData(152,i,4), i=1,3) / 0.3093820000E+04, -0.4253530000E-03, 0.1729000000E-05 /
  data (epsData(153,i,4), i=1,3) / 0.3465300000E+04, -0.3385490000E-03, 0.1074330000E-05 /
  data (epsData(154,i,4), i=1,3) / 0.3883400000E+04, -0.2692430000E-03, 0.6651660000E-06 /
  data (epsData(155,i,4), i=1,3) / 0.4352400000E+04, -0.2141230000E-03, 0.4107950000E-06 /
  data (epsData(156,i,4), i=1,3) / 0.4879390000E+04, -0.1702230000E-03, 0.2529070000E-06 /
  data (epsData(157,i,4), i=1,3) / 0.5469570000E+04, -0.1353740000E-03, 0.1555830000E-06 /
  data (epsData(158,i,4), i=1,3) / 0.6134190000E+04, -0.1075650000E-03, 0.9565400000E-07 /
  data (epsData(159,i,4), i=1,3) / 0.6880060000E+04, -0.8546560000E-04, 0.5849500000E-07 /
  data (epsData(160,i,4), i=1,3) / 0.7717630000E+04, -0.6789480000E-04, 0.3581390000E-07 /
  data (epsData(161,i,4), i=1,3) / 0.8655110000E+04, -0.5396580000E-04, 0.2173130000E-07 /
  data (epsData(162,i,4), i=1,3) / 0.9708370000E+04, -0.4288010000E-04, 0.1327570000E-07 /
  data (epsData(163,i,4), i=1,3) / 0.1089160000E+05, -0.3406170000E-04, 0.8064830000E-08 /
  data (epsData(164,i,4), i=1,3) / 0.1221660000E+05, -0.2706910000E-04, 0.4896630000E-08 /
  data (epsData(165,i,4), i=1,3) / 0.1370420000E+05, -0.2150820000E-04, 0.2975610000E-08 /
  data (epsData(166,i,4), i=1,3) / 0.1537410000E+05, -0.1708780000E-04, 0.1796750000E-08 /
  data (epsData(167,i,4), i=1,3) / 0.1724690000E+05, -0.1357690000E-04, 0.1106510000E-08 /
  data (epsData(168,i,4), i=1,3) / 0.1935060000E+05, -0.1078450000E-04, 0.6597490000E-09 /
  data (epsData(169,i,4), i=1,3) / 0.2171150000E+05, -0.8566060000E-05, 0.3995490000E-09 /
  data (epsData(170,i,4), i=1,3) / 0.2435810000E+05, -0.6805380000E-05, 0.2596010000E-09 /
  data (epsData(171,i,4), i=1,3) / 0.2732780000E+05, -0.5406460000E-05, 0.1595680000E-09 /
  data (epsData(172,i,4), i=1,3) / 0.3066210000E+05, -0.4294420000E-05, 0.1233050000E-09 /
  data (epsData(173,i,4), i=1,3) / 0.3440050000E+05, -0.3411650000E-05, 0.7236130000E-10 /
  data (epsData(174,i,4), i=1,3) / 0.3860030000E+05, -0.2709620000E-05, 0.5513630000E-10 /
  data (epsData(175,i,4), i=1,3) / 0.4331070000E+05, -0.2152230000E-05, 0.4530740000E-10 /
  data (epsData(176,i,4), i=1,3) / 0.4859290000E+05, -0.1709730000E-05, 0.1495760000E-10 /
  data (epsData(177,i,4), i=1,3) / 0.5451490000E+05, -0.1358450000E-05, 0.2066290000E-10 /
  data (epsData(178,i,4), i=1,3) / 0.6116530000E+05, -0.1079100000E-05, 0.2529330000E-10 /
  data (epsData(179,i,4), i=1,3) / 0.6862180000E+05, -0.8573130000E-06, 0.9877520000E-11 /
  data (epsData(180,i,4), i=1,3) / 0.7699770000E+05, -0.6809400000E-06, 0.3959610000E-11 /
  data (epsData(181,i,4), i=1,3) / 0.8639710000E+05, -0.5408320000E-06, 0.5863220000E-11 /
  data (epsData(182,i,4), i=1,3) / 0.9694550000E+05, -0.4295390000E-06, 0.8332650000E-11 /
  data (epsData(183,i,4), i=1,3) / 0.1000000000E+06, -0.4037610000E-06, 0.1709870000E-10 /
  data (epsData(1,i,5), i=1,3) / 0.2506580000E-02, -0.9191990000E+02, 0.3271380000E+03 /
  data (epsData(2,i,5), i=1,3) / 0.4693440000E-02, -0.7962870000E+02, 0.1154530000E+03 /
  data (epsData(3,i,5), i=1,3) / 0.7560590000E-02, -0.2823780000E+02, 0.5568140000E+02 /
  data (epsData(4,i,5), i=1,3) / 0.1110800000E-01, -0.5623920000E+01, 0.3220960000E+02 /
  data (epsData(5,i,5), i=1,3) / 0.1533570000E-01, 0.5048750000E+01, 0.2090890000E+02 /
  data (epsData(6,i,5), i=1,3) / 0.2024370000E-01, 0.1058180000E+02, 0.1467120000E+02 /
  data (epsData(7,i,5), i=1,3) / 0.2583200000E-01, 0.1368350000E+02, 0.1087840000E+02 /
  data (epsData(8,i,5), i=1,3) / 0.3210060000E-01, 0.1553510000E+02, 0.8401710000E+01 /
  data (epsData(9,i,5), i=1,3) / 0.3904940000E-01, 0.1669820000E+02, 0.6694230000E+01 /
  data (epsData(10,i,5), i=1,3) / 0.4667860000E-01, 0.1746050000E+02, 0.5465970000E+01 /
  data (epsData(11,i,5), i=1,3) / 0.5498800000E-01, 0.1797830000E+02, 0.4551900000E+01 /
  data (epsData(12,i,5), i=1,3) / 0.7364770000E-01, 0.1860320000E+02, 0.3305000000E+01 /
  data (epsData(13,i,5), i=1,3) / 0.1067390000E+00, 0.1906870000E+02, 0.2237730000E+01 /
  data (epsData(14,i,5), i=1,3) / 0.1912910000E+00, 0.1932790000E+02, 0.1276130000E+01 /
  data (epsData(15,i,5), i=1,3) / 0.4838160000E+00, 0.1946920000E+02, 0.5247750000E+00 /
  data (epsData(16,i,5), i=1,3) / 0.1606050000E+01, 0.2017140000E+02, 0.2072620000E+00 /
  data (epsData(17,i,5), i=1,3) / 0.2737710000E+01, 0.2196840000E+02, 0.1799700000E+00 /
  data (epsData(18,i,5), i=1,3) / 0.3519060000E+01, 0.2444510000E+02, 0.2113970000E+00 /
  data (epsData(19,i,5), i=1,3) / 0.4020090000E+01, 0.2731100000E+02, 0.2793260000E+00 /
  data (epsData(20,i,5), i=1,3) / 0.4321360000E+01, 0.3035980000E+02, 0.4292180000E+00 /
  data (epsData(21,i,5), i=1,3) / 0.4476080000E+01, 0.3289190000E+02, 0.7667480000E+00 /
  data (epsData(22,i,5), i=1,3) / 0.4554460000E+01, 0.3503840000E+02, 0.9200180000E+00 /
  data (epsData(23,i,5), i=1,3) / 0.4633520000E+01, 0.3764640000E+02, 0.2787390000E+01 /
  data (epsData(24,i,5), i=1,3) / 0.4712240000E+01, 0.3888570000E+02, 0.6630500000E+01 /
  data (epsData(25,i,5), i=1,3) / 0.4864630000E+01, 0.3792350000E+02, 0.1191760000E+02 /
  data (epsData(26,i,5), i=1,3) / 0.4947960000E+01, 0.3608420000E+02, 0.1353560000E+02 /
  data (epsData(27,i,5), i=1,3) / 0.5388790000E+01, 0.3265680000E+02, 0.1749800000E+02 /
  data (epsData(28,i,5), i=1,3) / 0.6181660000E+01, 0.2767960000E+02, 0.2470680000E+02 /
  data (epsData(29,i,5), i=1,3) / 0.6463300000E+01, 0.2429010000E+02, 0.2698700000E+02 /
  data (epsData(30,i,5), i=1,3) / 0.6716700000E+01, 0.2057250000E+02, 0.2779700000E+02 /
  data (epsData(31,i,5), i=1,3) / 0.6931670000E+01, 0.1846750000E+02, 0.2628340000E+02 /
  data (epsData(32,i,5), i=1,3) / 0.7157530000E+01, 0.1798750000E+02, 0.2618540000E+02 /
  data (epsData(33,i,5), i=1,3) / 0.8033730000E+01, 0.1284110000E+02, 0.2672810000E+02 /
  data (epsData(34,i,5), i=1,3) / 0.9930020000E+01, 0.4002030000E+01, 0.2551180000E+02 /
  data (epsData(35,i,5), i=1,3) / 0.1099130000E+02, -0.4692540000E+00, 0.2356310000E+02 /
  data (epsData(36,i,5), i=1,3) / 0.1195050000E+02, -0.3537090000E+01, 0.2084240000E+02 /
  data (epsData(37,i,5), i=1,3) / 0.1325850000E+02, -0.5884350000E+01, 0.1685260000E+02 /
  data (epsData(38,i,5), i=1,3) / 0.1448470000E+02, -0.6640760000E+01, 0.1360500000E+02 /
  data (epsData(39,i,5), i=1,3) / 0.1583170000E+02, -0.6669800000E+01, 0.1085280000E+02 /
  data (epsData(40,i,5), i=1,3) / 0.1749430000E+02, -0.6248710000E+01, 0.8465780000E+01 /
  data (epsData(41,i,5), i=1,3) / 0.1979910000E+02, -0.5625660000E+01, 0.6306860000E+01 /
  data (epsData(42,i,5), i=1,3) / 0.2274850000E+02, -0.4883750000E+01, 0.4408670000E+01 /
  data (epsData(43,i,5), i=1,3) / 0.2575020000E+02, -0.4140930000E+01, 0.3137460000E+01 /
  data (epsData(44,i,5), i=1,3) / 0.2874350000E+02, -0.3503890000E+01, 0.2322230000E+01 /
  data (epsData(45,i,5), i=1,3) / 0.3222480000E+02, -0.2925120000E+01, 0.1695420000E+01 /
  data (epsData(46,i,5), i=1,3) / 0.3606600000E+02, -0.2426530000E+01, 0.1228570000E+01 /
  data (epsData(47,i,5), i=1,3) / 0.4026710000E+02, -0.1998960000E+01, 0.8895880000E+00 /
  data (epsData(48,i,5), i=1,3) / 0.4482810000E+02, -0.1642940000E+01, 0.6492180000E+00 /
  data (epsData(49,i,5), i=1,3) / 0.4997100000E+02, -0.1340890000E+01, 0.4713670000E+00 /
  data (epsData(50,i,5), i=1,3) / 0.5574490000E+02, -0.1088780000E+01, 0.3391580000E+00 /
  data (epsData(51,i,5), i=1,3) / 0.6194400000E+02, -0.8850340000E+00, 0.2449890000E+00 /
  data (epsData(52,i,5), i=1,3) / 0.6856830000E+02, -0.7214570000E+00, 0.1796510000E+00 /
  data (epsData(53,i,5), i=1,3) / 0.7590870000E+02, -0.5862410000E+00, 0.1315220000E+00 /
  data (epsData(54,i,5), i=1,3) / 0.8401830000E+02, -0.4744810000E+00, 0.9575120000E-01 /
  data (epsData(55,i,5), i=1,3) / 0.9328090000E+02, -0.3788440000E+00, 0.6852210000E-01 /
  data (epsData(56,i,5), i=1,3) / 0.1034650000E+03, -0.3006120000E+00, 0.4926570000E-01 /
  data (epsData(57,i,5), i=1,3) / 0.1142610000E+03, -0.2382580000E+00, 0.3571620000E-01 /
  data (epsData(58,i,5), i=1,3) / 0.1264530000E+03, -0.1844830000E+00, 0.2565600000E-01 /
  data (epsData(59,i,5), i=1,3) / 0.1401690000E+03, -0.1375100000E+00, 0.1815820000E-01 /
  data (epsData(60,i,5), i=1,3) / 0.1568710000E+03, -0.9057060000E-01, 0.1264040000E-01 /
  data (epsData(61,i,5), i=1,3) / 0.1695000000E+03, -0.5576360000E-01, 0.9764100000E-02 /
  data (epsData(62,i,5), i=1,3) / 0.1750760000E+03, -0.3735860000E-01, 0.8765840000E-02 /
  data (epsData(63,i,5), i=1,3) / 0.1783730000E+03, -0.2399220000E-01, 0.8323870000E-02 /
  data (epsData(64,i,5), i=1,3) / 0.1802730000E+03, -0.1468720000E-01, 0.8096730000E-02 /
  data (epsData(65,i,5), i=1,3) / 0.1817050000E+03, -0.6392820000E-02, 0.7947860000E-02 /
  data (epsData(66,i,5), i=1,3) / 0.1826620000E+03, 0.6811160000E-04, 0.7864910000E-02 /
  data (epsData(67,i,5), i=1,3) / 0.1831420000E+03, 0.3677750000E-02, 0.7830910000E-02 /
  data (epsData(68,i,5), i=1,3) / 0.1836230000E+03, 0.7607200000E-02, 0.7804020000E-02 /
  data (epsData(69,i,5), i=1,3) / 0.1841040000E+03, 0.1192790000E-01, 0.7785460000E-02 /
  data (epsData(70,i,5), i=1,3) / 0.1845860000E+03, 0.1673900000E-01, 0.7778740000E-02 /
  data (epsData(71,i,5), i=1,3) / 0.1850690000E+03, 0.2218120000E-01, 0.7791480000E-02 /
  data (epsData(72,i,5), i=1,3) / 0.1855520000E+03, 0.2847000000E-01, 0.7830460000E-02 /
  data (epsData(73,i,5), i=1,3) / 0.1860360000E+03, 0.3596240000E-01, 0.7915740000E-02 /
  data (epsData(74,i,5), i=1,3) / 0.1865210000E+03, 0.4532690000E-01, 0.8090510000E-02 /
  data (epsData(75,i,5), i=1,3) / 0.1870070000E+03, 0.5818570000E-01, 0.8449810000E-02 /
  data (epsData(76,i,5), i=1,3) / 0.1874930000E+03, 0.7893050000E-01, 0.9708680000E-02 /
  data (epsData(77,i,5), i=1,3) / 0.1879790000E+03, 0.1081480000E+00, 0.3043660000E-01 /
  data (epsData(78,i,5), i=1,3) / 0.1880980000E+03, 0.1131890000E+00, 0.4712350000E-01 /
  data (epsData(79,i,5), i=1,3) / 0.1881960000E+03, 0.1101770000E+00, 0.6044140000E-01 /
  data (epsData(80,i,5), i=1,3) / 0.1883700000E+03, 0.1026710000E+00, 0.6919070000E-01 /
  data (epsData(81,i,5), i=1,3) / 0.1887510000E+03, 0.9336100000E-01, 0.8381540000E-01 /
  data (epsData(82,i,5), i=1,3) / 0.1897150000E+03, 0.7347810000E-01, 0.1112080000E+00 /
  data (epsData(83,i,5), i=1,3) / 0.1900170000E+03, 0.6371390000E-01, 0.1164380000E+00 /
  data (epsData(84,i,5), i=1,3) / 0.1902870000E+03, 0.5816240000E-01, 0.1145930000E+00 /
  data (epsData(85,i,5), i=1,3) / 0.1935730000E+03, 0.2101080000E-01, 0.1418870000E+00 /
  data (epsData(86,i,5), i=1,3) / 0.1949560000E+03, 0.1004300000E-02, 0.1457910000E+00 /
  data (epsData(87,i,5), i=1,3) / 0.1964910000E+03, -0.1561060000E-01, 0.1422940000E+00 /
  data (epsData(88,i,5), i=1,3) / 0.1993910000E+03, -0.3358620000E-01, 0.1338670000E+00 /
  data (epsData(89,i,5), i=1,3) / 0.2053730000E+03, -0.5386890000E-01, 0.1218520000E+00 /
  data (epsData(90,i,5), i=1,3) / 0.2113490000E+03, -0.6590150000E-01, 0.1079210000E+00 /
  data (epsData(91,i,5), i=1,3) / 0.2186130000E+03, -0.6963360000E-01, 0.9519960000E-01 /
  data (epsData(92,i,5), i=1,3) / 0.2397360000E+03, -0.7426920000E-01, 0.7284500000E-01 /
  data (epsData(93,i,5), i=1,3) / 0.2631380000E+03, -0.7109630000E-01, 0.5477760000E-01 /
  data (epsData(94,i,5), i=1,3) / 0.2939090000E+03, -0.6360220000E-01, 0.3823210000E-01 /
  data (epsData(95,i,5), i=1,3) / 0.3303890000E+03, -0.5382460000E-01, 0.2580640000E-01 /
  data (epsData(96,i,5), i=1,3) / 0.3682740000E+03, -0.4510040000E-01, 0.1768060000E-01 /
  data (epsData(97,i,5), i=1,3) / 0.4095220000E+03, -0.3727060000E-01, 0.1215060000E-01 /
  data (epsData(98,i,5), i=1,3) / 0.4532120000E+03, -0.3085060000E-01, 0.8439390000E-02 /
  data (epsData(99,i,5), i=1,3) / 0.5014830000E+03, -0.2533150000E-01, 0.5832010000E-02 /
  data (epsData(100,i,5), i=1,3) / 0.5552010000E+03, -0.2071400000E-01, 0.4014930000E-02 /
  data (epsData(101,i,5), i=1,3) / 0.6101280000E+03, -0.1714580000E-01, 0.2866460000E-02 /
  data (epsData(102,i,5), i=1,3) / 0.6729340000E+03, -0.1405260000E-01, 0.1977760000E-02 /
  data (epsData(103,i,5), i=1,3) / 0.7522950000E+03, -0.1119880000E-01, 0.1293230000E-02 /
  data (epsData(104,i,5), i=1,3) / 0.8414330000E+03, -0.8906330000E-02, 0.8417600000E-03 /
  data (epsData(105,i,5), i=1,3) / 0.9401370000E+03, -0.7094500000E-02, 0.5487530000E-03 /
  data (epsData(106,i,5), i=1,3) / 0.1050110000E+04, -0.5653220000E-02, 0.3576900000E-03 /
  data (epsData(107,i,5), i=1,3) / 0.1173360000E+04, -0.4501610000E-02, 0.2325530000E-03 /
  data (epsData(108,i,5), i=1,3) / 0.1311020000E+04, -0.3586100000E-02, 0.1509120000E-03 /
  data (epsData(109,i,5), i=1,3) / 0.1462930000E+04, -0.2865670000E-02, 0.9816150000E-04 /
  data (epsData(110,i,5), i=1,3) / 0.1631480000E+04, -0.2293880000E-02, 0.6366490000E-04 /
  data (epsData(111,i,5), i=1,3) / 0.1819420000E+04, -0.1836550000E-02, 0.4057980000E-04 /
  data (epsData(112,i,5), i=1,3) / 0.2031500000E+04, -0.1467380000E-02, 0.2602050000E-04 /
  data (epsData(113,i,5), i=1,3) / 0.2269980000E+04, -0.1171230000E-02, 0.1660040000E-04 /
  data (epsData(114,i,5), i=1,3) / 0.2539030000E+04, -0.9333380000E-03, 0.1051520000E-04 /
  data (epsData(115,i,5), i=1,3) / 0.2841500000E+04, -0.7432600000E-03, 0.6634190000E-05 /
  data (epsData(116,i,5), i=1,3) / 0.3180400000E+04, -0.5919580000E-03, 0.4174370000E-05 /
  data (epsData(117,i,5), i=1,3) / 0.3561050000E+04, -0.4712560000E-03, 0.2616440000E-05 /
  data (epsData(118,i,5), i=1,3) / 0.3989400000E+04, -0.3748690000E-03, 0.1632740000E-05 /
  data (epsData(119,i,5), i=1,3) / 0.4469830000E+04, -0.2981980000E-03, 0.1016120000E-05 /
  data (epsData(120,i,5), i=1,3) / 0.5009520000E+04, -0.2371280000E-03, 0.6304770000E-06 /
  data (epsData(121,i,5), i=1,3) / 0.5613780000E+04, -0.1886400000E-03, 0.3906000000E-06 /
  data (epsData(122,i,5), i=1,3) / 0.6293980000E+04, -0.1499440000E-03, 0.2421400000E-06 /
  data (epsData(123,i,5), i=1,3) / 0.7057060000E+04, -0.1191890000E-03, 0.1494250000E-06 /
  data (epsData(124,i,5), i=1,3) / 0.7913610000E+04, -0.9473010000E-04, 0.9158560000E-07 /
  data (epsData(125,i,5), i=1,3) / 0.8875380000E+04, -0.7527590000E-04, 0.5616020000E-07 /
  data (epsData(126,i,5), i=1,3) / 0.9955420000E+04, -0.5980540000E-04, 0.3426130000E-07 /
  data (epsData(127,i,5), i=1,3) / 0.1116820000E+05, -0.4750680000E-04, 0.2090620000E-07 /
  data (epsData(128,i,5), i=1,3) / 0.1252550000E+05, -0.3775850000E-04, 0.1277380000E-07 /
  data (epsData(129,i,5), i=1,3) / 0.1404860000E+05, -0.3000850000E-04, 0.7785020000E-08 /
  data (epsData(130,i,5), i=1,3) / 0.1575750000E+05, -0.2384850000E-04, 0.4741140000E-08 /
  data (epsData(131,i,5), i=1,3) / 0.1767770000E+05, -0.1894610000E-04, 0.2879650000E-08 /
  data (epsData(132,i,5), i=1,3) / 0.1983350000E+05, -0.1504950000E-04, 0.1750420000E-08 /
  data (epsData(133,i,5), i=1,3) / 0.2225130000E+05, -0.1195550000E-04, 0.1060070000E-08 /
  data (epsData(134,i,5), i=1,3) / 0.2496600000E+05, -0.9496170000E-05, 0.6508190000E-09 /
  data (epsData(135,i,5), i=1,3) / 0.2801010000E+05, -0.7543760000E-05, 0.3911960000E-09 /
  data (epsData(136,i,5), i=1,3) / 0.3142590000E+05, -0.5992660000E-05, 0.2483150000E-09 /
  data (epsData(137,i,5), i=1,3) / 0.3526020000E+05, -0.4760030000E-05, 0.1490830000E-09 /
  data (epsData(138,i,5), i=1,3) / 0.3956490000E+05, -0.3780460000E-05, 0.1057290000E-09 /
  data (epsData(139,i,5), i=1,3) / 0.4439750000E+05, -0.3002170000E-05, 0.6156990000E-10 /
  data (epsData(140,i,5), i=1,3) / 0.4981340000E+05, -0.2384800000E-05, 0.4259570000E-10 /
  data (epsData(141,i,5), i=1,3) / 0.5588990000E+05, -0.1894400000E-05, 0.3406710000E-10 /
  data (epsData(142,i,5), i=1,3) / 0.6270790000E+05, -0.1504830000E-05, 0.2661150000E-10 /
  data (epsData(143,i,5), i=1,3) / 0.7036800000E+05, -0.1195030000E-05, 0.1234920000E-10 /
  data (epsData(144,i,5), i=1,3) / 0.7895620000E+05, -0.9491870000E-06, 0.1727680000E-10 /
  data (epsData(145,i,5), i=1,3) / 0.8859820000E+05, -0.7538340000E-06, 0.8128960000E-11 /
  data (epsData(146,i,5), i=1,3) / 0.9941120000E+05, -0.5987630000E-06, 0.9431070000E-11 /
  data (epsData(147,i,5), i=1,3) / 0.1000000000E+06, -0.5918340000E-06, 0.1009700000E-10 /
  data (epsData(1,i,6), i=1,3) / 0.2506580000E-02, -0.4589820000E+01, 0.2420570000E+03 /
  data (epsData(2,i,6), i=1,3) / 0.4693440000E-02, -0.2129550000E+02, 0.9827940000E+02 /
  data (epsData(3,i,6), i=1,3) / 0.7560590000E-02, 0.2250110000E+01, 0.5264660000E+02 /
  data (epsData(4,i,6), i=1,3) / 0.1110800000E-01, 0.1324180000E+02, 0.3285770000E+02 /
  data (epsData(5,i,6), i=1,3) / 0.1533570000E-01, 0.1854850000E+02, 0.2253310000E+02 /
  data (epsData(6,i,6), i=1,3) / 0.2024370000E-01, 0.2133500000E+02, 0.1645890000E+02 /
  data (epsData(7,i,6), i=1,3) / 0.2583200000E-01, 0.2291020000E+02, 0.1257460000E+02 /
  data (epsData(8,i,6), i=1,3) / 0.3210060000E-01, 0.2385610000E+02, 0.9934770000E+01 /
  data (epsData(9,i,6), i=1,3) / 0.3904940000E-01, 0.2445290000E+02, 0.8055790000E+01 /
  data (epsData(10,i,6), i=1,3) / 0.5498800000E-01, 0.2511220000E+02, 0.5615140000E+01 /
  data (epsData(11,i,6), i=1,3) / 0.8399800000E-01, 0.2553270000E+02, 0.3617350000E+01 /
  data (epsData(12,i,6), i=1,3) / 0.1459540000E+00, 0.2577190000E+02, 0.2054730000E+01 /
  data (epsData(13,i,6), i=1,3) / 0.3208860000E+00, 0.2588980000E+02, 0.9549210000E+00 /
  data (epsData(14,i,6), i=1,3) / 0.1295500000E+01, 0.2614890000E+02, 0.2719420000E+00 /
  data (epsData(15,i,6), i=1,3) / 0.3314540000E+01, 0.2787980000E+02, 0.1561930000E+00 /
  data (epsData(16,i,6), i=1,3) / 0.4874780000E+01, 0.3101230000E+02, 0.1639860000E+00 /
  data (epsData(17,i,6), i=1,3) / 0.5901040000E+01, 0.3502030000E+02, 0.2116820000E+00 /
  data (epsData(18,i,6), i=1,3) / 0.6450900000E+01, 0.3883770000E+02, 0.2933310000E+00 /
  data (epsData(19,i,6), i=1,3) / 0.6735020000E+01, 0.4205060000E+02, 0.4066150000E+00 /
  data (epsData(20,i,6), i=1,3) / 0.6927830000E+01, 0.4548090000E+02, 0.7784530000E+00 /
  data (epsData(21,i,6), i=1,3) / 0.7025260000E+01, 0.4821780000E+02, 0.1024870000E+01 /
  data (epsData(22,i,6), i=1,3) / 0.7123360000E+01, 0.5206390000E+02, 0.2763190000E+01 /
  data (epsData(23,i,6), i=1,3) / 0.7212240000E+01, 0.5396230000E+02, 0.8276820000E+01 /
  data (epsData(24,i,6), i=1,3) / 0.7379940000E+01, 0.5230970000E+02, 0.1534130000E+02 /
  data (epsData(25,i,6), i=1,3) / 0.7466670000E+01, 0.4987630000E+02, 0.1676700000E+02 /
  data (epsData(26,i,6), i=1,3) / 0.7919740000E+01, 0.4612870000E+02, 0.2061270000E+02 /
  data (epsData(27,i,6), i=1,3) / 0.8593220000E+01, 0.4283430000E+02, 0.2722930000E+02 /
  data (epsData(28,i,6), i=1,3) / 0.8914660000E+01, 0.3937970000E+02, 0.3007920000E+02 /
  data (epsData(29,i,6), i=1,3) / 0.9216700000E+01, 0.3750800000E+02, 0.2982140000E+02 /
  data (epsData(30,i,6), i=1,3) / 0.9774540000E+01, 0.3448200000E+02, 0.3317150000E+02 /
  data (epsData(31,i,6), i=1,3) / 0.1193610000E+02, 0.2124570000E+02, 0.3939650000E+02 /
  data (epsData(32,i,6), i=1,3) / 0.1339910000E+02, 0.1148030000E+02, 0.4020020000E+02 /
  data (epsData(33,i,6), i=1,3) / 0.1465050000E+02, 0.4105910000E+01, 0.3811810000E+02 /
  data (epsData(34,i,6), i=1,3) / 0.1590760000E+02, -0.1534720000E+01, 0.3458310000E+02 /
  data (epsData(35,i,6), i=1,3) / 0.1749810000E+02, -0.6219610000E+01, 0.2936750000E+02 /
  data (epsData(36,i,6), i=1,3) / 0.1910510000E+02, -0.8635070000E+01, 0.2434700000E+02 /
  data (epsData(37,i,6), i=1,3) / 0.2094190000E+02, -0.9726520000E+01, 0.1969760000E+02 /
  data (epsData(38,i,6), i=1,3) / 0.2340010000E+02, -0.1005330000E+02, 0.1500510000E+02 /
  data (epsData(39,i,6), i=1,3) / 0.2599320000E+02, -0.9610470000E+01, 0.1123940000E+02 /
  data (epsData(40,i,6), i=1,3) / 0.2894330000E+02, -0.8599120000E+01, 0.8256810000E+01 /
  data (epsData(41,i,6), i=1,3) / 0.3235560000E+02, -0.7417630000E+01, 0.6046490000E+01 /
  data (epsData(42,i,6), i=1,3) / 0.3636750000E+02, -0.6296000000E+01, 0.4359690000E+01 /
  data (epsData(43,i,6), i=1,3) / 0.4091350000E+02, -0.5273730000E+01, 0.3072940000E+01 /
  data (epsData(44,i,6), i=1,3) / 0.4585890000E+02, -0.4360800000E+01, 0.2145880000E+01 /
  data (epsData(45,i,6), i=1,3) / 0.5088750000E+02, -0.3618010000E+01, 0.1542070000E+01 /
  data (epsData(46,i,6), i=1,3) / 0.5665800000E+02, -0.2966580000E+01, 0.1094380000E+01 /
  data (epsData(47,i,6), i=1,3) / 0.6322960000E+02, -0.2407070000E+01, 0.7661140000E+00 /
  data (epsData(48,i,6), i=1,3) / 0.7041340000E+02, -0.1952150000E+01, 0.5391310000E+00 /
  data (epsData(49,i,6), i=1,3) / 0.7767060000E+02, -0.1608750000E+01, 0.3899070000E+00 /
  data (epsData(50,i,6), i=1,3) / 0.8574950000E+02, -0.1320120000E+01, 0.2796870000E+00 /
  data (epsData(51,i,6), i=1,3) / 0.9501830000E+02, -0.1072270000E+01, 0.1969840000E+00 /
  data (epsData(52,i,6), i=1,3) / 0.1056030000E+03, -0.8637660000E+00, 0.1378710000E+00 /
  data (epsData(53,i,6), i=1,3) / 0.1169290000E+03, -0.7001820000E+00, 0.9737440000E-01 /
  data (epsData(54,i,6), i=1,3) / 0.1297510000E+03, -0.5638030000E+00, 0.6813830000E-01 /
  data (epsData(55,i,6), i=1,3) / 0.1442120000E+03, -0.4514210000E+00, 0.4709800000E-01 /
  data (epsData(56,i,6), i=1,3) / 0.1600340000E+03, -0.3619090000E+00, 0.3328940000E-01 /
  data (epsData(57,i,6), i=1,3) / 0.1772930000E+03, -0.2898710000E+00, 0.2312210000E-01 /
  data (epsData(58,i,6), i=1,3) / 0.1975210000E+03, -0.2277170000E+00, 0.1573890000E-01 /
  data (epsData(59,i,6), i=1,3) / 0.2200380000E+03, -0.1765030000E+00, 0.1067890000E-01 /
  data (epsData(60,i,6), i=1,3) / 0.2466720000E+03, -0.1297320000E+00, 0.7118010000E-02 /
  data (epsData(61,i,6), i=1,3) / 0.2697810000E+03, -0.9116930000E-01, 0.5214630000E-02 /
  data (epsData(62,i,6), i=1,3) / 0.2767610000E+03, -0.7513060000E-01, 0.4814770000E-02 /
  data (epsData(63,i,6), i=1,3) / 0.2802870000E+03, -0.6287700000E-01, 0.4693370000E-02 /
  data (epsData(64,i,6), i=1,3) / 0.2820600000E+03, -0.5332820000E-01, 0.4731360000E-02 /
  data (epsData(65,i,6), i=1,3) / 0.2826520000E+03, -0.4882660000E-01, 0.4803450000E-02 /
  data (epsData(66,i,6), i=1,3) / 0.2832450000E+03, -0.4277360000E-01, 0.4979770000E-02 /
  data (epsData(67,i,6), i=1,3) / 0.2838390000E+03, -0.3323140000E-01, 0.5792960000E-02 /
  data (epsData(68,i,6), i=1,3) / 0.2842090000E+03, -0.2508540000E-01, 0.1120110000E-01 /
  data (epsData(69,i,6), i=1,3) / 0.2843360000E+03, -0.2292830000E-01, 0.1570580000E-01 /
  data (epsData(70,i,6), i=1,3) / 0.2844130000E+03, -0.2293040000E-01, 0.1822430000E-01 /
  data (epsData(71,i,6), i=1,3) / 0.2849200000E+03, -0.2269630000E-01, 0.2828000000E-01 /
  data (epsData(72,i,6), i=1,3) / 0.2854660000E+03, -0.2452570000E-01, 0.3785100000E-01 /
  data (epsData(73,i,6), i=1,3) / 0.2858670000E+03, -0.2818190000E-01, 0.4280690000E-01 /
  data (epsData(74,i,6), i=1,3) / 0.2862170000E+03, -0.2989130000E-01, 0.4424000000E-01 /
  data (epsData(75,i,6), i=1,3) / 0.2869550000E+03, -0.3269670000E-01, 0.5121890000E-01 /
  data (epsData(76,i,6), i=1,3) / 0.2885430000E+03, -0.4186920000E-01, 0.6188000000E-01 /
  data (epsData(77,i,6), i=1,3) / 0.2902170000E+03, -0.5320200000E-01, 0.6688500000E-01 /
  data (epsData(78,i,6), i=1,3) / 0.2924780000E+03, -0.6377720000E-01, 0.6673730000E-01 /
  data (epsData(79,i,6), i=1,3) / 0.2976700000E+03, -0.7528170000E-01, 0.6413130000E-01 /
  data (epsData(80,i,6), i=1,3) / 0.3031450000E+03, -0.8327250000E-01, 0.5968210000E-01 /
  data (epsData(81,i,6), i=1,3) / 0.3106770000E+03, -0.8532680000E-01, 0.5315420000E-01 /
  data (epsData(82,i,6), i=1,3) / 0.3302630000E+03, -0.8576650000E-01, 0.4347470000E-01 /
  data (epsData(83,i,6), i=1,3) / 0.3668720000E+03, -0.7680440000E-01, 0.3096910000E-01 /
  data (epsData(84,i,6), i=1,3) / 0.4192050000E+03, -0.6332720000E-01, 0.2003130000E-01 /
  data (epsData(85,i,6), i=1,3) / 0.4734990000E+03, -0.5163810000E-01, 0.1321960000E-01 /
  data (epsData(86,i,6), i=1,3) / 0.5299590000E+03, -0.4213680000E-01, 0.8931420000E-02 /
  data (epsData(87,i,6), i=1,3) / 0.5898950000E+03, -0.3443870000E-01, 0.6102550000E-02 /
  data (epsData(88,i,6), i=1,3) / 0.6570780000E+03, -0.2794720000E-01, 0.4133260000E-02 /
  data (epsData(89,i,6), i=1,3) / 0.7317080000E+03, -0.2263250000E-01, 0.2802840000E-02 /
  data (epsData(90,i,6), i=1,3) / 0.8105990000E+03, -0.1845510000E-01, 0.1924060000E-02 /
  data (epsData(91,i,6), i=1,3) / 0.8977170000E+03, -0.1503390000E-01, 0.1315400000E-02 /
  data (epsData(92,i,6), i=1,3) / 0.9954330000E+03, -0.1220410000E-01, 0.8910210000E-03 /
  data (epsData(93,i,6), i=1,3) / 0.1104590000E+04, -0.9885940000E-02, 0.6000770000E-03 /
  data (epsData(94,i,6), i=1,3) / 0.1227180000E+04, -0.7985610000E-02, 0.4008080000E-03 /
  data (epsData(95,i,6), i=1,3) / 0.1366790000E+04, -0.6416880000E-02, 0.2637370000E-03 /
  data (epsData(96,i,6), i=1,3) / 0.1526420000E+04, -0.5126770000E-02, 0.1709490000E-03 /
  data (epsData(97,i,6), i=1,3) / 0.1706740000E+04, -0.4087280000E-02, 0.1110910000E-03 /
  data (epsData(98,i,6), i=1,3) / 0.1906690000E+04, -0.3265240000E-02, 0.7144290000E-04 /
  data (epsData(99,i,6), i=1,3) / 0.2130870000E+04, -0.2606860000E-02, 0.4595900000E-04 /
  data (epsData(100,i,6), i=1,3) / 0.2383050000E+04, -0.2078930000E-02, 0.2939680000E-04 /
  data (epsData(101,i,6), i=1,3) / 0.2665740000E+04, -0.1657520000E-02, 0.1873120000E-04 /
  data (epsData(102,i,6), i=1,3) / 0.2983500000E+04, -0.1320510000E-02, 0.1188590000E-04 /
  data (epsData(103,i,6), i=1,3) / 0.3339480000E+04, -0.1052080000E-02, 0.7519150000E-05 /
  data (epsData(104,i,6), i=1,3) / 0.3739150000E+04, -0.8378700000E-03, 0.4740520000E-05 /
  data (epsData(105,i,6), i=1,3) / 0.4188610000E+04, -0.6667890000E-03, 0.2975350000E-05 /
  data (epsData(106,i,6), i=1,3) / 0.4692440000E+04, -0.5306680000E-03, 0.1862760000E-05 /
  data (epsData(107,i,6), i=1,3) / 0.5257990000E+04, -0.4222270000E-03, 0.1162590000E-05 /
  data (epsData(108,i,6), i=1,3) / 0.5893510000E+04, -0.3357900000E-03, 0.7232990000E-06 /
  data (epsData(109,i,6), i=1,3) / 0.6608220000E+04, -0.2668930000E-03, 0.4513630000E-06 /
  data (epsData(110,i,6), i=1,3) / 0.7409300000E+04, -0.2121740000E-03, 0.2775700000E-06 /
  data (epsData(111,i,6), i=1,3) / 0.8307600000E+04, -0.1686840000E-03, 0.1721110000E-06 /
  data (epsData(112,i,6), i=1,3) / 0.9318720000E+04, -0.1340080000E-03, 0.1054990000E-06 /
  data (epsData(113,i,6), i=1,3) / 0.1045290000E+05, -0.1064670000E-03, 0.6481780000E-07 /
  data (epsData(114,i,6), i=1,3) / 0.1172510000E+05, -0.8459250000E-04, 0.3971200000E-07 /
  data (epsData(115,i,6), i=1,3) / 0.1315160000E+05, -0.6722090000E-04, 0.2434110000E-07 /
  data (epsData(116,i,6), i=1,3) / 0.1475510000E+05, -0.5339400000E-04, 0.1485080000E-07 /
  data (epsData(117,i,6), i=1,3) / 0.1655190000E+05, -0.4242420000E-04, 0.9063230000E-08 /
  data (epsData(118,i,6), i=1,3) / 0.1856830000E+05, -0.3370590000E-04, 0.5516880000E-08 /
  data (epsData(119,i,6), i=1,3) / 0.2082920000E+05, -0.2678290000E-04, 0.3354530000E-08 /
  data (epsData(120,i,6), i=1,3) / 0.2336740000E+05, -0.2127870000E-04, 0.2040720000E-08 /
  data (epsData(121,i,6), i=1,3) / 0.2621350000E+05, -0.1690760000E-04, 0.1238170000E-08 /
  data (epsData(122,i,6), i=1,3) / 0.2941370000E+05, -0.1342790000E-04, 0.7595340000E-09 /
  data (epsData(123,i,6), i=1,3) / 0.3299990000E+05, -0.1066750000E-04, 0.4704780000E-09 /
  data (epsData(124,i,6), i=1,3) / 0.3702720000E+05, -0.8472840000E-05, 0.2774470000E-09 /
  data (epsData(125,i,6), i=1,3) / 0.4154260000E+05, -0.6730850000E-05, 0.1874980000E-09 /
  data (epsData(126,i,6), i=1,3) / 0.4661300000E+05, -0.5346040000E-05, 0.1051290000E-09 /
  data (epsData(127,i,6), i=1,3) / 0.5230480000E+05, -0.4245740000E-05, 0.6510310000E-10 /
  data (epsData(128,i,6), i=1,3) / 0.5868930000E+05, -0.3372190000E-05, 0.5766560000E-10 /
  data (epsData(129,i,6), i=1,3) / 0.6585470000E+05, -0.2678260000E-05, 0.3836990000E-10 /
  data (epsData(130,i,6), i=1,3) / 0.7389410000E+05, -0.2127160000E-05, 0.2725950000E-10 /
  data (epsData(131,i,6), i=1,3) / 0.8291660000E+05, -0.1689420000E-05, 0.2576610000E-10 /
  data (epsData(132,i,6), i=1,3) / 0.9304350000E+05, -0.1341670000E-05, 0.8602730000E-11 /
  data (epsData(133,i,6), i=1,3) / 0.1000000000E+06, -0.1161610000E-05, 0.8199520000E-11 /
  data (epsData(1,i,7), i=1,3) / 0.2506580000E-02, -0.7982590000E+02, 0.1762020000E+03 /
  data (epsData(2,i,7), i=1,3) / 0.4693440000E-02, -0.6086670000E+02, 0.5603850000E+02 /
  data (epsData(3,i,7), i=1,3) / 0.7560590000E-02, -0.2626170000E+02, 0.2451660000E+02 /
  data (epsData(4,i,7), i=1,3) / 0.1110800000E-01, -0.1133600000E+02, 0.1303240000E+02 /
  data (epsData(5,i,7), i=1,3) / 0.1533570000E-01, -0.4348920000E+01, 0.7884390000E+01 /
  data (epsData(6,i,7), i=1,3) / 0.2024370000E-01, -0.7436670000E+00, 0.5222310000E+01 /
  data (epsData(7,i,7), i=1,3) / 0.2583200000E-01, 0.1270840000E+01, 0.3694980000E+01 /
  data (epsData(8,i,7), i=1,3) / 0.3210060000E-01, 0.2470380000E+01, 0.2747100000E+01 /
  data (epsData(9,i,7), i=1,3) / 0.3904940000E-01, 0.3222290000E+01, 0.2121820000E+01 /
  data (epsData(10,i,7), i=1,3) / 0.4667860000E-01, 0.3713900000E+01, 0.1688850000E+01 /
  data (epsData(11,i,7), i=1,3) / 0.5498800000E-01, 0.4046840000E+01, 0.1377060000E+01 /
  data (epsData(12,i,7), i=1,3) / 0.6397770000E-01, 0.4279140000E+01, 0.1145180000E+01 /
  data (epsData(13,i,7), i=1,3) / 0.8399800000E-01, 0.4567100000E+01, 0.8296130000E+00 /
  data (epsData(14,i,7), i=1,3) / 0.1191310000E+00, 0.4779920000E+01, 0.5566290000E+00 /
  data (epsData(15,i,7), i=1,3) / 0.1912910000E+00, 0.4919040000E+01, 0.3307450000E+00 /
  data (epsData(16,i,7), i=1,3) / 0.3866320000E+00, 0.4993600000E+01, 0.1570740000E+00 /
  data (epsData(17,i,7), i=1,3) / 0.1606050000E+01, 0.5031710000E+01, 0.4064040000E-01 /
  data (epsData(18,i,7), i=1,3) / 0.6357560000E+01, 0.5282420000E+01, 0.1506410000E-01 /
  data (epsData(19,i,7), i=1,3) / 0.1038290000E+02, 0.5833180000E+01, 0.1433060000E-01 /
  data (epsData(20,i,7), i=1,3) / 0.1329650000E+02, 0.6616820000E+01, 0.1778230000E-01 /
  data (epsData(21,i,7), i=1,3) / 0.1510250000E+02, 0.7485040000E+01, 0.2432620000E-01 /
  data (epsData(22,i,7), i=1,3) / 0.1627110000E+02, 0.8463130000E+01, 0.3658600000E-01 /
  data (epsData(23,i,7), i=1,3) / 0.1687170000E+02, 0.9346420000E+01, 0.5790940000E-01 /
  data (epsData(24,i,7), i=1,3) / 0.1717600000E+02, 0.1013550000E+02, 0.1177170000E+00 /
  data (epsData(25,i,7), i=1,3) / 0.1732920000E+02, 0.1080980000E+02, 0.1988760000E+00 /
  data (epsData(26,i,7), i=1,3) / 0.1750850000E+02, 0.1159220000E+02, 0.1218670000E+01 /
  data (epsData(27,i,7), i=1,3) / 0.1767990000E+02, 0.1150120000E+02, 0.2391740000E+01 /
  data (epsData(28,i,7), i=1,3) / 0.1780610000E+02, 0.1107640000E+02, 0.2709680000E+01 /
  data (epsData(29,i,7), i=1,3) / 0.1849190000E+02, 0.1044750000E+02, 0.3666220000E+01 /
  data (epsData(30,i,7), i=1,3) / 0.2223610000E+02, 0.8798150000E+01, 0.7608590000E+01 /
  data (epsData(31,i,7), i=1,3) / 0.2397770000E+02, 0.7191600000E+01, 0.9241300000E+01 /
  data (epsData(32,i,7), i=1,3) / 0.2577770000E+02, 0.5008540000E+01, 0.1012140000E+02 /
  data (epsData(33,i,7), i=1,3) / 0.2756280000E+02, 0.2909150000E+01, 0.1012480000E+02 /
  data (epsData(34,i,7), i=1,3) / 0.2939440000E+02, 0.1195510000E+01, 0.9545330000E+01 /
  data (epsData(35,i,7), i=1,3) / 0.3165530000E+02, -0.2929610000E+00, 0.8557160000E+01 /
  data (epsData(36,i,7), i=1,3) / 0.3426360000E+02, -0.1439110000E+01, 0.7312700000E+01 /
  data (epsData(37,i,7), i=1,3) / 0.3676700000E+02, -0.2022260000E+01, 0.6116030000E+01 /
  data (epsData(38,i,7), i=1,3) / 0.3942280000E+02, -0.2219330000E+01, 0.5074660000E+01 /
  data (epsData(39,i,7), i=1,3) / 0.4322660000E+02, -0.2245150000E+01, 0.4054200000E+01 /
  data (epsData(40,i,7), i=1,3) / 0.4932200000E+02, -0.2213490000E+01, 0.2952470000E+01 /
  data (epsData(41,i,7), i=1,3) / 0.5555040000E+02, -0.2079260000E+01, 0.2103730000E+01 /
  data (epsData(42,i,7), i=1,3) / 0.6175320000E+02, -0.1831080000E+01, 0.1514780000E+01 /
  data (epsData(43,i,7), i=1,3) / 0.6913430000E+02, -0.1558290000E+01, 0.1075710000E+01 /
  data (epsData(44,i,7), i=1,3) / 0.7681770000E+02, -0.1323860000E+01, 0.7678550000E+00 /
  data (epsData(45,i,7), i=1,3) / 0.8531280000E+02, -0.1102970000E+01, 0.5422420000E+00 /
  data (epsData(46,i,7), i=1,3) / 0.9471740000E+02, -0.9119470000E+00, 0.3828830000E+00 /
  data (epsData(47,i,7), i=1,3) / 0.1043350000E+03, -0.7587640000E+00, 0.2745960000E+00 /
  data (epsData(48,i,7), i=1,3) / 0.1155610000E+03, -0.6200880000E+00, 0.1935410000E+00 /
  data (epsData(49,i,7), i=1,3) / 0.1273020000E+03, -0.5108950000E+00, 0.1383110000E+00 /
  data (epsData(50,i,7), i=1,3) / 0.1407600000E+03, -0.4159820000E+00, 0.9753050000E-01 /
  data (epsData(51,i,7), i=1,3) / 0.1557320000E+03, -0.3371250000E+00, 0.6841610000E-01 /
  data (epsData(52,i,7), i=1,3) / 0.1731050000E+03, -0.2710350000E+00, 0.4755710000E-01 /
  data (epsData(53,i,7), i=1,3) / 0.1892760000E+03, -0.2242910000E+00, 0.3483920000E-01 /
  data (epsData(54,i,7), i=1,3) / 0.2096800000E+03, -0.1802070000E+00, 0.2408710000E-01 /
  data (epsData(55,i,7), i=1,3) / 0.2334910000E+03, -0.1427160000E+00, 0.1630460000E-01 /
  data (epsData(56,i,7), i=1,3) / 0.2595300000E+03, -0.1128810000E+00, 0.1115220000E-01 /
  data (epsData(57,i,7), i=1,3) / 0.2879020000E+03, -0.8883230000E-01, 0.7669390000E-02 /
  data (epsData(58,i,7), i=1,3) / 0.3199120000E+03, -0.6842580000E-01, 0.5227810000E-02 /
  data (epsData(59,i,7), i=1,3) / 0.3623330000E+03, -0.4721440000E-01, 0.3306230000E-02 /
  data (epsData(60,i,7), i=1,3) / 0.3874090000E+03, -0.3450960000E-01, 0.2589970000E-02 /
  data (epsData(61,i,7), i=1,3) / 0.3975780000E+03, -0.2746840000E-01, 0.2395340000E-02 /
  data (epsData(62,i,7), i=1,3) / 0.4023760000E+03, -0.2266440000E-01, 0.2328110000E-02 /
  data (epsData(63,i,7), i=1,3) / 0.4051330000E+03, -0.1876510000E-01, 0.2313770000E-02 /
  data (epsData(64,i,7), i=1,3) / 0.4065150000E+03, -0.1613360000E-01, 0.2324920000E-02 /
  data (epsData(65,i,7), i=1,3) / 0.4072070000E+03, -0.1450820000E-01, 0.2341690000E-02 /
  data (epsData(66,i,7), i=1,3) / 0.4079000000E+03, -0.1254670000E-01, 0.2373400000E-02 /
  data (epsData(67,i,7), i=1,3) / 0.4085940000E+03, -0.1001350000E-01, 0.2436790000E-02 /
  data (epsData(68,i,7), i=1,3) / 0.4092880000E+03, -0.6303980000E-02, 0.2620580000E-02 /
  data (epsData(69,i,7), i=1,3) / 0.4099010000E+03, -0.1496990000E-02, 0.4258240000E-02 /
  data (epsData(70,i,7), i=1,3) / 0.4099570000E+03, -0.7874290000E-03, 0.4893900000E-02 /
  data (epsData(71,i,7), i=1,3) / 0.4100650000E+03, 0.6427880000E-04, 0.6106340000E-02 /
  data (epsData(72,i,7), i=1,3) / 0.4102700000E+03, 0.1052620000E-02, 0.8284960000E-02 /
  data (epsData(73,i,7), i=1,3) / 0.4105000000E+03, 0.1700920000E-02, 0.1072650000E-01 /
  data (epsData(74,i,7), i=1,3) / 0.4106840000E+03, 0.1708790000E-02, 0.1272760000E-01 /
  data (epsData(75,i,7), i=1,3) / 0.4110050000E+03, 0.1027560000E-02, 0.1537960000E-01 /
  data (epsData(76,i,7), i=1,3) / 0.4113370000E+03, 0.5484230000E-03, 0.1713710000E-01 /
  data (epsData(77,i,7), i=1,3) / 0.4116630000E+03, -0.2835590000E-04, 0.1914410000E-01 /
  data (epsData(78,i,7), i=1,3) / 0.4131000000E+03, -0.2892660000E-02, 0.2569530000E-01 /
  data (epsData(79,i,7), i=1,3) / 0.4142430000E+03, -0.6240220000E-02, 0.2971410000E-01 /
  data (epsData(80,i,7), i=1,3) / 0.4158270000E+03, -0.1144220000E-01, 0.3277210000E-01 /
  data (epsData(81,i,7), i=1,3) / 0.4176560000E+03, -0.1636910000E-01, 0.3365430000E-01 /
  data (epsData(82,i,7), i=1,3) / 0.4212910000E+03, -0.2214170000E-01, 0.3335970000E-01 /
  data (epsData(83,i,7), i=1,3) / 0.4269670000E+03, -0.2817560000E-01, 0.3213580000E-01 /
  data (epsData(84,i,7), i=1,3) / 0.4325410000E+03, -0.3071420000E-01, 0.2953330000E-01 /
  data (epsData(85,i,7), i=1,3) / 0.4510930000E+03, -0.3448650000E-01, 0.2572570000E-01 /
  data (epsData(86,i,7), i=1,3) / 0.4781680000E+03, -0.3477440000E-01, 0.2076090000E-01 /
  data (epsData(87,i,7), i=1,3) / 0.5208020000E+03, -0.3236050000E-01, 0.1553870000E-01 /
  data (epsData(88,i,7), i=1,3) / 0.6002150000E+03, -0.2671080000E-01, 0.9671790000E-02 /
  data (epsData(89,i,7), i=1,3) / 0.6775200000E+03, -0.2189690000E-01, 0.6356800000E-02 /
  data (epsData(90,i,7), i=1,3) / 0.7568890000E+03, -0.1795020000E-01, 0.4289440000E-02 /
  data (epsData(91,i,7), i=1,3) / 0.8381120000E+03, -0.1483070000E-01, 0.2988490000E-02 /
  data (epsData(92,i,7), i=1,3) / 0.9253540000E+03, -0.1225160000E-01, 0.2080330000E-02 /
  data (epsData(93,i,7), i=1,3) / 0.1023420000E+04, -0.1004970000E-01, 0.1432270000E-02 /
  data (epsData(94,i,7), i=1,3) / 0.1132990000E+04, -0.8205110000E-02, 0.9780080000E-03 /
  data (epsData(95,i,7), i=1,3) / 0.1257990000E+04, -0.6645910000E-02, 0.6570270000E-03 /
  data (epsData(96,i,7), i=1,3) / 0.1402220000E+04, -0.5331750000E-02, 0.4353440000E-03 /
  data (epsData(97,i,7), i=1,3) / 0.1564030000E+04, -0.4271170000E-02, 0.2871950000E-03 /
  data (epsData(98,i,7), i=1,3) / 0.1736720000E+04, -0.3451520000E-02, 0.1916260000E-03 /
  data (epsData(99,i,7), i=1,3) / 0.1932930000E+04, -0.2774950000E-02, 0.1265210000E-03 /
  data (epsData(100,i,7), i=1,3) / 0.2154970000E+04, -0.2223980000E-02, 0.8259470000E-04 /
  data (epsData(101,i,7), i=1,3) / 0.2403630000E+04, -0.1780830000E-02, 0.5367720000E-04 /
  data (epsData(102,i,7), i=1,3) / 0.2682900000E+04, -0.1424260000E-02, 0.3470180000E-04 /
  data (epsData(103,i,7), i=1,3) / 0.2997320000E+04, -0.1137330000E-02, 0.2229420000E-04 /
  data (epsData(104,i,7), i=1,3) / 0.3350070000E+04, -0.9076840000E-03, 0.1426100000E-04 /
  data (epsData(105,i,7), i=1,3) / 0.3746630000E+04, -0.7237470000E-03, 0.9080240000E-05 /
  data (epsData(106,i,7), i=1,3) / 0.4190870000E+04, -0.5770660000E-03, 0.5762810000E-05 /
  data (epsData(107,i,7), i=1,3) / 0.4691730000E+04, -0.4594670000E-03, 0.3636580000E-05 /
  data (epsData(108,i,7), i=1,3) / 0.5254490000E+04, -0.3656500000E-03, 0.2286030000E-05 /
  data (epsData(109,i,7), i=1,3) / 0.5884680000E+04, -0.2910690000E-03, 0.1434030000E-05 /
  data (epsData(110,i,7), i=1,3) / 0.6591030000E+04, -0.2317170000E-03, 0.9033960000E-06 /
  data (epsData(111,i,7), i=1,3) / 0.7386360000E+04, -0.1842940000E-03, 0.5597380000E-06 /
  data (epsData(112,i,7), i=1,3) / 0.8278790000E+04, -0.1465590000E-03, 0.3487850000E-06 /
  data (epsData(113,i,7), i=1,3) / 0.9280420000E+04, -0.1165350000E-03, 0.2151950000E-06 /
  data (epsData(114,i,7), i=1,3) / 0.1040460000E+05, -0.9264960000E-04, 0.1332720000E-06 /
  data (epsData(115,i,7), i=1,3) / 0.1166620000E+05, -0.7365280000E-04, 0.8213420000E-07 /
  data (epsData(116,i,7), i=1,3) / 0.1308140000E+05, -0.5854990000E-04, 0.5053110000E-07 /
  data (epsData(117,i,7), i=1,3) / 0.1467300000E+05, -0.4651890000E-04, 0.3101210000E-07 /
  data (epsData(118,i,7), i=1,3) / 0.1645710000E+05, -0.3696700000E-04, 0.1899540000E-07 /
  data (epsData(119,i,7), i=1,3) / 0.1846030000E+05, -0.2937180000E-04, 0.1160280000E-07 /
  data (epsData(120,i,7), i=1,3) / 0.2070710000E+05, -0.2333840000E-04, 0.7081030000E-08 /
  data (epsData(121,i,7), i=1,3) / 0.2323040000E+05, -0.1854030000E-04, 0.4311880000E-08 /
  data (epsData(122,i,7), i=1,3) / 0.2606080000E+05, -0.1472960000E-04, 0.2618670000E-08 /
  data (epsData(123,i,7), i=1,3) / 0.2923790000E+05, -0.1170090000E-04, 0.1592920000E-08 /
  data (epsData(124,i,7), i=1,3) / 0.3280600000E+05, -0.9293140000E-05, 0.9665140000E-09 /
  data (epsData(125,i,7), i=1,3) / 0.3680720000E+05, -0.7381930000E-05, 0.5826570000E-09 /
  data (epsData(126,i,7), i=1,3) / 0.4129450000E+05, -0.5864390000E-05, 0.3589840000E-09 /
  data (epsData(127,i,7), i=1,3) / 0.4633460000E+05, -0.4657710000E-05, 0.2149330000E-09 /
  data (epsData(128,i,7), i=1,3) / 0.5198550000E+05, -0.3699980000E-05, 0.1300400000E-09 /
  data (epsData(129,i,7), i=1,3) / 0.5833480000E+05, -0.2938270000E-05, 0.7957660000E-10 /
  data (epsData(130,i,7), i=1,3) / 0.6545280000E+05, -0.2333890000E-05, 0.5430900000E-10 /
  data (epsData(131,i,7), i=1,3) / 0.7344080000E+05, -0.1853750000E-05, 0.3417660000E-10 /
  data (epsData(132,i,7), i=1,3) / 0.8240770000E+05, -0.1472260000E-05, 0.2588730000E-10 /
  data (epsData(133,i,7), i=1,3) / 0.9246320000E+05, -0.1169440000E-05, 0.1215130000E-10 /
  data (epsData(134,i,7), i=1,3) / 0.1000000000E+06, -0.9999510000E-06, 0.9591340000E-11 /
  data (epsData(1,i,8), i=1,3) / 0.2506580000E-02, -0.3590260000E+02, 0.9799700000E+02 /
  data (epsData(2,i,8), i=1,3) / 0.4693440000E-02, -0.2883560000E+02, 0.3289890000E+02 /
  data (epsData(3,i,8), i=1,3) / 0.7560590000E-02, -0.1154120000E+02, 0.1517800000E+02 /
  data (epsData(4,i,8), i=1,3) / 0.1110800000E-01, -0.4013820000E+01, 0.8464570000E+01 /
  data (epsData(5,i,8), i=1,3) / 0.1533570000E-01, -0.4769460000E+00, 0.5336860000E+01 /
  data (epsData(6,i,8), i=1,3) / 0.2024370000E-01, 0.1351970000E+01, 0.3659690000E+01 /
  data (epsData(7,i,8), i=1,3) / 0.2583200000E-01, 0.2375390000E+01, 0.2664960000E+01 /
  data (epsData(8,i,8), i=1,3) / 0.3210060000E-01, 0.2985440000E+01, 0.2028970000E+01 /
  data (epsData(9,i,8), i=1,3) / 0.3904940000E-01, 0.3368160000E+01, 0.1598250000E+01 /
  data (epsData(10,i,8), i=1,3) / 0.4667860000E-01, 0.3618560000E+01, 0.1293020000E+01 /
  data (epsData(11,i,8), i=1,3) / 0.5498800000E-01, 0.3788260000E+01, 0.1068740000E+01 /
  data (epsData(12,i,8), i=1,3) / 0.7364770000E-01, 0.3991590000E+01, 0.7672520000E+00 /
  data (epsData(13,i,8), i=1,3) / 0.1067390000E+00, 0.4135490000E+01, 0.5100310000E+00 /
  data (epsData(14,i,8), i=1,3) / 0.1754980000E+00, 0.4226430000E+01, 0.2999410000E+00 /
  data (epsData(15,i,8), i=1,3) / 0.3640360000E+00, 0.4268640000E+01, 0.1486900000E+00 /
  data (epsData(16,i,8), i=1,3) / 0.1653130000E+01, 0.4287390000E+01, 0.3506270000E-01 /
  data (epsData(17,i,8), i=1,3) / 0.6735020000E+01, 0.4494030000E+01, 0.1253010000E-01 /
  data (epsData(18,i,8), i=1,3) / 0.1098570000E+02, 0.4947840000E+01, 0.1185630000E-01 /
  data (epsData(19,i,8), i=1,3) / 0.1397750000E+02, 0.5578580000E+01, 0.1470320000E-01 /
  data (epsData(20,i,8), i=1,3) / 0.1582780000E+02, 0.6283210000E+01, 0.2034560000E-01 /
  data (epsData(21,i,8), i=1,3) / 0.1702350000E+02, 0.7103040000E+01, 0.3199170000E-01 /
  data (epsData(22,i,8), i=1,3) / 0.1763770000E+02, 0.7898130000E+01, 0.5656050000E-01 /
  data (epsData(23,i,8), i=1,3) / 0.1779300000E+02, 0.8233910000E+01, 0.6888450000E-01 /
  data (epsData(24,i,8), i=1,3) / 0.1794890000E+02, 0.8693870000E+01, 0.1547960000E+00 /
  data (epsData(25,i,8), i=1,3) / 0.1810550000E+02, 0.9478500000E+01, 0.2897480000E+00 /
  data (epsData(26,i,8), i=1,3) / 0.1821670000E+02, 0.9869700000E+01, 0.1277410000E+01 /
  data (epsData(27,i,8), i=1,3) / 0.1837990000E+02, 0.9595910000E+01, 0.2399490000E+01 /
  data (epsData(28,i,8), i=1,3) / 0.1846670000E+02, 0.9147580000E+01, 0.2609110000E+01 /
  data (epsData(29,i,8), i=1,3) / 0.1877180000E+02, 0.8465240000E+01, 0.2852630000E+01 /
  data (epsData(30,i,8), i=1,3) / 0.1905040000E+02, 0.8330740000E+01, 0.2874910000E+01 /
  data (epsData(31,i,8), i=1,3) / 0.2043170000E+02, 0.7758430000E+01, 0.3798930000E+01 /
  data (epsData(32,i,8), i=1,3) / 0.2343000000E+02, 0.6990050000E+01, 0.5581610000E+01 /
  data (epsData(33,i,8), i=1,3) / 0.2545050000E+02, 0.5897920000E+01, 0.6851910000E+01 /
  data (epsData(34,i,8), i=1,3) / 0.2757300000E+02, 0.4292530000E+01, 0.7594230000E+01 /
  data (epsData(35,i,8), i=1,3) / 0.2971580000E+02, 0.2715690000E+01, 0.7665520000E+01 /
  data (epsData(36,i,8), i=1,3) / 0.3221690000E+02, 0.1253020000E+01, 0.7270030000E+01 /
  data (epsData(37,i,8), i=1,3) / 0.3481290000E+02, 0.1068310000E+00, 0.6600620000E+01 /
  data (epsData(38,i,8), i=1,3) / 0.3762920000E+02, -0.6971330000E+00, 0.5722810000E+01 /
  data (epsData(39,i,8), i=1,3) / 0.4049150000E+02, -0.1116530000E+01, 0.4901670000E+01 /
  data (epsData(40,i,8), i=1,3) / 0.4394320000E+02, -0.1334730000E+01, 0.4141570000E+01 /
  data (epsData(41,i,8), i=1,3) / 0.5131480000E+02, -0.1601460000E+01, 0.3054240000E+01 /
  data (epsData(42,i,8), i=1,3) / 0.5717710000E+02, -0.1649780000E+01, 0.2334620000E+01 /
  data (epsData(43,i,8), i=1,3) / 0.6320040000E+02, -0.1552380000E+01, 0.1785050000E+01 /
  data (epsData(44,i,8), i=1,3) / 0.7180600000E+02, -0.1376860000E+01, 0.1262470000E+01 /
  data (epsData(45,i,8), i=1,3) / 0.8185100000E+02, -0.1170590000E+01, 0.8610550000E+00 /
  data (epsData(46,i,8), i=1,3) / 0.9123490000E+02, -0.9978080000E+00, 0.6166960000E+00 /
  data (epsData(47,i,8), i=1,3) / 0.1021740000E+03, -0.8285930000E+00, 0.4314680000E+00 /
  data (epsData(48,i,8), i=1,3) / 0.1135690000E+03, -0.6878040000E+00, 0.3050300000E+00 /
  data (epsData(49,i,8), i=1,3) / 0.1260750000E+03, -0.5672160000E+00, 0.2164870000E+00 /
  data (epsData(50,i,8), i=1,3) / 0.1397130000E+03, -0.4657010000E+00, 0.1531690000E+00 /
  data (epsData(51,i,8), i=1,3) / 0.1546080000E+03, -0.3814780000E+00, 0.1083560000E+00 /
  data (epsData(52,i,8), i=1,3) / 0.1715010000E+03, -0.3106890000E+00, 0.7683480000E-01 /
  data (epsData(53,i,8), i=1,3) / 0.1869580000E+03, -0.2603600000E+00, 0.5697750000E-01 /
  data (epsData(54,i,8), i=1,3) / 0.2088580000E+03, -0.2074560000E+00, 0.3854220000E-01 /
  data (epsData(55,i,8), i=1,3) / 0.2335450000E+03, -0.1645650000E+00, 0.2593880000E-01 /
  data (epsData(56,i,8), i=1,3) / 0.2605680000E+03, -0.1308050000E+00, 0.1764070000E-01 /
  data (epsData(57,i,8), i=1,3) / 0.2893060000E+03, -0.1046600000E+00, 0.1218700000E-01 /
  data (epsData(58,i,8), i=1,3) / 0.3205750000E+03, -0.8377280000E-01, 0.8451090000E-02 /
  data (epsData(59,i,8), i=1,3) / 0.3555270000E+03, -0.6655860000E-01, 0.5825000000E-02 /
  data (epsData(60,i,8), i=1,3) / 0.3950390000E+03, -0.5213180000E-01, 0.3975520000E-02 /
  data (epsData(61,i,8), i=1,3) / 0.4388480000E+03, -0.4004680000E-01, 0.2774430000E-02 /
  data (epsData(62,i,8), i=1,3) / 0.5111980000E+03, -0.2348100000E-01, 0.1572290000E-02 /
  data (epsData(63,i,8), i=1,3) / 0.5260830000E+03, -0.1894180000E-01, 0.1447640000E-02 /
  data (epsData(64,i,8), i=1,3) / 0.5336280000E+03, -0.1554670000E-01, 0.1395620000E-02 /
  data (epsData(65,i,8), i=1,3) / 0.5374250000E+03, -0.1299060000E-01, 0.1386120000E-02 /
  data (epsData(66,i,8), i=1,3) / 0.5397120000E+03, -0.1075200000E-01, 0.1400980000E-02 /
  data (epsData(67,i,8), i=1,3) / 0.5404760000E+03, -0.9762430000E-02, 0.1416140000E-02 /
  data (epsData(68,i,8), i=1,3) / 0.5412400000E+03, -0.8548240000E-02, 0.1443560000E-02 /
  data (epsData(69,i,8), i=1,3) / 0.5420050000E+03, -0.6934230000E-02, 0.1502030000E-02 /
  data (epsData(70,i,8), i=1,3) / 0.5427710000E+03, -0.4510760000E-02, 0.1733420000E-02 /
  data (epsData(71,i,8), i=1,3) / 0.5431050000E+03, -0.2946990000E-02, 0.2315000000E-02 /
  data (epsData(72,i,8), i=1,3) / 0.5432500000E+03, -0.2214070000E-02, 0.3036750000E-02 /
  data (epsData(73,i,8), i=1,3) / 0.5433860000E+03, -0.1700730000E-02, 0.3789050000E-02 /
  data (epsData(74,i,8), i=1,3) / 0.5435910000E+03, -0.1302430000E-02, 0.4985590000E-02 /
  data (epsData(75,i,8), i=1,3) / 0.5439850000E+03, -0.1070510000E-02, 0.6842360000E-02 /
  data (epsData(76,i,8), i=1,3) / 0.5442440000E+03, -0.1149460000E-02, 0.7950810000E-02 /
  data (epsData(77,i,8), i=1,3) / 0.5448150000E+03, -0.1444290000E-02, 0.9763120000E-02 /
  data (epsData(78,i,8), i=1,3) / 0.5459170000E+03, -0.2106600000E-02, 0.1299350000E-01 /
  data (epsData(79,i,8), i=1,3) / 0.5469220000E+03, -0.3456980000E-02, 0.1553060000E-01 /
  data (epsData(80,i,8), i=1,3) / 0.5482460000E+03, -0.5809110000E-02, 0.1778800000E-01 /
  data (epsData(81,i,8), i=1,3) / 0.5499590000E+03, -0.8766100000E-02, 0.1912750000E-01 /
  data (epsData(82,i,8), i=1,3) / 0.5524730000E+03, -0.1185550000E-01, 0.1953890000E-01 /
  data (epsData(83,i,8), i=1,3) / 0.5583870000E+03, -0.1625180000E-01, 0.1936130000E-01 /
  data (epsData(84,i,8), i=1,3) / 0.5641890000E+03, -0.1840950000E-01, 0.1803030000E-01 /
  data (epsData(85,i,8), i=1,3) / 0.5821890000E+03, -0.2136910000E-01, 0.1631750000E-01 /
  data (epsData(86,i,8), i=1,3) / 0.6110640000E+03, -0.2231280000E-01, 0.1357740000E-01 /
  data (epsData(87,i,8), i=1,3) / 0.6559530000E+03, -0.2155320000E-01, 0.1052530000E-01 /
  data (epsData(88,i,8), i=1,3) / 0.7421640000E+03, -0.1850590000E-01, 0.6945880000E-02 /
  data (epsData(89,i,8), i=1,3) / 0.8372970000E+03, -0.1531380000E-01, 0.4591920000E-02 /
  data (epsData(90,i,8), i=1,3) / 0.9361880000E+03, -0.1259340000E-01, 0.3103860000E-02 /
  data (epsData(91,i,8), i=1,3) / 0.1044420000E+04, -0.1029510000E-01, 0.2101660000E-02 /
  data (epsData(92,i,8), i=1,3) / 0.1158450000E+04, -0.8447340000E-02, 0.1440320000E-02 /
  data (epsData(93,i,8), i=1,3) / 0.1289180000E+04, -0.6850810000E-02, 0.9663360000E-03 /
  data (epsData(94,i,8), i=1,3) / 0.1438010000E+04, -0.5511980000E-02, 0.6462770000E-03 /
  data (epsData(95,i,8), i=1,3) / 0.1595570000E+04, -0.4474810000E-02, 0.4367440000E-03 /
  data (epsData(96,i,8), i=1,3) / 0.1773640000E+04, -0.3613160000E-02, 0.2907560000E-03 /
  data (epsData(97,i,8), i=1,3) / 0.1976790000E+04, -0.2898760000E-02, 0.1929490000E-03 /
  data (epsData(98,i,8), i=1,3) / 0.2202950000E+04, -0.2326510000E-02, 0.1276150000E-03 /
  data (epsData(99,i,8), i=1,3) / 0.2455330000E+04, -0.1866050000E-02, 0.8365980000E-04 /
  data (epsData(100,i,8), i=1,3) / 0.2737650000E+04, -0.1495460000E-02, 0.5460710000E-04 /
  data (epsData(101,i,8), i=1,3) / 0.3054160000E+04, -0.1197230000E-02, 0.3547410000E-04 /
  data (epsData(102,i,8), i=1,3) / 0.3409730000E+04, -0.9572650000E-03, 0.2291350000E-04 /
  data (epsData(103,i,8), i=1,3) / 0.3809890000E+04, -0.7643040000E-03, 0.1471750000E-04 /
  data (epsData(104,i,8), i=1,3) / 0.4258590000E+04, -0.6099640000E-03, 0.9414530000E-05 /
  data (epsData(105,i,8), i=1,3) / 0.4762410000E+04, -0.4864660000E-03, 0.5995870000E-05 /
  data (epsData(106,i,8), i=1,3) / 0.5328830000E+04, -0.3876510000E-03, 0.3801640000E-05 /
  data (epsData(107,i,8), i=1,3) / 0.5966160000E+04, -0.3086250000E-03, 0.2398880000E-05 /
  data (epsData(108,i,8), i=1,3) / 0.6680820000E+04, -0.2457040000E-03, 0.1516190000E-05 /
  data (epsData(109,i,8), i=1,3) / 0.7482640000E+04, -0.1955670000E-03, 0.9481530000E-06 /
  data (epsData(110,i,8), i=1,3) / 0.8385840000E+04, -0.1555070000E-03, 0.5934650000E-06 /
  data (epsData(111,i,8), i=1,3) / 0.9399730000E+04, -0.1236310000E-03, 0.3685320000E-06 /
  data (epsData(112,i,8), i=1,3) / 0.1053780000E+05, -0.9827660000E-04, 0.2283670000E-06 /
  data (epsData(113,i,8), i=1,3) / 0.1181500000E+05, -0.7811530000E-04, 0.1414550000E-06 /
  data (epsData(114,i,8), i=1,3) / 0.1324790000E+05, -0.6209040000E-04, 0.8747450000E-07 /
  data (epsData(115,i,8), i=1,3) / 0.1485470000E+05, -0.4935700000E-04, 0.5383590000E-07 /
  data (epsData(116,i,8), i=1,3) / 0.1666050000E+05, -0.3921900000E-04, 0.3310010000E-07 /
  data (epsData(117,i,8), i=1,3) / 0.1868780000E+05, -0.3115960000E-04, 0.2030620000E-07 /
  data (epsData(118,i,8), i=1,3) / 0.2096140000E+05, -0.2475880000E-04, 0.1242600000E-07 /
  data (epsData(119,i,8), i=1,3) / 0.2351430000E+05, -0.1966940000E-04, 0.7587580000E-08 /
  data (epsData(120,i,8), i=1,3) / 0.2637750000E+05, -0.1562770000E-04, 0.4628160000E-08 /
  data (epsData(121,i,8), i=1,3) / 0.2959080000E+05, -0.1241570000E-04, 0.2815450000E-08 /
  data (epsData(122,i,8), i=1,3) / 0.3319870000E+05, -0.9862320000E-05, 0.1711020000E-08 /
  data (epsData(123,i,8), i=1,3) / 0.3724370000E+05, -0.7835480000E-05, 0.1036690000E-08 /
  data (epsData(124,i,8), i=1,3) / 0.4178650000E+05, -0.6223810000E-05, 0.6290820000E-09 /
  data (epsData(125,i,8), i=1,3) / 0.4688780000E+05, -0.4942820000E-05, 0.3789950000E-09 /
  data (epsData(126,i,8), i=1,3) / 0.5260580000E+05, -0.3926440000E-05, 0.2283310000E-09 /
  data (epsData(127,i,8), i=1,3) / 0.5902840000E+05, -0.3118330000E-05, 0.1410290000E-09 /
  data (epsData(128,i,8), i=1,3) / 0.6622690000E+05, -0.2477180000E-05, 0.8606350000E-10 /
  data (epsData(129,i,8), i=1,3) / 0.7431310000E+05, -0.1967350000E-05, 0.5364950000E-10 /
  data (epsData(130,i,8), i=1,3) / 0.8338760000E+05, -0.1562420000E-05, 0.3454820000E-10 /
  data (epsData(131,i,8), i=1,3) / 0.9356090000E+05, -0.1241090000E-05, 0.1940590000E-10 /
  data (epsData(132,i,8), i=1,3) / 0.1000000000E+06, -0.1086460000E-05, 0.1449370000E-10 /
  data (epsData(1,i,9), i=1,3) / 0.2506580000E-02, -0.1581220000E+03, 0.2906160000E+03 /
  data (epsData(2,i,9), i=1,3) / 0.4693440000E-02, -0.1161640000E+03, 0.8710220000E+02 /
  data (epsData(3,i,9), i=1,3) / 0.7560590000E-02, -0.5309100000E+02, 0.3569450000E+02 /
  data (epsData(4,i,9), i=1,3) / 0.1110800000E-01, -0.2609550000E+02, 0.1775630000E+02 /
  data (epsData(5,i,9), i=1,3) / 0.1533570000E-01, -0.1349840000E+02, 0.1007850000E+02 /
  data (epsData(6,i,9), i=1,3) / 0.2024370000E-01, -0.7010610000E+01, 0.6292170000E+01 /
  data (epsData(7,i,9), i=1,3) / 0.2583200000E-01, -0.3389940000E+01, 0.4219620000E+01 /
  data (epsData(8,i,9), i=1,3) / 0.3210060000E-01, -0.1235970000E+01, 0.2990670000E+01 /
  data (epsData(9,i,9), i=1,3) / 0.3904940000E-01, 0.1132420000E+00, 0.2214370000E+01 /
  data (epsData(10,i,9), i=1,3) / 0.4667860000E-01, 0.9948840000E+00, 0.1698270000E+01 /
  data (epsData(11,i,9), i=1,3) / 0.5498800000E-01, 0.1591680000E+01, 0.1340400000E+01 /
  data (epsData(12,i,9), i=1,3) / 0.6397770000E-01, 0.2007910000E+01, 0.1083380000E+01 /
  data (epsData(13,i,9), i=1,3) / 0.7364770000E-01, 0.2305740000E+01, 0.8932100000E+00 /
  data (epsData(14,i,9), i=1,3) / 0.8399800000E-01, 0.2523610000E+01, 0.7488970000E+00 /
  data (epsData(15,i,9), i=1,3) / 0.9502860000E-01, 0.2686130000E+01, 0.6369620000E+00 /
  data (epsData(16,i,9), i=1,3) / 0.1191310000E+00, 0.2904450000E+01, 0.4773640000E+00 /
  data (epsData(17,i,9), i=1,3) / 0.1603860000E+00, 0.3084210000E+01, 0.3314250000E+00 /
  data (epsData(18,i,9), i=1,3) / 0.2427500000E+00, 0.3216940000E+01, 0.2039880000E+00 /
  data (epsData(19,i,9), i=1,3) / 0.4338630000E+00, 0.3293860000E+01, 0.1069410000E+00 /
  data (epsData(20,i,9), i=1,3) / 0.1295500000E+01, 0.3333240000E+01, 0.3541660000E-01 /
  data (epsData(21,i,9), i=1,3) / 0.6927830000E+01, 0.3475330000E+01, 0.9481040000E-02 /
  data (epsData(22,i,9), i=1,3) / 0.1185810000E+02, 0.3818460000E+01, 0.8664850000E-02 /
  data (epsData(23,i,9), i=1,3) / 0.1524620000E+02, 0.4299870000E+01, 0.1071990000E-01 /
  data (epsData(24,i,9), i=1,3) / 0.1732920000E+02, 0.4849420000E+01, 0.1505900000E-01 /
  data (epsData(25,i,9), i=1,3) / 0.1857940000E+02, 0.5453500000E+01, 0.2339000000E-01 /
  data (epsData(26,i,9), i=1,3) / 0.1922080000E+02, 0.6033640000E+01, 0.3887690000E-01 /
  data (epsData(27,i,9), i=1,3) / 0.1954560000E+02, 0.6598260000E+01, 0.8588790000E-01 /
  data (epsData(28,i,9), i=1,3) / 0.1970900000E+02, 0.7105940000E+01, 0.1780850000E+00 /
  data (epsData(29,i,9), i=1,3) / 0.1987310000E+02, 0.7670740000E+01, 0.8578140000E+00 /
  data (epsData(30,i,9), i=1,3) / 0.1995750000E+02, 0.7697550000E+01, 0.1438860000E+01 /
  data (epsData(31,i,9), i=1,3) / 0.2007990000E+02, 0.7374860000E+01, 0.2174880000E+01 /
  data (epsData(32,i,9), i=1,3) / 0.2012990000E+02, 0.7025740000E+01, 0.2277530000E+01 /
  data (epsData(33,i,9), i=1,3) / 0.2031670000E+02, 0.6569360000E+01, 0.2178070000E+01 /
  data (epsData(34,i,9), i=1,3) / 0.2068370000E+02, 0.6389980000E+01, 0.2339750000E+01 /
  data (epsData(35,i,9), i=1,3) / 0.2181330000E+02, 0.5811250000E+01, 0.2771570000E+01 /
  data (epsData(36,i,9), i=1,3) / 0.2372180000E+02, 0.5434470000E+01, 0.3204000000E+01 /
  data (epsData(37,i,9), i=1,3) / 0.2600760000E+02, 0.5122720000E+01, 0.4064640000E+01 /
  data (epsData(38,i,9), i=1,3) / 0.2828420000E+02, 0.4357450000E+01, 0.4924590000E+01 /
  data (epsData(39,i,9), i=1,3) / 0.3092060000E+02, 0.3163080000E+01, 0.5430910000E+01 /
  data (epsData(40,i,9), i=1,3) / 0.3377920000E+02, 0.1965640000E+01, 0.5452560000E+01 /
  data (epsData(41,i,9), i=1,3) / 0.3666360000E+02, 0.9802670000E+00, 0.5161450000E+01 /
  data (epsData(42,i,9), i=1,3) / 0.3965580000E+02, 0.2514340000E+00, 0.4666230000E+01 /
  data (epsData(43,i,9), i=1,3) / 0.4289360000E+02, -0.2210070000E+00, 0.4117290000E+01 /
  data (epsData(44,i,9), i=1,3) / 0.4671470000E+02, -0.5363780000E+00, 0.3592210000E+01 /
  data (epsData(45,i,9), i=1,3) / 0.5589850000E+02, -0.1031650000E+01, 0.2654280000E+01 /
  data (epsData(46,i,9), i=1,3) / 0.6146290000E+02, -0.1130020000E+01, 0.2146880000E+01 /
  data (epsData(47,i,9), i=1,3) / 0.6888650000E+02, -0.1127170000E+01, 0.1649530000E+01 /
  data (epsData(48,i,9), i=1,3) / 0.7789890000E+02, -0.1063380000E+01, 0.1217000000E+01 /
  data (epsData(49,i,9), i=1,3) / 0.8810820000E+02, -0.9503830000E+00, 0.8718240000E+00 /
  data (epsData(50,i,9), i=1,3) / 0.9996830000E+02, -0.8160620000E+00, 0.6075360000E+00 /
  data (epsData(51,i,9), i=1,3) / 0.1123570000E+03, -0.6886720000E+00, 0.4282810000E+00 /
  data (epsData(52,i,9), i=1,3) / 0.1256720000E+03, -0.5743340000E+00, 0.3024380000E+00 /
  data (epsData(53,i,9), i=1,3) / 0.1398270000E+03, -0.4774010000E+00, 0.2147940000E+00 /
  data (epsData(54,i,9), i=1,3) / 0.1551260000E+03, -0.3949930000E+00, 0.1531870000E+00 /
  data (epsData(55,i,9), i=1,3) / 0.1715690000E+03, -0.3272620000E+00, 0.1108870000E+00 /
  data (epsData(56,i,9), i=1,3) / 0.1878300000E+03, -0.2741350000E+00, 0.8152010000E-01 /
  data (epsData(57,i,9), i=1,3) / 0.2108170000E+03, -0.2182560000E+00, 0.5480590000E-01 /
  data (epsData(58,i,9), i=1,3) / 0.2366010000E+03, -0.1730400000E+00, 0.3674480000E-01 /
  data (epsData(59,i,9), i=1,3) / 0.2647450000E+03, -0.1375560000E+00, 0.2483980000E-01 /
  data (epsData(60,i,9), i=1,3) / 0.2958160000E+03, -0.1093340000E+00, 0.1684800000E-01 /
  data (epsData(61,i,9), i=1,3) / 0.3299490000E+03, -0.8697540000E-01, 0.1147950000E-01 /
  data (epsData(62,i,9), i=1,3) / 0.3677790000E+03, -0.6908070000E-01, 0.7825850000E-02 /
  data (epsData(63,i,9), i=1,3) / 0.4089740000E+03, -0.5491420000E-01, 0.5385300000E-02 /
  data (epsData(64,i,9), i=1,3) / 0.4538130000E+03, -0.4373340000E-01, 0.3775780000E-02 /
  data (epsData(65,i,9), i=1,3) / 0.5027910000E+03, -0.3459140000E-01, 0.2570590000E-02 /
  data (epsData(66,i,9), i=1,3) / 0.5615900000E+03, -0.2637140000E-01, 0.1721380000E-02 /
  data (epsData(67,i,9), i=1,3) / 0.6592230000E+03, -0.1515410000E-01, 0.9542660000E-03 /
  data (epsData(68,i,9), i=1,3) / 0.6761880000E+03, -0.1247840000E-01, 0.8673160000E-03 /
  data (epsData(69,i,9), i=1,3) / 0.6851950000E+03, -0.1041210000E-01, 0.8523450000E-03 /
  data (epsData(70,i,9), i=1,3) / 0.6901420000E+03, -0.8679840000E-02, 0.8491850000E-03 /
  data (epsData(71,i,9), i=1,3) / 0.6926250000E+03, -0.7404750000E-02, 0.8615890000E-03 /
  data (epsData(72,i,9), i=1,3) / 0.6942830000E+03, -0.6197640000E-02, 0.8869590000E-03 /
  data (epsData(73,i,9), i=1,3) / 0.6951140000E+03, -0.5364440000E-02, 0.9153300000E-03 /
  data (epsData(74,i,9), i=1,3) / 0.6959450000E+03, -0.4188000000E-02, 0.9832600000E-03 /
  data (epsData(75,i,9), i=1,3) / 0.6967050000E+03, -0.2584390000E-02, 0.1404400000E-02 /
  data (epsData(76,i,9), i=1,3) / 0.6969130000E+03, -0.1989160000E-02, 0.1946740000E-02 /
  data (epsData(77,i,9), i=1,3) / 0.6971170000E+03, -0.1646410000E-02, 0.2609340000E-02 /
  data (epsData(78,i,9), i=1,3) / 0.6973890000E+03, -0.1489320000E-02, 0.3417720000E-02 /
  data (epsData(79,i,9), i=1,3) / 0.6975500000E+03, -0.1529150000E-02, 0.3826210000E-02 /
  data (epsData(80,i,9), i=1,3) / 0.6988230000E+03, -0.1366270000E-02, 0.6120690000E-02 /
  data (epsData(81,i,9), i=1,3) / 0.6997060000E+03, -0.1649210000E-02, 0.7747150000E-02 /
  data (epsData(82,i,9), i=1,3) / 0.7006670000E+03, -0.2393790000E-02, 0.9246470000E-02 /
  data (epsData(83,i,9), i=1,3) / 0.7020150000E+03, -0.3792020000E-02, 0.1066090000E-01 /
  data (epsData(84,i,9), i=1,3) / 0.7037530000E+03, -0.5521790000E-02, 0.1152870000E-01 /
  data (epsData(85,i,9), i=1,3) / 0.7064150000E+03, -0.7433980000E-02, 0.1191400000E-01 /
  data (epsData(86,i,9), i=1,3) / 0.7117000000E+03, -0.9869590000E-02, 0.1194680000E-01 /
  data (epsData(87,i,9), i=1,3) / 0.7177890000E+03, -0.1129180000E-01, 0.1133680000E-01 /
  data (epsData(88,i,9), i=1,3) / 0.7339660000E+03, -0.1335080000E-01, 0.1068920000E-01 /
  data (epsData(89,i,9), i=1,3) / 0.7631520000E+03, -0.1441870000E-01, 0.9153180000E-02 /
  data (epsData(90,i,9), i=1,3) / 0.8095530000E+03, -0.1440240000E-01, 0.7360320000E-02 /
  data (epsData(91,i,9), i=1,3) / 0.8885450000E+03, -0.1310900000E-01, 0.5316040000E-02 /
  data (epsData(92,i,9), i=1,3) / 0.1014110000E+04, -0.1080290000E-01, 0.3383920000E-02 /
  data (epsData(93,i,9), i=1,3) / 0.1135670000E+04, -0.8918290000E-02, 0.2280430000E-02 /
  data (epsData(94,i,9), i=1,3) / 0.1270670000E+04, -0.7266440000E-02, 0.1527290000E-02 /
  data (epsData(95,i,9), i=1,3) / 0.1411870000E+04, -0.5953820000E-02, 0.1042340000E-02 /
  data (epsData(96,i,9), i=1,3) / 0.1571850000E+04, -0.4829020000E-02, 0.7016130000E-03 /
  data (epsData(97,i,9), i=1,3) / 0.1746770000E+04, -0.3920310000E-02, 0.4750530000E-03 /
  data (epsData(98,i,9), i=1,3) / 0.1942820000E+04, -0.3167600000E-02, 0.3175810000E-03 /
  data (epsData(99,i,9), i=1,3) / 0.2162690000E+04, -0.2551500000E-02, 0.2122640000E-03 /
  data (epsData(100,i,9), i=1,3) / 0.2411090000E+04, -0.2048300000E-02, 0.1410220000E-03 /
  data (epsData(101,i,9), i=1,3) / 0.2685490000E+04, -0.1645990000E-02, 0.9306080000E-04 /
  data (epsData(102,i,9), i=1,3) / 0.2992610000E+04, -0.1320870000E-02, 0.6108550000E-04 /
  data (epsData(103,i,9), i=1,3) / 0.3336920000E+04, -0.1058540000E-02, 0.3988050000E-04 /
  data (epsData(104,i,9), i=1,3) / 0.3723510000E+04, -0.8471320000E-03, 0.2590040000E-04 /
  data (epsData(105,i,9), i=1,3) / 0.4158160000E+04, -0.6769870000E-03, 0.1672250000E-04 /
  data (epsData(106,i,9), i=1,3) / 0.4645020000E+04, -0.5408010000E-03, 0.1075470000E-04 /
  data (epsData(107,i,9), i=1,3) / 0.5193530000E+04, -0.4313520000E-03, 0.6875030000E-05 /
  data (epsData(108,i,9), i=1,3) / 0.5809290000E+04, -0.3438550000E-03, 0.4376500000E-05 /
  data (epsData(109,i,9), i=1,3) / 0.6498220000E+04, -0.2741800000E-03, 0.2798580000E-05 /
  data (epsData(110,i,9), i=1,3) / 0.7272470000E+04, -0.2184720000E-03, 0.1764890000E-05 /
  data (epsData(111,i,9), i=1,3) / 0.8142860000E+04, -0.1739500000E-03, 0.1108850000E-05 /
  data (epsData(112,i,9), i=1,3) / 0.9121410000E+04, -0.1384180000E-03, 0.6956710000E-06 /
  data (epsData(113,i,9), i=1,3) / 0.1022140000E+05, -0.1100820000E-03, 0.4345670000E-06 /
  data (epsData(114,i,9), i=1,3) / 0.1145390000E+05, -0.8756830000E-04, 0.2707310000E-06 /
  data (epsData(115,i,9), i=1,3) / 0.1283840000E+05, -0.6963350000E-04, 0.1680530000E-06 /
  data (epsData(116,i,9), i=1,3) / 0.1439280000E+05, -0.5535970000E-04, 0.1040890000E-06 /
  data (epsData(117,i,9), i=1,3) / 0.1613750000E+05, -0.4400640000E-04, 0.6426280000E-07 /
  data (epsData(118,i,9), i=1,3) / 0.1809870000E+05, -0.3496630000E-04, 0.3960020000E-07 /
  data (epsData(119,i,9), i=1,3) / 0.2029570000E+05, -0.2779280000E-04, 0.2435020000E-07 /
  data (epsData(120,i,9), i=1,3) / 0.2276550000E+05, -0.2208080000E-04, 0.1492910000E-07 /
  data (epsData(121,i,9), i=1,3) / 0.2553860000E+05, -0.1754010000E-04, 0.9131780000E-08 /
  data (epsData(122,i,9), i=1,3) / 0.2864820000E+05, -0.1393530000E-04, 0.5575370000E-08 /
  data (epsData(123,i,9), i=1,3) / 0.3213710000E+05, -0.1107140000E-04, 0.3396820000E-08 /
  data (epsData(124,i,9), i=1,3) / 0.3605280000E+05, -0.8795470000E-05, 0.2065470000E-08 /
  data (epsData(125,i,9), i=1,3) / 0.4044790000E+05, -0.6986840000E-05, 0.1253720000E-08 /
  data (epsData(126,i,9), i=1,3) / 0.4538070000E+05, -0.5549790000E-05, 0.7599070000E-09 /
  data (epsData(127,i,9), i=1,3) / 0.5091560000E+05, -0.4408330000E-05, 0.4593070000E-09 /
  data (epsData(128,i,9), i=1,3) / 0.5712310000E+05, -0.3502010000E-05, 0.2781270000E-09 /
  data (epsData(129,i,9), i=1,3) / 0.6409540000E+05, -0.2781370000E-05, 0.1691030000E-09 /
  data (epsData(130,i,9), i=1,3) / 0.7191580000E+05, -0.2209230000E-05, 0.1013250000E-09 /
  data (epsData(131,i,9), i=1,3) / 0.8069060000E+05, -0.1754800000E-05, 0.6254020000E-10 /
  data (epsData(132,i,9), i=1,3) / 0.9053760000E+05, -0.1393800000E-05, 0.3634620000E-10 /
  data (epsData(133,i,9), i=1,3) / 0.1000000000E+06, -0.1142770000E-05, 0.2372770000E-10 /
  data (epsData(1,i,10), i=1,3) / 0.5723190000E+01, 0.2688560000E+01, 0.8474140000E-02 /
  data (epsData(2,i,10), i=1,3) / 0.1160550000E+02, 0.2914180000E+01, 0.6482710000E-02 /
  data (epsData(3,i,10), i=1,3) / 0.1568130000E+02, 0.3254450000E+01, 0.7567190000E-02 /
  data (epsData(4,i,10), i=1,3) / 0.1826280000E+02, 0.3661680000E+01, 0.1034730000E-01 /
  data (epsData(5,i,10), i=1,3) / 0.1987310000E+02, 0.4133890000E+01, 0.1604840000E-01 /
  data (epsData(6,i,10), i=1,3) / 0.2070380000E+02, 0.4592050000E+01, 0.2636780000E-01 /
  data (epsData(7,i,10), i=1,3) / 0.2104080000E+02, 0.4917100000E+01, 0.4047010000E-01 /
  data (epsData(8,i,10), i=1,3) / 0.2121040000E+02, 0.5170980000E+01, 0.5617340000E-01 /
  data (epsData(9,i,10), i=1,3) / 0.2138060000E+02, 0.5544690000E+01, 0.1403730000E+00 /
  data (epsData(10,i,10), i=1,3) / 0.2155150000E+02, 0.6042210000E+01, 0.4995380000E+00 /
  data (epsData(11,i,10), i=1,3) / 0.2166670000E+02, 0.6204320000E+01, 0.1159480000E+01 /
  data (epsData(12,i,10), i=1,3) / 0.2179830000E+02, 0.5881120000E+01, 0.1804860000E+01 /
  data (epsData(13,i,10), i=1,3) / 0.2192990000E+02, 0.5414660000E+01, 0.1936620000E+01 /
  data (epsData(14,i,10), i=1,3) / 0.2214080000E+02, 0.5136520000E+01, 0.1898330000E+01 /
  data (epsData(15,i,10), i=1,3) / 0.2318170000E+02, 0.4524260000E+01, 0.2205910000E+01 /
  data (epsData(16,i,10), i=1,3) / 0.2433410000E+02, 0.4144460000E+01, 0.2274500000E+01 /
  data (epsData(17,i,10), i=1,3) / 0.2596670000E+02, 0.4011850000E+01, 0.2409350000E+01 /
  data (epsData(18,i,10), i=1,3) / 0.2836680000E+02, 0.3838490000E+01, 0.2958790000E+01 /
  data (epsData(19,i,10), i=1,3) / 0.3096040000E+02, 0.3308230000E+01, 0.3566980000E+01 /
  data (epsData(20,i,10), i=1,3) / 0.3436990000E+02, 0.2371410000E+01, 0.3954390000E+01 /
  data (epsData(21,i,10), i=1,3) / 0.3786490000E+02, 0.1484440000E+01, 0.3954090000E+01 /
  data (epsData(22,i,10), i=1,3) / 0.4112920000E+02, 0.8325690000E+00, 0.3725230000E+01 /
  data (epsData(23,i,10), i=1,3) / 0.4540650000E+02, 0.2855110000E+00, 0.3325830000E+01 /
  data (epsData(24,i,10), i=1,3) / 0.5016670000E+02, -0.8337970000E-01, 0.2948200000E+01 /
  data (epsData(25,i,10), i=1,3) / 0.5764850000E+02, -0.5176270000E+00, 0.2434780000E+01 /
  data (epsData(26,i,10), i=1,3) / 0.6350030000E+02, -0.6897140000E+00, 0.2032240000E+01 /
  data (epsData(27,i,10), i=1,3) / 0.7131710000E+02, -0.7807690000E+00, 0.1617680000E+01 /
  data (epsData(28,i,10), i=1,3) / 0.8034950000E+02, -0.8057320000E+00, 0.1247920000E+01 /
  data (epsData(29,i,10), i=1,3) / 0.9064490000E+02, -0.7699530000E+00, 0.9340520000E+00 /
  data (epsData(30,i,10), i=1,3) / 0.1031950000E+03, -0.6931780000E+00, 0.6674460000E+00 /
  data (epsData(31,i,10), i=1,3) / 0.1179930000E+03, -0.5937620000E+00, 0.4598130000E+00 /
  data (epsData(32,i,10), i=1,3) / 0.1331960000E+03, -0.5011490000E+00, 0.3225480000E+00 /
  data (epsData(33,i,10), i=1,3) / 0.1491610000E+03, -0.4197590000E+00, 0.2287040000E+00 /
  data (epsData(34,i,10), i=1,3) / 0.1649070000E+03, -0.3535450000E+00, 0.1683630000E+00 /
  data (epsData(35,i,10), i=1,3) / 0.1865210000E+03, -0.2841930000E+00, 0.1129390000E+00 /
  data (epsData(36,i,10), i=1,3) / 0.2109150000E+03, -0.2260720000E+00, 0.7521790000E-01 /
  data (epsData(37,i,10), i=1,3) / 0.2372910000E+03, -0.1801360000E+00, 0.5057770000E-01 /
  data (epsData(38,i,10), i=1,3) / 0.2661960000E+03, -0.1435080000E+00, 0.3419940000E-01 /
  data (epsData(39,i,10), i=1,3) / 0.2980790000E+03, -0.1142290000E+00, 0.2320380000E-01 /
  data (epsData(40,i,10), i=1,3) / 0.3330580000E+03, -0.9103660000E-01, 0.1581810000E-01 /
  data (epsData(41,i,10), i=1,3) / 0.3716340000E+03, -0.7256780000E-01, 0.1080580000E-01 /
  data (epsData(42,i,10), i=1,3) / 0.4128520000E+03, -0.5819240000E-01, 0.7508510000E-02 /
  data (epsData(43,i,10), i=1,3) / 0.4585790000E+03, -0.4672280000E-01, 0.5252460000E-02 /
  data (epsData(44,i,10), i=1,3) / 0.5057220000E+03, -0.3785900000E-01, 0.3664340000E-02 /
  data (epsData(45,i,10), i=1,3) / 0.5634080000E+03, -0.2992330000E-01, 0.2481870000E-02 /
  data (epsData(46,i,10), i=1,3) / 0.6229920000E+03, -0.2380870000E-01, 0.1739570000E-02 /
  data (epsData(47,i,10), i=1,3) / 0.6942890000E+03, -0.1831730000E-01, 0.1160950000E-02 /
  data (epsData(48,i,10), i=1,3) / 0.8250840000E+03, -0.1026430000E-01, 0.6210750000E-03 /
  data (epsData(49,i,10), i=1,3) / 0.8467440000E+03, -0.8414280000E-02, 0.5647520000E-03 /
  data (epsData(50,i,10), i=1,3) / 0.8572910000E+03, -0.7045840000E-02, 0.5580240000E-03 /
  data (epsData(51,i,10), i=1,3) / 0.8626020000E+03, -0.5999300000E-02, 0.5577410000E-03 /
  data (epsData(52,i,10), i=1,3) / 0.8652660000E+03, -0.5247650000E-02, 0.5654570000E-03 /
  data (epsData(53,i,10), i=1,3) / 0.8670460000E+03, -0.4562400000E-02, 0.5799880000E-03 /
  data (epsData(54,i,10), i=1,3) / 0.8679370000E+03, -0.4114050000E-02, 0.5947850000E-03 /
  data (epsData(55,i,10), i=1,3) / 0.8688280000E+03, -0.3529130000E-02, 0.6235360000E-03 /
  data (epsData(56,i,10), i=1,3) / 0.8697200000E+03, -0.2670970000E-02, 0.7120750000E-03 /
  data (epsData(57,i,10), i=1,3) / 0.8702120000E+03, -0.1983880000E-02, 0.9996360000E-03 /
  data (epsData(58,i,10), i=1,3) / 0.8704300000E+03, -0.1698630000E-02, 0.1383440000E-02 /
  data (epsData(59,i,10), i=1,3) / 0.8706660000E+03, -0.1576670000E-02, 0.1802090000E-02 /
  data (epsData(60,i,10), i=1,3) / 0.8708580000E+03, -0.1574390000E-02, 0.2023380000E-02 /
  data (epsData(61,i,10), i=1,3) / 0.8715500000E+03, -0.1335170000E-02, 0.2695500000E-02 /
  data (epsData(62,i,10), i=1,3) / 0.8722690000E+03, -0.1148840000E-02, 0.3539290000E-02 /
  data (epsData(63,i,10), i=1,3) / 0.8731420000E+03, -0.1224160000E-02, 0.4587680000E-02 /
  data (epsData(64,i,10), i=1,3) / 0.8740940000E+03, -0.1619450000E-02, 0.5575030000E-02 /
  data (epsData(65,i,10), i=1,3) / 0.8754300000E+03, -0.2429830000E-02, 0.6530490000E-02 /
  data (epsData(66,i,10), i=1,3) / 0.8772530000E+03, -0.3526840000E-02, 0.7195860000E-02 /
  data (epsData(67,i,10), i=1,3) / 0.8799150000E+03, -0.4726580000E-02, 0.7539890000E-02 /
  data (epsData(68,i,10), i=1,3) / 0.8847750000E+03, -0.6171590000E-02, 0.7663870000E-02 /
  data (epsData(69,i,10), i=1,3) / 0.8914590000E+03, -0.7210450000E-02, 0.7415690000E-02 /
  data (epsData(70,i,10), i=1,3) / 0.9063490000E+03, -0.8620360000E-02, 0.7185760000E-02 /
  data (epsData(71,i,10), i=1,3) / 0.9363520000E+03, -0.9583030000E-02, 0.6307280000E-02 /
  data (epsData(72,i,10), i=1,3) / 0.9846250000E+03, -0.9835520000E-02, 0.5213030000E-02 /
  data (epsData(73,i,10), i=1,3) / 0.1065640000E+04, -0.9252010000E-02, 0.3922380000E-02 /
  data (epsData(74,i,10), i=1,3) / 0.1218560000E+04, -0.7686280000E-02, 0.2470330000E-02 /
  data (epsData(75,i,10), i=1,3) / 0.1373170000E+04, -0.6301670000E-02, 0.1629750000E-02 /
  data (epsData(76,i,10), i=1,3) / 0.1535000000E+04, -0.5153180000E-02, 0.1094030000E-02 /
  data (epsData(77,i,10), i=1,3) / 0.1707580000E+04, -0.4217200000E-02, 0.7437510000E-03 /
  data (epsData(78,i,10), i=1,3) / 0.1902420000E+04, -0.3419240000E-02, 0.5010350000E-03 /
  data (epsData(79,i,10), i=1,3) / 0.2116320000E+04, -0.2770030000E-02, 0.3364250000E-03 /
  data (epsData(80,i,10), i=1,3) / 0.2356030000E+04, -0.2234240000E-02, 0.2254860000E-03 /
  data (epsData(81,i,10), i=1,3) / 0.2632260000E+04, -0.1789190000E-02, 0.1494930000E-03 /
  data (epsData(82,i,10), i=1,3) / 0.2930360000E+04, -0.1440430000E-02, 0.9913960000E-04 /
  data (epsData(83,i,10), i=1,3) / 0.3264350000E+04, -0.1157250000E-02, 0.6536180000E-04 /
  data (epsData(84,i,10), i=1,3) / 0.3638930000E+04, -0.9280950000E-03, 0.4285000000E-04 /
  data (epsData(85,i,10), i=1,3) / 0.4057370000E+04, -0.7439170000E-03, 0.2798910000E-04 /
  data (epsData(86,i,10), i=1,3) / 0.4527470000E+04, -0.5953710000E-03, 0.1817380000E-04 /
  data (epsData(87,i,10), i=1,3) / 0.5056040000E+04, -0.4758050000E-03, 0.1173250000E-04 /
  data (epsData(88,i,10), i=1,3) / 0.5648280000E+04, -0.3800710000E-03, 0.7544690000E-05 /
  data (epsData(89,i,10), i=1,3) / 0.6312290000E+04, -0.3034350000E-03, 0.4852280000E-05 /
  data (epsData(90,i,10), i=1,3) / 0.7062940000E+04, -0.2417590000E-03, 0.3088610000E-05 /
  data (epsData(91,i,10), i=1,3) / 0.7902210000E+04, -0.1927000000E-03, 0.1955350000E-05 /
  data (epsData(92,i,10), i=1,3) / 0.8847420000E+04, -0.1534110000E-03, 0.1233610000E-05 /
  data (epsData(93,i,10), i=1,3) / 0.9908230000E+04, -0.1221060000E-03, 0.7736560000E-06 /
  data (epsData(94,i,10), i=1,3) / 0.1109850000E+05, -0.9717150000E-04, 0.4852420000E-06 /
  data (epsData(95,i,10), i=1,3) / 0.1243370000E+05, -0.7732070000E-04, 0.3034240000E-06 /
  data (epsData(96,i,10), i=1,3) / 0.1393500000E+05, -0.6148960000E-04, 0.1891470000E-06 /
  data (epsData(97,i,10), i=1,3) / 0.1562210000E+05, -0.4887920000E-04, 0.1173230000E-06 /
  data (epsData(98,i,10), i=1,3) / 0.1751650000E+05, -0.3884720000E-04, 0.7260480000E-07 /
  data (epsData(99,i,10), i=1,3) / 0.1964140000E+05, -0.3087570000E-04, 0.4481910000E-07 /
  data (epsData(100,i,10), i=1,3) / 0.2202780000E+05, -0.2453420000E-04, 0.2759740000E-07 /
  data (epsData(101,i,10), i=1,3) / 0.2470490000E+05, -0.1949600000E-04, 0.1694590000E-07 /
  data (epsData(102,i,10), i=1,3) / 0.2771040000E+05, -0.1549020000E-04, 0.1038670000E-07 /
  data (epsData(103,i,10), i=1,3) / 0.3108000000E+05, -0.1230960000E-04, 0.6353420000E-08 /
  data (epsData(104,i,10), i=1,3) / 0.3486600000E+05, -0.9778730000E-05, 0.3875310000E-08 /
  data (epsData(105,i,10), i=1,3) / 0.3911330000E+05, -0.7768610000E-05, 0.2358440000E-08 /
  data (epsData(106,i,10), i=1,3) / 0.4387790000E+05, -0.6171960000E-05, 0.1432610000E-08 /
  data (epsData(107,i,10), i=1,3) / 0.4922990000E+05, -0.4902220000E-05, 0.8680570000E-09 /
  data (epsData(108,i,10), i=1,3) / 0.5523060000E+05, -0.3894380000E-05, 0.5255900000E-09 /
  data (epsData(109,i,10), i=1,3) / 0.6196950000E+05, -0.3093140000E-05, 0.3190210000E-09 /
  data (epsData(110,i,10), i=1,3) / 0.6952530000E+05, -0.2457170000E-05, 0.1917030000E-09 /
  data (epsData(111,i,10), i=1,3) / 0.7801220000E+05, -0.1951500000E-05, 0.1163440000E-09 /
  data (epsData(112,i,10), i=1,3) / 0.8753520000E+05, -0.1549910000E-05, 0.6980460000E-10 /
  data (epsData(113,i,10), i=1,3) / 0.9822110000E+05, -0.1230980000E-05, 0.4165640000E-10 /
  data (epsData(114,i,10), i=1,3) / 0.1000000000E+06, -0.1187890000E-05, 0.3830070000E-10 /
  data (epsData(1,i,11), i=1,3) / 0.2506580000E-02, -0.1351890000E+03, 0.5342570000E+03 /
  data (epsData(2,i,11), i=1,3) / 0.4693440000E-02, -0.1277370000E+03, 0.1948370000E+03 /
  data (epsData(3,i,11), i=1,3) / 0.7560590000E-02, -0.5089160000E+02, 0.9653350000E+02 /
  data (epsData(4,i,11), i=1,3) / 0.1110800000E-01, -0.1676800000E+02, 0.5701680000E+02 /
  data (epsData(5,i,11), i=1,3) / 0.1533570000E-01, -0.6044330000E+00, 0.3760120000E+02 /
  data (epsData(6,i,11), i=1,3) / 0.2024370000E-01, 0.7793360000E+01, 0.2670070000E+02 /
  data (epsData(7,i,11), i=1,3) / 0.2583200000E-01, 0.1250860000E+02, 0.1997930000E+02 /
  data (epsData(8,i,11), i=1,3) / 0.3210060000E-01, 0.1532760000E+02, 0.1553970000E+02 /
  data (epsData(9,i,11), i=1,3) / 0.3904940000E-01, 0.1710210000E+02, 0.1245010000E+02 /
  data (epsData(10,i,11), i=1,3) / 0.4667860000E-01, 0.1826870000E+02, 0.1021040000E+02 /
  data (epsData(11,i,11), i=1,3) / 0.5498800000E-01, 0.1906610000E+02, 0.8532960000E+01 /
  data (epsData(12,i,11), i=1,3) / 0.7364770000E-01, 0.2005840000E+02, 0.6228070000E+01 /
  data (epsData(13,i,11), i=1,3) / 0.9502860000E-01, 0.2062930000E+02, 0.4876270000E+01 /
  data (epsData(14,i,11), i=1,3) / 0.1459540000E+00, 0.2094860000E+02, 0.3293160000E+01 /
  data (epsData(15,i,11), i=1,3) / 0.2612640000E+00, 0.2106620000E+02, 0.1894130000E+01 /
  data (epsData(16,i,11), i=1,3) / 0.5638470000E+00, 0.2124500000E+02, 0.9571970000E+00 /
  data (epsData(17,i,11), i=1,3) / 0.1337820000E+01, 0.2213990000E+02, 0.5358950000E+00 /
  data (epsData(18,i,11), i=1,3) / 0.2161410000E+01, 0.2424280000E+02, 0.4888700000E+00 /
  data (epsData(19,i,11), i=1,3) / 0.2861130000E+01, 0.2758540000E+02, 0.5761610000E+00 /
  data (epsData(20,i,11), i=1,3) / 0.3382030000E+01, 0.3182560000E+02, 0.7347270000E+00 /
  data (epsData(21,i,11), i=1,3) / 0.3729710000E+01, 0.3626510000E+02, 0.9743500000E+00 /
  data (epsData(22,i,11), i=1,3) / 0.4020090000E+01, 0.4183220000E+02, 0.1329690000E+01 /
  data (epsData(23,i,11), i=1,3) / 0.4245030000E+01, 0.4829970000E+02, 0.1819240000E+01 /
  data (epsData(24,i,11), i=1,3) / 0.4398380000E+01, 0.5470960000E+02, 0.2712550000E+01 /
  data (epsData(25,i,11), i=1,3) / 0.4476080000E+01, 0.5892540000E+02, 0.3213560000E+01 /
  data (epsData(26,i,11), i=1,3) / 0.4554460000E+01, 0.6477790000E+02, 0.3683010000E+01 /
  data (epsData(27,i,11), i=1,3) / 0.4713260000E+01, 0.7886120000E+02, 0.1147930000E+02 /
  data (epsData(28,i,11), i=1,3) / 0.4793680000E+01, 0.8790510000E+02, 0.1574710000E+02 /
  data (epsData(29,i,11), i=1,3) / 0.4874780000E+01, 0.1048800000E+03, 0.1976190000E+02 /
  data (epsData(30,i,11), i=1,3) / 0.4956560000E+01, 0.1154640000E+03, 0.5619690000E+02 /
  data (epsData(31,i,11), i=1,3) / 0.5039020000E+01, 0.1078390000E+03, 0.8540900000E+02 /
  data (epsData(32,i,11), i=1,3) / 0.5122160000E+01, 0.9016180000E+02, 0.1093610000E+03 /
  data (epsData(33,i,11), i=1,3) / 0.5205980000E+01, 0.6484570000E+02, 0.1247620000E+03 /
  data (epsData(34,i,11), i=1,3) / 0.5290480000E+01, 0.4140060000E+02, 0.1217910000E+03 /
  data (epsData(35,i,11), i=1,3) / 0.5461520000E+01, 0.1012830000E+02, 0.1161480000E+03 /
  data (epsData(36,i,11), i=1,3) / 0.5548060000E+01, -0.2876720000E+01, 0.1096110000E+03 /
  data (epsData(37,i,11), i=1,3) / 0.5635290000E+01, -0.1295570000E+02, 0.1022030000E+03 /
  data (epsData(38,i,11), i=1,3) / 0.5811770000E+01, -0.2784680000E+02, 0.8787580000E+02 /
  data (epsData(39,i,11), i=1,3) / 0.5990980000E+01, -0.3703000000E+02, 0.7236660000E+02 /
  data (epsData(40,i,11), i=1,3) / 0.6172910000E+01, -0.4154610000E+02, 0.5838430000E+02 /
  data (epsData(41,i,11), i=1,3) / 0.6357560000E+01, -0.4283590000E+02, 0.4577910000E+02 /
  data (epsData(42,i,11), i=1,3) / 0.6639630000E+01, -0.4081870000E+02, 0.3091170000E+02 /
  data (epsData(43,i,11), i=1,3) / 0.6927830000E+01, -0.3630540000E+02, 0.2013710000E+02 /
  data (epsData(44,i,11), i=1,3) / 0.7222150000E+01, -0.3069720000E+02, 0.1265220000E+02 /
  data (epsData(45,i,11), i=1,3) / 0.7321620000E+01, -0.2862940000E+02, 0.1086590000E+02 /
  data (epsData(46,i,11), i=1,3) / 0.7522590000E+01, -0.2498900000E+02, 0.8602520000E+01 /
  data (epsData(47,i,11), i=1,3) / 0.7624100000E+01, -0.2333910000E+02, 0.7788030000E+01 /
  data (epsData(48,i,11), i=1,3) / 0.7829160000E+01, -0.2084600000E+02, 0.6921460000E+01 /
  data (epsData(49,i,11), i=1,3) / 0.8460660000E+01, -0.1565900000E+02, 0.3746110000E+01 /
  data (epsData(50,i,11), i=1,3) / 0.8785590000E+01, -0.1336880000E+02, 0.2867290000E+01 /
  data (epsData(51,i,11), i=1,3) / 0.9228360000E+01, -0.1094970000E+02, 0.2194680000E+01 /
  data (epsData(52,i,11), i=1,3) / 0.9682010000E+01, -0.9080070000E+01, 0.1768050000E+01 /
  data (epsData(53,i,11), i=1,3) / 0.1026440000E+02, -0.7292170000E+01, 0.1452960000E+01 /
  data (epsData(54,i,11), i=1,3) / 0.1086380000E+02, -0.5942520000E+01, 0.1242080000E+01 /
  data (epsData(55,i,11), i=1,3) / 0.1160550000E+02, -0.4707280000E+01, 0.1044110000E+01 /
  data (epsData(56,i,11), i=1,3) / 0.1250170000E+02, -0.3605660000E+01, 0.8559290000E+00 /
  data (epsData(57,i,11), i=1,3) / 0.1343140000E+02, -0.2746340000E+01, 0.7027560000E+00 /
  data (epsData(58,i,11), i=1,3) / 0.1439430000E+02, -0.2054660000E+01, 0.5785790000E+00 /
  data (epsData(59,i,11), i=1,3) / 0.1539060000E+02, -0.1484030000E+01, 0.4790140000E+00 /
  data (epsData(60,i,11), i=1,3) / 0.1642020000E+02, -0.1004280000E+01, 0.4006110000E+00 /
  data (epsData(61,i,11), i=1,3) / 0.1732920000E+02, -0.6495440000E+00, 0.3480520000E+00 /
  data (epsData(62,i,11), i=1,3) / 0.1826280000E+02, -0.3367340000E+00, 0.3069720000E+00 /
  data (epsData(63,i,11), i=1,3) / 0.1905940000E+02, -0.1027370000E+00, 0.2799780000E+00 /
  data (epsData(64,i,11), i=1,3) / 0.1970900000E+02, 0.6957500000E-01, 0.2625250000E+00 /
  data (epsData(65,i,11), i=1,3) / 0.2070380000E+02, 0.3072690000E+00, 0.2414280000E+00 /
  data (epsData(66,i,11), i=1,3) / 0.2206830000E+02, 0.5938020000E+00, 0.2200740000E+00 /
  data (epsData(67,i,11), i=1,3) / 0.2548330000E+02, 0.1211130000E+01, 0.1799150000E+00 /
  data (epsData(68,i,11), i=1,3) / 0.2796150000E+02, 0.1696670000E+01, 0.1514240000E+00 /
  data (epsData(69,i,11), i=1,3) / 0.2914400000E+02, 0.2017530000E+01, 0.1395940000E+00 /
  data (epsData(70,i,11), i=1,3) / 0.2974440000E+02, 0.2251830000E+01, 0.1366340000E+00 /
  data (epsData(71,i,11), i=1,3) / 0.3014810000E+02, 0.2488920000E+01, 0.1404620000E+00 /
  data (epsData(72,i,11), i=1,3) / 0.3035100000E+02, 0.2669790000E+01, 0.1576870000E+00 /
  data (epsData(73,i,11), i=1,3) / 0.3055460000E+02, 0.2945870000E+01, 0.2223340000E+00 /
  data (epsData(74,i,11), i=1,3) / 0.3069900000E+02, 0.3149720000E+01, 0.4364390000E+00 /
  data (epsData(75,i,11), i=1,3) / 0.3074830000E+02, 0.3253490000E+01, 0.5922480000E+00 /
  data (epsData(76,i,11), i=1,3) / 0.3082220000E+02, 0.3215650000E+01, 0.8529630000E+00 /
  data (epsData(77,i,11), i=1,3) / 0.3100590000E+02, 0.2952150000E+01, 0.1167540000E+01 /
  data (epsData(78,i,11), i=1,3) / 0.3118040000E+02, 0.2735450000E+01, 0.1168720000E+01 /
  data (epsData(79,i,11), i=1,3) / 0.3166040000E+02, 0.2571260000E+01, 0.1161590000E+01 /
  data (epsData(80,i,11), i=1,3) / 0.3267260000E+02, 0.2545770000E+01, 0.1266280000E+01 /
  data (epsData(81,i,11), i=1,3) / 0.3381550000E+02, 0.2529090000E+01, 0.1435490000E+01 /
  data (epsData(82,i,11), i=1,3) / 0.3570800000E+02, 0.2535100000E+01, 0.1832220000E+01 /
  data (epsData(83,i,11), i=1,3) / 0.3776560000E+02, 0.2312620000E+01, 0.2230220000E+01 /
  data (epsData(84,i,11), i=1,3) / 0.4134620000E+02, 0.1803980000E+01, 0.2576290000E+01 /
  data (epsData(85,i,11), i=1,3) / 0.4581030000E+02, 0.1258560000E+01, 0.2700610000E+01 /
  data (epsData(86,i,11), i=1,3) / 0.5275460000E+02, 0.6111850000E+00, 0.2673290000E+01 /
  data (epsData(87,i,11), i=1,3) / 0.5863220000E+02, 0.1182860000E+00, 0.2529280000E+01 /
  data (epsData(88,i,11), i=1,3) / 0.6265950000E+02, -0.1376840000E+00, 0.2330210000E+01 /
  data (epsData(89,i,11), i=1,3) / 0.6703880000E+02, -0.3571840000E+00, 0.2130520000E+01 /
  data (epsData(90,i,11), i=1,3) / 0.7287300000E+02, -0.5211060000E+00, 0.1831250000E+01 /
  data (epsData(91,i,11), i=1,3) / 0.8149350000E+02, -0.6412250000E+00, 0.1480090000E+01 /
  data (epsData(92,i,11), i=1,3) / 0.9230360000E+02, -0.6942550000E+00, 0.1138290000E+01 /
  data (epsData(93,i,11), i=1,3) / 0.1035170000E+03, -0.6834370000E+00, 0.8611270000E+00 /
  data (epsData(94,i,11), i=1,3) / 0.1160350000E+03, -0.6282170000E+00, 0.6373030000E+00 /
  data (epsData(95,i,11), i=1,3) / 0.1317680000E+03, -0.5456080000E+00, 0.4510760000E+00 /
  data (epsData(96,i,11), i=1,3) / 0.1495600000E+03, -0.4621360000E+00, 0.3140280000E+00 /
  data (epsData(97,i,11), i=1,3) / 0.1682740000E+03, -0.3852820000E+00, 0.2196730000E+00 /
  data (epsData(98,i,11), i=1,3) / 0.1917340000E+03, -0.3108330000E+00, 0.1465060000E+00 /
  data (epsData(99,i,11), i=1,3) / 0.2152820000E+03, -0.2535490000E+00, 0.1018090000E+00 /
  data (epsData(100,i,11), i=1,3) / 0.2383670000E+03, -0.2101430000E+00, 0.7317250000E-01 /
  data (epsData(101,i,11), i=1,3) / 0.2604880000E+03, -0.1771320000E+00, 0.5465150000E-01 /
  data (epsData(102,i,11), i=1,3) / 0.2866720000E+03, -0.1470420000E+00, 0.3967530000E-01 /
  data (epsData(103,i,11), i=1,3) / 0.3174210000E+03, -0.1201700000E+00, 0.2799180000E-01 /
  data (epsData(104,i,11), i=1,3) / 0.3514040000E+03, -0.9794240000E-01, 0.1976750000E-01 /
  data (epsData(105,i,11), i=1,3) / 0.3888490000E+03, -0.7972380000E-01, 0.1394260000E-01 /
  data (epsData(106,i,11), i=1,3) / 0.4282080000E+03, -0.6535160000E-01, 0.1005560000E-01 /
  data (epsData(107,i,11), i=1,3) / 0.4775600000E+03, -0.5234310000E-01, 0.6890710000E-02 /
  data (epsData(108,i,11), i=1,3) / 0.5259090000E+03, -0.4275880000E-01, 0.4890280000E-02 /
  data (epsData(109,i,11), i=1,3) / 0.5881980000E+03, -0.3378720000E-01, 0.3253420000E-02 /
  data (epsData(110,i,11), i=1,3) / 0.6533510000E+03, -0.2700680000E-01, 0.2244210000E-02 /
  data (epsData(111,i,11), i=1,3) / 0.7227310000E+03, -0.2164880000E-01, 0.1556910000E-02 /
  data (epsData(112,i,11), i=1,3) / 0.8059190000E+03, -0.1691370000E-01, 0.1041630000E-02 /
  data (epsData(113,i,11), i=1,3) / 0.9038100000E+03, -0.1275080000E-01, 0.6851270000E-03 /
  data (epsData(114,i,11), i=1,3) / 0.1020680000E+04, -0.8407300000E-02, 0.4386480000E-03 /
  data (epsData(115,i,11), i=1,3) / 0.1046470000E+04, -0.7012490000E-02, 0.4007880000E-03 /
  data (epsData(116,i,11), i=1,3) / 0.1057690000E+04, -0.6032730000E-02, 0.4013330000E-03 /
  data (epsData(117,i,11), i=1,3) / 0.1063330000E+04, -0.5228560000E-02, 0.4063590000E-03 /
  data (epsData(118,i,11), i=1,3) / 0.1066170000E+04, -0.4587520000E-02, 0.4194150000E-03 /
  data (epsData(119,i,11), i=1,3) / 0.1068060000E+04, -0.3910430000E-02, 0.4456920000E-03 /
  data (epsData(120,i,11), i=1,3) / 0.1069000000E+04, -0.3372740000E-02, 0.4793930000E-03 /
  data (epsData(121,i,11), i=1,3) / 0.1069950000E+04, -0.2471580000E-02, 0.5815780000E-03 /
  data (epsData(122,i,11), i=1,3) / 0.1070810000E+04, -0.1342840000E-02, 0.1560030000E-02 /
  data (epsData(123,i,11), i=1,3) / 0.1070810000E+04, -0.1321060000E-02, 0.1585120000E-02 /
  data (epsData(124,i,11), i=1,3) / 0.1070820000E+04, -0.1308860000E-02, 0.1601470000E-02 /
  data (epsData(125,i,11), i=1,3) / 0.1070820000E+04, -0.1296300000E-02, 0.1620100000E-02 /
  data (epsData(126,i,11), i=1,3) / 0.1070830000E+04, -0.1283570000E-02, 0.1641010000E-02 /
  data (epsData(127,i,11), i=1,3) / 0.1070830000E+04, -0.1270220000E-02, 0.1664960000E-02 /
  data (epsData(128,i,11), i=1,3) / 0.1070910000E+04, -0.1176410000E-02, 0.1943980000E-02 /
  data (epsData(129,i,11), i=1,3) / 0.1070980000E+04, -0.1164240000E-02, 0.2198970000E-02 /
  data (epsData(130,i,11), i=1,3) / 0.1071070000E+04, -0.1221230000E-02, 0.2440050000E-02 /
  data (epsData(131,i,11), i=1,3) / 0.1071290000E+04, -0.1389640000E-02, 0.3003630000E-02 /
  data (epsData(132,i,11), i=1,3) / 0.1071650000E+04, -0.1841030000E-02, 0.3570500000E-02 /
  data (epsData(133,i,11), i=1,3) / 0.1072070000E+04, -0.2357930000E-02, 0.3930240000E-02 /
  data (epsData(134,i,11), i=1,3) / 0.1072610000E+04, -0.2918210000E-02, 0.4112820000E-02 /
  data (epsData(135,i,11), i=1,3) / 0.1073430000E+04, -0.3364880000E-02, 0.4188880000E-02 /
  data (epsData(136,i,11), i=1,3) / 0.1074620000E+04, -0.3961560000E-02, 0.4363880000E-02 /
  data (epsData(137,i,11), i=1,3) / 0.1076910000E+04, -0.4562640000E-02, 0.4306520000E-02 /
  data (epsData(138,i,11), i=1,3) / 0.1084000000E+04, -0.5310080000E-02, 0.4535880000E-02 /
  data (epsData(139,i,11), i=1,3) / 0.1095230000E+04, -0.6308410000E-02, 0.4924600000E-02 /
  data (epsData(140,i,11), i=1,3) / 0.1105410000E+04, -0.6947110000E-02, 0.4713430000E-02 /
  data (epsData(141,i,11), i=1,3) / 0.1146900000E+04, -0.7741910000E-02, 0.4159020000E-02 /
  data (epsData(142,i,11), i=1,3) / 0.1221670000E+04, -0.7734970000E-02, 0.3293400000E-02 /
  data (epsData(143,i,11), i=1,3) / 0.1346320000E+04, -0.6945190000E-02, 0.2335000000E-02 /
  data (epsData(144,i,11), i=1,3) / 0.1528510000E+04, -0.5713810000E-02, 0.1500700000E-02 /
  data (epsData(145,i,11), i=1,3) / 0.1712970000E+04, -0.4683190000E-02, 0.1004780000E-02 /
  data (epsData(146,i,11), i=1,3) / 0.1913520000E+04, -0.3817400000E-02, 0.6757700000E-03 /
  data (epsData(147,i,11), i=1,3) / 0.2132860000E+04, -0.3101370000E-02, 0.4561370000E-03 /
  data (epsData(148,i,11), i=1,3) / 0.2376210000E+04, -0.2510010000E-02, 0.3064140000E-03 /
  data (epsData(149,i,11), i=1,3) / 0.2635860000E+04, -0.2043060000E-02, 0.2103410000E-03 /
  data (epsData(150,i,11), i=1,3) / 0.2936540000E+04, -0.1646900000E-02, 0.1400060000E-03 /
  data (epsData(151,i,11), i=1,3) / 0.3270530000E+04, -0.1325980000E-02, 0.9294730000E-04 /
  data (epsData(152,i,11), i=1,3) / 0.3643230000E+04, -0.1066150000E-02, 0.6142590000E-04 /
  data (epsData(153,i,11), i=1,3) / 0.4061190000E+04, -0.8555870000E-03, 0.4037490000E-04 /
  data (epsData(154,i,11), i=1,3) / 0.4530090000E+04, -0.6855320000E-03, 0.2637950000E-04 /
  data (epsData(155,i,11), i=1,3) / 0.5056290000E+04, -0.5485550000E-03, 0.1714780000E-04 /
  data (epsData(156,i,11), i=1,3) / 0.5647120000E+04, -0.4384290000E-03, 0.1108600000E-04 /
  data (epsData(157,i,11), i=1,3) / 0.6308080000E+04, -0.3503300000E-03, 0.7172120000E-05 /
  data (epsData(158,i,11), i=1,3) / 0.7056340000E+04, -0.2792450000E-03, 0.4589180000E-05 /
  data (epsData(159,i,11), i=1,3) / 0.7891040000E+04, -0.2227620000E-03, 0.2923780000E-05 /
  data (epsData(160,i,11), i=1,3) / 0.8832100000E+04, -0.1774240000E-03, 0.1853490000E-05 /
  data (epsData(161,i,11), i=1,3) / 0.9885830000E+04, -0.1413450000E-03, 0.1169680000E-05 /
  data (epsData(162,i,11), i=1,3) / 0.1106920000E+05, -0.1125470000E-03, 0.7377840000E-06 /
  data (epsData(163,i,11), i=1,3) / 0.1239760000E+05, -0.8958650000E-04, 0.4633920000E-06 /
  data (epsData(164,i,11), i=1,3) / 0.1389220000E+05, -0.7125520000E-04, 0.2902010000E-06 /
  data (epsData(165,i,11), i=1,3) / 0.1556840000E+05, -0.5667600000E-04, 0.1807140000E-06 /
  data (epsData(166,i,11), i=1,3) / 0.1745160000E+05, -0.4506190000E-04, 0.1125880000E-06 /
  data (epsData(167,i,11), i=1,3) / 0.1956500000E+05, -0.3582420000E-04, 0.6965630000E-07 /
  data (epsData(168,i,11), i=1,3) / 0.2193960000E+05, -0.2846990000E-04, 0.4314880000E-07 /
  data (epsData(169,i,11), i=1,3) / 0.2460460000E+05, -0.2262420000E-04, 0.2668160000E-07 /
  data (epsData(170,i,11), i=1,3) / 0.2759760000E+05, -0.1797450000E-04, 0.1652070000E-07 /
  data (epsData(171,i,11), i=1,3) / 0.3095440000E+05, -0.1428200000E-04, 0.1011410000E-07 /
  data (epsData(172,i,11), i=1,3) / 0.3472060000E+05, -0.1134790000E-04, 0.6285130000E-08 /
  data (epsData(173,i,11), i=1,3) / 0.3894680000E+05, -0.9016460000E-05, 0.3849980000E-08 /
  data (epsData(174,i,11), i=1,3) / 0.4368910000E+05, -0.7163600000E-05, 0.2432040000E-08 /
  data (epsData(175,i,11), i=1,3) / 0.4900950000E+05, -0.5691720000E-05, 0.1406480000E-08 /
  data (epsData(176,i,11), i=1,3) / 0.5499340000E+05, -0.4519830000E-05, 0.8406540000E-09 /
  data (epsData(177,i,11), i=1,3) / 0.6169720000E+05, -0.3590520000E-05, 0.5177490000E-09 /
  data (epsData(178,i,11), i=1,3) / 0.6922450000E+05, -0.2851860000E-05, 0.3244490000E-09 /
  data (epsData(179,i,11), i=1,3) / 0.7767100000E+05, -0.2265140000E-05, 0.1934340000E-09 /
  data (epsData(180,i,11), i=1,3) / 0.8715080000E+05, -0.1799060000E-05, 0.1207080000E-09 /
  data (epsData(181,i,11), i=1,3) / 0.9777860000E+05, -0.1429160000E-05, 0.8762280000E-10 /
  data (epsData(182,i,11), i=1,3) / 0.1000000000E+06, -0.1366750000E-05, 0.9425140000E-10 /
  data (epsData(1,i,12), i=1,3) / 0.2506580000E-02, -0.6850750000E+03, 0.1377760000E+05 /
  data (epsData(2,i,12), i=1,3) / 0.4693440000E-02, -0.1671250000E+04, 0.5611520000E+04 /
  data (epsData(3,i,12), i=1,3) / 0.7560590000E-02, -0.3508580000E+03, 0.3012230000E+04 /
  data (epsData(4,i,12), i=1,3) / 0.1110800000E-01, 0.2669560000E+03, 0.1882570000E+04 /
  data (epsData(5,i,12), i=1,3) / 0.1533570000E-01, 0.5655300000E+03, 0.1292220000E+04 /
  data (epsData(6,i,12), i=1,3) / 0.2024370000E-01, 0.7224500000E+03, 0.9444900000E+03 /
  data (epsData(7,i,12), i=1,3) / 0.2583200000E-01, 0.8112780000E+03, 0.7219240000E+03 /
  data (epsData(8,i,12), i=1,3) / 0.3210060000E-01, 0.8647560000E+03, 0.5705620000E+03 /
  data (epsData(9,i,12), i=1,3) / 0.3904940000E-01, 0.8986650000E+03, 0.4627710000E+03 /
  data (epsData(10,i,12), i=1,3) / 0.4667860000E-01, 0.9211610000E+03, 0.3831810000E+03 /
  data (epsData(11,i,12), i=1,3) / 0.6397770000E-01, 0.9479410000E+03, 0.2755790000E+03 /
  data (epsData(12,i,12), i=1,3) / 0.9502860000E-01, 0.9690090000E+03, 0.1831630000E+03 /
  data (epsData(13,i,12), i=1,3) / 0.1459540000E+00, 0.9840000000E+03, 0.1219580000E+03 /
  data (epsData(14,i,12), i=1,3) / 0.2612640000E+00, 0.1013180000E+04, 0.6966690000E+02 /
  data (epsData(15,i,12), i=1,3) / 0.5098130000E+00, 0.1125400000E+04, 0.6025460000E+02 /
  data (epsData(16,i,12), i=1,3) / 0.5918850000E+00, 0.1203210000E+04, 0.5645310000E+02 /
  data (epsData(17,i,12), i=1,3) / 0.7422780000E+00, 0.1374370000E+04, 0.1294870000E+03 /
  data (epsData(18,i,12), i=1,3) / 0.8071970000E+00, 0.1491580000E+04, 0.1458520000E+03 /
  data (epsData(19,i,12), i=1,3) / 0.8406760000E+00, 0.1589130000E+04, 0.1517520000E+03 /
  data (epsData(20,i,12), i=1,3) / 0.8748370000E+00, 0.1755780000E+04, 0.1564190000E+03 /
  data (epsData(21,i,12), i=1,3) / 0.9096770000E+00, 0.1992160000E+04, 0.2824710000E+03 /
  data (epsData(22,i,12), i=1,3) / 0.9451980000E+00, 0.2123500000E+04, 0.6924280000E+03 /
  data (epsData(23,i,12), i=1,3) / 0.9813990000E+00, 0.2060120000E+04, 0.1049530000E+04 /
  data (epsData(24,i,12), i=1,3) / 0.1005440000E+01, 0.1919550000E+04, 0.1234050000E+04 /
  data (epsData(25,i,12), i=1,3) / 0.1087080000E+01, 0.1544250000E+04, 0.1501500000E+04 /
  data (epsData(26,i,12), i=1,3) / 0.1179940000E+01, 0.1124570000E+04, 0.1704930000E+04 /
  data (epsData(27,i,12), i=1,3) / 0.1212590000E+01, 0.9265780000E+03, 0.1714740000E+04 /
  data (epsData(28,i,12), i=1,3) / 0.1266670000E+01, 0.7110150000E+03, 0.1620000000E+04 /
  data (epsData(29,i,12), i=1,3) / 0.1370410000E+01, 0.4754600000E+03, 0.1466970000E+04 /
  data (epsData(30,i,12), i=1,3) / 0.1491170000E+01, 0.3116240000E+03, 0.1322140000E+04 /
  data (epsData(31,i,12), i=1,3) / 0.1600010000E+01, 0.2323060000E+03, 0.1205080000E+04 /
  data (epsData(32,i,12), i=1,3) / 0.1816680000E+01, 0.1305930000E+03, 0.1092910000E+04 /
  data (epsData(33,i,12), i=1,3) / 0.1884710000E+01, 0.1448200000E+03, 0.1044420000E+04 /
  data (epsData(34,i,12), i=1,3) / 0.1919740000E+01, 0.1688490000E+03, 0.1068180000E+04 /
  data (epsData(35,i,12), i=1,3) / 0.1955460000E+01, 0.1639310000E+03, 0.1125830000E+04 /
  data (epsData(36,i,12), i=1,3) / 0.2001360000E+01, 0.9901520000E+02, 0.1191010000E+04 /
  data (epsData(37,i,12), i=1,3) / 0.2136060000E+01, -0.1057080000E+03, 0.1175490000E+04 /
  data (epsData(38,i,12), i=1,3) / 0.2195920000E+01, -0.2052960000E+03, 0.1161170000E+04 /
  data (epsData(39,i,12), i=1,3) / 0.2266670000E+01, -0.3098290000E+03, 0.1066640000E+04 /
  data (epsData(40,i,12), i=1,3) / 0.2393200000E+01, -0.3923340000E+03, 0.9120690000E+03 /
  data (epsData(41,i,12), i=1,3) / 0.2544230000E+01, -0.4293370000E+03, 0.7616230000E+03 /
  data (epsData(42,i,12), i=1,3) / 0.2751370000E+01, -0.4399880000E+03, 0.6046760000E+03 /
  data (epsData(43,i,12), i=1,3) / 0.3066690000E+01, -0.4208360000E+03, 0.4357370000E+03 /
  data (epsData(44,i,12), i=1,3) / 0.3437100000E+01, -0.3807340000E+03, 0.2968470000E+03 /
  data (epsData(45,i,12), i=1,3) / 0.3812620000E+01, -0.3329220000E+03, 0.2002170000E+03 /
  data (epsData(46,i,12), i=1,3) / 0.4176910000E+01, -0.2850360000E+03, 0.1314950000E+03 /
  data (epsData(47,i,12), i=1,3) / 0.4515690000E+01, -0.2376680000E+03, 0.9169930000E+02 /
  data (epsData(48,i,12), i=1,3) / 0.4816720000E+01, -0.2047050000E+03, 0.7459400000E+02 /
  data (epsData(49,i,12), i=1,3) / 0.5266720000E+01, -0.1702450000E+03, 0.5529220000E+02 /
  data (epsData(50,i,12), i=1,3) / 0.5821840000E+01, -0.1392760000E+03, 0.3881390000E+02 /
  data (epsData(51,i,12), i=1,3) / 0.6420490000E+01, -0.1140190000E+03, 0.2642230000E+02 /
  data (epsData(52,i,12), i=1,3) / 0.7062670000E+01, -0.9321810000E+02, 0.1782780000E+02 /
  data (epsData(53,i,12), i=1,3) / 0.7748400000E+01, -0.7622560000E+02, 0.1200190000E+02 /
  data (epsData(54,i,12), i=1,3) / 0.8477660000E+01, -0.6244430000E+02, 0.8200550000E+01 /
  data (epsData(55,i,12), i=1,3) / 0.9250470000E+01, -0.5137340000E+02, 0.5774640000E+01 /
  data (epsData(56,i,12), i=1,3) / 0.1017190000E+02, -0.4153730000E+02, 0.4087820000E+01 /
  data (epsData(57,i,12), i=1,3) / 0.1114850000E+02, -0.3387680000E+02, 0.3059860000E+01 /
  data (epsData(58,i,12), i=1,3) / 0.1229810000E+02, -0.2730270000E+02, 0.2338930000E+01 /
  data (epsData(59,i,12), i=1,3) / 0.1364140000E+02, -0.2177910000E+02, 0.1813970000E+01 /
  data (epsData(60,i,12), i=1,3) / 0.1506690000E+02, -0.1754670000E+02, 0.1455430000E+01 /
  data (epsData(61,i,12), i=1,3) / 0.1671590000E+02, -0.1399670000E+02, 0.1185600000E+01 /
  data (epsData(62,i,12), i=1,3) / 0.1861290000E+02, -0.1107050000E+02, 0.9817340000E+00 /
  data (epsData(63,i,12), i=1,3) / 0.2062480000E+02, -0.8842250000E+01, 0.8280110000E+00 /
  data (epsData(64,i,12), i=1,3) / 0.2292010000E+02, -0.7003570000E+01, 0.6852280000E+00 /
  data (epsData(65,i,12), i=1,3) / 0.2552730000E+02, -0.5484570000E+01, 0.5416670000E+00 /
  data (epsData(66,i,12), i=1,3) / 0.2847700000E+02, -0.4214690000E+01, 0.4028290000E+00 /
  data (epsData(67,i,12), i=1,3) / 0.3160080000E+02, -0.3197090000E+01, 0.2945540000E+00 /
  data (epsData(68,i,12), i=1,3) / 0.3468760000E+02, -0.2412440000E+01, 0.2241480000E+00 /
  data (epsData(69,i,12), i=1,3) / 0.3792750000E+02, -0.1750750000E+01, 0.1804540000E+00 /
  data (epsData(70,i,12), i=1,3) / 0.4155200000E+02, -0.1137320000E+01, 0.1539300000E+00 /
  data (epsData(71,i,12), i=1,3) / 0.4632760000E+02, -0.3983620000E+00, 0.1349810000E+00 /
  data (epsData(72,i,12), i=1,3) / 0.4731540000E+02, -0.2265270000E+00, 0.1320520000E+00 /
  data (epsData(73,i,12), i=1,3) / 0.4781340000E+02, -0.1302200000E+00, 0.1308300000E+00 /
  data (epsData(74,i,12), i=1,3) / 0.4831410000E+02, -0.2131890000E-01, 0.1300910000E+00 /
  data (epsData(75,i,12), i=1,3) / 0.4856540000E+02, 0.4092670000E-01, 0.1301020000E+00 /
  data (epsData(76,i,12), i=1,3) / 0.4881750000E+02, 0.1123130000E+00, 0.1306350000E+00 /
  data (epsData(77,i,12), i=1,3) / 0.4907020000E+02, 0.2011710000E+00, 0.1327100000E+00 /
  data (epsData(78,i,12), i=1,3) / 0.4932360000E+02, 0.3237190000E+00, 0.1493580000E+00 /
  data (epsData(79,i,12), i=1,3) / 0.4953400000E+02, 0.4219820000E+00, 0.2510060000E+00 /
  data (epsData(80,i,12), i=1,3) / 0.4969590000E+02, 0.4561450000E+00, 0.3444070000E+00 /
  data (epsData(81,i,12), i=1,3) / 0.4995990000E+02, 0.4360200000E+00, 0.4497470000E+00 /
  data (epsData(82,i,12), i=1,3) / 0.5010690000E+02, 0.4155840000E+00, 0.4715470000E+00 /
  data (epsData(83,i,12), i=1,3) / 0.5043850000E+02, 0.4123610000E+00, 0.4972660000E+00 /
  data (epsData(84,i,12), i=1,3) / 0.5092420000E+02, 0.4675530000E+00, 0.5156850000E+00 /
  data (epsData(85,i,12), i=1,3) / 0.5144670000E+02, 0.5690320000E+00, 0.5793830000E+00 /
  data (epsData(86,i,12), i=1,3) / 0.5206710000E+02, 0.6672580000E+00, 0.7150410000E+00 /
  data (epsData(87,i,12), i=1,3) / 0.5272190000E+02, 0.7007350000E+00, 0.8715900000E+00 /
  data (epsData(88,i,12), i=1,3) / 0.5331880000E+02, 0.7447500000E+00, 0.9611500000E+00 /
  data (epsData(89,i,12), i=1,3) / 0.5404670000E+02, 0.7772630000E+00, 0.1155870000E+01 /
  data (epsData(90,i,12), i=1,3) / 0.5509470000E+02, 0.7034310000E+00, 0.1393110000E+01 /
  data (epsData(91,i,12), i=1,3) / 0.5644680000E+02, 0.5362150000E+00, 0.1577600000E+01 /
  data (epsData(92,i,12), i=1,3) / 0.5805770000E+02, 0.3547690000E+00, 0.1660760000E+01 /
  data (epsData(93,i,12), i=1,3) / 0.6019680000E+02, 0.2040210000E+00, 0.1677380000E+01 /
  data (epsData(94,i,12), i=1,3) / 0.6478030000E+02, 0.5396680000E-01, 0.1697380000E+01 /
  data (epsData(95,i,12), i=1,3) / 0.7070010000E+02, -0.1344610000E+00, 0.1751020000E+01 /
  data (epsData(96,i,12), i=1,3) / 0.7587880000E+02, -0.3423130000E+00, 0.1699070000E+01 /
  data (epsData(97,i,12), i=1,3) / 0.8162950000E+02, -0.4924380000E+00, 0.1544140000E+01 /
  data (epsData(98,i,12), i=1,3) / 0.8704630000E+02, -0.5328560000E+00, 0.1401710000E+01 /
  data (epsData(99,i,12), i=1,3) / 0.8863920000E+02, -0.5110190000E+00, 0.1409420000E+01 /
  data (epsData(100,i,12), i=1,3) / 0.8984420000E+02, -0.5768650000E+00, 0.1423330000E+01 /
  data (epsData(101,i,12), i=1,3) / 0.9319840000E+02, -0.6501170000E+00, 0.1308830000E+01 /
  data (epsData(102,i,12), i=1,3) / 0.1025790000E+03, -0.6940660000E+00, 0.1084690000E+01 /
  data (epsData(103,i,12), i=1,3) / 0.1167170000E+03, -0.7043380000E+00, 0.8112690000E+00 /
  data (epsData(104,i,12), i=1,3) / 0.1308450000E+03, -0.6512010000E+00, 0.6032720000E+00 /
  data (epsData(105,i,12), i=1,3) / 0.1498500000E+03, -0.5629550000E+00, 0.4202700000E+00 /
  data (epsData(106,i,12), i=1,3) / 0.1707020000E+03, -0.4736130000E+00, 0.2886540000E+00 /
  data (epsData(107,i,12), i=1,3) / 0.1915940000E+03, -0.3955300000E+00, 0.2048090000E+00 /
  data (epsData(108,i,12), i=1,3) / 0.2169340000E+03, -0.3221950000E+00, 0.1398340000E+00 /
  data (epsData(109,i,12), i=1,3) / 0.2436800000E+03, -0.2624890000E+00, 0.9653160000E-01 /
  data (epsData(110,i,12), i=1,3) / 0.2719570000E+03, -0.2142310000E+00, 0.6766300000E-01 /
  data (epsData(111,i,12), i=1,3) / 0.3036830000E+03, -0.1738620000E+00, 0.4688540000E-01 /
  data (epsData(112,i,12), i=1,3) / 0.3367770000E+03, -0.1420870000E+00, 0.3317690000E-01 /
  data (epsData(113,i,12), i=1,3) / 0.3734830000E+03, -0.1158530000E+00, 0.2332180000E-01 /
  data (epsData(114,i,12), i=1,3) / 0.4120560000E+03, -0.9510660000E-01, 0.1664090000E-01 /
  data (epsData(115,i,12), i=1,3) / 0.4523750000E+03, -0.7867800000E-01, 0.1217820000E-01 /
  data (epsData(116,i,12), i=1,3) / 0.5062930000E+03, -0.6273860000E-01, 0.8166880000E-02 /
  data (epsData(117,i,12), i=1,3) / 0.5610940000E+03, -0.5079540000E-01, 0.5746460000E-02 /
  data (epsData(118,i,12), i=1,3) / 0.6224520000E+03, -0.4099250000E-01, 0.3951060000E-02 /
  data (epsData(119,i,12), i=1,3) / 0.6948820000E+03, -0.3261470000E-01, 0.2641320000E-02 /
  data (epsData(120,i,12), i=1,3) / 0.7723750000E+03, -0.2612410000E-01, 0.1819820000E-02 /
  data (epsData(121,i,12), i=1,3) / 0.8572370000E+03, -0.2091530000E-01, 0.1247440000E-02 /
  data (epsData(122,i,12), i=1,3) / 0.9531890000E+03, -0.1659570000E-01, 0.8454630000E-03 /
  data (epsData(123,i,12), i=1,3) / 0.1066120000E+04, -0.1284720000E-01, 0.5603270000E-03 /
  data (epsData(124,i,12), i=1,3) / 0.1254330000E+04, -0.7878720000E-02, 0.3083140000E-03 /
  data (epsData(125,i,12), i=1,3) / 0.1280140000E+04, -0.6830590000E-02, 0.2845450000E-03 /
  data (epsData(126,i,12), i=1,3) / 0.1291200000E+04, -0.6048080000E-02, 0.2904890000E-03 /
  data (epsData(127,i,12), i=1,3) / 0.1296250000E+04, -0.5408330000E-02, 0.3001850000E-03 /
  data (epsData(128,i,12), i=1,3) / 0.1298280000E+04, -0.4990880000E-02, 0.3131960000E-03 /
  data (epsData(129,i,12), i=1,3) / 0.1299290000E+04, -0.4698540000E-02, 0.3261860000E-03 /
  data (epsData(130,i,12), i=1,3) / 0.1300300000E+04, -0.4292910000E-02, 0.3509010000E-03 /
  data (epsData(131,i,12), i=1,3) / 0.1301320000E+04, -0.3634610000E-02, 0.4095050000E-03 /
  data (epsData(132,i,12), i=1,3) / 0.1302340000E+04, -0.2293820000E-02, 0.6750320000E-03 /
  data (epsData(133,i,12), i=1,3) / 0.1303010000E+04, -0.1414960000E-02, 0.2740300000E-02 /
  data (epsData(134,i,12), i=1,3) / 0.1303040000E+04, -0.1386870000E-02, 0.2933380000E-02 /
  data (epsData(135,i,12), i=1,3) / 0.1303110000E+04, -0.1443020000E-02, 0.3342840000E-02 /
  data (epsData(136,i,12), i=1,3) / 0.1303150000E+04, -0.1533280000E-02, 0.3578270000E-02 /
  data (epsData(137,i,12), i=1,3) / 0.1303200000E+04, -0.1727960000E-02, 0.3849940000E-02 /
  data (epsData(138,i,12), i=1,3) / 0.1303210000E+04, -0.1827200000E-02, 0.3896090000E-02 /
  data (epsData(139,i,12), i=1,3) / 0.1303540000E+04, -0.2891200000E-02, 0.4407950000E-02 /
  data (epsData(140,i,12), i=1,3) / 0.1303990000E+04, -0.3910240000E-02, 0.4520130000E-02 /
  data (epsData(141,i,12), i=1,3) / 0.1304390000E+04, -0.4633010000E-02, 0.4366900000E-02 /
  data (epsData(142,i,12), i=1,3) / 0.1304960000E+04, -0.5181390000E-02, 0.3909660000E-02 /
  data (epsData(143,i,12), i=1,3) / 0.1305760000E+04, -0.5393290000E-02, 0.3509940000E-02 /
  data (epsData(144,i,12), i=1,3) / 0.1307980000E+04, -0.5704120000E-02, 0.3103060000E-02 /
  data (epsData(145,i,12), i=1,3) / 0.1313180000E+04, -0.5830260000E-02, 0.2996930000E-02 /
  data (epsData(146,i,12), i=1,3) / 0.1320530000E+04, -0.6224110000E-02, 0.3273950000E-02 /
  data (epsData(147,i,12), i=1,3) / 0.1331410000E+04, -0.6701450000E-02, 0.3179420000E-02 /
  data (epsData(148,i,12), i=1,3) / 0.1373730000E+04, -0.7260200000E-02, 0.2918570000E-02 /
  data (epsData(149,i,12), i=1,3) / 0.1463070000E+04, -0.7132590000E-02, 0.2315110000E-02 /
  data (epsData(150,i,12), i=1,3) / 0.1640440000E+04, -0.6153140000E-02, 0.1539960000E-02 /
  data (epsData(151,i,12), i=1,3) / 0.1862050000E+04, -0.4983570000E-02, 0.9897640000E-03 /
  data (epsData(152,i,12), i=1,3) / 0.2090960000E+04, -0.4039960000E-02, 0.6571810000E-03 /
  data (epsData(153,i,12), i=1,3) / 0.2336410000E+04, -0.3274940000E-02, 0.4407240000E-03 /
  data (epsData(154,i,12), i=1,3) / 0.2608410000E+04, -0.2644980000E-02, 0.2946800000E-03 /
  data (epsData(155,i,12), i=1,3) / 0.2895910000E+04, -0.2152560000E-02, 0.2023270000E-03 /
  data (epsData(156,i,12), i=1,3) / 0.3231400000E+04, -0.1731660000E-02, 0.1342750000E-03 /
  data (epsData(157,i,12), i=1,3) / 0.3602450000E+04, -0.1392960000E-02, 0.8898940000E-04 /
  data (epsData(158,i,12), i=1,3) / 0.4019580000E+04, -0.1117330000E-02, 0.5859590000E-04 /
  data (epsData(159,i,12), i=1,3) / 0.4483920000E+04, -0.8961170000E-03, 0.3846800000E-04 /
  data (epsData(160,i,12), i=1,3) / 0.5005010000E+04, -0.7175570000E-03, 0.2511620000E-04 /
  data (epsData(161,i,12), i=1,3) / 0.5589660000E+04, -0.5738670000E-03, 0.1631620000E-04 /
  data (epsData(162,i,12), i=1,3) / 0.6245630000E+04, -0.4584730000E-03, 0.1056690000E-04 /
  data (epsData(163,i,12), i=1,3) / 0.6987140000E+04, -0.3654790000E-03, 0.6794520000E-05 /
  data (epsData(164,i,12), i=1,3) / 0.7813140000E+04, -0.2916500000E-03, 0.4353920000E-05 /
  data (epsData(165,i,12), i=1,3) / 0.8742800000E+04, -0.2324360000E-03, 0.2767620000E-05 /
  data (epsData(166,i,12), i=1,3) / 0.9785360000E+04, -0.1852050000E-03, 0.1758550000E-05 /
  data (epsData(167,i,12), i=1,3) / 0.1095780000E+05, -0.1474470000E-03, 0.1112150000E-05 /
  data (epsData(168,i,12), i=1,3) / 0.1227160000E+05, -0.1173920000E-03, 0.7006530000E-06 /
  data (epsData(169,i,12), i=1,3) / 0.1374720000E+05, -0.9342210000E-04, 0.4404170000E-06 /
  data (epsData(170,i,12), i=1,3) / 0.1540790000E+05, -0.7428700000E-04, 0.2754190000E-06 /
  data (epsData(171,i,12), i=1,3) / 0.1727090000E+05, -0.5906820000E-04, 0.1715370000E-06 /
  data (epsData(172,i,12), i=1,3) / 0.1936330000E+05, -0.4695360000E-04, 0.1069980000E-06 /
  data (epsData(173,i,12), i=1,3) / 0.2171070000E+05, -0.3732320000E-04, 0.6645530000E-07 /
  data (epsData(174,i,12), i=1,3) / 0.2434700000E+05, -0.2966090000E-04, 0.4110600000E-07 /
  data (epsData(175,i,12), i=1,3) / 0.2730370000E+05, -0.2357330000E-04, 0.2536720000E-07 /
  data (epsData(176,i,12), i=1,3) / 0.3062800000E+05, -0.1872600000E-04, 0.1566060000E-07 /
  data (epsData(177,i,12), i=1,3) / 0.3435320000E+05, -0.1488000000E-04, 0.9647720000E-08 /
  data (epsData(178,i,12), i=1,3) / 0.3853560000E+05, -0.1182190000E-04, 0.5934960000E-08 /
  data (epsData(179,i,12), i=1,3) / 0.4323110000E+05, -0.9391200000E-05, 0.3621140000E-08 /
  data (epsData(180,i,12), i=1,3) / 0.4850140000E+05, -0.7459640000E-05, 0.2308010000E-08 /
  data (epsData(181,i,12), i=1,3) / 0.5440630000E+05, -0.5927350000E-05, 0.1352980000E-08 /
  data (epsData(182,i,12), i=1,3) / 0.6104310000E+05, -0.4707920000E-05, 0.8948890000E-09 /
  data (epsData(183,i,12), i=1,3) / 0.6848810000E+05, -0.3739620000E-05, 0.5073230000E-09 /
  data (epsData(184,i,12), i=1,3) / 0.7684580000E+05, -0.2970150000E-05, 0.3280780000E-09 /
  data (epsData(185,i,12), i=1,3) / 0.8622950000E+05, -0.2358750000E-05, 0.1941170000E-09 /
  data (epsData(186,i,12), i=1,3) / 0.9674240000E+05, -0.1873880000E-05, 0.1237950000E-09 /
  data (epsData(187,i,12), i=1,3) / 0.1000000000E+06, -0.1754240000E-05, 0.1154170000E-09 /
  data (epsData(1,i,13), i=1,3) / 0.2506580000E-02, -0.1503660000E+04, 0.4054640000E+04 /
  data (epsData(2,i,13), i=1,3) / 0.4693440000E-02, -0.1202850000E+04, 0.1357010000E+04 /
  data (epsData(3,i,13), i=1,3) / 0.7560590000E-02, -0.4825670000E+03, 0.6242630000E+03 /
  data (epsData(4,i,13), i=1,3) / 0.1110800000E-01, -0.1692460000E+03, 0.3472820000E+03 /
  data (epsData(5,i,13), i=1,3) / 0.1533570000E-01, -0.2205660000E+02, 0.2185120000E+03 /
  data (epsData(6,i,13), i=1,3) / 0.2024370000E-01, 0.5405190000E+02, 0.1495940000E+03 /
  data (epsData(7,i,13), i=1,3) / 0.2583200000E-01, 0.9664700000E+02, 0.1087880000E+03 /
  data (epsData(8,i,13), i=1,3) / 0.3210060000E-01, 0.1220490000E+03, 0.8273750000E+02 /
  data (epsData(9,i,13), i=1,3) / 0.3904940000E-01, 0.1380030000E+03, 0.6511670000E+02 /
  data (epsData(10,i,13), i=1,3) / 0.4667860000E-01, 0.1484640000E+03, 0.5264380000E+02 /
  data (epsData(11,i,13), i=1,3) / 0.5498800000E-01, 0.1555850000E+03, 0.4348690000E+02 /
  data (epsData(12,i,13), i=1,3) / 0.7364770000E-01, 0.1642470000E+03, 0.3119190000E+02 /
  data (epsData(13,i,13), i=1,3) / 0.1067390000E+00, 0.1710870000E+03, 0.2117730000E+02 /
  data (epsData(14,i,13), i=1,3) / 0.1912910000E+00, 0.1743550000E+03, 0.1306060000E+02 /
  data (epsData(15,i,13), i=1,3) / 0.4338630000E+00, 0.1781310000E+03, 0.6461130000E+01 /
  data (epsData(16,i,13), i=1,3) / 0.8406760000E+00, 0.1895230000E+03, 0.4752030000E+01 /
  data (epsData(17,i,13), i=1,3) / 0.1172610000E+01, 0.2082360000E+03, 0.4980770000E+01 /
  data (epsData(18,i,13), i=1,3) / 0.1424510000E+01, 0.2336400000E+03, 0.6402250000E+01 /
  data (epsData(19,i,13), i=1,3) / 0.1559640000E+01, 0.2553780000E+03, 0.7307910000E+01 /
  data (epsData(20,i,13), i=1,3) / 0.1700900000E+01, 0.2900560000E+03, 0.1576890000E+02 /
  data (epsData(21,i,13), i=1,3) / 0.1798470000E+01, 0.3236970000E+03, 0.2199850000E+02 /
  data (epsData(22,i,13), i=1,3) / 0.1848280000E+01, 0.3531640000E+03, 0.2465460000E+02 /
  data (epsData(23,i,13), i=1,3) / 0.1898770000E+01, 0.4101050000E+03, 0.2703580000E+02 /
  data (epsData(24,i,13), i=1,3) / 0.1949930000E+01, 0.4510980000E+03, 0.1296570000E+03 /
  data (epsData(25,i,13), i=1,3) / 0.2001360000E+01, 0.4278810000E+03, 0.2232480000E+03 /
  data (epsData(26,i,13), i=1,3) / 0.2076530000E+01, 0.3671990000E+03, 0.2712380000E+03 /
  data (epsData(27,i,13), i=1,3) / 0.2179940000E+01, 0.2895360000E+03, 0.3258450000E+03 /
  data (epsData(28,i,13), i=1,3) / 0.2212590000E+01, 0.2523680000E+03, 0.3318620000E+03 /
  data (epsData(29,i,13), i=1,3) / 0.2286060000E+01, 0.2007470000E+03, 0.3138500000E+03 /
  data (epsData(30,i,13), i=1,3) / 0.2416670000E+01, 0.1523020000E+03, 0.2855830000E+03 /
  data (epsData(31,i,13), i=1,3) / 0.2517360000E+01, 0.1330290000E+03, 0.2679820000E+03 /
  data (epsData(32,i,13), i=1,3) / 0.2783690000E+01, 0.9775460000E+02, 0.2439380000E+03 /
  data (epsData(33,i,13), i=1,3) / 0.3224510000E+01, 0.6104940000E+02, 0.2231940000E+03 /
  data (epsData(34,i,13), i=1,3) / 0.3572820000E+01, 0.4166340000E+02, 0.2110640000E+03 /
  data (epsData(35,i,13), i=1,3) / 0.3862620000E+01, 0.3333550000E+02, 0.2113380000E+03 /
  data (epsData(36,i,13), i=1,3) / 0.3913300000E+01, 0.3498720000E+02, 0.2143390000E+03 /
  data (epsData(37,i,13), i=1,3) / 0.3964660000E+01, 0.3301200000E+02, 0.2260880000E+03 /
  data (epsData(38,i,13), i=1,3) / 0.4012250000E+01, 0.2287750000E+02, 0.2342840000E+03 /
  data (epsData(39,i,13), i=1,3) / 0.4195920000E+01, -0.1650450000E+02, 0.2406600000E+03 /
  data (epsData(40,i,13), i=1,3) / 0.4306130000E+01, -0.4069890000E+02, 0.2266120000E+03 /
  data (epsData(41,i,13), i=1,3) / 0.4491170000E+01, -0.6151050000E+02, 0.2015080000E+03 /
  data (epsData(42,i,13), i=1,3) / 0.4783690000E+01, -0.7864800000E+02, 0.1674330000E+03 /
  data (epsData(43,i,13), i=1,3) / 0.5144240000E+01, -0.8702570000E+02, 0.1320420000E+03 /
  data (epsData(44,i,13), i=1,3) / 0.5437100000E+01, -0.8588570000E+02, 0.1055620000E+03 /
  data (epsData(45,i,13), i=1,3) / 0.5666700000E+01, -0.7939620000E+02, 0.9204860000E+02 /
  data (epsData(46,i,13), i=1,3) / 0.6016700000E+01, -0.7274670000E+02, 0.7900410000E+02 /
  data (epsData(47,i,13), i=1,3) / 0.6941890000E+01, -0.6414130000E+02, 0.5325720000E+02 /
  data (epsData(48,i,13), i=1,3) / 0.8041230000E+01, -0.5445790000E+02, 0.3207300000E+02 /
  data (epsData(49,i,13), i=1,3) / 0.8980020000E+01, -0.4552740000E+02, 0.1997330000E+02 /
  data (epsData(50,i,13), i=1,3) / 0.9837180000E+01, -0.3798400000E+02, 0.1295190000E+02 /
  data (epsData(51,i,13), i=1,3) / 0.1066680000E+02, -0.3174110000E+02, 0.8649170000E+01 /
  data (epsData(52,i,13), i=1,3) / 0.1155150000E+02, -0.2630920000E+02, 0.5805390000E+01 /
  data (epsData(53,i,13), i=1,3) / 0.1249130000E+02, -0.2175340000E+02, 0.3978060000E+01 /
  data (epsData(54,i,13), i=1,3) / 0.1348620000E+02, -0.1802200000E+02, 0.2821840000E+01 /
  data (epsData(55,i,13), i=1,3) / 0.1465630000E+02, -0.1468740000E+02, 0.2035260000E+01 /
  data (epsData(56,i,13), i=1,3) / 0.1589440000E+02, -0.1205450000E+02, 0.1561770000E+01 /
  data (epsData(57,i,13), i=1,3) / 0.1733490000E+02, -0.9790650000E+01, 0.1253430000E+01 /
  data (epsData(58,i,13), i=1,3) / 0.1900030000E+02, -0.7898910000E+01, 0.1050750000E+01 /
  data (epsData(59,i,13), i=1,3) / 0.2091490000E+02, -0.6350370000E+01, 0.9022480000E+00 /
  data (epsData(60,i,13), i=1,3) / 0.2326700000E+02, -0.5013720000E+01, 0.7592110000E+00 /
  data (epsData(61,i,13), i=1,3) / 0.2594460000E+02, -0.3940460000E+01, 0.6199790000E+00 /
  data (epsData(62,i,13), i=1,3) / 0.2898040000E+02, -0.3065850000E+01, 0.4911120000E+00 /
  data (epsData(63,i,13), i=1,3) / 0.3221270000E+02, -0.2382720000E+01, 0.3881180000E+00 /
  data (epsData(64,i,13), i=1,3) / 0.3564170000E+02, -0.1841510000E+01, 0.3115750000E+00 /
  data (epsData(65,i,13), i=1,3) / 0.3926730000E+02, -0.1409150000E+01, 0.2566570000E+00 /
  data (epsData(66,i,13), i=1,3) / 0.4308940000E+02, -0.1060580000E+01, 0.2170530000E+00 /
  data (epsData(67,i,13), i=1,3) / 0.4735070000E+02, -0.7607300000E+00, 0.1849140000E+00 /
  data (epsData(68,i,13), i=1,3) / 0.5208790000E+02, -0.4975290000E+00, 0.1554010000E+00 /
  data (epsData(69,i,13), i=1,3) / 0.5760940000E+02, -0.2419250000E+00, 0.1248280000E+00 /
  data (epsData(70,i,13), i=1,3) / 0.6314700000E+02, -0.1211910000E-02, 0.9849410000E-01 /
  data (epsData(71,i,13), i=1,3) / 0.6572780000E+02, 0.1176440000E+00, 0.8913160000E-01 /
  data (epsData(72,i,13), i=1,3) / 0.6806800000E+02, 0.2382090000E+00, 0.8176700000E-01 /
  data (epsData(73,i,13), i=1,3) / 0.6985180000E+02, 0.3482760000E+00, 0.7715350000E-01 /
  data (epsData(74,i,13), i=1,3) / 0.7105450000E+02, 0.4406320000E+00, 0.7499790000E-01 /
  data (epsData(75,i,13), i=1,3) / 0.7196370000E+02, 0.5354500000E+00, 0.7464400000E-01 /
  data (epsData(76,i,13), i=1,3) / 0.7226820000E+02, 0.5810260000E+00, 0.7776830000E-01 /
  data (epsData(77,i,13), i=1,3) / 0.7268610000E+02, 0.6403040000E+00, 0.1201170000E+00 /
  data (epsData(78,i,13), i=1,3) / 0.7346730000E+02, 0.7041040000E+00, 0.1763200000E+00 /
  data (epsData(79,i,13), i=1,3) / 0.7452280000E+02, 0.8147780000E+00, 0.2726740000E+00 /
  data (epsData(80,i,13), i=1,3) / 0.7552450000E+02, 0.8827180000E+00, 0.4118360000E+00 /
  data (epsData(81,i,13), i=1,3) / 0.7662900000E+02, 0.9274540000E+00, 0.5521450000E+00 /
  data (epsData(82,i,13), i=1,3) / 0.7768610000E+02, 0.9196870000E+00, 0.7353290000E+00 /
  data (epsData(83,i,13), i=1,3) / 0.7887660000E+02, 0.8281480000E+00, 0.8940070000E+00 /
  data (epsData(84,i,13), i=1,3) / 0.8060320000E+02, 0.6656000000E+00, 0.1005930000E+01 /
  data (epsData(85,i,13), i=1,3) / 0.8266480000E+02, 0.5230150000E+00, 0.1040130000E+01 /
  data (epsData(86,i,13), i=1,3) / 0.8669210000E+02, 0.3772720000E+00, 0.1064850000E+01 /
  data (epsData(87,i,13), i=1,3) / 0.9270580000E+02, 0.1917460000E+00, 0.1116170000E+01 /
  data (epsData(88,i,13), i=1,3) / 0.9738030000E+02, 0.4837510000E-01, 0.1093320000E+01 /
  data (epsData(89,i,13), i=1,3) / 0.1035630000E+03, -0.7790690000E-01, 0.1018780000E+01 /
  data (epsData(90,i,13), i=1,3) / 0.1122700000E+03, -0.1836720000E+00, 0.9085990000E+00 /
  data (epsData(91,i,13), i=1,3) / 0.1163010000E+03, -0.1930290000E+00, 0.8606900000E+00 /
  data (epsData(92,i,13), i=1,3) / 0.1175240000E+03, -0.1829200000E+00, 0.8685570000E+00 /
  data (epsData(93,i,13), i=1,3) / 0.1184590000E+03, -0.2077840000E+00, 0.8858770000E+00 /
  data (epsData(94,i,13), i=1,3) / 0.1227800000E+03, -0.2709930000E+00, 0.8321340000E+00 /
  data (epsData(95,i,13), i=1,3) / 0.1367450000E+03, -0.3599750000E+00, 0.6700670000E+00 /
  data (epsData(96,i,13), i=1,3) / 0.1498580000E+03, -0.3781730000E+00, 0.5336660000E+00 /
  data (epsData(97,i,13), i=1,3) / 0.1656320000E+03, -0.3654180000E+00, 0.4106240000E+00 /
  data (epsData(98,i,13), i=1,3) / 0.1876010000E+03, -0.3313920000E+00, 0.2907680000E+00 /
  data (epsData(99,i,13), i=1,3) / 0.2111960000E+03, -0.2862140000E+00, 0.2041630000E+00 /
  data (epsData(100,i,13), i=1,3) / 0.2406620000E+03, -0.2367540000E+00, 0.1373950000E+00 /
  data (epsData(101,i,13), i=1,3) / 0.2686750000E+03, -0.1978550000E+00, 0.9703360000E-01 /
  data (epsData(102,i,13), i=1,3) / 0.2980740000E+03, -0.1650440000E+00, 0.6928840000E-01 /
  data (epsData(103,i,13), i=1,3) / 0.3295500000E+03, -0.1373630000E+00, 0.4979340000E-01 /
  data (epsData(104,i,13), i=1,3) / 0.3635590000E+03, -0.1141280000E+00, 0.3583730000E-01 /
  data (epsData(105,i,13), i=1,3) / 0.4000980000E+03, -0.9476230000E-01, 0.2589560000E-01 /
  data (epsData(106,i,13), i=1,3) / 0.4411560000E+03, -0.7813250000E-01, 0.1852350000E-01 /
  data (epsData(107,i,13), i=1,3) / 0.4828840000E+03, -0.6515650000E-01, 0.1372640000E-01 /
  data (epsData(108,i,13), i=1,3) / 0.5414450000E+03, -0.5174360000E-01, 0.9126340000E-02 /
  data (epsData(109,i,13), i=1,3) / 0.6027340000E+03, -0.4164960000E-01, 0.6293470000E-02 /
  data (epsData(110,i,13), i=1,3) / 0.6629820000E+03, -0.3420280000E-01, 0.4488880000E-02 /
  data (epsData(111,i,13), i=1,3) / 0.7383670000E+03, -0.2735360000E-01, 0.3044690000E-02 /
  data (epsData(112,i,13), i=1,3) / 0.8232690000E+03, -0.2179550000E-01, 0.2048360000E-02 /
  data (epsData(113,i,13), i=1,3) / 0.9149400000E+03, -0.1745160000E-01, 0.1404700000E-02 /
  data (epsData(114,i,13), i=1,3) / 0.1014770000E+04, -0.1398280000E-01, 0.9663640000E-03 /
  data (epsData(115,i,13), i=1,3) / 0.1126440000E+04, -0.1112880000E-01, 0.6597010000E-03 /
  data (epsData(116,i,13), i=1,3) / 0.1254010000E+04, -0.8710780000E-02, 0.4442500000E-03 /
  data (epsData(117,i,13), i=1,3) / 0.1426750000E+04, -0.6199820000E-02, 0.2748560000E-03 /
  data (epsData(118,i,13), i=1,3) / 0.1508660000E+04, -0.4938120000E-02, 0.2232860000E-03 /
  data (epsData(119,i,13), i=1,3) / 0.1535150000E+04, -0.4286800000E-02, 0.2096780000E-03 /
  data (epsData(120,i,13), i=1,3) / 0.1546940000E+04, -0.3777380000E-02, 0.2138080000E-03 /
  data (epsData(121,i,13), i=1,3) / 0.1552320000E+04, -0.3365710000E-02, 0.2197990000E-03 /
  data (epsData(122,i,13), i=1,3) / 0.1554480000E+04, -0.3101320000E-02, 0.2277310000E-03 /
  data (epsData(123,i,13), i=1,3) / 0.1555560000E+04, -0.2918640000E-02, 0.2355750000E-03 /
  data (epsData(124,i,13), i=1,3) / 0.1556640000E+04, -0.2668540000E-02, 0.2502210000E-03 /
  data (epsData(125,i,13), i=1,3) / 0.1557730000E+04, -0.2268370000E-02, 0.2840810000E-03 /
  data (epsData(126,i,13), i=1,3) / 0.1558810000E+04, -0.1468580000E-02, 0.4317090000E-03 /
  data (epsData(127,i,13), i=1,3) / 0.1559610000E+04, -0.9751080000E-03, 0.1880080000E-02 /
  data (epsData(128,i,13), i=1,3) / 0.1559650000E+04, -0.9704480000E-03, 0.2026160000E-02 /
  data (epsData(129,i,13), i=1,3) / 0.1559740000E+04, -0.1072620000E-02, 0.2340350000E-02 /
  data (epsData(130,i,13), i=1,3) / 0.1559780000E+04, -0.1186210000E-02, 0.2498360000E-02 /
  data (epsData(131,i,13), i=1,3) / 0.1559810000E+04, -0.1328270000E-02, 0.2576690000E-02 /
  data (epsData(132,i,13), i=1,3) / 0.1560120000E+04, -0.2033210000E-02, 0.2696940000E-02 /
  data (epsData(133,i,13), i=1,3) / 0.1560380000E+04, -0.2400930000E-02, 0.2543600000E-02 /
  data (epsData(134,i,13), i=1,3) / 0.1560820000E+04, -0.2659120000E-02, 0.2352620000E-02 /
  data (epsData(135,i,13), i=1,3) / 0.1561670000E+04, -0.2845300000E-02, 0.2205580000E-02 /
  data (epsData(136,i,13), i=1,3) / 0.1563790000E+04, -0.3149010000E-02, 0.2152610000E-02 /
  data (epsData(137,i,13), i=1,3) / 0.1571880000E+04, -0.3665730000E-02, 0.2254860000E-02 /
  data (epsData(138,i,13), i=1,3) / 0.1578710000E+04, -0.4012230000E-02, 0.2234450000E-02 /
  data (epsData(139,i,13), i=1,3) / 0.1605910000E+04, -0.4420940000E-02, 0.2144010000E-02 /
  data (epsData(140,i,13), i=1,3) / 0.1641600000E+04, -0.4667700000E-02, 0.2007690000E-02 /
  data (epsData(141,i,13), i=1,3) / 0.1732040000E+04, -0.4677190000E-02, 0.1662240000E-02 /
  data (epsData(142,i,13), i=1,3) / 0.1913490000E+04, -0.4176080000E-02, 0.1162410000E-02 /
  data (epsData(143,i,13), i=1,3) / 0.2178610000E+04, -0.3393510000E-02, 0.7374020000E-03 /
  data (epsData(144,i,13), i=1,3) / 0.2446800000E+04, -0.2758180000E-02, 0.4881140000E-03 /
  data (epsData(145,i,13), i=1,3) / 0.2736400000E+04, -0.2236020000E-02, 0.3262200000E-03 /
  data (epsData(146,i,13), i=1,3) / 0.3059680000E+04, -0.1799400000E-02, 0.2169300000E-03 /
  data (epsData(147,i,13), i=1,3) / 0.3434910000E+04, -0.1436010000E-02, 0.1428040000E-03 /
  data (epsData(148,i,13), i=1,3) / 0.3829080000E+04, -0.1157180000E-02, 0.9498290000E-04 /
  data (epsData(149,i,13), i=1,3) / 0.4268490000E+04, -0.9307720000E-03, 0.6293470000E-04 /
  data (epsData(150,i,13), i=1,3) / 0.4759270000E+04, -0.7475620000E-03, 0.4150900000E-04 /
  data (epsData(151,i,13), i=1,3) / 0.5308880000E+04, -0.5994750000E-03, 0.2724150000E-04 /
  data (epsData(152,i,13), i=1,3) / 0.5923870000E+04, -0.4802320000E-03, 0.1779900000E-04 /
  data (epsData(153,i,13), i=1,3) / 0.6609100000E+04, -0.3848320000E-03, 0.1166140000E-04 /
  data (epsData(154,i,13), i=1,3) / 0.7380800000E+04, -0.3077580000E-03, 0.7519920000E-05 /
  data (epsData(155,i,13), i=1,3) / 0.8252520000E+04, -0.2455270000E-03, 0.4841300000E-05 /
  data (epsData(156,i,13), i=1,3) / 0.9226970000E+04, -0.1959420000E-03, 0.3095080000E-05 /
  data (epsData(157,i,13), i=1,3) / 0.1032280000E+05, -0.1562020000E-03, 0.1975780000E-05 /
  data (epsData(158,i,13), i=1,3) / 0.1155430000E+05, -0.1244280000E-03, 0.1253420000E-05 /
  data (epsData(159,i,13), i=1,3) / 0.1293750000E+05, -0.9906690000E-04, 0.7912780000E-06 /
  data (epsData(160,i,13), i=1,3) / 0.1448980000E+05, -0.7885230000E-04, 0.4988980000E-06 /
  data (epsData(161,i,13), i=1,3) / 0.1623540000E+05, -0.6272190000E-04, 0.3127690000E-06 /
  data (epsData(162,i,13), i=1,3) / 0.1819160000E+05, -0.4989830000E-04, 0.1956960000E-06 /
  data (epsData(163,i,13), i=1,3) / 0.2039150000E+05, -0.3967230000E-04, 0.1219560000E-06 /
  data (epsData(164,i,13), i=1,3) / 0.2285700000E+05, -0.3154770000E-04, 0.7581710000E-07 /
  data (epsData(165,i,13), i=1,3) / 0.2562840000E+05, -0.2507500000E-04, 0.4702290000E-07 /
  data (epsData(166,i,13), i=1,3) / 0.2873930000E+05, -0.1992780000E-04, 0.2911290000E-07 /
  data (epsData(167,i,13), i=1,3) / 0.3223280000E+05, -0.1583400000E-04, 0.1797600000E-07 /
  data (epsData(168,i,13), i=1,3) / 0.3614980000E+05, -0.1258300000E-04, 0.1100820000E-07 /
  data (epsData(169,i,13), i=1,3) / 0.4054940000E+05, -0.9996900000E-05, 0.6792850000E-08 /
  data (epsData(170,i,13), i=1,3) / 0.4549030000E+05, -0.7940870000E-05, 0.4191010000E-08 /
  data (epsData(171,i,13), i=1,3) / 0.5102900000E+05, -0.6309030000E-05, 0.2539210000E-08 /
  data (epsData(172,i,13), i=1,3) / 0.5725280000E+05, -0.5010890000E-05, 0.1583520000E-08 /
  data (epsData(173,i,13), i=1,3) / 0.6423630000E+05, -0.3979930000E-05, 0.9603290000E-09 /
  data (epsData(174,i,13), i=1,3) / 0.7207210000E+05, -0.3161110000E-05, 0.5761430000E-09 /
  data (epsData(175,i,13), i=1,3) / 0.8086680000E+05, -0.2510670000E-05, 0.3709550000E-09 /
  data (epsData(176,i,13), i=1,3) / 0.9072780000E+05, -0.1994410000E-05, 0.2108940000E-09 /
  data (epsData(177,i,13), i=1,3) / 0.1000000000E+06, -0.1642110000E-05, 0.1479950000E-09 /
  data (epsData(1,i,14), i=1,3) / 0.2506580000E-02, -0.1201230000E+04, 0.3156070000E+04 /
  data (epsData(2,i,14), i=1,3) / 0.4693440000E-02, -0.9452530000E+03, 0.1045400000E+04 /
  data (epsData(3,i,14), i=1,3) / 0.7560590000E-02, -0.3723450000E+03, 0.4762310000E+03 /
  data (epsData(4,i,14), i=1,3) / 0.1110800000E-01, -0.1236030000E+03, 0.2626820000E+03 /
  data (epsData(5,i,14), i=1,3) / 0.1533570000E-01, -0.6841540000E+01, 0.1641100000E+03 /
  data (epsData(6,i,14), i=1,3) / 0.2024370000E-01, 0.5350610000E+02, 0.1117000000E+03 /
  data (epsData(7,i,14), i=1,3) / 0.2583200000E-01, 0.8726990000E+02, 0.8084980000E+02 /
  data (epsData(8,i,14), i=1,3) / 0.3210060000E-01, 0.1074010000E+03, 0.6125530000E+02 /
  data (epsData(9,i,14), i=1,3) / 0.3904940000E-01, 0.1200400000E+03, 0.4806040000E+02 /
  data (epsData(10,i,14), i=1,3) / 0.4667860000E-01, 0.1283270000E+03, 0.3875620000E+02 /
  data (epsData(11,i,14), i=1,3) / 0.5498800000E-01, 0.1339650000E+03, 0.3194810000E+02 /
  data (epsData(12,i,14), i=1,3) / 0.7364770000E-01, 0.1408180000E+03, 0.2284240000E+02 /
  data (epsData(13,i,14), i=1,3) / 0.1067390000E+00, 0.1462080000E+03, 0.1547910000E+02 /
  data (epsData(14,i,14), i=1,3) / 0.1912910000E+00, 0.1487690000E+03, 0.9589420000E+01 /
  data (epsData(15,i,14), i=1,3) / 0.4585000000E+00, 0.1517680000E+03, 0.4570290000E+01 /
  data (epsData(16,i,14), i=1,3) / 0.9096770000E+00, 0.1613360000E+03, 0.3293900000E+01 /
  data (epsData(17,i,14), i=1,3) / 0.1295500000E+01, 0.1789100000E+03, 0.3452730000E+01 /
  data (epsData(18,i,14), i=1,3) / 0.1559640000E+01, 0.2016980000E+03, 0.4648770000E+01 /
  data (epsData(19,i,14), i=1,3) / 0.1749350000E+01, 0.2309330000E+03, 0.1035370000E+02 /
  data (epsData(20,i,14), i=1,3) / 0.1848280000E+01, 0.2579090000E+03, 0.1306130000E+02 /
  data (epsData(21,i,14), i=1,3) / 0.1898770000E+01, 0.2866690000E+03, 0.1420680000E+02 /
  data (epsData(22,i,14), i=1,3) / 0.1949930000E+01, 0.3100600000E+03, 0.5887090000E+02 /
  data (epsData(23,i,14), i=1,3) / 0.2005440000E+01, 0.3083680000E+03, 0.1020550000E+03 /
  data (epsData(24,i,14), i=1,3) / 0.2136060000E+01, 0.2761440000E+03, 0.1619600000E+03 /
  data (epsData(25,i,14), i=1,3) / 0.2195920000E+01, 0.2512920000E+03, 0.1844770000E+03 /
  data (epsData(26,i,14), i=1,3) / 0.2266670000E+01, 0.2221070000E+03, 0.1871750000E+03 /
  data (epsData(27,i,14), i=1,3) / 0.2465650000E+01, 0.1799900000E+03, 0.1885470000E+03 /
  data (epsData(28,i,14), i=1,3) / 0.2850360000E+01, 0.1341710000E+03, 0.1908290000E+03 /
  data (epsData(29,i,14), i=1,3) / 0.3714660000E+01, 0.6107470000E+02, 0.1943970000E+03 /
  data (epsData(30,i,14), i=1,3) / 0.4287120000E+01, 0.1559560000E+02, 0.1857990000E+03 /
  data (epsData(31,i,14), i=1,3) / 0.4694260000E+01, -0.1295730000E+02, 0.1703080000E+03 /
  data (epsData(32,i,14), i=1,3) / 0.5005500000E+01, -0.3109280000E+02, 0.1538190000E+03 /
  data (epsData(33,i,14), i=1,3) / 0.5266720000E+01, -0.4060370000E+02, 0.1368250000E+03 /
  data (epsData(34,i,14), i=1,3) / 0.5469790000E+01, -0.4190150000E+02, 0.1230020000E+03 /
  data (epsData(35,i,14), i=1,3) / 0.5678980000E+01, -0.4064870000E+02, 0.1163140000E+03 /
  data (epsData(36,i,14), i=1,3) / 0.6498380000E+01, -0.5040780000E+02, 0.9492290000E+02 /
  data (epsData(37,i,14), i=1,3) / 0.7400090000E+01, -0.5669100000E+02, 0.6911790000E+02 /
  data (epsData(38,i,14), i=1,3) / 0.8066670000E+01, -0.5549880000E+02, 0.5343540000E+02 /
  data (epsData(39,i,14), i=1,3) / 0.8884710000E+01, -0.5280000000E+02, 0.3718180000E+02 /
  data (epsData(40,i,14), i=1,3) / 0.9714660000E+01, -0.4675620000E+02, 0.2476950000E+02 /
  data (epsData(41,i,14), i=1,3) / 0.1063410000E+02, -0.3947040000E+02, 0.1576300000E+02 /
  data (epsData(42,i,14), i=1,3) / 0.1153880000E+02, -0.3303350000E+02, 0.1027250000E+02 /
  data (epsData(43,i,14), i=1,3) / 0.1249840000E+02, -0.2736430000E+02, 0.6657950000E+01 /
  data (epsData(44,i,14), i=1,3) / 0.1348620000E+02, -0.2269670000E+02, 0.4412360000E+01 /
  data (epsData(45,i,14), i=1,3) / 0.1457190000E+02, -0.1868370000E+02, 0.2946530000E+01 /
  data (epsData(46,i,14), i=1,3) / 0.1575560000E+02, -0.1534140000E+02, 0.2022460000E+01 /
  data (epsData(47,i,14), i=1,3) / 0.1703720000E+02, -0.1260250000E+02, 0.1454490000E+01 /
  data (epsData(48,i,14), i=1,3) / 0.1853620000E+02, -0.1022100000E+02, 0.1100810000E+01 /
  data (epsData(49,i,14), i=1,3) / 0.2015020000E+02, -0.8343570000E+01, 0.9074780000E+00 /
  data (epsData(50,i,14), i=1,3) / 0.2201690000E+02, -0.6771420000E+01, 0.7891370000E+00 /
  data (epsData(51,i,14), i=1,3) / 0.2431350000E+02, -0.5404420000E+01, 0.6970820000E+00 /
  data (epsData(52,i,14), i=1,3) / 0.2694450000E+02, -0.4305830000E+01, 0.6116310000E+00 /
  data (epsData(53,i,14), i=1,3) / 0.2994460000E+02, -0.3418330000E+01, 0.5283010000E+00 /
  data (epsData(54,i,14), i=1,3) / 0.3335040000E+02, -0.2700090000E+01, 0.4505920000E+00 /
  data (epsData(55,i,14), i=1,3) / 0.3720080000E+02, -0.2119840000E+01, 0.3791040000E+00 /
  data (epsData(56,i,14), i=1,3) / 0.4153660000E+02, -0.1648420000E+01, 0.3139940000E+00 /
  data (epsData(57,i,14), i=1,3) / 0.4617240000E+02, -0.1279450000E+01, 0.2613520000E+00 /
  data (epsData(58,i,14), i=1,3) / 0.5135070000E+02, -0.9763260000E+00, 0.2170450000E+00 /
  data (epsData(59,i,14), i=1,3) / 0.5711650000E+02, -0.7254380000E+00, 0.1764030000E+00 /
  data (epsData(60,i,14), i=1,3) / 0.6351660000E+02, -0.5106800000E+00, 0.1395610000E+00 /
  data (epsData(61,i,14), i=1,3) / 0.7001790000E+02, -0.3352020000E+00, 0.1120660000E+00 /
  data (epsData(62,i,14), i=1,3) / 0.7718560000E+02, -0.1720470000E+00, 0.9053160000E-01 /
  data (epsData(63,i,14), i=1,3) / 0.8604310000E+02, 0.1411070000E-01, 0.7182080000E-01 /
  data (epsData(64,i,14), i=1,3) / 0.8966880000E+02, 0.9631880000E-01, 0.6546830000E-01 /
  data (epsData(65,i,14), i=1,3) / 0.9269650000E+02, 0.1748410000E+00, 0.6064390000E-01 /
  data (epsData(66,i,14), i=1,3) / 0.9508940000E+02, 0.2498570000E+00, 0.5724160000E-01 /
  data (epsData(67,i,14), i=1,3) / 0.9716710000E+02, 0.3339070000E+00, 0.5481070000E-01 /
  data (epsData(68,i,14), i=1,3) / 0.9856580000E+02, 0.4134370000E+00, 0.5394430000E-01 /
  data (epsData(69,i,14), i=1,3) / 0.9891720000E+02, 0.4409500000E+00, 0.5422810000E-01 /
  data (epsData(70,i,14), i=1,3) / 0.9926920000E+02, 0.4761680000E+00, 0.5729380000E-01 /
  data (epsData(71,i,14), i=1,3) / 0.9983670000E+02, 0.5237210000E+00, 0.9176540000E-01 /
  data (epsData(72,i,14), i=1,3) / 0.1009250000E+03, 0.6017190000E+00, 0.1460660000E+00 /
  data (epsData(73,i,14), i=1,3) / 0.1018890000E+03, 0.6739100000E+00, 0.2296140000E+00 /
  data (epsData(74,i,14), i=1,3) / 0.1033590000E+03, 0.7372050000E+00, 0.3885490000E+00 /
  data (epsData(75,i,14), i=1,3) / 0.1043180000E+03, 0.7285080000E+00, 0.5290570000E+00 /
  data (epsData(76,i,14), i=1,3) / 0.1055680000E+03, 0.6472980000E+00, 0.6651710000E+00 /
  data (epsData(77,i,14), i=1,3) / 0.1071700000E+03, 0.5117460000E+00, 0.7501820000E+00 /
  data (epsData(78,i,14), i=1,3) / 0.1089680000E+03, 0.3983670000E+00, 0.7661150000E+00 /
  data (epsData(79,i,14), i=1,3) / 0.1120990000E+03, 0.2934970000E+00, 0.7638920000E+00 /
  data (epsData(80,i,14), i=1,3) / 0.1195760000E+03, 0.1221590000E+00, 0.7903950000E+00 /
  data (epsData(81,i,14), i=1,3) / 0.1240680000E+03, 0.2793790000E-01, 0.7642430000E+00 /
  data (epsData(82,i,14), i=1,3) / 0.1306370000E+03, -0.5394600000E-01, 0.7112870000E+00 /
  data (epsData(83,i,14), i=1,3) / 0.1389090000E+03, -0.1236530000E+00, 0.6423890000E+00 /
  data (epsData(84,i,14), i=1,3) / 0.1466370000E+03, -0.1476200000E+00, 0.5849410000E+00 /
  data (epsData(85,i,14), i=1,3) / 0.1499670000E+03, -0.1443400000E+00, 0.5814310000E+00 /
  data (epsData(86,i,14), i=1,3) / 0.1551860000E+03, -0.1889110000E+00, 0.5698780000E+00 /
  data (epsData(87,i,14), i=1,3) / 0.1669200000E+03, -0.2451900000E+00, 0.4878790000E+00 /
  data (epsData(88,i,14), i=1,3) / 0.1811350000E+03, -0.2635110000E+00, 0.3967470000E+00 /
  data (epsData(89,i,14), i=1,3) / 0.2014360000E+03, -0.2608240000E+00, 0.3003860000E+00 /
  data (epsData(90,i,14), i=1,3) / 0.2242000000E+03, -0.2419630000E+00, 0.2205250000E+00 /
  data (epsData(91,i,14), i=1,3) / 0.2507320000E+03, -0.2126320000E+00, 0.1583050000E+00 /
  data (epsData(92,i,14), i=1,3) / 0.2804530000E+03, -0.1819910000E+00, 0.1117100000E+00 /
  data (epsData(93,i,14), i=1,3) / 0.3135280000E+03, -0.1521220000E+00, 0.7788320000E-01 /
  data (epsData(94,i,14), i=1,3) / 0.3513750000E+03, -0.1247800000E+00, 0.5394700000E-01 /
  data (epsData(95,i,14), i=1,3) / 0.3891390000E+03, -0.1036000000E+00, 0.3840730000E-01 /
  data (epsData(96,i,14), i=1,3) / 0.4289080000E+03, -0.8620370000E-01, 0.2768030000E-01 /
  data (epsData(97,i,14), i=1,3) / 0.4743840000E+03, -0.7087160000E-01, 0.1957140000E-01 /
  data (epsData(98,i,14), i=1,3) / 0.5225770000E+03, -0.5857980000E-01, 0.1422210000E-01 /
  data (epsData(99,i,14), i=1,3) / 0.5779120000E+03, -0.4785020000E-01, 0.9961530000E-02 /
  data (epsData(100,i,14), i=1,3) / 0.6460720000E+03, -0.3824740000E-01, 0.6677440000E-02 /
  data (epsData(101,i,14), i=1,3) / 0.7166270000E+03, -0.3093570000E-01, 0.4667240000E-02 /
  data (epsData(102,i,14), i=1,3) / 0.7881340000E+03, -0.2541110000E-01, 0.3323490000E-02 /
  data (epsData(103,i,14), i=1,3) / 0.8761270000E+03, -0.2039690000E-01, 0.2269140000E-02 /
  data (epsData(104,i,14), i=1,3) / 0.9761090000E+03, -0.1627470000E-01, 0.1532860000E-02 /
  data (epsData(105,i,14), i=1,3) / 0.1087350000E+04, -0.1296320000E-01, 0.1034300000E-02 /
  data (epsData(106,i,14), i=1,3) / 0.1211030000E+04, -0.1029820000E-01, 0.6975220000E-03 /
  data (epsData(107,i,14), i=1,3) / 0.1349460000E+04, -0.8128730000E-02, 0.4687780000E-03 /
  data (epsData(108,i,14), i=1,3) / 0.1506770000E+04, -0.6313970000E-02, 0.3120270000E-03 /
  data (epsData(109,i,14), i=1,3) / 0.1772850000E+04, -0.3871740000E-02, 0.1684710000E-03 /
  data (epsData(110,i,14), i=1,3) / 0.1809790000E+04, -0.3347680000E-02, 0.1559820000E-03 /
  data (epsData(111,i,14), i=1,3) / 0.1824540000E+04, -0.2977020000E-02, 0.1572570000E-03 /
  data (epsData(112,i,14), i=1,3) / 0.1831390000E+04, -0.2674210000E-02, 0.1608410000E-03 /
  data (epsData(113,i,14), i=1,3) / 0.1834830000E+04, -0.2409820000E-02, 0.1690730000E-03 /
  data (epsData(114,i,14), i=1,3) / 0.1835970000E+04, -0.2273960000E-02, 0.1760550000E-03 /
  data (epsData(115,i,14), i=1,3) / 0.1837120000E+04, -0.2076590000E-02, 0.1909980000E-03 /
  data (epsData(116,i,14), i=1,3) / 0.1838270000E+04, -0.1730320000E-02, 0.2472470000E-03 /
  data (epsData(117,i,14), i=1,3) / 0.1839030000E+04, -0.1449400000E-02, 0.6153340000E-03 /
  data (epsData(118,i,14), i=1,3) / 0.1839180000E+04, -0.1442000000E-02, 0.7726800000E-03 /
  data (epsData(119,i,14), i=1,3) / 0.1839540000E+04, -0.1563200000E-02, 0.9762240000E-03 /
  data (epsData(120,i,14), i=1,3) / 0.1840220000E+04, -0.1727150000E-02, 0.1135010000E-02 /
  data (epsData(121,i,14), i=1,3) / 0.1842270000E+04, -0.2033790000E-02, 0.1396380000E-02 /
  data (epsData(122,i,14), i=1,3) / 0.1843980000E+04, -0.2190940000E-02, 0.1536670000E-02 /
  data (epsData(123,i,14), i=1,3) / 0.1846350000E+04, -0.2444530000E-02, 0.1607430000E-02 /
  data (epsData(124,i,14), i=1,3) / 0.1854430000E+04, -0.2914100000E-02, 0.1671780000E-02 /
  data (epsData(125,i,14), i=1,3) / 0.1861990000E+04, -0.3096820000E-02, 0.1580120000E-02 /
  data (epsData(126,i,14), i=1,3) / 0.1883080000E+04, -0.3308060000E-02, 0.1541010000E-02 /
  data (epsData(127,i,14), i=1,3) / 0.1919670000E+04, -0.3508860000E-02, 0.1464630000E-02 /
  data (epsData(128,i,14), i=1,3) / 0.2016310000E+04, -0.3560090000E-02, 0.1237690000E-02 /
  data (epsData(129,i,14), i=1,3) / 0.2211880000E+04, -0.3227820000E-02, 0.8861810000E-03 /
  data (epsData(130,i,14), i=1,3) / 0.2522950000E+04, -0.2621440000E-02, 0.5573770000E-03 /
  data (epsData(131,i,14), i=1,3) / 0.2835970000E+04, -0.2128180000E-02, 0.3673040000E-03 /
  data (epsData(132,i,14), i=1,3) / 0.3175240000E+04, -0.1721390000E-02, 0.2442180000E-03 /
  data (epsData(133,i,14), i=1,3) / 0.3504860000E+04, -0.1422990000E-02, 0.1721880000E-03 /
  data (epsData(134,i,14), i=1,3) / 0.3910890000E+04, -0.1148310000E-02, 0.1146710000E-03 /
  data (epsData(135,i,14), i=1,3) / 0.4361160000E+04, -0.9249860000E-03, 0.7622140000E-04 /
  data (epsData(136,i,14), i=1,3) / 0.4861750000E+04, -0.7441700000E-03, 0.5051370000E-04 /
  data (epsData(137,i,14), i=1,3) / 0.5420830000E+04, -0.5977690000E-03, 0.3332650000E-04 /
  data (epsData(138,i,14), i=1,3) / 0.6046340000E+04, -0.4794740000E-03, 0.2188180000E-04 /
  data (epsData(139,i,14), i=1,3) / 0.6750560000E+04, -0.3837990000E-03, 0.1429470000E-04 /
  data (epsData(140,i,14), i=1,3) / 0.7542620000E+04, -0.3065960000E-03, 0.9270430000E-05 /
  data (epsData(141,i,14), i=1,3) / 0.8426740000E+04, -0.2450170000E-03, 0.5997900000E-05 /
  data (epsData(142,i,14), i=1,3) / 0.9419420000E+04, -0.1955930000E-03, 0.3852200000E-05 /
  data (epsData(143,i,14), i=1,3) / 0.1053330000E+05, -0.1560460000E-03, 0.2462610000E-05 /
  data (epsData(144,i,14), i=1,3) / 0.1178580000E+05, -0.1243680000E-03, 0.1569410000E-05 /
  data (epsData(145,i,14), i=1,3) / 0.1318940000E+05, -0.9911100000E-04, 0.9978480000E-06 /
  data (epsData(146,i,14), i=1,3) / 0.1476920000E+05, -0.7890220000E-04, 0.6300740000E-06 /
  data (epsData(147,i,14), i=1,3) / 0.1654160000E+05, -0.6280180000E-04, 0.3968900000E-06 /
  data (epsData(148,i,14), i=1,3) / 0.1853290000E+05, -0.4996290000E-04, 0.2488070000E-06 /
  data (epsData(149,i,14), i=1,3) / 0.2076720000E+05, -0.3974360000E-04, 0.1557320000E-06 /
  data (epsData(150,i,14), i=1,3) / 0.2327660000E+05, -0.3160400000E-04, 0.9705680000E-07 /
  data (epsData(151,i,14), i=1,3) / 0.2609710000E+05, -0.2512010000E-04, 0.6025300000E-07 /
  data (epsData(152,i,14), i=1,3) / 0.2926260000E+05, -0.1996460000E-04, 0.3732890000E-07 /
  data (epsData(153,i,14), i=1,3) / 0.3281670000E+05, -0.1586460000E-04, 0.2305340000E-07 /
  data (epsData(154,i,14), i=1,3) / 0.3680080000E+05, -0.1260890000E-04, 0.1418510000E-07 /
  data (epsData(155,i,14), i=1,3) / 0.4127450000E+05, -0.1001930000E-04, 0.8749880000E-08 /
  data (epsData(156,i,14), i=1,3) / 0.4629710000E+05, -0.7960430000E-05, 0.5353220000E-08 /
  data (epsData(157,i,14), i=1,3) / 0.5193400000E+05, -0.6324250000E-05, 0.3284920000E-08 /
  data (epsData(158,i,14), i=1,3) / 0.5826540000E+05, -0.5023230000E-05, 0.1997780000E-08 /
  data (epsData(159,i,14), i=1,3) / 0.6536770000E+05, -0.3990170000E-05, 0.1222450000E-08 /
  data (epsData(160,i,14), i=1,3) / 0.7333380000E+05, -0.3169820000E-05, 0.7475430000E-09 /
  data (epsData(161,i,14), i=1,3) / 0.8228180000E+05, -0.2517570000E-05, 0.4613930000E-09 /
  data (epsData(162,i,14), i=1,3) / 0.9232200000E+05, -0.1999580000E-05, 0.2717710000E-09 /
  data (epsData(163,i,14), i=1,3) / 0.1000000000E+06, -0.1704780000E-05, 0.1958080000E-09 /
  data (epsData(1,i,15), i=1,3) / 0.2506580000E-02, -0.1913400000E+02, 0.5290860000E+03 /
  data (epsData(2,i,15), i=1,3) / 0.4693440000E-02, -0.5085830000E+02, 0.2124310000E+03 /
  data (epsData(3,i,15), i=1,3) / 0.7560590000E-02, 0.3296340000E+01, 0.1129490000E+03 /
  data (epsData(4,i,15), i=1,3) / 0.1110800000E-01, 0.2838790000E+02, 0.7014400000E+02 /
  data (epsData(5,i,15), i=1,3) / 0.1533570000E-01, 0.4046830000E+02, 0.4794120000E+02 /
  data (epsData(6,i,15), i=1,3) / 0.2024370000E-01, 0.4680190000E+02, 0.3493520000E+02 /
  data (epsData(7,i,15), i=1,3) / 0.2583200000E-01, 0.5037860000E+02, 0.2664510000E+02 /
  data (epsData(8,i,15), i=1,3) / 0.3210060000E-01, 0.5252500000E+02, 0.2102490000E+02 /
  data (epsData(9,i,15), i=1,3) / 0.3904940000E-01, 0.5387860000E+02, 0.1703210000E+02 /
  data (epsData(10,i,15), i=1,3) / 0.5498800000E-01, 0.5537320000E+02, 0.1185620000E+02 /
  data (epsData(11,i,15), i=1,3) / 0.8399800000E-01, 0.5632690000E+02, 0.7629090000E+01 /
  data (epsData(12,i,15), i=1,3) / 0.1459540000E+00, 0.5687280000E+02, 0.4329320000E+01 /
  data (epsData(13,i,15), i=1,3) / 0.3208860000E+00, 0.5717000000E+02, 0.2020580000E+01 /
  data (epsData(14,i,15), i=1,3) / 0.1212890000E+01, 0.5801800000E+02, 0.6424760000E+00 /
  data (epsData(15,i,15), i=1,3) / 0.2737710000E+01, 0.6227300000E+02, 0.4257530000E+00 /
  data (epsData(16,i,15), i=1,3) / 0.3946480000E+01, 0.6960110000E+02, 0.4572290000E+00 /
  data (epsData(17,i,15), i=1,3) / 0.4793680000E+01, 0.7904660000E+02, 0.5874070000E+00 /
  data (epsData(18,i,15), i=1,3) / 0.5290480000E+01, 0.8843510000E+02, 0.7837800000E+00 /
  data (epsData(19,i,15), i=1,3) / 0.5635290000E+01, 0.9947040000E+02, 0.1327960000E+01 /
  data (epsData(20,i,15), i=1,3) / 0.5811770000E+01, 0.1089390000E+03, 0.2377880000E+01 /
  data (epsData(21,i,15), i=1,3) / 0.5901040000E+01, 0.1177060000E+03, 0.3012870000E+01 /
  data (epsData(22,i,15), i=1,3) / 0.6005440000E+01, 0.1247360000E+03, 0.1796910000E+02 /
  data (epsData(23,i,15), i=1,3) / 0.6179940000E+01, 0.1220420000E+03, 0.3397800000E+02 /
  data (epsData(24,i,15), i=1,3) / 0.6266670000E+01, 0.1176070000E+03, 0.3719320000E+02 /
  data (epsData(25,i,15), i=1,3) / 0.6628920000E+01, 0.1151920000E+03, 0.4500410000E+02 /
  data (epsData(26,i,15), i=1,3) / 0.6884710000E+01, 0.1189590000E+03, 0.5133230000E+02 /
  data (epsData(27,i,15), i=1,3) / 0.7057480000E+01, 0.1204380000E+03, 0.6294890000E+02 /
  data (epsData(28,i,15), i=1,3) / 0.7266670000E+01, 0.1156740000E+03, 0.7408180000E+02 /
  data (epsData(29,i,15), i=1,3) / 0.8144240000E+01, 0.9482260000E+02, 0.1066530000E+03 /
  data (epsData(30,i,15), i=1,3) / 0.8572820000E+01, 0.7675800000E+02, 0.1197130000E+03 /
  data (epsData(31,i,15), i=1,3) / 0.8964660000E+01, 0.6029360000E+02, 0.1208560000E+03 /
  data (epsData(32,i,15), i=1,3) / 0.9400040000E+01, 0.4919850000E+02, 0.1244790000E+03 /
  data (epsData(33,i,15), i=1,3) / 0.9878960000E+01, 0.3131860000E+02, 0.1279830000E+03 /
  data (epsData(34,i,15), i=1,3) / 0.1067900000E+02, 0.4131610000E+01, 0.1266940000E+03 /
  data (epsData(35,i,15), i=1,3) / 0.1126670000E+02, -0.1580090000E+02, 0.1194350000E+03 /
  data (epsData(36,i,15), i=1,3) / 0.1200310000E+02, -0.3521870000E+02, 0.1046820000E+03 /
  data (epsData(37,i,15), i=1,3) / 0.1285040000E+02, -0.4911780000E+02, 0.8421020000E+02 /
  data (epsData(38,i,15), i=1,3) / 0.1371470000E+02, -0.5483830000E+02, 0.6359780000E+02 /
  data (epsData(39,i,15), i=1,3) / 0.1451570000E+02, -0.5446620000E+02, 0.4729480000E+02 /
  data (epsData(40,i,15), i=1,3) / 0.1540140000E+02, -0.5053150000E+02, 0.3353950000E+02 /
  data (epsData(41,i,15), i=1,3) / 0.1649840000E+02, -0.4397400000E+02, 0.2190910000E+02 /
  data (epsData(42,i,15), i=1,3) / 0.1766030000E+02, -0.3716680000E+02, 0.1419140000E+02 /
  data (epsData(43,i,15), i=1,3) / 0.1885860000E+02, -0.3112350000E+02, 0.9256300000E+01 /
  data (epsData(44,i,15), i=1,3) / 0.2006680000E+02, -0.2610600000E+02, 0.6181580000E+01 /
  data (epsData(45,i,15), i=1,3) / 0.2137300000E+02, -0.2178360000E+02, 0.4146350000E+01 /
  data (epsData(46,i,15), i=1,3) / 0.2277710000E+02, -0.1813670000E+02, 0.2831920000E+01 /
  data (epsData(47,i,15), i=1,3) / 0.2440870000E+02, -0.1489400000E+02, 0.1954980000E+01 /
  data (epsData(48,i,15), i=1,3) / 0.2615530000E+02, -0.1228460000E+02, 0.1445660000E+01 /
  data (epsData(49,i,15), i=1,3) / 0.2816490000E+02, -0.1006390000E+02, 0.1137050000E+01 /
  data (epsData(50,i,15), i=1,3) / 0.3062480000E+02, -0.8104710000E+01, 0.9449290000E+00 /
  data (epsData(51,i,15), i=1,3) / 0.3342930000E+02, -0.6527400000E+01, 0.8332720000E+00 /
  data (epsData(52,i,15), i=1,3) / 0.3679630000E+02, -0.5206640000E+01, 0.7544270000E+00 /
  data (epsData(53,i,15), i=1,3) / 0.4080360000E+02, -0.4131070000E+01, 0.6836680000E+00 /
  data (epsData(54,i,15), i=1,3) / 0.4553660000E+02, -0.3262940000E+01, 0.6029960000E+00 /
  data (epsData(55,i,15), i=1,3) / 0.5085920000E+02, -0.2585520000E+01, 0.5216500000E+00 /
  data (epsData(56,i,15), i=1,3) / 0.5706740000E+02, -0.2036820000E+01, 0.4403070000E+00 /
  data (epsData(57,i,15), i=1,3) / 0.6426760000E+02, -0.1589700000E+01, 0.3564270000E+00 /
  data (epsData(58,i,15), i=1,3) / 0.7228720000E+02, -0.1231040000E+01, 0.2828620000E+00 /
  data (epsData(59,i,15), i=1,3) / 0.8087910000E+02, -0.9495110000E+00, 0.2243680000E+00 /
  data (epsData(60,i,15), i=1,3) / 0.9036930000E+02, -0.7172050000E+00, 0.1766180000E+00 /
  data (epsData(61,i,15), i=1,3) / 0.1011670000E+03, -0.5131990000E+00, 0.1357530000E+00 /
  data (epsData(62,i,15), i=1,3) / 0.1148940000E+03, -0.2986210000E+00, 0.9860410000E-01 /
  data (epsData(63,i,15), i=1,3) / 0.1240700000E+03, -0.1556410000E+00, 0.8135070000E-01 /
  data (epsData(64,i,15), i=1,3) / 0.1280090000E+03, -0.8317800000E-01, 0.7525600000E-01 /
  data (epsData(65,i,15), i=1,3) / 0.1304050000E+03, -0.3049940000E-01, 0.7197030000E-01 /
  data (epsData(66,i,15), i=1,3) / 0.1316120000E+03, 0.5829710000E-03, 0.7044640000E-01 /
  data (epsData(67,i,15), i=1,3) / 0.1328260000E+03, 0.3690520000E-01, 0.6907430000E-01 /
  data (epsData(68,i,15), i=1,3) / 0.1336380000E+03, 0.6580510000E-01, 0.6830740000E-01 /
  data (epsData(69,i,15), i=1,3) / 0.1344530000E+03, 0.1021450000E+00, 0.6799250000E-01 /
  data (epsData(70,i,15), i=1,3) / 0.1348620000E+03, 0.1264150000E+00, 0.6983400000E-01 /
  data (epsData(71,i,15), i=1,3) / 0.1352130000E+03, 0.1443360000E+00, 0.8211420000E-01 /
  data (epsData(72,i,15), i=1,3) / 0.1358500000E+03, 0.1760200000E+00, 0.9174820000E-01 /
  data (epsData(73,i,15), i=1,3) / 0.1370670000E+03, 0.2439630000E+00, 0.1345270000E+00 /
  data (epsData(74,i,15), i=1,3) / 0.1382320000E+03, 0.3062340000E+00, 0.1976040000E+00 /
  data (epsData(75,i,15), i=1,3) / 0.1391350000E+03, 0.3436730000E+00, 0.2758190000E+00 /
  data (epsData(76,i,15), i=1,3) / 0.1401910000E+03, 0.3490140000E+00, 0.3820490000E+00 /
  data (epsData(77,i,15), i=1,3) / 0.1412300000E+03, 0.3146930000E+00, 0.4764660000E+00 /
  data (epsData(78,i,15), i=1,3) / 0.1422910000E+03, 0.2505260000E+00, 0.5444550000E+00 /
  data (epsData(79,i,15), i=1,3) / 0.1435510000E+03, 0.1698260000E+00, 0.5777250000E+00 /
  data (epsData(80,i,15), i=1,3) / 0.1451480000E+03, 0.9652570000E-01, 0.5816660000E+00 /
  data (epsData(81,i,15), i=1,3) / 0.1477680000E+03, 0.2969960000E-01, 0.5706850000E+00 /
  data (epsData(82,i,15), i=1,3) / 0.1565840000E+03, -0.9884550000E-01, 0.5631240000E+00 /
  data (epsData(83,i,15), i=1,3) / 0.1630490000E+03, -0.1591310000E+00, 0.5245430000E+00 /
  data (epsData(84,i,15), i=1,3) / 0.1725940000E+03, -0.2071410000E+00, 0.4669090000E+00 /
  data (epsData(85,i,15), i=1,3) / 0.1825650000E+03, -0.2166010000E+00, 0.4173200000E+00 /
  data (epsData(86,i,15), i=1,3) / 0.1888020000E+03, -0.2113720000E+00, 0.3996440000E+00 /
  data (epsData(87,i,15), i=1,3) / 0.1938170000E+03, -0.2267290000E+00, 0.3992850000E+00 /
  data (epsData(88,i,15), i=1,3) / 0.2068370000E+03, -0.2563300000E+00, 0.3476730000E+00 /
  data (epsData(89,i,15), i=1,3) / 0.2271720000E+03, -0.2580310000E+00, 0.2724730000E+00 /
  data (epsData(90,i,15), i=1,3) / 0.2551510000E+03, -0.2400540000E+00, 0.1969960000E+00 /
  data (epsData(91,i,15), i=1,3) / 0.2892920000E+03, -0.2079150000E+00, 0.1355110000E+00 /
  data (epsData(92,i,15), i=1,3) / 0.3278900000E+03, -0.1733440000E+00, 0.9133180000E-01 /
  data (epsData(93,i,15), i=1,3) / 0.3653330000E+03, -0.1453030000E+00, 0.6445790000E-01 /
  data (epsData(94,i,15), i=1,3) / 0.4056090000E+03, -0.1209780000E+00, 0.4552790000E-01 /
  data (epsData(95,i,15), i=1,3) / 0.4518140000E+03, -0.9907450000E-01, 0.3169650000E-01 /
  data (epsData(96,i,15), i=1,3) / 0.5005270000E+03, -0.8150840000E-01, 0.2237050000E-01 /
  data (epsData(97,i,15), i=1,3) / 0.5498670000E+03, -0.6786240000E-01, 0.1632750000E-01 /
  data (epsData(98,i,15), i=1,3) / 0.6126450000E+03, -0.5478170000E-01, 0.1115430000E-01 /
  data (epsData(99,i,15), i=1,3) / 0.6853720000E+03, -0.4377490000E-01, 0.7456310000E-02 /
  data (epsData(100,i,15), i=1,3) / 0.7638590000E+03, -0.3515840000E-01, 0.5098780000E-02 /
  data (epsData(101,i,15), i=1,3) / 0.8443200000E+03, -0.2863740000E-01, 0.3574980000E-02 /
  data (epsData(102,i,15), i=1,3) / 0.9320600000E+03, -0.2336490000E-01, 0.2503600000E-02 /
  data (epsData(103,i,15), i=1,3) / 0.1033870000E+04, -0.1885950000E-01, 0.1719030000E-02 /
  data (epsData(104,i,15), i=1,3) / 0.1150810000E+04, -0.1509680000E-01, 0.1162650000E-02 /
  data (epsData(105,i,15), i=1,3) / 0.1281480000E+04, -0.1205690000E-01, 0.7837650000E-03 /
  data (epsData(106,i,15), i=1,3) / 0.1427470000E+04, -0.9599550000E-02, 0.5266580000E-03 /
  data (epsData(107,i,15), i=1,3) / 0.1589610000E+04, -0.7617690000E-02, 0.3538710000E-03 /
  data (epsData(108,i,15), i=1,3) / 0.1772230000E+04, -0.5972620000E-02, 0.2332860000E-03 /
  data (epsData(109,i,15), i=1,3) / 0.2084150000E+04, -0.3759550000E-02, 0.1271970000E-03 /
  data (epsData(110,i,15), i=1,3) / 0.2122070000E+04, -0.3313070000E-02, 0.1188610000E-03 /
  data (epsData(111,i,15), i=1,3) / 0.2136470000E+04, -0.2968750000E-02, 0.1233150000E-03 /
  data (epsData(112,i,15), i=1,3) / 0.2141290000E+04, -0.2731060000E-02, 0.1302150000E-03 /
  data (epsData(113,i,15), i=1,3) / 0.2143710000E+04, -0.2495770000E-02, 0.1462570000E-03 /
  data (epsData(114,i,15), i=1,3) / 0.2144920000E+04, -0.2263180000E-02, 0.1930360000E-03 /
  data (epsData(115,i,15), i=1,3) / 0.2145600000E+04, -0.2101030000E-02, 0.3993460000E-03 /
  data (epsData(116,i,15), i=1,3) / 0.2146100000E+04, -0.2129570000E-02, 0.5818960000E-03 /
  data (epsData(117,i,15), i=1,3) / 0.2147960000E+04, -0.2288650000E-02, 0.8911270000E-03 /
  data (epsData(118,i,15), i=1,3) / 0.2150480000E+04, -0.2500770000E-02, 0.1119340000E-02 /
  data (epsData(119,i,15), i=1,3) / 0.2154100000E+04, -0.2746510000E-02, 0.1184810000E-02 /
  data (epsData(120,i,15), i=1,3) / 0.2161220000E+04, -0.3054920000E-02, 0.1234440000E-02 /
  data (epsData(121,i,15), i=1,3) / 0.2172170000E+04, -0.3229480000E-02, 0.1174660000E-02 /
  data (epsData(122,i,15), i=1,3) / 0.2188370000E+04, -0.3326890000E-02, 0.1133140000E-02 /
  data (epsData(123,i,15), i=1,3) / 0.2248020000E+04, -0.3478590000E-02, 0.1056580000E-02 /
  data (epsData(124,i,15), i=1,3) / 0.2399420000E+04, -0.3352090000E-02, 0.8380280000E-03 /
  data (epsData(125,i,15), i=1,3) / 0.2800820000E+04, -0.2655220000E-02, 0.4821530000E-03 /
  data (epsData(126,i,15), i=1,3) / 0.3157670000E+04, -0.2143040000E-02, 0.3147490000E-03 /
  data (epsData(127,i,15), i=1,3) / 0.3540350000E+04, -0.1727300000E-02, 0.2082180000E-03 /
  data (epsData(128,i,15), i=1,3) / 0.3970040000E+04, -0.1385170000E-02, 0.1386470000E-03 /
  data (epsData(129,i,15), i=1,3) / 0.4432590000E+04, -0.1115730000E-02, 0.9214580000E-04 /
  data (epsData(130,i,15), i=1,3) / 0.4947330000E+04, -0.8970130000E-03, 0.6102700000E-04 /
  data (epsData(131,i,15), i=1,3) / 0.5520700000E+04, -0.7203150000E-03, 0.4029160000E-04 /
  data (epsData(132,i,15), i=1,3) / 0.6161310000E+04, -0.5776780000E-03, 0.2651210000E-04 /
  data (epsData(133,i,15), i=1,3) / 0.6887960000E+04, -0.4614790000E-03, 0.1727310000E-04 /
  data (epsData(134,i,15), i=1,3) / 0.7699700000E+04, -0.3686030000E-03, 0.1122730000E-04 /
  data (epsData(135,i,15), i=1,3) / 0.8602190000E+04, -0.2947110000E-03, 0.7256620000E-05 /
  data (epsData(136,i,15), i=1,3) / 0.9617150000E+04, -0.2352790000E-03, 0.4684030000E-05 /
  data (epsData(137,i,15), i=1,3) / 0.1075740000E+05, -0.1876540000E-03, 0.3001880000E-05 /
  data (epsData(138,i,15), i=1,3) / 0.1203720000E+05, -0.1495820000E-03, 0.1916410000E-05 /
  data (epsData(139,i,15), i=1,3) / 0.1347620000E+05, -0.1191310000E-03, 0.1216900000E-05 /
  data (epsData(140,i,15), i=1,3) / 0.1508830000E+05, -0.9488010000E-04, 0.7711690000E-06 /
  data (epsData(141,i,15), i=1,3) / 0.1690180000E+05, -0.7550300000E-04, 0.4862230000E-06 /
  data (epsData(142,i,15), i=1,3) / 0.1893490000E+05, -0.6008390000E-04, 0.3057410000E-06 /
  data (epsData(143,i,15), i=1,3) / 0.2122120000E+05, -0.4778190000E-04, 0.1914250000E-06 /
  data (epsData(144,i,15), i=1,3) / 0.2378890000E+05, -0.3798750000E-04, 0.1194680000E-06 /
  data (epsData(145,i,15), i=1,3) / 0.2666860000E+05, -0.3020170000E-04, 0.7435980000E-07 /
  data (epsData(146,i,15), i=1,3) / 0.2990000000E+05, -0.2400970000E-04, 0.4612690000E-07 /
  data (epsData(147,i,15), i=1,3) / 0.3352710000E+05, -0.1908440000E-04, 0.2854520000E-07 /
  data (epsData(148,i,15), i=1,3) / 0.3759880000E+05, -0.1516720000E-04, 0.1760280000E-07 /
  data (epsData(149,i,15), i=1,3) / 0.4216930000E+05, -0.1205250000E-04, 0.1087030000E-07 /
  data (epsData(150,i,15), i=1,3) / 0.4729860000E+05, -0.9576800000E-05, 0.6651200000E-08 /
  data (epsData(151,i,15), i=1,3) / 0.5306130000E+05, -0.7607350000E-05, 0.4098540000E-08 /
  data (epsData(152,i,15), i=1,3) / 0.5952160000E+05, -0.6044180000E-05, 0.2525190000E-08 /
  data (epsData(153,i,15), i=1,3) / 0.6677540000E+05, -0.4801400000E-05, 0.1534650000E-08 /
  data (epsData(154,i,15), i=1,3) / 0.7491760000E+05, -0.3813840000E-05, 0.9507740000E-09 /
  data (epsData(155,i,15), i=1,3) / 0.8405880000E+05, -0.3029080000E-05, 0.5867870000E-09 /
  data (epsData(156,i,15), i=1,3) / 0.9431070000E+05, -0.2406130000E-05, 0.3409550000E-09 /
  data (epsData(157,i,15), i=1,3) / 0.1000000000E+06, -0.2140920000E-05, 0.2636740000E-09 /
  data (epsData(1,i,16), i=1,3) / 0.2506580000E-02, 0.8199490000E+02, 0.2552120000E+03 /
  data (epsData(2,i,16), i=1,3) / 0.4693440000E-02, 0.2973480000E+02, 0.1208800000E+03 /
  data (epsData(3,i,16), i=1,3) / 0.7560590000E-02, 0.3511560000E+02, 0.7087880000E+02 /
  data (epsData(4,i,16), i=1,3) / 0.1110800000E-01, 0.3899550000E+02, 0.4676270000E+02 /
  data (epsData(5,i,16), i=1,3) / 0.1533570000E-01, 0.4111250000E+02, 0.3324120000E+02 /
  data (epsData(6,i,16), i=1,3) / 0.2024370000E-01, 0.4229500000E+02, 0.2487800000E+02 /
  data (epsData(7,i,16), i=1,3) / 0.2583200000E-01, 0.4298970000E+02, 0.1933510000E+02 /
  data (epsData(8,i,16), i=1,3) / 0.3210060000E-01, 0.4341830000E+02, 0.1546770000E+02 /
  data (epsData(9,i,16), i=1,3) / 0.4667860000E-01, 0.4387900000E+02, 0.1055590000E+02 /
  data (epsData(10,i,16), i=1,3) / 0.7364770000E-01, 0.4416460000E+02, 0.6648940000E+01 /
  data (epsData(11,i,16), i=1,3) / 0.1322020000E+00, 0.4432780000E+02, 0.3708970000E+01 /
  data (epsData(12,i,16), i=1,3) / 0.3003320000E+00, 0.4440000000E+02, 0.1644630000E+01 /
  data (epsData(13,i,16), i=1,3) / 0.1172610000E+01, 0.4481860000E+02, 0.4910200000E+00 /
  data (epsData(14,i,16), i=1,3) / 0.2987270000E+01, 0.4768360000E+02, 0.2836600000E+00 /
  data (epsData(15,i,16), i=1,3) / 0.4476080000E+01, 0.5299800000E+02, 0.2906140000E+00 /
  data (epsData(16,i,16), i=1,3) / 0.5548060000E+01, 0.6014410000E+02, 0.3672150000E+00 /
  data (epsData(17,i,16), i=1,3) / 0.6172910000E+01, 0.6728600000E+02, 0.4959800000E+00 /
  data (epsData(18,i,16), i=1,3) / 0.6544930000E+01, 0.7426020000E+02, 0.6996170000E+00 /
  data (epsData(19,i,16), i=1,3) / 0.6735020000E+01, 0.7990270000E+02, 0.1257870000E+01 /
  data (epsData(20,i,16), i=1,3) / 0.6831080000E+01, 0.8428420000E+02, 0.1607610000E+01 /
  data (epsData(21,i,16), i=1,3) / 0.6927830000E+01, 0.9007220000E+02, 0.4494340000E+01 /
  data (epsData(22,i,16), i=1,3) / 0.7016670000E+01, 0.9278340000E+02, 0.1217100000E+02 /
  data (epsData(23,i,16), i=1,3) / 0.7195920000E+01, 0.9075320000E+02, 0.2239020000E+02 /
  data (epsData(24,i,16), i=1,3) / 0.7393200000E+01, 0.8755980000E+02, 0.2488840000E+02 /
  data (epsData(25,i,16), i=1,3) / 0.7816680000E+01, 0.8995630000E+02, 0.3028220000E+02 /
  data (epsData(26,i,16), i=1,3) / 0.7955460000E+01, 0.9318180000E+02, 0.3445510000E+02 /
  data (epsData(27,i,16), i=1,3) / 0.8229940000E+01, 0.9160900000E+02, 0.4493630000E+02 /
  data (epsData(28,i,16), i=1,3) / 0.8991850000E+01, 0.8545780000E+02, 0.6446340000E+02 /
  data (epsData(29,i,16), i=1,3) / 0.9307510000E+01, 0.7847070000E+02, 0.7337820000E+02 /
  data (epsData(30,i,16), i=1,3) / 0.9619420000E+01, 0.7343970000E+02, 0.7474170000E+02 /
  data (epsData(31,i,16), i=1,3) / 0.1006940000E+02, 0.6915170000E+02, 0.8396200000E+02 /
  data (epsData(32,i,16), i=1,3) / 0.1051570000E+02, 0.5920870000E+02, 0.9181510000E+02 /
  data (epsData(33,i,16), i=1,3) / 0.1133370000E+02, 0.4118880000E+02, 0.1038710000E+03 /
  data (epsData(34,i,16), i=1,3) / 0.1204120000E+02, 0.2016370000E+02, 0.1083750000E+03 /
  data (epsData(35,i,16), i=1,3) / 0.1273610000E+02, 0.2606110000E-01, 0.1060040000E+03 /
  data (epsData(36,i,16), i=1,3) / 0.1348620000E+02, -0.1896370000E+02, 0.9788740000E+02 /
  data (epsData(37,i,16), i=1,3) / 0.1426670000E+02, -0.3426210000E+02, 0.8468910000E+02 /
  data (epsData(38,i,16), i=1,3) / 0.1514420000E+02, -0.4423780000E+02, 0.6706060000E+02 /
  data (epsData(39,i,16), i=1,3) / 0.1596470000E+02, -0.4705840000E+02, 0.5150590000E+02 /
  data (epsData(40,i,16), i=1,3) / 0.1687900000E+02, -0.4571750000E+02, 0.3759710000E+02 /
  data (epsData(41,i,16), i=1,3) / 0.1796740000E+02, -0.4147070000E+02, 0.2567400000E+02 /
  data (epsData(42,i,16), i=1,3) / 0.1923000000E+02, -0.3562130000E+02, 0.1659890000E+02 /
  data (epsData(43,i,16), i=1,3) / 0.2047770000E+02, -0.3020990000E+02, 0.1094610000E+02 /
  data (epsData(44,i,16), i=1,3) / 0.2175560000E+02, -0.2548950000E+02, 0.7338950000E+01 /
  data (epsData(45,i,16), i=1,3) / 0.2314850000E+02, -0.2130100000E+02, 0.4873780000E+01 /
  data (epsData(46,i,16), i=1,3) / 0.2465630000E+02, -0.1771170000E+02, 0.3249680000E+01 /
  data (epsData(47,i,16), i=1,3) / 0.2627910000E+02, -0.1470690000E+02, 0.2210470000E+01 /
  data (epsData(48,i,16), i=1,3) / 0.2815530000E+02, -0.1205800000E+02, 0.1528610000E+01 /
  data (epsData(49,i,16), i=1,3) / 0.3031350000E+02, -0.9802040000E+01, 0.1110070000E+01 /
  data (epsData(50,i,16), i=1,3) / 0.3262480000E+02, -0.8026210000E+01, 0.8805360000E+00 /
  data (epsData(51,i,16), i=1,3) / 0.3542930000E+02, -0.6469150000E+01, 0.7483720000E+00 /
  data (epsData(52,i,16), i=1,3) / 0.3861300000E+02, -0.5218730000E+01, 0.6794170000E+00 /
  data (epsData(53,i,16), i=1,3) / 0.4240900000E+02, -0.4182440000E+01, 0.6329060000E+00 /
  data (epsData(54,i,16), i=1,3) / 0.4711080000E+02, -0.3305490000E+01, 0.5815020000E+00 /
  data (epsData(55,i,16), i=1,3) / 0.5240070000E+02, -0.2631050000E+01, 0.5275800000E+00 /
  data (epsData(56,i,16), i=1,3) / 0.5857360000E+02, -0.2094210000E+01, 0.4655000000E+00 /
  data (epsData(57,i,16), i=1,3) / 0.6626760000E+02, -0.1633110000E+01, 0.3855450000E+00 /
  data (epsData(58,i,16), i=1,3) / 0.7486140000E+02, -0.1272040000E+01, 0.3110370000E+00 /
  data (epsData(59,i,16), i=1,3) / 0.8410910000E+02, -0.9939810000E+00, 0.2488850000E+00 /
  data (epsData(60,i,16), i=1,3) / 0.9434080000E+02, -0.7695220000E+00, 0.1966880000E+00 /
  data (epsData(61,i,16), i=1,3) / 0.1059750000E+03, -0.5797780000E+00, 0.1521250000E+00 /
  data (epsData(62,i,16), i=1,3) / 0.1187740000E+03, -0.4192100000E+00, 0.1160730000E+00 /
  data (epsData(63,i,16), i=1,3) / 0.1336120000E+03, -0.2670000000E+00, 0.8670250000E-01 /
  data (epsData(64,i,16), i=1,3) / 0.1468700000E+03, -0.1360210000E+00, 0.6841380000E-01 /
  data (epsData(65,i,16), i=1,3) / 0.1524550000E+03, -0.7206540000E-01, 0.6236750000E-01 /
  data (epsData(66,i,16), i=1,3) / 0.1555100000E+03, -0.3025420000E-01, 0.5941010000E-01 /
  data (epsData(67,i,16), i=1,3) / 0.1572710000E+03, -0.2100310000E-02, 0.5782360000E-01 /
  data (epsData(68,i,16), i=1,3) / 0.1585980000E+03, 0.2220870000E-01, 0.5671010000E-01 /
  data (epsData(69,i,16), i=1,3) / 0.1599320000E+03, 0.5066170000E-01, 0.5569840000E-01 /
  data (epsData(70,i,16), i=1,3) / 0.1612720000E+03, 0.8608610000E-01, 0.5491720000E-01 /
  data (epsData(71,i,16), i=1,3) / 0.1617200000E+03, 0.1007780000E+00, 0.5480710000E-01 /
  data (epsData(72,i,16), i=1,3) / 0.1621690000E+03, 0.1183670000E+00, 0.5534420000E-01 /
  data (epsData(73,i,16), i=1,3) / 0.1626500000E+03, 0.1386010000E+00, 0.6332810000E-01 /
  data (epsData(74,i,16), i=1,3) / 0.1633500000E+03, 0.1634080000E+00, 0.7092860000E-01 /
  data (epsData(75,i,16), i=1,3) / 0.1647440000E+03, 0.2238660000E+00, 0.1031600000E+00 /
  data (epsData(76,i,16), i=1,3) / 0.1654630000E+03, 0.2596310000E+00, 0.1266840000E+00 /
  data (epsData(77,i,16), i=1,3) / 0.1665420000E+03, 0.3094220000E+00, 0.1888050000E+00 /
  data (epsData(78,i,16), i=1,3) / 0.1675670000E+03, 0.3349440000E+00, 0.2715620000E+00 /
  data (epsData(79,i,16), i=1,3) / 0.1687460000E+03, 0.3246660000E+00, 0.3763410000E+00 /
  data (epsData(80,i,16), i=1,3) / 0.1697080000E+03, 0.2804040000E+00, 0.4499800000E+00 /
  data (epsData(81,i,16), i=1,3) / 0.1709500000E+03, 0.2015110000E+00, 0.5013100000E+00 /
  data (epsData(82,i,16), i=1,3) / 0.1723080000E+03, 0.1281550000E+00, 0.5148300000E+00 /
  data (epsData(83,i,16), i=1,3) / 0.1740170000E+03, 0.6485860000E-01, 0.5074260000E+00 /
  data (epsData(84,i,16), i=1,3) / 0.1764020000E+03, 0.1590550000E-01, 0.4920280000E+00 /
  data (epsData(85,i,16), i=1,3) / 0.1841840000E+03, -0.8725860000E-01, 0.4714480000E+00 /
  data (epsData(86,i,16), i=1,3) / 0.1906490000E+03, -0.1380590000E+00, 0.4331130000E+00 /
  data (epsData(87,i,16), i=1,3) / 0.1999710000E+03, -0.1772110000E+00, 0.3820970000E+00 /
  data (epsData(88,i,16), i=1,3) / 0.2148060000E+03, -0.1943670000E+00, 0.3236860000E+00 /
  data (epsData(89,i,16), i=1,3) / 0.2273730000E+03, -0.1920200000E+00, 0.2821250000E+00 /
  data (epsData(90,i,16), i=1,3) / 0.2322500000E+03, -0.1842510000E+00, 0.2789230000E+00 /
  data (epsData(91,i,16), i=1,3) / 0.2416770000E+03, -0.1985170000E+00, 0.2663320000E+00 /
  data (epsData(92,i,16), i=1,3) / 0.2621280000E+03, -0.2054180000E+00, 0.2160390000E+00 /
  data (epsData(93,i,16), i=1,3) / 0.2920500000E+03, -0.1923820000E+00, 0.1588200000E+00 /
  data (epsData(94,i,16), i=1,3) / 0.3326750000E+03, -0.1661210000E+00, 0.1077470000E+00 /
  data (epsData(95,i,16), i=1,3) / 0.3737300000E+03, -0.1406610000E+00, 0.7454580000E-01 /
  data (epsData(96,i,16), i=1,3) / 0.4181740000E+03, -0.1170260000E+00, 0.5193860000E-01 /
  data (epsData(97,i,16), i=1,3) / 0.4641020000E+03, -0.9757890000E-01, 0.3670400000E-01 /
  data (epsData(98,i,16), i=1,3) / 0.5163060000E+03, -0.8009100000E-01, 0.2566130000E-01 /
  data (epsData(99,i,16), i=1,3) / 0.5683360000E+03, -0.6667420000E-01, 0.1861320000E-01 /
  data (epsData(100,i,16), i=1,3) / 0.6372620000E+03, -0.5339790000E-01, 0.1248860000E-01 /
  data (epsData(101,i,16), i=1,3) / 0.7150220000E+03, -0.4252580000E-01, 0.8312350000E-02 /
  data (epsData(102,i,16), i=1,3) / 0.7978260000E+03, -0.3414850000E-01, 0.5654920000E-02 /
  data (epsData(103,i,16), i=1,3) / 0.8843330000E+03, -0.2769990000E-01, 0.3931690000E-02 /
  data (epsData(104,i,16), i=1,3) / 0.9769800000E+03, -0.2259170000E-01, 0.2753270000E-02 /
  data (epsData(105,i,16), i=1,3) / 0.1080060000E+04, -0.1838080000E-01, 0.1918970000E-02 /
  data (epsData(106,i,16), i=1,3) / 0.1195530000E+04, -0.1490230000E-01, 0.1327090000E-02 /
  data (epsData(107,i,16), i=1,3) / 0.1325990000E+04, -0.1202060000E-01, 0.9089600000E-03 /
  data (epsData(108,i,16), i=1,3) / 0.1472060000E+04, -0.9664720000E-02, 0.6193640000E-03 /
  data (epsData(109,i,16), i=1,3) / 0.1636380000E+04, -0.7734900000E-02, 0.4191350000E-03 /
  data (epsData(110,i,16), i=1,3) / 0.1818450000E+04, -0.6169120000E-02, 0.2806890000E-03 /
  data (epsData(111,i,16), i=1,3) / 0.2030060000E+04, -0.4833880000E-02, 0.1862300000E-03 /
  data (epsData(112,i,16), i=1,3) / 0.2406330000E+04, -0.3001790000E-02, 0.9844980000E-04 /
  data (epsData(113,i,16), i=1,3) / 0.2447130000E+04, -0.2667940000E-02, 0.9243160000E-04 /
  data (epsData(114,i,16), i=1,3) / 0.2462150000E+04, -0.2420180000E-02, 0.9572320000E-04 /
  data (epsData(115,i,16), i=1,3) / 0.2467180000E+04, -0.2256840000E-02, 0.1001670000E-03 /
  data (epsData(116,i,16), i=1,3) / 0.2469700000E+04, -0.2108370000E-02, 0.1088270000E-03 /
  data (epsData(117,i,16), i=1,3) / 0.2470960000E+04, -0.1975290000E-02, 0.1246640000E-03 /
  data (epsData(118,i,16), i=1,3) / 0.2472090000E+04, -0.1811640000E-02, 0.2555530000E-03 /
  data (epsData(119,i,16), i=1,3) / 0.2472750000E+04, -0.1797810000E-02, 0.4032820000E-03 /
  data (epsData(120,i,16), i=1,3) / 0.2474180000E+04, -0.1861710000E-02, 0.6023120000E-03 /
  data (epsData(121,i,16), i=1,3) / 0.2476660000E+04, -0.2007730000E-02, 0.8012110000E-03 /
  data (epsData(122,i,16), i=1,3) / 0.2479960000E+04, -0.2185040000E-02, 0.8680930000E-03 /
  data (epsData(123,i,16), i=1,3) / 0.2487140000E+04, -0.2432970000E-02, 0.9323480000E-03 /
  data (epsData(124,i,16), i=1,3) / 0.2497350000E+04, -0.2572240000E-02, 0.8929840000E-03 /
  data (epsData(125,i,16), i=1,3) / 0.2514140000E+04, -0.2667290000E-02, 0.8619440000E-03 /
  data (epsData(126,i,16), i=1,3) / 0.2574520000E+04, -0.2785140000E-02, 0.8026750000E-03 /
  data (epsData(127,i,16), i=1,3) / 0.2742650000E+04, -0.2695120000E-02, 0.6436600000E-03 /
  data (epsData(128,i,16), i=1,3) / 0.3209040000E+04, -0.2125540000E-02, 0.3667720000E-03 /
  data (epsData(129,i,16), i=1,3) / 0.3616130000E+04, -0.1714980000E-02, 0.2395340000E-03 /
  data (epsData(130,i,16), i=1,3) / 0.4034140000E+04, -0.1394540000E-02, 0.1632300000E-03 /
  data (epsData(131,i,16), i=1,3) / 0.4517120000E+04, -0.1121790000E-02, 0.1078480000E-03 /
  data (epsData(132,i,16), i=1,3) / 0.5044430000E+04, -0.9030370000E-03, 0.7159320000E-04 /
  data (epsData(133,i,16), i=1,3) / 0.5630080000E+04, -0.7259920000E-03, 0.4740710000E-04 /
  data (epsData(134,i,16), i=1,3) / 0.6282500000E+04, -0.5829150000E-03, 0.3137340000E-04 /
  data (epsData(135,i,16), i=1,3) / 0.7023400000E+04, -0.4660390000E-03, 0.2051300000E-04 /
  data (epsData(136,i,16), i=1,3) / 0.7838180000E+04, -0.3736430000E-03, 0.1342520000E-04 /
  data (epsData(137,i,16), i=1,3) / 0.8757960000E+04, -0.2986930000E-03, 0.8705370000E-05 /
  data (epsData(138,i,16), i=1,3) / 0.9788260000E+04, -0.2386500000E-03, 0.5632380000E-05 /
  data (epsData(139,i,16), i=1,3) / 0.1094420000E+05, -0.1905060000E-03, 0.3628310000E-05 /
  data (epsData(140,i,16), i=1,3) / 0.1224310000E+05, -0.1519250000E-03, 0.2323480000E-05 /
  data (epsData(141,i,16), i=1,3) / 0.1370090000E+05, -0.1210860000E-03, 0.1481820000E-05 /
  data (epsData(142,i,16), i=1,3) / 0.1533940000E+05, -0.9643480000E-04, 0.9412520000E-06 /
  data (epsData(143,i,16), i=1,3) / 0.1717910000E+05, -0.7676740000E-04, 0.5954540000E-06 /
  data (epsData(144,i,16), i=1,3) / 0.1924230000E+05, -0.6110310000E-04, 0.3753430000E-06 /
  data (epsData(145,i,16), i=1,3) / 0.2155790000E+05, -0.4862250000E-04, 0.2358830000E-06 /
  data (epsData(146,i,16), i=1,3) / 0.2415880000E+05, -0.3867560000E-04, 0.1476460000E-06 /
  data (epsData(147,i,16), i=1,3) / 0.2708190000E+05, -0.3074920000E-04, 0.9209270000E-07 /
  data (epsData(148,i,16), i=1,3) / 0.3036190000E+05, -0.2444510000E-04, 0.5724720000E-07 /
  data (epsData(149,i,16), i=1,3) / 0.3404340000E+05, -0.1943090000E-04, 0.3549500000E-07 /
  data (epsData(150,i,16), i=1,3) / 0.3817580000E+05, -0.1544310000E-04, 0.2192900000E-07 /
  data (epsData(151,i,16), i=1,3) / 0.4281380000E+05, -0.1227260000E-04, 0.1353590000E-07 /
  data (epsData(152,i,16), i=1,3) / 0.4801800000E+05, -0.9752590000E-05, 0.8313820000E-08 /
  data (epsData(153,i,16), i=1,3) / 0.5386340000E+05, -0.7748090000E-05, 0.5114810000E-08 /
  data (epsData(154,i,16), i=1,3) / 0.6042360000E+05, -0.6155280000E-05, 0.3122110000E-08 /
  data (epsData(155,i,16), i=1,3) / 0.6778790000E+05, -0.4889430000E-05, 0.1915200000E-08 /
  data (epsData(156,i,16), i=1,3) / 0.7605180000E+05, -0.3883850000E-05, 0.1171890000E-08 /
  data (epsData(157,i,16), i=1,3) / 0.8532710000E+05, -0.3084940000E-05, 0.7171980000E-09 /
  data (epsData(158,i,16), i=1,3) / 0.9572600000E+05, -0.2450880000E-05, 0.4260970000E-09 /
  data (epsData(159,i,16), i=1,3) / 0.1000000000E+06, -0.2246730000E-05, 0.3505680000E-09 /
  data (epsData(1,i,17), i=1,3) / 0.2506580000E-02, 0.6859160000E+02, 0.1115330000E+03 /
  data (epsData(2,i,17), i=1,3) / 0.4693440000E-02, 0.3222610000E+02, 0.5956520000E+02 /
  data (epsData(3,i,17), i=1,3) / 0.7560590000E-02, 0.2698600000E+02, 0.3697670000E+02 /
  data (epsData(4,i,17), i=1,3) / 0.1110800000E-01, 0.2567180000E+02, 0.2516790000E+02 /
  data (epsData(5,i,17), i=1,3) / 0.1533570000E-01, 0.2523900000E+02, 0.1822970000E+02 /
  data (epsData(6,i,17), i=1,3) / 0.2024370000E-01, 0.2507030000E+02, 0.1381000000E+02 /
  data (epsData(7,i,17), i=1,3) / 0.2583200000E-01, 0.2499650000E+02, 0.1082240000E+02 /
  data (epsData(8,i,17), i=1,3) / 0.3210060000E-01, 0.2496150000E+02, 0.8709050000E+01 /
  data (epsData(9,i,17), i=1,3) / 0.4667860000E-01, 0.2493480000E+02, 0.5989160000E+01 /
  data (epsData(10,i,17), i=1,3) / 0.7364770000E-01, 0.2492590000E+02, 0.3795990000E+01 /
  data (epsData(11,i,17), i=1,3) / 0.1322020000E+00, 0.2492630000E+02, 0.2114690000E+01 /
  data (epsData(12,i,17), i=1,3) / 0.3003320000E+00, 0.2494330000E+02, 0.9309330000E+00 /
  data (epsData(13,i,17), i=1,3) / 0.1212890000E+01, 0.2507460000E+02, 0.2589040000E+00 /
  data (epsData(14,i,17), i=1,3) / 0.3729710000E+01, 0.2650000000E+02, 0.1231630000E+00 /
  data (epsData(15,i,17), i=1,3) / 0.5901040000E+01, 0.2945940000E+02, 0.1200710000E+00 /
  data (epsData(16,i,17), i=1,3) / 0.7421760000E+01, 0.3332430000E+02, 0.1452090000E+00 /
  data (epsData(17,i,17), i=1,3) / 0.8460660000E+01, 0.3780170000E+02, 0.1927190000E+00 /
  data (epsData(18,i,17), i=1,3) / 0.9116650000E+01, 0.4242530000E+02, 0.2710470000E+00 /
  data (epsData(19,i,17), i=1,3) / 0.9567580000E+01, 0.4784730000E+02, 0.4243760000E+00 /
  data (epsData(20,i,17), i=1,3) / 0.9797130000E+01, 0.5289460000E+02, 0.9504540000E+00 /
  data (epsData(21,i,17), i=1,3) / 0.9912920000E+01, 0.5741410000E+02, 0.2062640000E+01 /
  data (epsData(22,i,17), i=1,3) / 0.1001220000E+02, 0.5975850000E+02, 0.8258280000E+01 /
  data (epsData(23,i,17), i=1,3) / 0.1017990000E+02, 0.5834570000E+02, 0.1498970000E+02 /
  data (epsData(24,i,17), i=1,3) / 0.1026670000E+02, 0.5609860000E+02, 0.1625730000E+02 /
  data (epsData(25,i,17), i=1,3) / 0.1071970000E+02, 0.5406330000E+02, 0.1950760000E+02 /
  data (epsData(26,i,17), i=1,3) / 0.1143710000E+02, 0.5518020000E+02, 0.2695060000E+02 /
  data (epsData(27,i,17), i=1,3) / 0.1196470000E+02, 0.5308540000E+02, 0.3423970000E+02 /
  data (epsData(28,i,17), i=1,3) / 0.1234320000E+02, 0.5213360000E+02, 0.3673880000E+02 /
  data (epsData(29,i,17), i=1,3) / 0.1333370000E+02, 0.4823910000E+02, 0.4959600000E+02 /
  data (epsData(30,i,17), i=1,3) / 0.1426670000E+02, 0.3958860000E+02, 0.6206850000E+02 /
  data (epsData(31,i,17), i=1,3) / 0.1498000000E+02, 0.2858160000E+02, 0.6959520000E+02 /
  data (epsData(32,i,17), i=1,3) / 0.1592660000E+02, 0.1131560000E+02, 0.7370380000E+02 /
  data (epsData(33,i,17), i=1,3) / 0.1676240000E+02, -0.4430700000E+01, 0.7216980000E+02 /
  data (epsData(34,i,17), i=1,3) / 0.1745050000E+02, -0.1644500000E+02, 0.6646940000E+02 /
  data (epsData(35,i,17), i=1,3) / 0.1817990000E+02, -0.2551930000E+02, 0.5703860000E+02 /
  data (epsData(36,i,17), i=1,3) / 0.1902890000E+02, -0.3068000000E+02, 0.4553020000E+02 /
  data (epsData(37,i,17), i=1,3) / 0.1996470000E+02, -0.3218240000E+02, 0.3471880000E+02 /
  data (epsData(38,i,17), i=1,3) / 0.2100550000E+02, -0.3114980000E+02, 0.2517370000E+02 /
  data (epsData(39,i,17), i=1,3) / 0.2211570000E+02, -0.2840710000E+02, 0.1774250000E+02 /
  data (epsData(40,i,17), i=1,3) / 0.2357290000E+02, -0.2424120000E+02, 0.1138770000E+02 /
  data (epsData(41,i,17), i=1,3) / 0.2505320000E+02, -0.2041420000E+02, 0.7448930000E+01 /
  data (epsData(42,i,17), i=1,3) / 0.2659910000E+02, -0.1707900000E+02, 0.4904740000E+01 /
  data (epsData(43,i,17), i=1,3) / 0.2818010000E+02, -0.1431760000E+02, 0.3284720000E+01 /
  data (epsData(44,i,17), i=1,3) / 0.2989440000E+02, -0.1193340000E+02, 0.2201770000E+01 /
  data (epsData(45,i,17), i=1,3) / 0.3187920000E+02, -0.9796010000E+01, 0.1465280000E+01 /
  data (epsData(46,i,17), i=1,3) / 0.3401690000E+02, -0.8047660000E+01, 0.1026170000E+01 /
  data (epsData(47,i,17), i=1,3) / 0.3646600000E+02, -0.6554110000E+01, 0.7615010000E+00 /
  data (epsData(48,i,17), i=1,3) / 0.3925890000E+02, -0.5309590000E+01, 0.6215690000E+00 /
  data (epsData(49,i,17), i=1,3) / 0.4243040000E+02, -0.4299120000E+01, 0.5566770000E+00 /
  data (epsData(50,i,17), i=1,3) / 0.4621270000E+02, -0.3458200000E+01, 0.5241630000E+00 /
  data (epsData(51,i,17), i=1,3) / 0.5089890000E+02, -0.2745880000E+01, 0.4957350000E+00 /
  data (epsData(52,i,17), i=1,3) / 0.5640070000E+02, -0.2180380000E+01, 0.4651810000E+00 /
  data (epsData(53,i,17), i=1,3) / 0.6282020000E+02, -0.1738760000E+01, 0.4247760000E+00 /
  data (epsData(54,i,17), i=1,3) / 0.7107060000E+02, -0.1357090000E+01, 0.3652830000E+00 /
  data (epsData(55,i,17), i=1,3) / 0.8030870000E+02, -0.1065180000E+01, 0.3023210000E+00 /
  data (epsData(56,i,17), i=1,3) / 0.9091630000E+02, -0.8302320000E+00, 0.2426680000E+00 /
  data (epsData(57,i,17), i=1,3) / 0.1026960000E+03, -0.6441450000E+00, 0.1909900000E+00 /
  data (epsData(58,i,17), i=1,3) / 0.1157230000E+03, -0.4939490000E+00, 0.1478710000E+00 /
  data (epsData(59,i,17), i=1,3) / 0.1296800000E+03, -0.3727780000E+00, 0.1138240000E+00 /
  data (epsData(60,i,17), i=1,3) / 0.1449840000E+03, -0.2689140000E+00, 0.8699370000E-01 /
  data (epsData(61,i,17), i=1,3) / 0.1634870000E+03, -0.1650000000E+00, 0.6446460000E-01 /
  data (epsData(62,i,17), i=1,3) / 0.1785400000E+03, -0.8206960000E-01, 0.5106010000E-01 /
  data (epsData(63,i,17), i=1,3) / 0.1856270000E+03, -0.3703180000E-01, 0.4643780000E-01 /
  data (epsData(64,i,17), i=1,3) / 0.1894690000E+03, -0.8071430000E-02, 0.4402230000E-01 /
  data (epsData(65,i,17), i=1,3) / 0.1918930000E+03, 0.1319450000E-01, 0.4259550000E-01 /
  data (epsData(66,i,17), i=1,3) / 0.1943330000E+03, 0.3833690000E-01, 0.4125900000E-01 /
  data (epsData(67,i,17), i=1,3) / 0.1962980000E+03, 0.6285590000E-01, 0.4029070000E-01 /
  data (epsData(68,i,17), i=1,3) / 0.1982740000E+03, 0.9422460000E-01, 0.3962370000E-01 /
  data (epsData(69,i,17), i=1,3) / 0.1992650000E+03, 0.1150120000E+00, 0.3955990000E-01 /
  data (epsData(70,i,17), i=1,3) / 0.1997620000E+03, 0.1282460000E+00, 0.4047340000E-01 /
  data (epsData(71,i,17), i=1,3) / 0.2003480000E+03, 0.1423360000E+00, 0.4715140000E-01 /
  data (epsData(72,i,17), i=1,3) / 0.2014370000E+03, 0.1708080000E+00, 0.5387540000E-01 /
  data (epsData(73,i,17), i=1,3) / 0.2029200000E+03, 0.2201660000E+00, 0.7821670000E-01 /
  data (epsData(74,i,17), i=1,3) / 0.2041230000E+03, 0.2643040000E+00, 0.1162730000E+00 /
  data (epsData(75,i,17), i=1,3) / 0.2052000000E+03, 0.2962750000E+00, 0.1710880000E+00 /
  data (epsData(76,i,17), i=1,3) / 0.2062670000E+03, 0.3093670000E+00, 0.2414720000E+00 /
  data (epsData(77,i,17), i=1,3) / 0.2073150000E+03, 0.2915000000E+00, 0.3142500000E+00 /
  data (epsData(78,i,17), i=1,3) / 0.2085720000E+03, 0.2410580000E+00, 0.3769090000E+00 /
  data (epsData(79,i,17), i=1,3) / 0.2100670000E+03, 0.1693370000E+00, 0.4145320000E+00 /
  data (epsData(80,i,17), i=1,3) / 0.2116000000E+03, 0.1080120000E+00, 0.4174590000E+00 /
  data (epsData(81,i,17), i=1,3) / 0.2137680000E+03, 0.5484260000E-01, 0.4044160000E+00 /
  data (epsData(82,i,17), i=1,3) / 0.2172870000E+03, 0.6459660000E-02, 0.3911070000E+00 /
  data (epsData(83,i,17), i=1,3) / 0.2220850000E+03, -0.4860310000E-01, 0.3737100000E+00 /
  data (epsData(84,i,17), i=1,3) / 0.2277230000E+03, -0.8681260000E-01, 0.3419530000E+00 /
  data (epsData(85,i,17), i=1,3) / 0.2377070000E+03, -0.1202720000E+00, 0.2960580000E+00 /
  data (epsData(86,i,17), i=1,3) / 0.2540010000E+03, -0.1368420000E+00, 0.2479770000E+00 /
  data (epsData(87,i,17), i=1,3) / 0.2666600000E+03, -0.1356980000E+00, 0.2172110000E+00 /
  data (epsData(88,i,17), i=1,3) / 0.2718630000E+03, -0.1305630000E+00, 0.2150800000E+00 /
  data (epsData(89,i,17), i=1,3) / 0.2838790000E+03, -0.1434810000E+00, 0.2027950000E+00 /
  data (epsData(90,i,17), i=1,3) / 0.3084010000E+03, -0.1485090000E+00, 0.1624510000E+00 /
  data (epsData(91,i,17), i=1,3) / 0.3394930000E+03, -0.1410850000E+00, 0.1230620000E+00 /
  data (epsData(92,i,17), i=1,3) / 0.3840310000E+03, -0.1240230000E+00, 0.8473090000E-01 /
  data (epsData(93,i,17), i=1,3) / 0.4319450000E+03, -0.1051410000E+00, 0.5849000000E-01 /
  data (epsData(94,i,17), i=1,3) / 0.4844430000E+03, -0.8734370000E-01, 0.4010750000E-01 /
  data (epsData(95,i,17), i=1,3) / 0.5382240000E+03, -0.7273690000E-01, 0.2821490000E-01 /
  data (epsData(96,i,17), i=1,3) / 0.5966810000E+03, -0.6002120000E-01, 0.1996500000E-01 /
  data (epsData(97,i,17), i=1,3) / 0.6719130000E+03, -0.4791480000E-01, 0.1323050000E-01 /
  data (epsData(98,i,17), i=1,3) / 0.7549340000E+03, -0.3817330000E-01, 0.8787760000E-02 /
  data (epsData(99,i,17), i=1,3) / 0.8448590000E+03, -0.3051610000E-01, 0.5899490000E-02 /
  data (epsData(100,i,17), i=1,3) / 0.9414710000E+03, -0.2452740000E-01, 0.4021470000E-02 /
  data (epsData(101,i,17), i=1,3) / 0.1044560000E+04, -0.1984400000E-01, 0.2776380000E-02 /
  data (epsData(102,i,17), i=1,3) / 0.1158300000E+04, -0.1605060000E-01, 0.1913200000E-02 /
  data (epsData(103,i,17), i=1,3) / 0.1285700000E+04, -0.1294200000E-01, 0.1309460000E-02 /
  data (epsData(104,i,17), i=1,3) / 0.1429150000E+04, -0.1039410000E-01, 0.8887320000E-03 /
  data (epsData(105,i,17), i=1,3) / 0.1590620000E+04, -0.8315200000E-02, 0.6002460000E-03 /
  data (epsData(106,i,17), i=1,3) / 0.1775670000E+04, -0.6600450000E-02, 0.3972180000E-03 /
  data (epsData(107,i,17), i=1,3) / 0.1977190000E+04, -0.5252440000E-02, 0.2669570000E-03 /
  data (epsData(108,i,17), i=1,3) / 0.2203640000E+04, -0.4149170000E-02, 0.1782380000E-03 /
  data (epsData(109,i,17), i=1,3) / 0.2470020000E+04, -0.3188990000E-02, 0.1162010000E-03 /
  data (epsData(110,i,17), i=1,3) / 0.2745270000E+04, -0.2311840000E-02, 0.7809240000E-04 /
  data (epsData(111,i,17), i=1,3) / 0.2794030000E+04, -0.2049180000E-02, 0.7306420000E-04 /
  data (epsData(112,i,17), i=1,3) / 0.2812240000E+04, -0.1842760000E-02, 0.7556700000E-04 /
  data (epsData(113,i,17), i=1,3) / 0.2818780000E+04, -0.1674580000E-02, 0.8137510000E-04 /
  data (epsData(114,i,17), i=1,3) / 0.2820090000E+04, -0.1611960000E-02, 0.8575800000E-04 /
  data (epsData(115,i,17), i=1,3) / 0.2821400000E+04, -0.1515920000E-02, 0.9776050000E-04 /
  data (epsData(116,i,17), i=1,3) / 0.2822540000E+04, -0.1399880000E-02, 0.1847490000E-03 /
  data (epsData(117,i,17), i=1,3) / 0.2823470000E+04, -0.1378580000E-02, 0.3180500000E-03 /
  data (epsData(118,i,17), i=1,3) / 0.2825670000E+04, -0.1442820000E-02, 0.5065800000E-03 /
  data (epsData(119,i,17), i=1,3) / 0.2827970000E+04, -0.1560760000E-02, 0.6139330000E-03 /
  data (epsData(120,i,17), i=1,3) / 0.2836280000E+04, -0.1809670000E-02, 0.7150280000E-03 /
  data (epsData(121,i,17), i=1,3) / 0.2843830000E+04, -0.1912050000E-02, 0.6834710000E-03 /
  data (epsData(122,i,17), i=1,3) / 0.2862890000E+04, -0.2020690000E-02, 0.6674490000E-03 /
  data (epsData(123,i,17), i=1,3) / 0.2907440000E+04, -0.2114810000E-02, 0.6296540000E-03 /
  data (epsData(124,i,17), i=1,3) / 0.3061270000E+04, -0.2109150000E-02, 0.5312690000E-03 /
  data (epsData(125,i,17), i=1,3) / 0.3449700000E+04, -0.1798930000E-02, 0.3453860000E-03 /
  data (epsData(126,i,17), i=1,3) / 0.3909850000E+04, -0.1449180000E-02, 0.2210210000E-03 /
  data (epsData(127,i,17), i=1,3) / 0.4374360000E+04, -0.1175910000E-02, 0.1493110000E-03 /
  data (epsData(128,i,17), i=1,3) / 0.4902700000E+04, -0.9463060000E-03, 0.9846180000E-04 /
  data (epsData(129,i,17), i=1,3) / 0.5477800000E+04, -0.7619420000E-03, 0.6532120000E-04 /
  data (epsData(130,i,17), i=1,3) / 0.6116660000E+04, -0.6123520000E-03, 0.4322660000E-04 /
  data (epsData(131,i,17), i=1,3) / 0.6840550000E+04, -0.4899090000E-03, 0.2834520000E-04 /
  data (epsData(132,i,17), i=1,3) / 0.7646480000E+04, -0.3918120000E-03, 0.1857880000E-04 /
  data (epsData(133,i,17), i=1,3) / 0.8531590000E+04, -0.3143150000E-03, 0.1217190000E-04 /
  data (epsData(134,i,17), i=1,3) / 0.9530420000E+04, -0.2514030000E-03, 0.7924170000E-05 /
  data (epsData(135,i,17), i=1,3) / 0.1064880000E+05, -0.2009580000E-03, 0.5129960000E-05 /
  data (epsData(136,i,17), i=1,3) / 0.1190610000E+05, -0.1604230000E-03, 0.3305840000E-05 /
  data (epsData(137,i,17), i=1,3) / 0.1331730000E+05, -0.1279650000E-03, 0.2121240000E-05 /
  data (epsData(138,i,17), i=1,3) / 0.1490320000E+05, -0.1019810000E-03, 0.1352770000E-05 /
  data (epsData(139,i,17), i=1,3) / 0.1668300000E+05, -0.8123740000E-04, 0.8598860000E-06 /
  data (epsData(140,i,17), i=1,3) / 0.1868320000E+05, -0.6466840000E-04, 0.5445700000E-06 /
  data (epsData(141,i,17), i=1,3) / 0.2092730000E+05, -0.5146840000E-04, 0.3434780000E-06 /
  data (epsData(142,i,17), i=1,3) / 0.2344680000E+05, -0.4094900000E-04, 0.2158320000E-06 /
  data (epsData(143,i,17), i=1,3) / 0.2627700000E+05, -0.3256640000E-04, 0.1351730000E-06 /
  data (epsData(144,i,17), i=1,3) / 0.2945150000E+05, -0.2589920000E-04, 0.8436570000E-07 /
  data (epsData(145,i,17), i=1,3) / 0.3301930000E+05, -0.2058740000E-04, 0.5245840000E-07 /
  data (epsData(146,i,17), i=1,3) / 0.3702260000E+05, -0.1636410000E-04, 0.3250930000E-07 /
  data (epsData(147,i,17), i=1,3) / 0.4151410000E+05, -0.1300680000E-04, 0.2011250000E-07 /
  data (epsData(148,i,17), i=1,3) / 0.4656000000E+05, -0.1033510000E-04, 0.1238750000E-07 /
  data (epsData(149,i,17), i=1,3) / 0.5222610000E+05, -0.8210670000E-05, 0.7606780000E-08 /
  data (epsData(150,i,17), i=1,3) / 0.5858520000E+05, -0.6522620000E-05, 0.4657590000E-08 /
  data (epsData(151,i,17), i=1,3) / 0.6572080000E+05, -0.5181600000E-05, 0.2852260000E-08 /
  data (epsData(152,i,17), i=1,3) / 0.7372700000E+05, -0.4116350000E-05, 0.1747900000E-08 /
  data (epsData(153,i,17), i=1,3) / 0.8271200000E+05, -0.3269980000E-05, 0.1064260000E-08 /
  data (epsData(154,i,17), i=1,3) / 0.9279600000E+05, -0.2597570000E-05, 0.6418770000E-09 /
  data (epsData(155,i,17), i=1,3) / 0.1000000000E+06, -0.2237980000E-05, 0.4602690000E-09 /
  data (epsData(1,i,18), i=1,3) / 0.2506580000E-02, -0.1269040000E+03, 0.2981160000E+03 /
  data (epsData(2,i,18), i=1,3) / 0.4693440000E-02, -0.9794250000E+02, 0.9636350000E+02 /
  data (epsData(3,i,18), i=1,3) / 0.7560590000E-02, -0.4114290000E+02, 0.4286190000E+02 /
  data (epsData(4,i,18), i=1,3) / 0.1110800000E-01, -0.1658340000E+02, 0.2313940000E+02 /
  data (epsData(5,i,18), i=1,3) / 0.1533570000E-01, -0.5074700000E+01, 0.1419240000E+02 /
  data (epsData(6,i,18), i=1,3) / 0.2024370000E-01, 0.8671670000E+00, 0.9512260000E+01 /
  data (epsData(7,i,18), i=1,3) / 0.2583200000E-01, 0.4188620000E+01, 0.6798000000E+01 /
  data (epsData(8,i,18), i=1,3) / 0.3210060000E-01, 0.6166970000E+01, 0.5096800000E+01 /
  data (epsData(9,i,18), i=1,3) / 0.3904940000E-01, 0.7407330000E+01, 0.3964550000E+01 /
  data (epsData(10,i,18), i=1,3) / 0.4667860000E-01, 0.8218450000E+01, 0.3174300000E+01 /
  data (epsData(11,i,18), i=1,3) / 0.5498800000E-01, 0.8767850000E+01, 0.2601200000E+01 /
  data (epsData(12,i,18), i=1,3) / 0.6397770000E-01, 0.9151240000E+01, 0.2172310000E+01 /
  data (epsData(13,i,18), i=1,3) / 0.8399800000E-01, 0.9626550000E+01, 0.1584180000E+01 /
  data (epsData(14,i,18), i=1,3) / 0.1191310000E+00, 0.9977950000E+01, 0.1070230000E+01 /
  data (epsData(15,i,18), i=1,3) / 0.1912910000E+00, 0.1020780000E+02, 0.6402410000E+00 /
  data (epsData(16,i,18), i=1,3) / 0.4099070000E+00, 0.1033660000E+02, 0.2879860000E+00 /
  data (epsData(17,i,18), i=1,3) / 0.2161410000E+01, 0.1046520000E+02, 0.6168450000E-01 /
  data (epsData(18,i,18), i=1,3) / 0.6081600000E+01, 0.1117840000E+02, 0.3278580000E-01 /
  data (epsData(19,i,18), i=1,3) / 0.9228360000E+01, 0.1249190000E+02, 0.3340500000E-01 /
  data (epsData(20,i,18), i=1,3) / 0.1160550000E+02, 0.1430910000E+02, 0.4249740000E-01 /
  data (epsData(21,i,18), i=1,3) / 0.1329650000E+02, 0.1652720000E+02, 0.5750360000E-01 /
  data (epsData(22,i,18), i=1,3) / 0.1439430000E+02, 0.1892640000E+02, 0.8179340000E-01 /
  data (epsData(23,i,18), i=1,3) / 0.1510250000E+02, 0.2156980000E+02, 0.1324750000E+00 /
  data (epsData(24,i,18), i=1,3) / 0.1539060000E+02, 0.2343880000E+02, 0.2342830000E+00 /
  data (epsData(25,i,18), i=1,3) / 0.1553560000E+02, 0.2497980000E+02, 0.3539710000E+00 /
  data (epsData(26,i,18), i=1,3) / 0.1571220000E+02, 0.2681310000E+02, 0.2414380000E+01 /
  data (epsData(27,i,18), i=1,3) / 0.1583610000E+02, 0.2755560000E+02, 0.4147180000E+01 /
  data (epsData(28,i,18), i=1,3) / 0.1591670000E+02, 0.2708180000E+02, 0.5521890000E+01 /
  data (epsData(29,i,18), i=1,3) / 0.1618610000E+02, 0.2585420000E+02, 0.7020270000E+01 /
  data (epsData(30,i,18), i=1,3) / 0.1675040000E+02, 0.2581830000E+02, 0.8498890000E+01 /
  data (epsData(31,i,18), i=1,3) / 0.1830000000E+02, 0.2690560000E+02, 0.1500360000E+02 /
  data (epsData(32,i,18), i=1,3) / 0.1936980000E+02, 0.2700080000E+02, 0.2092740000E+02 /
  data (epsData(33,i,18), i=1,3) / 0.2032050000E+02, 0.2515020000E+02, 0.2769740000E+02 /
  data (epsData(34,i,18), i=1,3) / 0.2121470000E+02, 0.2053460000E+02, 0.3403840000E+02 /
  data (epsData(35,i,18), i=1,3) / 0.2209910000E+02, 0.1349700000E+02, 0.3871120000E+02 /
  data (epsData(36,i,18), i=1,3) / 0.2285560000E+02, 0.5981640000E+01, 0.4070260000E+02 /
  data (epsData(37,i,18), i=1,3) / 0.2355320000E+02, -0.1448460000E+01, 0.4004660000E+02 /
  data (epsData(38,i,18), i=1,3) / 0.2417770000E+02, -0.7261370000E+01, 0.3735530000E+02 /
  data (epsData(39,i,18), i=1,3) / 0.2493720000E+02, -0.1221410000E+02, 0.3270640000E+02 /
  data (epsData(40,i,18), i=1,3) / 0.2596280000E+02, -0.1593720000E+02, 0.2628520000E+02 /
  data (epsData(41,i,18), i=1,3) / 0.2704340000E+02, -0.1745870000E+02, 0.2002180000E+02 /
  data (epsData(42,i,18), i=1,3) / 0.2805020000E+02, -0.1719170000E+02, 0.1517160000E+02 /
  data (epsData(43,i,18), i=1,3) / 0.2931670000E+02, -0.1580620000E+02, 0.1075650000E+02 /
  data (epsData(44,i,18), i=1,3) / 0.3096670000E+02, -0.1366440000E+02, 0.7020230000E+01 /
  data (epsData(45,i,18), i=1,3) / 0.3270140000E+02, -0.1156580000E+02, 0.4565570000E+01 /
  data (epsData(46,i,18), i=1,3) / 0.3453000000E+02, -0.9677050000E+01, 0.2956630000E+01 /
  data (epsData(47,i,18), i=1,3) / 0.3645150000E+02, -0.8048590000E+01, 0.1910300000E+01 /
  data (epsData(48,i,18), i=1,3) / 0.3844850000E+02, -0.6682750000E+01, 0.1251070000E+01 /
  data (epsData(49,i,18), i=1,3) / 0.4056690000E+02, -0.5540140000E+01, 0.8513970000E+00 /
  data (epsData(50,i,18), i=1,3) / 0.4304210000E+02, -0.4526560000E+01, 0.5992160000E+00 /
  data (epsData(51,i,18), i=1,3) / 0.4576290000E+02, -0.3694230000E+01, 0.4610730000E+00 /
  data (epsData(52,i,18), i=1,3) / 0.4889220000E+02, -0.2995460000E+01, 0.3946420000E+00 /
  data (epsData(53,i,18), i=1,3) / 0.5247080000E+02, -0.2428370000E+01, 0.3720860000E+00 /
  data (epsData(54,i,18), i=1,3) / 0.5692800000E+02, -0.1943040000E+01, 0.3639530000E+00 /
  data (epsData(55,i,18), i=1,3) / 0.6241080000E+02, -0.1541420000E+01, 0.3554050000E+00 /
  data (epsData(56,i,18), i=1,3) / 0.6862040000E+02, -0.1238690000E+01, 0.3426140000E+00 /
  data (epsData(57,i,18), i=1,3) / 0.7637020000E+02, -0.9891080000E+00, 0.3163670000E+00 /
  data (epsData(58,i,18), i=1,3) / 0.8647790000E+02, -0.7730050000E+00, 0.2730280000E+00 /
  data (epsData(59,i,18), i=1,3) / 0.9848560000E+02, -0.6018570000E+00, 0.2245580000E+00 /
  data (epsData(60,i,18), i=1,3) / 0.1119720000E+03, -0.4684330000E+00, 0.1780760000E+00 /
  data (epsData(61,i,18), i=1,3) / 0.1270230000E+03, -0.3612180000E+00, 0.1375440000E+00 /
  data (epsData(62,i,18), i=1,3) / 0.1429380000E+03, -0.2768650000E+00, 0.1058110000E+00 /
  data (epsData(63,i,18), i=1,3) / 0.1600530000E+03, -0.2070780000E+00, 0.8127440000E-01 /
  data (epsData(64,i,18), i=1,3) / 0.1797220000E+03, -0.1434880000E+00, 0.6095310000E-01 /
  data (epsData(65,i,18), i=1,3) / 0.2056330000E+03, -0.7205160000E-01, 0.4284550000E-01 /
  data (epsData(66,i,18), i=1,3) / 0.2201240000E+03, -0.3085000000E-01, 0.3567270000E-01 /
  data (epsData(67,i,18), i=1,3) / 0.2273250000E+03, -0.7029200000E-02, 0.3256500000E-01 /
  data (epsData(68,i,18), i=1,3) / 0.2320240000E+03, 0.1106360000E-01, 0.3102170000E-01 /
  data (epsData(69,i,18), i=1,3) / 0.2362470000E+03, 0.3002280000E-01, 0.2964330000E-01 /
  data (epsData(70,i,18), i=1,3) / 0.2399780000E+03, 0.5047040000E-01, 0.2851860000E-01 /
  data (epsData(71,i,18), i=1,3) / 0.2432030000E+03, 0.7290590000E-01, 0.2765460000E-01 /
  data (epsData(72,i,18), i=1,3) / 0.2459090000E+03, 0.9800390000E-01, 0.2710610000E-01 /
  data (epsData(73,i,18), i=1,3) / 0.2475410000E+03, 0.1189390000E+00, 0.2702320000E-01 /
  data (epsData(74,i,18), i=1,3) / 0.2484280000E+03, 0.1347350000E+00, 0.2968020000E-01 /
  data (epsData(75,i,18), i=1,3) / 0.2494670000E+03, 0.1500230000E+00, 0.3444690000E-01 /
  data (epsData(76,i,18), i=1,3) / 0.2505230000E+03, 0.1724570000E+00, 0.4009690000E-01 /
  data (epsData(77,i,18), i=1,3) / 0.2519500000E+03, 0.2063860000E+00, 0.5776810000E-01 /
  data (epsData(78,i,18), i=1,3) / 0.2532340000E+03, 0.2413900000E+00, 0.8866450000E-01 /
  data (epsData(79,i,18), i=1,3) / 0.2543500000E+03, 0.2659380000E+00, 0.1320290000E+00 /
  data (epsData(80,i,18), i=1,3) / 0.2554170000E+03, 0.2742960000E+00, 0.1864130000E+00 /
  data (epsData(81,i,18), i=1,3) / 0.2565270000E+03, 0.2582960000E+00, 0.2429240000E+00 /
  data (epsData(82,i,18), i=1,3) / 0.2579500000E+03, 0.2162390000E+00, 0.2959850000E+00 /
  data (epsData(83,i,18), i=1,3) / 0.2594170000E+03, 0.1603660000E+00, 0.3233180000E+00 /
  data (epsData(84,i,18), i=1,3) / 0.2611360000E+03, 0.1083540000E+00, 0.3249340000E+00 /
  data (epsData(85,i,18), i=1,3) / 0.2636670000E+03, 0.6231540000E-01, 0.3137970000E+00 /
  data (epsData(86,i,18), i=1,3) / 0.2695450000E+03, -0.3462850000E-02, 0.2951750000E+00 /
  data (epsData(87,i,18), i=1,3) / 0.2739490000E+03, -0.3523030000E-01, 0.2718470000E+00 /
  data (epsData(88,i,18), i=1,3) / 0.2806040000E+03, -0.5931970000E-01, 0.2432210000E+00 /
  data (epsData(89,i,18), i=1,3) / 0.2883180000E+03, -0.7307170000E-01, 0.2163620000E+00 /
  data (epsData(90,i,18), i=1,3) / 0.3077780000E+03, -0.8786380000E-01, 0.1770420000E+00 /
  data (epsData(91,i,18), i=1,3) / 0.3219500000E+03, -0.8767020000E-01, 0.1547380000E+00 /
  data (epsData(92,i,18), i=1,3) / 0.3283170000E+03, -0.8389050000E-01, 0.1523680000E+00 /
  data (epsData(93,i,18), i=1,3) / 0.3413000000E+03, -0.9343790000E-01, 0.1444030000E+00 /
  data (epsData(94,i,18), i=1,3) / 0.3677300000E+03, -0.9812590000E-01, 0.1174310000E+00 /
  data (epsData(95,i,18), i=1,3) / 0.4030450000E+03, -0.9477170000E-01, 0.8958830000E-01 /
  data (epsData(96,i,18), i=1,3) / 0.4503910000E+03, -0.8526260000E-01, 0.6361960000E-01 /
  data (epsData(97,i,18), i=1,3) / 0.5070690000E+03, -0.7269640000E-01, 0.4343910000E-01 /
  data (epsData(98,i,18), i=1,3) / 0.5674540000E+03, -0.6080300000E-01, 0.2985820000E-01 /
  data (epsData(99,i,18), i=1,3) / 0.6319950000E+03, -0.5037240000E-01, 0.2071200000E-01 /
  data (epsData(100,i,18), i=1,3) / 0.7106610000E+03, -0.4065020000E-01, 0.1382310000E-01 /
  data (epsData(101,i,18), i=1,3) / 0.7959230000E+03, -0.3273770000E-01, 0.9306030000E-02 /
  data (epsData(102,i,18), i=1,3) / 0.8870950000E+03, -0.2645190000E-01, 0.6349740000E-02 /
  data (epsData(103,i,18), i=1,3) / 0.9844560000E+03, -0.2147100000E-01, 0.4387670000E-02 /
  data (epsData(104,i,18), i=1,3) / 0.1089610000E+04, -0.1747340000E-01, 0.3052540000E-02 /
  data (epsData(105,i,18), i=1,3) / 0.1204730000E+04, -0.1422510000E-01, 0.2124470000E-02 /
  data (epsData(106,i,18), i=1,3) / 0.1333780000E+04, -0.1153410000E-01, 0.1467020000E-02 /
  data (epsData(107,i,18), i=1,3) / 0.1480430000E+04, -0.9292460000E-02, 0.1000420000E-02 /
  data (epsData(108,i,18), i=1,3) / 0.1645070000E+04, -0.7460210000E-02, 0.6802800000E-03 /
  data (epsData(109,i,18), i=1,3) / 0.1834970000E+04, -0.5935830000E-02, 0.4514070000E-03 /
  data (epsData(110,i,18), i=1,3) / 0.2041200000E+04, -0.4741200000E-02, 0.3048560000E-03 /
  data (epsData(111,i,18), i=1,3) / 0.2271040000E+04, -0.3774410000E-02, 0.2048450000E-03 /
  data (epsData(112,i,18), i=1,3) / 0.2530690000E+04, -0.2976890000E-02, 0.1365290000E-03 /
  data (epsData(113,i,18), i=1,3) / 0.2840640000E+04, -0.2269000000E-02, 0.8839480000E-04 /
  data (epsData(114,i,18), i=1,3) / 0.3116570000E+04, -0.1692150000E-02, 0.6228550000E-04 /
  data (epsData(115,i,18), i=1,3) / 0.3172540000E+04, -0.1495060000E-02, 0.5822410000E-04 /
  data (epsData(116,i,18), i=1,3) / 0.3192820000E+04, -0.1349270000E-02, 0.5957140000E-04 /
  data (epsData(117,i,18), i=1,3) / 0.3200980000E+04, -0.1224530000E-02, 0.6301280000E-04 /
  data (epsData(118,i,18), i=1,3) / 0.3203700000E+04, -0.1140290000E-02, 0.6863790000E-04 /
  data (epsData(119,i,18), i=1,3) / 0.3205070000E+04, -0.1066550000E-02, 0.7986470000E-04 /
  data (epsData(120,i,18), i=1,3) / 0.3206100000E+04, -0.9896460000E-03, 0.1381670000E-03 /
  data (epsData(121,i,18), i=1,3) / 0.3207080000E+04, -0.9703410000E-03, 0.2386790000E-03 /
  data (epsData(122,i,18), i=1,3) / 0.3209510000E+04, -0.1016700000E-02, 0.3912800000E-03 /
  data (epsData(123,i,18), i=1,3) / 0.3212100000E+04, -0.1109190000E-02, 0.4724760000E-03 /
  data (epsData(124,i,18), i=1,3) / 0.3218970000E+04, -0.1268640000E-02, 0.5491790000E-03 /
  data (epsData(125,i,18), i=1,3) / 0.3225660000E+04, -0.1352370000E-02, 0.5240390000E-03 /
  data (epsData(126,i,18), i=1,3) / 0.3252970000E+04, -0.1481200000E-02, 0.5179370000E-03 /
  data (epsData(127,i,18), i=1,3) / 0.3313720000E+04, -0.1554230000E-02, 0.4796700000E-03 /
  data (epsData(128,i,18), i=1,3) / 0.3500100000E+04, -0.1540260000E-02, 0.3987870000E-03 /
  data (epsData(129,i,18), i=1,3) / 0.3970230000E+04, -0.1297070000E-02, 0.2533400000E-03 /
  data (epsData(130,i,18), i=1,3) / 0.4492710000E+04, -0.1046140000E-02, 0.1626780000E-03 /
  data (epsData(131,i,18), i=1,3) / 0.4978820000E+04, -0.8649130000E-03, 0.1135240000E-03 /
  data (epsData(132,i,18), i=1,3) / 0.5571260000E+04, -0.6977130000E-03, 0.7518950000E-04 /
  data (epsData(133,i,18), i=1,3) / 0.6224590000E+04, -0.5616430000E-03, 0.4982980000E-04 /
  data (epsData(134,i,18), i=1,3) / 0.6964220000E+04, -0.4497440000E-03, 0.3279980000E-04 /
  data (epsData(135,i,18), i=1,3) / 0.7769750000E+04, -0.3614830000E-03, 0.2167100000E-04 /
  data (epsData(136,i,18), i=1,3) / 0.8669770000E+04, -0.2900630000E-03, 0.1419920000E-04 /
  data (epsData(137,i,18), i=1,3) / 0.9683350000E+04, -0.2321550000E-03, 0.9290530000E-05 /
  data (epsData(138,i,18), i=1,3) / 0.1081530000E+05, -0.1857280000E-03, 0.6043750000E-05 /
  data (epsData(139,i,18), i=1,3) / 0.1208340000E+05, -0.1484680000E-03, 0.3913700000E-05 /
  data (epsData(140,i,18), i=1,3) / 0.1350520000E+05, -0.1185900000E-03, 0.2521610000E-05 /
  data (epsData(141,i,18), i=1,3) / 0.1510090000E+05, -0.9464580000E-04, 0.1619060000E-05 /
  data (epsData(142,i,18), i=1,3) / 0.1689340000E+05, -0.7547300000E-04, 0.1033960000E-05 /
  data (epsData(143,i,18), i=1,3) / 0.1890450000E+05, -0.6015610000E-04, 0.6577870000E-06 /
  data (epsData(144,i,18), i=1,3) / 0.2116720000E+05, -0.4790110000E-04, 0.4162410000E-06 /
  data (epsData(145,i,18), i=1,3) / 0.2370860000E+05, -0.3812390000E-04, 0.2623950000E-06 /
  data (epsData(146,i,18), i=1,3) / 0.2655890000E+05, -0.3033930000E-04, 0.1648180000E-06 /
  data (epsData(147,i,18), i=1,3) / 0.2976280000E+05, -0.2413070000E-04, 0.1031270000E-06 /
  data (epsData(148,i,18), i=1,3) / 0.3335800000E+05, -0.1919000000E-04, 0.6430770000E-07 /
  data (epsData(149,i,18), i=1,3) / 0.3739250000E+05, -0.1525900000E-04, 0.3998210000E-07 /
  data (epsData(150,i,18), i=1,3) / 0.4192670000E+05, -0.1212810000E-04, 0.2476010000E-07 /
  data (epsData(151,i,18), i=1,3) / 0.4701300000E+05, -0.9639670000E-05, 0.1528380000E-07 /
  data (epsData(152,i,18), i=1,3) / 0.5272450000E+05, -0.7660210000E-05, 0.9407000000E-08 /
  data (epsData(153,i,18), i=1,3) / 0.5913430000E+05, -0.6086860000E-05, 0.5776870000E-08 /
  data (epsData(154,i,18), i=1,3) / 0.6633570000E+05, -0.4835220000E-05, 0.3541400000E-08 /
  data (epsData(155,i,18), i=1,3) / 0.7441520000E+05, -0.3841130000E-05, 0.2163610000E-08 /
  data (epsData(156,i,18), i=1,3) / 0.8348150000E+05, -0.3051380000E-05, 0.1317370000E-08 /
  data (epsData(157,i,18), i=1,3) / 0.9365560000E+05, -0.2424050000E-05, 0.7975990000E-09 /
  data (epsData(158,i,18), i=1,3) / 0.1000000000E+06, -0.2127550000E-05, 0.5959660000E-09 /
  data (epsData(1,i,19), i=1,3) / 0.2506580000E-02, -0.2877150000E+03, 0.7924200000E+03 /
  data (epsData(2,i,19), i=1,3) / 0.4693440000E-02, -0.2318630000E+03, 0.2666690000E+03 /
  data (epsData(3,i,19), i=1,3) / 0.7560590000E-02, -0.9274340000E+02, 0.1233060000E+03 /
  data (epsData(4,i,19), i=1,3) / 0.1110800000E-01, -0.3216330000E+02, 0.6889820000E+02 /
  data (epsData(5,i,19), i=1,3) / 0.1533570000E-01, -0.3692900000E+01, 0.4350860000E+02 /
  data (epsData(6,i,19), i=1,3) / 0.2024370000E-01, 0.1103140000E+02, 0.2987370000E+02 /
  data (epsData(7,i,19), i=1,3) / 0.2583200000E-01, 0.1927220000E+02, 0.2177610000E+02 /
  data (epsData(8,i,19), i=1,3) / 0.3210060000E-01, 0.2418590000E+02, 0.1659300000E+02 /
  data (epsData(9,i,19), i=1,3) / 0.3904940000E-01, 0.2727020000E+02, 0.1307920000E+02 /
  data (epsData(10,i,19), i=1,3) / 0.4667860000E-01, 0.2929030000E+02, 0.1058710000E+02 /
  data (epsData(11,i,19), i=1,3) / 0.5498800000E-01, 0.3066220000E+02, 0.8754600000E+01 /
  data (epsData(12,i,19), i=1,3) / 0.7364770000E-01, 0.3231970000E+02, 0.6289240000E+01 /
  data (epsData(13,i,19), i=1,3) / 0.1067390000E+00, 0.3354120000E+02, 0.4263270000E+01 /
  data (epsData(14,i,19), i=1,3) / 0.1754980000E+00, 0.3416440000E+02, 0.2628860000E+01 /
  data (epsData(15,i,19), i=1,3) / 0.3866320000E+00, 0.3456020000E+02, 0.1220520000E+01 /
  data (epsData(16,i,19), i=1,3) / 0.1172610000E+01, 0.3587000000E+02, 0.5241180000E+00 /
  data (epsData(17,i,19), i=1,3) / 0.2001780000E+01, 0.3927340000E+02, 0.4565140000E+00 /
  data (epsData(18,i,19), i=1,3) / 0.2557670000E+01, 0.4375390000E+02, 0.5274350000E+00 /
  data (epsData(19,i,19), i=1,3) / 0.2987270000E+01, 0.4978550000E+02, 0.6990640000E+00 /
  data (epsData(20,i,19), i=1,3) / 0.3314540000E+01, 0.5768690000E+02, 0.1005150000E+01 /
  data (epsData(21,i,19), i=1,3) / 0.3519060000E+01, 0.6583240000E+02, 0.1515460000E+01 /
  data (epsData(22,i,19), i=1,3) / 0.3658810000E+01, 0.7451620000E+02, 0.1924290000E+01 /
  data (epsData(23,i,19), i=1,3) / 0.3729710000E+01, 0.8065030000E+02, 0.3065860000E+01 /
  data (epsData(24,i,19), i=1,3) / 0.3801290000E+01, 0.8798860000E+02, 0.4951750000E+01 /
  data (epsData(25,i,19), i=1,3) / 0.3873540000E+01, 0.9807330000E+02, 0.6713360000E+01 /
  data (epsData(26,i,19), i=1,3) / 0.3946480000E+01, 0.1161510000E+03, 0.8358570000E+01 /
  data (epsData(27,i,19), i=1,3) / 0.4020090000E+01, 0.1381150000E+03, 0.3039680000E+02 /
  data (epsData(28,i,19), i=1,3) / 0.4094390000E+01, 0.1384810000E+03, 0.8248240000E+02 /
  data (epsData(29,i,19), i=1,3) / 0.4169370000E+01, 0.1190890000E+03, 0.1055320000E+03 /
  data (epsData(30,i,19), i=1,3) / 0.4245030000E+01, 0.9534320000E+02, 0.1271030000E+03 /
  data (epsData(31,i,19), i=1,3) / 0.4321360000E+01, 0.6598000000E+02, 0.1365120000E+03 /
  data (epsData(32,i,19), i=1,3) / 0.4398380000E+01, 0.4121320000E+02, 0.1284890000E+03 /
  data (epsData(33,i,19), i=1,3) / 0.4476080000E+01, 0.2497190000E+02, 0.1208680000E+03 /
  data (epsData(34,i,19), i=1,3) / 0.4554460000E+01, 0.1257460000E+02, 0.1136260000E+03 /
  data (epsData(35,i,19), i=1,3) / 0.4633520000E+01, 0.2531950000E+01, 0.1060180000E+03 /
  data (epsData(36,i,19), i=1,3) / 0.4713260000E+01, -0.5385640000E+01, 0.9815350000E+02 /
  data (epsData(37,i,19), i=1,3) / 0.4874780000E+01, -0.1647710000E+02, 0.8359920000E+02 /
  data (epsData(38,i,19), i=1,3) / 0.5039020000E+01, -0.2333700000E+02, 0.7003940000E+02 /
  data (epsData(39,i,19), i=1,3) / 0.5205980000E+01, -0.2693300000E+02, 0.5779870000E+02 /
  data (epsData(40,i,19), i=1,3) / 0.5375660000E+01, -0.2832800000E+02, 0.4714120000E+02 /
  data (epsData(41,i,19), i=1,3) / 0.5548060000E+01, -0.2795340000E+02, 0.3779840000E+02 /
  data (epsData(42,i,19), i=1,3) / 0.5811770000E+01, -0.2548130000E+02, 0.2636750000E+02 /
  data (epsData(43,i,19), i=1,3) / 0.6081600000E+01, -0.2161900000E+02, 0.1779820000E+02 /
  data (epsData(44,i,19), i=1,3) / 0.6357560000E+01, -0.1719230000E+02, 0.1162790000E+02 /
  data (epsData(45,i,19), i=1,3) / 0.6639630000E+01, -0.1269760000E+02, 0.7267860000E+01 /
  data (epsData(46,i,19), i=1,3) / 0.6831080000E+01, -0.9745410000E+01, 0.5178420000E+01 /
  data (epsData(47,i,19), i=1,3) / 0.7025260000E+01, -0.6798250000E+01, 0.3587110000E+01 /
  data (epsData(48,i,19), i=1,3) / 0.7222150000E+01, -0.4159200000E+01, 0.2895620000E+01 /
  data (epsData(49,i,19), i=1,3) / 0.7321620000E+01, -0.2964470000E+01, 0.2669520000E+01 /
  data (epsData(50,i,19), i=1,3) / 0.7421760000E+01, -0.1916270000E+01, 0.2677470000E+01 /
  data (epsData(51,i,19), i=1,3) / 0.7522590000E+01, -0.1070600000E+01, 0.2684090000E+01 /
  data (epsData(52,i,19), i=1,3) / 0.7624100000E+01, -0.3603820000E+00, 0.2659110000E+01 /
  data (epsData(53,i,19), i=1,3) / 0.7829160000E+01, 0.8742440000E+00, 0.2484080000E+01 /
  data (epsData(54,i,19), i=1,3) / 0.8247440000E+01, 0.3004410000E+01, 0.1995670000E+01 /
  data (epsData(55,i,19), i=1,3) / 0.8568290000E+01, 0.4424240000E+01, 0.1746070000E+01 /
  data (epsData(56,i,19), i=1,3) / 0.9005620000E+01, 0.6023010000E+01, 0.1517400000E+01 /
  data (epsData(57,i,19), i=1,3) / 0.9797130000E+01, 0.8302550000E+01, 0.1207740000E+01 /
  data (epsData(58,i,19), i=1,3) / 0.1098570000E+02, 0.1099530000E+02, 0.9139920000E+00 /
  data (epsData(59,i,19), i=1,3) / 0.1384000000E+02, 0.1677720000E+02, 0.5205400000E+00 /
  data (epsData(60,i,19), i=1,3) / 0.1539060000E+02, 0.2093720000E+02, 0.4100050000E+00 /
  data (epsData(61,i,19), i=1,3) / 0.1642020000E+02, 0.2490600000E+02, 0.3783370000E+00 /
  data (epsData(62,i,19), i=1,3) / 0.1717600000E+02, 0.2916860000E+02, 0.3927270000E+00 /
  data (epsData(63,i,19), i=1,3) / 0.1763770000E+02, 0.3301210000E+02, 0.4430500000E+00 /
  data (epsData(64,i,19), i=1,3) / 0.1794890000E+02, 0.3697840000E+02, 0.5977070000E+00 /
  data (epsData(65,i,19), i=1,3) / 0.1810550000E+02, 0.4016270000E+02, 0.8574280000E+00 /
  data (epsData(66,i,19), i=1,3) / 0.1826280000E+02, 0.4389100000E+02, 0.3728960000E+01 /
  data (epsData(67,i,19), i=1,3) / 0.1832760000E+02, 0.4444040000E+02, 0.6007170000E+01 /
  data (epsData(68,i,19), i=1,3) / 0.1849590000E+02, 0.4409290000E+02, 0.1010020000E+02 /
  data (epsData(69,i,19), i=1,3) / 0.1869320000E+02, 0.4339380000E+02, 0.1097250000E+02 /
  data (epsData(70,i,19), i=1,3) / 0.1911670000E+02, 0.4645080000E+02, 0.1347750000E+02 /
  data (epsData(71,i,19), i=1,3) / 0.1956570000E+02, 0.5135640000E+02, 0.1894690000E+02 /
  data (epsData(72,i,19), i=1,3) / 0.1996670000E+02, 0.5565170000E+02, 0.2674500000E+02 /
  data (epsData(73,i,19), i=1,3) / 0.2036940000E+02, 0.5798270000E+02, 0.3826650000E+02 /
  data (epsData(74,i,19), i=1,3) / 0.2075750000E+02, 0.5545740000E+02, 0.5146830000E+02 /
  data (epsData(75,i,19), i=1,3) / 0.2111670000E+02, 0.4798970000E+02, 0.6223310000E+02 /
  data (epsData(76,i,19), i=1,3) / 0.2143470000E+02, 0.3968340000E+02, 0.6766130000E+02 /
  data (epsData(77,i,19), i=1,3) / 0.2205010000E+02, 0.2503440000E+02, 0.7776900000E+02 /
  data (epsData(78,i,19), i=1,3) / 0.2241570000E+02, 0.1250230000E+02, 0.8178280000E+02 /
  data (epsData(79,i,19), i=1,3) / 0.2272050000E+02, 0.8911520000E+00, 0.8188740000E+02 /
  data (epsData(80,i,19), i=1,3) / 0.2303610000E+02, -0.1042570000E+02, 0.7805330000E+02 /
  data (epsData(81,i,19), i=1,3) / 0.2344600000E+02, -0.2112490000E+02, 0.6936650000E+02 /
  data (epsData(82,i,19), i=1,3) / 0.2396030000E+02, -0.2830870000E+02, 0.5810490000E+02 /
  data (epsData(83,i,19), i=1,3) / 0.2459130000E+02, -0.3240550000E+02, 0.4642680000E+02 /
  data (epsData(84,i,19), i=1,3) / 0.2525560000E+02, -0.3360910000E+02, 0.3587690000E+02 /
  data (epsData(85,i,19), i=1,3) / 0.2595320000E+02, -0.3223000000E+02, 0.2706420000E+02 /
  data (epsData(86,i,19), i=1,3) / 0.2689910000E+02, -0.2879620000E+02, 0.1906060000E+02 /
  data (epsData(87,i,19), i=1,3) / 0.2801480000E+02, -0.2487500000E+02, 0.1330860000E+02 /
  data (epsData(88,i,19), i=1,3) / 0.2919850000E+02, -0.2141130000E+02, 0.9444750000E+01 /
  data (epsData(89,i,19), i=1,3) / 0.3057910000E+02, -0.1813520000E+02, 0.6519320000E+01 /
  data (epsData(90,i,19), i=1,3) / 0.3204210000E+02, -0.1535760000E+02, 0.4511310000E+01 /
  data (epsData(91,i,19), i=1,3) / 0.3373190000E+02, -0.1281450000E+02, 0.3019480000E+01 /
  data (epsData(92,i,19), i=1,3) / 0.3555140000E+02, -0.1066010000E+02, 0.2038330000E+01 /
  data (epsData(93,i,19), i=1,3) / 0.3767900000E+02, -0.8748640000E+01, 0.1351680000E+01 /
  data (epsData(94,i,19), i=1,3) / 0.4003000000E+02, -0.7144960000E+01, 0.9172600000E+00 /
  data (epsData(95,i,19), i=1,3) / 0.4255560000E+02, -0.5857010000E+01, 0.6809930000E+00 /
  data (epsData(96,i,19), i=1,3) / 0.4545630000E+02, -0.4772240000E+01, 0.5444250000E+00 /
  data (epsData(97,i,19), i=1,3) / 0.4881690000E+02, -0.3861160000E+01, 0.4795700000E+00 /
  data (epsData(98,i,19), i=1,3) / 0.5263740000E+02, -0.3129580000E+01, 0.4598240000E+00 /
  data (epsData(99,i,19), i=1,3) / 0.5744120000E+02, -0.2502000000E+01, 0.4527930000E+00 /
  data (epsData(100,i,19), i=1,3) / 0.6320900000E+02, -0.1997820000E+01, 0.4374170000E+00 /
  data (epsData(101,i,19), i=1,3) / 0.7006730000E+02, -0.1595940000E+01, 0.4108300000E+00 /
  data (epsData(102,i,19), i=1,3) / 0.7839390000E+02, -0.1270750000E+01, 0.3705070000E+00 /
  data (epsData(103,i,19), i=1,3) / 0.8867980000E+02, -0.9997110000E+00, 0.3179960000E+00 /
  data (epsData(104,i,19), i=1,3) / 0.1006520000E+03, -0.7894700000E+00, 0.2658510000E+00 /
  data (epsData(105,i,19), i=1,3) / 0.1154720000E+03, -0.6135120000E+00, 0.2076570000E+00 /
  data (epsData(106,i,19), i=1,3) / 0.1310690000E+03, -0.4809090000E+00, 0.1596460000E+00 /
  data (epsData(107,i,19), i=1,3) / 0.1476530000E+03, -0.3774550000E+00, 0.1232400000E+00 /
  data (epsData(108,i,19), i=1,3) / 0.1655530000E+03, -0.2948900000E+00, 0.9532390000E-01 /
  data (epsData(109,i,19), i=1,3) / 0.1861280000E+03, -0.2228220000E+00, 0.7134180000E-01 /
  data (epsData(110,i,19), i=1,3) / 0.2082070000E+03, -0.1617400000E+00, 0.5361870000E-01 /
  data (epsData(111,i,19), i=1,3) / 0.2385760000E+03, -0.9249900000E-01, 0.3733110000E-01 /
  data (epsData(112,i,19), i=1,3) / 0.2574220000E+03, -0.4901320000E-01, 0.3020610000E-01 /
  data (epsData(113,i,19), i=1,3) / 0.2663150000E+03, -0.2476270000E-01, 0.2745700000E-01 /
  data (epsData(114,i,19), i=1,3) / 0.2719610000E+03, -0.6499010000E-02, 0.2591270000E-01 /
  data (epsData(115,i,19), i=1,3) / 0.2742390000E+03, 0.2033480000E-02, 0.2531090000E-01 /
  data (epsData(116,i,19), i=1,3) / 0.2765270000E+03, 0.1118880000E-01, 0.2495820000E-01 /
  data (epsData(117,i,19), i=1,3) / 0.2799810000E+03, 0.2713120000E-01, 0.2430430000E-01 /
  data (epsData(118,i,19), i=1,3) / 0.2828770000E+03, 0.4334140000E-01, 0.2373520000E-01 /
  data (epsData(119,i,19), i=1,3) / 0.2857910000E+03, 0.6373530000E-01, 0.2324650000E-01 /
  data (epsData(120,i,19), i=1,3) / 0.2881340000E+03, 0.8492460000E-01, 0.2296260000E-01 /
  data (epsData(121,i,19), i=1,3) / 0.2898980000E+03, 0.1055230000E+00, 0.2287250000E-01 /
  data (epsData(122,i,19), i=1,3) / 0.2916680000E+03, 0.1330560000E+00, 0.2300800000E-01 /
  data (epsData(123,i,19), i=1,3) / 0.2928520000E+03, 0.1582380000E+00, 0.2339260000E-01 /
  data (epsData(124,i,19), i=1,3) / 0.2934450000E+03, 0.1744860000E+00, 0.2380150000E-01 /
  data (epsData(125,i,19), i=1,3) / 0.2940390000E+03, 0.1950470000E+00, 0.2467330000E-01 /
  data (epsData(126,i,19), i=1,3) / 0.2949930000E+03, 0.2357490000E+00, 0.3669410000E-01 /
  data (epsData(127,i,19), i=1,3) / 0.2956290000E+03, 0.2706710000E+00, 0.4910130000E-01 /
  data (epsData(128,i,19), i=1,3) / 0.2962670000E+03, 0.3149790000E+00, 0.7725230000E-01 /
  data (epsData(129,i,19), i=1,3) / 0.2967770000E+03, 0.3490170000E+00, 0.1195220000E+00 /
  data (epsData(130,i,19), i=1,3) / 0.2973570000E+03, 0.3693550000E+00, 0.1851830000E+00 /
  data (epsData(131,i,19), i=1,3) / 0.2982550000E+03, 0.3643680000E+00, 0.2886740000E+00 /
  data (epsData(132,i,19), i=1,3) / 0.2988730000E+03, 0.3274100000E+00, 0.3642610000E+00 /
  data (epsData(133,i,19), i=1,3) / 0.2995320000E+03, 0.2604620000E+00, 0.4131970000E+00 /
  data (epsData(134,i,19), i=1,3) / 0.3004350000E+03, 0.1801480000E+00, 0.4373920000E+00 /
  data (epsData(135,i,19), i=1,3) / 0.3014910000E+03, 0.9431370000E-01, 0.4516100000E+00 /
  data (epsData(136,i,19), i=1,3) / 0.3021980000E+03, 0.3371930000E-01, 0.4357440000E+00 /
  data (epsData(137,i,19), i=1,3) / 0.3030480000E+03, -0.1147210000E-01, 0.3982150000E+00 /
  data (epsData(138,i,19), i=1,3) / 0.3042560000E+03, -0.4419710000E-01, 0.3510350000E+00 /
  data (epsData(139,i,19), i=1,3) / 0.3053670000E+03, -0.5483480000E-01, 0.3158220000E+00 /
  data (epsData(140,i,19), i=1,3) / 0.3071300000E+03, -0.5848390000E-01, 0.2828730000E+00 /
  data (epsData(141,i,19), i=1,3) / 0.3105010000E+03, -0.6569940000E-01, 0.2522160000E+00 /
  data (epsData(142,i,19), i=1,3) / 0.3165670000E+03, -0.7874850000E-01, 0.2173000000E+00 /
  data (epsData(143,i,19), i=1,3) / 0.3271020000E+03, -0.8750820000E-01, 0.1834870000E+00 /
  data (epsData(144,i,19), i=1,3) / 0.3389670000E+03, -0.9119780000E-01, 0.1622350000E+00 /
  data (epsData(145,i,19), i=1,3) / 0.3547570000E+03, -0.9560890000E-01, 0.1379730000E+00 /
  data (epsData(146,i,19), i=1,3) / 0.3734000000E+03, -0.9041740000E-01, 0.1181850000E+00 /
  data (epsData(147,i,19), i=1,3) / 0.3796290000E+03, -0.8617320000E-01, 0.1176980000E+00 /
  data (epsData(148,i,19), i=1,3) / 0.3915340000E+03, -0.9217890000E-01, 0.1128420000E+00 /
  data (epsData(149,i,19), i=1,3) / 0.4226820000E+03, -0.9258550000E-01, 0.9066190000E-01 /
  data (epsData(150,i,19), i=1,3) / 0.4734270000E+03, -0.8498200000E-01, 0.6440750000E-01 /
  data (epsData(151,i,19), i=1,3) / 0.5340400000E+03, -0.7305290000E-01, 0.4406020000E-01 /
  data (epsData(152,i,19), i=1,3) / 0.5962960000E+03, -0.6175940000E-01, 0.3068550000E-01 /
  data (epsData(153,i,19), i=1,3) / 0.6671300000E+03, -0.5107770000E-01, 0.2103320000E-01 /
  data (epsData(154,i,19), i=1,3) / 0.7415720000E+03, -0.4226590000E-01, 0.1471670000E-01 /
  data (epsData(155,i,19), i=1,3) / 0.8144790000E+03, -0.3542200000E-01, 0.1066820000E-01 /
  data (epsData(156,i,19), i=1,3) / 0.8940540000E+03, -0.2957580000E-01, 0.7698300000E-02 /
  data (epsData(157,i,19), i=1,3) / 0.9830140000E+03, -0.2453030000E-01, 0.5501770000E-02 /
  data (epsData(158,i,19), i=1,3) / 0.1083940000E+04, -0.2017100000E-01, 0.3882660000E-02 /
  data (epsData(159,i,19), i=1,3) / 0.1197930000E+04, -0.1647770000E-01, 0.2711640000E-02 /
  data (epsData(160,i,19), i=1,3) / 0.1323850000E+04, -0.1344000000E-01, 0.1889110000E-02 /
  data (epsData(161,i,19), i=1,3) / 0.1462410000E+04, -0.1095820000E-01, 0.1315510000E-02 /
  data (epsData(162,i,19), i=1,3) / 0.1618220000E+04, -0.8894700000E-02, 0.9074580000E-03 /
  data (epsData(163,i,19), i=1,3) / 0.1795580000E+04, -0.7174650000E-02, 0.6167630000E-03 /
  data (epsData(164,i,19), i=1,3) / 0.1993670000E+04, -0.5768330000E-02, 0.4193260000E-03 /
  data (epsData(165,i,19), i=1,3) / 0.2217470000E+04, -0.4617480000E-02, 0.2825510000E-03 /
  data (epsData(166,i,19), i=1,3) / 0.2468190000E+04, -0.3682900000E-02, 0.1895090000E-03 /
  data (epsData(167,i,19), i=1,3) / 0.2749910000E+04, -0.2919820000E-02, 0.1264470000E-03 /
  data (epsData(168,i,19), i=1,3) / 0.3073820000E+04, -0.2275830000E-02, 0.8318190000E-04 /
  data (epsData(169,i,19), i=1,3) / 0.3518390000E+04, -0.1547840000E-02, 0.4991950000E-04 /
  data (epsData(170,i,19), i=1,3) / 0.3576400000E+04, -0.1383150000E-02, 0.4692210000E-04 /
  data (epsData(171,i,19), i=1,3) / 0.3596000000E+04, -0.1263980000E-02, 0.4862120000E-04 /
  data (epsData(172,i,19), i=1,3) / 0.3603030000E+04, -0.1172150000E-02, 0.5175320000E-04 /
  data (epsData(173,i,19), i=1,3) / 0.3605850000E+04, -0.1097140000E-02, 0.5752690000E-04 /
  data (epsData(174,i,19), i=1,3) / 0.3607260000E+04, -0.1026470000E-02, 0.6909020000E-04 /
  data (epsData(175,i,19), i=1,3) / 0.3608500000E+04, -0.9450110000E-03, 0.1535970000E-03 /
  data (epsData(176,i,19), i=1,3) / 0.3609120000E+04, -0.9486700000E-03, 0.2308400000E-03 /
  data (epsData(177,i,19), i=1,3) / 0.3610360000E+04, -0.1006780000E-02, 0.3185550000E-03 /
  data (epsData(178,i,19), i=1,3) / 0.3611530000E+04, -0.1066300000E-02, 0.3351110000E-03 /
  data (epsData(179,i,19), i=1,3) / 0.3615750000E+04, -0.1127250000E-02, 0.3970210000E-03 /
  data (epsData(180,i,19), i=1,3) / 0.3619180000E+04, -0.1205060000E-02, 0.4097220000E-03 /
  data (epsData(181,i,19), i=1,3) / 0.3627340000E+04, -0.1265350000E-02, 0.4076180000E-03 /
  data (epsData(182,i,19), i=1,3) / 0.3651270000E+04, -0.1368730000E-02, 0.4112940000E-03 /
  data (epsData(183,i,19), i=1,3) / 0.3722820000E+04, -0.1433610000E-02, 0.3766300000E-03 /
  data (epsData(184,i,19), i=1,3) / 0.3956760000E+04, -0.1393890000E-02, 0.3071400000E-03 /
  data (epsData(185,i,19), i=1,3) / 0.4653700000E+04, -0.1087400000E-02, 0.1712450000E-03 /
  data (epsData(186,i,19), i=1,3) / 0.5216070000E+04, -0.8837450000E-03, 0.1149400000E-03 /
  data (epsData(187,i,19), i=1,3) / 0.5853340000E+04, -0.7110030000E-03, 0.7552720000E-04 /
  data (epsData(188,i,19), i=1,3) / 0.6542570000E+04, -0.5727850000E-03, 0.5011590000E-04 /
  data (epsData(189,i,19), i=1,3) / 0.7303680000E+04, -0.4611860000E-03, 0.3329780000E-04 /
  data (epsData(190,i,19), i=1,3) / 0.8157310000E+04, -0.3700450000E-03, 0.2192720000E-04 /
  data (epsData(191,i,19), i=1,3) / 0.9105040000E+04, -0.2969730000E-03, 0.1443460000E-04 /
  data (epsData(192,i,19), i=1,3) / 0.1016920000E+05, -0.2377880000E-03, 0.9439280000E-05 /
  data (epsData(193,i,19), i=1,3) / 0.1136000000E+05, -0.1902420000E-03, 0.6144250000E-05 /
  data (epsData(194,i,19), i=1,3) / 0.1269560000E+05, -0.1520350000E-03, 0.3981690000E-05 /
  data (epsData(195,i,19), i=1,3) / 0.1419410000E+05, -0.1213890000E-03, 0.2564880000E-05 /
  data (epsData(196,i,19), i=1,3) / 0.1587640000E+05, -0.9683630000E-04, 0.1647560000E-05 /
  data (epsData(197,i,19), i=1,3) / 0.1776150000E+05, -0.7722680000E-04, 0.1053650000E-05 /
  data (epsData(198,i,19), i=1,3) / 0.1988080000E+05, -0.6153190000E-04, 0.6698130000E-06 /
  data (epsData(199,i,19), i=1,3) / 0.2226350000E+05, -0.4898730000E-04, 0.4241760000E-06 /
  data (epsData(200,i,19), i=1,3) / 0.2493810000E+05, -0.3898680000E-04, 0.2677320000E-06 /
  data (epsData(201,i,19), i=1,3) / 0.2794120000E+05, -0.3101720000E-04, 0.1683090000E-06 /
  data (epsData(202,i,19), i=1,3) / 0.3131370000E+05, -0.2466820000E-04, 0.1052840000E-06 /
  data (epsData(203,i,19), i=1,3) / 0.3510120000E+05, -0.1961280000E-04, 0.6585050000E-07 /
  data (epsData(204,i,19), i=1,3) / 0.3935400000E+05, -0.1558990000E-04, 0.4097730000E-07 /
  data (epsData(205,i,19), i=1,3) / 0.4412080000E+05, -0.1239450000E-04, 0.2553750000E-07 /
  data (epsData(206,i,19), i=1,3) / 0.4947780000E+05, -0.9849690000E-05, 0.1563900000E-07 /
  data (epsData(207,i,19), i=1,3) / 0.5549450000E+05, -0.7825680000E-05, 0.9660490000E-08 /
  data (epsData(208,i,19), i=1,3) / 0.6224530000E+05, -0.6217560000E-05, 0.5937210000E-08 /
  data (epsData(209,i,19), i=1,3) / 0.6982240000E+05, -0.4939620000E-05, 0.3631880000E-08 /
  data (epsData(210,i,19), i=1,3) / 0.7832260000E+05, -0.3924490000E-05, 0.2213040000E-08 /
  data (epsData(211,i,19), i=1,3) / 0.8785940000E+05, -0.3118060000E-05, 0.1347960000E-08 /
  data (epsData(212,i,19), i=1,3) / 0.9854740000E+05, -0.2478300000E-05, 0.8368930000E-09 /
  data (epsData(213,i,19), i=1,3) / 0.1000000000E+06, -0.2408490000E-05, 0.7966550000E-09 /
  data (epsData(1,i,20), i=1,3) / 0.2506580000E-02, -0.2455490000E+05, 0.6003120000E+05 /
  data (epsData(2,i,20), i=1,3) / 0.4693440000E-02, -0.2015310000E+05, 0.2011700000E+05 /
  data (epsData(3,i,20), i=1,3) / 0.7560590000E-02, -0.9518130000E+04, 0.9265560000E+04 /
  data (epsData(4,i,20), i=1,3) / 0.1110800000E-01, -0.4890840000E+04, 0.5159840000E+04 /
  data (epsData(5,i,20), i=1,3) / 0.1533570000E-01, -0.2716930000E+04, 0.3249380000E+04 /
  data (epsData(6,i,20), i=1,3) / 0.2024370000E-01, -0.1592890000E+04, 0.2226090000E+04 /
  data (epsData(7,i,20), i=1,3) / 0.2583200000E-01, -0.9639370000E+03, 0.1619770000E+04 /
  data (epsData(8,i,20), i=1,3) / 0.3210060000E-01, -0.5890430000E+03, 0.1232450000E+04 /
  data (epsData(9,i,20), i=1,3) / 0.3904940000E-01, -0.3538610000E+03, 0.9703220000E+03 /
  data (epsData(10,i,20), i=1,3) / 0.4667860000E-01, -0.1999930000E+03, 0.7846940000E+03 /
  data (epsData(11,i,20), i=1,3) / 0.5498800000E-01, -0.9572730000E+02, 0.6483630000E+03 /
  data (epsData(12,i,20), i=1,3) / 0.6397770000E-01, -0.2293970000E+02, 0.5452150000E+03 /
  data (epsData(13,i,20), i=1,3) / 0.7364770000E-01, 0.2918870000E+02, 0.4652250000E+03 /
  data (epsData(14,i,20), i=1,3) / 0.8399800000E-01, 0.6736020000E+02, 0.4018930000E+03 /
  data (epsData(15,i,20), i=1,3) / 0.9502860000E-01, 0.9586260000E+02, 0.3508570000E+03 /
  data (epsData(16,i,20), i=1,3) / 0.1067390000E+00, 0.1175170000E+03, 0.3091020000E+03 /
  data (epsData(17,i,20), i=1,3) / 0.1191310000E+00, 0.1342270000E+03, 0.2744840000E+03 /
  data (epsData(18,i,20), i=1,3) / 0.1322020000E+00, 0.1473030000E+03, 0.2454520000E+03 /
  data (epsData(19,i,20), i=1,3) / 0.1603860000E+00, 0.1659880000E+03, 0.1998210000E+03 /
  data (epsData(20,i,20), i=1,3) / 0.1912910000E+00, 0.1782730000E+03, 0.1659500000E+03 /
  data (epsData(21,i,20), i=1,3) / 0.2427500000E+00, 0.1899530000E+03, 0.1293940000E+03 /
  data (epsData(22,i,20), i=1,3) / 0.3003320000E+00, 0.1972300000E+03, 0.1037930000E+03 /
  data (epsData(23,i,20), i=1,3) / 0.4099070000E+00, 0.2049120000E+03, 0.7539230000E+02 /
  data (epsData(24,i,20), i=1,3) / 0.5918850000E+00, 0.2129110000E+03, 0.5183050000E+02 /
  data (epsData(25,i,20), i=1,3) / 0.9096770000E+00, 0.2255270000E+03, 0.3586550000E+02 /
  data (epsData(26,i,20), i=1,3) / 0.1295500000E+01, 0.2471860000E+03, 0.2637200000E+02 /
  data (epsData(27,i,20), i=1,3) / 0.1653130000E+01, 0.2822930000E+03, 0.2395480000E+02 /
  data (epsData(28,i,20), i=1,3) / 0.1848280000E+01, 0.3140490000E+03, 0.2267320000E+02 /
  data (epsData(29,i,20), i=1,3) / 0.2054310000E+01, 0.3645170000E+03, 0.2683900000E+02 /
  data (epsData(30,i,20), i=1,3) / 0.2161410000E+01, 0.4050420000E+03, 0.2826420000E+02 /
  data (epsData(31,i,20), i=1,3) / 0.2215970000E+01, 0.4351720000E+03, 0.2883760000E+02 /
  data (epsData(32,i,20), i=1,3) / 0.2327150000E+01, 0.5125010000E+03, 0.5107430000E+02 /
  data (epsData(33,i,20), i=1,3) / 0.2383760000E+01, 0.5709670000E+03, 0.6157880000E+02 /
  data (epsData(34,i,20), i=1,3) / 0.2441050000E+01, 0.6789980000E+03, 0.7117310000E+02 /
  data (epsData(35,i,20), i=1,3) / 0.2499020000E+01, 0.8050550000E+03, 0.2145350000E+03 /
  data (epsData(36,i,20), i=1,3) / 0.2557670000E+01, 0.7828580000E+03, 0.5390020000E+03 /
  data (epsData(37,i,20), i=1,3) / 0.2617000000E+01, 0.6342290000E+03, 0.6513710000E+03 /
  data (epsData(38,i,20), i=1,3) / 0.2677010000E+01, 0.5054570000E+03, 0.6984330000E+03 /
  data (epsData(39,i,20), i=1,3) / 0.2799080000E+01, 0.2683090000E+03, 0.7708690000E+03 /
  data (epsData(40,i,20), i=1,3) / 0.2861130000E+01, 0.1414170000E+03, 0.7423350000E+03 /
  data (epsData(41,i,20), i=1,3) / 0.2923860000E+01, 0.4377380000E+02, 0.6816960000E+03 /
  data (epsData(42,i,20), i=1,3) / 0.2987270000E+01, -0.3303140000E+02, 0.6248570000E+03 /
  data (epsData(43,i,20), i=1,3) / 0.3051370000E+01, -0.8566640000E+02, 0.5332540000E+03 /
  data (epsData(44,i,20), i=1,3) / 0.3116140000E+01, -0.1003300000E+03, 0.4460710000E+03 /
  data (epsData(45,i,20), i=1,3) / 0.3181590000E+01, -0.7575670000E+02, 0.3646610000E+03 /
  data (epsData(46,i,20), i=1,3) / 0.3247730000E+01, -0.3978450000E+02, 0.3508990000E+03 /
  data (epsData(47,i,20), i=1,3) / 0.3314540000E+01, -0.2587690000E+02, 0.3531620000E+03 /
  data (epsData(48,i,20), i=1,3) / 0.3382030000E+01, -0.3036390000E+02, 0.3549910000E+03 /
  data (epsData(49,i,20), i=1,3) / 0.3450210000E+01, -0.4796550000E+02, 0.3492430000E+03 /
  data (epsData(50,i,20), i=1,3) / 0.3519060000E+01, -0.5854270000E+02, 0.3116770000E+03 /
  data (epsData(51,i,20), i=1,3) / 0.3588600000E+01, -0.4420250000E+02, 0.2763770000E+03 /
  data (epsData(52,i,20), i=1,3) / 0.3658810000E+01, 0.7592820000E+01, 0.2431990000E+03 /
  data (epsData(53,i,20), i=1,3) / 0.3729710000E+01, 0.1135290000E+02, 0.3613450000E+03 /
  data (epsData(54,i,20), i=1,3) / 0.3801290000E+01, -0.4244500000E+02, 0.3472310000E+03 /
  data (epsData(55,i,20), i=1,3) / 0.3873540000E+01, -0.7131100000E+02, 0.3266550000E+03 /
  data (epsData(56,i,20), i=1,3) / 0.3946480000E+01, -0.8711800000E+02, 0.3044570000E+03 /
  data (epsData(57,i,20), i=1,3) / 0.4020090000E+01, -0.9406200000E+02, 0.2831280000E+03 /
  data (epsData(58,i,20), i=1,3) / 0.4476080000E+01, -0.1133860000E+03, 0.2135620000E+03 /
  data (epsData(59,i,20), i=1,3) / 0.4793680000E+01, -0.1175210000E+03, 0.1723010000E+03 /
  data (epsData(60,i,20), i=1,3) / 0.5205980000E+01, -0.1133950000E+03, 0.1309470000E+03 /
  data (epsData(61,i,20), i=1,3) / 0.5723190000E+01, -0.1023000000E+03, 0.9434200000E+02 /
  data (epsData(62,i,20), i=1,3) / 0.6357560000E+01, -0.8705290000E+02, 0.6354580000E+02 /
  data (epsData(63,i,20), i=1,3) / 0.6927830000E+01, -0.7338610000E+02, 0.4440040000E+02 /
  data (epsData(64,i,20), i=1,3) / 0.7522590000E+01, -0.5975800000E+02, 0.3015900000E+02 /
  data (epsData(65,i,20), i=1,3) / 0.8141850000E+01, -0.4743630000E+02, 0.2225910000E+02 /
  data (epsData(66,i,20), i=1,3) / 0.8568290000E+01, -0.4029620000E+02, 0.1884120000E+02 /
  data (epsData(67,i,20), i=1,3) / 0.9228360000E+01, -0.3212730000E+02, 0.1515110000E+02 /
  data (epsData(68,i,20), i=1,3) / 0.1002940000E+02, -0.2473610000E+02, 0.1209650000E+02 /
  data (epsData(69,i,20), i=1,3) / 0.1086380000E+02, -0.1882670000E+02, 0.9782880000E+01 /
  data (epsData(70,i,20), i=1,3) / 0.1173150000E+02, -0.1390940000E+02, 0.8006410000E+01 /
  data (epsData(71,i,20), i=1,3) / 0.1263250000E+02, -0.9731060000E+01, 0.6632190000E+01 /
  data (epsData(72,i,20), i=1,3) / 0.1370310000E+02, -0.5632940000E+01, 0.5450840000E+01 /
  data (epsData(73,i,20), i=1,3) / 0.1481720000E+02, -0.2027020000E+01, 0.4569060000E+01 /
  data (epsData(74,i,20), i=1,3) / 0.1582780000E+02, 0.8827730000E+00, 0.3979080000E+01 /
  data (epsData(75,i,20), i=1,3) / 0.1794890000E+02, 0.6644930000E+01, 0.3126060000E+01 /
  data (epsData(76,i,20), i=1,3) / 0.1938290000E+02, 0.1087750000E+02, 0.2740920000E+01 /
  data (epsData(77,i,20), i=1,3) / 0.2053630000E+02, 0.1498580000E+02, 0.2511140000E+01 /
  data (epsData(78,i,20), i=1,3) / 0.2155150000E+02, 0.1968710000E+02, 0.2344840000E+01 /
  data (epsData(79,i,20), i=1,3) / 0.2241620000E+02, 0.2520180000E+02, 0.2248120000E+01 /
  data (epsData(80,i,20), i=1,3) / 0.2312020000E+02, 0.3155250000E+02, 0.2223170000E+01 /
  data (epsData(81,i,20), i=1,3) / 0.2365530000E+02, 0.3841130000E+02, 0.2248950000E+01 /
  data (epsData(82,i,20), i=1,3) / 0.2401550000E+02, 0.4472690000E+02, 0.2308340000E+01 /
  data (epsData(83,i,20), i=1,3) / 0.2437840000E+02, 0.5343120000E+02, 0.2431540000E+01 /
  data (epsData(84,i,20), i=1,3) / 0.2456080000E+02, 0.5920370000E+02, 0.2558350000E+01 /
  data (epsData(85,i,20), i=1,3) / 0.2474400000E+02, 0.6644410000E+02, 0.2728440000E+01 /
  data (epsData(86,i,20), i=1,3) / 0.2492780000E+02, 0.7591820000E+02, 0.3015090000E+01 /
  data (epsData(87,i,20), i=1,3) / 0.2511230000E+02, 0.8899370000E+02, 0.3708450000E+01 /
  data (epsData(88,i,20), i=1,3) / 0.2529750000E+02, 0.1091390000E+03, 0.4647280000E+01 /
  data (epsData(89,i,20), i=1,3) / 0.2542180000E+02, 0.1281780000E+03, 0.1153980000E+02 /
  data (epsData(90,i,20), i=1,3) / 0.2555000000E+02, 0.1514940000E+03, 0.2279940000E+02 /
  data (epsData(91,i,20), i=1,3) / 0.2559590000E+02, 0.1658100000E+03, 0.2677650000E+02 /
  data (epsData(92,i,20), i=1,3) / 0.2562990000E+02, 0.1779750000E+03, 0.3924760000E+02 /
  data (epsData(93,i,20), i=1,3) / 0.2572690000E+02, 0.1918560000E+03, 0.7810650000E+02 /
  data (epsData(94,i,20), i=1,3) / 0.2586570000E+02, 0.1914120000E+03, 0.1327240000E+03 /
  data (epsData(95,i,20), i=1,3) / 0.2602890000E+02, 0.1700720000E+03, 0.1943910000E+03 /
  data (epsData(96,i,20), i=1,3) / 0.2615140000E+02, 0.1387610000E+03, 0.2394110000E+03 /
  data (epsData(97,i,20), i=1,3) / 0.2621670000E+02, 0.1125050000E+03, 0.2602420000E+03 /
  data (epsData(98,i,20), i=1,3) / 0.2639190000E+02, 0.3912340000E+02, 0.2932380000E+03 /
  data (epsData(99,i,20), i=1,3) / 0.2646670000E+02, -0.3840050000E+01, 0.3070600000E+03 /
  data (epsData(100,i,20), i=1,3) / 0.2650510000E+02, -0.3533570000E+02, 0.3108650000E+03 /
  data (epsData(101,i,20), i=1,3) / 0.2654420000E+02, -0.6538090000E+02, 0.2933790000E+03 /
  data (epsData(102,i,20), i=1,3) / 0.2662450000E+02, -0.1025260000E+03, 0.2578120000E+03 /
  data (epsData(103,i,20), i=1,3) / 0.2675000000E+02, -0.1318990000E+03, 0.2030430000E+03 /
  data (epsData(104,i,20), i=1,3) / 0.2679320000E+02, -0.1346010000E+03, 0.1844270000E+03 /
  data (epsData(105,i,20), i=1,3) / 0.2701940000E+02, -0.1396580000E+03, 0.1234830000E+03 /
  data (epsData(106,i,20), i=1,3) / 0.2711470000E+02, -0.1351610000E+03, 0.1001640000E+03 /
  data (epsData(107,i,20), i=1,3) / 0.2736470000E+02, -0.1185210000E+03, 0.5881810000E+02 /
  data (epsData(108,i,20), i=1,3) / 0.2741670000E+02, -0.1124380000E+03, 0.5200860000E+02 /
  data (epsData(109,i,20), i=1,3) / 0.2763170000E+02, -0.9479410000E+02, 0.3826670000E+02 /
  data (epsData(110,i,20), i=1,3) / 0.2774320000E+02, -0.8646860000E+02, 0.3275060000E+02 /
  data (epsData(111,i,20), i=1,3) / 0.2803410000E+02, -0.7133920000E+02, 0.2503650000E+02 /
  data (epsData(112,i,20), i=1,3) / 0.2834190000E+02, -0.6031380000E+02, 0.2061200000E+02 /
  data (epsData(113,i,20), i=1,3) / 0.2873370000E+02, -0.5073730000E+02, 0.1738120000E+02 /
  data (epsData(114,i,20), i=1,3) / 0.2922180000E+02, -0.4285700000E+02, 0.1499050000E+02 /
  data (epsData(115,i,20), i=1,3) / 0.2982050000E+02, -0.3713860000E+02, 0.1287680000E+02 /
  data (epsData(116,i,20), i=1,3) / 0.3063000000E+02, -0.3126100000E+02, 0.1008320000E+02 /
  data (epsData(117,i,20), i=1,3) / 0.3169130000E+02, -0.2603560000E+02, 0.7903870000E+01 /
  data (epsData(118,i,20), i=1,3) / 0.3295150000E+02, -0.2179280000E+02, 0.6141440000E+01 /
  data (epsData(119,i,20), i=1,3) / 0.3454850000E+02, -0.1809400000E+02, 0.4605960000E+01 /
  data (epsData(120,i,20), i=1,3) / 0.3642060000E+02, -0.1501530000E+02, 0.3358780000E+01 /
  data (epsData(121,i,20), i=1,3) / 0.3860060000E+02, -0.1239300000E+02, 0.2388850000E+01 /
  data (epsData(122,i,20), i=1,3) / 0.4112310000E+02, -0.1014840000E+02, 0.1673760000E+01 /
  data (epsData(123,i,20), i=1,3) / 0.4386600000E+02, -0.8286040000E+01, 0.1226090000E+01 /
  data (epsData(124,i,20), i=1,3) / 0.4636940000E+02, -0.7125930000E+01, 0.1035440000E+01 /
  data (epsData(125,i,20), i=1,3) / 0.5040760000E+02, -0.5688790000E+01, 0.7301250000E+00 /
  data (epsData(126,i,20), i=1,3) / 0.5436280000E+02, -0.4625380000E+01, 0.5704250000E+00 /
  data (epsData(127,i,20), i=1,3) / 0.5887470000E+02, -0.3744080000E+01, 0.4996890000E+00 /
  data (epsData(128,i,20), i=1,3) / 0.6389220000E+02, -0.3049570000E+01, 0.4763760000E+00 /
  data (epsData(129,i,20), i=1,3) / 0.7039870000E+02, -0.2442920000E+01, 0.4593970000E+00 /
  data (epsData(130,i,20), i=1,3) / 0.7783660000E+02, -0.1978530000E+01, 0.4302970000E+00 /
  data (epsData(131,i,20), i=1,3) / 0.8789390000E+02, -0.1555190000E+01, 0.3752000000E+00 /
  data (epsData(132,i,20), i=1,3) / 0.9872270000E+02, -0.1244740000E+01, 0.3246730000E+00 /
  data (epsData(133,i,20), i=1,3) / 0.1128730000E+03, -0.9755570000E+00, 0.2609980000E+00 /
  data (epsData(134,i,20), i=1,3) / 0.1283190000E+03, -0.7691750000E+00, 0.2013830000E+00 /
  data (epsData(135,i,20), i=1,3) / 0.1442230000E+03, -0.6155010000E+00, 0.1579960000E+00 /
  data (epsData(136,i,20), i=1,3) / 0.1623040000E+03, -0.4898730000E+00, 0.1228030000E+00 /
  data (epsData(137,i,20), i=1,3) / 0.1840600000E+03, -0.3815180000E+00, 0.9071420000E-01 /
  data (epsData(138,i,20), i=1,3) / 0.2053070000E+03, -0.3021060000E+00, 0.6936040000E-01 /
  data (epsData(139,i,20), i=1,3) / 0.2300630000E+03, -0.2332620000E+00, 0.5203540000E-01 /
  data (epsData(140,i,20), i=1,3) / 0.2565870000E+03, -0.1756920000E+00, 0.3872720000E-01 /
  data (epsData(141,i,20), i=1,3) / 0.2900590000E+03, -0.1162600000E+00, 0.2768170000E-01 /
  data (epsData(142,i,20), i=1,3) / 0.3149460000E+03, -0.7180800000E-01, 0.2195870000E-01 /
  data (epsData(143,i,20), i=1,3) / 0.3247420000E+03, -0.4994060000E-01, 0.2006100000E-01 /
  data (epsData(144,i,20), i=1,3) / 0.3309530000E+03, -0.3226460000E-01, 0.1924270000E-01 /
  data (epsData(145,i,20), i=1,3) / 0.3347130000E+03, -0.1863100000E-01, 0.1870250000E-01 /
  data (epsData(146,i,20), i=1,3) / 0.3372330000E+03, -0.7229590000E-02, 0.1842500000E-01 /
  data (epsData(147,i,20), i=1,3) / 0.3384970000E+03, -0.4461080000E-03, 0.1831350000E-01 /
  data (epsData(148,i,20), i=1,3) / 0.3397630000E+03, 0.7387120000E-02, 0.1823270000E-01 /
  data (epsData(149,i,20), i=1,3) / 0.3410330000E+03, 0.1670670000E-01, 0.1820270000E-01 /
  data (epsData(150,i,20), i=1,3) / 0.3423050000E+03, 0.2830750000E-01, 0.1826670000E-01 /
  data (epsData(151,i,20), i=1,3) / 0.3429420000E+03, 0.3542980000E-01, 0.1836560000E-01 /
  data (epsData(152,i,20), i=1,3) / 0.3435800000E+03, 0.4392360000E-01, 0.1854940000E-01 /
  data (epsData(153,i,20), i=1,3) / 0.3442180000E+03, 0.5457360000E-01, 0.1889560000E-01 /
  data (epsData(154,i,20), i=1,3) / 0.3448580000E+03, 0.6929210000E-01, 0.1962670000E-01 /
  data (epsData(155,i,20), i=1,3) / 0.3454970000E+03, 0.9335250000E-01, 0.2177350000E-01 /
  data (epsData(156,i,20), i=1,3) / 0.3461380000E+03, 0.1278460000E+00, 0.5072270000E-01 /
  data (epsData(157,i,20), i=1,3) / 0.3462280000E+03, 0.1304390000E+00, 0.6641400000E-01 /
  data (epsData(158,i,20), i=1,3) / 0.3463800000E+03, 0.1235810000E+00, 0.8876930000E-01 /
  data (epsData(159,i,20), i=1,3) / 0.3464480000E+03, 0.1145290000E+00, 0.9260610000E-01 /
  data (epsData(160,i,20), i=1,3) / 0.3467720000E+03, 0.9622990000E-01, 0.9697120000E-01 /
  data (epsData(161,i,20), i=1,3) / 0.3471550000E+03, 0.8899180000E-01, 0.9592160000E-01 /
  data (epsData(162,i,20), i=1,3) / 0.3479150000E+03, 0.8805910000E-01, 0.1017400000E+00 /
  data (epsData(163,i,20), i=1,3) / 0.3491420000E+03, 0.9073210000E-01, 0.1173680000E+00 /
  data (epsData(164,i,20), i=1,3) / 0.3495340000E+03, 0.9769530000E-01, 0.1279590000E+00 /
  data (epsData(165,i,20), i=1,3) / 0.3496700000E+03, 0.1005730000E+00, 0.1361900000E+00 /
  data (epsData(166,i,20), i=1,3) / 0.3498960000E+03, 0.9300940000E-01, 0.1522970000E+00 /
  data (epsData(167,i,20), i=1,3) / 0.3502440000E+03, 0.7694150000E-01, 0.1563330000E+00 /
  data (epsData(168,i,20), i=1,3) / 0.3507290000E+03, 0.6876470000E-01, 0.1557080000E+00 /
  data (epsData(169,i,20), i=1,3) / 0.3516650000E+03, 0.6109050000E-01, 0.1667270000E+00 /
  data (epsData(170,i,20), i=1,3) / 0.3531010000E+03, 0.3872660000E-01, 0.1816710000E+00 /
  data (epsData(171,i,20), i=1,3) / 0.3541980000E+03, 0.2179010000E-01, 0.1843900000E+00 /
  data (epsData(172,i,20), i=1,3) / 0.3568520000E+03, -0.1024180000E-01, 0.1874770000E+00 /
  data (epsData(173,i,20), i=1,3) / 0.3591860000E+03, -0.3266780000E-01, 0.1798050000E+00 /
  data (epsData(174,i,20), i=1,3) / 0.3627670000E+03, -0.5123050000E-01, 0.1646620000E+00 /
  data (epsData(175,i,20), i=1,3) / 0.3673820000E+03, -0.6094430000E-01, 0.1507290000E+00 /
  data (epsData(176,i,20), i=1,3) / 0.3773280000E+03, -0.7115550000E-01, 0.1367650000E+00 /
  data (epsData(177,i,20), i=1,3) / 0.3930510000E+03, -0.8652540000E-01, 0.1218940000E+00 /
  data (epsData(178,i,20), i=1,3) / 0.4074370000E+03, -0.8900980000E-01, 0.1059480000E+00 /
  data (epsData(179,i,20), i=1,3) / 0.4323720000E+03, -0.8506860000E-01, 0.8955590000E-01 /
  data (epsData(180,i,20), i=1,3) / 0.4381720000E+03, -0.8116730000E-01, 0.9045870000E-01 /
  data (epsData(181,i,20), i=1,3) / 0.4429770000E+03, -0.8591280000E-01, 0.9120760000E-01 /
  data (epsData(182,i,20), i=1,3) / 0.4579920000E+03, -0.8758010000E-01, 0.8204680000E-01 /
  data (epsData(183,i,20), i=1,3) / 0.5015850000E+03, -0.8397360000E-01, 0.6283830000E-01 /
  data (epsData(184,i,20), i=1,3) / 0.5568050000E+03, -0.7518410000E-01, 0.4537070000E-01 /
  data (epsData(185,i,20), i=1,3) / 0.6317800000E+03, -0.6287510000E-01, 0.3017780000E-01 /
  data (epsData(186,i,20), i=1,3) / 0.7102610000E+03, -0.5193200000E-01, 0.2045950000E-01 /
  data (epsData(187,i,20), i=1,3) / 0.7902150000E+03, -0.4303930000E-01, 0.1428430000E-01 /
  data (epsData(188,i,20), i=1,3) / 0.8735100000E+03, -0.3575520000E-01, 0.1013000000E-01 /
  data (epsData(189,i,20), i=1,3) / 0.9631070000E+03, -0.2967100000E-01, 0.7211980000E-02 /
  data (epsData(190,i,20), i=1,3) / 0.1060170000E+04, -0.2458910000E-01, 0.5137750000E-02 /
  data (epsData(191,i,20), i=1,3) / 0.1168450000E+04, -0.2026610000E-01, 0.3634890000E-02 /
  data (epsData(192,i,20), i=1,3) / 0.1291870000E+04, -0.1656460000E-01, 0.2533990000E-02 /
  data (epsData(193,i,20), i=1,3) / 0.1433040000E+04, -0.1342380000E-01, 0.1738690000E-02 /
  data (epsData(194,i,20), i=1,3) / 0.1593790000E+04, -0.1080130000E-01, 0.1181500000E-02 /
  data (epsData(195,i,20), i=1,3) / 0.1773290000E+04, -0.8680850000E-02, 0.8029210000E-03 /
  data (epsData(196,i,20), i=1,3) / 0.1968640000E+04, -0.7000900000E-02, 0.5455270000E-03 /
  data (epsData(197,i,20), i=1,3) / 0.2189640000E+04, -0.5619540000E-02, 0.3682890000E-03 /
  data (epsData(198,i,20), i=1,3) / 0.2437080000E+04, -0.4499750000E-02, 0.2475060000E-03 /
  data (epsData(199,i,20), i=1,3) / 0.2714880000E+04, -0.3590670000E-02, 0.1654580000E-03 /
  data (epsData(200,i,20), i=1,3) / 0.3026700000E+04, -0.2851930000E-02, 0.1100930000E-03 /
  data (epsData(201,i,20), i=1,3) / 0.3382420000E+04, -0.2237650000E-02, 0.7248180000E-04 /
  data (epsData(202,i,20), i=1,3) / 0.3953570000E+04, -0.1464970000E-02, 0.4017450000E-04 /
  data (epsData(203,i,20), i=1,3) / 0.4009460000E+04, -0.1325090000E-02, 0.3808660000E-04 /
  data (epsData(204,i,20), i=1,3) / 0.4026860000E+04, -0.1224040000E-02, 0.4047440000E-04 /
  data (epsData(205,i,20), i=1,3) / 0.4032690000E+04, -0.1148180000E-02, 0.4391690000E-04 /
  data (epsData(206,i,20), i=1,3) / 0.4035600000E+04, -0.1065120000E-02, 0.5166290000E-04 /
  data (epsData(207,i,20), i=1,3) / 0.4037060000E+04, -0.9771200000E-03, 0.6809880000E-04 /
  data (epsData(208,i,20), i=1,3) / 0.4038540000E+04, -0.8837100000E-03, 0.2460180000E-03 /
  data (epsData(209,i,20), i=1,3) / 0.4038750000E+04, -0.8964510000E-03, 0.3037110000E-03 /
  data (epsData(210,i,20), i=1,3) / 0.4039070000E+04, -0.9472370000E-03, 0.3574890000E-03 /
  data (epsData(211,i,20), i=1,3) / 0.4039680000E+04, -0.1033420000E-02, 0.3778650000E-03 /
  data (epsData(212,i,20), i=1,3) / 0.4041190000E+04, -0.1120450000E-02, 0.3674810000E-03 /
  data (epsData(213,i,20), i=1,3) / 0.4043900000E+04, -0.1212960000E-02, 0.3390670000E-03 /
  data (epsData(214,i,20), i=1,3) / 0.4046990000E+04, -0.1224340000E-02, 0.2999590000E-03 /
  data (epsData(215,i,20), i=1,3) / 0.4062570000E+04, -0.1278280000E-02, 0.3267380000E-03 /
  data (epsData(216,i,20), i=1,3) / 0.4082340000E+04, -0.1335300000E-02, 0.3186620000E-03 /
  data (epsData(217,i,20), i=1,3) / 0.4167570000E+04, -0.1376540000E-02, 0.2988670000E-03 /
  data (epsData(218,i,20), i=1,3) / 0.4453080000E+04, -0.1306890000E-02, 0.2377430000E-03 /
  data (epsData(219,i,20), i=1,3) / 0.5187620000E+04, -0.1021300000E-02, 0.1372820000E-03 /
  data (epsData(220,i,20), i=1,3) / 0.5775880000E+04, -0.8384750000E-03, 0.9430350000E-04 /
  data (epsData(221,i,20), i=1,3) / 0.6476170000E+04, -0.6744530000E-03, 0.6218760000E-04 /
  data (epsData(222,i,20), i=1,3) / 0.7236090000E+04, -0.5433810000E-03, 0.4139080000E-04 /
  data (epsData(223,i,20), i=1,3) / 0.8087100000E+04, -0.4362490000E-03, 0.2725480000E-04 /
  data (epsData(224,i,20), i=1,3) / 0.9038330000E+04, -0.3496230000E-03, 0.1794630000E-04 /
  data (epsData(225,i,20), i=1,3) / 0.1009850000E+05, -0.2799870000E-03, 0.1175220000E-04 /
  data (epsData(226,i,20), i=1,3) / 0.1128240000E+05, -0.2240900000E-03, 0.7669820000E-05 /
  data (epsData(227,i,20), i=1,3) / 0.1260940000E+05, -0.1791550000E-03, 0.4985200000E-05 /
  data (epsData(228,i,20), i=1,3) / 0.1409970000E+05, -0.1430540000E-03, 0.3220910000E-05 /
  data (epsData(229,i,20), i=1,3) / 0.1576920000E+05, -0.1141700000E-03, 0.2074500000E-05 /
  data (epsData(230,i,20), i=1,3) / 0.1764410000E+05, -0.9103960000E-04, 0.1329070000E-05 /
  data (epsData(231,i,20), i=1,3) / 0.1975100000E+05, -0.7253490000E-04, 0.8473610000E-06 /
  data (epsData(232,i,20), i=1,3) / 0.2211370000E+05, -0.5777590000E-04, 0.5386840000E-06 /
  data (epsData(233,i,20), i=1,3) / 0.2476900000E+05, -0.4598880000E-04, 0.3407640000E-06 /
  data (epsData(234,i,20), i=1,3) / 0.2774810000E+05, -0.3659870000E-04, 0.2147640000E-06 /
  data (epsData(235,i,20), i=1,3) / 0.3109670000E+05, -0.2910870000E-04, 0.1347490000E-06 /
  data (epsData(236,i,20), i=1,3) / 0.3485400000E+05, -0.2314900000E-04, 0.8440990000E-07 /
  data (epsData(237,i,20), i=1,3) / 0.3906930000E+05, -0.1840780000E-04, 0.5258910000E-07 /
  data (epsData(238,i,20), i=1,3) / 0.4380440000E+05, -0.1463270000E-04, 0.3278340000E-07 /
  data (epsData(239,i,20), i=1,3) / 0.4911330000E+05, -0.1163320000E-04, 0.2024240000E-07 /
  data (epsData(240,i,20), i=1,3) / 0.5507930000E+05, -0.9244720000E-05, 0.1261860000E-07 /
  data (epsData(241,i,20), i=1,3) / 0.6176840000E+05, -0.7347630000E-05, 0.7785830000E-08 /
  data (epsData(242,i,20), i=1,3) / 0.6928850000E+05, -0.5837190000E-05, 0.4732730000E-08 /
  data (epsData(243,i,20), i=1,3) / 0.7772840000E+05, -0.4636980000E-05, 0.2918340000E-08 /
  data (epsData(244,i,20), i=1,3) / 0.8720160000E+05, -0.3683410000E-05, 0.1765080000E-08 /
  data (epsData(245,i,20), i=1,3) / 0.9781110000E+05, -0.2927480000E-05, 0.1091330000E-08 /
  data (epsData(246,i,20), i=1,3) / 0.1000000000E+06, -0.2802870000E-05, 0.9993650000E-09 /
  data (epsData(1,i,21), i=1,3) / 0.2506580000E-02, -0.4741540000E+03, 0.1378920000E+04 /
  data (epsData(2,i,21), i=1,3) / 0.4693440000E-02, -0.3889200000E+03, 0.4699980000E+03 /
  data (epsData(3,i,21), i=1,3) / 0.7560590000E-02, -0.1535430000E+03, 0.2198740000E+03 /
  data (epsData(4,i,21), i=1,3) / 0.1110800000E-01, -0.5078610000E+02, 0.1240750000E+03 /
  data (epsData(5,i,21), i=1,3) / 0.1533570000E-01, -0.2444590000E+01, 0.7898390000E+02 /
  data (epsData(6,i,21), i=1,3) / 0.2024370000E-01, 0.2257080000E+02, 0.5458120000E+02 /
  data (epsData(7,i,21), i=1,3) / 0.2583200000E-01, 0.3657600000E+02, 0.3999070000E+02 /
  data (epsData(8,i,21), i=1,3) / 0.3210060000E-01, 0.4492800000E+02, 0.3059720000E+02 /
  data (epsData(9,i,21), i=1,3) / 0.3904940000E-01, 0.5016990000E+02, 0.2419740000E+02 /
  data (epsData(10,i,21), i=1,3) / 0.4667860000E-01, 0.5360150000E+02, 0.1963940000E+02 /
  data (epsData(11,i,21), i=1,3) / 0.5498800000E-01, 0.5592880000E+02, 0.1627560000E+02 /
  data (epsData(12,i,21), i=1,3) / 0.7364770000E-01, 0.5872310000E+02, 0.1173110000E+02 /
  data (epsData(13,i,21), i=1,3) / 0.1067390000E+00, 0.6071780000E+02, 0.7828870000E+01 /
  data (epsData(14,i,21), i=1,3) / 0.1754980000E+00, 0.6208190000E+02, 0.4656560000E+01 /
  data (epsData(15,i,21), i=1,3) / 0.4099070000E+00, 0.6285260000E+02, 0.2142890000E+01 /
  data (epsData(16,i,21), i=1,3) / 0.1133010000E+01, 0.6548320000E+02, 0.1015530000E+01 /
  data (epsData(17,i,21), i=1,3) / 0.1898770000E+01, 0.7190090000E+02, 0.8973860000E+00 /
  data (epsData(18,i,21), i=1,3) / 0.2441050000E+01, 0.8054290000E+02, 0.1038780000E+01 /
  data (epsData(19,i,21), i=1,3) / 0.2861130000E+01, 0.9202500000E+02, 0.1318620000E+01 /
  data (epsData(20,i,21), i=1,3) / 0.3116140000E+01, 0.1031280000E+03, 0.1749510000E+01 /
  data (epsData(21,i,21), i=1,3) / 0.3314540000E+01, 0.1161930000E+03, 0.2536880000E+01 /
  data (epsData(22,i,21), i=1,3) / 0.3450210000E+01, 0.1298020000E+03, 0.3088640000E+01 /
  data (epsData(23,i,21), i=1,3) / 0.3588600000E+01, 0.1498720000E+03, 0.7515620000E+01 /
  data (epsData(24,i,21), i=1,3) / 0.3658810000E+01, 0.1648340000E+03, 0.9789810000E+01 /
  data (epsData(25,i,21), i=1,3) / 0.3729710000E+01, 0.1912440000E+03, 0.1190850000E+02 /
  data (epsData(26,i,21), i=1,3) / 0.3801290000E+01, 0.2206020000E+03, 0.4556390000E+02 /
  data (epsData(27,i,21), i=1,3) / 0.3873540000E+01, 0.2219490000E+03, 0.1099990000E+03 /
  data (epsData(28,i,21), i=1,3) / 0.3946480000E+01, 0.2001920000E+03, 0.1413980000E+03 /
  data (epsData(29,i,21), i=1,3) / 0.4020090000E+01, 0.1728170000E+03, 0.1707280000E+03 /
  data (epsData(30,i,21), i=1,3) / 0.4094390000E+01, 0.1380800000E+03, 0.1864200000E+03 /
  data (epsData(31,i,21), i=1,3) / 0.4169370000E+01, 0.1081750000E+03, 0.1803610000E+03 /
  data (epsData(32,i,21), i=1,3) / 0.4321360000E+01, 0.7350360000E+02, 0.1690020000E+03 /
  data (epsData(33,i,21), i=1,3) / 0.4554460000E+01, 0.4238760000E+02, 0.1540350000E+03 /
  data (epsData(34,i,21), i=1,3) / 0.4793680000E+01, 0.2153200000E+02, 0.1409950000E+03 /
  data (epsData(35,i,21), i=1,3) / 0.5122160000E+01, 0.6580130000E+00, 0.1250490000E+03 /
  data (epsData(36,i,21), i=1,3) / 0.5461520000E+01, -0.1510910000E+02, 0.1085600000E+03 /
  data (epsData(37,i,21), i=1,3) / 0.5811770000E+01, -0.2572080000E+02, 0.9081790000E+02 /
  data (epsData(38,i,21), i=1,3) / 0.6081600000E+01, -0.2969100000E+02, 0.7717950000E+02 /
  data (epsData(39,i,21), i=1,3) / 0.6357560000E+01, -0.3012350000E+02, 0.6441920000E+02 /
  data (epsData(40,i,21), i=1,3) / 0.6639630000E+01, -0.2694770000E+02, 0.5319420000E+02 /
  data (epsData(41,i,21), i=1,3) / 0.6831080000E+01, -0.2216020000E+02, 0.4668380000E+02 /
  data (epsData(42,i,21), i=1,3) / 0.6927830000E+01, -0.1855390000E+02, 0.4443490000E+02 /
  data (epsData(43,i,21), i=1,3) / 0.7025260000E+01, -0.1343020000E+02, 0.4233910000E+02 /
  data (epsData(44,i,21), i=1,3) / 0.7123360000E+01, -0.3489940000E+01, 0.4038490000E+02 /
  data (epsData(45,i,21), i=1,3) / 0.7222150000E+01, 0.5175580000E+00, 0.6004200000E+02 /
  data (epsData(46,i,21), i=1,3) / 0.7321620000E+01, -0.6953720000E+01, 0.6894410000E+02 /
  data (epsData(47,i,21), i=1,3) / 0.7421760000E+01, -0.1857990000E+02, 0.7458660000E+02 /
  data (epsData(48,i,21), i=1,3) / 0.7522590000E+01, -0.2975630000E+02, 0.6896500000E+02 /
  data (epsData(49,i,21), i=1,3) / 0.7624100000E+01, -0.3594320000E+02, 0.6137090000E+02 /
  data (epsData(50,i,21), i=1,3) / 0.7726290000E+01, -0.3866790000E+02, 0.5435920000E+02 /
  data (epsData(51,i,21), i=1,3) / 0.7932710000E+01, -0.3972830000E+02, 0.4329140000E+02 /
  data (epsData(52,i,21), i=1,3) / 0.8036940000E+01, -0.3882470000E+02, 0.3856700000E+02 /
  data (epsData(53,i,21), i=1,3) / 0.8353710000E+01, -0.3494200000E+02, 0.2904910000E+02 /
  data (epsData(54,i,21), i=1,3) / 0.8785590000E+01, -0.2937870000E+02, 0.2107460000E+02 /
  data (epsData(55,i,21), i=1,3) / 0.9228360000E+01, -0.2403230000E+02, 0.1566000000E+02 /
  data (epsData(56,i,21), i=1,3) / 0.9797130000E+01, -0.1774200000E+02, 0.1136960000E+02 /
  data (epsData(57,i,21), i=1,3) / 0.1014660000E+02, -0.1364240000E+02, 0.9710260000E+01 /
  data (epsData(58,i,21), i=1,3) / 0.1026440000E+02, -0.1199530000E+02, 0.9422760000E+01 /
  data (epsData(59,i,21), i=1,3) / 0.1038290000E+02, -0.9793250000E+01, 0.9256170000E+01 /
  data (epsData(60,i,21), i=1,3) / 0.1050210000E+02, -0.8509160000E+01, 0.1202430000E+02 /
  data (epsData(61,i,21), i=1,3) / 0.1062200000E+02, -0.8971370000E+01, 0.1341900000E+02 /
  data (epsData(62,i,21), i=1,3) / 0.1074250000E+02, -0.9820570000E+01, 0.1372040000E+02 /
  data (epsData(63,i,21), i=1,3) / 0.1086380000E+02, -0.1034610000E+02, 0.1311930000E+02 /
  data (epsData(64,i,21), i=1,3) / 0.1110830000E+02, -0.1026400000E+02, 0.1198980000E+02 /
  data (epsData(65,i,21), i=1,3) / 0.1160550000E+02, -0.8886290000E+01, 0.1031460000E+02 /
  data (epsData(66,i,21), i=1,3) / 0.1224230000E+02, -0.6773470000E+01, 0.9089060000E+01 /
  data (epsData(67,i,21), i=1,3) / 0.1302890000E+02, -0.4587290000E+01, 0.8449190000E+01 /
  data (epsData(68,i,21), i=1,3) / 0.1384000000E+02, -0.2806300000E+01, 0.8261830000E+01 /
  data (epsData(69,i,21), i=1,3) / 0.1439430000E+02, -0.2026070000E+01, 0.8429480000E+01 /
  data (epsData(70,i,21), i=1,3) / 0.1510250000E+02, -0.1222530000E+01, 0.8225170000E+01 /
  data (epsData(71,i,21), i=1,3) / 0.1568130000E+02, -0.4839750000E+00, 0.8377970000E+01 /
  data (epsData(72,i,21), i=1,3) / 0.1687170000E+02, 0.2788390000E+00, 0.8861200000E+01 /
  data (epsData(73,i,21), i=1,3) / 0.1810550000E+02, 0.4669010000E+00, 0.9281340000E+01 /
  data (epsData(74,i,21), i=1,3) / 0.1970900000E+02, 0.2265530000E+00, 0.9475250000E+01 /
  data (epsData(75,i,21), i=1,3) / 0.2224190000E+02, -0.3847700000E+00, 0.9102990000E+01 /
  data (epsData(76,i,21), i=1,3) / 0.2456080000E+02, -0.7002480000E+00, 0.8368220000E+01 /
  data (epsData(77,i,21), i=1,3) / 0.2623360000E+02, -0.5429190000E+00, 0.7732250000E+01 /
  data (epsData(78,i,21), i=1,3) / 0.2718670000E+02, -0.6751830000E-01, 0.7359020000E+01 /
  data (epsData(79,i,21), i=1,3) / 0.2757270000E+02, 0.3640070000E+00, 0.7217630000E+01 /
  data (epsData(80,i,21), i=1,3) / 0.2776670000E+02, 0.7210110000E+00, 0.7163570000E+01 /
  data (epsData(81,i,21), i=1,3) / 0.2796150000E+02, 0.1264450000E+01, 0.7181270000E+01 /
  data (epsData(82,i,21), i=1,3) / 0.2815680000E+02, 0.2237890000E+01, 0.7340900000E+01 /
  data (epsData(83,i,21), i=1,3) / 0.2831670000E+02, 0.2736340000E+01, 0.8917800000E+01 /
  data (epsData(84,i,21), i=1,3) / 0.2846460000E+02, 0.2246530000E+01, 0.1012960000E+02 /
  data (epsData(85,i,21), i=1,3) / 0.2851260000E+02, 0.1793320000E+01, 0.1041600000E+02 /
  data (epsData(86,i,21), i=1,3) / 0.2869320000E+02, 0.8896830000E+00, 0.1042850000E+02 /
  data (epsData(87,i,21), i=1,3) / 0.2911670000E+02, -0.1048240000E+00, 0.1039520000E+02 /
  data (epsData(88,i,21), i=1,3) / 0.3021330000E+02, -0.1766670000E+01, 0.1019990000E+02 /
  data (epsData(89,i,21), i=1,3) / 0.3117900000E+02, -0.2757980000E+01, 0.9476740000E+01 /
  data (epsData(90,i,21), i=1,3) / 0.3249090000E+02, -0.3580800000E+01, 0.8478520000E+01 /
  data (epsData(91,i,21), i=1,3) / 0.3535320000E+02, -0.4071880000E+01, 0.6626590000E+01 /
  data (epsData(92,i,21), i=1,3) / 0.3790020000E+02, -0.4038710000E+01, 0.5302760000E+01 /
  data (epsData(93,i,21), i=1,3) / 0.4123390000E+02, -0.3663850000E+01, 0.4117660000E+01 /
  data (epsData(94,i,21), i=1,3) / 0.4567310000E+02, -0.3135880000E+01, 0.3141000000E+01 /
  data (epsData(95,i,21), i=1,3) / 0.5041770000E+02, -0.2610050000E+01, 0.2506400000E+01 /
  data (epsData(96,i,21), i=1,3) / 0.5224420000E+02, -0.2512160000E+01, 0.2409640000E+01 /
  data (epsData(97,i,21), i=1,3) / 0.5906240000E+02, -0.2162660000E+01, 0.1795350000E+01 /
  data (epsData(98,i,21), i=1,3) / 0.6610030000E+02, -0.1811700000E+01, 0.1400640000E+01 /
  data (epsData(99,i,21), i=1,3) / 0.7480650000E+02, -0.1501170000E+01, 0.1076200000E+01 /
  data (epsData(100,i,21), i=1,3) / 0.8442340000E+02, -0.1247200000E+01, 0.8312610000E+00 /
  data (epsData(101,i,21), i=1,3) / 0.9592020000E+02, -0.1025310000E+01, 0.6290270000E+00 /
  data (epsData(102,i,21), i=1,3) / 0.1093990000E+03, -0.8347850000E+00, 0.4665700000E+00 /
  data (epsData(103,i,21), i=1,3) / 0.1240160000E+03, -0.6788040000E+00, 0.3441170000E+00 /
  data (epsData(104,i,21), i=1,3) / 0.1395720000E+03, -0.5528690000E+00, 0.2575180000E+00 /
  data (epsData(105,i,21), i=1,3) / 0.1558740000E+03, -0.4539570000E+00, 0.1968460000E+00 /
  data (epsData(106,i,21), i=1,3) / 0.1768400000E+03, -0.3603460000E+00, 0.1418490000E+00 /
  data (epsData(107,i,21), i=1,3) / 0.1988200000E+03, -0.2863760000E+00, 0.1034750000E+00 /
  data (epsData(108,i,21), i=1,3) / 0.2225690000E+03, -0.2272480000E+00, 0.7658090000E-01 /
  data (epsData(109,i,21), i=1,3) / 0.2491250000E+03, -0.1769580000E+00, 0.5578340000E-01 /
  data (epsData(110,i,21), i=1,3) / 0.2775910000E+03, -0.1354160000E+00, 0.4115410000E-01 /
  data (epsData(111,i,21), i=1,3) / 0.3103390000E+03, -0.9742050000E-01, 0.2981540000E-01 /
  data (epsData(112,i,21), i=1,3) / 0.3542210000E+03, -0.5200490000E-01, 0.2009000000E-01 /
  data (epsData(113,i,21), i=1,3) / 0.3698300000E+03, -0.3302990000E-01, 0.1768880000E-01 /
  data (epsData(114,i,21), i=1,3) / 0.3777810000E+03, -0.2076910000E-01, 0.1663600000E-01 /
  data (epsData(115,i,21), i=1,3) / 0.3824650000E+03, -0.1197420000E-01, 0.1620480000E-01 /
  data (epsData(116,i,21), i=1,3) / 0.3858310000E+03, -0.4406490000E-02, 0.1592360000E-01 /
  data (epsData(117,i,21), i=1,3) / 0.3878580000E+03, 0.9488150000E-03, 0.1572500000E-01 /
  data (epsData(118,i,21), i=1,3) / 0.3898920000E+03, 0.7205800000E-02, 0.1554600000E-01 /
  data (epsData(119,i,21), i=1,3) / 0.3919320000E+03, 0.1477680000E-01, 0.1540560000E-01 /
  data (epsData(120,i,21), i=1,3) / 0.3939780000E+03, 0.2445460000E-01, 0.1534540000E-01 /
  data (epsData(121,i,21), i=1,3) / 0.3953460000E+03, 0.3294030000E-01, 0.1539700000E-01 /
  data (epsData(122,i,21), i=1,3) / 0.3960300000E+03, 0.3823590000E-01, 0.1548650000E-01 /
  data (epsData(123,i,21), i=1,3) / 0.3967160000E+03, 0.4471140000E-01, 0.1566420000E-01 /
  data (epsData(124,i,21), i=1,3) / 0.3974020000E+03, 0.5333240000E-01, 0.1604410000E-01 /
  data (epsData(125,i,21), i=1,3) / 0.3980890000E+03, 0.6652570000E-01, 0.1736990000E-01 /
  data (epsData(126,i,21), i=1,3) / 0.3987280000E+03, 0.8134280000E-01, 0.3142620000E-01 /
  data (epsData(127,i,21), i=1,3) / 0.3988960000E+03, 0.8170260000E-01, 0.3994400000E-01 /
  data (epsData(128,i,21), i=1,3) / 0.3993590000E+03, 0.7824510000E-01, 0.4782520000E-01 /
  data (epsData(129,i,21), i=1,3) / 0.4007690000E+03, 0.7625020000E-01, 0.6681530000E-01 /
  data (epsData(130,i,21), i=1,3) / 0.4027410000E+03, 0.7218680000E-01, 0.8373850000E-01 /
  data (epsData(131,i,21), i=1,3) / 0.4031980000E+03, 0.7541560000E-01, 0.8615610000E-01 /
  data (epsData(132,i,21), i=1,3) / 0.4036410000E+03, 0.7949420000E-01, 0.9557920000E-01 /
  data (epsData(133,i,21), i=1,3) / 0.4041440000E+03, 0.7471350000E-01, 0.1063220000E+00 /
  data (epsData(134,i,21), i=1,3) / 0.4057230000E+03, 0.5866780000E-01, 0.1252560000E+00 /
  data (epsData(135,i,21), i=1,3) / 0.4076410000E+03, 0.3642330000E-01, 0.1354490000E+00 /
  data (epsData(136,i,21), i=1,3) / 0.4108500000E+03, 0.8832550000E-02, 0.1387960000E+00 /
  data (epsData(137,i,21), i=1,3) / 0.4132000000E+03, -0.6904730000E-02, 0.1343730000E+00 /
  data (epsData(138,i,21), i=1,3) / 0.4166670000E+03, -0.2003580000E-01, 0.1252900000E+00 /
  data (epsData(139,i,21), i=1,3) / 0.4217500000E+03, -0.2893340000E-01, 0.1165050000E+00 /
  data (epsData(140,i,21), i=1,3) / 0.4334020000E+03, -0.4114650000E-01, 0.1074970000E+00 /
  data (epsData(141,i,21), i=1,3) / 0.4447930000E+03, -0.5221680000E-01, 0.9870180000E-01 /
  data (epsData(142,i,21), i=1,3) / 0.4585680000E+03, -0.5639130000E-01, 0.8725930000E-01 /
  data (epsData(143,i,21), i=1,3) / 0.4869440000E+03, -0.5751650000E-01, 0.7256490000E-01 /
  data (epsData(144,i,21), i=1,3) / 0.4955750000E+03, -0.5432230000E-01, 0.6992040000E-01 /
  data (epsData(145,i,21), i=1,3) / 0.4988170000E+03, -0.5388940000E-01, 0.7282220000E-01 /
  data (epsData(146,i,21), i=1,3) / 0.5034000000E+03, -0.5793780000E-01, 0.7176470000E-01 /
  data (epsData(147,i,21), i=1,3) / 0.5401450000E+03, -0.6020930000E-01, 0.5814350000E-01 /
  data (epsData(148,i,21), i=1,3) / 0.5931870000E+03, -0.5679700000E-01, 0.4342940000E-01 /
  data (epsData(149,i,21), i=1,3) / 0.6704370000E+03, -0.4936050000E-01, 0.2935550000E-01 /
  data (epsData(150,i,21), i=1,3) / 0.7560520000E+03, -0.4124750000E-01, 0.1978100000E-01 /
  data (epsData(151,i,21), i=1,3) / 0.8381520000E+03, -0.3472310000E-01, 0.1401710000E-01 /
  data (epsData(152,i,21), i=1,3) / 0.9246870000E+03, -0.2912220000E-01, 0.1001040000E-01 /
  data (epsData(153,i,21), i=1,3) / 0.1021860000E+04, -0.2417210000E-01, 0.7064010000E-02 /
  data (epsData(154,i,21), i=1,3) / 0.1129930000E+04, -0.1991510000E-01, 0.4947610000E-02 /
  data (epsData(155,i,21), i=1,3) / 0.1252160000E+04, -0.1625790000E-01, 0.3429020000E-02 /
  data (epsData(156,i,21), i=1,3) / 0.1394010000E+04, -0.1310640000E-01, 0.2339650000E-02 /
  data (epsData(157,i,21), i=1,3) / 0.1549270000E+04, -0.1058510000E-01, 0.1600270000E-02 /
  data (epsData(158,i,21), i=1,3) / 0.1713070000E+04, -0.8623610000E-02, 0.1111160000E-02 /
  data (epsData(159,i,21), i=1,3) / 0.1896980000E+04, -0.6997080000E-02, 0.7637670000E-03 /
  data (epsData(160,i,21), i=1,3) / 0.2102710000E+04, -0.5656920000E-02, 0.5236100000E-03 /
  data (epsData(161,i,21), i=1,3) / 0.2336750000E+04, -0.4547190000E-02, 0.3546290000E-03 /
  data (epsData(162,i,21), i=1,3) / 0.2598540000E+04, -0.3646450000E-02, 0.2390290000E-03 /
  data (epsData(163,i,21), i=1,3) / 0.2892510000E+04, -0.2913940000E-02, 0.1603200000E-03 /
  data (epsData(164,i,21), i=1,3) / 0.3221360000E+04, -0.2320410000E-02, 0.1070910000E-03 /
  data (epsData(165,i,21), i=1,3) / 0.3592860000E+04, -0.1832550000E-02, 0.7101600000E-04 /
  data (epsData(166,i,21), i=1,3) / 0.4038750000E+04, -0.1399040000E-02, 0.4563620000E-04 /
  data (epsData(167,i,21), i=1,3) / 0.4390380000E+04, -0.1080540000E-02, 0.3324720000E-04 /
  data (epsData(168,i,21), i=1,3) / 0.4456800000E+04, -0.9704810000E-03, 0.3140540000E-04 /
  data (epsData(169,i,21), i=1,3) / 0.4477740000E+04, -0.8906970000E-03, 0.3296770000E-04 /
  data (epsData(170,i,21), i=1,3) / 0.4485260000E+04, -0.8260470000E-03, 0.3572300000E-04 /
  data (epsData(171,i,21), i=1,3) / 0.4488270000E+04, -0.7714810000E-03, 0.4053950000E-04 /
  data (epsData(172,i,21), i=1,3) / 0.4489770000E+04, -0.7200270000E-03, 0.4830890000E-04 /
  data (epsData(173,i,21), i=1,3) / 0.4491280000E+04, -0.6247100000E-03, 0.8726730000E-04 /
  data (epsData(174,i,21), i=1,3) / 0.4492070000E+04, -0.5855620000E-03, 0.2213460000E-03 /
  data (epsData(175,i,21), i=1,3) / 0.4492270000E+04, -0.6062070000E-03, 0.2632040000E-03 /
  data (epsData(176,i,21), i=1,3) / 0.4492570000E+04, -0.6508070000E-03, 0.3000660000E-03 /
  data (epsData(177,i,21), i=1,3) / 0.4493140000E+04, -0.7161660000E-03, 0.3128060000E-03 /
  data (epsData(178,i,21), i=1,3) / 0.4494940000E+04, -0.8151070000E-03, 0.3045300000E-03 /
  data (epsData(179,i,21), i=1,3) / 0.4497750000E+04, -0.8740290000E-03, 0.2607760000E-03 /
  data (epsData(180,i,21), i=1,3) / 0.4501710000E+04, -0.8749520000E-03, 0.2381970000E-03 /
  data (epsData(181,i,21), i=1,3) / 0.4514470000E+04, -0.9164750000E-03, 0.2614990000E-03 /
  data (epsData(182,i,21), i=1,3) / 0.4534150000E+04, -0.9644970000E-03, 0.2563960000E-03 /
  data (epsData(183,i,21), i=1,3) / 0.4604070000E+04, -0.1006440000E-02, 0.2449090000E-03 /
  data (epsData(184,i,21), i=1,3) / 0.4854270000E+04, -0.9889500000E-03, 0.2050080000E-03 /
  data (epsData(185,i,21), i=1,3) / 0.5765620000E+04, -0.7595750000E-03, 0.1104420000E-03 /
  data (epsData(186,i,21), i=1,3) / 0.6467660000E+04, -0.6160650000E-03, 0.7379790000E-04 /
  data (epsData(187,i,21), i=1,3) / 0.7249100000E+04, -0.4959630000E-03, 0.4861970000E-04 /
  data (epsData(188,i,21), i=1,3) / 0.8113670000E+04, -0.3982330000E-03, 0.3207090000E-04 /
  data (epsData(189,i,21), i=1,3) / 0.9065820000E+04, -0.3199060000E-03, 0.2118830000E-04 /
  data (epsData(190,i,21), i=1,3) / 0.1012340000E+05, -0.2567430000E-03, 0.1395190000E-04 /
  data (epsData(191,i,21), i=1,3) / 0.1130620000E+05, -0.2057550000E-03, 0.9144360000E-05 /
  data (epsData(192,i,21), i=1,3) / 0.1263140000E+05, -0.1646580000E-03, 0.5967280000E-05 /
  data (epsData(193,i,21), i=1,3) / 0.1411420000E+05, -0.1316750000E-03, 0.3875390000E-05 /
  data (epsData(194,i,21), i=1,3) / 0.1577570000E+05, -0.1052080000E-03, 0.2508300000E-05 /
  data (epsData(195,i,21), i=1,3) / 0.1764120000E+05, -0.8397600000E-04, 0.1614380000E-05 /
  data (epsData(196,i,21), i=1,3) / 0.1973630000E+05, -0.6696970000E-04, 0.1033800000E-05 /
  data (epsData(197,i,21), i=1,3) / 0.2208920000E+05, -0.5336850000E-04, 0.6594740000E-06 /
  data (epsData(198,i,21), i=1,3) / 0.2473140000E+05, -0.4250490000E-04, 0.4186240000E-06 /
  data (epsData(199,i,21), i=1,3) / 0.2769870000E+05, -0.3383560000E-04, 0.2646750000E-06 /
  data (epsData(200,i,21), i=1,3) / 0.3103080000E+05, -0.2692300000E-04, 0.1665870000E-06 /
  data (epsData(201,i,21), i=1,3) / 0.3477230000E+05, -0.2141570000E-04, 0.1046430000E-06 /
  data (epsData(202,i,21), i=1,3) / 0.3897250000E+05, -0.1703070000E-04, 0.6532830000E-07 /
  data (epsData(203,i,21), i=1,3) / 0.4368610000E+05, -0.1354170000E-04, 0.4076670000E-07 /
  data (epsData(204,i,21), i=1,3) / 0.4897340000E+05, -0.1076740000E-04, 0.2528270000E-07 /
  data (epsData(205,i,21), i=1,3) / 0.5491780000E+05, -0.8556870000E-05, 0.1571880000E-07 /
  data (epsData(206,i,21), i=1,3) / 0.6158590000E+05, -0.6800490000E-05, 0.9735440000E-08 /
  data (epsData(207,i,21), i=1,3) / 0.6907510000E+05, -0.5403370000E-05, 0.5939450000E-08 /
  data (epsData(208,i,21), i=1,3) / 0.7747310000E+05, -0.4293780000E-05, 0.3665470000E-08 /
  data (epsData(209,i,21), i=1,3) / 0.8691260000E+05, -0.3410780000E-05, 0.2221200000E-08 /
  data (epsData(210,i,21), i=1,3) / 0.9747600000E+05, -0.2711330000E-05, 0.1371180000E-08 /
  data (epsData(211,i,21), i=1,3) / 0.1000000000E+06, -0.2579940000E-05, 0.1238110000E-08 /
  data (epsData(1,i,22), i=1,3) / 0.2506580000E-02, -0.8076590000E+03, 0.2550630000E+04 /
  data (epsData(2,i,22), i=1,3) / 0.4693440000E-02, -0.6772060000E+03, 0.8829340000E+03 /
  data (epsData(3,i,22), i=1,3) / 0.7560590000E-02, -0.2570480000E+03, 0.4187870000E+03 /
  data (epsData(4,i,22), i=1,3) / 0.1110800000E-01, -0.7290220000E+02, 0.2390300000E+03 /
  data (epsData(5,i,22), i=1,3) / 0.1533570000E-01, 0.1412590000E+02, 0.1535520000E+03 /
  data (epsData(6,i,22), i=1,3) / 0.2024370000E-01, 0.5918730000E+02, 0.1077600000E+03 /
  data (epsData(7,i,22), i=1,3) / 0.2583200000E-01, 0.8395530000E+02, 0.8022960000E+02 /
  data (epsData(8,i,22), i=1,3) / 0.3210060000E-01, 0.9844190000E+02, 0.6217570000E+02 /
  data (epsData(9,i,22), i=1,3) / 0.3904940000E-01, 0.1074330000E+03, 0.4968400000E+02 /
  data (epsData(10,i,22), i=1,3) / 0.4667860000E-01, 0.1132800000E+03, 0.4066200000E+02 /
  data (epsData(11,i,22), i=1,3) / 0.5498800000E-01, 0.1172350000E+03, 0.3392540000E+02 /
  data (epsData(12,i,22), i=1,3) / 0.7364770000E-01, 0.1220180000E+03, 0.2470080000E+02 /
  data (epsData(13,i,22), i=1,3) / 0.1067390000E+00, 0.1257690000E+03, 0.1690200000E+02 /
  data (epsData(14,i,22), i=1,3) / 0.1912910000E+00, 0.1275260000E+03, 0.1014870000E+02 /
  data (epsData(15,i,22), i=1,3) / 0.4338630000E+00, 0.1295290000E+03, 0.4932010000E+01 /
  data (epsData(16,i,22), i=1,3) / 0.9096770000E+00, 0.1369060000E+03, 0.3342610000E+01 /
  data (epsData(17,i,22), i=1,3) / 0.1295500000E+01, 0.1506470000E+03, 0.3531280000E+01 /
  data (epsData(18,i,22), i=1,3) / 0.1513920000E+01, 0.1650850000E+03, 0.4914630000E+01 /
  data (epsData(19,i,22), i=1,3) / 0.1606050000E+01, 0.1758000000E+03, 0.5741300000E+01 /
  data (epsData(20,i,22), i=1,3) / 0.1798470000E+01, 0.2024230000E+03, 0.1801370000E+02 /
  data (epsData(21,i,22), i=1,3) / 0.1848280000E+01, 0.2152420000E+03, 0.2035670000E+02 /
  data (epsData(22,i,22), i=1,3) / 0.1898770000E+01, 0.2404720000E+03, 0.2246350000E+02 /
  data (epsData(23,i,22), i=1,3) / 0.1949930000E+01, 0.2584240000E+03, 0.7013570000E+02 /
  data (epsData(24,i,22), i=1,3) / 0.2003060000E+01, 0.2463120000E+03, 0.1142520000E+03 /
  data (epsData(25,i,22), i=1,3) / 0.2164630000E+01, 0.1836540000E+03, 0.1636270000E+03 /
  data (epsData(26,i,22), i=1,3) / 0.2195920000E+01, 0.1655210000E+03, 0.1713860000E+03 /
  data (epsData(27,i,22), i=1,3) / 0.2229940000E+01, 0.1453950000E+03, 0.1664670000E+03 /
  data (epsData(28,i,22), i=1,3) / 0.2306130000E+01, 0.1206970000E+03, 0.1525650000E+03 /
  data (epsData(29,i,22), i=1,3) / 0.2416670000E+01, 0.1009340000E+03, 0.1348510000E+03 /
  data (epsData(30,i,22), i=1,3) / 0.2571780000E+01, 0.8733460000E+02, 0.1144650000E+03 /
  data (epsData(31,i,22), i=1,3) / 0.2751370000E+01, 0.8173910000E+02, 0.9581390000E+02 /
  data (epsData(32,i,22), i=1,3) / 0.2991850000E+01, 0.8384390000E+02, 0.7913500000E+02 /
  data (epsData(33,i,22), i=1,3) / 0.3224510000E+01, 0.9117360000E+02, 0.6843460000E+02 /
  data (epsData(34,i,22), i=1,3) / 0.3481660000E+01, 0.1052160000E+03, 0.6065140000E+02 /
  data (epsData(35,i,22), i=1,3) / 0.3619420000E+01, 0.1179230000E+03, 0.5754570000E+02 /
  data (epsData(36,i,22), i=1,3) / 0.3763300000E+01, 0.1369810000E+03, 0.5856300000E+02 /
  data (epsData(37,i,22), i=1,3) / 0.3812620000E+01, 0.1463170000E+03, 0.5882580000E+02 /
  data (epsData(38,i,22), i=1,3) / 0.3862620000E+01, 0.1600930000E+03, 0.5905920000E+02 /
  data (epsData(39,i,22), i=1,3) / 0.3913300000E+01, 0.1853370000E+03, 0.5926400000E+02 /
  data (epsData(40,i,22), i=1,3) / 0.3964660000E+01, 0.2059840000E+03, 0.9867070000E+02 /
  data (epsData(41,i,22), i=1,3) / 0.4016700000E+01, 0.2019470000E+03, 0.1395370000E+03 /
  data (epsData(42,i,22), i=1,3) / 0.4122830000E+01, 0.1738850000E+03, 0.1775750000E+03 /
  data (epsData(43,i,22), i=1,3) / 0.4176910000E+01, 0.1557880000E+03, 0.1953590000E+03 /
  data (epsData(44,i,22), i=1,3) / 0.4231670000E+01, 0.1321390000E+03, 0.2069040000E+03 /
  data (epsData(45,i,22), i=1,3) / 0.4287120000E+01, 0.1106880000E+03, 0.2031530000E+03 /
  data (epsData(46,i,22), i=1,3) / 0.4400040000E+01, 0.8429040000E+02, 0.1959590000E+03 /
  data (epsData(47,i,22), i=1,3) / 0.4574530000E+01, 0.5881450000E+02, 0.1860000000E+03 /
  data (epsData(48,i,22), i=1,3) / 0.4941890000E+01, 0.2419250000E+02, 0.1687390000E+03 /
  data (epsData(49,i,22), i=1,3) / 0.5266720000E+01, 0.1878430000E+01, 0.1534410000E+03 /
  data (epsData(50,i,22), i=1,3) / 0.5538840000E+01, -0.1260230000E+02, 0.1391230000E+03 /
  data (epsData(51,i,22), i=1,3) / 0.5894290000E+01, -0.2514320000E+02, 0.1192460000E+03 /
  data (epsData(52,i,22), i=1,3) / 0.6266740000E+01, -0.3053040000E+02, 0.9888060000E+02 /
  data (epsData(53,i,22), i=1,3) / 0.6656200000E+01, -0.2945810000E+02, 0.8140770000E+02 /
  data (epsData(54,i,22), i=1,3) / 0.6980020000E+01, -0.2542970000E+02, 0.7038750000E+02 /
  data (epsData(55,i,22), i=1,3) / 0.7230020000E+01, -0.1989680000E+02, 0.6344040000E+02 /
  data (epsData(56,i,22), i=1,3) / 0.7400090000E+01, -0.1331570000E+02, 0.6031330000E+02 /
  data (epsData(57,i,22), i=1,3) / 0.7486150000E+01, -0.6622170000E+01, 0.5881740000E+02 /
  data (epsData(58,i,22), i=1,3) / 0.7572890000E+01, -0.6479810000E+00, 0.6739520000E+02 /
  data (epsData(59,i,22), i=1,3) / 0.7660300000E+01, -0.2964960000E+01, 0.8040760000E+02 /
  data (epsData(60,i,22), i=1,3) / 0.7748400000E+01, -0.1054950000E+02, 0.8420480000E+02 /
  data (epsData(61,i,22), i=1,3) / 0.7837180000E+01, -0.1958060000E+02, 0.8703250000E+02 /
  data (epsData(62,i,22), i=1,3) / 0.7926630000E+01, -0.2751450000E+02, 0.8119800000E+02 /
  data (epsData(63,i,22), i=1,3) / 0.8016770000E+01, -0.3148840000E+02, 0.7557680000E+02 /
  data (epsData(64,i,22), i=1,3) / 0.8107590000E+01, -0.3316500000E+02, 0.7014430000E+02 /
  data (epsData(65,i,22), i=1,3) / 0.8384120000E+01, -0.3370290000E+02, 0.5935430000E+02 /
  data (epsData(66,i,22), i=1,3) / 0.8762360000E+01, -0.3126200000E+02, 0.4983260000E+02 /
  data (epsData(67,i,22), i=1,3) / 0.9350130000E+01, -0.2680800000E+02, 0.4077580000E+02 /
  data (epsData(68,i,22), i=1,3) / 0.1006680000E+02, -0.2209650000E+02, 0.3425720000E+02 /
  data (epsData(69,i,22), i=1,3) / 0.1092670000E+02, -0.1795270000E+02, 0.2962270000E+02 /
  data (epsData(70,i,22), i=1,3) / 0.1183010000E+02, -0.1510080000E+02, 0.2666290000E+02 /
  data (epsData(71,i,22), i=1,3) / 0.1339100000E+02, -0.1248750000E+02, 0.2279890000E+02 /
  data (epsData(72,i,22), i=1,3) / 0.1560570000E+02, -0.1062760000E+02, 0.1882490000E+02 /
  data (epsData(73,i,22), i=1,3) / 0.1906700000E+02, -0.9114340000E+01, 0.1407800000E+02 /
  data (epsData(74,i,22), i=1,3) / 0.2275170000E+02, -0.7658950000E+01, 0.1034420000E+02 /
  data (epsData(75,i,22), i=1,3) / 0.2643040000E+02, -0.6178850000E+01, 0.7891300000E+01 /
  data (epsData(76,i,22), i=1,3) / 0.3021270000E+02, -0.4664520000E+01, 0.6275320000E+01 /
  data (epsData(77,i,22), i=1,3) / 0.3160080000E+02, -0.3854410000E+01, 0.5906220000E+01 /
  data (epsData(78,i,22), i=1,3) / 0.3220600000E+02, -0.3251480000E+01, 0.5945530000E+01 /
  data (epsData(79,i,22), i=1,3) / 0.3262750000E+02, -0.2868780000E+01, 0.6472950000E+01 /
  data (epsData(80,i,22), i=1,3) / 0.3290610000E+02, -0.2996420000E+01, 0.6878930000E+01 /
  data (epsData(81,i,22), i=1,3) / 0.3359190000E+02, -0.3506340000E+01, 0.7184660000E+01 /
  data (epsData(82,i,22), i=1,3) / 0.3466940000E+02, -0.4100960000E+01, 0.7028820000E+01 /
  data (epsData(83,i,22), i=1,3) / 0.3702050000E+02, -0.4645460000E+01, 0.6198240000E+01 /
  data (epsData(84,i,22), i=1,3) / 0.4141680000E+02, -0.4693580000E+01, 0.4614250000E+01 /
  data (epsData(85,i,22), i=1,3) / 0.4540160000E+02, -0.4260550000E+01, 0.3485730000E+01 /
  data (epsData(86,i,22), i=1,3) / 0.5075030000E+02, -0.3555260000E+01, 0.2521940000E+01 /
  data (epsData(87,i,22), i=1,3) / 0.5630650000E+02, -0.2905300000E+01, 0.1942640000E+01 /
  data (epsData(88,i,22), i=1,3) / 0.5885000000E+02, -0.2627880000E+01, 0.1833420000E+01 /
  data (epsData(89,i,22), i=1,3) / 0.6410010000E+02, -0.2359910000E+01, 0.1536380000E+01 /
  data (epsData(90,i,22), i=1,3) / 0.7150160000E+02, -0.1962520000E+01, 0.1221380000E+01 /
  data (epsData(91,i,22), i=1,3) / 0.8116600000E+02, -0.1604950000E+01, 0.9476070000E+00 /
  data (epsData(92,i,22), i=1,3) / 0.9223660000E+02, -0.1309600000E+01, 0.7283520000E+00 /
  data (epsData(93,i,22), i=1,3) / 0.1047630000E+03, -0.1068850000E+01, 0.5535240000E+00 /
  data (epsData(94,i,22), i=1,3) / 0.1192740000E+03, -0.8664500000E+00, 0.4151370000E+00 /
  data (epsData(95,i,22), i=1,3) / 0.1351220000E+03, -0.7025330000E+00, 0.3101760000E+00 /
  data (epsData(96,i,22), i=1,3) / 0.1517400000E+03, -0.5756250000E+00, 0.2363520000E+00 /
  data (epsData(97,i,22), i=1,3) / 0.1699910000E+03, -0.4708690000E+00, 0.1783290000E+00 /
  data (epsData(98,i,22), i=1,3) / 0.1932760000E+03, -0.3719480000E+00, 0.1273580000E+00 /
  data (epsData(99,i,22), i=1,3) / 0.2183060000E+03, -0.2946630000E+00, 0.9260990000E-01 /
  data (epsData(100,i,22), i=1,3) / 0.2464790000E+03, -0.2310580000E+00, 0.6631500000E-01 /
  data (epsData(101,i,22), i=1,3) / 0.2763960000E+03, -0.1809240000E+00, 0.4832940000E-01 /
  data (epsData(102,i,22), i=1,3) / 0.3102750000E+03, -0.1384230000E+00, 0.3480960000E-01 /
  data (epsData(103,i,22), i=1,3) / 0.3484840000E+03, -0.1015380000E+00, 0.2481590000E-01 /
  data (epsData(104,i,22), i=1,3) / 0.4112070000E+03, -0.5069160000E-01, 0.1528450000E-01 /
  data (epsData(105,i,22), i=1,3) / 0.4273160000E+03, -0.3484570000E-01, 0.1368140000E-01 /
  data (epsData(106,i,22), i=1,3) / 0.4351470000E+03, -0.2487960000E-01, 0.1307240000E-01 /
  data (epsData(107,i,22), i=1,3) / 0.4401740000E+03, -0.1683400000E-01, 0.1267400000E-01 /
  data (epsData(108,i,22), i=1,3) / 0.4437850000E+03, -0.9631770000E-02, 0.1251160000E-01 /
  data (epsData(109,i,22), i=1,3) / 0.4459590000E+03, -0.4266750000E-02, 0.1240050000E-01 /
  data (epsData(110,i,22), i=1,3) / 0.4474120000E+03, -0.2371650000E-04, 0.1234760000E-01 /
  data (epsData(111,i,22), i=1,3) / 0.4488680000E+03, 0.5010640000E-02, 0.1232520000E-01 /
  data (epsData(112,i,22), i=1,3) / 0.4495970000E+03, 0.7947260000E-02, 0.1233310000E-01 /
  data (epsData(113,i,22), i=1,3) / 0.4503270000E+03, 0.1127630000E-01, 0.1236170000E-01 /
  data (epsData(114,i,22), i=1,3) / 0.4510570000E+03, 0.1514710000E-01, 0.1242540000E-01 /
  data (epsData(115,i,22), i=1,3) / 0.4517880000E+03, 0.1983420000E-01, 0.1255500000E-01 /
  data (epsData(116,i,22), i=1,3) / 0.4525200000E+03, 0.2597400000E-01, 0.1283600000E-01 /
  data (epsData(117,i,22), i=1,3) / 0.4532520000E+03, 0.3512920000E-01, 0.1385560000E-01 /
  data (epsData(118,i,22), i=1,3) / 0.4538280000E+03, 0.4397220000E-01, 0.2109730000E-01 /
  data (epsData(119,i,22), i=1,3) / 0.4540130000E+03, 0.4484660000E-01, 0.2586410000E-01 /
  data (epsData(120,i,22), i=1,3) / 0.4553730000E+03, 0.4563140000E-01, 0.4160610000E-01 /
  data (epsData(121,i,22), i=1,3) / 0.4574090000E+03, 0.4321070000E-01, 0.5900060000E-01 /
  data (epsData(122,i,22), i=1,3) / 0.4580670000E+03, 0.4167870000E-01, 0.6165830000E-01 /
  data (epsData(123,i,22), i=1,3) / 0.4599080000E+03, 0.4251950000E-01, 0.7495560000E-01 /
  data (epsData(124,i,22), i=1,3) / 0.4605270000E+03, 0.4100440000E-01, 0.8280050000E-01 /
  data (epsData(125,i,22), i=1,3) / 0.4621130000E+03, 0.2971580000E-01, 0.9449060000E-01 /
  data (epsData(126,i,22), i=1,3) / 0.4641670000E+03, 0.1518240000E-01, 0.1013800000E+00 /
  data (epsData(127,i,22), i=1,3) / 0.4676500000E+03, -0.3893740000E-02, 0.1047520000E+00 /
  data (epsData(128,i,22), i=1,3) / 0.4704980000E+03, -0.1601230000E-01, 0.1015470000E+00 /
  data (epsData(129,i,22), i=1,3) / 0.4747750000E+03, -0.2557690000E-01, 0.9553210000E-01 /
  data (epsData(130,i,22), i=1,3) / 0.4824920000E+03, -0.3433830000E-01, 0.8955320000E-01 /
  data (epsData(131,i,22), i=1,3) / 0.4983720000E+03, -0.4760530000E-01, 0.8103930000E-01 /
  data (epsData(132,i,22), i=1,3) / 0.5122010000E+03, -0.5180570000E-01, 0.7224960000E-01 /
  data (epsData(133,i,22), i=1,3) / 0.5448960000E+03, -0.5380760000E-01, 0.5922220000E-01 /
  data (epsData(134,i,22), i=1,3) / 0.5571950000E+03, -0.5099540000E-01, 0.5557820000E-01 /
  data (epsData(135,i,22), i=1,3) / 0.5610100000E+03, -0.4935640000E-01, 0.5764940000E-01 /
  data (epsData(136,i,22), i=1,3) / 0.5652430000E+03, -0.5262810000E-01, 0.5806450000E-01 /
  data (epsData(137,i,22), i=1,3) / 0.5891130000E+03, -0.5456200000E-01, 0.5135080000E-01 /
  data (epsData(138,i,22), i=1,3) / 0.6344480000E+03, -0.5319590000E-01, 0.4065380000E-01 /
  data (epsData(139,i,22), i=1,3) / 0.7126820000E+03, -0.4713040000E-01, 0.2810690000E-01 /
  data (epsData(140,i,22), i=1,3) / 0.8060810000E+03, -0.3951830000E-01, 0.1876620000E-01 /
  data (epsData(141,i,22), i=1,3) / 0.9003720000E+03, -0.3298310000E-01, 0.1296260000E-01 /
  data (epsData(142,i,22), i=1,3) / 0.1003740000E+04, -0.2718960000E-01, 0.8915400000E-02 /
  data (epsData(143,i,22), i=1,3) / 0.1122570000E+04, -0.2205960000E-01, 0.6053860000E-02 /
  data (epsData(144,i,22), i=1,3) / 0.1246790000E+04, -0.1802260000E-01, 0.4187270000E-02 /
  data (epsData(145,i,22), i=1,3) / 0.1376010000E+04, -0.1484630000E-01, 0.2950560000E-02 /
  data (epsData(146,i,22), i=1,3) / 0.1518670000E+04, -0.1219440000E-01, 0.2067320000E-02 /
  data (epsData(147,i,22), i=1,3) / 0.1682680000E+04, -0.9910010000E-02, 0.1422280000E-02 /
  data (epsData(148,i,22), i=1,3) / 0.1871640000E+04, -0.7977180000E-02, 0.9691250000E-03 /
  data (epsData(149,i,22), i=1,3) / 0.2076740000E+04, -0.6448260000E-02, 0.6628030000E-03 /
  data (epsData(150,i,22), i=1,3) / 0.2309060000E+04, -0.5186440000E-02, 0.4485820000E-03 /
  data (epsData(151,i,22), i=1,3) / 0.2567830000E+04, -0.4165910000E-02, 0.3027870000E-03 /
  data (epsData(152,i,22), i=1,3) / 0.2856810000E+04, -0.3340110000E-02, 0.2036260000E-03 /
  data (epsData(153,i,22), i=1,3) / 0.3180310000E+04, -0.2671230000E-02, 0.1364320000E-03 /
  data (epsData(154,i,22), i=1,3) / 0.3544010000E+04, -0.2127250000E-02, 0.9088100000E-04 /
  data (epsData(155,i,22), i=1,3) / 0.3954000000E+04, -0.1682310000E-02, 0.6015290000E-04 /
  data (epsData(156,i,22), i=1,3) / 0.4440330000E+04, -0.1293730000E-02, 0.3878150000E-04 /
  data (epsData(157,i,22), i=1,3) / 0.4862740000E+04, -0.9876080000E-03, 0.2745910000E-04 /
  data (epsData(158,i,22), i=1,3) / 0.4932530000E+04, -0.8922140000E-03, 0.2600960000E-04 /
  data (epsData(159,i,22), i=1,3) / 0.4952510000E+04, -0.8268770000E-03, 0.2762830000E-04 /
  data (epsData(160,i,22), i=1,3) / 0.4960230000E+04, -0.7660700000E-03, 0.3088990000E-04 /
  data (epsData(161,i,22), i=1,3) / 0.4961780000E+04, -0.7422010000E-03, 0.3320370000E-04 /
  data (epsData(162,i,22), i=1,3) / 0.4963320000E+04, -0.7049470000E-03, 0.3845570000E-04 /
  data (epsData(163,i,22), i=1,3) / 0.4964870000E+04, -0.6376180000E-03, 0.5872630000E-04 /
  data (epsData(164,i,22), i=1,3) / 0.4966080000E+04, -0.5938240000E-03, 0.1841970000E-03 /
  data (epsData(165,i,22), i=1,3) / 0.4966390000E+04, -0.6180800000E-03, 0.2285340000E-03 /
  data (epsData(166,i,22), i=1,3) / 0.4966880000E+04, -0.6695680000E-03, 0.2547000000E-03 /
  data (epsData(167,i,22), i=1,3) / 0.4968180000E+04, -0.7493680000E-03, 0.2584280000E-03 /
  data (epsData(168,i,22), i=1,3) / 0.4970040000E+04, -0.7984440000E-03, 0.2308180000E-03 /
  data (epsData(169,i,22), i=1,3) / 0.4972860000E+04, -0.8186340000E-03, 0.1998050000E-03 /
  data (epsData(170,i,22), i=1,3) / 0.4979610000E+04, -0.8240410000E-03, 0.2037680000E-03 /
  data (epsData(171,i,22), i=1,3) / 0.5001710000E+04, -0.8817700000E-03, 0.2132580000E-03 /
  data (epsData(172,i,22), i=1,3) / 0.5082410000E+04, -0.9239460000E-03, 0.2010570000E-03 /
  data (epsData(173,i,22), i=1,3) / 0.5348920000E+04, -0.9032960000E-03, 0.1686000000E-03 /
  data (epsData(174,i,22), i=1,3) / 0.6338890000E+04, -0.6921030000E-03, 0.9158770000E-04 /
  data (epsData(175,i,22), i=1,3) / 0.7165730000E+04, -0.5526390000E-03, 0.5945360000E-04 /
  data (epsData(176,i,22), i=1,3) / 0.8031490000E+04, -0.4442390000E-03, 0.3913460000E-04 /
  data (epsData(177,i,22), i=1,3) / 0.8991160000E+04, -0.3563500000E-03, 0.2580880000E-04 /
  data (epsData(178,i,22), i=1,3) / 0.1004700000E+05, -0.2860460000E-03, 0.1700040000E-04 /
  data (epsData(179,i,22), i=1,3) / 0.1122280000E+05, -0.2293970000E-03, 0.1116990000E-04 /
  data (epsData(180,i,22), i=1,3) / 0.1254000000E+05, -0.1836550000E-03, 0.7309020000E-05 /
  data (epsData(181,i,22), i=1,3) / 0.1401450000E+05, -0.1468840000E-03, 0.4760190000E-05 /
  data (epsData(182,i,22), i=1,3) / 0.1566210000E+05, -0.1174320000E-03, 0.3089180000E-05 /
  data (epsData(183,i,22), i=1,3) / 0.1751330000E+05, -0.9376200000E-04, 0.1993710000E-05 /
  data (epsData(184,i,22), i=1,3) / 0.1958820000E+05, -0.7482240000E-04, 0.1281540000E-05 /
  data (epsData(185,i,22), i=1,3) / 0.2192310000E+05, -0.5963220000E-04, 0.8191400000E-06 /
  data (epsData(186,i,22), i=1,3) / 0.2454430000E+05, -0.4749950000E-04, 0.5213330000E-06 /
  data (epsData(187,i,22), i=1,3) / 0.2748650000E+05, -0.3781890000E-04, 0.3304110000E-06 /
  data (epsData(188,i,22), i=1,3) / 0.3078850000E+05, -0.3010140000E-04, 0.2084920000E-06 /
  data (epsData(189,i,22), i=1,3) / 0.3449350000E+05, -0.2395330000E-04, 0.1312890000E-06 /
  data (epsData(190,i,22), i=1,3) / 0.3865640000E+05, -0.1905190000E-04, 0.8224390000E-07 /
  data (epsData(191,i,22), i=1,3) / 0.4333190000E+05, -0.1514820000E-04, 0.5122400000E-07 /
  data (epsData(192,i,22), i=1,3) / 0.4858020000E+05, -0.1204230000E-04, 0.3193370000E-07 /
  data (epsData(193,i,22), i=1,3) / 0.5446820000E+05, -0.9572890000E-05, 0.1974960000E-07 /
  data (epsData(194,i,22), i=1,3) / 0.6108630000E+05, -0.7606520000E-05, 0.1227170000E-07 /
  data (epsData(195,i,22), i=1,3) / 0.6851330000E+05, -0.6043900000E-05, 0.7536720000E-08 /
  data (epsData(196,i,22), i=1,3) / 0.7684580000E+05, -0.4802320000E-05, 0.4642970000E-08 /
  data (epsData(197,i,22), i=1,3) / 0.8620550000E+05, -0.3814980000E-05, 0.2821480000E-08 /
  data (epsData(198,i,22), i=1,3) / 0.9668410000E+05, -0.3032470000E-05, 0.1732630000E-08 /
  data (epsData(199,i,22), i=1,3) / 0.1000000000E+06, -0.2837960000E-05, 0.1511740000E-08 /
  data (epsData(1,i,23), i=1,3) / 0.2506580000E-02, -0.1155150000E+04, 0.3112360000E+04 /
  data (epsData(2,i,23), i=1,3) / 0.4693440000E-02, -0.9296810000E+03, 0.1044350000E+04 /
  data (epsData(3,i,23), i=1,3) / 0.7560590000E-02, -0.3798430000E+03, 0.4815980000E+03 /
  data (epsData(4,i,23), i=1,3) / 0.1110800000E-01, -0.1405470000E+03, 0.2684760000E+03 /
  data (epsData(5,i,23), i=1,3) / 0.1533570000E-01, -0.2811050000E+02, 0.1692180000E+03 /
  data (epsData(6,i,23), i=1,3) / 0.2024370000E-01, 0.3003340000E+02, 0.1160090000E+03 /
  data (epsData(7,i,23), i=1,3) / 0.2583200000E-01, 0.6257520000E+02, 0.8445960000E+02 /
  data (epsData(8,i,23), i=1,3) / 0.3210060000E-01, 0.8198110000E+02, 0.6429280000E+02 /
  data (epsData(9,i,23), i=1,3) / 0.3904940000E-01, 0.9416620000E+02, 0.5063730000E+02 /
  data (epsData(10,i,23), i=1,3) / 0.4667860000E-01, 0.1021530000E+03, 0.4096240000E+02 /
  data (epsData(11,i,23), i=1,3) / 0.5498800000E-01, 0.1075840000E+03, 0.3385400000E+02 /
  data (epsData(12,i,23), i=1,3) / 0.6397770000E-01, 0.1114010000E+03, 0.2847400000E+02 /
  data (epsData(13,i,23), i=1,3) / 0.8399800000E-01, 0.1162600000E+03, 0.2099560000E+02 /
  data (epsData(14,i,23), i=1,3) / 0.1191310000E+00, 0.1201590000E+03, 0.1496920000E+02 /
  data (epsData(15,i,23), i=1,3) / 0.2077630000E+00, 0.1220500000E+03, 0.9281070000E+01 /
  data (epsData(16,i,23), i=1,3) / 0.4585000000E+00, 0.1242830000E+03, 0.4714420000E+01 /
  data (epsData(17,i,23), i=1,3) / 0.9096770000E+00, 0.1315290000E+03, 0.3373340000E+01 /
  data (epsData(18,i,23), i=1,3) / 0.1295500000E+01, 0.1455460000E+03, 0.3609690000E+01 /
  data (epsData(19,i,23), i=1,3) / 0.1513920000E+01, 0.1604070000E+03, 0.5150800000E+01 /
  data (epsData(20,i,23), i=1,3) / 0.1606050000E+01, 0.1716100000E+03, 0.6053960000E+01 /
  data (epsData(21,i,23), i=1,3) / 0.1798470000E+01, 0.1988190000E+03, 0.1968970000E+02 /
  data (epsData(22,i,23), i=1,3) / 0.1848280000E+01, 0.2119010000E+03, 0.2228660000E+02 /
  data (epsData(23,i,23), i=1,3) / 0.1898770000E+01, 0.2379180000E+03, 0.2461700000E+02 /
  data (epsData(24,i,23), i=1,3) / 0.1949930000E+01, 0.2561530000E+03, 0.7473000000E+02 /
  data (epsData(25,i,23), i=1,3) / 0.2003060000E+01, 0.2423860000E+03, 0.1210420000E+03 /
  data (epsData(26,i,23), i=1,3) / 0.2164630000E+01, 0.1758280000E+03, 0.1698840000E+03 /
  data (epsData(27,i,23), i=1,3) / 0.2195920000E+01, 0.1571150000E+03, 0.1775430000E+03 /
  data (epsData(28,i,23), i=1,3) / 0.2229940000E+01, 0.1363950000E+03, 0.1722640000E+03 /
  data (epsData(29,i,23), i=1,3) / 0.2306130000E+01, 0.1108940000E+03, 0.1575660000E+03 /
  data (epsData(30,i,23), i=1,3) / 0.2416670000E+01, 0.9051040000E+02, 0.1388550000E+03 /
  data (epsData(31,i,23), i=1,3) / 0.2544230000E+01, 0.7867820000E+02, 0.1211230000E+03 /
  data (epsData(32,i,23), i=1,3) / 0.2719740000E+01, 0.7096580000E+02, 0.1025300000E+03 /
  data (epsData(33,i,23), i=1,3) / 0.2919740000E+01, 0.7002160000E+02, 0.8750670000E+02 /
  data (epsData(34,i,23), i=1,3) / 0.3184030000E+01, 0.7313880000E+02, 0.7386290000E+02 /
  data (epsData(35,i,23), i=1,3) / 0.3481660000E+01, 0.8242270000E+02, 0.6285950000E+02 /
  data (epsData(36,i,23), i=1,3) / 0.3714660000E+01, 0.9758720000E+02, 0.5605620000E+02 /
  data (epsData(37,i,23), i=1,3) / 0.3862620000E+01, 0.1144450000E+03, 0.5506740000E+02 /
  data (epsData(38,i,23), i=1,3) / 0.3964660000E+01, 0.1333490000E+03, 0.5481310000E+02 /
  data (epsData(39,i,23), i=1,3) / 0.4016700000E+01, 0.1521800000E+03, 0.5466900000E+02 /
  data (epsData(40,i,23), i=1,3) / 0.4069420000E+01, 0.1730780000E+03, 0.7518870000E+02 /
  data (epsData(41,i,23), i=1,3) / 0.4122830000E+01, 0.1786620000E+03, 0.1123280000E+03 /
  data (epsData(42,i,23), i=1,3) / 0.4176910000E+01, 0.1698410000E+03, 0.1363970000E+03 /
  data (epsData(43,i,23), i=1,3) / 0.4287120000E+01, 0.1440470000E+03, 0.1702710000E+03 /
  data (epsData(44,i,23), i=1,3) / 0.4343240000E+01, 0.1235600000E+03, 0.1854950000E+03 /
  data (epsData(45,i,23), i=1,3) / 0.4400040000E+01, 0.1033120000E+03, 0.1830220000E+03 /
  data (epsData(46,i,23), i=1,3) / 0.4515690000E+01, 0.7900770000E+02, 0.1781870000E+03 /
  data (epsData(47,i,23), i=1,3) / 0.4755150000E+01, 0.4878110000E+02, 0.1694910000E+03 /
  data (epsData(48,i,23), i=1,3) / 0.5069780000E+01, 0.2033950000E+02, 0.1585200000E+03 /
  data (epsData(49,i,23), i=1,3) / 0.5333730000E+01, 0.8844670000E+00, 0.1471310000E+03 /
  data (epsData(50,i,23), i=1,3) / 0.5608570000E+01, -0.1441050000E+02, 0.1326120000E+03 /
  data (epsData(51,i,23), i=1,3) / 0.5967420000E+01, -0.2632790000E+02, 0.1119940000E+03 /
  data (epsData(52,i,23), i=1,3) / 0.6343270000E+01, -0.3077400000E+02, 0.9266450000E+02 /
  data (epsData(53,i,23), i=1,3) / 0.6736140000E+01, -0.3037690000E+02, 0.7691260000E+02 /
  data (epsData(54,i,23), i=1,3) / 0.7230020000E+01, -0.2645860000E+02, 0.6191270000E+02 /
  data (epsData(55,i,23), i=1,3) / 0.7572890000E+01, -0.2031830000E+02, 0.5320640000E+02 /
  data (epsData(56,i,23), i=1,3) / 0.7748400000E+01, -0.1404000000E+02, 0.5009420000E+02 /
  data (epsData(57,i,23), i=1,3) / 0.7837180000E+01, -0.7788290000E+01, 0.4861650000E+02 /
  data (epsData(58,i,23), i=1,3) / 0.7926630000E+01, -0.1441470000E+01, 0.5537640000E+02 /
  data (epsData(59,i,23), i=1,3) / 0.8016770000E+01, -0.2713040000E+01, 0.6871110000E+02 /
  data (epsData(60,i,23), i=1,3) / 0.8107590000E+01, -0.9665210000E+01, 0.7262190000E+02 /
  data (epsData(61,i,23), i=1,3) / 0.8199090000E+01, -0.1779810000E+02, 0.7528380000E+02 /
  data (epsData(62,i,23), i=1,3) / 0.8291270000E+01, -0.2478750000E+02, 0.7031960000E+02 /
  data (epsData(63,i,23), i=1,3) / 0.8384120000E+01, -0.2829500000E+02, 0.6571390000E+02 /
  data (epsData(64,i,23), i=1,3) / 0.8477660000E+01, -0.2990740000E+02, 0.6126020000E+02 /
  data (epsData(65,i,23), i=1,3) / 0.8762360000E+01, -0.3094140000E+02, 0.5203390000E+02 /
  data (epsData(66,i,23), i=1,3) / 0.9151480000E+01, -0.2928820000E+02, 0.4368160000E+02 /
  data (epsData(67,i,23), i=1,3) / 0.9755580000E+01, -0.2568670000E+02, 0.3529980000E+02 /
  data (epsData(68,i,23), i=1,3) / 0.1049130000E+02, -0.2125090000E+02, 0.2908030000E+02 /
  data (epsData(69,i,23), i=1,3) / 0.1126040000E+02, -0.1758930000E+02, 0.2526500000E+02 /
  data (epsData(70,i,23), i=1,3) / 0.1206280000E+02, -0.1489040000E+02, 0.2285060000E+02 /
  data (epsData(71,i,23), i=1,3) / 0.1351580000E+02, -0.1200140000E+02, 0.1958350000E+02 /
  data (epsData(72,i,23), i=1,3) / 0.1533490000E+02, -0.9955550000E+01, 0.1677990000E+02 /
  data (epsData(73,i,23), i=1,3) / 0.1801690000E+02, -0.8401160000E+01, 0.1363500000E+02 /
  data (epsData(74,i,23), i=1,3) / 0.2208500000E+02, -0.6916170000E+01, 0.9980680000E+01 /
  data (epsData(75,i,23), i=1,3) / 0.2588650000E+02, -0.5677230000E+01, 0.7677590000E+01 /
  data (epsData(76,i,23), i=1,3) / 0.3001710000E+02, -0.4585810000E+01, 0.6059210000E+01 /
  data (epsData(77,i,23), i=1,3) / 0.3489890000E+02, -0.3431300000E+01, 0.4802940000E+01 /
  data (epsData(78,i,23), i=1,3) / 0.3639650000E+02, -0.2860000000E+01, 0.4587650000E+01 /
  data (epsData(79,i,23), i=1,3) / 0.3704860000E+02, -0.2458280000E+01, 0.4772080000E+01 /
  data (epsData(80,i,23), i=1,3) / 0.3746670000E+02, -0.2434480000E+01, 0.5137340000E+01 /
  data (epsData(81,i,23), i=1,3) / 0.3826670000E+02, -0.2787980000E+01, 0.5360920000E+01 /
  data (epsData(82,i,23), i=1,3) / 0.3965750000E+02, -0.3149200000E+01, 0.5166800000E+01 /
  data (epsData(83,i,23), i=1,3) / 0.4201680000E+02, -0.3446910000E+01, 0.4692140000E+01 /
  data (epsData(84,i,23), i=1,3) / 0.4623720000E+02, -0.3543920000E+01, 0.3691240000E+01 /
  data (epsData(85,i,23), i=1,3) / 0.5053490000E+02, -0.3248690000E+01, 0.2836280000E+01 /
  data (epsData(86,i,23), i=1,3) / 0.5646700000E+02, -0.2722900000E+01, 0.2095810000E+01 /
  data (epsData(87,i,23), i=1,3) / 0.6329870000E+02, -0.2201430000E+01, 0.1605140000E+01 /
  data (epsData(88,i,23), i=1,3) / 0.6645000000E+02, -0.1980070000E+01, 0.1512050000E+01 /
  data (epsData(89,i,23), i=1,3) / 0.7034120000E+02, -0.1875630000E+01, 0.1361630000E+01 /
  data (epsData(90,i,23), i=1,3) / 0.7870870000E+02, -0.1572680000E+01, 0.1080350000E+01 /
  data (epsData(91,i,23), i=1,3) / 0.9036710000E+02, -0.1278050000E+01, 0.8223220000E+00 /
  data (epsData(92,i,23), i=1,3) / 0.1031170000E+03, -0.1049170000E+01, 0.6276290000E+00 /
  data (epsData(93,i,23), i=1,3) / 0.1172430000E+03, -0.8600260000E+00, 0.4762200000E+00 /
  data (epsData(94,i,23), i=1,3) / 0.1333550000E+03, -0.7011270000E+00, 0.3565910000E+00 /
  data (epsData(95,i,23), i=1,3) / 0.1499810000E+03, -0.5790820000E+00, 0.2726960000E+00 /
  data (epsData(96,i,23), i=1,3) / 0.1691940000E+03, -0.4725750000E+00, 0.2033060000E+00 /
  data (epsData(97,i,23), i=1,3) / 0.1932840000E+03, -0.3742180000E+00, 0.1447280000E+00 /
  data (epsData(98,i,23), i=1,3) / 0.2185370000E+03, -0.2991330000E+00, 0.1054600000E+00 /
  data (epsData(99,i,23), i=1,3) / 0.2465740000E+03, -0.2377330000E+00, 0.7595130000E-01 /
  data (epsData(100,i,23), i=1,3) / 0.2764430000E+03, -0.1890990000E+00, 0.5561990000E-01 /
  data (epsData(101,i,23), i=1,3) / 0.3097510000E+03, -0.1486890000E+00, 0.4036900000E-01 /
  data (epsData(102,i,23), i=1,3) / 0.3467420000E+03, -0.1147610000E+00, 0.2916200000E-01 /
  data (epsData(103,i,23), i=1,3) / 0.3883310000E+03, -0.8545600000E-01, 0.2109140000E-01 /
  data (epsData(104,i,23), i=1,3) / 0.4398900000E+03, -0.5584840000E-01, 0.1466360000E-01 /
  data (epsData(105,i,23), i=1,3) / 0.4719190000E+03, -0.3688650000E-01, 0.1182630000E-01 /
  data (epsData(106,i,23), i=1,3) / 0.4869130000E+03, -0.2551250000E-01, 0.1071730000E-01 /
  data (epsData(107,i,23), i=1,3) / 0.4952760000E+03, -0.1703200000E-01, 0.1023650000E-01 /
  data (epsData(108,i,23), i=1,3) / 0.4998720000E+03, -0.1094490000E-01, 0.1004670000E-01 /
  data (epsData(109,i,23), i=1,3) / 0.5029500000E+03, -0.5948900000E-02, 0.9936150000E-02 /
  data (epsData(110,i,23), i=1,3) / 0.5052650000E+03, -0.1279310000E-02, 0.9879940000E-02 /
  data (epsData(111,i,23), i=1,3) / 0.5068120000E+03, 0.2519320000E-02, 0.9869000000E-02 /
  data (epsData(112,i,23), i=1,3) / 0.5075870000E+03, 0.4721650000E-02, 0.9877730000E-02 /
  data (epsData(113,i,23), i=1,3) / 0.5083620000E+03, 0.7204270000E-02, 0.9901820000E-02 /
  data (epsData(114,i,23), i=1,3) / 0.5091380000E+03, 0.1006810000E-01, 0.9951380000E-02 /
  data (epsData(115,i,23), i=1,3) / 0.5099140000E+03, 0.1349630000E-01, 0.1004490000E-01 /
  data (epsData(116,i,23), i=1,3) / 0.5106920000E+03, 0.1788980000E-01, 0.1024560000E-01 /
  data (epsData(117,i,23), i=1,3) / 0.5114700000E+03, 0.2422220000E-01, 0.1086350000E-01 /
  data (epsData(118,i,23), i=1,3) / 0.5121340000E+03, 0.3103280000E-01, 0.1550990000E-01 /
  data (epsData(119,i,23), i=1,3) / 0.5123300000E+03, 0.3198990000E-01, 0.1844450000E-01 /
  data (epsData(120,i,23), i=1,3) / 0.5134070000E+03, 0.3389580000E-01, 0.2775030000E-01 /
  data (epsData(121,i,23), i=1,3) / 0.5151050000E+03, 0.3483560000E-01, 0.3970850000E-01 /
  data (epsData(122,i,23), i=1,3) / 0.5159220000E+03, 0.3287830000E-01, 0.4473180000E-01 /
  data (epsData(123,i,23), i=1,3) / 0.5170800000E+03, 0.3243220000E-01, 0.5058810000E-01 /
  data (epsData(124,i,23), i=1,3) / 0.5191530000E+03, 0.2800740000E-01, 0.6029790000E-01 /
  data (epsData(125,i,23), i=1,3) / 0.5200480000E+03, 0.2686330000E-01, 0.6536990000E-01 /
  data (epsData(126,i,23), i=1,3) / 0.5220320000E+03, 0.1852740000E-01, 0.7316630000E-01 /
  data (epsData(127,i,23), i=1,3) / 0.5240670000E+03, 0.1030110000E-01, 0.7740900000E-01 /
  data (epsData(128,i,23), i=1,3) / 0.5268530000E+03, -0.6202370000E-04, 0.8096320000E-01 /
  data (epsData(129,i,23), i=1,3) / 0.5296300000E+03, -0.9247070000E-02, 0.8003430000E-01 /
  data (epsData(130,i,23), i=1,3) / 0.5345160000E+03, -0.1829210000E-01, 0.7613610000E-01 /
  data (epsData(131,i,23), i=1,3) / 0.5433270000E+03, -0.2681760000E-01, 0.7201530000E-01 /
  data (epsData(132,i,23), i=1,3) / 0.5566170000E+03, -0.3635360000E-01, 0.6636760000E-01 /
  data (epsData(133,i,23), i=1,3) / 0.5712710000E+03, -0.4053410000E-01, 0.5942230000E-01 /
  data (epsData(134,i,23), i=1,3) / 0.6058600000E+03, -0.4326430000E-01, 0.4869820000E-01 /
  data (epsData(135,i,23), i=1,3) / 0.6219470000E+03, -0.4108900000E-01, 0.4510290000E-01 /
  data (epsData(136,i,23), i=1,3) / 0.6264700000E+03, -0.3936570000E-01, 0.4659070000E-01 /
  data (epsData(137,i,23), i=1,3) / 0.6303090000E+03, -0.4179150000E-01, 0.4757860000E-01 /
  data (epsData(138,i,23), i=1,3) / 0.6477890000E+03, -0.4381690000E-01, 0.4381010000E-01 /
  data (epsData(139,i,23), i=1,3) / 0.6872740000E+03, -0.4394130000E-01, 0.3625150000E-01 /
  data (epsData(140,i,23), i=1,3) / 0.7634240000E+03, -0.4022790000E-01, 0.2597530000E-01 /
  data (epsData(141,i,23), i=1,3) / 0.8649990000E+03, -0.3409250000E-01, 0.1727140000E-01 /
  data (epsData(142,i,23), i=1,3) / 0.9682130000E+03, -0.2853180000E-01, 0.1184390000E-01 /
  data (epsData(143,i,23), i=1,3) / 0.1082040000E+04, -0.2349000000E-01, 0.8112450000E-02 /
  data (epsData(144,i,23), i=1,3) / 0.1202440000E+04, -0.1935250000E-01, 0.5628700000E-02 /
  data (epsData(145,i,23), i=1,3) / 0.1327150000E+04, -0.1603430000E-01, 0.3976190000E-02 /
  data (epsData(146,i,23), i=1,3) / 0.1466210000E+04, -0.1320120000E-01, 0.2786850000E-02 /
  data (epsData(147,i,23), i=1,3) / 0.1628340000E+04, -0.1070810000E-01, 0.1908170000E-02 /
  data (epsData(148,i,23), i=1,3) / 0.1809830000E+04, -0.8656370000E-02, 0.1306500000E-02 /
  data (epsData(149,i,23), i=1,3) / 0.2003700000E+04, -0.7041520000E-02, 0.9011110000E-03 /
  data (epsData(150,i,23), i=1,3) / 0.2219610000E+04, -0.5711640000E-02, 0.6210450000E-03 /
  data (epsData(151,i,23), i=1,3) / 0.2465450000E+04, -0.4604160000E-02, 0.4218760000E-03 /
  data (epsData(152,i,23), i=1,3) / 0.2740780000E+04, -0.3701710000E-02, 0.2851160000E-03 /
  data (epsData(153,i,23), i=1,3) / 0.3048000000E+04, -0.2971320000E-02, 0.1920220000E-03 /
  data (epsData(154,i,23), i=1,3) / 0.3391550000E+04, -0.2379680000E-02, 0.1288130000E-03 /
  data (epsData(155,i,23), i=1,3) / 0.3778420000E+04, -0.1897650000E-02, 0.8587190000E-04 /
  data (epsData(156,i,23), i=1,3) / 0.4213720000E+04, -0.1504860000E-02, 0.5694290000E-04 /
  data (epsData(157,i,23), i=1,3) / 0.4714180000E+04, -0.1174700000E-02, 0.3722960000E-04 /
  data (epsData(158,i,23), i=1,3) / 0.5351930000E+04, -0.8254660000E-03, 0.2298340000E-04 /
  data (epsData(159,i,23), i=1,3) / 0.5428220000E+04, -0.7473350000E-03, 0.2177370000E-04 /
  data (epsData(160,i,23), i=1,3) / 0.5450320000E+04, -0.6932390000E-03, 0.2307670000E-04 /
  data (epsData(161,i,23), i=1,3) / 0.5458240000E+04, -0.6482600000E-03, 0.2539150000E-04 /
  data (epsData(162,i,23), i=1,3) / 0.5461410000E+04, -0.6082210000E-03, 0.2973080000E-04 /
  data (epsData(163,i,23), i=1,3) / 0.5463000000E+04, -0.5687700000E-03, 0.3756790000E-04 /
  data (epsData(164,i,23), i=1,3) / 0.5464590000E+04, -0.5064670000E-03, 0.8844530000E-04 /
  data (epsData(165,i,23), i=1,3) / 0.5465110000E+04, -0.4976600000E-03, 0.1559000000E-03 /
  data (epsData(166,i,23), i=1,3) / 0.5465570000E+04, -0.5332670000E-03, 0.1989190000E-03 /
  data (epsData(167,i,23), i=1,3) / 0.5466390000E+04, -0.5908820000E-03, 0.2172650000E-03 /
  data (epsData(168,i,23), i=1,3) / 0.5467570000E+04, -0.6426790000E-03, 0.2051810000E-03 /
  data (epsData(169,i,23), i=1,3) / 0.5470490000E+04, -0.6783110000E-03, 0.1719750000E-03 /
  data (epsData(170,i,23), i=1,3) / 0.5476640000E+04, -0.6829900000E-03, 0.1641190000E-03 /
  data (epsData(171,i,23), i=1,3) / 0.5496640000E+04, -0.7273000000E-03, 0.1757300000E-03 /
  data (epsData(172,i,23), i=1,3) / 0.5565660000E+04, -0.7675890000E-03, 0.1678460000E-03 /
  data (epsData(173,i,23), i=1,3) / 0.5796360000E+04, -0.7654780000E-03, 0.1461800000E-03 /
  data (epsData(174,i,23), i=1,3) / 0.6755790000E+04, -0.6104890000E-03, 0.8437820000E-04 /
  data (epsData(175,i,23), i=1,3) / 0.7647470000E+04, -0.4876960000E-03, 0.5451140000E-04 /
  data (epsData(176,i,23), i=1,3) / 0.8573520000E+04, -0.3924430000E-03, 0.3592390000E-04 /
  data (epsData(177,i,23), i=1,3) / 0.9596140000E+04, -0.3151540000E-03, 0.2368410000E-04 /
  data (epsData(178,i,23), i=1,3) / 0.1072370000E+05, -0.2530720000E-03, 0.1561290000E-04 /
  data (epsData(179,i,23), i=1,3) / 0.1198110000E+05, -0.2029320000E-03, 0.1026070000E-04 /
  data (epsData(180,i,23), i=1,3) / 0.1338630000E+05, -0.1625260000E-03, 0.6716380000E-05 /
  data (epsData(181,i,23), i=1,3) / 0.1495740000E+05, -0.1300450000E-03, 0.4377090000E-05 /
  data (epsData(182,i,23), i=1,3) / 0.1672130000E+05, -0.1039100000E-03, 0.2837110000E-05 /
  data (epsData(183,i,23), i=1,3) / 0.1869890000E+05, -0.8295700000E-04, 0.1832340000E-05 /
  data (epsData(184,i,23), i=1,3) / 0.2091500000E+05, -0.6619620000E-04, 0.1177810000E-05 /
  data (epsData(185,i,23), i=1,3) / 0.2340650000E+05, -0.5276430000E-04, 0.7532300000E-06 /
  data (epsData(186,i,23), i=1,3) / 0.2620080000E+05, -0.4204150000E-04, 0.4798140000E-06 /
  data (epsData(187,i,23), i=1,3) / 0.2933960000E+05, -0.3347720000E-04, 0.3041680000E-06 /
  data (epsData(188,i,23), i=1,3) / 0.3286420000E+05, -0.2664500000E-04, 0.1920280000E-06 /
  data (epsData(189,i,23), i=1,3) / 0.3682040000E+05, -0.2120070000E-04, 0.1207930000E-06 /
  data (epsData(190,i,23), i=1,3) / 0.4125950000E+05, -0.1686590000E-04, 0.7575050000E-07 /
  data (epsData(191,i,23), i=1,3) / 0.4624520000E+05, -0.1341260000E-04, 0.4720740000E-07 /
  data (epsData(192,i,23), i=1,3) / 0.5184150000E+05, -0.1066420000E-04, 0.2949260000E-07 /
  data (epsData(193,i,23), i=1,3) / 0.5812710000E+05, -0.8476610000E-05, 0.1817470000E-07 /
  data (epsData(194,i,23), i=1,3) / 0.6518710000E+05, -0.6736030000E-05, 0.1127380000E-07 /
  data (epsData(195,i,23), i=1,3) / 0.7310960000E+05, -0.5352520000E-05, 0.6945580000E-08 /
  data (epsData(196,i,23), i=1,3) / 0.8200400000E+05, -0.4252740000E-05, 0.4273870000E-08 /
  data (epsData(197,i,23), i=1,3) / 0.9197820000E+05, -0.3379520000E-05, 0.2592630000E-08 /
  data (epsData(198,i,23), i=1,3) / 0.1000000000E+06, -0.2862940000E-05, 0.1832110000E-08 /
  data (epsData(1,i,24), i=1,3) / 0.2506580000E-02, -0.1056850000E+04, 0.2956940000E+04 /
  data (epsData(2,i,24), i=1,3) / 0.4693440000E-02, -0.8619770000E+03, 0.1001830000E+04 /
  data (epsData(3,i,24), i=1,3) / 0.7560590000E-02, -0.3504460000E+03, 0.4661270000E+03 /
  data (epsData(4,i,24), i=1,3) / 0.1110800000E-01, -0.1274010000E+03, 0.2618330000E+03 /
  data (epsData(5,i,24), i=1,3) / 0.1533570000E-01, -0.2251970000E+02, 0.1660610000E+03 /
  data (epsData(6,i,24), i=1,3) / 0.2024370000E-01, 0.3174140000E+02, 0.1144160000E+03 /
  data (epsData(7,i,24), i=1,3) / 0.2583200000E-01, 0.6211940000E+02, 0.8363350000E+02 /
  data (epsData(8,i,24), i=1,3) / 0.3210060000E-01, 0.8023930000E+02, 0.6386870000E+02 /
  data (epsData(9,i,24), i=1,3) / 0.3904940000E-01, 0.9161940000E+02, 0.5043380000E+02 /
  data (epsData(10,i,24), i=1,3) / 0.4667860000E-01, 0.9908030000E+02, 0.4088380000E+02 /
  data (epsData(11,i,24), i=1,3) / 0.5498800000E-01, 0.1041550000E+03, 0.3384750000E+02 /
  data (epsData(12,i,24), i=1,3) / 0.6397770000E-01, 0.1077240000E+03, 0.2850920000E+02 /
  data (epsData(13,i,24), i=1,3) / 0.8399800000E-01, 0.1122750000E+03, 0.2106770000E+02 /
  data (epsData(14,i,24), i=1,3) / 0.1191310000E+00, 0.1159420000E+03, 0.1507530000E+02 /
  data (epsData(15,i,24), i=1,3) / 0.2077630000E+00, 0.1176440000E+03, 0.9231730000E+01 /
  data (epsData(16,i,24), i=1,3) / 0.4838160000E+00, 0.1201590000E+03, 0.4584120000E+01 /
  data (epsData(17,i,24), i=1,3) / 0.9096770000E+00, 0.1272180000E+03, 0.3413970000E+01 /
  data (epsData(18,i,24), i=1,3) / 0.1295500000E+01, 0.1414150000E+03, 0.3709600000E+01 /
  data (epsData(19,i,24), i=1,3) / 0.1513920000E+01, 0.1565460000E+03, 0.5373420000E+01 /
  data (epsData(20,i,24), i=1,3) / 0.1606050000E+01, 0.1680840000E+03, 0.6357410000E+01 /
  data (epsData(21,i,24), i=1,3) / 0.1798470000E+01, 0.1954540000E+03, 0.2108610000E+02 /
  data (epsData(22,i,24), i=1,3) / 0.1848280000E+01, 0.2085250000E+03, 0.2389620000E+02 /
  data (epsData(23,i,24), i=1,3) / 0.1898770000E+01, 0.2346540000E+03, 0.2641810000E+02 /
  data (epsData(24,i,24), i=1,3) / 0.1949930000E+01, 0.2528330000E+03, 0.7732590000E+02 /
  data (epsData(25,i,24), i=1,3) / 0.2003060000E+01, 0.2384130000E+03, 0.1243350000E+03 /
  data (epsData(26,i,24), i=1,3) / 0.2136060000E+01, 0.1836530000E+03, 0.1654710000E+03 /
  data (epsData(27,i,24), i=1,3) / 0.2195920000E+01, 0.1518210000E+03, 0.1804850000E+03 /
  data (epsData(28,i,24), i=1,3) / 0.2229940000E+01, 0.1310430000E+03, 0.1752690000E+03 /
  data (epsData(29,i,24), i=1,3) / 0.2306130000E+01, 0.1052730000E+03, 0.1607090000E+03 /
  data (epsData(30,i,24), i=1,3) / 0.2416670000E+01, 0.8446910000E+02, 0.1421710000E+03 /
  data (epsData(31,i,24), i=1,3) / 0.2544230000E+01, 0.7227810000E+02, 0.1248130000E+03 /
  data (epsData(32,i,24), i=1,3) / 0.2719740000E+01, 0.6331240000E+02, 0.1070630000E+03 /
  data (epsData(33,i,24), i=1,3) / 0.2919740000E+01, 0.6009140000E+02, 0.9233630000E+02 /
  data (epsData(34,i,24), i=1,3) / 0.3224510000E+01, 0.5986100000E+02, 0.7603250000E+02 /
  data (epsData(35,i,24), i=1,3) / 0.3526900000E+01, 0.6619020000E+02, 0.6283370000E+02 /
  data (epsData(36,i,24), i=1,3) / 0.3763300000E+01, 0.7842320000E+02, 0.5440240000E+02 /
  data (epsData(37,i,24), i=1,3) / 0.3913300000E+01, 0.9255330000E+02, 0.5119550000E+02 /
  data (epsData(38,i,24), i=1,3) / 0.4016700000E+01, 0.1060190000E+03, 0.5049010000E+02 /
  data (epsData(39,i,24), i=1,3) / 0.4069420000E+01, 0.1162770000E+03, 0.5020170000E+02 /
  data (epsData(40,i,24), i=1,3) / 0.4122830000E+01, 0.1334430000E+03, 0.4990770000E+02 /
  data (epsData(41,i,24), i=1,3) / 0.4176910000E+01, 0.1528170000E+03, 0.6755590000E+02 /
  data (epsData(42,i,24), i=1,3) / 0.4231670000E+01, 0.1586810000E+03, 0.1011010000E+03 /
  data (epsData(43,i,24), i=1,3) / 0.4287120000E+01, 0.1514870000E+03, 0.1228210000E+03 /
  data (epsData(44,i,24), i=1,3) / 0.4400040000E+01, 0.1286670000E+03, 0.1546290000E+03 /
  data (epsData(45,i,24), i=1,3) / 0.4457530000E+01, 0.1106400000E+03, 0.1671680000E+03 /
  data (epsData(46,i,24), i=1,3) / 0.4515690000E+01, 0.9338690000E+02, 0.1657060000E+03 /
  data (epsData(47,i,24), i=1,3) / 0.4694260000E+01, 0.6409440000E+02, 0.1615370000E+03 /
  data (epsData(48,i,24), i=1,3) / 0.5069780000E+01, 0.2615170000E+02, 0.1530680000E+03 /
  data (epsData(49,i,24), i=1,3) / 0.5401420000E+01, 0.4210750000E+00, 0.1399610000E+03 /
  data (epsData(50,i,24), i=1,3) / 0.5750070000E+01, -0.1751380000E+02, 0.1209940000E+03 /
  data (epsData(51,i,24), i=1,3) / 0.6041230000E+01, -0.2535470000E+02, 0.1049600000E+03 /
  data (epsData(52,i,24), i=1,3) / 0.6420490000E+01, -0.2914960000E+02, 0.8748840000E+02 /
  data (epsData(53,i,24), i=1,3) / 0.6898040000E+01, -0.2956090000E+02, 0.7095400000E+02 /
  data (epsData(54,i,24), i=1,3) / 0.7400090000E+01, -0.2713940000E+02, 0.5705440000E+02 /
  data (epsData(55,i,24), i=1,3) / 0.7748400000E+01, -0.2268840000E+02, 0.4817980000E+02 /
  data (epsData(56,i,24), i=1,3) / 0.7926630000E+01, -0.1821220000E+02, 0.4422240000E+02 /
  data (epsData(57,i,24), i=1,3) / 0.8016770000E+01, -0.1494110000E+02, 0.4289130000E+02 /
  data (epsData(58,i,24), i=1,3) / 0.8107590000E+01, -0.1064820000E+02, 0.4207790000E+02 /
  data (epsData(59,i,24), i=1,3) / 0.8199090000E+01, -0.3464670000E+01, 0.4154270000E+02 /
  data (epsData(60,i,24), i=1,3) / 0.8291270000E+01, 0.7326240000E+00, 0.5377130000E+02 /
  data (epsData(61,i,24), i=1,3) / 0.8384120000E+01, -0.3091930000E+01, 0.6302190000E+02 /
  data (epsData(62,i,24), i=1,3) / 0.8477660000E+01, -0.1037980000E+02, 0.6710440000E+02 /
  data (epsData(63,i,24), i=1,3) / 0.8571880000E+01, -0.1809160000E+02, 0.6706850000E+02 /
  data (epsData(64,i,24), i=1,3) / 0.8666780000E+01, -0.2377540000E+02, 0.6271020000E+02 /
  data (epsData(65,i,24), i=1,3) / 0.8762360000E+01, -0.2665880000E+02, 0.5842250000E+02 /
  data (epsData(66,i,24), i=1,3) / 0.8955560000E+01, -0.2860790000E+02, 0.5186250000E+02 /
  data (epsData(67,i,24), i=1,3) / 0.9250470000E+01, -0.2880870000E+02, 0.4460670000E+02 /
  data (epsData(68,i,24), i=1,3) / 0.9755580000E+01, -0.2717410000E+02, 0.3613270000E+02 /
  data (epsData(69,i,24), i=1,3) / 0.1038420000E+02, -0.2371060000E+02, 0.2899630000E+02 /
  data (epsData(70,i,24), i=1,3) / 0.1114850000E+02, -0.1956670000E+02, 0.2397150000E+02 /
  data (epsData(71,i,24), i=1,3) / 0.1194610000E+02, -0.1621200000E+02, 0.2090220000E+02 /
  data (epsData(72,i,24), i=1,3) / 0.1339100000E+02, -0.1261560000E+02, 0.1749030000E+02 /
  data (epsData(73,i,24), i=1,3) / 0.1493390000E+02, -0.1023000000E+02, 0.1513730000E+02 /
  data (epsData(74,i,24), i=1,3) / 0.1700030000E+02, -0.8404820000E+01, 0.1287670000E+02 /
  data (epsData(75,i,24), i=1,3) / 0.2015030000E+02, -0.6866070000E+01, 0.1024210000E+02 /
  data (epsData(76,i,24), i=1,3) / 0.2377210000E+02, -0.5635700000E+01, 0.7990810000E+01 /
  data (epsData(77,i,24), i=1,3) / 0.2753650000E+02, -0.4634020000E+01, 0.6404180000E+01 /
  data (epsData(78,i,24), i=1,3) / 0.3180190000E+02, -0.3811370000E+01, 0.5195190000E+01 /
  data (epsData(79,i,24), i=1,3) / 0.3814890000E+02, -0.2909270000E+01, 0.3971170000E+01 /
  data (epsData(80,i,24), i=1,3) / 0.4085920000E+02, -0.2415500000E+01, 0.3604650000E+01 /
  data (epsData(81,i,24), i=1,3) / 0.4178440000E+02, -0.2099280000E+01, 0.3605620000E+01 /
  data (epsData(82,i,24), i=1,3) / 0.4236460000E+02, -0.1964630000E+01, 0.3872510000E+01 /
  data (epsData(83,i,24), i=1,3) / 0.4305040000E+02, -0.2152300000E+01, 0.4051410000E+01 /
  data (epsData(84,i,24), i=1,3) / 0.4437690000E+02, -0.2379800000E+01, 0.3956450000E+01 /
  data (epsData(85,i,24), i=1,3) / 0.4726270000E+02, -0.2617810000E+01, 0.3626190000E+01 /
  data (epsData(86,i,24), i=1,3) / 0.5134850000E+02, -0.2714790000E+01, 0.2971920000E+01 /
  data (epsData(87,i,24), i=1,3) / 0.5607920000E+02, -0.2508050000E+01, 0.2311110000E+01 /
  data (epsData(88,i,24), i=1,3) / 0.6278400000E+02, -0.2105430000E+01, 0.1723550000E+01 /
  data (epsData(89,i,24), i=1,3) / 0.7060900000E+02, -0.1703510000E+01, 0.1330650000E+01 /
  data (epsData(90,i,24), i=1,3) / 0.7421020000E+02, -0.1529430000E+01, 0.1253930000E+01 /
  data (epsData(91,i,24), i=1,3) / 0.7792180000E+02, -0.1475270000E+01, 0.1156360000E+01 /
  data (epsData(92,i,24), i=1,3) / 0.8703390000E+02, -0.1254370000E+01, 0.9282320000E+00 /
  data (epsData(93,i,24), i=1,3) / 0.1003880000E+03, -0.1023530000E+01, 0.7034820000E+00 /
  data (epsData(94,i,24), i=1,3) / 0.1155300000E+03, -0.8348090000E+00, 0.5267190000E+00 /
  data (epsData(95,i,24), i=1,3) / 0.1321180000E+03, -0.6825620000E+00, 0.3953830000E+00 /
  data (epsData(96,i,24), i=1,3) / 0.1492380000E+03, -0.5661500000E+00, 0.3024720000E+00 /
  data (epsData(97,i,24), i=1,3) / 0.1692870000E+03, -0.4625630000E+00, 0.2242860000E+00 /
  data (epsData(98,i,24), i=1,3) / 0.1941160000E+03, -0.3678350000E+00, 0.1596290000E+00 /
  data (epsData(99,i,24), i=1,3) / 0.2204780000E+03, -0.2944820000E+00, 0.1155970000E+00 /
  data (epsData(100,i,24), i=1,3) / 0.2494550000E+03, -0.2348750000E+00, 0.8306030000E-01 /
  data (epsData(101,i,24), i=1,3) / 0.2805170000E+03, -0.1876590000E+00, 0.6054340000E-01 /
  data (epsData(102,i,24), i=1,3) / 0.3152540000E+03, -0.1484650000E+00, 0.4368780000E-01 /
  data (epsData(103,i,24), i=1,3) / 0.3533080000E+03, -0.1163210000E+00, 0.3160570000E-01 /
  data (epsData(104,i,24), i=1,3) / 0.3954700000E+03, -0.8960080000E-01, 0.2294790000E-01 /
  data (epsData(105,i,24), i=1,3) / 0.4413010000E+03, -0.6720460000E-01, 0.1669750000E-01 /
  data (epsData(106,i,24), i=1,3) / 0.5130650000E+03, -0.3775220000E-01, 0.1054300000E-01 /
  data (epsData(107,i,24), i=1,3) / 0.5381440000E+03, -0.2603080000E-01, 0.9112790000E-02 /
  data (epsData(108,i,24), i=1,3) / 0.5509450000E+03, -0.1812710000E-01, 0.8460880000E-02 /
  data (epsData(109,i,24), i=1,3) / 0.5582220000E+03, -0.1219700000E-01, 0.8226190000E-02 /
  data (epsData(110,i,24), i=1,3) / 0.5622880000E+03, -0.7968730000E-02, 0.8080300000E-02 /
  data (epsData(111,i,24), i=1,3) / 0.5655540000E+03, -0.3756880000E-02, 0.8055180000E-02 /
  data (epsData(112,i,24), i=1,3) / 0.5671910000E+03, -0.1204520000E-02, 0.8029640000E-02 /
  data (epsData(113,i,24), i=1,3) / 0.5688300000E+03, 0.1822270000E-02, 0.8024330000E-02 /
  data (epsData(114,i,24), i=1,3) / 0.5696510000E+03, 0.3586340000E-02, 0.8034650000E-02 /
  data (epsData(115,i,24), i=1,3) / 0.5704720000E+03, 0.5584420000E-02, 0.8059370000E-02 /
  data (epsData(116,i,24), i=1,3) / 0.5712950000E+03, 0.7907450000E-02, 0.8107820000E-02 /
  data (epsData(117,i,24), i=1,3) / 0.5721170000E+03, 0.1072720000E-01, 0.8204020000E-02 /
  data (epsData(118,i,24), i=1,3) / 0.5729410000E+03, 0.1443270000E-01, 0.8422570000E-02 /
  data (epsData(119,i,24), i=1,3) / 0.5737650000E+03, 0.1977760000E-01, 0.9440450000E-02 /
  data (epsData(120,i,24), i=1,3) / 0.5741870000E+03, 0.2301210000E-01, 0.1236090000E-01 /
  data (epsData(121,i,24), i=1,3) / 0.5746170000E+03, 0.2414760000E-01, 0.1571710000E-01 /
  data (epsData(122,i,24), i=1,3) / 0.5759130000E+03, 0.2607620000E-01, 0.2283240000E-01 /
  data (epsData(123,i,24), i=1,3) / 0.5769790000E+03, 0.2714710000E-01, 0.2908000000E-01 /
  data (epsData(124,i,24), i=1,3) / 0.5778500000E+03, 0.2581180000E-01, 0.3349610000E-01 /
  data (epsData(125,i,24), i=1,3) / 0.5789980000E+03, 0.2532010000E-01, 0.3859880000E-01 /
  data (epsData(126,i,24), i=1,3) / 0.5805780000E+03, 0.2157260000E-01, 0.4516790000E-01 /
  data (epsData(127,i,24), i=1,3) / 0.5822720000E+03, 0.1704120000E-01, 0.4863520000E-01 /
  data (epsData(128,i,24), i=1,3) / 0.5835860000E+03, 0.1572240000E-01, 0.5021050000E-01 /
  data (epsData(129,i,24), i=1,3) / 0.5851070000E+03, 0.1339260000E-01, 0.5419550000E-01 /
  data (epsData(130,i,24), i=1,3) / 0.5886980000E+03, 0.6241760000E-02, 0.6052620000E-01 /
  data (epsData(131,i,24), i=1,3) / 0.5914530000E+03, -0.9745870000E-03, 0.6347930000E-01 /
  data (epsData(132,i,24), i=1,3) / 0.5946980000E+03, -0.8136510000E-02, 0.6301590000E-01 /
  data (epsData(133,i,24), i=1,3) / 0.6008670000E+03, -0.1573430000E-01, 0.6034450000E-01 /
  data (epsData(134,i,24), i=1,3) / 0.6162770000E+03, -0.2645290000E-01, 0.5569910000E-01 /
  data (epsData(135,i,24), i=1,3) / 0.6316320000E+03, -0.3131800000E-01, 0.5001200000E-01 /
  data (epsData(136,i,24), i=1,3) / 0.6671440000E+03, -0.3499230000E-01, 0.4131340000E-01 /
  data (epsData(137,i,24), i=1,3) / 0.6889510000E+03, -0.3353690000E-01, 0.3721510000E-01 /
  data (epsData(138,i,24), i=1,3) / 0.6950910000E+03, -0.3172800000E-01, 0.3789070000E-01 /
  data (epsData(139,i,24), i=1,3) / 0.6981770000E+03, -0.3294970000E-01, 0.3940950000E-01 /
  data (epsData(140,i,24), i=1,3) / 0.7082790000E+03, -0.3486950000E-01, 0.3763010000E-01 /
  data (epsData(141,i,24), i=1,3) / 0.7496090000E+03, -0.3609170000E-01, 0.3150790000E-01 /
  data (epsData(142,i,24), i=1,3) / 0.8200910000E+03, -0.3405060000E-01, 0.2362570000E-01 /
  data (epsData(143,i,24), i=1,3) / 0.9275150000E+03, -0.2933750000E-01, 0.1581300000E-01 /
  data (epsData(144,i,24), i=1,3) / 0.1047120000E+04, -0.2435600000E-01, 0.1054210000E-01 /
  data (epsData(145,i,24), i=1,3) / 0.1171240000E+04, -0.2011830000E-01, 0.7207830000E-02 /
  data (epsData(146,i,24), i=1,3) / 0.1297700000E+04, -0.1670040000E-01, 0.5040210000E-02 /
  data (epsData(147,i,24), i=1,3) / 0.1442080000E+04, -0.1366080000E-01, 0.3468050000E-02 /
  data (epsData(148,i,24), i=1,3) / 0.1605310000E+04, -0.1108630000E-01, 0.2375330000E-02 /
  data (epsData(149,i,24), i=1,3) / 0.1773740000E+04, -0.9096210000E-02, 0.1658130000E-02 /
  data (epsData(150,i,24), i=1,3) / 0.1966980000E+04, -0.7383020000E-02, 0.1138900000E-02 /
  data (epsData(151,i,24), i=1,3) / 0.2184680000E+04, -0.5965880000E-02, 0.7794130000E-03 /
  data (epsData(152,i,24), i=1,3) / 0.2427230000E+04, -0.4813230000E-02, 0.5304310000E-03 /
  data (epsData(153,i,24), i=1,3) / 0.2696330000E+04, -0.3879140000E-02, 0.3599680000E-03 /
  data (epsData(154,i,24), i=1,3) / 0.2997220000E+04, -0.3119250000E-02, 0.2431630000E-03 /
  data (epsData(155,i,24), i=1,3) / 0.3332430000E+04, -0.2504970000E-02, 0.1638060000E-03 /
  data (epsData(156,i,24), i=1,3) / 0.3708620000E+04, -0.2005590000E-02, 0.1097570000E-03 /
  data (epsData(157,i,24), i=1,3) / 0.4131540000E+04, -0.1599550000E-02, 0.7314190000E-04 /
  data (epsData(158,i,24), i=1,3) / 0.4607180000E+04, -0.1268900000E-02, 0.4848100000E-04 /
  data (epsData(159,i,24), i=1,3) / 0.5152880000E+04, -0.9916990000E-03, 0.3171250000E-04 /
  data (epsData(160,i,24), i=1,3) / 0.5866200000E+04, -0.6932010000E-03, 0.1935790000E-04 /
  data (epsData(161,i,24), i=1,3) / 0.5949120000E+04, -0.6285020000E-03, 0.1834780000E-04 /
  data (epsData(162,i,24), i=1,3) / 0.5973380000E+04, -0.5828490000E-03, 0.1944590000E-04 /
  data (epsData(163,i,24), i=1,3) / 0.5981500000E+04, -0.5480550000E-03, 0.2124740000E-04 /
  data (epsData(164,i,24), i=1,3) / 0.5984750000E+04, -0.5191370000E-03, 0.2426100000E-04 /
  data (epsData(165,i,24), i=1,3) / 0.5986380000E+04, -0.4927320000E-03, 0.2888940000E-04 /
  data (epsData(166,i,24), i=1,3) / 0.5988010000E+04, -0.4462860000E-03, 0.4813810000E-04 /
  data (epsData(167,i,24), i=1,3) / 0.5989100000E+04, -0.4238260000E-03, 0.1275840000E-03 /
  data (epsData(168,i,24), i=1,3) / 0.5989570000E+04, -0.4495550000E-03, 0.1639070000E-03 /
  data (epsData(169,i,24), i=1,3) / 0.5990390000E+04, -0.4984370000E-03, 0.1807220000E-03 /
  data (epsData(170,i,24), i=1,3) / 0.5991520000E+04, -0.5391270000E-03, 0.1684050000E-03 /
  data (epsData(171,i,24), i=1,3) / 0.5994490000E+04, -0.5668810000E-03, 0.1412140000E-03 /
  data (epsData(172,i,24), i=1,3) / 0.6001280000E+04, -0.5733910000E-03, 0.1377230000E-03 /
  data (epsData(173,i,24), i=1,3) / 0.6022110000E+04, -0.6108370000E-03, 0.1444940000E-03 /
  data (epsData(174,i,24), i=1,3) / 0.6097580000E+04, -0.6442040000E-03, 0.1396140000E-03 /
  data (epsData(175,i,24), i=1,3) / 0.6338740000E+04, -0.6435370000E-03, 0.1219490000E-03 /
  data (epsData(176,i,24), i=1,3) / 0.7326590000E+04, -0.5210620000E-03, 0.7255540000E-04 /
  data (epsData(177,i,24), i=1,3) / 0.8311500000E+04, -0.4150810000E-03, 0.4649200000E-04 /
  data (epsData(178,i,24), i=1,3) / 0.9319340000E+04, -0.3339820000E-03, 0.3057680000E-04 /
  data (epsData(179,i,24), i=1,3) / 0.1042420000E+05, -0.2686040000E-03, 0.2019720000E-04 /
  data (epsData(180,i,24), i=1,3) / 0.1164810000E+05, -0.2157430000E-03, 0.1332160000E-04 /
  data (epsData(181,i,24), i=1,3) / 0.1301170000E+05, -0.1730520000E-03, 0.8753680000E-05 /
  data (epsData(182,i,24), i=1,3) / 0.1453940000E+05, -0.1385660000E-03, 0.5732320000E-05 /
  data (epsData(183,i,24), i=1,3) / 0.1624720000E+05, -0.1108590000E-03, 0.3734480000E-05 /
  data (epsData(184,i,24), i=1,3) / 0.1816320000E+05, -0.8857950000E-04, 0.2421360000E-05 /
  data (epsData(185,i,24), i=1,3) / 0.2030940000E+05, -0.7073300000E-04, 0.1562740000E-05 /
  data (epsData(186,i,24), i=1,3) / 0.2271630000E+05, -0.5644160000E-04, 0.1004470000E-05 /
  data (epsData(187,i,24), i=1,3) / 0.2541790000E+05, -0.4500460000E-04, 0.6423380000E-06 /
  data (epsData(188,i,24), i=1,3) / 0.2845420000E+05, -0.3585440000E-04, 0.4089470000E-06 /
  data (epsData(189,i,24), i=1,3) / 0.3185890000E+05, -0.2855740000E-04, 0.2593590000E-06 /
  data (epsData(190,i,24), i=1,3) / 0.3568160000E+05, -0.2273520000E-04, 0.1637770000E-06 /
  data (epsData(191,i,24), i=1,3) / 0.3997800000E+05, -0.1808880000E-04, 0.1029180000E-06 /
  data (epsData(192,i,24), i=1,3) / 0.4480420000E+05, -0.1438610000E-04, 0.6445880000E-07 /
  data (epsData(193,i,24), i=1,3) / 0.5022180000E+05, -0.1143890000E-04, 0.4020570000E-07 /
  data (epsData(194,i,24), i=1,3) / 0.5630720000E+05, -0.9092520000E-05, 0.2491700000E-07 /
  data (epsData(195,i,24), i=1,3) / 0.6313580000E+05, -0.7227020000E-05, 0.1556530000E-07 /
  data (epsData(196,i,24), i=1,3) / 0.7079710000E+05, -0.5744130000E-05, 0.9509860000E-08 /
  data (epsData(197,i,24), i=1,3) / 0.7939890000E+05, -0.4564790000E-05, 0.5904530000E-08 /
  data (epsData(198,i,24), i=1,3) / 0.8905650000E+05, -0.3627190000E-05, 0.3594740000E-08 /
  data (epsData(199,i,24), i=1,3) / 0.9975030000E+05, -0.2892420000E-05, 0.2226990000E-08 /
  data (epsData(200,i,24), i=1,3) / 0.1000000000E+06, -0.2882100000E-05, 0.2203540000E-08 /
  data (epsData(1,i,25), i=1,3) / 0.2506580000E-02, -0.2519710000E+04, 0.6553040000E+04 /
  data (epsData(2,i,25), i=1,3) / 0.4693440000E-02, -0.2018280000E+04, 0.2185560000E+04 /
  data (epsData(3,i,25), i=1,3) / 0.7560590000E-02, -0.8456080000E+03, 0.1002150000E+04 /
  data (epsData(4,i,25), i=1,3) / 0.1110800000E-01, -0.3358270000E+03, 0.5559290000E+03 /
  data (epsData(5,i,25), i=1,3) / 0.1533570000E-01, -0.9640970000E+02, 0.3489750000E+03 /
  data (epsData(6,i,25), i=1,3) / 0.2024370000E-01, 0.2736810000E+02, 0.2384550000E+03 /
  data (epsData(7,i,25), i=1,3) / 0.2583200000E-01, 0.9663340000E+02, 0.1731430000E+03 /
  data (epsData(8,i,25), i=1,3) / 0.3210060000E-01, 0.1379370000E+03, 0.1315180000E+03 /
  data (epsData(9,i,25), i=1,3) / 0.3904940000E-01, 0.1638730000E+03, 0.1034040000E+03 /
  data (epsData(10,i,25), i=1,3) / 0.4667860000E-01, 0.1808770000E+03, 0.8352840000E+02 /
  data (epsData(11,i,25), i=1,3) / 0.5498800000E-01, 0.1924460000E+03, 0.6895270000E+02 /
  data (epsData(12,i,25), i=1,3) / 0.6397770000E-01, 0.2005870000E+03, 0.5793870000E+02 /
  data (epsData(13,i,25), i=1,3) / 0.8399800000E-01, 0.2110020000E+03, 0.4265780000E+02 /
  data (epsData(14,i,25), i=1,3) / 0.1067390000E+00, 0.2175830000E+03, 0.3353000000E+02 /
  data (epsData(15,i,25), i=1,3) / 0.1754980000E+00, 0.2223460000E+03, 0.2234820000E+02 /
  data (epsData(16,i,25), i=1,3) / 0.3866320000E+00, 0.2263570000E+03, 0.1097120000E+02 /
  data (epsData(17,i,25), i=1,3) / 0.8071970000E+00, 0.2399730000E+03, 0.7736770000E+01 /
  data (epsData(18,i,25), i=1,3) / 0.1133010000E+01, 0.2614850000E+03, 0.8197570000E+01 /
  data (epsData(19,i,25), i=1,3) / 0.1337820000E+01, 0.2851410000E+03, 0.9659540000E+01 /
  data (epsData(20,i,25), i=1,3) / 0.1513920000E+01, 0.3158860000E+03, 0.1309220000E+02 /
  data (epsData(21,i,25), i=1,3) / 0.1606050000E+01, 0.3445230000E+03, 0.1571280000E+02 /
  data (epsData(22,i,25), i=1,3) / 0.1749350000E+01, 0.3907200000E+03, 0.4636190000E+02 /
  data (epsData(23,i,25), i=1,3) / 0.1798470000E+01, 0.4107530000E+03, 0.5460710000E+02 /
  data (epsData(24,i,25), i=1,3) / 0.1848280000E+01, 0.4423130000E+03, 0.6202320000E+02 /
  data (epsData(25,i,25), i=1,3) / 0.1898770000E+01, 0.5059370000E+03, 0.6868030000E+02 /
  data (epsData(26,i,25), i=1,3) / 0.1949930000E+01, 0.5499710000E+03, 0.1943050000E+03 /
  data (epsData(27,i,25), i=1,3) / 0.2003060000E+01, 0.5138010000E+03, 0.3103100000E+03 /
  data (epsData(28,i,25), i=1,3) / 0.2110210000E+01, 0.4036130000E+03, 0.3947180000E+03 /
  data (epsData(29,i,25), i=1,3) / 0.2179940000E+01, 0.3247900000E+03, 0.4400840000E+03 /
  data (epsData(30,i,25), i=1,3) / 0.2212590000E+01, 0.2705870000E+03, 0.4457780000E+03 /
  data (epsData(31,i,25), i=1,3) / 0.2266670000E+01, 0.2115730000E+03, 0.4205690000E+03 /
  data (epsData(32,i,25), i=1,3) / 0.2370410000E+01, 0.1476550000E+03, 0.3772130000E+03 /
  data (epsData(33,i,25), i=1,3) / 0.2491170000E+01, 0.1062310000E+03, 0.3338070000E+03 /
  data (epsData(34,i,25), i=1,3) / 0.2688790000E+01, 0.6931330000E+02, 0.2845310000E+03 /
  data (epsData(35,i,25), i=1,3) / 0.2884710000E+01, 0.4876920000E+02, 0.2469290000E+03 /
  data (epsData(36,i,25), i=1,3) / 0.3224510000E+01, 0.2809900000E+02, 0.1959270000E+03 /
  data (epsData(37,i,25), i=1,3) / 0.3481660000E+01, 0.2568490000E+02, 0.1620540000E+03 /
  data (epsData(38,i,25), i=1,3) / 0.3666700000E+01, 0.3173820000E+02, 0.1406100000E+03 /
  data (epsData(39,i,25), i=1,3) / 0.3862620000E+01, 0.4534920000E+02, 0.1219690000E+03 /
  data (epsData(40,i,25), i=1,3) / 0.4016700000E+01, 0.6331130000E+02, 0.1121220000E+03 /
  data (epsData(41,i,25), i=1,3) / 0.4122830000E+01, 0.8288510000E+02, 0.1070090000E+03 /
  data (epsData(42,i,25), i=1,3) / 0.4176910000E+01, 0.1020350000E+03, 0.1045470000E+03 /
  data (epsData(43,i,25), i=1,3) / 0.4231670000E+01, 0.1200660000E+03, 0.1266060000E+03 /
  data (epsData(44,i,25), i=1,3) / 0.4287120000E+01, 0.1221660000E+03, 0.1553430000E+03 /
  data (epsData(45,i,25), i=1,3) / 0.4400040000E+01, 0.1067540000E+03, 0.1843350000E+03 /
  data (epsData(46,i,25), i=1,3) / 0.4457530000E+01, 0.9436190000E+02, 0.1976030000E+03 /
  data (epsData(47,i,25), i=1,3) / 0.4515690000E+01, 0.7860280000E+02, 0.2038180000E+03 /
  data (epsData(48,i,25), i=1,3) / 0.4634060000E+01, 0.5562800000E+02, 0.1997080000E+03 /
  data (epsData(49,i,25), i=1,3) / 0.4941890000E+01, 0.2325380000E+02, 0.1897640000E+03 /
  data (epsData(50,i,25), i=1,3) / 0.5200400000E+01, 0.1874600000E+01, 0.1789920000E+03 /
  data (epsData(51,i,25), i=1,3) / 0.5469790000E+01, -0.1540570000E+02, 0.1645530000E+03 /
  data (epsData(52,i,25), i=1,3) / 0.5894290000E+01, -0.3053360000E+02, 0.1376500000E+03 /
  data (epsData(53,i,25), i=1,3) / 0.6266740000E+01, -0.3407090000E+02, 0.1171410000E+03 /
  data (epsData(54,i,25), i=1,3) / 0.6816750000E+01, -0.3381520000E+02, 0.9538850000E+02 /
  data (epsData(55,i,25), i=1,3) / 0.7572890000E+01, -0.2970120000E+02, 0.7218510000E+02 /
  data (epsData(56,i,25), i=1,3) / 0.7926630000E+01, -0.2398320000E+02, 0.6280460000E+02 /
  data (epsData(57,i,25), i=1,3) / 0.8199090000E+01, -0.1710660000E+02, 0.5809460000E+02 /
  data (epsData(58,i,25), i=1,3) / 0.8384120000E+01, -0.1149880000E+02, 0.5664620000E+02 /
  data (epsData(59,i,25), i=1,3) / 0.8477660000E+01, -0.7767990000E+01, 0.5626590000E+02 /
  data (epsData(60,i,25), i=1,3) / 0.8571880000E+01, -0.1314880000E+01, 0.5591870000E+02 /
  data (epsData(61,i,25), i=1,3) / 0.8666780000E+01, 0.2375010000E+01, 0.6743370000E+02 /
  data (epsData(62,i,25), i=1,3) / 0.8762360000E+01, -0.1402980000E+01, 0.7600640000E+02 /
  data (epsData(63,i,25), i=1,3) / 0.8858620000E+01, -0.8407470000E+01, 0.7972350000E+02 /
  data (epsData(64,i,25), i=1,3) / 0.8955560000E+01, -0.1549480000E+02, 0.7906370000E+02 /
  data (epsData(65,i,25), i=1,3) / 0.9053180000E+01, -0.2042990000E+02, 0.7504280000E+02 /
  data (epsData(66,i,25), i=1,3) / 0.9151480000E+01, -0.2287770000E+02, 0.7117100000E+02 /
  data (epsData(67,i,25), i=1,3) / 0.9450470000E+01, -0.2507270000E+02, 0.6299820000E+02 /
  data (epsData(68,i,25), i=1,3) / 0.9962380000E+01, -0.2483630000E+02, 0.5343260000E+02 /
  data (epsData(69,i,25), i=1,3) / 0.1049130000E+02, -0.2239480000E+02, 0.4616740000E+02 /
  data (epsData(70,i,25), i=1,3) / 0.1126040000E+02, -0.1813200000E+02, 0.3985250000E+02 /
  data (epsData(71,i,25), i=1,3) / 0.1218010000E+02, -0.1430910000E+02, 0.3563970000E+02 /
  data (epsData(72,i,25), i=1,3) / 0.1376750000E+02, -0.1093250000E+02, 0.3137610000E+02 /
  data (epsData(73,i,25), i=1,3) / 0.1574210000E+02, -0.9078110000E+01, 0.2770270000E+02 /
  data (epsData(74,i,25), i=1,3) / 0.1952720000E+02, -0.8160000000E+01, 0.2215130000E+02 /
  data (epsData(75,i,25), i=1,3) / 0.2394460000E+02, -0.7446010000E+01, 0.1701970000E+02 /
  data (epsData(76,i,25), i=1,3) / 0.2847700000E+02, -0.6526790000E+01, 0.1350350000E+02 /
  data (epsData(77,i,25), i=1,3) / 0.3384950000E+02, -0.5753940000E+01, 0.1077840000E+02 /
  data (epsData(78,i,25), i=1,3) / 0.4201740000E+02, -0.5000840000E+01, 0.7929520000E+01 /
  data (epsData(79,i,25), i=1,3) / 0.4608240000E+02, -0.4467250000E+01, 0.6867030000E+01 /
  data (epsData(80,i,25), i=1,3) / 0.4733610000E+02, -0.4193470000E+01, 0.6853450000E+01 /
  data (epsData(81,i,25), i=1,3) / 0.4901260000E+02, -0.4382330000E+01, 0.6689920000E+01 /
  data (epsData(82,i,25), i=1,3) / 0.5612670000E+02, -0.4459830000E+01, 0.5233790000E+01 /
  data (epsData(83,i,25), i=1,3) / 0.6277710000E+02, -0.4053220000E+01, 0.4006440000E+01 /
  data (epsData(84,i,25), i=1,3) / 0.7144840000E+02, -0.3435600000E+01, 0.2994210000E+01 /
  data (epsData(85,i,25), i=1,3) / 0.8138050000E+02, -0.2840470000E+01, 0.2270690000E+01 /
  data (epsData(86,i,25), i=1,3) / 0.9744350000E+02, -0.2264900000E+01, 0.1543450000E+01 /
  data (epsData(87,i,25), i=1,3) / 0.1109060000E+03, -0.1881140000E+01, 0.1141260000E+01 /
  data (epsData(88,i,25), i=1,3) / 0.1258940000E+03, -0.1550480000E+01, 0.8409580000E+00 /
  data (epsData(89,i,25), i=1,3) / 0.1425870000E+03, -0.1272020000E+01, 0.6181530000E+00 /
  data (epsData(90,i,25), i=1,3) / 0.1643020000E+03, -0.1008730000E+01, 0.4235390000E+00 /
  data (epsData(91,i,25), i=1,3) / 0.1870740000E+03, -0.8079060000E+00, 0.2972170000E+00 /
  data (epsData(92,i,25), i=1,3) / 0.2084540000E+03, -0.6650700000E+00, 0.2205910000E+00 /
  data (epsData(93,i,25), i=1,3) / 0.2309180000E+03, -0.5507090000E+00, 0.1643940000E+00 /
  data (epsData(94,i,25), i=1,3) / 0.2561930000E+03, -0.4520150000E+00, 0.1209200000E+00 /
  data (epsData(95,i,25), i=1,3) / 0.2871300000E+03, -0.3626250000E+00, 0.8616350000E-01 /
  data (epsData(96,i,25), i=1,3) / 0.3211680000E+03, -0.2903310000E+00, 0.6129160000E-01 /
  data (epsData(97,i,25), i=1,3) / 0.3584350000E+03, -0.2319740000E+00, 0.4379720000E-01 /
  data (epsData(98,i,25), i=1,3) / 0.3984010000E+03, -0.1856590000E+00, 0.3179830000E-01 /
  data (epsData(99,i,25), i=1,3) / 0.4452720000E+03, -0.1458420000E+00, 0.2254110000E-01 /
  data (epsData(100,i,25), i=1,3) / 0.4923890000E+03, -0.1152050000E+00, 0.1637720000E-01 /
  data (epsData(101,i,25), i=1,3) / 0.5551260000E+03, -0.8313980000E-01, 0.1105680000E-01 /
  data (epsData(102,i,25), i=1,3) / 0.6061630000E+03, -0.5797970000E-01, 0.8330790000E-02 /
  data (epsData(103,i,25), i=1,3) / 0.6223270000E+03, -0.4696350000E-01, 0.7731520000E-02 /
  data (epsData(104,i,25), i=1,3) / 0.6300690000E+03, -0.3914440000E-01, 0.7553670000E-02 /
  data (epsData(105,i,25), i=1,3) / 0.6343940000E+03, -0.3262630000E-01, 0.7529410000E-02 /
  data (epsData(106,i,25), i=1,3) / 0.6361290000E+03, -0.2879720000E-01, 0.7601290000E-02 /
  data (epsData(107,i,25), i=1,3) / 0.6369970000E+03, -0.2625600000E-01, 0.7702910000E-02 /
  data (epsData(108,i,25), i=1,3) / 0.6378660000E+03, -0.2281450000E-01, 0.7977070000E-02 /
  data (epsData(109,i,25), i=1,3) / 0.6387410000E+03, -0.1840320000E-01, 0.1020220000E-01 /
  data (epsData(110,i,25), i=1,3) / 0.6391170000E+03, -0.1716290000E-01, 0.1233170000E-01 /
  data (epsData(111,i,25), i=1,3) / 0.6411000000E+03, -0.1357530000E-01, 0.2060590000E-01 /
  data (epsData(112,i,25), i=1,3) / 0.6419670000E+03, -0.1349480000E-01, 0.2493230000E-01 /
  data (epsData(113,i,25), i=1,3) / 0.6430430000E+03, -0.1384100000E-01, 0.2861250000E-01 /
  data (epsData(114,i,25), i=1,3) / 0.6443600000E+03, -0.1532990000E-01, 0.3391210000E-01 /
  data (epsData(115,i,25), i=1,3) / 0.6460500000E+03, -0.1919110000E-01, 0.3831160000E-01 /
  data (epsData(116,i,25), i=1,3) / 0.6479600000E+03, -0.2290610000E-01, 0.4003200000E-01 /
  data (epsData(117,i,25), i=1,3) / 0.6497210000E+03, -0.2367350000E-01, 0.4074740000E-01 /
  data (epsData(118,i,25), i=1,3) / 0.6538670000E+03, -0.2684730000E-01, 0.4728220000E-01 /
  data (epsData(119,i,25), i=1,3) / 0.6569530000E+03, -0.3179650000E-01, 0.5130510000E-01 /
  data (epsData(120,i,25), i=1,3) / 0.6610430000E+03, -0.3866700000E-01, 0.5180450000E-01 /
  data (epsData(121,i,25), i=1,3) / 0.6699850000E+03, -0.4582720000E-01, 0.4954400000E-01 /
  data (epsData(122,i,25), i=1,3) / 0.6847310000E+03, -0.5223690000E-01, 0.4577430000E-01 /
  data (epsData(123,i,25), i=1,3) / 0.7073570000E+03, -0.5483140000E-01, 0.4024630000E-01 /
  data (epsData(124,i,25), i=1,3) / 0.7465320000E+03, -0.5406250000E-01, 0.3304800000E-01 /
  data (epsData(125,i,25), i=1,3) / 0.7655120000E+03, -0.5096110000E-01, 0.3113970000E-01 /
  data (epsData(126,i,25), i=1,3) / 0.7700920000E+03, -0.5038380000E-01, 0.3286620000E-01 /
  data (epsData(127,i,25), i=1,3) / 0.7776990000E+03, -0.5189790000E-01, 0.3210000000E-01 /
  data (epsData(128,i,25), i=1,3) / 0.8340570000E+03, -0.5014650000E-01, 0.2576590000E-01 /
  data (epsData(129,i,25), i=1,3) / 0.9420210000E+03, -0.4294630000E-01, 0.1733770000E-01 /
  data (epsData(130,i,25), i=1,3) / 0.1061400000E+04, -0.3555850000E-01, 0.1169040000E-01 /
  data (epsData(131,i,25), i=1,3) / 0.1187450000E+04, -0.2925420000E-01, 0.7975960000E-02 /
  data (epsData(132,i,25), i=1,3) / 0.1328150000E+04, -0.2380780000E-01, 0.5431000000E-02 /
  data (epsData(133,i,25), i=1,3) / 0.1474440000E+04, -0.1952100000E-01, 0.3761970000E-02 /
  data (epsData(134,i,25), i=1,3) / 0.1640980000E+04, -0.1583760000E-01, 0.2569830000E-02 /
  data (epsData(135,i,25), i=1,3) / 0.1823960000E+04, -0.1285080000E-01, 0.1761710000E-02 /
  data (epsData(136,i,25), i=1,3) / 0.2028280000E+04, -0.1038700000E-01, 0.1197470000E-02 /
  data (epsData(137,i,25), i=1,3) / 0.2257780000E+04, -0.8367350000E-02, 0.8136990000E-03 /
  data (epsData(138,i,25), i=1,3) / 0.2515710000E+04, -0.6722200000E-02, 0.5483550000E-03 /
  data (epsData(139,i,25), i=1,3) / 0.2800890000E+04, -0.5403190000E-02, 0.3690720000E-03 /
  data (epsData(140,i,25), i=1,3) / 0.3120130000E+04, -0.4335380000E-02, 0.2474070000E-03 /
  data (epsData(141,i,25), i=1,3) / 0.3477990000E+04, -0.3472130000E-02, 0.1650820000E-03 /
  data (epsData(142,i,25), i=1,3) / 0.3879680000E+04, -0.2774960000E-02, 0.1096530000E-03 /
  data (epsData(143,i,25), i=1,3) / 0.4328890000E+04, -0.2214560000E-02, 0.7262450000E-04 /
  data (epsData(144,i,25), i=1,3) / 0.4834200000E+04, -0.1761320000E-02, 0.4786850000E-04 /
  data (epsData(145,i,25), i=1,3) / 0.5405760000E+04, -0.1391610000E-02, 0.3133970000E-04 /
  data (epsData(146,i,25), i=1,3) / 0.6113000000E+04, -0.1056230000E-02, 0.1962410000E-04 /
  data (epsData(147,i,25), i=1,3) / 0.6464380000E+04, -0.8883930000E-03, 0.1593990000E-04 /
  data (epsData(148,i,25), i=1,3) / 0.6518810000E+04, -0.8272820000E-03, 0.1543760000E-04 /
  data (epsData(149,i,25), i=1,3) / 0.6532120000E+04, -0.7836340000E-03, 0.1862150000E-04 /
  data (epsData(150,i,25), i=1,3) / 0.6535450000E+04, -0.7541270000E-03, 0.2238350000E-04 /
  data (epsData(151,i,25), i=1,3) / 0.6537120000E+04, -0.7247740000E-03, 0.2960840000E-04 /
  data (epsData(152,i,25), i=1,3) / 0.6538790000E+04, -0.6900730000E-03, 0.8074210000E-04 /
  data (epsData(153,i,25), i=1,3) / 0.6539370000E+04, -0.6971720000E-03, 0.1253150000E-03 /
  data (epsData(154,i,25), i=1,3) / 0.6540350000E+04, -0.7469360000E-03, 0.1499640000E-03 /
  data (epsData(155,i,25), i=1,3) / 0.6541940000E+04, -0.7860860000E-03, 0.1335070000E-03 /
  data (epsData(156,i,25), i=1,3) / 0.6545960000E+04, -0.8025520000E-03, 0.1128330000E-03 /
  data (epsData(157,i,25), i=1,3) / 0.6579960000E+04, -0.8405360000E-03, 0.1181710000E-03 /
  data (epsData(158,i,25), i=1,3) / 0.6692150000E+04, -0.8592000000E-03, 0.1144430000E-03 /
  data (epsData(159,i,25), i=1,3) / 0.7319730000E+04, -0.7670530000E-03, 0.8344640000E-04 /
  data (epsData(160,i,25), i=1,3) / 0.8324370000E+04, -0.6103840000E-03, 0.5301520000E-04 /
  data (epsData(161,i,25), i=1,3) / 0.9368230000E+04, -0.4880670000E-03, 0.3449770000E-04 /
  data (epsData(162,i,25), i=1,3) / 0.1049890000E+05, -0.3911310000E-03, 0.2268080000E-04 /
  data (epsData(163,i,25), i=1,3) / 0.1175230000E+05, -0.3131810000E-03, 0.1490080000E-04 /
  data (epsData(164,i,25), i=1,3) / 0.1314770000E+05, -0.2505900000E-03, 0.9776850000E-05 /
  data (epsData(165,i,25), i=1,3) / 0.1470840000E+05, -0.2003020000E-03, 0.6380750000E-05 /
  data (epsData(166,i,25), i=1,3) / 0.1645280000E+05, -0.1600180000E-03, 0.4150960000E-05 /
  data (epsData(167,i,25), i=1,3) / 0.1840590000E+05, -0.1277570000E-03, 0.2688280000E-05 /
  data (epsData(168,i,25), i=1,3) / 0.2060010000E+05, -0.1018830000E-03, 0.1732050000E-05 /
  data (epsData(169,i,25), i=1,3) / 0.2306180000E+05, -0.8120050000E-04, 0.1110990000E-05 /
  data (epsData(170,i,25), i=1,3) / 0.2582420000E+05, -0.6468180000E-04, 0.7095540000E-06 /
  data (epsData(171,i,25), i=1,3) / 0.2892680000E+05, -0.5149200000E-04, 0.4511740000E-06 /
  data (epsData(172,i,25), i=1,3) / 0.3240900000E+05, -0.4097730000E-04, 0.2856940000E-06 /
  data (epsData(173,i,25), i=1,3) / 0.3631430000E+05, -0.3260560000E-04, 0.1802660000E-06 /
  data (epsData(174,i,25), i=1,3) / 0.4069830000E+05, -0.2593620000E-04, 0.1133620000E-06 /
  data (epsData(175,i,25), i=1,3) / 0.4562360000E+05, -0.2062230000E-04, 0.7095370000E-07 /
  data (epsData(176,i,25), i=1,3) / 0.5115280000E+05, -0.1639360000E-04, 0.4420870000E-07 /
  data (epsData(177,i,25), i=1,3) / 0.5735500000E+05, -0.1303200000E-04, 0.2751230000E-07 /
  data (epsData(178,i,25), i=1,3) / 0.6431320000E+05, -0.1035940000E-04, 0.1711850000E-07 /
  data (epsData(179,i,25), i=1,3) / 0.7212840000E+05, -0.8232500000E-05, 0.1053130000E-07 /
  data (epsData(180,i,25), i=1,3) / 0.8090090000E+05, -0.6541700000E-05, 0.6490560000E-08 /
  data (epsData(181,i,25), i=1,3) / 0.9074720000E+05, -0.5197910000E-05, 0.3951200000E-08 /
  data (epsData(182,i,25), i=1,3) / 0.1000000000E+06, -0.4286140000E-05, 0.2633510000E-08 /
  data (epsData(1,i,26), i=1,3) / 0.2506580000E-02, -0.2731350000E+04, 0.6885180000E+04 /
  data (epsData(2,i,26), i=1,3) / 0.4693440000E-02, -0.2169930000E+04, 0.2279110000E+04 /
  data (epsData(3,i,26), i=1,3) / 0.7560590000E-02, -0.9184080000E+03, 0.1037600000E+04 /
  data (epsData(4,i,26), i=1,3) / 0.1110800000E-01, -0.3750780000E+03, 0.5720120000E+03 /
  data (epsData(5,i,26), i=1,3) / 0.1533570000E-01, -0.1200190000E+03, 0.3571990000E+03 /
  data (epsData(6,i,26), i=1,3) / 0.2024370000E-01, 0.1185980000E+02, 0.2430320000E+03 /
  data (epsData(7,i,26), i=1,3) / 0.2583200000E-01, 0.8578440000E+02, 0.1758550000E+03 /
  data (epsData(8,i,26), i=1,3) / 0.3210060000E-01, 0.1298110000E+03, 0.1336890000E+03 /
  data (epsData(9,i,26), i=1,3) / 0.3904940000E-01, 0.1572130000E+03, 0.1051920000E+03 /
  data (epsData(10,i,26), i=1,3) / 0.4667860000E-01, 0.1750860000E+03, 0.8502610000E+02 /
  data (epsData(11,i,26), i=1,3) / 0.5498800000E-01, 0.1872080000E+03, 0.7022540000E+02 /
  data (epsData(12,i,26), i=1,3) / 0.6397770000E-01, 0.1957180000E+03, 0.5903350000E+02 /
  data (epsData(13,i,26), i=1,3) / 0.8399800000E-01, 0.2065800000E+03, 0.4349270000E+02 /
  data (epsData(14,i,26), i=1,3) / 0.1067390000E+00, 0.2134340000E+03, 0.3422840000E+02 /
  data (epsData(15,i,26), i=1,3) / 0.1754980000E+00, 0.2183280000E+03, 0.2291750000E+02 /
  data (epsData(16,i,26), i=1,3) / 0.3866320000E+00, 0.2223650000E+03, 0.1125630000E+02 /
  data (epsData(17,i,26), i=1,3) / 0.7743970000E+00, 0.2346230000E+03, 0.8026730000E+01 /
  data (epsData(18,i,26), i=1,3) / 0.1094080000E+01, 0.2545020000E+03, 0.8334430000E+01 /
  data (epsData(19,i,26), i=1,3) / 0.1295500000E+01, 0.2755880000E+03, 0.8980830000E+01 /
  data (epsData(20,i,26), i=1,3) / 0.1513920000E+01, 0.3126250000E+03, 0.1373680000E+02 /
  data (epsData(21,i,26), i=1,3) / 0.1606050000E+01, 0.3416640000E+03, 0.1649770000E+02 /
  data (epsData(22,i,26), i=1,3) / 0.1749350000E+01, 0.3875920000E+03, 0.4862660000E+02 /
  data (epsData(23,i,26), i=1,3) / 0.1798470000E+01, 0.4071300000E+03, 0.5726830000E+02 /
  data (epsData(24,i,26), i=1,3) / 0.1848280000E+01, 0.4378510000E+03, 0.6503770000E+02 /
  data (epsData(25,i,26), i=1,3) / 0.1898770000E+01, 0.4997550000E+03, 0.7201590000E+02 /
  data (epsData(26,i,26), i=1,3) / 0.1949930000E+01, 0.5427740000E+03, 0.1944670000E+03 /
  data (epsData(27,i,26), i=1,3) / 0.2003060000E+01, 0.5083930000E+03, 0.3076140000E+03 /
  data (epsData(28,i,26), i=1,3) / 0.2122790000E+01, 0.3895190000E+03, 0.4014370000E+03 /
  data (epsData(29,i,26), i=1,3) / 0.2195920000E+01, 0.2994360000E+03, 0.4479560000E+03 /
  data (epsData(30,i,26), i=1,3) / 0.2229940000E+01, 0.2495090000E+03, 0.4372900000E+03 /
  data (epsData(31,i,26), i=1,3) / 0.2306130000E+01, 0.1855270000E+03, 0.4063930000E+03 /
  data (epsData(32,i,26), i=1,3) / 0.2416670000E+01, 0.1305170000E+03, 0.3667880000E+03 /
  data (epsData(33,i,26), i=1,3) / 0.2544230000E+01, 0.9341240000E+02, 0.3293080000E+03 /
  data (epsData(34,i,26), i=1,3) / 0.2751370000E+01, 0.5436920000E+02, 0.2839540000E+03 /
  data (epsData(35,i,26), i=1,3) / 0.2955460000E+01, 0.2982470000E+02, 0.2455960000E+03 /
  data (epsData(36,i,26), i=1,3) / 0.3184030000E+01, 0.1235280000E+02, 0.2071520000E+03 /
  data (epsData(37,i,26), i=1,3) / 0.3393220000E+01, 0.8373230000E+01, 0.1743710000E+03 /
  data (epsData(38,i,26), i=1,3) / 0.3572820000E+01, 0.1149380000E+02, 0.1496120000E+03 /
  data (epsData(39,i,26), i=1,3) / 0.3714660000E+01, 0.1850560000E+02, 0.1330760000E+03 /
  data (epsData(40,i,26), i=1,3) / 0.3913300000E+01, 0.3417820000E+02, 0.1145510000E+03 /
  data (epsData(41,i,26), i=1,3) / 0.4069420000E+01, 0.5336960000E+02, 0.1054030000E+03 /
  data (epsData(42,i,26), i=1,3) / 0.4122830000E+01, 0.6217150000E+02, 0.1028550000E+03 /
  data (epsData(43,i,26), i=1,3) / 0.4176910000E+01, 0.7481770000E+02, 0.1003720000E+03 /
  data (epsData(44,i,26), i=1,3) / 0.4231670000E+01, 0.9469790000E+02, 0.1010900000E+03 /
  data (epsData(45,i,26), i=1,3) / 0.4287120000E+01, 0.1089400000E+03, 0.1285490000E+03 /
  data (epsData(46,i,26), i=1,3) / 0.4343240000E+01, 0.1076070000E+03, 0.1514920000E+03 /
  data (epsData(47,i,26), i=1,3) / 0.4457530000E+01, 0.9230090000E+02, 0.1785950000E+03 /
  data (epsData(48,i,26), i=1,3) / 0.4515690000E+01, 0.7883300000E+02, 0.1912670000E+03 /
  data (epsData(49,i,26), i=1,3) / 0.4574530000E+01, 0.6418100000E+02, 0.1921840000E+03 /
  data (epsData(50,i,26), i=1,3) / 0.4755150000E+01, 0.3805650000E+02, 0.1883480000E+03 /
  data (epsData(51,i,26), i=1,3) / 0.5134750000E+01, 0.2624750000E+01, 0.1769940000E+03 /
  data (epsData(52,i,26), i=1,3) / 0.5469790000E+01, -0.1946010000E+02, 0.1578780000E+03 /
  data (epsData(53,i,26), i=1,3) / 0.5750070000E+01, -0.2980560000E+02, 0.1394990000E+03 /
  data (epsData(54,i,26), i=1,3) / 0.6041230000E+01, -0.3447150000E+02, 0.1224660000E+03 /
  data (epsData(55,i,26), i=1,3) / 0.6576950000E+01, -0.3608190000E+02, 0.9991570000E+02 /
  data (epsData(56,i,26), i=1,3) / 0.7314720000E+01, -0.3509290000E+02, 0.7622610000E+02 /
  data (epsData(57,i,26), i=1,3) / 0.7748400000E+01, -0.3139190000E+02, 0.6325930000E+02 /
  data (epsData(58,i,26), i=1,3) / 0.8107590000E+01, -0.2467700000E+02, 0.5560870000E+02 /
  data (epsData(59,i,26), i=1,3) / 0.8666780000E+01, -0.1374880000E+02, 0.4996130000E+02 /
  data (epsData(60,i,26), i=1,3) / 0.8858620000E+01, -0.7495350000E+01, 0.4922390000E+02 /
  data (epsData(61,i,26), i=1,3) / 0.8955560000E+01, -0.7544920000E+00, 0.4886320000E+02 /
  data (epsData(62,i,26), i=1,3) / 0.9053180000E+01, 0.1603060000E+01, 0.6322280000E+02 /
  data (epsData(63,i,26), i=1,3) / 0.9151480000E+01, -0.3516760000E+01, 0.6882820000E+02 /
  data (epsData(64,i,26), i=1,3) / 0.9250470000E+01, -0.1043730000E+02, 0.7258950000E+02 /
  data (epsData(65,i,26), i=1,3) / 0.9350130000E+01, -0.1699920000E+02, 0.7018270000E+02 /
  data (epsData(66,i,26), i=1,3) / 0.9450470000E+01, -0.2087470000E+02, 0.6668250000E+02 /
  data (epsData(67,i,26), i=1,3) / 0.9551490000E+01, -0.2284280000E+02, 0.6330200000E+02 /
  data (epsData(68,i,26), i=1,3) / 0.9962380000E+01, -0.2560110000E+02, 0.5434400000E+02 /
  data (epsData(69,i,26), i=1,3) / 0.1049130000E+02, -0.2515300000E+02, 0.4523920000E+02 /
  data (epsData(70,i,26), i=1,3) / 0.1103720000E+02, -0.2223890000E+02, 0.3881010000E+02 /
  data (epsData(71,i,26), i=1,3) / 0.1183010000E+02, -0.1802250000E+02, 0.3350560000E+02 /
  data (epsData(72,i,26), i=1,3) / 0.1265630000E+02, -0.1475720000E+02, 0.3035950000E+02 /
  data (epsData(73,i,26), i=1,3) / 0.1415020000E+02, -0.1135940000E+02, 0.2660190000E+02 /
  data (epsData(74,i,26), i=1,3) / 0.1601690000E+02, -0.9081600000E+01, 0.2354400000E+02 /
  data (epsData(75,i,26), i=1,3) / 0.1891490000E+02, -0.7613070000E+01, 0.1993040000E+02 /
  data (epsData(76,i,26), i=1,3) / 0.2377210000E+02, -0.6446560000E+01, 0.1519730000E+02 /
  data (epsData(77,i,26), i=1,3) / 0.2828750000E+02, -0.5540850000E+01, 0.1224650000E+02 /
  data (epsData(78,i,26), i=1,3) / 0.3343460000E+02, -0.4869130000E+01, 0.1001630000E+02 /
  data (epsData(79,i,26), i=1,3) / 0.4108940000E+02, -0.4345660000E+01, 0.7683920000E+01 /
  data (epsData(80,i,26), i=1,3) / 0.4932360000E+02, -0.3813670000E+01, 0.5798820000E+01 /
  data (epsData(81,i,26), i=1,3) / 0.5215600000E+02, -0.3442130000E+01, 0.5361750000E+01 /
  data (epsData(82,i,26), i=1,3) / 0.5348370000E+02, -0.3406260000E+01, 0.5411400000E+01 /
  data (epsData(83,i,26), i=1,3) / 0.6119130000E+02, -0.3527360000E+01, 0.4378920000E+01 /
  data (epsData(84,i,26), i=1,3) / 0.6813190000E+02, -0.3265280000E+01, 0.3420440000E+01 /
  data (epsData(85,i,26), i=1,3) / 0.7768040000E+02, -0.2791820000E+01, 0.2570460000E+01 /
  data (epsData(86,i,26), i=1,3) / 0.8951730000E+02, -0.2296360000E+01, 0.1903520000E+01 /
  data (epsData(87,i,26), i=1,3) / 0.9549090000E+02, -0.2139280000E+01, 0.1693940000E+01 /
  data (epsData(88,i,26), i=1,3) / 0.1089820000E+03, -0.1795830000E+01, 0.1261820000E+01 /
  data (epsData(89,i,26), i=1,3) / 0.1241990000E+03, -0.1492570000E+01, 0.9363240000E+00 /
  data (epsData(90,i,26), i=1,3) / 0.1414560000E+03, -0.1230160000E+01, 0.6884380000E+00 /
  data (epsData(91,i,26), i=1,3) / 0.1642160000E+03, -0.9773830000E+00, 0.4703490000E+00 /
  data (epsData(92,i,26), i=1,3) / 0.1882950000E+03, -0.7810050000E+00, 0.3268870000E+00 /
  data (epsData(93,i,26), i=1,3) / 0.2121260000E+03, -0.6360410000E+00, 0.2367870000E+00 /
  data (epsData(94,i,26), i=1,3) / 0.2367870000E+03, -0.5225700000E+00, 0.1734970000E+00 /
  data (epsData(95,i,26), i=1,3) / 0.2613230000E+03, -0.4349620000E+00, 0.1304170000E+00 /
  data (epsData(96,i,26), i=1,3) / 0.2903640000E+03, -0.3564050000E+00, 0.9557270000E-01 /
  data (epsData(97,i,26), i=1,3) / 0.3233760000E+03, -0.2892020000E+00, 0.6905110000E-01 /
  data (epsData(98,i,26), i=1,3) / 0.3601280000E+03, -0.2333250000E+00, 0.4973840000E-01 /
  data (epsData(99,i,26), i=1,3) / 0.3995800000E+03, -0.1885740000E+00, 0.3635100000E-01 /
  data (epsData(100,i,26), i=1,3) / 0.4479750000E+03, -0.1486460000E+00, 0.2551230000E-01 /
  data (epsData(101,i,26), i=1,3) / 0.4946930000E+03, -0.1194690000E+00, 0.1869340000E-01 /
  data (epsData(102,i,26), i=1,3) / 0.5529630000E+03, -0.9187440000E-01, 0.1296100000E-01 /
  data (epsData(103,i,26), i=1,3) / 0.6279030000E+03, -0.6413400000E-01, 0.8624980000E-02 /
  data (epsData(104,i,26), i=1,3) / 0.6741460000E+03, -0.4669470000E-01, 0.6835150000E-02 /
  data (epsData(105,i,26), i=1,3) / 0.6902860000E+03, -0.3812640000E-01, 0.6389500000E-02 /
  data (epsData(106,i,26), i=1,3) / 0.6984380000E+03, -0.3164560000E-01, 0.6212800000E-02 /
  data (epsData(107,i,26), i=1,3) / 0.7029910000E+03, -0.2603950000E-01, 0.6292540000E-02 /
  data (epsData(108,i,26), i=1,3) / 0.7048170000E+03, -0.2258410000E-01, 0.6404080000E-02 /
  data (epsData(109,i,26), i=1,3) / 0.7057310000E+03, -0.2011650000E-01, 0.6575700000E-02 /
  data (epsData(110,i,26), i=1,3) / 0.7066460000E+03, -0.1681780000E-01, 0.7535330000E-02 /
  data (epsData(111,i,26), i=1,3) / 0.7071060000E+03, -0.1526280000E-01, 0.9316860000E-02 /
  data (epsData(112,i,26), i=1,3) / 0.7090870000E+03, -0.1182630000E-01, 0.1552980000E-01 /
  data (epsData(113,i,26), i=1,3) / 0.7100000000E+03, -0.1147260000E-01, 0.1916100000E-01 /
  data (epsData(114,i,26), i=1,3) / 0.7115360000E+03, -0.1166170000E-01, 0.2395000000E-01 /
  data (epsData(115,i,26), i=1,3) / 0.7129990000E+03, -0.1362410000E-01, 0.2841170000E-01 /
  data (epsData(116,i,26), i=1,3) / 0.7149720000E+03, -0.1751920000E-01, 0.3158950000E-01 /
  data (epsData(117,i,26), i=1,3) / 0.7172170000E+03, -0.2063470000E-01, 0.3235530000E-01 /
  data (epsData(118,i,26), i=1,3) / 0.7194700000E+03, -0.2112630000E-01, 0.3244750000E-01 /
  data (epsData(119,i,26), i=1,3) / 0.7227170000E+03, -0.2120590000E-01, 0.3634770000E-01 /
  data (epsData(120,i,26), i=1,3) / 0.7262840000E+03, -0.2464940000E-01, 0.4101730000E-01 /
  data (epsData(121,i,26), i=1,3) / 0.7301980000E+03, -0.3022970000E-01, 0.4249130000E-01 /
  data (epsData(122,i,26), i=1,3) / 0.7385250000E+03, -0.3643100000E-01, 0.4116280000E-01 /
  data (epsData(123,i,26), i=1,3) / 0.7525880000E+03, -0.4197020000E-01, 0.3848710000E-01 /
  data (epsData(124,i,26), i=1,3) / 0.7762440000E+03, -0.4492680000E-01, 0.3403740000E-01 /
  data (epsData(125,i,26), i=1,3) / 0.8158080000E+03, -0.4500710000E-01, 0.2828110000E-01 /
  data (epsData(126,i,26), i=1,3) / 0.8403210000E+03, -0.4237190000E-01, 0.2614140000E-01 /
  data (epsData(127,i,26), i=1,3) / 0.8452890000E+03, -0.4169540000E-01, 0.2750280000E-01 /
  data (epsData(128,i,26), i=1,3) / 0.8521510000E+03, -0.4303300000E-01, 0.2713880000E-01 /
  data (epsData(129,i,26), i=1,3) / 0.9098540000E+03, -0.4204180000E-01, 0.2207950000E-01 /
  data (epsData(130,i,26), i=1,3) / 0.1021420000E+04, -0.3651140000E-01, 0.1509640000E-01 /
  data (epsData(131,i,26), i=1,3) / 0.1150940000E+04, -0.3031890000E-01, 0.1015740000E-01 /
  data (epsData(132,i,26), i=1,3) / 0.1290550000E+04, -0.2486280000E-01, 0.6893140000E-02 /
  data (epsData(133,i,26), i=1,3) / 0.1438180000E+04, -0.2040170000E-01, 0.4735680000E-02 /
  data (epsData(134,i,26), i=1,3) / 0.1603840000E+04, -0.1657520000E-01, 0.3229420000E-02 /
  data (epsData(135,i,26), i=1,3) / 0.1780800000E+04, -0.1352650000E-01, 0.2226860000E-02 /
  data (epsData(136,i,26), i=1,3) / 0.1981250000E+04, -0.1094750000E-01, 0.1518360000E-02 /
  data (epsData(137,i,26), i=1,3) / 0.2199780000E+04, -0.8881210000E-02, 0.1040530000E-02 /
  data (epsData(138,i,26), i=1,3) / 0.2444880000E+04, -0.7178280000E-02, 0.7110410000E-03 /
  data (epsData(139,i,26), i=1,3) / 0.2719910000E+04, -0.5784590000E-02, 0.4807980000E-03 /
  data (epsData(140,i,26), i=1,3) / 0.3027070000E+04, -0.4653400000E-02, 0.3238910000E-03 /
  data (epsData(141,i,26), i=1,3) / 0.3372320000E+04, -0.3733200000E-02, 0.2169410000E-03 /
  data (epsData(142,i,26), i=1,3) / 0.3758830000E+04, -0.2990320000E-02, 0.1447440000E-03 /
  data (epsData(143,i,26), i=1,3) / 0.4192000000E+04, -0.2391070000E-02, 0.9618390000E-04 /
  data (epsData(144,i,26), i=1,3) / 0.4677900000E+04, -0.1907870000E-02, 0.6365430000E-04 /
  data (epsData(145,i,26), i=1,3) / 0.5223460000E+04, -0.1517930000E-02, 0.4195160000E-04 /
  data (epsData(146,i,26), i=1,3) / 0.5840060000E+04, -0.1200230000E-02, 0.2746900000E-04 /
  data (epsData(147,i,26), i=1,3) / 0.6582550000E+04, -0.9203340000E-03, 0.1749900000E-04 /
  data (epsData(148,i,26), i=1,3) / 0.7026160000E+04, -0.7587880000E-03, 0.1362440000E-04 /
  data (epsData(149,i,26), i=1,3) / 0.7090270000E+04, -0.7048000000E-03, 0.1316390000E-04 /
  data (epsData(150,i,26), i=1,3) / 0.7105600000E+04, -0.6630630000E-03, 0.1636150000E-04 /
  data (epsData(151,i,26), i=1,3) / 0.7109010000E+04, -0.6346870000E-03, 0.2096670000E-04 /
  data (epsData(152,i,26), i=1,3) / 0.7110720000E+04, -0.6057970000E-03, 0.3247750000E-04 /
  data (epsData(153,i,26), i=1,3) / 0.7112000000E+04, -0.4303170000E-03, 0.8256570000E-04 /
  data (epsData(154,i,26), i=1,3) / 0.7112000000E+04, -0.5889040000E-03, 0.8278240000E-04 /
  data (epsData(155,i,26), i=1,3) / 0.7112660000E+04, -0.6070200000E-03, 0.1157230000E-03 /
  data (epsData(156,i,26), i=1,3) / 0.7113620000E+04, -0.6452420000E-03, 0.1231840000E-03 /
  data (epsData(157,i,26), i=1,3) / 0.7115680000E+04, -0.6726050000E-03, 0.1065050000E-03 /
  data (epsData(158,i,26), i=1,3) / 0.7120490000E+04, -0.6815120000E-03, 0.9466140000E-04 /
  data (epsData(159,i,26), i=1,3) / 0.7149490000E+04, -0.7124490000E-03, 0.9972380000E-04 /
  data (epsData(160,i,26), i=1,3) / 0.7265150000E+04, -0.7309620000E-03, 0.9703710000E-04 /
  data (epsData(161,i,26), i=1,3) / 0.7867170000E+04, -0.6646220000E-03, 0.7325310000E-04 /
  data (epsData(162,i,26), i=1,3) / 0.8955500000E+04, -0.5292740000E-03, 0.4638590000E-04 /
  data (epsData(163,i,26), i=1,3) / 0.1008220000E+05, -0.4231250000E-03, 0.3013710000E-04 /
  data (epsData(164,i,26), i=1,3) / 0.1130500000E+05, -0.3388070000E-03, 0.1979210000E-04 /
  data (epsData(165,i,26), i=1,3) / 0.1265520000E+05, -0.2713030000E-03, 0.1300170000E-04 /
  data (epsData(166,i,26), i=1,3) / 0.1415950000E+05, -0.2170550000E-03, 0.8509780000E-05 /
  data (epsData(167,i,26), i=1,3) / 0.1583720000E+05, -0.1735630000E-03, 0.5561320000E-05 /
  data (epsData(168,i,26), i=1,3) / 0.1771290000E+05, -0.1387000000E-03, 0.3618200000E-05 /
  data (epsData(169,i,26), i=1,3) / 0.1982090000E+05, -0.1106780000E-03, 0.2339690000E-05 /
  data (epsData(170,i,26), i=1,3) / 0.2218370000E+05, -0.8826610000E-04, 0.1507660000E-05 /
  data (epsData(171,i,26), i=1,3) / 0.2483180000E+05, -0.7036360000E-04, 0.9673220000E-06 /
  data (epsData(172,i,26), i=1,3) / 0.2780530000E+05, -0.5605320000E-04, 0.6175920000E-06 /
  data (epsData(173,i,26), i=1,3) / 0.3114650000E+05, -0.4462110000E-04, 0.3925970000E-06 /
  data (epsData(174,i,26), i=1,3) / 0.3489720000E+05, -0.3550670000E-04, 0.2485640000E-06 /
  data (epsData(175,i,26), i=1,3) / 0.3910430000E+05, -0.2824960000E-04, 0.1566940000E-06 /
  data (epsData(176,i,26), i=1,3) / 0.4382670000E+05, -0.2246970000E-04, 0.9851190000E-07 /
  data (epsData(177,i,26), i=1,3) / 0.4913080000E+05, -0.1786590000E-04, 0.6155890000E-07 /
  data (epsData(178,i,26), i=1,3) / 0.5508300000E+05, -0.1420350000E-04, 0.3844990000E-07 /
  data (epsData(179,i,26), i=1,3) / 0.6175660000E+05, -0.1129280000E-04, 0.2398220000E-07 /
  data (epsData(180,i,26), i=1,3) / 0.6925460000E+05, -0.8975440000E-05, 0.1479470000E-07 /
  data (epsData(181,i,26), i=1,3) / 0.7767340000E+05, -0.7132220000E-05, 0.9153070000E-08 /
  data (epsData(182,i,26), i=1,3) / 0.8712670000E+05, -0.5666740000E-05, 0.5598710000E-08 /
  data (epsData(183,i,26), i=1,3) / 0.9770590000E+05, -0.4505930000E-05, 0.3440350000E-08 /
  data (epsData(184,i,26), i=1,3) / 0.1000000000E+06, -0.4307880000E-05, 0.3120730000E-08 /
  data (epsData(1,i,27), i=1,3) / 0.2506580000E-02, -0.8700530000E+03, 0.2836570000E+04 /
  data (epsData(2,i,27), i=1,3) / 0.4693440000E-02, -0.7517100000E+03, 0.9952210000E+03 /
  data (epsData(3,i,27), i=1,3) / 0.7560590000E-02, -0.2995000000E+03, 0.4775860000E+03 /
  data (epsData(4,i,27), i=1,3) / 0.1110800000E-01, -0.1007980000E+03, 0.2751690000E+03 /
  data (epsData(5,i,27), i=1,3) / 0.1533570000E-01, -0.7071040000E+01, 0.1780770000E+03 /
  data (epsData(6,i,27), i=1,3) / 0.2024370000E-01, 0.4150870000E+02, 0.1246560000E+03 /
  data (epsData(7,i,27), i=1,3) / 0.2583200000E-01, 0.6874300000E+02, 0.9226460000E+02 /
  data (epsData(8,i,27), i=1,3) / 0.3210060000E-01, 0.8499980000E+02, 0.7117470000E+02 /
  data (epsData(9,i,27), i=1,3) / 0.3904940000E-01, 0.9521210000E+02, 0.5665700000E+02 /
  data (epsData(10,i,27), i=1,3) / 0.4667860000E-01, 0.1019190000E+03, 0.4622710000E+02 /
  data (epsData(11,i,27), i=1,3) / 0.5498800000E-01, 0.1065190000E+03, 0.3847340000E+02 /
  data (epsData(12,i,27), i=1,3) / 0.7364770000E-01, 0.1119530000E+03, 0.2817090000E+02 /
  data (epsData(13,i,27), i=1,3) / 0.1067390000E+00, 0.1156330000E+03, 0.1910030000E+02 /
  data (epsData(14,i,27), i=1,3) / 0.1754980000E+00, 0.1180330000E+03, 0.1143820000E+02 /
  data (epsData(15,i,27), i=1,3) / 0.3421210000E+00, 0.1194580000E+03, 0.6312030000E+01 /
  data (epsData(16,i,27), i=1,3) / 0.8748370000E+00, 0.1232320000E+03, 0.3135870000E+01 /
  data (epsData(17,i,27), i=1,3) / 0.1513920000E+01, 0.1335830000E+03, 0.2727030000E+01 /
  data (epsData(18,i,27), i=1,3) / 0.2001780000E+01, 0.1495680000E+03, 0.3169290000E+01 /
  data (epsData(19,i,27), i=1,3) / 0.2327150000E+01, 0.1691240000E+03, 0.4463730000E+01 /
  data (epsData(20,i,27), i=1,3) / 0.2499020000E+01, 0.1855930000E+03, 0.6281930000E+01 /
  data (epsData(21,i,27), i=1,3) / 0.2617000000E+01, 0.2040560000E+03, 0.9007780000E+01 /
  data (epsData(22,i,27), i=1,3) / 0.2799080000E+01, 0.2385010000E+03, 0.2636330000E+02 /
  data (epsData(23,i,27), i=1,3) / 0.2861130000E+01, 0.2613440000E+03, 0.3127540000E+02 /
  data (epsData(24,i,27), i=1,3) / 0.2923860000E+01, 0.2894880000E+03, 0.5962370000E+02 /
  data (epsData(25,i,27), i=1,3) / 0.2987270000E+01, 0.2969900000E+03, 0.1234420000E+03 /
  data (epsData(26,i,27), i=1,3) / 0.3012250000E+01, 0.2843900000E+03, 0.1409840000E+03 /
  data (epsData(27,i,27), i=1,3) / 0.3164630000E+01, 0.2262430000E+03, 0.2009510000E+03 /
  data (epsData(28,i,27), i=1,3) / 0.3195920000E+01, 0.2069260000E+03, 0.2117810000E+03 /
  data (epsData(29,i,27), i=1,3) / 0.3247960000E+01, 0.1768520000E+03, 0.2071330000E+03 /
  data (epsData(30,i,27), i=1,3) / 0.3370410000E+01, 0.1407770000E+03, 0.1928980000E+03 /
  data (epsData(31,i,27), i=1,3) / 0.3544230000E+01, 0.1135500000E+03, 0.1756370000E+03 /
  data (epsData(32,i,27), i=1,3) / 0.3850360000E+01, 0.8962410000E+02, 0.1513750000E+03 /
  data (epsData(33,i,27), i=1,3) / 0.3991850000E+01, 0.8862230000E+02, 0.1398730000E+03 /
  data (epsData(34,i,27), i=1,3) / 0.4144240000E+01, 0.9426550000E+02, 0.1298000000E+03 /
  data (epsData(35,i,27), i=1,3) / 0.4224510000E+01, 0.1073870000E+03, 0.1236620000E+03 /
  data (epsData(36,i,27), i=1,3) / 0.4265670000E+01, 0.1205600000E+03, 0.1256550000E+03 /
  data (epsData(37,i,27), i=1,3) / 0.4307510000E+01, 0.1297430000E+03, 0.1438040000E+03 /
  data (epsData(38,i,27), i=1,3) / 0.4350020000E+01, 0.1282140000E+03, 0.1615120000E+03 /
  data (epsData(39,i,27), i=1,3) / 0.4481660000E+01, 0.1078370000E+03, 0.1890110000E+03 /
  data (epsData(40,i,27), i=1,3) / 0.4526900000E+01, 0.9743950000E+02, 0.1972900000E+03 /
  data (epsData(41,i,27), i=1,3) / 0.4572820000E+01, 0.8446290000E+02, 0.2016220000E+03 /
  data (epsData(42,i,27), i=1,3) / 0.4714660000E+01, 0.5680370000E+02, 0.1952170000E+03 /
  data (epsData(43,i,27), i=1,3) / 0.4913300000E+01, 0.3147520000E+02, 0.1866950000E+03 /
  data (epsData(44,i,27), i=1,3) / 0.5231670000E+01, 0.1681880000E+01, 0.1682440000E+03 /
  data (epsData(45,i,27), i=1,3) / 0.5574530000E+01, -0.1632290000E+02, 0.1433500000E+03 /
  data (epsData(46,i,27), i=1,3) / 0.5878960000E+01, -0.2265030000E+02, 0.1244720000E+03 /
  data (epsData(47,i,27), i=1,3) / 0.6333730000E+01, -0.2681360000E+02, 0.1050820000E+03 /
  data (epsData(48,i,27), i=1,3) / 0.7041230000E+01, -0.3165430000E+02, 0.8051880000E+02 /
  data (epsData(49,i,27), i=1,3) / 0.7656200000E+01, -0.3013050000E+02, 0.6112500000E+02 /
  data (epsData(50,i,27), i=1,3) / 0.7898050000E+01, -0.2646770000E+02, 0.5435070000E+02 /
  data (epsData(51,i,27), i=1,3) / 0.8314720000E+01, -0.2010260000E+02, 0.4826450000E+02 /
  data (epsData(52,i,27), i=1,3) / 0.8837180000E+01, -0.1323630000E+02, 0.4289330000E+02 /
  data (epsData(53,i,27), i=1,3) / 0.9016770000E+01, -0.9753110000E+01, 0.4133280000E+02 /
  data (epsData(54,i,27), i=1,3) / 0.9199090000E+01, -0.4365670000E+01, 0.4073440000E+02 /
  data (epsData(55,i,27), i=1,3) / 0.9291270000E+01, 0.1106950000E+01, 0.4048570000E+02 /
  data (epsData(56,i,27), i=1,3) / 0.9384120000E+01, 0.4749030000E+01, 0.4989150000E+02 /
  data (epsData(57,i,27), i=1,3) / 0.9477660000E+01, 0.1973350000E+01, 0.5844410000E+02 /
  data (epsData(58,i,27), i=1,3) / 0.9571880000E+01, -0.3984250000E+01, 0.6214690000E+02 /
  data (epsData(59,i,27), i=1,3) / 0.9666780000E+01, -0.1032510000E+02, 0.6284320000E+02 /
  data (epsData(60,i,27), i=1,3) / 0.9762360000E+01, -0.1519350000E+02, 0.5988250000E+02 /
  data (epsData(61,i,27), i=1,3) / 0.9955560000E+01, -0.1952170000E+02, 0.5445120000E+02 /
  data (epsData(62,i,27), i=1,3) / 0.1035010000E+02, -0.2276110000E+02, 0.4653370000E+02 /
  data (epsData(63,i,27), i=1,3) / 0.1075560000E+02, -0.2292070000E+02, 0.3934840000E+02 /
  data (epsData(64,i,27), i=1,3) / 0.1127770000E+02, -0.2042340000E+02, 0.3283720000E+02 /
  data (epsData(65,i,27), i=1,3) / 0.1192670000E+02, -0.1694920000E+02, 0.2811490000E+02 /
  data (epsData(66,i,27), i=1,3) / 0.1271480000E+02, -0.1351840000E+02, 0.2483590000E+02 /
  data (epsData(67,i,27), i=1,3) / 0.1402060000E+02, -0.1008430000E+02, 0.2166940000E+02 /
  data (epsData(68,i,27), i=1,3) / 0.1540870000E+02, -0.7760790000E+01, 0.1960310000E+02 /
  data (epsData(69,i,27), i=1,3) / 0.1729450000E+02, -0.6083900000E+01, 0.1767920000E+02 /
  data (epsData(70,i,27), i=1,3) / 0.2052720000E+02, -0.4940320000E+01, 0.1509890000E+02 /
  data (epsData(71,i,27), i=1,3) / 0.2564120000E+02, -0.4155190000E+01, 0.1184050000E+02 /
  data (epsData(72,i,27), i=1,3) / 0.3043450000E+02, -0.3619810000E+01, 0.9779570000E+01 /
  data (epsData(73,i,27), i=1,3) / 0.3653660000E+02, -0.3270160000E+01, 0.8002430000E+01 /
  data (epsData(74,i,27), i=1,3) / 0.4683780000E+02, -0.3040860000E+01, 0.5862220000E+01 /
  data (epsData(75,i,27), i=1,3) / 0.5500120000E+02, -0.2758670000E+01, 0.4564980000E+01 /
  data (epsData(76,i,27), i=1,3) / 0.5879180000E+02, -0.2475100000E+01, 0.4155360000E+01 /
  data (epsData(77,i,27), i=1,3) / 0.6096670000E+02, -0.2444090000E+01, 0.4123800000E+01 /
  data (epsData(78,i,27), i=1,3) / 0.6860760000E+02, -0.2564820000E+01, 0.3451120000E+01 /
  data (epsData(79,i,27), i=1,3) / 0.7606490000E+02, -0.2408980000E+01, 0.2742020000E+01 /
  data (epsData(80,i,27), i=1,3) / 0.8714160000E+02, -0.2077870000E+01, 0.2054490000E+01 /
  data (epsData(81,i,27), i=1,3) / 0.9968440000E+02, -0.1732860000E+01, 0.1555240000E+01 /
  data (epsData(82,i,27), i=1,3) / 0.1199450000E+03, -0.1396160000E+01, 0.1064930000E+01 /
  data (epsData(83,i,27), i=1,3) / 0.1375940000E+03, -0.1162470000E+01, 0.7848570000E+00 /
  data (epsData(84,i,27), i=1,3) / 0.1576210000E+03, -0.9627460000E+00, 0.5700510000E+00 /
  data (epsData(85,i,27), i=1,3) / 0.1755000000E+03, -0.8172620000E+00, 0.4360680000E+00 /
  data (epsData(86,i,27), i=1,3) / 0.1972690000E+03, -0.6782000000E+00, 0.3212070000E+00 /
  data (epsData(87,i,27), i=1,3) / 0.2242710000E+03, -0.5480180000E+00, 0.2269040000E+00 /
  data (epsData(88,i,27), i=1,3) / 0.2541490000E+03, -0.4401360000E+00, 0.1591460000E+00 /
  data (epsData(89,i,27), i=1,3) / 0.2867630000E+03, -0.3534460000E+00, 0.1124300000E+00 /
  data (epsData(90,i,27), i=1,3) / 0.3219740000E+03, -0.2842810000E+00, 0.7968930000E-01 /
  data (epsData(91,i,27), i=1,3) / 0.3596450000E+03, -0.2293210000E+00, 0.5706720000E-01 /
  data (epsData(92,i,27), i=1,3) / 0.3990050000E+03, -0.1863860000E+00, 0.4178850000E-01 /
  data (epsData(93,i,27), i=1,3) / 0.4473020000E+03, -0.1480430000E+00, 0.2937690000E-01 /
  data (epsData(94,i,27), i=1,3) / 0.4926270000E+03, -0.1206320000E+00, 0.2175900000E-01 /
  data (epsData(95,i,27), i=1,3) / 0.5500700000E+03, -0.9466490000E-01, 0.1515940000E-01 /
  data (epsData(96,i,27), i=1,3) / 0.6139450000E+03, -0.7281410000E-01, 0.1070040000E-01 /
  data (epsData(97,i,27), i=1,3) / 0.6909920000E+03, -0.5185410000E-01, 0.7280080000E-02 /
  data (epsData(98,i,27), i=1,3) / 0.7422330000E+03, -0.3755730000E-01, 0.5732630000E-02 /
  data (epsData(99,i,27), i=1,3) / 0.7601030000E+03, -0.3044790000E-01, 0.5368170000E-02 /
  data (epsData(100,i,27), i=1,3) / 0.7686530000E+03, -0.2543920000E-01, 0.5238490000E-02 /
  data (epsData(101,i,27), i=1,3) / 0.7734280000E+03, -0.2116730000E-01, 0.5294190000E-02 /
  data (epsData(102,i,27), i=1,3) / 0.7753420000E+03, -0.1874820000E-01, 0.5355920000E-02 /
  data (epsData(103,i,27), i=1,3) / 0.7763000000E+03, -0.1714460000E-01, 0.5433640000E-02 /
  data (epsData(104,i,27), i=1,3) / 0.7772590000E+03, -0.1501320000E-01, 0.5633910000E-02 /
  data (epsData(105,i,27), i=1,3) / 0.7781770000E+03, -0.1245640000E-01, 0.6883030000E-02 /
  data (epsData(106,i,27), i=1,3) / 0.7789170000E+03, -0.1116740000E-01, 0.8730120000E-02 /
  data (epsData(107,i,27), i=1,3) / 0.7801690000E+03, -0.9143570000E-02, 0.1160270000E-01 /
  data (epsData(108,i,27), i=1,3) / 0.7811700000E+03, -0.8453280000E-02, 0.1472920000E-01 /
  data (epsData(109,i,27), i=1,3) / 0.7827560000E+03, -0.8395530000E-02, 0.1889510000E-01 /
  data (epsData(110,i,27), i=1,3) / 0.7841170000E+03, -0.9702650000E-02, 0.2243280000E-01 /
  data (epsData(111,i,27), i=1,3) / 0.7859590000E+03, -0.1266020000E-01, 0.2530640000E-01 /
  data (epsData(112,i,27), i=1,3) / 0.7881630000E+03, -0.1561180000E-01, 0.2633100000E-01 /
  data (epsData(113,i,27), i=1,3) / 0.7913010000E+03, -0.1725580000E-01, 0.2626520000E-01 /
  data (epsData(114,i,27), i=1,3) / 0.7934670000E+03, -0.1659920000E-01, 0.2693970000E-01 /
  data (epsData(115,i,27), i=1,3) / 0.7963350000E+03, -0.1658550000E-01, 0.2998580000E-01 /
  data (epsData(116,i,27), i=1,3) / 0.7995840000E+03, -0.1896750000E-01, 0.3364470000E-01 /
  data (epsData(117,i,27), i=1,3) / 0.8034980000E+03, -0.2341540000E-01, 0.3505230000E-01 /
  data (epsData(118,i,27), i=1,3) / 0.8121450000E+03, -0.2868170000E-01, 0.3424420000E-01 /
  data (epsData(119,i,27), i=1,3) / 0.8258880000E+03, -0.3316690000E-01, 0.3219470000E-01 /
  data (epsData(120,i,27), i=1,3) / 0.8512180000E+03, -0.3610190000E-01, 0.2862930000E-01 /
  data (epsData(121,i,27), i=1,3) / 0.8909230000E+03, -0.3655520000E-01, 0.2407630000E-01 /
  data (epsData(122,i,27), i=1,3) / 0.9197690000E+03, -0.3454030000E-01, 0.2196390000E-01 /
  data (epsData(123,i,27), i=1,3) / 0.9254700000E+03, -0.3377200000E-01, 0.2300260000E-01 /
  data (epsData(124,i,27), i=1,3) / 0.9319580000E+03, -0.3492190000E-01, 0.2292320000E-01 /
  data (epsData(125,i,27), i=1,3) / 0.9909520000E+03, -0.3453740000E-01, 0.1891570000E-01 /
  data (epsData(126,i,27), i=1,3) / 0.1103400000E+04, -0.3053250000E-01, 0.1323440000E-01 /
  data (epsData(127,i,27), i=1,3) / 0.1246470000E+04, -0.2534590000E-01, 0.8814770000E-02 /
  data (epsData(128,i,27), i=1,3) / 0.1396320000E+04, -0.2087880000E-01, 0.6004000000E-02 /
  data (epsData(129,i,27), i=1,3) / 0.1560740000E+04, -0.1703980000E-01, 0.4076920000E-02 /
  data (epsData(130,i,27), i=1,3) / 0.1733910000E+04, -0.1396490000E-01, 0.2817270000E-02 /
  data (epsData(131,i,27), i=1,3) / 0.1930240000E+04, -0.1133350000E-01, 0.1924880000E-02 /
  data (epsData(132,i,27), i=1,3) / 0.2142740000E+04, -0.9221530000E-02, 0.1320860000E-02 /
  data (epsData(133,i,27), i=1,3) / 0.2381220000E+04, -0.7464840000E-02, 0.9045040000E-03 /
  data (epsData(134,i,27), i=1,3) / 0.2651730000E+04, -0.6011700000E-02, 0.6121790000E-03 /
  data (epsData(135,i,27), i=1,3) / 0.2950250000E+04, -0.4843090000E-02, 0.4137710000E-03 /
  data (epsData(136,i,27), i=1,3) / 0.3283290000E+04, -0.3895900000E-02, 0.2786420000E-03 /
  data (epsData(137,i,27), i=1,3) / 0.3657080000E+04, -0.3126400000E-02, 0.1866410000E-03 /
  data (epsData(138,i,27), i=1,3) / 0.4074890000E+04, -0.2505620000E-02, 0.1246100000E-03 /
  data (epsData(139,i,27), i=1,3) / 0.4542330000E+04, -0.2005180000E-02, 0.8291710000E-04 /
  data (epsData(140,i,27), i=1,3) / 0.5068100000E+04, -0.1600240000E-02, 0.5487220000E-04 /
  data (epsData(141,i,27), i=1,3) / 0.5658890000E+04, -0.1273120000E-02, 0.3615530000E-04 /
  data (epsData(142,i,27), i=1,3) / 0.6324940000E+04, -0.1007120000E-02, 0.2378000000E-04 /
  data (epsData(143,i,27), i=1,3) / 0.7135380000E+04, -0.7705970000E-03, 0.1502500000E-04 /
  data (epsData(144,i,27), i=1,3) / 0.7614850000E+04, -0.6354650000E-03, 0.1169910000E-04 /
  data (epsData(145,i,27), i=1,3) / 0.7685540000E+04, -0.5894540000E-03, 0.1129770000E-04 /
  data (epsData(146,i,27), i=1,3) / 0.7701220000E+04, -0.5580380000E-03, 0.1365090000E-04 /
  data (epsData(147,i,27), i=1,3) / 0.7704700000E+04, -0.5400880000E-03, 0.1603980000E-04 /
  data (epsData(148,i,27), i=1,3) / 0.7706450000E+04, -0.5236490000E-03, 0.1991770000E-04 /
  data (epsData(149,i,27), i=1,3) / 0.7708200000E+04, -0.4981830000E-03, 0.3783420000E-04 /
  data (epsData(150,i,27), i=1,3) / 0.7709000000E+04, -0.3627120000E-03, 0.7010270000E-04 /
  data (epsData(151,i,27), i=1,3) / 0.7709000000E+04, -0.4903050000E-03, 0.7027250000E-04 /
  data (epsData(152,i,27), i=1,3) / 0.7709690000E+04, -0.5073100000E-03, 0.9732360000E-04 /
  data (epsData(153,i,27), i=1,3) / 0.7710670000E+04, -0.5388340000E-03, 0.1020170000E-03 /
  data (epsData(154,i,27), i=1,3) / 0.7712970000E+04, -0.5613550000E-03, 0.8784450000E-04 /
  data (epsData(155,i,27), i=1,3) / 0.7719540000E+04, -0.5696390000E-03, 0.8168940000E-04 /
  data (epsData(156,i,27), i=1,3) / 0.7748090000E+04, -0.5947200000E-03, 0.8411420000E-04 /
  data (epsData(157,i,27), i=1,3) / 0.7861240000E+04, -0.6116590000E-03, 0.8281700000E-04 /
  data (epsData(158,i,27), i=1,3) / 0.8417290000E+04, -0.5680990000E-03, 0.6503060000E-04 /
  data (epsData(159,i,27), i=1,3) / 0.9613170000E+04, -0.4514980000E-03, 0.4068800000E-04 /
  data (epsData(160,i,27), i=1,3) / 0.1082370000E+05, -0.3611640000E-03, 0.2646630000E-04 /
  data (epsData(161,i,27), i=1,3) / 0.1213680000E+05, -0.2892950000E-03, 0.1735290000E-04 /
  data (epsData(162,i,27), i=1,3) / 0.1358660000E+05, -0.2316850000E-03, 0.1138310000E-04 /
  data (epsData(163,i,27), i=1,3) / 0.1519980000E+05, -0.1854150000E-03, 0.7465320000E-05 /
  data (epsData(164,i,27), i=1,3) / 0.1699770000E+05, -0.1483250000E-03, 0.4878210000E-05 /
  data (epsData(165,i,27), i=1,3) / 0.1901230000E+05, -0.1185170000E-03, 0.3172040000E-05 /
  data (epsData(166,i,27), i=1,3) / 0.2127160000E+05, -0.9460150000E-04, 0.2053140000E-05 /
  data (epsData(167,i,27), i=1,3) / 0.2380770000E+05, -0.7544170000E-04, 0.1322370000E-05 /
  data (epsData(168,i,27), i=1,3) / 0.2665240000E+05, -0.6012620000E-04, 0.8477030000E-06 /
  data (epsData(169,i,27), i=1,3) / 0.2984360000E+05, -0.4789770000E-04, 0.5411690000E-06 /
  data (epsData(170,i,27), i=1,3) / 0.3342490000E+05, -0.3813930000E-04, 0.3440610000E-06 /
  data (epsData(171,i,27), i=1,3) / 0.3744610000E+05, -0.3035440000E-04, 0.2177710000E-06 /
  data (epsData(172,i,27), i=1,3) / 0.4195700000E+05, -0.2415390000E-04, 0.1373250000E-06 /
  data (epsData(173,i,27), i=1,3) / 0.4702000000E+05, -0.1921470000E-04, 0.8634630000E-07 /
  data (epsData(174,i,27), i=1,3) / 0.5270550000E+05, -0.1528040000E-04, 0.5394600000E-07 /
  data (epsData(175,i,27), i=1,3) / 0.5909210000E+05, -0.1214730000E-04, 0.3364940000E-07 /
  data (epsData(176,i,27), i=1,3) / 0.6626560000E+05, -0.9653870000E-05, 0.2087990000E-07 /
  data (epsData(177,i,27), i=1,3) / 0.7431250000E+05, -0.7672450000E-05, 0.1294600000E-07 /
  data (epsData(178,i,27), i=1,3) / 0.8334300000E+05, -0.6097450000E-05, 0.7981590000E-08 /
  data (epsData(179,i,27), i=1,3) / 0.9347560000E+05, -0.4846100000E-05, 0.4883620000E-08 /
  data (epsData(180,i,27), i=1,3) / 0.1000000000E+06, -0.4242840000E-05, 0.3674290000E-08 /
  data (epsData(1,i,28), i=1,3) / 0.2506580000E-02, 0.7964610000E+02, 0.8823440000E+03 /
  data (epsData(2,i,28), i=1,3) / 0.4693440000E-02, -0.3466640000E+02, 0.3848560000E+03 /
  data (epsData(3,i,28), i=1,3) / 0.7560590000E-02, 0.2118350000E+02, 0.2156050000E+03 /
  data (epsData(4,i,28), i=1,3) / 0.1110800000E-01, 0.4936520000E+02, 0.1384570000E+03 /
  data (epsData(5,i,28), i=1,3) / 0.1533570000E-01, 0.6334790000E+02, 0.9675880000E+02 /
  data (epsData(6,i,28), i=1,3) / 0.2024370000E-01, 0.7080030000E+02, 0.7159710000E+02 /
  data (epsData(7,i,28), i=1,3) / 0.2583200000E-01, 0.7505510000E+02, 0.5520620000E+02 /
  data (epsData(8,i,28), i=1,3) / 0.3210060000E-01, 0.7762970000E+02, 0.4391210000E+02 /
  data (epsData(9,i,28), i=1,3) / 0.3904940000E-01, 0.7926560000E+02, 0.3578830000E+02 /
  data (epsData(10,i,28), i=1,3) / 0.4667860000E-01, 0.8034970000E+02, 0.2974360000E+02 /
  data (epsData(11,i,28), i=1,3) / 0.6397770000E-01, 0.8163030000E+02, 0.2150370000E+02 /
  data (epsData(12,i,28), i=1,3) / 0.9502860000E-01, 0.8258320000E+02, 0.1447840000E+02 /
  data (epsData(13,i,28), i=1,3) / 0.1603860000E+00, 0.8295530000E+02, 0.8698270000E+01 /
  data (epsData(14,i,28), i=1,3) / 0.3421210000E+00, 0.8335130000E+02, 0.4189150000E+01 /
  data (epsData(15,i,28), i=1,3) / 0.9451980000E+00, 0.8517710000E+02, 0.1857790000E+01 /
  data (epsData(16,i,28), i=1,3) / 0.1798470000E+01, 0.9180030000E+02, 0.1423680000E+01 /
  data (epsData(17,i,28), i=1,3) / 0.2499020000E+01, 0.1030820000E+03, 0.1613470000E+01 /
  data (epsData(18,i,28), i=1,3) / 0.2987270000E+01, 0.1173920000E+03, 0.2095080000E+01 /
  data (epsData(19,i,28), i=1,3) / 0.3314540000E+01, 0.1337150000E+03, 0.2963580000E+01 /
  data (epsData(20,i,28), i=1,3) / 0.3519060000E+01, 0.1498800000E+03, 0.4432680000E+01 /
  data (epsData(21,i,28), i=1,3) / 0.3588600000E+01, 0.1580910000E+03, 0.4868790000E+01 /
  data (epsData(22,i,28), i=1,3) / 0.3801290000E+01, 0.1889700000E+03, 0.1664340000E+02 /
  data (epsData(23,i,28), i=1,3) / 0.3873540000E+01, 0.2085580000E+03, 0.2031060000E+02 /
  data (epsData(24,i,28), i=1,3) / 0.3946480000E+01, 0.2281300000E+03, 0.4827290000E+02 /
  data (epsData(25,i,28), i=1,3) / 0.4005440000E+01, 0.2290160000E+03, 0.7935350000E+02 /
  data (epsData(26,i,28), i=1,3) / 0.4179940000E+01, 0.2088610000E+03, 0.1247580000E+03 /
  data (epsData(27,i,28), i=1,3) / 0.4229940000E+01, 0.1987140000E+03, 0.1286150000E+03 /
  data (epsData(28,i,28), i=1,3) / 0.4266670000E+01, 0.2000890000E+03, 0.1275710000E+03 /
  data (epsData(29,i,28), i=1,3) / 0.4286060000E+01, 0.2048270000E+03, 0.1290110000E+03 /
  data (epsData(30,i,28), i=1,3) / 0.4348310000E+01, 0.2071680000E+03, 0.1567900000E+03 /
  data (epsData(31,i,28), i=1,3) / 0.4416670000E+01, 0.1911080000E+03, 0.1794830000E+03 /
  data (epsData(32,i,28), i=1,3) / 0.4571780000E+01, 0.1478980000E+03, 0.2131160000E+03 /
  data (epsData(33,i,28), i=1,3) / 0.4658520000E+01, 0.1191010000E+03, 0.2139310000E+03 /
  data (epsData(34,i,28), i=1,3) / 0.4884710000E+01, 0.6927550000E+02, 0.2081320000E+03 /
  data (epsData(35,i,28), i=1,3) / 0.5184030000E+01, 0.2330420000E+02, 0.1873360000E+03 /
  data (epsData(36,i,28), i=1,3) / 0.5350020000E+01, 0.6050310000E+01, 0.1703230000E+03 /
  data (epsData(37,i,28), i=1,3) / 0.5619420000E+01, -0.9977430000E+01, 0.1460600000E+03 /
  data (epsData(38,i,28), i=1,3) / 0.5862620000E+01, -0.1653350000E+02, 0.1281580000E+03 /
  data (epsData(39,i,28), i=1,3) / 0.6176910000E+01, -0.2102650000E+02, 0.1109160000E+03 /
  data (epsData(40,i,28), i=1,3) / 0.6694260000E+01, -0.2532780000E+02, 0.9046400000E+02 /
  data (epsData(41,i,28), i=1,3) / 0.7538840000E+01, -0.2881260000E+02, 0.6394130000E+02 /
  data (epsData(42,i,28), i=1,3) / 0.7821840000E+01, -0.2668850000E+02, 0.5457360000E+02 /
  data (epsData(43,i,28), i=1,3) / 0.8041230000E+01, -0.2311800000E+02, 0.5010320000E+02 /
  data (epsData(44,i,28), i=1,3) / 0.8576950000E+01, -0.1759230000E+02, 0.4323470000E+02 /
  data (epsData(45,i,28), i=1,3) / 0.9062670000E+01, -0.1254660000E+02, 0.3813870000E+02 /
  data (epsData(46,i,28), i=1,3) / 0.9314720000E+01, -0.8773040000E+01, 0.3605870000E+02 /
  data (epsData(47,i,28), i=1,3) / 0.9486150000E+01, -0.4822190000E+01, 0.3528690000E+02 /
  data (epsData(48,i,28), i=1,3) / 0.9572890000E+01, -0.1840760000E+01, 0.3505810000E+02 /
  data (epsData(49,i,28), i=1,3) / 0.9660300000E+01, 0.3735870000E+01, 0.3484610000E+02 /
  data (epsData(50,i,28), i=1,3) / 0.9748400000E+01, 0.6750380000E+01, 0.4553310000E+02 /
  data (epsData(51,i,28), i=1,3) / 0.9837180000E+01, 0.3482340000E+01, 0.5258340000E+02 /
  data (epsData(52,i,28), i=1,3) / 0.9926630000E+01, -0.1976910000E+01, 0.5618920000E+02 /
  data (epsData(53,i,28), i=1,3) / 0.1001680000E+02, -0.7952760000E+01, 0.5718020000E+02 /
  data (epsData(54,i,28), i=1,3) / 0.1010760000E+02, -0.1270450000E+02, 0.5467760000E+02 /
  data (epsData(55,i,28), i=1,3) / 0.1029130000E+02, -0.1736690000E+02, 0.4992790000E+02 /
  data (epsData(56,i,28), i=1,3) / 0.1066680000E+02, -0.2143950000E+02, 0.4244140000E+02 /
  data (epsData(57,i,28), i=1,3) / 0.1105320000E+02, -0.2207280000E+02, 0.3529290000E+02 /
  data (epsData(58,i,28), i=1,3) / 0.1145050000E+02, -0.2044700000E+02, 0.2978110000E+02 /
  data (epsData(59,i,28), i=1,3) / 0.1206680000E+02, -0.1715820000E+02, 0.2465300000E+02 /
  data (epsData(60,i,28), i=1,3) / 0.1281680000E+02, -0.1374190000E+02, 0.2112620000E+02 /
  data (epsData(61,i,28), i=1,3) / 0.1383010000E+02, -0.1064120000E+02, 0.1844310000E+02 /
  data (epsData(62,i,28), i=1,3) / 0.1489850000E+02, -0.8238630000E+01, 0.1659420000E+02 /
  data (epsData(63,i,28), i=1,3) / 0.1627910000E+02, -0.6219670000E+01, 0.1512550000E+02 /
  data (epsData(64,i,28), i=1,3) / 0.1815530000E+02, -0.4686120000E+01, 0.1379170000E+02 /
  data (epsData(65,i,28), i=1,3) / 0.2106700000E+02, -0.3641630000E+01, 0.1218080000E+02 /
  data (epsData(66,i,28), i=1,3) / 0.2611770000E+02, -0.2938280000E+01, 0.9873300000E+01 /
  data (epsData(67,i,28), i=1,3) / 0.3124160000E+02, -0.2519110000E+01, 0.8232380000E+01 /
  data (epsData(68,i,28), i=1,3) / 0.3775060000E+02, -0.2304360000E+01, 0.6818410000E+01 /
  data (epsData(69,i,28), i=1,3) / 0.4956400000E+02, -0.2254880000E+01, 0.4966220000E+01 /
  data (epsData(70,i,28), i=1,3) / 0.5869520000E+02, -0.2111020000E+01, 0.3865740000E+01 /
  data (epsData(71,i,28), i=1,3) / 0.6486140000E+02, -0.1883440000E+01, 0.3314720000E+01 /
  data (epsData(72,i,28), i=1,3) / 0.6796330000E+02, -0.1786770000E+01, 0.3236980000E+01 /
  data (epsData(73,i,28), i=1,3) / 0.7515150000E+02, -0.1912090000E+01, 0.2858390000E+01 /
  data (epsData(74,i,28), i=1,3) / 0.8300030000E+02, -0.1856060000E+01, 0.2306630000E+01 /
  data (epsData(75,i,28), i=1,3) / 0.9582220000E+02, -0.1613410000E+01, 0.1702600000E+01 /
  data (epsData(76,i,28), i=1,3) / 0.1091930000E+03, -0.1358120000E+01, 0.1303760000E+01 /
  data (epsData(77,i,28), i=1,3) / 0.1313840000E+03, -0.1106350000E+01, 0.9047500000E+00 /
  data (epsData(78,i,28), i=1,3) / 0.1505840000E+03, -0.9333450000E+00, 0.6733640000E+00 /
  data (epsData(79,i,28), i=1,3) / 0.1719500000E+03, -0.7788020000E+00, 0.4919210000E+00 /
  data (epsData(80,i,28), i=1,3) / 0.1972020000E+03, -0.6364540000E+00, 0.3496470000E+00 /
  data (epsData(81,i,28), i=1,3) / 0.2256200000E+03, -0.5154980000E+00, 0.2458950000E+00 /
  data (epsData(82,i,28), i=1,3) / 0.2558420000E+03, -0.4176830000E+00, 0.1737870000E+00 /
  data (epsData(83,i,28), i=1,3) / 0.2886060000E+03, -0.3383030000E+00, 0.1236540000E+00 /
  data (epsData(84,i,28), i=1,3) / 0.3236250000E+03, -0.2745750000E+00, 0.8838570000E-01 /
  data (epsData(85,i,28), i=1,3) / 0.3612070000E+03, -0.2230550000E+00, 0.6367670000E-01 /
  data (epsData(86,i,28), i=1,3) / 0.4005840000E+03, -0.1823180000E+00, 0.4683510000E-01 /
  data (epsData(87,i,28), i=1,3) / 0.4502720000E+03, -0.1449810000E+00, 0.3264700000E-01 /
  data (epsData(88,i,28), i=1,3) / 0.4944380000E+03, -0.1194530000E+00, 0.2448110000E-01 /
  data (epsData(89,i,28), i=1,3) / 0.5544140000E+03, -0.9371020000E-01, 0.1684620000E-01 /
  data (epsData(90,i,28), i=1,3) / 0.6187450000E+03, -0.7330650000E-01, 0.1191650000E-01 /
  data (epsData(91,i,28), i=1,3) / 0.6847750000E+03, -0.5716590000E-01, 0.8595170000E-02 /
  data (epsData(92,i,28), i=1,3) / 0.8054070000E+03, -0.3266150000E-01, 0.5028510000E-02 /
  data (epsData(93,i,28), i=1,3) / 0.8299240000E+03, -0.2594980000E-01, 0.4551390000E-02 /
  data (epsData(94,i,28), i=1,3) / 0.8408460000E+03, -0.2151080000E-01, 0.4448350000E-02 /
  data (epsData(95,i,28), i=1,3) / 0.8468380000E+03, -0.1780440000E-01, 0.4418530000E-02 /
  data (epsData(96,i,28), i=1,3) / 0.8498440000E+03, -0.1494010000E-01, 0.4475150000E-02 /
  data (epsData(97,i,28), i=1,3) / 0.8508470000E+03, -0.1363000000E-01, 0.4608490000E-02 /
  data (epsData(98,i,28), i=1,3) / 0.8518510000E+03, -0.1193490000E-01, 0.4783120000E-02 /
  data (epsData(99,i,28), i=1,3) / 0.8528100000E+03, -0.9911510000E-02, 0.5780640000E-02 /
  data (epsData(100,i,28), i=1,3) / 0.8547170000E+03, -0.7202780000E-02, 0.9164570000E-02 /
  data (epsData(101,i,28), i=1,3) / 0.8557060000E+03, -0.6506230000E-02, 0.1162030000E-01 /
  data (epsData(102,i,28), i=1,3) / 0.8572770000E+03, -0.6281290000E-02, 0.1502560000E-01 /
  data (epsData(103,i,28), i=1,3) / 0.8586270000E+03, -0.7195300000E-02, 0.1798310000E-01 /
  data (epsData(104,i,28), i=1,3) / 0.8603530000E+03, -0.9388590000E-02, 0.2043370000E-01 /
  data (epsData(105,i,28), i=1,3) / 0.8625300000E+03, -0.1193110000E-01, 0.2156140000E-01 /
  data (epsData(106,i,28), i=1,3) / 0.8656340000E+03, -0.1382460000E-01, 0.2168750000E-01 /
  data (epsData(107,i,28), i=1,3) / 0.8690140000E+03, -0.1389680000E-01, 0.2180670000E-01 /
  data (epsData(108,i,28), i=1,3) / 0.8718630000E+03, -0.1315460000E-01, 0.2358710000E-01 /
  data (epsData(109,i,28), i=1,3) / 0.8756600000E+03, -0.1447680000E-01, 0.2729470000E-01 /
  data (epsData(110,i,28), i=1,3) / 0.8798300000E+03, -0.1819860000E-01, 0.2909690000E-01 /
  data (epsData(111,i,28), i=1,3) / 0.8873730000E+03, -0.2232240000E-01, 0.2880010000E-01 /
  data (epsData(112,i,28), i=1,3) / 0.9008170000E+03, -0.2627270000E-01, 0.2734520000E-01 /
  data (epsData(113,i,28), i=1,3) / 0.9274570000E+03, -0.2931320000E-01, 0.2444520000E-01 /
  data (epsData(114,i,28), i=1,3) / 0.9673580000E+03, -0.3010170000E-01, 0.2077690000E-01 /
  data (epsData(115,i,28), i=1,3) / 0.1001600000E+04, -0.2861200000E-01, 0.1864630000E-01 /
  data (epsData(116,i,28), i=1,3) / 0.1008660000E+04, -0.2776720000E-01, 0.1941180000E-01 /
  data (epsData(117,i,28), i=1,3) / 0.1015170000E+04, -0.2873400000E-01, 0.1949520000E-01 /
  data (epsData(118,i,28), i=1,3) / 0.1077170000E+04, -0.2870240000E-01, 0.1619490000E-01 /
  data (epsData(119,i,28), i=1,3) / 0.1197390000E+04, -0.2553980000E-01, 0.1136320000E-01 /
  data (epsData(120,i,28), i=1,3) / 0.1357650000E+04, -0.2112220000E-01, 0.7477340000E-02 /
  data (epsData(121,i,28), i=1,3) / 0.1523220000E+04, -0.1735820000E-01, 0.5046570000E-02 /
  data (epsData(122,i,28), i=1,3) / 0.1696420000E+04, -0.1427030000E-01, 0.3470680000E-02 /
  data (epsData(123,i,28), i=1,3) / 0.1890310000E+04, -0.1162270000E-01, 0.2374050000E-02 /
  data (epsData(124,i,28), i=1,3) / 0.2102150000E+04, -0.9454360000E-02, 0.1622040000E-02 /
  data (epsData(125,i,28), i=1,3) / 0.2336760000E+04, -0.7668750000E-02, 0.1111340000E-02 /
  data (epsData(126,i,28), i=1,3) / 0.2595630000E+04, -0.6216530000E-02, 0.7624760000E-03 /
  data (epsData(127,i,28), i=1,3) / 0.2885510000E+04, -0.5022640000E-02, 0.5178160000E-03 /
  data (epsData(128,i,28), i=1,3) / 0.3210060000E+04, -0.4046550000E-02, 0.3497240000E-03 /
  data (epsData(129,i,28), i=1,3) / 0.3573550000E+04, -0.3252640000E-02, 0.2351440000E-03 /
  data (epsData(130,i,28), i=1,3) / 0.3978850000E+04, -0.2611900000E-02, 0.1576430000E-03 /
  data (epsData(131,i,28), i=1,3) / 0.4433270000E+04, -0.2093150000E-02, 0.1051950000E-03 /
  data (epsData(132,i,28), i=1,3) / 0.4943030000E+04, -0.1674010000E-02, 0.6988630000E-04 /
  data (epsData(133,i,28), i=1,3) / 0.5512850000E+04, -0.1336870000E-02, 0.4629750000E-04 /
  data (epsData(134,i,28), i=1,3) / 0.6154720000E+04, -0.1063580000E-02, 0.3054210000E-04 /
  data (epsData(135,i,28), i=1,3) / 0.6886110000E+04, -0.8392430000E-03, 0.1993740000E-04 /
  data (epsData(136,i,28), i=1,3) / 0.7786860000E+04, -0.6370080000E-03, 0.1248830000E-04 /
  data (epsData(137,i,28), i=1,3) / 0.8237440000E+04, -0.5352930000E-03, 0.1005690000E-04 /
  data (epsData(138,i,28), i=1,3) / 0.8307950000E+04, -0.4981500000E-03, 0.9737410000E-05 /
  data (epsData(139,i,28), i=1,3) / 0.8325750000E+04, -0.4679660000E-03, 0.1212440000E-04 /
  data (epsData(140,i,28), i=1,3) / 0.8329310000E+04, -0.4501550000E-03, 0.1500920000E-04 /
  data (epsData(141,i,28), i=1,3) / 0.8331100000E+04, -0.4331260000E-03, 0.2067170000E-04 /
  data (epsData(142,i,28), i=1,3) / 0.8332880000E+04, -0.4175140000E-03, 0.5490770000E-04 /
  data (epsData(143,i,28), i=1,3) / 0.8333000000E+04, -0.3380580000E-03, 0.5960160000E-04 /
  data (epsData(144,i,28), i=1,3) / 0.8333000000E+04, -0.4170090000E-03, 0.5973420000E-04 /
  data (epsData(145,i,28), i=1,3) / 0.8333720000E+04, -0.4306460000E-03, 0.8183720000E-04 /
  data (epsData(146,i,28), i=1,3) / 0.8334760000E+04, -0.4563670000E-03, 0.8449600000E-04 /
  data (epsData(147,i,28), i=1,3) / 0.8337580000E+04, -0.4744880000E-03, 0.7202130000E-04 /
  data (epsData(148,i,28), i=1,3) / 0.8376350000E+04, -0.5022840000E-03, 0.7173470000E-04 /
  data (epsData(149,i,28), i=1,3) / 0.8493540000E+04, -0.5171120000E-03, 0.7067430000E-04 /
  data (epsData(150,i,28), i=1,3) / 0.9066040000E+04, -0.4832420000E-03, 0.5611750000E-04 /
  data (epsData(151,i,28), i=1,3) / 0.1034490000E+05, -0.3852310000E-03, 0.3523030000E-04 /
  data (epsData(152,i,28), i=1,3) / 0.1164850000E+05, -0.3082520000E-03, 0.2287910000E-04 /
  data (epsData(153,i,28), i=1,3) / 0.1305770000E+05, -0.2470630000E-03, 0.1501080000E-04 /
  data (epsData(154,i,28), i=1,3) / 0.1461900000E+05, -0.1978450000E-03, 0.9855480000E-05 /
  data (epsData(155,i,28), i=1,3) / 0.1634980000E+05, -0.1584330000E-03, 0.6463030000E-05 /
  data (epsData(156,i,28), i=1,3) / 0.1828350000E+05, -0.1267460000E-03, 0.4222120000E-05 /
  data (epsData(157,i,28), i=1,3) / 0.2045030000E+05, -0.1012740000E-03, 0.2743830000E-05 /
  data (epsData(158,i,28), i=1,3) / 0.2287950000E+05, -0.8084610000E-04, 0.1775470000E-05 /
  data (epsData(159,i,28), i=1,3) / 0.2560380000E+05, -0.6448760000E-04, 0.1144100000E-05 /
  data (epsData(160,i,28), i=1,3) / 0.2866180000E+05, -0.5140050000E-04, 0.7334690000E-06 /
  data (epsData(161,i,28), i=1,3) / 0.3209370000E+05, -0.4094540000E-04, 0.4679590000E-06 /
  data (epsData(162,i,28), i=1,3) / 0.3594630000E+05, -0.3260030000E-04, 0.2973960000E-06 /
  data (epsData(163,i,28), i=1,3) / 0.4027260000E+05, -0.2594340000E-04, 0.1880560000E-06 /
  data (epsData(164,i,28), i=1,3) / 0.4512550000E+05, -0.2064220000E-04, 0.1185540000E-06 /
  data (epsData(165,i,28), i=1,3) / 0.5057140000E+05, -0.1642060000E-04, 0.7437500000E-07 /
  data (epsData(166,i,28), i=1,3) / 0.5669270000E+05, -0.1305530000E-04, 0.4649300000E-07 /
  data (epsData(167,i,28), i=1,3) / 0.6355700000E+05, -0.1038030000E-04, 0.2910590000E-07 /
  data (epsData(168,i,28), i=1,3) / 0.7126840000E+05, -0.8250470000E-05, 0.1794840000E-07 /
  data (epsData(169,i,28), i=1,3) / 0.7992640000E+05, -0.6556670000E-05, 0.1112920000E-07 /
  data (epsData(170,i,28), i=1,3) / 0.8963550000E+05, -0.5211440000E-05, 0.6827850000E-08 /
  data (epsData(171,i,28), i=1,3) / 0.9998410000E+05, -0.4195130000E-05, 0.4302270000E-08 /
  data (epsData(172,i,28), i=1,3) / 0.1000000000E+06, -0.4196100000E-05, 0.4299320000E-08 /
  data (epsData(1,i,29), i=1,3) / 0.2506580000E-02, -0.7111520000E+02, 0.4352460000E+03 /
  data (epsData(2,i,29), i=1,3) / 0.4693440000E-02, -0.7430300000E+02, 0.1633410000E+03 /
  data (epsData(3,i,29), i=1,3) / 0.7560590000E-02, -0.1689570000E+02, 0.8275170000E+02 /
  data (epsData(4,i,29), i=1,3) / 0.1110800000E-01, 0.8843370000E+01, 0.4968920000E+02 /
  data (epsData(5,i,29), i=1,3) / 0.1533570000E-01, 0.2108140000E+02, 0.3316730000E+02 /
  data (epsData(6,i,29), i=1,3) / 0.2024370000E-01, 0.2745310000E+02, 0.2376340000E+02 /
  data (epsData(7,i,29), i=1,3) / 0.2583200000E-01, 0.3103520000E+02, 0.1790080000E+02 /
  data (epsData(8,i,29), i=1,3) / 0.3210060000E-01, 0.3317820000E+02, 0.1399430000E+02 /
  data (epsData(9,i,29), i=1,3) / 0.3904940000E-01, 0.3452710000E+02, 0.1125630000E+02 /
  data (epsData(10,i,29), i=1,3) / 0.4667860000E-01, 0.3541290000E+02, 0.9260110000E+01 /
  data (epsData(11,i,29), i=1,3) / 0.6397770000E-01, 0.3644040000E+02, 0.6598250000E+01 /
  data (epsData(12,i,29), i=1,3) / 0.9502860000E-01, 0.3716890000E+02, 0.4348480000E+01 /
  data (epsData(13,i,29), i=1,3) / 0.1754980000E+00, 0.3756660000E+02, 0.2447000000E+01 /
  data (epsData(14,i,29), i=1,3) / 0.4838160000E+00, 0.3776600000E+02, 0.9426470000E+00 /
  data (epsData(15,i,29), i=1,3) / 0.1559640000E+01, 0.3917950000E+02, 0.3931040000E+00 /
  data (epsData(16,i,29), i=1,3) / 0.2677010000E+01, 0.4290690000E+02, 0.3583180000E+00 /
  data (epsData(17,i,29), i=1,3) / 0.3450210000E+01, 0.4791340000E+02, 0.4397430000E+00 /
  data (epsData(18,i,29), i=1,3) / 0.4020090000E+01, 0.5436580000E+02, 0.6321070000E+00 /
  data (epsData(19,i,29), i=1,3) / 0.4321360000E+01, 0.5989740000E+02, 0.9022390000E+00 /
  data (epsData(20,i,29), i=1,3) / 0.4554460000E+01, 0.6653680000E+02, 0.1414100000E+01 /
  data (epsData(21,i,29), i=1,3) / 0.4713260000E+01, 0.7322420000E+02, 0.3504010000E+01 /
  data (epsData(22,i,29), i=1,3) / 0.4793680000E+01, 0.7734740000E+02, 0.4810950000E+01 /
  data (epsData(23,i,29), i=1,3) / 0.4874780000E+01, 0.8389940000E+02, 0.6040410000E+01 /
  data (epsData(24,i,29), i=1,3) / 0.4956560000E+01, 0.9060230000E+02, 0.1507340000E+02 /
  data (epsData(25,i,29), i=1,3) / 0.5027550000E+01, 0.9210640000E+02, 0.2493510000E+02 /
  data (epsData(26,i,29), i=1,3) / 0.5150000000E+01, 0.8783280000E+02, 0.3945430000E+02 /
  data (epsData(27,i,29), i=1,3) / 0.5195920000E+01, 0.8333060000E+02, 0.4454060000E+02 /
  data (epsData(28,i,29), i=1,3) / 0.5266670000E+01, 0.7643170000E+02, 0.4552900000E+02 /
  data (epsData(29,i,29), i=1,3) / 0.5544230000E+01, 0.6350990000E+02, 0.4642750000E+02 /
  data (epsData(30,i,29), i=1,3) / 0.5816680000E+01, 0.5933970000E+02, 0.4237900000E+02 /
  data (epsData(31,i,29), i=1,3) / 0.6066690000E+01, 0.6180030000E+02, 0.3840230000E+02 /
  data (epsData(32,i,29), i=1,3) / 0.6265670000E+01, 0.6914010000E+02, 0.3520200000E+02 /
  data (epsData(33,i,29), i=1,3) / 0.6393220000E+01, 0.7851200000E+02, 0.3327110000E+02 /
  data (epsData(34,i,29), i=1,3) / 0.6481660000E+01, 0.8926540000E+02, 0.3196960000E+02 /
  data (epsData(35,i,29), i=1,3) / 0.6526900000E+01, 0.9736740000E+02, 0.3132470000E+02 /
  data (epsData(36,i,29), i=1,3) / 0.6572820000E+01, 0.1090590000E+03, 0.3068520000E+02 /
  data (epsData(37,i,29), i=1,3) / 0.6619420000E+01, 0.1290740000E+03, 0.3005120000E+02 /
  data (epsData(38,i,29), i=1,3) / 0.6666700000E+01, 0.1724370000E+03, 0.2942290000E+02 /
  data (epsData(39,i,29), i=1,3) / 0.6714660000E+01, 0.1746470000E+03, 0.1443900000E+03 /
  data (epsData(40,i,29), i=1,3) / 0.6763300000E+01, 0.1305600000E+03, 0.1523700000E+03 /
  data (epsData(41,i,29), i=1,3) / 0.6812620000E+01, 0.1034430000E+03, 0.1583180000E+03 /
  data (epsData(42,i,29), i=1,3) / 0.6862620000E+01, 0.8312690000E+02, 0.1581520000E+03 /
  data (epsData(43,i,29), i=1,3) / 0.7016700000E+01, 0.4123570000E+02, 0.1553450000E+03 /
  data (epsData(44,i,29), i=1,3) / 0.7122830000E+01, 0.2055670000E+02, 0.1469980000E+03 /
  data (epsData(45,i,29), i=1,3) / 0.7287120000E+01, -0.2908580000E+01, 0.1351360000E+03 /
  data (epsData(46,i,29), i=1,3) / 0.7400040000E+01, -0.1492030000E+02, 0.1239390000E+03 /
  data (epsData(47,i,29), i=1,3) / 0.7634060000E+01, -0.3019910000E+02, 0.1035230000E+03 /
  data (epsData(48,i,29), i=1,3) / 0.7816720000E+01, -0.3691830000E+02, 0.8833630000E+02 /
  data (epsData(49,i,29), i=1,3) / 0.8005490000E+01, -0.3987690000E+02, 0.7422890000E+02 /
  data (epsData(50,i,29), i=1,3) / 0.8333730000E+01, -0.3970390000E+02, 0.5584800000E+02 /
  data (epsData(51,i,29), i=1,3) / 0.8538840000E+01, -0.3732110000E+02, 0.4611300000E+02 /
  data (epsData(52,i,29), i=1,3) / 0.9041230000E+01, -0.3065420000E+02, 0.3211300000E+02 /
  data (epsData(53,i,29), i=1,3) / 0.9190890000E+01, -0.2805590000E+02, 0.2940330000E+02 /
  data (epsData(54,i,29), i=1,3) / 0.9736140000E+01, -0.2253430000E+02, 0.2286840000E+02 /
  data (epsData(55,i,29), i=1,3) / 0.1048610000E+02, -0.1677170000E+02, 0.1658660000E+02 /
  data (epsData(56,i,29), i=1,3) / 0.1110760000E+02, -0.1289440000E+02, 0.1377760000E+02 /
  data (epsData(57,i,29), i=1,3) / 0.1157190000E+02, -0.1048360000E+02, 0.1237410000E+02 /
  data (epsData(58,i,29), i=1,3) / 0.1235010000E+02, -0.7458420000E+01, 0.1085650000E+02 /
  data (epsData(59,i,29), i=1,3) / 0.1327770000E+02, -0.4930270000E+01, 0.9881240000E+01 /
  data (epsData(60,i,29), i=1,3) / 0.1426040000E+02, -0.3118230000E+01, 0.9346240000E+01 /
  data (epsData(61,i,29), i=1,3) / 0.1541680000E+02, -0.1741130000E+01, 0.9028420000E+01 /
  data (epsData(62,i,29), i=1,3) / 0.1689440000E+02, -0.7429870000E+00, 0.8817530000E+01 /
  data (epsData(63,i,29), i=1,3) / 0.1887920000E+02, -0.1802770000E+00, 0.8601150000E+01 /
  data (epsData(64,i,29), i=1,3) / 0.2161290000E+02, -0.1003310000E+00, 0.8196760000E+01 /
  data (epsData(65,i,29), i=1,3) / 0.3091070000E+02, -0.6477930000E+00, 0.6231410000E+01 /
  data (epsData(66,i,29), i=1,3) / 0.3832340000E+02, -0.8320840000E+00, 0.5078720000E+01 /
  data (epsData(67,i,29), i=1,3) / 0.4786640000E+02, -0.9998510000E+00, 0.4018930000E+01 /
  data (epsData(68,i,29), i=1,3) / 0.5807060000E+02, -0.1070170000E+01, 0.3157780000E+01 /
  data (epsData(69,i,29), i=1,3) / 0.6672780000E+02, -0.9995910000E+00, 0.2589430000E+01 /
  data (epsData(70,i,29), i=1,3) / 0.7205450000E+02, -0.8497040000E+00, 0.2329580000E+01 /
  data (epsData(71,i,29), i=1,3) / 0.7418560000E+02, -0.7080100000E+00, 0.2296320000E+01 /
  data (epsData(72,i,29), i=1,3) / 0.7538610000E+02, -0.6394930000E+00, 0.2397110000E+01 /
  data (epsData(73,i,29), i=1,3) / 0.7762690000E+02, -0.7628700000E+00, 0.2536690000E+01 /
  data (epsData(74,i,29), i=1,3) / 0.8024190000E+02, -0.9838970000E+00, 0.2489960000E+01 /
  data (epsData(75,i,29), i=1,3) / 0.8515860000E+02, -0.1204270000E+01, 0.2204580000E+01 /
  data (epsData(76,i,29), i=1,3) / 0.9090570000E+02, -0.1254280000E+01, 0.1838670000E+01 /
  data (epsData(77,i,29), i=1,3) / 0.9890040000E+02, -0.1169620000E+01, 0.1478580000E+01 /
  data (epsData(78,i,29), i=1,3) / 0.1116970000E+03, -0.1006850000E+01, 0.1123770000E+01 /
  data (epsData(79,i,29), i=1,3) / 0.1229410000E+03, -0.8711260000E+00, 0.9399230000E+00 /
  data (epsData(80,i,29), i=1,3) / 0.1451410000E+03, -0.7330670000E+00, 0.6821950000E+00 /
  data (epsData(81,i,29), i=1,3) / 0.1711010000E+03, -0.6049580000E+00, 0.4824920000E+00 /
  data (epsData(82,i,29), i=1,3) / 0.1989220000E+03, -0.4985840000E+00, 0.3421620000E+00 /
  data (epsData(83,i,29), i=1,3) / 0.2284090000E+03, -0.4107750000E+00, 0.2435990000E+00 /
  data (epsData(84,i,29), i=1,3) / 0.2587940000E+03, -0.3393460000E+00, 0.1757510000E+00 /
  data (epsData(85,i,29), i=1,3) / 0.2915620000E+03, -0.2794310000E+00, 0.1269850000E+00 /
  data (epsData(86,i,29), i=1,3) / 0.3268020000E+03, -0.2295710000E+00, 0.9182220000E-01 /
  data (epsData(87,i,29), i=1,3) / 0.3642270000E+03, -0.1886810000E+00, 0.6689540000E-01 /
  data (epsData(88,i,29), i=1,3) / 0.4023250000E+03, -0.1563660000E+00, 0.5004350000E-01 /
  data (epsData(89,i,29), i=1,3) / 0.4531940000E+03, -0.1248110000E+00, 0.3478070000E-01 /
  data (epsData(90,i,29), i=1,3) / 0.4960900000E+03, -0.1038630000E+00, 0.2645190000E-01 /
  data (epsData(91,i,29), i=1,3) / 0.5583790000E+03, -0.8128360000E-01, 0.1807820000E-01 /
  data (epsData(92,i,29), i=1,3) / 0.6238160000E+03, -0.6392920000E-01, 0.1276750000E-01 /
  data (epsData(93,i,29), i=1,3) / 0.6876710000E+03, -0.5086990000E-01, 0.9364790000E-02 /
  data (epsData(94,i,29), i=1,3) / 0.7684120000E+03, -0.3793370000E-01, 0.6498870000E-02 /
  data (epsData(95,i,29), i=1,3) / 0.8675730000E+03, -0.2376100000E-01, 0.4347800000E-02 /
  data (epsData(96,i,29), i=1,3) / 0.8970580000E+03, -0.1826560000E-01, 0.3908040000E-02 /
  data (epsData(97,i,29), i=1,3) / 0.9114970000E+03, -0.1433190000E-01, 0.3705200000E-02 /
  data (epsData(98,i,29), i=1,3) / 0.9187660000E+03, -0.1154290000E-01, 0.3705250000E-02 /
  data (epsData(99,i,29), i=1,3) / 0.9229350000E+03, -0.9358940000E-02, 0.3693970000E-02 /
  data (epsData(100,i,29), i=1,3) / 0.9260690000E+03, -0.7141160000E-02, 0.3719480000E-02 /
  data (epsData(101,i,29), i=1,3) / 0.9281610000E+03, -0.5118380000E-02, 0.3781920000E-02 /
  data (epsData(102,i,29), i=1,3) / 0.9292090000E+03, -0.3798480000E-02, 0.3848440000E-02 /
  data (epsData(103,i,29), i=1,3) / 0.9302570000E+03, -0.2094570000E-02, 0.3973520000E-02 /
  data (epsData(104,i,29), i=1,3) / 0.9313050000E+03, 0.3512270000E-03, 0.4272180000E-02 /
  data (epsData(105,i,29), i=1,3) / 0.9323550000E+03, 0.4053080000E-02, 0.5783220000E-02 /
  data (epsData(106,i,29), i=1,3) / 0.9327220000E+03, 0.5491720000E-02, 0.7982030000E-02 /
  data (epsData(107,i,29), i=1,3) / 0.9329130000E+03, 0.5574070000E-02, 0.9419030000E-02 /
  data (epsData(108,i,29), i=1,3) / 0.9332440000E+03, 0.5119870000E-02, 0.1114470000E-01 /
  data (epsData(109,i,29), i=1,3) / 0.9338050000E+03, 0.4381300000E-02, 0.1283810000E-01 /
  data (epsData(110,i,29), i=1,3) / 0.9348770000E+03, 0.3248140000E-02, 0.1571810000E-01 /
  data (epsData(111,i,29), i=1,3) / 0.9355170000E+03, 0.2063040000E-02, 0.1667490000E-01 /
  data (epsData(112,i,29), i=1,3) / 0.9394620000E+03, -0.1423450000E-02, 0.2104450000E-01 /
  data (epsData(113,i,29), i=1,3) / 0.9419600000E+03, -0.4366010000E-02, 0.2258850000E-01 /
  data (epsData(114,i,29), i=1,3) / 0.9443410000E+03, -0.6694360000E-02, 0.2260620000E-01 /
  data (epsData(115,i,29), i=1,3) / 0.9475580000E+03, -0.8104600000E-02, 0.2173540000E-01 /
  data (epsData(116,i,29), i=1,3) / 0.9503820000E+03, -0.7670740000E-02, 0.2120400000E-01 /
  data (epsData(117,i,29), i=1,3) / 0.9518050000E+03, -0.6535740000E-02, 0.2160540000E-01 /
  data (epsData(118,i,29), i=1,3) / 0.9527660000E+03, -0.6050540000E-02, 0.2305600000E-01 /
  data (epsData(119,i,29), i=1,3) / 0.9548160000E+03, -0.7266810000E-02, 0.2529150000E-01 /
  data (epsData(120,i,29), i=1,3) / 0.9602620000E+03, -0.1122760000E-01, 0.2764300000E-01 /
  data (epsData(121,i,29), i=1,3) / 0.9648390000E+03, -0.1457040000E-01, 0.2743160000E-01 /
  data (epsData(122,i,29), i=1,3) / 0.9737290000E+03, -0.1769790000E-01, 0.2571660000E-01 /
  data (epsData(123,i,29), i=1,3) / 0.9897920000E+03, -0.2031980000E-01, 0.2342820000E-01 /
  data (epsData(124,i,29), i=1,3) / 0.1019050000E+04, -0.2219110000E-01, 0.2048740000E-01 /
  data (epsData(125,i,29), i=1,3) / 0.1062810000E+04, -0.2242900000E-01, 0.1730930000E-01 /
  data (epsData(126,i,29), i=1,3) / 0.1091620000E+04, -0.2125310000E-01, 0.1598320000E-01 /
  data (epsData(127,i,29), i=1,3) / 0.1097690000E+04, -0.2082420000E-01, 0.1668520000E-01 /
  data (epsData(128,i,29), i=1,3) / 0.1105630000E+04, -0.2163030000E-01, 0.1659920000E-01 /
  data (epsData(129,i,29), i=1,3) / 0.1182420000E+04, -0.2167330000E-01, 0.1343490000E-01 /
  data (epsData(130,i,29), i=1,3) / 0.1309530000E+04, -0.1945700000E-01, 0.9535150000E-02 /
  data (epsData(131,i,29), i=1,3) / 0.1480340000E+04, -0.1624310000E-01, 0.6315590000E-02 /
  data (epsData(132,i,29), i=1,3) / 0.1655750000E+04, -0.1345100000E-01, 0.4303490000E-02 /
  data (epsData(133,i,29), i=1,3) / 0.1846820000E+04, -0.1103720000E-01, 0.2942270000E-02 /
  data (epsData(134,i,29), i=1,3) / 0.2054470000E+04, -0.9025950000E-02, 0.2017080000E-02 /
  data (epsData(135,i,29), i=1,3) / 0.2281100000E+04, -0.7369290000E-02, 0.1389270000E-02 /
  data (epsData(136,i,29), i=1,3) / 0.2531780000E+04, -0.5995260000E-02, 0.9558590000E-03 /
  data (epsData(137,i,29), i=1,3) / 0.2815680000E+04, -0.4848430000E-02, 0.6510360000E-03 /
  data (epsData(138,i,29), i=1,3) / 0.3128490000E+04, -0.3919720000E-02, 0.4424480000E-03 /
  data (epsData(139,i,29), i=1,3) / 0.3478240000E+04, -0.3160460000E-02, 0.2992410000E-03 /
  data (epsData(140,i,29), i=1,3) / 0.3867380000E+04, -0.2545430000E-02, 0.2018800000E-03 /
  data (epsData(141,i,29), i=1,3) / 0.4302640000E+04, -0.2046140000E-02, 0.1356200000E-03 /
  data (epsData(142,i,29), i=1,3) / 0.4789740000E+04, -0.1641790000E-02, 0.9071980000E-04 /
  data (epsData(143,i,29), i=1,3) / 0.5337600000E+04, -0.1313630000E-02, 0.6034050000E-04 /
  data (epsData(144,i,29), i=1,3) / 0.5951560000E+04, -0.1048770000E-02, 0.3997610000E-04 /
  data (epsData(145,i,29), i=1,3) / 0.6638920000E+04, -0.8352460000E-03, 0.2651630000E-04 /
  data (epsData(146,i,29), i=1,3) / 0.7418520000E+04, -0.6600050000E-03, 0.1729300000E-04 /
  data (epsData(147,i,29), i=1,3) / 0.8393600000E+04, -0.4988810000E-03, 0.1081570000E-04 /
  data (epsData(148,i,29), i=1,3) / 0.8868620000E+04, -0.4193090000E-03, 0.8749120000E-05 /
  data (epsData(149,i,29), i=1,3) / 0.8951240000E+04, -0.3877510000E-03, 0.8445600000E-05 /
  data (epsData(150,i,29), i=1,3) / 0.8971210000E+04, -0.3632000000E-03, 0.1029060000E-04 /
  data (epsData(151,i,29), i=1,3) / 0.8974850000E+04, -0.3510480000E-03, 0.1215290000E-04 /
  data (epsData(152,i,29), i=1,3) / 0.8976670000E+04, -0.3402790000E-03, 0.1519280000E-04 /
  data (epsData(153,i,29), i=1,3) / 0.8978490000E+04, -0.3259680000E-03, 0.2906110000E-04 /
  data (epsData(154,i,29), i=1,3) / 0.8979000000E+04, -0.2539110000E-03, 0.3962990000E-04 /
  data (epsData(155,i,29), i=1,3) / 0.8979000000E+04, -0.3228000000E-03, 0.3970890000E-04 /
  data (epsData(156,i,29), i=1,3) / 0.8979960000E+04, -0.3309700000E-03, 0.5763050000E-04 /
  data (epsData(157,i,29), i=1,3) / 0.8981880000E+04, -0.3517770000E-03, 0.6523710000E-04 /
  data (epsData(158,i,29), i=1,3) / 0.8985110000E+04, -0.3678150000E-03, 0.5903310000E-04 /
  data (epsData(159,i,29), i=1,3) / 0.9036460000E+04, -0.3919250000E-03, 0.6210090000E-04 /
  data (epsData(160,i,29), i=1,3) / 0.9183300000E+04, -0.4040260000E-03, 0.6006320000E-04 /
  data (epsData(161,i,29), i=1,3) / 0.9845450000E+04, -0.3752480000E-03, 0.4686240000E-04 /
  data (epsData(162,i,29), i=1,3) / 0.1122920000E+05, -0.2996530000E-03, 0.2943920000E-04 /
  data (epsData(163,i,29), i=1,3) / 0.1264070000E+05, -0.2400160000E-03, 0.1912960000E-04 /
  data (epsData(164,i,29), i=1,3) / 0.1417100000E+05, -0.1924150000E-03, 0.1253020000E-04 /
  data (epsData(165,i,29), i=1,3) / 0.1585720000E+05, -0.1542450000E-03, 0.8235130000E-05 /
  data (epsData(166,i,29), i=1,3) / 0.1773180000E+05, -0.1235590000E-03, 0.5399000000E-05 /
  data (epsData(167,i,29), i=1,3) / 0.1982820000E+05, -0.9884650000E-04, 0.3523290000E-05 /
  data (epsData(168,i,29), i=1,3) / 0.2217480000E+05, -0.7899850000E-04, 0.2290800000E-05 /
  data (epsData(169,i,29), i=1,3) / 0.2480270000E+05, -0.6308650000E-04, 0.1482620000E-05 /
  data (epsData(170,i,29), i=1,3) / 0.2775020000E+05, -0.5033740000E-04, 0.9551140000E-06 /
  data (epsData(171,i,29), i=1,3) / 0.3105880000E+05, -0.4013150000E-04, 0.6122680000E-06 /
  data (epsData(172,i,29), i=1,3) / 0.3477190000E+05, -0.3197580000E-04, 0.3907470000E-06 /
  data (epsData(173,i,29), i=1,3) / 0.3893920000E+05, -0.2546500000E-04, 0.2481710000E-06 /
  data (epsData(174,i,29), i=1,3) / 0.4361730000E+05, -0.2027080000E-04, 0.1569880000E-06 /
  data (epsData(175,i,29), i=1,3) / 0.4887000000E+05, -0.1612950000E-04, 0.9883740000E-07 /
  data (epsData(176,i,29), i=1,3) / 0.5476940000E+05, -0.1282910000E-04, 0.6198300000E-07 /
  data (epsData(177,i,29), i=1,3) / 0.6138740000E+05, -0.1020300000E-04, 0.3881180000E-07 /
  data (epsData(178,i,29), i=1,3) / 0.6882730000E+05, -0.8110480000E-05, 0.2415440000E-07 /
  data (epsData(179,i,29), i=1,3) / 0.7717120000E+05, -0.6447360000E-05, 0.1494470000E-07 /
  data (epsData(180,i,29), i=1,3) / 0.8654130000E+05, -0.5124400000E-05, 0.9195730000E-08 /
  data (epsData(181,i,29), i=1,3) / 0.9701690000E+05, -0.4077340000E-05, 0.5669430000E-08 /
  data (epsData(182,i,29), i=1,3) / 0.1000000000E+06, -0.3847870000E-05, 0.4997330000E-08 /
  data (epsData(1,i,30), i=1,3) / 0.2506580000E-02, 0.2354940000E+04, 0.5311040000E+04 /
  data (epsData(2,i,30), i=1,3) / 0.4693440000E-02, 0.8540530000E+03, 0.2721450000E+04 /
  data (epsData(3,i,30), i=1,3) / 0.7560590000E-02, 0.7340650000E+03, 0.1658390000E+04 /
  data (epsData(4,i,30), i=1,3) / 0.1110800000E-01, 0.7228600000E+03, 0.1117730000E+04 /
  data (epsData(5,i,30), i=1,3) / 0.1533570000E-01, 0.7254630000E+03, 0.8049020000E+03 /
  data (epsData(6,i,30), i=1,3) / 0.2024370000E-01, 0.7291950000E+03, 0.6074900000E+03 /
  data (epsData(7,i,30), i=1,3) / 0.2583200000E-01, 0.7322240000E+03, 0.4748700000E+03 /
  data (epsData(8,i,30), i=1,3) / 0.3210060000E-01, 0.7344960000E+03, 0.3814540000E+03 /
  data (epsData(9,i,30), i=1,3) / 0.3904940000E-01, 0.7362140000E+03, 0.3131620000E+03 /
  data (epsData(10,i,30), i=1,3) / 0.5498800000E-01, 0.7387090000E+03, 0.2219990000E+03 /
  data (epsData(11,i,30), i=1,3) / 0.8399800000E-01, 0.7417330000E+03, 0.1451110000E+03 /
  data (epsData(12,i,30), i=1,3) / 0.1322020000E+00, 0.7468670000E+03, 0.9212860000E+02 /
  data (epsData(13,i,30), i=1,3) / 0.2427500000E+00, 0.7661950000E+03, 0.5015660000E+02 /
  data (epsData(14,i,30), i=1,3) / 0.3866320000E+00, 0.8101970000E+03, 0.4494920000E+02 /
  data (epsData(15,i,30), i=1,3) / 0.5364900000E+00, 0.8760650000E+03, 0.4178020000E+02 /
  data (epsData(16,i,30), i=1,3) / 0.5918850000E+00, 0.9226800000E+03, 0.3992560000E+02 /
  data (epsData(17,i,30), i=1,3) / 0.7422780000E+00, 0.1060160000E+04, 0.9926270000E+02 /
  data (epsData(18,i,30), i=1,3) / 0.8071970000E+00, 0.1154860000E+04, 0.1122430000E+03 /
  data (epsData(19,i,30), i=1,3) / 0.8406760000E+00, 0.1233570000E+04, 0.1169380000E+03 /
  data (epsData(20,i,30), i=1,3) / 0.8748370000E+00, 0.1367490000E+04, 0.1206640000E+03 /
  data (epsData(21,i,30), i=1,3) / 0.9096770000E+00, 0.1557460000E+04, 0.2202620000E+03 /
  data (epsData(22,i,30), i=1,3) / 0.9451980000E+00, 0.1666260000E+04, 0.5440870000E+03 /
  data (epsData(23,i,30), i=1,3) / 0.9813990000E+00, 0.1625080000E+04, 0.8261720000E+03 /
  data (epsData(24,i,30), i=1,3) / 0.1005440000E+01, 0.1527290000E+04, 0.9756060000E+03 /
  data (epsData(25,i,30), i=1,3) / 0.1110210000E+01, 0.1151580000E+04, 0.1291090000E+04 /
  data (epsData(26,i,30), i=1,3) / 0.1179940000E+01, 0.8735770000E+03, 0.1432270000E+04 /
  data (epsData(27,i,30), i=1,3) / 0.1212590000E+01, 0.7002960000E+03, 0.1447570000E+04 /
  data (epsData(28,i,30), i=1,3) / 0.1266670000E+01, 0.5070030000E+03, 0.1376140000E+04 /
  data (epsData(29,i,30), i=1,3) / 0.1370410000E+01, 0.2807260000E+03, 0.1256870000E+04 /
  data (epsData(30,i,30), i=1,3) / 0.1517360000E+01, 0.5895510000E+02, 0.1111510000E+04 /
  data (epsData(31,i,30), i=1,3) / 0.1600010000E+01, -0.2593400000E+02, 0.1014870000E+04 /
  data (epsData(32,i,30), i=1,3) / 0.1751370000E+01, -0.1244190000E+03, 0.8703220000E+03 /
  data (epsData(33,i,30), i=1,3) / 0.1919740000E+01, -0.1956790000E+03, 0.7225560000E+03 /
  data (epsData(34,i,30), i=1,3) / 0.2105120000E+01, -0.2204950000E+03, 0.5906540000E+03 /
  data (epsData(35,i,30), i=1,3) / 0.2307510000E+01, -0.2240620000E+03, 0.4849760000E+03 /
  data (epsData(36,i,30), i=1,3) / 0.2572820000E+01, -0.2137410000E+03, 0.3880370000E+03 /
  data (epsData(37,i,30), i=1,3) / 0.2913300000E+01, -0.2000040000E+03, 0.3039870000E+03 /
  data (epsData(38,i,30), i=1,3) / 0.3400040000E+01, -0.1847940000E+03, 0.2203130000E+03 /
  data (epsData(39,i,30), i=1,3) / 0.4069780000E+01, -0.1625930000E+03, 0.1309980000E+03 /
  data (epsData(40,i,30), i=1,3) / 0.4266720000E+01, -0.1474150000E+03, 0.1097530000E+03 /
  data (epsData(41,i,30), i=1,3) / 0.4538840000E+01, -0.1279630000E+03, 0.9120990000E+02 /
  data (epsData(42,i,30), i=1,3) / 0.4967420000E+01, -0.1070450000E+03, 0.7476430000E+02 /
  data (epsData(43,i,30), i=1,3) / 0.5576950000E+01, -0.8665700000E+02, 0.5661880000E+02 /
  data (epsData(44,i,30), i=1,3) / 0.6230020000E+01, -0.7045650000E+02, 0.4297320000E+02 /
  data (epsData(45,i,30), i=1,3) / 0.7016770000E+01, -0.5536320000E+02, 0.3107940000E+02 /
  data (epsData(46,i,30), i=1,3) / 0.7858620000E+01, -0.4194670000E+02, 0.2211160000E+02 /
  data (epsData(47,i,30), i=1,3) / 0.8858640000E+01, -0.2725600000E+02, 0.1501070000E+02 /
  data (epsData(48,i,30), i=1,3) / 0.9277690000E+01, -0.2038600000E+02, 0.1294190000E+02 /
  data (epsData(49,i,30), i=1,3) / 0.9491300000E+01, -0.1617820000E+02, 0.1211490000E+02 /
  data (epsData(50,i,30), i=1,3) / 0.9599130000E+01, -0.1368950000E+02, 0.1179610000E+02 /
  data (epsData(51,i,30), i=1,3) / 0.9707630000E+01, -0.1070770000E+02, 0.1154060000E+02 /
  data (epsData(52,i,30), i=1,3) / 0.9816820000E+01, -0.7200650000E+01, 0.1190810000E+02 /
  data (epsData(53,i,30), i=1,3) / 0.9926680000E+01, -0.2938040000E+01, 0.1259880000E+02 /
  data (epsData(54,i,30), i=1,3) / 0.1003720000E+02, 0.2322560000E+01, 0.1499400000E+02 /
  data (epsData(55,i,30), i=1,3) / 0.1011220000E+02, 0.5647050000E+01, 0.1907300000E+02 /
  data (epsData(56,i,30), i=1,3) / 0.1018710000E+02, 0.7095820000E+01, 0.2364480000E+02 /
  data (epsData(57,i,30), i=1,3) / 0.1029830000E+02, 0.7499940000E+01, 0.2893350000E+02 /
  data (epsData(58,i,30), i=1,3) / 0.1035000000E+02, 0.8492750000E+01, 0.2996110000E+02 /
  data (epsData(59,i,30), i=1,3) / 0.1036460000E+02, 0.9407490000E+01, 0.3023560000E+02 /
  data (epsData(60,i,30), i=1,3) / 0.1037990000E+02, 0.1106920000E+02, 0.3066390000E+02 /
  data (epsData(61,i,30), i=1,3) / 0.1039590000E+02, 0.1254690000E+02, 0.3339390000E+02 /
  data (epsData(62,i,30), i=1,3) / 0.1042990000E+02, 0.1288870000E+02, 0.3866600000E+02 /
  data (epsData(63,i,30), i=1,3) / 0.1046670000E+02, 0.1095250000E+02, 0.4421190000E+02 /
  data (epsData(64,i,30), i=1,3) / 0.1048610000E+02, 0.8930580000E+01, 0.4640530000E+02 /
  data (epsData(65,i,30), i=1,3) / 0.1064080000E+02, -0.2920730000E+01, 0.5372530000E+02 /
  data (epsData(66,i,30), i=1,3) / 0.1069120000E+02, -0.8075420000E+01, 0.5510950000E+02 /
  data (epsData(67,i,30), i=1,3) / 0.1077180000E+02, -0.1429220000E+02, 0.5317940000E+02 /
  data (epsData(68,i,30), i=1,3) / 0.1095140000E+02, -0.2245710000E+02, 0.4885550000E+02 /
  data (epsData(69,i,30), i=1,3) / 0.1122890000E+02, -0.2949190000E+02, 0.4150700000E+02 /
  data (epsData(70,i,30), i=1,3) / 0.1150750000E+02, -0.3258710000E+02, 0.3385130000E+02 /
  data (epsData(71,i,30), i=1,3) / 0.1186670000E+02, -0.3218200000E+02, 0.2591510000E+02 /
  data (epsData(72,i,30), i=1,3) / 0.1226940000E+02, -0.2965140000E+02, 0.2028100000E+02 /
  data (epsData(73,i,30), i=1,3) / 0.1283410000E+02, -0.2585440000E+02, 0.1563470000E+02 /
  data (epsData(74,i,30), i=1,3) / 0.1340040000E+02, -0.2276040000E+02, 0.1312700000E+02 /
  data (epsData(75,i,30), i=1,3) / 0.1462050000E+02, -0.1787130000E+02, 0.9370530000E+01 /
  data (epsData(76,i,30), i=1,3) / 0.1568610000E+02, -0.1431070000E+02, 0.7626170000E+01 /
  data (epsData(77,i,30), i=1,3) / 0.1686680000E+02, -0.1127090000E+02, 0.6623230000E+01 /
  data (epsData(78,i,30), i=1,3) / 0.1816240000E+02, -0.8793610000E+01, 0.6076400000E+01 /
  data (epsData(79,i,30), i=1,3) / 0.1968620000E+02, -0.6699890000E+01, 0.5801560000E+01 /
  data (epsData(80,i,30), i=1,3) / 0.2146690000E+02, -0.5044110000E+01, 0.5707080000E+01 /
  data (epsData(81,i,30), i=1,3) / 0.2380570000E+02, -0.3721010000E+01, 0.5660160000E+01 /
  data (epsData(82,i,30), i=1,3) / 0.2666290000E+02, -0.2837010000E+01, 0.5521630000E+01 /
  data (epsData(83,i,30), i=1,3) / 0.3112010000E+02, -0.2129100000E+01, 0.5119930000E+01 /
  data (epsData(84,i,30), i=1,3) / 0.3724940000E+02, -0.1608100000E+01, 0.4560660000E+01 /
  data (epsData(85,i,30), i=1,3) / 0.4503050000E+02, -0.1350610000E+01, 0.4007270000E+01 /
  data (epsData(86,i,30), i=1,3) / 0.6765720000E+02, -0.1286460000E+01, 0.2620400000E+01 /
  data (epsData(87,i,30), i=1,3) / 0.8123870000E+02, -0.1154820000E+01, 0.2025020000E+01 /
  data (epsData(88,i,30), i=1,3) / 0.8791760000E+02, -0.1039090000E+01, 0.1825720000E+01 /
  data (epsData(89,i,30), i=1,3) / 0.9352280000E+02, -0.1035920000E+01, 0.1779390000E+01 /
  data (epsData(90,i,30), i=1,3) / 0.1004370000E+03, -0.1108800000E+01, 0.1588090000E+01 /
  data (epsData(91,i,30), i=1,3) / 0.1105050000E+03, -0.1061440000E+01, 0.1283040000E+01 /
  data (epsData(92,i,30), i=1,3) / 0.1262310000E+03, -0.9188800000E+00, 0.9707180000E+00 /
  data (epsData(93,i,30), i=1,3) / 0.1392320000E+03, -0.7979660000E+00, 0.8117280000E+00 /
  data (epsData(94,i,30), i=1,3) / 0.1463720000E+03, -0.7746550000E+00, 0.7451470000E+00 /
  data (epsData(95,i,30), i=1,3) / 0.1741970000E+03, -0.6406910000E+00, 0.5193970000E+00 /
  data (epsData(96,i,30), i=1,3) / 0.2029850000E+03, -0.5288910000E+00, 0.3688350000E+00 /
  data (epsData(97,i,30), i=1,3) / 0.2342680000E+03, -0.4353430000E+00, 0.2607740000E+00 /
  data (epsData(98,i,30), i=1,3) / 0.2651270000E+03, -0.3612650000E+00, 0.1893150000E+00 /
  data (epsData(99,i,30), i=1,3) / 0.2984760000E+03, -0.2989360000E+00, 0.1375440000E+00 /
  data (epsData(100,i,30), i=1,3) / 0.3342080000E+03, -0.2469060000E+00, 0.9996390000E-01 /
  data (epsData(101,i,30), i=1,3) / 0.3713150000E+03, -0.2046960000E+00, 0.7365610000E-01 /
  data (epsData(102,i,30), i=1,3) / 0.4098390000E+03, -0.1705130000E+00, 0.5535860000E-01 /
  data (epsData(103,i,30), i=1,3) / 0.4631560000E+03, -0.1358710000E+00, 0.3795500000E-01 /
  data (epsData(104,i,30), i=1,3) / 0.5084160000E+03, -0.1129380000E+00, 0.2871530000E-01 /
  data (epsData(105,i,30), i=1,3) / 0.5710710000E+03, -0.8927770000E-01, 0.1976250000E-01 /
  data (epsData(106,i,30), i=1,3) / 0.6402900000E+03, -0.7039320000E-01, 0.1371570000E-01 /
  data (epsData(107,i,30), i=1,3) / 0.7076050000E+03, -0.5650300000E-01, 0.1000310000E-01 /
  data (epsData(108,i,30), i=1,3) / 0.7847750000E+03, -0.4430400000E-01, 0.7118790000E-02 /
  data (epsData(109,i,30), i=1,3) / 0.8828600000E+03, -0.3219590000E-01, 0.4817350000E-02 /
  data (epsData(110,i,30), i=1,3) / 0.9675980000E+03, -0.2226540000E-01, 0.3550010000E-02 /
  data (epsData(111,i,30), i=1,3) / 0.9954220000E+03, -0.1777390000E-01, 0.3226870000E-02 /
  data (epsData(112,i,30), i=1,3) / 0.1008420000E+04, -0.1462210000E-01, 0.3160370000E-02 /
  data (epsData(113,i,30), i=1,3) / 0.1014950000E+04, -0.1222700000E-01, 0.3145670000E-02 /
  data (epsData(114,i,30), i=1,3) / 0.1018230000E+04, -0.1044730000E-01, 0.3180230000E-02 /
  data (epsData(115,i,30), i=1,3) / 0.1020420000E+04, -0.8655190000E-02, 0.3287580000E-02 /
  data (epsData(116,i,30), i=1,3) / 0.1021510000E+04, -0.7360010000E-02, 0.3578480000E-02 /
  data (epsData(117,i,30), i=1,3) / 0.1022320000E+04, -0.6404730000E-02, 0.4346820000E-02 /
  data (epsData(118,i,30), i=1,3) / 0.1023510000E+04, -0.5191740000E-02, 0.5519280000E-02 /
  data (epsData(119,i,30), i=1,3) / 0.1024430000E+04, -0.4470510000E-02, 0.6887380000E-02 /
  data (epsData(120,i,30), i=1,3) / 0.1026140000E+04, -0.3998980000E-02, 0.9421060000E-02 /
  data (epsData(121,i,30), i=1,3) / 0.1027460000E+04, -0.4359340000E-02, 0.1144470000E-01 /
  data (epsData(122,i,30), i=1,3) / 0.1028950000E+04, -0.5462840000E-02, 0.1317040000E-01 /
  data (epsData(123,i,30), i=1,3) / 0.1031060000E+04, -0.7216490000E-02, 0.1436470000E-01 /
  data (epsData(124,i,30), i=1,3) / 0.1033950000E+04, -0.8846450000E-02, 0.1477900000E-01 /
  data (epsData(125,i,30), i=1,3) / 0.1039640000E+04, -0.1000400000E-01, 0.1497270000E-01 /
  data (epsData(126,i,30), i=1,3) / 0.1044270000E+04, -0.9709470000E-02, 0.1532970000E-01 /
  data (epsData(127,i,30), i=1,3) / 0.1047470000E+04, -0.9115180000E-02, 0.1686150000E-01 /
  data (epsData(128,i,30), i=1,3) / 0.1051010000E+04, -0.1000720000E-01, 0.1917490000E-01 /
  data (epsData(129,i,30), i=1,3) / 0.1055200000E+04, -0.1240420000E-01, 0.2027620000E-01 /
  data (epsData(130,i,30), i=1,3) / 0.1063370000E+04, -0.1519300000E-01, 0.2007860000E-01 /
  data (epsData(131,i,30), i=1,3) / 0.1085620000E+04, -0.1864760000E-01, 0.1879010000E-01 /
  data (epsData(132,i,30), i=1,3) / 0.1121000000E+04, -0.2053160000E-01, 0.1648720000E-01 /
  data (epsData(133,i,30), i=1,3) / 0.1172720000E+04, -0.2047720000E-01, 0.1391010000E-01 /
  data (epsData(134,i,30), i=1,3) / 0.1192620000E+04, -0.1946730000E-01, 0.1347880000E-01 /
  data (epsData(135,i,30), i=1,3) / 0.1198600000E+04, -0.1956980000E-01, 0.1423590000E-01 /
  data (epsData(136,i,30), i=1,3) / 0.1232130000E+04, -0.2036340000E-01, 0.1312540000E-01 /
  data (epsData(137,i,30), i=1,3) / 0.1335960000E+04, -0.1937570000E-01, 0.1002380000E-01 /
  data (epsData(138,i,30), i=1,3) / 0.1519560000E+04, -0.1633440000E-01, 0.6486230000E-02 /
  data (epsData(139,i,30), i=1,3) / 0.1707480000E+04, -0.1351150000E-01, 0.4361870000E-02 /
  data (epsData(140,i,30), i=1,3) / 0.1907430000E+04, -0.1110240000E-01, 0.2968380000E-02 /
  data (epsData(141,i,30), i=1,3) / 0.2124520000E+04, -0.9084090000E-02, 0.2029400000E-02 /
  data (epsData(142,i,30), i=1,3) / 0.2362900000E+04, -0.7402110000E-02, 0.1391090000E-02 /
  data (epsData(143,i,30), i=1,3) / 0.2622910000E+04, -0.6028720000E-02, 0.9583800000E-03 /
  data (epsData(144,i,30), i=1,3) / 0.2918240000E+04, -0.4876480000E-02, 0.6519900000E-03 /
  data (epsData(145,i,30), i=1,3) / 0.3244660000E+04, -0.3940410000E-02, 0.4423750000E-03 /
  data (epsData(146,i,30), i=1,3) / 0.3606630000E+04, -0.3180790000E-02, 0.2995470000E-03 /
  data (epsData(147,i,30), i=1,3) / 0.4011710000E+04, -0.2561360000E-02, 0.2018490000E-03 /
  data (epsData(148,i,30), i=1,3) / 0.4465100000E+04, -0.2058390000E-02, 0.1354330000E-03 /
  data (epsData(149,i,30), i=1,3) / 0.4972630000E+04, -0.1651240000E-02, 0.9048730000E-04 /
  data (epsData(150,i,30), i=1,3) / 0.5541000000E+04, -0.1322310000E-02, 0.6021190000E-04 /
  data (epsData(151,i,30), i=1,3) / 0.6180400000E+04, -0.1055920000E-02, 0.3986170000E-04 /
  data (epsData(152,i,30), i=1,3) / 0.6905390000E+04, -0.8391880000E-03, 0.2622330000E-04 /
  data (epsData(153,i,30), i=1,3) / 0.7110620000E+04, -0.7895400000E-03, 0.2350930000E-04 /
  data (epsData(154,i,30), i=1,3) / 0.7112000000E+04, -0.7389600000E-03, 0.2349200000E-04 /
  data (epsData(155,i,30), i=1,3) / 0.7112000000E+04, -0.7892210000E-03, 0.2349200000E-04 /
  data (epsData(156,i,30), i=1,3) / 0.7945110000E+04, -0.6248240000E-03, 0.1535820000E-04 /
  data (epsData(157,i,30), i=1,3) / 0.8963430000E+04, -0.4775140000E-03, 0.9686310000E-05 /
  data (epsData(158,i,30), i=1,3) / 0.9542390000E+04, -0.3958590000E-03, 0.7612880000E-05 /
  data (epsData(159,i,30), i=1,3) / 0.9630460000E+04, -0.3671820000E-03, 0.7348670000E-05 /
  data (epsData(160,i,30), i=1,3) / 0.9650860000E+04, -0.3449930000E-03, 0.9190990000E-05 /
  data (epsData(161,i,30), i=1,3) / 0.9654580000E+04, -0.3337360000E-03, 0.1111300000E-04 /
  data (epsData(162,i,30), i=1,3) / 0.9656440000E+04, -0.3236860000E-03, 0.1428690000E-04 /
  data (epsData(163,i,30), i=1,3) / 0.9658300000E+04, -0.3104670000E-03, 0.2825500000E-04 /
  data (epsData(164,i,30), i=1,3) / 0.9659000000E+04, -0.2291780000E-03, 0.4355210000E-04 /
  data (epsData(165,i,30), i=1,3) / 0.9659000000E+04, -0.3080530000E-03, 0.4363290000E-04 /
  data (epsData(166,i,30), i=1,3) / 0.9659850000E+04, -0.3199140000E-03, 0.5864770000E-04 /
  data (epsData(167,i,30), i=1,3) / 0.9661120000E+04, -0.3371570000E-03, 0.5848660000E-04 /
  data (epsData(168,i,30), i=1,3) / 0.9664750000E+04, -0.3480380000E-03, 0.5099950000E-04 /
  data (epsData(169,i,30), i=1,3) / 0.9693830000E+04, -0.3666150000E-03, 0.5274900000E-04 /
  data (epsData(170,i,30), i=1,3) / 0.9807170000E+04, -0.3802230000E-03, 0.5294900000E-04 /
  data (epsData(171,i,30), i=1,3) / 0.1029460000E+05, -0.3667250000E-03, 0.4461060000E-04 /
  data (epsData(172,i,30), i=1,3) / 0.1189000000E+05, -0.2883210000E-03, 0.2679880000E-04 /
  data (epsData(173,i,30), i=1,3) / 0.1339020000E+05, -0.2307730000E-03, 0.1738420000E-04 /
  data (epsData(174,i,30), i=1,3) / 0.1501170000E+05, -0.1849720000E-03, 0.1139440000E-04 /
  data (epsData(175,i,30), i=1,3) / 0.1680900000E+05, -0.1480890000E-03, 0.7465520000E-05 /
  data (epsData(176,i,30), i=1,3) / 0.1880600000E+05, -0.1185080000E-03, 0.4886650000E-05 /
  data (epsData(177,i,30), i=1,3) / 0.2103300000E+05, -0.9478080000E-04, 0.3186710000E-05 /
  data (epsData(178,i,30), i=1,3) / 0.2352830000E+05, -0.7571480000E-04, 0.2069930000E-05 /
  data (epsData(179,i,30), i=1,3) / 0.2632430000E+05, -0.6043480000E-04, 0.1338670000E-05 /
  data (epsData(180,i,30), i=1,3) / 0.2945990000E+05, -0.4820210000E-04, 0.8612370000E-06 /
  data (epsData(181,i,30), i=1,3) / 0.3297820000E+05, -0.3841920000E-04, 0.5516300000E-06 /
  data (epsData(182,i,30), i=1,3) / 0.3692400000E+05, -0.3060900000E-04, 0.3518270000E-06 /
  data (epsData(183,i,30), i=1,3) / 0.4135510000E+05, -0.2437170000E-04, 0.2233080000E-06 /
  data (epsData(184,i,30), i=1,3) / 0.4633130000E+05, -0.1939560000E-04, 0.1410690000E-06 /
  data (epsData(185,i,30), i=1,3) / 0.5191250000E+05, -0.1543320000E-04, 0.8888770000E-07 /
  data (epsData(186,i,30), i=1,3) / 0.5818110000E+05, -0.1227530000E-04, 0.5562400000E-07 /
  data (epsData(187,i,30), i=1,3) / 0.6522160000E+05, -0.9760380000E-05, 0.3478190000E-07 /
  data (epsData(188,i,30), i=1,3) / 0.7312270000E+05, -0.7759630000E-05, 0.2166520000E-07 /
  data (epsData(189,i,30), i=1,3) / 0.8199420000E+05, -0.6167960000E-05, 0.1340300000E-07 /
  data (epsData(190,i,30), i=1,3) / 0.9194200000E+05, -0.4903860000E-05, 0.8240920000E-08 /
  data (epsData(191,i,30), i=1,3) / 0.1000000000E+06, -0.4159190000E-05, 0.5787310000E-08 /
  data (epsData(1,i,31), i=1,3) / 0.2506580000E-02, 0.3347560000E+03, 0.6998160000E+04 /
  data (epsData(2,i,31), i=1,3) / 0.4693440000E-02, -0.4517050000E+03, 0.2992550000E+04 /
  data (epsData(3,i,31), i=1,3) / 0.7560590000E-02, 0.5871900000E+02, 0.1656720000E+04 /
  data (epsData(4,i,31), i=1,3) / 0.1110800000E-01, 0.3089990000E+03, 0.1056110000E+04 /
  data (epsData(5,i,31), i=1,3) / 0.1533570000E-01, 0.4320000000E+03, 0.7345300000E+03 /
  data (epsData(6,i,31), i=1,3) / 0.2024370000E-01, 0.4972530000E+03, 0.5417590000E+03 /
  data (epsData(7,i,31), i=1,3) / 0.2583200000E-01, 0.5344310000E+03, 0.4167790000E+03 /
  data (epsData(8,i,31), i=1,3) / 0.3210060000E-01, 0.5569370000E+03, 0.3309610000E+03 /
  data (epsData(9,i,31), i=1,3) / 0.3904940000E-01, 0.5712900000E+03, 0.2693970000E+03 /
  data (epsData(10,i,31), i=1,3) / 0.4667860000E-01, 0.5808820000E+03, 0.2236800000E+03 /
  data (epsData(11,i,31), i=1,3) / 0.6397770000E-01, 0.5924900000E+03, 0.1614960000E+03 /
  data (epsData(12,i,31), i=1,3) / 0.9502860000E-01, 0.6021160000E+03, 0.1077160000E+03 /
  data (epsData(13,i,31), i=1,3) / 0.1603860000E+00, 0.6123170000E+03, 0.6493430000E+02 /
  data (epsData(14,i,31), i=1,3) / 0.2804580000E+00, 0.6360570000E+03, 0.3761010000E+02 /
  data (epsData(15,i,31), i=1,3) / 0.4838160000E+00, 0.6999420000E+03, 0.3885030000E+02 /
  data (epsData(16,i,31), i=1,3) / 0.5638470000E+00, 0.7469310000E+03, 0.3688420000E+02 /
  data (epsData(17,i,31), i=1,3) / 0.5918850000E+00, 0.7734140000E+03, 0.3610490000E+02 /
  data (epsData(18,i,31), i=1,3) / 0.7422780000E+00, 0.9028790000E+03, 0.1047410000E+03 /
  data (epsData(19,i,31), i=1,3) / 0.8071970000E+00, 0.9957970000E+03, 0.1201050000E+03 /
  data (epsData(20,i,31), i=1,3) / 0.8406760000E+00, 0.1078320000E+04, 0.1257220000E+03 /
  data (epsData(21,i,31), i=1,3) / 0.8748370000E+00, 0.1228070000E+04, 0.1302220000E+03 /
  data (epsData(22,i,31), i=1,3) / 0.9096770000E+00, 0.1443780000E+04, 0.2602800000E+03 /
  data (epsData(23,i,31), i=1,3) / 0.9451980000E+00, 0.1540850000E+04, 0.6838050000E+03 /
  data (epsData(24,i,31), i=1,3) / 0.9813990000E+00, 0.1416840000E+04, 0.1052810000E+04 /
  data (epsData(25,i,31), i=1,3) / 0.1001360000E+01, 0.1242530000E+04, 0.1222430000E+04 /
  data (epsData(26,i,31), i=1,3) / 0.1034010000E+01, 0.9966570000E+03, 0.1250300000E+04 /
  data (epsData(27,i,31), i=1,3) / 0.1122790000E+01, 0.5986050000E+03, 0.1299880000E+04 /
  data (epsData(28,i,31), i=1,3) / 0.1179940000E+01, 0.3566800000E+03, 0.1316610000E+04 /
  data (epsData(29,i,31), i=1,3) / 0.1195920000E+01, 0.2686660000E+03, 0.1319650000E+04 /
  data (epsData(30,i,31), i=1,3) / 0.1212590000E+01, 0.1768310000E+03, 0.1276670000E+04 /
  data (epsData(31,i,31), i=1,3) / 0.1247960000E+01, 0.4899970000E+02, 0.1161840000E+04 /
  data (epsData(32,i,31), i=1,3) / 0.1286060000E+01, -0.3472000000E+02, 0.1049940000E+04 /
  data (epsData(33,i,31), i=1,3) / 0.1348310000E+01, -0.1144580000E+03, 0.8901810000E+03 /
  data (epsData(34,i,31), i=1,3) / 0.1416670000E+01, -0.1539920000E+03, 0.7417580000E+03 /
  data (epsData(35,i,31), i=1,3) / 0.1491170000E+01, -0.1566330000E+03, 0.6059950000E+03 /
  data (epsData(36,i,31), i=1,3) / 0.1600010000E+01, -0.1260560000E+03, 0.4762360000E+03 /
  data (epsData(37,i,31), i=1,3) / 0.1719740000E+01, -0.8128170000E+02, 0.3749690000E+03 /
  data (epsData(38,i,31), i=1,3) / 0.1783690000E+01, -0.4681620000E+02, 0.3300640000E+03 /
  data (epsData(39,i,31), i=1,3) / 0.1816680000E+01, -0.2374440000E+02, 0.3111910000E+03 /
  data (epsData(40,i,31), i=1,3) / 0.1850360000E+01, 0.4419900000E+01, 0.2952130000E+03 /
  data (epsData(41,i,31), i=1,3) / 0.1884710000E+01, 0.4517540000E+02, 0.2798740000E+03 /
  data (epsData(42,i,31), i=1,3) / 0.1919740000E+01, 0.8968890000E+02, 0.3002270000E+03 /
  data (epsData(43,i,31), i=1,3) / 0.1955460000E+01, 0.1127120000E+03, 0.3459860000E+03 /
  data (epsData(44,i,31), i=1,3) / 0.2000000000E+01, 0.1026490000E+03, 0.3982550000E+03 /
  data (epsData(45,i,31), i=1,3) / 0.2066670000E+01, 0.7007280000E+02, 0.4133470000E+03 /
  data (epsData(46,i,31), i=1,3) / 0.2164630000E+01, 0.3056980000E+02, 0.4340340000E+03 /
  data (epsData(47,i,31), i=1,3) / 0.2195920000E+01, 0.8327650000E+01, 0.4400320000E+03 /
  data (epsData(48,i,31), i=1,3) / 0.2229940000E+01, -0.1671940000E+02, 0.4275900000E+03 /
  data (epsData(49,i,31), i=1,3) / 0.2326880000E+01, -0.4888010000E+02, 0.3884520000E+03 /
  data (epsData(50,i,31), i=1,3) / 0.2440820000E+01, -0.6302690000E+02, 0.3493960000E+03 /
  data (epsData(51,i,31), i=1,3) / 0.2628920000E+01, -0.6758520000E+02, 0.3050200000E+03 /
  data (epsData(52,i,31), i=1,3) / 0.2884710000E+01, -0.6697270000E+02, 0.2647360000E+03 /
  data (epsData(53,i,31), i=1,3) / 0.3307510000E+01, -0.6696810000E+02, 0.2237510000E+03 /
  data (epsData(54,i,31), i=1,3) / 0.3964660000E+01, -0.7429180000E+02, 0.1821660000E+03 /
  data (epsData(55,i,31), i=1,3) / 0.4941890000E+01, -0.9133090000E+02, 0.1254560000E+03 /
  data (epsData(56,i,31), i=1,3) / 0.5266720000E+01, -0.8896050000E+02, 0.1021010000E+03 /
  data (epsData(57,i,31), i=1,3) / 0.5538840000E+01, -0.8117400000E+02, 0.8732750000E+02 /
  data (epsData(58,i,31), i=1,3) / 0.6041230000E+01, -0.7098800000E+02, 0.7412060000E+02 /
  data (epsData(59,i,31), i=1,3) / 0.6898040000E+01, -0.6134910000E+02, 0.5658210000E+02 /
  data (epsData(60,i,31), i=1,3) / 0.8107590000E+01, -0.5184070000E+02, 0.3863460000E+02 /
  data (epsData(61,i,31), i=1,3) / 0.9450470000E+01, -0.4234620000E+02, 0.2437780000E+02 /
  data (epsData(62,i,31), i=1,3) / 0.1059910000E+02, -0.3480810000E+02, 0.1633750000E+02 /
  data (epsData(63,i,31), i=1,3) / 0.1171480000E+02, -0.2848310000E+02, 0.1123260000E+02 /
  data (epsData(64,i,31), i=1,3) / 0.1289850000E+02, -0.2293200000E+02, 0.7731320000E+01 /
  data (epsData(65,i,31), i=1,3) / 0.1415020000E+02, -0.1818420000E+02, 0.5365460000E+01 /
  data (epsData(66,i,31), i=1,3) / 0.1547000000E+02, -0.1412040000E+02, 0.3771350000E+01 /
  data (epsData(67,i,31), i=1,3) / 0.1743190000E+02, -0.8896230000E+01, 0.2392500000E+01 /
  data (epsData(68,i,31), i=1,3) / 0.1801690000E+02, -0.6973940000E+01, 0.2166970000E+01 /
  data (epsData(69,i,31), i=1,3) / 0.1816490000E+02, -0.6307790000E+01, 0.2158330000E+01 /
  data (epsData(70,i,31), i=1,3) / 0.1831350000E+02, -0.5436510000E+01, 0.2189820000E+01 /
  data (epsData(71,i,31), i=1,3) / 0.1846290000E+02, -0.4356850000E+01, 0.2616120000E+01 /
  data (epsData(72,i,31), i=1,3) / 0.1861290000E+02, -0.2805010000E+01, 0.3340290000E+01 /
  data (epsData(73,i,31), i=1,3) / 0.1870140000E+02, -0.2446230000E+01, 0.5656270000E+01 /
  data (epsData(74,i,31), i=1,3) / 0.1885000000E+02, -0.3706850000E+01, 0.7510550000E+01 /
  data (epsData(75,i,31), i=1,3) / 0.1889590000E+02, -0.4447400000E+01, 0.8071470000E+01 /
  data (epsData(76,i,31), i=1,3) / 0.1894800000E+02, -0.5341210000E+01, 0.7937650000E+01 /
  data (epsData(77,i,31), i=1,3) / 0.1909320000E+02, -0.6473860000E+01, 0.7394320000E+01 /
  data (epsData(78,i,31), i=1,3) / 0.1935850000E+02, -0.7289590000E+01, 0.6490320000E+01 /
  data (epsData(79,i,31), i=1,3) / 0.1965550000E+02, -0.7434110000E+01, 0.5713470000E+01 /
  data (epsData(80,i,31), i=1,3) / 0.2022690000E+02, -0.7146530000E+01, 0.4839690000E+01 /
  data (epsData(81,i,31), i=1,3) / 0.2145520000E+02, -0.6197730000E+01, 0.3993140000E+01 /
  data (epsData(82,i,31), i=1,3) / 0.2296670000E+02, -0.5228200000E+01, 0.3320420000E+01 /
  data (epsData(83,i,31), i=1,3) / 0.2471680000E+02, -0.4071370000E+01, 0.2897740000E+01 /
  data (epsData(84,i,31), i=1,3) / 0.2676680000E+02, -0.3012600000E+01, 0.2708510000E+01 /
  data (epsData(85,i,31), i=1,3) / 0.2899810000E+02, -0.2140910000E+01, 0.2668260000E+01 /
  data (epsData(86,i,31), i=1,3) / 0.3136990000E+02, -0.1499730000E+01, 0.2737300000E+01 /
  data (epsData(87,i,31), i=1,3) / 0.3427710000E+02, -0.1027400000E+01, 0.2838750000E+01 /
  data (epsData(88,i,31), i=1,3) / 0.3780540000E+02, -0.7229340000E+00, 0.2891700000E+01 /
  data (epsData(89,i,31), i=1,3) / 0.4349630000E+02, -0.4835500000E+00, 0.2855830000E+01 /
  data (epsData(90,i,31), i=1,3) / 0.5138760000E+02, -0.3896260000E+00, 0.2749990000E+01 /
  data (epsData(91,i,31), i=1,3) / 0.6180820000E+02, -0.4849640000E+00, 0.2533600000E+01 /
  data (epsData(92,i,31), i=1,3) / 0.7643710000E+02, -0.6426620000E+00, 0.2086800000E+01 /
  data (epsData(93,i,31), i=1,3) / 0.9319970000E+02, -0.6837890000E+00, 0.1644980000E+01 /
  data (epsData(94,i,31), i=1,3) / 0.1006000000E+03, -0.6513270000E+00, 0.1510160000E+01 /
  data (epsData(95,i,31), i=1,3) / 0.1074670000E+03, -0.7017460000E+00, 0.1430040000E+01 /
  data (epsData(96,i,31), i=1,3) / 0.1195170000E+03, -0.7382240000E+00, 0.1174870000E+01 /
  data (epsData(97,i,31), i=1,3) / 0.1357670000E+03, -0.6862470000E+00, 0.9182540000E+00 /
  data (epsData(98,i,31), i=1,3) / 0.1565710000E+03, -0.6001830000E+00, 0.6997380000E+00 /
  data (epsData(99,i,31), i=1,3) / 0.1642360000E+03, -0.5814340000E+00, 0.6529180000E+00 /
  data (epsData(100,i,31), i=1,3) / 0.1905220000E+03, -0.5140720000E+00, 0.4787660000E+00 /
  data (epsData(101,i,31), i=1,3) / 0.2180810000E+03, -0.4436070000E+00, 0.3528320000E+00 /
  data (epsData(102,i,31), i=1,3) / 0.2490160000E+03, -0.3759260000E+00, 0.2548700000E+00 /
  data (epsData(103,i,31), i=1,3) / 0.2795160000E+03, -0.3191150000E+00, 0.1890570000E+00 /
  data (epsData(104,i,31), i=1,3) / 0.3131060000E+03, -0.2682970000E+00, 0.1390080000E+00 /
  data (epsData(105,i,31), i=1,3) / 0.3493060000E+03, -0.2243050000E+00, 0.1020080000E+00 /
  data (epsData(106,i,31), i=1,3) / 0.3882170000E+03, -0.1867460000E+00, 0.7502930000E-01 /
  data (epsData(107,i,31), i=1,3) / 0.4277240000E+03, -0.1567330000E+00, 0.5681120000E-01 /
  data (epsData(108,i,31), i=1,3) / 0.4861810000E+03, -0.1234690000E+00, 0.3798730000E-01 /
  data (epsData(109,i,31), i=1,3) / 0.5453090000E+03, -0.9914550000E-01, 0.2678140000E-01 /
  data (epsData(110,i,31), i=1,3) / 0.5961500000E+03, -0.8274590000E-01, 0.2021420000E-01 /
  data (epsData(111,i,31), i=1,3) / 0.6699420000E+03, -0.6505360000E-01, 0.1381030000E-01 /
  data (epsData(112,i,31), i=1,3) / 0.7494920000E+03, -0.5117520000E-01, 0.9613590000E-02 /
  data (epsData(113,i,31), i=1,3) / 0.8290610000E+03, -0.4065900000E-01, 0.6938360000E-02 /
  data (epsData(114,i,31), i=1,3) / 0.9177190000E+03, -0.3150650000E-01, 0.4954570000E-02 /
  data (epsData(115,i,31), i=1,3) / 0.1057720000E+04, -0.1908550000E-01, 0.3076910000E-02 /
  data (epsData(116,i,31), i=1,3) / 0.1088940000E+04, -0.1535750000E-01, 0.2783090000E-02 /
  data (epsData(117,i,31), i=1,3) / 0.1103620000E+04, -0.1270270000E-01, 0.2710720000E-02 /
  data (epsData(118,i,31), i=1,3) / 0.1110430000E+04, -0.1080700000E-01, 0.2696470000E-02 /
  data (epsData(119,i,31), i=1,3) / 0.1113840000E+04, -0.9365260000E-02, 0.2730570000E-02 /
  data (epsData(120,i,31), i=1,3) / 0.1114980000E+04, -0.8695150000E-02, 0.2771200000E-02 /
  data (epsData(121,i,31), i=1,3) / 0.1116500000E+04, -0.7540800000E-02, 0.3052710000E-02 /
  data (epsData(122,i,31), i=1,3) / 0.1118110000E+04, -0.6214900000E-02, 0.3950370000E-02 /
  data (epsData(123,i,31), i=1,3) / 0.1119410000E+04, -0.5480500000E-02, 0.5154350000E-02 /
  data (epsData(124,i,31), i=1,3) / 0.1120900000E+04, -0.4976160000E-02, 0.6631640000E-02 /
  data (epsData(125,i,31), i=1,3) / 0.1122330000E+04, -0.5089240000E-02, 0.8204610000E-02 /
  data (epsData(126,i,31), i=1,3) / 0.1123950000E+04, -0.5812150000E-02, 0.9537200000E-02 /
  data (epsData(127,i,31), i=1,3) / 0.1126460000E+04, -0.7023970000E-02, 0.1052890000E-01 /
  data (epsData(128,i,31), i=1,3) / 0.1130560000E+04, -0.8183040000E-02, 0.1110870000E-01 /
  data (epsData(129,i,31), i=1,3) / 0.1139390000E+04, -0.9037170000E-02, 0.1184320000E-01 /
  data (epsData(130,i,31), i=1,3) / 0.1143510000E+04, -0.8669090000E-02, 0.1219680000E-01 /
  data (epsData(131,i,31), i=1,3) / 0.1146530000E+04, -0.8293960000E-02, 0.1331430000E-01 /
  data (epsData(132,i,31), i=1,3) / 0.1149870000E+04, -0.8889480000E-02, 0.1493000000E-01 /
  data (epsData(133,i,31), i=1,3) / 0.1154470000E+04, -0.1061120000E-01, 0.1573050000E-01 /
  data (epsData(134,i,31), i=1,3) / 0.1166370000E+04, -0.1295340000E-01, 0.1581380000E-01 /
  data (epsData(135,i,31), i=1,3) / 0.1187280000E+04, -0.1533120000E-01, 0.1524790000E-01 /
  data (epsData(136,i,31), i=1,3) / 0.1228240000E+04, -0.1697370000E-01, 0.1348270000E-01 /
  data (epsData(137,i,31), i=1,3) / 0.1278200000E+04, -0.1689190000E-01, 0.1164170000E-01 /
  data (epsData(138,i,31), i=1,3) / 0.1295890000E+04, -0.1618620000E-01, 0.1144420000E-01 /
  data (epsData(139,i,31), i=1,3) / 0.1302680000E+04, -0.1631390000E-01, 0.1200740000E-01 /
  data (epsData(140,i,31), i=1,3) / 0.1330020000E+04, -0.1701940000E-01, 0.1139370000E-01 /
  data (epsData(141,i,31), i=1,3) / 0.1430170000E+04, -0.1654010000E-01, 0.8938540000E-02 /
  data (epsData(142,i,31), i=1,3) / 0.1599560000E+04, -0.1443150000E-01, 0.6100270000E-02 /
  data (epsData(143,i,31), i=1,3) / 0.1803620000E+04, -0.1195440000E-01, 0.4055840000E-02 /
  data (epsData(144,i,31), i=1,3) / 0.2018120000E+04, -0.9828140000E-02, 0.2746880000E-02 /
  data (epsData(145,i,31), i=1,3) / 0.2249930000E+04, -0.8040440000E-02, 0.1872350000E-02 /
  data (epsData(146,i,31), i=1,3) / 0.2502520000E+04, -0.6558410000E-02, 0.1280780000E-02 /
  data (epsData(147,i,31), i=1,3) / 0.2781990000E+04, -0.5330360000E-02, 0.8798140000E-03 /
  data (epsData(148,i,31), i=1,3) / 0.3096860000E+04, -0.4310210000E-02, 0.5968950000E-03 /
  data (epsData(149,i,31), i=1,3) / 0.3442010000E+04, -0.3486280000E-02, 0.4053650000E-03 /
  data (epsData(150,i,31), i=1,3) / 0.3826720000E+04, -0.2813490000E-02, 0.2742690000E-03 /
  data (epsData(151,i,31), i=1,3) / 0.4257200000E+04, -0.2265070000E-02, 0.1846990000E-03 /
  data (epsData(152,i,31), i=1,3) / 0.4738790000E+04, -0.1819960000E-02, 0.1238470000E-03 /
  data (epsData(153,i,31), i=1,3) / 0.5277500000E+04, -0.1459950000E-02, 0.8272130000E-04 /
  data (epsData(154,i,31), i=1,3) / 0.5880310000E+04, -0.1169290000E-02, 0.5504770000E-04 /
  data (epsData(155,i,31), i=1,3) / 0.6549640000E+04, -0.9366060000E-03, 0.3681480000E-04 /
  data (epsData(156,i,31), i=1,3) / 0.7301670000E+04, -0.7479590000E-03, 0.2430960000E-04 /
  data (epsData(157,i,31), i=1,3) / 0.7708360000E+04, -0.6680150000E-03, 0.1979490000E-04 /
  data (epsData(158,i,31), i=1,3) / 0.7709000000E+04, -0.6304150000E-03, 0.1978870000E-04 /
  data (epsData(159,i,31), i=1,3) / 0.7709000000E+04, -0.6678990000E-03, 0.1978870000E-04 /
  data (epsData(160,i,31), i=1,3) / 0.8614980000E+04, -0.5280810000E-03, 0.1290710000E-04 /
  data (epsData(161,i,31), i=1,3) / 0.9762470000E+04, -0.3978830000E-03, 0.7997710000E-05 /
  data (epsData(162,i,31), i=1,3) / 0.1025720000E+05, -0.3393040000E-03, 0.6616330000E-05 /
  data (epsData(163,i,31), i=1,3) / 0.1033980000E+05, -0.3156080000E-03, 0.6412890000E-05 /
  data (epsData(164,i,31), i=1,3) / 0.1035870000E+05, -0.2982680000E-03, 0.8011450000E-05 /
  data (epsData(165,i,31), i=1,3) / 0.1036250000E+05, -0.2893720000E-03, 0.9565060000E-05 /
  data (epsData(166,i,31), i=1,3) / 0.1036440000E+05, -0.2817620000E-03, 0.1206410000E-04 /
  data (epsData(167,i,31), i=1,3) / 0.1036630000E+05, -0.2725420000E-03, 0.2248670000E-04 /
  data (epsData(168,i,31), i=1,3) / 0.1036700000E+05, -0.2121290000E-03, 0.3231330000E-04 /
  data (epsData(169,i,31), i=1,3) / 0.1036700000E+05, -0.2712400000E-03, 0.3301120000E-04 /
  data (epsData(170,i,31), i=1,3) / 0.1036860000E+05, -0.2831940000E-03, 0.4286290000E-04 /
  data (epsData(171,i,31), i=1,3) / 0.1037400000E+05, -0.2968760000E-03, 0.4464940000E-04 /
  data (epsData(172,i,31), i=1,3) / 0.1038980000E+05, -0.3112760000E-03, 0.4609900000E-04 /
  data (epsData(173,i,31), i=1,3) / 0.1047440000E+05, -0.3259940000E-03, 0.4649120000E-04 /
  data (epsData(174,i,31), i=1,3) / 0.1082080000E+05, -0.3248250000E-03, 0.4145960000E-04 /
  data (epsData(175,i,31), i=1,3) / 0.1280650000E+05, -0.2474580000E-03, 0.2284900000E-04 /
  data (epsData(176,i,31), i=1,3) / 0.1442260000E+05, -0.1979690000E-03, 0.1480900000E-04 /
  data (epsData(177,i,31), i=1,3) / 0.1617410000E+05, -0.1585540000E-03, 0.9686530000E-05 /
  data (epsData(178,i,31), i=1,3) / 0.1810880000E+05, -0.1269460000E-03, 0.6346890000E-05 /
  data (epsData(179,i,31), i=1,3) / 0.2025510000E+05, -0.1016320000E-03, 0.4154560000E-05 /
  data (epsData(180,i,31), i=1,3) / 0.2265540000E+05, -0.8126480000E-04, 0.2708550000E-05 /
  data (epsData(181,i,31), i=1,3) / 0.2534380000E+05, -0.6491180000E-04, 0.1757030000E-05 /
  data (epsData(182,i,31), i=1,3) / 0.2835300000E+05, -0.5181960000E-04, 0.1135810000E-05 /
  data (epsData(183,i,31), i=1,3) / 0.3172910000E+05, -0.4133260000E-04, 0.7305300000E-06 /
  data (epsData(184,i,31), i=1,3) / 0.3551810000E+05, -0.3294400000E-04, 0.4676110000E-06 /
  data (epsData(185,i,31), i=1,3) / 0.3976790000E+05, -0.2624640000E-04, 0.2979910000E-06 /
  data (epsData(186,i,31), i=1,3) / 0.4453990000E+05, -0.2089830000E-04, 0.1889680000E-06 /
  data (epsData(187,i,31), i=1,3) / 0.4989780000E+05, -0.1663240000E-04, 0.1193510000E-06 /
  data (epsData(188,i,31), i=1,3) / 0.5591300000E+05, -0.1323260000E-04, 0.7502010000E-07 /
  data (epsData(189,i,31), i=1,3) / 0.6265720000E+05, -0.1052750000E-04, 0.4723350000E-07 /
  data (epsData(190,i,31), i=1,3) / 0.7023330000E+05, -0.8372310000E-05, 0.2933040000E-07 /
  data (epsData(191,i,31), i=1,3) / 0.7874230000E+05, -0.6656260000E-05, 0.1826560000E-07 /
  data (epsData(192,i,31), i=1,3) / 0.8828910000E+05, -0.5292150000E-05, 0.1127500000E-07 /
  data (epsData(193,i,31), i=1,3) / 0.9888680000E+05, -0.4220430000E-05, 0.6984290000E-08 /
  data (epsData(194,i,31), i=1,3) / 0.1000000000E+06, -0.4141080000E-05, 0.6658770000E-08 /
  data (epsData(1,i,32), i=1,3) / 0.2506580000E-02, -0.3900640000E+03, 0.1284350000E+04 /
  data (epsData(2,i,32), i=1,3) / 0.4693440000E-02, -0.3277130000E+03, 0.4462510000E+03 /
  data (epsData(3,i,32), i=1,3) / 0.7560590000E-02, -0.1180400000E+03, 0.2123520000E+03 /
  data (epsData(4,i,32), i=1,3) / 0.1110800000E-01, -0.2612150000E+02, 0.1215250000E+03 /
  data (epsData(5,i,32), i=1,3) / 0.1533570000E-01, 0.1719450000E+02, 0.7823020000E+02 /
  data (epsData(6,i,32), i=1,3) / 0.2024370000E-01, 0.3963130000E+02, 0.5453790000E+02 /
  data (epsData(7,i,32), i=1,3) / 0.2583200000E-01, 0.5220120000E+02, 0.4023620000E+02 /
  data (epsData(8,i,32), i=1,3) / 0.3210060000E-01, 0.5970110000E+02, 0.3095390000E+02 /
  data (epsData(9,i,32), i=1,3) / 0.3904940000E-01, 0.6441030000E+02, 0.2458660000E+02 /
  data (epsData(10,i,32), i=1,3) / 0.4667860000E-01, 0.6749450000E+02, 0.2002550000E+02 /
  data (epsData(11,i,32), i=1,3) / 0.5498800000E-01, 0.6958740000E+02, 0.1664310000E+02 /
  data (epsData(12,i,32), i=1,3) / 0.7364770000E-01, 0.7210310000E+02, 0.1204780000E+02 /
  data (epsData(13,i,32), i=1,3) / 0.1067390000E+00, 0.7390800000E+02, 0.8077210000E+01 /
  data (epsData(14,i,32), i=1,3) / 0.1912910000E+00, 0.7528040000E+02, 0.4387700000E+01 /
  data (epsData(15,i,32), i=1,3) / 0.5918850000E+00, 0.7668550000E+02, 0.1703270000E+01 /
  data (epsData(16,i,32), i=1,3) / 0.1337820000E+01, 0.8189640000E+02, 0.1108780000E+01 /
  data (epsData(17,i,32), i=1,3) / 0.1898770000E+01, 0.9044740000E+02, 0.1194500000E+01 /
  data (epsData(18,i,32), i=1,3) / 0.2271220000E+01, 0.1006780000E+03, 0.1466960000E+01 /
  data (epsData(19,i,32), i=1,3) / 0.2557670000E+01, 0.1144450000E+03, 0.2113470000E+01 /
  data (epsData(20,i,32), i=1,3) / 0.2737710000E+01, 0.1293630000E+03, 0.4570810000E+01 /
  data (epsData(21,i,32), i=1,3) / 0.2799080000E+01, 0.1368560000E+03, 0.5478940000E+01 /
  data (epsData(22,i,32), i=1,3) / 0.2861130000E+01, 0.1488680000E+03, 0.6315200000E+01 /
  data (epsData(23,i,32), i=1,3) / 0.2923860000E+01, 0.1639380000E+03, 0.1697450000E+02 /
  data (epsData(24,i,32), i=1,3) / 0.3000000000E+01, 0.1713030000E+03, 0.4729610000E+02 /
  data (epsData(25,i,32), i=1,3) / 0.3150000000E+01, 0.1564020000E+03, 0.8246170000E+02 /
  data (epsData(26,i,32), i=1,3) / 0.3212590000E+01, 0.1429270000E+03, 0.9304300000E+02 /
  data (epsData(27,i,32), i=1,3) / 0.3348310000E+01, 0.1240470000E+03, 0.9641720000E+02 /
  data (epsData(28,i,32), i=1,3) / 0.3719740000E+01, 0.9762050000E+02, 0.1030490000E+03 /
  data (epsData(29,i,32), i=1,3) / 0.4437100000E+01, 0.6444520000E+02, 0.1112260000E+03 /
  data (epsData(30,i,32), i=1,3) / 0.4862620000E+01, 0.4944070000E+02, 0.1133850000E+03 /
  data (epsData(31,i,32), i=1,3) / 0.4964660000E+01, 0.4737460000E+02, 0.1170640000E+03 /
  data (epsData(32,i,32), i=1,3) / 0.5212590000E+01, 0.3061460000E+02, 0.1221400000E+03 /
  data (epsData(33,i,32), i=1,3) / 0.5517360000E+01, 0.1281070000E+02, 0.1164950000E+03 /
  data (epsData(34,i,32), i=1,3) / 0.5955460000E+01, -0.6538770000E+01, 0.1050740000E+03 /
  data (epsData(35,i,32), i=1,3) / 0.6265670000E+01, -0.1673660000E+02, 0.9337110000E+02 /
  data (epsData(36,i,32), i=1,3) / 0.6481660000E+01, -0.1866210000E+02, 0.8319070000E+02 /
  data (epsData(37,i,32), i=1,3) / 0.6812620000E+01, -0.1825540000E+02, 0.7579410000E+02 /
  data (epsData(38,i,32), i=1,3) / 0.8821840000E+01, -0.2945820000E+02, 0.4761320000E+02 /
  data (epsData(39,i,32), i=1,3) / 0.9736140000E+01, -0.3028220000E+02, 0.3580130000E+02 /
  data (epsData(40,i,32), i=1,3) / 0.1066030000E+02, -0.2863650000E+02, 0.2639650000E+02 /
  data (epsData(41,i,32), i=1,3) / 0.1176240000E+02, -0.2534380000E+02, 0.1829960000E+02 /
  data (epsData(42,i,32), i=1,3) / 0.1296240000E+02, -0.2144850000E+02, 0.1240580000E+02 /
  data (epsData(43,i,32), i=1,3) / 0.1414850000E+02, -0.1793370000E+02, 0.8635260000E+01 /
  data (epsData(44,i,32), i=1,3) / 0.1541680000E+02, -0.1476630000E+02, 0.6014090000E+01 /
  data (epsData(45,i,32), i=1,3) / 0.1676760000E+02, -0.1202850000E+02, 0.4221250000E+01 /
  data (epsData(46,i,32), i=1,3) / 0.1820060000E+02, -0.9721700000E+01, 0.2994590000E+01 /
  data (epsData(47,i,32), i=1,3) / 0.1971590000E+02, -0.7799030000E+01, 0.2155450000E+01 /
  data (epsData(48,i,32), i=1,3) / 0.2131350000E+02, -0.6198520000E+01, 0.1576660000E+01 /
  data (epsData(49,i,32), i=1,3) / 0.2315030000E+02, -0.4742670000E+01, 0.1144020000E+01 /
  data (epsData(50,i,32), i=1,3) / 0.2508500000E+02, -0.3510650000E+01, 0.8483710000E+00 /
  data (epsData(51,i,32), i=1,3) / 0.2799360000E+02, -0.1895240000E+01, 0.5829440000E+00 /
  data (epsData(52,i,32), i=1,3) / 0.2852730000E+02, -0.1537360000E+01, 0.5551050000E+00 /
  data (epsData(53,i,32), i=1,3) / 0.2870650000E+02, -0.1386630000E+01, 0.5516930000E+00 /
  data (epsData(54,i,32), i=1,3) / 0.2888650000E+02, -0.1207940000E+01, 0.5701940000E+00 /
  data (epsData(55,i,32), i=1,3) / 0.2906710000E+02, -0.9711300000E+00, 0.6115580000E+00 /
  data (epsData(56,i,32), i=1,3) / 0.2922180000E+02, -0.8078590000E+00, 0.8516380000E+00 /
  data (epsData(57,i,32), i=1,3) / 0.2936460000E+02, -0.7879610000E+00, 0.1050740000E+01 /
  data (epsData(58,i,32), i=1,3) / 0.2942990000E+02, -0.8321250000E+00, 0.1108330000E+01 /
  data (epsData(59,i,32), i=1,3) / 0.2964080000E+02, -0.8248430000E+00, 0.1170070000E+01 /
  data (epsData(60,i,32), i=1,3) / 0.2971740000E+02, -0.7768260000E+00, 0.1207980000E+01 /
  data (epsData(61,i,32), i=1,3) / 0.2983400000E+02, -0.7698340000E+00, 0.1335120000E+01 /
  data (epsData(62,i,32), i=1,3) / 0.2999590000E+02, -0.8341300000E+00, 0.1470160000E+01 /
  data (epsData(63,i,32), i=1,3) / 0.3026570000E+02, -0.9391290000E+00, 0.1499250000E+01 /
  data (epsData(64,i,32), i=1,3) / 0.3098400000E+02, -0.9889750000E+00, 0.1504510000E+01 /
  data (epsData(65,i,32), i=1,3) / 0.3274190000E+02, -0.8949740000E+00, 0.1472160000E+01 /
  data (epsData(66,i,32), i=1,3) / 0.3384120000E+02, -0.7473880000E+00, 0.1408590000E+01 /
  data (epsData(67,i,32), i=1,3) / 0.3906040000E+02, -0.1367960000E+00, 0.1369440000E+01 /
  data (epsData(68,i,32), i=1,3) / 0.4119100000E+02, 0.7450310000E-01, 0.1424920000E+01 /
  data (epsData(69,i,32), i=1,3) / 0.4423430000E+02, 0.2704030000E+00, 0.1543600000E+01 /
  data (epsData(70,i,32), i=1,3) / 0.4795030000E+02, 0.3790630000E+00, 0.1675180000E+01 /
  data (epsData(71,i,32), i=1,3) / 0.5332730000E+02, 0.4085580000E+00, 0.1793670000E+01 /
  data (epsData(72,i,32), i=1,3) / 0.6123460000E+02, 0.3400020000E+00, 0.1882300000E+01 /
  data (epsData(73,i,32), i=1,3) / 0.7005100000E+02, 0.1664800000E+00, 0.1891960000E+01 /
  data (epsData(74,i,32), i=1,3) / 0.7943490000E+02, -0.3681080000E-01, 0.1789720000E+01 /
  data (epsData(75,i,32), i=1,3) / 0.9595140000E+02, -0.2561220000E+00, 0.1517490000E+01 /
  data (epsData(76,i,32), i=1,3) / 0.1141570000E+03, -0.3793410000E+00, 0.1229960000E+01 /
  data (epsData(77,i,32), i=1,3) / 0.1212410000E+03, -0.3809340000E+00, 0.1145080000E+01 /
  data (epsData(78,i,32), i=1,3) / 0.1316620000E+03, -0.4475400000E+00, 0.1036530000E+01 /
  data (epsData(79,i,32), i=1,3) / 0.1466720000E+03, -0.4693490000E+00, 0.8500870000E+00 /
  data (epsData(80,i,32), i=1,3) / 0.1687380000E+03, -0.4490930000E+00, 0.6512980000E+00 /
  data (epsData(81,i,32), i=1,3) / 0.1823870000E+03, -0.4216620000E+00, 0.5686820000E+00 /
  data (epsData(82,i,32), i=1,3) / 0.2127880000E+03, -0.3909110000E+00, 0.4127710000E+00 /
  data (epsData(83,i,32), i=1,3) / 0.2505180000E+03, -0.3331140000E+00, 0.2819540000E+00 /
  data (epsData(84,i,32), i=1,3) / 0.2860090000E+03, -0.2831480000E+00, 0.2014820000E+00 /
  data (epsData(85,i,32), i=1,3) / 0.3220490000E+03, -0.2398220000E+00, 0.1466490000E+00 /
  data (epsData(86,i,32), i=1,3) / 0.3603740000E+03, -0.2016640000E+00, 0.1069980000E+00 /
  data (epsData(87,i,32), i=1,3) / 0.4016220000E+03, -0.1683810000E+00, 0.7815550000E-01 /
  data (epsData(88,i,32), i=1,3) / 0.4441120000E+03, -0.1412960000E+00, 0.5863830000E-01 /
  data (epsData(89,i,32), i=1,3) / 0.5027930000E+03, -0.1126180000E+00, 0.3978810000E-01 /
  data (epsData(90,i,32), i=1,3) / 0.5666340000E+03, -0.9005890000E-01, 0.2743140000E-01 /
  data (epsData(91,i,32), i=1,3) / 0.6206120000E+03, -0.7510550000E-01, 0.2069750000E-01 /
  data (epsData(92,i,32), i=1,3) / 0.6930450000E+03, -0.5997410000E-01, 0.1444640000E-01 /
  data (epsData(93,i,32), i=1,3) / 0.7781830000E+03, -0.4700570000E-01, 0.9857120000E-02 /
  data (epsData(94,i,32), i=1,3) / 0.8688870000E+03, -0.3684040000E-01, 0.6881730000E-02 /
  data (epsData(95,i,32), i=1,3) / 0.9618600000E+03, -0.2886840000E-01, 0.4928960000E-02 /
  data (epsData(96,i,32), i=1,3) / 0.1073900000E+04, -0.2115060000E-01, 0.3406710000E-02 /
  data (epsData(97,i,32), i=1,3) / 0.1160830000E+04, -0.1518030000E-01, 0.2614480000E-02 /
  data (epsData(98,i,32), i=1,3) / 0.1191100000E+04, -0.1228000000E-01, 0.2398260000E-02 /
  data (epsData(99,i,32), i=1,3) / 0.1205220000E+04, -0.1018660000E-01, 0.2350010000E-02 /
  data (epsData(100,i,32), i=1,3) / 0.1212320000E+04, -0.8497050000E-02, 0.2349380000E-02 /
  data (epsData(101,i,32), i=1,3) / 0.1214690000E+04, -0.7632200000E-02, 0.2379400000E-02 /
  data (epsData(102,i,32), i=1,3) / 0.1215880000E+04, -0.7052020000E-02, 0.2423070000E-02 /
  data (epsData(103,i,32), i=1,3) / 0.1217420000E+04, -0.6096160000E-02, 0.2719080000E-02 /
  data (epsData(104,i,32), i=1,3) / 0.1218910000E+04, -0.5110560000E-02, 0.3468690000E-02 /
  data (epsData(105,i,32), i=1,3) / 0.1220680000E+04, -0.4416860000E-02, 0.4819970000E-02 /
  data (epsData(106,i,32), i=1,3) / 0.1221900000E+04, -0.4230340000E-02, 0.5940280000E-02 /
  data (epsData(107,i,32), i=1,3) / 0.1223290000E+04, -0.4573850000E-02, 0.7088560000E-02 /
  data (epsData(108,i,32), i=1,3) / 0.1225710000E+04, -0.5535890000E-02, 0.8190370000E-02 /
  data (epsData(109,i,32), i=1,3) / 0.1229020000E+04, -0.6491150000E-02, 0.8751010000E-02 /
  data (epsData(110,i,32), i=1,3) / 0.1237750000E+04, -0.7518070000E-02, 0.9456980000E-02 /
  data (epsData(111,i,32), i=1,3) / 0.1246200000E+04, -0.7698770000E-02, 0.9848920000E-02 /
  data (epsData(112,i,32), i=1,3) / 0.1249810000E+04, -0.7176420000E-02, 0.1034320000E-01 /
  data (epsData(113,i,32), i=1,3) / 0.1253250000E+04, -0.7179440000E-02, 0.1164410000E-01 /
  data (epsData(114,i,32), i=1,3) / 0.1257140000E+04, -0.8279610000E-02, 0.1254260000E-01 /
  data (epsData(115,i,32), i=1,3) / 0.1266090000E+04, -0.9805990000E-02, 0.1280670000E-01 /
  data (epsData(116,i,32), i=1,3) / 0.1285820000E+04, -0.1189440000E-01, 0.1279650000E-01 /
  data (epsData(117,i,32), i=1,3) / 0.1325490000E+04, -0.1361040000E-01, 0.1160340000E-01 /
  data (epsData(118,i,32), i=1,3) / 0.1382680000E+04, -0.1399380000E-01, 0.9965970000E-02 /
  data (epsData(119,i,32), i=1,3) / 0.1410980000E+04, -0.1332640000E-01, 0.9495550000E-02 /
  data (epsData(120,i,32), i=1,3) / 0.1420620000E+04, -0.1334580000E-01, 0.1002330000E-01 /
  data (epsData(121,i,32), i=1,3) / 0.1444810000E+04, -0.1395130000E-01, 0.9628440000E-02 /
  data (epsData(122,i,32), i=1,3) / 0.1547900000E+04, -0.1371030000E-01, 0.7629800000E-02 /
  data (epsData(123,i,32), i=1,3) / 0.1727430000E+04, -0.1205780000E-01, 0.5248790000E-02 /
  data (epsData(124,i,32), i=1,3) / 0.1948640000E+04, -0.1000810000E-01, 0.3475840000E-02 /
  data (epsData(125,i,32), i=1,3) / 0.2177910000E+04, -0.8252420000E-02, 0.2361040000E-02 /
  data (epsData(126,i,32), i=1,3) / 0.2426770000E+04, -0.6761320000E-02, 0.1611990000E-02 /
  data (epsData(127,i,32), i=1,3) / 0.2698770000E+04, -0.5515460000E-02, 0.1104670000E-02 /
  data (epsData(128,i,32), i=1,3) / 0.3000160000E+04, -0.4486530000E-02, 0.7582910000E-03 /
  data (epsData(129,i,32), i=1,3) / 0.3334900000E+04, -0.3637270000E-02, 0.5158290000E-03 /
  data (epsData(130,i,32), i=1,3) / 0.3705220000E+04, -0.2943880000E-02, 0.3506420000E-03 /
  data (epsData(131,i,32), i=1,3) / 0.4117740000E+04, -0.2377460000E-02, 0.2374840000E-03 /
  data (epsData(132,i,32), i=1,3) / 0.4578900000E+04, -0.1915520000E-02, 0.1601120000E-03 /
  data (epsData(133,i,32), i=1,3) / 0.5094210000E+04, -0.1540560000E-02, 0.1075230000E-03 /
  data (epsData(134,i,32), i=1,3) / 0.5672330000E+04, -0.1236060000E-02, 0.7183640000E-04 /
  data (epsData(135,i,32), i=1,3) / 0.6318410000E+04, -0.9902960000E-03, 0.4800720000E-04 /
  data (epsData(136,i,32), i=1,3) / 0.7048800000E+04, -0.7905700000E-03, 0.3175540000E-04 /
  data (epsData(137,i,32), i=1,3) / 0.7856480000E+04, -0.6314640000E-03, 0.2100430000E-04 /
  data (epsData(138,i,32), i=1,3) / 0.8332650000E+04, -0.5584330000E-03, 0.1678460000E-04 /
  data (epsData(139,i,32), i=1,3) / 0.8333000000E+04, -0.5298330000E-03, 0.1678200000E-04 /
  data (epsData(140,i,32), i=1,3) / 0.8333000000E+04, -0.5583840000E-03, 0.1678200000E-04 /
  data (epsData(141,i,32), i=1,3) / 0.9316050000E+04, -0.4407250000E-03, 0.1092580000E-04 /
  data (epsData(142,i,32), i=1,3) / 0.1069660000E+05, -0.3186700000E-03, 0.6434160000E-05 /
  data (epsData(143,i,32), i=1,3) / 0.1101890000E+05, -0.2832460000E-03, 0.5736480000E-05 /
  data (epsData(144,i,32), i=1,3) / 0.1108040000E+05, -0.2652500000E-03, 0.5611280000E-05 /
  data (epsData(145,i,32), i=1,3) / 0.1109590000E+05, -0.2521690000E-03, 0.7161560000E-05 /
  data (epsData(146,i,32), i=1,3) / 0.1109980000E+05, -0.2442210000E-03, 0.8855120000E-05 /
  data (epsData(147,i,32), i=1,3) / 0.1110170000E+05, -0.2378750000E-03, 0.1221400000E-04 /
  data (epsData(148,i,32), i=1,3) / 0.1110300000E+05, -0.1960600000E-03, 0.1902520000E-04 /
  data (epsData(149,i,32), i=1,3) / 0.1110300000E+05, -0.2339340000E-03, 0.1938340000E-04 /
  data (epsData(150,i,32), i=1,3) / 0.1110540000E+05, -0.2379520000E-03, 0.3110180000E-04 /
  data (epsData(151,i,32), i=1,3) / 0.1111270000E+05, -0.2520770000E-03, 0.4038490000E-04 /
  data (epsData(152,i,32), i=1,3) / 0.1113020000E+05, -0.2652410000E-03, 0.4042780000E-04 /
  data (epsData(153,i,32), i=1,3) / 0.1122870000E+05, -0.2779940000E-03, 0.4029780000E-04 /
  data (epsData(154,i,32), i=1,3) / 0.1163450000E+05, -0.2754880000E-03, 0.3549200000E-04 /
  data (epsData(155,i,32), i=1,3) / 0.1377740000E+05, -0.2092460000E-03, 0.1949440000E-04 /
  data (epsData(156,i,32), i=1,3) / 0.1552350000E+05, -0.1672130000E-03, 0.1259600000E-04 /
  data (epsData(157,i,32), i=1,3) / 0.1740650000E+05, -0.1339350000E-03, 0.8240270000E-05 /
  data (epsData(158,i,32), i=1,3) / 0.1948380000E+05, -0.1072770000E-03, 0.5401340000E-05 /
  data (epsData(159,i,32), i=1,3) / 0.2179530000E+05, -0.8585810000E-04, 0.3531250000E-05 /
  data (epsData(160,i,32), i=1,3) / 0.2437520000E+05, -0.6866470000E-04, 0.2301690000E-05 /
  data (epsData(161,i,32), i=1,3) / 0.2726260000E+05, -0.5486440000E-04, 0.1493420000E-05 /
  data (epsData(162,i,32), i=1,3) / 0.3050110000E+05, -0.4379250000E-04, 0.9645030000E-06 /
  data (epsData(163,i,32), i=1,3) / 0.3413020000E+05, -0.3493430000E-04, 0.6199980000E-06 /
  data (epsData(164,i,32), i=1,3) / 0.3820370000E+05, -0.2784650000E-04, 0.3966850000E-06 /
  data (epsData(165,i,32), i=1,3) / 0.4277280000E+05, -0.2218680000E-04, 0.2526470000E-06 /
  data (epsData(166,i,32), i=1,3) / 0.4790260000E+05, -0.1766760000E-04, 0.1601460000E-06 /
  data (epsData(167,i,32), i=1,3) / 0.5366040000E+05, -0.1406330000E-04, 0.1010830000E-06 /
  data (epsData(168,i,32), i=1,3) / 0.6013020000E+05, -0.1118800000E-04, 0.6346880000E-07 /
  data (epsData(169,i,32), i=1,3) / 0.6740690000E+05, -0.8894830000E-05, 0.3980250000E-07 /
  data (epsData(170,i,32), i=1,3) / 0.7555500000E+05, -0.7074380000E-05, 0.2473420000E-07 /
  data (epsData(171,i,32), i=1,3) / 0.8470290000E+05, -0.5625510000E-05, 0.1536300000E-07 /
  data (epsData(172,i,32), i=1,3) / 0.9494790000E+05, -0.4476170000E-05, 0.9484690000E-08 /
  data (epsData(173,i,32), i=1,3) / 0.1000000000E+06, -0.4051920000E-05, 0.7626150000E-08 /
  data (epsData(1,i,33), i=1,3) / 0.2506580000E-02, -0.1122230000E+04, 0.3779860000E+04 /
  data (epsData(2,i,33), i=1,3) / 0.4693440000E-02, -0.9493050000E+03, 0.1318590000E+04 /
  data (epsData(3,i,33), i=1,3) / 0.7560590000E-02, -0.3381700000E+03, 0.6296480000E+03 /
  data (epsData(4,i,33), i=1,3) / 0.1110800000E-01, -0.7000550000E+02, 0.3613490000E+03 /
  data (epsData(5,i,33), i=1,3) / 0.1533570000E-01, 0.5641210000E+02, 0.2331280000E+03 /
  data (epsData(6,i,33), i=1,3) / 0.2024370000E-01, 0.1219080000E+03, 0.1628030000E+03 /
  data (epsData(7,i,33), i=1,3) / 0.2583200000E-01, 0.1586060000E+03, 0.1202710000E+03 /
  data (epsData(8,i,33), i=1,3) / 0.3210060000E-01, 0.1805040000E+03, 0.9262180000E+02 /
  data (epsData(9,i,33), i=1,3) / 0.3904940000E-01, 0.1942550000E+03, 0.7363040000E+02 /
  data (epsData(10,i,33), i=1,3) / 0.4667860000E-01, 0.2032610000E+03, 0.6001130000E+02 /
  data (epsData(11,i,33), i=1,3) / 0.5498800000E-01, 0.2093720000E+03, 0.4990210000E+02 /
  data (epsData(12,i,33), i=1,3) / 0.7364770000E-01, 0.2167160000E+03, 0.3615280000E+02 /
  data (epsData(13,i,33), i=1,3) / 0.1067390000E+00, 0.2219720000E+03, 0.2424790000E+02 /
  data (epsData(14,i,33), i=1,3) / 0.1912910000E+00, 0.2260060000E+03, 0.1314890000E+02 /
  data (epsData(15,i,33), i=1,3) / 0.5918850000E+00, 0.2300500000E+03, 0.5079090000E+01 /
  data (epsData(16,i,33), i=1,3) / 0.1337820000E+01, 0.2448270000E+03, 0.3262510000E+01 /
  data (epsData(17,i,33), i=1,3) / 0.1949930000E+01, 0.2720370000E+03, 0.3494210000E+01 /
  data (epsData(18,i,33), i=1,3) / 0.2327150000E+01, 0.3034900000E+03, 0.4463070000E+01 /
  data (epsData(19,i,33), i=1,3) / 0.2557670000E+01, 0.3357470000E+03, 0.5953640000E+01 /
  data (epsData(20,i,33), i=1,3) / 0.2737710000E+01, 0.3769050000E+03, 0.1267080000E+02 /
  data (epsData(21,i,33), i=1,3) / 0.2799080000E+01, 0.3975150000E+03, 0.1515360000E+02 /
  data (epsData(22,i,33), i=1,3) / 0.2861130000E+01, 0.4304940000E+03, 0.1743780000E+02 /
  data (epsData(23,i,33), i=1,3) / 0.2923860000E+01, 0.4717870000E+03, 0.4669770000E+02 /
  data (epsData(24,i,33), i=1,3) / 0.3000000000E+01, 0.4915530000E+03, 0.1299510000E+03 /
  data (epsData(25,i,33), i=1,3) / 0.3150000000E+01, 0.4506880000E+03, 0.2243860000E+03 /
  data (epsData(26,i,33), i=1,3) / 0.3212590000E+01, 0.4142610000E+03, 0.2526980000E+03 /
  data (epsData(27,i,33), i=1,3) / 0.3370410000E+01, 0.3583780000E+03, 0.2618890000E+03 /
  data (epsData(28,i,33), i=1,3) / 0.3783690000E+01, 0.2897880000E+03, 0.2809150000E+03 /
  data (epsData(29,i,33), i=1,3) / 0.4913300000E+01, 0.1601980000E+03, 0.3299900000E+03 /
  data (epsData(30,i,33), i=1,3) / 0.5574530000E+01, 0.7582490000E+02, 0.3324360000E+03 /
  data (epsData(31,i,33), i=1,3) / 0.5941890000E+01, 0.2721620000E+02, 0.3167650000E+03 /
  data (epsData(32,i,33), i=1,3) / 0.6266720000E+01, -0.1897710000E+00, 0.2856540000E+03 /
  data (epsData(33,i,33), i=1,3) / 0.6469790000E+01, -0.6352150000E+01, 0.2672390000E+03 /
  data (epsData(34,i,33), i=1,3) / 0.6750070000E+01, -0.1094470000E+02, 0.2573630000E+03 /
  data (epsData(35,i,33), i=1,3) / 0.7736140000E+01, -0.5118790000E+02, 0.2264140000E+03 /
  data (epsData(36,i,33), i=1,3) / 0.8544230000E+01, -0.8018780000E+02, 0.1918060000E+03 /
  data (epsData(37,i,33), i=1,3) / 0.9393220000E+01, -0.9497520000E+02, 0.1517240000E+03 /
  data (epsData(38,i,33), i=1,3) / 0.1034320000E+02, -0.9814540000E+02, 0.1141110000E+03 /
  data (epsData(39,i,33), i=1,3) / 0.1133370000E+02, -0.9368980000E+02, 0.8345590000E+02 /
  data (epsData(40,i,33), i=1,3) / 0.1242050000E+02, -0.8438670000E+02, 0.5885150000E+02 /
  data (epsData(41,i,33), i=1,3) / 0.1374840000E+02, -0.7167000000E+02, 0.3893050000E+02 /
  data (epsData(42,i,33), i=1,3) / 0.1505320000E+02, -0.6034280000E+02, 0.2650820000E+02 /
  data (epsData(43,i,33), i=1,3) / 0.1649130000E+02, -0.4993530000E+02, 0.1784660000E+02 /
  data (epsData(44,i,33), i=1,3) / 0.1794610000E+02, -0.4150240000E+02, 0.1232460000E+02 /
  data (epsData(45,i,33), i=1,3) / 0.1951580000E+02, -0.3433950000E+02, 0.8525070000E+01 /
  data (epsData(46,i,33), i=1,3) / 0.2120060000E+02, -0.2837550000E+02, 0.5935410000E+01 /
  data (epsData(47,i,33), i=1,3) / 0.2314350000E+02, -0.2312010000E+02, 0.4056270000E+01 /
  data (epsData(48,i,33), i=1,3) / 0.2521970000E+02, -0.1887880000E+02, 0.2802350000E+01 /
  data (epsData(49,i,33), i=1,3) / 0.2759220000E+02, -0.1522440000E+02, 0.1909900000E+01 /
  data (epsData(50,i,33), i=1,3) / 0.3011770000E+02, -0.1229960000E+02, 0.1330260000E+01 /
  data (epsData(51,i,33), i=1,3) / 0.3298040000E+02, -0.9801660000E+01, 0.9269450000E+00 /
  data (epsData(52,i,33), i=1,3) / 0.3621270000E+02, -0.7673580000E+01, 0.6519050000E+00 /
  data (epsData(53,i,33), i=1,3) / 0.4111080000E+02, -0.5134180000E+01, 0.4293500000E+00 /
  data (epsData(54,i,33), i=1,3) / 0.4153660000E+02, -0.4818900000E+01, 0.4577260000E+00 /
  data (epsData(55,i,33), i=1,3) / 0.4186460000E+02, -0.4649910000E+01, 0.6990970000E+00 /
  data (epsData(56,i,33), i=1,3) / 0.4251670000E+02, -0.4602520000E+01, 0.8032490000E+00 /
  data (epsData(57,i,33), i=1,3) / 0.4552180000E+02, -0.3960650000E+01, 0.8605110000E+00 /
  data (epsData(58,i,33), i=1,3) / 0.4987190000E+02, -0.3112120000E+01, 0.8504500000E+00 /
  data (epsData(59,i,33), i=1,3) / 0.5385020000E+02, -0.2426080000E+01, 0.8864270000E+00 /
  data (epsData(60,i,33), i=1,3) / 0.5846360000E+02, -0.1849020000E+01, 0.1003590000E+01 /
  data (epsData(61,i,33), i=1,3) / 0.6347210000E+02, -0.1432700000E+01, 0.1129300000E+01 /
  data (epsData(62,i,33), i=1,3) / 0.6971710000E+02, -0.1094810000E+01, 0.1250890000E+01 /
  data (epsData(63,i,33), i=1,3) / 0.7631320000E+02, -0.8917760000E+00, 0.1352240000E+01 /
  data (epsData(64,i,33), i=1,3) / 0.8360580000E+02, -0.7960920000E+00, 0.1406690000E+01 /
  data (epsData(65,i,33), i=1,3) / 0.9666830000E+02, -0.7459420000E+00, 0.1351680000E+01 /
  data (epsData(66,i,33), i=1,3) / 0.1379690000E+03, -0.6625320000E+00, 0.9372440000E+00 /
  data (epsData(67,i,33), i=1,3) / 0.1464480000E+03, -0.6403390000E+00, 0.8871860000E+00 /
  data (epsData(68,i,33), i=1,3) / 0.1709960000E+03, -0.6087930000E+00, 0.6835590000E+00 /
  data (epsData(69,i,33), i=1,3) / 0.2003510000E+03, -0.5282650000E+00, 0.5007040000E+00 /
  data (epsData(70,i,33), i=1,3) / 0.2228500000E+03, -0.4817190000E+00, 0.4129280000E+00 /
  data (epsData(71,i,33), i=1,3) / 0.2618780000E+03, -0.4035470000E+00, 0.2841330000E+00 /
  data (epsData(72,i,33), i=1,3) / 0.2998870000E+03, -0.3375440000E+00, 0.2020040000E+00 /
  data (epsData(73,i,33), i=1,3) / 0.3379960000E+03, -0.2832050000E+00, 0.1468200000E+00 /
  data (epsData(74,i,33), i=1,3) / 0.3795670000E+03, -0.2353570000E+00, 0.1059940000E+00 /
  data (epsData(75,i,33), i=1,3) / 0.4273220000E+03, -0.1918730000E+00, 0.7531060000E-01 /
  data (epsData(76,i,33), i=1,3) / 0.4832920000E+03, -0.1549250000E+00, 0.5248830000E-01 /
  data (epsData(77,i,33), i=1,3) / 0.5293840000E+03, -0.1305240000E+00, 0.3971170000E-01 /
  data (epsData(78,i,33), i=1,3) / 0.5985200000E+03, -0.1033620000E+00, 0.2685880000E-01 /
  data (epsData(79,i,33), i=1,3) / 0.6711310000E+03, -0.8270830000E-01, 0.1873210000E-01 /
  data (epsData(80,i,33), i=1,3) / 0.7387430000E+03, -0.6807370000E-01, 0.1381860000E-01 /
  data (epsData(81,i,33), i=1,3) / 0.8182170000E+03, -0.5506570000E-01, 0.9878000000E-02 /
  data (epsData(82,i,33), i=1,3) / 0.9149490000E+03, -0.4335560000E-01, 0.6814360000E-02 /
  data (epsData(83,i,33), i=1,3) / 0.1024030000E+04, -0.3361870000E-01, 0.4674950000E-02 /
  data (epsData(84,i,33), i=1,3) / 0.1153360000E+04, -0.2485180000E-01, 0.3136510000E-02 /
  data (epsData(85,i,33), i=1,3) / 0.1271570000E+04, -0.1743360000E-01, 0.2248870000E-02 /
  data (epsData(86,i,33), i=1,3) / 0.1303130000E+04, -0.1452410000E-01, 0.2073960000E-02 /
  data (epsData(87,i,33), i=1,3) / 0.1315390000E+04, -0.1271050000E-01, 0.2068360000E-02 /
  data (epsData(88,i,33), i=1,3) / 0.1320310000E+04, -0.1151400000E-01, 0.2090490000E-02 /
  data (epsData(89,i,33), i=1,3) / 0.1322780000E+04, -0.1056650000E-01, 0.2164080000E-02 /
  data (epsData(90,i,33), i=1,3) / 0.1324990000E+04, -0.9363060000E-02, 0.2720080000E-02 /
  data (epsData(91,i,33), i=1,3) / 0.1326800000E+04, -0.8569910000E-02, 0.3791270000E-02 /
  data (epsData(92,i,33), i=1,3) / 0.1328340000E+04, -0.8278400000E-02, 0.4983900000E-02 /
  data (epsData(93,i,33), i=1,3) / 0.1330170000E+04, -0.8682030000E-02, 0.6239100000E-02 /
  data (epsData(94,i,33), i=1,3) / 0.1333090000E+04, -0.9711270000E-02, 0.7178720000E-02 /
  data (epsData(95,i,33), i=1,3) / 0.1337890000E+04, -0.1064350000E-01, 0.7556530000E-02 /
  data (epsData(96,i,33), i=1,3) / 0.1351620000E+04, -0.1145420000E-01, 0.8183890000E-02 /
  data (epsData(97,i,33), i=1,3) / 0.1359190000E+04, -0.1111040000E-01, 0.8350900000E-02 /
  data (epsData(98,i,33), i=1,3) / 0.1362920000E+04, -0.1064340000E-01, 0.9193350000E-02 /
  data (epsData(99,i,33), i=1,3) / 0.1366250000E+04, -0.1110700000E-01, 0.1019370000E-01 /
  data (epsData(100,i,33), i=1,3) / 0.1372300000E+04, -0.1220280000E-01, 0.1059130000E-01 /
  data (epsData(101,i,33), i=1,3) / 0.1394590000E+04, -0.1408690000E-01, 0.1070970000E-01 /
  data (epsData(102,i,33), i=1,3) / 0.1444140000E+04, -0.1543980000E-01, 0.9645220000E-02 /
  data (epsData(103,i,33), i=1,3) / 0.1506370000E+04, -0.1523070000E-01, 0.8307930000E-02 /
  data (epsData(104,i,33), i=1,3) / 0.1526250000E+04, -0.1462780000E-01, 0.8275110000E-02 /
  data (epsData(105,i,33), i=1,3) / 0.1538270000E+04, -0.1490020000E-01, 0.8660670000E-02 /
  data (epsData(106,i,33), i=1,3) / 0.1602460000E+04, -0.1509430000E-01, 0.7580580000E-02 /
  data (epsData(107,i,33), i=1,3) / 0.1766440000E+04, -0.1366220000E-01, 0.5434790000E-02 /
  data (epsData(108,i,33), i=1,3) / 0.2001430000E+04, -0.1130110000E-01, 0.3542660000E-02 /
  data (epsData(109,i,33), i=1,3) / 0.2245140000E+04, -0.9267680000E-02, 0.2380170000E-02 /
  data (epsData(110,i,33), i=1,3) / 0.2507730000E+04, -0.7562970000E-02, 0.1612920000E-02 /
  data (epsData(111,i,33), i=1,3) / 0.2794040000E+04, -0.6150590000E-02, 0.1098050000E-02 /
  data (epsData(112,i,33), i=1,3) / 0.3111090000E+04, -0.4989550000E-02, 0.7502580000E-03 /
  data (epsData(113,i,33), i=1,3) / 0.3461880000E+04, -0.4038800000E-02, 0.5086590000E-03 /
  data (epsData(114,i,33), i=1,3) / 0.3851280000E+04, -0.3262540000E-02, 0.3442500000E-03 /
  data (epsData(115,i,33), i=1,3) / 0.4286040000E+04, -0.2629300000E-02, 0.2319960000E-03 /
  data (epsData(116,i,33), i=1,3) / 0.4770640000E+04, -0.2116000000E-02, 0.1558760000E-03 /
  data (epsData(117,i,33), i=1,3) / 0.5314770000E+04, -0.1698520000E-02, 0.1041560000E-03 /
  data (epsData(118,i,33), i=1,3) / 0.5923000000E+04, -0.1361660000E-02, 0.6936200000E-04 /
  data (epsData(119,i,33), i=1,3) / 0.6597490000E+04, -0.1092400000E-02, 0.4641710000E-04 /
  data (epsData(120,i,33), i=1,3) / 0.7356640000E+04, -0.8739080000E-03, 0.3063820000E-04 /
  data (epsData(121,i,33), i=1,3) / 0.8213890000E+04, -0.6964990000E-03, 0.2016970000E-04 /
  data (epsData(122,i,33), i=1,3) / 0.8976350000E+04, -0.5795090000E-03, 0.1437960000E-04 /
  data (epsData(123,i,33), i=1,3) / 0.8979000000E+04, -0.5454260000E-03, 0.1436340000E-04 /
  data (epsData(124,i,33), i=1,3) / 0.8979000000E+04, -0.5791520000E-03, 0.1436330000E-04 /
  data (epsData(125,i,33), i=1,3) / 0.1004590000E+05, -0.4572360000E-03, 0.9333070000E-05 /
  data (epsData(126,i,33), i=1,3) / 0.1174770000E+05, -0.3096700000E-03, 0.5104070000E-05 /
  data (epsData(127,i,33), i=1,3) / 0.1184170000E+05, -0.2897210000E-03, 0.4953460000E-05 /
  data (epsData(128,i,33), i=1,3) / 0.1186150000E+05, -0.2750470000E-03, 0.6729330000E-05 /
  data (epsData(129,i,33), i=1,3) / 0.1186540000E+05, -0.2665700000E-03, 0.9709640000E-05 /
  data (epsData(130,i,33), i=1,3) / 0.1186700000E+05, -0.2313640000E-03, 0.1512670000E-04 /
  data (epsData(131,i,33), i=1,3) / 0.1186700000E+05, -0.2625240000E-03, 0.1537550000E-04 /
  data (epsData(132,i,33), i=1,3) / 0.1187050000E+05, -0.2655960000E-03, 0.2808960000E-04 /
  data (epsData(133,i,33), i=1,3) / 0.1187880000E+05, -0.2781280000E-03, 0.3569070000E-04 /
  data (epsData(134,i,33), i=1,3) / 0.1190230000E+05, -0.2898070000E-03, 0.3496420000E-04 /
  data (epsData(135,i,33), i=1,3) / 0.1203650000E+05, -0.2993860000E-03, 0.3483380000E-04 /
  data (epsData(136,i,33), i=1,3) / 0.1271290000E+05, -0.2837560000E-03, 0.2859270000E-04 /
  data (epsData(137,i,33), i=1,3) / 0.1474350000E+05, -0.2192750000E-03, 0.1690390000E-04 /
  data (epsData(138,i,33), i=1,3) / 0.1659170000E+05, -0.1751910000E-03, 0.1096580000E-04 /
  data (epsData(139,i,33), i=1,3) / 0.1860220000E+05, -0.1401760000E-03, 0.7170440000E-05 /
  data (epsData(140,i,33), i=1,3) / 0.2082810000E+05, -0.1121430000E-03, 0.4692780000E-05 /
  data (epsData(141,i,33), i=1,3) / 0.2330790000E+05, -0.8965830000E-04, 0.3062500000E-05 /
  data (epsData(142,i,33), i=1,3) / 0.2608160000E+05, -0.7161610000E-04, 0.1990270000E-05 /
  data (epsData(143,i,33), i=1,3) / 0.2918640000E+05, -0.5716610000E-04, 0.1288320000E-05 /
  data (epsData(144,i,33), i=1,3) / 0.3266660000E+05, -0.4559850000E-04, 0.8300430000E-06 /
  data (epsData(145,i,33), i=1,3) / 0.3656960000E+05, -0.3634900000E-04, 0.5323970000E-06 /
  data (epsData(146,i,33), i=1,3) / 0.4095240000E+05, -0.2895410000E-04, 0.3397720000E-06 /
  data (epsData(147,i,33), i=1,3) / 0.4586960000E+05, -0.2305440000E-04, 0.2158690000E-06 /
  data (epsData(148,i,33), i=1,3) / 0.5139050000E+05, -0.1834810000E-04, 0.1365620000E-06 /
  data (epsData(149,i,33), i=1,3) / 0.5758640000E+05, -0.1459840000E-04, 0.8598270000E-07 /
  data (epsData(150,i,33), i=1,3) / 0.6452910000E+05, -0.1161630000E-04, 0.5412850000E-07 /
  data (epsData(151,i,33), i=1,3) / 0.7233920000E+05, -0.9236460000E-05, 0.3379960000E-07 /
  data (epsData(152,i,33), i=1,3) / 0.8110290000E+05, -0.7343830000E-05, 0.2095910000E-07 /
  data (epsData(153,i,33), i=1,3) / 0.9093560000E+05, -0.5839500000E-05, 0.1295900000E-07 /
  data (epsData(154,i,33), i=1,3) / 0.1000000000E+06, -0.4846280000E-05, 0.8710850000E-08 /
  data (epsData(1,i,34), i=1,3) / 0.2506580000E-02, -0.1070040000E+04, 0.3716480000E+04 /
  data (epsData(2,i,34), i=1,3) / 0.4693440000E-02, -0.9131650000E+03, 0.1303030000E+04 /
  data (epsData(3,i,34), i=1,3) / 0.7560590000E-02, -0.3196550000E+03, 0.6249240000E+03 /
  data (epsData(4,i,34), i=1,3) / 0.1110800000E-01, -0.5891120000E+02, 0.3598890000E+03 /
  data (epsData(5,i,34), i=1,3) / 0.1533570000E-01, 0.6406720000E+02, 0.2328170000E+03 /
  data (epsData(6,i,34), i=1,3) / 0.2024370000E-01, 0.1277980000E+03, 0.1629280000E+03 /
  data (epsData(7,i,34), i=1,3) / 0.2583200000E-01, 0.1635150000E+03, 0.1205600000E+03 /
  data (epsData(8,i,34), i=1,3) / 0.3210060000E-01, 0.1848290000E+03, 0.9296350000E+02 /
  data (epsData(9,i,34), i=1,3) / 0.3904940000E-01, 0.1982150000E+03, 0.7397680000E+02 /
  data (epsData(10,i,34), i=1,3) / 0.4667860000E-01, 0.2069830000E+03, 0.6034250000E+02 /
  data (epsData(11,i,34), i=1,3) / 0.5498800000E-01, 0.2129330000E+03, 0.5021050000E+02 /
  data (epsData(12,i,34), i=1,3) / 0.7364770000E-01, 0.2200830000E+03, 0.3641190000E+02 /
  data (epsData(13,i,34), i=1,3) / 0.1067390000E+00, 0.2251990000E+03, 0.2444450000E+02 /
  data (epsData(14,i,34), i=1,3) / 0.1912910000E+00, 0.2291220000E+03, 0.1326730000E+02 /
  data (epsData(15,i,34), i=1,3) / 0.5918850000E+00, 0.2329860000E+03, 0.5089810000E+01 /
  data (epsData(16,i,34), i=1,3) / 0.1337820000E+01, 0.2469520000E+03, 0.3221880000E+01 /
  data (epsData(17,i,34), i=1,3) / 0.1949930000E+01, 0.2725230000E+03, 0.3391540000E+01 /
  data (epsData(18,i,34), i=1,3) / 0.2327150000E+01, 0.3018910000E+03, 0.4269350000E+01 /
  data (epsData(19,i,34), i=1,3) / 0.2557670000E+01, 0.3318690000E+03, 0.5636060000E+01 /
  data (epsData(20,i,34), i=1,3) / 0.2737710000E+01, 0.3700030000E+03, 0.1185180000E+02 /
  data (epsData(21,i,34), i=1,3) / 0.2799080000E+01, 0.3890800000E+03, 0.1414980000E+02 /
  data (epsData(22,i,34), i=1,3) / 0.2861130000E+01, 0.4196280000E+03, 0.1626400000E+02 /
  data (epsData(23,i,34), i=1,3) / 0.2923860000E+01, 0.4578100000E+03, 0.4354020000E+02 /
  data (epsData(24,i,34), i=1,3) / 0.3000000000E+01, 0.4754260000E+03, 0.1211760000E+03 /
  data (epsData(25,i,34), i=1,3) / 0.3150000000E+01, 0.4365150000E+03, 0.2069030000E+03 /
  data (epsData(26,i,34), i=1,3) / 0.3212590000E+01, 0.4025610000E+03, 0.2324080000E+03 /
  data (epsData(27,i,34), i=1,3) / 0.3370410000E+01, 0.3514110000E+03, 0.2385640000E+03 /
  data (epsData(28,i,34), i=1,3) / 0.3783690000E+01, 0.2932440000E+03, 0.2523260000E+03 /
  data (epsData(29,i,34), i=1,3) / 0.5016700000E+01, 0.1810960000E+03, 0.3119250000E+03 /
  data (epsData(30,i,34), i=1,3) / 0.5515690000E+01, 0.1214230000E+03, 0.3245240000E+03 /
  data (epsData(31,i,34), i=1,3) / 0.5816720000E+01, 0.8154490000E+02, 0.3160340000E+03 /
  data (epsData(32,i,34), i=1,3) / 0.6005490000E+01, 0.6067910000E+02, 0.3026150000E+03 /
  data (epsData(33,i,34), i=1,3) / 0.6200400000E+01, 0.5272510000E+02, 0.2855120000E+03 /
  data (epsData(34,i,34), i=1,3) / 0.6678980000E+01, 0.3551920000E+02, 0.2795260000E+03 /
  data (epsData(35,i,34), i=1,3) / 0.7420490000E+01, -0.8985470000E+01, 0.2669610000E+03 /
  data (epsData(36,i,34), i=1,3) / 0.8062670000E+01, -0.4569410000E+02, 0.2435250000E+03 /
  data (epsData(37,i,34), i=1,3) / 0.8926630000E+01, -0.7799690000E+02, 0.2024680000E+03 /
  data (epsData(38,i,34), i=1,3) / 0.9858620000E+01, -0.9505790000E+02, 0.1603890000E+03 /
  data (epsData(39,i,34), i=1,3) / 0.1085860000E+02, -0.1006640000E+03, 0.1208630000E+03 /
  data (epsData(40,i,34), i=1,3) / 0.1181680000E+02, -0.9719700000E+02, 0.9014330000E+02 /
  data (epsData(41,i,34), i=1,3) / 0.1302890000E+02, -0.8741130000E+02, 0.6228410000E+02 /
  data (epsData(42,i,34), i=1,3) / 0.1445750000E+02, -0.7419550000E+02, 0.4080700000E+02 /
  data (epsData(43,i,34), i=1,3) / 0.1589430000E+02, -0.6209410000E+02, 0.2728530000E+02 /
  data (epsData(44,i,34), i=1,3) / 0.1731470000E+02, -0.5204080000E+02, 0.1879020000E+02 /
  data (epsData(45,i,34), i=1,3) / 0.1885860000E+02, -0.4322110000E+02, 0.1290410000E+02 /
  data (epsData(46,i,34), i=1,3) / 0.2049130000E+02, -0.3588960000E+02, 0.8952660000E+01 /
  data (epsData(47,i,34), i=1,3) / 0.2229810000E+02, -0.2961610000E+02, 0.6185710000E+01 /
  data (epsData(48,i,34), i=1,3) / 0.2427910000E+02, -0.2436370000E+02, 0.4270410000E+01 /
  data (epsData(49,i,34), i=1,3) / 0.2657470000E+02, -0.1977730000E+02, 0.2879980000E+01 /
  data (epsData(50,i,34), i=1,3) / 0.2906700000E+02, -0.1605420000E+02, 0.1958470000E+01 /
  data (epsData(51,i,34), i=1,3) / 0.3175580000E+02, -0.1304610000E+02, 0.1347570000E+01 /
  data (epsData(52,i,34), i=1,3) / 0.3481710000E+02, -0.1049270000E+02, 0.9288110000E+00 /
  data (epsData(53,i,34), i=1,3) / 0.3828750000E+02, -0.8355320000E+01, 0.6473290000E+00 /
  data (epsData(54,i,34), i=1,3) / 0.4220600000E+02, -0.6579570000E+01, 0.4581760000E+00 /
  data (epsData(55,i,34), i=1,3) / 0.4661320000E+02, -0.5102230000E+01, 0.3297990000E+00 /
  data (epsData(56,i,34), i=1,3) / 0.5178440000E+02, -0.3783050000E+01, 0.2407440000E+00 /
  data (epsData(57,i,34), i=1,3) / 0.5414490000E+02, -0.3210290000E+01, 0.2183210000E+00 /
  data (epsData(58,i,34), i=1,3) / 0.5477990000E+02, -0.3027440000E+01, 0.2965810000E+00 /
  data (epsData(59,i,34), i=1,3) / 0.5578610000E+02, -0.2867880000E+01, 0.4037050000E+00 /
  data (epsData(60,i,34), i=1,3) / 0.5946740000E+02, -0.2475390000E+01, 0.5167600000E+00 /
  data (epsData(61,i,34), i=1,3) / 0.6689100000E+02, -0.1731010000E+01, 0.5409390000E+00 /
  data (epsData(62,i,34), i=1,3) / 0.7211290000E+02, -0.1320240000E+01, 0.6341140000E+00 /
  data (epsData(63,i,34), i=1,3) / 0.7796600000E+02, -0.1002250000E+01, 0.7529830000E+00 /
  data (epsData(64,i,34), i=1,3) / 0.8510090000E+02, -0.7500950000E+00, 0.8852800000E+00 /
  data (epsData(65,i,34), i=1,3) / 0.9164890000E+02, -0.6264200000E+00, 0.9924320000E+00 /
  data (epsData(66,i,34), i=1,3) / 0.9885070000E+02, -0.5786320000E+00, 0.1063340000E+01 /
  data (epsData(67,i,34), i=1,3) / 0.1121210000E+03, -0.5627510000E+00, 0.1061010000E+01 /
  data (epsData(68,i,34), i=1,3) / 0.1378370000E+03, -0.5631450000E+00, 0.9152410000E+00 /
  data (epsData(69,i,34), i=1,3) / 0.1591440000E+03, -0.5289820000E+00, 0.7583970000E+00 /
  data (epsData(70,i,34), i=1,3) / 0.1699010000E+03, -0.5240460000E+00, 0.7164260000E+00 /
  data (epsData(71,i,34), i=1,3) / 0.1996110000E+03, -0.4977060000E+00, 0.5405750000E+00 /
  data (epsData(72,i,34), i=1,3) / 0.2279380000E+03, -0.4397870000E+00, 0.4149010000E+00 /
  data (epsData(73,i,34), i=1,3) / 0.2794950000E+03, -0.3657650000E+00, 0.2686090000E+00 /
  data (epsData(74,i,34), i=1,3) / 0.3194650000E+03, -0.3090340000E+00, 0.1914690000E+00 /
  data (epsData(75,i,34), i=1,3) / 0.3637490000E+03, -0.2560810000E+00, 0.1347140000E+00 /
  data (epsData(76,i,34), i=1,3) / 0.4128570000E+03, -0.2092080000E+00, 0.9452530000E-01 /
  data (epsData(77,i,34), i=1,3) / 0.4577590000E+03, -0.1755080000E+00, 0.7040870000E-01 /
  data (epsData(78,i,34), i=1,3) / 0.5225330000E+03, -0.1389410000E+00, 0.4689070000E-01 /
  data (epsData(79,i,34), i=1,3) / 0.5876700000E+03, -0.1120570000E+00, 0.3276010000E-01 /
  data (epsData(80,i,34), i=1,3) / 0.6456860000E+03, -0.9338600000E-01, 0.2442720000E-01 /
  data (epsData(81,i,34), i=1,3) / 0.7218720000E+03, -0.7497510000E-01, 0.1702100000E-01 /
  data (epsData(82,i,34), i=1,3) / 0.8115560000E+03, -0.5921960000E-01, 0.1159530000E-01 /
  data (epsData(83,i,34), i=1,3) / 0.9097010000E+03, -0.4676180000E-01, 0.7954460000E-02 /
  data (epsData(84,i,34), i=1,3) / 0.1013420000E+04, -0.3707200000E-01, 0.5576860000E-02 /
  data (epsData(85,i,34), i=1,3) / 0.1125610000E+04, -0.2919960000E-01, 0.3928790000E-02 /
  data (epsData(86,i,34), i=1,3) / 0.1259450000E+04, -0.2192070000E-01, 0.2682180000E-02 /
  data (epsData(87,i,34), i=1,3) / 0.1382480000E+04, -0.1558690000E-01, 0.1953240000E-02 /
  data (epsData(88,i,34), i=1,3) / 0.1412730000E+04, -0.1326610000E-01, 0.1809880000E-02 /
  data (epsData(89,i,34), i=1,3) / 0.1426730000E+04, -0.1149190000E-01, 0.1807320000E-02 /
  data (epsData(90,i,34), i=1,3) / 0.1431840000E+04, -0.1030430000E-01, 0.1848620000E-02 /
  data (epsData(91,i,34), i=1,3) / 0.1434290000E+04, -0.9328960000E-02, 0.2070650000E-02 /
  data (epsData(92,i,34), i=1,3) / 0.1436360000E+04, -0.8373280000E-02, 0.2816510000E-02 /
  data (epsData(93,i,34), i=1,3) / 0.1438090000E+04, -0.7857300000E-02, 0.3895710000E-02 /
  data (epsData(94,i,34), i=1,3) / 0.1439830000E+04, -0.7987450000E-02, 0.5099810000E-02 /
  data (epsData(95,i,34), i=1,3) / 0.1441970000E+04, -0.8700920000E-02, 0.6063040000E-02 /
  data (epsData(96,i,34), i=1,3) / 0.1445040000E+04, -0.9620590000E-02, 0.6467670000E-02 /
  data (epsData(97,i,34), i=1,3) / 0.1453010000E+04, -0.1046830000E-01, 0.6704430000E-02 /
  data (epsData(98,i,34), i=1,3) / 0.1466380000E+04, -0.1099500000E-01, 0.6991260000E-02 /
  data (epsData(99,i,34), i=1,3) / 0.1474150000E+04, -0.1058150000E-01, 0.6985960000E-02 /
  data (epsData(100,i,34), i=1,3) / 0.1477980000E+04, -0.1001420000E-01, 0.7666940000E-02 /
  data (epsData(101,i,34), i=1,3) / 0.1481350000E+04, -0.1032710000E-01, 0.8654860000E-02 /
  data (epsData(102,i,34), i=1,3) / 0.1486580000E+04, -0.1126280000E-01, 0.9020550000E-02 /
  data (epsData(103,i,34), i=1,3) / 0.1509350000E+04, -0.1284820000E-01, 0.9026730000E-02 /
  data (epsData(104,i,34), i=1,3) / 0.1564170000E+04, -0.1393050000E-01, 0.8118220000E-02 /
  data (epsData(105,i,34), i=1,3) / 0.1631580000E+04, -0.1361710000E-01, 0.7006410000E-02 /
  data (epsData(106,i,34), i=1,3) / 0.1651610000E+04, -0.1310990000E-01, 0.7021760000E-02 /
  data (epsData(107,i,34), i=1,3) / 0.1665330000E+04, -0.1336540000E-01, 0.7345020000E-02 /
  data (epsData(108,i,34), i=1,3) / 0.1750090000E+04, -0.1338590000E-01, 0.6241800000E-02 /
  data (epsData(109,i,34), i=1,3) / 0.1948100000E+04, -0.1182040000E-01, 0.4317330000E-02 /
  data (epsData(110,i,34), i=1,3) / 0.2201500000E+04, -0.9740780000E-02, 0.2834370000E-02 /
  data (epsData(111,i,34), i=1,3) / 0.2466060000E+04, -0.7977070000E-02, 0.1909450000E-02 /
  data (epsData(112,i,34), i=1,3) / 0.2752880000E+04, -0.6501030000E-02, 0.1294230000E-02 /
  data (epsData(113,i,34), i=1,3) / 0.3057890000E+04, -0.5312690000E-02, 0.8927320000E-03 /
  data (epsData(114,i,34), i=1,3) / 0.3407880000E+04, -0.4298770000E-02, 0.6050990000E-03 /
  data (epsData(115,i,34), i=1,3) / 0.3791600000E+04, -0.3478000000E-02, 0.4100660000E-03 /
  data (epsData(116,i,34), i=1,3) / 0.4218820000E+04, -0.2807420000E-02, 0.2770440000E-03 /
  data (epsData(117,i,34), i=1,3) / 0.4695060000E+04, -0.2262020000E-02, 0.1865230000E-03 /
  data (epsData(118,i,34), i=1,3) / 0.5227210000E+04, -0.1819260000E-02, 0.1251080000E-03 /
  data (epsData(119,i,34), i=1,3) / 0.5823540000E+04, -0.1460170000E-02, 0.8351590000E-04 /
  data (epsData(120,i,34), i=1,3) / 0.6486330000E+04, -0.1172020000E-02, 0.5598860000E-04 /
  data (epsData(121,i,34), i=1,3) / 0.7225460000E+04, -0.9401470000E-03, 0.3719740000E-04 /
  data (epsData(122,i,34), i=1,3) / 0.8061360000E+04, -0.7511580000E-03, 0.2447030000E-04 /
  data (epsData(123,i,34), i=1,3) / 0.9006160000E+04, -0.5979190000E-03, 0.1608620000E-04 /
  data (epsData(124,i,34), i=1,3) / 0.9658900000E+04, -0.5171850000E-03, 0.1230760000E-04 /
  data (epsData(125,i,34), i=1,3) / 0.9659000000E+04, -0.5009750000E-03, 0.1230700000E-04 /
  data (epsData(126,i,34), i=1,3) / 0.9659000000E+04, -0.5171740000E-03, 0.1230700000E-04 /
  data (epsData(127,i,34), i=1,3) / 0.1080940000E+05, -0.4080570000E-03, 0.7992450000E-05 /
  data (epsData(128,i,34), i=1,3) / 0.1253550000E+05, -0.2818860000E-03, 0.4514500000E-05 /
  data (epsData(129,i,34), i=1,3) / 0.1263330000E+05, -0.2639890000E-03, 0.4382460000E-05 /
  data (epsData(130,i,34), i=1,3) / 0.1265140000E+05, -0.2529030000E-03, 0.5794500000E-05 /
  data (epsData(131,i,34), i=1,3) / 0.1265750000E+05, -0.2425060000E-03, 0.1085520000E-04 /
  data (epsData(132,i,34), i=1,3) / 0.1265800000E+05, -0.2186630000E-03, 0.1265960000E-04 /
  data (epsData(133,i,34), i=1,3) / 0.1265800000E+05, -0.2413010000E-03, 0.1284500000E-04 /
  data (epsData(134,i,34), i=1,3) / 0.1266220000E+05, -0.2437900000E-03, 0.2509060000E-04 /
  data (epsData(135,i,34), i=1,3) / 0.1267150000E+05, -0.2550000000E-03, 0.3153420000E-04 /
  data (epsData(136,i,34), i=1,3) / 0.1270230000E+05, -0.2656240000E-03, 0.3117810000E-04 /
  data (epsData(137,i,34), i=1,3) / 0.1287410000E+05, -0.2730910000E-03, 0.3019130000E-04 /
  data (epsData(138,i,34), i=1,3) / 0.1385640000E+05, -0.2493540000E-03, 0.2314930000E-04 /
  data (epsData(139,i,34), i=1,3) / 0.1591430000E+05, -0.1947350000E-03, 0.1416130000E-04 /
  data (epsData(140,i,34), i=1,3) / 0.1789530000E+05, -0.1556200000E-03, 0.9195770000E-05 /
  data (epsData(141,i,34), i=1,3) / 0.2005970000E+05, -0.1244940000E-03, 0.6015440000E-05 /
  data (epsData(142,i,34), i=1,3) / 0.2245830000E+05, -0.9957570000E-04, 0.3932320000E-05 /
  data (epsData(143,i,34), i=1,3) / 0.2513520000E+05, -0.7957760000E-04, 0.2562930000E-05 /
  data (epsData(144,i,34), i=1,3) / 0.2812640000E+05, -0.6355760000E-04, 0.1664330000E-05 /
  data (epsData(145,i,34), i=1,3) / 0.3147470000E+05, -0.5073050000E-04, 0.1076050000E-05 /
  data (epsData(146,i,34), i=1,3) / 0.3523390000E+05, -0.4045040000E-04, 0.6922410000E-06 /
  data (epsData(147,i,34), i=1,3) / 0.3944990000E+05, -0.3223530000E-04, 0.4433170000E-06 /
  data (epsData(148,i,34), i=1,3) / 0.4417700000E+05, -0.2567920000E-04, 0.2826820000E-06 /
  data (epsData(149,i,34), i=1,3) / 0.4948590000E+05, -0.2044370000E-04, 0.1793650000E-06 /
  data (epsData(150,i,34), i=1,3) / 0.5544410000E+05, -0.1626990000E-04, 0.1132920000E-06 /
  data (epsData(151,i,34), i=1,3) / 0.6212740000E+05, -0.1294590000E-04, 0.7155360000E-07 /
  data (epsData(152,i,34), i=1,3) / 0.6962870000E+05, -0.1029880000E-04, 0.4477830000E-07 /
  data (epsData(153,i,34), i=1,3) / 0.7805870000E+05, -0.8188960000E-05, 0.2792580000E-07 /
  data (epsData(154,i,34), i=1,3) / 0.8751330000E+05, -0.6512080000E-05, 0.1732460000E-07 /
  data (epsData(155,i,34), i=1,3) / 0.9803640000E+05, -0.5190790000E-05, 0.1074730000E-07 /
  data (epsData(156,i,34), i=1,3) / 0.1000000000E+06, -0.5009950000E-05, 0.9886900000E-08 /
  data (epsData(1,i,35), i=1,3) / 0.2506580000E-02, -0.1013780000E+04, 0.3618660000E+04 /
  data (epsData(2,i,35), i=1,3) / 0.4693440000E-02, -0.8721430000E+03, 0.1274270000E+04 /
  data (epsData(3,i,35), i=1,3) / 0.7560590000E-02, -0.3004940000E+03, 0.6134060000E+03 /
  data (epsData(4,i,35), i=1,3) / 0.1110800000E-01, -0.4908720000E+02, 0.3543040000E+03 /
  data (epsData(5,i,35), i=1,3) / 0.1533570000E-01, 0.6953810000E+02, 0.2297320000E+03 /
  data (epsData(6,i,35), i=1,3) / 0.2024370000E-01, 0.1310280000E+03, 0.1610540000E+03 /
  data (epsData(7,i,35), i=1,3) / 0.2583200000E-01, 0.1654940000E+03, 0.1193370000E+03 /
  data (epsData(8,i,35), i=1,3) / 0.3210060000E-01, 0.1860660000E+03, 0.9211830000E+02 /
  data (epsData(9,i,35), i=1,3) / 0.3904940000E-01, 0.1989860000E+03, 0.7336620000E+02 /
  data (epsData(10,i,35), i=1,3) / 0.4667860000E-01, 0.2074490000E+03, 0.5988490000E+02 /
  data (epsData(11,i,35), i=1,3) / 0.6397770000E-01, 0.2172090000E+03, 0.4218750000E+02 /
  data (epsData(12,i,35), i=1,3) / 0.9502860000E-01, 0.2238100000E+03, 0.2750930000E+02 /
  data (epsData(13,i,35), i=1,3) / 0.1603860000E+00, 0.2279090000E+03, 0.1585480000E+02 /
  data (epsData(14,i,35), i=1,3) / 0.3421210000E+00, 0.2304030000E+03, 0.7986740000E+01 /
  data (epsData(15,i,35), i=1,3) / 0.1018280000E+01, 0.2384270000E+03, 0.3518500000E+01 /
  data (epsData(16,i,35), i=1,3) / 0.1749350000E+01, 0.2599700000E+03, 0.3135280000E+01 /
  data (epsData(17,i,35), i=1,3) / 0.2215970000E+01, 0.2875100000E+03, 0.3778550000E+01 /
  data (epsData(18,i,35), i=1,3) / 0.2499020000E+01, 0.3168950000E+03, 0.5106560000E+01 /
  data (epsData(19,i,35), i=1,3) / 0.2677010000E+01, 0.3480170000E+03, 0.8939630000E+01 /
  data (epsData(20,i,35), i=1,3) / 0.2799080000E+01, 0.3797030000E+03, 0.1353100000E+02 /
  data (epsData(21,i,35), i=1,3) / 0.2861130000E+01, 0.4087760000E+03, 0.1555540000E+02 /
  data (epsData(22,i,35), i=1,3) / 0.2923860000E+01, 0.4450760000E+03, 0.4195610000E+02 /
  data (epsData(23,i,35), i=1,3) / 0.3000000000E+01, 0.4607810000E+03, 0.1171380000E+03 /
  data (epsData(24,i,35), i=1,3) / 0.3150000000E+01, 0.4211920000E+03, 0.1973880000E+03 /
  data (epsData(25,i,35), i=1,3) / 0.3212590000E+01, 0.3877440000E+03, 0.2209710000E+03 /
  data (epsData(26,i,35), i=1,3) / 0.3348310000E+01, 0.3429330000E+03, 0.2232070000E+03 /
  data (epsData(27,i,35), i=1,3) / 0.3658520000E+01, 0.2969110000E+03, 0.2278670000E+03 /
  data (epsData(28,i,35), i=1,3) / 0.4224510000E+01, 0.2553780000E+03, 0.2451190000E+03 /
  data (epsData(29,i,35), i=1,3) / 0.5016700000E+01, 0.1985920000E+03, 0.2849220000E+03 /
  data (epsData(30,i,35), i=1,3) / 0.5343240000E+01, 0.1618170000E+03, 0.2981680000E+03 /
  data (epsData(31,i,35), i=1,3) / 0.5694270000E+01, 0.1215710000E+03, 0.2908390000E+03 /
  data (epsData(32,i,35), i=1,3) / 0.5878960000E+01, 0.1119200000E+03, 0.2796140000E+03 /
  data (epsData(33,i,35), i=1,3) / 0.6333730000E+01, 0.9497120000E+02, 0.2808360000E+03 /
  data (epsData(34,i,35), i=1,3) / 0.7115720000E+01, 0.4500580000E+02, 0.2829900000E+03 /
  data (epsData(35,i,35), i=1,3) / 0.7816750000E+01, -0.2456170000E+01, 0.2699670000E+03 /
  data (epsData(36,i,35), i=1,3) / 0.8486150000E+01, -0.3933520000E+02, 0.2456820000E+03 /
  data (epsData(37,i,35), i=1,3) / 0.9477660000E+01, -0.7556190000E+02, 0.2042040000E+03 /
  data (epsData(38,i,35), i=1,3) / 0.1055150000E+02, -0.9722970000E+02, 0.1582510000E+03 /
  data (epsData(39,i,35), i=1,3) / 0.1149130000E+02, -0.1023870000E+03, 0.1211430000E+03 /
  data (epsData(40,i,35), i=1,3) / 0.1248620000E+02, -0.9837830000E+02, 0.8984860000E+02 /
  data (epsData(41,i,35), i=1,3) / 0.1365630000E+02, -0.8877000000E+02, 0.6361940000E+02 /
  data (epsData(42,i,35), i=1,3) / 0.1502200000E+02, -0.7646220000E+02, 0.4310030000E+02 /
  data (epsData(43,i,35), i=1,3) / 0.1647000000E+02, -0.6439560000E+02, 0.2912860000E+02 /
  data (epsData(44,i,35), i=1,3) / 0.1800030000E+02, -0.5365180000E+02, 0.1981820000E+02 /
  data (epsData(45,i,35), i=1,3) / 0.1961290000E+02, -0.4458540000E+02, 0.1365410000E+02 /
  data (epsData(46,i,35), i=1,3) / 0.2130780000E+02, -0.3712530000E+02, 0.9521160000E+01 /
  data (epsData(47,i,35), i=1,3) / 0.2325070000E+02, -0.3053380000E+02, 0.6505510000E+01 /
  data (epsData(48,i,35), i=1,3) / 0.2546600000E+02, -0.2486210000E+02, 0.4345480000E+01 /
  data (epsData(49,i,35), i=1,3) / 0.2771970000E+02, -0.2048830000E+02, 0.3000950000E+01 /
  data (epsData(50,i,35), i=1,3) / 0.3046980000E+02, -0.1651720000E+02, 0.1941730000E+01 /
  data (epsData(51,i,35), i=1,3) / 0.3338410000E+02, -0.1336280000E+02, 0.1274650000E+01 /
  data (epsData(52,i,35), i=1,3) / 0.3660020000E+02, -0.1077700000E+02, 0.8531400000E+00 /
  data (epsData(53,i,35), i=1,3) / 0.4020060000E+02, -0.8647520000E+01, 0.5868010000E+00 /
  data (epsData(54,i,35), i=1,3) / 0.4421970000E+02, -0.6907670000E+01, 0.4160680000E+00 /
  data (epsData(55,i,35), i=1,3) / 0.4877210000E+02, -0.5464540000E+01, 0.3017490000E+00 /
  data (epsData(56,i,35), i=1,3) / 0.5385790000E+02, -0.4282260000E+01, 0.2269080000E+00 /
  data (epsData(57,i,35), i=1,3) / 0.5968760000E+02, -0.3281250000E+01, 0.1751460000E+00 /
  data (epsData(58,i,35), i=1,3) / 0.6725100000E+02, -0.2291590000E+01, 0.1363330000E+00 /
  data (epsData(59,i,35), i=1,3) / 0.6890580000E+02, -0.2051640000E+01, 0.1406960000E+00 /
  data (epsData(60,i,35), i=1,3) / 0.7028610000E+02, -0.1906950000E+01, 0.2402560000E+00 /
  data (epsData(61,i,35), i=1,3) / 0.7419090000E+02, -0.1679640000E+01, 0.3333360000E+00 /
  data (epsData(62,i,35), i=1,3) / 0.8240870000E+02, -0.1192900000E+01, 0.3661020000E+00 /
  data (epsData(63,i,35), i=1,3) / 0.8846600000E+02, -0.9015800000E+00, 0.4425170000E+00 /
  data (epsData(64,i,35), i=1,3) / 0.9572320000E+02, -0.6558780000E+00, 0.5543690000E+00 /
  data (epsData(65,i,35), i=1,3) / 0.1028990000E+03, -0.5011020000E+00, 0.6703680000E+00 /
  data (epsData(66,i,35), i=1,3) / 0.1095520000E+03, -0.4335350000E+00, 0.7653780000E+00 /
  data (epsData(67,i,35), i=1,3) / 0.1170700000E+03, -0.4165300000E+00, 0.8279030000E+00 /
  data (epsData(68,i,35), i=1,3) / 0.1302870000E+03, -0.4229360000E+00, 0.8481490000E+00 /
  data (epsData(69,i,35), i=1,3) / 0.1533440000E+03, -0.4501330000E+00, 0.7709110000E+00 /
  data (epsData(70,i,35), i=1,3) / 0.1793530000E+03, -0.4347370000E+00, 0.6317680000E+00 /
  data (epsData(71,i,35), i=1,3) / 0.1911770000E+03, -0.4281320000E+00, 0.6023510000E+00 /
  data (epsData(72,i,35), i=1,3) / 0.2223230000E+03, -0.4202600000E+00, 0.4636990000E+00 /
  data (epsData(73,i,35), i=1,3) / 0.2545530000E+03, -0.3749130000E+00, 0.3518850000E+00 /
  data (epsData(74,i,35), i=1,3) / 0.3130680000E+03, -0.3112620000E+00, 0.2228250000E+00 /
  data (epsData(75,i,35), i=1,3) / 0.3606440000E+03, -0.2591720000E+00, 0.1545930000E+00 /
  data (epsData(76,i,35), i=1,3) / 0.4074220000E+03, -0.2165150000E+00, 0.1102310000E+00 /
  data (epsData(77,i,35), i=1,3) / 0.4602490000E+03, -0.1772390000E+00, 0.7789060000E-01 /
  data (epsData(78,i,35), i=1,3) / 0.5252240000E+03, -0.1419070000E+00, 0.5237250000E-01 /
  data (epsData(79,i,35), i=1,3) / 0.5823490000E+03, -0.1176110000E+00, 0.3848040000E-01 /
  data (epsData(80,i,35), i=1,3) / 0.6442590000E+03, -0.9720860000E-01, 0.2793460000E-01 /
  data (epsData(81,i,35), i=1,3) / 0.7258240000E+03, -0.7726910000E-01, 0.1902910000E-01 /
  data (epsData(82,i,35), i=1,3) / 0.8151780000E+03, -0.6144410000E-01, 0.1303170000E-01 /
  data (epsData(83,i,35), i=1,3) / 0.9076080000E+03, -0.4939350000E-01, 0.9213670000E-02 /
  data (epsData(84,i,35), i=1,3) / 0.1002070000E+04, -0.4014390000E-01, 0.6654680000E-02 /
  data (epsData(85,i,35), i=1,3) / 0.1105090000E+04, -0.3248990000E-01, 0.4798990000E-02 /
  data (epsData(86,i,35), i=1,3) / 0.1223920000E+04, -0.2576170000E-01, 0.3398940000E-02 /
  data (epsData(87,i,35), i=1,3) / 0.1368920000E+04, -0.1937170000E-01, 0.2321120000E-02 /
  data (epsData(88,i,35), i=1,3) / 0.1495320000E+04, -0.1406090000E-01, 0.1713390000E-02 /
  data (epsData(89,i,35), i=1,3) / 0.1529300000E+04, -0.1190650000E-01, 0.1585790000E-02 /
  data (epsData(90,i,35), i=1,3) / 0.1542490000E+04, -0.1049170000E-01, 0.1591620000E-02 /
  data (epsData(91,i,35), i=1,3) / 0.1547790000E+04, -0.9468030000E-02, 0.1628730000E-02 /
  data (epsData(92,i,35), i=1,3) / 0.1550490000E+04, -0.8566360000E-02, 0.1834360000E-02 /
  data (epsData(93,i,35), i=1,3) / 0.1552760000E+04, -0.7658470000E-02, 0.2536990000E-02 /
  data (epsData(94,i,35), i=1,3) / 0.1554340000E+04, -0.7224880000E-02, 0.3452840000E-02 /
  data (epsData(95,i,35), i=1,3) / 0.1556200000E+04, -0.7352720000E-02, 0.4638240000E-02 /
  data (epsData(96,i,35), i=1,3) / 0.1558170000E+04, -0.8033440000E-02, 0.5468230000E-02 /
  data (epsData(97,i,35), i=1,3) / 0.1561140000E+04, -0.8939360000E-02, 0.5788570000E-02 /
  data (epsData(98,i,35), i=1,3) / 0.1567680000E+04, -0.9690570000E-02, 0.5831130000E-02 /
  data (epsData(99,i,35), i=1,3) / 0.1582480000E+04, -0.1035090000E-01, 0.6004220000E-02 /
  data (epsData(100,i,35), i=1,3) / 0.1593590000E+04, -0.1007860000E-01, 0.5876470000E-02 /
  data (epsData(101,i,35), i=1,3) / 0.1598290000E+04, -0.9444730000E-02, 0.6199010000E-02 /
  data (epsData(102,i,35), i=1,3) / 0.1601490000E+04, -0.9284160000E-02, 0.7057930000E-02 /
  data (epsData(103,i,35), i=1,3) / 0.1605040000E+04, -0.9925930000E-02, 0.7700030000E-02 /
  data (epsData(104,i,35), i=1,3) / 0.1612910000E+04, -0.1078220000E-01, 0.7738670000E-02 /
  data (epsData(105,i,35), i=1,3) / 0.1635780000E+04, -0.1182370000E-01, 0.7570060000E-02 /
  data (epsData(106,i,35), i=1,3) / 0.1697770000E+04, -0.1251710000E-01, 0.6776410000E-02 /
  data (epsData(107,i,35), i=1,3) / 0.1766020000E+04, -0.1212950000E-01, 0.5960040000E-02 /
  data (epsData(108,i,35), i=1,3) / 0.1784760000E+04, -0.1171140000E-01, 0.6135760000E-02 /
  data (epsData(109,i,35), i=1,3) / 0.1807350000E+04, -0.1207680000E-01, 0.6181690000E-02 /
  data (epsData(110,i,35), i=1,3) / 0.1929270000E+04, -0.1170330000E-01, 0.4955650000E-02 /
  data (epsData(111,i,35), i=1,3) / 0.2190600000E+04, -0.9861200000E-02, 0.3195610000E-02 /
  data (epsData(112,i,35), i=1,3) / 0.2465950000E+04, -0.8090910000E-02, 0.2121490000E-02 /
  data (epsData(113,i,35), i=1,3) / 0.2759270000E+04, -0.6603100000E-02, 0.1429410000E-02 /
  data (epsData(114,i,35), i=1,3) / 0.3071460000E+04, -0.5388300000E-02, 0.9803640000E-03 /
  data (epsData(115,i,35), i=1,3) / 0.3419380000E+04, -0.4380640000E-02, 0.6685830000E-03 /
  data (epsData(116,i,35), i=1,3) / 0.3807150000E+04, -0.3545210000E-02, 0.4526300000E-03 /
  data (epsData(117,i,35), i=1,3) / 0.4236090000E+04, -0.2864890000E-02, 0.3061470000E-03 /
  data (epsData(118,i,35), i=1,3) / 0.4714180000E+04, -0.2310210000E-02, 0.2063620000E-03 /
  data (epsData(119,i,35), i=1,3) / 0.5249970000E+04, -0.1858000000E-02, 0.1384010000E-03 /
  data (epsData(120,i,35), i=1,3) / 0.5847040000E+04, -0.1492850000E-02, 0.9258070000E-04 /
  data (epsData(121,i,35), i=1,3) / 0.6509230000E+04, -0.1199930000E-02, 0.6222490000E-04 /
  data (epsData(122,i,35), i=1,3) / 0.7248600000E+04, -0.9635340000E-03, 0.4141020000E-04 /
  data (epsData(123,i,35), i=1,3) / 0.8088440000E+04, -0.7699450000E-03, 0.2726470000E-04 /
  data (epsData(124,i,35), i=1,3) / 0.9032160000E+04, -0.6139660000E-03, 0.1795460000E-04 /
  data (epsData(125,i,35), i=1,3) / 0.1008870000E+05, -0.4886080000E-03, 0.1175940000E-04 /
  data (epsData(126,i,35), i=1,3) / 0.1036660000E+05, -0.4618020000E-03, 0.1059500000E-04 /
  data (epsData(127,i,35), i=1,3) / 0.1036700000E+05, -0.4429660000E-03, 0.1059320000E-04 /
  data (epsData(128,i,35), i=1,3) / 0.1036700000E+05, -0.4617570000E-03, 0.1059300000E-04 /
  data (epsData(129,i,35), i=1,3) / 0.1160660000E+05, -0.3639540000E-03, 0.6865640000E-05 /
  data (epsData(130,i,35), i=1,3) / 0.1334730000E+05, -0.2564500000E-03, 0.4007180000E-05 /
  data (epsData(131,i,35), i=1,3) / 0.1344880000E+05, -0.2405960000E-03, 0.3890460000E-05 /
  data (epsData(132,i,35), i=1,3) / 0.1346940000E+05, -0.2287820000E-03, 0.5670510000E-05 /
  data (epsData(133,i,35), i=1,3) / 0.1347350000E+05, -0.2223190000E-03, 0.9465880000E-05 /
  data (epsData(134,i,35), i=1,3) / 0.1347400000E+05, -0.2019510000E-03, 0.1083430000E-04 /
  data (epsData(135,i,35), i=1,3) / 0.1347400000E+05, -0.2213320000E-03, 0.1097690000E-04 /
  data (epsData(136,i,35), i=1,3) / 0.1347850000E+05, -0.2231670000E-03, 0.2185570000E-04 /
  data (epsData(137,i,35), i=1,3) / 0.1349000000E+05, -0.2341530000E-03, 0.2791370000E-04 /
  data (epsData(138,i,35), i=1,3) / 0.1353260000E+05, -0.2437780000E-03, 0.2741680000E-04 /
  data (epsData(139,i,35), i=1,3) / 0.1378160000E+05, -0.2481520000E-03, 0.2598980000E-04 /
  data (epsData(140,i,35), i=1,3) / 0.1672820000E+05, -0.1809560000E-03, 0.1301490000E-04 /
  data (epsData(141,i,35), i=1,3) / 0.1882400000E+05, -0.1444660000E-03, 0.8437230000E-05 /
  data (epsData(142,i,35), i=1,3) / 0.2110270000E+05, -0.1155680000E-03, 0.5513600000E-05 /
  data (epsData(143,i,35), i=1,3) / 0.2362790000E+05, -0.9242980000E-04, 0.3603870000E-05 /
  data (epsData(144,i,35), i=1,3) / 0.2644620000E+05, -0.7386030000E-04, 0.2348080000E-05 /
  data (epsData(145,i,35), i=1,3) / 0.2959610000E+05, -0.5898340000E-04, 0.1524120000E-05 /
  data (epsData(146,i,35), i=1,3) / 0.3312470000E+05, -0.4706600000E-04, 0.9850220000E-06 /
  data (epsData(147,i,35), i=1,3) / 0.3708140000E+05, -0.3752890000E-04, 0.6334480000E-06 /
  data (epsData(148,i,35), i=1,3) / 0.4151810000E+05, -0.2990850000E-04, 0.4055650000E-06 /
  data (epsData(149,i,35), i=1,3) / 0.4649730000E+05, -0.2382190000E-04, 0.2584770000E-06 /
  data (epsData(150,i,35), i=1,3) / 0.5208590000E+05, -0.1896510000E-04, 0.1639330000E-06 /
  data (epsData(151,i,35), i=1,3) / 0.5836100000E+05, -0.1509150000E-04, 0.1034930000E-06 /
  data (epsData(152,i,35), i=1,3) / 0.6540210000E+05, -0.1200670000E-04, 0.6510210000E-07 /
  data (epsData(153,i,35), i=1,3) / 0.7331320000E+05, -0.9547940000E-05, 0.4085460000E-07 /
  data (epsData(154,i,35), i=1,3) / 0.8218710000E+05, -0.7592830000E-05, 0.2541500000E-07 /
  data (epsData(155,i,35), i=1,3) / 0.9213880000E+05, -0.6039470000E-05, 0.1576070000E-07 /
  data (epsData(156,i,35), i=1,3) / 0.1000000000E+06, -0.5154620000E-05, 0.1117820000E-07 /
  data (epsData(1,i,36), i=1,3) / 0.2506580000E-02, -0.8154130000E+02, 0.2532260000E+03 /
  data (epsData(2,i,36), i=1,3) / 0.4693440000E-02, -0.6686540000E+02, 0.8679730000E+02 /
  data (epsData(3,i,36), i=1,3) / 0.7560590000E-02, -0.2418850000E+02, 0.4081100000E+02 /
  data (epsData(4,i,36), i=1,3) / 0.1110800000E-01, -0.5535300000E+01, 0.2312680000E+02 /
  data (epsData(5,i,36), i=1,3) / 0.1533570000E-01, 0.3244290000E+01, 0.1477190000E+02 /
  data (epsData(6,i,36), i=1,3) / 0.2024370000E-01, 0.7788830000E+01, 0.1023540000E+02 /
  data (epsData(7,i,36), i=1,3) / 0.2583200000E-01, 0.1033380000E+02, 0.7515190000E+01 /
  data (epsData(8,i,36), i=1,3) / 0.3210060000E-01, 0.1185190000E+02, 0.5759610000E+01 /
  data (epsData(9,i,36), i=1,3) / 0.3904940000E-01, 0.1280510000E+02, 0.4561050000E+01 /
  data (epsData(10,i,36), i=1,3) / 0.4667860000E-01, 0.1342960000E+02, 0.3705930000E+01 /
  data (epsData(11,i,36), i=1,3) / 0.5498800000E-01, 0.1385370000E+02, 0.3073910000E+01 /
  data (epsData(12,i,36), i=1,3) / 0.7364770000E-01, 0.1436590000E+02, 0.2218570000E+01 /
  data (epsData(13,i,36), i=1,3) / 0.1067390000E+00, 0.1475200000E+02, 0.1500060000E+01 /
  data (epsData(14,i,36), i=1,3) / 0.1912910000E+00, 0.1494570000E+02, 0.8747150000E+00 /
  data (epsData(15,i,36), i=1,3) / 0.5364900000E+00, 0.1499960000E+02, 0.3228030000E+00 /
  data (epsData(16,i,36), i=1,3) / 0.2987270000E+01, 0.1532290000E+02, 0.7157960000E-01 /
  data (epsData(17,i,36), i=1,3) / 0.6357560000E+01, 0.1664590000E+02, 0.5155120000E-01 /
  data (epsData(18,i,36), i=1,3) / 0.8895260000E+01, 0.1872300000E+02, 0.5701450000E-01 /
  data (epsData(19,i,36), i=1,3) / 0.1074250000E+02, 0.2139240000E+02, 0.7244060000E-01 /
  data (epsData(20,i,36), i=1,3) / 0.1211350000E+02, 0.2474600000E+02, 0.1011740000E+00 /
  data (epsData(21,i,36), i=1,3) / 0.1302890000E+02, 0.2855040000E+02, 0.1514790000E+00 /
  data (epsData(22,i,36), i=1,3) / 0.1356690000E+02, 0.3242800000E+02, 0.2493310000E+00 /
  data (epsData(23,i,36), i=1,3) / 0.1370310000E+02, 0.3395160000E+02, 0.2940130000E+00 /
  data (epsData(24,i,36), i=1,3) / 0.1384000000E+02, 0.3597400000E+02, 0.5402030000E+00 /
  data (epsData(25,i,36), i=1,3) / 0.1397750000E+02, 0.3927450000E+02, 0.7778940000E+00 /
  data (epsData(26,i,36), i=1,3) / 0.1411670000E+02, 0.4166980000E+02, 0.5834390000E+01 /
  data (epsData(27,i,36), i=1,3) / 0.1427990000E+02, 0.4085480000E+02, 0.1032520000E+02 /
  data (epsData(28,i,36), i=1,3) / 0.1436670000E+02, 0.3933790000E+02, 0.1119580000E+02 /
  data (epsData(29,i,36), i=1,3) / 0.1478880000E+02, 0.3814200000E+02, 0.1331860000E+02 /
  data (epsData(30,i,36), i=1,3) / 0.1553710000E+02, 0.3917040000E+02, 0.1870620000E+02 /
  data (epsData(31,i,36), i=1,3) / 0.1606470000E+02, 0.3862210000E+02, 0.2447950000E+02 /
  data (epsData(32,i,36), i=1,3) / 0.1655750000E+02, 0.3580090000E+02, 0.2805970000E+02 /
  data (epsData(33,i,36), i=1,3) / 0.1697900000E+02, 0.3541580000E+02, 0.3097050000E+02 /
  data (epsData(34,i,36), i=1,3) / 0.1785010000E+02, 0.3099810000E+02, 0.3911620000E+02 /
  data (epsData(35,i,36), i=1,3) / 0.1859840000E+02, 0.2430870000E+02, 0.4495630000E+02 /
  data (epsData(36,i,36), i=1,3) / 0.1941470000E+02, 0.1509200000E+02, 0.4834800000E+02 /
  data (epsData(37,i,36), i=1,3) / 0.2039130000E+02, 0.4162690000E+01, 0.4842200000E+02 /
  data (epsData(38,i,36), i=1,3) / 0.2115320000E+02, -0.3743240000E+01, 0.4612170000E+02 /
  data (epsData(39,i,36), i=1,3) / 0.2185560000E+02, -0.9841070000E+01, 0.4213580000E+02 /
  data (epsData(40,i,36), i=1,3) / 0.2280760000E+02, -0.1513350000E+02, 0.3534480000E+02 /
  data (epsData(41,i,36), i=1,3) / 0.2381480000E+02, -0.1759440000E+02, 0.2857430000E+02 /
  data (epsData(42,i,36), i=1,3) / 0.2499850000E+02, -0.1828910000E+02, 0.2210660000E+02 /
  data (epsData(43,i,36), i=1,3) / 0.2637910000E+02, -0.1759920000E+02, 0.1631410000E+02 /
  data (epsData(44,i,36), i=1,3) / 0.2796570000E+02, -0.1589880000E+02, 0.1162940000E+02 /
  data (epsData(45,i,36), i=1,3) / 0.2990000000E+02, -0.1365030000E+02, 0.7941510000E+01 /
  data (epsData(46,i,36), i=1,3) / 0.3192050000E+02, -0.1156730000E+02, 0.5521480000E+01 /
  data (epsData(47,i,36), i=1,3) / 0.3416680000E+02, -0.9667870000E+01, 0.3814280000E+01 /
  data (epsData(48,i,36), i=1,3) / 0.3664850000E+02, -0.8013590000E+01, 0.2610600000E+01 /
  data (epsData(49,i,36), i=1,3) / 0.3926760000E+02, -0.6633180000E+01, 0.1795900000E+01 /
  data (epsData(50,i,36), i=1,3) / 0.4207470000E+02, -0.5472800000E+01, 0.1240090000E+01 /
  data (epsData(51,i,36), i=1,3) / 0.4518190000E+02, -0.4479240000E+01, 0.8566640000E+00 /
  data (epsData(52,i,36), i=1,3) / 0.4858910000E+02, -0.3647790000E+01, 0.5992180000E+00 /
  data (epsData(53,i,36), i=1,3) / 0.5229630000E+02, -0.2962310000E+01, 0.4286680000E+00 /
  data (epsData(54,i,36), i=1,3) / 0.5650190000E+02, -0.2376390000E+01, 0.3101880000E+00 /
  data (epsData(55,i,36), i=1,3) / 0.6103660000E+02, -0.1898670000E+01, 0.2317870000E+00 /
  data (epsData(56,i,36), i=1,3) / 0.6590070000E+02, -0.1508200000E+01, 0.1804430000E+00 /
  data (epsData(57,i,36), i=1,3) / 0.7158240000E+02, -0.1162000000E+01, 0.1434260000E+00 /
  data (epsData(58,i,36), i=1,3) / 0.7791750000E+02, -0.8665090000E+00, 0.1165390000E+00 /
  data (epsData(59,i,36), i=1,3) / 0.8495720000E+02, -0.6057950000E+00, 0.9652570000E-01 /
  data (epsData(60,i,36), i=1,3) / 0.9245680000E+02, -0.3522790000E+00, 0.8228720000E-01 /
  data (epsData(61,i,36), i=1,3) / 0.9335180000E+02, -0.3114450000E+00, 0.8184650000E-01 /
  data (epsData(62,i,36), i=1,3) / 0.9382760000E+02, -0.2818990000E+00, 0.9128330000E-01 /
  data (epsData(63,i,36), i=1,3) / 0.9426570000E+02, -0.2688230000E+00, 0.1061310000E+00 /
  data (epsData(64,i,36), i=1,3) / 0.9500540000E+02, -0.2414850000E+00, 0.1247510000E+00 /
  data (epsData(65,i,36), i=1,3) / 0.9560000000E+02, -0.2333600000E+00, 0.1454950000E+00 /
  data (epsData(66,i,36), i=1,3) / 0.9769430000E+02, -0.2101980000E+00, 0.1773280000E+00 /
  data (epsData(67,i,36), i=1,3) / 0.9989800000E+02, -0.1895010000E+00, 0.1918100000E+00 /
  data (epsData(68,i,36), i=1,3) / 0.1037080000E+03, -0.1311520000E+00, 0.1968030000E+00 /
  data (epsData(69,i,36), i=1,3) / 0.1083350000E+03, -0.5322580000E-01, 0.2184890000E+00 /
  data (epsData(70,i,36), i=1,3) / 0.1126820000E+03, 0.1491540000E-01, 0.2480560000E+00 /
  data (epsData(71,i,36), i=1,3) / 0.1172920000E+03, 0.7458200000E-01, 0.2918620000E+00 /
  data (epsData(72,i,36), i=1,3) / 0.1234090000E+03, 0.1318700000E+00, 0.3587600000E+00 /
  data (epsData(73,i,36), i=1,3) / 0.1300490000E+03, 0.1620030000E+00, 0.4404620000E+00 /
  data (epsData(74,i,36), i=1,3) / 0.1366670000E+03, 0.1546590000E+00, 0.5147860000E+00 /
  data (epsData(75,i,36), i=1,3) / 0.1448950000E+03, 0.1138730000E+00, 0.5736330000E+00 /
  data (epsData(76,i,36), i=1,3) / 0.1567280000E+03, 0.4463540000E-01, 0.6094710000E+00 /
  data (epsData(77,i,36), i=1,3) / 0.1701380000E+03, -0.3004420000E-01, 0.6061070000E+00 /
  data (epsData(78,i,36), i=1,3) / 0.1852150000E+03, -0.9554140000E-01, 0.5691310000E+00 /
  data (epsData(79,i,36), i=1,3) / 0.2059020000E+03, -0.1436850000E+00, 0.4991300000E+00 /
  data (epsData(80,i,36), i=1,3) / 0.2142090000E+03, -0.1439890000E+00, 0.4767660000E+00 /
  data (epsData(81,i,36), i=1,3) / 0.2263160000E+03, -0.1698070000E+00, 0.4601570000E+00 /
  data (epsData(82,i,36), i=1,3) / 0.2502170000E+03, -0.2032110000E+00, 0.3868940000E+00 /
  data (epsData(83,i,36), i=1,3) / 0.2785440000E+03, -0.2092700000E+00, 0.3107750000E+00 /
  data (epsData(84,i,36), i=1,3) / 0.2986370000E+03, -0.2033100000E+00, 0.2731790000E+00 /
  data (epsData(85,i,36), i=1,3) / 0.3371280000E+03, -0.1970770000E+00, 0.2059270000E+00 /
  data (epsData(86,i,36), i=1,3) / 0.3837170000E+03, -0.1753390000E+00, 0.1473550000E+00 /
  data (epsData(87,i,36), i=1,3) / 0.4382870000E+03, -0.1487670000E+00, 0.1017220000E+00 /
  data (epsData(88,i,36), i=1,3) / 0.4929060000E+03, -0.1250050000E+00, 0.7291580000E-01 /
  data (epsData(89,i,36), i=1,3) / 0.5658780000E+03, -0.1001190000E+00, 0.4793010000E-01 /
  data (epsData(90,i,36), i=1,3) / 0.6439240000E+03, -0.7995310000E-01, 0.3196490000E-01 /
  data (epsData(91,i,36), i=1,3) / 0.7194870000E+03, -0.6519210000E-01, 0.2262500000E-01 /
  data (epsData(92,i,36), i=1,3) / 0.7899950000E+03, -0.5437740000E-01, 0.1677710000E-01 /
  data (epsData(93,i,36), i=1,3) / 0.8738660000E+03, -0.4450570000E-01, 0.1203340000E-01 /
  data (epsData(94,i,36), i=1,3) / 0.9777200000E+03, -0.3539940000E-01, 0.8291930000E-02 /
  data (epsData(95,i,36), i=1,3) / 0.1093230000E+04, -0.2796990000E-01, 0.5708540000E-02 /
  data (epsData(96,i,36), i=1,3) / 0.1219990000E+04, -0.2195850000E-01, 0.3944800000E-02 /
  data (epsData(97,i,36), i=1,3) / 0.1362090000E+04, -0.1690070000E-01, 0.2713930000E-02 /
  data (epsData(98,i,36), i=1,3) / 0.1562470000E+04, -0.1121110000E-01, 0.1688020000E-02 /
  data (epsData(99,i,36), i=1,3) / 0.1630390000E+04, -0.8954790000E-02, 0.1447670000E-02 /
  data (epsData(100,i,36), i=1,3) / 0.1657500000E+04, -0.7545350000E-02, 0.1367950000E-02 /
  data (epsData(101,i,36), i=1,3) / 0.1669800000E+04, -0.6504160000E-02, 0.1378690000E-02 /
  data (epsData(102,i,36), i=1,3) / 0.1675280000E+04, -0.5702370000E-02, 0.1404950000E-02 /
  data (epsData(103,i,36), i=1,3) / 0.1678020000E+04, -0.5037480000E-02, 0.1487020000E-02 /
  data (epsData(104,i,36), i=1,3) / 0.1680160000E+04, -0.4315320000E-02, 0.1837860000E-02 /
  data (epsData(105,i,36), i=1,3) / 0.1681800000E+04, -0.3753790000E-02, 0.2427700000E-02 /
  data (epsData(106,i,36), i=1,3) / 0.1683140000E+04, -0.3487570000E-02, 0.3170310000E-02 /
  data (epsData(107,i,36), i=1,3) / 0.1684690000E+04, -0.3592790000E-02, 0.4108150000E-02 /
  data (epsData(108,i,36), i=1,3) / 0.1686160000E+04, -0.4067250000E-02, 0.4780590000E-02 /
  data (epsData(109,i,36), i=1,3) / 0.1688230000E+04, -0.4813160000E-02, 0.5144550000E-02 /
  data (epsData(110,i,36), i=1,3) / 0.1691330000E+04, -0.5515600000E-02, 0.5175600000E-02 /
  data (epsData(111,i,36), i=1,3) / 0.1697670000E+04, -0.6030960000E-02, 0.5167300000E-02 /
  data (epsData(112,i,36), i=1,3) / 0.1709420000E+04, -0.6616090000E-02, 0.5191020000E-02 /
  data (epsData(113,i,36), i=1,3) / 0.1722970000E+04, -0.6620990000E-02, 0.5021320000E-02 /
  data (epsData(114,i,36), i=1,3) / 0.1730670000E+04, -0.6173540000E-02, 0.5061940000E-02 /
  data (epsData(115,i,36), i=1,3) / 0.1734650000E+04, -0.5672630000E-02, 0.5556350000E-02 /
  data (epsData(116,i,36), i=1,3) / 0.1737760000E+04, -0.5802990000E-02, 0.6370840000E-02 /
  data (epsData(117,i,36), i=1,3) / 0.1741440000E+04, -0.6483510000E-02, 0.6702150000E-02 /
  data (epsData(118,i,36), i=1,3) / 0.1751150000E+04, -0.7214230000E-02, 0.6635890000E-02 /
  data (epsData(119,i,36), i=1,3) / 0.1773280000E+04, -0.7991420000E-02, 0.6395800000E-02 /
  data (epsData(120,i,36), i=1,3) / 0.1834920000E+04, -0.8639490000E-02, 0.5763850000E-02 /
  data (epsData(121,i,36), i=1,3) / 0.1900920000E+04, -0.8477990000E-02, 0.5104340000E-02 /
  data (epsData(122,i,36), i=1,3) / 0.1920710000E+04, -0.8160140000E-02, 0.5143460000E-02 /
  data (epsData(123,i,36), i=1,3) / 0.1933540000E+04, -0.8333960000E-02, 0.5412400000E-02 /
  data (epsData(124,i,36), i=1,3) / 0.1993600000E+04, -0.8623390000E-02, 0.4918140000E-02 /
  data (epsData(125,i,36), i=1,3) / 0.2165600000E+04, -0.8114000000E-02, 0.3699370000E-02 /
  data (epsData(126,i,36), i=1,3) / 0.2470500000E+04, -0.6736600000E-02, 0.2342060000E-02 /
  data (epsData(127,i,36), i=1,3) / 0.2772310000E+04, -0.5543800000E-02, 0.1565090000E-02 /
  data (epsData(128,i,36), i=1,3) / 0.3094010000E+04, -0.4536820000E-02, 0.1060970000E-02 /
  data (epsData(129,i,36), i=1,3) / 0.3448350000E+04, -0.3693650000E-02, 0.7249180000E-03 /
  data (epsData(130,i,36), i=1,3) / 0.3838070000E+04, -0.3000330000E-02, 0.4917930000E-03 /
  data (epsData(131,i,36), i=1,3) / 0.4266670000E+04, -0.2432760000E-02, 0.3340560000E-03 /
  data (epsData(132,i,36), i=1,3) / 0.4742390000E+04, -0.1968090000E-02, 0.2264130000E-03 /
  data (epsData(133,i,36), i=1,3) / 0.5272670000E+04, -0.1588510000E-02, 0.1528810000E-03 /
  data (epsData(134,i,36), i=1,3) / 0.5864690000E+04, -0.1279550000E-02, 0.1028270000E-03 /
  data (epsData(135,i,36), i=1,3) / 0.6517300000E+04, -0.1031900000E-02, 0.6961530000E-04 /
  data (epsData(136,i,36), i=1,3) / 0.7246730000E+04, -0.8307900000E-03, 0.4661870000E-04 /
  data (epsData(137,i,36), i=1,3) / 0.8076070000E+04, -0.6652580000E-03, 0.3085220000E-04 /
  data (epsData(138,i,36), i=1,3) / 0.9008790000E+04, -0.5314130000E-03, 0.2041570000E-04 /
  data (epsData(139,i,36), i=1,3) / 0.1005050000E+05, -0.4239010000E-03, 0.1343530000E-04 /
  data (epsData(140,i,36), i=1,3) / 0.1109980000E+05, -0.3446830000E-03, 0.9175300000E-05 /
  data (epsData(141,i,36), i=1,3) / 0.1110300000E+05, -0.3224210000E-03, 0.9164850000E-05 /
  data (epsData(142,i,36), i=1,3) / 0.1110300000E+05, -0.3444720000E-03, 0.9164690000E-05 /
  data (epsData(143,i,36), i=1,3) / 0.1242710000E+05, -0.2709660000E-03, 0.5945720000E-05 /
  data (epsData(144,i,36), i=1,3) / 0.1416240000E+05, -0.1935880000E-03, 0.3591040000E-05 /
  data (epsData(145,i,36), i=1,3) / 0.1429020000E+05, -0.1803220000E-03, 0.3468280000E-05 /
  data (epsData(146,i,36), i=1,3) / 0.1431730000E+05, -0.1709530000E-03, 0.4438610000E-05 /
  data (epsData(147,i,36), i=1,3) / 0.1432360000E+05, -0.1649770000E-03, 0.6093990000E-05 /
  data (epsData(148,i,36), i=1,3) / 0.1432570000E+05, -0.1620950000E-03, 0.8679560000E-05 /
  data (epsData(149,i,36), i=1,3) / 0.1432600000E+05, -0.1460040000E-03, 0.9378160000E-05 /
  data (epsData(150,i,36), i=1,3) / 0.1432600000E+05, -0.1615600000E-03, 0.9488230000E-05 /
  data (epsData(151,i,36), i=1,3) / 0.1433030000E+05, -0.1624610000E-03, 0.1841070000E-04 /
  data (epsData(152,i,36), i=1,3) / 0.1434130000E+05, -0.1716180000E-03, 0.2439390000E-04 /
  data (epsData(153,i,36), i=1,3) / 0.1437850000E+05, -0.1803700000E-03, 0.2426510000E-04 /
  data (epsData(154,i,36), i=1,3) / 0.1457870000E+05, -0.1863850000E-03, 0.2334640000E-04 /
  data (epsData(155,i,36), i=1,3) / 0.1553590000E+05, -0.1738950000E-03, 0.1852360000E-04 /
  data (epsData(156,i,36), i=1,3) / 0.1797210000E+05, -0.1348590000E-03, 0.1102610000E-04 /
  data (epsData(157,i,36), i=1,3) / 0.2021490000E+05, -0.1078530000E-03, 0.7151450000E-05 /
  data (epsData(158,i,36), i=1,3) / 0.2265620000E+05, -0.8635900000E-04, 0.4674480000E-05 /
  data (epsData(159,i,36), i=1,3) / 0.2536030000E+05, -0.6911790000E-04, 0.3054400000E-05 /
  data (epsData(160,i,36), i=1,3) / 0.2837400000E+05, -0.5527660000E-04, 0.1991460000E-05 /
  data (epsData(161,i,36), i=1,3) / 0.3174260000E+05, -0.4416960000E-04, 0.1293480000E-05 /
  data (epsData(162,i,36), i=1,3) / 0.3551590000E+05, -0.3526330000E-04, 0.8361730000E-06 /
  data (epsData(163,i,36), i=1,3) / 0.3974100000E+05, -0.2813790000E-04, 0.5382120000E-06 /
  data (epsData(164,i,36), i=1,3) / 0.4448450000E+05, -0.2243230000E-04, 0.3446190000E-06 /
  data (epsData(165,i,36), i=1,3) / 0.4980710000E+05, -0.1787320000E-04, 0.2196450000E-06 /
  data (epsData(166,i,36), i=1,3) / 0.5577950000E+05, -0.1423410000E-04, 0.1393030000E-06 /
  data (epsData(167,i,36), i=1,3) / 0.6247500000E+05, -0.1133420000E-04, 0.8838400000E-07 /
  data (epsData(168,i,36), i=1,3) / 0.6999310000E+05, -0.9021540000E-05, 0.5544590000E-07 /
  data (epsData(169,i,36), i=1,3) / 0.7844150000E+05, -0.7176950000E-05, 0.3471690000E-07 /
  data (epsData(170,i,36), i=1,3) / 0.8791590000E+05, -0.5710260000E-05, 0.2161340000E-07 /
  data (epsData(171,i,36), i=1,3) / 0.9839520000E+05, -0.4561920000E-05, 0.1348500000E-07 /
  data (epsData(172,i,36), i=1,3) / 0.1000000000E+06, -0.4440660000E-05, 0.1259440000E-07 /
  data (epsData(1,i,37), i=1,3) / 0.2506580000E-02, -0.8635120000E+03, 0.2179300000E+04 /
  data (epsData(2,i,37), i=1,3) / 0.4693440000E-02, -0.7003980000E+03, 0.7286490000E+03 /
  data (epsData(3,i,37), i=1,3) / 0.7560590000E-02, -0.3124550000E+03, 0.3348920000E+03 /
  data (epsData(4,i,37), i=1,3) / 0.1110800000E-01, -0.1437340000E+03, 0.1861550000E+03 /
  data (epsData(5,i,37), i=1,3) / 0.1533570000E-01, -0.6448120000E+02, 0.1170530000E+03 /
  data (epsData(6,i,37), i=1,3) / 0.2024370000E-01, -0.2350710000E+02, 0.8009210000E+02 /
  data (epsData(7,i,37), i=1,3) / 0.2583200000E-01, -0.5811700000E+00, 0.5821970000E+02 /
  data (epsData(8,i,37), i=1,3) / 0.3210060000E-01, 0.1308390000E+02, 0.4426270000E+02 /
  data (epsData(9,i,37), i=1,3) / 0.3904940000E-01, 0.2165680000E+02, 0.3482620000E+02 /
  data (epsData(10,i,37), i=1,3) / 0.4667860000E-01, 0.2726630000E+02, 0.2814880000E+02 /
  data (epsData(11,i,37), i=1,3) / 0.5498800000E-01, 0.3106840000E+02, 0.2324820000E+02 /
  data (epsData(12,i,37), i=1,3) / 0.6397770000E-01, 0.3372390000E+02, 0.1954260000E+02 /
  data (epsData(13,i,37), i=1,3) / 0.7364770000E-01, 0.3562720000E+02, 0.1667050000E+02 /
  data (epsData(14,i,37), i=1,3) / 0.9502860000E-01, 0.3806710000E+02, 0.1256640000E+02 /
  data (epsData(15,i,37), i=1,3) / 0.1191310000E+00, 0.3948100000E+02, 0.9827650000E+01 /
  data (epsData(16,i,37), i=1,3) / 0.1603860000E+00, 0.4068320000E+02, 0.7151800000E+01 /
  data (epsData(17,i,37), i=1,3) / 0.2249170000E+00, 0.4157470000E+02, 0.5150670000E+01 /
  data (epsData(18,i,37), i=1,3) / 0.4338630000E+00, 0.4209650000E+02, 0.2888530000E+01 /
  data (epsData(19,i,37), i=1,3) / 0.1055840000E+01, 0.4321560000E+02, 0.1539820000E+01 /
  data (epsData(20,i,37), i=1,3) / 0.1848280000E+01, 0.4645710000E+02, 0.1401560000E+01 /
  data (epsData(21,i,37), i=1,3) / 0.2441050000E+01, 0.5141610000E+02, 0.1730540000E+01 /
  data (epsData(22,i,37), i=1,3) / 0.2861130000E+01, 0.5780270000E+02, 0.2348480000E+01 /
  data (epsData(23,i,37), i=1,3) / 0.3116140000E+01, 0.6417190000E+02, 0.3287370000E+01 /
  data (epsData(24,i,37), i=1,3) / 0.3314540000E+01, 0.7201470000E+02, 0.5059760000E+01 /
  data (epsData(25,i,37), i=1,3) / 0.3450210000E+01, 0.8015820000E+02, 0.6543520000E+01 /
  data (epsData(26,i,37), i=1,3) / 0.3519060000E+01, 0.8631010000E+02, 0.9068420000E+01 /
  data (epsData(27,i,37), i=1,3) / 0.3729710000E+01, 0.1021930000E+03, 0.2640790000E+02 /
  data (epsData(28,i,37), i=1,3) / 0.3801290000E+01, 0.1125860000E+03, 0.3283430000E+02 /
  data (epsData(29,i,37), i=1,3) / 0.3873540000E+01, 0.1172690000E+03, 0.6041210000E+02 /
  data (epsData(30,i,37), i=1,3) / 0.3946480000E+01, 0.1096780000E+03, 0.8094650000E+02 /
  data (epsData(31,i,37), i=1,3) / 0.4020090000E+01, 0.9663260000E+02, 0.9771290000E+02 /
  data (epsData(32,i,37), i=1,3) / 0.4094390000E+01, 0.7636630000E+02, 0.1133700000E+03 /
  data (epsData(33,i,37), i=1,3) / 0.4169370000E+01, 0.5571820000E+02, 0.1100960000E+03 /
  data (epsData(34,i,37), i=1,3) / 0.4245030000E+01, 0.4175130000E+02, 0.1063250000E+03 /
  data (epsData(35,i,37), i=1,3) / 0.4398380000E+01, 0.2030020000E+02, 0.9920400000E+02 /
  data (epsData(36,i,37), i=1,3) / 0.4476080000E+01, 0.1144050000E+02, 0.9258180000E+02 /
  data (epsData(37,i,37), i=1,3) / 0.4633520000E+01, -0.2103140000E+00, 0.8033110000E+02 /
  data (epsData(38,i,37), i=1,3) / 0.4793680000E+01, -0.7745620000E+01, 0.6881990000E+02 /
  data (epsData(39,i,37), i=1,3) / 0.4956560000E+01, -0.1210630000E+02, 0.5814980000E+02 /
  data (epsData(40,i,37), i=1,3) / 0.5122160000E+01, -0.1430440000E+02, 0.4869690000E+02 /
  data (epsData(41,i,37), i=1,3) / 0.5290480000E+01, -0.1490940000E+02, 0.4021770000E+02 /
  data (epsData(42,i,37), i=1,3) / 0.5548060000E+01, -0.1384900000E+02, 0.2984610000E+02 /
  data (epsData(43,i,37), i=1,3) / 0.5811770000E+01, -0.1134820000E+02, 0.2176000000E+02 /
  data (epsData(44,i,37), i=1,3) / 0.5990980000E+01, -0.9198050000E+01, 0.1743380000E+02 /
  data (epsData(45,i,37), i=1,3) / 0.6172910000E+01, -0.7003870000E+01, 0.1378470000E+02 /
  data (epsData(46,i,37), i=1,3) / 0.6450900000E+01, -0.3546670000E+01, 0.9526880000E+01 /
  data (epsData(47,i,37), i=1,3) / 0.6639630000E+01, -0.1210050000E+01, 0.7337690000E+01 /
  data (epsData(48,i,37), i=1,3) / 0.6735020000E+01, -0.6437220000E-01, 0.6364680000E+01 /
  data (epsData(49,i,37), i=1,3) / 0.6831080000E+01, 0.1105590000E+01, 0.5506740000E+01 /
  data (epsData(50,i,37), i=1,3) / 0.7025260000E+01, 0.3367260000E+01, 0.4151210000E+01 /
  data (epsData(51,i,37), i=1,3) / 0.7123360000E+01, 0.4511150000E+01, 0.3589210000E+01 /
  data (epsData(52,i,37), i=1,3) / 0.7321620000E+01, 0.6626140000E+01, 0.2905010000E+01 /
  data (epsData(53,i,37), i=1,3) / 0.7421760000E+01, 0.7636730000E+01, 0.2650480000E+01 /
  data (epsData(54,i,37), i=1,3) / 0.7624100000E+01, 0.9331720000E+01, 0.2554780000E+01 /
  data (epsData(55,i,37), i=1,3) / 0.8036940000E+01, 0.1174540000E+02, 0.2300160000E+01 /
  data (epsData(56,i,37), i=1,3) / 0.8895260000E+01, 0.1559860000E+02, 0.1563120000E+01 /
  data (epsData(57,i,37), i=1,3) / 0.1062200000E+02, 0.2167530000E+02, 0.9230580000E+00 /
  data (epsData(58,i,37), i=1,3) / 0.1237170000E+02, 0.2826380000E+02, 0.6315800000E+00 /
  data (epsData(59,i,37), i=1,3) / 0.1343140000E+02, 0.3382040000E+02, 0.5582770000E+00 /
  data (epsData(60,i,37), i=1,3) / 0.1425470000E+02, 0.4027140000E+02, 0.5674150000E+00 /
  data (epsData(61,i,37), i=1,3) / 0.1467550000E+02, 0.4525790000E+02, 0.6379620000E+00 /
  data (epsData(62,i,37), i=1,3) / 0.1495950000E+02, 0.5032490000E+02, 0.8514830000E+00 /
  data (epsData(63,i,37), i=1,3) / 0.1510250000E+02, 0.5425810000E+02, 0.1164970000E+01 /
  data (epsData(64,i,37), i=1,3) / 0.1524620000E+02, 0.5925400000E+02, 0.3969330000E+01 /
  data (epsData(65,i,37), i=1,3) / 0.1532180000E+02, 0.6060020000E+02, 0.7699860000E+01 /
  data (epsData(66,i,37), i=1,3) / 0.1549590000E+02, 0.6049640000E+02, 0.1352370000E+02 /
  data (epsData(67,i,37), i=1,3) / 0.1571670000E+02, 0.6007150000E+02, 0.1532980000E+02 /
  data (epsData(68,i,37), i=1,3) / 0.1611670000E+02, 0.6505930000E+02, 0.1963920000E+02 /
  data (epsData(69,i,37), i=1,3) / 0.1625550000E+02, 0.6835210000E+02, 0.2319260000E+02 /
  data (epsData(70,i,37), i=1,3) / 0.1654800000E+02, 0.6945720000E+02, 0.3252120000E+02 /
  data (epsData(71,i,37), i=1,3) / 0.1711670000E+02, 0.6991120000E+02, 0.4852010000E+02 /
  data (epsData(72,i,37), i=1,3) / 0.1760750000E+02, 0.6451960000E+02, 0.6545070000E+02 /
  data (epsData(73,i,37), i=1,3) / 0.1796670000E+02, 0.5512300000E+02, 0.7624910000E+02 /
  data (epsData(74,i,37), i=1,3) / 0.1842280000E+02, 0.4051310000E+02, 0.8380870000E+02 /
  data (epsData(75,i,37), i=1,3) / 0.1924190000E+02, 0.1666430000E+02, 0.8765600000E+02 /
  data (epsData(76,i,37), i=1,3) / 0.1970140000E+02, 0.3479630000E+01, 0.8567160000E+02 /
  data (epsData(77,i,37), i=1,3) / 0.2041570000E+02, -0.1245720000E+02, 0.7819540000E+02 /
  data (epsData(78,i,37), i=1,3) / 0.2103610000E+02, -0.2287360000E+02, 0.6863290000E+02 /
  data (epsData(79,i,37), i=1,3) / 0.2178610000E+02, -0.2961080000E+02, 0.5617740000E+02 /
  data (epsData(80,i,37), i=1,3) / 0.2259130000E+02, -0.3206690000E+02, 0.4478980000E+02 /
  data (epsData(81,i,37), i=1,3) / 0.2355050000E+02, -0.3198610000E+02, 0.3425400000E+02 /
  data (epsData(82,i,37), i=1,3) / 0.2468420000E+02, -0.2987770000E+02, 0.2507690000E+02 /
  data (epsData(83,i,37), i=1,3) / 0.2601480000E+02, -0.2640240000E+02, 0.1787470000E+02 /
  data (epsData(84,i,37), i=1,3) / 0.2744340000E+02, -0.2287180000E+02, 0.1296350000E+02 /
  data (epsData(85,i,37), i=1,3) / 0.2910160000E+02, -0.1947360000E+02, 0.9313910000E+01 /
  data (epsData(86,i,37), i=1,3) / 0.3104420000E+02, -0.1632450000E+02, 0.6582110000E+01 /
  data (epsData(87,i,37), i=1,3) / 0.3325510000E+02, -0.1357290000E+02, 0.4596220000E+01 /
  data (epsData(88,i,37), i=1,3) / 0.3573000000E+02, -0.1121520000E+02, 0.3159610000E+01 /
  data (epsData(89,i,37), i=1,3) / 0.3846240000E+02, -0.9212190000E+01, 0.2143110000E+01 /
  data (epsData(90,i,37), i=1,3) / 0.4139850000E+02, -0.7561660000E+01, 0.1462720000E+01 /
  data (epsData(91,i,37), i=1,3) / 0.4465530000E+02, -0.6172460000E+01, 0.1001450000E+01 /
  data (epsData(92,i,37), i=1,3) / 0.4818190000E+02, -0.5039960000E+01, 0.7000560000E+00 /
  data (epsData(93,i,37), i=1,3) / 0.5210040000E+02, -0.4098990000E+01, 0.5012420000E+00 /
  data (epsData(94,i,37), i=1,3) / 0.5659870000E+02, -0.3299730000E+01, 0.3641490000E+00 /
  data (epsData(95,i,37), i=1,3) / 0.6172810000E+02, -0.2630290000E+01, 0.2698000000E+00 /
  data (epsData(96,i,37), i=1,3) / 0.6731730000E+02, -0.2093830000E+01, 0.2082940000E+00 /
  data (epsData(97,i,37), i=1,3) / 0.7360820000E+02, -0.1647500000E+01, 0.1640760000E+00 /
  data (epsData(98,i,37), i=1,3) / 0.8039510000E+02, -0.1286950000E+01, 0.1328480000E+00 /
  data (epsData(99,i,37), i=1,3) / 0.8795720000E+02, -0.9820550000E+00, 0.1104040000E+00 /
  data (epsData(100,i,37), i=1,3) / 0.9635180000E+02, -0.7199840000E+00, 0.9417520000E-01 /
  data (epsData(101,i,37), i=1,3) / 0.1105110000E+03, -0.3550970000E+00, 0.7519780000E-01 /
  data (epsData(102,i,37), i=1,3) / 0.1115030000E+03, -0.3183300000E+00, 0.7494660000E-01 /
  data (epsData(103,i,37), i=1,3) / 0.1118360000E+03, -0.3006740000E+00, 0.7766900000E-01 /
  data (epsData(104,i,37), i=1,3) / 0.1121960000E+03, -0.2866290000E+00, 0.9489820000E-01 /
  data (epsData(105,i,37), i=1,3) / 0.1129200000E+03, -0.2686510000E+00, 0.1092450000E+00 /
  data (epsData(106,i,37), i=1,3) / 0.1133060000E+03, -0.2648440000E+00, 0.1253640000E+00 /
  data (epsData(107,i,37), i=1,3) / 0.1146670000E+03, -0.2618200000E+00, 0.1478910000E+00 /
  data (epsData(108,i,37), i=1,3) / 0.1160050000E+03, -0.2551260000E+00, 0.1474190000E+00 /
  data (epsData(109,i,37), i=1,3) / 0.1198590000E+03, -0.1936110000E+00, 0.1537540000E+00 /
  data (epsData(110,i,37), i=1,3) / 0.1264700000E+03, -0.9144030000E-01, 0.1862380000E+00 /
  data (epsData(111,i,37), i=1,3) / 0.1299150000E+03, -0.4013600000E-01, 0.2159440000E+00 /
  data (epsData(112,i,37), i=1,3) / 0.1342590000E+03, 0.7983760000E-02, 0.2618830000E+00 /
  data (epsData(113,i,37), i=1,3) / 0.1404340000E+03, 0.5481500000E-01, 0.3284980000E+00 /
  data (epsData(114,i,37), i=1,3) / 0.1467510000E+03, 0.7607130000E-01, 0.4007280000E+00 /
  data (epsData(115,i,37), i=1,3) / 0.1527840000E+03, 0.6831730000E-01, 0.4606560000E+00 /
  data (epsData(116,i,37), i=1,3) / 0.1613440000E+03, 0.3451670000E-01, 0.5119700000E+00 /
  data (epsData(117,i,37), i=1,3) / 0.1732870000E+03, -0.1873210000E-01, 0.5445230000E+00 /
  data (epsData(118,i,37), i=1,3) / 0.1878180000E+03, -0.8431510000E-01, 0.5427940000E+00 /
  data (epsData(119,i,37), i=1,3) / 0.2035660000E+03, -0.1370410000E+00, 0.5065560000E+00 /
  data (epsData(120,i,37), i=1,3) / 0.2274250000E+03, -0.1712280000E+00, 0.4361830000E+00 /
  data (epsData(121,i,37), i=1,3) / 0.2375020000E+03, -0.1651450000E+00, 0.4111240000E+00 /
  data (epsData(122,i,37), i=1,3) / 0.2425010000E+03, -0.1678660000E+00, 0.4193980000E+00 /
  data (epsData(123,i,37), i=1,3) / 0.2517700000E+03, -0.1902540000E+00, 0.4028030000E+00 /
  data (epsData(124,i,37), i=1,3) / 0.2704720000E+03, -0.2109780000E+00, 0.3509450000E+00 /
  data (epsData(125,i,37), i=1,3) / 0.3033950000E+03, -0.2125410000E+00, 0.2753700000E+00 /
  data (epsData(126,i,37), i=1,3) / 0.3276200000E+03, -0.2009090000E+00, 0.2342610000E+00 /
  data (epsData(127,i,37), i=1,3) / 0.3737700000E+03, -0.1887590000E+00, 0.1727530000E+00 /
  data (epsData(128,i,37), i=1,3) / 0.4292200000E+03, -0.1627060000E+00, 0.1191970000E+00 /
  data (epsData(129,i,37), i=1,3) / 0.4858410000E+03, -0.1374780000E+00, 0.8387780000E-01 /
  data (epsData(130,i,37), i=1,3) / 0.5454860000E+03, -0.1153660000E+00, 0.6007010000E-01 /
  data (epsData(131,i,37), i=1,3) / 0.5997790000E+03, -0.9826790000E-01, 0.4497440000E-01 /
  data (epsData(132,i,37), i=1,3) / 0.6785150000E+03, -0.7906560000E-01, 0.3055610000E-01 /
  data (epsData(133,i,37), i=1,3) / 0.7656650000E+03, -0.6322310000E-01, 0.2075480000E-01 /
  data (epsData(134,i,37), i=1,3) / 0.8598910000E+03, -0.5059110000E-01, 0.1422210000E-01 /
  data (epsData(135,i,37), i=1,3) / 0.9576990000E+03, -0.4086010000E-01, 0.1001950000E-01 /
  data (epsData(136,i,37), i=1,3) / 0.1056750000E+04, -0.3340050000E-01, 0.7249490000E-02 /
  data (epsData(137,i,37), i=1,3) / 0.1163010000E+04, -0.2729680000E-01, 0.5265540000E-02 /
  data (epsData(138,i,37), i=1,3) / 0.1281040000E+04, -0.2211830000E-01, 0.3799810000E-02 /
  data (epsData(139,i,37), i=1,3) / 0.1415370000E+04, -0.1759790000E-01, 0.2705570000E-02 /
  data (epsData(140,i,37), i=1,3) / 0.1575430000E+04, -0.1337180000E-01, 0.1871410000E-02 /
  data (epsData(141,i,37), i=1,3) / 0.1727060000E+04, -0.9558250000E-02, 0.1350760000E-02 /
  data (epsData(142,i,37), i=1,3) / 0.1770180000E+04, -0.7969970000E-02, 0.1241900000E-02 /
  data (epsData(143,i,37), i=1,3) / 0.1785640000E+04, -0.7013440000E-02, 0.1235450000E-02 /
  data (epsData(144,i,37), i=1,3) / 0.1794100000E+04, -0.6189080000E-02, 0.1244480000E-02 /
  data (epsData(145,i,37), i=1,3) / 0.1798350000E+04, -0.5513040000E-02, 0.1276050000E-02 /
  data (epsData(146,i,37), i=1,3) / 0.1801180000E+04, -0.4786580000E-02, 0.1343220000E-02 /
  data (epsData(147,i,37), i=1,3) / 0.1802590000E+04, -0.4219220000E-02, 0.1433960000E-02 /
  data (epsData(148,i,37), i=1,3) / 0.1804100000E+04, -0.3386990000E-02, 0.1788860000E-02 /
  data (epsData(149,i,37), i=1,3) / 0.1804990000E+04, -0.2751650000E-02, 0.2364090000E-02 /
  data (epsData(150,i,37), i=1,3) / 0.1805620000E+04, -0.2385590000E-02, 0.3016970000E-02 /
  data (epsData(151,i,37), i=1,3) / 0.1806230000E+04, -0.2260490000E-02, 0.3837510000E-02 /
  data (epsData(152,i,37), i=1,3) / 0.1806880000E+04, -0.2478200000E-02, 0.4690960000E-02 /
  data (epsData(153,i,37), i=1,3) / 0.1807890000E+04, -0.3184140000E-02, 0.5661630000E-02 /
  data (epsData(154,i,37), i=1,3) / 0.1808820000E+04, -0.4067970000E-02, 0.6206850000E-02 /
  data (epsData(155,i,37), i=1,3) / 0.1809840000E+04, -0.5023380000E-02, 0.6282470000E-02 /
  data (epsData(156,i,37), i=1,3) / 0.1811550000E+04, -0.6011380000E-02, 0.5902400000E-02 /
  data (epsData(157,i,37), i=1,3) / 0.1813950000E+04, -0.6559090000E-02, 0.5278760000E-02 /
  data (epsData(158,i,37), i=1,3) / 0.1817880000E+04, -0.6697380000E-02, 0.4835960000E-02 /
  data (epsData(159,i,37), i=1,3) / 0.1839930000E+04, -0.7117820000E-02, 0.4448750000E-02 /
  data (epsData(160,i,37), i=1,3) / 0.1857880000E+04, -0.6621840000E-02, 0.4374380000E-02 /
  data (epsData(161,i,37), i=1,3) / 0.1862580000E+04, -0.6087480000E-02, 0.4435170000E-02 /
  data (epsData(162,i,37), i=1,3) / 0.1864660000E+04, -0.5537840000E-02, 0.4752890000E-02 /
  data (epsData(163,i,37), i=1,3) / 0.1866020000E+04, -0.5261550000E-02, 0.5403010000E-02 /
  data (epsData(164,i,37), i=1,3) / 0.1867400000E+04, -0.5545720000E-02, 0.6162090000E-02 /
  data (epsData(165,i,37), i=1,3) / 0.1869150000E+04, -0.6310940000E-02, 0.6611310000E-02 /
  data (epsData(166,i,37), i=1,3) / 0.1871450000E+04, -0.7061830000E-02, 0.6427870000E-02 /
  data (epsData(167,i,37), i=1,3) / 0.1875520000E+04, -0.7435490000E-02, 0.5982550000E-02 /
  data (epsData(168,i,37), i=1,3) / 0.1905910000E+04, -0.8063700000E-02, 0.5504590000E-02 /
  data (epsData(169,i,37), i=1,3) / 0.1980010000E+04, -0.8459340000E-02, 0.4872670000E-02 /
  data (epsData(170,i,37), i=1,3) / 0.2048760000E+04, -0.8168250000E-02, 0.4360650000E-02 /
  data (epsData(171,i,37), i=1,3) / 0.2066310000E+04, -0.7870310000E-02, 0.4487760000E-02 /
  data (epsData(172,i,37), i=1,3) / 0.2086430000E+04, -0.8145190000E-02, 0.4601600000E-02 /
  data (epsData(173,i,37), i=1,3) / 0.2202160000E+04, -0.8106310000E-02, 0.3841600000E-02 /
  data (epsData(174,i,37), i=1,3) / 0.2450080000E+04, -0.7144910000E-02, 0.2652260000E-02 /
  data (epsData(175,i,37), i=1,3) / 0.2767410000E+04, -0.5884950000E-02, 0.1735830000E-02 /
  data (epsData(176,i,37), i=1,3) / 0.3097220000E+04, -0.4819600000E-02, 0.1167890000E-02 /
  data (epsData(177,i,37), i=1,3) / 0.3439260000E+04, -0.3962850000E-02, 0.8080800000E-03 /
  data (epsData(178,i,37), i=1,3) / 0.3833260000E+04, -0.3218120000E-02, 0.5482230000E-03 /
  data (epsData(179,i,37), i=1,3) / 0.4264730000E+04, -0.2609850000E-02, 0.3719720000E-03 /
  data (epsData(180,i,37), i=1,3) / 0.4743260000E+04, -0.2111410000E-02, 0.2518610000E-03 /
  data (epsData(181,i,37), i=1,3) / 0.5275110000E+04, -0.1704920000E-02, 0.1701280000E-03 /
  data (epsData(182,i,37), i=1,3) / 0.5868680000E+04, -0.1373780000E-02, 0.1144840000E-03 /
  data (epsData(183,i,37), i=1,3) / 0.6522350000E+04, -0.1108410000E-02, 0.7755690000E-04 /
  data (epsData(184,i,37), i=1,3) / 0.7254350000E+04, -0.8924230000E-03, 0.5192840000E-04 /
  data (epsData(185,i,37), i=1,3) / 0.8084830000E+04, -0.7150030000E-03, 0.3440130000E-04 /
  data (epsData(186,i,37), i=1,3) / 0.9019910000E+04, -0.5714160000E-03, 0.2276540000E-04 /
  data (epsData(187,i,37), i=1,3) / 0.1006530000E+05, -0.4560770000E-03, 0.1498120000E-04 /
  data (epsData(188,i,37), i=1,3) / 0.1123670000E+05, -0.3632220000E-03, 0.9828250000E-05 /
  data (epsData(189,i,37), i=1,3) / 0.1186530000E+05, -0.3242630000E-03, 0.7977600000E-05 /
  data (epsData(190,i,37), i=1,3) / 0.1186700000E+05, -0.3065020000E-03, 0.7973190000E-05 /
  data (epsData(191,i,37), i=1,3) / 0.1186700000E+05, -0.3241610000E-03, 0.7973070000E-05 /
  data (epsData(192,i,37), i=1,3) / 0.1329090000E+05, -0.2547630000E-03, 0.5160470000E-05 /
  data (epsData(193,i,37), i=1,3) / 0.1503750000E+05, -0.1856170000E-03, 0.3202540000E-05 /
  data (epsData(194,i,37), i=1,3) / 0.1516550000E+05, -0.1735660000E-03, 0.3099830000E-05 /
  data (epsData(195,i,37), i=1,3) / 0.1519310000E+05, -0.1641000000E-03, 0.4344060000E-05 /
  data (epsData(196,i,37), i=1,3) / 0.1519950000E+05, -0.1574590000E-03, 0.8476200000E-05 /
  data (epsData(197,i,37), i=1,3) / 0.1520000000E+05, -0.1396550000E-03, 0.9604500000E-05 /
  data (epsData(198,i,37), i=1,3) / 0.1520000000E+05, -0.1568560000E-03, 0.9718130000E-05 /
  data (epsData(199,i,37), i=1,3) / 0.1520380000E+05, -0.1588680000E-03, 0.1642490000E-04 /
  data (epsData(200,i,37), i=1,3) / 0.1521510000E+05, -0.1666150000E-03, 0.2106810000E-04 /
  data (epsData(201,i,37), i=1,3) / 0.1526380000E+05, -0.1748810000E-03, 0.2137180000E-04 /
  data (epsData(202,i,37), i=1,3) / 0.1551680000E+05, -0.1790090000E-03, 0.2039180000E-04 /
  data (epsData(203,i,37), i=1,3) / 0.1696830000E+05, -0.1586240000E-03, 0.1494540000E-04 /
  data (epsData(204,i,37), i=1,3) / 0.1932640000E+05, -0.1254130000E-03, 0.9277960000E-05 /
  data (epsData(205,i,37), i=1,3) / 0.2171910000E+05, -0.1002930000E-03, 0.6027990000E-05 /
  data (epsData(206,i,37), i=1,3) / 0.2433650000E+05, -0.8027490000E-04, 0.3940030000E-05 /
  data (epsData(207,i,37), i=1,3) / 0.2724430000E+05, -0.6420500000E-04, 0.2571300000E-05 /
  data (epsData(208,i,37), i=1,3) / 0.3048990000E+05, -0.5130800000E-04, 0.1673370000E-05 /
  data (epsData(209,i,37), i=1,3) / 0.3411810000E+05, -0.4097450000E-04, 0.1084680000E-05 /
  data (epsData(210,i,37), i=1,3) / 0.3817680000E+05, -0.3270610000E-04, 0.7003380000E-06 /
  data (epsData(211,i,37), i=1,3) / 0.4272750000E+05, -0.2608720000E-04, 0.4500900000E-06 /
  data (epsData(212,i,37), i=1,3) / 0.4783560000E+05, -0.2079140000E-04, 0.2876620000E-06 /
  data (epsData(213,i,37), i=1,3) / 0.5356590000E+05, -0.1656300000E-04, 0.1831090000E-06 /
  data (epsData(214,i,37), i=1,3) / 0.6000100000E+05, -0.1318660000E-04, 0.1158420000E-06 /
  data (epsData(215,i,37), i=1,3) / 0.6723750000E+05, -0.1049060000E-04, 0.7321670000E-07 /
  data (epsData(216,i,37), i=1,3) / 0.7534630000E+05, -0.8347170000E-05, 0.4581350000E-07 /
  data (epsData(217,i,37), i=1,3) / 0.8444560000E+05, -0.6640880000E-05, 0.2865080000E-07 /
  data (epsData(218,i,37), i=1,3) / 0.9463180000E+05, -0.5287710000E-05, 0.1781940000E-07 /
  data (epsData(219,i,37), i=1,3) / 0.1000000000E+06, -0.4768830000E-05, 0.1415230000E-07 /
  data (epsData(1,i,38), i=1,3) / 0.2506580000E-02, -0.1844240000E+04, 0.5885370000E+04 /
  data (epsData(2,i,38), i=1,3) / 0.4693440000E-02, -0.1646500000E+04, 0.2088710000E+04 /
  data (epsData(3,i,38), i=1,3) / 0.7560590000E-02, -0.7350700000E+03, 0.1012110000E+04 /
  data (epsData(4,i,38), i=1,3) / 0.1110800000E-01, -0.3334370000E+03, 0.5876450000E+03 /
  data (epsData(5,i,38), i=1,3) / 0.1533570000E-01, -0.1437750000E+03, 0.3825600000E+03 /
  data (epsData(6,i,38), i=1,3) / 0.2024370000E-01, -0.4541340000E+02, 0.2690190000E+03 /
  data (epsData(7,i,38), i=1,3) / 0.2583200000E-01, 0.9741780000E+01, 0.1998080000E+03 /
  data (epsData(8,i,38), i=1,3) / 0.3210060000E-01, 0.4267660000E+02, 0.1545200000E+03 /
  data (epsData(9,i,38), i=1,3) / 0.3904940000E-01, 0.6337580000E+02, 0.1232440000E+03 /
  data (epsData(10,i,38), i=1,3) / 0.4667860000E-01, 0.7695030000E+02, 0.1007140000E+03 /
  data (epsData(11,i,38), i=1,3) / 0.5498800000E-01, 0.8618120000E+02, 0.8392720000E+02 /
  data (epsData(12,i,38), i=1,3) / 0.6397770000E-01, 0.9266280000E+02, 0.7107090000E+02 /
  data (epsData(13,i,38), i=1,3) / 0.7364770000E-01, 0.9735220000E+02, 0.6099740000E+02 /
  data (epsData(14,i,38), i=1,3) / 0.9502860000E-01, 0.1035820000E+03, 0.4641750000E+02 /
  data (epsData(15,i,38), i=1,3) / 0.1191310000E+00, 0.1075130000E+03, 0.3699460000E+02 /
  data (epsData(16,i,38), i=1,3) / 0.1603860000E+00, 0.1103300000E+03, 0.2843480000E+02 /
  data (epsData(17,i,38), i=1,3) / 0.2612640000E+00, 0.1124110000E+03, 0.1788030000E+02 /
  data (epsData(18,i,38), i=1,3) / 0.5638470000E+00, 0.1162810000E+03, 0.1056970000E+02 /
  data (epsData(19,i,38), i=1,3) / 0.9096770000E+00, 0.1252640000E+03, 0.9653320000E+01 /
  data (epsData(20,i,38), i=1,3) / 0.1172610000E+01, 0.1372140000E+03, 0.1032190000E+02 /
  data (epsData(21,i,38), i=1,3) / 0.1380830000E+01, 0.1519070000E+03, 0.1315790000E+02 /
  data (epsData(22,i,38), i=1,3) / 0.1468870000E+01, 0.1616210000E+03, 0.1399380000E+02 /
  data (epsData(23,i,38), i=1,3) / 0.1653130000E+01, 0.1864050000E+03, 0.2257280000E+02 /
  data (epsData(24,i,38), i=1,3) / 0.1700900000E+01, 0.1961070000E+03, 0.2414130000E+02 /
  data (epsData(25,i,38), i=1,3) / 0.1749350000E+01, 0.2109560000E+03, 0.2552490000E+02 /
  data (epsData(26,i,38), i=1,3) / 0.1798470000E+01, 0.2273890000E+03, 0.3890630000E+02 /
  data (epsData(27,i,38), i=1,3) / 0.1898770000E+01, 0.2464550000E+03, 0.7335850000E+02 /
  data (epsData(28,i,38), i=1,3) / 0.2001780000E+01, 0.2659270000E+03, 0.1015380000E+03 /
  data (epsData(29,i,38), i=1,3) / 0.2054310000E+01, 0.2922700000E+03, 0.1136170000E+03 /
  data (epsData(30,i,38), i=1,3) / 0.2107520000E+01, 0.3129980000E+03, 0.1773640000E+03 /
  data (epsData(31,i,38), i=1,3) / 0.2161410000E+01, 0.2977970000E+03, 0.2497970000E+03 /
  data (epsData(32,i,38), i=1,3) / 0.2271220000E+01, 0.2212170000E+03, 0.3295820000E+03 /
  data (epsData(33,i,38), i=1,3) / 0.2327150000E+01, 0.1690590000E+03, 0.3630860000E+03 /
  data (epsData(34,i,38), i=1,3) / 0.2383760000E+01, 0.1050630000E+03, 0.3766870000E+03 /
  data (epsData(35,i,38), i=1,3) / 0.2441050000E+01, 0.4981230000E+02, 0.3533740000E+03 /
  data (epsData(36,i,38), i=1,3) / 0.2499020000E+01, 0.1283260000E+02, 0.3315330000E+03 /
  data (epsData(37,i,38), i=1,3) / 0.2617000000E+01, -0.3998610000E+02, 0.2918520000E+03 /
  data (epsData(38,i,38), i=1,3) / 0.2677010000E+01, -0.6249590000E+02, 0.2720590000E+03 /
  data (epsData(39,i,38), i=1,3) / 0.2737710000E+01, -0.8042470000E+02, 0.2458920000E+03 /
  data (epsData(40,i,38), i=1,3) / 0.2861130000E+01, -0.9924740000E+02, 0.1986230000E+03 /
  data (epsData(41,i,38), i=1,3) / 0.2987270000E+01, -0.1044600000E+03, 0.1581090000E+03 /
  data (epsData(42,i,38), i=1,3) / 0.3116140000E+01, -0.1029530000E+03, 0.1262130000E+03 /
  data (epsData(43,i,38), i=1,3) / 0.3247730000E+01, -0.9572910000E+02, 0.9824870000E+02 /
  data (epsData(44,i,38), i=1,3) / 0.3382030000E+01, -0.8588250000E+02, 0.8180000000E+02 /
  data (epsData(45,i,38), i=1,3) / 0.3588600000E+01, -0.7577830000E+02, 0.6253720000E+02 /
  data (epsData(46,i,38), i=1,3) / 0.3873540000E+01, -0.6437320000E+02, 0.4327970000E+02 /
  data (epsData(47,i,38), i=1,3) / 0.4169370000E+01, -0.5330210000E+02, 0.2807050000E+02 /
  data (epsData(48,i,38), i=1,3) / 0.4476080000E+01, -0.4227980000E+02, 0.1763710000E+02 /
  data (epsData(49,i,38), i=1,3) / 0.4793680000E+01, -0.3236310000E+02, 0.1099130000E+02 /
  data (epsData(50,i,38), i=1,3) / 0.5039020000E+01, -0.2595760000E+02, 0.7714300000E+01 /
  data (epsData(51,i,38), i=1,3) / 0.5290480000E+01, -0.2039510000E+02, 0.5481330000E+01 /
  data (epsData(52,i,38), i=1,3) / 0.5548060000E+01, -0.1559830000E+02, 0.3926350000E+01 /
  data (epsData(53,i,38), i=1,3) / 0.5811770000E+01, -0.1148790000E+02, 0.2851260000E+01 /
  data (epsData(54,i,38), i=1,3) / 0.6081600000E+01, -0.7927250000E+01, 0.2118750000E+01 /
  data (epsData(55,i,38), i=1,3) / 0.6264890000E+01, -0.5780890000E+01, 0.1757350000E+01 /
  data (epsData(56,i,38), i=1,3) / 0.6450900000E+01, -0.3830430000E+01, 0.1647770000E+01 /
  data (epsData(57,i,38), i=1,3) / 0.6544930000E+01, -0.2935590000E+01, 0.1596670000E+01 /
  data (epsData(58,i,38), i=1,3) / 0.6639630000E+01, -0.2093050000E+01, 0.1618620000E+01 /
  data (epsData(59,i,38), i=1,3) / 0.6735020000E+01, -0.1333300000E+01, 0.1658800000E+01 /
  data (epsData(60,i,38), i=1,3) / 0.6831080000E+01, -0.6510770000E+00, 0.1696380000E+01 /
  data (epsData(61,i,38), i=1,3) / 0.6927830000E+01, -0.2389740000E-01, 0.1699090000E+01 /
  data (epsData(62,i,38), i=1,3) / 0.7123360000E+01, 0.1139090000E+01, 0.1667310000E+01 /
  data (epsData(63,i,38), i=1,3) / 0.7421760000E+01, 0.2718320000E+01, 0.1524590000E+01 /
  data (epsData(64,i,38), i=1,3) / 0.7829160000E+01, 0.4657210000E+01, 0.1331850000E+01 /
  data (epsData(65,i,38), i=1,3) / 0.8353710000E+01, 0.6789360000E+01, 0.1181580000E+01 /
  data (epsData(66,i,38), i=1,3) / 0.9228360000E+01, 0.9632260000E+01, 0.1010160000E+01 /
  data (epsData(67,i,38), i=1,3) / 0.1086380000E+02, 0.1383140000E+02, 0.7421550000E+00 /
  data (epsData(68,i,38), i=1,3) / 0.1343140000E+02, 0.2014560000E+02, 0.4861920000E+00 /
  data (epsData(69,i,38), i=1,3) / 0.1495950000E+02, 0.2503290000E+02, 0.4205220000E+00 /
  data (epsData(70,i,38), i=1,3) / 0.1612260000E+02, 0.3025980000E+02, 0.4123440000E+00 /
  data (epsData(71,i,38), i=1,3) / 0.1702350000E+02, 0.3609330000E+02, 0.4411750000E+00 /
  data (epsData(72,i,38), i=1,3) / 0.1779300000E+02, 0.4346270000E+02, 0.5128360000E+00 /
  data (epsData(73,i,38), i=1,3) / 0.1826280000E+02, 0.4996030000E+02, 0.6004400000E+00 /
  data (epsData(74,i,38), i=1,3) / 0.1873870000E+02, 0.5944280000E+02, 0.7681520000E+00 /
  data (epsData(75,i,38), i=1,3) / 0.1905940000E+02, 0.6884290000E+02, 0.9818720000E+00 /
  data (epsData(76,i,38), i=1,3) / 0.1922080000E+02, 0.7517280000E+02, 0.1183760000E+01 /
  data (epsData(77,i,38), i=1,3) / 0.1938290000E+02, 0.8331470000E+02, 0.1409560000E+01 /
  data (epsData(78,i,38), i=1,3) / 0.1954560000E+02, 0.9435310000E+02, 0.1975780000E+01 /
  data (epsData(79,i,38), i=1,3) / 0.1970900000E+02, 0.1110940000E+03, 0.2773740000E+01 /
  data (epsData(80,i,38), i=1,3) / 0.1987310000E+02, 0.1395860000E+03, 0.7064190000E+01 /
  data (epsData(81,i,38), i=1,3) / 0.2003790000E+02, 0.1838260000E+03, 0.3723920000E+02 /
  data (epsData(82,i,38), i=1,3) / 0.2010310000E+02, 0.1920110000E+03, 0.8235360000E+02 /
  data (epsData(83,i,38), i=1,3) / 0.2023610000E+02, 0.1697870000E+03, 0.1411490000E+03 /
  data (epsData(84,i,38), i=1,3) / 0.2029590000E+02, 0.1452430000E+03, 0.1671090000E+03 /
  data (epsData(85,i,38), i=1,3) / 0.2032990000E+02, 0.1267740000E+03, 0.1688660000E+03 /
  data (epsData(86,i,38), i=1,3) / 0.2044830000E+02, 0.9113050000E+02, 0.1688600000E+03 /
  data (epsData(87,i,38), i=1,3) / 0.2059120000E+02, 0.5674680000E+02, 0.1688380000E+03 /
  data (epsData(88,i,38), i=1,3) / 0.2064420000E+02, 0.4315440000E+02, 0.1621100000E+03 /
  data (epsData(89,i,38), i=1,3) / 0.2075850000E+02, 0.2756570000E+02, 0.1449870000E+03 /
  data (epsData(90,i,38), i=1,3) / 0.2085140000E+02, 0.2341890000E+02, 0.1313430000E+03 /
  data (epsData(91,i,38), i=1,3) / 0.2088370000E+02, 0.2457420000E+02, 0.1266510000E+03 /
  data (epsData(92,i,38), i=1,3) / 0.2091670000E+02, 0.2694190000E+02, 0.1243340000E+03 /
  data (epsData(93,i,38), i=1,3) / 0.2116670000E+02, 0.3229320000E+02, 0.1249130000E+03 /
  data (epsData(94,i,38), i=1,3) / 0.2120510000E+02, 0.3588670000E+02, 0.1260910000E+03 /
  data (epsData(95,i,38), i=1,3) / 0.2124420000E+02, 0.3811200000E+02, 0.1344970000E+03 /
  data (epsData(96,i,38), i=1,3) / 0.2130540000E+02, 0.3287040000E+02, 0.1469790000E+03 /
  data (epsData(97,i,38), i=1,3) / 0.2145000000E+02, 0.1020230000E+02, 0.1628840000E+03 /
  data (epsData(98,i,38), i=1,3) / 0.2149590000E+02, -0.2390180000E+01, 0.1678430000E+03 /
  data (epsData(99,i,38), i=1,3) / 0.2154800000E+02, -0.1737360000E+02, 0.1625660000E+03 /
  data (epsData(100,i,38), i=1,3) / 0.2169320000E+02, -0.3934360000E+02, 0.1455530000E+03 /
  data (epsData(101,i,38), i=1,3) / 0.2187180000E+02, -0.5503870000E+02, 0.1241600000E+03 /
  data (epsData(102,i,38), i=1,3) / 0.2205140000E+02, -0.5977240000E+02, 0.1017440000E+03 /
  data (epsData(103,i,38), i=1,3) / 0.2211670000E+02, -0.5713150000E+02, 0.9520020000E+02 /
  data (epsData(104,i,38), i=1,3) / 0.2232890000E+02, -0.5655170000E+02, 0.8775810000E+02 /
  data (epsData(105,i,38), i=1,3) / 0.2248400000E+02, -0.6080970000E+02, 0.7988720000E+02 /
  data (epsData(106,i,38), i=1,3) / 0.2278170000E+02, -0.6165690000E+02, 0.6155480000E+02 /
  data (epsData(107,i,38), i=1,3) / 0.2296670000E+02, -0.5875880000E+02, 0.5111450000E+02 /
  data (epsData(108,i,38), i=1,3) / 0.2316260000E+02, -0.5350720000E+02, 0.4448300000E+02 /
  data (epsData(109,i,38), i=1,3) / 0.2342280000E+02, -0.4772800000E+02, 0.3848570000E+02 /
  data (epsData(110,i,38), i=1,3) / 0.2381570000E+02, -0.4160010000E+02, 0.3321050000E+02 /
  data (epsData(111,i,38), i=1,3) / 0.2436980000E+02, -0.3637520000E+02, 0.2873710000E+02 /
  data (epsData(112,i,38), i=1,3) / 0.2549090000E+02, -0.3047440000E+02, 0.2136030000E+02 /
  data (epsData(113,i,38), i=1,3) / 0.2687290000E+02, -0.2524100000E+02, 0.1531030000E+02 /
  data (epsData(114,i,38), i=1,3) / 0.2825560000E+02, -0.2138230000E+02, 0.1176480000E+02 /
  data (epsData(115,i,38), i=1,3) / 0.3000760000E+02, -0.1795930000E+02, 0.8716040000E+01 /
  data (epsData(116,i,38), i=1,3) / 0.3207710000E+02, -0.1498430000E+02, 0.6279720000E+01 /
  data (epsData(117,i,38), i=1,3) / 0.3450060000E+02, -0.1238850000E+02, 0.4373400000E+01 /
  data (epsData(118,i,38), i=1,3) / 0.3716970000E+02, -0.1019590000E+02, 0.2968180000E+01 /
  data (epsData(119,i,38), i=1,3) / 0.3992890000E+02, -0.8416860000E+01, 0.2043300000E+01 /
  data (epsData(120,i,38), i=1,3) / 0.4309090000E+02, -0.6854800000E+01, 0.1381160000E+01 /
  data (epsData(121,i,38), i=1,3) / 0.4645150000E+02, -0.5604750000E+01, 0.9630930000E+00 /
  data (epsData(122,i,38), i=1,3) / 0.5016690000E+02, -0.4569430000E+01, 0.6827380000E+00 /
  data (epsData(123,i,38), i=1,3) / 0.5433190000E+02, -0.3708880000E+01, 0.4926700000E+00 /
  data (epsData(124,i,38), i=1,3) / 0.5915070000E+02, -0.2977040000E+01, 0.3577980000E+00 /
  data (epsData(125,i,38), i=1,3) / 0.6443650000E+02, -0.2384230000E+01, 0.2679770000E+00 /
  data (epsData(126,i,38), i=1,3) / 0.7033460000E+02, -0.1897630000E+01, 0.2103590000E+00 /
  data (epsData(127,i,38), i=1,3) / 0.7730070000E+02, -0.1481540000E+01, 0.1693520000E+00 /
  data (epsData(128,i,38), i=1,3) / 0.8496340000E+02, -0.1148180000E+01, 0.1388310000E+00 /
  data (epsData(129,i,38), i=1,3) / 0.9332270000E+02, -0.8788860000E+00, 0.1171060000E+00 /
  data (epsData(130,i,38), i=1,3) / 0.1026730000E+03, -0.6532040000E+00, 0.9963850000E-01 /
  data (epsData(131,i,38), i=1,3) / 0.1134000000E+03, -0.4530780000E+00, 0.8344410000E-01 /
  data (epsData(132,i,38), i=1,3) / 0.1305220000E+03, -0.1825850000E+00, 0.6482060000E-01 /
  data (epsData(133,i,38), i=1,3) / 0.1326530000E+03, -0.1399190000E+00, 0.6335010000E-01 /
  data (epsData(134,i,38), i=1,3) / 0.1333680000E+03, -0.1202920000E+00, 0.6326670000E-01 /
  data (epsData(135,i,38), i=1,3) / 0.1337270000E+03, -0.1064660000E+00, 0.6383960000E-01 /
  data (epsData(136,i,38), i=1,3) / 0.1340870000E+03, -0.8593700000E-01, 0.6954480000E-01 /
  data (epsData(137,i,38), i=1,3) / 0.1342410000E+03, -0.7950060000E-01, 0.8430160000E-01 /
  data (epsData(138,i,38), i=1,3) / 0.1343960000E+03, -0.8237650000E-01, 0.9767370000E-01 /
  data (epsData(139,i,38), i=1,3) / 0.1346910000E+03, -0.9084120000E-01, 0.1024200000E+00 /
  data (epsData(140,i,38), i=1,3) / 0.1350850000E+03, -0.9192280000E-01, 0.9938420000E-01 /
  data (epsData(141,i,38), i=1,3) / 0.1353440000E+03, -0.8537160000E-01, 0.9809040000E-01 /
  data (epsData(142,i,38), i=1,3) / 0.1358670000E+03, -0.7092100000E-01, 0.1077810000E+00 /
  data (epsData(143,i,38), i=1,3) / 0.1359630000E+03, -0.6721420000E-01, 0.1133570000E+00 /
  data (epsData(144,i,38), i=1,3) / 0.1361800000E+03, -0.7041570000E-01, 0.1268850000E+00 /
  data (epsData(145,i,38), i=1,3) / 0.1363700000E+03, -0.7769710000E-01, 0.1290030000E+00 /
  data (epsData(146,i,38), i=1,3) / 0.1366890000E+03, -0.8358800000E-01, 0.1276100000E+00 /
  data (epsData(147,i,38), i=1,3) / 0.1371440000E+03, -0.8112340000E-01, 0.1236720000E+00 /
  data (epsData(148,i,38), i=1,3) / 0.1382870000E+03, -0.7462950000E-01, 0.1253960000E+00 /
  data (epsData(149,i,38), i=1,3) / 0.1426670000E+03, -0.2820880000E-01, 0.1329860000E+00 /
  data (epsData(150,i,38), i=1,3) / 0.1452600000E+03, 0.6800420000E-02, 0.1437090000E+00 /
  data (epsData(151,i,38), i=1,3) / 0.1480220000E+03, 0.4236480000E-01, 0.1647230000E+00 /
  data (epsData(152,i,38), i=1,3) / 0.1515770000E+03, 0.7811870000E-01, 0.2005780000E+00 /
  data (epsData(153,i,38), i=1,3) / 0.1581180000E+03, 0.1208970000E+00, 0.2702490000E+00 /
  data (epsData(154,i,38), i=1,3) / 0.1638220000E+03, 0.1352900000E+00, 0.3388660000E+00 /
  data (epsData(155,i,38), i=1,3) / 0.1691110000E+03, 0.1214870000E+00, 0.3955260000E+00 /
  data (epsData(156,i,38), i=1,3) / 0.1757840000E+03, 0.8611660000E-01, 0.4382820000E+00 /
  data (epsData(157,i,38), i=1,3) / 0.1920680000E+03, 0.1736500000E-01, 0.4793240000E+00 /
  data (epsData(158,i,38), i=1,3) / 0.2061090000E+03, -0.4574030000E-01, 0.4837240000E+00 /
  data (epsData(159,i,38), i=1,3) / 0.2193440000E+03, -0.9423160000E-01, 0.4583160000E+00 /
  data (epsData(160,i,38), i=1,3) / 0.2426700000E+03, -0.1349510000E+00, 0.3997530000E+00 /
  data (epsData(161,i,38), i=1,3) / 0.2625690000E+03, -0.1418220000E+00, 0.3527820000E+00 /
  data (epsData(162,i,38), i=1,3) / 0.2675980000E+03, -0.1287290000E+00, 0.3467040000E+00 /
  data (epsData(163,i,38), i=1,3) / 0.2692960000E+03, -0.1195500000E+00, 0.3575910000E+00 /
  data (epsData(164,i,38), i=1,3) / 0.2705440000E+03, -0.1292700000E+00, 0.3766500000E+00 /
  data (epsData(165,i,38), i=1,3) / 0.2718630000E+03, -0.1510630000E+00, 0.3724910000E+00 /
  data (epsData(166,i,38), i=1,3) / 0.2746560000E+03, -0.1578540000E+00, 0.3534450000E+00 /
  data (epsData(167,i,38), i=1,3) / 0.2792600000E+03, -0.1526540000E+00, 0.3443030000E+00 /
  data (epsData(168,i,38), i=1,3) / 0.2813290000E+03, -0.1610310000E+00, 0.3509570000E+00 /
  data (epsData(169,i,38), i=1,3) / 0.2860480000E+03, -0.1724970000E+00, 0.3329020000E+00 /
  data (epsData(170,i,38), i=1,3) / 0.3123580000E+03, -0.1821470000E+00, 0.2752910000E+00 /
  data (epsData(171,i,38), i=1,3) / 0.3470530000E+03, -0.1780540000E+00, 0.2162920000E+00 /
  data (epsData(172,i,38), i=1,3) / 0.3621010000E+03, -0.1718350000E+00, 0.2009510000E+00 /
  data (epsData(173,i,38), i=1,3) / 0.4055170000E+03, -0.1632930000E+00, 0.1520770000E+00 /
  data (epsData(174,i,38), i=1,3) / 0.4619690000E+03, -0.1423620000E+00, 0.1067650000E+00 /
  data (epsData(175,i,38), i=1,3) / 0.5234740000E+03, -0.1199220000E+00, 0.7482200000E-01 /
  data (epsData(176,i,38), i=1,3) / 0.6033030000E+03, -0.9644500000E-01, 0.4892710000E-01 /
  data (epsData(177,i,38), i=1,3) / 0.6873850000E+03, -0.7726230000E-01, 0.3264530000E-01 /
  data (epsData(178,i,38), i=1,3) / 0.7747860000E+03, -0.6228570000E-01, 0.2236850000E-01 /
  data (epsData(179,i,38), i=1,3) / 0.8608420000E+03, -0.5103490000E-01, 0.1600400000E-01 /
  data (epsData(180,i,38), i=1,3) / 0.9477960000E+03, -0.4226450000E-01, 0.1171400000E-01 /
  data (epsData(181,i,38), i=1,3) / 0.1043620000E+04, -0.3483150000E-01, 0.8517230000E-02 /
  data (epsData(182,i,38), i=1,3) / 0.1156010000E+04, -0.2822580000E-01, 0.6050580000E-02 /
  data (epsData(183,i,38), i=1,3) / 0.1283650000E+04, -0.2261560000E-01, 0.4255650000E-02 /
  data (epsData(184,i,38), i=1,3) / 0.1423170000E+04, -0.1802080000E-01, 0.3000380000E-02 /
  data (epsData(185,i,38), i=1,3) / 0.1577630000E+04, -0.1414060000E-01, 0.2109130000E-02 /
  data (epsData(186,i,38), i=1,3) / 0.1800780000E+04, -0.9592790000E-02, 0.1326780000E-02 /
  data (epsData(187,i,38), i=1,3) / 0.1878850000E+04, -0.7733800000E-02, 0.1146260000E-02 /
  data (epsData(188,i,38), i=1,3) / 0.1907750000E+04, -0.6628990000E-02, 0.1086810000E-02 /
  data (epsData(189,i,38), i=1,3) / 0.1919390000E+04, -0.5863110000E-02, 0.1063730000E-02 /
  data (epsData(190,i,38), i=1,3) / 0.1926680000E+04, -0.5150140000E-02, 0.1118040000E-02 /
  data (epsData(191,i,38), i=1,3) / 0.1931070000E+04, -0.4426260000E-02, 0.1159690000E-02 /
  data (epsData(192,i,38), i=1,3) / 0.1932530000E+04, -0.4073320000E-02, 0.1190130000E-02 /
  data (epsData(193,i,38), i=1,3) / 0.1933990000E+04, -0.3612340000E-02, 0.1240780000E-02 /
  data (epsData(194,i,38), i=1,3) / 0.1935460000E+04, -0.2961590000E-02, 0.1334550000E-02 /
  data (epsData(195,i,38), i=1,3) / 0.1936920000E+04, -0.1917750000E-02, 0.1542600000E-02 /
  data (epsData(196,i,38), i=1,3) / 0.1938390000E+04, -0.7340780000E-04, 0.2204530000E-02 /
  data (epsData(197,i,38), i=1,3) / 0.1939850000E+04, 0.1522910000E-02, 0.6815370000E-02 /
  data (epsData(198,i,38), i=1,3) / 0.1940030000E+04, 0.1477570000E-02, 0.8079330000E-02 /
  data (epsData(199,i,38), i=1,3) / 0.1940150000E+04, 0.1136560000E-02, 0.8973200000E-02 /
  data (epsData(200,i,38), i=1,3) / 0.1940270000E+04, 0.5343500000E-03, 0.9664770000E-02 /
  data (epsData(201,i,38), i=1,3) / 0.1940470000E+04, -0.5746570000E-03, 0.1064040000E-01 /
  data (epsData(202,i,38), i=1,3) / 0.1940570000E+04, -0.1398640000E-02, 0.1090560000E-01 /
  data (epsData(203,i,38), i=1,3) / 0.1940820000E+04, -0.3034310000E-02, 0.1119400000E-01 /
  data (epsData(204,i,38), i=1,3) / 0.1941180000E+04, -0.5202970000E-02, 0.1060800000E-01 /
  data (epsData(205,i,38), i=1,3) / 0.1941530000E+04, -0.6562640000E-02, 0.9500440000E-02 /
  data (epsData(206,i,38), i=1,3) / 0.1941910000E+04, -0.7255290000E-02, 0.8111980000E-02 /
  data (epsData(207,i,38), i=1,3) / 0.1942400000E+04, -0.7296350000E-02, 0.6802430000E-02 /
  data (epsData(208,i,38), i=1,3) / 0.1943200000E+04, -0.6931640000E-02, 0.5720150000E-02 /
  data (epsData(209,i,38), i=1,3) / 0.1944420000E+04, -0.6630710000E-02, 0.5130520000E-02 /
  data (epsData(210,i,38), i=1,3) / 0.1946860000E+04, -0.6320130000E-02, 0.4570380000E-02 /
  data (epsData(211,i,38), i=1,3) / 0.1951890000E+04, -0.6272620000E-02, 0.4414390000E-02 /
  data (epsData(212,i,38), i=1,3) / 0.1964430000E+04, -0.6558850000E-02, 0.4011990000E-02 /
  data (epsData(213,i,38), i=1,3) / 0.1980020000E+04, -0.6366860000E-02, 0.3808810000E-02 /
  data (epsData(214,i,38), i=1,3) / 0.1995520000E+04, -0.5991890000E-02, 0.3795420000E-02 /
  data (epsData(215,i,38), i=1,3) / 0.2001440000E+04, -0.5443500000E-02, 0.3821200000E-02 /
  data (epsData(216,i,38), i=1,3) / 0.2003770000E+04, -0.4907380000E-02, 0.3937150000E-02 /
  data (epsData(217,i,38), i=1,3) / 0.2004960000E+04, -0.4395230000E-02, 0.4140900000E-02 /
  data (epsData(218,i,38), i=1,3) / 0.2005850000E+04, -0.3764480000E-02, 0.4587670000E-02 /
  data (epsData(219,i,38), i=1,3) / 0.2006150000E+04, -0.3506060000E-02, 0.4899620000E-02 /
  data (epsData(220,i,38), i=1,3) / 0.2006450000E+04, -0.3279150000E-02, 0.5377020000E-02 /
  data (epsData(221,i,38), i=1,3) / 0.2006750000E+04, -0.3189400000E-02, 0.6066240000E-02 /
  data (epsData(222,i,38), i=1,3) / 0.2007140000E+04, -0.3487100000E-02, 0.7118090000E-02 /
  data (epsData(223,i,38), i=1,3) / 0.2007520000E+04, -0.4424260000E-02, 0.7859780000E-02 /
  data (epsData(224,i,38), i=1,3) / 0.2007960000E+04, -0.5622090000E-02, 0.7929710000E-02 /
  data (epsData(225,i,38), i=1,3) / 0.2008440000E+04, -0.6589460000E-02, 0.7454410000E-02 /
  data (epsData(226,i,38), i=1,3) / 0.2009070000E+04, -0.7068400000E-02, 0.6588670000E-02 /
  data (epsData(227,i,38), i=1,3) / 0.2009940000E+04, -0.7032800000E-02, 0.5888230000E-02 /
  data (epsData(228,i,38), i=1,3) / 0.2012230000E+04, -0.6876110000E-02, 0.5366650000E-02 /
  data (epsData(229,i,38), i=1,3) / 0.2017300000E+04, -0.6814730000E-02, 0.5149000000E-02 /
  data (epsData(230,i,38), i=1,3) / 0.2042270000E+04, -0.7192900000E-02, 0.4773180000E-02 /
  data (epsData(231,i,38), i=1,3) / 0.2116350000E+04, -0.7496460000E-02, 0.4251570000E-02 /
  data (epsData(232,i,38), i=1,3) / 0.2195790000E+04, -0.7208040000E-02, 0.3740190000E-02 /
  data (epsData(233,i,38), i=1,3) / 0.2213950000E+04, -0.6945080000E-02, 0.3801580000E-02 /
  data (epsData(234,i,38), i=1,3) / 0.2222670000E+04, -0.7056400000E-02, 0.4015960000E-02 /
  data (epsData(235,i,38), i=1,3) / 0.2283660000E+04, -0.7262800000E-02, 0.3719480000E-02 /
  data (epsData(236,i,38), i=1,3) / 0.2486650000E+04, -0.6763000000E-02, 0.2771020000E-02 /
  data (epsData(237,i,38), i=1,3) / 0.2836850000E+04, -0.5567300000E-02, 0.1751000000E-02 /
  data (epsData(238,i,38), i=1,3) / 0.3182920000E+04, -0.4562660000E-02, 0.1168660000E-02 /
  data (epsData(239,i,38), i=1,3) / 0.3540130000E+04, -0.3749520000E-02, 0.8050670000E-03 /
  data (epsData(240,i,38), i=1,3) / 0.3945700000E+04, -0.3050480000E-02, 0.5467740000E-03 /
  data (epsData(241,i,38), i=1,3) / 0.4389860000E+04, -0.2476770000E-02, 0.3711810000E-03 /
  data (epsData(242,i,38), i=1,3) / 0.4882820000E+04, -0.2004970000E-02, 0.2513930000E-03 /
  data (epsData(243,i,38), i=1,3) / 0.5430300000E+04, -0.1619830000E-02, 0.1699150000E-03 /
  data (epsData(244,i,38), i=1,3) / 0.6042460000E+04, -0.1305090000E-02, 0.1143210000E-03 /
  data (epsData(245,i,38), i=1,3) / 0.6727680000E+04, -0.1049610000E-02, 0.7672740000E-04 /
  data (epsData(246,i,38), i=1,3) / 0.7496070000E+04, -0.8418650000E-03, 0.5109390000E-04 /
  data (epsData(247,i,38), i=1,3) / 0.8359500000E+04, -0.6738560000E-03, 0.3392770000E-04 /
  data (epsData(248,i,38), i=1,3) / 0.9319320000E+04, -0.5393940000E-03, 0.2241540000E-04 /
  data (epsData(249,i,38), i=1,3) / 0.1039540000E+05, -0.4310030000E-03, 0.1480360000E-04 /
  data (epsData(250,i,38), i=1,3) / 0.1160450000E+05, -0.3434490000E-03, 0.9722590000E-05 /
  data (epsData(251,i,38), i=1,3) / 0.1265430000E+05, -0.2867800000E-03, 0.6976720000E-05 /
  data (epsData(252,i,38), i=1,3) / 0.1265800000E+05, -0.2694860000E-03, 0.6968860000E-05 /
  data (epsData(253,i,38), i=1,3) / 0.1265800000E+05, -0.2866010000E-03, 0.6968760000E-05 /
  data (epsData(254,i,38), i=1,3) / 0.1418100000E+05, -0.2249970000E-03, 0.4496260000E-05 /
  data (epsData(255,i,38), i=1,3) / 0.1593320000E+05, -0.1665350000E-03, 0.2868090000E-05 /
  data (epsData(256,i,38), i=1,3) / 0.1606970000E+05, -0.1554680000E-03, 0.2773070000E-05 /
  data (epsData(257,i,38), i=1,3) / 0.1609560000E+05, -0.1477740000E-03, 0.3828300000E-05 /
  data (epsData(258,i,38), i=1,3) / 0.1610210000E+05, -0.1422790000E-03, 0.5992390000E-05 /
  data (epsData(259,i,38), i=1,3) / 0.1610430000E+05, -0.1399780000E-03, 0.9374240000E-05 /
  data (epsData(260,i,38), i=1,3) / 0.1610500000E+05, -0.1184410000E-03, 0.1124030000E-04 /
  data (epsData(261,i,38), i=1,3) / 0.1610500000E+05, -0.1395360000E-03, 0.1137040000E-04 /
  data (epsData(262,i,38), i=1,3) / 0.1610860000E+05, -0.1434800000E-03, 0.1699180000E-04 /
  data (epsData(263,i,38), i=1,3) / 0.1611610000E+05, -0.1490150000E-03, 0.1779600000E-04 /
  data (epsData(264,i,38), i=1,3) / 0.1615750000E+05, -0.1561070000E-03, 0.1898000000E-04 /
  data (epsData(265,i,38), i=1,3) / 0.1638540000E+05, -0.1607240000E-03, 0.1830680000E-04 /
  data (epsData(266,i,38), i=1,3) / 0.1759080000E+05, -0.1472910000E-03, 0.1415140000E-04 /
  data (epsData(267,i,38), i=1,3) / 0.2028130000E+05, -0.1144520000E-03, 0.8514480000E-05 /
  data (epsData(268,i,38), i=1,3) / 0.2280090000E+05, -0.9149540000E-04, 0.5523890000E-05 /
  data (epsData(269,i,38), i=1,3) / 0.2555070000E+05, -0.7323370000E-04, 0.3609360000E-05 /
  data (epsData(270,i,38), i=1,3) / 0.2860280000E+05, -0.5858230000E-04, 0.2356000000E-05 /
  data (epsData(271,i,38), i=1,3) / 0.3200300000E+05, -0.4683870000E-04, 0.1534140000E-05 /
  data (epsData(272,i,38), i=1,3) / 0.3580740000E+05, -0.3741450000E-04, 0.9944890000E-06 /
  data (epsData(273,i,38), i=1,3) / 0.4007090000E+05, -0.2985970000E-04, 0.6418560000E-06 /
  data (epsData(274,i,38), i=1,3) / 0.4485160000E+05, -0.2381250000E-04, 0.4121850000E-06 /
  data (epsData(275,i,38), i=1,3) / 0.5020980000E+05, -0.1898190000E-04, 0.2635060000E-06 /
  data (epsData(276,i,38), i=1,3) / 0.5622480000E+05, -0.1512140000E-04, 0.1675800000E-06 /
  data (epsData(277,i,38), i=1,3) / 0.6296740000E+05, -0.1204370000E-04, 0.1066550000E-06 /
  data (epsData(278,i,38), i=1,3) / 0.7054290000E+05, -0.9587000000E-05, 0.6692090000E-07 /
  data (epsData(279,i,38), i=1,3) / 0.7904970000E+05, -0.7628500000E-05, 0.4206300000E-07 /
  data (epsData(280,i,38), i=1,3) / 0.8858900000E+05, -0.6071010000E-05, 0.2625180000E-07 /
  data (epsData(281,i,38), i=1,3) / 0.9902380000E+05, -0.4865550000E-05, 0.1649930000E-07 /
  data (epsData(282,i,38), i=1,3) / 0.1000000000E+06, -0.4796530000E-05, 0.1582930000E-07 /
  data (epsData(1,i,39), i=1,3) / 0.2506580000E-02, -0.2866750000E+04, 0.8853460000E+04 /
  data (epsData(2,i,39), i=1,3) / 0.4693440000E-02, -0.2474430000E+04, 0.3094930000E+04 /
  data (epsData(3,i,39), i=1,3) / 0.7560590000E-02, -0.1049780000E+04, 0.1480540000E+04 /
  data (epsData(4,i,39), i=1,3) / 0.1110800000E-01, -0.4236140000E+03, 0.8508950000E+03 /
  data (epsData(5,i,39), i=1,3) / 0.1533570000E-01, -0.1270330000E+03, 0.5495830000E+03 /
  data (epsData(6,i,39), i=1,3) / 0.2024370000E-01, 0.2941140000E+02, 0.3841330000E+03 /
  data (epsData(7,i,39), i=1,3) / 0.2583200000E-01, 0.1238940000E+03, 0.2839720000E+03 /
  data (epsData(8,i,39), i=1,3) / 0.3210060000E-01, 0.1791160000E+03, 0.2407920000E+03 /
  data (epsData(9,i,39), i=1,3) / 0.3904940000E-01, 0.2026170000E+03, 0.2091130000E+03 /
  data (epsData(10,i,39), i=1,3) / 0.4667860000E-01, 0.2123780000E+03, 0.1819900000E+03 /
  data (epsData(11,i,39), i=1,3) / 0.5498800000E-01, 0.2167430000E+03, 0.1591210000E+03 /
  data (epsData(12,i,39), i=1,3) / 0.7364770000E-01, 0.2195380000E+03, 0.1237280000E+03 /
  data (epsData(13,i,39), i=1,3) / 0.1067390000E+00, 0.2197960000E+03, 0.8847260000E+02 /
  data (epsData(14,i,39), i=1,3) / 0.1603860000E+00, 0.2197340000E+03, 0.6041730000E+02 /
  data (epsData(15,i,39), i=1,3) / 0.2427500000E+00, 0.2213970000E+03, 0.4060360000E+02 /
  data (epsData(16,i,39), i=1,3) / 0.3421210000E+00, 0.2273320000E+03, 0.3036080000E+02 /
  data (epsData(17,i,39), i=1,3) / 0.5918850000E+00, 0.2384690000E+03, 0.2438610000E+02 /
  data (epsData(18,i,39), i=1,3) / 0.8748370000E+00, 0.2676120000E+03, 0.2706870000E+02 /
  data (epsData(19,i,39), i=1,3) / 0.1133010000E+01, 0.3211130000E+03, 0.4292350000E+02 /
  data (epsData(20,i,39), i=1,3) / 0.1172610000E+01, 0.3352470000E+03, 0.4454260000E+02 /
  data (epsData(21,i,39), i=1,3) / 0.1212890000E+01, 0.3573740000E+03, 0.4588320000E+02 /
  data (epsData(22,i,39), i=1,3) / 0.1253850000E+01, 0.3816430000E+03, 0.6626600000E+02 /
  data (epsData(23,i,39), i=1,3) / 0.1295500000E+01, 0.3972810000E+03, 0.9394290000E+02 /
  data (epsData(24,i,39), i=1,3) / 0.1424510000E+01, 0.4217600000E+03, 0.1593540000E+03 /
  data (epsData(25,i,39), i=1,3) / 0.1468870000E+01, 0.4377260000E+03, 0.1762820000E+03 /
  data (epsData(26,i,39), i=1,3) / 0.1513920000E+01, 0.4753750000E+03, 0.1911910000E+03 /
  data (epsData(27,i,39), i=1,3) / 0.1559640000E+01, 0.5091520000E+03, 0.2760490000E+03 /
  data (epsData(28,i,39), i=1,3) / 0.1606050000E+01, 0.4964410000E+03, 0.3796370000E+03 /
  data (epsData(29,i,39), i=1,3) / 0.1653130000E+01, 0.4488660000E+03, 0.4535730000E+03 /
  data (epsData(30,i,39), i=1,3) / 0.1749350000E+01, 0.3280140000E+03, 0.5510640000E+03 /
  data (epsData(31,i,39), i=1,3) / 0.1798470000E+01, 0.2479960000E+03, 0.5919070000E+03 /
  data (epsData(32,i,39), i=1,3) / 0.1848280000E+01, 0.1564370000E+03, 0.5973320000E+03 /
  data (epsData(33,i,39), i=1,3) / 0.1898770000E+01, 0.8111130000E+02, 0.5617490000E+03 /
  data (epsData(34,i,39), i=1,3) / 0.1949930000E+01, 0.2973740000E+02, 0.5285800000E+03 /
  data (epsData(35,i,39), i=1,3) / 0.2054310000E+01, -0.4279760000E+02, 0.4687450000E+03 /
  data (epsData(36,i,39), i=1,3) / 0.2161410000E+01, -0.9909220000E+02, 0.4096960000E+03 /
  data (epsData(37,i,39), i=1,3) / 0.2271220000E+01, -0.1321570000E+03, 0.3449130000E+03 /
  data (epsData(38,i,39), i=1,3) / 0.2383760000E+01, -0.1482150000E+03, 0.2888130000E+03 /
  data (epsData(39,i,39), i=1,3) / 0.2557670000E+01, -0.1549390000E+03, 0.2188360000E+03 /
  data (epsData(40,i,39), i=1,3) / 0.2737710000E+01, -0.1483900000E+03, 0.1625140000E+03 /
  data (epsData(41,i,39), i=1,3) / 0.2987270000E+01, -0.1303920000E+03, 0.1085650000E+03 /
  data (epsData(42,i,39), i=1,3) / 0.3181590000E+01, -0.1137820000E+03, 0.7997810000E+02 /
  data (epsData(43,i,39), i=1,3) / 0.3382030000E+01, -0.9715440000E+02, 0.5816860000E+02 /
  data (epsData(44,i,39), i=1,3) / 0.3658810000E+01, -0.7687500000E+02, 0.3811790000E+02 /
  data (epsData(45,i,39), i=1,3) / 0.3946480000E+01, -0.5912450000E+02, 0.2507870000E+02 /
  data (epsData(46,i,39), i=1,3) / 0.4245030000E+01, -0.4405270000E+02, 0.1664120000E+02 /
  data (epsData(47,i,39), i=1,3) / 0.4476080000E+01, -0.3437270000E+02, 0.1236620000E+02 /
  data (epsData(48,i,39), i=1,3) / 0.4713260000E+01, -0.2581380000E+02, 0.9247240000E+01 /
  data (epsData(49,i,39), i=1,3) / 0.4956560000E+01, -0.1818960000E+02, 0.6940950000E+01 /
  data (epsData(50,i,39), i=1,3) / 0.5205980000E+01, -0.1126340000E+02, 0.5248940000E+01 /
  data (epsData(51,i,39), i=1,3) / 0.5461520000E+01, -0.4756050000E+01, 0.4019950000E+01 /
  data (epsData(52,i,39), i=1,3) / 0.5635290000E+01, -0.4525080000E+00, 0.3421370000E+01 /
  data (epsData(53,i,39), i=1,3) / 0.5723190000E+01, 0.1775960000E+01, 0.3162230000E+01 /
  data (epsData(54,i,39), i=1,3) / 0.5901040000E+01, 0.6519270000E+01, 0.2858210000E+01 /
  data (epsData(55,i,39), i=1,3) / 0.5990980000E+01, 0.9238560000E+01, 0.2773550000E+01 /
  data (epsData(56,i,39), i=1,3) / 0.6081600000E+01, 0.1225190000E+02, 0.3054830000E+01 /
  data (epsData(57,i,39), i=1,3) / 0.6172910000E+01, 0.1583240000E+02, 0.3391020000E+01 /
  data (epsData(58,i,39), i=1,3) / 0.6264890000E+01, 0.2134190000E+02, 0.3708730000E+01 /
  data (epsData(59,i,39), i=1,3) / 0.6357560000E+01, 0.2507950000E+02, 0.1208990000E+02 /
  data (epsData(60,i,39), i=1,3) / 0.6450900000E+01, 0.2451670000E+02, 0.1780150000E+02 /
  data (epsData(61,i,39), i=1,3) / 0.6544930000E+01, 0.2187740000E+02, 0.2250430000E+02 /
  data (epsData(62,i,39), i=1,3) / 0.6639630000E+01, 0.1855260000E+02, 0.2382970000E+02 /
  data (epsData(63,i,39), i=1,3) / 0.6735020000E+01, 0.1626720000E+02, 0.2360350000E+02 /
  data (epsData(64,i,39), i=1,3) / 0.6927830000E+01, 0.1400610000E+02, 0.2308670000E+02 /
  data (epsData(65,i,39), i=1,3) / 0.7222150000E+01, 0.1280670000E+02, 0.2204150000E+02 /
  data (epsData(66,i,39), i=1,3) / 0.7624100000E+01, 0.1300110000E+02, 0.2109310000E+02 /
  data (epsData(67,i,39), i=1,3) / 0.8353710000E+01, 0.1471420000E+02, 0.2115570000E+02 /
  data (epsData(68,i,39), i=1,3) / 0.9116650000E+01, 0.1581630000E+02, 0.2300320000E+02 /
  data (epsData(69,i,39), i=1,3) / 0.1002940000E+02, 0.1497080000E+02, 0.2573230000E+02 /
  data (epsData(70,i,39), i=1,3) / 0.1074250000E+02, 0.1244330000E+02, 0.2741280000E+02 /
  data (epsData(71,i,39), i=1,3) / 0.1110830000E+02, 0.1098490000E+02, 0.2635930000E+02 /
  data (epsData(72,i,39), i=1,3) / 0.1148020000E+02, 0.1091700000E+02, 0.2562930000E+02 /
  data (epsData(73,i,39), i=1,3) / 0.1211350000E+02, 0.1083830000E+02, 0.2609500000E+02 /
  data (epsData(74,i,39), i=1,3) / 0.1329650000E+02, 0.8470300000E+01, 0.2712980000E+02 /
  data (epsData(75,i,39), i=1,3) / 0.1481720000E+02, 0.4502620000E+01, 0.2666080000E+02 /
  data (epsData(76,i,39), i=1,3) / 0.1627110000E+02, 0.1430830000E+01, 0.2463610000E+02 /
  data (epsData(77,i,39), i=1,3) / 0.1779300000E+02, -0.5146440000E+00, 0.2184650000E+02 /
  data (epsData(78,i,39), i=1,3) / 0.1938290000E+02, -0.1086450000E+01, 0.1873010000E+02 /
  data (epsData(79,i,39), i=1,3) / 0.2053630000E+02, -0.3281300000E+00, 0.1651650000E+02 /
  data (epsData(80,i,39), i=1,3) / 0.2121040000E+02, 0.8265780000E+00, 0.1528860000E+02 /
  data (epsData(81,i,39), i=1,3) / 0.2172310000E+02, 0.2333380000E+01, 0.1440020000E+02 /
  data (epsData(82,i,39), i=1,3) / 0.2206830000E+02, 0.3915930000E+01, 0.1383380000E+02 /
  data (epsData(83,i,39), i=1,3) / 0.2241620000E+02, 0.6484850000E+01, 0.1332360000E+02 /
  data (epsData(84,i,39), i=1,3) / 0.2259110000E+02, 0.8580870000E+01, 0.1313310000E+02 /
  data (epsData(85,i,39), i=1,3) / 0.2276680000E+02, 0.1192620000E+02, 0.1322310000E+02 /
  data (epsData(86,i,39), i=1,3) / 0.2294310000E+02, 0.1859580000E+02, 0.1375230000E+02 /
  data (epsData(87,i,39), i=1,3) / 0.2310310000E+02, 0.2117280000E+02, 0.2808200000E+02 /
  data (epsData(88,i,39), i=1,3) / 0.2326460000E+02, 0.1431850000E+02, 0.3578150000E+02 /
  data (epsData(89,i,39), i=1,3) / 0.2329590000E+02, 0.1183570000E+02, 0.3724780000E+02 /
  data (epsData(90,i,39), i=1,3) / 0.2332990000E+02, 0.8940780000E+01, 0.3682480000E+02 /
  data (epsData(91,i,39), i=1,3) / 0.2342690000E+02, 0.4583870000E+01, 0.3485870000E+02 /
  data (epsData(92,i,39), i=1,3) / 0.2356570000E+02, 0.1397420000E+01, 0.3210040000E+02 /
  data (epsData(93,i,39), i=1,3) / 0.2378880000E+02, -0.5012280000E+00, 0.2894030000E+02 /
  data (epsData(94,i,39), i=1,3) / 0.2391670000E+02, -0.6723290000E+00, 0.2737800000E+02 /
  data (epsData(95,i,39), i=1,3) / 0.2412890000E+02, 0.9781620000E-01, 0.2583270000E+02 /
  data (epsData(96,i,39), i=1,3) / 0.2424420000E+02, 0.1555950000E+01, 0.2519760000E+02 /
  data (epsData(97,i,39), i=1,3) / 0.2428400000E+02, 0.2728930000E+01, 0.2504080000E+02 /
  data (epsData(98,i,39), i=1,3) / 0.2432450000E+02, 0.4023290000E+01, 0.2626590000E+02 /
  data (epsData(99,i,39), i=1,3) / 0.2436570000E+02, 0.4464510000E+01, 0.2841640000E+02 /
  data (epsData(100,i,39), i=1,3) / 0.2440540000E+02, 0.3592900000E+01, 0.3028320000E+02 /
  data (epsData(101,i,39), i=1,3) / 0.2457990000E+02, -0.3482750000E+00, 0.3306170000E+02 /
  data (epsData(102,i,39), i=1,3) / 0.2462990000E+02, -0.2275110000E+01, 0.3304820000E+02 /
  data (epsData(103,i,39), i=1,3) / 0.2479320000E+02, -0.5243510000E+01, 0.3120670000E+02 /
  data (epsData(104,i,39), i=1,3) / 0.2505850000E+02, -0.7268690000E+01, 0.2885240000E+02 /
  data (epsData(105,i,39), i=1,3) / 0.2554420000E+02, -0.8870720000E+01, 0.2598330000E+02 /
  data (epsData(106,i,39), i=1,3) / 0.2685750000E+02, -0.1135630000E+02, 0.2141190000E+02 /
  data (epsData(107,i,39), i=1,3) / 0.2807900000E+02, -0.1263600000E+02, 0.1760080000E+02 /
  data (epsData(108,i,39), i=1,3) / 0.2971470000E+02, -0.1240470000E+02, 0.1371320000E+02 /
  data (epsData(109,i,39), i=1,3) / 0.3267770000E+02, -0.1144470000E+02, 0.9397740000E+01 /
  data (epsData(110,i,39), i=1,3) / 0.3591580000E+02, -0.1010360000E+02, 0.6197100000E+01 /
  data (epsData(111,i,39), i=1,3) / 0.3954350000E+02, -0.8468000000E+01, 0.3852780000E+01 /
  data (epsData(112,i,39), i=1,3) / 0.4286600000E+02, -0.7064230000E+01, 0.2531020000E+01 /
  data (epsData(113,i,39), i=1,3) / 0.4631570000E+02, -0.5860110000E+01, 0.1713200000E+01 /
  data (epsData(114,i,39), i=1,3) / 0.5018410000E+02, -0.4804080000E+01, 0.1145020000E+01 /
  data (epsData(115,i,39), i=1,3) / 0.5421680000E+02, -0.3947170000E+01, 0.7845440000E+00 /
  data (epsData(116,i,39), i=1,3) / 0.5837470000E+02, -0.3259910000E+01, 0.5562870000E+00 /
  data (epsData(117,i,39), i=1,3) / 0.6322920000E+02, -0.2645270000E+01, 0.3940670000E+00 /
  data (epsData(118,i,39), i=1,3) / 0.6878040000E+02, -0.2119310000E+01, 0.2903570000E+00 /
  data (epsData(119,i,39), i=1,3) / 0.7502810000E+02, -0.1686510000E+01, 0.2277220000E+00 /
  data (epsData(120,i,39), i=1,3) / 0.8220070000E+02, -0.1327180000E+01, 0.1863110000E+00 /
  data (epsData(121,i,39), i=1,3) / 0.9011410000E+02, -0.1038580000E+01, 0.1579850000E+00 /
  data (epsData(122,i,39), i=1,3) / 0.9904210000E+02, -0.8012030000E+00, 0.1374260000E+00 /
  data (epsData(123,i,39), i=1,3) / 0.1093530000E+03, -0.6001760000E+00, 0.1189190000E+00 /
  data (epsData(124,i,39), i=1,3) / 0.1208690000E+03, -0.4305840000E+00, 0.1019220000E+00 /
  data (epsData(125,i,39), i=1,3) / 0.1336650000E+03, -0.2815860000E+00, 0.8765730000E-01 /
  data (epsData(126,i,39), i=1,3) / 0.1500890000E+03, -0.1112560000E+00, 0.7317060000E-01 /
  data (epsData(127,i,39), i=1,3) / 0.1531530000E+03, -0.7240660000E-01, 0.7084610000E-01 /
  data (epsData(128,i,39), i=1,3) / 0.1543140000E+03, -0.5350420000E-01, 0.7016170000E-01 /
  data (epsData(129,i,39), i=1,3) / 0.1550910000E+03, -0.3567130000E-01, 0.7007740000E-01 /
  data (epsData(130,i,39), i=1,3) / 0.1554800000E+03, -0.2158420000E-01, 0.7186650000E-01 /
  data (epsData(131,i,39), i=1,3) / 0.1558280000E+03, -0.8411740000E-02, 0.8659510000E-01 /
  data (epsData(132,i,39), i=1,3) / 0.1559650000E+03, -0.8348950000E-02, 0.9656270000E-01 /
  data (epsData(133,i,39), i=1,3) / 0.1560480000E+03, -0.1211970000E-01, 0.9982920000E-01 /
  data (epsData(134,i,39), i=1,3) / 0.1564000000E+03, -0.1770140000E-01, 0.1037660000E+00 /
  data (epsData(135,i,39), i=1,3) / 0.1568670000E+03, -0.1862830000E-01, 0.1038880000E+00 /
  data (epsData(136,i,39), i=1,3) / 0.1572820000E+03, -0.1391560000E-01, 0.1027660000E+00 /
  data (epsData(137,i,39), i=1,3) / 0.1575630000E+03, -0.6449840000E-02, 0.1045940000E+00 /
  data (epsData(138,i,39), i=1,3) / 0.1576130000E+03, -0.4078450000E-02, 0.1056590000E+00 /
  data (epsData(139,i,39), i=1,3) / 0.1577340000E+03, -0.1738130000E-02, 0.1127590000E+00 /
  data (epsData(140,i,39), i=1,3) / 0.1578960000E+03, -0.4208290000E-02, 0.1205890000E+00 /
  data (epsData(141,i,39), i=1,3) / 0.1582170000E+03, -0.1055490000E-01, 0.1229090000E+00 /
  data (epsData(142,i,39), i=1,3) / 0.1589250000E+03, -0.1375170000E-01, 0.1219700000E+00 /
  data (epsData(143,i,39), i=1,3) / 0.1602750000E+03, -0.7963510000E-02, 0.1197040000E+00 /
  data (epsData(144,i,39), i=1,3) / 0.1624360000E+03, 0.8082090000E-02, 0.1198140000E+00 /
  data (epsData(145,i,39), i=1,3) / 0.1643670000E+03, 0.2812610000E-01, 0.1211230000E+00 /
  data (epsData(146,i,39), i=1,3) / 0.1670730000E+03, 0.6025780000E-01, 0.1329580000E+00 /
  data (epsData(147,i,39), i=1,3) / 0.1701090000E+03, 0.9114290000E-01, 0.1569610000E+00 /
  data (epsData(148,i,39), i=1,3) / 0.1764840000E+03, 0.1319870000E+00, 0.2155530000E+00 /
  data (epsData(149,i,39), i=1,3) / 0.1819480000E+03, 0.1530000000E+00, 0.2728860000E+00 /
  data (epsData(150,i,39), i=1,3) / 0.1873010000E+03, 0.1488490000E+00, 0.3330170000E+00 /
  data (epsData(151,i,39), i=1,3) / 0.1929670000E+03, 0.1212450000E+00, 0.3793710000E+00 /
  data (epsData(152,i,39), i=1,3) / 0.2003250000E+03, 0.8221120000E-01, 0.4074480000E+00 /
  data (epsData(153,i,39), i=1,3) / 0.2168470000E+03, 0.1245400000E-01, 0.4374980000E+00 /
  data (epsData(154,i,39), i=1,3) / 0.2293620000E+03, -0.4112760000E-01, 0.4329560000E+00 /
  data (epsData(155,i,39), i=1,3) / 0.2441020000E+03, -0.8567070000E-01, 0.4039460000E+00 /
  data (epsData(156,i,39), i=1,3) / 0.2737090000E+03, -0.1273970000E+00, 0.3405820000E+00 /
  data (epsData(157,i,39), i=1,3) / 0.2927050000E+03, -0.1307230000E+00, 0.3003190000E+00 /
  data (epsData(158,i,39), i=1,3) / 0.2978970000E+03, -0.1204980000E+00, 0.2963870000E+00 /
  data (epsData(159,i,39), i=1,3) / 0.3004670000E+03, -0.1246360000E+00, 0.3037080000E+00 /
  data (epsData(160,i,39), i=1,3) / 0.3098210000E+03, -0.1343060000E+00, 0.2935810000E+00 /
  data (epsData(161,i,39), i=1,3) / 0.3182530000E+03, -0.1478160000E+00, 0.2836570000E+00 /
  data (epsData(162,i,39), i=1,3) / 0.3449970000E+03, -0.1597610000E+00, 0.2380820000E+00 /
  data (epsData(163,i,39), i=1,3) / 0.3773530000E+03, -0.1566460000E+00, 0.1920110000E+00 /
  data (epsData(164,i,39), i=1,3) / 0.3944000000E+03, -0.1508130000E+00, 0.1772340000E+00 /
  data (epsData(165,i,39), i=1,3) / 0.4398320000E+03, -0.1442420000E+00, 0.1347250000E+00 /
  data (epsData(166,i,39), i=1,3) / 0.4994330000E+03, -0.1262050000E+00, 0.9476070000E-01 /
  data (epsData(167,i,39), i=1,3) / 0.5610620000E+03, -0.1077740000E+00, 0.6816150000E-01 /
  data (epsData(168,i,39), i=1,3) / 0.6371810000E+03, -0.8869080000E-01, 0.4645340000E-01 /
  data (epsData(169,i,39), i=1,3) / 0.7220230000E+03, -0.7181130000E-01, 0.3149280000E-01 /
  data (epsData(170,i,39), i=1,3) / 0.8148860000E+03, -0.5776170000E-01, 0.2143880000E-01 /
  data (epsData(171,i,39), i=1,3) / 0.9150190000E+03, -0.4643470000E-01, 0.1472860000E-01 /
  data (epsData(172,i,39), i=1,3) / 0.1025780000E+04, -0.3715680000E-01, 0.1011970000E-01 /
  data (epsData(173,i,39), i=1,3) / 0.1147180000E+04, -0.2967510000E-01, 0.6977360000E-02 /
  data (epsData(174,i,39), i=1,3) / 0.1281400000E+04, -0.2360230000E-01, 0.4817480000E-02 /
  data (epsData(175,i,39), i=1,3) / 0.1428970000E+04, -0.1870320000E-01, 0.3337700000E-02 /
  data (epsData(176,i,39), i=1,3) / 0.1589130000E+04, -0.1475570000E-01, 0.2323940000E-02 /
  data (epsData(177,i,39), i=1,3) / 0.1766260000E+04, -0.1139940000E-01, 0.1601910000E-02 /
  data (epsData(178,i,39), i=1,3) / 0.1993760000E+04, -0.7613890000E-02, 0.1055520000E-02 /
  data (epsData(179,i,39), i=1,3) / 0.2042840000E+04, -0.6414680000E-02, 0.9695020000E-03 /
  data (epsData(180,i,39), i=1,3) / 0.2059370000E+04, -0.5719260000E-02, 0.9427660000E-03 /
  data (epsData(181,i,39), i=1,3) / 0.2068420000E+04, -0.5135860000E-02, 0.9756260000E-03 /
  data (epsData(182,i,39), i=1,3) / 0.2072950000E+04, -0.4645390000E-02, 0.1002740000E-02 /
  data (epsData(183,i,39), i=1,3) / 0.2075980000E+04, -0.4094150000E-02, 0.1065320000E-02 /
  data (epsData(184,i,39), i=1,3) / 0.2077490000E+04, -0.3631420000E-02, 0.1158610000E-02 /
  data (epsData(185,i,39), i=1,3) / 0.2079010000E+04, -0.2922370000E-02, 0.1512510000E-02 /
  data (epsData(186,i,39), i=1,3) / 0.2080070000E+04, -0.2620020000E-02, 0.2661360000E-02 /
  data (epsData(187,i,39), i=1,3) / 0.2080390000E+04, -0.2829850000E-02, 0.3023170000E-02 /
  data (epsData(188,i,39), i=1,3) / 0.2080820000E+04, -0.3194300000E-02, 0.3234050000E-02 /
  data (epsData(189,i,39), i=1,3) / 0.2081710000E+04, -0.3602390000E-02, 0.3287840000E-02 /
  data (epsData(190,i,39), i=1,3) / 0.2084660000E+04, -0.4216370000E-02, 0.3493950000E-02 /
  data (epsData(191,i,39), i=1,3) / 0.2090900000E+04, -0.4928390000E-02, 0.3830690000E-02 /
  data (epsData(192,i,39), i=1,3) / 0.2096610000E+04, -0.5486420000E-02, 0.3716670000E-02 /
  data (epsData(193,i,39), i=1,3) / 0.2107050000E+04, -0.5773930000E-02, 0.3429200000E-02 /
  data (epsData(194,i,39), i=1,3) / 0.2140000000E+04, -0.5735600000E-02, 0.3249990000E-02 /
  data (epsData(195,i,39), i=1,3) / 0.2150730000E+04, -0.5355620000E-02, 0.3213350000E-02 /
  data (epsData(196,i,39), i=1,3) / 0.2154180000E+04, -0.4958440000E-02, 0.3308570000E-02 /
  data (epsData(197,i,39), i=1,3) / 0.2155460000E+04, -0.4679160000E-02, 0.3565250000E-02 /
  data (epsData(198,i,39), i=1,3) / 0.2156420000E+04, -0.4723620000E-02, 0.4002750000E-02 /
  data (epsData(199,i,39), i=1,3) / 0.2157860000E+04, -0.5038800000E-02, 0.4164030000E-02 /
  data (epsData(200,i,39), i=1,3) / 0.2164490000E+04, -0.5544910000E-02, 0.4379670000E-02 /
  data (epsData(201,i,39), i=1,3) / 0.2173530000E+04, -0.6044320000E-02, 0.4335620000E-02 /
  data (epsData(202,i,39), i=1,3) / 0.2195780000E+04, -0.6358900000E-02, 0.4101110000E-02 /
  data (epsData(203,i,39), i=1,3) / 0.2263820000E+04, -0.6657190000E-02, 0.3693580000E-02 /
  data (epsData(204,i,39), i=1,3) / 0.2352470000E+04, -0.6416200000E-02, 0.3230090000E-02 /
  data (epsData(205,i,39), i=1,3) / 0.2370980000E+04, -0.6164550000E-02, 0.3299270000E-02 /
  data (epsData(206,i,39), i=1,3) / 0.2378570000E+04, -0.6270470000E-02, 0.3490480000E-02 /
  data (epsData(207,i,39), i=1,3) / 0.2440960000E+04, -0.6465830000E-02, 0.3243430000E-02 /
  data (epsData(208,i,39), i=1,3) / 0.2650980000E+04, -0.6045130000E-02, 0.2437650000E-02 /
  data (epsData(209,i,39), i=1,3) / 0.3031160000E+04, -0.4961990000E-02, 0.1525700000E-02 /
  data (epsData(210,i,39), i=1,3) / 0.3400490000E+04, -0.4065190000E-02, 0.1018090000E-02 /
  data (epsData(211,i,39), i=1,3) / 0.3771980000E+04, -0.3357340000E-02, 0.7076400000E-03 /
  data (epsData(212,i,39), i=1,3) / 0.4208590000E+04, -0.2725300000E-02, 0.4782250000E-03 /
  data (epsData(213,i,39), i=1,3) / 0.4683080000E+04, -0.2211780000E-02, 0.3243750000E-03 /
  data (epsData(214,i,39), i=1,3) / 0.5208120000E+04, -0.1790830000E-02, 0.2197690000E-03 /
  data (epsData(215,i,39), i=1,3) / 0.5793350000E+04, -0.1446050000E-02, 0.1483500000E-03 /
  data (epsData(216,i,39), i=1,3) / 0.6430990000E+04, -0.1170810000E-02, 0.1011370000E-03 /
  data (epsData(217,i,39), i=1,3) / 0.7148530000E+04, -0.9447300000E-03, 0.6808340000E-04 /
  data (epsData(218,i,39), i=1,3) / 0.7951970000E+04, -0.7605230000E-03, 0.4552470000E-04 /
  data (epsData(219,i,39), i=1,3) / 0.8862330000E+04, -0.6094290000E-03, 0.3025840000E-04 /
  data (epsData(220,i,39), i=1,3) / 0.9879750000E+04, -0.4879170000E-03, 0.1999630000E-04 /
  data (epsData(221,i,39), i=1,3) / 0.1102260000E+05, -0.3897280000E-03, 0.1319370000E-04 /
  data (epsData(222,i,39), i=1,3) / 0.1230510000E+05, -0.3105610000E-03, 0.8657090000E-05 /
  data (epsData(223,i,39), i=1,3) / 0.1347330000E+05, -0.2571310000E-03, 0.6115250000E-05 /
  data (epsData(224,i,39), i=1,3) / 0.1347400000E+05, -0.2453470000E-03, 0.6114010000E-05 /
  data (epsData(225,i,39), i=1,3) / 0.1347400000E+05, -0.2571010000E-03, 0.6113920000E-05 /
  data (epsData(226,i,39), i=1,3) / 0.1509800000E+05, -0.2016960000E-03, 0.3945150000E-05 /
  data (epsData(227,i,39), i=1,3) / 0.1685990000E+05, -0.1514410000E-03, 0.2573910000E-05 /
  data (epsData(228,i,39), i=1,3) / 0.1700090000E+05, -0.1416250000E-03, 0.2492750000E-05 /
  data (epsData(229,i,39), i=1,3) / 0.1702720000E+05, -0.1350700000E-03, 0.3420650000E-05 /
  data (epsData(230,i,39), i=1,3) / 0.1703380000E+05, -0.1306990000E-03, 0.4910060000E-05 /
  data (epsData(231,i,39), i=1,3) / 0.1703600000E+05, -0.1285360000E-03, 0.6827280000E-05 /
  data (epsData(232,i,39), i=1,3) / 0.1703800000E+05, -0.1040140000E-03, 0.1089120000E-04 /
  data (epsData(233,i,39), i=1,3) / 0.1703800000E+05, -0.1273090000E-03, 0.1100840000E-04 /
  data (epsData(234,i,39), i=1,3) / 0.1704160000E+05, -0.1313850000E-03, 0.1592680000E-04 /
  data (epsData(235,i,39), i=1,3) / 0.1704850000E+05, -0.1359310000E-03, 0.1561330000E-04 /
  data (epsData(236,i,39), i=1,3) / 0.1709000000E+05, -0.1419540000E-03, 0.1686270000E-04 /
  data (epsData(237,i,39), i=1,3) / 0.1731470000E+05, -0.1462440000E-03, 0.1634390000E-04 /
  data (epsData(238,i,39), i=1,3) / 0.1851820000E+05, -0.1349200000E-03, 0.1281100000E-04 /
  data (epsData(239,i,39), i=1,3) / 0.2138040000E+05, -0.1046310000E-03, 0.7668590000E-05 /
  data (epsData(240,i,39), i=1,3) / 0.2403910000E+05, -0.8362330000E-04, 0.4970820000E-05 /
  data (epsData(241,i,39), i=1,3) / 0.2694070000E+05, -0.6691790000E-04, 0.3245980000E-05 /
  data (epsData(242,i,39), i=1,3) / 0.3015760000E+05, -0.5353330000E-04, 0.2118230000E-05 /
  data (epsData(243,i,39), i=1,3) / 0.3375100000E+05, -0.4278010000E-04, 0.1377640000E-05 /
  data (epsData(244,i,39), i=1,3) / 0.3776850000E+05, -0.3416270000E-04, 0.8922350000E-06 /
  data (epsData(245,i,39), i=1,3) / 0.4227100000E+05, -0.2725760000E-04, 0.5752400000E-06 /
  data (epsData(246,i,39), i=1,3) / 0.4731190000E+05, -0.2174000000E-04, 0.3692740000E-06 /
  data (epsData(247,i,39), i=1,3) / 0.5297220000E+05, -0.1732460000E-04, 0.2358660000E-06 /
  data (epsData(248,i,39), i=1,3) / 0.5932190000E+05, -0.1379980000E-04, 0.1499510000E-06 /
  data (epsData(249,i,39), i=1,3) / 0.6645810000E+05, -0.1098420000E-04, 0.9494840000E-07 /
  data (epsData(250,i,39), i=1,3) / 0.7445830000E+05, -0.8743010000E-05, 0.5980930000E-07 /
  data (epsData(251,i,39), i=1,3) / 0.8344190000E+05, -0.6956900000E-05, 0.3748140000E-07 /
  data (epsData(252,i,39), i=1,3) / 0.9349550000E+05, -0.5540410000E-05, 0.2337640000E-07 /
  data (epsData(253,i,39), i=1,3) / 0.1000000000E+06, -0.4892550000E-05, 0.1765670000E-07 /
  data (epsData(1,i,40), i=1,3) / 0.2506580000E-02, -0.2409270000E+05, 0.6591120000E+05 /
  data (epsData(2,i,40), i=1,3) / 0.4693440000E-02, -0.2045320000E+05, 0.2268200000E+05 /
  data (epsData(3,i,40), i=1,3) / 0.7560590000E-02, -0.9446170000E+04, 0.1070260000E+05 /
  data (epsData(4,i,40), i=1,3) / 0.1110800000E-01, -0.4631130000E+04, 0.6082670000E+04 /
  data (epsData(5,i,40), i=1,3) / 0.1533570000E-01, -0.2364000000E+04, 0.3894300000E+04 /
  data (epsData(6,i,40), i=1,3) / 0.2024370000E-01, -0.1190210000E+04, 0.2703300000E+04 /
  data (epsData(7,i,40), i=1,3) / 0.2583200000E-01, -0.5327450000E+03, 0.1987740000E+04 /
  data (epsData(8,i,40), i=1,3) / 0.3210060000E-01, -0.1404680000E+03, 0.1525140000E+04 /
  data (epsData(9,i,40), i=1,3) / 0.3904940000E-01, 0.1059260000E+03, 0.1208870000E+04 /
  data (epsData(10,i,40), i=1,3) / 0.4667860000E-01, 0.2674300000E+03, 0.9829480000E+03 /
  data (epsData(11,i,40), i=1,3) / 0.5498800000E-01, 0.3771950000E+03, 0.8158030000E+03 /
  data (epsData(12,i,40), i=1,3) / 0.6397770000E-01, 0.4541920000E+03, 0.6885470000E+03 /
  data (epsData(13,i,40), i=1,3) / 0.7364770000E-01, 0.5097640000E+03, 0.5893330000E+03 /
  data (epsData(14,i,40), i=1,3) / 0.8399800000E-01, 0.5509550000E+03, 0.5104190000E+03 /
  data (epsData(15,i,40), i=1,3) / 0.9502860000E-01, 0.5822890000E+03, 0.4465750000E+03 /
  data (epsData(16,i,40), i=1,3) / 0.1191310000E+00, 0.6264180000E+03, 0.3505770000E+03 /
  data (epsData(17,i,40), i=1,3) / 0.1603860000E+00, 0.6689580000E+03, 0.2561450000E+03 /
  data (epsData(18,i,40), i=1,3) / 0.2249170000E+00, 0.7138730000E+03, 0.1801520000E+03 /
  data (epsData(19,i,40), i=1,3) / 0.2804580000E+00, 0.7650700000E+03, 0.1441070000E+03 /
  data (epsData(20,i,40), i=1,3) / 0.3208860000E+00, 0.7927380000E+03, 0.1671690000E+03 /
  data (epsData(21,i,40), i=1,3) / 0.4338630000E+00, 0.8194040000E+03, 0.1787070000E+03 /
  data (epsData(22,i,40), i=1,3) / 0.5098130000E+00, 0.8675950000E+03, 0.1719160000E+03 /
  data (epsData(23,i,40), i=1,3) / 0.5638470000E+00, 0.9503410000E+03, 0.1652580000E+03 /
  data (epsData(24,i,40), i=1,3) / 0.5918850000E+00, 0.1014310000E+04, 0.2027260000E+03 /
  data (epsData(25,i,40), i=1,3) / 0.6206030000E+00, 0.1054190000E+04, 0.2856080000E+03 /
  data (epsData(26,i,40), i=1,3) / 0.6800800000E+00, 0.1058370000E+04, 0.4123890000E+03 /
  data (epsData(27,i,40), i=1,3) / 0.7743970000E+00, 0.1039810000E+04, 0.5315350000E+03 /
  data (epsData(28,i,40), i=1,3) / 0.8406760000E+00, 0.1083310000E+04, 0.5783240000E+03 /
  data (epsData(29,i,40), i=1,3) / 0.8748370000E+00, 0.1182680000E+04, 0.5946270000E+03 /
  data (epsData(30,i,40), i=1,3) / 0.9096770000E+00, 0.1262690000E+04, 0.8219210000E+03 /
  data (epsData(31,i,40), i=1,3) / 0.9451980000E+00, 0.1223270000E+04, 0.1053430000E+04 /
  data (epsData(32,i,40), i=1,3) / 0.9813990000E+00, 0.1093420000E+04, 0.1249940000E+04 /
  data (epsData(33,i,40), i=1,3) / 0.1055840000E+01, 0.7909250000E+03, 0.1415630000E+04 /
  data (epsData(34,i,40), i=1,3) / 0.1133010000E+01, 0.4764690000E+03, 0.1531600000E+04 /
  data (epsData(35,i,40), i=1,3) / 0.1172610000E+01, 0.2707850000E+03, 0.1574700000E+04 /
  data (epsData(36,i,40), i=1,3) / 0.1212890000E+01, 0.7080950000E+02, 0.1484280000E+04 /
  data (epsData(37,i,40), i=1,3) / 0.1253850000E+01, -0.6570260000E+02, 0.1373950000E+04 /
  data (epsData(38,i,40), i=1,3) / 0.1295500000E+01, -0.1600430000E+03, 0.1272820000E+04 /
  data (epsData(39,i,40), i=1,3) / 0.1380830000E+01, -0.2861760000E+03, 0.1094730000E+04 /
  data (epsData(40,i,40), i=1,3) / 0.1513920000E+01, -0.4061200000E+03, 0.8642550000E+03 /
  data (epsData(41,i,40), i=1,3) / 0.1606050000E+01, -0.4394090000E+03, 0.7156830000E+03 /
  data (epsData(42,i,40), i=1,3) / 0.1749350000E+01, -0.4411070000E+03, 0.5347210000E+03 /
  data (epsData(43,i,40), i=1,3) / 0.1898770000E+01, -0.4145670000E+03, 0.3995760000E+03 /
  data (epsData(44,i,40), i=1,3) / 0.2054310000E+01, -0.3771450000E+03, 0.2945270000E+03 /
  data (epsData(45,i,40), i=1,3) / 0.2215970000E+01, -0.3342440000E+03, 0.2191890000E+03 /
  data (epsData(46,i,40), i=1,3) / 0.2441050000E+01, -0.2789990000E+03, 0.1460000000E+03 /
  data (epsData(47,i,40), i=1,3) / 0.2677010000E+01, -0.2305420000E+03, 0.9726530000E+02 /
  data (epsData(48,i,40), i=1,3) / 0.2923860000E+01, -0.1889690000E+03, 0.6592950000E+02 /
  data (epsData(49,i,40), i=1,3) / 0.3116140000E+01, -0.1621370000E+03, 0.4937470000E+02 /
  data (epsData(50,i,40), i=1,3) / 0.3382030000E+01, -0.1323240000E+03, 0.3358960000E+02 /
  data (epsData(51,i,40), i=1,3) / 0.3658810000E+01, -0.1079760000E+03, 0.2297710000E+02 /
  data (epsData(52,i,40), i=1,3) / 0.3946480000E+01, -0.8806010000E+02, 0.1577840000E+02 /
  data (epsData(53,i,40), i=1,3) / 0.4245030000E+01, -0.7166810000E+02, 0.1086150000E+02 /
  data (epsData(54,i,40), i=1,3) / 0.4554460000E+01, -0.5807190000E+02, 0.7501550000E+01 /
  data (epsData(55,i,40), i=1,3) / 0.4874780000E+01, -0.4669350000E+02, 0.5222300000E+01 /
  data (epsData(56,i,40), i=1,3) / 0.5205980000E+01, -0.3701840000E+02, 0.3761390000E+01 /
  data (epsData(57,i,40), i=1,3) / 0.5461520000E+01, -0.3062050000E+02, 0.3239970000E+01 /
  data (epsData(58,i,40), i=1,3) / 0.5811770000E+01, -0.2384670000E+02, 0.3088260000E+01 /
  data (epsData(59,i,40), i=1,3) / 0.6264890000E+01, -0.1684230000E+02, 0.2508810000E+01 /
  data (epsData(60,i,40), i=1,3) / 0.6735020000E+01, -0.1044590000E+02, 0.1929120000E+01 /
  data (epsData(61,i,40), i=1,3) / 0.7321620000E+01, -0.2946070000E+01, 0.1560460000E+01 /
  data (epsData(62,i,40), i=1,3) / 0.7421760000E+01, -0.1597540000E+01, 0.1532450000E+01 /
  data (epsData(63,i,40), i=1,3) / 0.7522590000E+01, -0.1633020000E+00, 0.1519440000E+01 /
  data (epsData(64,i,40), i=1,3) / 0.7624100000E+01, 0.1427240000E+01, 0.1509240000E+01 /
  data (epsData(65,i,40), i=1,3) / 0.7726290000E+01, 0.3241820000E+01, 0.1612620000E+01 /
  data (epsData(66,i,40), i=1,3) / 0.7829160000E+01, 0.5433940000E+01, 0.1818340000E+01 /
  data (epsData(67,i,40), i=1,3) / 0.7932710000E+01, 0.8715820000E+01, 0.2015550000E+01 /
  data (epsData(68,i,40), i=1,3) / 0.8036940000E+01, 0.1150650000E+02, 0.6227670000E+01 /
  data (epsData(69,i,40), i=1,3) / 0.8141850000E+01, 0.1166890000E+02, 0.1084930000E+02 /
  data (epsData(70,i,40), i=1,3) / 0.8247440000E+01, 0.9724910000E+01, 0.1419650000E+02 /
  data (epsData(71,i,40), i=1,3) / 0.8353710000E+01, 0.7382450000E+01, 0.1480280000E+02 /
  data (epsData(72,i,40), i=1,3) / 0.8460660000E+01, 0.5912550000E+01, 0.1453230000E+02 /
  data (epsData(73,i,40), i=1,3) / 0.8568290000E+01, 0.5089140000E+01, 0.1427020000E+02 /
  data (epsData(74,i,40), i=1,3) / 0.8785590000E+01, 0.4327350000E+01, 0.1361780000E+02 /
  data (epsData(75,i,40), i=1,3) / 0.9116650000E+01, 0.4348730000E+01, 0.1281030000E+02 /
  data (epsData(76,i,40), i=1,3) / 0.9567580000E+01, 0.5322930000E+01, 0.1228500000E+02 /
  data (epsData(77,i,40), i=1,3) / 0.1026440000E+02, 0.7169950000E+01, 0.1264460000E+02 /
  data (epsData(78,i,40), i=1,3) / 0.1098570000E+02, 0.8471980000E+01, 0.1387000000E+02 /
  data (epsData(79,i,40), i=1,3) / 0.1198550000E+02, 0.9034990000E+01, 0.1588660000E+02 /
  data (epsData(80,i,40), i=1,3) / 0.1250170000E+02, 0.8360150000E+01, 0.1702640000E+02 /
  data (epsData(81,i,40), i=1,3) / 0.1289610000E+02, 0.7942590000E+01, 0.1671870000E+02 /
  data (epsData(82,i,40), i=1,3) / 0.1356690000E+02, 0.8338940000E+01, 0.1713190000E+02 /
  data (epsData(83,i,40), i=1,3) / 0.1467550000E+02, 0.7684020000E+01, 0.1876010000E+02 /
  data (epsData(84,i,40), i=1,3) / 0.1597480000E+02, 0.5831970000E+01, 0.2003360000E+02 /
  data (epsData(85,i,40), i=1,3) / 0.1794890000E+02, 0.2353760000E+01, 0.2022880000E+02 /
  data (epsData(86,i,40), i=1,3) / 0.1970900000E+02, -0.4237120000E+00, 0.1907570000E+02 /
  data (epsData(87,i,40), i=1,3) / 0.2138060000E+02, -0.2354950000E+01, 0.1723920000E+02 /
  data (epsData(88,i,40), i=1,3) / 0.2329790000E+02, -0.3513420000E+01, 0.1474150000E+02 /
  data (epsData(89,i,40), i=1,3) / 0.2492780000E+02, -0.3394510000E+01, 0.1251860000E+02 /
  data (epsData(90,i,40), i=1,3) / 0.2585710000E+02, -0.2546430000E+01, 0.1130250000E+02 /
  data (epsData(91,i,40), i=1,3) / 0.2642280000E+02, -0.1363540000E+01, 0.1062320000E+02 /
  data (epsData(92,i,40), i=1,3) / 0.2661280000E+02, -0.6644640000E+00, 0.1042330000E+02 /
  data (epsData(93,i,40), i=1,3) / 0.2680340000E+02, 0.4103420000E+00, 0.1032350000E+02 /
  data (epsData(94,i,40), i=1,3) / 0.2699470000E+02, 0.2717530000E+01, 0.1030670000E+02 /
  data (epsData(95,i,40), i=1,3) / 0.2710310000E+02, 0.3478430000E+01, 0.1329720000E+02 /
  data (epsData(96,i,40), i=1,3) / 0.2726460000E+02, 0.2190860000E+01, 0.1554960000E+02 /
  data (epsData(97,i,40), i=1,3) / 0.2731260000E+02, 0.1306650000E+01, 0.1601100000E+02 /
  data (epsData(98,i,40), i=1,3) / 0.2742690000E+02, 0.7557720000E-01, 0.1579090000E+02 /
  data (epsData(99,i,40), i=1,3) / 0.2770000000E+02, -0.1135290000E+01, 0.1529180000E+02 /
  data (epsData(100,i,40), i=1,3) / 0.2805550000E+02, -0.1607410000E+01, 0.1475460000E+02 /
  data (epsData(101,i,40), i=1,3) / 0.2832450000E+02, -0.1309090000E+01, 0.1453160000E+02 /
  data (epsData(102,i,40), i=1,3) / 0.2840750000E+02, -0.7991230000E+00, 0.1457440000E+02 /
  data (epsData(103,i,40), i=1,3) / 0.2849320000E+02, -0.6175350000E+00, 0.1560030000E+02 /
  data (epsData(104,i,40), i=1,3) / 0.2869590000E+02, -0.1784400000E+01, 0.1676300000E+02 /
  data (epsData(105,i,40), i=1,3) / 0.2891670000E+02, -0.3050480000E+01, 0.1651840000E+02 /
  data (epsData(106,i,40), i=1,3) / 0.2980750000E+02, -0.5221470000E+01, 0.1542990000E+02 /
  data (epsData(107,i,40), i=1,3) / 0.3163470000E+02, -0.7503950000E+01, 0.1293120000E+02 /
  data (epsData(108,i,40), i=1,3) / 0.3269090000E+02, -0.7844270000E+01, 0.1139230000E+02 /
  data (epsData(109,i,40), i=1,3) / 0.3451680000E+02, -0.8084980000E+01, 0.9411330000E+01 /
  data (epsData(110,i,40), i=1,3) / 0.3764850000E+02, -0.7964920000E+01, 0.6774790000E+01 /
  data (epsData(111,i,40), i=1,3) / 0.4090870000E+02, -0.7337210000E+01, 0.4649460000E+01 /
  data (epsData(112,i,40), i=1,3) / 0.4451690000E+02, -0.6338930000E+01, 0.3013900000E+01 /
  data (epsData(113,i,40), i=1,3) / 0.4825580000E+02, -0.5297770000E+01, 0.1960390000E+01 /
  data (epsData(114,i,40), i=1,3) / 0.5208170000E+02, -0.4401640000E+01, 0.1324580000E+01 /
  data (epsData(115,i,40), i=1,3) / 0.5643720000E+02, -0.3609410000E+01, 0.8869980000E+00 /
  data (epsData(116,i,40), i=1,3) / 0.6078010000E+02, -0.2983600000E+01, 0.6131010000E+00 /
  data (epsData(117,i,40), i=1,3) / 0.6531590000E+02, -0.2465730000E+01, 0.4350190000E+00 /
  data (epsData(118,i,40), i=1,3) / 0.7068500000E+02, -0.1994830000E+01, 0.3141650000E+00 /
  data (epsData(119,i,40), i=1,3) / 0.7688750000E+02, -0.1592700000E+01, 0.2394500000E+00 /
  data (epsData(120,i,40), i=1,3) / 0.8392340000E+02, -0.1261910000E+01, 0.1939510000E+00 /
  data (epsData(121,i,40), i=1,3) / 0.9179250000E+02, -0.9957220000E+00, 0.1639480000E+00 /
  data (epsData(122,i,40), i=1,3) / 0.1007560000E+03, -0.7752670000E+00, 0.1410480000E+00 /
  data (epsData(123,i,40), i=1,3) / 0.1108870000E+03, -0.5931100000E+00, 0.1232640000E+00 /
  data (epsData(124,i,40), i=1,3) / 0.1222620000E+03, -0.4427320000E+00, 0.1084550000E+00 /
  data (epsData(125,i,40), i=1,3) / 0.1349570000E+03, -0.3163510000E+00, 0.9520670000E-01 /
  data (epsData(126,i,40), i=1,3) / 0.1501540000E+03, -0.1982330000E+00, 0.8141110000E-01 /
  data (epsData(127,i,40), i=1,3) / 0.1706160000E+03, -0.5623330000E-01, 0.6611900000E-01 /
  data (epsData(128,i,40), i=1,3) / 0.1746910000E+03, -0.2276680000E-01, 0.6352230000E-01 /
  data (epsData(129,i,40), i=1,3) / 0.1763400000E+03, -0.6223400000E-02, 0.6258170000E-01 /
  data (epsData(130,i,40), i=1,3) / 0.1771690000E+03, 0.3901320000E-02, 0.6218760000E-01 /
  data (epsData(131,i,40), i=1,3) / 0.1775840000E+03, 0.1001440000E-01, 0.6205530000E-01 /
  data (epsData(132,i,40), i=1,3) / 0.1780000000E+03, 0.1772030000E-01, 0.6205230000E-01 /
  data (epsData(133,i,40), i=1,3) / 0.1784170000E+03, 0.2916400000E-01, 0.6275850000E-01 /
  data (epsData(134,i,40), i=1,3) / 0.1788340000E+03, 0.4092930000E-01, 0.7753660000E-01 /
  data (epsData(135,i,40), i=1,3) / 0.1789800000E+03, 0.3947880000E-01, 0.8672700000E-01 /
  data (epsData(136,i,40), i=1,3) / 0.1790670000E+03, 0.3548620000E-01, 0.8845010000E-01 /
  data (epsData(137,i,40), i=1,3) / 0.1794000000E+03, 0.2994660000E-01, 0.8983370000E-01 /
  data (epsData(138,i,40), i=1,3) / 0.1799840000E+03, 0.2909870000E-01, 0.8789550000E-01 /
  data (epsData(139,i,40), i=1,3) / 0.1805630000E+03, 0.3488590000E-01, 0.8633620000E-01 /
  data (epsData(140,i,40), i=1,3) / 0.1809230000E+03, 0.4281630000E-01, 0.8821840000E-01 /
  data (epsData(141,i,40), i=1,3) / 0.1810320000E+03, 0.4701840000E-01, 0.9087840000E-01 /
  data (epsData(142,i,40), i=1,3) / 0.1811670000E+03, 0.4793710000E-01, 0.9802600000E-01 /
  data (epsData(143,i,40), i=1,3) / 0.1813130000E+03, 0.4442000000E-01, 0.1039950000E+00 /
  data (epsData(144,i,40), i=1,3) / 0.1817590000E+03, 0.3725820000E-01, 0.1049380000E+00 /
  data (epsData(145,i,40), i=1,3) / 0.1826730000E+03, 0.3555410000E-01, 0.1020150000E+00 /
  data (epsData(146,i,40), i=1,3) / 0.1853670000E+03, 0.4985210000E-01, 0.9853490000E-01 /
  data (epsData(147,i,40), i=1,3) / 0.1877670000E+03, 0.7094600000E-01, 0.9984930000E-01 /
  data (epsData(148,i,40), i=1,3) / 0.1908150000E+03, 0.1009570000E+00, 0.1125640000E+00 /
  data (epsData(149,i,40), i=1,3) / 0.1944350000E+03, 0.1292730000E+00, 0.1393080000E+00 /
  data (epsData(150,i,40), i=1,3) / 0.2020200000E+03, 0.1650410000E+00, 0.2027400000E+00 /
  data (epsData(151,i,40), i=1,3) / 0.2075770000E+03, 0.1734080000E+00, 0.2596150000E+00 /
  data (epsData(152,i,40), i=1,3) / 0.2129500000E+03, 0.1567040000E+00, 0.3100230000E+00 /
  data (epsData(153,i,40), i=1,3) / 0.2201890000E+03, 0.1177330000E+00, 0.3487090000E+00 /
  data (epsData(154,i,40), i=1,3) / 0.2363420000E+03, 0.4778520000E-01, 0.3829880000E+00 /
  data (epsData(155,i,40), i=1,3) / 0.2496730000E+03, -0.7847570000E-02, 0.3844460000E+00 /
  data (epsData(156,i,40), i=1,3) / 0.2627690000E+03, -0.4932170000E-01, 0.3655440000E+00 /
  data (epsData(157,i,40), i=1,3) / 0.2892990000E+03, -0.9439680000E-01, 0.3178780000E+00 /
  data (epsData(158,i,40), i=1,3) / 0.3148220000E+03, -0.1116280000E+00, 0.2700600000E+00 /
  data (epsData(159,i,40), i=1,3) / 0.3270320000E+03, -0.1049220000E+00, 0.2509810000E+00 /
  data (epsData(160,i,40), i=1,3) / 0.3306170000E+03, -0.1000020000E+00, 0.2551520000E+00 /
  data (epsData(161,i,40), i=1,3) / 0.3389480000E+03, -0.1102880000E+00, 0.2528730000E+00 /
  data (epsData(162,i,40), i=1,3) / 0.3527600000E+03, -0.1236370000E+00, 0.2408030000E+00 /
  data (epsData(163,i,40), i=1,3) / 0.3828200000E+03, -0.1352410000E+00, 0.1995420000E+00 /
  data (epsData(164,i,40), i=1,3) / 0.4183180000E+03, -0.1314040000E+00, 0.1604310000E+00 /
  data (epsData(165,i,40), i=1,3) / 0.4328160000E+03, -0.1275980000E+00, 0.1516920000E+00 /
  data (epsData(166,i,40), i=1,3) / 0.4756150000E+03, -0.1236870000E+00, 0.1187640000E+00 /
  data (epsData(167,i,40), i=1,3) / 0.5419800000E+03, -0.1080650000E+00, 0.8252580000E-01 /
  data (epsData(168,i,40), i=1,3) / 0.6066330000E+03, -0.9308560000E-01, 0.5970940000E-01 /
  data (epsData(169,i,40), i=1,3) / 0.6714540000E+03, -0.7966370000E-01, 0.4389820000E-01 /
  data (epsData(170,i,40), i=1,3) / 0.7529930000E+03, -0.6584230000E-01, 0.3071570000E-01 /
  data (epsData(171,i,40), i=1,3) / 0.8463860000E+03, -0.5350670000E-01, 0.2118650000E-01 /
  data (epsData(172,i,40), i=1,3) / 0.9482700000E+03, -0.4329690000E-01, 0.1466530000E-01 /
  data (epsData(173,i,40), i=1,3) / 0.1060370000E+04, -0.3488660000E-01, 0.1015810000E-01 /
  data (epsData(174,i,40), i=1,3) / 0.1184460000E+04, -0.2798050000E-01, 0.7028650000E-02 /
  data (epsData(175,i,40), i=1,3) / 0.1319700000E+04, -0.2241830000E-01, 0.4892570000E-02 /
  data (epsData(176,i,40), i=1,3) / 0.1466080000E+04, -0.1796190000E-01, 0.3431640000E-02 /
  data (epsData(177,i,40), i=1,3) / 0.1622330000E+04, -0.1440090000E-01, 0.2428920000E-02 /
  data (epsData(178,i,40), i=1,3) / 0.1796370000E+04, -0.1136200000E-01, 0.1696470000E-02 /
  data (epsData(179,i,40), i=1,3) / 0.2032060000E+04, -0.8119360000E-02, 0.1110560000E-02 /
  data (epsData(180,i,40), i=1,3) / 0.2153870000E+04, -0.6343600000E-02, 0.9059490000E-03 /
  data (epsData(181,i,40), i=1,3) / 0.2192360000E+04, -0.5454200000E-02, 0.8515690000E-03 /
  data (epsData(182,i,40), i=1,3) / 0.2207880000E+04, -0.4825990000E-02, 0.8599580000E-03 /
  data (epsData(183,i,40), i=1,3) / 0.2215670000E+04, -0.4274390000E-02, 0.8816670000E-03 /
  data (epsData(184,i,40), i=1,3) / 0.2218780000E+04, -0.3888600000E-02, 0.9195260000E-03 /
  data (epsData(185,i,40), i=1,3) / 0.2220340000E+04, -0.3591910000E-02, 0.9700430000E-03 /
  data (epsData(186,i,40), i=1,3) / 0.2221910000E+04, -0.3159740000E-02, 0.1129250000E-02 /
  data (epsData(187,i,40), i=1,3) / 0.2223150000E+04, -0.2834780000E-02, 0.1679910000E-02 /
  data (epsData(188,i,40), i=1,3) / 0.2223850000E+04, -0.2899250000E-02, 0.2026050000E-02 /
  data (epsData(189,i,40), i=1,3) / 0.2226010000E+04, -0.3222910000E-02, 0.2570600000E-02 /
  data (epsData(190,i,40), i=1,3) / 0.2228660000E+04, -0.3545320000E-02, 0.2877830000E-02 /
  data (epsData(191,i,40), i=1,3) / 0.2232710000E+04, -0.4040470000E-02, 0.3258630000E-02 /
  data (epsData(192,i,40), i=1,3) / 0.2237290000E+04, -0.4569590000E-02, 0.3281980000E-02 /
  data (epsData(193,i,40), i=1,3) / 0.2245820000E+04, -0.4956480000E-02, 0.3074570000E-02 /
  data (epsData(194,i,40), i=1,3) / 0.2260490000E+04, -0.5092890000E-02, 0.2950130000E-02 /
  data (epsData(195,i,40), i=1,3) / 0.2282720000E+04, -0.5167090000E-02, 0.2838940000E-02 /
  data (epsData(196,i,40), i=1,3) / 0.2300390000E+04, -0.4844900000E-02, 0.2781480000E-02 /
  data (epsData(197,i,40), i=1,3) / 0.2305340000E+04, -0.4508640000E-02, 0.2847580000E-02 /
  data (epsData(198,i,40), i=1,3) / 0.2307160000E+04, -0.4296070000E-02, 0.3108610000E-02 /
  data (epsData(199,i,40), i=1,3) / 0.2310270000E+04, -0.4480140000E-02, 0.3513590000E-02 /
  data (epsData(200,i,40), i=1,3) / 0.2318890000E+04, -0.5019310000E-02, 0.3820960000E-02 /
  data (epsData(201,i,40), i=1,3) / 0.2333670000E+04, -0.5440390000E-02, 0.3642040000E-02 /
  data (epsData(202,i,40), i=1,3) / 0.2397220000E+04, -0.5804120000E-02, 0.3310340000E-02 /
  data (epsData(203,i,40), i=1,3) / 0.2500890000E+04, -0.5696320000E-02, 0.2855890000E-02 /
  data (epsData(204,i,40), i=1,3) / 0.2527970000E+04, -0.5431210000E-02, 0.2842790000E-02 /
  data (epsData(205,i,40), i=1,3) / 0.2535820000E+04, -0.5459890000E-02, 0.3053540000E-02 /
  data (epsData(206,i,40), i=1,3) / 0.2589460000E+04, -0.5673470000E-02, 0.2888690000E-02 /
  data (epsData(207,i,40), i=1,3) / 0.2795410000E+04, -0.5380600000E-02, 0.2219180000E-02 /
  data (epsData(208,i,40), i=1,3) / 0.3211140000E+04, -0.4405600000E-02, 0.1365160000E-02 /
  data (epsData(209,i,40), i=1,3) / 0.3606120000E+04, -0.3604860000E-02, 0.9067480000E-03 /
  data (epsData(210,i,40), i=1,3) / 0.4032080000E+04, -0.2933650000E-02, 0.6132200000E-03 /
  data (epsData(211,i,40), i=1,3) / 0.4501100000E+04, -0.2379090000E-02, 0.4129400000E-03 /
  data (epsData(212,i,40), i=1,3) / 0.5008060000E+04, -0.1930480000E-02, 0.2799970000E-03 /
  data (epsData(213,i,40), i=1,3) / 0.5570990000E+04, -0.1561880000E-02, 0.1893990000E-03 /
  data (epsData(214,i,40), i=1,3) / 0.6198200000E+04, -0.1260240000E-02, 0.1277290000E-03 /
  data (epsData(215,i,40), i=1,3) / 0.6915790000E+04, -0.1009990000E-02, 0.8515680000E-04 /
  data (epsData(216,i,40), i=1,3) / 0.7705420000E+04, -0.8109200000E-03, 0.5689350000E-04 /
  data (epsData(217,i,40), i=1,3) / 0.8574740000E+04, -0.6522270000E-03, 0.3797820000E-04 /
  data (epsData(218,i,40), i=1,3) / 0.9557830000E+04, -0.5224500000E-03, 0.2522310000E-04 /
  data (epsData(219,i,40), i=1,3) / 0.1065810000E+05, -0.4179480000E-03, 0.1666000000E-04 /
  data (epsData(220,i,40), i=1,3) / 0.1189190000E+05, -0.3337330000E-03, 0.1097030000E-04 /
  data (epsData(221,i,40), i=1,3) / 0.1327770000E+05, -0.2657700000E-03, 0.7195910000E-05 /
  data (epsData(222,i,40), i=1,3) / 0.1432270000E+05, -0.2269310000E-03, 0.5374900000E-05 /
  data (epsData(223,i,40), i=1,3) / 0.1432600000E+05, -0.2138320000E-03, 0.5370220000E-05 /
  data (epsData(224,i,40), i=1,3) / 0.1432600000E+05, -0.2268190000E-03, 0.5370150000E-05 /
  data (epsData(225,i,40), i=1,3) / 0.1606340000E+05, -0.1775690000E-03, 0.3453140000E-05 /
  data (epsData(226,i,40), i=1,3) / 0.1781140000E+05, -0.1354190000E-03, 0.2316660000E-05 /
  data (epsData(227,i,40), i=1,3) / 0.1795880000E+05, -0.1267080000E-03, 0.2242860000E-05 /
  data (epsData(228,i,40), i=1,3) / 0.1798550000E+05, -0.1211670000E-03, 0.3034840000E-05 /
  data (epsData(229,i,40), i=1,3) / 0.1799440000E+05, -0.1161170000E-03, 0.5049480000E-05 /
  data (epsData(230,i,40), i=1,3) / 0.1799670000E+05, -0.1143700000E-03, 0.7510130000E-05 /
  data (epsData(231,i,40), i=1,3) / 0.1799800000E+05, -0.9315580000E-04, 0.1012670000E-04 /
  data (epsData(232,i,40), i=1,3) / 0.1799800000E+05, -0.1138920000E-03, 0.1022760000E-04 /
  data (epsData(233,i,40), i=1,3) / 0.1800180000E+05, -0.1178940000E-03, 0.1443410000E-04 /
  data (epsData(234,i,40), i=1,3) / 0.1801090000E+05, -0.1218590000E-03, 0.1398880000E-04 /
  data (epsData(235,i,40), i=1,3) / 0.1805110000E+05, -0.1268280000E-03, 0.1503640000E-04 /
  data (epsData(236,i,40), i=1,3) / 0.1827230000E+05, -0.1307890000E-03, 0.1463200000E-04 /
  data (epsData(237,i,40), i=1,3) / 0.1945980000E+05, -0.1216860000E-03, 0.1164650000E-04 /
  data (epsData(238,i,40), i=1,3) / 0.2250970000E+05, -0.9414840000E-04, 0.6922240000E-05 /
  data (epsData(239,i,40), i=1,3) / 0.2531430000E+05, -0.7522540000E-04, 0.4482220000E-05 /
  data (epsData(240,i,40), i=1,3) / 0.2837230000E+05, -0.6019020000E-04, 0.2924640000E-05 /
  data (epsData(241,i,40), i=1,3) / 0.3176330000E+05, -0.4814210000E-04, 0.1907580000E-05 /
  data (epsData(242,i,40), i=1,3) / 0.3554630000E+05, -0.3847570000E-04, 0.1240200000E-05 /
  data (epsData(243,i,40), i=1,3) / 0.3977180000E+05, -0.3073420000E-04, 0.8035590000E-06 /
  data (epsData(244,i,40), i=1,3) / 0.4450680000E+05, -0.2452920000E-04, 0.5180150000E-06 /
  data (epsData(245,i,40), i=1,3) / 0.4981900000E+05, -0.1956020000E-04, 0.3323040000E-06 /
  data (epsData(246,i,40), i=1,3) / 0.5577360000E+05, -0.1559080000E-04, 0.2122370000E-06 /
  data (epsData(247,i,40), i=1,3) / 0.6244830000E+05, -0.1242310000E-04, 0.1354760000E-06 /
  data (epsData(248,i,40), i=1,3) / 0.6994290000E+05, -0.9894160000E-05, 0.8548210000E-07 /
  data (epsData(249,i,40), i=1,3) / 0.7836140000E+05, -0.7875740000E-05, 0.5385280000E-07 /
  data (epsData(250,i,40), i=1,3) / 0.8780740000E+05, -0.6269080000E-05, 0.3369050000E-07 /
  data (epsData(251,i,40), i=1,3) / 0.9821100000E+05, -0.5016740000E-05, 0.2116820000E-07 /
  data (epsData(252,i,40), i=1,3) / 0.1000000000E+06, -0.4877190000E-05, 0.1962590000E-07 /
  data (epsData(1,i,41), i=1,3) / 0.2506580000E-02, -0.4447810000E+05, 0.8058020000E+05 /
  data (epsData(2,i,41), i=1,3) / 0.4693440000E-02, -0.3246840000E+05, 0.2396390000E+05 /
  data (epsData(3,i,41), i=1,3) / 0.7560590000E-02, -0.1476880000E+05, 0.9730410000E+04 /
  data (epsData(4,i,41), i=1,3) / 0.1110800000E-01, -0.7199710000E+04, 0.4791900000E+04 /
  data (epsData(5,i,41), i=1,3) / 0.1533570000E-01, -0.3668680000E+04, 0.2691650000E+04 /
  data (epsData(6,i,41), i=1,3) / 0.2024370000E-01, -0.1849980000E+04, 0.1663030000E+04 /
  data (epsData(7,i,41), i=1,3) / 0.2583200000E-01, -0.8344660000E+03, 0.1104070000E+04 /
  data (epsData(8,i,41), i=1,3) / 0.3210060000E-01, -0.2294570000E+03, 0.7750750000E+03 /
  data (epsData(9,i,41), i=1,3) / 0.3904940000E-01, 0.1506990000E+03, 0.5687940000E+03 /
  data (epsData(10,i,41), i=1,3) / 0.4667860000E-01, 0.4006730000E+03, 0.4326550000E+03 /
  data (epsData(11,i,41), i=1,3) / 0.5498800000E-01, 0.5718990000E+03, 0.3389540000E+03 /
  data (epsData(12,i,41), i=1,3) / 0.6397770000E-01, 0.6938200000E+03, 0.2722770000E+03 /
  data (epsData(13,i,41), i=1,3) / 0.7364770000E-01, 0.7842790000E+03, 0.2232360000E+03 /
  data (epsData(14,i,41), i=1,3) / 0.8399800000E-01, 0.8549110000E+03, 0.1862290000E+03 /
  data (epsData(15,i,41), i=1,3) / 0.1067390000E+00, 0.9716060000E+03, 0.1352180000E+03 /
  data (epsData(16,i,41), i=1,3) / 0.1191310000E+00, 0.1032570000E+04, 0.1293800000E+03 /
  data (epsData(17,i,41), i=1,3) / 0.1322020000E+00, 0.1072990000E+04, 0.1729300000E+03 /
  data (epsData(18,i,41), i=1,3) / 0.1603860000E+00, 0.1080190000E+04, 0.2169190000E+03 /
  data (epsData(19,i,41), i=1,3) / 0.2249170000E+00, 0.1068730000E+04, 0.2260620000E+03 /
  data (epsData(20,i,41), i=1,3) / 0.3003320000E+00, 0.1118140000E+04, 0.2026580000E+03 /
  data (epsData(21,i,41), i=1,3) / 0.3421210000E+00, 0.1191740000E+04, 0.1885720000E+03 /
  data (epsData(22,i,41), i=1,3) / 0.3640360000E+00, 0.1253880000E+04, 0.1815660000E+03 /
  data (epsData(23,i,41), i=1,3) / 0.3866320000E+00, 0.1352290000E+04, 0.1746970000E+03 /
  data (epsData(24,i,41), i=1,3) / 0.4099070000E+00, 0.1538420000E+04, 0.1680060000E+03 /
  data (epsData(25,i,41), i=1,3) / 0.4338630000E+00, 0.1715440000E+04, 0.4297000000E+03 /
  data (epsData(26,i,41), i=1,3) / 0.4585000000E+00, 0.1739390000E+04, 0.7462210000E+03 /
  data (epsData(27,i,41), i=1,3) / 0.4838160000E+00, 0.1644230000E+04, 0.1003750000E+04 /
  data (epsData(28,i,41), i=1,3) / 0.5098130000E+00, 0.1472940000E+04, 0.1212500000E+04 /
  data (epsData(29,i,41), i=1,3) / 0.5364900000E+00, 0.1277380000E+04, 0.1302470000E+04 /
  data (epsData(30,i,41), i=1,3) / 0.5918850000E+00, 0.9628220000E+03, 0.1374890000E+04 /
  data (epsData(31,i,41), i=1,3) / 0.6800800000E+00, 0.5563580000E+03, 0.1409020000E+04 /
  data (epsData(32,i,41), i=1,3) / 0.7108380000E+00, 0.4009800000E+03, 0.1407070000E+04 /
  data (epsData(33,i,41), i=1,3) / 0.7422780000E+00, 0.2552450000E+03, 0.1321880000E+04 /
  data (epsData(34,i,41), i=1,3) / 0.7743970000E+00, 0.1551130000E+03, 0.1226070000E+04 /
  data (epsData(35,i,41), i=1,3) / 0.8071970000E+00, 0.8588330000E+02, 0.1139340000E+04 /
  data (epsData(36,i,41), i=1,3) / 0.8748370000E+00, -0.4822110000E+01, 0.9890710000E+03 /
  data (epsData(37,i,41), i=1,3) / 0.9451980000E+00, -0.5949630000E+02, 0.8643200000E+03 /
  data (epsData(38,i,41), i=1,3) / 0.1055840000E+01, -0.1072410000E+03, 0.7111350000E+03 /
  data (epsData(39,i,41), i=1,3) / 0.1172610000E+01, -0.1252350000E+03, 0.5876260000E+03 /
  data (epsData(40,i,41), i=1,3) / 0.1295500000E+01, -0.1283950000E+03, 0.4909780000E+03 /
  data (epsData(41,i,41), i=1,3) / 0.1424510000E+01, -0.1238880000E+03, 0.4127110000E+03 /
  data (epsData(42,i,41), i=1,3) / 0.1606050000E+01, -0.1103910000E+03, 0.3316680000E+03 /
  data (epsData(43,i,41), i=1,3) / 0.1848280000E+01, -0.8942530000E+02, 0.2576740000E+03 /
  data (epsData(44,i,41), i=1,3) / 0.2054310000E+01, -0.7184000000E+02, 0.2147150000E+03 /
  data (epsData(45,i,41), i=1,3) / 0.2271220000E+01, -0.5516990000E+02, 0.1820220000E+03 /
  data (epsData(46,i,41), i=1,3) / 0.2557670000E+01, -0.3695580000E+02, 0.1524050000E+03 /
  data (epsData(47,i,41), i=1,3) / 0.2861130000E+01, -0.2181550000E+02, 0.1317890000E+03 /
  data (epsData(48,i,41), i=1,3) / 0.3181590000E+01, -0.9768910000E+01, 0.1174310000E+03 /
  data (epsData(49,i,41), i=1,3) / 0.3519060000E+01, -0.5859770000E+00, 0.1072850000E+03 /
  data (epsData(50,i,41), i=1,3) / 0.3873540000E+01, 0.6358440000E+01, 0.1000850000E+03 /
  data (epsData(51,i,41), i=1,3) / 0.4245030000E+01, 0.1064180000E+02, 0.9494590000E+02 /
  data (epsData(52,i,41), i=1,3) / 0.4398380000E+01, 0.1099220000E+02, 0.9177150000E+02 /
  data (epsData(53,i,41), i=1,3) / 0.4633520000E+01, 0.1491530000E+02, 0.8573500000E+02 /
  data (epsData(54,i,41), i=1,3) / 0.4874780000E+01, 0.2225320000E+02, 0.8077240000E+02 /
  data (epsData(55,i,41), i=1,3) / 0.5122160000E+01, 0.3483510000E+02, 0.7891160000E+02 /
  data (epsData(56,i,41), i=1,3) / 0.5205980000E+01, 0.4082930000E+02, 0.8144610000E+02 /
  data (epsData(57,i,41), i=1,3) / 0.5375660000E+01, 0.5169180000E+02, 0.9000240000E+02 /
  data (epsData(58,i,41), i=1,3) / 0.5461520000E+01, 0.6044870000E+02, 0.9571970000E+02 /
  data (epsData(59,i,41), i=1,3) / 0.5548060000E+01, 0.6275830000E+02, 0.1197340000E+03 /
  data (epsData(60,i,41), i=1,3) / 0.5635290000E+01, 0.5405950000E+02, 0.1349310000E+03 /
  data (epsData(61,i,41), i=1,3) / 0.5723190000E+01, 0.3917800000E+02, 0.1487280000E+03 /
  data (epsData(62,i,41), i=1,3) / 0.5811770000E+01, 0.2165240000E+02, 0.1502910000E+03 /
  data (epsData(63,i,41), i=1,3) / 0.5901040000E+01, 0.8106800000E+01, 0.1457470000E+03 /
  data (epsData(64,i,41), i=1,3) / 0.6081600000E+01, -0.1026580000E+02, 0.1363240000E+03 /
  data (epsData(65,i,41), i=1,3) / 0.6264890000E+01, -0.2164720000E+02, 0.1232600000E+03 /
  data (epsData(66,i,41), i=1,3) / 0.6544930000E+01, -0.2945420000E+02, 0.1058550000E+03 /
  data (epsData(67,i,41), i=1,3) / 0.6831080000E+01, -0.3134000000E+02, 0.9152470000E+02 /
  data (epsData(68,i,41), i=1,3) / 0.7222150000E+01, -0.2954670000E+02, 0.7714980000E+02 /
  data (epsData(69,i,41), i=1,3) / 0.7726290000E+01, -0.2509610000E+02, 0.6560790000E+02 /
  data (epsData(70,i,41), i=1,3) / 0.8353710000E+01, -0.2016350000E+02, 0.5727150000E+02 /
  data (epsData(71,i,41), i=1,3) / 0.9116650000E+01, -0.1616040000E+02, 0.5140480000E+02 /
  data (epsData(72,i,41), i=1,3) / 0.9912920000E+01, -0.1420570000E+02, 0.4794390000E+02 /
  data (epsData(73,i,41), i=1,3) / 0.1185810000E+02, -0.1350570000E+02, 0.4043970000E+02 /
  data (epsData(74,i,41), i=1,3) / 0.1642020000E+02, -0.1496070000E+02, 0.2678680000E+02 /
  data (epsData(75,i,41), i=1,3) / 0.1938290000E+02, -0.1458340000E+02, 0.1942690000E+02 /
  data (epsData(76,i,41), i=1,3) / 0.2189530000E+02, -0.1306530000E+02, 0.1447150000E+02 /
  data (epsData(77,i,41), i=1,3) / 0.2474400000E+02, -0.1069070000E+02, 0.1048270000E+02 /
  data (epsData(78,i,41), i=1,3) / 0.2718670000E+02, -0.8269390000E+01, 0.8049450000E+01 /
  data (epsData(79,i,41), i=1,3) / 0.2874710000E+02, -0.6175450000E+01, 0.6846270000E+01 /
  data (epsData(80,i,41), i=1,3) / 0.2954360000E+02, -0.4543550000E+01, 0.6338450000E+01 /
  data (epsData(81,i,41), i=1,3) / 0.2994590000E+02, -0.3277350000E+01, 0.6134810000E+01 /
  data (epsData(82,i,41), i=1,3) / 0.3014810000E+02, -0.2363150000E+01, 0.6067190000E+01 /
  data (epsData(83,i,41), i=1,3) / 0.3035100000E+02, -0.9961730000E+00, 0.6058730000E+01 /
  data (epsData(84,i,41), i=1,3) / 0.3055460000E+02, 0.1014520000E+01, 0.6706240000E+01 /
  data (epsData(85,i,41), i=1,3) / 0.3075880000E+02, 0.3258120000E+01, 0.9595920000E+01 /
  data (epsData(86,i,41), i=1,3) / 0.3080310000E+02, 0.3269980000E+01, 0.1125410000E+02 /
  data (epsData(87,i,41), i=1,3) / 0.3093610000E+02, 0.1921980000E+01, 0.1395760000E+02 /
  data (epsData(88,i,41), i=1,3) / 0.3099590000E+02, 0.6172860000E+00, 0.1515930000E+02 /
  data (epsData(89,i,41), i=1,3) / 0.3102990000E+02, -0.3532510000E+00, 0.1513890000E+02 /
  data (epsData(90,i,41), i=1,3) / 0.3114830000E+02, -0.2076350000E+01, 0.1473970000E+02 /
  data (epsData(91,i,41), i=1,3) / 0.3137180000E+02, -0.3798560000E+01, 0.1395100000E+02 /
  data (epsData(92,i,41), i=1,3) / 0.3165040000E+02, -0.4688480000E+01, 0.1291450000E+02 /
  data (epsData(93,i,41), i=1,3) / 0.3198400000E+02, -0.4860990000E+01, 0.1204270000E+02 /
  data (epsData(94,i,41), i=1,3) / 0.3223710000E+02, -0.4469920000E+01, 0.1165020000E+02 /
  data (epsData(95,i,41), i=1,3) / 0.3246670000E+02, -0.3695780000E+01, 0.1171400000E+02 /
  data (epsData(96,i,41), i=1,3) / 0.3251470000E+02, -0.3315250000E+01, 0.1192370000E+02 /
  data (epsData(97,i,41), i=1,3) / 0.3260030000E+02, -0.3179130000E+01, 0.1308760000E+02 /
  data (epsData(98,i,41), i=1,3) / 0.3279590000E+02, -0.4433920000E+01, 0.1446890000E+02 /
  data (epsData(99,i,41), i=1,3) / 0.3294830000E+02, -0.5607360000E+01, 0.1424190000E+02 /
  data (epsData(100,i,41), i=1,3) / 0.3338370000E+02, -0.7075760000E+01, 0.1340800000E+02 /
  data (epsData(101,i,41), i=1,3) / 0.3431470000E+02, -0.8343830000E+01, 0.1197230000E+02 /
  data (epsData(102,i,41), i=1,3) / 0.3535510000E+02, -0.9069370000E+01, 0.1041320000E+02 /
  data (epsData(103,i,41), i=1,3) / 0.3733610000E+02, -0.9286230000E+01, 0.8159660000E+01 /
  data (epsData(104,i,41), i=1,3) / 0.4056240000E+02, -0.8716880000E+01, 0.5731750000E+01 /
  data (epsData(105,i,41), i=1,3) / 0.4449440000E+02, -0.7704590000E+01, 0.3680710000E+01 /
  data (epsData(106,i,41), i=1,3) / 0.4891350000E+02, -0.6421620000E+01, 0.2200130000E+01 /
  data (epsData(107,i,41), i=1,3) / 0.5252010000E+02, -0.5440330000E+01, 0.1481850000E+01 /
  data (epsData(108,i,41), i=1,3) / 0.5672690000E+02, -0.4498880000E+01, 0.9881590000E+00 /
  data (epsData(109,i,41), i=1,3) / 0.6197290000E+02, -0.3625680000E+01, 0.6374910000E+00 /
  data (epsData(110,i,41), i=1,3) / 0.6779100000E+02, -0.2901670000E+01, 0.4119360000E+00 /
  data (epsData(111,i,41), i=1,3) / 0.7377310000E+02, -0.2341630000E+01, 0.2896070000E+00 /
  data (epsData(112,i,41), i=1,3) / 0.8028650000E+02, -0.1891090000E+01, 0.2211470000E+00 /
  data (epsData(113,i,41), i=1,3) / 0.8783460000E+02, -0.1510930000E+01, 0.1792580000E+00 /
  data (epsData(114,i,41), i=1,3) / 0.9665100000E+02, -0.1194380000E+01, 0.1520190000E+00 /
  data (epsData(115,i,41), i=1,3) / 0.1068180000E+03, -0.9334790000E+00, 0.1303780000E+00 /
  data (epsData(116,i,41), i=1,3) / 0.1181280000E+03, -0.7242940000E+00, 0.1140140000E+00 /
  data (epsData(117,i,41), i=1,3) / 0.1309000000E+03, -0.5537400000E+00, 0.1007720000E+00 /
  data (epsData(118,i,41), i=1,3) / 0.1459160000E+03, -0.4080050000E+00, 0.8731770000E-01 /
  data (epsData(119,i,41), i=1,3) / 0.1623090000E+03, -0.2876840000E+00, 0.7465180000E-01 /
  data (epsData(120,i,41), i=1,3) / 0.1838000000E+03, -0.1607510000E+00, 0.6075660000E-01 /
  data (epsData(121,i,41), i=1,3) / 0.1965970000E+03, -0.8577920000E-01, 0.5404770000E-01 /
  data (epsData(122,i,41), i=1,3) / 0.2001130000E+03, -0.5968090000E-01, 0.5251190000E-01 /
  data (epsData(123,i,41), i=1,3) / 0.2014420000E+03, -0.4540250000E-01, 0.5220710000E-01 /
  data (epsData(124,i,41), i=1,3) / 0.2018870000E+03, -0.3756620000E-01, 0.5251070000E-01 /
  data (epsData(125,i,41), i=1,3) / 0.2023490000E+03, -0.2871550000E-01, 0.6136930000E-01 /
  data (epsData(126,i,41), i=1,3) / 0.2024960000E+03, -0.2930070000E-01, 0.6676100000E-01 /
  data (epsData(127,i,41), i=1,3) / 0.2029000000E+03, -0.3289020000E-01, 0.6976230000E-01 /
  data (epsData(128,i,41), i=1,3) / 0.2036500000E+03, -0.3324410000E-01, 0.7021230000E-01 /
  data (epsData(129,i,41), i=1,3) / 0.2044770000E+03, -0.2776760000E-01, 0.6921330000E-01 /
  data (epsData(130,i,41), i=1,3) / 0.2048750000E+03, -0.2136670000E-01, 0.7070350000E-01 /
  data (epsData(131,i,41), i=1,3) / 0.2050410000E+03, -0.1889230000E-01, 0.7524120000E-01 /
  data (epsData(132,i,41), i=1,3) / 0.2052130000E+03, -0.2021960000E-01, 0.7954490000E-01 /
  data (epsData(133,i,41), i=1,3) / 0.2057200000E+03, -0.2363370000E-01, 0.8158970000E-01 /
  data (epsData(134,i,41), i=1,3) / 0.2069650000E+03, -0.2383850000E-01, 0.8105560000E-01 /
  data (epsData(135,i,41), i=1,3) / 0.2088220000E+03, -0.1649640000E-01, 0.7890420000E-01 /
  data (epsData(136,i,41), i=1,3) / 0.2112910000E+03, -0.2390500000E-02, 0.7771290000E-01 /
  data (epsData(137,i,41), i=1,3) / 0.2135990000E+03, 0.1487760000E-01, 0.7972570000E-01 /
  data (epsData(138,i,41), i=1,3) / 0.2167680000E+03, 0.3937640000E-01, 0.8923470000E-01 /
  data (epsData(139,i,41), i=1,3) / 0.2205770000E+03, 0.6496330000E-01, 0.1081020000E+00 /
  data (epsData(140,i,41), i=1,3) / 0.2264290000E+03, 0.9580070000E-01, 0.1444650000E+00 /
  data (epsData(141,i,41), i=1,3) / 0.2314770000E+03, 0.1131070000E+00, 0.1847030000E+00 /
  data (epsData(142,i,41), i=1,3) / 0.2370580000E+03, 0.1133200000E+00, 0.2323920000E+00 /
  data (epsData(143,i,41), i=1,3) / 0.2429450000E+03, 0.9586250000E-01, 0.2716810000E+00 /
  data (epsData(144,i,41), i=1,3) / 0.2543750000E+03, 0.5413470000E-01, 0.3131160000E+00 /
  data (epsData(145,i,41), i=1,3) / 0.2676000000E+03, 0.4001170000E-02, 0.3329050000E+00 /
  data (epsData(146,i,41), i=1,3) / 0.2795000000E+03, -0.3656250000E-01, 0.3294430000E+00 /
  data (epsData(147,i,41), i=1,3) / 0.2962690000E+03, -0.7404350000E-01, 0.3076230000E+00 /
  data (epsData(148,i,41), i=1,3) / 0.3242090000E+03, -0.1064360000E+00, 0.2652150000E+00 /
  data (epsData(149,i,41), i=1,3) / 0.3473890000E+03, -0.1119720000E+00, 0.2296840000E+00 /
  data (epsData(150,i,41), i=1,3) / 0.3576930000E+03, -0.1036770000E+00, 0.2179420000E+00 /
  data (epsData(151,i,41), i=1,3) / 0.3608670000E+03, -0.9846420000E-01, 0.2234410000E+00 /
  data (epsData(152,i,41), i=1,3) / 0.3658300000E+03, -0.1077460000E+00, 0.2245420000E+00 /
  data (epsData(153,i,41), i=1,3) / 0.3787340000E+03, -0.1162770000E+00, 0.2172640000E+00 /
  data (epsData(154,i,41), i=1,3) / 0.4021990000E+03, -0.1288200000E+00, 0.1899660000E+00 /
  data (epsData(155,i,41), i=1,3) / 0.4416530000E+03, -0.1280540000E+00, 0.1503680000E+00 /
  data (epsData(156,i,41), i=1,3) / 0.4687230000E+03, -0.1214180000E+00, 0.1329090000E+00 /
  data (epsData(157,i,41), i=1,3) / 0.5172800000E+03, -0.1158670000E+00, 0.1027010000E+00 /
  data (epsData(158,i,41), i=1,3) / 0.5878320000E+03, -0.1005660000E+00, 0.7154120000E-01 /
  data (epsData(159,i,41), i=1,3) / 0.6788880000E+03, -0.8214820000E-01, 0.4673210000E-01 /
  data (epsData(160,i,41), i=1,3) / 0.7722950000E+03, -0.6669430000E-01, 0.3140460000E-01 /
  data (epsData(161,i,41), i=1,3) / 0.8707350000E+03, -0.5405920000E-01, 0.2148180000E-01 /
  data (epsData(162,i,41), i=1,3) / 0.9762090000E+03, -0.4377910000E-01, 0.1484560000E-01 /
  data (epsData(163,i,41), i=1,3) / 0.1090210000E+04, -0.3542660000E-01, 0.1032800000E-01 /
  data (epsData(164,i,41), i=1,3) / 0.1215710000E+04, -0.2856480000E-01, 0.7185600000E-02 /
  data (epsData(165,i,41), i=1,3) / 0.1352700000E+04, -0.2300140000E-01, 0.5022350000E-02 /
  data (epsData(166,i,41), i=1,3) / 0.1498820000E+04, -0.1858550000E-01, 0.3551500000E-02 /
  data (epsData(167,i,41), i=1,3) / 0.1657360000E+04, -0.1499360000E-01, 0.2519640000E-02 /
  data (epsData(168,i,41), i=1,3) / 0.1832670000E+04, -0.1196690000E-01, 0.1769260000E-02 /
  data (epsData(169,i,41), i=1,3) / 0.2044820000E+04, -0.9162480000E-02, 0.1214390000E-02 /
  data (epsData(170,i,41), i=1,3) / 0.2281700000E+04, -0.6357550000E-02, 0.8278140000E-03 /
  data (epsData(171,i,41), i=1,3) / 0.2333930000E+04, -0.5422830000E-02, 0.7646220000E-03 /
  data (epsData(172,i,41), i=1,3) / 0.2351510000E+04, -0.4855990000E-02, 0.7670810000E-03 /
  data (epsData(173,i,41), i=1,3) / 0.2361130000E+04, -0.4349330000E-02, 0.7832260000E-03 /
  data (epsData(174,i,41), i=1,3) / 0.2365950000E+04, -0.3860300000E-02, 0.8288700000E-03 /
  data (epsData(175,i,41), i=1,3) / 0.2367560000E+04, -0.3582890000E-02, 0.8751510000E-03 /
  data (epsData(176,i,41), i=1,3) / 0.2369160000E+04, -0.3167330000E-02, 0.1000230000E-02 /
  data (epsData(177,i,41), i=1,3) / 0.2370770000E+04, -0.2730300000E-02, 0.1640070000E-02 /
  data (epsData(178,i,41), i=1,3) / 0.2371230000E+04, -0.2734110000E-02, 0.2013630000E-02 /
  data (epsData(179,i,41), i=1,3) / 0.2371780000E+04, -0.2947780000E-02, 0.2294950000E-02 /
  data (epsData(180,i,41), i=1,3) / 0.2372860000E+04, -0.3262110000E-02, 0.2463700000E-02 /
  data (epsData(181,i,41), i=1,3) / 0.2375340000E+04, -0.3685380000E-02, 0.2655950000E-02 /
  data (epsData(182,i,41), i=1,3) / 0.2383020000E+04, -0.4384350000E-02, 0.2963770000E-02 /
  data (epsData(183,i,41), i=1,3) / 0.2389780000E+04, -0.4809970000E-02, 0.2839790000E-02 /
  data (epsData(184,i,41), i=1,3) / 0.2404750000E+04, -0.5036500000E-02, 0.2631190000E-02 /
  data (epsData(185,i,41), i=1,3) / 0.2435070000E+04, -0.5105530000E-02, 0.2489840000E-02 /
  data (epsData(186,i,41), i=1,3) / 0.2457060000E+04, -0.4776210000E-02, 0.2434840000E-02 /
  data (epsData(187,i,41), i=1,3) / 0.2462620000E+04, -0.4454860000E-02, 0.2492390000E-02 /
  data (epsData(188,i,41), i=1,3) / 0.2464400000E+04, -0.4226240000E-02, 0.2669530000E-02 /
  data (epsData(189,i,41), i=1,3) / 0.2465750000E+04, -0.4255490000E-02, 0.2999010000E-02 /
  data (epsData(190,i,41), i=1,3) / 0.2469580000E+04, -0.4588730000E-02, 0.3216760000E-02 /
  data (epsData(191,i,41), i=1,3) / 0.2480580000E+04, -0.5083710000E-02, 0.3320960000E-02 /
  data (epsData(192,i,41), i=1,3) / 0.2501820000E+04, -0.5352360000E-02, 0.3132180000E-02 /
  data (epsData(193,i,41), i=1,3) / 0.2581810000E+04, -0.5573590000E-02, 0.2792140000E-02 /
  data (epsData(194,i,41), i=1,3) / 0.2677830000E+04, -0.5332850000E-02, 0.2458650000E-02 /
  data (epsData(195,i,41), i=1,3) / 0.2695950000E+04, -0.5142350000E-02, 0.2522020000E-02 /
  data (epsData(196,i,41), i=1,3) / 0.2703750000E+04, -0.5224570000E-02, 0.2678280000E-02 /
  data (epsData(197,i,41), i=1,3) / 0.2772500000E+04, -0.5368240000E-02, 0.2494050000E-02 /
  data (epsData(198,i,41), i=1,3) / 0.3015410000E+04, -0.4981330000E-02, 0.1863460000E-02 /
  data (epsData(199,i,41), i=1,3) / 0.3443070000E+04, -0.4076870000E-02, 0.1169500000E-02 /
  data (epsData(200,i,41), i=1,3) / 0.3864700000E+04, -0.3325970000E-02, 0.7774110000E-03 /
  data (epsData(201,i,41), i=1,3) / 0.4324950000E+04, -0.2700880000E-02, 0.5240410000E-03 /
  data (epsData(202,i,41), i=1,3) / 0.4818870000E+04, -0.2194550000E-02, 0.3544850000E-03 /
  data (epsData(203,i,41), i=1,3) / 0.5362920000E+04, -0.1778700000E-02, 0.2399690000E-03 /
  data (epsData(204,i,41), i=1,3) / 0.5966900000E+04, -0.1437920000E-02, 0.1620930000E-03 /
  data (epsData(205,i,41), i=1,3) / 0.6632430000E+04, -0.1162900000E-02, 0.1100330000E-03 /
  data (epsData(206,i,41), i=1,3) / 0.7377180000E+04, -0.9377440000E-03, 0.7381520000E-04 /
  data (epsData(207,i,41), i=1,3) / 0.8222010000E+04, -0.7522570000E-03, 0.4924280000E-04 /
  data (epsData(208,i,41), i=1,3) / 0.9154690000E+04, -0.6044590000E-03, 0.3283060000E-04 /
  data (epsData(209,i,41), i=1,3) / 0.1020790000E+05, -0.4839310000E-03, 0.2175290000E-04 /
  data (epsData(210,i,41), i=1,3) / 0.1138540000E+05, -0.3870560000E-03, 0.1435870000E-04 /
  data (epsData(211,i,41), i=1,3) / 0.1270750000E+05, -0.3089260000E-03, 0.9439010000E-05 /
  data (epsData(212,i,41), i=1,3) / 0.1419420000E+05, -0.2458480000E-03, 0.6169460000E-05 /
  data (epsData(213,i,41), i=1,3) / 0.1519990000E+05, -0.2131710000E-03, 0.4745860000E-05 /
  data (epsData(214,i,41), i=1,3) / 0.1520000000E+05, -0.2070610000E-03, 0.4745740000E-05 /
  data (epsData(215,i,41), i=1,3) / 0.1520000000E+05, -0.2131670000E-03, 0.4745680000E-05 /
  data (epsData(216,i,41), i=1,3) / 0.1704670000E+05, -0.1668620000E-03, 0.3048950000E-05 /
  data (epsData(217,i,41), i=1,3) / 0.1880170000E+05, -0.1290570000E-03, 0.2086080000E-05 /
  data (epsData(218,i,41), i=1,3) / 0.1894930000E+05, -0.1210820000E-03, 0.2023770000E-05 /
  data (epsData(219,i,41), i=1,3) / 0.1897640000E+05, -0.1154900000E-03, 0.3005020000E-05 /
  data (epsData(220,i,41), i=1,3) / 0.1898320000E+05, -0.1117260000E-03, 0.5035220000E-05 /
  data (epsData(221,i,41), i=1,3) / 0.1898550000E+05, -0.1105250000E-03, 0.7679000000E-05 /
  data (epsData(222,i,41), i=1,3) / 0.1898600000E+05, -0.9529980000E-04, 0.8471510000E-05 /
  data (epsData(223,i,41), i=1,3) / 0.1898600000E+05, -0.1104190000E-03, 0.8549560000E-05 /
  data (epsData(224,i,41), i=1,3) / 0.1899050000E+05, -0.1138250000E-03, 0.1253510000E-04 /
  data (epsData(225,i,41), i=1,3) / 0.1900320000E+05, -0.1175930000E-03, 0.1275360000E-04 /
  data (epsData(226,i,41), i=1,3) / 0.1905610000E+05, -0.1221680000E-03, 0.1353020000E-04 /
  data (epsData(227,i,41), i=1,3) / 0.1937110000E+05, -0.1247700000E-03, 0.1287150000E-04 /
  data (epsData(228,i,41), i=1,3) / 0.2193620000E+05, -0.1034920000E-03, 0.8278190000E-05 /
  data (epsData(229,i,41), i=1,3) / 0.2480720000E+05, -0.8229550000E-04, 0.5267690000E-05 /
  data (epsData(230,i,41), i=1,3) / 0.2784960000E+05, -0.6578730000E-04, 0.3424810000E-05 /
  data (epsData(231,i,41), i=1,3) / 0.3120090000E+05, -0.5260770000E-04, 0.2233700000E-05 /
  data (epsData(232,i,41), i=1,3) / 0.3492590000E+05, -0.4205460000E-04, 0.1454120000E-05 /
  data (epsData(233,i,41), i=1,3) / 0.3908920000E+05, -0.3358920000E-04, 0.9430480000E-06 /
  data (epsData(234,i,41), i=1,3) / 0.4374660000E+05, -0.2681160000E-04, 0.6091010000E-06 /
  data (epsData(235,i,41), i=1,3) / 0.4896540000E+05, -0.2138690000E-04, 0.3916910000E-06 /
  data (epsData(236,i,41), i=1,3) / 0.5481540000E+05, -0.1705050000E-04, 0.2506050000E-06 /
  data (epsData(237,i,41), i=1,3) / 0.6138380000E+05, -0.1358320000E-04, 0.1598730000E-06 /
  data (epsData(238,i,41), i=1,3) / 0.6876770000E+05, -0.1081330000E-04, 0.1013250000E-06 /
  data (epsData(239,i,41), i=1,3) / 0.7704410000E+05, -0.8607610000E-05, 0.6377160000E-07 /
  data (epsData(240,i,41), i=1,3) / 0.8633370000E+05, -0.6851100000E-05, 0.3993430000E-07 /
  data (epsData(241,i,41), i=1,3) / 0.9665100000E+05, -0.5469440000E-05, 0.2504240000E-07 /
  data (epsData(242,i,41), i=1,3) / 0.1000000000E+06, -0.5152910000E-05, 0.2175570000E-07 /
  data (epsData(1,i,42), i=1,3) / 0.2506580000E-02, -0.1834070000E+05, 0.4068220000E+05 /
  data (epsData(2,i,42), i=1,3) / 0.4693440000E-02, -0.1332720000E+05, 0.1262370000E+05 /
  data (epsData(3,i,42), i=1,3) / 0.7560590000E-02, -0.4972990000E+04, 0.5380210000E+04 /
  data (epsData(4,i,42), i=1,3) / 0.1110800000E-01, -0.1366410000E+04, 0.2787990000E+04 /
  data (epsData(5,i,42), i=1,3) / 0.1533570000E-01, 0.3484810000E+03, 0.1647450000E+04 /
  data (epsData(6,i,42), i=1,3) / 0.2024370000E-01, 0.1292340000E+04, 0.1068540000E+04 /
  data (epsData(7,i,42), i=1,3) / 0.2583200000E-01, 0.1965820000E+04, 0.7422990000E+03 /
  data (epsData(8,i,42), i=1,3) / 0.3210060000E-01, 0.2342910000E+04, 0.1013030000E+04 /
  data (epsData(9,i,42), i=1,3) / 0.3904940000E-01, 0.2341400000E+04, 0.1163940000E+04 /
  data (epsData(10,i,42), i=1,3) / 0.5498800000E-01, 0.2092200000E+04, 0.1141510000E+04 /
  data (epsData(11,i,42), i=1,3) / 0.8399800000E-01, 0.1818040000E+04, 0.9217580000E+03 /
  data (epsData(12,i,42), i=1,3) / 0.1191310000E+00, 0.1693330000E+04, 0.7186950000E+03 /
  data (epsData(13,i,42), i=1,3) / 0.1603860000E+00, 0.1665230000E+04, 0.5652440000E+03 /
  data (epsData(14,i,42), i=1,3) / 0.2077630000E+00, 0.1718970000E+04, 0.4523170000E+03 /
  data (epsData(15,i,42), i=1,3) / 0.2612640000E+00, 0.1890660000E+04, 0.3685160000E+03 /
  data (epsData(16,i,42), i=1,3) / 0.2804580000E+00, 0.1999730000E+04, 0.3454980000E+03 /
  data (epsData(17,i,42), i=1,3) / 0.3003320000E+00, 0.2171180000E+04, 0.3244870000E+03 /
  data (epsData(18,i,42), i=1,3) / 0.3208860000E+00, 0.2494770000E+04, 0.3052700000E+03 /
  data (epsData(19,i,42), i=1,3) / 0.3421210000E+00, 0.2795640000E+04, 0.7627110000E+03 /
  data (epsData(20,i,42), i=1,3) / 0.3640360000E+00, 0.2824540000E+04, 0.1299150000E+04 /
  data (epsData(21,i,42), i=1,3) / 0.3866320000E+00, 0.2654560000E+04, 0.1723490000E+04 /
  data (epsData(22,i,42), i=1,3) / 0.4099070000E+00, 0.2373630000E+04, 0.2057300000E+04 /
  data (epsData(23,i,42), i=1,3) / 0.4338630000E+00, 0.2025120000E+04, 0.2264000000E+04 /
  data (epsData(24,i,42), i=1,3) / 0.4585000000E+00, 0.1701140000E+04, 0.2297710000E+04 /
  data (epsData(25,i,42), i=1,3) / 0.5098130000E+00, 0.1213570000E+04, 0.2314120000E+04 /
  data (epsData(26,i,42), i=1,3) / 0.6206030000E+00, 0.3607700000E+03, 0.2225580000E+04 /
  data (epsData(27,i,42), i=1,3) / 0.6500010000E+00, 0.1444450000E+03, 0.2070180000E+04 /
  data (epsData(28,i,42), i=1,3) / 0.6800800000E+00, -0.3500390000E+00, 0.1898770000E+04 /
  data (epsData(29,i,42), i=1,3) / 0.7108380000E+00, -0.9551010000E+02, 0.1745180000E+04 /
  data (epsData(30,i,42), i=1,3) / 0.7422780000E+00, -0.1624420000E+03, 0.1607210000E+04 /
  data (epsData(31,i,42), i=1,3) / 0.8071970000E+00, -0.2455310000E+03, 0.1370830000E+04 /
  data (epsData(32,i,42), i=1,3) / 0.8748370000E+00, -0.2885220000E+03, 0.1177470000E+04 /
  data (epsData(33,i,42), i=1,3) / 0.9813990000E+00, -0.3145140000E+03, 0.9430710000E+03 /
  data (epsData(34,i,42), i=1,3) / 0.1094080000E+01, -0.3076220000E+03, 0.7605270000E+03 /
  data (epsData(35,i,42), i=1,3) / 0.1212890000E+01, -0.2869120000E+03, 0.6203110000E+03 /
  data (epsData(36,i,42), i=1,3) / 0.1380830000E+01, -0.2508200000E+03, 0.4781650000E+03 /
  data (epsData(37,i,42), i=1,3) / 0.1559640000E+01, -0.2117590000E+03, 0.3743770000E+03 /
  data (epsData(38,i,42), i=1,3) / 0.1749350000E+01, -0.1744760000E+03, 0.2975450000E+03 /
  data (epsData(39,i,42), i=1,3) / 0.1949930000E+01, -0.1399600000E+03, 0.2406730000E+03 /
  data (epsData(40,i,42), i=1,3) / 0.2161410000E+01, -0.1094440000E+03, 0.1979970000E+03 /
  data (epsData(41,i,42), i=1,3) / 0.2441050000E+01, -0.7721060000E+02, 0.1598470000E+03 /
  data (epsData(42,i,42), i=1,3) / 0.2737710000E+01, -0.5100210000E+02, 0.1335590000E+03 /
  data (epsData(43,i,42), i=1,3) / 0.3051370000E+01, -0.3016320000E+02, 0.1153180000E+03 /
  data (epsData(44,i,42), i=1,3) / 0.3382030000E+01, -0.1378200000E+02, 0.1024310000E+03 /
  data (epsData(45,i,42), i=1,3) / 0.3658810000E+01, -0.3165640000E+01, 0.9481910000E+02 /
  data (epsData(46,i,42), i=1,3) / 0.4020090000E+01, 0.6025830000E+01, 0.8780580000E+02 /
  data (epsData(47,i,42), i=1,3) / 0.4476080000E+01, 0.1613400000E+02, 0.7729170000E+02 /
  data (epsData(48,i,42), i=1,3) / 0.4874780000E+01, 0.2895740000E+02, 0.7051810000E+02 /
  data (epsData(49,i,42), i=1,3) / 0.5122160000E+01, 0.3926540000E+02, 0.6855580000E+02 /
  data (epsData(50,i,42), i=1,3) / 0.5290480000E+01, 0.4881640000E+02, 0.6809280000E+02 /
  data (epsData(51,i,42), i=1,3) / 0.5375660000E+01, 0.5555730000E+02, 0.6963430000E+02 /
  data (epsData(52,i,42), i=1,3) / 0.5548060000E+01, 0.6794500000E+02, 0.7949520000E+02 /
  data (epsData(53,i,42), i=1,3) / 0.5635290000E+01, 0.7774510000E+02, 0.8397710000E+02 /
  data (epsData(54,i,42), i=1,3) / 0.5723190000E+01, 0.8278960000E+02, 0.1077400000E+03 /
  data (epsData(55,i,42), i=1,3) / 0.5811770000E+01, 0.7583300000E+02, 0.1268150000E+03 /
  data (epsData(56,i,42), i=1,3) / 0.5901040000E+01, 0.6112080000E+02, 0.1423020000E+03 /
  data (epsData(57,i,42), i=1,3) / 0.5990980000E+01, 0.4230460000E+02, 0.1475060000E+03 /
  data (epsData(58,i,42), i=1,3) / 0.6081600000E+01, 0.2675140000E+02, 0.1423520000E+03 /
  data (epsData(59,i,42), i=1,3) / 0.6264890000E+01, 0.7291970000E+01, 0.1321670000E+03 /
  data (epsData(60,i,42), i=1,3) / 0.6357560000E+01, 0.4687660000E+00, 0.1248730000E+03 /
  data (epsData(61,i,42), i=1,3) / 0.6544930000E+01, -0.7149760000E+01, 0.1112950000E+03 /
  data (epsData(62,i,42), i=1,3) / 0.6735020000E+01, -0.1022380000E+02, 0.9936220000E+02 /
  data (epsData(63,i,42), i=1,3) / 0.6927830000E+01, -0.1012670000E+02, 0.8894610000E+02 /
  data (epsData(64,i,42), i=1,3) / 0.7123360000E+01, -0.7987070000E+01, 0.8015800000E+02 /
  data (epsData(65,i,42), i=1,3) / 0.7321620000E+01, -0.3758860000E+01, 0.7334310000E+02 /
  data (epsData(66,i,42), i=1,3) / 0.7421760000E+01, -0.7017130000E+00, 0.7039530000E+02 /
  data (epsData(67,i,42), i=1,3) / 0.7522590000E+01, 0.4130430000E+01, 0.6793470000E+02 /
  data (epsData(68,i,42), i=1,3) / 0.7624100000E+01, 0.7815140000E+01, 0.7204180000E+02 /
  data (epsData(69,i,42), i=1,3) / 0.7726290000E+01, 0.7986450000E+01, 0.7511280000E+02 /
  data (epsData(70,i,42), i=1,3) / 0.7829160000E+01, 0.5884050000E+01, 0.7753390000E+02 /
  data (epsData(71,i,42), i=1,3) / 0.7932710000E+01, 0.3281690000E+01, 0.7633830000E+02 /
  data (epsData(72,i,42), i=1,3) / 0.8141850000E+01, 0.1208050000E+01, 0.7284600000E+02 /
  data (epsData(73,i,42), i=1,3) / 0.8568290000E+01, 0.8596270000E+00, 0.6682070000E+02 /
  data (epsData(74,i,42), i=1,3) / 0.9116650000E+01, 0.2651620000E+01, 0.6128210000E+02 /
  data (epsData(75,i,42), i=1,3) / 0.9797130000E+01, 0.5181420000E+01, 0.5796040000E+02 /
  data (epsData(76,i,42), i=1,3) / 0.1074250000E+02, 0.6693610000E+01, 0.5584690000E+02 /
  data (epsData(77,i,42), i=1,3) / 0.1148020000E+02, 0.5731790000E+01, 0.5518080000E+02 /
  data (epsData(78,i,42), i=1,3) / 0.1356690000E+02, 0.3179580000E+01, 0.5116430000E+02 /
  data (epsData(79,i,42), i=1,3) / 0.1702350000E+02, -0.3290740000E+01, 0.4620170000E+02 /
  data (epsData(80,i,42), i=1,3) / 0.2036950000E+02, -0.9638730000E+01, 0.3962990000E+02 /
  data (epsData(81,i,42), i=1,3) / 0.2437840000E+02, -0.1418190000E+02, 0.3082520000E+02 /
  data (epsData(82,i,42), i=1,3) / 0.2776670000E+02, -0.1552160000E+02, 0.2374270000E+02 /
  data (epsData(83,i,42), i=1,3) / 0.3075880000E+02, -0.1493240000E+02, 0.1856370000E+02 /
  data (epsData(84,i,42), i=1,3) / 0.3347570000E+02, -0.1343590000E+02, 0.1508410000E+02 /
  data (epsData(85,i,42), i=1,3) / 0.3520480000E+02, -0.1200730000E+02, 0.1407660000E+02 /
  data (epsData(86,i,42), i=1,3) / 0.3660510000E+02, -0.1191870000E+02, 0.1403320000E+02 /
  data (epsData(87,i,42), i=1,3) / 0.3882450000E+02, -0.1262090000E+02, 0.1301320000E+02 /
  data (epsData(88,i,42), i=1,3) / 0.4225620000E+02, -0.1293320000E+02, 0.9895010000E+01 /
  data (epsData(89,i,42), i=1,3) / 0.4630760000E+02, -0.1196480000E+02, 0.6811960000E+01 /
  data (epsData(90,i,42), i=1,3) / 0.5093490000E+02, -0.1037850000E+02, 0.4277790000E+01 /
  data (epsData(91,i,42), i=1,3) / 0.5590780000E+02, -0.8599780000E+01, 0.2559450000E+01 /
  data (epsData(92,i,42), i=1,3) / 0.6041710000E+02, -0.7194890000E+01, 0.1652550000E+01 /
  data (epsData(93,i,42), i=1,3) / 0.6468170000E+02, -0.6093040000E+01, 0.1128590000E+01 /
  data (epsData(94,i,42), i=1,3) / 0.7025320000E+02, -0.4972210000E+01, 0.7025400000E+00 /
  data (epsData(95,i,42), i=1,3) / 0.7640060000E+02, -0.4023460000E+01, 0.4395600000E+00 /
  data (epsData(96,i,42), i=1,3) / 0.8312010000E+02, -0.3248630000E+01, 0.2956430000E+00 /
  data (epsData(97,i,42), i=1,3) / 0.9063450000E+02, -0.2613630000E+01, 0.2182670000E+00 /
  data (epsData(98,i,42), i=1,3) / 0.9890670000E+02, -0.2107150000E+01, 0.1813450000E+00 /
  data (epsData(99,i,42), i=1,3) / 0.1085150000E+03, -0.1685970000E+01, 0.1597950000E+00 /
  data (epsData(100,i,42), i=1,3) / 0.1198210000E+03, -0.1332680000E+01, 0.1457590000E+00 /
  data (epsData(101,i,42), i=1,3) / 0.1326930000E+03, -0.1049090000E+01, 0.1350390000E+00 /
  data (epsData(102,i,42), i=1,3) / 0.1478960000E+03, -0.8133730000E+00, 0.1220680000E+00 /
  data (epsData(103,i,42), i=1,3) / 0.1649750000E+03, -0.6230750000E+00, 0.1076390000E+00 /
  data (epsData(104,i,42), i=1,3) / 0.1852450000E+03, -0.4577630000E+00, 0.9131890000E-01 /
  data (epsData(105,i,42), i=1,3) / 0.2113690000E+03, -0.2933580000E+00, 0.7365310000E-01 /
  data (epsData(106,i,42), i=1,3) / 0.2251450000E+03, -0.2076100000E+00, 0.6586610000E-01 /
  data (epsData(107,i,42), i=1,3) / 0.2270280000E+03, -0.1902780000E+00, 0.6516530000E-01 /
  data (epsData(108,i,42), i=1,3) / 0.2275000000E+03, -0.1830800000E+00, 0.6543400000E-01 /
  data (epsData(109,i,42), i=1,3) / 0.2280360000E+03, -0.1756250000E+00, 0.7520240000E-01 /
  data (epsData(110,i,42), i=1,3) / 0.2287170000E+03, -0.1793920000E+00, 0.7944590000E-01 /
  data (epsData(111,i,42), i=1,3) / 0.2305340000E+03, -0.1704150000E+00, 0.7783170000E-01 /
  data (epsData(112,i,42), i=1,3) / 0.2311410000E+03, -0.1623240000E+00, 0.8309550000E-01 /
  data (epsData(113,i,42), i=1,3) / 0.2318200000E+03, -0.1656530000E+00, 0.8802450000E-01 /
  data (epsData(114,i,42), i=1,3) / 0.2342350000E+03, -0.1592750000E+00, 0.8464130000E-01 /
  data (epsData(115,i,42), i=1,3) / 0.2385500000E+03, -0.1326200000E+00, 0.8274510000E-01 /
  data (epsData(116,i,42), i=1,3) / 0.2436390000E+03, -0.9483100000E-01, 0.9284880000E-01 /
  data (epsData(117,i,42), i=1,3) / 0.2489370000E+03, -0.6022680000E-01, 0.1145550000E+00 /
  data (epsData(118,i,42), i=1,3) / 0.2537410000E+03, -0.3359240000E-01, 0.1411250000E+00 /
  data (epsData(119,i,42), i=1,3) / 0.2579580000E+03, -0.1684700000E-01, 0.1708650000E+00 /
  data (epsData(120,i,42), i=1,3) / 0.2627420000E+03, -0.1031130000E-01, 0.2072520000E+00 /
  data (epsData(121,i,42), i=1,3) / 0.2676940000E+03, -0.1626750000E-01, 0.2398980000E+00 /
  data (epsData(122,i,42), i=1,3) / 0.2756740000E+03, -0.3610910000E-01, 0.2739050000E+00 /
  data (epsData(123,i,42), i=1,3) / 0.2868920000E+03, -0.6825550000E-01, 0.3002450000E+00 /
  data (epsData(124,i,42), i=1,3) / 0.3002860000E+03, -0.1064400000E+00, 0.3061240000E+00 /
  data (epsData(125,i,42), i=1,3) / 0.3164770000E+03, -0.1379610000E+00, 0.2904190000E+00 /
  data (epsData(126,i,42), i=1,3) / 0.3479050000E+03, -0.1632410000E+00, 0.2479390000E+00 /
  data (epsData(127,i,42), i=1,3) / 0.3765870000E+03, -0.1623160000E+00, 0.2085880000E+00 /
  data (epsData(128,i,42), i=1,3) / 0.3911730000E+03, -0.1503530000E+00, 0.1931820000E+00 /
  data (epsData(129,i,42), i=1,3) / 0.3954370000E+03, -0.1457450000E+00, 0.1982700000E+00 /
  data (epsData(130,i,42), i=1,3) / 0.4082940000E+03, -0.1531460000E+00, 0.1932290000E+00 /
  data (epsData(131,i,42), i=1,3) / 0.4326890000E+03, -0.1596490000E+00, 0.1741130000E+00 /
  data (epsData(132,i,42), i=1,3) / 0.4780520000E+03, -0.1520760000E+00, 0.1350560000E+00 /
  data (epsData(133,i,42), i=1,3) / 0.5089340000E+03, -0.1416860000E+00, 0.1180270000E+00 /
  data (epsData(134,i,42), i=1,3) / 0.5671610000E+03, -0.1289680000E+00, 0.8816200000E-01 /
  data (epsData(135,i,42), i=1,3) / 0.6421630000E+03, -0.1094750000E+00, 0.6186740000E-01 /
  data (epsData(136,i,42), i=1,3) / 0.7202030000E+03, -0.9186010000E-01, 0.4380150000E-01 /
  data (epsData(137,i,42), i=1,3) / 0.8068570000E+03, -0.7594310000E-01, 0.3076720000E-01 /
  data (epsData(138,i,42), i=1,3) / 0.9030530000E+03, -0.6215890000E-01, 0.2150620000E-01 /
  data (epsData(139,i,42), i=1,3) / 0.1009270000E+04, -0.5056740000E-01, 0.1499990000E-01 /
  data (epsData(140,i,42), i=1,3) / 0.1125310000E+04, -0.4104140000E-01, 0.1048210000E-01 /
  data (epsData(141,i,42), i=1,3) / 0.1251370000E+04, -0.3329960000E-01, 0.7352660000E-02 /
  data (epsData(142,i,42), i=1,3) / 0.1389110000E+04, -0.2699040000E-01, 0.5174280000E-02 /
  data (epsData(143,i,42), i=1,3) / 0.1538520000E+04, -0.2189000000E-01, 0.3660190000E-02 /
  data (epsData(144,i,42), i=1,3) / 0.1702160000E+04, -0.1771880000E-01, 0.2588480000E-02 /
  data (epsData(145,i,42), i=1,3) / 0.1883300000E+04, -0.1422670000E-01, 0.1814760000E-02 /
  data (epsData(146,i,42), i=1,3) / 0.2097740000E+04, -0.1112990000E-01, 0.1251680000E-02 /
  data (epsData(147,i,42), i=1,3) / 0.2442800000E+04, -0.7090790000E-02, 0.7334590000E-03 /
  data (epsData(148,i,42), i=1,3) / 0.2491810000E+04, -0.6186790000E-02, 0.6837510000E-03 /
  data (epsData(149,i,42), i=1,3) / 0.2509930000E+04, -0.5501720000E-02, 0.7071080000E-03 /
  data (epsData(150,i,42), i=1,3) / 0.2514880000E+04, -0.5118130000E-02, 0.7400430000E-03 /
  data (epsData(151,i,42), i=1,3) / 0.2516540000E+04, -0.4913740000E-02, 0.7717010000E-03 /
  data (epsData(152,i,42), i=1,3) / 0.2518190000E+04, -0.4622460000E-02, 0.8516310000E-03 /
  data (epsData(153,i,42), i=1,3) / 0.2520090000E+04, -0.4258430000E-02, 0.1311620000E-02 /
  data (epsData(154,i,42), i=1,3) / 0.2521140000E+04, -0.4318070000E-02, 0.1670300000E-02 /
  data (epsData(155,i,42), i=1,3) / 0.2523820000E+04, -0.4624100000E-02, 0.2104720000E-02 /
  data (epsData(156,i,42), i=1,3) / 0.2530660000E+04, -0.5211780000E-02, 0.2551510000E-02 /
  data (epsData(157,i,42), i=1,3) / 0.2536920000E+04, -0.5653590000E-02, 0.2529020000E-02 /
  data (epsData(158,i,42), i=1,3) / 0.2552690000E+04, -0.5945780000E-02, 0.2346770000E-02 /
  data (epsData(159,i,42), i=1,3) / 0.2588880000E+04, -0.6030840000E-02, 0.2197830000E-02 /
  data (epsData(160,i,42), i=1,3) / 0.2617360000E+04, -0.5700690000E-02, 0.2137550000E-02 /
  data (epsData(161,i,42), i=1,3) / 0.2623270000E+04, -0.5417340000E-02, 0.2191590000E-02 /
  data (epsData(162,i,42), i=1,3) / 0.2625630000E+04, -0.5265590000E-02, 0.2442850000E-02 /
  data (epsData(163,i,42), i=1,3) / 0.2631960000E+04, -0.5524310000E-02, 0.2816820000E-02 /
  data (epsData(164,i,42), i=1,3) / 0.2641610000E+04, -0.5894440000E-02, 0.2884530000E-02 /
  data (epsData(165,i,42), i=1,3) / 0.2675160000E+04, -0.6172600000E-02, 0.2712030000E-02 /
  data (epsData(166,i,42), i=1,3) / 0.2784140000E+04, -0.6159080000E-02, 0.2342400000E-02 /
  data (epsData(167,i,42), i=1,3) / 0.2855390000E+04, -0.5852890000E-02, 0.2172420000E-02 /
  data (epsData(168,i,42), i=1,3) / 0.2867110000E+04, -0.5725580000E-02, 0.2320350000E-02 /
  data (epsData(169,i,42), i=1,3) / 0.2882020000E+04, -0.5859810000E-02, 0.2345160000E-02 /
  data (epsData(170,i,42), i=1,3) / 0.3046770000E+04, -0.5709610000E-02, 0.1958900000E-02 /
  data (epsData(171,i,42), i=1,3) / 0.3534610000E+04, -0.4596150000E-02, 0.1163610000E-02 /
  data (epsData(172,i,42), i=1,3) / 0.3970550000E+04, -0.3743730000E-02, 0.7714250000E-03 /
  data (epsData(173,i,42), i=1,3) / 0.4445690000E+04, -0.3034070000E-02, 0.5194320000E-03 /
  data (epsData(174,i,42), i=1,3) / 0.4959780000E+04, -0.2458950000E-02, 0.3499420000E-03 /
  data (epsData(175,i,42), i=1,3) / 0.5527110000E+04, -0.1988010000E-02, 0.2358630000E-03 /
  data (epsData(176,i,42), i=1,3) / 0.6153910000E+04, -0.1605370000E-02, 0.1591560000E-03 /
  data (epsData(177,i,42), i=1,3) / 0.6870130000E+04, -0.1287310000E-02, 0.1059530000E-03 /
  data (epsData(178,i,42), i=1,3) / 0.7664500000E+04, -0.1032390000E-02, 0.7063060000E-04 /
  data (epsData(179,i,42), i=1,3) / 0.8536240000E+04, -0.8301120000E-03, 0.4711900000E-04 /
  data (epsData(180,i,42), i=1,3) / 0.9519760000E+04, -0.6651460000E-03, 0.3126220000E-04 /
  data (epsData(181,i,42), i=1,3) / 0.1062080000E+05, -0.5323520000E-03, 0.2063770000E-04 /
  data (epsData(182,i,42), i=1,3) / 0.1185500000E+05, -0.4254800000E-03, 0.1358660000E-04 /
  data (epsData(183,i,42), i=1,3) / 0.1324030000E+05, -0.3394730000E-03, 0.8914770000E-05 /
  data (epsData(184,i,42), i=1,3) / 0.1479730000E+05, -0.2702140000E-03, 0.5816270000E-05 /
  data (epsData(185,i,42), i=1,3) / 0.1610420000E+05, -0.2268000000E-03, 0.4199230000E-05 /
  data (epsData(186,i,42), i=1,3) / 0.1610500000E+05, -0.2185060000E-03, 0.4198430000E-05 /
  data (epsData(187,i,42), i=1,3) / 0.1610500000E+05, -0.2267750000E-03, 0.4198380000E-05 /
  data (epsData(188,i,42), i=1,3) / 0.1807410000E+05, -0.1776080000E-03, 0.2690860000E-05 /
  data (epsData(189,i,42), i=1,3) / 0.1984530000E+05, -0.1395130000E-03, 0.1872830000E-05 /
  data (epsData(190,i,42), i=1,3) / 0.1997260000E+05, -0.1318710000E-03, 0.1827460000E-05 /
  data (epsData(191,i,42), i=1,3) / 0.1999330000E+05, -0.1268050000E-03, 0.3211870000E-05 /
  data (epsData(192,i,42), i=1,3) / 0.1999790000E+05, -0.1243350000E-03, 0.5275820000E-05 /
  data (epsData(193,i,42), i=1,3) / 0.2000000000E+05, -0.1068020000E-03, 0.7774970000E-05 /
  data (epsData(194,i,42), i=1,3) / 0.2000000000E+05, -0.1236780000E-03, 0.7841160000E-05 /
  data (epsData(195,i,42), i=1,3) / 0.2000570000E+05, -0.1274890000E-03, 0.1132750000E-04 /
  data (epsData(196,i,42), i=1,3) / 0.2005750000E+05, -0.1332770000E-03, 0.1206990000E-04 /
  data (epsData(197,i,42), i=1,3) / 0.2035460000E+05, -0.1357120000E-03, 0.1166170000E-04 /
  data (epsData(198,i,42), i=1,3) / 0.2461380000E+05, -0.9887970000E-04, 0.5892140000E-05 /
  data (epsData(199,i,42), i=1,3) / 0.2768030000E+05, -0.7889290000E-04, 0.3815270000E-05 /
  data (epsData(200,i,42), i=1,3) / 0.3103820000E+05, -0.6302200000E-04, 0.2484340000E-05 /
  data (epsData(201,i,42), i=1,3) / 0.3476530000E+05, -0.5034010000E-04, 0.1616690000E-05 /
  data (epsData(202,i,42), i=1,3) / 0.3892820000E+05, -0.4018220000E-04, 0.1048400000E-05 /
  data (epsData(203,i,42), i=1,3) / 0.4358480000E+05, -0.3205630000E-04, 0.6771850000E-06 /
  data (epsData(204,i,42), i=1,3) / 0.4880300000E+05, -0.2555710000E-04, 0.4354710000E-06 /
  data (epsData(205,i,42), i=1,3) / 0.5465200000E+05, -0.2036570000E-04, 0.2786230000E-06 /
  data (epsData(206,i,42), i=1,3) / 0.6122090000E+05, -0.1621620000E-04, 0.1773730000E-06 /
  data (epsData(207,i,42), i=1,3) / 0.6860340000E+05, -0.1290480000E-04, 0.1127070000E-06 /
  data (epsData(208,i,42), i=1,3) / 0.7686510000E+05, -0.1027250000E-04, 0.7098590000E-07 /
  data (epsData(209,i,42), i=1,3) / 0.8613810000E+05, -0.8176030000E-05, 0.4448690000E-07 /
  data (epsData(210,i,42), i=1,3) / 0.9645740000E+05, -0.6523470000E-05, 0.2790520000E-07 /
  data (epsData(211,i,42), i=1,3) / 0.1000000000E+06, -0.6117330000E-05, 0.2405220000E-07 /
  data (epsData(1,i,43), i=1,3) / 0.2506580000E-02, -0.5626100000E+05, 0.1068880000E+06 /
  data (epsData(2,i,43), i=1,3) / 0.4693440000E-02, -0.4177700000E+05, 0.3250820000E+05 /
  data (epsData(3,i,43), i=1,3) / 0.7560590000E-02, -0.1911060000E+05, 0.1354890000E+05 /
  data (epsData(4,i,43), i=1,3) / 0.1110800000E-01, -0.9392530000E+04, 0.6862330000E+04 /
  data (epsData(5,i,43), i=1,3) / 0.1533570000E-01, -0.4854300000E+04, 0.3966340000E+04 /
  data (epsData(6,i,43), i=1,3) / 0.2024370000E-01, -0.2515750000E+04, 0.2520130000E+04 /
  data (epsData(7,i,43), i=1,3) / 0.2583200000E-01, -0.1209970000E+04, 0.1718240000E+04 /
  data (epsData(8,i,43), i=1,3) / 0.3210060000E-01, -0.4325350000E+03, 0.1236570000E+04 /
  data (epsData(9,i,43), i=1,3) / 0.3904940000E-01, 0.5508460000E+02, 0.9284330000E+03 /
  data (epsData(10,i,43), i=1,3) / 0.4667860000E-01, 0.3744730000E+03, 0.7210510000E+03 /
  data (epsData(11,i,43), i=1,3) / 0.5498800000E-01, 0.5915680000E+03, 0.5755580000E+03 /
  data (epsData(12,i,43), i=1,3) / 0.6397770000E-01, 0.7440450000E+03, 0.4699000000E+03 /
  data (epsData(13,i,43), i=1,3) / 0.7364770000E-01, 0.8544110000E+03, 0.3909100000E+03 /
  data (epsData(14,i,43), i=1,3) / 0.8399800000E-01, 0.9366470000E+03, 0.3303820000E+03 /
  data (epsData(15,i,43), i=1,3) / 0.9502860000E-01, 0.9997450000E+03, 0.2830090000E+03 /
  data (epsData(16,i,43), i=1,3) / 0.1191310000E+00, 0.1090580000E+04, 0.2146620000E+03 /
  data (epsData(17,i,43), i=1,3) / 0.1603860000E+00, 0.1185010000E+04, 0.1510920000E+03 /
  data (epsData(18,i,43), i=1,3) / 0.2249170000E+00, 0.1321710000E+04, 0.1028070000E+03 /
  data (epsData(19,i,43), i=1,3) / 0.2427500000E+00, 0.1367930000E+04, 0.1275940000E+03 /
  data (epsData(20,i,43), i=1,3) / 0.2804580000E+00, 0.1408500000E+04, 0.1925440000E+03 /
  data (epsData(21,i,43), i=1,3) / 0.3640360000E+00, 0.1461150000E+04, 0.2418450000E+03 /
  data (epsData(22,i,43), i=1,3) / 0.4338630000E+00, 0.1581790000E+04, 0.2452940000E+03 /
  data (epsData(23,i,43), i=1,3) / 0.4838160000E+00, 0.1774750000E+04, 0.2404210000E+03 /
  data (epsData(24,i,43), i=1,3) / 0.5098130000E+00, 0.1975270000E+04, 0.2367570000E+03 /
  data (epsData(25,i,43), i=1,3) / 0.5364900000E+00, 0.2298450000E+04, 0.3390310000E+03 /
  data (epsData(26,i,43), i=1,3) / 0.5638470000E+00, 0.2495760000E+04, 0.9117220000E+03 /
  data (epsData(27,i,43), i=1,3) / 0.5918850000E+00, 0.2406310000E+04, 0.1389900000E+04 /
  data (epsData(28,i,43), i=1,3) / 0.6206030000E+00, 0.2145330000E+04, 0.1788300000E+04 /
  data (epsData(29,i,43), i=1,3) / 0.6500010000E+00, 0.1812970000E+04, 0.1974840000E+04 /
  data (epsData(30,i,43), i=1,3) / 0.6800800000E+00, 0.1522960000E+04, 0.2040690000E+04 /
  data (epsData(31,i,43), i=1,3) / 0.7743970000E+00, 0.8324680000E+03, 0.2146230000E+04 /
  data (epsData(32,i,43), i=1,3) / 0.8071970000E+00, 0.5899690000E+03, 0.2158550000E+04 /
  data (epsData(33,i,43), i=1,3) / 0.8406760000E+00, 0.3231250000E+03, 0.2113460000E+04 /
  data (epsData(34,i,43), i=1,3) / 0.8748370000E+00, 0.1032240000E+03, 0.1935970000E+04 /
  data (epsData(35,i,43), i=1,3) / 0.9096770000E+00, -0.3449940000E+02, 0.1775740000E+04 /
  data (epsData(36,i,43), i=1,3) / 0.9451980000E+00, -0.1299570000E+03, 0.1630840000E+04 /
  data (epsData(37,i,43), i=1,3) / 0.9813990000E+00, -0.1989510000E+03, 0.1499550000E+04 /
  data (epsData(38,i,43), i=1,3) / 0.1055840000E+01, -0.2858950000E+03, 0.1272130000E+04 /
  data (epsData(39,i,43), i=1,3) / 0.1133010000E+01, -0.3301330000E+03, 0.1083650000E+04 /
  data (epsData(40,i,43), i=1,3) / 0.1253850000E+01, -0.3501470000E+03, 0.8564490000E+03 /
  data (epsData(41,i,43), i=1,3) / 0.1380830000E+01, -0.3383550000E+03, 0.6816020000E+03 /
  data (epsData(42,i,43), i=1,3) / 0.1513920000E+01, -0.3092940000E+03, 0.5507310000E+03 /
  data (epsData(43,i,43), i=1,3) / 0.1700900000E+01, -0.2671220000E+03, 0.4217420000E+03 /
  data (epsData(44,i,43), i=1,3) / 0.1898770000E+01, -0.2240880000E+03, 0.3323610000E+03 /
  data (epsData(45,i,43), i=1,3) / 0.2107520000E+01, -0.1861240000E+03, 0.2671740000E+03 /
  data (epsData(46,i,43), i=1,3) / 0.2383760000E+01, -0.1473020000E+03, 0.2094230000E+03 /
  data (epsData(47,i,43), i=1,3) / 0.2677010000E+01, -0.1163340000E+03, 0.1680730000E+03 /
  data (epsData(48,i,43), i=1,3) / 0.2987270000E+01, -0.9105010000E+02, 0.1370700000E+03 /
  data (epsData(49,i,43), i=1,3) / 0.3314540000E+01, -0.6967510000E+02, 0.1134830000E+03 /
  data (epsData(50,i,43), i=1,3) / 0.3658810000E+01, -0.5126080000E+02, 0.9579860000E+02 /
  data (epsData(51,i,43), i=1,3) / 0.4020090000E+01, -0.3540730000E+02, 0.8292550000E+02 /
  data (epsData(52,i,43), i=1,3) / 0.4321360000E+01, -0.2451480000E+02, 0.7595040000E+02 /
  data (epsData(53,i,43), i=1,3) / 0.4713260000E+01, -0.1469010000E+02, 0.6953550000E+02 /
  data (epsData(54,i,43), i=1,3) / 0.5290480000E+01, -0.2237440000E+01, 0.5929910000E+02 /
  data (epsData(55,i,43), i=1,3) / 0.5723190000E+01, 0.8674650000E+01, 0.5494880000E+02 /
  data (epsData(56,i,43), i=1,3) / 0.5990980000E+01, 0.1629220000E+02, 0.5342830000E+02 /
  data (epsData(57,i,43), i=1,3) / 0.6172910000E+01, 0.2291050000E+02, 0.5322070000E+02 /
  data (epsData(58,i,43), i=1,3) / 0.6264890000E+01, 0.2786820000E+02, 0.5319500000E+02 /
  data (epsData(59,i,43), i=1,3) / 0.6450900000E+01, 0.3726700000E+02, 0.6063100000E+02 /
  data (epsData(60,i,43), i=1,3) / 0.6544930000E+01, 0.4391400000E+02, 0.6404910000E+02 /
  data (epsData(61,i,43), i=1,3) / 0.6639630000E+01, 0.4689940000E+02, 0.8070170000E+02 /
  data (epsData(62,i,43), i=1,3) / 0.6735020000E+01, 0.4132250000E+02, 0.9361590000E+02 /
  data (epsData(63,i,43), i=1,3) / 0.6831080000E+01, 0.2992600000E+02, 0.1039680000E+03 /
  data (epsData(64,i,43), i=1,3) / 0.6927830000E+01, 0.1679180000E+02, 0.1042200000E+03 /
  data (epsData(65,i,43), i=1,3) / 0.7025260000E+01, 0.7194180000E+01, 0.9977920000E+02 /
  data (epsData(66,i,43), i=1,3) / 0.7123360000E+01, 0.7262600000E+00, 0.9551160000E+02 /
  data (epsData(67,i,43), i=1,3) / 0.7321620000E+01, -0.7361200000E+01, 0.8560260000E+02 /
  data (epsData(68,i,43), i=1,3) / 0.7522590000E+01, -0.1018530000E+02, 0.7652010000E+02 /
  data (epsData(69,i,43), i=1,3) / 0.7726290000E+01, -0.1066860000E+02, 0.6926900000E+02 /
  data (epsData(70,i,43), i=1,3) / 0.8036940000E+01, -0.9244070000E+01, 0.6125450000E+02 /
  data (epsData(71,i,43), i=1,3) / 0.8460660000E+01, -0.6164630000E+01, 0.5413490000E+02 /
  data (epsData(72,i,43), i=1,3) / 0.9005620000E+01, -0.2242590000E+01, 0.4831120000E+02 /
  data (epsData(73,i,43), i=1,3) / 0.9682010000E+01, 0.2079820000E+01, 0.4406280000E+02 /
  data (epsData(74,i,43), i=1,3) / 0.1062200000E+02, 0.7136240000E+01, 0.4109090000E+02 /
  data (epsData(75,i,43), i=1,3) / 0.1135550000E+02, 0.1127010000E+02, 0.3988610000E+02 /
  data (epsData(76,i,43), i=1,3) / 0.1160550000E+02, 0.1428370000E+02, 0.3976940000E+02 /
  data (epsData(77,i,43), i=1,3) / 0.1173150000E+02, 0.1744330000E+02, 0.3982400000E+02 /
  data (epsData(78,i,43), i=1,3) / 0.1185810000E+02, 0.1910560000E+02, 0.4603610000E+02 /
  data (epsData(79,i,43), i=1,3) / 0.1198550000E+02, 0.1707670000E+02, 0.5025520000E+02 /
  data (epsData(80,i,43), i=1,3) / 0.1211350000E+02, 0.1356170000E+02, 0.5234050000E+02 /
  data (epsData(81,i,43), i=1,3) / 0.1224230000E+02, 0.1053500000E+02, 0.5146680000E+02 /
  data (epsData(82,i,43), i=1,3) / 0.1263250000E+02, 0.6662680000E+01, 0.4889590000E+02 /
  data (epsData(83,i,43), i=1,3) / 0.1329650000E+02, 0.4706630000E+01, 0.4588600000E+02 /
  data (epsData(84,i,43), i=1,3) / 0.1510250000E+02, 0.2385210000E+01, 0.4158630000E+02 /
  data (epsData(85,i,43), i=1,3) / 0.1794890000E+02, -0.3740900000E+00, 0.3718040000E+02 /
  data (epsData(86,i,43), i=1,3) / 0.2138060000E+02, -0.3086340000E+01, 0.3371780000E+02 /
  data (epsData(87,i,43), i=1,3) / 0.2585710000E+02, -0.8038330000E+01, 0.2970880000E+02 /
  data (epsData(88,i,43), i=1,3) / 0.2954360000E+02, -0.1172270000E+02, 0.2480930000E+02 /
  data (epsData(89,i,43), i=1,3) / 0.3305020000E+02, -0.1319120000E+02, 0.1952570000E+02 /
  data (epsData(90,i,43), i=1,3) / 0.3675350000E+02, -0.1301870000E+02, 0.1493340000E+02 /
  data (epsData(91,i,43), i=1,3) / 0.4001020000E+02, -0.1207110000E+02, 0.1217010000E+02 /
  data (epsData(92,i,43), i=1,3) / 0.4915860000E+02, -0.1092990000E+02, 0.6599550000E+01 /
  data (epsData(93,i,43), i=1,3) / 0.5445020000E+02, -0.9511010000E+01, 0.4058440000E+01 /
  data (epsData(94,i,43), i=1,3) / 0.5906360000E+02, -0.8101080000E+01, 0.2604000000E+01 /
  data (epsData(95,i,43), i=1,3) / 0.6338910000E+02, -0.6892460000E+01, 0.1733740000E+01 /
  data (epsData(96,i,43), i=1,3) / 0.6858750000E+02, -0.5691240000E+01, 0.1094040000E+01 /
  data (epsData(97,i,43), i=1,3) / 0.7481470000E+02, -0.4588250000E+01, 0.6630800000E+00 /
  data (epsData(98,i,43), i=1,3) / 0.8126760000E+02, -0.3712840000E+01, 0.4163850000E+00 /
  data (epsData(99,i,43), i=1,3) / 0.8812480000E+02, -0.3015080000E+01, 0.2862420000E+00 /
  data (epsData(100,i,43), i=1,3) / 0.9559870000E+02, -0.2450290000E+01, 0.2170570000E+00 /
  data (epsData(101,i,43), i=1,3) / 0.1041130000E+03, -0.1980890000E+01, 0.1833160000E+00 /
  data (epsData(102,i,43), i=1,3) / 0.1143200000E+03, -0.1579450000E+01, 0.1640520000E+00 /
  data (epsData(103,i,43), i=1,3) / 0.1261210000E+03, -0.1251590000E+01, 0.1525750000E+00 /
  data (epsData(104,i,43), i=1,3) / 0.1399180000E+03, -0.9841100000E+00, 0.1427830000E+00 /
  data (epsData(105,i,43), i=1,3) / 0.1559020000E+03, -0.7671050000E+00, 0.1295880000E+00 /
  data (epsData(106,i,43), i=1,3) / 0.1746520000E+03, -0.5869750000E+00, 0.1141660000E+00 /
  data (epsData(107,i,43), i=1,3) / 0.1960690000E+03, -0.4377680000E+00, 0.9698270000E-01 /
  data (epsData(108,i,43), i=1,3) / 0.2221930000E+03, -0.3001970000E+00, 0.7886340000E-01 /
  data (epsData(109,i,43), i=1,3) / 0.2487840000E+03, -0.1734950000E+00, 0.6402550000E-01 /
  data (epsData(110,i,43), i=1,3) / 0.2527570000E+03, -0.1469620000E+00, 0.6243370000E-01 /
  data (epsData(111,i,43), i=1,3) / 0.2532570000E+03, -0.1412370000E+00, 0.6256860000E-01 /
  data (epsData(112,i,43), i=1,3) / 0.2537570000E+03, -0.1331590000E+00, 0.6547920000E-01 /
  data (epsData(113,i,43), i=1,3) / 0.2540960000E+03, -0.1323260000E+00, 0.7554350000E-01 /
  data (epsData(114,i,43), i=1,3) / 0.2546840000E+03, -0.1371490000E+00, 0.7605280000E-01 /
  data (epsData(115,i,43), i=1,3) / 0.2569700000E+03, -0.1276170000E+00, 0.7344650000E-01 /
  data (epsData(116,i,43), i=1,3) / 0.2575090000E+03, -0.1206880000E+00, 0.7538670000E-01 /
  data (epsData(117,i,43), i=1,3) / 0.2578300000E+03, -0.1213140000E+00, 0.8188870000E-01 /
  data (epsData(118,i,43), i=1,3) / 0.2587050000E+03, -0.1236970000E+00, 0.8078440000E-01 /
  data (epsData(119,i,43), i=1,3) / 0.2617160000E+03, -0.1105250000E+00, 0.7702030000E-01 /
  data (epsData(120,i,43), i=1,3) / 0.2654590000E+03, -0.8631520000E-01, 0.7883010000E-01 /
  data (epsData(121,i,43), i=1,3) / 0.2697500000E+03, -0.5769520000E-01, 0.9071240000E-01 /
  data (epsData(122,i,43), i=1,3) / 0.2746670000E+03, -0.2962850000E-01, 0.1123330000E+00 /
  data (epsData(123,i,43), i=1,3) / 0.2788590000E+03, -0.9593780000E-02, 0.1364970000E+00 /
  data (epsData(124,i,43), i=1,3) / 0.2825800000E+03, 0.2606820000E-02, 0.1629870000E+00 /
  data (epsData(125,i,43), i=1,3) / 0.2870010000E+03, 0.6020410000E-02, 0.1966130000E+00 /
  data (epsData(126,i,43), i=1,3) / 0.2913510000E+03, -0.1361240000E-02, 0.2252990000E+00 /
  data (epsData(127,i,43), i=1,3) / 0.2980850000E+03, -0.2142160000E-01, 0.2540970000E+00 /
  data (epsData(128,i,43), i=1,3) / 0.3096010000E+03, -0.5608330000E-01, 0.2796610000E+00 /
  data (epsData(129,i,43), i=1,3) / 0.3225570000E+03, -0.9325910000E-01, 0.2850780000E+00 /
  data (epsData(130,i,43), i=1,3) / 0.3376110000E+03, -0.1234550000E+00, 0.2714090000E+00 /
  data (epsData(131,i,43), i=1,3) / 0.3704530000E+03, -0.1502800000E+00, 0.2306130000E+00 /
  data (epsData(132,i,43), i=1,3) / 0.3995490000E+03, -0.1511530000E+00, 0.1942410000E+00 /
  data (epsData(133,i,43), i=1,3) / 0.4158110000E+03, -0.1406940000E+00, 0.1787580000E+00 /
  data (epsData(134,i,43), i=1,3) / 0.4239910000E+03, -0.1388440000E+00, 0.1826520000E+00 /
  data (epsData(135,i,43), i=1,3) / 0.4369670000E+03, -0.1452110000E+00, 0.1741530000E+00 /
  data (epsData(136,i,43), i=1,3) / 0.4540780000E+03, -0.1436530000E+00, 0.1638440000E+00 /
  data (epsData(137,i,43), i=1,3) / 0.4946700000E+03, -0.1432120000E+00, 0.1332380000E+00 /
  data (epsData(138,i,43), i=1,3) / 0.5388690000E+03, -0.1312650000E+00, 0.1068180000E+00 /
  data (epsData(139,i,43), i=1,3) / 0.6300600000E+03, -0.1125080000E+00, 0.7098820000E-01 /
  data (epsData(140,i,43), i=1,3) / 0.7222980000E+03, -0.9256310000E-01, 0.4735540000E-01 /
  data (epsData(141,i,43), i=1,3) / 0.8213620000E+03, -0.7511440000E-01, 0.3191420000E-01 /
  data (epsData(142,i,43), i=1,3) / 0.9247540000E+03, -0.6103240000E-01, 0.2194130000E-01 /
  data (epsData(143,i,43), i=1,3) / 0.1035450000E+04, -0.4957900000E-01, 0.1522930000E-01 /
  data (epsData(144,i,43), i=1,3) / 0.1152960000E+04, -0.4040470000E-01, 0.1069340000E-01 /
  data (epsData(145,i,43), i=1,3) / 0.1281040000E+04, -0.3288230000E-01, 0.7525390000E-02 /
  data (epsData(146,i,43), i=1,3) / 0.1420250000E+04, -0.2675850000E-01, 0.5318050000E-02 /
  data (epsData(147,i,43), i=1,3) / 0.1572670000E+04, -0.2174660000E-01, 0.3764330000E-02 /
  data (epsData(148,i,43), i=1,3) / 0.1739860000E+04, -0.1764250000E-01, 0.2659320000E-02 /
  data (epsData(149,i,43), i=1,3) / 0.1923730000E+04, -0.1422890000E-01, 0.1875080000E-02 /
  data (epsData(150,i,43), i=1,3) / 0.2139860000E+04, -0.1124580000E-01, 0.1298090000E-02 /
  data (epsData(151,i,43), i=1,3) / 0.2398600000E+04, -0.8524010000E-02, 0.8700670000E-03 /
  data (epsData(152,i,43), i=1,3) / 0.2607510000E+04, -0.6438060000E-02, 0.6480850000E-03 /
  data (epsData(153,i,43), i=1,3) / 0.2653000000E+04, -0.5679680000E-02, 0.6096150000E-03 /
  data (epsData(154,i,43), i=1,3) / 0.2668270000E+04, -0.5174440000E-02, 0.6336350000E-03 /
  data (epsData(155,i,43), i=1,3) / 0.2673370000E+04, -0.4840760000E-02, 0.6670950000E-03 /
  data (epsData(156,i,43), i=1,3) / 0.2675080000E+04, -0.4659270000E-02, 0.7057010000E-03 /
  data (epsData(157,i,43), i=1,3) / 0.2677470000E+04, -0.4355070000E-02, 0.9675130000E-03 /
  data (epsData(158,i,43), i=1,3) / 0.2680330000E+04, -0.4314330000E-02, 0.1439170000E-02 /
  data (epsData(159,i,43), i=1,3) / 0.2684760000E+04, -0.4545150000E-02, 0.1947880000E-02 /
  data (epsData(160,i,43), i=1,3) / 0.2689800000E+04, -0.4969190000E-02, 0.2139250000E-02 /
  data (epsData(161,i,43), i=1,3) / 0.2701430000E+04, -0.5358610000E-02, 0.2080800000E-02 /
  data (epsData(162,i,43), i=1,3) / 0.2734180000E+04, -0.5613370000E-02, 0.1987130000E-02 /
  data (epsData(163,i,43), i=1,3) / 0.2779520000E+04, -0.5419570000E-02, 0.1872710000E-02 /
  data (epsData(164,i,43), i=1,3) / 0.2791030000E+04, -0.5145090000E-02, 0.1885570000E-02 /
  data (epsData(165,i,43), i=1,3) / 0.2795070000E+04, -0.4989410000E-02, 0.2105180000E-02 /
  data (epsData(166,i,43), i=1,3) / 0.2802150000E+04, -0.5158400000E-02, 0.2446070000E-02 /
  data (epsData(167,i,43), i=1,3) / 0.2813250000E+04, -0.5445610000E-02, 0.2461420000E-02 /
  data (epsData(168,i,43), i=1,3) / 0.2865600000E+04, -0.5701900000E-02, 0.2302680000E-02 /
  data (epsData(169,i,43), i=1,3) / 0.2999950000E+04, -0.5535770000E-02, 0.1957820000E-02 /
  data (epsData(170,i,43), i=1,3) / 0.3038180000E+04, -0.5289590000E-02, 0.1932930000E-02 /
  data (epsData(171,i,43), i=1,3) / 0.3046820000E+04, -0.5305540000E-02, 0.2097400000E-02 /
  data (epsData(172,i,43), i=1,3) / 0.3128380000E+04, -0.5381710000E-02, 0.1938960000E-02 /
  data (epsData(173,i,43), i=1,3) / 0.3470450000E+04, -0.4753890000E-02, 0.1350460000E-02 /
  data (epsData(174,i,43), i=1,3) / 0.3932400000E+04, -0.3867130000E-02, 0.8686010000E-03 /
  data (epsData(175,i,43), i=1,3) / 0.4376540000E+04, -0.3184590000E-02, 0.5975320000E-03 /
  data (epsData(176,i,43), i=1,3) / 0.4892840000E+04, -0.2580430000E-02, 0.4014350000E-03 /
  data (epsData(177,i,43), i=1,3) / 0.5455240000E+04, -0.2089220000E-02, 0.2706150000E-03 /
  data (epsData(178,i,43), i=1,3) / 0.6077930000E+04, -0.1687430000E-02, 0.1822690000E-03 /
  data (epsData(179,i,43), i=1,3) / 0.6781000000E+04, -0.1356660000E-02, 0.1218510000E-03 /
  data (epsData(180,i,43), i=1,3) / 0.7564590000E+04, -0.1088650000E-02, 0.8135950000E-04 /
  data (epsData(181,i,43), i=1,3) / 0.8428100000E+04, -0.8751630000E-03, 0.5436780000E-04 /
  data (epsData(182,i,43), i=1,3) / 0.9396390000E+04, -0.7019430000E-03, 0.3608930000E-04 /
  data (epsData(183,i,43), i=1,3) / 0.1047960000E+05, -0.5624270000E-03, 0.2388540000E-04 /
  data (epsData(184,i,43), i=1,3) / 0.1169600000E+05, -0.4497720000E-03, 0.1575270000E-04 /
  data (epsData(185,i,43), i=1,3) / 0.1305980000E+05, -0.3591690000E-03, 0.1034930000E-04 /
  data (epsData(186,i,43), i=1,3) / 0.1459070000E+05, -0.2863210000E-03, 0.6777140000E-05 /
  data (epsData(187,i,43), i=1,3) / 0.1631090000E+05, -0.2276500000E-03, 0.4416210000E-05 /
  data (epsData(188,i,43), i=1,3) / 0.1703420000E+05, -0.2080610000E-03, 0.3736680000E-05 /
  data (epsData(189,i,43), i=1,3) / 0.1703800000E+05, -0.1987700000E-03, 0.3733460000E-05 /
  data (epsData(190,i,43), i=1,3) / 0.1703800000E+05, -0.2079630000E-03, 0.3733420000E-05 /
  data (epsData(191,i,43), i=1,3) / 0.1912670000E+05, -0.1627480000E-03, 0.2387050000E-05 /
  data (epsData(192,i,43), i=1,3) / 0.2088960000E+05, -0.1294030000E-03, 0.1696470000E-05 /
  data (epsData(193,i,43), i=1,3) / 0.2101650000E+05, -0.1225120000E-03, 0.1656440000E-05 /
  data (epsData(194,i,43), i=1,3) / 0.2103750000E+05, -0.1177080000E-03, 0.3137000000E-05 /
  data (epsData(195,i,43), i=1,3) / 0.2104210000E+05, -0.1155310000E-03, 0.5476060000E-05 /
  data (epsData(196,i,43), i=1,3) / 0.2104400000E+05, -0.9879580000E-04, 0.7655440000E-05 /
  data (epsData(197,i,43), i=1,3) / 0.2104400000E+05, -0.1152250000E-03, 0.7715410000E-05 /
  data (epsData(198,i,43), i=1,3) / 0.2105000000E+05, -0.1191450000E-03, 0.1028160000E-04 /
  data (epsData(199,i,43), i=1,3) / 0.2110720000E+05, -0.1241250000E-03, 0.1088950000E-04 /
  data (epsData(200,i,43), i=1,3) / 0.2146080000E+05, -0.1258800000E-03, 0.1040610000E-04 /
  data (epsData(201,i,43), i=1,3) / 0.2579150000E+05, -0.9237450000E-04, 0.5377630000E-05 /
  data (epsData(202,i,43), i=1,3) / 0.2900810000E+05, -0.7368040000E-04, 0.3479560000E-05 /
  data (epsData(203,i,43), i=1,3) / 0.3252840000E+05, -0.5885060000E-04, 0.2265320000E-05 /
  data (epsData(204,i,43), i=1,3) / 0.3643450000E+05, -0.4700670000E-04, 0.1473450000E-05 /
  data (epsData(205,i,43), i=1,3) / 0.4079820000E+05, -0.3751930000E-04, 0.9551640000E-06 /
  data (epsData(206,i,43), i=1,3) / 0.4567700000E+05, -0.2993410000E-04, 0.6169530000E-06 /
  data (epsData(207,i,43), i=1,3) / 0.5114480000E+05, -0.2386640000E-04, 0.3964770000E-06 /
  data (epsData(208,i,43), i=1,3) / 0.5727250000E+05, -0.1902030000E-04, 0.2537000000E-06 /
  data (epsData(209,i,43), i=1,3) / 0.6412820000E+05, -0.1516020000E-04, 0.1623140000E-06 /
  data (epsData(210,i,43), i=1,3) / 0.7184330000E+05, -0.1207000000E-04, 0.1026200000E-06 /
  data (epsData(211,i,43), i=1,3) / 0.8049610000E+05, -0.9608930000E-05, 0.6463280000E-07 /
  data (epsData(212,i,43), i=1,3) / 0.9020230000E+05, -0.7650440000E-05, 0.4056400000E-07 /
  data (epsData(213,i,43), i=1,3) / 0.9990540000E+05, -0.6267620000E-05, 0.2661660000E-07 /
  data (epsData(214,i,43), i=1,3) / 0.1000000000E+06, -0.6294880000E-05, 0.2651000000E-07 /
  data (epsData(1,i,44), i=1,3) / 0.2506580000E-02, -0.7245710000E+05, 0.1323170000E+06 /
  data (epsData(2,i,44), i=1,3) / 0.4693440000E-02, -0.5279120000E+05, 0.3937760000E+05 /
  data (epsData(3,i,44), i=1,3) / 0.7560590000E-02, -0.2375730000E+05, 0.1600230000E+05 /
  data (epsData(4,i,44), i=1,3) / 0.1110800000E-01, -0.1133900000E+05, 0.7887830000E+04 /
  data (epsData(5,i,44), i=1,3) / 0.1533570000E-01, -0.5543450000E+04, 0.4434900000E+04 /
  data (epsData(6,i,44), i=1,3) / 0.2024370000E-01, -0.2555140000E+04, 0.2742750000E+04 /
  data (epsData(7,i,44), i=1,3) / 0.2583200000E-01, -0.8819370000E+03, 0.1822600000E+04 /
  data (epsData(8,i,44), i=1,3) / 0.3210060000E-01, 0.1214560000E+03, 0.1280650000E+04 /
  data (epsData(9,i,44), i=1,3) / 0.3904940000E-01, 0.7613520000E+03, 0.9406860000E+03 /
  data (epsData(10,i,44), i=1,3) / 0.4667860000E-01, 0.1196340000E+04, 0.7161540000E+03 /
  data (epsData(11,i,44), i=1,3) / 0.5498800000E-01, 0.1518830000E+04, 0.5614560000E+03 /
  data (epsData(12,i,44), i=1,3) / 0.6397770000E-01, 0.1801260000E+04, 0.4510330000E+03 /
  data (epsData(13,i,44), i=1,3) / 0.7364770000E-01, 0.2061020000E+04, 0.4627330000E+03 /
  data (epsData(14,i,44), i=1,3) / 0.8399800000E-01, 0.2193540000E+04, 0.6591840000E+03 /
  data (epsData(15,i,44), i=1,3) / 0.9502860000E-01, 0.2182670000E+04, 0.7677200000E+03 /
  data (epsData(16,i,44), i=1,3) / 0.1191310000E+00, 0.2065720000E+04, 0.8398160000E+03 /
  data (epsData(17,i,44), i=1,3) / 0.1603860000E+00, 0.1908270000E+04, 0.7951090000E+03 /
  data (epsData(18,i,44), i=1,3) / 0.2249170000E+00, 0.1839880000E+04, 0.6681960000E+03 /
  data (epsData(19,i,44), i=1,3) / 0.2804580000E+00, 0.1922130000E+04, 0.5758360000E+03 /
  data (epsData(20,i,44), i=1,3) / 0.3208860000E+00, 0.2090970000E+04, 0.5211400000E+03 /
  data (epsData(21,i,44), i=1,3) / 0.3421210000E+00, 0.2257380000E+04, 0.4960050000E+03 /
  data (epsData(22,i,44), i=1,3) / 0.3640360000E+00, 0.2578710000E+04, 0.4723110000E+03 /
  data (epsData(23,i,44), i=1,3) / 0.3866320000E+00, 0.2885860000E+04, 0.9204330000E+03 /
  data (epsData(24,i,44), i=1,3) / 0.4099070000E+00, 0.2922990000E+04, 0.1470720000E+04 /
  data (epsData(25,i,44), i=1,3) / 0.4338630000E+00, 0.2750790000E+04, 0.1911650000E+04 /
  data (epsData(26,i,44), i=1,3) / 0.4585000000E+00, 0.2455180000E+04, 0.2263190000E+04 /
  data (epsData(27,i,44), i=1,3) / 0.4838160000E+00, 0.2103450000E+04, 0.2449340000E+04 /
  data (epsData(28,i,44), i=1,3) / 0.5364900000E+00, 0.1516220000E+04, 0.2545230000E+04 /
  data (epsData(29,i,44), i=1,3) / 0.6500010000E+00, 0.5419720000E+03, 0.2546090000E+04 /
  data (epsData(30,i,44), i=1,3) / 0.6800800000E+00, 0.2683290000E+03, 0.2465050000E+04 /
  data (epsData(31,i,44), i=1,3) / 0.7108380000E+00, 0.4763170000E+02, 0.2262340000E+04 /
  data (epsData(32,i,44), i=1,3) / 0.7422780000E+00, -0.9063300000E+02, 0.2080380000E+04 /
  data (epsData(33,i,44), i=1,3) / 0.7743970000E+00, -0.1864330000E+03, 0.1916670000E+04 /
  data (epsData(34,i,44), i=1,3) / 0.8406760000E+00, -0.3067490000E+03, 0.1635610000E+04 /
  data (epsData(35,i,44), i=1,3) / 0.9096770000E+00, -0.3723850000E+03, 0.1405130000E+04 /
  data (epsData(36,i,44), i=1,3) / 0.1018280000E+01, -0.4207820000E+03, 0.1122550000E+04 /
  data (epsData(37,i,44), i=1,3) / 0.1133010000E+01, -0.4202960000E+03, 0.8975480000E+03 /
  data (epsData(38,i,44), i=1,3) / 0.1253850000E+01, -0.3983120000E+03, 0.7249900000E+03 /
  data (epsData(39,i,44), i=1,3) / 0.1424510000E+01, -0.3555690000E+03, 0.5504270000E+03 /
  data (epsData(40,i,44), i=1,3) / 0.1606050000E+01, -0.3075410000E+03, 0.4233840000E+03 /
  data (epsData(41,i,44), i=1,3) / 0.1798470000E+01, -0.2607740000E+03, 0.3295540000E+03 /
  data (epsData(42,i,44), i=1,3) / 0.2001780000E+01, -0.2171850000E+03, 0.2604310000E+03 /
  data (epsData(43,i,44), i=1,3) / 0.2215970000E+01, -0.1783690000E+03, 0.2089040000E+03 /
  data (epsData(44,i,44), i=1,3) / 0.2499020000E+01, -0.1373020000E+03, 0.1629660000E+03 /
  data (epsData(45,i,44), i=1,3) / 0.2799080000E+01, -0.1038150000E+03, 0.1311640000E+03 /
  data (epsData(46,i,44), i=1,3) / 0.3116140000E+01, -0.7684870000E+02, 0.1087260000E+03 /
  data (epsData(47,i,44), i=1,3) / 0.3450210000E+01, -0.5490990000E+02, 0.9276190000E+02 /
  data (epsData(48,i,44), i=1,3) / 0.3801290000E+01, -0.3789190000E+02, 0.8167630000E+02 /
  data (epsData(49,i,44), i=1,3) / 0.4554460000E+01, -0.1201850000E+02, 0.6283560000E+02 /
  data (epsData(50,i,44), i=1,3) / 0.5039020000E+01, 0.4822160000E+01, 0.5506620000E+02 /
  data (epsData(51,i,44), i=1,3) / 0.5290480000E+01, 0.1513250000E+02, 0.5235630000E+02 /
  data (epsData(52,i,44), i=1,3) / 0.5461520000E+01, 0.2408550000E+02, 0.5150600000E+02 /
  data (epsData(53,i,44), i=1,3) / 0.5548060000E+01, 0.2942390000E+02, 0.5144230000E+02 /
  data (epsData(54,i,44), i=1,3) / 0.5635290000E+01, 0.3655710000E+02, 0.5136320000E+02 /
  data (epsData(55,i,44), i=1,3) / 0.5723190000E+01, 0.4468330000E+02, 0.5525110000E+02 /
  data (epsData(56,i,44), i=1,3) / 0.5901040000E+01, 0.6070240000E+02, 0.6693200000E+02 /
  data (epsData(57,i,44), i=1,3) / 0.5990980000E+01, 0.7120970000E+02, 0.7913720000E+02 /
  data (epsData(58,i,44), i=1,3) / 0.6081600000E+01, 0.7203160000E+02, 0.1085870000E+03 /
  data (epsData(59,i,44), i=1,3) / 0.6172910000E+01, 0.6006810000E+02, 0.1276750000E+03 /
  data (epsData(60,i,44), i=1,3) / 0.6264890000E+01, 0.3899250000E+02, 0.1457420000E+03 /
  data (epsData(61,i,44), i=1,3) / 0.6357560000E+01, 0.1713610000E+02, 0.1407320000E+03 /
  data (epsData(62,i,44), i=1,3) / 0.6450900000E+01, 0.3119640000E+01, 0.1352080000E+03 /
  data (epsData(63,i,44), i=1,3) / 0.6544930000E+01, -0.7905260000E+01, 0.1299100000E+03 /
  data (epsData(64,i,44), i=1,3) / 0.6639630000E+01, -0.1685270000E+02, 0.1223860000E+03 /
  data (epsData(65,i,44), i=1,3) / 0.6831080000E+01, -0.2749930000E+02, 0.1068640000E+03 /
  data (epsData(66,i,44), i=1,3) / 0.7025260000E+01, -0.3229650000E+02, 0.9319710000E+02 /
  data (epsData(67,i,44), i=1,3) / 0.7222150000E+01, -0.3351950000E+02, 0.8121370000E+02 /
  data (epsData(68,i,44), i=1,3) / 0.7522590000E+01, -0.3163520000E+02, 0.6692940000E+02 /
  data (epsData(69,i,44), i=1,3) / 0.7829160000E+01, -0.2746560000E+02, 0.5639490000E+02 /
  data (epsData(70,i,44), i=1,3) / 0.8247440000E+01, -0.2081830000E+02, 0.4680020000E+02 /
  data (epsData(71,i,44), i=1,3) / 0.8676600000E+01, -0.1412670000E+02, 0.4044620000E+02 /
  data (epsData(72,i,44), i=1,3) / 0.9228360000E+01, -0.5814000000E+01, 0.3562410000E+02 /
  data (epsData(73,i,44), i=1,3) / 0.9682010000E+01, 0.1383400000E+01, 0.3422990000E+02 /
  data (epsData(74,i,44), i=1,3) / 0.9797130000E+01, 0.3700700000E+01, 0.3421180000E+02 /
  data (epsData(75,i,44), i=1,3) / 0.9912920000E+01, 0.7215370000E+01, 0.3415650000E+02 /
  data (epsData(76,i,44), i=1,3) / 0.1002940000E+02, 0.9443770000E+01, 0.3952650000E+02 /
  data (epsData(77,i,44), i=1,3) / 0.1014660000E+02, 0.8556180000E+01, 0.4314760000E+02 /
  data (epsData(78,i,44), i=1,3) / 0.1026440000E+02, 0.6208550000E+01, 0.4552000000E+02 /
  data (epsData(79,i,44), i=1,3) / 0.1038290000E+02, 0.3977690000E+01, 0.4505860000E+02 /
  data (epsData(80,i,44), i=1,3) / 0.1074250000E+02, 0.1213170000E+01, 0.4340850000E+02 /
  data (epsData(81,i,44), i=1,3) / 0.1135550000E+02, 0.1540390000E+00, 0.4031120000E+02 /
  data (epsData(82,i,44), i=1,3) / 0.1211350000E+02, 0.9598730000E+00, 0.3756680000E+02 /
  data (epsData(83,i,44), i=1,3) / 0.1302890000E+02, 0.2125210000E+01, 0.3610520000E+02 /
  data (epsData(84,i,44), i=1,3) / 0.1687170000E+02, 0.1433260000E+01, 0.3239360000E+02 /
  data (epsData(85,i,44), i=1,3) / 0.2121040000E+02, -0.1683590000E+01, 0.2965300000E+02 /
  data (epsData(86,i,44), i=1,3) / 0.2737940000E+02, -0.7050780000E+01, 0.2455360000E+02 /
  data (epsData(87,i,44), i=1,3) / 0.3179010000E+02, -0.9587340000E+01, 0.1976730000E+02 /
  data (epsData(88,i,44), i=1,3) / 0.3608570000E+02, -0.1037990000E+02, 0.1545410000E+02 /
  data (epsData(89,i,44), i=1,3) / 0.4041840000E+02, -0.1013520000E+02, 0.1193910000E+02 /
  data (epsData(90,i,44), i=1,3) / 0.4395140000E+02, -0.9546600000E+01, 0.1009240000E+02 /
  data (epsData(91,i,44), i=1,3) / 0.5111670000E+02, -0.9360310000E+01, 0.6781660000E+01 /
  data (epsData(92,i,44), i=1,3) / 0.5601480000E+02, -0.8608140000E+01, 0.4607000000E+01 /
  data (epsData(93,i,44), i=1,3) / 0.6130030000E+02, -0.7460850000E+01, 0.2902170000E+01 /
  data (epsData(94,i,44), i=1,3) / 0.6688400000E+02, -0.6218620000E+01, 0.1767540000E+01 /
  data (epsData(95,i,44), i=1,3) / 0.7258750000E+02, -0.5120260000E+01, 0.1085030000E+01 /
  data (epsData(96,i,44), i=1,3) / 0.7860860000E+02, -0.4199020000E+01, 0.6764820000E+00 /
  data (epsData(97,i,44), i=1,3) / 0.8529810000E+02, -0.3406910000E+01, 0.4146980000E+00 /
  data (epsData(98,i,44), i=1,3) / 0.9237310000E+02, -0.2768390000E+01, 0.2727050000E+00 /
  data (epsData(99,i,44), i=1,3) / 0.1001650000E+03, -0.2244420000E+01, 0.1974390000E+00 /
  data (epsData(100,i,44), i=1,3) / 0.1087510000E+03, -0.1821530000E+01, 0.1626280000E+00 /
  data (epsData(101,i,44), i=1,3) / 0.1188380000E+03, -0.1464440000E+01, 0.1444410000E+00 /
  data (epsData(102,i,44), i=1,3) / 0.1307920000E+03, -0.1163680000E+01, 0.1348440000E+00 /
  data (epsData(103,i,44), i=1,3) / 0.1448010000E+03, -0.9181650000E+00, 0.1278180000E+00 /
  data (epsData(104,i,44), i=1,3) / 0.1614020000E+03, -0.7159900000E+00, 0.1182270000E+00 /
  data (epsData(105,i,44), i=1,3) / 0.1805300000E+03, -0.5533330000E+00, 0.1055040000E+00 /
  data (epsData(106,i,44), i=1,3) / 0.2028170000E+03, -0.4180490000E+00, 0.9075190000E-01 /
  data (epsData(107,i,44), i=1,3) / 0.2286060000E+03, -0.3028930000E+00, 0.7527150000E-01 /
  data (epsData(108,i,44), i=1,3) / 0.2653150000E+03, -0.1729140000E+00, 0.5743180000E-01 /
  data (epsData(109,i,44), i=1,3) / 0.2771990000E+03, -0.1276260000E+00, 0.5286500000E-01 /
  data (epsData(110,i,44), i=1,3) / 0.2793020000E+03, -0.1150560000E+00, 0.5248870000E-01 /
  data (epsData(111,i,44), i=1,3) / 0.2800220000E+03, -0.1078360000E+00, 0.5689060000E-01 /
  data (epsData(112,i,44), i=1,3) / 0.2804410000E+03, -0.1092150000E+00, 0.6119430000E-01 /
  data (epsData(113,i,44), i=1,3) / 0.2818130000E+03, -0.1091580000E+00, 0.6103270000E-01 /
  data (epsData(114,i,44), i=1,3) / 0.2837500000E+03, -0.1004480000E+00, 0.6065970000E-01 /
  data (epsData(115,i,44), i=1,3) / 0.2842870000E+03, -0.9661020000E-01, 0.6457260000E-01 /
  data (epsData(116,i,44), i=1,3) / 0.2850500000E+03, -0.9908090000E-01, 0.6661730000E-01 /
  data (epsData(117,i,44), i=1,3) / 0.2876700000E+03, -0.9278550000E-01, 0.6390110000E-01 /
  data (epsData(118,i,44), i=1,3) / 0.2918530000E+03, -0.7556960000E-01, 0.6266620000E-01 /
  data (epsData(119,i,44), i=1,3) / 0.2976700000E+03, -0.4708270000E-01, 0.6959150000E-01 /
  data (epsData(120,i,44), i=1,3) / 0.3033050000E+03, -0.2022050000E-01, 0.8428930000E-01 /
  data (epsData(121,i,44), i=1,3) / 0.3075490000E+03, -0.1535610000E-02, 0.1010340000E+00 /
  data (epsData(122,i,44), i=1,3) / 0.3114420000E+03, 0.1210080000E-01, 0.1212880000E+00 /
  data (epsData(123,i,44), i=1,3) / 0.3162580000E+03, 0.2072960000E-01, 0.1501350000E+00 /
  data (epsData(124,i,44), i=1,3) / 0.3212410000E+03, 0.1920150000E-01, 0.1790630000E+00 /
  data (epsData(125,i,44), i=1,3) / 0.3275510000E+03, 0.7774480000E-02, 0.2077260000E+00 /
  data (epsData(126,i,44), i=1,3) / 0.3356710000E+03, -0.1391280000E-01, 0.2330780000E+00 /
  data (epsData(127,i,44), i=1,3) / 0.3468000000E+03, -0.4695590000E-01, 0.2488060000E+00 /
  data (epsData(128,i,44), i=1,3) / 0.3590180000E+03, -0.7808970000E-01, 0.2471290000E+00 /
  data (epsData(129,i,44), i=1,3) / 0.3772400000E+03, -0.1061100000E+00, 0.2301580000E+00 /
  data (epsData(130,i,44), i=1,3) / 0.4082910000E+03, -0.1267600000E+00, 0.1960540000E+00 /
  data (epsData(131,i,44), i=1,3) / 0.4401000000E+03, -0.1268670000E+00, 0.1632760000E+00 /
  data (epsData(132,i,44), i=1,3) / 0.4585790000E+03, -0.1174650000E+00, 0.1491530000E+00 /
  data (epsData(133,i,44), i=1,3) / 0.4640940000E+03, -0.1136340000E+00, 0.1525610000E+00 /
  data (epsData(134,i,44), i=1,3) / 0.4780130000E+03, -0.1196040000E+00, 0.1488070000E+00 /
  data (epsData(135,i,44), i=1,3) / 0.5034200000E+03, -0.1232200000E+00, 0.1361740000E+00 /
  data (epsData(136,i,44), i=1,3) / 0.5523790000E+03, -0.1190950000E+00, 0.1065890000E+00 /
  data (epsData(137,i,44), i=1,3) / 0.5887340000E+03, -0.1107350000E+00, 0.9224210000E-01 /
  data (epsData(138,i,44), i=1,3) / 0.6571390000E+03, -0.1005790000E+00, 0.6811300000E-01 /
  data (epsData(139,i,44), i=1,3) / 0.7580520000E+03, -0.8283880000E-01, 0.4470070000E-01 /
  data (epsData(140,i,44), i=1,3) / 0.8591780000E+03, -0.6793150000E-01, 0.3041530000E-01 /
  data (epsData(141,i,44), i=1,3) / 0.9604030000E+03, -0.5607030000E-01, 0.2138320000E-01 /
  data (epsData(142,i,44), i=1,3) / 0.1068600000E+04, -0.4619210000E-01, 0.1514450000E-01 /
  data (epsData(143,i,44), i=1,3) / 0.1185090000E+04, -0.3801020000E-01, 0.1077660000E-01 /
  data (epsData(144,i,44), i=1,3) / 0.1312160000E+04, -0.3119950000E-01, 0.7671920000E-02 /
  data (epsData(145,i,44), i=1,3) / 0.1451460000E+04, -0.2553960000E-01, 0.5461120000E-02 /
  data (epsData(146,i,44), i=1,3) / 0.1606500000E+04, -0.2080410000E-01, 0.3869590000E-02 /
  data (epsData(147,i,44), i=1,3) / 0.1778130000E+04, -0.1688560000E-01, 0.2725410000E-02 /
  data (epsData(148,i,44), i=1,3) / 0.1967200000E+04, -0.1362920000E-01, 0.1922320000E-02 /
  data (epsData(149,i,44), i=1,3) / 0.2186380000E+04, -0.1083150000E-01, 0.1333980000E-02 /
  data (epsData(150,i,44), i=1,3) / 0.2437530000E+04, -0.8416120000E-02, 0.9112190000E-03 /
  data (epsData(151,i,44), i=1,3) / 0.2758240000E+04, -0.5768800000E-02, 0.5889150000E-03 /
  data (epsData(152,i,44), i=1,3) / 0.2810120000E+04, -0.5075960000E-02, 0.5512500000E-03 /
  data (epsData(153,i,44), i=1,3) / 0.2829300000E+04, -0.4537590000E-02, 0.5748690000E-03 /
  data (epsData(154,i,44), i=1,3) / 0.2834540000E+04, -0.4214770000E-02, 0.6133020000E-03 /
  data (epsData(155,i,44), i=1,3) / 0.2836290000E+04, -0.4034280000E-02, 0.6618610000E-03 /
  data (epsData(156,i,44), i=1,3) / 0.2838600000E+04, -0.3782260000E-02, 0.9460390000E-03 /
  data (epsData(157,i,44), i=1,3) / 0.2841130000E+04, -0.3796040000E-02, 0.1339190000E-02 /
  data (epsData(158,i,44), i=1,3) / 0.2846490000E+04, -0.4077280000E-02, 0.1806900000E-02 /
  data (epsData(159,i,44), i=1,3) / 0.2852290000E+04, -0.4469940000E-02, 0.1948200000E-02 /
  data (epsData(160,i,44), i=1,3) / 0.2866020000E+04, -0.4812610000E-02, 0.1869010000E-02 /
  data (epsData(161,i,44), i=1,3) / 0.2902660000E+04, -0.5008210000E-02, 0.1757090000E-02 /
  data (epsData(162,i,44), i=1,3) / 0.2951640000E+04, -0.4816690000E-02, 0.1649580000E-02 /
  data (epsData(163,i,44), i=1,3) / 0.2964570000E+04, -0.4558800000E-02, 0.1662160000E-02 /
  data (epsData(164,i,44), i=1,3) / 0.2968530000E+04, -0.4407190000E-02, 0.1845360000E-02 /
  data (epsData(165,i,44), i=1,3) / 0.2976150000E+04, -0.4562880000E-02, 0.2164630000E-02 /
  data (epsData(166,i,44), i=1,3) / 0.2986920000E+04, -0.4819540000E-02, 0.2190260000E-02 /
  data (epsData(167,i,44), i=1,3) / 0.3043750000E+04, -0.5051670000E-02, 0.2030650000E-02 /
  data (epsData(168,i,44), i=1,3) / 0.3182520000E+04, -0.4894790000E-02, 0.1731460000E-02 /
  data (epsData(169,i,44), i=1,3) / 0.3219750000E+04, -0.4681160000E-02, 0.1715780000E-02 /
  data (epsData(170,i,44), i=1,3) / 0.3228420000E+04, -0.4700180000E-02, 0.1856540000E-02 /
  data (epsData(171,i,44), i=1,3) / 0.3313520000E+04, -0.4769630000E-02, 0.1723390000E-02 /
  data (epsData(172,i,44), i=1,3) / 0.3673070000E+04, -0.4220280000E-02, 0.1203080000E-02 /
  data (epsData(173,i,44), i=1,3) / 0.4161330000E+04, -0.3434320000E-02, 0.7736740000E-03 /
  data (epsData(174,i,44), i=1,3) / 0.4652110000E+04, -0.2804920000E-02, 0.5234350000E-03 /
  data (epsData(175,i,44), i=1,3) / 0.5202940000E+04, -0.2270390000E-02, 0.3508730000E-03 /
  data (epsData(176,i,44), i=1,3) / 0.5800170000E+04, -0.1838010000E-02, 0.2364720000E-03 /
  data (epsData(177,i,44), i=1,3) / 0.6442480000E+04, -0.1493430000E-02, 0.1617190000E-03 /
  data (epsData(178,i,44), i=1,3) / 0.7163060000E+04, -0.1208760000E-02, 0.1093030000E-03 /
  data (epsData(179,i,44), i=1,3) / 0.7971540000E+04, -0.9748830000E-03, 0.7329290000E-04 /
  data (epsData(180,i,44), i=1,3) / 0.8885910000E+04, -0.7826530000E-03, 0.4890220000E-04 /
  data (epsData(181,i,44), i=1,3) / 0.9904860000E+04, -0.6280380000E-03, 0.3244420000E-04 /
  data (epsData(182,i,44), i=1,3) / 0.1104720000E+05, -0.5031250000E-03, 0.2150320000E-04 /
  data (epsData(183,i,44), i=1,3) / 0.1232880000E+05, -0.4023720000E-03, 0.1417140000E-04 /
  data (epsData(184,i,44), i=1,3) / 0.1376430000E+05, -0.3214030000E-03, 0.9318170000E-05 /
  data (epsData(185,i,44), i=1,3) / 0.1537790000E+05, -0.2561940000E-03, 0.6093720000E-05 /
  data (epsData(186,i,44), i=1,3) / 0.1718900000E+05, -0.2037260000E-03, 0.3972680000E-05 /
  data (epsData(187,i,44), i=1,3) / 0.1799380000E+05, -0.1852680000E-03, 0.3329320000E-05 /
  data (epsData(188,i,44), i=1,3) / 0.1799800000E+05, -0.1768810000E-03, 0.3326380000E-05 /
  data (epsData(189,i,44), i=1,3) / 0.1799800000E+05, -0.1851770000E-03, 0.3326340000E-05 /
  data (epsData(190,i,44), i=1,3) / 0.2021510000E+05, -0.1446600000E-03, 0.2123690000E-05 /
  data (epsData(191,i,44), i=1,3) / 0.2195560000E+05, -0.1164350000E-03, 0.1541620000E-05 /
  data (epsData(192,i,44), i=1,3) / 0.2208890000E+05, -0.1102290000E-03, 0.1505600000E-05 /
  data (epsData(193,i,44), i=1,3) / 0.2211020000E+05, -0.1061460000E-03, 0.2794810000E-05 /
  data (epsData(194,i,44), i=1,3) / 0.2211490000E+05, -0.1043470000E-03, 0.4599430000E-05 /
  data (epsData(195,i,44), i=1,3) / 0.2211700000E+05, -0.9001690000E-04, 0.6408770000E-05 /
  data (epsData(196,i,44), i=1,3) / 0.2211700000E+05, -0.1039790000E-03, 0.6455160000E-05 /
  data (epsData(197,i,44), i=1,3) / 0.2212430000E+05, -0.1073090000E-03, 0.8988760000E-05 /
  data (epsData(198,i,44), i=1,3) / 0.2218470000E+05, -0.1116810000E-03, 0.9803460000E-05 /
  data (epsData(199,i,44), i=1,3) / 0.2255750000E+05, -0.1132500000E-03, 0.9382960000E-05 /
  data (epsData(200,i,44), i=1,3) / 0.2710900000E+05, -0.8311100000E-04, 0.4845300000E-05 /
  data (epsData(201,i,44), i=1,3) / 0.3049100000E+05, -0.6628490000E-04, 0.3133640000E-05 /
  data (epsData(202,i,44), i=1,3) / 0.3418970000E+05, -0.5294780000E-04, 0.2039190000E-05 /
  data (epsData(203,i,44), i=1,3) / 0.3829540000E+05, -0.4229090000E-04, 0.1325780000E-05 /
  data (epsData(204,i,44), i=1,3) / 0.4287940000E+05, -0.3375910000E-04, 0.8593690000E-06 /
  data (epsData(205,i,44), i=1,3) / 0.4801070000E+05, -0.2692960000E-04, 0.5545050000E-06 /
  data (epsData(206,i,44), i=1,3) / 0.5376050000E+05, -0.2146880000E-04, 0.3561370000E-06 /
  data (epsData(207,i,44), i=1,3) / 0.6020870000E+05, -0.1710540000E-04, 0.2275670000E-06 /
  data (epsData(208,i,44), i=1,3) / 0.6745850000E+05, -0.1361690000E-04, 0.1451020000E-06 /
  data (epsData(209,i,44), i=1,3) / 0.7556940000E+05, -0.1084360000E-04, 0.9159820000E-07 /
  data (epsData(210,i,44), i=1,3) / 0.8467380000E+05, -0.8632980000E-05, 0.5779060000E-07 /
  data (epsData(211,i,44), i=1,3) / 0.9482940000E+05, -0.6885140000E-05, 0.3630170000E-07 /
  data (epsData(212,i,44), i=1,3) / 0.1000000000E+06, -0.6262780000E-05, 0.2915480000E-07 /
  data (epsData(1,i,45), i=1,3) / 0.2506580000E-02, -0.3201010000E+04, 0.8181080000E+04 /
  data (epsData(2,i,45), i=1,3) / 0.4693440000E-02, -0.2552290000E+04, 0.2717220000E+04 /
  data (epsData(3,i,45), i=1,3) / 0.7560590000E-02, -0.1075530000E+04, 0.1241040000E+04 /
  data (epsData(4,i,45), i=1,3) / 0.1110800000E-01, -0.4340420000E+03, 0.6860980000E+03 /
  data (epsData(5,i,45), i=1,3) / 0.1533570000E-01, -0.1328630000E+03, 0.4294560000E+03 /
  data (epsData(6,i,45), i=1,3) / 0.2024370000E-01, 0.2281650000E+02, 0.2927630000E+03 /
  data (epsData(7,i,45), i=1,3) / 0.2583200000E-01, 0.1099230000E+03, 0.2121740000E+03 /
  data (epsData(8,i,45), i=1,3) / 0.3210060000E-01, 0.1618600000E+03, 0.1609190000E+03 /
  data (epsData(9,i,45), i=1,3) / 0.3904940000E-01, 0.1944710000E+03, 0.1263620000E+03 /
  data (epsData(10,i,45), i=1,3) / 0.4667860000E-01, 0.2158480000E+03, 0.1019690000E+03 /
  data (epsData(11,i,45), i=1,3) / 0.5498800000E-01, 0.2303920000E+03, 0.8410480000E+02 /
  data (epsData(12,i,45), i=1,3) / 0.6397770000E-01, 0.2406240000E+03, 0.7062110000E+02 /
  data (epsData(13,i,45), i=1,3) / 0.8399800000E-01, 0.2537010000E+03, 0.5193920000E+02 /
  data (epsData(14,i,45), i=1,3) / 0.1067390000E+00, 0.2619540000E+03, 0.4076420000E+02 /
  data (epsData(15,i,45), i=1,3) / 0.1754980000E+00, 0.2679760000E+03, 0.2725010000E+02 /
  data (epsData(16,i,45), i=1,3) / 0.3866320000E+00, 0.2728720000E+03, 0.1331520000E+02 /
  data (epsData(17,i,45), i=1,3) / 0.8071970000E+00, 0.2896530000E+03, 0.9219190000E+01 /
  data (epsData(18,i,45), i=1,3) / 0.1133010000E+01, 0.3160160000E+03, 0.9561520000E+01 /
  data (epsData(19,i,45), i=1,3) / 0.1337820000E+01, 0.3446790000E+03, 0.1102600000E+02 /
  data (epsData(20,i,45), i=1,3) / 0.1513920000E+01, 0.3818290000E+03, 0.1454690000E+02 /
  data (epsData(21,i,45), i=1,3) / 0.1606050000E+01, 0.4150320000E+03, 0.1712250000E+02 /
  data (epsData(22,i,45), i=1,3) / 0.1749350000E+01, 0.4727800000E+03, 0.4675490000E+02 /
  data (epsData(23,i,45), i=1,3) / 0.1798470000E+01, 0.4990450000E+03, 0.5471950000E+02 /
  data (epsData(24,i,45), i=1,3) / 0.1848280000E+01, 0.5394090000E+03, 0.6187420000E+02 /
  data (epsData(25,i,45), i=1,3) / 0.1898770000E+01, 0.6176380000E+03, 0.6829380000E+02 /
  data (epsData(26,i,45), i=1,3) / 0.1949930000E+01, 0.6765480000E+03, 0.2096370000E+03 /
  data (epsData(27,i,45), i=1,3) / 0.2003060000E+01, 0.6534740000E+03, 0.3415410000E+03 /
  data (epsData(28,i,45), i=1,3) / 0.2136060000E+01, 0.5103240000E+03, 0.5077440000E+03 /
  data (epsData(29,i,45), i=1,3) / 0.2195920000E+01, 0.4113670000E+03, 0.5687770000E+03 /
  data (epsData(30,i,45), i=1,3) / 0.2229940000E+01, 0.3453280000E+03, 0.5600060000E+03 /
  data (epsData(31,i,45), i=1,3) / 0.2326880000E+01, 0.2415840000E+03, 0.5220900000E+03 /
  data (epsData(32,i,45), i=1,3) / 0.2465650000E+01, 0.1525240000E+03, 0.4748840000E+03 /
  data (epsData(33,i,45), i=1,3) / 0.2600010000E+01, 0.9265800000E+02, 0.4275960000E+03 /
  data (epsData(34,i,45), i=1,3) / 0.2783690000E+01, 0.4418730000E+02, 0.3703340000E+03 /
  data (epsData(35,i,45), i=1,3) / 0.2955460000E+01, 0.1644800000E+02, 0.3235590000E+03 /
  data (epsData(36,i,45), i=1,3) / 0.3184030000E+01, -0.3587440000E+01, 0.2722780000E+03 /
  data (epsData(37,i,45), i=1,3) / 0.3393220000E+01, -0.1129970000E+02, 0.2338810000E+03 /
  data (epsData(38,i,45), i=1,3) / 0.3666700000E+01, -0.1278170000E+02, 0.1949800000E+03 /
  data (epsData(39,i,45), i=1,3) / 0.3964660000E+01, -0.8541430000E+01, 0.1640810000E+03 /
  data (epsData(40,i,45), i=1,3) / 0.4343240000E+01, 0.1163720000E+00, 0.1370880000E+03 /
  data (epsData(41,i,45), i=1,3) / 0.4694260000E+01, 0.9054410000E+01, 0.1201970000E+03 /
  data (epsData(42,i,45), i=1,3) / 0.5069780000E+01, 0.1936720000E+02, 0.1089020000E+03 /
  data (epsData(43,i,45), i=1,3) / 0.5469790000E+01, 0.3012010000E+02, 0.1002020000E+03 /
  data (epsData(44,i,45), i=1,3) / 0.5678980000E+01, 0.3947850000E+02, 0.9676030000E+02 /
  data (epsData(45,i,45), i=1,3) / 0.5821840000E+01, 0.5076170000E+02, 0.9580360000E+02 /
  data (epsData(46,i,45), i=1,3) / 0.6041230000E+01, 0.6904670000E+02, 0.1082330000E+03 /
  data (epsData(47,i,45), i=1,3) / 0.6115720000E+01, 0.7962310000E+02, 0.1147120000E+03 /
  data (epsData(48,i,45), i=1,3) / 0.6190890000E+01, 0.8358030000E+02, 0.1410650000E+03 /
  data (epsData(49,i,45), i=1,3) / 0.6266740000E+01, 0.7558350000E+02, 0.1594490000E+03 /
  data (epsData(50,i,45), i=1,3) / 0.6343270000E+01, 0.6175460000E+02, 0.1752380000E+03 /
  data (epsData(51,i,45), i=1,3) / 0.6420490000E+01, 0.4122410000E+02, 0.1867750000E+03 /
  data (epsData(52,i,45), i=1,3) / 0.6498380000E+01, 0.2167260000E+02, 0.1811040000E+03 /
  data (epsData(53,i,45), i=1,3) / 0.6656200000E+01, -0.1831830000E+01, 0.1702620000E+03 /
  data (epsData(54,i,45), i=1,3) / 0.6816750000E+01, -0.1842950000E+02, 0.1561320000E+03 /
  data (epsData(55,i,45), i=1,3) / 0.7062670000E+01, -0.3058150000E+02, 0.1342310000E+03 /
  data (epsData(56,i,45), i=1,3) / 0.7314720000E+01, -0.3478380000E+02, 0.1152340000E+03 /
  data (epsData(57,i,45), i=1,3) / 0.7660300000E+01, -0.3375040000E+02, 0.9560800000E+02 /
  data (epsData(58,i,45), i=1,3) / 0.8016770000E+01, -0.2931350000E+02, 0.8136720000E+02 /
  data (epsData(59,i,45), i=1,3) / 0.8477660000E+01, -0.2251300000E+02, 0.6914950000E+02 /
  data (epsData(60,i,45), i=1,3) / 0.9053180000E+01, -0.1459130000E+02, 0.6018110000E+02 /
  data (epsData(61,i,45), i=1,3) / 0.9551490000E+01, -0.9280110000E+01, 0.5643770000E+02 /
  data (epsData(62,i,45), i=1,3) / 0.1049130000E+02, -0.3619460000E+01, 0.5165310000E+02 /
  data (epsData(63,i,45), i=1,3) / 0.1137300000E+02, -0.1879080000E+00, 0.4920740000E+02 /
  data (epsData(64,i,45), i=1,3) / 0.1253620000E+02, 0.2000920000E+01, 0.4750250000E+02 /
  data (epsData(65,i,45), i=1,3) / 0.1440870000E+02, 0.2107660000E+01, 0.4593150000E+02 /
  data (epsData(66,i,45), i=1,3) / 0.1714350000E+02, -0.1203090000E+01, 0.4338460000E+02 /
  data (epsData(67,i,45), i=1,3) / 0.2110540000E+02, -0.7164350000E+01, 0.3756900000E+02 /
  data (epsData(68,i,45), i=1,3) / 0.2499360000E+02, -0.1074500000E+02, 0.3067150000E+02 /
  data (epsData(69,i,45), i=1,3) / 0.2943450000E+02, -0.1244070000E+02, 0.2421920000E+02 /
  data (epsData(70,i,45), i=1,3) / 0.3532340000E+02, -0.1329450000E+02, 0.1748010000E+02 /
  data (epsData(71,i,45), i=1,3) / 0.4062960000E+02, -0.1296260000E+02, 0.1242960000E+02 /
  data (epsData(72,i,45), i=1,3) / 0.4510820000E+02, -0.1185810000E+02, 0.9048340000E+01 /
  data (epsData(73,i,45), i=1,3) / 0.4911260000E+02, -0.1063960000E+02, 0.7122520000E+01 /
  data (epsData(74,i,45), i=1,3) / 0.5835860000E+02, -0.8710330000E+01, 0.3854590000E+01 /
  data (epsData(75,i,45), i=1,3) / 0.6410570000E+02, -0.7423230000E+01, 0.2392380000E+01 /
  data (epsData(76,i,45), i=1,3) / 0.7075070000E+02, -0.6007040000E+01, 0.1348890000E+01 /
  data (epsData(77,i,45), i=1,3) / 0.7735790000E+02, -0.4846590000E+01, 0.7831380000E+00 /
  data (epsData(78,i,45), i=1,3) / 0.8421670000E+02, -0.3921580000E+01, 0.4812290000E+00 /
  data (epsData(79,i,45), i=1,3) / 0.9193620000E+02, -0.3147290000E+01, 0.2977310000E+00 /
  data (epsData(80,i,45), i=1,3) / 0.1000250000E+03, -0.2543550000E+01, 0.2060980000E+00 /
  data (epsData(81,i,45), i=1,3) / 0.1086420000E+03, -0.2069050000E+01, 0.1641520000E+00 /
  data (epsData(82,i,45), i=1,3) / 0.1186660000E+03, -0.1671070000E+01, 0.1460370000E+00 /
  data (epsData(83,i,45), i=1,3) / 0.1307750000E+03, -0.1329980000E+01, 0.1360000000E+00 /
  data (epsData(84,i,45), i=1,3) / 0.1448790000E+03, -0.1052920000E+01, 0.1292310000E+00 /
  data (epsData(85,i,45), i=1,3) / 0.1610810000E+03, -0.8310390000E+00, 0.1217360000E+00 /
  data (epsData(86,i,45), i=1,3) / 0.1809540000E+03, -0.6428770000E+00, 0.1092770000E+00 /
  data (epsData(87,i,45), i=1,3) / 0.2042620000E+03, -0.4896220000E+00, 0.9432220000E-01 /
  data (epsData(88,i,45), i=1,3) / 0.2309190000E+03, -0.3650180000E+00, 0.7855740000E-01 /
  data (epsData(89,i,45), i=1,3) / 0.2611790000E+03, -0.2595190000E+00, 0.6336020000E-01 /
  data (epsData(90,i,45), i=1,3) / 0.3023740000E+03, -0.1366530000E+00, 0.4768030000E-01 /
  data (epsData(91,i,45), i=1,3) / 0.3062290000E+03, -0.1204530000E+00, 0.4677400000E-01 /
  data (epsData(92,i,45), i=1,3) / 0.3072030000E+03, -0.1131860000E+00, 0.5010030000E-01 /
  data (epsData(93,i,45), i=1,3) / 0.3077170000E+03, -0.1140260000E+00, 0.5401980000E-01 /
  data (epsData(94,i,45), i=1,3) / 0.3097160000E+03, -0.1117870000E+00, 0.5332640000E-01 /
  data (epsData(95,i,45), i=1,3) / 0.3116210000E+03, -0.1037470000E+00, 0.5365820000E-01 /
  data (epsData(96,i,45), i=1,3) / 0.3122480000E+03, -0.1025240000E+00, 0.5762190000E-01 /
  data (epsData(97,i,45), i=1,3) / 0.3139170000E+03, -0.1016620000E+00, 0.5661920000E-01 /
  data (epsData(98,i,45), i=1,3) / 0.3190510000E+03, -0.8487160000E-01, 0.5462380000E-01 /
  data (epsData(99,i,45), i=1,3) / 0.3253700000E+03, -0.5859400000E-01, 0.5959440000E-01 /
  data (epsData(100,i,45), i=1,3) / 0.3318200000E+03, -0.3114170000E-01, 0.7278180000E-01 /
  data (epsData(101,i,45), i=1,3) / 0.3363300000E+03, -0.1300740000E-01, 0.8809000000E-01 /
  data (epsData(102,i,45), i=1,3) / 0.3401130000E+03, -0.6765790000E-03, 0.1051930000E+00 /
  data (epsData(103,i,45), i=1,3) / 0.3445880000E+03, 0.8018740000E-02, 0.1287220000E+00 /
  data (epsData(104,i,45), i=1,3) / 0.3496180000E+03, 0.8962720000E-02, 0.1554370000E+00 /
  data (epsData(105,i,45), i=1,3) / 0.3559820000E+03, 0.5250730000E-03, 0.1838310000E+00 /
  data (epsData(106,i,45), i=1,3) / 0.3633710000E+03, -0.1700600000E-01, 0.2076450000E+00 /
  data (epsData(107,i,45), i=1,3) / 0.3730500000E+03, -0.4417760000E-01, 0.2242720000E+00 /
  data (epsData(108,i,45), i=1,3) / 0.3845020000E+03, -0.7366130000E-01, 0.2262600000E+00 /
  data (epsData(109,i,45), i=1,3) / 0.4003720000E+03, -0.1003290000E+00, 0.2145930000E+00 /
  data (epsData(110,i,45), i=1,3) / 0.4291040000E+03, -0.1224290000E+00, 0.1862430000E+00 /
  data (epsData(111,i,45), i=1,3) / 0.4618700000E+03, -0.1263720000E+00, 0.1548070000E+00 /
  data (epsData(112,i,45), i=1,3) / 0.4911840000E+03, -0.1170130000E+00, 0.1330740000E+00 /
  data (epsData(113,i,45), i=1,3) / 0.4984130000E+03, -0.1110230000E+00, 0.1339390000E+00 /
  data (epsData(114,i,45), i=1,3) / 0.5112160000E+03, -0.1154230000E+00, 0.1332740000E+00 /
  data (epsData(115,i,45), i=1,3) / 0.5302270000E+03, -0.1158610000E+00, 0.1261040000E+00 /
  data (epsData(116,i,45), i=1,3) / 0.5746400000E+03, -0.1155470000E+00, 0.1027810000E+00 /
  data (epsData(117,i,45), i=1,3) / 0.6230750000E+03, -0.1059830000E+00, 0.8280420000E-01 /
  data (epsData(118,i,45), i=1,3) / 0.7236470000E+03, -0.9114020000E-01, 0.5579240000E-01 /
  data (epsData(119,i,45), i=1,3) / 0.8133600000E+03, -0.7717790000E-01, 0.3929080000E-01 /
  data (epsData(120,i,45), i=1,3) / 0.9042350000E+03, -0.6509100000E-01, 0.2828750000E-01 /
  data (epsData(121,i,45), i=1,3) / 0.1002400000E+04, -0.5448530000E-01, 0.2039680000E-01 /
  data (epsData(122,i,45), i=1,3) / 0.1109790000E+04, -0.4531400000E-01, 0.1467120000E-01 /
  data (epsData(123,i,45), i=1,3) / 0.1227090000E+04, -0.3752520000E-01, 0.1054140000E-01 /
  data (epsData(124,i,45), i=1,3) / 0.1356150000E+04, -0.3093310000E-01, 0.7549960000E-02 /
  data (epsData(125,i,45), i=1,3) / 0.1499980000E+04, -0.2534770000E-01, 0.5374320000E-02 /
  data (epsData(126,i,45), i=1,3) / 0.1662690000E+04, -0.2060390000E-01, 0.3783670000E-02 /
  data (epsData(127,i,45), i=1,3) / 0.1848310000E+04, -0.1658080000E-01, 0.2621020000E-02 /
  data (epsData(128,i,45), i=1,3) / 0.2050640000E+04, -0.1333080000E-01, 0.1836990000E-02 /
  data (epsData(129,i,45), i=1,3) / 0.2280440000E+04, -0.1060610000E-01, 0.1270500000E-02 /
  data (epsData(130,i,45), i=1,3) / 0.2540330000E+04, -0.8299810000E-02, 0.8698480000E-03 /
  data (epsData(131,i,45), i=1,3) / 0.2925780000E+04, -0.5482750000E-02, 0.5279320000E-03 /
  data (epsData(132,i,45), i=1,3) / 0.2977340000E+04, -0.4867680000E-02, 0.4961110000E-03 /
  data (epsData(133,i,45), i=1,3) / 0.2995260000E+04, -0.4428640000E-02, 0.5189160000E-03 /
  data (epsData(134,i,45), i=1,3) / 0.3000640000E+04, -0.4152560000E-02, 0.5530370000E-03 /
  data (epsData(135,i,45), i=1,3) / 0.3002440000E+04, -0.4001930000E-02, 0.5959190000E-03 /
  data (epsData(136,i,45), i=1,3) / 0.3004920000E+04, -0.3787700000E-02, 0.8373630000E-03 /
  data (epsData(137,i,45), i=1,3) / 0.3007890000E+04, -0.3788820000E-02, 0.1189950000E-02 /
  data (epsData(138,i,45), i=1,3) / 0.3012820000E+04, -0.3999710000E-02, 0.1573890000E-02 /
  data (epsData(139,i,45), i=1,3) / 0.3018720000E+04, -0.4335580000E-02, 0.1701290000E-02 /
  data (epsData(140,i,45), i=1,3) / 0.3035640000E+04, -0.4661060000E-02, 0.1647440000E-02 /
  data (epsData(141,i,45), i=1,3) / 0.3072570000E+04, -0.4822070000E-02, 0.1561600000E-02 /
  data (epsData(142,i,45), i=1,3) / 0.3127680000E+04, -0.4650210000E-02, 0.1458420000E-02 /
  data (epsData(143,i,45), i=1,3) / 0.3143320000E+04, -0.4416280000E-02, 0.1465940000E-02 /
  data (epsData(144,i,45), i=1,3) / 0.3148180000E+04, -0.4274580000E-02, 0.1637570000E-02 /
  data (epsData(145,i,45), i=1,3) / 0.3156060000E+04, -0.4407870000E-02, 0.1906550000E-02 /
  data (epsData(146,i,45), i=1,3) / 0.3169530000E+04, -0.4635100000E-02, 0.1920300000E-02 /
  data (epsData(147,i,45), i=1,3) / 0.3239750000E+04, -0.4810280000E-02, 0.1769060000E-02 /
  data (epsData(148,i,45), i=1,3) / 0.3384300000E+04, -0.4592290000E-02, 0.1511560000E-02 /
  data (epsData(149,i,45), i=1,3) / 0.3409410000E+04, -0.4422050000E-02, 0.1540960000E-02 /
  data (epsData(150,i,45), i=1,3) / 0.3417490000E+04, -0.4472970000E-02, 0.1650380000E-02 /
  data (epsData(151,i,45), i=1,3) / 0.3531210000E+04, -0.4487730000E-02, 0.1494370000E-02 /
  data (epsData(152,i,45), i=1,3) / 0.3991290000E+04, -0.3811220000E-02, 0.9729470000E-03 /
  data (epsData(153,i,45), i=1,3) / 0.4509210000E+04, -0.3087840000E-02, 0.6317360000E-03 /
  data (epsData(154,i,45), i=1,3) / 0.5062840000E+04, -0.2496550000E-02, 0.4210210000E-03 /
  data (epsData(155,i,45), i=1,3) / 0.5653450000E+04, -0.2021580000E-02, 0.2827140000E-03 /
  data (epsData(156,i,45), i=1,3) / 0.6300400000E+04, -0.1634910000E-02, 0.1909990000E-03 /
  data (epsData(157,i,45), i=1,3) / 0.7033670000E+04, -0.1314470000E-02, 0.1276200000E-03 /
  data (epsData(158,i,45), i=1,3) / 0.7825140000E+04, -0.1061910000E-02, 0.8597310000E-04 /
  data (epsData(159,i,45), i=1,3) / 0.8718910000E+04, -0.8538820000E-03, 0.5733980000E-04 /
  data (epsData(160,i,45), i=1,3) / 0.9721370000E+04, -0.6852260000E-03, 0.3818400000E-04 /
  data (epsData(161,i,45), i=1,3) / 0.1084230000E+05, -0.5491530000E-03, 0.2531520000E-04 /
  data (epsData(162,i,45), i=1,3) / 0.1209940000E+05, -0.4394220000E-03, 0.1670720000E-04 /
  data (epsData(163,i,45), i=1,3) / 0.1351020000E+05, -0.3510680000E-03, 0.1097520000E-04 /
  data (epsData(164,i,45), i=1,3) / 0.1509080000E+05, -0.2801380000E-03, 0.7194840000E-05 /
  data (epsData(165,i,45), i=1,3) / 0.1686750000E+05, -0.2230580000E-03, 0.4693340000E-05 /
  data (epsData(166,i,45), i=1,3) / 0.1886660000E+05, -0.1770280000E-03, 0.3049120000E-05 /
  data (epsData(167,i,45), i=1,3) / 0.1898150000E+05, -0.1748070000E-03, 0.2978230000E-05 /
  data (epsData(168,i,45), i=1,3) / 0.1898600000E+05, -0.1672290000E-03, 0.2975470000E-05 /
  data (epsData(169,i,45), i=1,3) / 0.1898600000E+05, -0.1747200000E-03, 0.2975440000E-05 /
  data (epsData(170,i,45), i=1,3) / 0.2133350000E+05, -0.1363930000E-03, 0.1895620000E-05 /
  data (epsData(171,i,45), i=1,3) / 0.2306280000E+05, -0.1110610000E-03, 0.1400390000E-05 /
  data (epsData(172,i,45), i=1,3) / 0.2319330000E+05, -0.1053700000E-03, 0.1370120000E-05 /
  data (epsData(173,i,45), i=1,3) / 0.2321490000E+05, -0.1013570000E-03, 0.2982680000E-05 /
  data (epsData(174,i,45), i=1,3) / 0.2321970000E+05, -0.1001270000E-03, 0.5548680000E-05 /
  data (epsData(175,i,45), i=1,3) / 0.2322000000E+05, -0.9063230000E-04, 0.5788480000E-05 /
  data (epsData(176,i,45), i=1,3) / 0.2322000000E+05, -0.1001280000E-03, 0.5827050000E-05 /
  data (epsData(177,i,45), i=1,3) / 0.2322800000E+05, -0.1031010000E-03, 0.8025720000E-05 /
  data (epsData(178,i,45), i=1,3) / 0.2329480000E+05, -0.1070400000E-03, 0.8858280000E-05 /
  data (epsData(179,i,45), i=1,3) / 0.2373460000E+05, -0.1079470000E-03, 0.8410750000E-05 /
  data (epsData(180,i,45), i=1,3) / 0.2831940000E+05, -0.7987310000E-04, 0.4457050000E-05 /
  data (epsData(181,i,45), i=1,3) / 0.3185590000E+05, -0.6367560000E-04, 0.2880900000E-05 /
  data (epsData(182,i,45), i=1,3) / 0.3572190000E+05, -0.5085240000E-04, 0.1873670000E-05 /
  data (epsData(183,i,45), i=1,3) / 0.4001420000E+05, -0.4060970000E-04, 0.1217810000E-05 /
  data (epsData(184,i,45), i=1,3) / 0.4480910000E+05, -0.3240890000E-04, 0.7886480000E-06 /
  data (epsData(185,i,45), i=1,3) / 0.5017520000E+05, -0.2584910000E-04, 0.5084950000E-06 /
  data (epsData(186,i,45), i=1,3) / 0.5618970000E+05, -0.2060400000E-04, 0.3263520000E-06 /
  data (epsData(187,i,45), i=1,3) / 0.6291970000E+05, -0.1642250000E-04, 0.2095370000E-06 /
  data (epsData(188,i,45), i=1,3) / 0.7047450000E+05, -0.1308230000E-04, 0.1327280000E-06 /
  data (epsData(189,i,45), i=1,3) / 0.7896540000E+05, -0.1041410000E-04, 0.8411780000E-07 /
  data (epsData(190,i,45), i=1,3) / 0.8847940000E+05, -0.8292750000E-05, 0.5292520000E-07 /
  data (epsData(191,i,45), i=1,3) / 0.9880330000E+05, -0.6664450000E-05, 0.3363150000E-07 /
  data (epsData(192,i,45), i=1,3) / 0.1000000000E+06, -0.6560790000E-05, 0.3198090000E-07 /
  data (epsData(1,i,46), i=1,3) / 0.2506580000E-02, -0.1551410000E+04, 0.4082210000E+04 /
  data (epsData(2,i,46), i=1,3) / 0.4693440000E-02, -0.1247490000E+04, 0.1365700000E+04 /
  data (epsData(3,i,46), i=1,3) / 0.7560590000E-02, -0.5217150000E+03, 0.6280340000E+03 /
  data (epsData(4,i,46), i=1,3) / 0.1110800000E-01, -0.2060270000E+03, 0.3492700000E+03 /
  data (epsData(5,i,46), i=1,3) / 0.1533570000E-01, -0.5773070000E+02, 0.2197050000E+03 /
  data (epsData(6,i,46), i=1,3) / 0.2024370000E-01, 0.1894780000E+02, 0.1503800000E+03 /
  data (epsData(7,i,46), i=1,3) / 0.2583200000E-01, 0.6186000000E+02, 0.1093410000E+03 /
  data (epsData(8,i,46), i=1,3) / 0.3210060000E-01, 0.8744940000E+02, 0.8314620000E+02 /
  data (epsData(9,i,46), i=1,3) / 0.3904940000E-01, 0.1035180000E+03, 0.6543110000E+02 /
  data (epsData(10,i,46), i=1,3) / 0.4667860000E-01, 0.1140520000E+03, 0.5289320000E+02 /
  data (epsData(11,i,46), i=1,3) / 0.5498800000E-01, 0.1212180000E+03, 0.4368960000E+02 /
  data (epsData(12,i,46), i=1,3) / 0.6397770000E-01, 0.1262580000E+03, 0.3672930000E+02 /
  data (epsData(13,i,46), i=1,3) / 0.8399800000E-01, 0.1326980000E+03, 0.2706310000E+02 /
  data (epsData(14,i,46), i=1,3) / 0.1067390000E+00, 0.1367510000E+03, 0.2126390000E+02 /
  data (epsData(15,i,46), i=1,3) / 0.1754980000E+00, 0.1396630000E+03, 0.1411290000E+02 /
  data (epsData(16,i,46), i=1,3) / 0.3866320000E+00, 0.1417830000E+03, 0.6860120000E+01 /
  data (epsData(17,i,46), i=1,3) / 0.8406760000E+00, 0.1492800000E+03, 0.4541590000E+01 /
  data (epsData(18,i,46), i=1,3) / 0.1253850000E+01, 0.1643760000E+03, 0.4645560000E+01 /
  data (epsData(19,i,46), i=1,3) / 0.1513920000E+01, 0.1828150000E+03, 0.6273960000E+01 /
  data (epsData(20,i,46), i=1,3) / 0.1606050000E+01, 0.1941210000E+03, 0.7105740000E+01 /
  data (epsData(21,i,46), i=1,3) / 0.1798470000E+01, 0.2218530000E+03, 0.1869590000E+02 /
  data (epsData(22,i,46), i=1,3) / 0.1848280000E+01, 0.2338810000E+03, 0.2089500000E+02 /
  data (epsData(23,i,46), i=1,3) / 0.1898770000E+01, 0.2553030000E+03, 0.2286560000E+02 /
  data (epsData(24,i,46), i=1,3) / 0.1949930000E+01, 0.2735960000E+03, 0.5577420000E+02 /
  data (epsData(25,i,46), i=1,3) / 0.2016670000E+01, 0.2756700000E+03, 0.9394410000E+02 /
  data (epsData(26,i,46), i=1,3) / 0.2122790000E+01, 0.2576530000E+03, 0.1417280000E+03 /
  data (epsData(27,i,46), i=1,3) / 0.2195920000E+01, 0.2332970000E+03, 0.1691140000E+03 /
  data (epsData(28,i,46), i=1,3) / 0.2286060000E+01, 0.2023800000E+03, 0.1776180000E+03 /
  data (epsData(29,i,46), i=1,3) / 0.2544230000E+01, 0.1446350000E+03, 0.1894020000E+03 /
  data (epsData(30,i,46), i=1,3) / 0.2850360000E+01, 0.9260570000E+02, 0.1844230000E+03 /
  data (epsData(31,i,46), i=1,3) / 0.3144240000E+01, 0.5712680000E+02, 0.1676990000E+03 /
  data (epsData(32,i,46), i=1,3) / 0.3481660000E+01, 0.3253000000E+02, 0.1436400000E+03 /
  data (epsData(33,i,46), i=1,3) / 0.3812620000E+01, 0.2059350000E+02, 0.1221340000E+03 /
  data (epsData(34,i,46), i=1,3) / 0.4016700000E+01, 0.1778340000E+02, 0.1110200000E+03 /
  data (epsData(35,i,46), i=1,3) / 0.4515690000E+01, 0.1374460000E+02, 0.9445660000E+02 /
  data (epsData(36,i,46), i=1,3) / 0.5069780000E+01, 0.1017440000E+02, 0.8066690000E+02 /
  data (epsData(37,i,46), i=1,3) / 0.5821840000E+01, 0.8163360000E+01, 0.6732420000E+02 /
  data (epsData(38,i,46), i=1,3) / 0.6656200000E+01, 0.7421130000E+01, 0.5680280000E+02 /
  data (epsData(39,i,46), i=1,3) / 0.7660300000E+01, 0.7811140000E+01, 0.4794550000E+02 /
  data (epsData(40,i,46), i=1,3) / 0.8858620000E+01, 0.9139960000E+01, 0.4138070000E+02 /
  data (epsData(41,i,46), i=1,3) / 0.1027770000E+02, 0.1024240000E+02, 0.3737460000E+02 /
  data (epsData(42,i,46), i=1,3) / 0.1206280000E+02, 0.1015250000E+02, 0.3509100000E+02 /
  data (epsData(43,i,46), i=1,3) / 0.1533490000E+02, 0.7154260000E+01, 0.3305040000E+02 /
  data (epsData(44,i,46), i=1,3) / 0.2046600000E+02, 0.4139090000E+00, 0.2914590000E+02 /
  data (epsData(45,i,46), i=1,3) / 0.2411770000E+02, -0.3248220000E+01, 0.2516640000E+02 /
  data (epsData(46,i,46), i=1,3) / 0.2772320000E+02, -0.5242580000E+01, 0.2135350000E+02 /
  data (epsData(47,i,46), i=1,3) / 0.3261280000E+02, -0.6693210000E+01, 0.1735190000E+02 /
  data (epsData(48,i,46), i=1,3) / 0.3971800000E+02, -0.7866200000E+01, 0.1270880000E+02 /
  data (epsData(49,i,46), i=1,3) / 0.4559390000E+02, -0.7990950000E+01, 0.9451050000E+01 /
  data (epsData(50,i,46), i=1,3) / 0.5008790000E+02, -0.7547040000E+01, 0.7523500000E+01 /
  data (epsData(51,i,46), i=1,3) / 0.6295050000E+02, -0.6664620000E+01, 0.3713040000E+01 /
  data (epsData(52,i,46), i=1,3) / 0.6971690000E+02, -0.5718110000E+01, 0.2216720000E+01 /
  data (epsData(53,i,46), i=1,3) / 0.7678910000E+02, -0.4670800000E+01, 0.1251310000E+01 /
  data (epsData(54,i,46), i=1,3) / 0.8410900000E+02, -0.3742290000E+01, 0.7047730000E+00 /
  data (epsData(55,i,46), i=1,3) / 0.9152050000E+02, -0.3013570000E+01, 0.4188840000E+00 /
  data (epsData(56,i,46), i=1,3) / 0.9937910000E+02, -0.2429230000E+01, 0.2547920000E+00 /
  data (epsData(57,i,46), i=1,3) / 0.1076840000E+03, -0.1962610000E+01, 0.1724020000E+00 /
  data (epsData(58,i,46), i=1,3) / 0.1165000000E+03, -0.1597480000E+01, 0.1370840000E+00 /
  data (epsData(59,i,46), i=1,3) / 0.1268840000E+03, -0.1288430000E+01, 0.1218370000E+00 /
  data (epsData(60,i,46), i=1,3) / 0.1391010000E+03, -0.1030130000E+01, 0.1145020000E+00 /
  data (epsData(61,i,46), i=1,3) / 0.1535520000E+03, -0.8163010000E+00, 0.1103480000E+00 /
  data (epsData(62,i,46), i=1,3) / 0.1697720000E+03, -0.6498280000E+00, 0.1051770000E+00 /
  data (epsData(63,i,46), i=1,3) / 0.1888750000E+03, -0.5116850000E+00, 0.9762850000E-01 /
  data (epsData(64,i,46), i=1,3) / 0.2120140000E+03, -0.3941060000E+00, 0.8569190000E-01 /
  data (epsData(65,i,46), i=1,3) / 0.2393370000E+03, -0.2956770000E+00, 0.7226920000E-01 /
  data (epsData(66,i,46), i=1,3) / 0.2703570000E+03, -0.2134620000E+00, 0.5903160000E-01 /
  data (epsData(67,i,46), i=1,3) / 0.3124820000E+03, -0.1252390000E+00, 0.4471500000E-01 /
  data (epsData(68,i,46), i=1,3) / 0.3317030000E+03, -0.8377610000E-01, 0.3969850000E-01 /
  data (epsData(69,i,46), i=1,3) / 0.3345960000E+03, -0.7497900000E-01, 0.3925240000E-01 /
  data (epsData(70,i,46), i=1,3) / 0.3358000000E+03, -0.7093970000E-01, 0.4184010000E-01 /
  data (epsData(71,i,46), i=1,3) / 0.3380790000E+03, -0.6778540000E-01, 0.4322230000E-01 /
  data (epsData(72,i,46), i=1,3) / 0.3406230000E+03, -0.6134150000E-01, 0.4442940000E-01 /
  data (epsData(73,i,46), i=1,3) / 0.3427870000E+03, -0.5904860000E-01, 0.4581350000E-01 /
  data (epsData(74,i,46), i=1,3) / 0.3490990000E+03, -0.4510090000E-01, 0.4487370000E-01 /
  data (epsData(75,i,46), i=1,3) / 0.3553580000E+03, -0.2678520000E-01, 0.4739970000E-01 /
  data (epsData(76,i,46), i=1,3) / 0.3609170000E+03, -0.9094850000E-02, 0.5333640000E-01 /
  data (epsData(77,i,46), i=1,3) / 0.3658500000E+03, 0.7232040000E-02, 0.6266430000E-01 /
  data (epsData(78,i,46), i=1,3) / 0.3703020000E+03, 0.2100430000E-01, 0.7530000000E-01 /
  data (epsData(79,i,46), i=1,3) / 0.3751130000E+03, 0.3269550000E-01, 0.9326970000E-01 /
  data (epsData(80,i,46), i=1,3) / 0.3812200000E+03, 0.3965990000E-01, 0.1199650000E+00 /
  data (epsData(81,i,46), i=1,3) / 0.3880780000E+03, 0.3651700000E-01, 0.1493490000E+00 /
  data (epsData(82,i,46), i=1,3) / 0.3957420000E+03, 0.2220120000E-01, 0.1762230000E+00 /
  data (epsData(83,i,46), i=1,3) / 0.4036850000E+03, 0.7795710000E-03, 0.1932210000E+00 /
  data (epsData(84,i,46), i=1,3) / 0.4127880000E+03, -0.2377410000E-01, 0.2008370000E+00 /
  data (epsData(85,i,46), i=1,3) / 0.4241810000E+03, -0.4926570000E-01, 0.1984460000E+00 /
  data (epsData(86,i,46), i=1,3) / 0.4419030000E+03, -0.7380430000E-01, 0.1847980000E+00 /
  data (epsData(87,i,46), i=1,3) / 0.4704110000E+03, -0.9195150000E-01, 0.1590430000E+00 /
  data (epsData(88,i,46), i=1,3) / 0.5033850000E+03, -0.9535190000E-01, 0.1328560000E+00 /
  data (epsData(89,i,46), i=1,3) / 0.5272700000E+03, -0.8849420000E-01, 0.1183690000E+00 /
  data (epsData(90,i,46), i=1,3) / 0.5332200000E+03, -0.8350510000E-01, 0.1202990000E+00 /
  data (epsData(91,i,46), i=1,3) / 0.5452340000E+03, -0.8928580000E-01, 0.1206780000E+00 /
  data (epsData(92,i,46), i=1,3) / 0.5597340000E+03, -0.8961490000E-01, 0.1139630000E+00 /
  data (epsData(93,i,46), i=1,3) / 0.5836070000E+03, -0.9391320000E-01, 0.1054990000E+00 /
  data (epsData(94,i,46), i=1,3) / 0.6382690000E+03, -0.9077490000E-01, 0.8245650000E-01 /
  data (epsData(95,i,46), i=1,3) / 0.6761770000E+03, -0.8500700000E-01, 0.7217030000E-01 /
  data (epsData(96,i,46), i=1,3) / 0.7526030000E+03, -0.7749350000E-01, 0.5361960000E-01 /
  data (epsData(97,i,46), i=1,3) / 0.8494060000E+03, -0.6607510000E-01, 0.3730190000E-01 /
  data (epsData(98,i,46), i=1,3) / 0.9446780000E+03, -0.5603770000E-01, 0.2682120000E-01 /
  data (epsData(99,i,46), i=1,3) / 0.1045900000E+04, -0.4719460000E-01, 0.1940830000E-01 /
  data (epsData(100,i,46), i=1,3) / 0.1156530000E+04, -0.3944340000E-01, 0.1400570000E-01 /
  data (epsData(101,i,46), i=1,3) / 0.1279650000E+04, -0.3267880000E-01, 0.1002760000E-01 /
  data (epsData(102,i,46), i=1,3) / 0.1417680000E+04, -0.2683960000E-01, 0.7113680000E-02 /
  data (epsData(103,i,46), i=1,3) / 0.1576460000E+04, -0.2175980000E-01, 0.4965540000E-02 /
  data (epsData(104,i,46), i=1,3) / 0.1755410000E+04, -0.1752570000E-01, 0.3448820000E-02 /
  data (epsData(105,i,46), i=1,3) / 0.1949170000E+04, -0.1412920000E-01, 0.2397670000E-02 /
  data (epsData(106,i,46), i=1,3) / 0.2164550000E+04, -0.1133780000E-01, 0.1673690000E-02 /
  data (epsData(107,i,46), i=1,3) / 0.2404370000E+04, -0.9035230000E-02, 0.1160060000E-02 /
  data (epsData(108,i,46), i=1,3) / 0.2676880000E+04, -0.7070090000E-02, 0.7951730000E-03 /
  data (epsData(109,i,46), i=1,3) / 0.3083840000E+04, -0.4650530000E-02, 0.4814690000E-03 /
  data (epsData(110,i,46), i=1,3) / 0.3142160000E+04, -0.4086930000E-02, 0.4504150000E-03 /
  data (epsData(111,i,46), i=1,3) / 0.3162370000E+04, -0.3668480000E-02, 0.4700570000E-03 /
  data (epsData(112,i,46), i=1,3) / 0.3167890000E+04, -0.3430940000E-02, 0.4960890000E-03 /
  data (epsData(113,i,46), i=1,3) / 0.3169740000E+04, -0.3303810000E-02, 0.5217710000E-03 /
  data (epsData(114,i,46), i=1,3) / 0.3171580000E+04, -0.3129140000E-02, 0.5900630000E-03 /
  data (epsData(115,i,46), i=1,3) / 0.3173490000E+04, -0.2953870000E-02, 0.8535490000E-03 /
  data (epsData(116,i,46), i=1,3) / 0.3175230000E+04, -0.3008280000E-02, 0.1129340000E-02 /
  data (epsData(117,i,46), i=1,3) / 0.3178490000E+04, -0.3193260000E-02, 0.1340720000E-02 /
  data (epsData(118,i,46), i=1,3) / 0.3185020000E+04, -0.3504070000E-02, 0.1584090000E-02 /
  data (epsData(119,i,46), i=1,3) / 0.3192270000E+04, -0.3774790000E-02, 0.1566930000E-02 /
  data (epsData(120,i,46), i=1,3) / 0.3212780000E+04, -0.3981960000E-02, 0.1482480000E-02 /
  data (epsData(121,i,46), i=1,3) / 0.3262870000E+04, -0.4068760000E-02, 0.1368490000E-02 /
  data (epsData(122,i,46), i=1,3) / 0.3312760000E+04, -0.3880970000E-02, 0.1293020000E-02 /
  data (epsData(123,i,46), i=1,3) / 0.3327060000E+04, -0.3665340000E-02, 0.1308180000E-02 /
  data (epsData(124,i,46), i=1,3) / 0.3331070000E+04, -0.3532530000E-02, 0.1456850000E-02 /
  data (epsData(125,i,46), i=1,3) / 0.3339370000E+04, -0.3681950000E-02, 0.1711020000E-02 /
  data (epsData(126,i,46), i=1,3) / 0.3350580000E+04, -0.3884060000E-02, 0.1729970000E-02 /
  data (epsData(127,i,46), i=1,3) / 0.3412000000E+04, -0.4056250000E-02, 0.1598800000E-02 /
  data (epsData(128,i,46), i=1,3) / 0.3564330000E+04, -0.3917950000E-02, 0.1363340000E-02 /
  data (epsData(129,i,46), i=1,3) / 0.3601250000E+04, -0.3752750000E-02, 0.1359470000E-02 /
  data (epsData(130,i,46), i=1,3) / 0.3611150000E+04, -0.3779650000E-02, 0.1458850000E-02 /
  data (epsData(131,i,46), i=1,3) / 0.3708020000E+04, -0.3821930000E-02, 0.1361530000E-02 /
  data (epsData(132,i,46), i=1,3) / 0.4120290000E+04, -0.3364000000E-02, 0.9418960000E-03 /
  data (epsData(133,i,46), i=1,3) / 0.4669670000E+04, -0.2728800000E-02, 0.6042150000E-03 /
  data (epsData(134,i,46), i=1,3) / 0.5245860000E+04, -0.2209310000E-02, 0.4021260000E-03 /
  data (epsData(135,i,46), i=1,3) / 0.5857690000E+04, -0.1791520000E-02, 0.2700930000E-03 /
  data (epsData(136,i,46), i=1,3) / 0.6514250000E+04, -0.1456510000E-02, 0.1842390000E-03 /
  data (epsData(137,i,46), i=1,3) / 0.7239000000E+04, -0.1182320000E-02, 0.1249620000E-03 /
  data (epsData(138,i,46), i=1,3) / 0.8061350000E+04, -0.9532610000E-03, 0.8369180000E-04 /
  data (epsData(139,i,46), i=1,3) / 0.8989100000E+04, -0.7655330000E-03, 0.5592800000E-04 /
  data (epsData(140,i,46), i=1,3) / 0.1001630000E+05, -0.6150020000E-03, 0.3724690000E-04 /
  data (epsData(141,i,46), i=1,3) / 0.1116840000E+05, -0.4931230000E-03, 0.2469820000E-04 /
  data (epsData(142,i,46), i=1,3) / 0.1245720000E+05, -0.3949350000E-03, 0.1634390000E-04 /
  data (epsData(143,i,46), i=1,3) / 0.1390330000E+05, -0.3157800000E-03, 0.1077100000E-04 /
  data (epsData(144,i,46), i=1,3) / 0.1552720000E+05, -0.2520410000E-03, 0.7058450000E-05 /
  data (epsData(145,i,46), i=1,3) / 0.1735200000E+05, -0.2007520000E-03, 0.4608820000E-05 /
  data (epsData(146,i,46), i=1,3) / 0.1940450000E+05, -0.1594110000E-03, 0.2997950000E-05 /
  data (epsData(147,i,46), i=1,3) / 0.1999900000E+05, -0.1497100000E-03, 0.2668070000E-05 /
  data (epsData(148,i,46), i=1,3) / 0.2000000000E+05, -0.1442270000E-03, 0.2667540000E-05 /
  data (epsData(149,i,46), i=1,3) / 0.2000000000E+05, -0.1496930000E-03, 0.2667510000E-05 /
  data (epsData(150,i,46), i=1,3) / 0.2248710000E+05, -0.1165410000E-03, 0.1694470000E-05 /
  data (epsData(151,i,46), i=1,3) / 0.2418110000E+05, -0.9587640000E-04, 0.1277530000E-05 /
  data (epsData(152,i,46), i=1,3) / 0.2432300000E+05, -0.9076880000E-04, 0.1249070000E-05 /
  data (epsData(153,i,46), i=1,3) / 0.2434480000E+05, -0.8758310000E-04, 0.2480660000E-05 /
  data (epsData(154,i,46), i=1,3) / 0.2434970000E+05, -0.8647160000E-04, 0.4123610000E-05 /
  data (epsData(155,i,46), i=1,3) / 0.2435000000E+05, -0.7923330000E-04, 0.4300470000E-05 /
  data (epsData(156,i,46), i=1,3) / 0.2435000000E+05, -0.8642500000E-04, 0.4326740000E-05 /
  data (epsData(157,i,46), i=1,3) / 0.2435850000E+05, -0.8827030000E-04, 0.6817270000E-05 /
  data (epsData(158,i,46), i=1,3) / 0.2441380000E+05, -0.9170970000E-04, 0.7934810000E-05 /
  data (epsData(159,i,46), i=1,3) / 0.2472000000E+05, -0.9357880000E-04, 0.7799640000E-05 /
  data (epsData(160,i,46), i=1,3) / 0.3001750000E+05, -0.6773980000E-04, 0.3878900000E-05 /
  data (epsData(161,i,46), i=1,3) / 0.3375280000E+05, -0.5403280000E-04, 0.2508150000E-05 /
  data (epsData(162,i,46), i=1,3) / 0.3783970000E+05, -0.4316810000E-04, 0.1631350000E-05 /
  data (epsData(163,i,46), i=1,3) / 0.4238050000E+05, -0.3448070000E-04, 0.1059920000E-05 /
  data (epsData(164,i,46), i=1,3) / 0.4745560000E+05, -0.2752010000E-04, 0.6859790000E-06 /
  data (epsData(165,i,46), i=1,3) / 0.5313420000E+05, -0.2195240000E-04, 0.4420770000E-06 /
  data (epsData(166,i,46), i=1,3) / 0.5949130000E+05, -0.1750430000E-04, 0.2836570000E-06 /
  data (epsData(167,i,46), i=1,3) / 0.6664150000E+05, -0.1394150000E-04, 0.1812900000E-06 /
  data (epsData(168,i,46), i=1,3) / 0.7463510000E+05, -0.1110870000E-04, 0.1152220000E-06 /
  data (epsData(169,i,46), i=1,3) / 0.8361350000E+05, -0.8846940000E-05, 0.7283950000E-07 /
  data (epsData(170,i,46), i=1,3) / 0.9364410000E+05, -0.7055280000E-05, 0.4586910000E-07 /
  data (epsData(171,i,46), i=1,3) / 0.1000000000E+06, -0.6261000000E-05, 0.3499690000E-07 /
  data (epsData(1,i,47), i=1,3) / 0.2506580000E-02, 0.2303820000E+03, 0.1186630000E+04 /
  data (epsData(2,i,47), i=1,3) / 0.4693440000E-02, 0.5044030000E+01, 0.5532460000E+03 /
  data (epsData(3,i,47), i=1,3) / 0.7560590000E-02, 0.3996850000E+02, 0.3217260000E+03 /
  data (epsData(4,i,47), i=1,3) / 0.1110800000E-01, 0.6193710000E+02, 0.2112530000E+03 /
  data (epsData(5,i,47), i=1,3) / 0.1533570000E-01, 0.7355510000E+02, 0.1497270000E+03 /
  data (epsData(6,i,47), i=1,3) / 0.2024370000E-01, 0.7995210000E+02, 0.1118390000E+03 /
  data (epsData(7,i,47), i=1,3) / 0.2583200000E-01, 0.8368170000E+02, 0.8680410000E+02 /
  data (epsData(8,i,47), i=1,3) / 0.3210060000E-01, 0.8597570000E+02, 0.6937450000E+02 /
  data (epsData(9,i,47), i=1,3) / 0.3904940000E-01, 0.8745770000E+02, 0.5674080000E+02 /
  data (epsData(10,i,47), i=1,3) / 0.4667860000E-01, 0.8846570000E+02, 0.4728490000E+02 /
  data (epsData(11,i,47), i=1,3) / 0.6397770000E-01, 0.8973860000E+02, 0.3445230000E+02 /
  data (epsData(12,i,47), i=1,3) / 0.9502860000E-01, 0.9035320000E+02, 0.2337800000E+02 /
  data (epsData(13,i,47), i=1,3) / 0.1459540000E+00, 0.9056940000E+02, 0.1530680000E+02 /
  data (epsData(14,i,47), i=1,3) / 0.2612640000E+00, 0.9079630000E+02, 0.8590490000E+01 /
  data (epsData(15,i,47), i=1,3) / 0.5918850000E+00, 0.9143550000E+02, 0.4134810000E+01 /
  data (epsData(16,i,47), i=1,3) / 0.1424510000E+01, 0.9533120000E+02, 0.2309440000E+01 /
  data (epsData(17,i,47), i=1,3) / 0.2215970000E+01, 0.1039390000E+03, 0.2174090000E+01 /
  data (epsData(18,i,47), i=1,3) / 0.2799080000E+01, 0.1161370000E+03, 0.2593080000E+01 /
  data (epsData(19,i,47), i=1,3) / 0.3181590000E+01, 0.1301630000E+03, 0.3422120000E+01 /
  data (epsData(20,i,47), i=1,3) / 0.3450210000E+01, 0.1468590000E+03, 0.5083070000E+01 /
  data (epsData(21,i,47), i=1,3) / 0.3588600000E+01, 0.1611650000E+03, 0.6133670000E+01 /
  data (epsData(22,i,47), i=1,3) / 0.3801290000E+01, 0.1913130000E+03, 0.2014380000E+02 /
  data (epsData(23,i,47), i=1,3) / 0.3873540000E+01, 0.2111980000E+03, 0.2443980000E+02 /
  data (epsData(24,i,47), i=1,3) / 0.3946480000E+01, 0.2313540000E+03, 0.5520360000E+02 /
  data (epsData(25,i,47), i=1,3) / 0.4008500000E+01, 0.2331120000E+03, 0.9089030000E+02 /
  data (epsData(26,i,47), i=1,3) / 0.4122790000E+01, 0.2127020000E+03, 0.1369930000E+03 /
  data (epsData(27,i,47), i=1,3) / 0.4195920000E+01, 0.1852020000E+03, 0.1638110000E+03 /
  data (epsData(28,i,47), i=1,3) / 0.4247960000E+01, 0.1616200000E+03, 0.1639630000E+03 /
  data (epsData(29,i,47), i=1,3) / 0.4416670000E+01, 0.1224300000E+03, 0.1596050000E+03 /
  data (epsData(30,i,47), i=1,3) / 0.4571780000E+01, 0.9618180000E+02, 0.1524630000E+03 /
  data (epsData(31,i,47), i=1,3) / 0.4850360000E+01, 0.6928060000E+02, 0.1345550000E+03 /
  data (epsData(32,i,47), i=1,3) / 0.5144240000E+01, 0.5507940000E+02, 0.1168210000E+03 /
  data (epsData(33,i,47), i=1,3) / 0.5437100000E+01, 0.5048090000E+02, 0.1014440000E+03 /
  data (epsData(34,i,47), i=1,3) / 0.5714660000E+01, 0.5338820000E+02, 0.8996720000E+02 /
  data (epsData(35,i,47), i=1,3) / 0.5964660000E+01, 0.6298520000E+02, 0.8359090000E+02 /
  data (epsData(36,i,47), i=1,3) / 0.6069420000E+01, 0.7160060000E+02, 0.8240020000E+02 /
  data (epsData(37,i,47), i=1,3) / 0.6287120000E+01, 0.8879210000E+02, 0.9422940000E+02 /
  data (epsData(38,i,47), i=1,3) / 0.6343240000E+01, 0.9760910000E+02, 0.9710120000E+02 /
  data (epsData(39,i,47), i=1,3) / 0.6400040000E+01, 0.1055910000E+03, 0.1131720000E+03 /
  data (epsData(40,i,47), i=1,3) / 0.6457530000E+01, 0.1046480000E+03, 0.1334700000E+03 /
  data (epsData(41,i,47), i=1,3) / 0.6574540000E+01, 0.8781080000E+02, 0.1612310000E+03 /
  data (epsData(42,i,47), i=1,3) / 0.6634060000E+01, 0.7241900000E+02, 0.1747760000E+03 /
  data (epsData(43,i,47), i=1,3) / 0.6694260000E+01, 0.5387760000E+02, 0.1779170000E+03 /
  data (epsData(44,i,47), i=1,3) / 0.6816720000E+01, 0.2732100000E+02, 0.1706550000E+03 /
  data (epsData(45,i,47), i=1,3) / 0.6941890000E+01, 0.8221740000E+01, 0.1636390000E+03 /
  data (epsData(46,i,47), i=1,3) / 0.7069780000E+01, -0.7374470000E+01, 0.1517410000E+03 /
  data (epsData(47,i,47), i=1,3) / 0.7266720000E+01, -0.2174080000E+02, 0.1335080000E+03 /
  data (epsData(48,i,47), i=1,3) / 0.7469790000E+01, -0.2973790000E+02, 0.1163440000E+03 /
  data (epsData(49,i,47), i=1,3) / 0.7750070000E+01, -0.3382730000E+02, 0.9636710000E+02 /
  data (epsData(50,i,47), i=1,3) / 0.8041230000E+01, -0.3342910000E+02, 0.8011240000E+02 /
  data (epsData(51,i,47), i=1,3) / 0.8420490000E+01, -0.2978220000E+02, 0.6450720000E+02 /
  data (epsData(52,i,47), i=1,3) / 0.8898040000E+01, -0.2353960000E+02, 0.5153870000E+02 /
  data (epsData(53,i,47), i=1,3) / 0.9230020000E+01, -0.1899400000E+02, 0.4577360000E+02 /
  data (epsData(54,i,47), i=1,3) / 0.9660300000E+01, -0.1471580000E+02, 0.4097640000E+02 /
  data (epsData(55,i,47), i=1,3) / 0.1038410000E+02, -0.9212090000E+01, 0.3460540000E+02 /
  data (epsData(56,i,47), i=1,3) / 0.1105320000E+02, -0.4894700000E+01, 0.3109390000E+02 /
  data (epsData(57,i,47), i=1,3) / 0.1175560000E+02, -0.1426340000E+01, 0.2888560000E+02 /
  data (epsData(58,i,47), i=1,3) / 0.1259910000E+02, 0.1585640000E+01, 0.2743450000E+02 /
  data (epsData(59,i,47), i=1,3) / 0.1371480000E+02, 0.4105230000E+01, 0.2662610000E+02 /
  data (epsData(60,i,47), i=1,3) / 0.1526690000E+02, 0.5702300000E+01, 0.2643130000E+02 /
  data (epsData(61,i,47), i=1,3) / 0.1774210000E+02, 0.5524290000E+01, 0.2663010000E+02 /
  data (epsData(62,i,47), i=1,3) / 0.2091490000E+02, 0.3061660000E+01, 0.2621580000E+02 /
  data (epsData(63,i,47), i=1,3) / 0.2492010000E+02, -0.5768150000E+00, 0.2412880000E+02 /
  data (epsData(64,i,47), i=1,3) / 0.2991060000E+02, -0.3623350000E+01, 0.2069470000E+02 /
  data (epsData(65,i,47), i=1,3) / 0.3796520000E+02, -0.6781610000E+01, 0.1585230000E+02 /
  data (epsData(66,i,47), i=1,3) / 0.4519250000E+02, -0.8432120000E+01, 0.1154740000E+02 /
  data (epsData(67,i,47), i=1,3) / 0.5056540000E+02, -0.8601470000E+01, 0.8468090000E+01 /
  data (epsData(68,i,47), i=1,3) / 0.5573560000E+02, -0.8026540000E+01, 0.6123730000E+01 /
  data (epsData(69,i,47), i=1,3) / 0.6170140000E+02, -0.7116010000E+01, 0.4274270000E+01 /
  data (epsData(70,i,47), i=1,3) / 0.6812050000E+02, -0.6206790000E+01, 0.2862120000E+01 /
  data (epsData(71,i,47), i=1,3) / 0.7447710000E+02, -0.5321090000E+01, 0.1825740000E+01 /
  data (epsData(72,i,47), i=1,3) / 0.8031290000E+02, -0.4529520000E+01, 0.1169890000E+01 /
  data (epsData(73,i,47), i=1,3) / 0.8669360000E+02, -0.3768470000E+01, 0.7192630000E+00 /
  data (epsData(74,i,47), i=1,3) / 0.9370360000E+02, -0.3085880000E+01, 0.4393650000E+00 /
  data (epsData(75,i,47), i=1,3) / 0.1015770000E+03, -0.2504820000E+01, 0.2796870000E+00 /
  data (epsData(76,i,47), i=1,3) / 0.1100670000E+03, -0.2031060000E+01, 0.1830540000E+00 /
  data (epsData(77,i,47), i=1,3) / 0.1187720000E+03, -0.1662890000E+01, 0.1399430000E+00 /
  data (epsData(78,i,47), i=1,3) / 0.1288500000E+03, -0.1351240000E+01, 0.1236510000E+00 /
  data (epsData(79,i,47), i=1,3) / 0.1413280000E+03, -0.1078500000E+01, 0.1168020000E+00 /
  data (epsData(80,i,47), i=1,3) / 0.1561470000E+03, -0.8534350000E+00, 0.1133230000E+00 /
  data (epsData(81,i,47), i=1,3) / 0.1724580000E+03, -0.6823170000E+00, 0.1091480000E+00 /
  data (epsData(82,i,47), i=1,3) / 0.1921870000E+03, -0.5377820000E+00, 0.1021200000E+00 /
  data (epsData(83,i,47), i=1,3) / 0.2158070000E+03, -0.4173220000E+00, 0.9081140000E-01 /
  data (epsData(84,i,47), i=1,3) / 0.2438250000E+03, -0.3175020000E+00, 0.7710140000E-01 /
  data (epsData(85,i,47), i=1,3) / 0.2757910000E+03, -0.2354400000E+00, 0.6321800000E-01 /
  data (epsData(86,i,47), i=1,3) / 0.3125240000E+03, -0.1639240000E+00, 0.5010680000E-01 /
  data (epsData(87,i,47), i=1,3) / 0.3586200000E+03, -0.8466800000E-01, 0.3781890000E-01 /
  data (epsData(88,i,47), i=1,3) / 0.3664330000E+03, -0.6749810000E-01, 0.3629080000E-01 /
  data (epsData(89,i,47), i=1,3) / 0.3676460000E+03, -0.6318960000E-01, 0.3641380000E-01 /
  data (epsData(90,i,47), i=1,3) / 0.3684650000E+03, -0.6004660000E-01, 0.3949100000E-01 /
  data (epsData(91,i,47), i=1,3) / 0.3695660000E+03, -0.6086480000E-01, 0.4103150000E-01 /
  data (epsData(92,i,47), i=1,3) / 0.3732800000E+03, -0.5413730000E-01, 0.4001930000E-01 /
  data (epsData(93,i,47), i=1,3) / 0.3741230000E+03, -0.5113560000E-01, 0.4204110000E-01 /
  data (epsData(94,i,47), i=1,3) / 0.3751840000E+03, -0.5165330000E-01, 0.4293830000E-01 /
  data (epsData(95,i,47), i=1,3) / 0.3800170000E+03, -0.4279850000E-01, 0.4124240000E-01 /
  data (epsData(96,i,47), i=1,3) / 0.3858940000E+03, -0.2738190000E-01, 0.4254750000E-01 /
  data (epsData(97,i,47), i=1,3) / 0.3915270000E+03, -0.1108090000E-01, 0.4736370000E-01 /
  data (epsData(98,i,47), i=1,3) / 0.3962920000E+03, 0.3552940000E-02, 0.5491910000E-01 /
  data (epsData(99,i,47), i=1,3) / 0.4006670000E+03, 0.1655570000E-01, 0.6563460000E-01 /
  data (epsData(100,i,47), i=1,3) / 0.4052280000E+03, 0.2778980000E-01, 0.8085970000E-01 /
  data (epsData(101,i,47), i=1,3) / 0.4103710000E+03, 0.3527250000E-01, 0.1016640000E+00 /
  data (epsData(102,i,47), i=1,3) / 0.4171080000E+03, 0.3521410000E-01, 0.1298150000E+00 /
  data (epsData(103,i,47), i=1,3) / 0.4238950000E+03, 0.2548700000E-01, 0.1545160000E+00 /
  data (epsData(104,i,47), i=1,3) / 0.4306210000E+03, 0.9702500000E-02, 0.1722050000E+00 /
  data (epsData(105,i,47), i=1,3) / 0.4374790000E+03, -0.8762610000E-02, 0.1825950000E+00 /
  data (epsData(106,i,47), i=1,3) / 0.4462880000E+03, -0.3098060000E-01, 0.1864270000E+00 /
  data (epsData(107,i,47), i=1,3) / 0.4583570000E+03, -0.5438080000E-01, 0.1813130000E+00 /
  data (epsData(108,i,47), i=1,3) / 0.4795300000E+03, -0.7750650000E-01, 0.1647880000E+00 /
  data (epsData(109,i,47), i=1,3) / 0.5090050000E+03, -0.9066860000E-01, 0.1403620000E+00 /
  data (epsData(110,i,47), i=1,3) / 0.5449860000E+03, -0.9167790000E-01, 0.1159540000E+00 /
  data (epsData(111,i,47), i=1,3) / 0.5694380000E+03, -0.8481440000E-01, 0.1035230000E+00 /
  data (epsData(112,i,47), i=1,3) / 0.5769670000E+03, -0.8146760000E-01, 0.1051800000E+00 /
  data (epsData(113,i,47), i=1,3) / 0.5921050000E+03, -0.8529100000E-01, 0.1031080000E+00 /
  data (epsData(114,i,47), i=1,3) / 0.6122910000E+03, -0.8514690000E-01, 0.9805050000E-01 /
  data (epsData(115,i,47), i=1,3) / 0.6615370000E+03, -0.8603880000E-01, 0.8023240000E-01 /
  data (epsData(116,i,47), i=1,3) / 0.7135390000E+03, -0.7956390000E-01, 0.6527500000E-01 /
  data (epsData(117,i,47), i=1,3) / 0.8318530000E+03, -0.6855380000E-01, 0.4293860000E-01 /
  data (epsData(118,i,47), i=1,3) / 0.9482750000E+03, -0.5672250000E-01, 0.2881950000E-01 /
  data (epsData(119,i,47), i=1,3) / 0.1059150000E+04, -0.4734790000E-01, 0.2034010000E-01 /
  data (epsData(120,i,47), i=1,3) / 0.1173560000E+04, -0.3956730000E-01, 0.1460820000E-01 /
  data (epsData(121,i,47), i=1,3) / 0.1297400000E+04, -0.3291800000E-01, 0.1050330000E-01 /
  data (epsData(122,i,47), i=1,3) / 0.1434170000E+04, -0.2721170000E-01, 0.7514690000E-02 /
  data (epsData(123,i,47), i=1,3) / 0.1589790000E+04, -0.2225490000E-01, 0.5304410000E-02 /
  data (epsData(124,i,47), i=1,3) / 0.1767880000E+04, -0.1799670000E-01, 0.3696910000E-02 /
  data (epsData(125,i,47), i=1,3) / 0.1969030000E+04, -0.1445190000E-01, 0.2545160000E-02 /
  data (epsData(126,i,47), i=1,3) / 0.2184990000E+04, -0.1164140000E-01, 0.1782190000E-02 /
  data (epsData(127,i,47), i=1,3) / 0.2427290000E+04, -0.9311310000E-02, 0.1235970000E-02 /
  data (epsData(128,i,47), i=1,3) / 0.2697940000E+04, -0.7373200000E-02, 0.8524500000E-03 /
  data (epsData(129,i,47), i=1,3) / 0.3023760000E+04, -0.5596010000E-02, 0.5693830000E-03 /
  data (epsData(130,i,47), i=1,3) / 0.3272750000E+04, -0.4296810000E-02, 0.4296530000E-03 /
  data (epsData(131,i,47), i=1,3) / 0.3325210000E+04, -0.3828980000E-02, 0.4059830000E-03 /
  data (epsData(132,i,47), i=1,3) / 0.3344080000E+04, -0.3462990000E-02, 0.4287730000E-03 /
  data (epsData(133,i,47), i=1,3) / 0.3347860000E+04, -0.3311490000E-02, 0.4500800000E-03 /
  data (epsData(134,i,47), i=1,3) / 0.3351270000E+04, -0.3098490000E-02, 0.5636160000E-03 /
  data (epsData(135,i,47), i=1,3) / 0.3355040000E+04, -0.3016500000E-02, 0.8755850000E-03 /
  data (epsData(136,i,47), i=1,3) / 0.3359710000E+04, -0.3135910000E-02, 0.1179700000E-02 /
  data (epsData(137,i,47), i=1,3) / 0.3365860000E+04, -0.3392570000E-02, 0.1311530000E-02 /
  data (epsData(138,i,47), i=1,3) / 0.3383690000E+04, -0.3667610000E-02, 0.1312170000E-02 /
  data (epsData(139,i,47), i=1,3) / 0.3419270000E+04, -0.3825140000E-02, 0.1259670000E-02 /
  data (epsData(140,i,47), i=1,3) / 0.3493390000E+04, -0.3735810000E-02, 0.1152680000E-02 /
  data (epsData(141,i,47), i=1,3) / 0.3520540000E+04, -0.3518310000E-02, 0.1145610000E-02 /
  data (epsData(142,i,47), i=1,3) / 0.3526690000E+04, -0.3390780000E-02, 0.1270170000E-02 /
  data (epsData(143,i,47), i=1,3) / 0.3534540000E+04, -0.3471130000E-02, 0.1471990000E-02 /
  data (epsData(144,i,47), i=1,3) / 0.3550860000E+04, -0.3650610000E-02, 0.1498500000E-02 /
  data (epsData(145,i,47), i=1,3) / 0.3625400000E+04, -0.3785800000E-02, 0.1392940000E-02 /
  data (epsData(146,i,47), i=1,3) / 0.3778510000E+04, -0.3615530000E-02, 0.1199020000E-02 /
  data (epsData(147,i,47), i=1,3) / 0.3803210000E+04, -0.3484660000E-02, 0.1223430000E-02 /
  data (epsData(148,i,47), i=1,3) / 0.3810980000E+04, -0.3521270000E-02, 0.1315350000E-02 /
  data (epsData(149,i,47), i=1,3) / 0.3919240000E+04, -0.3557970000E-02, 0.1210720000E-02 /
  data (epsData(150,i,47), i=1,3) / 0.4379970000E+04, -0.3091410000E-02, 0.8202670000E-03 /
  data (epsData(151,i,47), i=1,3) / 0.4943900000E+04, -0.2518750000E-02, 0.5356710000E-03 /
  data (epsData(152,i,47), i=1,3) / 0.5537230000E+04, -0.2048060000E-02, 0.3587120000E-03 /
  data (epsData(153,i,47), i=1,3) / 0.6182140000E+04, -0.1660160000E-02, 0.2409610000E-03 /
  data (epsData(154,i,47), i=1,3) / 0.6902200000E+04, -0.1338800000E-02, 0.1615810000E-03 /
  data (epsData(155,i,47), i=1,3) / 0.7701200000E+04, -0.1077760000E-02, 0.1081760000E-03 /
  data (epsData(156,i,47), i=1,3) / 0.8568800000E+04, -0.8705240000E-03, 0.7273030000E-04 /
  data (epsData(157,i,47), i=1,3) / 0.9551470000E+04, -0.6994220000E-03, 0.4857250000E-04 /
  data (epsData(158,i,47), i=1,3) / 0.1064760000E+05, -0.5614090000E-03, 0.3229250000E-04 /
  data (epsData(159,i,47), i=1,3) / 0.1187570000E+05, -0.4498940000E-03, 0.2139610000E-04 /
  data (epsData(160,i,47), i=1,3) / 0.1325170000E+05, -0.3600330000E-03, 0.1413030000E-04 /
  data (epsData(161,i,47), i=1,3) / 0.1479390000E+05, -0.2877450000E-03, 0.9286010000E-05 /
  data (epsData(162,i,47), i=1,3) / 0.1652330000E+05, -0.2296400000E-03, 0.6088520000E-05 /
  data (epsData(163,i,47), i=1,3) / 0.1846390000E+05, -0.1829350000E-03, 0.3974330000E-05 /
  data (epsData(164,i,47), i=1,3) / 0.2065280000E+05, -0.1451740000E-03, 0.2581190000E-05 /
  data (epsData(165,i,47), i=1,3) / 0.2104290000E+05, -0.1396280000E-03, 0.2400010000E-05 /
  data (epsData(166,i,47), i=1,3) / 0.2104400000E+05, -0.1346670000E-03, 0.2399550000E-05 /
  data (epsData(167,i,47), i=1,3) / 0.2104400000E+05, -0.1396120000E-03, 0.2399530000E-05 /
  data (epsData(168,i,47), i=1,3) / 0.2367290000E+05, -0.1085660000E-03, 0.1521460000E-05 /
  data (epsData(169,i,47), i=1,3) / 0.2534760000E+05, -0.9021290000E-04, 0.1166120000E-05 /
  data (epsData(170,i,47), i=1,3) / 0.2548660000E+05, -0.8557810000E-04, 0.1141810000E-05 /
  data (epsData(171,i,47), i=1,3) / 0.2550870000E+05, -0.8236920000E-04, 0.2574900000E-05 /
  data (epsData(172,i,47), i=1,3) / 0.2551370000E+05, -0.8153680000E-04, 0.4510230000E-05 /
  data (epsData(173,i,47), i=1,3) / 0.2551400000E+05, -0.7367350000E-04, 0.4693990000E-05 /
  data (epsData(174,i,47), i=1,3) / 0.2551400000E+05, -0.8154180000E-04, 0.4720410000E-05 /
  data (epsData(175,i,47), i=1,3) / 0.2552330000E+05, -0.8382180000E-04, 0.6454750000E-05 /
  data (epsData(176,i,47), i=1,3) / 0.2559370000E+05, -0.8696830000E-04, 0.7247620000E-05 /
  data (epsData(177,i,47), i=1,3) / 0.2604630000E+05, -0.8789250000E-04, 0.6941030000E-05 /
  data (epsData(178,i,47), i=1,3) / 0.3117040000E+05, -0.6478420000E-04, 0.3635640000E-05 /
  data (epsData(179,i,47), i=1,3) / 0.3505670000E+05, -0.5166110000E-04, 0.2348900000E-05 /
  data (epsData(180,i,47), i=1,3) / 0.3930960000E+05, -0.4125890000E-04, 0.1526650000E-05 /
  data (epsData(181,i,47), i=1,3) / 0.4403330000E+05, -0.3294690000E-04, 0.9911340000E-06 /
  data (epsData(182,i,47), i=1,3) / 0.4930890000E+05, -0.2629430000E-04, 0.6413010000E-06 /
  data (epsData(183,i,47), i=1,3) / 0.5521290000E+05, -0.2097240000E-04, 0.4131370000E-06 /
  data (epsData(184,i,47), i=1,3) / 0.6181980000E+05, -0.1672270000E-04, 0.2658160000E-06 /
  data (epsData(185,i,47), i=1,3) / 0.6922840000E+05, -0.1332900000E-04, 0.1696770000E-06 /
  data (epsData(186,i,47), i=1,3) / 0.7754300000E+05, -0.1061800000E-04, 0.1076930000E-06 /
  data (epsData(187,i,47), i=1,3) / 0.8686880000E+05, -0.8458470000E-05, 0.6795960000E-07 /
  data (epsData(188,i,47), i=1,3) / 0.9715680000E+05, -0.6771360000E-05, 0.4303600000E-07 /
  data (epsData(189,i,47), i=1,3) / 0.1000000000E+06, -0.6473950000E-05, 0.3822720000E-07 /
  data (epsData(1,i,48), i=1,3) / 0.2506580000E-02, -0.2102450000E+04, 0.5705950000E+04 /
  data (epsData(2,i,48), i=1,3) / 0.4693440000E-02, -0.1757190000E+04, 0.1948550000E+04 /
  data (epsData(3,i,48), i=1,3) / 0.7560590000E-02, -0.7873730000E+03, 0.9131340000E+03 /
  data (epsData(4,i,48), i=1,3) / 0.1110800000E-01, -0.3638020000E+03, 0.5160220000E+03 /
  data (epsData(5,i,48), i=1,3) / 0.1533570000E-01, -0.1644770000E+03, 0.3288700000E+03 /
  data (epsData(6,i,48), i=1,3) / 0.2024370000E-01, -0.6128400000E+02, 0.2274720000E+03 /
  data (epsData(7,i,48), i=1,3) / 0.2583200000E-01, -0.3449260000E+01, 0.1667860000E+03 /
  data (epsData(8,i,48), i=1,3) / 0.3210060000E-01, 0.3112790000E+02, 0.1276830000E+03 /
  data (epsData(9,i,48), i=1,3) / 0.3904940000E-01, 0.5296770000E+02, 0.1010230000E+03 /
  data (epsData(10,i,48), i=1,3) / 0.4667860000E-01, 0.6752640000E+02, 0.8202420000E+02 /
  data (epsData(11,i,48), i=1,3) / 0.5498800000E-01, 0.7801650000E+02, 0.6799600000E+02 /
  data (epsData(12,i,48), i=1,3) / 0.6397770000E-01, 0.8523860000E+02, 0.5926840000E+02 /
  data (epsData(13,i,48), i=1,3) / 0.7364770000E-01, 0.8948420000E+02, 0.5226250000E+02 /
  data (epsData(14,i,48), i=1,3) / 0.9502860000E-01, 0.9390210000E+02, 0.4139910000E+02 /
  data (epsData(15,i,48), i=1,3) / 0.1322020000E+00, 0.9664860000E+02, 0.3038160000E+02 /
  data (epsData(16,i,48), i=1,3) / 0.1912910000E+00, 0.9806270000E+02, 0.2133700000E+02 /
  data (epsData(17,i,48), i=1,3) / 0.3208860000E+00, 0.9933770000E+02, 0.1290290000E+02 /
  data (epsData(18,i,48), i=1,3) / 0.6800800000E+00, 0.1019410000E+03, 0.7304820000E+01 /
  data (epsData(19,i,48), i=1,3) / 0.1212890000E+01, 0.1086950000E+03, 0.5848160000E+01 /
  data (epsData(20,i,48), i=1,3) / 0.1653130000E+01, 0.1196750000E+03, 0.6373770000E+01 /
  data (epsData(21,i,48), i=1,3) / 0.2001780000E+01, 0.1342120000E+03, 0.7862000000E+01 /
  data (epsData(22,i,48), i=1,3) / 0.2215970000E+01, 0.1481450000E+03, 0.9618380000E+01 /
  data (epsData(23,i,48), i=1,3) / 0.2383760000E+01, 0.1627890000E+03, 0.1187620000E+02 /
  data (epsData(24,i,48), i=1,3) / 0.2499020000E+01, 0.1776210000E+03, 0.1486540000E+02 /
  data (epsData(25,i,48), i=1,3) / 0.2617000000E+01, 0.1952980000E+03, 0.2021540000E+02 /
  data (epsData(26,i,48), i=1,3) / 0.2677010000E+01, 0.2073760000E+03, 0.2255680000E+02 /
  data (epsData(27,i,48), i=1,3) / 0.2737710000E+01, 0.2255710000E+03, 0.2469850000E+02 /
  data (epsData(28,i,48), i=1,3) / 0.2799080000E+01, 0.2456270000E+03, 0.4122320000E+02 /
  data (epsData(29,i,48), i=1,3) / 0.2987270000E+01, 0.2883740000E+03, 0.1002230000E+03 /
  data (epsData(30,i,48), i=1,3) / 0.3051370000E+01, 0.3218280000E+03, 0.1169570000E+03 /
  data (epsData(31,i,48), i=1,3) / 0.3116140000E+01, 0.3398650000E+03, 0.2078090000E+03 /
  data (epsData(32,i,48), i=1,3) / 0.3181590000E+01, 0.3124880000E+03, 0.2843270000E+03 /
  data (epsData(33,i,48), i=1,3) / 0.3247730000E+01, 0.2653940000E+03, 0.3350530000E+03 /
  data (epsData(34,i,48), i=1,3) / 0.3314540000E+01, 0.2033770000E+03, 0.3820020000E+03 /
  data (epsData(35,i,48), i=1,3) / 0.3382030000E+01, 0.1271720000E+03, 0.3985970000E+03 /
  data (epsData(36,i,48), i=1,3) / 0.3450210000E+01, 0.6319100000E+02, 0.3722810000E+03 /
  data (epsData(37,i,48), i=1,3) / 0.3519060000E+01, 0.2114500000E+02, 0.3474480000E+03 /
  data (epsData(38,i,48), i=1,3) / 0.3658810000E+01, -0.3757080000E+02, 0.3016140000E+03 /
  data (epsData(39,i,48), i=1,3) / 0.3729710000E+01, -0.5885060000E+02, 0.2744520000E+03 /
  data (epsData(40,i,48), i=1,3) / 0.3801290000E+01, -0.7338680000E+02, 0.2488530000E+03 /
  data (epsData(41,i,48), i=1,3) / 0.3873540000E+01, -0.8307330000E+02, 0.2247220000E+03 /
  data (epsData(42,i,48), i=1,3) / 0.3946480000E+01, -0.8825880000E+02, 0.2019700000E+03 /
  data (epsData(43,i,48), i=1,3) / 0.4169370000E+01, -0.9337710000E+02, 0.1519730000E+03 /
  data (epsData(44,i,48), i=1,3) / 0.4321360000E+01, -0.9036300000E+02, 0.1257840000E+03 /
  data (epsData(45,i,48), i=1,3) / 0.4554460000E+01, -0.8426060000E+02, 0.9598540000E+02 /
  data (epsData(46,i,48), i=1,3) / 0.4874780000E+01, -0.7385440000E+02, 0.6794630000E+02 /
  data (epsData(47,i,48), i=1,3) / 0.5205980000E+01, -0.6285860000E+02, 0.4759420000E+02 /
  data (epsData(48,i,48), i=1,3) / 0.5548060000E+01, -0.5169300000E+02, 0.3292960000E+02 /
  data (epsData(49,i,48), i=1,3) / 0.5901040000E+01, -0.4104540000E+02, 0.2276780000E+02 /
  data (epsData(50,i,48), i=1,3) / 0.6172910000E+01, -0.3357610000E+02, 0.1757390000E+02 /
  data (epsData(51,i,48), i=1,3) / 0.6357560000E+01, -0.2906160000E+02, 0.1498710000E+02 /
  data (epsData(52,i,48), i=1,3) / 0.6639630000E+01, -0.2355500000E+02, 0.1285170000E+02 /
  data (epsData(53,i,48), i=1,3) / 0.7222150000E+01, -0.1510580000E+02, 0.8193230000E+01 /
  data (epsData(54,i,48), i=1,3) / 0.7624100000E+01, -0.9869340000E+01, 0.5998890000E+01 /
  data (epsData(55,i,48), i=1,3) / 0.7932710000E+01, -0.6280560000E+01, 0.4812630000E+01 /
  data (epsData(56,i,48), i=1,3) / 0.8247440000E+01, -0.2937180000E+01, 0.3906270000E+01 /
  data (epsData(57,i,48), i=1,3) / 0.8460660000E+01, -0.8114250000E+00, 0.3415430000E+01 /
  data (epsData(58,i,48), i=1,3) / 0.8785590000E+01, 0.2293710000E+01, 0.2841070000E+01 /
  data (epsData(59,i,48), i=1,3) / 0.9340750000E+01, 0.7531970000E+01, 0.2197290000E+01 /
  data (epsData(60,i,48), i=1,3) / 0.9682010000E+01, 0.1105010000E+02, 0.1980370000E+01 /
  data (epsData(61,i,48), i=1,3) / 0.9912920000E+01, 0.1382250000E+02, 0.1939470000E+01 /
  data (epsData(62,i,48), i=1,3) / 0.1014660000E+02, 0.1735350000E+02, 0.2081250000E+01 /
  data (epsData(63,i,48), i=1,3) / 0.1026440000E+02, 0.1975330000E+02, 0.2199340000E+01 /
  data (epsData(64,i,48), i=1,3) / 0.1038290000E+02, 0.2259470000E+02, 0.3220450000E+01 /
  data (epsData(65,i,48), i=1,3) / 0.1050210000E+02, 0.2597890000E+02, 0.4589840000E+01 /
  data (epsData(66,i,48), i=1,3) / 0.1062200000E+02, 0.3055930000E+02, 0.7527070000E+01 /
  data (epsData(67,i,48), i=1,3) / 0.1070540000E+02, 0.3212430000E+02, 0.1429720000E+02 /
  data (epsData(68,i,48), i=1,3) / 0.1085000000E+02, 0.2914000000E+02, 0.2210400000E+02 /
  data (epsData(69,i,48), i=1,3) / 0.1089590000E+02, 0.2668280000E+02, 0.2449760000E+02 /
  data (epsData(70,i,48), i=1,3) / 0.1094800000E+02, 0.2366860000E+02, 0.2446390000E+02 /
  data (epsData(71,i,48), i=1,3) / 0.1109320000E+02, 0.1992570000E+02, 0.2380600000E+02 /
  data (epsData(72,i,48), i=1,3) / 0.1124420000E+02, 0.1772540000E+02, 0.2291340000E+02 /
  data (epsData(73,i,48), i=1,3) / 0.1138880000E+02, 0.1745460000E+02, 0.2208370000E+02 /
  data (epsData(74,i,48), i=1,3) / 0.1155040000E+02, 0.1843940000E+02, 0.2173170000E+02 /
  data (epsData(75,i,48), i=1,3) / 0.1158470000E+02, 0.1923710000E+02, 0.2171560000E+02 /
  data (epsData(76,i,48), i=1,3) / 0.1161970000E+02, 0.2018920000E+02, 0.2261160000E+02 /
  data (epsData(77,i,48), i=1,3) / 0.1170030000E+02, 0.2022670000E+02, 0.2622780000E+02 /
  data (epsData(78,i,48), i=1,3) / 0.1186460000E+02, 0.1663830000E+02, 0.3041220000E+02 /
  data (epsData(79,i,48), i=1,3) / 0.1191260000E+02, 0.1439070000E+02, 0.3121150000E+02 /
  data (epsData(80,i,48), i=1,3) / 0.1204830000E+02, 0.1057470000E+02, 0.2996720000E+02 /
  data (epsData(81,i,48), i=1,3) / 0.1230000000E+02, 0.6744980000E+01, 0.2767860000E+02 /
  data (epsData(82,i,48), i=1,3) / 0.1255040000E+02, 0.4784250000E+01, 0.2528590000E+02 /
  data (epsData(83,i,48), i=1,3) / 0.1292450000E+02, 0.3298440000E+01, 0.2252010000E+02 /
  data (epsData(84,i,48), i=1,3) / 0.1346330000E+02, 0.2425600000E+01, 0.1943870000E+02 /
  data (epsData(85,i,48), i=1,3) / 0.1398710000E+02, 0.2464870000E+01, 0.1723460000E+02 /
  data (epsData(86,i,48), i=1,3) / 0.1464190000E+02, 0.2651140000E+01, 0.1502780000E+02 /
  data (epsData(87,i,48), i=1,3) / 0.1581570000E+02, 0.3903910000E+01, 0.1246250000E+02 /
  data (epsData(88,i,48), i=1,3) / 0.1693000000E+02, 0.5477680000E+01, 0.1119060000E+02 /
  data (epsData(89,i,48), i=1,3) / 0.1817770000E+02, 0.7010340000E+01, 0.1083360000E+02 /
  data (epsData(90,i,48), i=1,3) / 0.1976680000E+02, 0.8331560000E+01, 0.1127670000E+02 /
  data (epsData(91,i,48), i=1,3) / 0.2188010000E+02, 0.9018390000E+01, 0.1256480000E+02 /
  data (epsData(92,i,48), i=1,3) / 0.2463390000E+02, 0.8401920000E+01, 0.1437270000E+02 /
  data (epsData(93,i,48), i=1,3) / 0.2771690000E+02, 0.6516870000E+01, 0.1570860000E+02 /
  data (epsData(94,i,48), i=1,3) / 0.3162010000E+02, 0.3680850000E+01, 0.1601950000E+02 /
  data (epsData(95,i,48), i=1,3) / 0.3761060000E+02, 0.2252540000E+00, 0.1513980000E+02 /
  data (epsData(96,i,48), i=1,3) / 0.4438760000E+02, -0.3112130000E+01, 0.1344540000E+02 /
  data (epsData(97,i,48), i=1,3) / 0.4919180000E+02, -0.5081420000E+01, 0.1151990000E+02 /
  data (epsData(98,i,48), i=1,3) / 0.5408470000E+02, -0.6284980000E+01, 0.9156190000E+01 /
  data (epsData(99,i,48), i=1,3) / 0.5877020000E+02, -0.6585470000E+01, 0.6985340000E+01 /
  data (epsData(100,i,48), i=1,3) / 0.6441740000E+02, -0.6304130000E+01, 0.4938620000E+01 /
  data (epsData(101,i,48), i=1,3) / 0.7186240000E+02, -0.5552890000E+01, 0.3023330000E+01 /
  data (epsData(102,i,48), i=1,3) / 0.7904350000E+02, -0.4687460000E+01, 0.1797790000E+01 /
  data (epsData(103,i,48), i=1,3) / 0.8619150000E+02, -0.3848900000E+01, 0.1046260000E+01 /
  data (epsData(104,i,48), i=1,3) / 0.9350080000E+02, -0.3120090000E+01, 0.6084200000E+00 /
  data (epsData(105,i,48), i=1,3) / 0.1009410000E+03, -0.2532210000E+01, 0.3637850000E+00 /
  data (epsData(106,i,48), i=1,3) / 0.1084740000E+03, -0.2065510000E+01, 0.2331000000E+00 /
  data (epsData(107,i,48), i=1,3) / 0.1180780000E+03, -0.1637990000E+01, 0.1549610000E+00 /
  data (epsData(108,i,48), i=1,3) / 0.1264130000E+03, -0.1355490000E+01, 0.1268850000E+00 /
  data (epsData(109,i,48), i=1,3) / 0.1366580000E+03, -0.1102870000E+01, 0.1159070000E+00 /
  data (epsData(110,i,48), i=1,3) / 0.1495840000E+03, -0.8783880000E+00, 0.1123620000E+00 /
  data (epsData(111,i,48), i=1,3) / 0.1650420000E+03, -0.6938400000E+00, 0.1108360000E+00 /
  data (epsData(112,i,48), i=1,3) / 0.1827160000E+03, -0.5501990000E+00, 0.1072390000E+00 /
  data (epsData(113,i,48), i=1,3) / 0.2024840000E+03, -0.4391950000E+00, 0.1010560000E+00 /
  data (epsData(114,i,48), i=1,3) / 0.2270040000E+03, -0.3428000000E+00, 0.9043930000E-01 /
  data (epsData(115,i,48), i=1,3) / 0.2557320000E+03, -0.2633920000E+00, 0.7731370000E-01 /
  data (epsData(116,i,48), i=1,3) / 0.2885910000E+03, -0.1977080000E+00, 0.6362750000E-01 /
  data (epsData(117,i,48), i=1,3) / 0.3269520000E+03, -0.1395370000E+00, 0.5045480000E-01 /
  data (epsData(118,i,48), i=1,3) / 0.3871620000E+03, -0.6198450000E-01, 0.3551410000E-01 /
  data (epsData(119,i,48), i=1,3) / 0.4002130000E+03, -0.4166350000E-01, 0.3317730000E-01 /
  data (epsData(120,i,48), i=1,3) / 0.4039970000E+03, -0.3306350000E-01, 0.3285600000E-01 /
  data (epsData(121,i,48), i=1,3) / 0.4046300000E+03, -0.3067890000E-01, 0.3327540000E-01 /
  data (epsData(122,i,48), i=1,3) / 0.4053360000E+03, -0.2860610000E-01, 0.3663090000E-01 /
  data (epsData(123,i,48), i=1,3) / 0.4059510000E+03, -0.3009190000E-01, 0.3784240000E-01 /
  data (epsData(124,i,48), i=1,3) / 0.4081420000E+03, -0.2879160000E-01, 0.3658950000E-01 /
  data (epsData(125,i,48), i=1,3) / 0.4107730000E+03, -0.2338380000E-01, 0.3546140000E-01 /
  data (epsData(126,i,48), i=1,3) / 0.4116780000E+03, -0.2006470000E-01, 0.3636490000E-01 /
  data (epsData(127,i,48), i=1,3) / 0.4121670000E+03, -0.1965350000E-01, 0.3841800000E-01 /
  data (epsData(128,i,48), i=1,3) / 0.4130050000E+03, -0.2001240000E-01, 0.3851500000E-01 /
  data (epsData(129,i,48), i=1,3) / 0.4157940000E+03, -0.1577330000E-01, 0.3754140000E-01 /
  data (epsData(130,i,48), i=1,3) / 0.4195530000E+03, -0.7299010000E-02, 0.3772830000E-01 /
  data (epsData(131,i,48), i=1,3) / 0.4240500000E+03, 0.4047530000E-02, 0.4028440000E-01 /
  data (epsData(132,i,48), i=1,3) / 0.4288150000E+03, 0.1706740000E-01, 0.4563520000E-01 /
  data (epsData(133,i,48), i=1,3) / 0.4340180000E+03, 0.3168200000E-01, 0.5588460000E-01 /
  data (epsData(134,i,48), i=1,3) / 0.4387580000E+03, 0.4336020000E-01, 0.7007930000E-01 /
  data (epsData(135,i,48), i=1,3) / 0.4439580000E+03, 0.5137420000E-01, 0.9014670000E-01 /
  data (epsData(136,i,48), i=1,3) / 0.4505300000E+03, 0.5145960000E-01, 0.1173250000E+00 /
  data (epsData(137,i,48), i=1,3) / 0.4567200000E+03, 0.4278260000E-01, 0.1398870000E+00 /
  data (epsData(138,i,48), i=1,3) / 0.4633710000E+03, 0.2747780000E-01, 0.1578140000E+00 /
  data (epsData(139,i,48), i=1,3) / 0.4719010000E+03, 0.4444640000E-02, 0.1706260000E+00 /
  data (epsData(140,i,48), i=1,3) / 0.4817010000E+03, -0.2036650000E-01, 0.1725330000E+00 /
  data (epsData(141,i,48), i=1,3) / 0.4935690000E+03, -0.4157030000E-01, 0.1662020000E+00 /
  data (epsData(142,i,48), i=1,3) / 0.5162960000E+03, -0.6389830000E-01, 0.1487310000E+00 /
  data (epsData(143,i,48), i=1,3) / 0.5447700000E+03, -0.7549710000E-01, 0.1270990000E+00 /
  data (epsData(144,i,48), i=1,3) / 0.5824040000E+03, -0.7770880000E-01, 0.1046720000E+00 /
  data (epsData(145,i,48), i=1,3) / 0.6130510000E+03, -0.7229680000E-01, 0.9068070000E-01 /
  data (epsData(146,i,48), i=1,3) / 0.6222940000E+03, -0.6832240000E-01, 0.9098680000E-01 /
  data (epsData(147,i,48), i=1,3) / 0.6350130000E+03, -0.7106900000E-01, 0.9070690000E-01 /
  data (epsData(148,i,48), i=1,3) / 0.6570200000E+03, -0.7076340000E-01, 0.8527500000E-01 /
  data (epsData(149,i,48), i=1,3) / 0.6867810000E+03, -0.7309150000E-01, 0.7740500000E-01 /
  data (epsData(150,i,48), i=1,3) / 0.7492320000E+03, -0.6940420000E-01, 0.6054680000E-01 /
  data (epsData(151,i,48), i=1,3) / 0.7750060000E+03, -0.6657660000E-01, 0.5683190000E-01 /
  data (epsData(152,i,48), i=1,3) / 0.8590890000E+03, -0.6116060000E-01, 0.4216440000E-01 /
  data (epsData(153,i,48), i=1,3) / 0.9810750000E+03, -0.5101610000E-01, 0.2812640000E-01 /
  data (epsData(154,i,48), i=1,3) / 0.1098020000E+04, -0.4267700000E-01, 0.1973480000E-01 /
  data (epsData(155,i,48), i=1,3) / 0.1217170000E+04, -0.3575330000E-01, 0.1415470000E-01 /
  data (epsData(156,i,48), i=1,3) / 0.1345970000E+04, -0.2979330000E-01, 0.1016400000E-01 /
  data (epsData(157,i,48), i=1,3) / 0.1490140000E+04, -0.2458830000E-01, 0.7229420000E-02 /
  data (epsData(158,i,48), i=1,3) / 0.1658100000E+04, -0.1996380000E-01, 0.5041120000E-02 /
  data (epsData(159,i,48), i=1,3) / 0.1843700000E+04, -0.1616910000E-01, 0.3516650000E-02 /
  data (epsData(160,i,48), i=1,3) / 0.2045620000E+04, -0.1308580000E-01, 0.2450520000E-02 /
  data (epsData(161,i,48), i=1,3) / 0.2270650000E+04, -0.1054190000E-01, 0.1714170000E-02 /
  data (epsData(162,i,48), i=1,3) / 0.2521700000E+04, -0.8443150000E-02, 0.1188390000E-02 /
  data (epsData(163,i,48), i=1,3) / 0.2800470000E+04, -0.6707580000E-02, 0.8216070000E-03 /
  data (epsData(164,i,48), i=1,3) / 0.3126610000E+04, -0.5169000000E-02, 0.5561320000E-03 /
  data (epsData(165,i,48), i=1,3) / 0.3452170000E+04, -0.3777760000E-02, 0.3907110000E-03 /
  data (epsData(166,i,48), i=1,3) / 0.3511700000E+04, -0.3346860000E-02, 0.3675660000E-03 /
  data (epsData(167,i,48), i=1,3) / 0.3531040000E+04, -0.3041780000E-02, 0.3855780000E-03 /
  data (epsData(168,i,48), i=1,3) / 0.3536850000E+04, -0.2832680000E-02, 0.4269700000E-03 /
  data (epsData(169,i,48), i=1,3) / 0.3540400000E+04, -0.2677750000E-02, 0.5991530000E-03 /
  data (epsData(170,i,48), i=1,3) / 0.3545050000E+04, -0.2678580000E-02, 0.9057520000E-03 /
  data (epsData(171,i,48), i=1,3) / 0.3550930000E+04, -0.2882530000E-02, 0.1085810000E-02 /
  data (epsData(172,i,48), i=1,3) / 0.3565630000E+04, -0.3117280000E-02, 0.1143910000E-02 /
  data (epsData(173,i,48), i=1,3) / 0.3592420000E+04, -0.3290250000E-02, 0.1140830000E-02 /
  data (epsData(174,i,48), i=1,3) / 0.3667070000E+04, -0.3321370000E-02, 0.1051890000E-02 /
  data (epsData(175,i,48), i=1,3) / 0.3717280000E+04, -0.3143100000E-02, 0.1008430000E-02 /
  data (epsData(176,i,48), i=1,3) / 0.3727540000E+04, -0.2996170000E-02, 0.1048440000E-02 /
  data (epsData(177,i,48), i=1,3) / 0.3735600000E+04, -0.2987890000E-02, 0.1249580000E-02 /
  data (epsData(178,i,48), i=1,3) / 0.3748940000E+04, -0.3130210000E-02, 0.1311910000E-02 /
  data (epsData(179,i,48), i=1,3) / 0.3794660000E+04, -0.3269770000E-02, 0.1280710000E-02 /
  data (epsData(180,i,48), i=1,3) / 0.3950170000E+04, -0.3219490000E-02, 0.1106550000E-02 /
  data (epsData(181,i,48), i=1,3) / 0.4009300000E+04, -0.3074570000E-02, 0.1068740000E-02 /
  data (epsData(182,i,48), i=1,3) / 0.4018520000E+04, -0.3019180000E-02, 0.1157100000E-02 /
  data (epsData(183,i,48), i=1,3) / 0.4028780000E+04, -0.3095830000E-02, 0.1169320000E-02 /
  data (epsData(184,i,48), i=1,3) / 0.4194820000E+04, -0.3075880000E-02, 0.1029600000E-02 /
  data (epsData(185,i,48), i=1,3) / 0.4780290000E+04, -0.2565860000E-02, 0.6496530000E-03 /
  data (epsData(186,i,48), i=1,3) / 0.5331410000E+04, -0.2124530000E-02, 0.4433730000E-03 /
  data (epsData(187,i,48), i=1,3) / 0.5972400000E+04, -0.1723390000E-02, 0.2956410000E-03 /
  data (epsData(188,i,48), i=1,3) / 0.6658220000E+04, -0.1399400000E-02, 0.1998570000E-03 /
  data (epsData(189,i,48), i=1,3) / 0.7425420000E+04, -0.1130330000E-02, 0.1338500000E-03 /
  data (epsData(190,i,48), i=1,3) / 0.8285360000E+04, -0.9091290000E-03, 0.8964610000E-04 /
  data (epsData(191,i,48), i=1,3) / 0.9223300000E+04, -0.7333080000E-03, 0.6010460000E-04 /
  data (epsData(192,i,48), i=1,3) / 0.1027770000E+05, -0.5894280000E-03, 0.4017230000E-04 /
  data (epsData(193,i,48), i=1,3) / 0.1145460000E+05, -0.4732170000E-03, 0.2673100000E-04 /
  data (epsData(194,i,48), i=1,3) / 0.1277430000E+05, -0.3792410000E-03, 0.1770520000E-04 /
  data (epsData(195,i,48), i=1,3) / 0.1425380000E+05, -0.3034470000E-03, 0.1167070000E-04 /
  data (epsData(196,i,48), i=1,3) / 0.1591300000E+05, -0.2424580000E-03, 0.7675800000E-05 /
  data (epsData(197,i,48), i=1,3) / 0.1777030000E+05, -0.1935120000E-03, 0.5030260000E-05 /
  data (epsData(198,i,48), i=1,3) / 0.1986030000E+05, -0.1540420000E-03, 0.3279560000E-05 /
  data (epsData(199,i,48), i=1,3) / 0.2211300000E+05, -0.1233090000E-03, 0.2166800000E-05 /
  data (epsData(200,i,48), i=1,3) / 0.2211700000E+05, -0.1178950000E-03, 0.2165270000E-05 /
  data (epsData(201,i,48), i=1,3) / 0.2211700000E+05, -0.1232620000E-03, 0.2165260000E-05 /
  data (epsData(202,i,48), i=1,3) / 0.2489740000E+05, -0.9560580000E-04, 0.1368260000E-05 /
  data (epsData(203,i,48), i=1,3) / 0.2653500000E+05, -0.8020350000E-04, 0.1067900000E-05 /
  data (epsData(204,i,48), i=1,3) / 0.2668080000E+05, -0.7604370000E-04, 0.1045390000E-05 /
  data (epsData(205,i,48), i=1,3) / 0.2670330000E+05, -0.7331710000E-04, 0.2146500000E-05 /
  data (epsData(206,i,48), i=1,3) / 0.2671080000E+05, -0.7224570000E-04, 0.4566260000E-05 /
  data (epsData(207,i,48), i=1,3) / 0.2671100000E+05, -0.6514450000E-04, 0.4667990000E-05 /
  data (epsData(208,i,48), i=1,3) / 0.2671100000E+05, -0.7226860000E-04, 0.4691540000E-05 /
  data (epsData(209,i,48), i=1,3) / 0.2671970000E+05, -0.7436890000E-04, 0.5993650000E-05 /
  data (epsData(210,i,48), i=1,3) / 0.2678970000E+05, -0.7723980000E-04, 0.6580430000E-05 /
  data (epsData(211,i,48), i=1,3) / 0.2723320000E+05, -0.7825250000E-04, 0.6333710000E-05 /
  data (epsData(212,i,48), i=1,3) / 0.3270800000E+05, -0.5743070000E-04, 0.3273020000E-05 /
  data (epsData(213,i,48), i=1,3) / 0.3678330000E+05, -0.4580420000E-04, 0.2114330000E-05 /
  data (epsData(214,i,48), i=1,3) / 0.4123970000E+05, -0.3659200000E-04, 0.1374220000E-05 /
  data (epsData(215,i,48), i=1,3) / 0.4619140000E+05, -0.2922520000E-04, 0.8919980000E-06 /
  data (epsData(216,i,48), i=1,3) / 0.5172180000E+05, -0.2332700000E-04, 0.5769150000E-06 /
  data (epsData(217,i,48), i=1,3) / 0.5790980000E+05, -0.1860890000E-04, 0.3715470000E-06 /
  data (epsData(218,i,48), i=1,3) / 0.6480990000E+05, -0.1485310000E-04, 0.2392220000E-06 /
  data (epsData(219,i,48), i=1,3) / 0.7258830000E+05, -0.1183430000E-04, 0.1525600000E-06 /
  data (epsData(220,i,48), i=1,3) / 0.8130390000E+05, -0.9429220000E-05, 0.9661110000E-07 /
  data (epsData(221,i,48), i=1,3) / 0.9106010000E+05, -0.7518010000E-05, 0.6100810000E-07 /
  data (epsData(222,i,48), i=1,3) / 0.9991130000E+05, -0.6296210000E-05, 0.4181780000E-07 /
  data (epsData(223,i,48), i=1,3) / 0.1000000000E+06, -0.6318620000E-05, 0.4166250000E-07 /
  data (epsData(1,i,49), i=1,3) / 0.2506580000E-02, -0.2186480000E+03, 0.1503930000E+04 /
  data (epsData(2,i,49), i=1,3) / 0.4693440000E-02, -0.2810970000E+03, 0.5981280000E+03 /
  data (epsData(3,i,49), i=1,3) / 0.7560590000E-02, -0.1162780000E+03, 0.3159720000E+03 /
  data (epsData(4,i,49), i=1,3) / 0.1110800000E-01, -0.4126500000E+02, 0.1953760000E+03 /
  data (epsData(5,i,49), i=1,3) / 0.1533570000E-01, -0.5351670000E+01, 0.1331360000E+03 /
  data (epsData(6,i,49), i=1,3) / 0.2024370000E-01, 0.1342210000E+02, 0.9681470000E+02 /
  data (epsData(7,i,49), i=1,3) / 0.2583200000E-01, 0.2400450000E+02, 0.7372870000E+02 /
  data (epsData(8,i,49), i=1,3) / 0.3210060000E-01, 0.3034620000E+02, 0.5811170000E+02 /
  data (epsData(9,i,49), i=1,3) / 0.3904940000E-01, 0.3434130000E+02, 0.4703570000E+02 /
  data (epsData(10,i,49), i=1,3) / 0.4667860000E-01, 0.3696420000E+02, 0.3888400000E+02 /
  data (epsData(11,i,49), i=1,3) / 0.5498800000E-01, 0.3874700000E+02, 0.3270330000E+02 /
  data (epsData(12,i,49), i=1,3) / 0.6397770000E-01, 0.3999530000E+02, 0.2790120000E+02 /
  data (epsData(13,i,49), i=1,3) / 0.8399800000E-01, 0.4155110000E+02, 0.2102160000E+02 /
  data (epsData(14,i,49), i=1,3) / 0.1067390000E+00, 0.4242340000E+02, 0.1641980000E+02 /
  data (epsData(15,i,49), i=1,3) / 0.1459540000E+00, 0.4314080000E+02, 0.1191890000E+02 /
  data (epsData(16,i,49), i=1,3) / 0.2077630000E+00, 0.4364200000E+02, 0.8383720000E+01 /
  data (epsData(17,i,49), i=1,3) / 0.3421210000E+00, 0.4396990000E+02, 0.5201530000E+01 /
  data (epsData(18,i,49), i=1,3) / 0.7108380000E+00, 0.4427530000E+02, 0.2720780000E+01 /
  data (epsData(19,i,49), i=1,3) / 0.1606050000E+01, 0.4585270000E+02, 0.1564570000E+01 /
  data (epsData(20,i,49), i=1,3) / 0.2617000000E+01, 0.4997330000E+02, 0.1431200000E+01 /
  data (epsData(21,i,49), i=1,3) / 0.3382030000E+01, 0.5608020000E+02, 0.1714140000E+01 /
  data (epsData(22,i,49), i=1,3) / 0.3873540000E+01, 0.6288290000E+02, 0.2226500000E+01 /
  data (epsData(23,i,49), i=1,3) / 0.4245030000E+01, 0.7128550000E+02, 0.3139820000E+01 /
  data (epsData(24,i,49), i=1,3) / 0.4476080000E+01, 0.7962110000E+02, 0.4480740000E+01 /
  data (epsData(25,i,49), i=1,3) / 0.4633520000E+01, 0.8817960000E+02, 0.5945460000E+01 /
  data (epsData(26,i,49), i=1,3) / 0.4713260000E+01, 0.9408470000E+02, 0.8556380000E+01 /
  data (epsData(27,i,49), i=1,3) / 0.4874780000E+01, 0.1056070000E+03, 0.1806760000E+02 /
  data (epsData(28,i,49), i=1,3) / 0.4956560000E+01, 0.1143470000E+03, 0.2262890000E+02 /
  data (epsData(29,i,49), i=1,3) / 0.5039020000E+01, 0.1211100000E+03, 0.4001480000E+02 /
  data (epsData(30,i,49), i=1,3) / 0.5122160000E+01, 0.1202020000E+03, 0.5727560000E+02 /
  data (epsData(31,i,49), i=1,3) / 0.5205980000E+01, 0.1135510000E+03, 0.7263720000E+02 /
  data (epsData(32,i,49), i=1,3) / 0.5290480000E+01, 0.1019470000E+03, 0.8459200000E+02 /
  data (epsData(33,i,49), i=1,3) / 0.5375660000E+01, 0.9362110000E+02, 0.8286320000E+02 /
  data (epsData(34,i,49), i=1,3) / 0.5461520000E+01, 0.8944630000E+02, 0.9078340000E+02 /
  data (epsData(35,i,49), i=1,3) / 0.5548060000E+01, 0.8120550000E+02, 0.1002920000E+03 /
  data (epsData(36,i,49), i=1,3) / 0.5635290000E+01, 0.6922640000E+02, 0.1065210000E+03 /
  data (epsData(37,i,49), i=1,3) / 0.5723190000E+01, 0.5508200000E+02, 0.1092720000E+03 /
  data (epsData(38,i,49), i=1,3) / 0.5811770000E+01, 0.4293420000E+02, 0.1036850000E+03 /
  data (epsData(39,i,49), i=1,3) / 0.5901040000E+01, 0.3509670000E+02, 0.9862390000E+02 /
  data (epsData(40,i,49), i=1,3) / 0.6081600000E+01, 0.2399970000E+02, 0.8977130000E+02 /
  data (epsData(41,i,49), i=1,3) / 0.6264890000E+01, 0.1800070000E+02, 0.8138660000E+02 /
  data (epsData(42,i,49), i=1,3) / 0.6544930000E+01, 0.1285540000E+02, 0.7291640000E+02 /
  data (epsData(43,i,49), i=1,3) / 0.7025260000E+01, 0.6762720000E+01, 0.6445180000E+02 /
  data (epsData(44,i,49), i=1,3) / 0.8141850000E+01, -0.6102360000E+01, 0.5021910000E+02 /
  data (epsData(45,i,49), i=1,3) / 0.8785590000E+01, -0.1102320000E+02, 0.4086680000E+02 /
  data (epsData(46,i,49), i=1,3) / 0.9116650000E+01, -0.1217480000E+02, 0.3539490000E+02 /
  data (epsData(47,i,49), i=1,3) / 0.9340750000E+01, -0.1169220000E+02, 0.3179890000E+02 /
  data (epsData(48,i,49), i=1,3) / 0.9682010000E+01, -0.1023000000E+02, 0.2820320000E+02 /
  data (epsData(49,i,49), i=1,3) / 0.1050210000E+02, -0.8465880000E+01, 0.2286560000E+02 /
  data (epsData(50,i,49), i=1,3) / 0.1185810000E+02, -0.6364000000E+01, 0.1611590000E+02 /
  data (epsData(51,i,49), i=1,3) / 0.1289610000E+02, -0.4398620000E+01, 0.1206950000E+02 /
  data (epsData(52,i,49), i=1,3) / 0.1370310000E+02, -0.2549260000E+01, 0.9583010000E+01 /
  data (epsData(53,i,49), i=1,3) / 0.1439430000E+02, -0.7717600000E+00, 0.7884190000E+01 /
  data (epsData(54,i,49), i=1,3) / 0.1495950000E+02, 0.8417240000E+00, 0.6758580000E+01 /
  data (epsData(55,i,49), i=1,3) / 0.1553560000E+02, 0.2750180000E+01, 0.5830200000E+01 /
  data (epsData(56,i,49), i=1,3) / 0.1597480000E+02, 0.4585260000E+01, 0.5287310000E+01 /
  data (epsData(57,i,49), i=1,3) / 0.1627110000E+02, 0.6272460000E+01, 0.5056470000E+01 /
  data (epsData(58,i,49), i=1,3) / 0.1642020000E+02, 0.7442330000E+01, 0.5016270000E+01 /
  data (epsData(59,i,49), i=1,3) / 0.1657000000E+02, 0.8900750000E+01, 0.5372000000E+01 /
  data (epsData(60,i,49), i=1,3) / 0.1672050000E+02, 0.1069400000E+02, 0.6157690000E+01 /
  data (epsData(61,i,49), i=1,3) / 0.1687170000E+02, 0.1234570000E+02, 0.8777200000E+01 /
  data (epsData(62,i,49), i=1,3) / 0.1692180000E+02, 0.1235290000E+02, 0.1010130000E+02 /
  data (epsData(63,i,49), i=1,3) / 0.1705000000E+02, 0.1141000000E+02, 0.1278240000E+02 /
  data (epsData(64,i,49), i=1,3) / 0.1709590000E+02, 0.1044580000E+02, 0.1373300000E+02 /
  data (epsData(65,i,49), i=1,3) / 0.1716670000E+02, 0.8982180000E+01, 0.1364180000E+02 /
  data (epsData(66,i,49), i=1,3) / 0.1731670000E+02, 0.7784140000E+01, 0.1330470000E+02 /
  data (epsData(67,i,49), i=1,3) / 0.1747180000E+02, 0.6994370000E+01, 0.1313200000E+02 /
  data (epsData(68,i,49), i=1,3) / 0.1758880000E+02, 0.7094150000E+01, 0.1280950000E+02 /
  data (epsData(69,i,49), i=1,3) / 0.1765140000E+02, 0.7359050000E+01, 0.1333820000E+02 /
  data (epsData(70,i,49), i=1,3) / 0.1776670000E+02, 0.6748530000E+01, 0.1430040000E+02 /
  data (epsData(71,i,49), i=1,3) / 0.1789590000E+02, 0.5387530000E+01, 0.1513220000E+02 /
  data (epsData(72,i,49), i=1,3) / 0.1800610000E+02, 0.4114640000E+01, 0.1455800000E+02 /
  data (epsData(73,i,49), i=1,3) / 0.1827180000E+02, 0.2653040000E+01, 0.1325910000E+02 /
  data (epsData(74,i,49), i=1,3) / 0.1855040000E+02, 0.2145080000E+01, 0.1187050000E+02 /
  data (epsData(75,i,49), i=1,3) / 0.1892450000E+02, 0.2148340000E+01, 0.1052590000E+02 /
  data (epsData(76,i,49), i=1,3) / 0.1946330000E+02, 0.2567100000E+01, 0.9290730000E+01 /
  data (epsData(77,i,49), i=1,3) / 0.2033410000E+02, 0.3437540000E+01, 0.8253890000E+01 /
  data (epsData(78,i,49), i=1,3) / 0.2204330000E+02, 0.4777810000E+01, 0.7206210000E+01 /
  data (epsData(79,i,49), i=1,3) / 0.2353720000E+02, 0.6138190000E+01, 0.7027110000E+01 /
  data (epsData(80,i,49), i=1,3) / 0.2525150000E+02, 0.7391830000E+01, 0.7559840000E+01 /
  data (epsData(81,i,49), i=1,3) / 0.2730020000E+02, 0.8253050000E+01, 0.8805250000E+01 /
  data (epsData(82,i,49), i=1,3) / 0.3010870000E+02, 0.8202470000E+01, 0.1086050000E+02 /
  data (epsData(83,i,49), i=1,3) / 0.3298730000E+02, 0.6989570000E+01, 0.1260510000E+02 /
  data (epsData(84,i,49), i=1,3) / 0.3680540000E+02, 0.4643090000E+01, 0.1370760000E+02 /
  data (epsData(85,i,49), i=1,3) / 0.4361070000E+02, 0.7794210000E+00, 0.1394900000E+02 /
  data (epsData(86,i,49), i=1,3) / 0.4872230000E+02, -0.2169900000E+01, 0.1322020000E+02 /
  data (epsData(87,i,49), i=1,3) / 0.5274860000E+02, -0.4240860000E+01, 0.1181240000E+02 /
  data (epsData(88,i,49), i=1,3) / 0.5678940000E+02, -0.5734560000E+01, 0.9915210000E+01 /
  data (epsData(89,i,49), i=1,3) / 0.6105070000E+02, -0.6413710000E+01, 0.7744250000E+01 /
  data (epsData(90,i,49), i=1,3) / 0.6578790000E+02, -0.6438710000E+01, 0.5685370000E+01 /
  data (epsData(91,i,49), i=1,3) / 0.7157980000E+02, -0.5929900000E+01, 0.3786970000E+01 /
  data (epsData(92,i,49), i=1,3) / 0.7864600000E+02, -0.5025090000E+01, 0.2277950000E+01 /
  data (epsData(93,i,49), i=1,3) / 0.8577910000E+02, -0.4155020000E+01, 0.1358200000E+01 /
  data (epsData(94,i,49), i=1,3) / 0.9276700000E+02, -0.3411750000E+01, 0.8110550000E+00 /
  data (epsData(95,i,49), i=1,3) / 0.9978750000E+02, -0.2796920000E+01, 0.4929780000E+00 /
  data (epsData(96,i,49), i=1,3) / 0.1072510000E+03, -0.2283530000E+01, 0.3070790000E+00 /
  data (epsData(97,i,49), i=1,3) / 0.1154060000E+03, -0.1856810000E+01, 0.1956820000E+00 /
  data (epsData(98,i,49), i=1,3) / 0.1229770000E+03, -0.1542150000E+01, 0.1480320000E+00 /
  data (epsData(99,i,49), i=1,3) / 0.1334360000E+03, -0.1233150000E+01, 0.1198680000E+00 /
  data (epsData(100,i,49), i=1,3) / 0.1446720000E+03, -0.9948190000E+00, 0.1129830000E+00 /
  data (epsData(101,i,49), i=1,3) / 0.1586070000E+03, -0.7899090000E+00, 0.1121070000E+00 /
  data (epsData(102,i,49), i=1,3) / 0.1746360000E+03, -0.6284930000E+00, 0.1114660000E+00 /
  data (epsData(103,i,49), i=1,3) / 0.1942500000E+03, -0.4944970000E+00, 0.1071790000E+00 /
  data (epsData(104,i,49), i=1,3) / 0.2155840000E+03, -0.3952150000E+00, 0.1000090000E+00 /
  data (epsData(105,i,49), i=1,3) / 0.2413050000E+03, -0.3112030000E+00, 0.8901400000E-01 /
  data (epsData(106,i,49), i=1,3) / 0.2733220000E+03, -0.2376960000E+00, 0.7507910000E-01 /
  data (epsData(107,i,49), i=1,3) / 0.3106790000E+03, -0.1762490000E+00, 0.6068730000E-01 /
  data (epsData(108,i,49), i=1,3) / 0.3532940000E+03, -0.1234310000E+00, 0.4744110000E-01 /
  data (epsData(109,i,49), i=1,3) / 0.4177300000E+03, -0.5525390000E-01, 0.3319730000E-01 /
  data (epsData(110,i,49), i=1,3) / 0.4350790000E+03, -0.3327580000E-01, 0.3047970000E-01 /
  data (epsData(111,i,49), i=1,3) / 0.4416300000E+03, -0.2226170000E-01, 0.2962880000E-01 /
  data (epsData(112,i,49), i=1,3) / 0.4429480000E+03, -0.1891990000E-01, 0.2971520000E-01 /
  data (epsData(113,i,49), i=1,3) / 0.4436090000E+03, -0.1675540000E-01, 0.3064400000E-01 /
  data (epsData(114,i,49), i=1,3) / 0.4440800000E+03, -0.1608610000E-01, 0.3297910000E-01 /
  data (epsData(115,i,49), i=1,3) / 0.4446200000E+03, -0.1725680000E-01, 0.3340210000E-01 /
  data (epsData(116,i,49), i=1,3) / 0.4459170000E+03, -0.1624330000E-01, 0.3223260000E-01 /
  data (epsData(117,i,49), i=1,3) / 0.4494730000E+03, -0.9566710000E-02, 0.3146590000E-01 /
  data (epsData(118,i,49), i=1,3) / 0.4508560000E+03, -0.5854540000E-02, 0.3177500000E-01 /
  data (epsData(119,i,49), i=1,3) / 0.4514490000E+03, -0.4150950000E-02, 0.3316000000E-01 /
  data (epsData(120,i,49), i=1,3) / 0.4521200000E+03, -0.4218730000E-02, 0.3395040000E-01 /
  data (epsData(121,i,49), i=1,3) / 0.4536320000E+03, -0.1768410000E-02, 0.3363320000E-01 /
  data (epsData(122,i,49), i=1,3) / 0.4572370000E+03, 0.6270140000E-02, 0.3475180000E-01 /
  data (epsData(123,i,49), i=1,3) / 0.4618170000E+03, 0.1792180000E-01, 0.3850820000E-01 /
  data (epsData(124,i,49), i=1,3) / 0.4666870000E+03, 0.3138600000E-01, 0.4597540000E-01 /
  data (epsData(125,i,49), i=1,3) / 0.4716510000E+03, 0.4466840000E-01, 0.5871490000E-01 /
  data (epsData(126,i,49), i=1,3) / 0.4761960000E+03, 0.5385390000E-01, 0.7532200000E-01 /
  data (epsData(127,i,49), i=1,3) / 0.4816060000E+03, 0.5754720000E-01, 0.9882670000E-01 /
  data (epsData(128,i,49), i=1,3) / 0.4877710000E+03, 0.5137700000E-01, 0.1241910000E+00 /
  data (epsData(129,i,49), i=1,3) / 0.4942660000E+03, 0.3673510000E-01, 0.1445210000E+00 /
  data (epsData(130,i,49), i=1,3) / 0.5018180000E+03, 0.1579290000E-01, 0.1583740000E+00 /
  data (epsData(131,i,49), i=1,3) / 0.5094180000E+03, -0.4949720000E-02, 0.1633270000E+00 /
  data (epsData(132,i,49), i=1,3) / 0.5181530000E+03, -0.2492500000E-01, 0.1614220000E+00 /
  data (epsData(133,i,49), i=1,3) / 0.5317410000E+03, -0.4503200000E-01, 0.1512390000E+00 /
  data (epsData(134,i,49), i=1,3) / 0.5561740000E+03, -0.6313310000E-01, 0.1327320000E+00 /
  data (epsData(135,i,49), i=1,3) / 0.5846960000E+03, -0.7116000000E-01, 0.1133520000E+00 /
  data (epsData(136,i,49), i=1,3) / 0.6272440000E+03, -0.7223090000E-01, 0.9173090000E-01 /
  data (epsData(137,i,49), i=1,3) / 0.6594100000E+03, -0.6670390000E-01, 0.7927760000E-01 /
  data (epsData(138,i,49), i=1,3) / 0.6689790000E+03, -0.6274970000E-01, 0.7962820000E-01 /
  data (epsData(139,i,49), i=1,3) / 0.6826730000E+03, -0.6518570000E-01, 0.7920190000E-01 /
  data (epsData(140,i,49), i=1,3) / 0.7119080000E+03, -0.6515620000E-01, 0.7390040000E-01 /
  data (epsData(141,i,49), i=1,3) / 0.7742390000E+03, -0.6490800000E-01, 0.5911390000E-01 /
  data (epsData(142,i,49), i=1,3) / 0.8244320000E+03, -0.6024230000E-01, 0.4993510000E-01 /
  data (epsData(143,i,49), i=1,3) / 0.9102080000E+03, -0.5564350000E-01, 0.3816730000E-01 /
  data (epsData(144,i,49), i=1,3) / 0.1035740000E+04, -0.4672610000E-01, 0.2572120000E-01 /
  data (epsData(145,i,49), i=1,3) / 0.1153210000E+04, -0.3942090000E-01, 0.1830540000E-01 /
  data (epsData(146,i,49), i=1,3) / 0.1281760000E+04, -0.3288410000E-01, 0.1298900000E-01 /
  data (epsData(147,i,49), i=1,3) / 0.1429800000E+04, -0.2692780000E-01, 0.9041170000E-02 /
  data (epsData(148,i,49), i=1,3) / 0.1595610000E+04, -0.2187730000E-01, 0.6283810000E-02 /
  data (epsData(149,i,49), i=1,3) / 0.1765660000E+04, -0.1796580000E-01, 0.4458140000E-02 /
  data (epsData(150,i,49), i=1,3) / 0.1956410000E+04, -0.1464610000E-01, 0.3131090000E-02 /
  data (epsData(151,i,49), i=1,3) / 0.2166140000E+04, -0.1189220000E-01, 0.2205330000E-02 /
  data (epsData(152,i,49), i=1,3) / 0.2405420000E+04, -0.9576890000E-02, 0.1536640000E-02 /
  data (epsData(153,i,49), i=1,3) / 0.2669120000E+04, -0.7682650000E-02, 0.1067310000E-02 /
  data (epsData(154,i,49), i=1,3) / 0.2965090000E+04, -0.6100160000E-02, 0.7366240000E-03 /
  data (epsData(155,i,49), i=1,3) / 0.3312450000E+04, -0.4693710000E-02, 0.4972500000E-03 /
  data (epsData(156,i,49), i=1,3) / 0.3642710000E+04, -0.3474530000E-02, 0.3541740000E-03 /
  data (epsData(157,i,49), i=1,3) / 0.3703720000E+04, -0.3090900000E-02, 0.3337180000E-03 /
  data (epsData(158,i,49), i=1,3) / 0.3723530000E+04, -0.2813020000E-02, 0.3513590000E-03 /
  data (epsData(159,i,49), i=1,3) / 0.3729490000E+04, -0.2615190000E-02, 0.4006710000E-03 /
  data (epsData(160,i,49), i=1,3) / 0.3732880000E+04, -0.2487000000E-02, 0.5677460000E-03 /
  data (epsData(161,i,49), i=1,3) / 0.3736760000E+04, -0.2513760000E-02, 0.8066200000E-03 /
  data (epsData(162,i,49), i=1,3) / 0.3742150000E+04, -0.2679620000E-02, 0.9357520000E-03 /
  data (epsData(163,i,49), i=1,3) / 0.3759600000E+04, -0.2890120000E-02, 0.1013450000E-02 /
  data (epsData(164,i,49), i=1,3) / 0.3783610000E+04, -0.3029980000E-02, 0.1016880000E-02 /
  data (epsData(165,i,49), i=1,3) / 0.3856150000E+04, -0.3083590000E-02, 0.9508540000E-03 /
  data (epsData(166,i,49), i=1,3) / 0.3922870000E+04, -0.2927270000E-02, 0.8962040000E-03 /
  data (epsData(167,i,49), i=1,3) / 0.3937480000E+04, -0.2784150000E-02, 0.9153660000E-03 /
  data (epsData(168,i,49), i=1,3) / 0.3944020000E+04, -0.2735120000E-02, 0.1063580000E-02 /
  data (epsData(169,i,49), i=1,3) / 0.3954760000E+04, -0.2848990000E-02, 0.1147050000E-02 /
  data (epsData(170,i,49), i=1,3) / 0.3994070000E+04, -0.2982950000E-02, 0.1148440000E-02 /
  data (epsData(171,i,49), i=1,3) / 0.4147610000E+04, -0.2970940000E-02, 0.1003910000E-02 /
  data (epsData(172,i,49), i=1,3) / 0.4227780000E+04, -0.2828680000E-02, 0.9508820000E-03 /
  data (epsData(173,i,49), i=1,3) / 0.4238750000E+04, -0.2774470000E-02, 0.1023240000E-02 /
  data (epsData(174,i,49), i=1,3) / 0.4262980000E+04, -0.2852390000E-02, 0.1039960000E-02 /
  data (epsData(175,i,49), i=1,3) / 0.4527780000E+04, -0.2745220000E-02, 0.8481600000E-03 /
  data (epsData(176,i,49), i=1,3) / 0.5245480000E+04, -0.2189030000E-02, 0.5040470000E-03 /
  data (epsData(177,i,49), i=1,3) / 0.5915930000E+04, -0.1767960000E-02, 0.3304400000E-03 /
  data (epsData(178,i,49), i=1,3) / 0.6606950000E+04, -0.1435720000E-02, 0.2219790000E-03 /
  data (epsData(179,i,49), i=1,3) / 0.7356820000E+04, -0.1165810000E-02, 0.1501790000E-03 /
  data (epsData(180,i,49), i=1,3) / 0.8204880000E+04, -0.9396840000E-03, 0.1006980000E-03 /
  data (epsData(181,i,49), i=1,3) / 0.9131490000E+04, -0.7590790000E-03, 0.6776960000E-04 /
  data (epsData(182,i,49), i=1,3) / 0.1017510000E+05, -0.6105260000E-03, 0.4530760000E-04 /
  data (epsData(183,i,49), i=1,3) / 0.1134130000E+05, -0.4902870000E-03, 0.3016200000E-04 /
  data (epsData(184,i,49), i=1,3) / 0.1264580000E+05, -0.3931800000E-03, 0.2001570000E-04 /
  data (epsData(185,i,49), i=1,3) / 0.1410770000E+05, -0.3148190000E-03, 0.1322150000E-04 /
  data (epsData(186,i,49), i=1,3) / 0.1574210000E+05, -0.2518550000E-03, 0.8715770000E-05 /
  data (epsData(187,i,49), i=1,3) / 0.1757820000E+05, -0.2011130000E-03, 0.5717600000E-05 /
  data (epsData(188,i,49), i=1,3) / 0.1963760000E+05, -0.1603350000E-03, 0.3736860000E-05 /
  data (epsData(189,i,49), i=1,3) / 0.2195780000E+05, -0.1274110000E-03, 0.2430420000E-05 /
  data (epsData(190,i,49), i=1,3) / 0.2321700000E+05, -0.1134680000E-03, 0.1958990000E-05 /
  data (epsData(191,i,49), i=1,3) / 0.2322000000E+05, -0.1087660000E-03, 0.1958010000E-05 /
  data (epsData(192,i,49), i=1,3) / 0.2322000000E+05, -0.1134370000E-03, 0.1957990000E-05 /
  data (epsData(193,i,49), i=1,3) / 0.2615820000E+05, -0.8781280000E-04, 0.1233510000E-05 /
  data (epsData(194,i,49), i=1,3) / 0.2776640000E+05, -0.7433510000E-04, 0.9777830000E-06 /
  data (epsData(195,i,49), i=1,3) / 0.2791180000E+05, -0.7049060000E-04, 0.9581300000E-06 /
  data (epsData(196,i,49), i=1,3) / 0.2793460000E+05, -0.6793260000E-04, 0.2277480000E-05 /
  data (epsData(197,i,49), i=1,3) / 0.2793960000E+05, -0.6741020000E-04, 0.3742260000E-05 /
  data (epsData(198,i,49), i=1,3) / 0.2794000000E+05, -0.6162700000E-04, 0.3878530000E-05 /
  data (epsData(199,i,49), i=1,3) / 0.2794000000E+05, -0.6742240000E-04, 0.3892730000E-05 /
  data (epsData(200,i,49), i=1,3) / 0.2795510000E+05, -0.6946820000E-04, 0.5599760000E-05 /
  data (epsData(201,i,49), i=1,3) / 0.2804970000E+05, -0.7208950000E-04, 0.6005820000E-05 /
  data (epsData(202,i,49), i=1,3) / 0.2869010000E+05, -0.7213670000E-04, 0.5617410000E-05 /
  data (epsData(203,i,49), i=1,3) / 0.3402290000E+05, -0.5380470000E-04, 0.3038780000E-05 /
  data (epsData(204,i,49), i=1,3) / 0.3827300000E+05, -0.4289390000E-04, 0.1960480000E-05 /
  data (epsData(205,i,49), i=1,3) / 0.4291680000E+05, -0.3425740000E-04, 0.1273370000E-05 /
  data (epsData(206,i,49), i=1,3) / 0.4806940000E+05, -0.2736180000E-04, 0.8263680000E-06 /
  data (epsData(207,i,49), i=1,3) / 0.5382180000E+05, -0.2184260000E-04, 0.5344230000E-06 /
  data (epsData(208,i,49), i=1,3) / 0.6026310000E+05, -0.1742380000E-04, 0.3440080000E-06 /
  data (epsData(209,i,49), i=1,3) / 0.6750750000E+05, -0.1388140000E-04, 0.2207920000E-06 /
  data (epsData(210,i,49), i=1,3) / 0.7559640000E+05, -0.1106550000E-04, 0.1403730000E-06 /
  data (epsData(211,i,49), i=1,3) / 0.8467330000E+05, -0.8818200000E-05, 0.8916610000E-07 /
  data (epsData(212,i,49), i=1,3) / 0.9477210000E+05, -0.7045140000E-05, 0.5645390000E-07 /
  data (epsData(213,i,49), i=1,3) / 0.9998790000E+05, -0.6411280000E-05, 0.4536300000E-07 /
  data (epsData(214,i,49), i=1,3) / 0.1000000000E+06, -0.6438350000E-05, 0.4534010000E-07 /
  data (epsData(1,i,50), i=1,3) / 0.2506580000E-02, -0.1567720000E+08, 0.3461540000E+08 /
  data (epsData(2,i,50), i=1,3) / 0.4693440000E-02, -0.1243130000E+08, 0.1124790000E+08 /
  data (epsData(3,i,50), i=1,3) / 0.7560590000E-02, -0.5899560000E+07, 0.5029190000E+07 /
  data (epsData(4,i,50), i=1,3) / 0.1110800000E-01, -0.3066500000E+07, 0.2728060000E+07 /
  data (epsData(5,i,50), i=1,3) / 0.1533570000E-01, -0.1738020000E+07, 0.1700650000E+07 /
  data (epsData(6,i,50), i=1,3) / 0.2024370000E-01, -0.1063840000E+07, 0.1166490000E+07 /
  data (epsData(7,i,50), i=1,3) / 0.2583200000E-01, -0.6954860000E+06, 0.8495930000E+06 /
  data (epsData(8,i,50), i=1,3) / 0.3210060000E-01, -0.4787780000E+06, 0.6469420000E+06 /
  data (epsData(9,i,50), i=1,3) / 0.3904940000E-01, -0.3439670000E+06, 0.5096690000E+06 /
  data (epsData(10,i,50), i=1,3) / 0.4667860000E-01, -0.2563270000E+06, 0.4123780000E+06 /
  data (epsData(11,i,50), i=1,3) / 0.5498800000E-01, -0.1972610000E+06, 0.3408770000E+06 /
  data (epsData(12,i,50), i=1,3) / 0.6397770000E-01, -0.1562400000E+06, 0.2867470000E+06 /
  data (epsData(13,i,50), i=1,3) / 0.7364770000E-01, -0.1270240000E+06, 0.2447490000E+06 /
  data (epsData(14,i,50), i=1,3) / 0.8399800000E-01, -0.1057740000E+06, 0.2114830000E+06 /
  data (epsData(15,i,50), i=1,3) / 0.9502860000E-01, -0.9004730000E+05, 0.1846660000E+06 /
  data (epsData(16,i,50), i=1,3) / 0.1067390000E+00, -0.7824890000E+05, 0.1627180000E+06 /
  data (epsData(17,i,50), i=1,3) / 0.1191310000E+00, -0.6931490000E+05, 0.1445170000E+06 /
  data (epsData(18,i,50), i=1,3) / 0.1322020000E+00, -0.6252530000E+05, 0.1292480000E+06 /
  data (epsData(19,i,50), i=1,3) / 0.1459540000E+00, -0.5739360000E+05, 0.1163090000E+06 /
  data (epsData(20,i,50), i=1,3) / 0.1603860000E+00, -0.5360460000E+05, 0.1052430000E+06 /
  data (epsData(21,i,50), i=1,3) / 0.1754980000E+00, -0.5099560000E+05, 0.9570330000E+05 /
  data (epsData(22,i,50), i=1,3) / 0.1912910000E+00, -0.4964100000E+05, 0.8741850000E+05 /
  data (epsData(23,i,50), i=1,3) / 0.2077630000E+00, -0.5020740000E+05, 0.8017610000E+05 /
  data (epsData(24,i,50), i=1,3) / 0.2249170000E+00, -0.5097900000E+05, 0.6947050000E+05 /
  data (epsData(25,i,50), i=1,3) / 0.2427500000E+00, -0.4959670000E+05, 0.5888110000E+05 /
  data (epsData(26,i,50), i=1,3) / 0.2612640000E+00, -0.4679790000E+05, 0.5015330000E+05 /
  data (epsData(27,i,50), i=1,3) / 0.3003320000E+00, -0.4031120000E+05, 0.3687010000E+05 /
  data (epsData(28,i,50), i=1,3) / 0.3421210000E+00, -0.3414690000E+05, 0.2751970000E+05 /
  data (epsData(29,i,50), i=1,3) / 0.3866320000E+00, -0.2876280000E+05, 0.2080310000E+05 /
  data (epsData(30,i,50), i=1,3) / 0.4338630000E+00, -0.2418180000E+05, 0.1589250000E+05 /
  data (epsData(31,i,50), i=1,3) / 0.4838160000E+00, -0.2029170000E+05, 0.1224630000E+05 /
  data (epsData(32,i,50), i=1,3) / 0.5364900000E+00, -0.1692960000E+05, 0.9644970000E+04 /
  data (epsData(33,i,50), i=1,3) / 0.5918850000E+00, -0.1428090000E+05, 0.7806000000E+04 /
  data (epsData(34,i,50), i=1,3) / 0.6500010000E+00, -0.1214340000E+05, 0.6369820000E+04 /
  data (epsData(35,i,50), i=1,3) / 0.7108380000E+00, -0.1037430000E+05, 0.5236240000E+04 /
  data (epsData(36,i,50), i=1,3) / 0.8071970000E+00, -0.8201590000E+04, 0.3950280000E+04 /
  data (epsData(37,i,50), i=1,3) / 0.8748370000E+00, -0.7028980000E+04, 0.3409650000E+04 /
  data (epsData(38,i,50), i=1,3) / 0.9813990000E+00, -0.5693810000E+04, 0.2780990000E+04 /
  data (epsData(39,i,50), i=1,3) / 0.1094080000E+01, -0.4675210000E+04, 0.2298540000E+04 /
  data (epsData(40,i,50), i=1,3) / 0.1212890000E+01, -0.3881310000E+04, 0.1939880000E+04 /
  data (epsData(41,i,50), i=1,3) / 0.1337820000E+01, -0.3264750000E+04, 0.1657480000E+04 /
  data (epsData(42,i,50), i=1,3) / 0.1513920000E+01, -0.2634430000E+04, 0.1364960000E+04 /
  data (epsData(43,i,50), i=1,3) / 0.1700900000E+01, -0.2164230000E+04, 0.1141610000E+04 /
  data (epsData(44,i,50), i=1,3) / 0.1898770000E+01, -0.1806410000E+04, 0.9625340000E+03 /
  data (epsData(45,i,50), i=1,3) / 0.2161410000E+01, -0.1471070000E+04, 0.7845510000E+03 /
  data (epsData(46,i,50), i=1,3) / 0.2499020000E+01, -0.1177680000E+04, 0.6163260000E+03 /
  data (epsData(47,i,50), i=1,3) / 0.2861130000E+01, -0.9618600000E+03, 0.4829360000E+03 /
  data (epsData(48,i,50), i=1,3) / 0.3247730000E+01, -0.7967740000E+03, 0.3752860000E+03 /
  data (epsData(49,i,50), i=1,3) / 0.3729710000E+01, -0.6455460000E+03, 0.2726920000E+03 /
  data (epsData(50,i,50), i=1,3) / 0.4245030000E+01, -0.5152600000E+03, 0.1953550000E+03 /
  data (epsData(51,i,50), i=1,3) / 0.4633520000E+01, -0.4392420000E+03, 0.1621530000E+03 /
  data (epsData(52,i,50), i=1,3) / 0.5205980000E+01, -0.3599060000E+03, 0.1264630000E+03 /
  data (epsData(53,i,50), i=1,3) / 0.5901040000E+01, -0.2898670000E+03, 0.9409250000E+02 /
  data (epsData(54,i,50), i=1,3) / 0.6639630000E+01, -0.2319630000E+03, 0.7090660000E+02 /
  data (epsData(55,i,50), i=1,3) / 0.6927830000E+01, -0.2170110000E+03, 0.6935830000E+02 /
  data (epsData(56,i,50), i=1,3) / 0.8141850000E+01, -0.1670520000E+03, 0.4096280000E+02 /
  data (epsData(57,i,50), i=1,3) / 0.9005620000E+01, -0.1377340000E+03, 0.2863320000E+02 /
  data (epsData(58,i,50), i=1,3) / 0.9912920000E+01, -0.1135030000E+03, 0.2017110000E+02 /
  data (epsData(59,i,50), i=1,3) / 0.1086380000E+02, -0.9372430000E+02, 0.1444480000E+02 /
  data (epsData(60,i,50), i=1,3) / 0.1198550000E+02, -0.7583850000E+02, 0.9936900000E+01 /
  data (epsData(61,i,50), i=1,3) / 0.1316240000E+02, -0.6150260000E+02, 0.6948600000E+01 /
  data (epsData(62,i,50), i=1,3) / 0.1453460000E+02, -0.4884930000E+02, 0.4758520000E+01 /
  data (epsData(63,i,50), i=1,3) / 0.1597480000E+02, -0.3882600000E+02, 0.3339530000E+01 /
  data (epsData(64,i,50), i=1,3) / 0.1748310000E+02, -0.3078760000E+02, 0.2407780000E+01 /
  data (epsData(65,i,50), i=1,3) / 0.1922080000E+02, -0.2363720000E+02, 0.1736070000E+01 /
  data (epsData(66,i,50), i=1,3) / 0.2121040000E+02, -0.1721610000E+02, 0.1267090000E+01 /
  data (epsData(67,i,50), i=1,3) / 0.2312020000E+02, -0.1173620000E+02, 0.1044650000E+01 /
  data (epsData(68,i,50), i=1,3) / 0.2347620000E+02, -0.1048890000E+02, 0.1088740000E+01 /
  data (epsData(69,i,50), i=1,3) / 0.2383500000E+02, -0.9030800000E+01, 0.1693160000E+01 /
  data (epsData(70,i,50), i=1,3) / 0.2397650000E+02, -0.8694840000E+01, 0.2340410000E+01 /
  data (epsData(71,i,50), i=1,3) / 0.2411260000E+02, -0.8748000000E+01, 0.2844490000E+01 /
  data (epsData(72,i,50), i=1,3) / 0.2450000000E+02, -0.8583120000E+01, 0.3010910000E+01 /
  data (epsData(73,i,50), i=1,3) / 0.2481970000E+02, -0.8015650000E+01, 0.3215790000E+01 /
  data (epsData(74,i,50), i=1,3) / 0.2509590000E+02, -0.7844340000E+01, 0.3775670000E+01 /
  data (epsData(75,i,50), i=1,3) / 0.2561970000E+02, -0.7711670000E+01, 0.3798970000E+01 /
  data (epsData(76,i,50), i=1,3) / 0.2707690000E+02, -0.6180770000E+01, 0.3652640000E+01 /
  data (epsData(77,i,50), i=1,3) / 0.2963610000E+02, -0.3241280000E+01, 0.3831690000E+01 /
  data (epsData(78,i,50), i=1,3) / 0.3119130000E+02, -0.1534280000E+01, 0.4313180000E+01 /
  data (epsData(79,i,50), i=1,3) / 0.3255320000E+02, -0.2160350000E+00, 0.4951120000E+01 /
  data (epsData(80,i,50), i=1,3) / 0.3382670000E+02, 0.8218360000E+00, 0.5736000000E+01 /
  data (epsData(81,i,50), i=1,3) / 0.3543620000E+02, 0.1772530000E+01, 0.6925600000E+01 /
  data (epsData(82,i,50), i=1,3) / 0.3756990000E+02, 0.2348020000E+01, 0.8615170000E+01 /
  data (epsData(83,i,50), i=1,3) / 0.3961590000E+02, 0.2222900000E+01, 0.1009630000E+02 /
  data (epsData(84,i,50), i=1,3) / 0.4196700000E+02, 0.1521290000E+01, 0.1133850000E+02 /
  data (epsData(85,i,50), i=1,3) / 0.4598910000E+02, -0.5108630000E-01, 0.1251500000E+02 /
  data (epsData(86,i,50), i=1,3) / 0.5025040000E+02, -0.2060910000E+01, 0.1302460000E+02 /
  data (epsData(87,i,50), i=1,3) / 0.5410080000E+02, -0.4197180000E+01, 0.1275150000E+02 /
  data (epsData(88,i,50), i=1,3) / 0.5779890000E+02, -0.6114580000E+01, 0.1158590000E+02 /
  data (epsData(89,i,50), i=1,3) / 0.6171730000E+02, -0.7492950000E+01, 0.9666490000E+01 /
  data (epsData(90,i,50), i=1,3) / 0.6632960000E+02, -0.8014310000E+01, 0.7276250000E+01 /
  data (epsData(91,i,50), i=1,3) / 0.7121410000E+02, -0.7758430000E+01, 0.5177780000E+01 /
  data (epsData(92,i,50), i=1,3) / 0.7637060000E+02, -0.7059700000E+01, 0.3526710000E+01 /
  data (epsData(93,i,50), i=1,3) / 0.8124420000E+02, -0.6250460000E+01, 0.2435170000E+01 /
  data (epsData(94,i,50), i=1,3) / 0.8727900000E+02, -0.5314970000E+01, 0.1576490000E+01 /
  data (epsData(95,i,50), i=1,3) / 0.9401680000E+02, -0.4445640000E+01, 0.9773910000E+00 /
  data (epsData(96,i,50), i=1,3) / 0.1006670000E+03, -0.3727600000E+01, 0.6096770000E+00 /
  data (epsData(97,i,50), i=1,3) / 0.1078480000E+03, -0.3101370000E+01, 0.3792930000E+00 /
  data (epsData(98,i,50), i=1,3) / 0.1154500000E+03, -0.2579340000E+01, 0.2448160000E+00 /
  data (epsData(99,i,50), i=1,3) / 0.1240850000E+03, -0.2126080000E+01, 0.1667390000E+00 /
  data (epsData(100,i,50), i=1,3) / 0.1340180000E+03, -0.1728350000E+01, 0.1222300000E+00 /
  data (epsData(101,i,50), i=1,3) / 0.1441530000E+03, -0.1432270000E+01, 0.1207370000E+00 /
  data (epsData(102,i,50), i=1,3) / 0.1585290000E+03, -0.1140110000E+01, 0.1180550000E+00 /
  data (epsData(103,i,50), i=1,3) / 0.1748180000E+03, -0.9100870000E+00, 0.1161170000E+00 /
  data (epsData(104,i,50), i=1,3) / 0.1926170000E+03, -0.7351410000E+00, 0.1131410000E+00 /
  data (epsData(105,i,50), i=1,3) / 0.2144930000E+03, -0.5851390000E+00, 0.1067390000E+00 /
  data (epsData(106,i,50), i=1,3) / 0.2392470000E+03, -0.4668480000E+00, 0.9644480000E-01 /
  data (epsData(107,i,50), i=1,3) / 0.2674310000E+03, -0.3724110000E+00, 0.8396790000E-01 /
  data (epsData(108,i,50), i=1,3) / 0.2967060000E+03, -0.2992320000E+00, 0.7154630000E-01 /
  data (epsData(109,i,50), i=1,3) / 0.3279240000E+03, -0.2407570000E+00, 0.6003730000E-01 /
  data (epsData(110,i,50), i=1,3) / 0.3580740000E+03, -0.1942940000E+00, 0.5071440000E-01 /
  data (epsData(111,i,50), i=1,3) / 0.3765390000E+03, -0.1717170000E+00, 0.4581580000E-01 /
  data (epsData(112,i,50), i=1,3) / 0.3834370000E+03, -0.1619730000E+00, 0.4410880000E-01 /
  data (epsData(113,i,50), i=1,3) / 0.4083530000E+03, -0.1353300000E+00, 0.3852480000E-01 /
  data (epsData(114,i,50), i=1,3) / 0.4163080000E+03, -0.1256480000E+00, 0.3694640000E-01 /
  data (epsData(115,i,50), i=1,3) / 0.4306640000E+03, -0.1114300000E+00, 0.3427820000E-01 /
  data (epsData(116,i,50), i=1,3) / 0.4319300000E+03, -0.1084940000E+00, 0.3405450000E-01 /
  data (epsData(117,i,50), i=1,3) / 0.4338330000E+03, -0.1073130000E+00, 0.3372020000E-01 /
  data (epsData(118,i,50), i=1,3) / 0.4357420000E+03, -0.1038840000E+00, 0.3338870000E-01 /
  data (epsData(119,i,50), i=1,3) / 0.4370180000E+03, -0.1045330000E+00, 0.3316880000E-01 /
  data (epsData(120,i,50), i=1,3) / 0.4427950000E+03, -0.9709550000E-01, 0.3219180000E-01 /
  data (epsData(121,i,50), i=1,3) / 0.4447330000E+03, -0.9542030000E-01, 0.3187090000E-01 /
  data (epsData(122,i,50), i=1,3) / 0.4473260000E+03, -0.9220010000E-01, 0.3144710000E-01 /
  data (epsData(123,i,50), i=1,3) / 0.4499310000E+03, -0.9033510000E-01, 0.3102790000E-01 /
  data (epsData(124,i,50), i=1,3) / 0.4512370000E+03, -0.8679780000E-01, 0.3081970000E-01 /
  data (epsData(125,i,50), i=1,3) / 0.4538570000E+03, -0.8414790000E-01, 0.3040640000E-01 /
  data (epsData(126,i,50), i=1,3) / 0.4578090000E+03, -0.7997460000E-01, 0.2979350000E-01 /
  data (epsData(127,i,50), i=1,3) / 0.4617840000E+03, -0.7633330000E-01, 0.2918920000E-01 /
  data (epsData(128,i,50), i=1,3) / 0.4651160000E+03, -0.7102880000E-01, 0.2879150000E-01 /
  data (epsData(129,i,50), i=1,3) / 0.4684650000E+03, -0.6576640000E-01, 0.2832160000E-01 /
  data (epsData(130,i,50), i=1,3) / 0.4698090000E+03, -0.6696950000E-01, 0.2813520000E-01 /
  data (epsData(131,i,50), i=1,3) / 0.4718310000E+03, -0.6021260000E-01, 0.2785900000E-01 /
  data (epsData(132,i,50), i=1,3) / 0.4731820000E+03, -0.6200750000E-01, 0.2767690000E-01 /
  data (epsData(133,i,50), i=1,3) / 0.4765720000E+03, -0.5424950000E-01, 0.2730520000E-01 /
  data (epsData(134,i,50), i=1,3) / 0.4792960000E+03, -0.5208260000E-01, 0.2697030000E-01 /
  data (epsData(135,i,50), i=1,3) / 0.4806620000E+03, -0.4721290000E-01, 0.2681260000E-01 /
  data (epsData(136,i,50), i=1,3) / 0.4827160000E+03, -0.4587390000E-01, 0.2660940000E-01 /
  data (epsData(137,i,50), i=1,3) / 0.4847760000E+03, -0.3875140000E-01, 0.2709320000E-01 /
  data (epsData(138,i,50), i=1,3) / 0.4849000000E+03, -0.4086940000E-01, 0.2732620000E-01 /
  data (epsData(139,i,50), i=1,3) / 0.4849030000E+03, -0.3930070000E-01, 0.2733140000E-01 /
  data (epsData(140,i,50), i=1,3) / 0.4849170000E+03, -0.4079080000E-01, 0.2735750000E-01 /
  data (epsData(141,i,50), i=1,3) / 0.4849340000E+03, -0.4019360000E-01, 0.2739090000E-01 /
  data (epsData(142,i,50), i=1,3) / 0.4849410000E+03, -0.3842820000E-01, 0.2740470000E-01 /
  data (epsData(143,i,50), i=1,3) / 0.4849490000E+03, -0.3881270000E-01, 0.2741980000E-01 /
  data (epsData(144,i,50), i=1,3) / 0.4849570000E+03, -0.4089550000E-01, 0.2743610000E-01 /
  data (epsData(145,i,50), i=1,3) / 0.4849770000E+03, -0.3888120000E-01, 0.2747280000E-01 /
  data (epsData(146,i,50), i=1,3) / 0.4850650000E+03, -0.4121080000E-01, 0.2764230000E-01 /
  data (epsData(147,i,50), i=1,3) / 0.4850960000E+03, -0.3924790000E-01, 0.2770250000E-01 /
  data (epsData(148,i,50), i=1,3) / 0.4852060000E+03, -0.3969940000E-01, 0.2781090000E-01 /
  data (epsData(149,i,50), i=1,3) / 0.4852270000E+03, -0.3847650000E-01, 0.2783060000E-01 /
  data (epsData(150,i,50), i=1,3) / 0.4852930000E+03, -0.3879740000E-01, 0.2789350000E-01 /
  data (epsData(151,i,50), i=1,3) / 0.4854170000E+03, -0.3843090000E-01, 0.2799930000E-01 /
  data (epsData(152,i,50), i=1,3) / 0.4854720000E+03, -0.3739260000E-01, 0.2801400000E-01 /
  data (epsData(153,i,50), i=1,3) / 0.4855290000E+03, -0.3935610000E-01, 0.2802940000E-01 /
  data (epsData(154,i,50), i=1,3) / 0.4855890000E+03, -0.3937120000E-01, 0.2804560000E-01 /
  data (epsData(155,i,50), i=1,3) / 0.4856200000E+03, -0.3811910000E-01, 0.2805390000E-01 /
  data (epsData(156,i,50), i=1,3) / 0.4856840000E+03, -0.4032740000E-01, 0.2807120000E-01 /
  data (epsData(157,i,50), i=1,3) / 0.4857500000E+03, -0.3755630000E-01, 0.2807920000E-01 /
  data (epsData(158,i,50), i=1,3) / 0.4858200000E+03, -0.3935220000E-01, 0.2808420000E-01 /
  data (epsData(159,i,50), i=1,3) / 0.4858920000E+03, -0.3859280000E-01, 0.2808930000E-01 /
  data (epsData(160,i,50), i=1,3) / 0.4860050000E+03, -0.3902740000E-01, 0.2809740000E-01 /
  data (epsData(161,i,50), i=1,3) / 0.4860840000E+03, -0.4040790000E-01, 0.2809860000E-01 /
  data (epsData(162,i,50), i=1,3) / 0.4862070000E+03, -0.3683860000E-01, 0.2810040000E-01 /
  data (epsData(163,i,50), i=1,3) / 0.4862930000E+03, -0.3778020000E-01, 0.2810160000E-01 /
  data (epsData(164,i,50), i=1,3) / 0.4863370000E+03, -0.3662740000E-01, 0.2810160000E-01 /
  data (epsData(165,i,50), i=1,3) / 0.4864270000E+03, -0.3673760000E-01, 0.2810120000E-01 /
  data (epsData(166,i,50), i=1,3) / 0.4864730000E+03, -0.3568480000E-01, 0.2810100000E-01 /
  data (epsData(167,i,50), i=1,3) / 0.4866150000E+03, -0.3779470000E-01, 0.2810040000E-01 /
  data (epsData(168,i,50), i=1,3) / 0.4867630000E+03, -0.3715440000E-01, 0.2809920000E-01 /
  data (epsData(169,i,50), i=1,3) / 0.4868130000E+03, -0.3744030000E-01, 0.2809880000E-01 /
  data (epsData(170,i,50), i=1,3) / 0.4868650000E+03, -0.3614580000E-01, 0.2809840000E-01 /
  data (epsData(171,i,50), i=1,3) / 0.4869690000E+03, -0.3788330000E-01, 0.2809770000E-01 /
  data (epsData(172,i,50), i=1,3) / 0.4871320000E+03, -0.3656530000E-01, 0.2809670000E-01 /
  data (epsData(173,i,50), i=1,3) / 0.4875340000E+03, -0.3640420000E-01, 0.2809590000E-01 /
  data (epsData(174,i,50), i=1,3) / 0.4875940000E+03, -0.3536390000E-01, 0.2809620000E-01 /
  data (epsData(175,i,50), i=1,3) / 0.4879700000E+03, -0.3579060000E-01, 0.2810020000E-01 /
  data (epsData(176,i,50), i=1,3) / 0.4880350000E+03, -0.3446720000E-01, 0.2810120000E-01 /
  data (epsData(177,i,50), i=1,3) / 0.4881000000E+03, -0.3447340000E-01, 0.2810230000E-01 /
  data (epsData(178,i,50), i=1,3) / 0.4881670000E+03, -0.3639080000E-01, 0.2810410000E-01 /
  data (epsData(179,i,50), i=1,3) / 0.4883010000E+03, -0.3303760000E-01, 0.2810770000E-01 /
  data (epsData(180,i,50), i=1,3) / 0.4883700000E+03, -0.3271490000E-01, 0.2810960000E-01 /
  data (epsData(181,i,50), i=1,3) / 0.4885790000E+03, -0.3517790000E-01, 0.2811310000E-01 /
  data (epsData(182,i,50), i=1,3) / 0.4886500000E+03, -0.3457940000E-01, 0.2811430000E-01 /
  data (epsData(183,i,50), i=1,3) / 0.4887220000E+03, -0.3212940000E-01, 0.2811420000E-01 /
  data (epsData(184,i,50), i=1,3) / 0.4889410000E+03, -0.3291710000E-01, 0.2810500000E-01 /
  data (epsData(185,i,50), i=1,3) / 0.4890910000E+03, -0.3507850000E-01, 0.2809760000E-01 /
  data (epsData(186,i,50), i=1,3) / 0.4892430000E+03, -0.3221600000E-01, 0.2808930000E-01 /
  data (epsData(187,i,50), i=1,3) / 0.4895560000E+03, -0.3087330000E-01, 0.2808960000E-01 /
  data (epsData(188,i,50), i=1,3) / 0.4897170000E+03, -0.3217060000E-01, 0.2809930000E-01 /
  data (epsData(189,i,50), i=1,3) / 0.4898800000E+03, -0.3075410000E-01, 0.2811210000E-01 /
  data (epsData(190,i,50), i=1,3) / 0.4903860000E+03, -0.3199760000E-01, 0.2816450000E-01 /
  data (epsData(191,i,50), i=1,3) / 0.4905600000E+03, -0.2822680000E-01, 0.2818450000E-01 /
  data (epsData(192,i,50), i=1,3) / 0.4907370000E+03, -0.2998370000E-01, 0.2820630000E-01 /
  data (epsData(193,i,50), i=1,3) / 0.4908270000E+03, -0.2871600000E-01, 0.2821810000E-01 /
  data (epsData(194,i,50), i=1,3) / 0.4910990000E+03, -0.2937780000E-01, 0.2825710000E-01 /
  data (epsData(195,i,50), i=1,3) / 0.4912840000E+03, -0.2852830000E-01, 0.2828680000E-01 /
  data (epsData(196,i,50), i=1,3) / 0.4913780000E+03, -0.2719220000E-01, 0.2830270000E-01 /
  data (epsData(197,i,50), i=1,3) / 0.4915670000E+03, -0.2768910000E-01, 0.2833640000E-01 /
  data (epsData(198,i,50), i=1,3) / 0.4916620000E+03, -0.2690330000E-01, 0.2835470000E-01 /
  data (epsData(199,i,50), i=1,3) / 0.4917590000E+03, -0.2779750000E-01, 0.2837440000E-01 /
  data (epsData(200,i,50), i=1,3) / 0.4919530000E+03, -0.2560960000E-01, 0.2841690000E-01 /
  data (epsData(201,i,50), i=1,3) / 0.4921500000E+03, -0.2657260000E-01, 0.2846750000E-01 /
  data (epsData(202,i,50), i=1,3) / 0.4923500000E+03, -0.2459290000E-01, 0.2853120000E-01 /
  data (epsData(203,i,50), i=1,3) / 0.4924510000E+03, -0.2456210000E-01, 0.2856690000E-01 /
  data (epsData(204,i,50), i=1,3) / 0.4925530000E+03, -0.2621120000E-01, 0.2861210000E-01 /
  data (epsData(205,i,50), i=1,3) / 0.4928620000E+03, -0.2348710000E-01, 0.2880960000E-01 /
  data (epsData(206,i,50), i=1,3) / 0.4929670000E+03, -0.2179670000E-01, 0.2891720000E-01 /
  data (epsData(207,i,50), i=1,3) / 0.4931780000E+03, -0.2398440000E-01, 0.2916930000E-01 /
  data (epsData(208,i,50), i=1,3) / 0.4932000000E+03, -0.2485050000E-01, 0.2920260000E-01 /
  data (epsData(209,i,50), i=1,3) / 0.4932010000E+03, -0.2317830000E-01, 0.2920410000E-01 /
  data (epsData(210,i,50), i=1,3) / 0.4932030000E+03, -0.2142120000E-01, 0.2920660000E-01 /
  data (epsData(211,i,50), i=1,3) / 0.4932120000E+03, -0.2214490000E-01, 0.2922000000E-01 /
  data (epsData(212,i,50), i=1,3) / 0.4932170000E+03, -0.2354170000E-01, 0.2922640000E-01 /
  data (epsData(213,i,50), i=1,3) / 0.4932280000E+03, -0.2122010000E-01, 0.2924230000E-01 /
  data (epsData(214,i,50), i=1,3) / 0.4932340000E+03, -0.2251010000E-01, 0.2925170000E-01 /
  data (epsData(215,i,50), i=1,3) / 0.4932410000E+03, -0.2281420000E-01, 0.2926210000E-01 /
  data (epsData(216,i,50), i=1,3) / 0.4932490000E+03, -0.2199910000E-01, 0.2927360000E-01 /
  data (epsData(217,i,50), i=1,3) / 0.4932670000E+03, -0.2444540000E-01, 0.2929940000E-01 /
  data (epsData(218,i,50), i=1,3) / 0.4932870000E+03, -0.2345990000E-01, 0.2932910000E-01 /
  data (epsData(219,i,50), i=1,3) / 0.4932980000E+03, -0.2453170000E-01, 0.2934540000E-01 /
  data (epsData(220,i,50), i=1,3) / 0.4933100000E+03, -0.2399830000E-01, 0.2936280000E-01 /
  data (epsData(221,i,50), i=1,3) / 0.4933230000E+03, -0.2196930000E-01, 0.2938110000E-01 /
  data (epsData(222,i,50), i=1,3) / 0.4933500000E+03, -0.2225560000E-01, 0.2942080000E-01 /
  data (epsData(223,i,50), i=1,3) / 0.4933800000E+03, -0.2343350000E-01, 0.2946440000E-01 /
  data (epsData(224,i,50), i=1,3) / 0.4933960000E+03, -0.2278950000E-01, 0.2948770000E-01 /
  data (epsData(225,i,50), i=1,3) / 0.4934130000E+03, -0.2320880000E-01, 0.2950540000E-01 /
  data (epsData(226,i,50), i=1,3) / 0.4934480000E+03, -0.2112860000E-01, 0.2953850000E-01 /
  data (epsData(227,i,50), i=1,3) / 0.4934670000E+03, -0.2318880000E-01, 0.2955600000E-01 /
  data (epsData(228,i,50), i=1,3) / 0.4934860000E+03, -0.2399540000E-01, 0.2957410000E-01 /
  data (epsData(229,i,50), i=1,3) / 0.4935060000E+03, -0.2260220000E-01, 0.2959310000E-01 /
  data (epsData(230,i,50), i=1,3) / 0.4935270000E+03, -0.2307740000E-01, 0.2961320000E-01 /
  data (epsData(231,i,50), i=1,3) / 0.4935700000E+03, -0.2161810000E-01, 0.2965530000E-01 /
  data (epsData(232,i,50), i=1,3) / 0.4935930000E+03, -0.2378040000E-01, 0.2967740000E-01 /
  data (epsData(233,i,50), i=1,3) / 0.4936660000E+03, -0.2378850000E-01, 0.2974740000E-01 /
  data (epsData(234,i,50), i=1,3) / 0.4936910000E+03, -0.2307010000E-01, 0.2977210000E-01 /
  data (epsData(235,i,50), i=1,3) / 0.4937170000E+03, -0.2123830000E-01, 0.2979080000E-01 /
  data (epsData(236,i,50), i=1,3) / 0.4937720000E+03, -0.2089080000E-01, 0.2982270000E-01 /
  data (epsData(237,i,50), i=1,3) / 0.4938590000E+03, -0.2133900000E-01, 0.2987510000E-01 /
  data (epsData(238,i,50), i=1,3) / 0.4939200000E+03, -0.2234550000E-01, 0.2991260000E-01 /
  data (epsData(239,i,50), i=1,3) / 0.4939840000E+03, -0.2162010000E-01, 0.2995180000E-01 /
  data (epsData(240,i,50), i=1,3) / 0.4940500000E+03, -0.2279290000E-01, 0.2998660000E-01 /
  data (epsData(241,i,50), i=1,3) / 0.4941920000E+03, -0.2054170000E-01, 0.3005890000E-01 /
  data (epsData(242,i,50), i=1,3) / 0.4942670000E+03, -0.2214190000E-01, 0.3009800000E-01 /
  data (epsData(243,i,50), i=1,3) / 0.4943440000E+03, -0.2007030000E-01, 0.3013660000E-01 /
  data (epsData(244,i,50), i=1,3) / 0.4944660000E+03, -0.1995250000E-01, 0.3019710000E-01 /
  data (epsData(245,i,50), i=1,3) / 0.4945080000E+03, -0.1871340000E-01, 0.3021860000E-01 /
  data (epsData(246,i,50), i=1,3) / 0.4945500000E+03, -0.1849770000E-01, 0.3024040000E-01 /
  data (epsData(247,i,50), i=1,3) / 0.4946370000E+03, -0.2159360000E-01, 0.3028450000E-01 /
  data (epsData(248,i,50), i=1,3) / 0.4946820000E+03, -0.2134940000E-01, 0.3030660000E-01 /
  data (epsData(249,i,50), i=1,3) / 0.4947270000E+03, -0.1997130000E-01, 0.3032990000E-01 /
  data (epsData(250,i,50), i=1,3) / 0.4948670000E+03, -0.2015960000E-01, 0.3040320000E-01 /
  data (epsData(251,i,50), i=1,3) / 0.4949150000E+03, -0.2115920000E-01, 0.3042850000E-01 /
  data (epsData(252,i,50), i=1,3) / 0.4950130000E+03, -0.1800290000E-01, 0.3048130000E-01 /
  data (epsData(253,i,50), i=1,3) / 0.4953230000E+03, -0.1854830000E-01, 0.3065580000E-01 /
  data (epsData(254,i,50), i=1,3) / 0.4953770000E+03, -0.1918100000E-01, 0.3068600000E-01 /
  data (epsData(255,i,50), i=1,3) / 0.4954870000E+03, -0.1898710000E-01, 0.3075250000E-01 /
  data (epsData(256,i,50), i=1,3) / 0.4955430000E+03, -0.1748790000E-01, 0.3078690000E-01 /
  data (epsData(257,i,50), i=1,3) / 0.4956580000E+03, -0.1838720000E-01, 0.3085690000E-01 /
  data (epsData(258,i,50), i=1,3) / 0.4957160000E+03, -0.1757450000E-01, 0.3089250000E-01 /
  data (epsData(259,i,50), i=1,3) / 0.4957750000E+03, -0.1880320000E-01, 0.3092850000E-01 /
  data (epsData(260,i,50), i=1,3) / 0.4958940000E+03, -0.1747010000E-01, 0.3100180000E-01 /
  data (epsData(261,i,50), i=1,3) / 0.4959550000E+03, -0.1766460000E-01, 0.3104270000E-01 /
  data (epsData(262,i,50), i=1,3) / 0.4960790000E+03, -0.1466580000E-01, 0.3112640000E-01 /
  data (epsData(263,i,50), i=1,3) / 0.4963350000E+03, -0.1500790000E-01, 0.3129990000E-01 /
  data (epsData(264,i,50), i=1,3) / 0.4964670000E+03, -0.1606730000E-01, 0.3139400000E-01 /
  data (epsData(265,i,50), i=1,3) / 0.4965340000E+03, -0.1507740000E-01, 0.3144410000E-01 /
  data (epsData(266,i,50), i=1,3) / 0.4966700000E+03, -0.1578430000E-01, 0.3154580000E-01 /
  data (epsData(267,i,50), i=1,3) / 0.4968090000E+03, -0.1489170000E-01, 0.3164810000E-01 /
  data (epsData(268,i,50), i=1,3) / 0.4968790000E+03, -0.1538520000E-01, 0.3169980000E-01 /
  data (epsData(269,i,50), i=1,3) / 0.4969500000E+03, -0.1387560000E-01, 0.3175530000E-01 /
  data (epsData(270,i,50), i=1,3) / 0.4970220000E+03, -0.1385800000E-01, 0.3181200000E-01 /
  data (epsData(271,i,50), i=1,3) / 0.4970940000E+03, -0.1478110000E-01, 0.3186720000E-01 /
  data (epsData(272,i,50), i=1,3) / 0.4971670000E+03, -0.1271380000E-01, 0.3192290000E-01 /
  data (epsData(273,i,50), i=1,3) / 0.4972410000E+03, -0.1188480000E-01, 0.3197900000E-01 /
  data (epsData(274,i,50), i=1,3) / 0.4973160000E+03, -0.1289500000E-01, 0.3203540000E-01 /
  data (epsData(275,i,50), i=1,3) / 0.4974670000E+03, -0.1268220000E-01, 0.3215320000E-01 /
  data (epsData(276,i,50), i=1,3) / 0.4975430000E+03, -0.1341870000E-01, 0.3221590000E-01 /
  data (epsData(277,i,50), i=1,3) / 0.4976200000E+03, -0.1236600000E-01, 0.3227980000E-01 /
  data (epsData(278,i,50), i=1,3) / 0.4979360000E+03, -0.1304070000E-01, 0.3255400000E-01 /
  data (epsData(279,i,50), i=1,3) / 0.4981800000E+03, -0.1195580000E-01, 0.3278950000E-01 /
  data (epsData(280,i,50), i=1,3) / 0.4982630000E+03, -0.9834640000E-02, 0.3287030000E-01 /
  data (epsData(281,i,50), i=1,3) / 0.4983460000E+03, -0.9022190000E-02, 0.3295190000E-01 /
  data (epsData(282,i,50), i=1,3) / 0.4986000000E+03, -0.9043960000E-02, 0.3321530000E-01 /
  data (epsData(283,i,50), i=1,3) / 0.4987730000E+03, -0.9533190000E-02, 0.3339710000E-01 /
  data (epsData(284,i,50), i=1,3) / 0.4988600000E+03, -0.8774200000E-02, 0.3348930000E-01 /
  data (epsData(285,i,50), i=1,3) / 0.4990370000E+03, -0.9408920000E-02, 0.3368630000E-01 /
  data (epsData(286,i,50), i=1,3) / 0.4991270000E+03, -0.7967340000E-02, 0.3378750000E-01 /
  data (epsData(287,i,50), i=1,3) / 0.4992170000E+03, -0.7387610000E-02, 0.3388970000E-01 /
  data (epsData(288,i,50), i=1,3) / 0.4995840000E+03, -0.8333410000E-02, 0.3432080000E-01 /
  data (epsData(289,i,50), i=1,3) / 0.4997720000E+03, -0.6912630000E-02, 0.3454860000E-01 /
  data (epsData(290,i,50), i=1,3) / 0.4999620000E+03, -0.5044880000E-02, 0.3478440000E-01 /
  data (epsData(291,i,50), i=1,3) / 0.5000590000E+03, -0.5319110000E-02, 0.3490840000E-01 /
  data (epsData(292,i,50), i=1,3) / 0.5002530000E+03, -0.3231940000E-02, 0.3515940000E-01 /
  data (epsData(293,i,50), i=1,3) / 0.5003510000E+03, -0.3782010000E-02, 0.3528660000E-01 /
  data (epsData(294,i,50), i=1,3) / 0.5004500000E+03, -0.3304760000E-02, 0.3541830000E-01 /
  data (epsData(295,i,50), i=1,3) / 0.5005500000E+03, -0.4459730000E-02, 0.3555380000E-01 /
  data (epsData(296,i,50), i=1,3) / 0.5006500000E+03, -0.4568380000E-02, 0.3569080000E-01 /
  data (epsData(297,i,50), i=1,3) / 0.5007510000E+03, -0.2187830000E-02, 0.3582930000E-01 /
  data (epsData(298,i,50), i=1,3) / 0.5009560000E+03, -0.4268790000E-02, 0.3611300000E-01 /
  data (epsData(299,i,50), i=1,3) / 0.5010590000E+03, -0.2989910000E-02, 0.3626190000E-01 /
  data (epsData(300,i,50), i=1,3) / 0.5013720000E+03, -0.6617970000E-03, 0.3671560000E-01 /
  data (epsData(301,i,50), i=1,3) / 0.5014780000E+03, -0.5214050000E-03, 0.3687460000E-01 /
  data (epsData(302,i,50), i=1,3) / 0.5016910000E+03, -0.2200240000E-02, 0.3720240000E-01 /
  data (epsData(303,i,50), i=1,3) / 0.5017990000E+03, -0.2822360000E-03, 0.3736780000E-01 /
  data (epsData(304,i,50), i=1,3) / 0.5019080000E+03, 0.6091320000E-03, 0.3753660000E-01 /
  data (epsData(305,i,50), i=1,3) / 0.5020170000E+03, -0.6638600000E-03, 0.3771330000E-01 /
  data (epsData(306,i,50), i=1,3) / 0.5021270000E+03, 0.5486710000E-03, 0.3789130000E-01 /
  data (epsData(307,i,50), i=1,3) / 0.5022370000E+03, 0.2738400000E-02, 0.3807190000E-01 /
  data (epsData(308,i,50), i=1,3) / 0.5023480000E+03, 0.3115630000E-02, 0.3825340000E-01 /
  data (epsData(309,i,50), i=1,3) / 0.5024600000E+03, 0.1979340000E-02, 0.3844100000E-01 /
  data (epsData(310,i,50), i=1,3) / 0.5030300000E+03, 0.2345260000E-02, 0.3943260000E-01 /
  data (epsData(311,i,50), i=1,3) / 0.5032630000E+03, 0.2364650000E-02, 0.3985680000E-01 /
  data (epsData(312,i,50), i=1,3) / 0.5039770000E+03, 0.7213850000E-02, 0.4123930000E-01 /
  data (epsData(313,i,50), i=1,3) / 0.5040990000E+03, 0.5440540000E-02, 0.4148740000E-01 /
  data (epsData(314,i,50), i=1,3) / 0.5042210000E+03, 0.6650600000E-02, 0.4173740000E-01 /
  data (epsData(315,i,50), i=1,3) / 0.5044670000E+03, 0.5847100000E-02, 0.4225600000E-01 /
  data (epsData(316,i,50), i=1,3) / 0.5045910000E+03, 0.9037890000E-02, 0.4252600000E-01 /
  data (epsData(317,i,50), i=1,3) / 0.5049680000E+03, 0.8481430000E-02, 0.4336390000E-01 /
  data (epsData(318,i,50), i=1,3) / 0.5050940000E+03, 0.9233780000E-02, 0.4366140000E-01 /
  data (epsData(319,i,50), i=1,3) / 0.5052220000E+03, 0.1150520000E-01, 0.4396120000E-01 /
  data (epsData(320,i,50), i=1,3) / 0.5056090000E+03, 0.1037750000E-01, 0.4488260000E-01 /
  data (epsData(321,i,50), i=1,3) / 0.5058700000E+03, 0.1335770000E-01, 0.4552660000E-01 /
  data (epsData(322,i,50), i=1,3) / 0.5062670000E+03, 0.1382350000E-01, 0.4655890000E-01 /
  data (epsData(323,i,50), i=1,3) / 0.5065350000E+03, 0.1191680000E-01, 0.4726440000E-01 /
  data (epsData(324,i,50), i=1,3) / 0.5068060000E+03, 0.1631210000E-01, 0.4800350000E-01 /
  data (epsData(325,i,50), i=1,3) / 0.5069420000E+03, 0.1480580000E-01, 0.4839340000E-01 /
  data (epsData(326,i,50), i=1,3) / 0.5072170000E+03, 0.1593890000E-01, 0.4917840000E-01 /
  data (epsData(327,i,50), i=1,3) / 0.5074940000E+03, 0.1495920000E-01, 0.4998220000E-01 /
  data (epsData(328,i,50), i=1,3) / 0.5077750000E+03, 0.1787540000E-01, 0.5082840000E-01 /
  data (epsData(329,i,50), i=1,3) / 0.5079160000E+03, 0.1830230000E-01, 0.5126960000E-01 /
  data (epsData(330,i,50), i=1,3) / 0.5080580000E+03, 0.1732620000E-01, 0.5171210000E-01 /
  data (epsData(331,i,50), i=1,3) / 0.5086320000E+03, 0.1952800000E-01, 0.5353680000E-01 /
  data (epsData(332,i,50), i=1,3) / 0.5087770000E+03, 0.1873650000E-01, 0.5402660000E-01 /
  data (epsData(333,i,50), i=1,3) / 0.5093650000E+03, 0.2226540000E-01, 0.5602140000E-01 /
  data (epsData(334,i,50), i=1,3) / 0.5099640000E+03, 0.2231040000E-01, 0.5817190000E-01 /
  data (epsData(335,i,50), i=1,3) / 0.5102670000E+03, 0.2621690000E-01, 0.5928350000E-01 /
  data (epsData(336,i,50), i=1,3) / 0.5105730000E+03, 0.2372280000E-01, 0.6043200000E-01 /
  data (epsData(337,i,50), i=1,3) / 0.5108820000E+03, 0.2447320000E-01, 0.6164020000E-01 /
  data (epsData(338,i,50), i=1,3) / 0.5110370000E+03, 0.2615670000E-01, 0.6224990000E-01 /
  data (epsData(339,i,50), i=1,3) / 0.5116660000E+03, 0.2682910000E-01, 0.6476650000E-01 /
  data (epsData(340,i,50), i=1,3) / 0.5123050000E+03, 0.2902650000E-01, 0.6744500000E-01 /
  data (epsData(341,i,50), i=1,3) / 0.5127920000E+03, 0.2907890000E-01, 0.6954770000E-01 /
  data (epsData(342,i,50), i=1,3) / 0.5131200000E+03, 0.3021890000E-01, 0.7099450000E-01 /
  data (epsData(343,i,50), i=1,3) / 0.5134510000E+03, 0.2988600000E-01, 0.7247200000E-01 /
  data (epsData(344,i,50), i=1,3) / 0.5137840000E+03, 0.2954550000E-01, 0.7398620000E-01 /
  data (epsData(345,i,50), i=1,3) / 0.5146290000E+03, 0.3237110000E-01, 0.7788820000E-01 /
  data (epsData(346,i,50), i=1,3) / 0.5149720000E+03, 0.3039650000E-01, 0.7948240000E-01 /
  data (epsData(347,i,50), i=1,3) / 0.5154920000E+03, 0.3437110000E-01, 0.8194990000E-01 /
  data (epsData(348,i,50), i=1,3) / 0.5163710000E+03, 0.3342980000E-01, 0.8614740000E-01 /
  data (epsData(349,i,50), i=1,3) / 0.5174480000E+03, 0.3235260000E-01, 0.9134490000E-01 /
  data (epsData(350,i,50), i=1,3) / 0.5185500000E+03, 0.3381880000E-01, 0.9668430000E-01 /
  data (epsData(351,i,50), i=1,3) / 0.5196770000E+03, 0.3411440000E-01, 0.1020950000E+00 /
  data (epsData(352,i,50), i=1,3) / 0.5200580000E+03, 0.3075890000E-01, 0.1039300000E+00 /
  data (epsData(353,i,50), i=1,3) / 0.5206340000E+03, 0.3321180000E-01, 0.1066010000E+00 /
  data (epsData(354,i,50), i=1,3) / 0.5232040000E+03, 0.2862780000E-01, 0.1179080000E+00 /
  data (epsData(355,i,50), i=1,3) / 0.5240170000E+03, 0.2509280000E-01, 0.1212240000E+00 /
  data (epsData(356,i,50), i=1,3) / 0.5260990000E+03, 0.2155410000E-01, 0.1289980000E+00 /
  data (epsData(357,i,50), i=1,3) / 0.5280310000E+03, 0.1702320000E-01, 0.1354120000E+00 /
  data (epsData(358,i,50), i=1,3) / 0.5295710000E+03, 0.1172890000E-01, 0.1398760000E+00 /
  data (epsData(359,i,50), i=1,3) / 0.5306920000E+03, 0.1109380000E-01, 0.1425960000E+00 /
  data (epsData(360,i,50), i=1,3) / 0.5325200000E+03, 0.1885870000E-02, 0.1464480000E+00 /
  data (epsData(361,i,50), i=1,3) / 0.5343930000E+03, -0.1396140000E-02, 0.1497480000E+00 /
  data (epsData(362,i,50), i=1,3) / 0.5358250000E+03, -0.6645620000E-02, 0.1515580000E+00 /
  data (epsData(363,i,50), i=1,3) / 0.5370380000E+03, -0.1015270000E-01, 0.1529180000E+00 /
  data (epsData(364,i,50), i=1,3) / 0.5377740000E+03, -0.1084540000E-01, 0.1536060000E+00 /
  data (epsData(365,i,50), i=1,3) / 0.5392630000E+03, -0.1709100000E-01, 0.1548390000E+00 /
  data (epsData(366,i,50), i=1,3) / 0.5441430000E+03, -0.3130320000E-01, 0.1563390000E+00 /
  data (epsData(367,i,50), i=1,3) / 0.5454680000E+03, -0.3252790000E-01, 0.1563700000E+00 /
  data (epsData(368,i,50), i=1,3) / 0.5470800000E+03, -0.3603810000E-01, 0.1560070000E+00 /
  data (epsData(369,i,50), i=1,3) / 0.5492680000E+03, -0.4285570000E-01, 0.1551240000E+00 /
  data (epsData(370,i,50), i=1,3) / 0.5514990000E+03, -0.4675230000E-01, 0.1539660000E+00 /
  data (epsData(371,i,50), i=1,3) / 0.5532010000E+03, -0.4943230000E-01, 0.1529280000E+00 /
  data (epsData(372,i,50), i=1,3) / 0.5584540000E+03, -0.5925980000E-01, 0.1487160000E+00 /
  data (epsData(373,i,50), i=1,3) / 0.5630010000E+03, -0.6424800000E-01, 0.1446770000E+00 /
  data (epsData(374,i,50), i=1,3) / 0.5661160000E+03, -0.6785360000E-01, 0.1418760000E+00 /
  data (epsData(375,i,50), i=1,3) / 0.5686580000E+03, -0.6918480000E-01, 0.1396620000E+00 /
  data (epsData(376,i,50), i=1,3) / 0.5722240000E+03, -0.7307930000E-01, 0.1366050000E+00 /
  data (epsData(377,i,50), i=1,3) / 0.5758720000E+03, -0.7562420000E-01, 0.1337000000E+00 /
  data (epsData(378,i,50), i=1,3) / 0.5785770000E+03, -0.7599210000E-01, 0.1316290000E+00 /
  data (epsData(379,i,50), i=1,3) / 0.5851750000E+03, -0.7976410000E-01, 0.1268340000E+00 /
  data (epsData(380,i,50), i=1,3) / 0.5905580000E+03, -0.8239520000E-01, 0.1230970000E+00 /
  data (epsData(381,i,50), i=1,3) / 0.5931230000E+03, -0.8287610000E-01, 0.1213560000E+00 /
  data (epsData(382,i,50), i=1,3) / 0.5972200000E+03, -0.8284730000E-01, 0.1186160000E+00 /
  data (epsData(383,i,50), i=1,3) / 0.5979740000E+03, -0.8472560000E-01, 0.1181120000E+00 /
  data (epsData(384,i,50), i=1,3) / 0.5987300000E+03, -0.8509920000E-01, 0.1176100000E+00 /
  data (epsData(385,i,50), i=1,3) / 0.6006330000E+03, -0.8488570000E-01, 0.1163560000E+00 /
  data (epsData(386,i,50), i=1,3) / 0.6013990000E+03, -0.8361390000E-01, 0.1158550000E+00 /
  data (epsData(387,i,50), i=1,3) / 0.6044910000E+03, -0.8707510000E-01, 0.1138580000E+00 /
  data (epsData(388,i,50), i=1,3) / 0.6060530000E+03, -0.8522170000E-01, 0.1128660000E+00 /
  data (epsData(389,i,50), i=1,3) / 0.6080200000E+03, -0.8643570000E-01, 0.1116290000E+00 /
  data (epsData(390,i,50), i=1,3) / 0.6108040000E+03, -0.8728990000E-01, 0.1099040000E+00 /
  data (epsData(391,i,50), i=1,3) / 0.6136210000E+03, -0.8810520000E-01, 0.1081990000E+00 /
  data (epsData(392,i,50), i=1,3) / 0.6148380000E+03, -0.8795290000E-01, 0.1074900000E+00 /
  data (epsData(393,i,50), i=1,3) / 0.6164710000E+03, -0.8747930000E-01, 0.1065460000E+00 /
  data (epsData(394,i,50), i=1,3) / 0.6181140000E+03, -0.8694100000E-01, 0.1056060000E+00 /
  data (epsData(395,i,50), i=1,3) / 0.6197690000E+03, -0.8649130000E-01, 0.1046680000E+00 /
  data (epsData(396,i,50), i=1,3) / 0.6226900000E+03, -0.8746240000E-01, 0.1030620000E+00 /
  data (epsData(397,i,50), i=1,3) / 0.6243750000E+03, -0.8605670000E-01, 0.1021910000E+00 /
  data (epsData(398,i,50), i=1,3) / 0.6260700000E+03, -0.8742600000E-01, 0.1013220000E+00 /
  data (epsData(399,i,50), i=1,3) / 0.6307880000E+03, -0.8682370000E-01, 0.9895270000E-01 /
  data (epsData(400,i,50), i=1,3) / 0.6320900000E+03, -0.8653450000E-01, 0.9832810000E-01 /
  data (epsData(401,i,50), i=1,3) / 0.6333970000E+03, -0.8893240000E-01, 0.9770680000E-01 /
  data (epsData(402,i,50), i=1,3) / 0.6347100000E+03, -0.8532110000E-01, 0.9708710000E-01 /
  data (epsData(403,i,50), i=1,3) / 0.6377980000E+03, -0.8724520000E-01, 0.9564620000E-01 /
  data (epsData(404,i,50), i=1,3) / 0.6386870000E+03, -0.8501260000E-01, 0.9523600000E-01 /
  data (epsData(405,i,50), i=1,3) / 0.6400250000E+03, -0.8668090000E-01, 0.9462190000E-01 /
  data (epsData(406,i,50), i=1,3) / 0.6454370000E+03, -0.8619530000E-01, 0.9218040000E-01 /
  data (epsData(407,i,50), i=1,3) / 0.6477210000E+03, -0.8848880000E-01, 0.9117600000E-01 /
  data (epsData(408,i,50), i=1,3) / 0.6495610000E+03, -0.8516100000E-01, 0.9037640000E-01 /
  data (epsData(409,i,50), i=1,3) / 0.6523410000E+03, -0.8655570000E-01, 0.8918210000E-01 /
  data (epsData(410,i,50), i=1,3) / 0.6584470000E+03, -0.8619670000E-01, 0.8662240000E-01 /
  data (epsData(411,i,50), i=1,3) / 0.6603490000E+03, -0.8695050000E-01, 0.8584160000E-01 /
  data (epsData(412,i,50), i=1,3) / 0.6613050000E+03, -0.8607880000E-01, 0.8545230000E-01 /
  data (epsData(413,i,50), i=1,3) / 0.6622620000E+03, -0.8399670000E-01, 0.8506360000E-01 /
  data (epsData(414,i,50), i=1,3) / 0.6632230000E+03, -0.8605200000E-01, 0.8467560000E-01 /
  data (epsData(415,i,50), i=1,3) / 0.6641860000E+03, -0.8530480000E-01, 0.8428820000E-01 /
  data (epsData(416,i,50), i=1,3) / 0.6656360000E+03, -0.8498620000E-01, 0.8370830000E-01 /
  data (epsData(417,i,50), i=1,3) / 0.6666070000E+03, -0.8862900000E-01, 0.8332270000E-01 /
  data (epsData(418,i,50), i=1,3) / 0.6680670000E+03, -0.8449820000E-01, 0.8274680000E-01 /
  data (epsData(419,i,50), i=1,3) / 0.6734740000E+03, -0.8412620000E-01, 0.8068560000E-01 /
  data (epsData(420,i,50), i=1,3) / 0.6749620000E+03, -0.8277630000E-01, 0.8013490000E-01 /
  data (epsData(421,i,50), i=1,3) / 0.6759580000E+03, -0.8440120000E-01, 0.7976860000E-01 /
  data (epsData(422,i,50), i=1,3) / 0.6764570000E+03, -0.8159340000E-01, 0.7958570000E-01 /
  data (epsData(423,i,50), i=1,3) / 0.6784600000E+03, -0.8422240000E-01, 0.7885630000E-01 /
  data (epsData(424,i,50), i=1,3) / 0.7114400000E+03, -0.7584180000E-01, 0.6894540000E-01 /
  data (epsData(425,i,50), i=1,3) / 0.7199150000E+03, -0.7320530000E-01, 0.7073190000E-01 /
  data (epsData(426,i,50), i=1,3) / 0.7360290000E+03, -0.7480610000E-01, 0.6880420000E-01 /
  data (epsData(427,i,50), i=1,3) / 0.7653170000E+03, -0.7309530000E-01, 0.6440700000E-01 /
  data (epsData(428,i,50), i=1,3) / 0.8415360000E+03, -0.6937970000E-01, 0.4978360000E-01 /
  data (epsData(429,i,50), i=1,3) / 0.8908080000E+03, -0.6419680000E-01, 0.4342830000E-01 /
  data (epsData(430,i,50), i=1,3) / 0.9967700000E+03, -0.5674940000E-01, 0.3123590000E-01 /
  data (epsData(431,i,50), i=1,3) / 0.1124140000E+04, -0.4736380000E-01, 0.2142470000E-01 /
  data (epsData(432,i,50), i=1,3) / 0.1264730000E+04, -0.3880230000E-01, 0.1472610000E-01 /
  data (epsData(433,i,50), i=1,3) / 0.1409410000E+04, -0.3197580000E-01, 0.1034250000E-01 /
  data (epsData(434,i,50), i=1,3) / 0.1561920000E+04, -0.2639620000E-01, 0.7342650000E-02 /
  data (epsData(435,i,50), i=1,3) / 0.1739240000E+04, -0.2144460000E-01, 0.5107030000E-02 /
  data (epsData(436,i,50), i=1,3) / 0.1934770000E+04, -0.1738770000E-01, 0.3559180000E-02 /
  data (epsData(437,i,50), i=1,3) / 0.2146500000E+04, -0.1410750000E-01, 0.2483040000E-02 /
  data (epsData(438,i,50), i=1,3) / 0.2384780000E+04, -0.1137850000E-01, 0.1730880000E-02 /
  data (epsData(439,i,50), i=1,3) / 0.2649860000E+04, -0.9142920000E-02, 0.1197640000E-02 /
  data (epsData(440,i,50), i=1,3) / 0.2944860000E+04, -0.7305420000E-02, 0.8259600000E-03 /
  data (epsData(441,i,50), i=1,3) / 0.3278850000E+04, -0.5760560000E-02, 0.5643850000E-03 /
  data (epsData(442,i,50), i=1,3) / 0.3753310000E+04, -0.4073150000E-02, 0.3485750000E-03 /
  data (epsData(443,i,50), i=1,3) / 0.3885180000E+04, -0.3505950000E-02, 0.3079860000E-03 /
  data (epsData(444,i,50), i=1,3) / 0.3915570000E+04, -0.3236870000E-02, 0.3120830000E-03 /
  data (epsData(445,i,50), i=1,3) / 0.3925730000E+04, -0.3039760000E-02, 0.3320820000E-03 /
  data (epsData(446,i,50), i=1,3) / 0.3930110000E+04, -0.2867950000E-02, 0.4286550000E-03 /
  data (epsData(447,i,50), i=1,3) / 0.3934310000E+04, -0.2836150000E-02, 0.6806870000E-03 /
  data (epsData(448,i,50), i=1,3) / 0.3939660000E+04, -0.2991910000E-02, 0.8231530000E-03 /
  data (epsData(449,i,50), i=1,3) / 0.3960850000E+04, -0.3201860000E-02, 0.9116300000E-03 /
  data (epsData(450,i,50), i=1,3) / 0.4012010000E+04, -0.3347170000E-02, 0.8947830000E-03 /
  data (epsData(451,i,50), i=1,3) / 0.4120840000E+04, -0.3237660000E-02, 0.8059410000E-03 /
  data (epsData(452,i,50), i=1,3) / 0.4153280000E+04, -0.3061360000E-02, 0.8040800000E-03 /
  data (epsData(453,i,50), i=1,3) / 0.4159820000E+04, -0.2977160000E-02, 0.9092030000E-03 /
  data (epsData(454,i,50), i=1,3) / 0.4170430000E+04, -0.3076310000E-02, 0.1013810000E-02 /
  data (epsData(455,i,50), i=1,3) / 0.4213180000E+04, -0.3190540000E-02, 0.1021390000E-02 /
  data (epsData(456,i,50), i=1,3) / 0.4390890000E+04, -0.3124640000E-02, 0.8833030000E-03 /
  data (epsData(457,i,50), i=1,3) / 0.4459120000E+04, -0.2982980000E-02, 0.8527840000E-03 /
  data (epsData(458,i,50), i=1,3) / 0.4473380000E+04, -0.2977880000E-02, 0.9337740000E-03 /
  data (epsData(459,i,50), i=1,3) / 0.4566770000E+04, -0.3004240000E-02, 0.8829150000E-03 /
  data (epsData(460,i,50), i=1,3) / 0.5137430000E+04, -0.2565890000E-02, 0.5840380000E-03 /
  data (epsData(461,i,50), i=1,3) / 0.5757110000E+04, -0.2107420000E-02, 0.3918910000E-03 /
  data (epsData(462,i,50), i=1,3) / 0.6450510000E+04, -0.1707910000E-02, 0.2612490000E-03 /
  data (epsData(463,i,50), i=1,3) / 0.7195050000E+04, -0.1385100000E-02, 0.1761630000E-03 /
  data (epsData(464,i,50), i=1,3) / 0.8024790000E+04, -0.1118620000E-02, 0.1179740000E-03 /
  data (epsData(465,i,50), i=1,3) / 0.8954410000E+04, -0.8997480000E-03, 0.7895900000E-04 /
  data (epsData(466,i,50), i=1,3) / 0.9980790000E+04, -0.7239560000E-03, 0.5271210000E-04 /
  data (epsData(467,i,50), i=1,3) / 0.1112780000E+05, -0.5815650000E-03, 0.3509020000E-04 /
  data (epsData(468,i,50), i=1,3) / 0.1241260000E+05, -0.4663040000E-03, 0.2328720000E-04 /
  data (epsData(469,i,50), i=1,3) / 0.1385350000E+05, -0.3732890000E-03, 0.1539630000E-04 /
  data (epsData(470,i,50), i=1,3) / 0.1546840000E+05, -0.2984490000E-03, 0.1012150000E-04 /
  data (epsData(471,i,50), i=1,3) / 0.1727810000E+05, -0.2383480000E-03, 0.6640220000E-05 /
  data (epsData(472,i,50), i=1,3) / 0.1931110000E+05, -0.1900300000E-03, 0.4338040000E-05 /
  data (epsData(473,i,50), i=1,3) / 0.2159070000E+05, -0.1512840000E-03, 0.2824120000E-05 /
  data (epsData(474,i,50), i=1,3) / 0.2416220000E+05, -0.1199780000E-03, 0.1828630000E-05 /
  data (epsData(475,i,50), i=1,3) / 0.2434910000E+05, -0.1180740000E-03, 0.1775370000E-05 /
  data (epsData(476,i,50), i=1,3) / 0.2435000000E+05, -0.1145030000E-03, 0.1775120000E-05 /
  data (epsData(477,i,50), i=1,3) / 0.2435000000E+05, -0.1180640000E-03, 0.1775100000E-05 /
  data (epsData(478,i,50), i=1,3) / 0.2743180000E+05, -0.9155180000E-04, 0.1117670000E-05 /
  data (epsData(479,i,50), i=1,3) / 0.2905190000E+05, -0.7813750000E-04, 0.8936770000E-06 /
  data (epsData(480,i,50), i=1,3) / 0.2917660000E+05, -0.7458370000E-04, 0.8790120000E-06 /
  data (epsData(481,i,50), i=1,3) / 0.2919970000E+05, -0.7229860000E-04, 0.2985670000E-05 /
  data (epsData(482,i,50), i=1,3) / 0.2920000000E+05, -0.6802000000E-04, 0.3057550000E-05 /
  data (epsData(483,i,50), i=1,3) / 0.2920000000E+05, -0.7228400000E-04, 0.3070860000E-05 /
  data (epsData(484,i,50), i=1,3) / 0.2921940000E+05, -0.7414040000E-04, 0.5176520000E-05 /
  data (epsData(485,i,50), i=1,3) / 0.2934020000E+05, -0.7652790000E-04, 0.5481810000E-05 /
  data (epsData(486,i,50), i=1,3) / 0.3036540000E+05, -0.7460440000E-04, 0.4888220000E-05 /
  data (epsData(487,i,50), i=1,3) / 0.3534060000E+05, -0.5692130000E-04, 0.2832470000E-05 /
  data (epsData(488,i,50), i=1,3) / 0.3975160000E+05, -0.4534850000E-04, 0.1827750000E-05 /
  data (epsData(489,i,50), i=1,3) / 0.4458130000E+05, -0.3619260000E-04, 0.1186070000E-05 /
  data (epsData(490,i,50), i=1,3) / 0.4995180000E+05, -0.2888140000E-04, 0.7685260000E-06 /
  data (epsData(491,i,50), i=1,3) / 0.5594960000E+05, -0.2303780000E-04, 0.4962410000E-06 /
  data (epsData(492,i,50), i=1,3) / 0.6264710000E+05, -0.1837700000E-04, 0.3205540000E-06 /
  data (epsData(493,i,50), i=1,3) / 0.7015170000E+05, -0.1465350000E-04, 0.2047530000E-06 /
  data (epsData(494,i,50), i=1,3) / 0.7858730000E+05, -0.1167330000E-04, 0.1304870000E-06 /
  data (epsData(495,i,50), i=1,3) / 0.8802450000E+05, -0.9304680000E-05, 0.8267450000E-07 /
  data (epsData(496,i,50), i=1,3) / 0.9827110000E+05, -0.7484990000E-05, 0.5287600000E-07 /
  data (epsData(497,i,50), i=1,3) / 0.1000000000E+06, -0.7328460000E-05, 0.4921690000E-07 /
  data (epsData(1,i,51), i=1,3) / 0.2506580000E-02, -0.1919650000E+05, 0.4537770000E+05 /
  data (epsData(2,i,51), i=1,3) / 0.4693440000E-02, -0.1531830000E+05, 0.1493210000E+05 /
  data (epsData(3,i,51), i=1,3) / 0.7560590000E-02, -0.6970150000E+04, 0.6759470000E+04 /
  data (epsData(4,i,51), i=1,3) / 0.1110800000E-01, -0.3349770000E+04, 0.3707640000E+04 /
  data (epsData(5,i,51), i=1,3) / 0.1533570000E-01, -0.1651140000E+04, 0.2305430000E+04 /
  data (epsData(6,i,51), i=1,3) / 0.2024370000E-01, -0.7734740000E+03, 0.1563070000E+04 /
  data (epsData(7,i,51), i=1,3) / 0.2583200000E-01, -0.2825410000E+03, 0.1127760000E+04 /
  data (epsData(8,i,51), i=1,3) / 0.3210060000E-01, 0.1009770000E+02, 0.8522240000E+03 /
  data (epsData(9,i,51), i=1,3) / 0.3904940000E-01, 0.1937870000E+03, 0.6672280000E+03 /
  data (epsData(10,i,51), i=1,3) / 0.4667860000E-01, 0.3141550000E+03, 0.5371160000E+03 /
  data (epsData(11,i,51), i=1,3) / 0.5498800000E-01, 0.3959950000E+03, 0.4421230000E+03 /
  data (epsData(12,i,51), i=1,3) / 0.6397770000E-01, 0.4535160000E+03, 0.3706180000E+03 /
  data (epsData(13,i,51), i=1,3) / 0.7364770000E-01, 0.4952900000E+03, 0.3154110000E+03 /
  data (epsData(14,i,51), i=1,3) / 0.8399800000E-01, 0.5268760000E+03, 0.2718670000E+03 /
  data (epsData(15,i,51), i=1,3) / 0.9502860000E-01, 0.5526110000E+03, 0.2368940000E+03 /
  data (epsData(16,i,51), i=1,3) / 0.1191310000E+00, 0.5859170000E+03, 0.1966460000E+03 /
  data (epsData(17,i,51), i=1,3) / 0.1603860000E+00, 0.6021570000E+03, 0.1547620000E+03 /
  data (epsData(18,i,51), i=1,3) / 0.2612640000E+00, 0.6112480000E+03, 0.1009600000E+03 /
  data (epsData(19,i,51), i=1,3) / 0.4338630000E+00, 0.6239090000E+03, 0.6573960000E+02 /
  data (epsData(20,i,51), i=1,3) / 0.7422780000E+00, 0.6468970000E+03, 0.4907370000E+02 /
  data (epsData(21,i,51), i=1,3) / 0.1055840000E+01, 0.6898770000E+03, 0.4757660000E+02 /
  data (epsData(22,i,51), i=1,3) / 0.1295500000E+01, 0.7445790000E+03, 0.5085390000E+02 /
  data (epsData(23,i,51), i=1,3) / 0.1513920000E+01, 0.8226010000E+03, 0.6983230000E+02 /
  data (epsData(24,i,51), i=1,3) / 0.1606050000E+01, 0.8835320000E+03, 0.7853390000E+02 /
  data (epsData(25,i,51), i=1,3) / 0.1700900000E+01, 0.9402630000E+03, 0.1388570000E+03 /
  data (epsData(26,i,51), i=1,3) / 0.1848280000E+01, 0.1022020000E+04, 0.2081770000E+03 /
  data (epsData(27,i,51), i=1,3) / 0.1898770000E+01, 0.1088270000E+04, 0.2266560000E+03 /
  data (epsData(28,i,51), i=1,3) / 0.1949930000E+01, 0.1143340000E+04, 0.3484860000E+03 /
  data (epsData(29,i,51), i=1,3) / 0.2027550000E+01, 0.1136580000E+04, 0.5108600000E+03 /
  data (epsData(30,i,51), i=1,3) / 0.2150000000E+01, 0.1030320000E+04, 0.7124790000E+03 /
  data (epsData(31,i,51), i=1,3) / 0.2212590000E+01, 0.9279220000E+03, 0.7842070000E+03 /
  data (epsData(32,i,51), i=1,3) / 0.2370410000E+01, 0.7532720000E+03, 0.8136720000E+03 /
  data (epsData(33,i,51), i=1,3) / 0.2600010000E+01, 0.5810370000E+03, 0.8253130000E+03 /
  data (epsData(34,i,51), i=1,3) / 0.3028930000E+01, 0.3758920000E+03, 0.8031420000E+03 /
  data (epsData(35,i,51), i=1,3) / 0.3666700000E+01, 0.1595860000E+03, 0.7540480000E+03 /
  data (epsData(36,i,51), i=1,3) / 0.4176910000E+01, 0.2127730000E+02, 0.6881210000E+03 /
  data (epsData(37,i,51), i=1,3) / 0.4634060000E+01, -0.7355600000E+02, 0.6130010000E+03 /
  data (epsData(38,i,51), i=1,3) / 0.5266720000E+01, -0.1573070000E+03, 0.4998380000E+03 /
  data (epsData(39,i,51), i=1,3) / 0.5894290000E+01, -0.1985080000E+03, 0.3887190000E+03 /
  data (epsData(40,i,51), i=1,3) / 0.6266740000E+01, -0.1963960000E+03, 0.3318550000E+03 /
  data (epsData(41,i,51), i=1,3) / 0.6736140000E+01, -0.1902520000E+03, 0.2865040000E+03 /
  data (epsData(42,i,51), i=1,3) / 0.7884710000E+01, -0.1870230000E+03, 0.1950150000E+03 /
  data (epsData(43,i,51), i=1,3) / 0.8812620000E+01, -0.1708810000E+03, 0.1378690000E+03 /
  data (epsData(44,i,51), i=1,3) / 0.9878960000E+01, -0.1470850000E+03, 0.9254150000E+02 /
  data (epsData(45,i,51), i=1,3) / 0.1096740000E+02, -0.1234910000E+03, 0.6251700000E+02 /
  data (epsData(46,i,51), i=1,3) / 0.1206270000E+02, -0.1029670000E+03, 0.4300460000E+02 /
  data (epsData(47,i,51), i=1,3) / 0.1319910000E+02, -0.8548060000E+02, 0.2999150000E+02 /
  data (epsData(48,i,51), i=1,3) / 0.1445050000E+02, -0.7013220000E+02, 0.2071640000E+02 /
  data (epsData(49,i,51), i=1,3) / 0.1581680000E+02, -0.5706170000E+02, 0.1431380000E+02 /
  data (epsData(50,i,51), i=1,3) / 0.1729810000E+02, -0.4617820000E+02, 0.9935400000E+01 /
  data (epsData(51,i,51), i=1,3) / 0.1889440000E+02, -0.3722540000E+02, 0.6966510000E+01 /
  data (epsData(52,i,51), i=1,3) / 0.2060570000E+02, -0.2990320000E+02, 0.4945310000E+01 /
  data (epsData(53,i,51), i=1,3) / 0.2243190000E+02, -0.2390010000E+02, 0.3546520000E+01 /
  data (epsData(54,i,51), i=1,3) / 0.2452720000E+02, -0.1857710000E+02, 0.2498440000E+01 /
  data (epsData(55,i,51), i=1,3) / 0.2675580000E+02, -0.1414400000E+02, 0.1791760000E+01 /
  data (epsData(56,i,51), i=1,3) / 0.2929150000E+02, -0.1007880000E+02, 0.1292610000E+01 /
  data (epsData(57,i,51), i=1,3) / 0.3143040000E+02, -0.6907230000E+01, 0.1050090000E+01 /
  data (epsData(58,i,51), i=1,3) / 0.3179630000E+02, -0.6237620000E+01, 0.1092330000E+01 /
  data (epsData(59,i,51), i=1,3) / 0.3198040000E+02, -0.5850310000E+01, 0.1173700000E+01 /
  data (epsData(60,i,51), i=1,3) / 0.3225000000E+02, -0.5477770000E+01, 0.1610740000E+01 /
  data (epsData(61,i,51), i=1,3) / 0.3249320000E+02, -0.5399160000E+01, 0.1753940000E+01 /
  data (epsData(62,i,51), i=1,3) / 0.3312890000E+02, -0.4811920000E+01, 0.1883080000E+01 /
  data (epsData(63,i,51), i=1,3) / 0.3339830000E+02, -0.4526200000E+01, 0.2097940000E+01 /
  data (epsData(64,i,51), i=1,3) / 0.3384420000E+02, -0.4352540000E+01, 0.2258980000E+01 /
  data (epsData(65,i,51), i=1,3) / 0.3575750000E+02, -0.2977520000E+01, 0.2393760000E+01 /
  data (epsData(66,i,51), i=1,3) / 0.3726740000E+02, -0.1802460000E+01, 0.2556520000E+01 /
  data (epsData(67,i,51), i=1,3) / 0.3896030000E+02, -0.4690760000E+00, 0.2894260000E+01 /
  data (epsData(68,i,51), i=1,3) / 0.4045150000E+02, 0.6858800000E+00, 0.3387660000E+01 /
  data (epsData(69,i,51), i=1,3) / 0.4189910000E+02, 0.1724530000E+01, 0.4109350000E+01 /
  data (epsData(70,i,51), i=1,3) / 0.4348010000E+02, 0.2631290000E+01, 0.5169160000E+01 /
  data (epsData(71,i,51), i=1,3) / 0.4532200000E+02, 0.3257110000E+01, 0.6636650000E+01 /
  data (epsData(72,i,51), i=1,3) / 0.4745530000E+02, 0.3302550000E+01, 0.8417580000E+01 /
  data (epsData(73,i,51), i=1,3) / 0.4961350000E+02, 0.2681620000E+01, 0.9939990000E+01 /
  data (epsData(74,i,51), i=1,3) / 0.5256700000E+02, 0.1339080000E+01, 0.1136740000E+02 /
  data (epsData(75,i,51), i=1,3) / 0.5576600000E+02, -0.4621310000E+00, 0.1227210000E+02 /
  data (epsData(76,i,51), i=1,3) / 0.5902320000E+02, -0.2580450000E+01, 0.1255200000E+02 /
  data (epsData(77,i,51), i=1,3) / 0.6210360000E+02, -0.4688210000E+01, 0.1197860000E+02 /
  data (epsData(78,i,51), i=1,3) / 0.6535800000E+02, -0.6447520000E+01, 0.1056340000E+02 /
  data (epsData(79,i,51), i=1,3) / 0.6922750000E+02, -0.7566880000E+01, 0.8387620000E+01 /
  data (epsData(80,i,51), i=1,3) / 0.7331740000E+02, -0.7752240000E+01, 0.6265560000E+01 /
  data (epsData(81,i,51), i=1,3) / 0.7787360000E+02, -0.7351030000E+01, 0.4432500000E+01 /
  data (epsData(82,i,51), i=1,3) / 0.8293490000E+02, -0.6606020000E+01, 0.2960580000E+01 /
  data (epsData(83,i,51), i=1,3) / 0.8881660000E+02, -0.5650040000E+01, 0.1847760000E+01 /
  data (epsData(84,i,51), i=1,3) / 0.9502780000E+02, -0.4719670000E+01, 0.1160780000E+01 /
  data (epsData(85,i,51), i=1,3) / 0.1019840000E+03, -0.3917880000E+01, 0.7459000000E+00 /
  data (epsData(86,i,51), i=1,3) / 0.1094790000E+03, -0.3244420000E+01, 0.4609900000E+00 /
  data (epsData(87,i,51), i=1,3) / 0.1168590000E+03, -0.2715260000E+01, 0.3001510000E+00 /
  data (epsData(88,i,51), i=1,3) / 0.1250010000E+03, -0.2258830000E+01, 0.2022860000E+00 /
  data (epsData(89,i,51), i=1,3) / 0.1342300000E+03, -0.1864530000E+01, 0.1493590000E+00 /
  data (epsData(90,i,51), i=1,3) / 0.1460180000E+03, -0.1492500000E+01, 0.1189470000E+00 /
  data (epsData(91,i,51), i=1,3) / 0.1558940000E+03, -0.1257600000E+01, 0.1211440000E+00 /
  data (epsData(92,i,51), i=1,3) / 0.1711940000E+03, -0.1011380000E+01, 0.1229370000E+00 /
  data (epsData(93,i,51), i=1,3) / 0.1902410000E+03, -0.7982390000E+00, 0.1197530000E+00 /
  data (epsData(94,i,51), i=1,3) / 0.2126310000E+03, -0.6292810000E+00, 0.1135660000E+00 /
  data (epsData(95,i,51), i=1,3) / 0.2392600000E+03, -0.4938500000E+00, 0.1028300000E+00 /
  data (epsData(96,i,51), i=1,3) / 0.2716050000E+03, -0.3821500000E+00, 0.8863020000E-01 /
  data (epsData(97,i,51), i=1,3) / 0.3081800000E+03, -0.2954130000E+00, 0.7324400000E-01 /
  data (epsData(98,i,51), i=1,3) / 0.3486380000E+03, -0.2267700000E+00, 0.5872030000E-01 /
  data (epsData(99,i,51), i=1,3) / 0.3943540000E+03, -0.1693410000E+00, 0.4594990000E-01 /
  data (epsData(100,i,51), i=1,3) / 0.4492980000E+03, -0.1151690000E+00, 0.3460410000E-01 /
  data (epsData(101,i,51), i=1,3) / 0.4981300000E+03, -0.6914220000E-01, 0.2716940000E-01 /
  data (epsData(102,i,51), i=1,3) / 0.5168750000E+03, -0.4679190000E-01, 0.2507130000E-01 /
  data (epsData(103,i,51), i=1,3) / 0.5260780000E+03, -0.3215870000E-01, 0.2428320000E-01 /
  data (epsData(104,i,51), i=1,3) / 0.5287720000E+03, -0.2660160000E-01, 0.2509500000E-01 /
  data (epsData(105,i,51), i=1,3) / 0.5334300000E+03, -0.1726230000E-01, 0.2566710000E-01 /
  data (epsData(106,i,51), i=1,3) / 0.5364780000E+03, -0.9189860000E-02, 0.2654110000E-01 /
  data (epsData(107,i,51), i=1,3) / 0.5383850000E+03, -0.3445820000E-02, 0.2835760000E-01 /
  data (epsData(108,i,51), i=1,3) / 0.5409700000E+03, 0.4454450000E-02, 0.3146390000E-01 /
  data (epsData(109,i,51), i=1,3) / 0.5444560000E+03, 0.1565150000E-01, 0.3822600000E-01 /
  data (epsData(110,i,51), i=1,3) / 0.5482770000E+03, 0.2772180000E-01, 0.4934410000E-01 /
  data (epsData(111,i,51), i=1,3) / 0.5516550000E+03, 0.3682820000E-01, 0.6342130000E-01 /
  data (epsData(112,i,51), i=1,3) / 0.5550270000E+03, 0.4185680000E-01, 0.8120820000E-01 /
  data (epsData(113,i,51), i=1,3) / 0.5592720000E+03, 0.3998730000E-01, 0.1045090000E+00 /
  data (epsData(114,i,51), i=1,3) / 0.5635990000E+03, 0.3047520000E-01, 0.1250900000E+00 /
  data (epsData(115,i,51), i=1,3) / 0.5685220000E+03, 0.1390970000E-01, 0.1417650000E+00 /
  data (epsData(116,i,51), i=1,3) / 0.5740940000E+03, -0.7170370000E-02, 0.1508840000E+00 /
  data (epsData(117,i,51), i=1,3) / 0.5808510000E+03, -0.2849680000E-01, 0.1520740000E+00 /
  data (epsData(118,i,51), i=1,3) / 0.5900350000E+03, -0.4968750000E-01, 0.1464070000E+00 /
  data (epsData(119,i,51), i=1,3) / 0.6042530000E+03, -0.6795030000E-01, 0.1323740000E+00 /
  data (epsData(120,i,51), i=1,3) / 0.6235600000E+03, -0.7775470000E-01, 0.1164010000E+00 /
  data (epsData(121,i,51), i=1,3) / 0.6567090000E+03, -0.8324330000E-01, 0.9663340000E-01 /
  data (epsData(122,i,51), i=1,3) / 0.7080040000E+03, -0.8097690000E-01, 0.7580970000E-01 /
  data (epsData(123,i,51), i=1,3) / 0.7535660000E+03, -0.7372160000E-01, 0.6248600000E-01 /
  data (epsData(124,i,51), i=1,3) / 0.7674290000E+03, -0.6874780000E-01, 0.6103150000E-01 /
  data (epsData(125,i,51), i=1,3) / 0.7762300000E+03, -0.6938240000E-01, 0.6272480000E-01 /
  data (epsData(126,i,51), i=1,3) / 0.8131410000E+03, -0.6773040000E-01, 0.5654640000E-01 /
  data (epsData(127,i,51), i=1,3) / 0.8499660000E+03, -0.6731710000E-01, 0.5175130000E-01 /
  data (epsData(128,i,51), i=1,3) / 0.9315070000E+03, -0.6098740000E-01, 0.3973330000E-01 /
  data (epsData(129,i,51), i=1,3) / 0.9784770000E+03, -0.5791710000E-01, 0.3551680000E-01 /
  data (epsData(130,i,51), i=1,3) / 0.1124790000E+04, -0.4793640000E-01, 0.2318260000E-01 /
  data (epsData(131,i,51), i=1,3) / 0.1257530000E+04, -0.4003950000E-01, 0.1625540000E-01 /
  data (epsData(132,i,51), i=1,3) / 0.1408460000E+04, -0.3281300000E-01, 0.1124600000E-01 /
  data (epsData(133,i,51), i=1,3) / 0.1569610000E+04, -0.2689380000E-01, 0.7873030000E-02 /
  data (epsData(134,i,51), i=1,3) / 0.1740190000E+04, -0.2210790000E-01, 0.5562190000E-02 /
  data (epsData(135,i,51), i=1,3) / 0.1936000000E+04, -0.1794080000E-01, 0.3872240000E-02 /
  data (epsData(136,i,51), i=1,3) / 0.2152080000E+04, -0.1453130000E-01, 0.2688710000E-02 /
  data (epsData(137,i,51), i=1,3) / 0.2386680000E+04, -0.1178130000E-01, 0.1886720000E-02 /
  data (epsData(138,i,51), i=1,3) / 0.2652760000E+04, -0.9479840000E-02, 0.1305910000E-02 /
  data (epsData(139,i,51), i=1,3) / 0.2946330000E+04, -0.7606950000E-02, 0.9030540000E-03 /
  data (epsData(140,i,51), i=1,3) / 0.3275620000E+04, -0.6053060000E-02, 0.6206870000E-03 /
  data (epsData(141,i,51), i=1,3) / 0.3658730000E+04, -0.4698410000E-02, 0.4187050000E-03 /
  data (epsData(142,i,51), i=1,3) / 0.4054520000E+04, -0.3457400000E-02, 0.2899010000E-03 /
  data (epsData(143,i,51), i=1,3) / 0.4110300000E+04, -0.3134820000E-02, 0.2760140000E-03 /
  data (epsData(144,i,51), i=1,3) / 0.4126920000E+04, -0.2917010000E-02, 0.2981340000E-03 /
  data (epsData(145,i,51), i=1,3) / 0.4132350000E+04, -0.2739210000E-02, 0.3692870000E-03 /
  data (epsData(146,i,51), i=1,3) / 0.4135610000E+04, -0.2672250000E-02, 0.5542330000E-03 /
  data (epsData(147,i,51), i=1,3) / 0.4139350000E+04, -0.2768120000E-02, 0.7182010000E-03 /
  data (epsData(148,i,51), i=1,3) / 0.4146290000E+04, -0.2911840000E-02, 0.7659560000E-03 /
  data (epsData(149,i,51), i=1,3) / 0.4174870000E+04, -0.3089460000E-02, 0.8170190000E-03 /
  data (epsData(150,i,51), i=1,3) / 0.4249200000E+04, -0.3161140000E-02, 0.7824480000E-03 /
  data (epsData(151,i,51), i=1,3) / 0.4356280000E+04, -0.3001890000E-02, 0.7126910000E-03 /
  data (epsData(152,i,51), i=1,3) / 0.4378340000E+04, -0.2858690000E-02, 0.7248260000E-03 /
  data (epsData(153,i,51), i=1,3) / 0.4384190000E+04, -0.2800650000E-02, 0.8347560000E-03 /
  data (epsData(154,i,51), i=1,3) / 0.4393740000E+04, -0.2895190000E-02, 0.9076290000E-03 /
  data (epsData(155,i,51), i=1,3) / 0.4441730000E+04, -0.2994340000E-02, 0.9114240000E-03 /
  data (epsData(156,i,51), i=1,3) / 0.4633920000E+04, -0.2909880000E-02, 0.7844390000E-03 /
  data (epsData(157,i,51), i=1,3) / 0.4694140000E+04, -0.2781480000E-02, 0.7653160000E-03 /
  data (epsData(158,i,51), i=1,3) / 0.4706600000E+04, -0.2783740000E-02, 0.8414000000E-03 /
  data (epsData(159,i,51), i=1,3) / 0.4794990000E+04, -0.2812090000E-02, 0.7989120000E-03 /
  data (epsData(160,i,51), i=1,3) / 0.5392610000E+04, -0.2399820000E-02, 0.5287980000E-03 /
  data (epsData(161,i,51), i=1,3) / 0.6081850000E+04, -0.1947030000E-02, 0.3466240000E-03 /
  data (epsData(162,i,51), i=1,3) / 0.6823970000E+04, -0.1571970000E-02, 0.2296340000E-03 /
  data (epsData(163,i,51), i=1,3) / 0.7610240000E+04, -0.1274200000E-02, 0.1549850000E-03 /
  data (epsData(164,i,51), i=1,3) / 0.8478970000E+04, -0.1030870000E-02, 0.1043400000E-03 /
  data (epsData(165,i,51), i=1,3) / 0.9455710000E+04, -0.8298760000E-03, 0.6976150000E-04 /
  data (epsData(166,i,51), i=1,3) / 0.1054170000E+05, -0.6674340000E-03, 0.4652100000E-04 /
  data (epsData(167,i,51), i=1,3) / 0.1175480000E+05, -0.5359640000E-03, 0.3095970000E-04 /
  data (epsData(168,i,51), i=1,3) / 0.1311260000E+05, -0.4297090000E-03, 0.2054670000E-04 /
  data (epsData(169,i,51), i=1,3) / 0.1463750000E+05, -0.3438800000E-03, 0.1356110000E-04 /
  data (epsData(170,i,51), i=1,3) / 0.1634100000E+05, -0.2750390000E-03, 0.8927180000E-05 /
  data (epsData(171,i,51), i=1,3) / 0.1825210000E+05, -0.2196830000E-03, 0.5853660000E-05 /
  data (epsData(172,i,51), i=1,3) / 0.2040130000E+05, -0.1751340000E-03, 0.3818270000E-05 /
  data (epsData(173,i,51), i=1,3) / 0.2281290000E+05, -0.1393860000E-03, 0.2484450000E-05 /
  data (epsData(174,i,51), i=1,3) / 0.2551360000E+05, -0.1106780000E-03, 0.1612250000E-05 /
  data (epsData(175,i,51), i=1,3) / 0.2551400000E+05, -0.1078410000E-03, 0.1612150000E-05 /
  data (epsData(176,i,51), i=1,3) / 0.2551400000E+05, -0.1106740000E-03, 0.1612140000E-05 /
  data (epsData(177,i,51), i=1,3) / 0.2876180000E+05, -0.8568990000E-04, 0.1012480000E-05 /
  data (epsData(178,i,51), i=1,3) / 0.3034830000E+05, -0.7375940000E-04, 0.8213360000E-06 /
  data (epsData(179,i,51), i=1,3) / 0.3046700000E+05, -0.7064620000E-04, 0.8090050000E-06 /
  data (epsData(180,i,51), i=1,3) / 0.3049040000E+05, -0.6862700000E-04, 0.2560200000E-05 /
  data (epsData(181,i,51), i=1,3) / 0.3049100000E+05, -0.6416380000E-04, 0.2693230000E-05 /
  data (epsData(182,i,51), i=1,3) / 0.3049100000E+05, -0.6858990000E-04, 0.2704110000E-05 /
  data (epsData(183,i,51), i=1,3) / 0.3050910000E+05, -0.7005800000E-04, 0.4700740000E-05 /
  data (epsData(184,i,51), i=1,3) / 0.3062130000E+05, -0.7230720000E-04, 0.5010750000E-05 /
  data (epsData(185,i,51), i=1,3) / 0.3160780000E+05, -0.7080330000E-04, 0.4515930000E-05 /
  data (epsData(186,i,51), i=1,3) / 0.3682970000E+05, -0.5391540000E-04, 0.2604850000E-05 /
  data (epsData(187,i,51), i=1,3) / 0.4142490000E+05, -0.4294960000E-04, 0.1680260000E-05 /
  data (epsData(188,i,51), i=1,3) / 0.4645890000E+05, -0.3427330000E-04, 0.1089950000E-05 /
  data (epsData(189,i,51), i=1,3) / 0.5205300000E+05, -0.2735130000E-04, 0.7061940000E-06 /
  data (epsData(190,i,51), i=1,3) / 0.5830080000E+05, -0.2181890000E-04, 0.4558050000E-06 /
  data (epsData(191,i,51), i=1,3) / 0.6529430000E+05, -0.1739860000E-04, 0.2932620000E-06 /
  data (epsData(192,i,51), i=1,3) / 0.7314740000E+05, -0.1386130000E-04, 0.1877340000E-06 /
  data (epsData(193,i,51), i=1,3) / 0.8193260000E+05, -0.1104660000E-04, 0.1193010000E-06 /
  data (epsData(194,i,51), i=1,3) / 0.9176360000E+05, -0.8810640000E-05, 0.7556720000E-07 /
  data (epsData(195,i,51), i=1,3) / 0.9995020000E+05, -0.7502450000E-05, 0.5348200000E-07 /
  data (epsData(196,i,51), i=1,3) / 0.1000000000E+06, -0.7565540000E-05, 0.5337170000E-07 /
  data (epsData(1,i,52), i=1,3) / 0.2506580000E-02, -0.1969210000E+05, 0.4709060000E+05 /
  data (epsData(2,i,52), i=1,3) / 0.4693440000E-02, -0.1577410000E+05, 0.1554880000E+05 /
  data (epsData(3,i,52), i=1,3) / 0.7560590000E-02, -0.7170530000E+04, 0.7061870000E+04 /
  data (epsData(4,i,52), i=1,3) / 0.1110800000E-01, -0.3437190000E+04, 0.3884850000E+04 /
  data (epsData(5,i,52), i=1,3) / 0.1533570000E-01, -0.1685140000E+04, 0.2421610000E+04 /
  data (epsData(6,i,52), i=1,3) / 0.2024370000E-01, -0.7797380000E+03, 0.1645200000E+04 /
  data (epsData(7,i,52), i=1,3) / 0.2583200000E-01, -0.2732450000E+03, 0.1189010000E+04 /
  data (epsData(8,i,52), i=1,3) / 0.3210060000E-01, 0.2868820000E+02, 0.8997410000E+03 /
  data (epsData(9,i,52), i=1,3) / 0.3904940000E-01, 0.2182210000E+03, 0.7052200000E+03 /
  data (epsData(10,i,52), i=1,3) / 0.4667860000E-01, 0.3424210000E+03, 0.5682240000E+03 /
  data (epsData(11,i,52), i=1,3) / 0.5498800000E-01, 0.4268670000E+03, 0.4680870000E+03 /
  data (epsData(12,i,52), i=1,3) / 0.6397770000E-01, 0.4862190000E+03, 0.3926330000E+03 /
  data (epsData(13,i,52), i=1,3) / 0.7364770000E-01, 0.5293200000E+03, 0.3343250000E+03 /
  data (epsData(14,i,52), i=1,3) / 0.8399800000E-01, 0.5619040000E+03, 0.2883010000E+03 /
  data (epsData(15,i,52), i=1,3) / 0.9502860000E-01, 0.5884390000E+03, 0.2513110000E+03 /
  data (epsData(16,i,52), i=1,3) / 0.1191310000E+00, 0.6227860000E+03, 0.2083260000E+03 /
  data (epsData(17,i,52), i=1,3) / 0.1603860000E+00, 0.6395760000E+03, 0.1636510000E+03 /
  data (epsData(18,i,52), i=1,3) / 0.2612640000E+00, 0.6489590000E+03, 0.1065630000E+03 /
  data (epsData(19,i,52), i=1,3) / 0.4338630000E+00, 0.6617890000E+03, 0.6920530000E+02 /
  data (epsData(20,i,52), i=1,3) / 0.7743970000E+00, 0.6880120000E+03, 0.5100720000E+02 /
  data (epsData(21,i,52), i=1,3) / 0.1094080000E+01, 0.7344710000E+03, 0.5007730000E+02 /
  data (epsData(22,i,52), i=1,3) / 0.1337820000E+01, 0.7960420000E+03, 0.5634390000E+02 /
  data (epsData(23,i,52), i=1,3) / 0.1513920000E+01, 0.8591450000E+03, 0.7025240000E+02 /
  data (epsData(24,i,52), i=1,3) / 0.1606050000E+01, 0.9187580000E+03, 0.7857710000E+02 /
  data (epsData(25,i,52), i=1,3) / 0.1700900000E+01, 0.9745450000E+03, 0.1364650000E+03 /
  data (epsData(26,i,52), i=1,3) / 0.1848280000E+01, 0.1055530000E+04, 0.2029530000E+03 /
  data (epsData(27,i,52), i=1,3) / 0.1898770000E+01, 0.1120120000E+04, 0.2206680000E+03 /
  data (epsData(28,i,52), i=1,3) / 0.1949930000E+01, 0.1174190000E+04, 0.3373760000E+03 /
  data (epsData(29,i,52), i=1,3) / 0.2027550000E+01, 0.1170370000E+04, 0.4933330000E+03 /
  data (epsData(30,i,52), i=1,3) / 0.2150000000E+01, 0.1072380000E+04, 0.6880510000E+03 /
  data (epsData(31,i,52), i=1,3) / 0.2212590000E+01, 0.9770710000E+03, 0.7578070000E+03 /
  data (epsData(32,i,52), i=1,3) / 0.2370410000E+01, 0.8141150000E+03, 0.7914750000E+03 /
  data (epsData(33,i,52), i=1,3) / 0.2628920000E+01, 0.6391820000E+03, 0.8116590000E+03 /
  data (epsData(34,i,52), i=1,3) / 0.3184030000E+01, 0.4076480000E+03, 0.8104820000E+03 /
  data (epsData(35,i,52), i=1,3) / 0.3862620000E+01, 0.1832050000E+03, 0.7847020000E+03 /
  data (epsData(36,i,52), i=1,3) / 0.4343240000E+01, 0.4688760000E+02, 0.7326020000E+03 /
  data (epsData(37,i,52), i=1,3) / 0.4878960000E+01, -0.6720780000E+02, 0.6496940000E+03 /
  data (epsData(38,i,52), i=1,3) / 0.5538840000E+01, -0.1641260000E+03, 0.5362170000E+03 /
  data (epsData(39,i,52), i=1,3) / 0.6041230000E+01, -0.1980370000E+03, 0.4398600000E+03 /
  data (epsData(40,i,52), i=1,3) / 0.6656200000E+01, -0.2075970000E+03, 0.3630520000E+03 /
  data (epsData(41,i,52), i=1,3) / 0.7400090000E+01, -0.2166780000E+03, 0.2798920000E+03 /
  data (epsData(42,i,52), i=1,3) / 0.8199090000E+01, -0.2079830000E+03, 0.2090770000E+03 /
  data (epsData(43,i,52), i=1,3) / 0.9151480000E+01, -0.1881110000E+03, 0.1468990000E+03 /
  data (epsData(44,i,52), i=1,3) / 0.1027770000E+02, -0.1603560000E+03, 0.9702520000E+02 /
  data (epsData(45,i,52), i=1,3) / 0.1137300000E+02, -0.1346190000E+03, 0.6592980000E+02 /
  data (epsData(46,i,52), i=1,3) / 0.1237040000E+02, -0.1145130000E+03, 0.4794110000E+02 /
  data (epsData(47,i,52), i=1,3) / 0.1361940000E+02, -0.9435300000E+02, 0.3257650000E+02 /
  data (epsData(48,i,52), i=1,3) / 0.1494190000E+02, -0.7735560000E+02, 0.2227970000E+02 /
  data (epsData(49,i,52), i=1,3) / 0.1634330000E+02, -0.6334950000E+02, 0.1542910000E+02 /
  data (epsData(50,i,52), i=1,3) / 0.1783720000E+02, -0.5184850000E+02, 0.1077910000E+02 /
  data (epsData(51,i,52), i=1,3) / 0.1945050000E+02, -0.4234170000E+02, 0.7591630000E+01 /
  data (epsData(52,i,52), i=1,3) / 0.2126040000E+02, -0.3424300000E+02, 0.5311930000E+01 /
  data (epsData(53,i,52), i=1,3) / 0.2326690000E+02, -0.2744270000E+02, 0.3696900000E+01 /
  data (epsData(54,i,52), i=1,3) / 0.2547000000E+02, -0.2178200000E+02, 0.2581330000E+01 /
  data (epsData(55,i,52), i=1,3) / 0.2786970000E+02, -0.1709560000E+02, 0.1827930000E+01 /
  data (epsData(56,i,52), i=1,3) / 0.3046600000E+02, -0.1320520000E+02, 0.1317840000E+01 /
  data (epsData(57,i,52), i=1,3) / 0.3325890000E+02, -0.9931010000E+01, 0.9756610000E+00 /
  data (epsData(58,i,52), i=1,3) / 0.3643040000E+02, -0.6928630000E+01, 0.7262680000E+00 /
  data (epsData(59,i,52), i=1,3) / 0.3962800000E+02, -0.4169150000E+01, 0.5745730000E+00 /
  data (epsData(60,i,52), i=1,3) / 0.4001720000E+02, -0.3760830000E+01, 0.5795690000E+00 /
  data (epsData(61,i,52), i=1,3) / 0.4021270000E+02, -0.3526710000E+01, 0.6240280000E+00 /
  data (epsData(62,i,52), i=1,3) / 0.4051020000E+02, -0.3229490000E+01, 0.8376310000E+00 /
  data (epsData(63,i,52), i=1,3) / 0.4077040000E+02, -0.3138830000E+01, 0.9512620000E+00 /
  data (epsData(64,i,52), i=1,3) / 0.4146670000E+02, -0.2724610000E+01, 0.1028540000E+01 /
  data (epsData(65,i,52), i=1,3) / 0.4188170000E+02, -0.2382960000E+01, 0.1139100000E+01 /
  data (epsData(66,i,52), i=1,3) / 0.4216670000E+02, -0.2261850000E+01, 0.1270680000E+01 /
  data (epsData(67,i,52), i=1,3) / 0.4316570000E+02, -0.1779870000E+01, 0.1392700000E+01 /
  data (epsData(68,i,52), i=1,3) / 0.4477900000E+02, -0.8797550000E+00, 0.1561070000E+01 /
  data (epsData(69,i,52), i=1,3) / 0.4655620000E+02, 0.1933960000E+00, 0.1820340000E+01 /
  data (epsData(70,i,52), i=1,3) / 0.4800760000E+02, 0.1144450000E+01, 0.2143130000E+01 /
  data (epsData(71,i,52), i=1,3) / 0.4955320000E+02, 0.2206740000E+01, 0.2707770000E+01 /
  data (epsData(72,i,52), i=1,3) / 0.5104850000E+02, 0.3177170000E+01, 0.3550930000E+01 /
  data (epsData(73,i,52), i=1,3) / 0.5255630000E+02, 0.3967790000E+01, 0.4695470000E+01 /
  data (epsData(74,i,52), i=1,3) / 0.5417910000E+02, 0.4466770000E+01, 0.6202120000E+01 /
  data (epsData(75,i,52), i=1,3) / 0.5619450000E+02, 0.4382770000E+01, 0.8246950000E+01 /
  data (epsData(76,i,52), i=1,3) / 0.5821350000E+02, 0.3532250000E+01, 0.1005660000E+02 /
  data (epsData(77,i,52), i=1,3) / 0.6052480000E+02, 0.1987850000E+01, 0.1156310000E+02 /
  data (epsData(78,i,52), i=1,3) / 0.6315890000E+02, -0.1317600000E+00, 0.1252170000E+02 /
  data (epsData(79,i,52), i=1,3) / 0.6596710000E+02, -0.2604580000E+01, 0.1273540000E+02 /
  data (epsData(80,i,52), i=1,3) / 0.6856710000E+02, -0.4837020000E+01, 0.1199560000E+02 /
  data (epsData(81,i,52), i=1,3) / 0.7130050000E+02, -0.6565610000E+01, 0.1047470000E+02 /
  data (epsData(82,i,52), i=1,3) / 0.7479890000E+02, -0.7649920000E+01, 0.8172710000E+01 /
  data (epsData(83,i,52), i=1,3) / 0.7827100000E+02, -0.7770300000E+01, 0.6157710000E+01 /
  data (epsData(84,i,52), i=1,3) / 0.8238540000E+02, -0.7345120000E+01, 0.4370430000E+01 /
  data (epsData(85,i,52), i=1,3) / 0.8746410000E+02, -0.6543740000E+01, 0.2865050000E+01 /
  data (epsData(86,i,52), i=1,3) / 0.9310630000E+02, -0.5608320000E+01, 0.1801100000E+01 /
  data (epsData(87,i,52), i=1,3) / 0.9852140000E+02, -0.4787910000E+01, 0.1170810000E+01 /
  data (epsData(88,i,52), i=1,3) / 0.1038440000E+03, -0.4091460000E+01, 0.8161050000E+00 /
  data (epsData(89,i,52), i=1,3) / 0.1118990000E+03, -0.3346610000E+01, 0.5075810000E+00 /
  data (epsData(90,i,52), i=1,3) / 0.1191700000E+03, -0.2811510000E+01, 0.3321070000E+00 /
  data (epsData(91,i,52), i=1,3) / 0.1275480000E+03, -0.2330570000E+01, 0.2192020000E+00 /
  data (epsData(92,i,52), i=1,3) / 0.1372650000E+03, -0.1904940000E+01, 0.1570700000E+00 /
  data (epsData(93,i,52), i=1,3) / 0.1491130000E+03, -0.1531330000E+01, 0.1251940000E+00 /
  data (epsData(94,i,52), i=1,3) / 0.1627310000E+03, -0.1222130000E+01, 0.1133940000E+00 /
  data (epsData(95,i,52), i=1,3) / 0.1738980000E+03, -0.1036050000E+01, 0.1222620000E+00 /
  data (epsData(96,i,52), i=1,3) / 0.1922170000E+03, -0.8300020000E+00, 0.1229790000E+00 /
  data (epsData(97,i,52), i=1,3) / 0.2142200000E+03, -0.6583020000E+00, 0.1168950000E+00 /
  data (epsData(98,i,52), i=1,3) / 0.2388930000E+03, -0.5263140000E+00, 0.1076800000E+00 /
  data (epsData(99,i,52), i=1,3) / 0.2685870000E+03, -0.4161990000E+00, 0.9482440000E-01 /
  data (epsData(100,i,52), i=1,3) / 0.3056930000E+03, -0.3216050000E+00, 0.7903140000E-01 /
  data (epsData(101,i,52), i=1,3) / 0.3472060000E+03, -0.2477300000E+00, 0.6363500000E-01 /
  data (epsData(102,i,52), i=1,3) / 0.3931240000E+03, -0.1886940000E+00, 0.5002900000E-01 /
  data (epsData(103,i,52), i=1,3) / 0.4449220000E+03, -0.1383390000E+00, 0.3844210000E-01 /
  data (epsData(104,i,52), i=1,3) / 0.5323720000E+03, -0.6726850000E-01, 0.2536510000E-01 /
  data (epsData(105,i,52), i=1,3) / 0.5523160000E+03, -0.4735560000E-01, 0.2316690000E-01 /
  data (epsData(106,i,52), i=1,3) / 0.5639520000E+03, -0.3242670000E-01, 0.2230340000E-01 /
  data (epsData(107,i,52), i=1,3) / 0.5705740000E+03, -0.2133310000E-01, 0.2182970000E-01 /
  data (epsData(108,i,52), i=1,3) / 0.5736290000E+03, -0.1478050000E-01, 0.2239820000E-01 /
  data (epsData(109,i,52), i=1,3) / 0.5774980000E+03, -0.6058050000E-02, 0.2301740000E-01 /
  data (epsData(110,i,52), i=1,3) / 0.5800530000E+03, 0.1274470000E-02, 0.2377900000E-01 /
  data (epsData(111,i,52), i=1,3) / 0.5820370000E+03, 0.8191360000E-02, 0.2508110000E-01 /
  data (epsData(112,i,52), i=1,3) / 0.5842500000E+03, 0.1693150000E-01, 0.2845920000E-01 /
  data (epsData(113,i,52), i=1,3) / 0.5870790000E+03, 0.2827080000E-01, 0.3566910000E-01 /
  data (epsData(114,i,52), i=1,3) / 0.5900670000E+03, 0.3902040000E-01, 0.4726090000E-01 /
  data (epsData(115,i,52), i=1,3) / 0.5933460000E+03, 0.4824010000E-01, 0.6380440000E-01 /
  data (epsData(116,i,52), i=1,3) / 0.5964670000E+03, 0.5303860000E-01, 0.8336860000E-01 /
  data (epsData(117,i,52), i=1,3) / 0.5998630000E+03, 0.4995600000E-01, 0.1067460000E+00 /
  data (epsData(118,i,52), i=1,3) / 0.6036510000E+03, 0.3797000000E-01, 0.1279280000E+00 /
  data (epsData(119,i,52), i=1,3) / 0.6087500000E+03, 0.1623940000E-01, 0.1465740000E+00 /
  data (epsData(120,i,52), i=1,3) / 0.6136060000E+03, -0.7228890000E-02, 0.1543310000E+00 /
  data (epsData(121,i,52), i=1,3) / 0.6184490000E+03, -0.2716900000E-01, 0.1527500000E+00 /
  data (epsData(122,i,52), i=1,3) / 0.6269940000E+03, -0.4913150000E-01, 0.1432310000E+00 /
  data (epsData(123,i,52), i=1,3) / 0.6400210000E+03, -0.6658650000E-01, 0.1275300000E+00 /
  data (epsData(124,i,52), i=1,3) / 0.6563160000E+03, -0.7508400000E-01, 0.1116910000E+00 /
  data (epsData(125,i,52), i=1,3) / 0.6866690000E+03, -0.7965460000E-01, 0.9308140000E-01 /
  data (epsData(126,i,52), i=1,3) / 0.7297780000E+03, -0.7866540000E-01, 0.7546630000E-01 /
  data (epsData(127,i,52), i=1,3) / 0.7882300000E+03, -0.7237900000E-01, 0.5874970000E-01 /
  data (epsData(128,i,52), i=1,3) / 0.8171650000E+03, -0.6582170000E-01, 0.5332860000E-01 /
  data (epsData(129,i,52), i=1,3) / 0.8251460000E+03, -0.6401830000E-01, 0.5536070000E-01 /
  data (epsData(130,i,52), i=1,3) / 0.8400850000E+03, -0.6550840000E-01, 0.5395080000E-01 /
  data (epsData(131,i,52), i=1,3) / 0.8723730000E+03, -0.6283660000E-01, 0.4958950000E-01 /
  data (epsData(132,i,52), i=1,3) / 0.9065070000E+03, -0.6254870000E-01, 0.4579570000E-01 /
  data (epsData(133,i,52), i=1,3) / 0.9940710000E+03, -0.5631500000E-01, 0.3505600000E-01 /
  data (epsData(134,i,52), i=1,3) / 0.1187270000E+04, -0.4472250000E-01, 0.2105180000E-01 /
  data (epsData(135,i,52), i=1,3) / 0.1334020000E+04, -0.3694740000E-01, 0.1449640000E-01 /
  data (epsData(136,i,52), i=1,3) / 0.1491750000E+04, -0.3036080000E-01, 0.1009110000E-01 /
  data (epsData(137,i,52), i=1,3) / 0.1655670000E+04, -0.2505260000E-01, 0.7130860000E-02 /
  data (epsData(138,i,52), i=1,3) / 0.1844450000E+04, -0.2037250000E-01, 0.4959310000E-02 /
  data (epsData(139,i,52), i=1,3) / 0.2046530000E+04, -0.1662870000E-01, 0.3482590000E-02 /
  data (epsData(140,i,52), i=1,3) / 0.2271730000E+04, -0.1349410000E-01, 0.2427600000E-02 /
  data (epsData(141,i,52), i=1,3) / 0.2526150000E+04, -0.1088350000E-01, 0.1686800000E-02 /
  data (epsData(142,i,52), i=1,3) / 0.2806990000E+04, -0.8761220000E-02, 0.1166730000E-02 /
  data (epsData(143,i,52), i=1,3) / 0.3120080000E+04, -0.7018580000E-02, 0.8039710000E-03 /
  data (epsData(144,i,52), i=1,3) / 0.3471060000E+04, -0.5576890000E-02, 0.5509430000E-03 /
  data (epsData(145,i,52), i=1,3) / 0.3880540000E+04, -0.4316830000E-02, 0.3701620000E-03 /
  data (epsData(146,i,52), i=1,3) / 0.4262260000E+04, -0.3256340000E-02, 0.2643940000E-03 /
  data (epsData(147,i,52), i=1,3) / 0.4319340000E+04, -0.2959690000E-02, 0.2520580000E-03 /
  data (epsData(148,i,52), i=1,3) / 0.4334220000E+04, -0.2792260000E-02, 0.2695890000E-03 /
  data (epsData(149,i,52), i=1,3) / 0.4338470000E+04, -0.2693620000E-02, 0.2876250000E-03 /
  data (epsData(150,i,52), i=1,3) / 0.4342440000E+04, -0.2553260000E-02, 0.3952300000E-03 /
  data (epsData(151,i,52), i=1,3) / 0.4345820000E+04, -0.2556640000E-02, 0.5954000000E-03 /
  data (epsData(152,i,52), i=1,3) / 0.4350490000E+04, -0.2704680000E-02, 0.6985080000E-03 /
  data (epsData(153,i,52), i=1,3) / 0.4365430000E+04, -0.2842790000E-02, 0.7220480000E-03 /
  data (epsData(154,i,52), i=1,3) / 0.4397340000E+04, -0.2956340000E-02, 0.7355570000E-03 /
  data (epsData(155,i,52), i=1,3) / 0.4501070000E+04, -0.2969340000E-02, 0.6845240000E-03 /
  data (epsData(156,i,52), i=1,3) / 0.4594920000E+04, -0.2797020000E-02, 0.6379200000E-03 /
  data (epsData(157,i,52), i=1,3) / 0.4611040000E+04, -0.2677220000E-02, 0.6553260000E-03 /
  data (epsData(158,i,52), i=1,3) / 0.4616580000E+04, -0.2639950000E-02, 0.7686580000E-03 /
  data (epsData(159,i,52), i=1,3) / 0.4625330000E+04, -0.2731350000E-02, 0.8181710000E-03 /
  data (epsData(160,i,52), i=1,3) / 0.4682420000E+04, -0.2818860000E-02, 0.8122420000E-03 /
  data (epsData(161,i,52), i=1,3) / 0.4891210000E+04, -0.2708670000E-02, 0.6949900000E-03 /
  data (epsData(162,i,52), i=1,3) / 0.4936680000E+04, -0.2600010000E-02, 0.6885630000E-03 /
  data (epsData(163,i,52), i=1,3) / 0.4947710000E+04, -0.2610430000E-02, 0.7579040000E-03 /
  data (epsData(164,i,52), i=1,3) / 0.5032400000E+04, -0.2638460000E-02, 0.7226700000E-03 /
  data (epsData(165,i,52), i=1,3) / 0.5664070000E+04, -0.2245000000E-02, 0.4767560000E-03 /
  data (epsData(166,i,52), i=1,3) / 0.6392310000E+04, -0.1817170000E-02, 0.3119070000E-03 /
  data (epsData(167,i,52), i=1,3) / 0.7166500000E+04, -0.1468500000E-02, 0.2070960000E-03 /
  data (epsData(168,i,52), i=1,3) / 0.8000400000E+04, -0.1187810000E-02, 0.1388620000E-03 /
  data (epsData(169,i,52), i=1,3) / 0.8931470000E+04, -0.9565620000E-03, 0.9295100000E-04 /
  data (epsData(170,i,52), i=1,3) / 0.9959710000E+04, -0.7701170000E-03, 0.6206570000E-04 /
  data (epsData(171,i,52), i=1,3) / 0.1110560000E+05, -0.6191320000E-03, 0.4139490000E-04 /
  data (epsData(172,i,52), i=1,3) / 0.1238520000E+05, -0.4970010000E-03, 0.2752830000E-04 /
  data (epsData(173,i,52), i=1,3) / 0.1381990000E+05, -0.3982430000E-03, 0.1824570000E-04 /
  data (epsData(174,i,52), i=1,3) / 0.1543010000E+05, -0.3185870000E-03, 0.1201830000E-04 /
  data (epsData(175,i,52), i=1,3) / 0.1723150000E+05, -0.2546600000E-03, 0.7901710000E-05 /
  data (epsData(176,i,52), i=1,3) / 0.1925500000E+05, -0.2032380000E-03, 0.5171160000E-05 /
  data (epsData(177,i,52), i=1,3) / 0.2152350000E+05, -0.1620170000E-03, 0.3373890000E-05 /
  data (epsData(178,i,52), i=1,3) / 0.2407080000E+05, -0.1289170000E-03, 0.2192490000E-05 /
  data (epsData(179,i,52), i=1,3) / 0.2670740000E+05, -0.1040590000E-03, 0.1466860000E-05 /
  data (epsData(180,i,52), i=1,3) / 0.2671100000E+05, -0.1004440000E-03, 0.1466100000E-05 /
  data (epsData(181,i,52), i=1,3) / 0.2671100000E+05, -0.1040290000E-03, 0.1466090000E-05 /
  data (epsData(182,i,52), i=1,3) / 0.3013140000E+05, -0.8041810000E-04, 0.9181670000E-06 /
  data (epsData(183,i,52), i=1,3) / 0.3167700000E+05, -0.6981360000E-04, 0.7554360000E-06 /
  data (epsData(184,i,52), i=1,3) / 0.3179200000E+05, -0.6681110000E-04, 0.7448700000E-06 /
  data (epsData(185,i,52), i=1,3) / 0.3181310000E+05, -0.6530330000E-04, 0.2251330000E-05 /
  data (epsData(186,i,52), i=1,3) / 0.3181400000E+05, -0.6094160000E-04, 0.2422980000E-05 /
  data (epsData(187,i,52), i=1,3) / 0.3181400000E+05, -0.6524710000E-04, 0.2430360000E-05 /
  data (epsData(188,i,52), i=1,3) / 0.3183360000E+05, -0.6657640000E-04, 0.4313260000E-05 /
  data (epsData(189,i,52), i=1,3) / 0.3195680000E+05, -0.6862820000E-04, 0.4578980000E-05 /
  data (epsData(190,i,52), i=1,3) / 0.3309550000E+05, -0.6670400000E-04, 0.4077970000E-05 /
  data (epsData(191,i,52), i=1,3) / 0.3842570000E+05, -0.5101260000E-04, 0.2381580000E-05 /
  data (epsData(192,i,52), i=1,3) / 0.4321550000E+05, -0.4063440000E-04, 0.1536320000E-05 /
  data (epsData(193,i,52), i=1,3) / 0.4846510000E+05, -0.3242450000E-04, 0.9963420000E-06 /
  data (epsData(194,i,52), i=1,3) / 0.5430130000E+05, -0.2587390000E-04, 0.6450950000E-06 /
  data (epsData(195,i,52), i=1,3) / 0.6083020000E+05, -0.2063220000E-04, 0.4159130000E-06 /
  data (epsData(196,i,52), i=1,3) / 0.6815500000E+05, -0.1643920000E-04, 0.2676530000E-06 /
  data (epsData(197,i,52), i=1,3) / 0.7634210000E+05, -0.1310160000E-04, 0.1705550000E-06 /
  data (epsData(198,i,52), i=1,3) / 0.8551410000E+05, -0.1044290000E-04, 0.1084260000E-06 /
  data (epsData(199,i,52), i=1,3) / 0.9567770000E+05, -0.8354030000E-05, 0.6895800000E-07 /
  data (epsData(200,i,52), i=1,3) / 0.9998700000E+05, -0.7750550000E-05, 0.5774800000E-07 /
  data (epsData(201,i,52), i=1,3) / 0.1000000000E+06, -0.7777630000E-05, 0.5771700000E-07 /
  data (epsData(1,i,53), i=1,3) / 0.2506580000E-02, -0.1953660000E+05, 0.4730900000E+05 /
  data (epsData(2,i,53), i=1,3) / 0.4693440000E-02, -0.1571500000E+05, 0.1567800000E+05 /
  data (epsData(3,i,53), i=1,3) / 0.7560590000E-02, -0.7135890000E+04, 0.7145470000E+04 /
  data (epsData(4,i,53), i=1,3) / 0.1110800000E-01, -0.3410740000E+04, 0.3942960000E+04 /
  data (epsData(5,i,53), i=1,3) / 0.1533570000E-01, -0.1662070000E+04, 0.2464220000E+04 /
  data (epsData(6,i,53), i=1,3) / 0.2024370000E-01, -0.7582830000E+03, 0.1677730000E+04 /
  data (epsData(7,i,53), i=1,3) / 0.2583200000E-01, -0.2526400000E+03, 0.1214640000E+04 /
  data (epsData(8,i,53), i=1,3) / 0.3210060000E-01, 0.4880740000E+02, 0.9204330000E+03 /
  data (epsData(9,i,53), i=1,3) / 0.3904940000E-01, 0.2380450000E+03, 0.7222760000E+03 /
  data (epsData(10,i,53), i=1,3) / 0.4667860000E-01, 0.3620560000E+03, 0.5825210000E+03 /
  data (epsData(11,i,53), i=1,3) / 0.5498800000E-01, 0.4463740000E+03, 0.4802420000E+03 /
  data (epsData(12,i,53), i=1,3) / 0.6397770000E-01, 0.5056360000E+03, 0.4030930000E+03 /
  data (epsData(13,i,53), i=1,3) / 0.7364770000E-01, 0.5486690000E+03, 0.3434200000E+03 /
  data (epsData(14,i,53), i=1,3) / 0.8399800000E-01, 0.5811950000E+03, 0.2962820000E+03 /
  data (epsData(15,i,53), i=1,3) / 0.9502860000E-01, 0.6076730000E+03, 0.2583700000E+03 /
  data (epsData(16,i,53), i=1,3) / 0.1191310000E+00, 0.6419480000E+03, 0.2138830000E+03 /
  data (epsData(17,i,53), i=1,3) / 0.1603860000E+00, 0.6587460000E+03, 0.1677030000E+03 /
  data (epsData(18,i,53), i=1,3) / 0.2612640000E+00, 0.6681180000E+03, 0.1089980000E+03 /
  data (epsData(19,i,53), i=1,3) / 0.4338630000E+00, 0.6807030000E+03, 0.7060470000E+02 /
  data (epsData(20,i,53), i=1,3) / 0.7743970000E+00, 0.7061330000E+03, 0.5154310000E+02 /
  data (epsData(21,i,53), i=1,3) / 0.1094080000E+01, 0.7510260000E+03, 0.5001490000E+02 /
  data (epsData(22,i,53), i=1,3) / 0.1337820000E+01, 0.8102070000E+03, 0.5564620000E+02 /
  data (epsData(23,i,53), i=1,3) / 0.1513920000E+01, 0.8706560000E+03, 0.6861050000E+02 /
  data (epsData(24,i,53), i=1,3) / 0.1606050000E+01, 0.9273050000E+03, 0.7635730000E+02 /
  data (epsData(25,i,53), i=1,3) / 0.1749350000E+01, 0.1001080000E+04, 0.1535280000E+03 /
  data (epsData(26,i,53), i=1,3) / 0.1848280000E+01, 0.1058050000E+04, 0.1926790000E+03 /
  data (epsData(27,i,53), i=1,3) / 0.1898770000E+01, 0.1119150000E+04, 0.2092350000E+03 /
  data (epsData(28,i,53), i=1,3) / 0.1949930000E+01, 0.1170500000E+04, 0.3182940000E+03 /
  data (epsData(29,i,53), i=1,3) / 0.2034010000E+01, 0.1165560000E+04, 0.4748740000E+03 /
  data (epsData(30,i,53), i=1,3) / 0.2150000000E+01, 0.1079480000E+04, 0.6461070000E+03 /
  data (epsData(31,i,53), i=1,3) / 0.2212590000E+01, 0.9924230000E+03, 0.7115240000E+03 /
  data (epsData(32,i,53), i=1,3) / 0.2393200000E+01, 0.8280500000E+03, 0.7487960000E+03 /
  data (epsData(33,i,53), i=1,3) / 0.2688790000E+01, 0.6606570000E+03, 0.7713640000E+03 /
  data (epsData(34,i,53), i=1,3) / 0.3666700000E+01, 0.3308340000E+03, 0.8062380000E+03 /
  data (epsData(35,i,53), i=1,3) / 0.4231670000E+01, 0.1566320000E+03, 0.7862770000E+03 /
  data (epsData(36,i,53), i=1,3) / 0.4755150000E+01, 0.1921990000E+02, 0.7297060000E+03 /
  data (epsData(37,i,53), i=1,3) / 0.5333730000E+01, -0.1021590000E+03, 0.6430920000E+03 /
  data (epsData(38,i,53), i=1,3) / 0.5678980000E+01, -0.1541400000E+03, 0.5744810000E+03 /
  data (epsData(39,i,53), i=1,3) / 0.6190890000E+01, -0.1863920000E+03, 0.4882780000E+03 /
  data (epsData(40,i,53), i=1,3) / 0.7062670000E+01, -0.2248520000E+03, 0.3692450000E+03 /
  data (epsData(41,i,53), i=1,3) / 0.7837180000E+01, -0.2283920000E+03, 0.2821080000E+03 /
  data (epsData(42,i,53), i=1,3) / 0.8762360000E+01, -0.2169200000E+03, 0.2024180000E+03 /
  data (epsData(43,i,53), i=1,3) / 0.9755580000E+01, -0.1940730000E+03, 0.1400790000E+03 /
  data (epsData(44,i,53), i=1,3) / 0.1081680000E+02, -0.1663150000E+03, 0.9524180000E+02 /
  data (epsData(45,i,53), i=1,3) / 0.1194610000E+02, -0.1391780000E+03, 0.6461360000E+02 /
  data (epsData(46,i,53), i=1,3) / 0.1302060000E+02, -0.1174440000E+03, 0.4583770000E+02 /
  data (epsData(47,i,53), i=1,3) / 0.1421260000E+02, -0.9776310000E+02, 0.3244500000E+02 /
  data (epsData(48,i,53), i=1,3) / 0.1561940000E+02, -0.7993080000E+02, 0.2213710000E+02 /
  data (epsData(49,i,53), i=1,3) / 0.1706980000E+02, -0.6568180000E+02, 0.1542860000E+02 /
  data (epsData(50,i,53), i=1,3) / 0.1865620000E+02, -0.5378010000E+02, 0.1079040000E+02 /
  data (epsData(51,i,53), i=1,3) / 0.2047770000E+02, -0.4348650000E+02, 0.7388570000E+01 /
  data (epsData(52,i,53), i=1,3) / 0.2238420000E+02, -0.3532170000E+02, 0.5127460000E+01 /
  data (epsData(53,i,53), i=1,3) / 0.2453620000E+02, -0.2836480000E+02, 0.3542950000E+01 /
  data (epsData(54,i,53), i=1,3) / 0.2693390000E+02, -0.2256670000E+02, 0.2459580000E+01 /
  data (epsData(55,i,53), i=1,3) / 0.2957710000E+02, -0.1778530000E+02, 0.1729590000E+01 /
  data (epsData(56,i,53), i=1,3) / 0.3246600000E+02, -0.1386700000E+02, 0.1241060000E+01 /
  data (epsData(57,i,53), i=1,3) / 0.3560040000E+02, -0.1063070000E+02, 0.9011790000E+00 /
  data (epsData(58,i,53), i=1,3) / 0.3898040000E+02, -0.7906780000E+01, 0.6622680000E+00 /
  data (epsData(59,i,53), i=1,3) / 0.4280360000E+02, -0.5414550000E+01, 0.4885840000E+00 /
  data (epsData(60,i,53), i=1,3) / 0.4796520000E+02, -0.2423350000E+01, 0.3589190000E+00 /
  data (epsData(61,i,53), i=1,3) / 0.4861320000E+02, -0.1963770000E+01, 0.3751120000E+00 /
  data (epsData(62,i,53), i=1,3) / 0.4883050000E+02, -0.1782640000E+01, 0.4179550000E+00 /
  data (epsData(63,i,53), i=1,3) / 0.4907990000E+02, -0.1642180000E+01, 0.5523320000E+00 /
  data (epsData(64,i,53), i=1,3) / 0.4955850000E+02, -0.1466000000E+01, 0.6210320000E+00 /
  data (epsData(65,i,53), i=1,3) / 0.5029320000E+02, -0.1081440000E+01, 0.6793620000E+00 /
  data (epsData(66,i,53), i=1,3) / 0.5056670000E+02, -0.9063230000E+00, 0.7334720000E+00 /
  data (epsData(67,i,53), i=1,3) / 0.5082990000E+02, -0.7990270000E+00, 0.8221070000E+00 /
  data (epsData(68,i,53), i=1,3) / 0.5151970000E+02, -0.5192440000E+00, 0.9059430000E+00 /
  data (epsData(69,i,53), i=1,3) / 0.5283170000E+02, 0.7458750000E-01, 0.1046810000E+01 /
  data (epsData(70,i,53), i=1,3) / 0.5479090000E+02, 0.1083960000E+01, 0.1270270000E+01 /
  data (epsData(71,i,53), i=1,3) / 0.5617290000E+02, 0.1920500000E+01, 0.1508190000E+01 /
  data (epsData(72,i,53), i=1,3) / 0.5745860000E+02, 0.2798750000E+01, 0.1875940000E+01 /
  data (epsData(73,i,53), i=1,3) / 0.5898420000E+02, 0.3906840000E+01, 0.2605890000E+01 /
  data (epsData(74,i,53), i=1,3) / 0.6054610000E+02, 0.4983140000E+01, 0.3777240000E+01 /
  data (epsData(75,i,53), i=1,3) / 0.6199100000E+02, 0.5760540000E+01, 0.5285760000E+01 /
  data (epsData(76,i,53), i=1,3) / 0.6340160000E+02, 0.6100880000E+01, 0.7130250000E+01 /
  data (epsData(77,i,53), i=1,3) / 0.6503430000E+02, 0.5694940000E+01, 0.9460800000E+01 /
  data (epsData(78,i,53), i=1,3) / 0.6661690000E+02, 0.4468720000E+01, 0.1147170000E+02 /
  data (epsData(79,i,53), i=1,3) / 0.6828190000E+02, 0.2558030000E+01, 0.1297520000E+02 /
  data (epsData(80,i,53), i=1,3) / 0.7035580000E+02, -0.1494970000E+00, 0.1381910000E+02 /
  data (epsData(81,i,53), i=1,3) / 0.7254460000E+02, -0.2966950000E+01, 0.1369330000E+02 /
  data (epsData(82,i,53), i=1,3) / 0.7448650000E+02, -0.5174290000E+01, 0.1274120000E+02 /
  data (epsData(83,i,53), i=1,3) / 0.7669870000E+02, -0.6943840000E+01, 0.1100550000E+02 /
  data (epsData(84,i,53), i=1,3) / 0.7960190000E+02, -0.8016300000E+01, 0.8542340000E+01 /
  data (epsData(85,i,53), i=1,3) / 0.8244950000E+02, -0.8122660000E+01, 0.6480040000E+01 /
  data (epsData(86,i,53), i=1,3) / 0.8586730000E+02, -0.7682870000E+01, 0.4664470000E+01 /
  data (epsData(87,i,53), i=1,3) / 0.9038440000E+02, -0.6845920000E+01, 0.3089320000E+01 /
  data (epsData(88,i,53), i=1,3) / 0.9566740000E+02, -0.5876970000E+01, 0.1955650000E+01 /
  data (epsData(89,i,53), i=1,3) / 0.1010180000E+03, -0.5018770000E+01, 0.1257330000E+01 /
  data (epsData(90,i,53), i=1,3) / 0.1069440000E+03, -0.4232880000E+01, 0.7874980000E+00 /
  data (epsData(91,i,53), i=1,3) / 0.1134920000E+03, -0.3530030000E+01, 0.4788390000E+00 /
  data (epsData(92,i,53), i=1,3) / 0.1204010000E+03, -0.2926570000E+01, 0.3047090000E+00 /
  data (epsData(93,i,53), i=1,3) / 0.1252870000E+03, -0.2594790000E+01, 0.2788690000E+00 /
  data (epsData(94,i,53), i=1,3) / 0.1370170000E+03, -0.2050490000E+01, 0.1823520000E+00 /
  data (epsData(95,i,53), i=1,3) / 0.1490990000E+03, -0.1638700000E+01, 0.1360060000E+00 /
  data (epsData(96,i,53), i=1,3) / 0.1618590000E+03, -0.1326410000E+01, 0.1188300000E+00 /
  data (epsData(97,i,53), i=1,3) / 0.1774230000E+03, -0.1055210000E+01, 0.1120850000E+00 /
  data (epsData(98,i,53), i=1,3) / 0.1901910000E+03, -0.8913650000E+00, 0.1197680000E+00 /
  data (epsData(99,i,53), i=1,3) / 0.2102480000E+03, -0.7199190000E+00, 0.1202400000E+00 /
  data (epsData(100,i,53), i=1,3) / 0.2353750000E+03, -0.5694390000E+00, 0.1122320000E+00 /
  data (epsData(101,i,53), i=1,3) / 0.2653520000E+03, -0.4481330000E+00, 0.1002110000E+00 /
  data (epsData(102,i,53), i=1,3) / 0.3000310000E+03, -0.3520930000E+00, 0.8556950000E-01 /
  data (epsData(103,i,53), i=1,3) / 0.3405210000E+03, -0.2739370000E+00, 0.6995980000E-01 /
  data (epsData(104,i,53), i=1,3) / 0.3866280000E+03, -0.2107080000E+00, 0.5540250000E-01 /
  data (epsData(105,i,53), i=1,3) / 0.4381610000E+03, -0.1589540000E+00, 0.4284110000E-01 /
  data (epsData(106,i,53), i=1,3) / 0.4975270000E+03, -0.1130940000E+00, 0.3225820000E-01 /
  data (epsData(107,i,53), i=1,3) / 0.5703610000E+03, -0.6228820000E-01, 0.2329790000E-01 /
  data (epsData(108,i,53), i=1,3) / 0.5931060000E+03, -0.4229600000E-01, 0.2109300000E-01 /
  data (epsData(109,i,53), i=1,3) / 0.6058570000E+03, -0.2731070000E-01, 0.2023690000E-01 /
  data (epsData(110,i,53), i=1,3) / 0.6126870000E+03, -0.1684920000E-01, 0.1992660000E-01 /
  data (epsData(111,i,53), i=1,3) / 0.6172710000E+03, -0.7971450000E-02, 0.1974340000E-01 /
  data (epsData(112,i,53), i=1,3) / 0.6194800000E+03, -0.2606890000E-02, 0.2005920000E-01 /
  data (epsData(113,i,53), i=1,3) / 0.6218160000E+03, 0.3177880000E-02, 0.2059180000E-01 /
  data (epsData(114,i,53), i=1,3) / 0.6244460000E+03, 0.1114700000E-01, 0.2126400000E-01 /
  data (epsData(115,i,53), i=1,3) / 0.6265500000E+03, 0.1912650000E-01, 0.2231310000E-01 /
  data (epsData(116,i,53), i=1,3) / 0.6285600000E+03, 0.2851960000E-01, 0.2461120000E-01 /
  data (epsData(117,i,53), i=1,3) / 0.6304430000E+03, 0.3890180000E-01, 0.2904970000E-01 /
  data (epsData(118,i,53), i=1,3) / 0.6327650000E+03, 0.5205910000E-01, 0.3944130000E-01 /
  data (epsData(119,i,53), i=1,3) / 0.6347670000E+03, 0.6136300000E-01, 0.5315870000E-01 /
  data (epsData(120,i,53), i=1,3) / 0.6374670000E+03, 0.6713430000E-01, 0.7534520000E-01 /
  data (epsData(121,i,53), i=1,3) / 0.6410980000E+03, 0.6407200000E-01, 0.1047400000E+00 /
  data (epsData(122,i,53), i=1,3) / 0.6442700000E+03, 0.5313060000E-01, 0.1271590000E+00 /
  data (epsData(123,i,53), i=1,3) / 0.6483270000E+03, 0.3273540000E-01, 0.1480260000E+00 /
  data (epsData(124,i,53), i=1,3) / 0.6532660000E+03, 0.3629900000E-02, 0.1617980000E+00 /
  data (epsData(125,i,53), i=1,3) / 0.6570880000E+03, -0.2035310000E-01, 0.1629120000E+00 /
  data (epsData(126,i,53), i=1,3) / 0.6612090000E+03, -0.4017150000E-01, 0.1549560000E+00 /
  data (epsData(127,i,53), i=1,3) / 0.6671710000E+03, -0.5525950000E-01, 0.1408040000E+00 /
  data (epsData(128,i,53), i=1,3) / 0.6771140000E+03, -0.6666960000E-01, 0.1241590000E+00 /
  data (epsData(129,i,53), i=1,3) / 0.6922380000E+03, -0.7381690000E-01, 0.1069660000E+00 /
  data (epsData(130,i,53), i=1,3) / 0.7161770000E+03, -0.7633860000E-01, 0.9062010000E-01 /
  data (epsData(131,i,53), i=1,3) / 0.7573690000E+03, -0.7577110000E-01, 0.7331200000E-01 /
  data (epsData(132,i,53), i=1,3) / 0.8226450000E+03, -0.7004680000E-01, 0.5583960000E-01 /
  data (epsData(133,i,53), i=1,3) / 0.8673940000E+03, -0.6279730000E-01, 0.4754140000E-01 /
  data (epsData(134,i,53), i=1,3) / 0.8778790000E+03, -0.5947170000E-01, 0.4845770000E-01 /
  data (epsData(135,i,53), i=1,3) / 0.8892940000E+03, -0.6114910000E-01, 0.4879580000E-01 /
  data (epsData(136,i,53), i=1,3) / 0.9302420000E+03, -0.5855020000E-01, 0.4351120000E-01 /
  data (epsData(137,i,53), i=1,3) / 0.9501050000E+03, -0.5850350000E-01, 0.4245660000E-01 /
  data (epsData(138,i,53), i=1,3) / 0.1045820000E+04, -0.5303080000E-01, 0.3202590000E-01 /
  data (epsData(139,i,53), i=1,3) / 0.1092920000E+04, -0.5004450000E-01, 0.2910890000E-01 /
  data (epsData(140,i,53), i=1,3) / 0.1249310000E+04, -0.4178480000E-01, 0.1924450000E-01 /
  data (epsData(141,i,53), i=1,3) / 0.1404710000E+04, -0.3444990000E-01, 0.1323680000E-01 /
  data (epsData(142,i,53), i=1,3) / 0.1565090000E+04, -0.2847990000E-01, 0.9295550000E-02 /
  data (epsData(143,i,53), i=1,3) / 0.1745380000E+04, -0.2325970000E-01, 0.6463210000E-02 /
  data (epsData(144,i,53), i=1,3) / 0.1940620000E+04, -0.1899770000E-01, 0.4525550000E-02 /
  data (epsData(145,i,53), i=1,3) / 0.2154590000E+04, -0.1547650000E-01, 0.3160470000E-02 /
  data (epsData(146,i,53), i=1,3) / 0.2390770000E+04, -0.1256580000E-01, 0.2214480000E-02 /
  data (epsData(147,i,53), i=1,3) / 0.2657560000E+04, -0.1014520000E-01, 0.1535730000E-02 /
  data (epsData(148,i,53), i=1,3) / 0.2953710000E+04, -0.8163260000E-02, 0.1060780000E-02 /
  data (epsData(149,i,53), i=1,3) / 0.3282130000E+04, -0.6545220000E-02, 0.7312220000E-03 /
  data (epsData(150,i,53), i=1,3) / 0.3650040000E+04, -0.5206960000E-02, 0.5014310000E-03 /
  data (epsData(151,i,53), i=1,3) / 0.4080570000E+04, -0.4034130000E-02, 0.3367490000E-03 /
  data (epsData(152,i,53), i=1,3) / 0.4479420000E+04, -0.3050830000E-02, 0.2409440000E-03 /
  data (epsData(153,i,53), i=1,3) / 0.4535630000E+04, -0.2783300000E-02, 0.2303550000E-03 /
  data (epsData(154,i,53), i=1,3) / 0.4550840000E+04, -0.2619680000E-02, 0.2493660000E-03 /
  data (epsData(155,i,53), i=1,3) / 0.4555200000E+04, -0.2516510000E-02, 0.2729840000E-03 /
  data (epsData(156,i,53), i=1,3) / 0.4558910000E+04, -0.2393090000E-02, 0.3970400000E-03 /
  data (epsData(157,i,53), i=1,3) / 0.4562060000E+04, -0.2422010000E-02, 0.5798510000E-03 /
  data (epsData(158,i,53), i=1,3) / 0.4566370000E+04, -0.2567950000E-02, 0.6578320000E-03 /
  data (epsData(159,i,53), i=1,3) / 0.4578430000E+04, -0.2683520000E-02, 0.6520240000E-03 /
  data (epsData(160,i,53), i=1,3) / 0.4610880000E+04, -0.2789230000E-02, 0.6646450000E-03 /
  data (epsData(161,i,53), i=1,3) / 0.4720830000E+04, -0.2803620000E-02, 0.6201710000E-03 /
  data (epsData(162,i,53), i=1,3) / 0.4831300000E+04, -0.2634470000E-02, 0.5682570000E-03 /
  data (epsData(163,i,53), i=1,3) / 0.4850560000E+04, -0.2516330000E-02, 0.5866600000E-03 /
  data (epsData(164,i,53), i=1,3) / 0.4855820000E+04, -0.2469680000E-02, 0.6856030000E-03 /
  data (epsData(165,i,53), i=1,3) / 0.4863390000E+04, -0.2560400000E-02, 0.7433020000E-03 /
  data (epsData(166,i,53), i=1,3) / 0.4926500000E+04, -0.2644420000E-02, 0.7271050000E-03 /
  data (epsData(167,i,53), i=1,3) / 0.5149370000E+04, -0.2523510000E-02, 0.6207340000E-03 /
  data (epsData(168,i,53), i=1,3) / 0.5186730000E+04, -0.2430170000E-02, 0.6245090000E-03 /
  data (epsData(169,i,53), i=1,3) / 0.5199890000E+04, -0.2453070000E-02, 0.6848230000E-03 /
  data (epsData(170,i,53), i=1,3) / 0.5364820000E+04, -0.2435540000E-02, 0.6188310000E-03 /
  data (epsData(171,i,53), i=1,3) / 0.6259700000E+04, -0.1919850000E-02, 0.3598890000E-03 /
  data (epsData(172,i,53), i=1,3) / 0.7034670000E+04, -0.1554110000E-02, 0.2378920000E-03 /
  data (epsData(173,i,53), i=1,3) / 0.7858630000E+04, -0.1259300000E-02, 0.1595790000E-03 /
  data (epsData(174,i,53), i=1,3) / 0.8776460000E+04, -0.1015380000E-02, 0.1066220000E-03 /
  data (epsData(175,i,53), i=1,3) / 0.9784300000E+04, -0.8190940000E-03, 0.7148650000E-04 /
  data (epsData(176,i,53), i=1,3) / 0.1091110000E+05, -0.6588350000E-03, 0.4776070000E-04 /
  data (epsData(177,i,53), i=1,3) / 0.1217060000E+05, -0.5289900000E-03, 0.3177430000E-04 /
  data (epsData(178,i,53), i=1,3) / 0.1357890000E+05, -0.4241490000E-03, 0.2105110000E-04 /
  data (epsData(179,i,53), i=1,3) / 0.1515320000E+05, -0.3397480000E-03, 0.1393020000E-04 /
  data (epsData(180,i,53), i=1,3) / 0.1691870000E+05, -0.2717610000E-03, 0.9171220000E-05 /
  data (epsData(181,i,53), i=1,3) / 0.1889710000E+05, -0.2171450000E-03, 0.6018100000E-05 /
  data (epsData(182,i,53), i=1,3) / 0.2111860000E+05, -0.1732450000E-03, 0.3931540000E-05 /
  data (epsData(183,i,53), i=1,3) / 0.2361700000E+05, -0.1379560000E-03, 0.2558060000E-05 /
  data (epsData(184,i,53), i=1,3) / 0.2642620000E+05, -0.1095910000E-03, 0.1657650000E-05 /
  data (epsData(185,i,53), i=1,3) / 0.2793610000E+05, -0.9770590000E-04, 0.1336480000E-05 /
  data (epsData(186,i,53), i=1,3) / 0.2794000000E+05, -0.9467170000E-04, 0.1335760000E-05 /
  data (epsData(187,i,53), i=1,3) / 0.2794000000E+05, -0.9767730000E-04, 0.1335760000E-05 /
  data (epsData(188,i,53), i=1,3) / 0.3153950000E+05, -0.7537920000E-04, 0.8338990000E-06 /
  data (epsData(189,i,53), i=1,3) / 0.3303680000E+05, -0.6599820000E-04, 0.6960550000E-06 /
  data (epsData(190,i,53), i=1,3) / 0.3314800000E+05, -0.6325290000E-04, 0.6869940000E-06 /
  data (epsData(191,i,53), i=1,3) / 0.3316660000E+05, -0.6208930000E-04, 0.1855180000E-05 /
  data (epsData(192,i,53), i=1,3) / 0.3316900000E+05, -0.5745570000E-04, 0.2200540000E-05 /
  data (epsData(193,i,53), i=1,3) / 0.3316900000E+05, -0.6195110000E-04, 0.2206890000E-05 /
  data (epsData(194,i,53), i=1,3) / 0.3319040000E+05, -0.6318560000E-04, 0.3954370000E-05 /
  data (epsData(195,i,53), i=1,3) / 0.3333090000E+05, -0.6504550000E-04, 0.4192190000E-05 /
  data (epsData(196,i,53), i=1,3) / 0.3466910000E+05, -0.6258700000E-04, 0.3671690000E-05 /
  data (epsData(197,i,53), i=1,3) / 0.4015160000E+05, -0.4799660000E-04, 0.2162600000E-05 /
  data (epsData(198,i,53), i=1,3) / 0.4514870000E+05, -0.3823170000E-04, 0.1395260000E-05 /
  data (epsData(199,i,53), i=1,3) / 0.5063430000E+05, -0.3050090000E-04, 0.9042210000E-06 /
  data (epsData(200,i,53), i=1,3) / 0.5673680000E+05, -0.2433290000E-04, 0.5849660000E-06 /
  data (epsData(201,i,53), i=1,3) / 0.6355210000E+05, -0.1940680000E-04, 0.3772850000E-06 /
  data (epsData(202,i,53), i=1,3) / 0.7120320000E+05, -0.1546410000E-04, 0.2418720000E-06 /
  data (epsData(203,i,53), i=1,3) / 0.7976020000E+05, -0.1232520000E-04, 0.1545410000E-06 /
  data (epsData(204,i,53), i=1,3) / 0.8933300000E+05, -0.9829260000E-05, 0.9825810000E-07 /
  data (epsData(205,i,53), i=1,3) / 0.9925030000E+05, -0.8003160000E-05, 0.6426960000E-07 /
  data (epsData(206,i,53), i=1,3) / 0.1000000000E+06, -0.7975270000E-05, 0.6231270000E-07 /
  data (epsData(1,i,54), i=1,3) / 0.2506580000E-02, -0.9720670000E+03, 0.2102710000E+04 /
  data (epsData(2,i,54), i=1,3) / 0.4693440000E-02, -0.7539030000E+03, 0.6727640000E+03 /
  data (epsData(3,i,54), i=1,3) / 0.7560590000E-02, -0.3454810000E+03, 0.2961570000E+03 /
  data (epsData(4,i,54), i=1,3) / 0.1110800000E-01, -0.1691630000E+03, 0.1583510000E+03 /
  data (epsData(5,i,54), i=1,3) / 0.1533570000E-01, -0.8659340000E+02, 0.9630150000E+02 /
  data (epsData(6,i,54), i=1,3) / 0.2024370000E-01, -0.4397840000E+02, 0.6407650000E+02 /
  data (epsData(7,i,54), i=1,3) / 0.2583200000E-01, -0.2016180000E+02, 0.4551220000E+02 /
  data (epsData(8,i,54), i=1,3) / 0.3210060000E-01, -0.5976910000E+01, 0.3394770000E+02 /
  data (epsData(9,i,54), i=1,3) / 0.3904940000E-01, 0.2917480000E+01, 0.2629300000E+02 /
  data (epsData(10,i,54), i=1,3) / 0.4667860000E-01, 0.8736180000E+01, 0.2097640000E+02 /
  data (epsData(11,i,54), i=1,3) / 0.5498800000E-01, 0.1268110000E+02, 0.1713740000E+02 /
  data (epsData(12,i,54), i=1,3) / 0.6397770000E-01, 0.1543930000E+02, 0.1427530000E+02 /
  data (epsData(13,i,54), i=1,3) / 0.7364770000E-01, 0.1742190000E+02, 0.1208420000E+02 /
  data (epsData(14,i,54), i=1,3) / 0.8399800000E-01, 0.1888750000E+02, 0.1036880000E+02 /
  data (epsData(15,i,54), i=1,3) / 0.9502860000E-01, 0.2001310000E+02, 0.8999890000E+01 /
  data (epsData(16,i,54), i=1,3) / 0.1191310000E+00, 0.2150350000E+02, 0.7156370000E+01 /
  data (epsData(17,i,54), i=1,3) / 0.1603860000E+00, 0.2255310000E+02, 0.5342760000E+01 /
  data (epsData(18,i,54), i=1,3) / 0.2249170000E+00, 0.2315410000E+02, 0.3825920000E+01 /
  data (epsData(19,i,54), i=1,3) / 0.3640360000E+00, 0.2353740000E+02, 0.2373240000E+01 /
  data (epsData(20,i,54), i=1,3) / 0.7422780000E+00, 0.2372820000E+02, 0.1202640000E+01 /
  data (epsData(21,i,54), i=1,3) / 0.2327150000E+01, 0.2413820000E+02, 0.4525040000E+00 /
  data (epsData(22,i,54), i=1,3) / 0.5039020000E+01, 0.2587730000E+02, 0.3161230000E+00 /
  data (epsData(23,i,54), i=1,3) / 0.7321620000E+01, 0.2899470000E+02, 0.3431750000E+00 /
  data (epsData(24,i,54), i=1,3) / 0.9005620000E+01, 0.3317080000E+02, 0.4383840000E+00 /
  data (epsData(25,i,54), i=1,3) / 0.1014660000E+02, 0.3791700000E+02, 0.5947650000E+00 /
  data (epsData(26,i,54), i=1,3) / 0.1098570000E+02, 0.4363610000E+02, 0.8683350000E+00 /
  data (epsData(27,i,54), i=1,3) / 0.1148020000E+02, 0.4915940000E+02, 0.1304990000E+01 /
  data (epsData(28,i,54), i=1,3) / 0.1173150000E+02, 0.5352700000E+02, 0.1972960000E+01 /
  data (epsData(29,i,54), i=1,3) / 0.1198550000E+02, 0.5966950000E+02, 0.4191140000E+01 /
  data (epsData(30,i,54), i=1,3) / 0.1217650000E+02, 0.6339790000E+02, 0.1088620000E+02 /
  data (epsData(31,i,54), i=1,3) / 0.1231260000E+02, 0.6296910000E+02, 0.1529540000E+02 /
  data (epsData(32,i,54), i=1,3) / 0.1285140000E+02, 0.6327060000E+02, 0.2302420000E+02 /
  data (epsData(33,i,54), i=1,3) / 0.1332450000E+02, 0.6651030000E+02, 0.3174170000E+02 /
  data (epsData(34,i,54), i=1,3) / 0.1364800000E+02, 0.6547080000E+02, 0.4133890000E+02 /
  data (epsData(35,i,54), i=1,3) / 0.1439190000E+02, 0.5678080000E+02, 0.5759490000E+02 /
  data (epsData(36,i,54), i=1,3) / 0.1501940000E+02, 0.4505830000E+02, 0.6843230000E+02 /
  data (epsData(37,i,54), i=1,3) / 0.1552280000E+02, 0.3314730000E+02, 0.7267180000E+02 /
  data (epsData(38,i,54), i=1,3) / 0.1634190000E+02, 0.1588100000E+02, 0.7491380000E+02 /
  data (epsData(39,i,54), i=1,3) / 0.1680140000E+02, 0.6142820000E+01, 0.7275970000E+02 /
  data (epsData(40,i,54), i=1,3) / 0.1774330000E+02, -0.8327280000E+01, 0.6536750000E+02 /
  data (epsData(41,i,54), i=1,3) / 0.1863000000E+02, -0.1796770000E+02, 0.5591780000E+02 /
  data (epsData(42,i,54), i=1,3) / 0.1969130000E+02, -0.2418530000E+02, 0.4404440000E+02 /
  data (epsData(43,i,54), i=1,3) / 0.2065050000E+02, -0.2589720000E+02, 0.3440750000E+02 /
  data (epsData(44,i,54), i=1,3) / 0.2167770000E+02, -0.2525510000E+02, 0.2622170000E+02 /
  data (epsData(45,i,54), i=1,3) / 0.2288620000E+02, -0.2311030000E+02, 0.1925820000E+02 /
  data (epsData(46,i,54), i=1,3) / 0.2452450000E+02, -0.1976390000E+02, 0.1308520000E+02 /
  data (epsData(47,i,54), i=1,3) / 0.2617900000E+02, -0.1661600000E+02, 0.9128460000E+01 /
  data (epsData(48,i,54), i=1,3) / 0.2795620000E+02, -0.1377610000E+02, 0.6437230000E+01 /
  data (epsData(49,i,54), i=1,3) / 0.2987190000E+02, -0.1133360000E+02, 0.4557930000E+01 /
  data (epsData(50,i,54), i=1,3) / 0.3200760000E+02, -0.9137430000E+01, 0.3181580000E+01 /
  data (epsData(51,i,54), i=1,3) / 0.3419850000E+02, -0.7362110000E+01, 0.2300880000E+01 /
  data (epsData(52,i,54), i=1,3) / 0.3663490000E+02, -0.5825530000E+01, 0.1663460000E+01 /
  data (epsData(53,i,54), i=1,3) / 0.3916970000E+02, -0.4570460000E+01, 0.1229770000E+01 /
  data (epsData(54,i,54), i=1,3) / 0.4192480000E+02, -0.3491030000E+01, 0.9150100000E+00 /
  data (epsData(55,i,54), i=1,3) / 0.4490040000E+02, -0.2555170000E+01, 0.6840280000E+00 /
  data (epsData(56,i,54), i=1,3) / 0.4791300000E+02, -0.1771440000E+01, 0.5270410000E+00 /
  data (epsData(57,i,54), i=1,3) / 0.5092800000E+02, -0.1107290000E+01, 0.4179830000E+00 /
  data (epsData(58,i,54), i=1,3) / 0.5391280000E+02, -0.5242880000E+00, 0.3406640000E+00 /
  data (epsData(59,i,54), i=1,3) / 0.5683660000E+02, 0.4972120000E-02, 0.2844750000E+00 /
  data (epsData(60,i,54), i=1,3) / 0.6147240000E+02, 0.8307060000E+00, 0.2224070000E+00 /
  data (epsData(61,i,54), i=1,3) / 0.6449250000E+02, 0.1423020000E+01, 0.1955620000E+00 /
  data (epsData(62,i,54), i=1,3) / 0.6640820000E+02, 0.1871590000E+01, 0.1838690000E+00 /
  data (epsData(63,i,54), i=1,3) / 0.6713780000E+02, 0.2091430000E+01, 0.1869970000E+00 /
  data (epsData(64,i,54), i=1,3) / 0.6755750000E+02, 0.2244470000E+01, 0.2446900000E+00 /
  data (epsData(65,i,54), i=1,3) / 0.6831670000E+02, 0.2388920000E+01, 0.3196470000E+00 /
  data (epsData(66,i,54), i=1,3) / 0.6941330000E+02, 0.2694120000E+01, 0.3785030000E+00 /
  data (epsData(67,i,54), i=1,3) / 0.7116670000E+02, 0.3119400000E+01, 0.5496700000E+00 /
  data (epsData(68,i,54), i=1,3) / 0.7346740000E+02, 0.3868190000E+01, 0.7275140000E+00 /
  data (epsData(69,i,54), i=1,3) / 0.7516030000E+02, 0.4635470000E+01, 0.9645610000E+00 /
  data (epsData(70,i,54), i=1,3) / 0.7675050000E+02, 0.5558490000E+01, 0.1418710000E+01 /
  data (epsData(71,i,54), i=1,3) / 0.7831680000E+02, 0.6615580000E+01, 0.2265940000E+01 /
  data (epsData(72,i,54), i=1,3) / 0.7968010000E+02, 0.7545380000E+01, 0.3485600000E+01 /
  data (epsData(73,i,54), i=1,3) / 0.8101580000E+02, 0.8219670000E+01, 0.5257310000E+01 /
  data (epsData(74,i,54), i=1,3) / 0.8216990000E+02, 0.8294290000E+01, 0.7265990000E+01 /
  data (epsData(75,i,54), i=1,3) / 0.8351690000E+02, 0.7404420000E+01, 0.9762250000E+01 /
  data (epsData(76,i,54), i=1,3) / 0.8464350000E+02, 0.5836320000E+01, 0.1153020000E+02 /
  data (epsData(77,i,54), i=1,3) / 0.8581360000E+02, 0.3677960000E+01, 0.1267160000E+02 /
  data (epsData(78,i,54), i=1,3) / 0.8718190000E+02, 0.1030330000E+01, 0.1301280000E+02 /
  data (epsData(79,i,54), i=1,3) / 0.8844450000E+02, -0.1147150000E+01, 0.1257650000E+02 /
  data (epsData(80,i,54), i=1,3) / 0.8991700000E+02, -0.3154870000E+01, 0.1146350000E+02 /
  data (epsData(81,i,54), i=1,3) / 0.9161770000E+02, -0.4712060000E+01, 0.9820870000E+01 /
  data (epsData(82,i,54), i=1,3) / 0.9356710000E+02, -0.5589110000E+01, 0.7905220000E+01 /
  data (epsData(83,i,54), i=1,3) / 0.9559870000E+02, -0.5825720000E+01, 0.6213660000E+01 /
  data (epsData(84,i,54), i=1,3) / 0.9810590000E+02, -0.5644230000E+01, 0.4630580000E+01 /
  data (epsData(85,i,54), i=1,3) / 0.1013500000E+03, -0.5134900000E+01, 0.3257700000E+01 /
  data (epsData(86,i,54), i=1,3) / 0.1052070000E+03, -0.4495800000E+01, 0.2239150000E+01 /
  data (epsData(87,i,54), i=1,3) / 0.1095170000E+03, -0.3875190000E+01, 0.1530400000E+01 /
  data (epsData(88,i,54), i=1,3) / 0.1143200000E+03, -0.3302900000E+01, 0.1032410000E+01 /
  data (epsData(89,i,54), i=1,3) / 0.1196560000E+03, -0.2791390000E+01, 0.6838770000E+00 /
  data (epsData(90,i,54), i=1,3) / 0.1255680000E+03, -0.2338800000E+01, 0.4427440000E+00 /
  data (epsData(91,i,54), i=1,3) / 0.1321000000E+03, -0.1944250000E+01, 0.2812180000E+00 /
  data (epsData(92,i,54), i=1,3) / 0.1393010000E+03, -0.1597690000E+01, 0.1800740000E+00 /
  data (epsData(93,i,54), i=1,3) / 0.1452790000E+03, -0.1348640000E+01, 0.1516170000E+00 /
  data (epsData(94,i,54), i=1,3) / 0.1492160000E+03, -0.1249980000E+01, 0.1640490000E+00 /
  data (epsData(95,i,54), i=1,3) / 0.1617000000E+03, -0.9884670000E+00, 0.1258020000E+00 /
  data (epsData(96,i,54), i=1,3) / 0.1741340000E+03, -0.7948470000E+00, 0.1082990000E+00 /
  data (epsData(97,i,54), i=1,3) / 0.1893250000E+03, -0.6294110000E+00, 0.1022260000E+00 /
  data (epsData(98,i,54), i=1,3) / 0.2072740000E+03, -0.4934000000E+00, 0.1011710000E+00 /
  data (epsData(99,i,54), i=1,3) / 0.2193080000E+03, -0.4278860000E+00, 0.1082870000E+00 /
  data (epsData(100,i,54), i=1,3) / 0.2414130000E+03, -0.3496530000E+00, 0.1061740000E+00 /
  data (epsData(101,i,54), i=1,3) / 0.2692680000E+03, -0.2799890000E+00, 0.9796990000E-01 /
  data (epsData(102,i,54), i=1,3) / 0.3069530000E+03, -0.2171280000E+00, 0.8481910000E-01 /
  data (epsData(103,i,54), i=1,3) / 0.3525250000E+03, -0.1665650000E+00, 0.6900120000E-01 /
  data (epsData(104,i,54), i=1,3) / 0.4040240000E+03, -0.1263600000E+00, 0.5401690000E-01 /
  data (epsData(105,i,54), i=1,3) / 0.4606970000E+03, -0.9360560000E-01, 0.4131000000E-01 /
  data (epsData(106,i,54), i=1,3) / 0.5260310000E+03, -0.6364420000E-01, 0.3067890000E-01 /
  data (epsData(107,i,54), i=1,3) / 0.5975610000E+03, -0.3296690000E-01, 0.2262070000E-01 /
  data (epsData(108,i,54), i=1,3) / 0.6240360000E+03, -0.1919280000E-01, 0.2028490000E-01 /
  data (epsData(109,i,54), i=1,3) / 0.6398870000E+03, -0.8861310000E-02, 0.1907010000E-01 /
  data (epsData(110,i,54), i=1,3) / 0.6498500000E+03, -0.7311760000E-03, 0.1834210000E-01 /
  data (epsData(111,i,54), i=1,3) / 0.6560380000E+03, 0.5546050000E-02, 0.1790310000E-01 /
  data (epsData(112,i,54), i=1,3) / 0.6607070000E+03, 0.1076260000E-01, 0.1783910000E-01 /
  data (epsData(113,i,54), i=1,3) / 0.6669720000E+03, 0.1953610000E-01, 0.1751600000E-01 /
  data (epsData(114,i,54), i=1,3) / 0.6724890000E+03, 0.2964190000E-01, 0.1749390000E-01 /
  data (epsData(115,i,54), i=1,3) / 0.6764340000E+03, 0.3948160000E-01, 0.1772110000E-01 /
  data (epsData(116,i,54), i=1,3) / 0.6804410000E+03, 0.5245110000E-01, 0.1885430000E-01 /
  data (epsData(117,i,54), i=1,3) / 0.6831620000E+03, 0.6505580000E-01, 0.2047370000E-01 /
  data (epsData(118,i,54), i=1,3) / 0.6853270000E+03, 0.7886090000E-01, 0.2407200000E-01 /
  data (epsData(119,i,54), i=1,3) / 0.6871770000E+03, 0.9379160000E-01, 0.3130900000E-01 /
  data (epsData(120,i,54), i=1,3) / 0.6891360000E+03, 0.1113140000E+00, 0.4706630000E-01 /
  data (epsData(121,i,54), i=1,3) / 0.6906190000E+03, 0.1215790000E+00, 0.6703310000E-01 /
  data (epsData(122,i,54), i=1,3) / 0.6923340000E+03, 0.1228800000E+00, 0.9584700000E-01 /
  data (epsData(123,i,54), i=1,3) / 0.6939800000E+03, 0.1118750000E+00, 0.1204050000E+00 /
  data (epsData(124,i,54), i=1,3) / 0.6962500000E+03, 0.8882650000E-01, 0.1395110000E+00 /
  data (epsData(125,i,54), i=1,3) / 0.6984860000E+03, 0.6870540000E-01, 0.1455480000E+00 /
  data (epsData(126,i,54), i=1,3) / 0.7011500000E+03, 0.5626490000E-01, 0.1463560000E+00 /
  data (epsData(127,i,54), i=1,3) / 0.7035750000E+03, 0.5146750000E-01, 0.1550740000E+00 /
  data (epsData(128,i,54), i=1,3) / 0.7059150000E+03, 0.3764100000E-01, 0.1695000000E+00 /
  data (epsData(129,i,54), i=1,3) / 0.7084290000E+03, 0.1346880000E-01, 0.1759810000E+00 /
  data (epsData(130,i,54), i=1,3) / 0.7109450000E+03, -0.9443980000E-02, 0.1717910000E+00 /
  data (epsData(131,i,54), i=1,3) / 0.7137960000E+03, -0.2776390000E-01, 0.1591230000E+00 /
  data (epsData(132,i,54), i=1,3) / 0.7176060000E+03, -0.3942920000E-01, 0.1414820000E+00 /
  data (epsData(133,i,54), i=1,3) / 0.7223230000E+03, -0.4414440000E-01, 0.1257800000E+00 /
  data (epsData(134,i,54), i=1,3) / 0.7311450000E+03, -0.4702260000E-01, 0.1093220000E+00 /
  data (epsData(135,i,54), i=1,3) / 0.7481470000E+03, -0.4977120000E-01, 0.9223850000E-01 /
  data (epsData(136,i,54), i=1,3) / 0.7750600000E+03, -0.5086940000E-01, 0.7736590000E-01 /
  data (epsData(137,i,54), i=1,3) / 0.8180710000E+03, -0.5072190000E-01, 0.6297670000E-01 /
  data (epsData(138,i,54), i=1,3) / 0.8844380000E+03, -0.4776540000E-01, 0.4860710000E-01 /
  data (epsData(139,i,54), i=1,3) / 0.9290100000E+03, -0.4314820000E-01, 0.4170460000E-01 /
  data (epsData(140,i,54), i=1,3) / 0.9410660000E+03, -0.4009890000E-01, 0.4145550000E-01 /
  data (epsData(141,i,54), i=1,3) / 0.9478500000E+03, -0.4060520000E-01, 0.4344060000E-01 /
  data (epsData(142,i,54), i=1,3) / 0.9702010000E+03, -0.4212000000E-01, 0.4092400000E-01 /
  data (epsData(143,i,54), i=1,3) / 0.1002040000E+04, -0.4033400000E-01, 0.3770350000E-01 /
  data (epsData(144,i,54), i=1,3) / 0.1015980000E+04, -0.4088290000E-01, 0.3745560000E-01 /
  data (epsData(145,i,54), i=1,3) / 0.1099460000E+04, -0.3887730000E-01, 0.2964170000E-01 /
  data (epsData(146,i,54), i=1,3) / 0.1156870000E+04, -0.3617340000E-01, 0.2616110000E-01 /
  data (epsData(147,i,54), i=1,3) / 0.1284130000E+04, -0.3241880000E-01, 0.1908490000E-01 /
  data (epsData(148,i,54), i=1,3) / 0.1447340000E+04, -0.2703750000E-01, 0.1300460000E-01 /
  data (epsData(149,i,54), i=1,3) / 0.1616730000E+04, -0.2244000000E-01, 0.9065680000E-02 /
  data (epsData(150,i,54), i=1,3) / 0.1801190000E+04, -0.1845360000E-01, 0.6319950000E-02 /
  data (epsData(151,i,54), i=1,3) / 0.2000010000E+04, -0.1515750000E-01, 0.4443530000E-02 /
  data (epsData(152,i,54), i=1,3) / 0.2219190000E+04, -0.1238390000E-01, 0.3109280000E-02 /
  data (epsData(153,i,54), i=1,3) / 0.2459440000E+04, -0.1008710000E-01, 0.2186440000E-02 /
  data (epsData(154,i,54), i=1,3) / 0.2729850000E+04, -0.8170910000E-02, 0.1523720000E-02 /
  data (epsData(155,i,54), i=1,3) / 0.3028810000E+04, -0.6596650000E-02, 0.1058280000E-02 /
  data (epsData(156,i,54), i=1,3) / 0.3360560000E+04, -0.5302230000E-02, 0.7332240000E-03 /
  data (epsData(157,i,54), i=1,3) / 0.3732360000E+04, -0.4226850000E-02, 0.5050010000E-03 /
  data (epsData(158,i,54), i=1,3) / 0.4159290000E+04, -0.3301790000E-02, 0.3429230000E-03 /
  data (epsData(159,i,54), i=1,3) / 0.4684330000E+04, -0.2329810000E-02, 0.2236530000E-03 /
  data (epsData(160,i,54), i=1,3) / 0.4757070000E+04, -0.2089210000E-02, 0.2115610000E-03 /
  data (epsData(161,i,54), i=1,3) / 0.4777030000E+04, -0.1930660000E-02, 0.2241160000E-03 /
  data (epsData(162,i,54), i=1,3) / 0.4783700000E+04, -0.1807910000E-02, 0.2475270000E-03 /
  data (epsData(163,i,54), i=1,3) / 0.4787030000E+04, -0.1701440000E-02, 0.3250880000E-03 /
  data (epsData(164,i,54), i=1,3) / 0.4789540000E+04, -0.1671190000E-02, 0.4716100000E-03 /
  data (epsData(165,i,54), i=1,3) / 0.4792200000E+04, -0.1752730000E-02, 0.5944330000E-03 /
  data (epsData(166,i,54), i=1,3) / 0.4795830000E+04, -0.1875760000E-02, 0.6207740000E-03 /
  data (epsData(167,i,54), i=1,3) / 0.4805270000E+04, -0.1966200000E-02, 0.5967280000E-03 /
  data (epsData(168,i,54), i=1,3) / 0.4833580000E+04, -0.2063880000E-02, 0.6011780000E-03 /
  data (epsData(169,i,54), i=1,3) / 0.4927950000E+04, -0.2116710000E-02, 0.5727150000E-03 /
  data (epsData(170,i,54), i=1,3) / 0.5070160000E+04, -0.2000740000E-02, 0.5144980000E-03 /
  data (epsData(171,i,54), i=1,3) / 0.5102100000E+04, -0.1888880000E-02, 0.5162710000E-03 /
  data (epsData(172,i,54), i=1,3) / 0.5108910000E+04, -0.1809290000E-02, 0.5762910000E-03 /
  data (epsData(173,i,54), i=1,3) / 0.5113380000E+04, -0.1845270000E-02, 0.6695590000E-03 /
  data (epsData(174,i,54), i=1,3) / 0.5121720000E+04, -0.1922110000E-02, 0.6699390000E-03 /
  data (epsData(175,i,54), i=1,3) / 0.5205450000E+04, -0.1993620000E-02, 0.6419380000E-03 /
  data (epsData(176,i,54), i=1,3) / 0.5421140000E+04, -0.1904640000E-02, 0.5548500000E-03 /
  data (epsData(177,i,54), i=1,3) / 0.5451930000E+04, -0.1833580000E-02, 0.5621770000E-03 /
  data (epsData(178,i,54), i=1,3) / 0.5463780000E+04, -0.1852210000E-02, 0.6143930000E-03 /
  data (epsData(179,i,54), i=1,3) / 0.5566240000E+04, -0.1880620000E-02, 0.5821330000E-03 /
  data (epsData(180,i,54), i=1,3) / 0.6171140000E+04, -0.1650430000E-02, 0.4044100000E-03 /
  data (epsData(181,i,54), i=1,3) / 0.6983230000E+04, -0.1338340000E-02, 0.2624000000E-03 /
  data (epsData(182,i,54), i=1,3) / 0.7823750000E+04, -0.1085450000E-02, 0.1741610000E-03 /
  data (epsData(183,i,54), i=1,3) / 0.8745140000E+04, -0.8767650000E-03, 0.1162390000E-03 /
  data (epsData(184,i,54), i=1,3) / 0.9744490000E+04, -0.7093110000E-03, 0.7820570000E-04 /
  data (epsData(185,i,54), i=1,3) / 0.1085980000E+05, -0.5718110000E-03, 0.5241240000E-04 /
  data (epsData(186,i,54), i=1,3) / 0.1210070000E+05, -0.4602800000E-03, 0.3503710000E-04 /
  data (epsData(187,i,54), i=1,3) / 0.1348960000E+05, -0.3697220000E-03, 0.2331560000E-04 /
  data (epsData(188,i,54), i=1,3) / 0.1503900000E+05, -0.2966810000E-03, 0.1548620000E-04 /
  data (epsData(189,i,54), i=1,3) / 0.1677980000E+05, -0.2375710000E-03, 0.1022730000E-04 /
  data (epsData(190,i,54), i=1,3) / 0.1873370000E+05, -0.1899270000E-03, 0.6729030000E-05 /
  data (epsData(191,i,54), i=1,3) / 0.2091680000E+05, -0.1517570000E-03, 0.4415310000E-05 /
  data (epsData(192,i,54), i=1,3) / 0.2336980000E+05, -0.1210320000E-03, 0.2884860000E-05 /
  data (epsData(193,i,54), i=1,3) / 0.2612510000E+05, -0.9632340000E-04, 0.1877240000E-05 /
  data (epsData(194,i,54), i=1,3) / 0.2919670000E+05, -0.7652740000E-04, 0.1220630000E-05 /
  data (epsData(195,i,54), i=1,3) / 0.2920000000E+05, -0.7382550000E-04, 0.1220100000E-05 /
  data (epsData(196,i,54), i=1,3) / 0.2920000000E+05, -0.7650890000E-04, 0.1220090000E-05 /
  data (epsData(197,i,54), i=1,3) / 0.3305390000E+05, -0.5844740000E-04, 0.7533280000E-06 /
  data (epsData(198,i,54), i=1,3) / 0.3440800000E+05, -0.5167950000E-04, 0.6440760000E-06 /
  data (epsData(199,i,54), i=1,3) / 0.3453650000E+05, -0.4930780000E-04, 0.6347240000E-06 /
  data (epsData(200,i,54), i=1,3) / 0.3456080000E+05, -0.4792970000E-04, 0.1974310000E-05 /
  data (epsData(201,i,54), i=1,3) / 0.3456100000E+05, -0.4523830000E-04, 0.2004470000E-05 /
  data (epsData(202,i,54), i=1,3) / 0.3456100000E+05, -0.4791930000E-04, 0.2009860000E-05 /
  data (epsData(203,i,54), i=1,3) / 0.3457980000E+05, -0.4883090000E-04, 0.3543180000E-05 /
  data (epsData(204,i,54), i=1,3) / 0.3467500000E+05, -0.5052720000E-04, 0.3846190000E-05 /
  data (epsData(205,i,54), i=1,3) / 0.3541770000E+05, -0.5059280000E-04, 0.3617650000E-05 /
  data (epsData(206,i,54), i=1,3) / 0.4193730000E+05, -0.3774710000E-04, 0.1964750000E-05 /
  data (epsData(207,i,54), i=1,3) / 0.4716620000E+05, -0.3008560000E-04, 0.1265930000E-05 /
  data (epsData(208,i,54), i=1,3) / 0.5289000000E+05, -0.2401980000E-04, 0.8203040000E-06 /
  data (epsData(209,i,54), i=1,3) / 0.5925200000E+05, -0.1917500000E-04, 0.5307720000E-06 /
  data (epsData(210,i,54), i=1,3) / 0.6635570000E+05, -0.1530140000E-04, 0.3427270000E-06 /
  data (epsData(211,i,54), i=1,3) / 0.7431220000E+05, -0.1220520000E-04, 0.2200890000E-06 /
  data (epsData(212,i,54), i=1,3) / 0.8321320000E+05, -0.9736200000E-05, 0.1405840000E-06 /
  data (epsData(213,i,54), i=1,3) / 0.9312650000E+05, -0.7783330000E-05, 0.8956630000E-07 /
  data (epsData(214,i,54), i=1,3) / 0.9995230000E+05, -0.6852250000E-05, 0.6730640000E-07 /
  data (epsData(215,i,54), i=1,3) / 0.1000000000E+06, -0.6951080000E-05, 0.6717400000E-07 /
  data (epsData(1,i,55), i=1,3) / 0.2506580000E-02, -0.1753880000E+04, 0.4258530000E+04 /
  data (epsData(2,i,55), i=1,3) / 0.4693440000E-02, -0.1413800000E+04, 0.1413210000E+04 /
  data (epsData(3,i,55), i=1,3) / 0.7560590000E-02, -0.6437430000E+03, 0.6449370000E+03 /
  data (epsData(4,i,55), i=1,3) / 0.1110800000E-01, -0.3092950000E+03, 0.3562950000E+03 /
  data (epsData(5,i,55), i=1,3) / 0.1533570000E-01, -0.1522850000E+03, 0.2228880000E+03 /
  data (epsData(6,i,55), i=1,3) / 0.2024370000E-01, -0.7113410000E+02, 0.1518700000E+03 /
  data (epsData(7,i,55), i=1,3) / 0.2583200000E-01, -0.2573670000E+02, 0.1100220000E+03 /
  data (epsData(8,i,55), i=1,3) / 0.3210060000E-01, 0.1320800000E+01, 0.8341660000E+02 /
  data (epsData(9,i,55), i=1,3) / 0.3904940000E-01, 0.1829660000E+02, 0.6548610000E+02 /
  data (epsData(10,i,55), i=1,3) / 0.4667860000E-01, 0.2940780000E+02, 0.5283360000E+02 /
  data (epsData(11,i,55), i=1,3) / 0.5498800000E-01, 0.3694450000E+02, 0.4356970000E+02 /
  data (epsData(12,i,55), i=1,3) / 0.6397770000E-01, 0.4221670000E+02, 0.3657920000E+02 /
  data (epsData(13,i,55), i=1,3) / 0.7364770000E-01, 0.4600880000E+02, 0.3117050000E+02 /
  data (epsData(14,i,55), i=1,3) / 0.8399800000E-01, 0.4881370000E+02, 0.2689660000E+02 /
  data (epsData(15,i,55), i=1,3) / 0.9502860000E-01, 0.5096870000E+02, 0.2345830000E+02 /
  data (epsData(16,i,55), i=1,3) / 0.1191310000E+00, 0.5383360000E+02, 0.1865670000E+02 /
  data (epsData(17,i,55), i=1,3) / 0.1603860000E+00, 0.5589780000E+02, 0.1389760000E+02 /
  data (epsData(18,i,55), i=1,3) / 0.2249170000E+00, 0.5722980000E+02, 0.9983600000E+01 /
  data (epsData(19,i,55), i=1,3) / 0.3866320000E+00, 0.5793310000E+02, 0.6174920000E+01 /
  data (epsData(20,i,55), i=1,3) / 0.8406760000E+00, 0.5878280000E+02, 0.3143600000E+01 /
  data (epsData(21,i,55), i=1,3) / 0.1700900000E+01, 0.6213580000E+02, 0.2178190000E+01 /
  data (epsData(22,i,55), i=1,3) / 0.2327150000E+01, 0.6761550000E+02, 0.2354620000E+01 /
  data (epsData(23,i,55), i=1,3) / 0.2737710000E+01, 0.7454060000E+02, 0.3197360000E+01 /
  data (epsData(24,i,55), i=1,3) / 0.2923860000E+01, 0.7972150000E+02, 0.3749100000E+01 /
  data (epsData(25,i,55), i=1,3) / 0.3116140000E+01, 0.8748410000E+02, 0.5829200000E+01 /
  data (epsData(26,i,55), i=1,3) / 0.3181590000E+01, 0.9165220000E+02, 0.6418000000E+01 /
  data (epsData(27,i,55), i=1,3) / 0.3247730000E+01, 0.9694550000E+02, 0.8858630000E+01 /
  data (epsData(28,i,55), i=1,3) / 0.3450210000E+01, 0.1098380000E+03, 0.2379590000E+02 /
  data (epsData(29,i,55), i=1,3) / 0.3519060000E+01, 0.1187350000E+03, 0.2808670000E+02 /
  data (epsData(30,i,55), i=1,3) / 0.3588600000E+01, 0.1237700000E+03, 0.5089250000E+02 /
  data (epsData(31,i,55), i=1,3) / 0.3658810000E+01, 0.1179510000E+03, 0.6963100000E+02 /
  data (epsData(32,i,55), i=1,3) / 0.3729710000E+01, 0.1070530000E+03, 0.8371240000E+02 /
  data (epsData(33,i,55), i=1,3) / 0.3801290000E+01, 0.9098910000E+02, 0.9682420000E+02 /
  data (epsData(34,i,55), i=1,3) / 0.3873540000E+01, 0.7310330000E+02, 0.9776660000E+02 /
  data (epsData(35,i,55), i=1,3) / 0.3946480000E+01, 0.5953890000E+02, 0.9387810000E+02 /
  data (epsData(36,i,55), i=1,3) / 0.4094390000E+01, 0.4081800000E+02, 0.8663800000E+02 /
  data (epsData(37,i,55), i=1,3) / 0.4169370000E+01, 0.3315400000E+02, 0.8158480000E+02 /
  data (epsData(38,i,55), i=1,3) / 0.4321360000E+01, 0.2278700000E+02, 0.7028280000E+02 /
  data (epsData(39,i,55), i=1,3) / 0.4476080000E+01, 0.1666000000E+02, 0.6001220000E+02 /
  data (epsData(40,i,55), i=1,3) / 0.4633520000E+01, 0.1333110000E+02, 0.5056300000E+02 /
  data (epsData(41,i,55), i=1,3) / 0.4793680000E+01, 0.1218940000E+02, 0.4240130000E+02 /
  data (epsData(42,i,55), i=1,3) / 0.4956560000E+01, 0.1218250000E+02, 0.3545810000E+02 /
  data (epsData(43,i,55), i=1,3) / 0.5122160000E+01, 0.1323040000E+02, 0.2984520000E+02 /
  data (epsData(44,i,55), i=1,3) / 0.5375660000E+01, 0.1509200000E+02, 0.2316390000E+02 /
  data (epsData(45,i,55), i=1,3) / 0.5811770000E+01, 0.1876880000E+02, 0.1462150000E+02 /
  data (epsData(46,i,55), i=1,3) / 0.6172910000E+01, 0.2285080000E+02, 0.9394620000E+01 /
  data (epsData(47,i,55), i=1,3) / 0.6544930000E+01, 0.2736070000E+02, 0.6050020000E+01 /
  data (epsData(48,i,55), i=1,3) / 0.7025260000E+01, 0.3260490000E+02, 0.3794330000E+01 /
  data (epsData(49,i,55), i=1,3) / 0.7624100000E+01, 0.3837250000E+02, 0.2580820000E+01 /
  data (epsData(50,i,55), i=1,3) / 0.9453820000E+01, 0.5360210000E+02, 0.2029390000E+01 /
  data (epsData(51,i,55), i=1,3) / 0.1038290000E+02, 0.6456140000E+02, 0.2162730000E+01 /
  data (epsData(52,i,55), i=1,3) / 0.1098570000E+02, 0.7517440000E+02, 0.2592180000E+01 /
  data (epsData(53,i,55), i=1,3) / 0.1135550000E+02, 0.8468310000E+02, 0.3202640000E+01 /
  data (epsData(54,i,55), i=1,3) / 0.1160550000E+02, 0.9377520000E+02, 0.4209420000E+01 /
  data (epsData(55,i,55), i=1,3) / 0.1173150000E+02, 0.9985750000E+02, 0.5330010000E+01 /
  data (epsData(56,i,55), i=1,3) / 0.1198550000E+02, 0.1157860000E+03, 0.1084420000E+02 /
  data (epsData(57,i,55), i=1,3) / 0.1214900000E+02, 0.1258920000E+03, 0.2534240000E+02 /
  data (epsData(58,i,55), i=1,3) / 0.1231260000E+02, 0.1272590000E+03, 0.4006020000E+02 /
  data (epsData(59,i,55), i=1,3) / 0.1288370000E+02, 0.1264460000E+03, 0.7550340000E+02 /
  data (epsData(60,i,55), i=1,3) / 0.1328400000E+02, 0.1167510000E+03, 0.1029800000E+03 /
  data (epsData(61,i,55), i=1,3) / 0.1371940000E+02, 0.9581760000E+02, 0.1270280000E+03 /
  data (epsData(62,i,55), i=1,3) / 0.1420850000E+02, 0.6458760000E+02, 0.1397710000E+03 /
  data (epsData(63,i,55), i=1,3) / 0.1450610000E+02, 0.4640560000E+02, 0.1404770000E+03 /
  data (epsData(64,i,55), i=1,3) / 0.1488880000E+02, 0.3161560000E+02, 0.1378070000E+03 /
  data (epsData(65,i,55), i=1,3) / 0.1546570000E+02, 0.8973220000E+01, 0.1389620000E+03 /
  data (epsData(66,i,55), i=1,3) / 0.1596330000E+02, -0.1469410000E+02, 0.1337030000E+03 /
  data (epsData(67,i,55), i=1,3) / 0.1632280000E+02, -0.3007000000E+02, 0.1243930000E+03 /
  data (epsData(68,i,55), i=1,3) / 0.1677450000E+02, -0.4283270000E+02, 0.1084900000E+03 /
  data (epsData(69,i,55), i=1,3) / 0.1740040000E+02, -0.5091940000E+02, 0.8865360000E+02 /
  data (epsData(70,i,55), i=1,3) / 0.1809430000E+02, -0.5406130000E+02, 0.6948640000E+02 /
  data (epsData(71,i,55), i=1,3) / 0.1909800000E+02, -0.5174830000E+02, 0.4968630000E+02 /
  data (epsData(72,i,55), i=1,3) / 0.2003720000E+02, -0.4702370000E+02, 0.3658400000E+02 /
  data (epsData(73,i,55), i=1,3) / 0.2125320000E+02, -0.4051680000E+02, 0.2570820000E+02 /
  data (epsData(74,i,55), i=1,3) / 0.2258420000E+02, -0.3428730000E+02, 0.1821450000E+02 /
  data (epsData(75,i,55), i=1,3) / 0.2400750000E+02, -0.2889360000E+02, 0.1318680000E+02 /
  data (epsData(76,i,55), i=1,3) / 0.2570550000E+02, -0.2381800000E+02, 0.9215450000E+01 /
  data (epsData(77,i,55), i=1,3) / 0.2751680000E+02, -0.1960740000E+02, 0.6558520000E+01 /
  data (epsData(78,i,55), i=1,3) / 0.2946240000E+02, -0.1611310000E+02, 0.4757180000E+01 /
  data (epsData(79,i,55), i=1,3) / 0.3173720000E+02, -0.1303200000E+02, 0.3396820000E+01 /
  data (epsData(80,i,55), i=1,3) / 0.3421580000E+02, -0.1049510000E+02, 0.2433570000E+01 /
  data (epsData(81,i,55), i=1,3) / 0.3699450000E+02, -0.8335540000E+01, 0.1730160000E+01 /
  data (epsData(82,i,55), i=1,3) / 0.3991970000E+02, -0.6587840000E+01, 0.1243800000E+01 /
  data (epsData(83,i,55), i=1,3) / 0.4311700000E+02, -0.5101600000E+01, 0.8994340000E+00 /
  data (epsData(84,i,55), i=1,3) / 0.4658650000E+02, -0.3837320000E+01, 0.6600900000E+00 /
  data (epsData(85,i,55), i=1,3) / 0.5013450000E+02, -0.2802170000E+01, 0.5018660000E+00 /
  data (epsData(86,i,55), i=1,3) / 0.5392810000E+02, -0.1888080000E+01, 0.3898980000E+00 /
  data (epsData(87,i,55), i=1,3) / 0.5774860000E+02, -0.1093450000E+01, 0.3168510000E+00 /
  data (epsData(88,i,55), i=1,3) / 0.6178940000E+02, -0.3338380000E+00, 0.2619970000E+00 /
  data (epsData(89,i,55), i=1,3) / 0.6508470000E+02, 0.2750680000E+00, 0.2278810000E+00 /
  data (epsData(90,i,55), i=1,3) / 0.6801540000E+02, 0.8450950000E+00, 0.2019850000E+00 /
  data (epsData(91,i,55), i=1,3) / 0.7104400000E+02, 0.1505570000E+01, 0.1821290000E+00 /
  data (epsData(92,i,55), i=1,3) / 0.7364270000E+02, 0.2184030000E+01, 0.1694610000E+00 /
  data (epsData(93,i,55), i=1,3) / 0.7577060000E+02, 0.2882540000E+01, 0.1640790000E+00 /
  data (epsData(94,i,55), i=1,3) / 0.7712270000E+02, 0.3464040000E+01, 0.1690350000E+00 /
  data (epsData(95,i,55), i=1,3) / 0.7772990000E+02, 0.3766830000E+01, 0.2530210000E+00 /
  data (epsData(96,i,55), i=1,3) / 0.7921470000E+02, 0.4515110000E+01, 0.3516170000E+00 /
  data (epsData(97,i,55), i=1,3) / 0.8098400000E+02, 0.5692970000E+01, 0.5708950000E+00 /
  data (epsData(98,i,55), i=1,3) / 0.8231570000E+02, 0.6940670000E+01, 0.8406990000E+00 /
  data (epsData(99,i,55), i=1,3) / 0.8333880000E+02, 0.8272720000E+01, 0.1244630000E+01 /
  data (epsData(100,i,55), i=1,3) / 0.8422050000E+02, 0.9759300000E+01, 0.1910010000E+01 /
  data (epsData(101,i,55), i=1,3) / 0.8503000000E+02, 0.1146440000E+02, 0.3020220000E+01 /
  data (epsData(102,i,55), i=1,3) / 0.8581680000E+02, 0.1336380000E+02, 0.4897320000E+01 /
  data (epsData(103,i,55), i=1,3) / 0.8646680000E+02, 0.1489030000E+02, 0.7439130000E+01 /
  data (epsData(104,i,55), i=1,3) / 0.8705050000E+02, 0.1577060000E+02, 0.1072700000E+02 /
  data (epsData(105,i,55), i=1,3) / 0.8755560000E+02, 0.1556650000E+02, 0.1435730000E+02 /
  data (epsData(106,i,55), i=1,3) / 0.8797190000E+02, 0.1421070000E+02, 0.1756500000E+02 /
  data (epsData(107,i,55), i=1,3) / 0.8850760000E+02, 0.1103970000E+02, 0.2079280000E+02 /
  data (epsData(108,i,55), i=1,3) / 0.8917300000E+02, 0.6511820000E+01, 0.2223050000E+02 /
  data (epsData(109,i,55), i=1,3) / 0.9021690000E+02, 0.4644120000E+00, 0.2285400000E+02 /
  data (epsData(110,i,55), i=1,3) / 0.9082060000E+02, -0.3366050000E+01, 0.2220240000E+02 /
  data (epsData(111,i,55), i=1,3) / 0.9144140000E+02, -0.6665880000E+01, 0.1987990000E+02 /
  data (epsData(112,i,55), i=1,3) / 0.9220870000E+02, -0.8811780000E+01, 0.1646670000E+02 /
  data (epsData(113,i,55), i=1,3) / 0.9300060000E+02, -0.9485590000E+01, 0.1342380000E+02 /
  data (epsData(114,i,55), i=1,3) / 0.9395540000E+02, -0.9404190000E+01, 0.1077330000E+02 /
  data (epsData(115,i,55), i=1,3) / 0.9537710000E+02, -0.8840850000E+01, 0.8187300000E+01 /
  data (epsData(116,i,55), i=1,3) / 0.9732720000E+02, -0.7981360000E+01, 0.5988790000E+01 /
  data (epsData(117,i,55), i=1,3) / 0.9972010000E+02, -0.7057350000E+01, 0.4336170000E+01 /
  data (epsData(118,i,55), i=1,3) / 0.1026170000E+03, -0.6146870000E+01, 0.3080090000E+01 /
  data (epsData(119,i,55), i=1,3) / 0.1058990000E+03, -0.5311690000E+01, 0.2180110000E+01 /
  data (epsData(120,i,55), i=1,3) / 0.1098040000E+03, -0.4525520000E+01, 0.1514390000E+01 /
  data (epsData(121,i,55), i=1,3) / 0.1141970000E+03, -0.3848180000E+01, 0.1050330000E+01 /
  data (epsData(122,i,55), i=1,3) / 0.1193520000E+03, -0.3242390000E+01, 0.7097200000E+00 /
  data (epsData(123,i,55), i=1,3) / 0.1253640000E+03, -0.2706700000E+01, 0.4639910000E+00 /
  data (epsData(124,i,55), i=1,3) / 0.1323350000E+03, -0.2235150000E+01, 0.2897550000E+00 /
  data (epsData(125,i,55), i=1,3) / 0.1400870000E+03, -0.1834060000E+01, 0.1806600000E+00 /
  data (epsData(126,i,55), i=1,3) / 0.1486790000E+03, -0.1497600000E+01, 0.1192000000E+00 /
  data (epsData(127,i,55), i=1,3) / 0.1588240000E+03, -0.1188930000E+01, 0.9150230000E-01 /
  data (epsData(128,i,55), i=1,3) / 0.1624840000E+03, -0.1103170000E+01, 0.1140520000E+00 /
  data (epsData(129,i,55), i=1,3) / 0.1736660000E+03, -0.9219050000E+00, 0.1167050000E+00 /
  data (epsData(130,i,55), i=1,3) / 0.1905500000E+03, -0.7262250000E+00, 0.1077440000E+00 /
  data (epsData(131,i,55), i=1,3) / 0.2094410000E+03, -0.5726300000E+00, 0.1051110000E+00 /
  data (epsData(132,i,55), i=1,3) / 0.2318310000E+03, -0.4455850000E+00, 0.1055820000E+00 /
  data (epsData(133,i,55), i=1,3) / 0.2486140000E+03, -0.3876760000E+00, 0.1085660000E+00 /
  data (epsData(134,i,55), i=1,3) / 0.2803880000E+03, -0.3060790000E+00, 0.9826360000E-01 /
  data (epsData(135,i,55), i=1,3) / 0.3211190000E+03, -0.2365410000E+00, 0.8388190000E-01 /
  data (epsData(136,i,55), i=1,3) / 0.3673050000E+03, -0.1833710000E+00, 0.6799880000E-01 /
  data (epsData(137,i,55), i=1,3) / 0.4200790000E+03, -0.1405030000E+00, 0.5326190000E-01 /
  data (epsData(138,i,55), i=1,3) / 0.4780590000E+03, -0.1058440000E+00, 0.4075960000E-01 /
  data (epsData(139,i,55), i=1,3) / 0.5444790000E+03, -0.7500500000E-01, 0.3041960000E-01 /
  data (epsData(140,i,55), i=1,3) / 0.6327360000E+03, -0.3871910000E-01, 0.2116600000E-01 /
  data (epsData(141,i,55), i=1,3) / 0.6635710000E+03, -0.2356920000E-01, 0.1882020000E-01 /
  data (epsData(142,i,55), i=1,3) / 0.6805880000E+03, -0.1285320000E-01, 0.1763790000E-01 /
  data (epsData(143,i,55), i=1,3) / 0.6907990000E+03, -0.4727830000E-02, 0.1696180000E-01 /
  data (epsData(144,i,55), i=1,3) / 0.6979350000E+03, 0.2261350000E-02, 0.1650540000E-01 /
  data (epsData(145,i,55), i=1,3) / 0.7035230000E+03, 0.8947870000E-02, 0.1615640000E-01 /
  data (epsData(146,i,55), i=1,3) / 0.7099510000E+03, 0.1843940000E-01, 0.1618770000E-01 /
  data (epsData(147,i,55), i=1,3) / 0.7148000000E+03, 0.2763530000E-01, 0.1605830000E-01 /
  data (epsData(148,i,55), i=1,3) / 0.7188600000E+03, 0.3773270000E-01, 0.1604350000E-01 /
  data (epsData(149,i,55), i=1,3) / 0.7229360000E+03, 0.5148060000E-01, 0.1651510000E-01 /
  data (epsData(150,i,55), i=1,3) / 0.7253910000E+03, 0.6281240000E-01, 0.1691600000E-01 /
  data (epsData(151,i,55), i=1,3) / 0.7279930000E+03, 0.7931030000E-01, 0.1839450000E-01 /
  data (epsData(152,i,55), i=1,3) / 0.7299340000E+03, 0.9687510000E-01, 0.2074140000E-01 /
  data (epsData(153,i,55), i=1,3) / 0.7314170000E+03, 0.1159800000E+00, 0.2489170000E-01 /
  data (epsData(154,i,55), i=1,3) / 0.7326170000E+03, 0.1372320000E+00, 0.3233000000E-01 /
  data (epsData(155,i,55), i=1,3) / 0.7335550000E+03, 0.1586910000E+00, 0.4390260000E-01 /
  data (epsData(156,i,55), i=1,3) / 0.7344590000E+03, 0.1832640000E+00, 0.6460720000E-01 /
  data (epsData(157,i,55), i=1,3) / 0.7351990000E+03, 0.2038190000E+00, 0.9330390000E-01 /
  data (epsData(158,i,55), i=1,3) / 0.7358600000E+03, 0.2168350000E+00, 0.1326030000E+00 /
  data (epsData(159,i,55), i=1,3) / 0.7364300000E+03, 0.2160830000E+00, 0.1767410000E+00 /
  data (epsData(160,i,55), i=1,3) / 0.7370170000E+03, 0.1942550000E+00, 0.2258230000E+00 /
  data (epsData(161,i,55), i=1,3) / 0.7376210000E+03, 0.1483900000E+00, 0.2628630000E+00 /
  data (epsData(162,i,55), i=1,3) / 0.7382410000E+03, 0.9286670000E-01, 0.2705030000E+00 /
  data (epsData(163,i,55), i=1,3) / 0.7387500000E+03, 0.5531920000E-01, 0.2604750000E+00 /
  data (epsData(164,i,55), i=1,3) / 0.7394020000E+03, 0.2310860000E-01, 0.2371830000E+00 /
  data (epsData(165,i,55), i=1,3) / 0.7400700000E+03, 0.5580490000E-02, 0.2121110000E+00 /
  data (epsData(166,i,55), i=1,3) / 0.7408270000E+03, -0.2802100000E-02, 0.1888710000E+00 /
  data (epsData(167,i,55), i=1,3) / 0.7418070000E+03, -0.5119170000E-02, 0.1662240000E+00 /
  data (epsData(168,i,55), i=1,3) / 0.7430160000E+03, -0.2048500000E-02, 0.1466850000E+00 /
  data (epsData(169,i,55), i=1,3) / 0.7443940000E+03, 0.6395150000E-02, 0.1319090000E+00 /
  data (epsData(170,i,55), i=1,3) / 0.7457300000E+03, 0.1885900000E-01, 0.1236850000E+00 /
  data (epsData(171,i,55), i=1,3) / 0.7468840000E+03, 0.3363650000E-01, 0.1223690000E+00 /
  data (epsData(172,i,55), i=1,3) / 0.7478500000E+03, 0.4924670000E-01, 0.1276420000E+00 /
  data (epsData(173,i,55), i=1,3) / 0.7487780000E+03, 0.6595890000E-01, 0.1424880000E+00 /
  data (epsData(174,i,55), i=1,3) / 0.7494270000E+03, 0.7573280000E-01, 0.1617510000E+00 /
  data (epsData(175,i,55), i=1,3) / 0.7499860000E+03, 0.7857850000E-01, 0.1854890000E+00 /
  data (epsData(176,i,55), i=1,3) / 0.7505630000E+03, 0.7048240000E-01, 0.2149690000E+00 /
  data (epsData(177,i,55), i=1,3) / 0.7510360000E+03, 0.5207480000E-01, 0.2371740000E+00 /
  data (epsData(178,i,55), i=1,3) / 0.7515210000E+03, 0.2542390000E-01, 0.2515470000E+00 /
  data (epsData(179,i,55), i=1,3) / 0.7521410000E+03, -0.1132370000E-01, 0.2512260000E+00 /
  data (epsData(180,i,55), i=1,3) / 0.7527790000E+03, -0.3985600000E-01, 0.2378310000E+00 /
  data (epsData(181,i,55), i=1,3) / 0.7537010000E+03, -0.6223060000E-01, 0.2120750000E+00 /
  data (epsData(182,i,55), i=1,3) / 0.7547940000E+03, -0.7145540000E-01, 0.1866960000E+00 /
  data (epsData(183,i,55), i=1,3) / 0.7563700000E+03, -0.7385520000E-01, 0.1625020000E+00 /
  data (epsData(184,i,55), i=1,3) / 0.7589660000E+03, -0.7174990000E-01, 0.1390590000E+00 /
  data (epsData(185,i,55), i=1,3) / 0.7634940000E+03, -0.6681510000E-01, 0.1178240000E+00 /
  data (epsData(186,i,55), i=1,3) / 0.7701010000E+03, -0.6140440000E-01, 0.1019990000E+00 /
  data (epsData(187,i,55), i=1,3) / 0.7802840000E+03, -0.5746150000E-01, 0.9036500000E-01 /
  data (epsData(188,i,55), i=1,3) / 0.8054570000E+03, -0.5533180000E-01, 0.7496890000E-01 /
  data (epsData(189,i,55), i=1,3) / 0.8589050000E+03, -0.5321190000E-01, 0.5805180000E-01 /
  data (epsData(190,i,55), i=1,3) / 0.9354220000E+03, -0.4880890000E-01, 0.4364700000E-01 /
  data (epsData(191,i,55), i=1,3) / 0.9862590000E+03, -0.4366660000E-01, 0.3700740000E-01 /
  data (epsData(192,i,55), i=1,3) / 0.9997390000E+03, -0.4050480000E-01, 0.3639290000E-01 /
  data (epsData(193,i,55), i=1,3) / 0.1004390000E+04, -0.3940850000E-01, 0.3865610000E-01 /
  data (epsData(194,i,55), i=1,3) / 0.1009760000E+04, -0.4194180000E-01, 0.3939980000E-01 /
  data (epsData(195,i,55), i=1,3) / 0.1025990000E+04, -0.4256900000E-01, 0.3684760000E-01 /
  data (epsData(196,i,55), i=1,3) / 0.1065310000E+04, -0.4049110000E-01, 0.3310370000E-01 /
  data (epsData(197,i,55), i=1,3) / 0.1073690000E+04, -0.3992470000E-01, 0.3393560000E-01 /
  data (epsData(198,i,55), i=1,3) / 0.1093990000E+04, -0.4066770000E-01, 0.3192920000E-01 /
  data (epsData(199,i,55), i=1,3) / 0.1193450000E+04, -0.3695060000E-01, 0.2456810000E-01 /
  data (epsData(200,i,55), i=1,3) / 0.1244540000E+04, -0.3533030000E-01, 0.2241830000E-01 /
  data (epsData(201,i,55), i=1,3) / 0.1417420000E+04, -0.2967890000E-01, 0.1489180000E-01 /
  data (epsData(202,i,55), i=1,3) / 0.1586730000E+04, -0.2470810000E-01, 0.1032220000E-01 /
  data (epsData(203,i,55), i=1,3) / 0.1772670000E+04, -0.2032720000E-01, 0.7175280000E-02 /
  data (epsData(204,i,55), i=1,3) / 0.1971250000E+04, -0.1669660000E-01, 0.5014500000E-02 /
  data (epsData(205,i,55), i=1,3) / 0.2191730000E+04, -0.1362660000E-01, 0.3504910000E-02 /
  data (epsData(206,i,55), i=1,3) / 0.2431210000E+04, -0.1110790000E-01, 0.2450410000E-02 /
  data (epsData(207,i,55), i=1,3) / 0.2699680000E+04, -0.9004340000E-02, 0.1711040000E-02 /
  data (epsData(208,i,55), i=1,3) / 0.2997930000E+04, -0.7274200000E-02, 0.1186650000E-02 /
  data (epsData(209,i,55), i=1,3) / 0.3326780000E+04, -0.5862510000E-02, 0.8226300000E-03 /
  data (epsData(210,i,55), i=1,3) / 0.3694410000E+04, -0.4696230000E-02, 0.5673770000E-03 /
  data (epsData(211,i,55), i=1,3) / 0.4107330000E+04, -0.3724510000E-02, 0.3888760000E-03 /
  data (epsData(212,i,55), i=1,3) / 0.4611960000E+04, -0.2823760000E-02, 0.2566590000E-03 /
  data (epsData(213,i,55), i=1,3) / 0.4924240000E+04, -0.2269260000E-02, 0.2027020000E-03 /
  data (epsData(214,i,55), i=1,3) / 0.4982900000E+04, -0.2066680000E-02, 0.1942150000E-03 /
  data (epsData(215,i,55), i=1,3) / 0.5001040000E+04, -0.1911920000E-02, 0.2096090000E-03 /
  data (epsData(216,i,55), i=1,3) / 0.5005580000E+04, -0.1832500000E-02, 0.2226670000E-03 /
  data (epsData(217,i,55), i=1,3) / 0.5007860000E+04, -0.1770260000E-02, 0.2389240000E-03 /
  data (epsData(218,i,55), i=1,3) / 0.5010130000E+04, -0.1680200000E-02, 0.2804940000E-03 /
  data (epsData(219,i,55), i=1,3) / 0.5012490000E+04, -0.1574730000E-02, 0.4438480000E-03 /
  data (epsData(220,i,55), i=1,3) / 0.5013910000E+04, -0.1609230000E-02, 0.6172880000E-03 /
  data (epsData(221,i,55), i=1,3) / 0.5015540000E+04, -0.1760230000E-02, 0.7158080000E-03 /
  data (epsData(222,i,55), i=1,3) / 0.5017660000E+04, -0.1912610000E-02, 0.6931510000E-03 /
  data (epsData(223,i,55), i=1,3) / 0.5021600000E+04, -0.1998340000E-02, 0.6056910000E-03 /
  data (epsData(224,i,55), i=1,3) / 0.5029990000E+04, -0.2015940000E-02, 0.5588460000E-03 /
  data (epsData(225,i,55), i=1,3) / 0.5083350000E+04, -0.2104170000E-02, 0.5450630000E-03 /
  data (epsData(226,i,55), i=1,3) / 0.5241280000E+04, -0.2071730000E-02, 0.4902840000E-03 /
  data (epsData(227,i,55), i=1,3) / 0.5337350000E+04, -0.1936540000E-02, 0.4579840000E-03 /
  data (epsData(228,i,55), i=1,3) / 0.5354200000E+04, -0.1839920000E-02, 0.4749550000E-03 /
  data (epsData(229,i,55), i=1,3) / 0.5358300000E+04, -0.1768850000E-02, 0.5240620000E-03 /
  data (epsData(230,i,55), i=1,3) / 0.5360810000E+04, -0.1766850000E-02, 0.6339460000E-03 /
  data (epsData(231,i,55), i=1,3) / 0.5363980000E+04, -0.1878090000E-02, 0.6733550000E-03 /
  data (epsData(232,i,55), i=1,3) / 0.5370770000E+04, -0.1935050000E-02, 0.6179690000E-03 /
  data (epsData(233,i,55), i=1,3) / 0.5480230000E+04, -0.1972140000E-02, 0.5718510000E-03 /
  data (epsData(234,i,55), i=1,3) / 0.5692380000E+04, -0.1863790000E-02, 0.4979640000E-03 /
  data (epsData(235,i,55), i=1,3) / 0.5713590000E+04, -0.1806930000E-02, 0.5148770000E-03 /
  data (epsData(236,i,55), i=1,3) / 0.5735940000E+04, -0.1847300000E-02, 0.5559990000E-03 /
  data (epsData(237,i,55), i=1,3) / 0.6047380000E+04, -0.1780270000E-02, 0.4637470000E-03 /
  data (epsData(238,i,55), i=1,3) / 0.6900940000E+04, -0.1446430000E-02, 0.2923440000E-03 /
  data (epsData(239,i,55), i=1,3) / 0.7764120000E+04, -0.1169840000E-02, 0.1916640000E-03 /
  data (epsData(240,i,55), i=1,3) / 0.8680230000E+04, -0.9467070000E-03, 0.1279000000E-03 /
  data (epsData(241,i,55), i=1,3) / 0.9690180000E+04, -0.7641620000E-03, 0.8569220000E-04 /
  data (epsData(242,i,55), i=1,3) / 0.1080290000E+05, -0.6162140000E-03, 0.5740920000E-04 /
  data (epsData(243,i,55), i=1,3) / 0.1204100000E+05, -0.4960810000E-03, 0.3837930000E-04 /
  data (epsData(244,i,55), i=1,3) / 0.1342340000E+05, -0.3986780000E-03, 0.2556970000E-04 /
  data (epsData(245,i,55), i=1,3) / 0.1496720000E+05, -0.3199750000E-03, 0.1698360000E-04 /
  data (epsData(246,i,55), i=1,3) / 0.1670190000E+05, -0.2562570000E-03, 0.1121800000E-04 /
  data (epsData(247,i,55), i=1,3) / 0.1864840000E+05, -0.2048980000E-03, 0.7385520000E-05 /
  data (epsData(248,i,55), i=1,3) / 0.2082650000E+05, -0.1637060000E-03, 0.4847680000E-05 /
  data (epsData(249,i,55), i=1,3) / 0.2327210000E+05, -0.1305920000E-03, 0.3167870000E-05 /
  data (epsData(250,i,55), i=1,3) / 0.2601650000E+05, -0.1040110000E-03, 0.2061930000E-05 /
  data (epsData(251,i,55), i=1,3) / 0.2910650000E+05, -0.8259020000E-04, 0.1336270000E-05 /
  data (epsData(252,i,55), i=1,3) / 0.3048620000E+05, -0.7502140000E-04, 0.1116640000E-05 /
  data (epsData(253,i,55), i=1,3) / 0.3049100000E+05, -0.7241070000E-04, 0.1115960000E-05 /
  data (epsData(254,i,55), i=1,3) / 0.3049100000E+05, -0.7499640000E-04, 0.1115950000E-05 /
  data (epsData(255,i,55), i=1,3) / 0.3453960000E+05, -0.5723470000E-04, 0.6870550000E-06 /
  data (epsData(256,i,55), i=1,3) / 0.3584640000E+05, -0.5100920000E-04, 0.5941030000E-06 /
  data (epsData(257,i,55), i=1,3) / 0.3596030000E+05, -0.4894270000E-04, 0.5868150000E-06 /
  data (epsData(258,i,55), i=1,3) / 0.3598490000E+05, -0.4772150000E-04, 0.1893130000E-05 /
  data (epsData(259,i,55), i=1,3) / 0.3598500000E+05, -0.4566050000E-04, 0.1904650000E-05 /
  data (epsData(260,i,55), i=1,3) / 0.3598500000E+05, -0.4771760000E-04, 0.1909420000E-05 /
  data (epsData(261,i,55), i=1,3) / 0.3600780000E+05, -0.4872440000E-04, 0.3279940000E-05 /
  data (epsData(262,i,55), i=1,3) / 0.3613540000E+05, -0.5024160000E-04, 0.3531620000E-05 /
  data (epsData(263,i,55), i=1,3) / 0.3724030000E+05, -0.4934540000E-04, 0.3205880000E-05 /
  data (epsData(264,i,55), i=1,3) / 0.4349180000E+05, -0.3746250000E-04, 0.1830170000E-05 /
  data (epsData(265,i,55), i=1,3) / 0.4891110000E+05, -0.2985190000E-04, 0.1179260000E-05 /
  data (epsData(266,i,55), i=1,3) / 0.5484730000E+05, -0.2382850000E-04, 0.7640070000E-06 /
  data (epsData(267,i,55), i=1,3) / 0.6144710000E+05, -0.1901960000E-04, 0.4942850000E-06 /
  data (epsData(268,i,55), i=1,3) / 0.6880750000E+05, -0.1518160000E-04, 0.3195740000E-06 /
  data (epsData(269,i,55), i=1,3) / 0.7706380000E+05, -0.1210780000E-04, 0.2046500000E-06 /
  data (epsData(270,i,55), i=1,3) / 0.8629490000E+05, -0.9661130000E-05, 0.1304790000E-06 /
  data (epsData(271,i,55), i=1,3) / 0.9643000000E+05, -0.7757050000E-05, 0.8365410000E-07 /
  data (epsData(272,i,55), i=1,3) / 0.9997920000E+05, -0.7328380000E-05, 0.7235830000E-07 /
  data (epsData(273,i,55), i=1,3) / 0.1000000000E+06, -0.7384930000E-05, 0.7229610000E-07 /
  data (epsData(1,i,56), i=1,3) / 0.2506580000E-02, -0.3336110000E+05, 0.6327360000E+05 /
  data (epsData(2,i,56), i=1,3) / 0.4693440000E-02, -0.2492770000E+05, 0.1931370000E+05 /
  data (epsData(3,i,56), i=1,3) / 0.7560590000E-02, -0.1158900000E+05, 0.8082810000E+04 /
  data (epsData(4,i,56), i=1,3) / 0.1110800000E-01, -0.5867580000E+04, 0.4111410000E+04 /
  data (epsData(5,i,56), i=1,3) / 0.1533570000E-01, -0.3195310000E+04, 0.2386400000E+04 /
  data (epsData(6,i,56), i=1,3) / 0.2024370000E-01, -0.1818210000E+04, 0.1522350000E+04 /
  data (epsData(7,i,56), i=1,3) / 0.2583200000E-01, -0.1049320000E+04, 0.1041790000E+04 /
  data (epsData(8,i,56), i=1,3) / 0.3210060000E-01, -0.5916480000E+03, 0.7522610000E+03 /
  data (epsData(9,i,56), i=1,3) / 0.3904940000E-01, -0.3047400000E+03, 0.5665010000E+03 /
  data (epsData(10,i,56), i=1,3) / 0.4667860000E-01, -0.1170110000E+03, 0.4411340000E+03 /
  data (epsData(11,i,56), i=1,3) / 0.5498800000E-01, 0.1035840000E+02, 0.3529510000E+03 /
  data (epsData(12,i,56), i=1,3) / 0.6397770000E-01, 0.9954700000E+02, 0.2887560000E+03 /
  data (epsData(13,i,56), i=1,3) / 0.7364770000E-01, 0.1638060000E+03, 0.2406550000E+03 /
  data (epsData(14,i,56), i=1,3) / 0.8399800000E-01, 0.2113800000E+03, 0.2037200000E+03 /
  data (epsData(15,i,56), i=1,3) / 0.9502860000E-01, 0.2476090000E+03, 0.1747570000E+03 /
  data (epsData(16,i,56), i=1,3) / 0.1067390000E+00, 0.2761540000E+03, 0.1516300000E+03 /
  data (epsData(17,i,56), i=1,3) / 0.1322020000E+00, 0.3222730000E+03, 0.1174360000E+03 /
  data (epsData(18,i,56), i=1,3) / 0.1459540000E+00, 0.3396780000E+03, 0.1153140000E+03 /
  data (epsData(19,i,56), i=1,3) / 0.1912910000E+00, 0.3580290000E+03, 0.1048000000E+03 /
  data (epsData(20,i,56), i=1,3) / 0.3640360000E+00, 0.3811900000E+03, 0.6856860000E+02 /
  data (epsData(21,i,56), i=1,3) / 0.4338630000E+00, 0.4029550000E+03, 0.5990610000E+02 /
  data (epsData(22,i,56), i=1,3) / 0.5098130000E+00, 0.4222090000E+03, 0.7485640000E+02 /
  data (epsData(23,i,56), i=1,3) / 0.6206030000E+00, 0.4478980000E+03, 0.8149400000E+02 /
  data (epsData(24,i,56), i=1,3) / 0.6800800000E+00, 0.4759010000E+03, 0.8171210000E+02 /
  data (epsData(25,i,56), i=1,3) / 0.7108380000E+00, 0.5016440000E+03, 0.8134930000E+02 /
  data (epsData(26,i,56), i=1,3) / 0.7422780000E+00, 0.5367270000E+03, 0.9539460000E+02 /
  data (epsData(27,i,56), i=1,3) / 0.7743970000E+00, 0.5623630000E+03, 0.1399970000E+03 /
  data (epsData(28,i,56), i=1,3) / 0.8406760000E+00, 0.5720710000E+03, 0.2104700000E+03 /
  data (epsData(29,i,56), i=1,3) / 0.9451980000E+00, 0.5766950000E+03, 0.2808700000E+03 /
  data (epsData(30,i,56), i=1,3) / 0.9813990000E+00, 0.5928100000E+03, 0.2972760000E+03 /
  data (epsData(31,i,56), i=1,3) / 0.1018280000E+01, 0.6386310000E+03, 0.3109060000E+03 /
  data (epsData(32,i,56), i=1,3) / 0.1055840000E+01, 0.6868580000E+03, 0.4091150000E+03 /
  data (epsData(33,i,56), i=1,3) / 0.1094080000E+01, 0.6799860000E+03, 0.5478910000E+03 /
  data (epsData(34,i,56), i=1,3) / 0.1133010000E+01, 0.6105900000E+03, 0.6692630000E+03 /
  data (epsData(35,i,56), i=1,3) / 0.1212890000E+01, 0.4378380000E+03, 0.7700610000E+03 /
  data (epsData(36,i,56), i=1,3) / 0.1295500000E+01, 0.2491500000E+03, 0.8436270000E+03 /
  data (epsData(37,i,56), i=1,3) / 0.1337820000E+01, 0.1232970000E+03, 0.8607350000E+03 /
  data (epsData(38,i,56), i=1,3) / 0.1380830000E+01, 0.1140360000E+02, 0.7899540000E+03 /
  data (epsData(39,i,56), i=1,3) / 0.1424510000E+01, -0.5766160000E+02, 0.7250630000E+03 /
  data (epsData(40,i,56), i=1,3) / 0.1468870000E+01, -0.1056840000E+03, 0.6655130000E+03 /
  data (epsData(41,i,56), i=1,3) / 0.1559640000E+01, -0.1674400000E+03, 0.5605160000E+03 /
  data (epsData(42,i,56), i=1,3) / 0.1653130000E+01, -0.2035150000E+03, 0.4700190000E+03 /
  data (epsData(43,i,56), i=1,3) / 0.1749350000E+01, -0.2197810000E+03, 0.3878870000E+03 /
  data (epsData(44,i,56), i=1,3) / 0.1848280000E+01, -0.2211480000E+03, 0.3181360000E+03 /
  data (epsData(45,i,56), i=1,3) / 0.1949930000E+01, -0.2108390000E+03, 0.2605060000E+03 /
  data (epsData(46,i,56), i=1,3) / 0.2161410000E+01, -0.1866810000E+03, 0.1840280000E+03 /
  data (epsData(47,i,56), i=1,3) / 0.2383760000E+01, -0.1615170000E+03, 0.1293520000E+03 /
  data (epsData(48,i,56), i=1,3) / 0.2617000000E+01, -0.1369960000E+03, 0.8983160000E+02 /
  data (epsData(49,i,56), i=1,3) / 0.2861130000E+01, -0.1137000000E+03, 0.6117870000E+02 /
  data (epsData(50,i,56), i=1,3) / 0.3116140000E+01, -0.9247980000E+02, 0.4119910000E+02 /
  data (epsData(51,i,56), i=1,3) / 0.3382030000E+01, -0.7379530000E+02, 0.2768680000E+02 /
  data (epsData(52,i,56), i=1,3) / 0.3588600000E+01, -0.6150950000E+02, 0.2058390000E+02 /
  data (epsData(53,i,56), i=1,3) / 0.3801290000E+01, -0.5075510000E+02, 0.1524470000E+02 /
  data (epsData(54,i,56), i=1,3) / 0.4094390000E+01, -0.3871090000E+02, 0.1035970000E+02 /
  data (epsData(55,i,56), i=1,3) / 0.4398380000E+01, -0.2878090000E+02, 0.7078200000E+01 /
  data (epsData(56,i,56), i=1,3) / 0.4633520000E+01, -0.2241270000E+02, 0.5284040000E+01 /
  data (epsData(57,i,56), i=1,3) / 0.4874780000E+01, -0.1685900000E+02, 0.4132420000E+01 /
  data (epsData(58,i,56), i=1,3) / 0.5122160000E+01, -0.1205930000E+02, 0.3254230000E+01 /
  data (epsData(59,i,56), i=1,3) / 0.5375660000E+01, -0.7843720000E+01, 0.2589040000E+01 /
  data (epsData(60,i,56), i=1,3) / 0.5548060000E+01, -0.5300030000E+01, 0.2231870000E+01 /
  data (epsData(61,i,56), i=1,3) / 0.5811770000E+01, -0.1787080000E+01, 0.1833620000E+01 /
  data (epsData(62,i,56), i=1,3) / 0.5901040000E+01, -0.6798670000E+00, 0.1775700000E+01 /
  data (epsData(63,i,56), i=1,3) / 0.5990980000E+01, 0.3674570000E+00, 0.1739050000E+01 /
  data (epsData(64,i,56), i=1,3) / 0.6081600000E+01, 0.1371740000E+01, 0.1704640000E+01 /
  data (epsData(65,i,56), i=1,3) / 0.6172910000E+01, 0.2339230000E+01, 0.1700950000E+01 /
  data (epsData(66,i,56), i=1,3) / 0.6357560000E+01, 0.4096500000E+01, 0.1765050000E+01 /
  data (epsData(67,i,56), i=1,3) / 0.6639630000E+01, 0.6383670000E+01, 0.1788960000E+01 /
  data (epsData(68,i,56), i=1,3) / 0.7123360000E+01, 0.9718540000E+01, 0.1662950000E+01 /
  data (epsData(69,i,56), i=1,3) / 0.7932710000E+01, 0.1450530000E+02, 0.1402670000E+01 /
  data (epsData(70,i,56), i=1,3) / 0.1038290000E+02, 0.2721360000E+02, 0.1083420000E+01 /
  data (epsData(71,i,56), i=1,3) / 0.1148020000E+02, 0.3446450000E+02, 0.1140840000E+01 /
  data (epsData(72,i,56), i=1,3) / 0.1224230000E+02, 0.4147400000E+02, 0.1326430000E+01 /
  data (epsData(73,i,56), i=1,3) / 0.1289610000E+02, 0.5020440000E+02, 0.1695360000E+01 /
  data (epsData(74,i,56), i=1,3) / 0.1329650000E+02, 0.5794040000E+02, 0.2174500000E+01 /
  data (epsData(75,i,56), i=1,3) / 0.1370310000E+02, 0.6945150000E+02, 0.3097670000E+01 /
  data (epsData(76,i,56), i=1,3) / 0.1397750000E+02, 0.8128230000E+02, 0.4425400000E+01 /
  data (epsData(77,i,56), i=1,3) / 0.1411580000E+02, 0.8965320000E+02, 0.5407110000E+01 /
  data (epsData(78,i,56), i=1,3) / 0.1425470000E+02, 0.1005990000E+03, 0.7837830000E+01 /
  data (epsData(79,i,56), i=1,3) / 0.1439430000E+02, 0.1171290000E+03, 0.1018910000E+02 /
  data (epsData(80,i,56), i=1,3) / 0.1453460000E+02, 0.1363670000E+03, 0.2688250000E+02 /
  data (epsData(81,i,56), i=1,3) / 0.1467550000E+02, 0.1610310000E+03, 0.4362410000E+02 /
  data (epsData(82,i,56), i=1,3) / 0.1480540000E+02, 0.1707550000E+03, 0.1055320000E+03 /
  data (epsData(83,i,56), i=1,3) / 0.1493610000E+02, 0.1464420000E+03, 0.1515960000E+03 /
  data (epsData(84,i,56), i=1,3) / 0.1499590000E+02, 0.1243410000E+03, 0.1721680000E+03 /
  data (epsData(85,i,56), i=1,3) / 0.1504800000E+02, 0.1020310000E+03, 0.1741000000E+03 /
  data (epsData(86,i,56), i=1,3) / 0.1529120000E+02, 0.3899440000E+02, 0.1767260000E+03 /
  data (epsData(87,i,56), i=1,3) / 0.1537180000E+02, 0.1857380000E+02, 0.1678470000E+03 /
  data (epsData(88,i,56), i=1,3) / 0.1551970000E+02, -0.4793320000E+01, 0.1498840000E+03 /
  data (epsData(89,i,56), i=1,3) / 0.1565040000E+02, -0.2014470000E+02, 0.1330190000E+03 /
  data (epsData(90,i,56), i=1,3) / 0.1579190000E+02, -0.2890430000E+02, 0.1127660000E+03 /
  data (epsData(91,i,56), i=1,3) / 0.1590510000E+02, -0.2997060000E+02, 0.9729710000E+02 /
  data (epsData(92,i,56), i=1,3) / 0.1615000000E+02, -0.2612650000E+02, 0.7524040000E+02 /
  data (epsData(93,i,56), i=1,3) / 0.1623710000E+02, -0.2219150000E+02, 0.6900360000E+02 /
  data (epsData(94,i,56), i=1,3) / 0.1646670000E+02, -0.1221770000E+02, 0.5825080000E+02 /
  data (epsData(95,i,56), i=1,3) / 0.1656330000E+02, -0.6052670000E+01, 0.5511560000E+02 /
  data (epsData(96,i,56), i=1,3) / 0.1661260000E+02, -0.2353020000E+01, 0.5435460000E+02 /
  data (epsData(97,i,56), i=1,3) / 0.1681670000E+02, 0.1093440000E+02, 0.5753620000E+02 /
  data (epsData(98,i,56), i=1,3) / 0.1686940000E+02, 0.1600750000E+02, 0.5876340000E+02 /
  data (epsData(99,i,56), i=1,3) / 0.1692280000E+02, 0.2176700000E+02, 0.6398950000E+02 /
  data (epsData(100,i,56), i=1,3) / 0.1697690000E+02, 0.2446130000E+02, 0.7460540000E+02 /
  data (epsData(101,i,56), i=1,3) / 0.1701220000E+02, 0.2251930000E+02, 0.8046140000E+02 /
  data (epsData(102,i,56), i=1,3) / 0.1715000000E+02, 0.1313420000E+02, 0.9580660000E+02 /
  data (epsData(103,i,56), i=1,3) / 0.1719590000E+02, 0.6538800000E+01, 0.1009620000E+03 /
  data (epsData(104,i,56), i=1,3) / 0.1724800000E+02, -0.1609530000E+01, 0.1005530000E+03 /
  data (epsData(105,i,56), i=1,3) / 0.1744080000E+02, -0.1811450000E+02, 0.9734450000E+02 /
  data (epsData(106,i,56), i=1,3) / 0.1754420000E+02, -0.2716350000E+02, 0.9405770000E+02 /
  data (epsData(107,i,56), i=1,3) / 0.1775140000E+02, -0.3712490000E+02, 0.8278720000E+02 /
  data (epsData(108,i,56), i=1,3) / 0.1802890000E+02, -0.4360740000E+02, 0.6816890000E+02 /
  data (epsData(109,i,56), i=1,3) / 0.1843710000E+02, -0.4412050000E+02, 0.5130570000E+02 /
  data (epsData(110,i,56), i=1,3) / 0.1866670000E+02, -0.4121790000E+02, 0.4396740000E+02 /
  data (epsData(111,i,56), i=1,3) / 0.1906940000E+02, -0.3612710000E+02, 0.3708820000E+02 /
  data (epsData(112,i,56), i=1,3) / 0.1963410000E+02, -0.3164390000E+02, 0.3170890000E+02 /
  data (epsData(113,i,56), i=1,3) / 0.2020040000E+02, -0.2924220000E+02, 0.2851170000E+02 /
  data (epsData(114,i,56), i=1,3) / 0.2060860000E+02, -0.2914700000E+02, 0.2579600000E+02 /
  data (epsData(115,i,56), i=1,3) / 0.2096740000E+02, -0.2756010000E+02, 0.2279890000E+02 /
  data (epsData(116,i,56), i=1,3) / 0.2181680000E+02, -0.2455910000E+02, 0.1903590000E+02 /
  data (epsData(117,i,56), i=1,3) / 0.2347770000E+02, -0.2060110000E+02, 0.1345000000E+02 /
  data (epsData(118,i,56), i=1,3) / 0.2517190000E+02, -0.1724280000E+02, 0.9765690000E+01 /
  data (epsData(119,i,56), i=1,3) / 0.2706280000E+02, -0.1426870000E+02, 0.7092260000E+01 /
  data (epsData(120,i,56), i=1,3) / 0.2915020000E+02, -0.1171010000E+02, 0.5146630000E+01 /
  data (epsData(121,i,56), i=1,3) / 0.3136670000E+02, -0.9590410000E+01, 0.3780730000E+01 /
  data (epsData(122,i,56), i=1,3) / 0.3397900000E+02, -0.7661520000E+01, 0.2686440000E+01 /
  data (epsData(123,i,56), i=1,3) / 0.3668410000E+02, -0.6090080000E+01, 0.1932090000E+01 /
  data (epsData(124,i,56), i=1,3) / 0.3967300000E+02, -0.4723910000E+01, 0.1383270000E+01 /
  data (epsData(125,i,56), i=1,3) / 0.4270870000E+02, -0.3622710000E+01, 0.1019900000E+01 /
  data (epsData(126,i,56), i=1,3) / 0.4587710000E+02, -0.2700440000E+01, 0.7723140000E+00 /
  data (epsData(127,i,56), i=1,3) / 0.4924450000E+02, -0.1906210000E+01, 0.5992110000E+00 /
  data (epsData(128,i,56), i=1,3) / 0.5276600000E+02, -0.1218900000E+01, 0.4754170000E+00 /
  data (epsData(129,i,56), i=1,3) / 0.5621060000E+02, -0.6401900000E+00, 0.3876270000E+00 /
  data (epsData(130,i,56), i=1,3) / 0.5970050000E+02, -0.1087840000E+00, 0.3187980000E+00 /
  data (epsData(131,i,56), i=1,3) / 0.6405060000E+02, 0.5217860000E+00, 0.2555190000E+00 /
  data (epsData(132,i,56), i=1,3) / 0.6870070000E+02, 0.1219890000E+01, 0.2076720000E+00 /
  data (epsData(133,i,56), i=1,3) / 0.7316640000E+02, 0.1986700000E+01, 0.1915620000E+00 /
  data (epsData(134,i,56), i=1,3) / 0.7661410000E+02, 0.2719300000E+01, 0.1775240000E+00 /
  data (epsData(135,i,56), i=1,3) / 0.7941650000E+02, 0.3498260000E+01, 0.1724100000E+00 /
  data (epsData(136,i,56), i=1,3) / 0.8177060000E+02, 0.4374070000E+01, 0.1738590000E+00 /
  data (epsData(137,i,56), i=1,3) / 0.8390940000E+02, 0.5476800000E+01, 0.1844330000E+00 /
  data (epsData(138,i,56), i=1,3) / 0.8554210000E+02, 0.6661830000E+01, 0.2035710000E+00 /
  data (epsData(139,i,56), i=1,3) / 0.8692140000E+02, 0.8070440000E+01, 0.2350560000E+00 /
  data (epsData(140,i,56), i=1,3) / 0.8803710000E+02, 0.9686870000E+01, 0.2819780000E+00 /
  data (epsData(141,i,56), i=1,3) / 0.8888100000E+02, 0.1140470000E+02, 0.3453440000E+00 /
  data (epsData(142,i,56), i=1,3) / 0.8944700000E+02, 0.1296630000E+02, 0.4205860000E+00 /
  data (epsData(143,i,56), i=1,3) / 0.9014800000E+02, 0.1556380000E+02, 0.7533670000E+00 /
  data (epsData(144,i,56), i=1,3) / 0.9071670000E+02, 0.1848820000E+02, 0.1097160000E+01 /
  data (epsData(145,i,56), i=1,3) / 0.9116570000E+02, 0.2189810000E+02, 0.1572900000E+01 /
  data (epsData(146,i,56), i=1,3) / 0.9151940000E+02, 0.2578630000E+02, 0.2293740000E+01 /
  data (epsData(147,i,56), i=1,3) / 0.9181330000E+02, 0.3039900000E+02, 0.3496450000E+01 /
  data (epsData(148,i,56), i=1,3) / 0.9202280000E+02, 0.3502360000E+02, 0.5207570000E+01 /
  data (epsData(149,i,56), i=1,3) / 0.9218710000E+02, 0.3976160000E+02, 0.6941290000E+01 /
  data (epsData(150,i,56), i=1,3) / 0.9241570000E+02, 0.4778130000E+02, 0.1340050000E+02 /
  data (epsData(151,i,56), i=1,3) / 0.9247450000E+02, 0.5069630000E+02, 0.1511550000E+02 /
  data (epsData(152,i,56), i=1,3) / 0.9253410000E+02, 0.5398820000E+02, 0.1878140000E+02 /
  data (epsData(153,i,56), i=1,3) / 0.9267650000E+02, 0.5823190000E+02, 0.3097160000E+02 /
  data (epsData(154,i,56), i=1,3) / 0.9279590000E+02, 0.6062400000E+02, 0.4113090000E+02 /
  data (epsData(155,i,56), i=1,3) / 0.9286670000E+02, 0.6140520000E+02, 0.5156660000E+02 /
  data (epsData(156,i,56), i=1,3) / 0.9299320000E+02, 0.5424950000E+02, 0.7063680000E+02 /
  data (epsData(157,i,56), i=1,3) / 0.9306570000E+02, 0.4542730000E+02, 0.8150860000E+02 /
  data (epsData(158,i,56), i=1,3) / 0.9309120000E+02, 0.4054250000E+02, 0.8532860000E+02 /
  data (epsData(159,i,56), i=1,3) / 0.9314420000E+02, 0.2997270000E+02, 0.8734080000E+02 /
  data (epsData(160,i,56), i=1,3) / 0.9335140000E+02, -0.1441700000E+01, 0.9055470000E+02 /
  data (epsData(161,i,56), i=1,3) / 0.9338370000E+02, -0.8366230000E+01, 0.9105350000E+02 /
  data (epsData(162,i,56), i=1,3) / 0.9341670000E+02, -0.1627000000E+02, 0.8860340000E+02 /
  data (epsData(163,i,56), i=1,3) / 0.9348470000E+02, -0.2773890000E+02, 0.7760020000E+02 /
  data (epsData(164,i,56), i=1,3) / 0.9359190000E+02, -0.3628060000E+02, 0.6033540000E+02 /
  data (epsData(165,i,56), i=1,3) / 0.9366670000E+02, -0.3745220000E+02, 0.4832340000E+02 /
  data (epsData(166,i,56), i=1,3) / 0.9370510000E+02, -0.3597990000E+02, 0.4260190000E+02 /
  data (epsData(167,i,56), i=1,3) / 0.9395000000E+02, -0.2872040000E+02, 0.2418680000E+02 /
  data (epsData(168,i,56), i=1,3) / 0.9399320000E+02, -0.2631350000E+02, 0.2095870000E+02 /
  data (epsData(169,i,56), i=1,3) / 0.9403710000E+02, -0.2366690000E+02, 0.1956830000E+02 /
  data (epsData(170,i,56), i=1,3) / 0.9421940000E+02, -0.1669840000E+02, 0.1524290000E+02 /
  data (epsData(171,i,56), i=1,3) / 0.9431470000E+02, -0.1303280000E+02, 0.1324420000E+02 /
  data (epsData(172,i,56), i=1,3) / 0.9446260000E+02, -0.8023260000E+01, 0.1226360000E+02 /
  data (epsData(173,i,56), i=1,3) / 0.9456470000E+02, -0.4783410000E+01, 0.1159080000E+02 /
  data (epsData(174,i,56), i=1,3) / 0.9461670000E+02, -0.2944850000E+01, 0.1144250000E+02 /
  data (epsData(175,i,56), i=1,3) / 0.9483170000E+02, 0.4097750000E+01, 0.1251890000E+02 /
  data (epsData(176,i,56), i=1,3) / 0.9488710000E+02, 0.6405750000E+01, 0.1279490000E+02 /
  data (epsData(177,i,56), i=1,3) / 0.9494320000E+02, 0.8839150000E+01, 0.1420930000E+02 /
  data (epsData(178,i,56), i=1,3) / 0.9511570000E+02, 0.1462910000E+02, 0.1957020000E+02 /
  data (epsData(179,i,56), i=1,3) / 0.9517450000E+02, 0.1721520000E+02, 0.2139050000E+02 /
  data (epsData(180,i,56), i=1,3) / 0.9523410000E+02, 0.1998240000E+02, 0.2536000000E+02 /
  data (epsData(181,i,56), i=1,3) / 0.9529430000E+02, 0.2148480000E+02, 0.3097530000E+02 /
  data (epsData(182,i,56), i=1,3) / 0.9535510000E+02, 0.2145900000E+02, 0.3663980000E+02 /
  data (epsData(183,i,56), i=1,3) / 0.9541670000E+02, 0.2024620000E+02, 0.4235230000E+02 /
  data (epsData(184,i,56), i=1,3) / 0.9547900000E+02, 0.1775720000E+02, 0.4811360000E+02 /
  data (epsData(185,i,56), i=1,3) / 0.9560550000E+02, 0.9861230000E+01, 0.5774850000E+02 /
  data (epsData(186,i,56), i=1,3) / 0.9566980000E+02, 0.4503010000E+01, 0.6241890000E+02 /
  data (epsData(187,i,56), i=1,3) / 0.9573470000E+02, -0.2763440000E+01, 0.6712590000E+02 /
  data (epsData(188,i,56), i=1,3) / 0.9580040000E+02, -0.1408720000E+02, 0.7180440000E+02 /
  data (epsData(189,i,56), i=1,3) / 0.9586670000E+02, -0.2533710000E+02, 0.6554570000E+02 /
  data (epsData(190,i,56), i=1,3) / 0.9593370000E+02, -0.3229530000E+02, 0.5924080000E+02 /
  data (epsData(191,i,56), i=1,3) / 0.9606980000E+02, -0.4008380000E+02, 0.4649190000E+02 /
  data (epsData(192,i,56), i=1,3) / 0.9620860000E+02, -0.4284270000E+02, 0.3509460000E+02 /
  data (epsData(193,i,56), i=1,3) / 0.9635010000E+02, -0.4192810000E+02, 0.2397360000E+02 /
  data (epsData(194,i,56), i=1,3) / 0.9642180000E+02, -0.3950070000E+02, 0.1944070000E+02 /
  data (epsData(195,i,56), i=1,3) / 0.9664120000E+02, -0.3361140000E+02, 0.1320170000E+02 /
  data (epsData(196,i,56), i=1,3) / 0.9671570000E+02, -0.3162500000E+02, 0.1137460000E+02 /
  data (epsData(197,i,56), i=1,3) / 0.9694330000E+02, -0.2707090000E+02, 0.8991910000E+01 /
  data (epsData(198,i,56), i=1,3) / 0.9709840000E+02, -0.2465260000E+02, 0.7894980000E+01 /
  data (epsData(199,i,56), i=1,3) / 0.9741670000E+02, -0.2115080000E+02, 0.6499200000E+01 /
  data (epsData(200,i,56), i=1,3) / 0.9783000000E+02, -0.1808850000E+02, 0.5412970000E+01 /
  data (epsData(201,i,56), i=1,3) / 0.9834840000E+02, -0.1541370000E+02, 0.4566290000E+01 /
  data (epsData(202,i,56), i=1,3) / 0.9898410000E+02, -0.1316090000E+02, 0.3956880000E+01 /
  data (epsData(203,i,56), i=1,3) / 0.9985050000E+02, -0.1111670000E+02, 0.3436140000E+01 /
  data (epsData(204,i,56), i=1,3) / 0.1009840000E+03, -0.9369770000E+01, 0.2991580000E+01 /
  data (epsData(205,i,56), i=1,3) / 0.1025460000E+03, -0.7840190000E+01, 0.2569770000E+01 /
  data (epsData(206,i,56), i=1,3) / 0.1046220000E+03, -0.6571780000E+01, 0.2157090000E+01 /
  data (epsData(207,i,56), i=1,3) / 0.1074580000E+03, -0.5486470000E+01, 0.1723560000E+01 /
  data (epsData(208,i,56), i=1,3) / 0.1112250000E+03, -0.4551340000E+01, 0.1280570000E+01 /
  data (epsData(209,i,56), i=1,3) / 0.1155940000E+03, -0.3788740000E+01, 0.9155000000E+00 /
  data (epsData(210,i,56), i=1,3) / 0.1206170000E+03, -0.3152880000E+01, 0.6442790000E+00 /
  data (epsData(211,i,56), i=1,3) / 0.1265650000E+03, -0.2606910000E+01, 0.4397860000E+00 /
  data (epsData(212,i,56), i=1,3) / 0.1335560000E+03, -0.2137970000E+01, 0.2878410000E+00 /
  data (epsData(213,i,56), i=1,3) / 0.1414580000E+03, -0.1743120000E+01, 0.1836830000E+00 /
  data (epsData(214,i,56), i=1,3) / 0.1497780000E+03, -0.1430040000E+01, 0.1239470000E+00 /
  data (epsData(215,i,56), i=1,3) / 0.1593530000E+03, -0.1159170000E+01, 0.9104910000E-01 /
  data (epsData(216,i,56), i=1,3) / 0.1709690000E+03, -0.9158180000E+00, 0.7605660000E-01 /
  data (epsData(217,i,56), i=1,3) / 0.1783180000E+03, -0.7840080000E+00, 0.9091850000E-01 /
  data (epsData(218,i,56), i=1,3) / 0.1840860000E+03, -0.7313470000E+00, 0.1053120000E+00 /
  data (epsData(219,i,56), i=1,3) / 0.1934820000E+03, -0.6372970000E+00, 0.1100950000E+00 /
  data (epsData(220,i,56), i=1,3) / 0.2166130000E+03, -0.4869490000E+00, 0.1088950000E+00 /
  data (epsData(221,i,56), i=1,3) / 0.2413750000E+03, -0.3775510000E+00, 0.1047880000E+00 /
  data (epsData(222,i,56), i=1,3) / 0.2562550000E+03, -0.3288300000E+00, 0.1076520000E+00 /
  data (epsData(223,i,56), i=1,3) / 0.2919010000E+03, -0.2584220000E+00, 0.9803770000E-01 /
  data (epsData(224,i,56), i=1,3) / 0.3378570000E+03, -0.1977820000E+00, 0.8148600000E-01 /
  data (epsData(225,i,56), i=1,3) / 0.3876490000E+03, -0.1531770000E+00, 0.6525090000E-01 /
  data (epsData(226,i,56), i=1,3) / 0.4448340000E+03, -0.1170150000E+00, 0.5034370000E-01 /
  data (epsData(227,i,56), i=1,3) / 0.5056610000E+03, -0.8819630000E-01, 0.3841580000E-01 /
  data (epsData(228,i,56), i=1,3) / 0.5748700000E+03, -0.6223730000E-01, 0.2868700000E-01 /
  data (epsData(229,i,56), i=1,3) / 0.6598620000E+03, -0.3346600000E-01, 0.2050600000E-01 /
  data (epsData(230,i,56), i=1,3) / 0.6947870000E+03, -0.1957670000E-01, 0.1801020000E-01 /
  data (epsData(231,i,56), i=1,3) / 0.7143710000E+03, -0.9624630000E-02, 0.1673800000E-01 /
  data (epsData(232,i,56), i=1,3) / 0.7263260000E+03, -0.1939640000E-02, 0.1600630000E-01 /
  data (epsData(233,i,56), i=1,3) / 0.7343810000E+03, 0.4419100000E-02, 0.1553690000E-01 /
  data (epsData(234,i,56), i=1,3) / 0.7425040000E+03, 0.1233810000E-01, 0.1510920000E-01 /
  data (epsData(235,i,56), i=1,3) / 0.7490510000E+03, 0.2043600000E-01, 0.1477650000E-01 /
  data (epsData(236,i,56), i=1,3) / 0.7548160000E+03, 0.2964940000E-01, 0.1448900000E-01 /
  data (epsData(237,i,56), i=1,3) / 0.7589540000E+03, 0.3835350000E-01, 0.1428580000E-01 /
  data (epsData(238,i,56), i=1,3) / 0.7631090000E+03, 0.4904970000E-01, 0.1501120000E-01 /
  data (epsData(239,i,56), i=1,3) / 0.7672810000E+03, 0.6368180000E-01, 0.1527000000E-01 /
  data (epsData(240,i,56), i=1,3) / 0.7706300000E+03, 0.8060000000E-01, 0.1580500000E-01 /
  data (epsData(241,i,56), i=1,3) / 0.7731500000E+03, 0.9886750000E-01, 0.1666340000E-01 /
  data (epsData(242,i,56), i=1,3) / 0.7748330000E+03, 0.1157810000E+00, 0.1771790000E-01 /
  data (epsData(243,i,56), i=1,3) / 0.7765190000E+03, 0.1392680000E+00, 0.2017860000E-01 /
  data (epsData(244,i,56), i=1,3) / 0.7773630000E+03, 0.1546320000E+00, 0.2178220000E-01 /
  data (epsData(245,i,56), i=1,3) / 0.7782070000E+03, 0.1742310000E+00, 0.2414690000E-01 /
  data (epsData(246,i,56), i=1,3) / 0.7790530000E+03, 0.2001270000E+00, 0.2787780000E-01 /
  data (epsData(247,i,56), i=1,3) / 0.7798990000E+03, 0.2355990000E+00, 0.3439850000E-01 /
  data (epsData(248,i,56), i=1,3) / 0.7806650000E+03, 0.2805400000E+00, 0.4609950000E-01 /
  data (epsData(249,i,56), i=1,3) / 0.7812510000E+03, 0.3316720000E+00, 0.6313520000E-01 /
  data (epsData(250,i,56), i=1,3) / 0.7816840000E+03, 0.3851410000E+00, 0.8549000000E-01 /
  data (epsData(251,i,56), i=1,3) / 0.7819820000E+03, 0.4334890000E+00, 0.1106710000E+00 /
  data (epsData(252,i,56), i=1,3) / 0.7822150000E+03, 0.4801500000E+00, 0.1388540000E+00 /
  data (epsData(253,i,56), i=1,3) / 0.7825170000E+03, 0.5551510000E+00, 0.2007950000E+00 /
  data (epsData(254,i,56), i=1,3) / 0.7827870000E+03, 0.6315150000E+00, 0.2965580000E+00 /
  data (epsData(255,i,56), i=1,3) / 0.7830160000E+03, 0.6859640000E+00, 0.4421010000E+00 /
  data (epsData(256,i,56), i=1,3) / 0.7831340000E+03, 0.7102550000E+00, 0.5332860000E+00 /
  data (epsData(257,i,56), i=1,3) / 0.7833170000E+03, 0.6872370000E+00, 0.7267240000E+00 /
  data (epsData(258,i,56), i=1,3) / 0.7835050000E+03, 0.5859100000E+00, 0.9273700000E+00 /
  data (epsData(259,i,56), i=1,3) / 0.7836350000E+03, 0.4600370000E+00, 0.1065050000E+01 /
  data (epsData(260,i,56), i=1,3) / 0.7837000000E+03, 0.3592390000E+00, 0.1134560000E+01 /
  data (epsData(261,i,56), i=1,3) / 0.7839700000E+03, -0.3572360000E-01, 0.1159290000E+01 /
  data (epsData(262,i,56), i=1,3) / 0.7840390000E+03, -0.1486710000E+00, 0.1124270000E+01 /
  data (epsData(263,i,56), i=1,3) / 0.7841790000E+03, -0.3112580000E+00, 0.9880860000E+00 /
  data (epsData(264,i,56), i=1,3) / 0.7843940000E+03, -0.4442240000E+00, 0.7765290000E+00 /
  data (epsData(265,i,56), i=1,3) / 0.7845410000E+03, -0.4757950000E+00, 0.6300590000E+00 /
  data (epsData(266,i,56), i=1,3) / 0.7846160000E+03, -0.4698800000E+00, 0.5619440000E+00 /
  data (epsData(267,i,56), i=1,3) / 0.7848430000E+03, -0.4404910000E+00, 0.4234210000E+00 /
  data (epsData(268,i,56), i=1,3) / 0.7849980000E+03, -0.4035160000E+00, 0.3552100000E+00 /
  data (epsData(269,i,56), i=1,3) / 0.7852360000E+03, -0.3556920000E+00, 0.2787860000E+00 /
  data (epsData(270,i,56), i=1,3) / 0.7855630000E+03, -0.2974220000E+00, 0.2167470000E+00 /
  data (epsData(271,i,56), i=1,3) / 0.7859000000E+03, -0.2499610000E+00, 0.1765020000E+00 /
  data (epsData(272,i,56), i=1,3) / 0.7862480000E+03, -0.2110980000E+00, 0.1490380000E+00 /
  data (epsData(273,i,56), i=1,3) / 0.7866990000E+03, -0.1715570000E+00, 0.1256420000E+00 /
  data (epsData(274,i,56), i=1,3) / 0.7872620000E+03, -0.1344570000E+00, 0.1084120000E+00 /
  data (epsData(275,i,56), i=1,3) / 0.7879510000E+03, -0.1009810000E+00, 0.9563380000E-01 /
  data (epsData(276,i,56), i=1,3) / 0.7887780000E+03, -0.7116940000E-01, 0.8667640000E-01 /
  data (epsData(277,i,56), i=1,3) / 0.7897600000E+03, -0.4427470000E-01, 0.8075100000E-01 /
  data (epsData(278,i,56), i=1,3) / 0.7910360000E+03, -0.1642810000E-01, 0.7685240000E-01 /
  data (epsData(279,i,56), i=1,3) / 0.7927790000E+03, 0.1826680000E-01, 0.7543000000E-01 /
  data (epsData(280,i,56), i=1,3) / 0.7938350000E+03, 0.4212850000E-01, 0.7664340000E-01 /
  data (epsData(281,i,56), i=1,3) / 0.7946550000E+03, 0.6523180000E-01, 0.7940400000E-01 /
  data (epsData(282,i,56), i=1,3) / 0.7953580000E+03, 0.9107040000E-01, 0.8435060000E-01 /
  data (epsData(283,i,56), i=1,3) / 0.7959670000E+03, 0.1211550000E+00, 0.9282960000E-01 /
  data (epsData(284,i,56), i=1,3) / 0.7964840000E+03, 0.1555570000E+00, 0.1055730000E+00 /
  data (epsData(285,i,56), i=1,3) / 0.7968840000E+03, 0.1922630000E+00, 0.1236920000E+00 /
  data (epsData(286,i,56), i=1,3) / 0.7971370000E+03, 0.2218530000E+00, 0.1412050000E+00 /
  data (epsData(287,i,56), i=1,3) / 0.7974150000E+03, 0.2611160000E+00, 0.1710140000E+00 /
  data (epsData(288,i,56), i=1,3) / 0.7977170000E+03, 0.3125980000E+00, 0.2275720000E+00 /
  data (epsData(289,i,56), i=1,3) / 0.7979870000E+03, 0.3560560000E+00, 0.3153170000E+00 /
  data (epsData(290,i,56), i=1,3) / 0.7981580000E+03, 0.3716960000E+00, 0.4080790000E+00 /
  data (epsData(291,i,56), i=1,3) / 0.7983340000E+03, 0.3598010000E+00, 0.5125150000E+00 /
  data (epsData(292,i,56), i=1,3) / 0.7985170000E+03, 0.2992330000E+00, 0.6432890000E+00 /
  data (epsData(293,i,56), i=1,3) / 0.7985790000E+03, 0.2596500000E+00, 0.6878560000E+00 /
  data (epsData(294,i,56), i=1,3) / 0.7988350000E+03, 0.6178230000E-01, 0.7855220000E+00 /
  data (epsData(295,i,56), i=1,3) / 0.7989000000E+03, -0.1126380000E-01, 0.8083150000E+00 /
  data (epsData(296,i,56), i=1,3) / 0.7989670000E+03, -0.8448190000E-01, 0.7829260000E+00 /
  data (epsData(297,i,56), i=1,3) / 0.7991700000E+03, -0.2353500000E+00, 0.7052620000E+00 /
  data (epsData(298,i,56), i=1,3) / 0.7993090000E+03, -0.3154830000E+00, 0.6224190000E+00 /
  data (epsData(299,i,56), i=1,3) / 0.7994500000E+03, -0.3567490000E+00, 0.5295750000E+00 /
  data (epsData(300,i,56), i=1,3) / 0.7996670000E+03, -0.3706990000E+00, 0.4146260000E+00 /
  data (epsData(301,i,56), i=1,3) / 0.7998160000E+03, -0.3597470000E+00, 0.3450460000E+00 /
  data (epsData(302,i,56), i=1,3) / 0.8001200000E+03, -0.3223860000E+00, 0.2593650000E+00 /
  data (epsData(303,i,56), i=1,3) / 0.8005170000E+03, -0.2741850000E+00, 0.1999660000E+00 /
  data (epsData(304,i,56), i=1,3) / 0.8009300000E+03, -0.2361970000E+00, 0.1646330000E+00 /
  data (epsData(305,i,56), i=1,3) / 0.8014480000E+03, -0.2006590000E+00, 0.1395980000E+00 /
  data (epsData(306,i,56), i=1,3) / 0.8020840000E+03, -0.1701600000E+00, 0.1224790000E+00 /
  data (epsData(307,i,56), i=1,3) / 0.8029500000E+03, -0.1423660000E+00, 0.1099420000E+00 /
  data (epsData(308,i,56), i=1,3) / 0.8040840000E+03, -0.1190730000E+00, 0.1013660000E+00 /
  data (epsData(309,i,56), i=1,3) / 0.8057630000E+03, -0.9832360000E-01, 0.9508680000E-01 /
  data (epsData(310,i,56), i=1,3) / 0.8082390000E+03, -0.8151290000E-01, 0.9057060000E-01 /
  data (epsData(311,i,56), i=1,3) / 0.8121630000E+03, -0.6864900000E-01, 0.8673130000E-01 /
  data (epsData(312,i,56), i=1,3) / 0.8204960000E+03, -0.5811860000E-01, 0.8036340000E-01 /
  data (epsData(313,i,56), i=1,3) / 0.8350200000E+03, -0.5244280000E-01, 0.7319080000E-01 /
  data (epsData(314,i,56), i=1,3) / 0.8639680000E+03, -0.4923740000E-01, 0.6281090000E-01 /
  data (epsData(315,i,56), i=1,3) / 0.9185620000E+03, -0.4688880000E-01, 0.5008460000E-01 /
  data (epsData(316,i,56), i=1,3) / 0.9952850000E+03, -0.4276960000E-01, 0.3841630000E-01 /
  data (epsData(317,i,56), i=1,3) / 0.1042620000E+04, -0.3847650000E-01, 0.3293440000E-01 /
  data (epsData(318,i,56), i=1,3) / 0.1057320000E+04, -0.3579130000E-01, 0.3245370000E-01 /
  data (epsData(319,i,56), i=1,3) / 0.1062110000E+04, -0.3446220000E-01, 0.3431330000E-01 /
  data (epsData(320,i,56), i=1,3) / 0.1065690000E+04, -0.3625220000E-01, 0.3554020000E-01 /
  data (epsData(321,i,56), i=1,3) / 0.1076330000E+04, -0.3737240000E-01, 0.3391840000E-01 /
  data (epsData(322,i,56), i=1,3) / 0.1121860000E+04, -0.3626670000E-01, 0.2991380000E-01 /
  data (epsData(323,i,56), i=1,3) / 0.1136860000E+04, -0.3498500000E-01, 0.2996750000E-01 /
  data (epsData(324,i,56), i=1,3) / 0.1148390000E+04, -0.3581970000E-01, 0.2923800000E-01 /
  data (epsData(325,i,56), i=1,3) / 0.1252220000E+04, -0.3307250000E-01, 0.2244370000E-01 /
  data (epsData(326,i,56), i=1,3) / 0.1301170000E+04, -0.3118750000E-01, 0.2063330000E-01 /
  data (epsData(327,i,56), i=1,3) / 0.1458720000E+04, -0.2719990000E-01, 0.1448550000E-01 /
  data (epsData(328,i,56), i=1,3) / 0.1635200000E+04, -0.2270280000E-01, 0.1001040000E-01 /
  data (epsData(329,i,56), i=1,3) / 0.1828750000E+04, -0.1869020000E-01, 0.6933890000E-02 /
  data (epsData(330,i,56), i=1,3) / 0.2034050000E+04, -0.1536490000E-01, 0.4846200000E-02 /
  data (epsData(331,i,56), i=1,3) / 0.2258770000E+04, -0.1258550000E-01, 0.3400600000E-02 /
  data (epsData(332,i,56), i=1,3) / 0.2505500000E+04, -0.1026600000E-01, 0.2379550000E-02 /
  data (epsData(333,i,56), i=1,3) / 0.2784520000E+04, -0.8312830000E-02, 0.1656260000E-02 /
  data (epsData(334,i,56), i=1,3) / 0.3092420000E+04, -0.6717340000E-02, 0.1148730000E-02 /
  data (epsData(335,i,56), i=1,3) / 0.3430890000E+04, -0.5418330000E-02, 0.7970720000E-03 /
  data (epsData(336,i,56), i=1,3) / 0.3807760000E+04, -0.4348320000E-02, 0.5508720000E-03 /
  data (epsData(337,i,56), i=1,3) / 0.4231170000E+04, -0.3456800000E-02, 0.3783450000E-03 /
  data (epsData(338,i,56), i=1,3) / 0.4732790000E+04, -0.2660490000E-02, 0.2531630000E-03 /
  data (epsData(339,i,56), i=1,3) / 0.5142300000E+04, -0.2054260000E-02, 0.1877290000E-03 /
  data (epsData(340,i,56), i=1,3) / 0.5213560000E+04, -0.1855540000E-02, 0.1786500000E-03 /
  data (epsData(341,i,56), i=1,3) / 0.5232060000E+04, -0.1728100000E-02, 0.1901020000E-03 /
  data (epsData(342,i,56), i=1,3) / 0.5239000000E+04, -0.1627620000E-02, 0.2059800000E-03 /
  data (epsData(343,i,56), i=1,3) / 0.5241320000E+04, -0.1567450000E-02, 0.2222220000E-03 /
  data (epsData(344,i,56), i=1,3) / 0.5243640000E+04, -0.1478060000E-02, 0.2638190000E-03 /
  data (epsData(345,i,56), i=1,3) / 0.5245960000E+04, -0.1387770000E-02, 0.4276440000E-03 /
  data (epsData(346,i,56), i=1,3) / 0.5247200000E+04, -0.1423940000E-02, 0.6312900000E-03 /
  data (epsData(347,i,56), i=1,3) / 0.5248140000E+04, -0.1574750000E-02, 0.6912360000E-03 /
  data (epsData(348,i,56), i=1,3) / 0.5249400000E+04, -0.1692810000E-02, 0.6355210000E-03 /
  data (epsData(349,i,56), i=1,3) / 0.5252150000E+04, -0.1750430000E-02, 0.5527570000E-03 /
  data (epsData(350,i,56), i=1,3) / 0.5263760000E+04, -0.1797470000E-02, 0.5200550000E-03 /
  data (epsData(351,i,56), i=1,3) / 0.5293560000E+04, -0.1859100000E-02, 0.4975560000E-03 /
  data (epsData(352,i,56), i=1,3) / 0.5407540000E+04, -0.1891810000E-02, 0.4684520000E-03 /
  data (epsData(353,i,56), i=1,3) / 0.5580380000E+04, -0.1770930000E-02, 0.4173070000E-03 /
  data (epsData(354,i,56), i=1,3) / 0.5612790000E+04, -0.1679590000E-02, 0.4184080000E-03 /
  data (epsData(355,i,56), i=1,3) / 0.5620590000E+04, -0.1601490000E-02, 0.4475040000E-03 /
  data (epsData(356,i,56), i=1,3) / 0.5622730000E+04, -0.1564410000E-02, 0.5010810000E-03 /
  data (epsData(357,i,56), i=1,3) / 0.5624630000E+04, -0.1599530000E-02, 0.5905560000E-03 /
  data (epsData(358,i,56), i=1,3) / 0.5626880000E+04, -0.1675040000E-02, 0.5822850000E-03 /
  data (epsData(359,i,56), i=1,3) / 0.5635770000E+04, -0.1710030000E-02, 0.5519060000E-03 /
  data (epsData(360,i,56), i=1,3) / 0.5705340000E+04, -0.1759330000E-02, 0.5269870000E-03 /
  data (epsData(361,i,56), i=1,3) / 0.5952020000E+04, -0.1677910000E-02, 0.4535510000E-03 /
  data (epsData(362,i,56), i=1,3) / 0.5986270000E+04, -0.1616370000E-02, 0.4574900000E-03 /
  data (epsData(363,i,56), i=1,3) / 0.5994660000E+04, -0.1623580000E-02, 0.5012290000E-03 /
  data (epsData(364,i,56), i=1,3) / 0.6070670000E+04, -0.1656400000E-02, 0.4867700000E-03 /
  data (epsData(365,i,56), i=1,3) / 0.6668140000E+04, -0.1478050000E-02, 0.3503340000E-03 /
  data (epsData(366,i,56), i=1,3) / 0.7559900000E+04, -0.1195790000E-02, 0.2255950000E-03 /
  data (