/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>

namespace arm_conv {
namespace depthwise {

void sme2_fp32_planar_5x5_s2_4rows_mla_za_impl(
  const float *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const float *weights,
  const float *bias,
  float **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  float act_min,
  float act_max
)
{
  struct Args
  {
    const float *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const float *weights;
    const float *bias;
    long unsigned int input_cols, output_cols;
    float **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
    float clamp_min, clamp_max;
  };

  Args args = { inptr, ld_in_vl, pad_top, 11u - std::min(11u, pad_top + valid_input_rows), pad_left, weights, bias, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels, act_min, act_max };

  __asm__ __volatile__(
    "ldr x5, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x20, #0xb\n"
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x6, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ptrue p2.b\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "ldr x7, [%x[args], %[offsetof_Args_n_channels]]\n"
    "ld1rw { z17.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_min]]\n"
    "sub x20, x20, x5\n"
    "ldr x17, [%x[args], %[offsetof_Args_current_channel]]\n"
    "ld1rw { z16.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_max]]\n"
    "whilelt p1.s, XZR, x7\n"
    "whilelt p9.s, XZR, x20\n"
    "whilelt p8.s, XZR, x6\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "1:"  // Channel loop
    "ldr x20, [%x[args], %[offsetof_Args_bias]]\n"
    "fmov z20.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z20.s }, p1/Z, [x20, x17, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x16, [%x[args], %[offsetof_Args_input_cols]]\n"
    "mov x22, #0xb\n"
    "add x20, x6, x5\n"
    "mov z21.d, z20.d\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "lsl x21, %x[ld_in_row], #0x2\n"
    "mov z22.d, z20.d\n"
    "mov z23.d, z20.d\n"
    "ldr x14, [%x[args], %[offsetof_Args_inptr]]\n"
    "mov x8, #0x0\n"
    "sub x22, x22, x20\n"
    "sub x20, x16, #0x1\n"
    "ldr x13, [%x[args], %[offsetof_Args_output_cols]]\n"
    "orr x20, x20, %x[ld_in_col], LSL #18\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa14149e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "orr x20, x7, x20, LSL #20\n"
    "madd x21, x21, x6, x14\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "lsl x20, x20, #0x2\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "3:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b44abc  // rprfm pldstrm, x20, [x21]\n"
    "add x21, x21, %x[ld_in_col], LSL #2\n"
    "bgt 3b\n"
    "ldr x23, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x21, %x[ld_in_row], #0x2\n"
    ".inst 0xc0040e80  // mova za.d[x8, #0], { z20.d-z23.d }\n"
    "mov x22, #0x4\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    "msub x14, x6, x21, x14\n"
    ".inst 0xc0040e81  // mova za.d[x8, #1], { z20.d-z23.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040e82  // mova za.d[x8, #2], { z20.d-z23.d }\n"
    "ldp x11, x10, [x23], #0x10\n"
    ".inst 0xc0040e83  // mova za.d[x8, #3], { z20.d-z23.d }\n"
    "ldp x9, x28, [x20], #0x10\n"
    ".inst 0xc0040e84  // mova za.d[x8, #4], { z20.d-z23.d }\n"
    "ldp x27, x26, [x23], #0x10\n"
    "ldp x25, x24, [x20], #0x10\n"
    "cbz x21, 5f\n"
    "cmp x21, x22\n"
    "csel x20, x21, x22, LT\n"
    "sub x21, x21, x20\n"
    "sub x22, x22, x20\n"
    "cbz x21, 5f\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    "and x22, x21, #0x1\n"
    "add x21, x21, #0x1\n"
    "lsr x21, x21, #0x1\n"
    "sub x13, x13, x21\n"
    ".inst 0xc1b0ca38  // fclamp { z24.s-z27.s }, z17.s, z16.s\n"
    "4:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1w { z24.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    "st1w { z25.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z26.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z27.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    "bgt 4b\n"
    "5:"  // Left padding: End
    "adds XZR, x6, x5\n"
    "bne 12f\n"
    "cbz x22, 10f\n"
    "cmp x22, #0x1\n"
    "sub x16, x16, x22\n"
    "beq 9f\n"
    "cmp x22, #0x2\n"
    "beq 8f\n"
    "cmp x22, #0x3\n"
    "beq 7f\n"
    "6:"  // Unpadded: 4 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z9.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z12.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z7.s\n"
    ".inst 0xa04049e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z13.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z29.s }, p1/Z, [x20]\n"
    ".inst 0xc1341920  // fmla za.s[x8, 0], { z9.s-z12.s }, z4.s\n"
    ".inst 0xa14049e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1321b20  // fmla za.s[x8, 0], { z25.s-z28.s }, z2.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1311940  // fmla za.s[x8, 0], { z10.s-z13.s }, z1.s\n"
    ".inst 0xc1341b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z4.s\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "7:"  // Unpadded: 3 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z7.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z29.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z8.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z30.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z9.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z31.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z0.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1b80  // fmla za.s[x8, 0], { z28.s-z31.s }, z15.s\n"
    ".inst 0xa04049e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z1.s }, p1/Z, [x20]\n"
    ".inst 0xc13c18e0  // fmla za.s[x8, 0], { z7.s-z10.s }, z12.s\n"
    ".inst 0xa14049e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1331ba0  // fmla za.s[x8, 0], { z29.s-z0.s }, z3.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xa04149e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13e1900  // fmla za.s[x8, 0], { z8.s-z11.s }, z14.s\n"
    ".inst 0xc1351bc0  // fmla za.s[x8, 0], { z30.s-z1.s }, z5.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "8:"  // Unpadded: 2 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z10.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z11.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z12.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z13.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z29.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361b20  // fmla za.s[x8, 0], { z25.s-z28.s }, z6.s\n"
    ".inst 0xa04149e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1371b21  // fmla za.s[x8, 1], { z25.s-z28.s }, z7.s\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z14.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z30.s }, p1/Z, [x20]\n"
    ".inst 0xc1301940  // fmla za.s[x8, 0], { z10.s-z13.s }, z0.s\n"
    ".inst 0xa04149e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1341941  // fmla za.s[x8, 1], { z10.s-z13.s }, z4.s\n"
    ".inst 0xa14049e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1321b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z2.s\n"
    ".inst 0xa14149e2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1361b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z6.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1381960  // fmla za.s[x8, 0], { z11.s-z14.s }, z8.s\n"
    ".inst 0xc1311961  // fmla za.s[x8, 1], { z11.s-z14.s }, z1.s\n"
    ".inst 0xc1321b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z2.s\n"
    ".inst 0xa14149e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1341b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z4.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "9:"  // Unpadded: 1 priming loads
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z3.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z4.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z5.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z6.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z14.s\n"
    ".inst 0xa04149e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13f1b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z15.s\n"
    ".inst 0xa14049e2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z7.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z29.s }, p1/Z, [x20]\n"
    ".inst 0xc1311860  // fmla za.s[x8, 0], { z3.s-z6.s }, z1.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13c1861  // fmla za.s[x8, 1], { z3.s-z6.s }, z12.s\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1391b20  // fmla za.s[x8, 0], { z25.s-z28.s }, z9.s\n"
    ".inst 0xa04149e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13a1b21  // fmla za.s[x8, 1], { z25.s-z28.s }, z10.s\n"
    ".inst 0xa04049ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1311880  // fmla za.s[x8, 0], { z4.s-z7.s }, z1.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc13f1881  // fmla za.s[x8, 1], { z4.s-z7.s }, z15.s\n"
    ".inst 0xc1391b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z9.s\n"
    ".inst 0xa14149e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13d1b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z13.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "10:"  // Unpadded: 0 priming loads
    "cmp x16, #0x2\n"
    "blt 20f\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    "ld1w { z9.s }, p1/Z, [x14]\n"
    "sub x16, x16, #0x2\n"
    "ld1w { z26.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "sub x13, x13, #0x1\n"
    "ld1w { z10.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "lsr x20, x16, #0x1\n"
    "ld1w { z27.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "cmp x20, x13\n"
    "ld1w { z11.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "csel x23, x20, x13, LT\n"
    "ld1w { z28.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z12.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "and x16, x16, #0x1\n"
    "ld1w { z29.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "sub x13, x13, x23\n"
    "ld1w { z13.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z14.s }, p1/Z, [x21]\n"
    "cbz x23, 19f\n"
    "11:"  // Unpadded: Main loop
    ".inst 0xc1381920  // fmla za.s[x8, 0], { z9.s-z12.s }, z8.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    "add x22, x14, %x[ld_in_row], LSL #2\n"
    "subs x23, x23, #0x1\n"
    ".inst 0xc1361921  // fmla za.s[x8, 1], { z9.s-z12.s }, z6.s\n"
    ".inst 0xa04149e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1371922  // fmla za.s[x8, 2], { z9.s-z12.s }, z7.s\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z31.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1351b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z5.s\n"
    "ld1w { z1.s }, p2/Z, [x15, #4, MUL VL]\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1301b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z0.s\n"
    ".inst 0xa14149e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1341b42  // fmla za.s[x8, 2], { z26.s-z29.s }, z4.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z26.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1381940  // fmla za.s[x8, 0], { z10.s-z13.s }, z8.s\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1321941  // fmla za.s[x8, 1], { z10.s-z13.s }, z2.s\n"
    ".inst 0xa04149e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1361942  // fmla za.s[x8, 2], { z10.s-z13.s }, z6.s\n"
    ".inst 0xa04049e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15]\n"
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z0.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1311b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z1.s\n"
    ".inst 0xc1371b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z7.s\n"
    ".inst 0xc1341b62  // fmla za.s[x8, 2], { z27.s-z30.s }, z4.s\n"
    "ld1w { z27.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351960  // fmla za.s[x8, 0], { z11.s-z14.s }, z5.s\n"
    ".inst 0xc1321961  // fmla za.s[x8, 1], { z11.s-z14.s }, z2.s\n"
    ".inst 0xa0414a82  // ld1w { z2.s-z3.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1381962  // fmla za.s[x8, 2], { z11.s-z14.s }, z8.s\n"
    "ld1w { z1.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xa1404a82  // ld1w { z2.s, z10.s }, pn10.b/Z, [x20]\n"
    ".inst 0xc0060c0c  // mova { z12.d-z15.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "addvl x20, x20, #5\n"
    "ld1w { z2.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0040e84  // mova za.d[x8, #4], { z20.d-z23.d }\n"
    ".inst 0xa0404a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20]\n"
    ".inst 0xa0414a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc1331be0  // fmla za.s[x8, 0], { z31.s-z2.s }, z3.s\n"
    "ld1w { z29.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1b0ca2c  // fclamp { z12.s-z15.s }, z17.s, z16.s\n"
    ".inst 0xc13a1be1  // fmla za.s[x8, 1], { z31.s-z2.s }, z10.s\n"
    "ld1w { z3.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xa0404a8a  // ld1w { z10.s-z11.s }, pn10.b/Z, [x20]\n"
    ".inst 0xa0414a88  // ld1w { z8.s-z9.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc1371b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z7.s\n"
    "ld1w { z30.s }, p1/Z, [x22]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "st1w { z12.s }, p1, [x11]\n"
    ".inst 0xc1351b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z5.s\n"
    ".inst 0xa1404a84  // ld1w { z4.s, z12.s }, pn10.b/Z, [x20]\n"
    "add x11, x11, x9, LSL #2\n"
    "st1w { z13.s }, p1, [x10]\n"
    ".inst 0xa0414a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z14.s }, p1, [x27]\n"
    "ld1w { z4.s }, p1/Z, [x22]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z15.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    ".inst 0xc1391800  // fmla za.s[x8, 0], { z0.s-z3.s }, z9.s\n"
    ".inst 0xa0414a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc13b1801  // fmla za.s[x8, 1], { z0.s-z3.s }, z11.s\n"
    ".inst 0xa1404a86  // ld1w { z6.s, z14.s }, pn10.b/Z, [x20]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z9.s }, p1/Z, [x14]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z26.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z5.s\n"
    "ld1w { z10.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z12.s\n"
    "ld1w { z27.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1371820  // fmla za.s[x8, 0], { z1.s-z4.s }, z7.s\n"
    ".inst 0xa04149e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13e1821  // fmla za.s[x8, 1], { z1.s-z4.s }, z14.s\n"
    "ld1w { z11.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z12.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z29.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z13.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z30.s }, p1/Z, [x21]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z14.s }, p1/Z, [x21]\n"
    "bgt 11b\n"
    "b 19f\n"
    "12:"  // Padded
    "cbz x22, 17f\n"
    "cmp x22, #0x1\n"
    "sub x16, x16, x22\n"
    "beq 16f\n"
    "cmp x22, #0x2\n"
    "beq 15f\n"
    "cmp x22, #0x3\n"
    "beq 14f\n"
    "13:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z29.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z8.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z30.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z9.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z31.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z0.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    ".inst 0xc1371ba0  // fmla za.s[x8, 0], { z29.s-z0.s }, z7.s\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z1.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1341900  // fmla za.s[x8, 0], { z8.s-z11.s }, z4.s\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361bc0  // fmla za.s[x8, 0], { z30.s-z1.s }, z6.s\n"
    ".inst 0xa04049ee  // ld1w { z14.s-z15.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z2.s }, p0/Z, [x20]\n"
    ".inst 0xc1341920  // fmla za.s[x8, 0], { z9.s-z12.s }, z4.s\n"
    ".inst 0xc13e1be0  // fmla za.s[x8, 0], { z31.s-z2.s }, z14.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "14:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z30.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z31.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z0.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z1.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    ".inst 0xc13f1bc0  // fmla za.s[x8, 0], { z30.s-z1.s }, z15.s\n"
    "ld1w { z29.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z2.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z12.s\n"
    ".inst 0xa04049ec  // ld1w { z12.s-z13.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z30.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1be0  // fmla za.s[x8, 0], { z31.s-z2.s }, z14.s\n"
    ".inst 0xa04049ea  // ld1w { z10.s-z11.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z3.s }, p0/Z, [x20]\n"
    ".inst 0xa04149e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13d1b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z13.s\n"
    ".inst 0xc13b1800  // fmla za.s[x8, 0], { z0.s-z3.s }, z11.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "15:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z25.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z13.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    ".inst 0xc1361b20  // fmla za.s[x8, 0], { z25.s-z28.s }, z6.s\n"
    "ld1w { z14.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371b21  // fmla za.s[x8, 1], { z25.s-z28.s }, z7.s\n"
    ".inst 0xa14049e2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa14149e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z29.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1301960  // fmla za.s[x8, 0], { z11.s-z14.s }, z0.s\n"
    ".inst 0xa04149e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1341961  // fmla za.s[x8, 1], { z11.s-z14.s }, z4.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z15.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1311b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z1.s\n"
    ".inst 0xa14149e3  // ld1w { z3.s, z11.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1321b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z2.s\n"
    "ld1w { z30.s }, p0/Z, [x20]\n"
    ".inst 0xa04049e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1381980  // fmla za.s[x8, 0], { z12.s-z15.s }, z8.s\n"
    ".inst 0xc1341981  // fmla za.s[x8, 1], { z12.s-z15.s }, z4.s\n"
    ".inst 0xc1331b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z3.s\n"
    ".inst 0xa14149e6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1301b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z0.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "16:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z25.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "ld1w { z8.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z9.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    ".inst 0xc13e1b20  // fmla za.s[x8, 0], { z25.s-z28.s }, z14.s\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1b21  // fmla za.s[x8, 1], { z25.s-z28.s }, z15.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z29.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1311900  // fmla za.s[x8, 0], { z8.s-z11.s }, z1.s\n"
    ".inst 0xa14149e5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13c1901  // fmla za.s[x8, 1], { z8.s-z11.s }, z12.s\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1331b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z3.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc13f1b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z15.s\n"
    "ld1w { z30.s }, p0/Z, [x20]\n"
    ".inst 0xa04049e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13d1920  // fmla za.s[x8, 0], { z9.s-z12.s }, z13.s\n"
    ".inst 0xc1351921  // fmla za.s[x8, 1], { z9.s-z12.s }, z5.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1311b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z1.s\n"
    ".inst 0xa04149e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1331b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z3.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa04049e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "17:"  // Padded: 0 priming loads
    "cmp x16, #0x2\n"
    "blt 20f\n"
    "mov x12, #0x0\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x16, x16, #0x2\n"
    "sub x13, x13, #0x1\n"
    "lsr x20, x16, #0x1\n"
    "cmp x20, x13\n"
    "and x16, x16, #0x1\n"
    "ld1w { z9.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "csel x23, x20, x13, LT\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    "sub x13, x13, x23\n"
    "ld1w { z26.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z10.s }, p0/Z, [x21]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z27.s }, p0/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z11.s }, p0/Z, [x21]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z12.s }, p0/Z, [x21]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    "ld1w { z29.s }, p0/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z13.s }, p0/Z, [x21]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z30.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z14.s }, p0/Z, [x21]\n"
    "cbz x23, 19f\n"
    "18:"  // Padded: Main loop
    ".inst 0xc1381920  // fmla za.s[x8, 0], { z9.s-z12.s }, z8.s\n"
    "mov x12, #0x0\n"
    "ld1w { z15.s }, p2/Z, [x15, #4, MUL VL]\n"
    "add x22, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361921  // fmla za.s[x8, 1], { z9.s-z12.s }, z6.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xa04149e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "subs x23, x23, #0x1\n"
    ".inst 0xc1371922  // fmla za.s[x8, 2], { z9.s-z12.s }, z7.s\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1351b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z5.s\n"
    "ld1w { z8.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1301b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z0.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1341b42  // fmla za.s[x8, 2], { z26.s-z29.s }, z4.s\n"
    ".inst 0xa14049e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15]\n"
    "ld1w { z31.s }, p0/Z, [x22]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1940  // fmla za.s[x8, 0], { z10.s-z13.s }, z15.s\n"
    "ld1w { z15.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1321941  // fmla za.s[x8, 1], { z10.s-z13.s }, z2.s\n"
    ".inst 0xa04149e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1361942  // fmla za.s[x8, 2], { z10.s-z13.s }, z6.s\n"
    "ld1w { z9.s }, p0/Z, [x22]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    ".inst 0xa14049e2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13f1b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z15.s\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1301b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z0.s\n"
    ".inst 0xc1311b62  // fmla za.s[x8, 2], { z27.s-z30.s }, z1.s\n"
    "ld1w { z0.s }, p0/Z, [x22]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351960  // fmla za.s[x8, 0], { z11.s-z14.s }, z5.s\n"
    ".inst 0xc1341961  // fmla za.s[x8, 1], { z11.s-z14.s }, z4.s\n"
    ".inst 0xa0414a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1321962  // fmla za.s[x8, 2], { z11.s-z14.s }, z2.s\n"
    "ld1w { z10.s }, p0/Z, [x22]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xa0404a82  // ld1w { z2.s-z3.s }, pn10.b/Z, [x20]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    ".inst 0xa0404a8e  // ld1w { z14.s-z15.s }, pn10.b/Z, [x20]\n"
    "ld1w { z1.s }, p0/Z, [x22]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0040e84  // mova za.d[x8, #4], { z20.d-z23.d }\n"
    ".inst 0xa1414a86  // ld1w { z6.s, z14.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "ld1w { z11.s }, p0/Z, [x22]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1b0ca38  // fclamp { z24.s-z27.s }, z17.s, z16.s\n"
    "mov x12, #0x8\n"
    "ld1w { z2.s }, p0/Z, [x22]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371900  // fmla za.s[x8, 0], { z8.s-z11.s }, z7.s\n"
    ".inst 0xc1331901  // fmla za.s[x8, 1], { z8.s-z11.s }, z3.s\n"
    ".inst 0xa0404a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20]\n"
    "st1w { z24.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xa0414a8c  // ld1w { z12.s-z13.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "st1w { z25.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "ld1w { z12.s }, p0/Z, [x22]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "st1w { z26.s }, p1, [x27]\n"
    ".inst 0xc13e1be0  // fmla za.s[x8, 0], { z31.s-z2.s }, z14.s\n"
    ".inst 0xa1414a86  // ld1w { z6.s, z14.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z27.s }, p1, [x26]\n"
    ".inst 0xc13f1be1  // fmla za.s[x8, 1], { z31.s-z2.s }, z15.s\n"
    ".inst 0xa1404a87  // ld1w { z7.s, z15.s }, pn10.b/Z, [x20]\n"
    "addvl x20, x20, #5\n"
    "add x26, x26, x24, LSL #2\n"
    "ld1w { z3.s }, p0/Z, [x22]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x22, x22, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x0\n"
    ".inst 0xc13d1920  // fmla za.s[x8, 0], { z9.s-z12.s }, z13.s\n"
    ".inst 0xa0414a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "ld1w { z13.s }, p0/Z, [x22]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1351921  // fmla za.s[x8, 1], { z9.s-z12.s }, z5.s\n"
    ".inst 0xa0404a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z9.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc13e1800  // fmla za.s[x8, 0], { z0.s-z3.s }, z14.s\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13f1801  // fmla za.s[x8, 1], { z0.s-z3.s }, z15.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    "ld1w { z26.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371940  // fmla za.s[x8, 0], { z10.s-z13.s }, z7.s\n"
    ".inst 0xa04149e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1351941  // fmla za.s[x8, 1], { z10.s-z13.s }, z5.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z10.s }, p0/Z, [x21]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z27.s }, p0/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z11.s }, p0/Z, [x21]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z12.s }, p0/Z, [x21]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    "ld1w { z29.s }, p0/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z13.s }, p0/Z, [x21]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z30.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "ld1w { z14.s }, p0/Z, [x21]\n"
    "bgt 18b\n"
    "19:"  // Main loop tail
    ".inst 0xc1381920  // fmla za.s[x8, 0], { z9.s-z12.s }, z8.s\n"
    "mov x12, #0x0\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    "add x21, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361921  // fmla za.s[x8, 1], { z9.s-z12.s }, z6.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xa04149e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1371922  // fmla za.s[x8, 2], { z9.s-z12.s }, z7.s\n"
    ".inst 0xa04049e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1351b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z5.s\n"
    "ld1w { z9.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x14, x14, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1301b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z0.s\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1341b42  // fmla za.s[x8, 2], { z26.s-z29.s }, z4.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "ld1w { z31.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1381940  // fmla za.s[x8, 0], { z10.s-z13.s }, z8.s\n"
    "ld1w { z15.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xc1321941  // fmla za.s[x8, 1], { z10.s-z13.s }, z2.s\n"
    ".inst 0xa04149e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1361942  // fmla za.s[x8, 2], { z10.s-z13.s }, z6.s\n"
    "ld1w { z10.s }, p0/Z, [x21]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    ".inst 0xa04049e2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13f1b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z15.s\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1301b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z0.s\n"
    ".inst 0xc1371b62  // fmla za.s[x8, 2], { z27.s-z30.s }, z7.s\n"
    "ld1w { z0.s }, p0/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361960  // fmla za.s[x8, 0], { z11.s-z14.s }, z6.s\n"
    ".inst 0xc1341961  // fmla za.s[x8, 1], { z11.s-z14.s }, z4.s\n"
    ".inst 0xa0414a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1321962  // fmla za.s[x8, 2], { z11.s-z14.s }, z2.s\n"
    "ld1w { z11.s }, p0/Z, [x21]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xa0404a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20]\n"
    "addvl x20, x20, #5\n"
    ".inst 0xc0060c18  // mova { z24.d-z27.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    ".inst 0xa0404a82  // ld1w { z2.s-z3.s }, pn10.b/Z, [x20]\n"
    "ld1w { z1.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0040e84  // mova za.d[x8, #4], { z20.d-z23.d }\n"
    ".inst 0xa0414a8e  // ld1w { z14.s-z15.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "ld1w { z12.s }, p0/Z, [x21]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1b0ca38  // fclamp { z24.s-z27.s }, z17.s, z16.s\n"
    "mov x12, #0x8\n"
    ".inst 0xc1351920  // fmla za.s[x8, 0], { z9.s-z12.s }, z5.s\n"
    "ld1w { z2.s }, p0/Z, [x21]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371921  // fmla za.s[x8, 1], { z9.s-z12.s }, z7.s\n"
    ".inst 0xa0404a88  // ld1w { z8.s-z9.s }, pn10.b/Z, [x20]\n"
    "st1w { z24.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    ".inst 0xa1414a86  // ld1w { z6.s, z14.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "addvl x20, x20, #5\n"
    "st1w { z25.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "ld1w { z13.s }, p0/Z, [x21]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    "st1w { z26.s }, p1, [x27]\n"
    ".inst 0xc13f1be0  // fmla za.s[x8, 0], { z31.s-z2.s }, z15.s\n"
    ".inst 0xa0414a84  // ld1w { z4.s-z5.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z27.s }, p1, [x26]\n"
    ".inst 0xc1331be1  // fmla za.s[x8, 1], { z31.s-z2.s }, z3.s\n"
    ".inst 0xa0404a86  // ld1w { z6.s-z7.s }, pn10.b/Z, [x20]\n"
    "addvl x20, x20, #5\n"
    "add x26, x26, x24, LSL #2\n"
    "ld1w { z3.s }, p0/Z, [x21]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x21, x21, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1940  // fmla za.s[x8, 0], { z10.s-z13.s }, z14.s\n"
    ".inst 0xa0414a8e  // ld1w { z14.s-z15.s }, pn10.b/Z, [x20, #0x2, MUL VL]\n"
    ".inst 0xc1391941  // fmla za.s[x8, 1], { z10.s-z13.s }, z9.s\n"
    "ld1w { z14.s }, p0/Z, [x21]\n"
    ".inst 0xa0404a88  // ld1w { z8.s-z9.s }, pn10.b/Z, [x20]\n"
    "ldr x15, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1351800  // fmla za.s[x8, 0], { z0.s-z3.s }, z5.s\n"
    ".inst 0xc1371801  // fmla za.s[x8, 1], { z0.s-z3.s }, z7.s\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc13f1960  // fmla za.s[x8, 0], { z11.s-z14.s }, z15.s\n"
    ".inst 0xa04149e6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1391961  // fmla za.s[x8, 1], { z11.s-z14.s }, z9.s\n"
    ".inst 0xa14049e7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x15]\n"
    "addvl x15, x15, #5\n"
    ".inst 0xa14049e4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x15]\n"
    ".inst 0xa04149e0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "ld1w { z5.s }, p2/Z, [x15, #4, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "20:"  // Main loop skip tail
    "cbz x16, 21f\n"  // Skip remainder inputs
    "mov x12, #0x0\n"
    "add x20, x14, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x13, x13, #0x1\n"
    "ld1w { z25.s }, p0/Z, [x14]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z11.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z12.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z27.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z13.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z28.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x8\n"
    ".inst 0xc1381b20  // fmla za.s[x8, 0], { z25.s-z28.s }, z8.s\n"
    "ld1w { z14.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z8.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1361b21  // fmla za.s[x8, 1], { z25.s-z28.s }, z6.s\n"
    ".inst 0xc1371b22  // fmla za.s[x8, 2], { z25.s-z28.s }, z7.s\n"
    ".inst 0xa14049e1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x15]\n"
    "ld1w { z29.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351960  // fmla za.s[x8, 0], { z11.s-z14.s }, z5.s\n"
    ".inst 0xa14149e2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z10.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1301961  // fmla za.s[x8, 1], { z11.s-z14.s }, z0.s\n"
    "ld1w { z15.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1341962  // fmla za.s[x8, 2], { z11.s-z14.s }, z4.s\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049e3  // ld1w { z3.s, z11.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc1381b40  // fmla za.s[x8, 0], { z26.s-z29.s }, z8.s\n"
    ".inst 0xa14149e0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    "addvl x15, x15, #5\n"
    "ld1w { z30.s }, p0/Z, [x20]\n"
    ".inst 0xc1321b41  // fmla za.s[x8, 1], { z26.s-z29.s }, z2.s\n"
    "ld1w { z6.s }, p2/Z, [x15, #4, MUL VL]\n"
    ".inst 0xc1311b42  // fmla za.s[x8, 2], { z26.s-z29.s }, z1.s\n"
    ".inst 0xa04049e8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x15]\n"
    ".inst 0xc13a1980  // fmla za.s[x8, 0], { z12.s-z15.s }, z10.s\n"
    ".inst 0xa04149e4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x15, #0x2, MUL VL]\n"
    ".inst 0xc1301981  // fmla za.s[x8, 1], { z12.s-z15.s }, z0.s\n"
    ".inst 0xc1331982  // fmla za.s[x8, 2], { z12.s-z15.s }, z3.s\n"
    ".inst 0xc1361b60  // fmla za.s[x8, 0], { z27.s-z30.s }, z6.s\n"
    ".inst 0xc1341b61  // fmla za.s[x8, 1], { z27.s-z30.s }, z4.s\n"
    ".inst 0xc1381b62  // fmla za.s[x8, 2], { z27.s-z30.s }, z8.s\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc0040e84  // mova za.d[x8, #4], { z20.d-z23.d }\n"
    ".inst 0xc1b0ca24  // fclamp { z4.s-z7.s }, z17.s, z16.s\n"
    "st1w { z4.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    "st1w { z5.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z6.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z7.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    "21:"  // Tail input: End
    "cbz x13, 23f\n"
    "22:"  // Right padding loop
    ".inst 0xc0060c08  // mova { z8.d-z11.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "subs x13, x13, #0x1\n"
    ".inst 0xc0040e84  // mova za.d[x8, #4], { z20.d-z23.d }\n"
    ".inst 0xc1b0ca28  // fclamp { z8.s-z11.s }, z17.s, z16.s\n"
    "st1w { z8.s }, p1, [x11]\n"
    "add x11, x11, x9, LSL #2\n"
    "st1w { z9.s }, p1, [x10]\n"
    "add x10, x10, x28, LSL #2\n"
    "st1w { z10.s }, p1, [x27]\n"
    "add x27, x27, x25, LSL #2\n"
    "st1w { z11.s }, p1, [x26]\n"
    "add x26, x26, x24, LSL #2\n"
    "bgt 22b\n"
    "23:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x17\n"
    "whilelt p1.s, x17, x7\n"
    "incb x20, ALL, MUL #16\n"
    "incb x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21, LSL #2\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_bias] "I" (offsetof(Args, bias)), [offsetof_Args_clamp_max] "I" (offsetof(Args, clamp_max)), [offsetof_Args_clamp_min] "I" (offsetof(Args, clamp_min)), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights))
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
