# arch-tag: a61e4b6e-1447-4704-a7f3-947a2ffe11de
# Copyright (C) 2003--2005 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Internal module providing patchlog parsing.

This module implements some of public interface for the
pybaz_ package. But for convenience reasons the author prefers
to store this code in a file separate from ``__init__.py``.

.. _pybaz: pybaz-module.html

This module is strictly internal and should never be used.
"""

import os
import email
import email.Parser

import errors
from pathname import FileName
from _nameparser import NameParser
from _escaping import name_unescape
from deprecation import deprecated_callable

__all__ = [
    'Patchlog',
    ]


def _factory():
    import pybaz
    return pybaz.factory

def _is_unsafe(obj):
    import _builtin
    return isinstance(obj, _builtin._unsafe)

def _backend():
    import pybaz
    return pybaz.backend


class Patchlog(object):

    """Log entry associated to a revision.

    May be produced by `Revision.patchlog` or `ArchSourceTree.iter_logs()`. It
    provides an extensive summary of the associated changeset, a natural
    language description of the changes, and any number of user-defined
    extension headers.

    Patchlogs are formatted according to RFC-822, and are parsed using the
    standard email-handling facilities.

    Note that the patchlog text is not actually parsed before it is needed.
    That deferred evaluation feature is implemented in the `_parse` method.

    The fundamental accessors are `__getitem__`, which give the text of a named
    patchlog header, and the `description` property which gives the text of the
    patchlog body, that is anything after the headers.

    Additional properties provide appropriate standard conversion of the
    standard headers.
    """

    def __init__(self, revision, tree=None, fromlib=False):
        """Patchlog associated to the given revision.

        The patchlog may be retrieved from the provided ``tree``, from the
        revision library if ``fromlib`` is set, or from the archive.

        :param tree: source tree to retrieve the patchlog from.
        :type tree: `ArchSourceTree`, None
        :param fromlib: retrieve the patchlog from the revision library.
        :type fromlib: bool
        :raise ValueError: tree and fromlib are both set.
        """
        if tree is not None and fromlib:
            raise ValueError("both tree and fromlib arguments are set.")
        if _factory().isRevision(revision):
            self.__revision = revision.fullname
        elif _is_unsafe(revision):
            self.__revision, = revision
        else:
            p = NameParser(revision)
            if not p.has_archive() or not p.has_patchlevel:
                raise errors.NamespaceError(revision,
                                            'fully-qualified revision')
            self.__revision = revision
        if tree is None:
            self.__tree = None
        else:
            self.__tree = str(tree)
        self.__fromlib = bool(fromlib)
        self.__email = None

    def tree(self): return self.__tree
    tree = property(tree)

    def fromlib(self): return self.__fromlib
    fromlib = property(fromlib)

    def __repr__(self):
        fromtree = self.tree is not None
        fromlib = self.fromlib
        if not fromtree and not fromlib:
            return 'Patchlog(%r)' % (self.__revision,)
        elif fromtree and not fromlib:
            return 'Patchlog(%r, tree=%r)' % (self.__revision, self.tree)
        elif not fromtree and fromlib:
            return 'Patchlog(%r, fromlib=True)' % (self.__revision,)
        else:
            raise AssertionError

    def _parse(self):
        """Deferred parsing of the log text."""
        if self.__email is not None:
            return self.__email
        if self.tree is not None:
            revision = _factory().Revision(self.__revision)
            log_path = os.path.join(
                str(self.tree), '{arch}', revision.category.nonarch,
                revision.branch.nonarch, revision.version.nonarch,
                revision.archive.name, 'patch-log', revision.patchlevel)
            s = open(log_path).read()
        elif self.fromlib:
            s = _backend().text_cmd(('library-log', self.__revision))
        else:
            s = _backend().text_cmd(('cat-archive-log', self.__revision))
        self.__email = email.Parser.Parser().parsestr(s)
        return self.__email

    def __getitem__(self, header):
        """Text of a patchlog header by name.

        :param header: name of a patchlog header.
        :type header: str
        :return: text of the header, or None if the header is not present.
        :rtype: str, None
        """
        return self._parse()[header]

    def items(self):
        """List of 2-tuples containing all headers and values.

        :rtype: list of 2-tuple of str
        """
        return self._parse().items()

    def get_revision(self):
        """Deprecated.

        Revision associated to this patchlog.

        :rtype: `Revision`
        :see: `Patchlog.revision`
        """
        deprecated_callable(self.get_revision, (Patchlog, 'revision'))
        return self.revision

    def _get_revision(self):
        assert self.__revision == self['Archive']+'/'+self['Revision']
        return _factory().Revision(self.__revision)

    revision = property(_get_revision, doc="""
    Revision associated to this patchlog.

    :type: `Revision`
    """)

    def get_summary(self):
        """Deprecated.

        Patchlog summary, a one-line natural language description.

        :rtype: str
        :see: `Patchlog.summary`
        """
        deprecated_callable(self.get_summary, (Patchlog, 'summary'))
        return self.summary

    def _get_summary(self):
        return self['Summary']

    summary = property(_get_summary, doc="""
    Patchlog summary, a one-line natural language description.

    :type: str
    """)

    def get_description(self):
        """Deprecated.

        Patchlog body, a long natural language description.

        :rtype: str
        :see: `Patchlog.description`
        """
        deprecated_callable(self.get_description, (Patchlog, 'description'))
        return self.description

    def _get_description(self):
        m = self._parse()
        assert not m.is_multipart()
        return m.get_payload()

    description = property(_get_description, doc="""
    Patchlog body, a long natural language description.

    :type: str
    """)

    def get_date(self):
        """Deprecated.

        For the description of the local time tuple, see the
        documentation of the `time` module.

        :rtype: local time tuple
        :see: `Patchlog.date`
        """
        deprecated_callable(self.get_date, (Patchlog, 'date'))
        return self.date

    def _get_date(self):
        d = self['Standard-date']
        from time import strptime
        return strptime(d, '%Y-%m-%d %H:%M:%S %Z')

    date = property(_get_date, doc="""
    Time of the associated revision.

    For the description of the local time tuple, see the documentation
    of the `time` module.

    :type: local time tuple
    """)

    def get_creator(self):
        """Deprecated.

        User id of the the creator of the associated revision.

        :rtype: str
        :see: `Patchlog.creator`
        """
        deprecated_callable(self.get_creator, (Patchlog, 'creator'))
        return self.creator

    def _get_creator(self):
        return self['Creator']

    creator = property(_get_creator, doc="""
    User id of the the creator of the associated revision.

    :type: str
    """)

    def get_continuation(self):
        """Deprecated.

        Ancestor of tag revisions.
        None for commit and import revisions.

        :rtype: `Revision`, None.
        :see: `Patchlog.continuation_of`
        """
        deprecated_callable(
            self.get_continuation, (Patchlog, 'continuation_of'))
        return self.continuation_of

    def _get_continuation(self):
        deprecated_callable(
            (Patchlog, 'continuation'), (Patchlog, 'continuation_of'))
        return self.continuation_of

    continuation = property(_get_continuation, doc="""
    Deprecated.

    Ancestor of tag revisions.
    None for commit and import revisions.

    :type: `Revision`, None.
    :see: `Patchlog.continuation_of`
    """)

    def _get_continuation_of(self):
        c = self['Continuation-of']
        if c is None:
            return None
        return _factory().Revision(c)

    continuation_of = property(_get_continuation_of, doc="""
    Ancestor of tag revisions.
    None for commit and import revisions.

    :type: `Revision`, None.
    """)

    def get_new_patches(self):
        """Deprecated.

        New-patches header as an iterable of Revision.

        :rtype: iterable of `Revision`
        :see: `Patchlog.new_patches`
        """
        deprecated_callable(
            Patchlog.get_new_patches, (Patchlog, 'new_patches'))
        return self.new_patches

    def _get_new_patches(self):
        patches = self['New-patches'].split()
        if '!!!!!nothing-should-depend-on-this' in patches:
            patches.remove('!!!!!nothing-should-depend-on-this')
            patches.append("%s" % self.revision)
        return map(_factory().Revision, patches)

    new_patches = property(_get_new_patches, doc="""
    New-patches header as an iterable of Revision.

    Patchlogs added in this revision.

    :type: iterable of `Revision`
    """)

    def get_merged_patches(self):
        """Deprecated.

        Revisions merged in this revision. That is the revisions
        listed in the New-patches header except the revision of the
        patchlog.

        :rtype: iterable of `Revision`
        :see: `Patchlog.merged_patches`
        """
        deprecated_callable(
            Patchlog.get_merged_patches, (Patchlog, 'merged_patches'))
        return self.merged_patches

    def _get_merged_patches(self):
        rvsn = self.revision.fullname
        return filter(lambda(r): r.fullname != rvsn, self.new_patches)

    merged_patches = property(_get_merged_patches, doc="""
    Revisions merged in this revision.

    That is the revisions listed in the New-patches header except the
    revision of the patchlog.

    :type: iterable of `Revision`
    """)

    def get_new_files(self):
        """Deprecated.

        Source files added in the associated revision.

        :rtype: iterable of `FileName`
        :see: `Patchlog.new_files`
        """
        deprecated_callable(self.get_new_files, (Patchlog, 'new_files'))
        return self.new_files

    def _get_new_files(self):
        s = self['New-files']
        if s is None: return []
        return [ FileName(name_unescape(f)) for f in s.split() ]

    new_files = property(_get_new_files, doc="""
    Source files added in the associated revision.

    :type: iterable of `FileName`
    """)

    def get_modified_files(self):
        """Deprecated.

        Names of source files modified in the associated revision.

        :rtype: iterable of `FileName`
        """
        deprecated_callable(
            self.get_modified_files, (Patchlog, 'modified_files'))
        return self.modified_files

    def _get_modified_files(self):
        s = self['Modified-files']
        if s is None: return []
        return [ FileName(name_unescape(f)) for f in s.split() ]

    modified_files = property(_get_modified_files, doc="""
    Names of source files modified in the associated revision.

    :type: iterable of `FileName`
    """)

    def get_renamed_files(self):
        """Deprecated.

        Source files renames in the associated revision.

        Dictionnary whose keys are old names and whose values are the
        corresponding new names. Explicit file ids are listed in
        addition to their associated source file.

        :rtype: dict
        """
        deprecated_callable(
            self.get_renamed_files, (Patchlog, 'renamed_files'))
        return self.renamed_files

    def _get_renamed_files(self):
        renamed_files = self['Renamed-files']
        renames = {}
        if renamed_files is None:
            return renames
        names = [name_unescape(name) for name in renamed_files.split()]
        for i in range(len(names)/2):
            renames[names[i*2]] = names[i*2+1]
        return renames

    renamed_files = property(_get_renamed_files, doc="""
    Source files renames in the associated revision.

    Dictionnary whose keys are old names and whose values are the
    corresponding new names. Explicit file ids are listed in
    addition to their associated source file.

    :type: dict
    """)

    def get_removed_files(self):
        """Deprecated.

        Names of source files removed in the associated revision.

        :rtype: iterable of `FileName`
        """
        deprecated_callable(
            self.get_removed_files, (Patchlog, 'removed_files'))
        return self.removed_files

    def _get_removed_files(self):
        s = self['Removed-files']
        if s is None: return []
        return [ FileName(name_unescape(f)) for f in s.split() ]

    removed_files = property(_get_removed_files, doc="""
    Names of source files removed in the associated revision.

    :type: iterable of `FileName`
    """)
