/*
 *  xfce-file-chooser - a wrapper around GtkFileChooser(Dialog) (gtk 2.4+)
 *                      and/or GtkFileChooser (gtk 2.0, 2.2)
 *
 *  Copyright (c) 2004 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if defined(HAVE_STDARG_H)
#include <stdarg.h>
#elif defined(HAVE_VARARGS_H)
#include <varargs.h>
#endif

#include <gtk/gtkversion.h>

#include "xfce-filechooser.h"

#if GTK_CHECK_VERSION(2, 4, 0)
#include <gtk/gtkfilechooserdialog.h>
#include <gtk/gtkfilefilter.h>
#else
#include <gtk/gtk.h>
#endif

#if !GTK_CHECK_VERSION(2, 4, 0)
enum {
	SHOW_PREVIEW_LABEL = 1,
	HIDE_PREVIEW_LABEL
};
#endif

#if !GTK_CHECK_VERSION(2, 4, 0)
static void
_multiple_foreach(GtkTreeModel *model, GtkTreePath  *path, GtkTreeIter  *itr,
		gpointer data)
{
	gchar **filename = data;
	
	if(!*filename)
		gtk_tree_model_get(model, itr, 0, filename, -1);
}

static gchar *
_get_label_from_treesel(GtkTreeSelection *selection)
{
	gchar *filename = NULL;
	
	gtk_tree_selection_selected_foreach(selection, _multiple_foreach, &filename);
	
	return filename;
}
#endif

static void
#if GTK_CHECK_VERSION(2, 4, 0)
_preview_needs_update(GtkFileChooser *chooser, gpointer user_data)
#else
_preview_needs_update(GtkTreeSelection *selection, gpointer user_data)
#endif
{
	PreviewUpdateFunc func = NULL;
	gpointer data = NULL;
#if !GTK_CHECK_VERSION(2, 4, 0)
	XfceFileChooser *chooser = XFCE_FILE_CHOOSER(user_data);
	GtkWidget *frame = g_object_get_data(G_OBJECT(chooser), "xfce-fc-pframe");
	
	if(GPOINTER_TO_INT(g_object_get_data(G_OBJECT(chooser), "xfce-fc-plabel"))
			== HIDE_PREVIEW_LABEL)
	{
		gtk_frame_set_label(GTK_FRAME(frame), NULL);
	} else {
		gchar *filename = _get_label_from_treesel(selection);
		gtk_frame_set_label(GTK_FRAME(frame), filename);
		if(filename)
			g_free(filename);
	}
		
#endif
	
	func = g_object_get_data(G_OBJECT(chooser), "xfce-fc-previewfunc");
	data = g_object_get_data(G_OBJECT(chooser), "xfce-fc-previewdata");
	
	if(func)
		(*func)(XFCE_FILE_CHOOSER(chooser), data);
}

static XfceFileChooser *
xfce_file_chooser_new_valist(const gchar *title, GtkWindow *parent,
		XfceFileChooserAction action, const gchar *first_button_text,
		va_list ap)
{
	XfceFileChooser *chooser;
#if GTK_CHECK_VERSION(2, 4, 0)

	chooser = XFCE_FILE_CHOOSER(gtk_file_chooser_dialog_new(title, parent,
			action, NULL));
	
	if(first_button_text) {
		const gchar *text = first_button_text;
		GtkResponseType resp;
		
		while(text) {
			resp = va_arg(ap, gint);
			gtk_dialog_add_button(GTK_DIALOG(chooser), text, resp);
			text = va_arg(ap, const gchar *);
		}
	}
	
	g_signal_connect(G_OBJECT(chooser), "update-preview",
			G_CALLBACK(_preview_needs_update), NULL);
#else
	GtkTreeSelection *sel;
	GtkWidget *action_area;
	GList *buttons, *l;
	
	chooser = XFCE_FILE_CHOOSER(gtk_file_selection_new(title));

	if(parent) {
		gtk_window_set_transient_for(GTK_WINDOW(chooser), parent);
		gtk_window_set_destroy_with_parent(GTK_WINDOW(chooser), TRUE);
	}
	gtk_dialog_set_has_separator(GTK_DIALOG(chooser), FALSE);
	
	/* i dislike just ditching the buttons gtk 2.2 gives us, but i want to make
	 * sure the caller is getting the right responses that s/he expects */
	action_area = GTK_DIALOG(chooser)->action_area;
	buttons = gtk_container_get_children(GTK_CONTAINER(action_area));
	for(l = buttons; l; l = l->next) {
		if(l->data)
			gtk_container_remove(GTK_CONTAINER(action_area), GTK_WIDGET(l->data));
	}
	
	if(first_button_text) {
		const gchar *text = first_button_text;
		GtkResponseType resp;
		
		while(text) {
			resp = va_arg(ap, gint);
			gtk_dialog_add_button(GTK_DIALOG(chooser), text, resp);
			text = va_arg(ap, const gchar *);
		}
	}
	
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(GTK_FILE_SELECTION(chooser)->file_list));
	g_signal_connect(G_OBJECT(sel), "changed",
			G_CALLBACK(_preview_needs_update), chooser); 
#endif
	
	return chooser;
}

/**
 * xfce_file_chooser_dialog_new:
 * @title: The dialog's window title, or %NULL.
 * @parent: The dialog's transient parent window, or %NULL.
 * @action: Open or save mode for dialog.
 * @first_button_text: Stock ID or text for the first button.
 * @...: Response ID for the first button, followed by additional (button, ID)
 *       pairs, terminated with a %NULL argument.
 *
 * Deprecated: Use xfce_file_chooser_new() instead.
 *
 * Return value: a new #XfceFileChooser dialog widget.
 **/
GtkWidget *
xfce_file_chooser_dialog_new(const gchar *title, GtkWindow *parent, 
		XfceFileChooserAction action, const gchar *first_button_text, ...)
{
	XfceFileChooser *chooser;
	va_list ap;
	
	va_start(ap, first_button_text);
	chooser = xfce_file_chooser_new_valist(title, parent, action,
			first_button_text, ap);
	va_end(ap);
	
	return GTK_WIDGET(chooser);
}

/**
 * xfce_file_chooser_new:
 * @title: The dialog's window title, or %NULL.
 * @parent: The dialog's transient parent window, or %NULL.
 * @action: Open or save mode for dialog.
 * @first_button_text: Stock ID or text for the first button.
 * @...: Response ID for the first button, followed by additional (button, ID)
 *       pairs, terminated with a %NULL argument.
 *
 * Constructs a new file chooser dialog.  If compiled against GTK+ 2.4 (or
 * above), this will create a
 * <ulink url="http://developer.gnome.org/doc/API/2.0/gtk/GtkFileChooserDialog.html">GtkFileChooserDialog</ulink>.
 * For GTK+ 2.0 and 2.2, this creates a
 * <ulink url="http://developer.gnome.org/doc/API/2.0/gtk/GtkFileSelection.html">GtkFileSelection</ulink>.
 *
 * Since 4.1
 *
 * Return value: a new #XfceFileChooser dialog widget.
 **/
GtkWidget *
xfce_file_chooser_new(const gchar *title, GtkWindow *parent, 
		XfceFileChooserAction action, const gchar *first_button_text, ...)
{
	XfceFileChooser *chooser;
	va_list ap;
	
	va_start(ap, first_button_text);
	chooser = xfce_file_chooser_new_valist(title, parent, action,
			first_button_text, ap);
	va_end(ap);
	
	return GTK_WIDGET(chooser);
}

/**
 * xfce_file_chooser_set_select_multiple:
 * @chooser: An #XfceFileChooser.
 * @select_multiple: %TRUE it multiple files can be selected.
 *
 * Sets whether multiple files can be selected in the file chooser.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_select_multiple(XfceFileChooser *chooser,
		gboolean select_multiple)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(chooser), select_multiple);
#else
	gtk_file_selection_set_select_multiple(GTK_FILE_SELECTION(chooser), select_multiple);
#endif
}

/**
 * xfce_file_chooser_get_select_multiple:
 * @chooser: An #XfceFileChooser.
 *
 * Gets whether multiple files can be selected in the file chooser.  See
 * xfce_file_chooser_set_select_multiple().
 *
 * Since 4.1
 *
 * Return value: %TRUE if multiple files can be selected, %FALSE otherwise.
 **/
gboolean
xfce_file_chooser_get_select_multiple(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_select_multiple(GTK_FILE_CHOOSER(chooser));
#else
	return gtk_file_selection_get_select_multiple(GTK_FILE_SELECTION(chooser));
#endif
}

/**
 * xfce_file_chooser_set_current_name:
 * @chooser: An #XfceFileChooser.
 * @name: The filename to use (UTF-8 only).
 *
 * Sets the name to be displayed in the file chooser's entry box, as if entered
 * by the user.  This function is meant to be used, e.g., to suggest a file
 * name to a user in a "Save as..." dialog.
 *
 * To preselect a particular existing filename, see
 * xfce_file_chooser_set_filename().
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_current_name(XfceFileChooser *chooser, const gchar *name)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(chooser), name);
#else
	gtk_entry_set_text(GTK_ENTRY(GTK_FILE_SELECTION(chooser)->selection_entry), name);
#endif
}

/**
 * xfce_file_chooser_get_filename:
 * @chooser: An #XfceFileChooser.
 *
 * Returns the currently selected file in the file chooser.  If multiple files
 * are selected, behavior is indeterminate.
 *
 * Since 4.1
 *
 * Return value: The currently selected filename, or %NULL if no file is
 *               selected.  This string is allocated by libxfcegui4 and should
 *               be released with g_free() when it is no longer needed.
 **/
gchar *
xfce_file_chooser_get_filename(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
#else
	return g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(chooser)));
#endif
}

/**
 * xfce_file_chooser_set_filename:
 * @chooser: An #XfceFileChooser.
 * @filename: A filename to set.
 *
 * Sets @filename as the currently selected file in the file chooser, changing
 * the file chooser's directory view if necessary.
 *
 * Note that the file must exist, or nothing will occur except for possibly a
 * directory change.  If you wish to suggest a file name to the user, use
 * xfce_file_chooser_set_current_name() instead.
 *
 * Since 4.1
 *
 * Return value: %TRUE if the file exists and was selected, %FALSE otherwise.
 **/
gboolean
xfce_file_chooser_set_filename(XfceFileChooser *chooser, const gchar *filename)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(chooser), filename);
#else
	const gchar *filename_check;

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(chooser), filename);
	filename_check = gtk_file_selection_get_filename(GTK_FILE_SELECTION(chooser));
	if(!filename_check || !g_file_test(filename_check, G_FILE_TEST_EXISTS))
		return FALSE;
	else
		return TRUE;
#endif
}

/**
 * xfce_file_chooser_get_filenames:
 * @chooser: An #XfceFileChooser.
 *
 * Gets a list of file names selected in the file chooser.  This is intended for
 * use when xfce_file_chooser_set_select_multiple() is used to allow the
 * selection of multiple files.
 *
 * Since 4.1
 *
 * Return value: A <ulink url="http://developer.gnome.org/doc/API/2.0/glib/glib-Singly-Linked-Lists.html">GSList</ulink>
 *               containing the filenames of all selected files in the file
 *               chooser.  Release the list with g_slist_free() and the
 *               filenames with g_free() when they are no longer needed.
 **/
GSList *
xfce_file_chooser_get_filenames(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(chooser));
#else
	GSList *l = NULL;
	gint i;
	gchar **files;

	files = gtk_file_selection_get_selections(GTK_FILE_SELECTION(chooser));
	for(i=0; files[i]; i++)
		l = g_slist_append(l, files[i]);
	
	/* free the array of pointers, but not the pointers themselves */
	g_free(files);
	
	return l;
#endif	
}

/**
 * xfce_file_chooser_set_current_folder:
 * @chooser: An #XfceFileChooser.
 * @filename: The full path of the new current folder.
 *
 * Sets the current folder of @chooser to @filename.
 *
 * Since 4.1
 *
 * Return value: %TRUE if the folder exists and could be changed, %FALSE
 *               otherwise.
 **/
gboolean
xfce_file_chooser_set_current_folder(XfceFileChooser *chooser,
		const gchar *filename)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(chooser), filename);
#else
	if(g_file_test(filename, G_FILE_TEST_IS_DIR)) {
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(chooser), filename);
		return TRUE;
	} else
		return FALSE;
#endif
}

/**
 * xfce_file_chooser_get_current_folder:
 * @chooser: An #XfceFileChooser.
 *
 * Gets the current folder visible in the file chooser.
 *
 * Since 4.1
 *
 * Return value: The currently-selected folder.  Release with g_free() when no
 *               longer needed.
 **/
gchar *
xfce_file_chooser_get_current_folder(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(chooser));
#else
	gchar *dir, *p;
	
	dir = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(chooser)));
	if(dir) {
		if((p=g_strrstr(dir, G_DIR_SEPARATOR_S)))
			*p = 0;
	}
	
	return dir;
#endif
}

/**
 * xfce_file_chooser_set_preview_widget:
 * @chooser: An #XfceFileChooser.
 * @preview_widget: A widget used to preview file data.
 *
 * Sets an app-supplied widget to be shown as a "preview widget" for the file
 * chooser.  To implement the preview, set a callback function using
 * xfce_file_chooser_set_preview_callback().  In this callback function, use
 * xfce_file_chooser_get_filename() to get the currently selected file.
 * Finally, use xfce_file_chooser_set_preview_widget_active() to tell the file
 * chooser whether or not a preview should be shown for that file.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_preview_widget(XfceFileChooser *chooser,
		GtkWidget *preview_widget)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_preview_widget(GTK_FILE_CHOOSER(chooser), preview_widget);
#else
	/* following code based loosely on Xfce's PreviewFileSelection widget */
	GtkWidget *hbox, *frame = NULL, *old_widget, *label;

	old_widget = g_object_get_data(G_OBJECT(chooser), "xfce-fc-pwidget");
	label = g_object_get_data(G_OBJECT(chooser), "xfce-fc-plabel");
	if(old_widget) {
		frame = g_object_get_data(G_OBJECT(chooser), "xfce-fc-pframe");
		gtk_widget_hide_all(old_widget);
		gtk_container_remove(GTK_CONTAINER(frame), old_widget);
		gtk_widget_destroy(old_widget);
		if(label) {
			gtk_container_remove(GTK_CONTAINER(frame), label);
			gtk_widget_destroy(label);
		}
	}
	
	if(!frame) {
		hbox = GTK_FILE_SELECTION(chooser)->file_list;
		while(hbox && !GTK_IS_HBOX(hbox)) {
			hbox = gtk_widget_get_parent(hbox);
		}
		
		if(!hbox) {
			g_warning("Can't find GtkFileSelection's HBox, unable to set preview widget");
			return;
		}
		
		frame = gtk_frame_new(NULL);
		gtk_frame_set_label_align(GTK_FRAME(frame), 0.5, 0.5);
		gtk_box_pack_end(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
		
		g_object_set_data(G_OBJECT(chooser), "xfce-fc-pframe", frame);
	}
	
	gtk_container_add(GTK_CONTAINER(frame), preview_widget);
	g_object_set_data(G_OBJECT(chooser), "xfce-fc-pwidget", preview_widget);
#endif
}

/**
 * xfce_file_chooser_get_preview_widget:
 * @chooser: An #XfceFileChooser.
 *
 * Retrieves the preview widget set by xfce_file_chooser_set_preview_widget().
 *
 * Since 4.1
 *
 * Return value: A <ulink url="http://developer.gnome.org/dock/API/2.0/gtk/GtkWidget.html">GtkWidget</ulink>,
 *               or %NULL if no preview widget was set.
 **/
GtkWidget *
xfce_file_chooser_get_preview_widget(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_preview_widget(GTK_FILE_CHOOSER(chooser));
#else
	return g_object_get_data(G_OBJECT(chooser), "xfce-fc-pwidget");
#endif
}

/**
 * xfce_file_chooser_set_preview_callback:
 * @chooser: An #XfceFileChooser.
 * @func: A callback function.
 * @user_data: Data to be passed to @func.
 *
 * Sets a callback function to be executed whenever the preview widget requires
 * updating.  Unfortunately, this method limits the application to a single
 * callback function, but this should be sufficient for most uses.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_preview_callback(XfceFileChooser *chooser,
		PreviewUpdateFunc func, gpointer user_data)
{	
	g_object_set_data(G_OBJECT(chooser), "xfce-fc-previewfunc", func);
	g_object_set_data(G_OBJECT(chooser), "xfce-fc-previewdata", user_data);
}

/**
 * xfce_file_chooser_set_preview_widget_active:
 * @chooser: An #XfceFileChooser.
 * @active: %TRUE or %FALSE depending on whether or not the preview should be
 *          shown.
 *
 * Sets whether or not the currently selected file supports some type of
 * preview, handled by the widget set with
 * xfce_file_chooser_set_preview_widget().  This should be used in the preview
 * callback every time the selected file is changed.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_preview_widget_active(XfceFileChooser *chooser,
		gboolean active)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_preview_widget_active(GTK_FILE_CHOOSER(chooser), active);
#else
	GtkWidget *pframe = GTK_WIDGET(g_object_get_data(G_OBJECT(chooser),
			"xfce-fc-pframe"));
	if(pframe) {
		if(active)
			gtk_widget_show_all(pframe);
		else
			gtk_widget_hide_all(pframe);
	}
#endif
}

/**
 * xfce_file_chooser_get_preview_widget_active:
 * @chooser: An #XfceFileChooser.
 *
 * Gets whether or not the preview widget is displayed for the current file.
 * See xfce_file_chooser_set_preview_widget_active().
 *
 * Since 4.1
 *
 * Return value: %TRUE if the preview widget is active, %FALSE otherwise.
 **/
gboolean
xfce_file_chooser_get_preview_widget_active(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_preview_widget_active(GTK_FILE_CHOOSER(chooser));
#else
	GtkWidget *pframe = GTK_WIDGET(g_object_get_data(G_OBJECT(chooser),
			"xfce-fc-pwidget"));
	if(pframe)
		return GTK_WIDGET_VISIBLE(pframe);
	else
		return FALSE;
#endif
}

/**
 * xfce_file_chooser_set_use_preview_label:
 * @chooser: An #XfceFileChooser.
 * @use_label: Whether to display a stock label with the name of the previewed
 *             file.
 *
 * Sets whether the file chooser should display a text label above the preview
 * widget containing the name of the selected file.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_use_preview_label(XfceFileChooser *chooser,
		gboolean use_label)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_use_preview_label(GTK_FILE_CHOOSER(chooser), use_label);
#else
	g_object_set_data(G_OBJECT(chooser), "xfce-fc-plabel",
			GINT_TO_POINTER(use_label ? SHOW_PREVIEW_LABEL : HIDE_PREVIEW_LABEL));
#endif
}

/**
 * xfce_file_chooser_set_extra_widget:
 * @chooser: An #XfceFileChooser.
 * @extra_widget: An extra widget to display.
 *
 * Sets a widget to display in the file chooser to provide extra options to the
 * user.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_extra_widget(XfceFileChooser *chooser,
		GtkWidget *extra_widget)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(chooser), extra_widget);
#else
	GtkWidget *old, *box;
	
	box = gtk_widget_get_parent(GTK_FILE_SELECTION(chooser)->selection_entry);
	old = g_object_get_data(G_OBJECT(chooser), "xfce-fc-ewidget");
	
	if(old == extra_widget)
		return;
	if(old) {
		if(GTK_IS_BOX(box))
			gtk_container_remove(GTK_CONTAINER(box), old);
		gtk_widget_destroy(old);
	}
	
	if(extra_widget && GTK_IS_BOX(box))
		gtk_box_pack_start(GTK_BOX(box), extra_widget, FALSE, FALSE, 0);
	
	g_object_set_data(G_OBJECT(chooser), "xfce-fc-ewidget", extra_widget);
#endif
}

/**
 * xfce_file_chooser_get_extra_widget:
 * @chooser: An #XfceFileChooser.
 *
 * Gets the widget set with xfce_file_chooser_set_extra_widget().
 *
 * Since 4.1
 *
 * Return value: The extra <ulink url="http://developer.gnome.org/doc/API/2.0/gtk/GtkWidget.html">GtkWidget</ulink>.
 **/
GtkWidget *
xfce_file_chooser_get_extra_widget(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_extra_widget(GTK_FILE_CHOOSER(chooser));
#else
	return g_object_get_data(G_OBJECT(chooser), "xfce-fc-ewidget");
#endif
}

/**
 * xfce_file_chooser_add_filter:
 * @chooser: An #XfceFileChooser.
 * @filter: An #XfceFileFilter created with xfce_file_filter_new().
 *
 * Adds a filter to the list of filters that the user can select to restrict
 * the view of the file chooser.
 *
 * Note: This currently does nothing when libxfcegui4 is compiled against GTK+
 * 2.2 or lower.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_add_filter(XfceFileChooser *chooser, XfceFileFilter *filter)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), GTK_FILE_FILTER(filter));
#else
	/* noop */
#endif
}

/**
 * xfce_file_chooser_remove_filter:
 * @chooser: An #XfceFileChooser.
 * @filter: An #XfceFileFilter.
 *
 * Removes @filter from the list of filters displayed by the file chooser.
 *
 * Note: This currently does nothing when libxfcegui4 is compiled against GTK+
 * 2.2 or lower.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_remove_filter(XfceFileChooser *chooser, XfceFileFilter *filter)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_remove_filter(GTK_FILE_CHOOSER(chooser), GTK_FILE_FILTER(filter));
#else
	/* noop */
#endif
}

/**
 * xfce_file_chooser_set_filter:
 * @chooser: An #XfceFileChooser.
 * @filter: An #XfceFileFilter.
 *
 * Sets the file chooser's currently selected file filter to @filter.  This
 * filter must have been added previously using xfce_file_chooser_add_filter().
 *
 * Note: This currently does nothing when libxfcegui4 is compiled against GTK+
 * 2.2 or lower.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_filter(XfceFileChooser *chooser, XfceFileFilter *filter)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(chooser), GTK_FILE_FILTER(filter));
#else
	/* noop */
#endif
}

/* noops for gtk 2.[02] */

/**
 * xfce_file_chooser_set_local_only:
 * @chooser: An #XfceFileChooser.
 * @local_only: Whether to only allow the display of local files.
 *
 * Sets whether or not the file chooser can display and/or allow the selection
 * of non-local files.
 *
 * Deprecated: This function isn't really useful in the context of Xfce.
 *
 * Since 4.1
 **/
void
xfce_file_chooser_set_local_only(XfceFileChooser *chooser, gboolean local_only)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(chooser), local_only);
#else
	/* noop */
	return;
#endif
}

/**
 * xfce_file_chooser_get_local_only:
 * @chooser: An #XfceFileChooser.
 *
 * Gets whether or not the file chooser can display and/or allow the selection
 * of non-local files.
 *
 * Deprecated: This function isn't really useful in the context of Xfce.
 *
 * Since 4.1
 *
 * Return value: %TRUE if non-local files are allowed, %FALSE otherwise.
 **/
gboolean
xfce_file_chooser_get_local_only(XfceFileChooser *chooser)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_get_local_only(GTK_FILE_CHOOSER(chooser));
#else
	/* noop */
	return TRUE;
#endif
}

/**
 * xfce_file_chooser_add_shortcut_folder:
 * @chooser: An #XfceFileChooser.
 * @folder: The full path to a folder name.
 * @error: Location to store any errors, or %NULL.
 *
 * Adds a folder to be displayed in the shortcut folders list in the file
 * chooser.  The folder name is validated, so only valid paths may be added.
 *
 * Note: This currently does nothing when libxfcegui4 is compiled against GTK+
 * 2.2 or lower.
 *
 * Since 4.1
 *
 * Return value: %TRUE if the folder given is valid, %FALSE otherwise.  If
 *               @error is non-%NULL, it will be set with any error that occurs.
 **/
gboolean
xfce_file_chooser_add_shortcut_folder(XfceFileChooser *chooser,
		const gchar *folder, GError **error)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_add_shortcut_folder(GTK_FILE_CHOOSER(chooser),
			folder, error);
#else
	/* noop */
	return TRUE;
#endif
}

/**
 * xfce_file_chooser_remove_shortcut_folder:
 * @chooser: An #XfceFileChooser.
 * @folder: The full path to an existing shortcut folder name.
 * @error: Location to store any errors, or %NULL.
 *
 * Removes the specified folder from the file chooser's shortcut folder list.
 *
 * Note: This currently does nothing when libxfcegui4 is compiled against GTK+
 * 2.2 or lower.
 *
 * Since 4.1
 *
 * Return value: %TRUE if the removal succeeded, %FALSE otherwise.  If @error is
 *               non-%NULL, it will be set with any error that occurs.
 **/ 
gboolean
xfce_file_chooser_remove_shortcut_folder(XfceFileChooser *chooser,
		const gchar *folder, GError **error)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gtk_file_chooser_remove_shortcut_folder(GTK_FILE_CHOOSER(chooser),
			folder, error);
#else
	/* noop*/
	return TRUE;
#endif
}

/* file filter routines */

/**
 * xfce_file_filter_new:
 *
 * Create a new empty file filter.
 *
 * Note: this will always return %NULL when libxfcegui4 is compiled against GTK+
 * 2.2 or below, as file filters are not supported.
 *
 * Since 4.1
 *
 * Return value: An #XfceFileFilter, or %NULL (GTK+ 2.2 and below).
 **/
XfceFileFilter *
xfce_file_filter_new()
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return GTK_OBJECT(gtk_file_filter_new());
#else
	return NULL;
#endif
}

/**
 * xfce_file_filter_set_name:
 * @filter: An #XfceFileFilter.
 * @name: The name of the filter.
 *
 * Sets a name to be displayed in in the option menu which describes the
 * filter's purpose.
 *
 * Note: this function has no effect when libxfcegui4 is compiled against GTK+
 * 2.2 or below, as file filters are not supported.
 *
 * Since 4.1
 **/
void
xfce_file_filter_set_name(XfceFileFilter *filter, const gchar *name)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_filter_set_name(GTK_FILE_FILTER(filter), name);
#endif
}

/**
 * xfce_file_filter_add_mime_type:
 * @filter: An #XfceFileFilter.
 * @mime_type: A registered MIME type.
 *
 * Adds a MIME type to the file filter.  All files matching this MIME type will
 * be displayed in the file chooser when the filter is selected.
 *
 * Note: this function has no effect when libxfcegui4 is compiled against GTK+
 * 2.2 or below, as file filters are not supported.
 *
 * Since 4.1
 **/
void
xfce_file_filter_add_mime_type(XfceFileFilter *filter, const gchar *mime_type)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_filter_add_mime_type(GTK_FILE_FILTER(filter), mime_type);
#endif
}

/**
 * xfce_file_filter_add_pattern:
 * @filter: An #XfceFileFilter.
 * @pattern: A file pattern.
 *
 * Adds a shell-glob-type file pattern to the file filter.  All files matching
 * the pattern will be displayed in the file chooser when the filter is
 * selected.
 *
 * Note: this function has no effect when libxfcegui4 is compiled against GTK+
 * 2.2 or below, as file filters are not supported.
 *
 * Since 4.1
 **/
void
xfce_file_filter_add_pattern(XfceFileFilter *filter, const gchar *pattern)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gtk_file_filter_add_pattern(GTK_FILE_FILTER(filter), pattern);
#endif
}
