/*
 * bitmap.c
 * 
 * Copyright (c) 2003 Alexandre Pigolkine
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, 
 * and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, 
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all copies or substantial 
 * portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT 
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Authors:
 *   Alexandre Pigolkine (pigolkine@gmx.de)
 *   Vladimir Vukicevic (vladimir@pobox.com)
 *   Jordi Mas (jordi@ximian.com)
 *   Jonathan Gilbert (logic@deltaq.org)
 */

#include <glib.h>
#include "gdip.h"
#include "gdipImage.h"
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


/*
	Those are the only pixel formats that we really support	
*/
BOOL
gdip_is_a_supported_pixelformat (PixelFormat fmt)
{
	switch (fmt) {
	
	case Format1bppIndexed:
	case Format4bppIndexed:
	case Format8bppIndexed:
 	case Format24bppRgb:
	case Format32bppArgb:
	case Format32bppPArgb:
	case Format32bppRgb:
		return TRUE;
	default:
		return FALSE;
	}
}

/*
	Returns TRUE if the Bitmap contains indexed (palettized) data.
*/
BOOL
gdip_is_an_indexed_pixelformat (PixelFormat fmt)
{
	return ((fmt & PixelFormatIndexed) != 0);
}

void 
gdip_bitmap_init (GpBitmap *bitmap)
{
	gdip_image_init (&bitmap->image);

	bitmap->image.type = imageBitmap;
	bitmap->data.Width = 0;
	bitmap->data.Height = 0;
	bitmap->data.Stride = 0;
	bitmap->data.PixelFormat = Format32bppArgb;
	bitmap->data.Scan0 = 0;
	bitmap->data.Reserved = 0;
	bitmap->data.ByteCount = 0;
	bitmap->data.Bytes = NULL;
	bitmap->data.Left = 0;
	bitmap->data.Top = 0;
	bitmap->cairo_format = CAIRO_FORMAT_ARGB32; 
		
	bitmap->hBitmapDC = 0;
	bitmap->hInitialBitmap = 0;
	bitmap->hBitmap = 0;
}



void
gdip_bitmap_clone (GpBitmap *bitmap, GpBitmap **clonedbitmap)
{
	GpBitmap *result = (GpBitmap *) GdipAlloc (sizeof (GpBitmap));	
	memcpy (result, bitmap, sizeof (GpBitmap));
	
	result->data.Scan0 = GdipAlloc (bitmap->data.Stride * bitmap->data.Height);
	memcpy (result->data.Scan0, bitmap->data.Scan0, bitmap->data.Stride * bitmap->data.Height);
	*clonedbitmap = result;
	result->data.Reserved =  GBD_OWN_SCAN0; /* Also overwrites a possible GDB_LOCKED */

	if (bitmap->data.ByteCount > 0 && bitmap->data.Bytes != NULL){
		result->data.Bytes = GdipAlloc (bitmap->data.ByteCount);
		if (result->data.Bytes == NULL)
			bitmap->data.ByteCount = 0;
		else
			memcpy (result->data.Bytes, bitmap->data.Bytes, bitmap->data.ByteCount);
	}
	
	
	/*TODO: We should also copy palette info when we support it*/
}


GpBitmap *
gdip_bitmap_new ()
{
	GpBitmap *result = (GpBitmap *) GdipAlloc (sizeof (GpBitmap));

	if (result)
		gdip_bitmap_init (result);

	return result;
}

/*
 * This should only be called from GdipDisposeImage, and it should *not* free
 * the structure, that one is freed by GdipDisposeImage
 */
void 
gdip_bitmap_dispose (GpBitmap *bitmap)
{
	if (((bitmap->data.Reserved & GBD_OWN_SCAN0) != 0) && bitmap->data.Scan0 != NULL) {
		GdipFree (bitmap->data.Scan0);
		bitmap->data.Scan0 = NULL;
	}

	if ((bitmap->data.ByteCount > 0) && (bitmap->data.Bytes != NULL)) {
		GdipFree (bitmap->data.Bytes);
		bitmap->data.Bytes = NULL;
	}
}

/* The default indexed palettes. This code was generated by a tiny C# program.
 */

static const unsigned int default_Format1bppIndexed_palette[2] =
  {
    0x000000, 0xFFFFFF
  };
static const unsigned int default_Format4bppIndexed_palette[16] =
  {
    0x000000, 0x800000, 0x008000, 0x808000, 0x000080, 0x800080, 0x008080, 0x808080, 0xC0C0C0, 0xFF0000, 0x00FF00, 0xFFFF00, 0x0000FF, 0xFF00FF, 0x00FFFF, 0xFFFFFF
  };
static const unsigned int default_Format8bppIndexed_palette[256] =
  {
    0x000000, 0x800000, 0x008000, 0x808000, 0x000080, 0x800080, 0x008080, 0x808080, 0xC0C0C0, 0xFF0000, 0x00FF00, 0xFFFF00, 0x0000FF, 0xFF00FF, 0x00FFFF, 0xFFFFFF,
    0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000,
    0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000000, 0x000033, 0x000066, 0x000099, 0x0000CC, 0x0000FF, 0x003300, 0x003333,
    0x003366, 0x003399, 0x0033CC, 0x0033FF, 0x006600, 0x006633, 0x006666, 0x006699, 0x0066CC, 0x0066FF, 0x009900, 0x009933, 0x009966, 0x009999, 0x0099CC, 0x0099FF,
    0x00CC00, 0x00CC33, 0x00CC66, 0x00CC99, 0x00CCCC, 0x00CCFF, 0x00FF00, 0x00FF33, 0x00FF66, 0x00FF99, 0x00FFCC, 0x00FFFF, 0x330000, 0x330033, 0x330066, 0x330099,
    0x3300CC, 0x3300FF, 0x333300, 0x333333, 0x333366, 0x333399, 0x3333CC, 0x3333FF, 0x336600, 0x336633, 0x336666, 0x336699, 0x3366CC, 0x3366FF, 0x339900, 0x339933,
    0x339966, 0x339999, 0x3399CC, 0x3399FF, 0x33CC00, 0x33CC33, 0x33CC66, 0x33CC99, 0x33CCCC, 0x33CCFF, 0x33FF00, 0x33FF33, 0x33FF66, 0x33FF99, 0x33FFCC, 0x33FFFF,
    0x660000, 0x660033, 0x660066, 0x660099, 0x6600CC, 0x6600FF, 0x663300, 0x663333, 0x663366, 0x663399, 0x6633CC, 0x6633FF, 0x666600, 0x666633, 0x666666, 0x666699,
    0x6666CC, 0x6666FF, 0x669900, 0x669933, 0x669966, 0x669999, 0x6699CC, 0x6699FF, 0x66CC00, 0x66CC33, 0x66CC66, 0x66CC99, 0x66CCCC, 0x66CCFF, 0x66FF00, 0x66FF33,
    0x66FF66, 0x66FF99, 0x66FFCC, 0x66FFFF, 0x990000, 0x990033, 0x990066, 0x990099, 0x9900CC, 0x9900FF, 0x993300, 0x993333, 0x993366, 0x993399, 0x9933CC, 0x9933FF,
    0x996600, 0x996633, 0x996666, 0x996699, 0x9966CC, 0x9966FF, 0x999900, 0x999933, 0x999966, 0x999999, 0x9999CC, 0x9999FF, 0x99CC00, 0x99CC33, 0x99CC66, 0x99CC99,
    0x99CCCC, 0x99CCFF, 0x99FF00, 0x99FF33, 0x99FF66, 0x99FF99, 0x99FFCC, 0x99FFFF, 0xCC0000, 0xCC0033, 0xCC0066, 0xCC0099, 0xCC00CC, 0xCC00FF, 0xCC3300, 0xCC3333,
    0xCC3366, 0xCC3399, 0xCC33CC, 0xCC33FF, 0xCC6600, 0xCC6633, 0xCC6666, 0xCC6699, 0xCC66CC, 0xCC66FF, 0xCC9900, 0xCC9933, 0xCC9966, 0xCC9999, 0xCC99CC, 0xCC99FF,
    0xCCCC00, 0xCCCC33, 0xCCCC66, 0xCCCC99, 0xCCCCCC, 0xCCCCFF, 0xCCFF00, 0xCCFF33, 0xCCFF66, 0xCCFF99, 0xCCFFCC, 0xCCFFFF, 0xFF0000, 0xFF0033, 0xFF0066, 0xFF0099,
    0xFF00CC, 0xFF00FF, 0xFF3300, 0xFF3333, 0xFF3366, 0xFF3399, 0xFF33CC, 0xFF33FF, 0xFF6600, 0xFF6633, 0xFF6666, 0xFF6699, 0xFF66CC, 0xFF66FF, 0xFF9900, 0xFF9933,
    0xFF9966, 0xFF9999, 0xFF99CC, 0xFF99FF, 0xFFCC00, 0xFFCC33, 0xFFCC66, 0xFFCC99, 0xFFCCCC, 0xFFCCFF, 0xFFFF00, 0xFFFF33, 0xFFFF66, 0xFFFF99, 0xFFFFCC, 0xFFFFFF
  };

GpStatus
GdipCreateBitmapFromFile (GDIPCONST WCHAR* filename, GpBitmap **bitmap)
{
	return GdipLoadImageFromFile (filename, (GpImage **) bitmap);
}

GpStatus 
GdipCreateBitmapFromFileICM (GDIPCONST WCHAR* filename, GpBitmap **bitmap)
{
	return GdipCreateBitmapFromFile (filename, bitmap);
}

GpStatus 
GdipCreateBitmapFromScan0 (int width, int height, int stride, int format, void *scan0, GpBitmap **bitmap)
{
	GpBitmap *result = 0;
        bool own_scan0 = FALSE;
	int cairo_format = 0;
	
	switch (format) {
	case Format24bppRgb:
		cairo_format = CAIRO_FORMAT_RGB24;  
		break;
	case Format32bppArgb:
	case Format32bppRgb:
	case Format32bppPArgb:
		cairo_format = CAIRO_FORMAT_ARGB32;
		break;
	case Format8bppIndexed:
	case Format4bppIndexed:
		cairo_format = CAIRO_FORMAT_A8;        
		break;
	case Format1bppIndexed:
		cairo_format = CAIRO_FORMAT_A1;
		break;
	default:
		*bitmap = NULL;
		return NotImplemented;
	}

	result = gdip_bitmap_new ();
	if (result == NULL)
		return OutOfMemory;
			
	result->data.Width = result->image.width = width;
	result->data.Height = result->image.height = height;
	result->data.PixelFormat = result->image.pixFormat = format;
	result->cairo_format = cairo_format;

	if (stride == 0) {
		if (gdip_is_an_indexed_pixelformat (format)) {
			int bpp = gdip_get_pixel_format_depth(format);
			int bits_per_row = bpp * width;
			stride = (bits_per_row + 7) / 8;
		}
		else {
			int bpp  = gdip_get_pixel_format_components (format);
			bpp = bpp * gdip_get_pixel_format_depth (format);
			stride = (bpp * width) / 8;
		}

		/* make sure the stride aligns the next row to a pixman_bits_t boundary */
		stride += (sizeof(pixman_bits_t)-1);
		stride &= ~(sizeof(pixman_bits_t)-1);
	}

	if (scan0 == NULL) {
		scan0 = GdipAlloc (stride * height);
		if (scan0 == NULL)
			return OutOfMemory;

		if ((gdip_get_pixel_format_bpp (format) < 16) || gdip_is_an_alpha_pixelformat (format)) {
			memset (scan0, 0, stride * height);
		}
		else {
			/* Since the pixel format is not an alpha pixel format (i.e., it is
			 * either Format24bppRgb or Format32bppRgb), the image should be
			 * initially black, not initially transparent. Thus, we need to set
			 * the alpha channel, which the user code doesn't think exists but
			 * Cairo is still paying attention to, to 0xFF.
			 */
			int x, y;
			ARGB solid_black;

			/* Make sure the alpha channel is at the right end of the word. */
			set_pixel_bgra (&solid_black, 0,
				0, 0, 0, 0xFF);

			for (y=0; y < height; y++) {
				ARGB *scan = (ARGB *)((char *)scan0 + y * stride);
				for (x=0; x < width; x++) {
					scan[x] = solid_black;
				}
			}
		}
		own_scan0 = TRUE;
	}
	
	
	result->data.Scan0 = scan0;
	result->data.Stride = stride;
	
	if (own_scan0)
		result->data.Reserved |= GBD_OWN_SCAN0;

	/* Initialize the image frames */
	if (result->image.frameDimensionCount == 0) {
		result->image.frameDimensionCount = 1;
			result->image.frameDimensionList = (FrameInfo *) GdipAlloc (sizeof (FrameInfo));
			result->image.frameDimensionList[0].count = 1; /*multiple frames are already taken care of in respectic codecs*/
			memcpy (&(result->image.frameDimensionList[0].frameDimension), &gdip_image_frameDimension_page_guid, sizeof (CLSID));
			result->image.frameDimensionList[0].frames = &(((GpBitmap *) result)->data);
		}

	/* Make sure indexed images have a palette */
	if (gdip_is_an_indexed_pixelformat (format)) {
		int palette_entries = 1 << gdip_get_pixel_format_depth(format);
		int header_size = sizeof(ColorPalette) - sizeof(ARGB);
		int bytes_needed = header_size + palette_entries * sizeof(ARGB);
		const unsigned int *default_palette;
		int i;

		result->image.palette = GdipAlloc (bytes_needed);

		if (result->image.palette == NULL)
			return OutOfMemory;

		result->image.palette->Flags = 0;
		result->image.palette->Count = palette_entries;

		

		switch (format)
		{
			case Format1bppIndexed: default_palette = default_Format1bppIndexed_palette; break;
			case Format4bppIndexed: default_palette = default_Format4bppIndexed_palette; break;
			case Format8bppIndexed: default_palette = default_Format8bppIndexed_palette; break;

			/* The compiler doesn't know that these are the only possible
			 * indexed pixel formats This suppresses a warning:
			 */
			default: default_palette = NULL;
		}

		for (i=0; i < palette_entries; i++) {
			set_pixel_bgra (result->image.palette->Entries, i * 4,
				0xFF & (default_palette[i] >> 16),
				0xFF & (default_palette[i] >>  8),
				0xFF &  default_palette[i]       ,
				0xFF /* alpha */);
		}
	}

	*bitmap = result;

	return Ok;
}

GpStatus
GdipCreateBitmapFromGraphics (int width, int height, GpGraphics *graphics, GpBitmap **bitmap)
{
	GpBitmap 		*result = 0;
	int 			bmpSize = 0;
	int 			stride = width;
	/*
	 * FIXME: should get the stride based on the format of the graphics object.
	 */
	stride *= 4;

	while (stride % sizeof(pixman_bits_t))
		stride++;
	
	bmpSize = stride * height;
	result = gdip_bitmap_new ();
	result->data.Width = result->image.width = width;
	result->data.Height = result->image.height = height;
	result->data.PixelFormat = result->image.pixFormat = Format32bppArgb;
	result->data.Stride = stride;

	result->data.Scan0 = GdipAlloc (bmpSize);
	if (result->data.Scan0 == NULL){
		GdipFree (result);
		return OutOfMemory;
	}

	result->data.Reserved |= GBD_OWN_SCAN0;
	*bitmap = result;
	return Ok;
}

GpStatus
GdipCreateBitmapFromHBITMAP(void *hbm, void *hpal, GpBitmap** bitmap)
{
	return(NotImplemented);
}

GpStatus
GdipCreateHBITMAPFromBitmap(GpBitmap* bitmap, void **hbmReturn, unsigned long background)
{
	return(NotImplemented);
}

GpStatus
GdipCreateBitmapFromHICON(void *hicon, GpBitmap** bitmap)
{
	return(NotImplemented);
}

GpStatus
GdipCreateHICONFromBitmap(GpBitmap* bitmap, void **hbmReturn)
{
	return(NotImplemented);
}

GpStatus
GdipCreateBitmapFromResource(void *hInstance, GDIPCONST WCHAR *lpBitmapName, GpBitmap** bitmap)
{
	return(NotImplemented);
}

GpStatus
GdipCloneBitmapAreaI (int x, int y, int width, int height, PixelFormat format,
					  GpBitmap *original, GpBitmap **bitmap)
{
	GpBitmap *result = 0;
	GdipBitmapData bd;
	Rect sr = { x, y, width, height };
	Rect dr = { 0, 0, width, height };
	GpStatus st;

	g_return_val_if_fail (original != NULL, InvalidParameter);
	g_return_val_if_fail (bitmap != NULL, InvalidParameter);

	g_return_val_if_fail (x + width <= original->data.Width, InvalidParameter);
	g_return_val_if_fail (y + height <= original->data.Height, InvalidParameter);

	bd.Scan0 = NULL;
	bd.PixelFormat = format;
	
	st = gdip_bitmap_clone_data_rect (&original->data, &sr, &bd, &dr);
	if (st != Ok)
		return st;

	result = gdip_bitmap_new ();
	if (result == NULL){
		GdipFree (bd.Scan0);
		return OutOfMemory;
	}
	
	result->cairo_format = original->cairo_format;
	memcpy (&result->data, &bd, sizeof (GdipBitmapData));
	result->image.pixFormat = format;
	result->image.format = original->image.format;
	result->image.height = result->data.Height;
	result->image.width = result->data.Width;
		
	*bitmap = result;
	return Ok;
}

GpStatus
GdipCloneBitmapArea (float x, float y, float w, float h, PixelFormat format,
					 GpBitmap *original, GpBitmap **bitmap)
{
	return GdipCloneBitmapAreaI ((int) x, (int) y, (int) w, (int) h, format,
								 original, bitmap);
}

static void
gdip_copy_strides (void *dst, int dstStride, void *src, int srcStride, int realBytes, int height)
{
	int i;
	for (i = 0; i < height; i++) {
		memcpy (dst, src, realBytes);
		dst += dstStride;
		src += srcStride;
	}
}

/*
 * Copy srcRect region in srcData to destRect region in destData.  No conversion is done.  Assumes
 * BitmapData is straight from a GpBitmap.  src and dest rects must be the same width/height and
 * bits must be of the same PixelFormat.
 */

GpStatus
gdip_bitmap_clone_data_rect (GdipBitmapData *srcData, Rect *srcRect, GdipBitmapData *destData, Rect *destRect)
{
	int dest_components, dest_deph; 
	
	g_return_val_if_fail (srcData != NULL, InvalidParameter);
	g_return_val_if_fail (srcRect != NULL, InvalidParameter);
	g_return_val_if_fail (destData != NULL, InvalidParameter);
	g_return_val_if_fail (destRect != NULL, InvalidParameter);

	g_return_val_if_fail (srcRect->Width == destRect->Width, InvalidParameter);
	g_return_val_if_fail (srcRect->Height == destRect->Height, InvalidParameter);
	
	if (!gdip_is_a_supported_pixelformat (srcData->PixelFormat) ||
		!gdip_is_a_supported_pixelformat (destData->PixelFormat))
			return NotImplemented;
	
	dest_components = gdip_get_pixel_format_components (destData->PixelFormat);	
	dest_deph = gdip_get_pixel_format_depth (destData->PixelFormat);

	
	if (destData->Scan0 == NULL) {
		byte *data;

		destData->Stride = (((( destRect->Width * dest_components * dest_deph) /8)  + (sizeof(pixman_bits_t)-1)) & ~(sizeof(pixman_bits_t)-1));

		data = GdipAlloc (destData->Stride * destRect->Height);
		if (data == NULL) {
			return OutOfMemory;
		}
		
		destData->Scan0 = data;
		destData->Width = destRect->Width;
		destData->Height = destRect->Height;
		destData->PixelFormat = srcData->PixelFormat;
		destData->Reserved = GBD_OWN_SCAN0;
	}

	if (!gdip_is_an_indexed_pixelformat (srcData->PixelFormat)) {
		gdip_copy_strides (destData->Scan0, destData->Stride,
			srcData->Scan0 + (srcData->Stride * srcRect->Y) + (gdip_get_pixel_format_components (srcData->PixelFormat) 
			* srcRect->X), srcData->Stride,   destRect->Width * dest_components,  destRect->Height);
	}
	else {
		int src_depth = gdip_get_pixel_format_depth (srcData->PixelFormat);

		/* first, check if the bits are aligned onto a byte boundary */
		int src_first_x_bit_index = srcRect->X * src_depth;
		int width_bits = destRect->Width * src_depth;

		int src_first_x_bit_offset_into_byte = src_first_x_bit_index & 7;

		if (src_first_x_bit_offset_into_byte == 0) {
			/* the fast path: no mid-byte bit mangling required :-)
			 * this will always be the case for 8-bit images.
			 * basically, the source bits are aligned to the destination
			 * bytes, and it doesn't matter if the width isn't a multiple
			 * of 8 bits, because the remainder is guaranteed to be
			 * allocated by the stride, and will be ignored because the
			 * width will indicate fewer pixels than actually end up being
			 * copied.
			 */
			gdip_copy_strides (
				destData->Scan0, destData->Stride,
				srcData->Scan0 + (src_first_x_bit_index / 8) + (srcData->Stride * srcRect->Y),
				srcData->Stride, width_bits / 8, destRect->Height);
		}
		else {
			/* the not-so-fast path: no bits are aligned, so the entire image requires bit juggling. */

			unsigned char *src_scan0 = srcData->Scan0;
			unsigned char *dest_scan0 = destData->Scan0;
			int left_shift = src_first_x_bit_offset_into_byte;
			int x, y;

			/* move the src_scan0 up to the first byte with pixel data involved in the copy */
			src_scan0 += srcRect->Y * srcData->Stride;
			src_scan0 += (src_first_x_bit_offset_into_byte / 8);

			for (y=0; y < destRect->Height; y++) {
				unsigned char *src_scan = src_scan0 + y * srcData->Stride;
				unsigned char *dest_scan = dest_scan0 + y * destData->Stride;

				unsigned short buffer;

				/* jump-start the packing function. it avoids double-sampling the source bits by
				 * using buffer as a shift register; bits 8-15 are the current packed dest pixel,
				 * and some of bits 0-7 are not used, to permit alignment of the pixel data.
				 */
				buffer = src_scan[0] << left_shift;

				for (x=1; x < destRect->Width; x++) {
					buffer <<= 8;
					buffer |= src_scan[x] << left_shift;
					dest_scan[0] = (buffer >> 8);
				}
			}
		}
	}

	return Ok;
}

/*
 * According to GDI+ testing,
 * RGB[A] PixelFormats can be converted to/from any other RGB[A] pixel format, with or without alpha.
 * We should support all of these:
 *
 * 32bpp argb - 16bpp argb 1555
 * 32bpp argb - 16bpp rgb 555
 * 32bpp argb - 16bpp rgb 565
 * 32bpp argb - 24bpp rgb 888
 * 32bpp argb - 32bpp Pargb
 * 32bpp argb - 32bpp rgb
 *
 * Others can only be converted to itself. (i.e. 16bppGrayScale, 1bppIndexed, 4bppIndexed, 8bppIndexed)
 *
 */

static int
gdip_is_pixel_format_conversion_valid (PixelFormat src, PixelFormat dest)
{
	if (src == dest)
		return 1;

	/* non-GDI supported formats can't be converted */
	if (!(src & PixelFormatGDI))
		return 0;

	if ((src & PixelFormatIndexed)      )/*     || (dest & PixelFormatIndexed)) */
		return 0;

	/* These are the RGB formats */
	if ((src & PixelFormatGDI) && !(src & PixelFormatExtended)) {
		/* all of these should be supported, but we only report the
		 * ones we really can do for now
		 */
		/* We can't handle converting to/from the 565/555/1555 ones */
		if ((src & 0xff00) == 16 || (dest & 0xff00) == 16)
			return 0;
		return 1;
	}

	return 0;
}

/* Format24bppRgb is internally stored by Cairo as a four bytes. Convert it to 3-byte (RGB) */	
int
gdip_from_ARGB_to_RGB (BYTE *src, int width, int height, int stride, BYTE **dest, int* dest_stride)
{
	int x, y;
	BYTE *result;
	BYTE *pos_src, *pos_dest;
	int src_components = 4; /* ARGB */
	int dest_components = 3; /* RGB */
	
	*dest_stride = dest_components * 8;
	*dest_stride = (*dest_stride * width) / 8;
	*dest_stride = (*dest_stride + (sizeof(pixman_bits_t)-1)) & ~(sizeof(pixman_bits_t)-1);		
	
	result = GdipAlloc (*dest_stride * height);
	if (result == NULL)
		return OutOfMemory;
	
	memset (result, 0, *dest_stride * height);
	
	for (y = 0, pos_src = src, pos_dest = result; y < height; y++, pos_src += stride, pos_dest += *dest_stride) {		
		for (x = 0; x < width; x++) {
			pos_dest[0 + (dest_components * x)] = pos_src[0 + (src_components * x)];
			pos_dest[1 + (dest_components * x)] = pos_src[1 + (src_components * x)];
			pos_dest[2 + (dest_components * x)] = pos_src[2 + (src_components * x)];
		}
	}
	
	*dest = result;
	return Ok;
}


/* Format24bppRgb is internally stored by Cairo as a three bytes. Convert it to 4-byte (ARGB) */	
int
gdip_from_RGB_to_ARGB (BYTE *src, int width, int height, int stride, BYTE **dest, int* dest_stride)
{
	int x, y;
	BYTE *result;
	BYTE *pos_src, *pos_dest;
	int src_components = 3; /* RGB */
	int dest_components = 4; /* ARGB */
	
	*dest_stride = dest_components * 8;
	*dest_stride = (*dest_stride * width) / 8;
	*dest_stride = (*dest_stride + (sizeof(pixman_bits_t)-1)) & ~(sizeof(pixman_bits_t)-1);		
	
	result = GdipAlloc (*dest_stride * height);
	if (result == NULL)
		return OutOfMemory;
	
	memset (result, 0, *dest_stride * height);
	
	for (y = 0, pos_src = src, pos_dest = result; y < height; y++, pos_src += stride, pos_dest += *dest_stride) {		
		for (x = 0; x < width; x++) {
			pos_dest[0 + (dest_components * x)] = pos_src[0 + (src_components * x)];
			pos_dest[1 + (dest_components * x)] = pos_src[1 + (src_components * x)];
			pos_dest[2 + (dest_components * x)] = pos_src[2 + (src_components * x)];
			pos_dest[3 + (dest_components * x)] = 0xff;
		}
	}
	
	*dest = result;
	return Ok;
}

GpStatus
gdip_init_pixel_stream (StreamingState *state, BitmapData *data, int x, int y, int w, int h)
{
	if ((state == NULL) || (data == NULL) || (data->Scan0 == NULL))
		return InvalidParameter;

	/* Ensure that the rectangle requested is sensible. */
	if ((x < 0) || (y < 0) || (x >= data->Width) || (y >= data->Height))
		return InvalidParameter;

	if ((x + w > data->Width) || (y + h > data->Height))
		return InvalidParameter;

	/* Initialize the StreamingState structure to point at the first pixel. */
	state->region.X = x;
	state->region.Y = y;
	state->region.Width = w;
	state->region.Height = h;

	state->x = x;
	state->y = y;

	state->p = -1; /* ensure that the buffer will be preloaded on the first call, for indexed formats */

	switch (data->PixelFormat)
	{
		case Format1bppIndexed: state->one_pixel_mask = 0x01; state->one_pixel_shift = 1; state->pixels_per_byte = 8; break;
		case Format4bppIndexed: state->one_pixel_mask = 0x0F; state->one_pixel_shift = 4; state->pixels_per_byte = 2; break;
		case Format8bppIndexed: state->one_pixel_mask = 0xFF; state->one_pixel_shift = 8; state->pixels_per_byte = 1; break;
		default:
			state->pixels_per_byte = 0; /* indicate full RGB processing */
			break;
	}

	state->data = data;

	/* The following computation will compute the byte pointer that _contains_ the first
	 * pixel; this doesn't necessarily mean that the pixel is aligned to the byte. This
	 * will be handled in gdip_pixel_stream_get_next () each time it starts a new row.
	 */
	state->scan = (unsigned char *)(data->Scan0)
	            + y * data->Stride
	            + x * gdip_get_pixel_format_bpp (data->PixelFormat) / 8;

	return Ok;
}

BOOL
gdip_pixel_stream_has_next (StreamingState *state)
{
	if (state != NULL)
		return (state->p >= 0)
		    || ((state->y < (state->region.Y + state->region.Height))
		     && (state->x < (state->region.X + state->region.Width)));
	else
		return FALSE;
}

unsigned int /* <-- can be an ARGB or a palette index */
gdip_pixel_stream_get_next (StreamingState *state)
{
	unsigned int ret;

	if (state == NULL) {
		int bright_pink;
		set_pixel_bgra(&bright_pink, 0,
			0xFF, 0x00, 0xFF, 0xFF); /* bright pink; hopefully this will get somebody's attention :-) */
		return bright_pink;
	}

	/* Note: This function does not check whether the end of the region has been hit. This function can
	 * potentially overrun memory buffers! gdip_pixel_stream_has_next () must be used in conjunction
	 * with this function.
	 */

	if (state->pixels_per_byte == 1) {
		/* A fast path for 8-bit indexed data: pixels are byte-aligned, so no special unpacking is required. */
		ret = *state->scan;

		state->scan++;
		state->x++;

		if (state->x >= (state->region.X + state->region.Width)) {
			state->x = state->region.X;
			state->y++;

			state->scan = (unsigned char *)(state->data->Scan0)
			            + state->y * state->data->Stride
			            + state->x;
		}
	}
	else if (state->pixels_per_byte > 0) {
		/* We have an indexed format (the RGB formats can't fit a whole pixel into a single byte). */
		if (state->p < 0) {
			state->buffer = *state->scan;
			state->scan++;
			state->p = 0;

			if (state->x == state->region.X) {
				/* First pixel of the row; check whether it is aligned to the byte or not. */
				int index_into_byte = state->x & (state->pixels_per_byte - 1);

				if (index_into_byte != 0) {
					/* Not aligned; need to advance the buffer to the
					 * first pixel in the stream region.
					 */
					state->buffer <<= (index_into_byte * state->one_pixel_shift);
					state->p = index_into_byte;
				}
			}
		}

		state->buffer <<= state->one_pixel_shift;

		ret = (state->buffer >> 8) & state->one_pixel_mask;

		state->x++;
		state->p++;

		/* Have we hit the end of the buffer? */
		if (state->p >= state->pixels_per_byte)	
			state->p = -1;

		if (state->x >= (state->region.X + state->region.Width)) {
			state->x = state->region.X;
			state->y++;

			state->scan = (unsigned char *)(state->data->Scan0)
			            + state->y * state->data->Stride
			            + state->x * gdip_get_pixel_format_bpp (state->data->PixelFormat) / 8;

			state->p = -1;
		}
	}
	else {
		/* We have an RGB format. In the current implementation, these are always stored as
		 * CAIRO_FORMAT_ARGB. This makes this section very easy to implement. If native
		 * support for 15- and 16-bit pixel formats needs to be added in the future, though,
		 * then this is where it needs to be done.
		 */
		ret = *(unsigned int *)state->scan;

		/* Special case: 24-bit data needs to have the cairo format alpha component forced
		 * to 0xFF, or many operations will do nothing (or do strange things if the alpha
		 * channel contains garbage).
		 */
		if (state->data->PixelFormat == Format24bppRgb) {
			int force_alpha;
			set_pixel_bgra(&force_alpha, 0,
				0, 0, 0, 0xFF);
			ret |= force_alpha;
		}

		state->scan += 4;
		state->x++;

		if (state->x >= (state->region.X + state->region.Width)) {
			state->x = state->region.X;
			state->y++;

			state->scan = (unsigned char *)(state->data->Scan0)
			            + state->y * state->data->Stride
			            + state->x * 4;
		}
	}

	return ret;
}

void
gdip_pixel_stream_set_next (StreamingState *state, unsigned int pixel_value)
{
	if (state == NULL)
		return;

	/* Note: This function does not check whether the end of the region has been hit. This function can
	 * potentially overrun memory buffers! gdip_pixel_stream_has_next () must be used in conjunction
	 * with this function.
	 */

	if (state->pixels_per_byte == 1) {
		/* A fast path for 8-bit indexed data: pixels are byte-aligned, so no special packing is required. */
		*state->scan = pixel_value & 0xFF;

		state->scan++;
		state->x++;

		if (state->x >= (state->region.X + state->region.Width)) {
			state->x = state->region.X;
			state->y++;

			state->scan = (unsigned char *)(state->data->Scan0)
			            + state->y * state->data->Stride
			            + state->x;
		}
	}
	else if (state->pixels_per_byte > 0) {
		/* We have an indexed format (the RGB formats can't fit a whole pixel into a single byte). */
		if (state->p < 0) {
			state->p = 0;

			if (state->x == state->region.X) {
				/* First pixel of the row; check whether it is aligned to the byte or not. */
				int index_into_byte = state->x & (state->pixels_per_byte - 1);

				if (index_into_byte == 0) {
					/* It is aligned; all we need to do is clear the buffer. */
					state->buffer = 0;
				}
				else {
					/* It is not aligned; the buffer needs to be pre-loaded with those
					 * pixels that are to the left of the first pixel to be set.
					 */
					state->buffer = (*state->scan << (index_into_byte * state->one_pixel_shift));
					state->p = index_into_byte;
				}
			}
		}

		state->buffer <<= state->one_pixel_shift;
		state->buffer |= ((pixel_value & state->one_pixel_mask) << 8);

		state->x++;
		state->p++;

		/* Have we hit the end of the buffer? */
		if (state->p >= state->pixels_per_byte)	{
			*state->scan = (state->buffer >> 8);
			state->scan++;
			state->p = -1;
		}

		if (state->x >= (state->region.X + state->region.Width)) {
			if (state->p >= 0) {
				int existing_mask = 0;

				while (state->p < state->pixels_per_byte) {
					existing_mask <<= state->one_pixel_shift;
					existing_mask |= state->one_pixel_mask;

					state->buffer <<= state->one_pixel_shift;
					state->p++;
				}

				*state->scan = (*state->scan & existing_mask) | (state->buffer >> 8);
			}

			state->x = state->region.X;
			state->y++;

			state->scan = (unsigned char *)(state->data->Scan0)
			            + state->y * state->data->Stride
			            + state->x * gdip_get_pixel_format_bpp (state->data->PixelFormat) / 8;

			state->p = -1;
		}
	}
	else {
		/* We have an RGB format. In the current implementation, these are always stored as
		 * CAIRO_FORMAT_ARGB. This makes this section very easy to implement. If native
		 * support for 15- and 16-bit pixel formats needs to be added in the future, though,
		 * then this is where it needs to be done.
		 */
		*(unsigned int *)state->scan = pixel_value;

		state->scan += 4;
		state->x++;

		if (state->x >= (state->region.X + state->region.Width)) {
			state->x = state->region.X;
			state->y++;

			state->scan = (unsigned char *)(state->data->Scan0)
			            + state->y * state->data->Stride
			            + state->x * 4;
		}
	}
}

/**
 * srcData - input data
 * srcRect - rectangle of input data to place in destData
 * destData - where to place output; only the PixelFormat field is needed,
 *            which specifies the output type.
 * destRect - destination rectangle in output.
 *
 * assumes that the pixel format conversion has already been validated.
 */

GpStatus
gdip_bitmap_change_rect_pixel_format (GdipBitmapData *srcData, Rect *srcRect, GdipBitmapData *destData, Rect *destRect)
{
	PixelFormat srcFormat, destFormat;
	StreamingState srcStream, destStream;
	Rect effectiveDestRect;

	GpStatus status;

	g_return_val_if_fail (srcData != NULL, InvalidParameter);
	g_return_val_if_fail (srcRect != NULL, InvalidParameter);
	g_return_val_if_fail (destData != NULL, InvalidParameter);
	g_return_val_if_fail (destRect != NULL, InvalidParameter);

	if ((srcRect->X < 0) || (srcRect->Y < 0) || (srcRect->X >= srcData->Width) || (srcRect->Y >= srcData->Height))
		return InvalidParameter;
	if ((srcRect->X + srcRect->Width > srcData->Width) || (srcRect->Y + srcRect->Height > srcData->Height))
		return InvalidParameter;

	if ((destRect->X < 0) || (destRect->Y < 0))
		return InvalidParameter;

	g_return_val_if_fail (srcRect->Width <= destRect->Width, InvalidParameter);
	g_return_val_if_fail (srcRect->Height <= destRect->Height, InvalidParameter);

	srcFormat = srcData->PixelFormat;
	destFormat = destData->PixelFormat;

	if (!gdip_is_pixel_format_conversion_valid (srcFormat, destFormat))
		return InvalidParameter;

	if (destData->Scan0 == NULL) {
		/* Allocate a buffer on behalf of the caller. */
		int scans = destRect->Y + destRect->Height;

		int row_bits = destRect->Width * gdip_get_pixel_format_bpp (destFormat);
		int row_bytes = (row_bits + 7) / 8;

		int stride = (row_bytes + sizeof(pixman_bits_t) - 1) & ~(sizeof(pixman_bits_t) - 1);

		void *dest_scan0 = GdipAlloc(stride * scans);

		if (dest_scan0 == NULL)
			return OutOfMemory;

		destData->Width = destRect->X + destRect->Width;
		destData->Height = destRect->Y + destRect->Height;
		destData->Stride = stride;
		destData->Scan0 = dest_scan0;
		destData->Reserved = GBD_OWN_SCAN0;
	}
	else {
		/* Check that the destRect lies fully within the destData buffer. */
		if ((destRect->X + destRect->Width > destData->Width) || (destRect->Y + destRect->Height > destData->Height))
			return InvalidParameter;
	}

	effectiveDestRect = *destRect;

	if (effectiveDestRect.Width > srcRect->Width)
		effectiveDestRect.Width = srcRect->Width;
	if (effectiveDestRect.Height > srcRect->Height)
		effectiveDestRect.Height = srcRect->Height;

	/* Fire up the pixel streams. */
	status = gdip_init_pixel_stream (&srcStream, srcData, srcRect->X, srcRect->Y, srcRect->Width, srcRect->Height);

	if (status != Ok)
		return status;

	status = gdip_init_pixel_stream (&destStream, destData, effectiveDestRect.X, effectiveDestRect.Y, effectiveDestRect.Width, effectiveDestRect.Height);

	if (status != Ok)
		return status;

	/* Move the data. */
	while (gdip_pixel_stream_has_next (&srcStream))
		gdip_pixel_stream_set_next (&destStream, gdip_pixel_stream_get_next (&srcStream));

	return Ok;
}


#define NEW_LOCKBITS_IMPL

#ifdef NEW_LOCKBITS_IMPL
BOOL
gdip_is_a_32bit_pixelformat (PixelFormat format)
{
	switch (format)
	{
		case Format32bppRgb:
		case Format32bppArgb:
		case Format32bppPArgb: /* all of these use CAIRO_FORMAT_ARGB, which is 4 bytes wide */
			return TRUE;
		default:
			return FALSE;
	}
}

BOOL
gdip_is_an_alpha_pixelformat (PixelFormat format)
{
	return ((format & PixelFormatAlpha) != 0);
}

BOOL
gdip_can_window_without_copy (BitmapData *data, Rect *rect, int format)
{
	int bpp = gdip_get_pixel_format_bpp (format);

	if (format != data->PixelFormat) {
		/* can't possibly reinterpret bits from one indexed pixel
		 * format as being of another indexed pixel format...
		 */
		if (gdip_is_an_indexed_pixelformat (format)
		 || gdip_is_an_indexed_pixelformat (data->PixelFormat))
			return FALSE;

		/* ...but we can probably handle 24-bit<->32-bit and
		 * 32-bit alpha<->32-bit opaque without copying data,
		 * since these are all stored as CAIRO_FORMAT_ARGB
		 * internally.
		 */
		if (!gdip_is_a_32bit_pixelformat (format)
		 || !gdip_is_a_32bit_pixelformat (data->PixelFormat))
			return FALSE;
	}

	/* okay, so the pixel formats are compatible. now, make sure
	 * the rectangle lies on byte boundaries; if it doesn't, then
	 * pixels will have to be shuffled. =/
	 */

	/* 8bpp and above are guaranteed to be byte-aligned */
	if (bpp >= 8)
		return TRUE;
	else {
		int left_bit_offset = rect->X * bpp;
		int width_bit_count = rect->Width * bpp;

		/* check whether the values are byte-aligned */
		return ((left_bit_offset & 7) | (width_bit_count & 7)) == 0;
	}
}

void
gdip_make_alpha_opaque (BitmapData *data)
{
	unsigned char *scan0 = (unsigned char *)data->Scan0;
	int y, x, o, f;

	/* sanity check; make sure we aren't mangling any image data */
	if ((data->PixelFormat != Format32bppArgb)
	 && (data->PixelFormat != Format32bppRgb))
		return;

	f = data->Stride - 4 * data->Width;
	for (y=0, o=0; y < data->Height; y++, o += f)
		for (x=0; x < data->Width; x++, o += 4)
			scan0[o + 3] = 0xff; /* set alpha to fully-opaque */
}

GpStatus
GdipBitmapLockBits (GpBitmap *bitmap, Rect *srcRect, int flags, int format, GdipBitmapData *locked_data)
{
	BitmapData *root_data = &bitmap->data;

	g_return_val_if_fail (bitmap != NULL, InvalidParameter);
	g_return_val_if_fail (srcRect != NULL, InvalidParameter);
	g_return_val_if_fail (flags != 0, InvalidParameter);
	g_return_val_if_fail (locked_data != NULL, InvalidParameter);

	/* There is no way to set ImageLockModeUserInputBuf with S.D, so we don't
	 * support it
	 */
	if (flags & ImageLockModeUserInputBuf)
		return NotImplemented;

	/* Is this bitmap already locked? */
	if (root_data->Reserved & GBD_LOCKED)
		return InvalidParameter;

	if (!bitmap || !srcRect || !locked_data)
		return InvalidParameter;

	/* Make sure the srcRect makes sense */
	if ((srcRect->X < 0) || (srcRect->Y < 0) || (srcRect->Width < 0) || (srcRect->Height < 0))
		return InvalidParameter;

	if ((srcRect->X + srcRect->Width > root_data->Width) || (srcRect->Y + srcRect->Height > root_data->Height))
		return InvalidParameter;

	if (gdip_is_a_supported_pixelformat (format) == FALSE)
		return NotImplemented;

	/* Common stuff */
	if (flags == ImageLockModeRead)
		locked_data->Reserved |= GBD_READ_ONLY;
	else
		locked_data->Reserved &= ~GBD_READ_ONLY;	

	locked_data->Reserved |= GBD_LOCKED;
	locked_data->Reserved |= GBD_OWN_SCAN0;
	root_data->Reserved |= GBD_LOCKED;

	if (gdip_can_window_without_copy (root_data, srcRect, format)) {
		/* Compute the start of the window; it's guaranteed to be on
		 * a byte boundary since the preceding function call
		 * returned true
		 */
		locked_data->Scan0 = root_data->Scan0
		                   + srcRect->Y * root_data->Stride
		                   + srcRect->X * gdip_get_pixel_format_bpp(root_data->PixelFormat) / 8;

		/* Set up the window's dimensions */
		locked_data->Width = srcRect->Width;
		locked_data->Height = srcRect->Height;
		locked_data->Stride = root_data->Stride; /* the stride hasn't changed */
		locked_data->PixelFormat = root_data->PixelFormat;
		
		/* Make sure the bits don't get deallocated, since Scan0 doesn't
		 * necessarily point at the start of a memory block, and
		 * the bits are owned by the root bitmap anyway
		 */
		locked_data->Reserved &= ~GBD_OWN_SCAN0;

		/* If the source pixel format doesn't have alpha and the dest pixel format
		 * wants it, then overwrite the alpha byte in the original bitmap. it is
		 * unused anyway :-)
		 */
		if (!gdip_is_an_alpha_pixelformat (root_data->PixelFormat)
                 && gdip_is_an_alpha_pixelformat (locked_data->PixelFormat)) {
			gdip_make_alpha_opaque (locked_data);
		}

		return Ok;
	}
	else {
		/* If we get here, then something more drastic is needed. either the user's rectangle
		 * doesn't line up on byte boundaries, or the pixel format needs to be changed.
		 */
		int dest_pixel_format_bpp = gdip_get_pixel_format_bpp (format);
		int dest_stride = (srcRect->Width * dest_pixel_format_bpp + 7) / 8;
		int dest_size = srcRect->Height * dest_stride;

		unsigned char *dest_scan0 = GdipAlloc(dest_size);

		Rect destRect = { 0, 0, srcRect->Width, srcRect->Height };

		GpStatus status = Ok;

		if (dest_scan0 == NULL)
			return OutOfMemory;

		locked_data->Scan0 = dest_scan0;
		locked_data->Width = srcRect->Width;
		locked_data->Height = srcRect->Height;
		locked_data->Stride = dest_stride;
		locked_data->PixelFormat = format;

		/* If the user wants the original data to be readable, then convert the bits. */
		if ((flags & ImageLockModeRead) != 0) {
			status = gdip_bitmap_change_rect_pixel_format (root_data, srcRect, locked_data, &destRect);
			if (status != Ok)
				GdipFree(dest_scan0);
		}

		return status;
	}
}

GpStatus 
GdipBitmapUnlockBits (GpBitmap *bitmap, GdipBitmapData *locked_data)
{
	BitmapData *root_data = &bitmap->data;

	g_return_val_if_fail (bitmap != NULL, InvalidParameter);
	g_return_val_if_fail (locked_data != NULL, InvalidParameter);

	/* Make sure the bitmap is locked when the unlock happens */
	if (!(bitmap->data.Reserved & GBD_LOCKED))
		return InvalidParameter;

	/* It is not safe to assume that the correct BitmapData has been passed in.
	 * Sanity check: Make sure the locked data is in fact locked.
	 */
	if ((locked_data->Reserved & GBD_LOCKED) == 0)
		return InvalidParameter;

	/* Sanity check: Make sure the locked data's size is consistent with having
	 * been returned from LockBits ().
	 */
	if ((locked_data->Width > root_data->Width) || (locked_data->Height > root_data->Height))
		return InvalidParameter;

	/* Check whether the locked data window was created without
	 * duplicating the bits to be locked.
	 */
	if ((int)(((char *)locked_data->Scan0) - ((char *)root_data->Scan0)) < root_data->Height * root_data->Stride) {
		/* Since the locked data's Scan0 is inside the root data's Scan0, it is
		 * likely that the data was locked with the fast path of LockBits. However,
		 * a couple more sanity checks are required: The locked data's stride must
		 * be the same as the root data's stride, and the end of the locked data's
		 * bits must also lie within the root data's bits.
		 */
		char *byte_after_last_byte = (char *)locked_data->Scan0
		                           + (locked_data->Height - 1) * locked_data->Stride
		                           + (locked_data->Width * gdip_get_pixel_format_bpp (locked_data->PixelFormat) + 7) / 8;

		if ((locked_data->Stride != root_data->Stride)
		 || ((int)(byte_after_last_byte - (char *)root_data->Scan0) > root_data->Height * root_data->Stride))
			return InvalidParameter;

		/* We can be reasonably sure by this point that the locked data is authentic,
		 * so now the only thing that needs to be done is alpha fixup.
		 */
		if ((locked_data->Reserved & GBD_READ_ONLY) != 0) {
			if (!gdip_is_an_alpha_pixelformat (locked_data->PixelFormat)
			 && gdip_is_an_alpha_pixelformat (root_data->PixelFormat)) {
				gdip_make_alpha_opaque (locked_data);
			}
		}
	}
	else {
		/* We need to copy the locked data back to the root data's Scan0
		 * (but of course only if the ImageLockMode specified writability)
		 */
		if ((locked_data->Reserved & GBD_READ_ONLY) != 0) {
			/* FIXME: destRect isn't necessarily equal to srcRect. this is a bug (the old implementation had it too). */
			Rect srcRect = { 0, 0, locked_data->Width, locked_data->Height };
			Rect destRect = srcRect;

			GpStatus status = gdip_bitmap_change_rect_pixel_format (root_data, &srcRect, locked_data, &destRect);
			if (status != Ok)
				return status;
		}
	}

	if ((locked_data->Reserved & GBD_OWN_SCAN0) != 0) {
		GdipFree(locked_data->Scan0);
		locked_data->Scan0 = NULL;
		locked_data->Reserved &= ~GBD_OWN_SCAN0;
	}

	locked_data->Reserved &= ~GBD_LOCKED;
	root_data->Reserved &= ~GBD_LOCKED;

	return Ok;
}

#else

/* Microsoft GDI+ returns BitmapLock buffers in BGR (not RGB) */	
GpStatus 
GdipBitmapLockBits (GpBitmap *bitmap, Rect *srcRect, int flags, int format, GdipBitmapData *result)
{
	int src_stride;
	BYTE *src_scan0, *pos;
	BOOL allocated = FALSE;

	g_return_val_if_fail (bitmap != NULL, InvalidParameter);
	g_return_val_if_fail (srcRect != NULL, InvalidParameter);
	g_return_val_if_fail (flags != 0, InvalidParameter);
	g_return_val_if_fail (result != NULL, InvalidParameter);

	/* There is no way to set ImageLockModeUserInputBuf with S.D, so we don't
	 * support it
	 */
	if (flags & ImageLockModeUserInputBuf)
		return NotImplemented;

	/* Is this bitmap already locked? */
	if (bitmap->data.Reserved & GBD_LOCKED)
		return InvalidParameter;
		
	if (!bitmap || !srcRect || !result)
		return InvalidParameter;
		
	if (gdip_is_a_supported_pixelformat (format) == FALSE)
		return NotImplemented;	
	
	if (format == Format24bppRgb) {
		if (gdip_from_ARGB_to_RGB (bitmap->data.Scan0, bitmap->data.Width, bitmap->data.Height, 
					   bitmap->data.Stride, &src_scan0, &src_stride) != Ok)
			return OutOfMemory;
			
		allocated = TRUE;		
	}
	else {
		src_stride = bitmap->data.Stride;
		src_scan0 =  bitmap->data.Scan0;		
	}
		
	
	/* Special case -- the entire image is requested, with
	 * no format translation.
	 */
	if (srcRect->X == 0 && srcRect->Width == bitmap->data.Width &&
	    srcRect->Y == 0 && srcRect->Height == bitmap->data.Height &&
	    format == bitmap->data.PixelFormat)  {   

	    	result->Width = bitmap->data.Width;
		result->Height = bitmap->data.Height;
		result->Stride = src_stride;
		result->PixelFormat = format;
		result->Scan0 = GdipAlloc (src_stride * bitmap->data.Height);
		if (result->Scan0 == NULL)
			return OutOfMemory;

		if (allocated)
			GdipFree (src_scan0);
		
		memcpy (result->Scan0, src_scan0, src_stride * bitmap->data.Height);
		
	} else {
		/* the user wants a subrect and/or wants a pixel format conversion */
		GdipBitmapData convert;
		Rect destRect = {0, 0, srcRect->Width, srcRect->Height};
		GdipBitmapData src;

		convert.PixelFormat = format;
		convert.Scan0 = NULL;
		
		memcpy (&src, &bitmap->data, sizeof (GdipBitmapData));
		src.Scan0 = src_scan0;
		src.Stride = src_stride;
		
		if (gdip_bitmap_change_rect_pixel_format (&src, srcRect, &convert, &destRect) != Ok) 
			return InvalidParameter;
		
		result->Width = convert.Width;
		result->Height = convert.Height;
		result->Stride = convert.Stride;
		result->PixelFormat = convert.PixelFormat;
		result->Reserved = convert.Reserved;
		
		if (flags == ImageLockModeRead)
			result->Reserved &= GBD_READ_ONLY;
			
		result->Scan0 = convert.Scan0;	
	}

	if (flags == ImageLockModeRead)
		result->Reserved |= GBD_READ_ONLY;
	else
		result->Reserved &= ~GBD_READ_ONLY;	

		
	result->Reserved |= GBD_LOCKED;
	result->Reserved |= GBD_OWN_SCAN0;
	bitmap->data.Reserved |= GBD_LOCKED;
	
	if (allocated)
		GdipFree (src_scan0);
	
	return Ok;
}

GpStatus 
GdipBitmapUnlockBits (GpBitmap *bitmap, GdipBitmapData *bitmap_data)
{
	int src_stride;
	BYTE *src_scan0, *pos;
	BOOL allocated = FALSE;
	
	g_return_val_if_fail (bitmap != NULL, InvalidParameter);
	g_return_val_if_fail (bitmap_data != NULL, InvalidParameter);

	/* Make sure the bitmap is locked when the unlock happens */
	if (!(bitmap->data.Reserved & GBD_LOCKED))
		return InvalidParameter;
		
	if (bitmap_data->PixelFormat == Format24bppRgb) {
		if (gdip_from_RGB_to_ARGB (bitmap_data->Scan0, bitmap_data->Width, bitmap_data->Height, 
					   bitmap_data->Stride, &src_scan0, &src_stride) != Ok)
			return OutOfMemory;
	
		allocated = TRUE;		
	}
	else {
		src_stride = bitmap_data->Stride;
		src_scan0 =  bitmap_data->Scan0;	
	}
	
	/* Special case -- the entire image was requested, with
	 * no format translation, do direct copy
	 */
	if (bitmap->data.Width == bitmap_data->Width &&
	    bitmap->data.Height == bitmap_data->Height &&
	    bitmap->data.PixelFormat == bitmap_data->PixelFormat)  {   
	
		memcpy (bitmap->data.Scan0, src_scan0, bitmap->data.Stride * bitmap->data.Height);
		
	} else {
		/* the user requested a subrect and/or wants a pixel format conversion */
		GdipBitmapData convert_src;
		Rect rect = {0, 0, bitmap_data->Width, bitmap_data->Height};
		
		memcpy (&convert_src, bitmap_data, sizeof (GdipBitmapData));
		convert_src.Scan0 = src_scan0;
		convert_src.Stride = src_stride;
	
		if (gdip_bitmap_change_rect_pixel_format (&convert_src, &rect, &bitmap->data, &rect) != Ok) 
			return InvalidParameter;
			
	}

	if (bitmap_data->Reserved & GBD_OWN_SCAN0)
		GdipFree (bitmap_data->Scan0);

	/* unlock the bitmap */
	bitmap->data.Reserved &= ~GBD_LOCKED;
		
	if (allocated)
		GdipFree (src_scan0);
	
	return Ok;
}
#endif /* NEW_LOCKBITS_IMPL */

GpStatus
GdipBitmapSetPixel (GpBitmap *bitmap, int x, int y, ARGB color)
{
	GdipBitmapData *data = &bitmap->data;
	unsigned char *v;
	
	if (bitmap == 0) 
		return InvalidParameter;
	
	if (x < 0 || x > data->Width)
		return InvalidParameter;

	if (y < 0 || y > data->Height)
		return InvalidParameter;

	if (gdip_is_an_indexed_pixelformat (data->PixelFormat))
		return InvalidParameter;

	/* BMP Locked */
	if (bitmap->data.Reserved & GBD_LOCKED)
		return InvalidParameter;

	v = (unsigned char *)(data->Scan0) + y * data->Stride;
	switch (data->PixelFormat) {
		case Format24bppRgb:
		case Format32bppRgb:
			color |= 0xFF000000; /* force the alpha for Cairo */
			/* fall through */
		case Format32bppArgb:
		case Format32bppPArgb:
		{
			ARGB *scan = (ARGB *)v;

			scan[x] = color;
			break;
		}
		default:
			return NotImplemented;
	} 

	return Ok;		
}

GpStatus
GdipBitmapGetPixel (GpBitmap *bitmap, int x, int y, ARGB *color)
{
	GdipBitmapData *data = &bitmap->data;
	
	if (bitmap == NULL || color == NULL) 
		return InvalidParameter;

	if (x < 0 || x >= data->Width)
		return InvalidParameter;

	if (y < 0 || y >= data->Height)
		return InvalidParameter;

	/* BMP Locked */
	if (bitmap->data.Reserved & GBD_LOCKED)
		return InvalidParameter;
		
	if (gdip_is_an_indexed_pixelformat (data->PixelFormat)) {
		StreamingState pixel_stream;
		GpStatus status;
		unsigned int palette_index;

		if (bitmap->image.palette == NULL)
			return InvalidParameter;

		status = gdip_init_pixel_stream (&pixel_stream, data, x, y, 1, 1);

		if (status != Ok)
			return status;

		palette_index = gdip_pixel_stream_get_next (&pixel_stream);

		if (palette_index >= bitmap->image.palette->Count)
			return InvalidParameter;

		*color = bitmap->image.palette->Entries[palette_index];
	}
	else {
		unsigned char *v;

		v = ((unsigned char *)data->Scan0) + y * data->Stride;
		switch (data->PixelFormat)
		{
			case Format24bppRgb:
			case Format32bppArgb:
			case Format32bppPArgb:
			case Format32bppRgb:
			{
				ARGB *scan = (ARGB *)v;

				*color = scan[x];
				break;
			}
			default:
				return NotImplemented;
		}
	} 
	
	return Ok;
}

GpStatus
GdipBitmapSetResolution (GpBitmap *bitmap, float xdpi, float ydpi)
{
	g_return_val_if_fail (bitmap != NULL, InvalidParameter);

	bitmap->image.horizontalResolution = xdpi;
	bitmap->image.verticalResolution = ydpi;

	bitmap->image.imageFlags |= ImageFlagsHasRealDPI;

	return Ok;
}

cairo_surface_t *
gdip_bitmap_ensure_surface (GpBitmap *bitmap)
{
	if (bitmap->image.surface == NULL &&
		bitmap->data.Scan0 != NULL)	{		
		
		switch (bitmap->data.PixelFormat) {
		case Format24bppRgb:
			bitmap->image.surface = cairo_image_surface_create_for_data
				((unsigned char *)bitmap->data.Scan0,
				 CAIRO_FORMAT_RGB24,
				 bitmap->data.Width,
				 bitmap->data.Height,
				 bitmap->data.Stride);
			break;
		case Format32bppArgb:
		case Format32bppRgb:
		case Format32bppPArgb:
			bitmap->image.surface = cairo_image_surface_create_for_data
				((unsigned char *)bitmap->data.Scan0,
				 CAIRO_FORMAT_ARGB32,
				 bitmap->data.Width,
				 bitmap->data.Height,
				 bitmap->data.Stride);
			break;
		default:
			g_warning ("gdip_bitmap_ensure_surface: Unable to create a surface for raw bitmap data of format 0x%08x", bitmap->data.PixelFormat);
			break;
		}
	}

	return bitmap->image.surface;
}

GpBitmap *
gdip_convert_indexed_to_rgb (GpBitmap *indexed_bmp)
{
	BitmapData *data = &indexed_bmp->data;
	ColorPalette *palette = indexed_bmp->image.palette;

	int rgb_stride, rgb_bytes, force_alpha;
	int one_pixel_mask, one_pixel_shift, pixels_per_byte;
	int *rgb_scan0;
	int p, x, y;

	GpBitmap *ret;
	GpStatus status;

	if (!gdip_is_an_indexed_pixelformat (data->PixelFormat))
		return NULL;

	if (palette == NULL)
		return NULL;

	switch (data->PixelFormat)
	{
		case Format1bppIndexed: one_pixel_mask = 0x01; one_pixel_shift = 1; pixels_per_byte = 8; break;
		case Format4bppIndexed: one_pixel_mask = 0x0F; one_pixel_shift = 4; pixels_per_byte = 2; break;
		case Format8bppIndexed: one_pixel_mask = 0xFF; one_pixel_shift = 8; pixels_per_byte = 1; break;
		default: /* something is wrong!! */
			return NULL;
	}

	if ((palette->Flags & PaletteFlagsHasAlpha) == 0)
		set_pixel_bgra (&force_alpha, 0,
			0, 0, 0, 0xFF); /* full alpha bits set */
	else
		force_alpha = 0;

	rgb_stride = data->Width * 4;

	/* ensure pixman_bits_t alignment */
	rgb_stride += (sizeof(pixman_bits_t)-1);
	rgb_stride &= ~(sizeof(pixman_bits_t)-1);

	rgb_bytes = data->Height * rgb_stride;

	/* allocate the RGB frame */
	rgb_scan0 = GdipAlloc (rgb_bytes);

	if (rgb_scan0 == NULL) /* out of memory?? */
		return NULL;

	/* convert the indexed pixels into RGB values and store them into the RGB frame */
	for (y=0; y < data->Height; y++)
	{
		unsigned char *indexed_scan = (unsigned char *)(data->Scan0) + y * data->Stride;
		int *rgb_scan = rgb_scan0 + (y * rgb_stride) / sizeof(int);

		for (x=0; x < data->Width; x += pixels_per_byte)
		{
			int pixels_this_byte = pixels_per_byte;
			unsigned short sample = *indexed_scan;

			indexed_scan++;

			if (x + pixels_this_byte >= data->Width)
				pixels_this_byte = data->Width - x;

			for (p=0; p < pixels_this_byte; p++)
			{
				int index;

				sample <<= one_pixel_shift;
				index = (sample >> 8) & one_pixel_mask;

				rgb_scan[x + p] = palette->Entries[index] | force_alpha;
			}
		}
	}

	/* try to get a GpBitmap out of it :-) */
	status = GdipCreateBitmapFromScan0 (data->Width, data->Height, rgb_stride, Format32bppRgb, rgb_scan0, &ret);

	if ((status != Ok) || (ret == NULL)) {
		GdipFree (ret);
		GdipFree (rgb_scan0);
		return NULL;
	}
	else {
		ret->data.Reserved |= GBD_OWN_SCAN0;
		return ret;
	}
}

