// pkg_tree.cc
//
//  Copyright 1999,2000,2001 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  A package-tree displayer.

#include "aptitude.h"

#include "pkg_tree.h"

#include "download.h"
#include "load_grouppolicy.h"
#include "load_sortpolicy.h"
#include "pkg_columnizer.h"
#include "pkg_grouppolicy.h"
#include "pkg_item.h"
#include "pkg_node.h"
#include "pkg_sortpolicy.h"
#include "pkg_subtree.h"
#include "pkg_ver_item.h"
#include "ui.h"
#include "view_changelog.h"
#include "vs_progress.h"

#include <vscreen/columnify.h>
#include <vscreen/vs_treeitem.h>

#include <generic/apt.h>
#include <generic/apt_undo_group.h>
#include <generic/config_signal.h>
#include <generic/undo.h>

#include <apt-pkg/progress.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/algorithms.h>

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>
#include <sigc++/retype_return.h>

keybindings *pkg_tree::bindings=NULL;

vs_editline::history_list pkg_tree::limit_history, pkg_tree::grouping_history,
  pkg_tree::sorting_history, pkg_menu_tree::search_history;

class pkg_matcher_search:public vs_tree_search_func
{
  pkg_matcher *matcher;
public:
  pkg_matcher_search(pkg_matcher *_matcher):matcher(_matcher) {}

  bool operator()(const vs_treeitem &item)
  {
    // EWW
    const pkg_item *pitem=dynamic_cast<const pkg_item *>(&item);
    if(pitem)
      return matcher->matches(pitem->get_package(), pitem->visible_version());
    else {
      const pkg_ver_item *pvitem=dynamic_cast<const pkg_ver_item *>(&item);

      if(pvitem)
	return matcher->matches(pvitem->get_package(),
				pvitem->get_version());
      else
	return false;
    }
  }
};

pkg_menu_tree::pkg_menu_tree()
  :last_search_matcher(NULL), doing_incsearch(false),
   pre_incsearch_selected(get_end())
{
  aptcfg->connect(PACKAGE "::UI::Incremental-Search",
		  SigC::slot(*this, &pkg_tree::do_cancel_incsearch));
}

pkg_menu_tree::~pkg_menu_tree()
{
  delete last_search_matcher;
}

pkg_tree_node *pkg_menu_tree::selection()
{
  vs_treeiterator i=get_selected();

  if(i==get_end())
    return NULL;
  else
    return dynamic_cast<pkg_tree_node *>(&*i);
}

bool pkg_menu_tree::pkg_or_ver_selected()
{
  pkg_tree_node *curr=selection();

  if(curr && (dynamic_cast<pkg_item*>(curr) ||
	      dynamic_cast<pkg_ver_item*>(curr)))
    return true;
  else
    return false;
}

bool pkg_menu_tree::package_enabled()
{
  return get_visible() && selection()!=NULL;
}

bool pkg_menu_tree::package_action(void (pkg_tree_node::* action)(undo_group *))
{
  if(!get_visible())
    return false;

  pkg_tree_node *curr=selection();

  if(curr)
    {
      undo_group *grp=new apt_undo_group;

      (curr->*action)(grp);

      if(!grp->empty())
	apt_undos->add_item(grp);
      else
	delete grp;

      if(aptcfg->FindB(PACKAGE "::UI::Advance-On-Action", false))
	level_line_down();

      package_states_changed();

      return true;
    }
  else
    return false;
}

bool pkg_menu_tree::package_install()
{
  return package_action(&pkg_tree_node::select);
}

bool pkg_menu_tree::package_remove()
{
  return package_action(&pkg_tree_node::remove);
}

bool pkg_menu_tree::package_purge()
{
  return package_action(&pkg_tree_node::purge);
}

bool pkg_menu_tree::package_keep()
{
  return package_action(&pkg_tree_node::keep);
}

bool pkg_menu_tree::package_hold()
{
  return package_action(&pkg_tree_node::hold);
}

bool pkg_menu_tree::package_mark_auto()
{
  return package_action(&pkg_tree_node::mark_auto);
}

bool pkg_menu_tree::package_unmark_auto()
{
  return package_action(&pkg_tree_node::unmark_auto);
}

bool pkg_menu_tree::package_forbid_enabled()
{
  return get_visible() && pkg_or_ver_selected();
}

bool pkg_menu_tree::package_forbid()
{
  if(!get_visible())
    return false;

  pkg_tree_node *curr=selection();

  if(!curr)
    return false;

  pkg_item *pitem=dynamic_cast<pkg_item*>(curr);
  if(pitem)
    {
      undo_group *grp=new apt_undo_group;

      pitem->forbid_upgrade(grp);

      if(!grp->empty())
	apt_undos->add_item(grp);
      else
	delete grp;

      if(aptcfg->FindB(PACKAGE "::UI::Advance-On-Action", false))
	level_line_down();

      package_states_changed();

      return true;
    }
  else
    {
      pkg_ver_item *pvitem=dynamic_cast<pkg_ver_item*>(curr);

      if(pvitem)
	{
	  undo_group *grp=new apt_undo_group;

	  pvitem->forbid_version(grp);

	  if(!grp->empty())
	    apt_undos->add_item(grp);
	  else
	    delete grp;

	  if(aptcfg->FindB(PACKAGE "::UI::Advance-On-Action", false))
	    level_line_down();

	  package_states_changed();

	  return true;
	}
      else
	return false;
    }
}

bool pkg_menu_tree::package_changelog_enabled()
{
  return get_visible() && pkg_or_ver_selected();
}

bool pkg_menu_tree::package_changelog()
{
  if(!get_visible())
    return false;

  pkg_tree_node *curr=selection();  

  if(!curr)
    return false;

  pkg_item *pitem=dynamic_cast<pkg_item*>(curr);
  if(pitem)
    {
      pitem->show_changelog();
      return true;
    }
  else
    {
      pkg_ver_item *pvitem=dynamic_cast<pkg_ver_item*>(curr);

      if(pvitem)
	{
	  view_changelog(pvitem->get_version());
	  return true;
	}
      else
	return false;
    }
}

bool pkg_menu_tree::package_information_enabled()
{
  return get_visible() && pkg_or_ver_selected();
}

bool pkg_menu_tree::package_information()
{
  if(!get_visible())
    return false;

  pkg_tree_node *curr=selection();  

  if(!curr)
    return false;

  pkg_item *pitem=dynamic_cast<pkg_item*>(curr);
  if(pitem)
    {
      pitem->show_information();
      return true;
    }
  else
    {
      pkg_ver_item *pvitem=dynamic_cast<pkg_ver_item*>(curr);

      if(pvitem)
	{
	  pvitem->show_information();
	  return true;
	}
      else
	return false;
    }
}

bool pkg_menu_tree::find_search_enabled()
{
  return get_visible();
}

bool pkg_menu_tree::find_search()
{
  prompt_string(_("Search for: "),
		"",
		SigC::slot(*this, &pkg_tree::do_search),
		SigC::slot(*this, &pkg_tree::do_cancel_incsearch),
		SigC::slot(*this, &pkg_tree::do_incsearch),
		&search_history);

  return true;
}

bool pkg_menu_tree::find_research_enabled()
{
  return last_search_matcher!=NULL;
}

bool pkg_menu_tree::find_research()
{
  if(last_search_matcher)
    {
      pkg_matcher_search searcher(last_search_matcher);
      search_for(searcher);

      return true;
    }
  else
    {
      beep();
      return true;
    }
}

bool pkg_menu_tree::find_limit_enabled()
{
  return false;
}

bool pkg_menu_tree::find_limit()
{
  return false;
}

bool pkg_menu_tree::find_reset_limit_enabled()
{
  return false;
}

bool pkg_menu_tree::find_reset_limit()
{
  return false;
}

bool pkg_menu_tree::find_broken_enabled()
{
  return get_visible();
}

bool pkg_menu_tree::find_broken()
{
  if(!get_visible())
    return false;

  do_search("~b");

  return true;
}

void pkg_menu_tree::do_search(std::string s)
{
  if(s.size()!=0)
    {
      delete last_search_matcher;
      last_search_term=s;
      last_search_matcher=parse_pattern(s);
    }

  if(doing_incsearch)
    doing_incsearch=false;
  else
    {
      if(last_search_term.size()!=0 && last_search_matcher)
	{
	  pkg_matcher_search searcher(last_search_matcher);
	  search_for(searcher);
	}
      else
	beep();
    }
}

void pkg_menu_tree::do_incsearch(std::string s)
{
  if(!aptcfg->FindB(PACKAGE "::UI::Incremental-Search", true))
    return;

  if(!doing_incsearch)
    {
      doing_incsearch=true;
      pre_incsearch_selected=get_selected();
    }

  pkg_matcher *m=parse_pattern(s, false);

  set_selection(pre_incsearch_selected);

  if(m)
    {
      pkg_matcher_search searcher(m);
      search_for(searcher);
    }

  delete m;
}

void pkg_menu_tree::do_cancel_incsearch()
{
  if(doing_incsearch)
    {
      set_selection(pre_incsearch_selected);
      doing_incsearch=false;
    }
}

void pkg_menu_tree::reset_incsearch()
{
  doing_incsearch=false;
  pre_incsearch_selected=get_end();
}

bool pkg_menu_tree::handle_char(chtype ch)
{
  // ick -- but having our own bindings is also ugly. hm.
  if(pkg_tree::bindings->key_matches(ch, "Search"))
    find_search();
  else if(pkg_tree::bindings->key_matches(ch, "ReSearch"))
    find_research();
  else if(pkg_tree::bindings->key_matches(ch, "SearchBroken"))
    find_broken();
  else
    return vs_tree::handle_char(ch);

  return true;
}

void pkg_tree::init_bindings()
{
  bindings=new keybindings(vs_tree::bindings);
}

void pkg_tree::init(const char *def_limit,
		    OpProgress &progress)
{
  if(def_limit)
    limitstr=def_limit;
  else
    limitstr=aptcfg->Find(PACKAGE "::Pkg-Display-Limit", "");

  if(!limitstr.empty())
    limit=parse_pattern(limitstr);

  if(grouping)
    build_tree(progress);

  cache_closed.connect(slot(*this, &pkg_tree::handle_cache_close));

  cache_reloaded.connect(SigC::hide_return(SigC::slot<bool, pkg_tree, pkg_tree>(*this, &pkg_tree::build_tree)));
}

pkg_tree::pkg_tree(const char *def_grouping,
		   pkg_grouppolicy_factory *_grouping,
		   const char *def_limit,
		   OpProgress &progress):
  grouping(_grouping), groupingstr(def_grouping),
  sorting(parse_sortpolicy(aptcfg->Find(PACKAGE "::UI::Default-Sorting",
					 "name"))),
  limit(NULL)
{
  init(def_limit, progress);
}

pkg_tree::pkg_tree(const char *def_grouping,
		   pkg_grouppolicy_factory *_grouping,
		   const char *def_limit):
  grouping(_grouping), groupingstr(def_grouping),
  sorting(parse_sortpolicy(aptcfg->Find(PACKAGE "::UI::Default-Sorting",
					 "name"))),
  limit(NULL)
{
  OpProgress p;

  init(def_limit, p);
}

void pkg_tree::handle_cache_close()
{
  set_root(NULL);
}

pkg_tree::~pkg_tree()
{
  delete sorting;
}

void pkg_tree::set_grouping(pkg_grouppolicy_factory *_grouping)
{
  // Reject NULL groupings (should never happen!)
  if(!_grouping)
    return;

  pkg_grouppolicy_factory *oldgrouping=grouping;

  grouping=_grouping;

  build_tree();

  delete oldgrouping;
}

void pkg_tree::set_grouping(string s)
{
  groupingstr=s;

  pkg_grouppolicy_factory *grp=parse_grouppolicy(s);

  if(grp)
    set_grouping(grp);
}

void pkg_tree::set_sorting(pkg_sortpolicy *_sorting)
{
  delete sorting;

  sorting=_sorting;

  // ummmm
  if(grouping)
    build_tree();
}

void pkg_tree::set_sorting(string s)
{
  pkg_sortpolicy *policy=parse_sortpolicy(s);

  if(policy)
    set_sorting(policy);
}

bool pkg_tree::build_tree(OpProgress &progress)
{
  bool rval;

  reset_incsearch();

  set_root(NULL);

  reset_incsearch();

  if(grouping && apt_cache_file)
    {
      bool empty=true, cache_empty=true;

      pkg_subtree *mytree=new pkg_subtree(_("All Packages"), true);
      pkg_grouppolicy *grouper=grouping->instantiate(&selected_signal,
						     &selected_desc_signal);

      mytree->set_depth(-1);

      int num=0;
      int total=(*apt_cache_file)->Head().PackageCount;

      for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin(); !i.end(); i++)
	{
	  cache_empty=false;

	  progress.OverallProgress(num, total, 1, _("Building view"));

	  if((!limit) || limit->matches(i, pkg_item::visible_version(i)))
	    {
	      empty=false;
	      grouper->add_package(i, mytree);
	    }
	}

      progress.OverallProgress(total, total, 1, _("Building view"));

      mytree->sort(pkg_sortpolicy_wrapper(sorting));

      set_root(mytree);

      delete grouper;

      rval=cache_empty || !empty;
    }
  else
    rval=true;

  vscreen_update();

  return rval;
}

bool pkg_tree::build_tree()
{
  vs_progress *p=gen_progress_bar();
  bool rval=build_tree(*p);
  p->destroy();

  return rval;
}

void pkg_tree::set_limit(string _limit)
{
  pkg_matcher *old_limit=limit;
  string old_limitstr=limitstr;

  pkg_matcher *new_limit=parse_pattern(_limit);
  if(_limit.empty() || new_limit)
    {
      limit=new_limit;
      limitstr=_limit;

      if(!build_tree())
	{
	  char buf[512];

	  snprintf(buf, 512, _("No packages matched the pattern \"%s\"."),
		   _limit.c_str());

	  show_message(buf);
	  delete new_limit;
	  limit=old_limit;
	  limitstr=old_limitstr;

	  build_tree();
	}
      else
	delete old_limit;
    }
}

bool pkg_tree::find_limit_enabled()
{
  return get_visible();
}

bool pkg_tree::find_limit()
{
  prompt_string(_("Enter the new package tree limit: "),
		limitstr,
		SigC::slot(*this, &pkg_tree::set_limit),
		NULL,
		NULL,
		&limit_history);

  return true;
}

bool pkg_tree::find_reset_limit_enabled()
{
  return get_visible() && limit!=NULL;
}

bool pkg_tree::find_reset_limit()
{
  if(!get_visible())
    return false;

  delete limit;
  limit=NULL;
  limitstr="";

  build_tree();

  return true;
}

bool pkg_tree::handle_char(chtype ch)
{
  if(bindings->key_matches(ch, "ChangePkgTreeLimit"))
    find_limit();
  else if(bindings->key_matches(ch, "ChangePkgTreeGrouping"))
    prompt_string(_("Enter the new package grouping mechanism for this display: "),
		  groupingstr,
		  SigC::slot(*this,
			     (void (pkg_tree::*) (string)) &pkg_tree::set_grouping),
		  NULL,
		  NULL,
		  &grouping_history);
  else if(bindings->key_matches(ch, "ChangePkgTreeSorting"))
    prompt_string(_("Enter the new package sorting mechanism for this display: "),
		  "",
		  SigC::slot(*this,
			     (void (pkg_tree::*) (string)) &pkg_tree::set_sorting),
		  NULL,
		  NULL,
		  &sorting_history);
  else
    return pkg_menu_tree::handle_char(ch);

  return true;
}
